#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <poll.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <libraw1394/raw1394.h>
// For reading the utsname struct of the remote system:
#include <sys/utsname.h>

#include "fireproxy.h"

#define CSR_BASE	0xfffff0000000ULL
#define NR_CPUS		32
#ifdef __x86_64__
#define START_KERNEL_MAP	0xffffffff80000000UL
#define KERNELBASE		0xffff810000000000UL
#else
#define KERNELBASE	0xc0000000
#endif

#define inline

enum {
	sym_bad  = 0,
	sym_text = 0x00000001,
	sym_data = 0x00000002,
	sym_bss  = 0x00000004
};

extern raw1394handle_t		g_handle;
extern nodeid_t			g_target;
extern int			g_target_ok;
extern u_int64_t		g_target_uid;
extern int			g_change_occurred;
extern int verbose,rem_ptrsize;

FILE * logfile;
FILE * outfile;

unsigned long phys_to_virt(unsigned long phys_addr)
{
	return (phys_addr + KERNELBASE);
}

unsigned long virt_to_phys(unsigned long virt_addr)
{
#ifdef __x86_64__
	return virt_addr >= START_KERNEL_MAP ? 
		virt_addr - START_KERNEL_MAP : virt_addr - KERNELBASE;
#else
	return virt_addr - KERNELBASE;
#endif
}

int verbose = 0;
int rem_ptrsize = sizeof(void *);
int auto_update;

raw1394handle_t		g_handle;
nodeid_t		g_target;
int			g_target_ok;
u_int64_t		g_target_uid;
int			g_change_occurred;
char*			g_sysmap;
size_t			g_sysmap_size;
int			g_machine;

static int
got_bus_reset(raw1394handle_t hndl, unsigned int generation)
{
	raw1394_update_generation(g_handle, generation);
	g_target_ok = 0;
	g_change_occurred = 1;
	printf("Bus reset !\n");
	return 0;
}

static int
setup_handle(int port_num)
{
	int port_count;
	struct raw1394_portinfo ports[16];
	
	g_handle = raw1394_new_handle();
	if (!g_handle) {
		perror("libraw1394 couldn't be initialized");
		return -1;
	}

	port_count = raw1394_get_port_info(g_handle, ports, 16);
	if (port_count <= port_num) {
		fprintf(logfile, "Port %d not available (%d ports detected).\n",
			port_num, port_count);
		return -1;
	}
	raw1394_set_port(g_handle, port_num);
	printf("Port %d (%s) opened, %d nodes detected\n", port_num,
		ports[port_num].name, ports[port_num].nodes);

	raw1394_set_bus_reset_handler(g_handle, got_bus_reset);

	return 0;
}

static void
select_target(void)
{
	int i, tgt, count, local;

	count = raw1394_get_nodecount(g_handle);
	local = raw1394_get_local_id(g_handle) & 0x3f;
	
	printf("%d nodes available, local node is: %d\n", count, local);
	for (i=0; i<count; i++) {
		quadlet_t uuid[2];
		int rc;

		printf(" %d: %04x, uuid: ", i, i | 0xffc0);
		rc = raw1394_read(g_handle, i | 0xffc0, CSR_BASE + 0x40c, 4, &uuid[0]);
		if (rc >= 0)
			rc = raw1394_read(g_handle, i | 0xffc0, CSR_BASE + 0x410, 4, &uuid[1]);
		if (!rc)
			printf("%08x %08x%s\n", uuid[0], uuid[1], i == local ? " [LOCAL]" : "");
		else {
			fflush(stdout);
			perror("Error reading uuid");
		}
	}
	printf("pick a target node: ");
	tgt = 0;
	if (tgt == local) {
		tgt++;
	}
	if (tgt < 0 || tgt >= raw1394_get_nodecount(g_handle)) {
		printf("wrong node number !\n");
		exit(4);
        }
	else if (tgt == local) {
		printf("can't pick local node !\n");
		exit(4);
	}
	g_target = tgt | 0xffc0;
	g_target_ok = 1;
	g_change_occurred = 1;
}

static void
status(void)
{
	printf("Target : ");
	if (g_target_ok)
		printf("%04x\n", g_target);
	else
		printf("<unspecified>\n");

	printf("Gen    : %d\n", raw1394_get_generation(g_handle));
}

static unsigned long
__find_symbol(const char* symbol, int *sym_type, int exact)
{
	const char *p, *cur;
	const char *match = 0; // just to make gcc happy
	int goodness = 0;
	unsigned long result = 0;
	
	if (!g_sysmap || !g_sysmap_size)
		return 0;

	cur = g_sysmap;
	while(cur) {
		cur = strstr(cur, symbol);
		if (cur) {
			int gd = 1;

			/* best match if equal, better match if
			 * begins with
			 */
			if (cur == g_sysmap || *(cur-1) == ' ') {
				gd++;
				if (cur[strlen(symbol)] == 10)
					gd++;
			}
			if (gd == 3 || (!exact && gd > goodness)) {
				match = cur;
				goodness = gd;
				if (gd == 3)
					break;
			}
			cur++;
		}
	}	
	if (goodness) {
		p = match;
		while(p > g_sysmap && *p != 10)
			p--;
		if (*p == 10) p++;
		errno = 0;
		result = strtoul(p, (char **)&p, 16);
		if (errno == ERANGE) {
			printf("Symbol %s out of range. Non matching pointer size?\n",symbol);
			return 0;
		}
		if (result > 0xffffffff) 
			rem_ptrsize = 8; 
		else
			rem_ptrsize = 4;
			
		if (sym_type) {
			while(*p == ' ')
				p++;
			switch(toupper(*p)) {
				case 'T': *sym_type = sym_text; break;
				case 'D': *sym_type = sym_data; break;
				case 'B': *sym_type = sym_bss; break;
				default:  *sym_type = sym_bad; break;
			}
		}
	}
	return result;
}		

static unsigned long
find_symbol(const char* symbol, int sym_type)
{
	int found_type;
	unsigned long result;
	
	result = __find_symbol(symbol, &found_type, 1);
	if (result && ((found_type & sym_type) == 0))
		result = 0;
	return result;
}

int fwmem_read(unsigned long target_addr, void *data)
{
	if (!g_target_ok) {
		printf("fwmem_read - no target...\n");
		return -2;
	}
	if (verbose)
		printf("fwmem_read: target_addr %lx -> %p\n", target_addr, data);
	int rc = raw1394_read(g_handle, g_target,
			target_addr, 4, data);
	if (rc < 0) {
		perror("fwmem_read/raw1394_read");
		printf("remote read failed, target_addr=%lx\n", target_addr);
	} else
		if (verbose)
			printf("fwmem_read: SUCCESS: %lx\n", *(unsigned long *)data);
	return rc;
}

void* fwmem_readblk(void* dest, unsigned long target_addr, unsigned int len)
{
	unsigned int* cdst = (unsigned int*)dest;
	unsigned long end = target_addr + len;

	//target_addr = virt_to_phys(target_addr);

	int startoffset = (unsigned long)target_addr & 0x3;

	target_addr -= startoffset;

	if (!g_target_ok) {
		printf("readblk - no target...\n");
		return NULL;
	}
	while(target_addr < end) {
		if (fwmem_read(target_addr, cdst)) {
			printf("readblk - aborting...\n");
			return NULL;
		}
		cdst++;
		target_addr += 4;
	}
	return dest + startoffset;
}

static int fwmem_writechar(unsigned long target_addr, char ch)
{
	printf("fwmem_writechar( 0x%08lx << '%c' )\n", target_addr, ch);
	int rc = raw1394_write(g_handle, g_target, target_addr, 1, (quadlet_t *)&ch);
	if (rc < 0) {
		perror("fwmem_writechar/raw1394_write");
	}
	return rc;
}
int fwmem_writeblk(unsigned long target_addr, void *data, unsigned int len)
{
	char *dest = (char *)target_addr, *ptr = data;
	int rc = 0;
	printf("fwmem_writeblk( 0x%08lx << 0x%p, %d )\n", target_addr, data, len);
	while (len-- > 0 &&
		!(rc = fwmem_writechar((unsigned long)dest++, *ptr++)));
		//!(rc = raw1394_write(g_handle, g_target, dest++, 1, (quadlet_t *)ptr++)));
	if (rc < 0) {
		perror("fwmem_writeblk");
	}
	return rc;
}

/*
struct utsname {
        char sysname[65];
        char nodename[65];
        char release[65];
        char version[65];
        char machine[65];
        char domainname[65];
};
*/
typedef struct utsname utsname_t;

static void
attach_kernel(void)
{
	unsigned long target_addr;
	
	if (!g_target_ok)
		return;

	target_addr = find_symbol("_machine", sym_data | sym_bss);
	if (target_addr == 0) { 
		printf("not a ppc\n");
	} else {
		printf("Got _machine at %lx, reading %lx...\n",
			target_addr, virt_to_phys(target_addr));
		fwmem_read(virt_to_phys(target_addr), &g_machine);
		printf("g_machine: %x\n", g_machine);
		switch(g_machine) {
			case 1:
				printf("Found PReP kernel\n"); break;
			case 2:
				printf("Found PMac kernel\n"); break;
			case 4:
				printf("Found CHRP kernel\n"); break;
			default:
				printf("Unrecognized kernel type\n");
				return;
		}
	}
	// Todo: add error handling from here to strcmp:
	nodeaddr_t system_utsname_addr = find_symbol("system_utsname",
			sym_data|sym_bss);
	printf("utsname addr: %lx\n", system_utsname_addr);
	system_utsname_addr = virt_to_phys( system_utsname_addr);
	printf("utsname addr (phys): %lx\n", system_utsname_addr);

	utsname_t *utsname;
	utsname = malloc (sizeof(utsname_t));
	memset(utsname,0,sizeof(utsname_t));

	fwmem_readblk(utsname, system_utsname_addr, sizeof(utsname_t));

	if (strcmp(utsname->sysname, "Linux")) {
		printf("System.map does not match remote system!\n");
	} else
		printf("Attached to node '%s'\nSystem : %s\nVersion: %s (%s)\n",
			utsname->nodename,
			utsname->machine,
			utsname->release,
			utsname->version);

#if 0
	// Testing basic write functinality:
	char * test;

	fwmem_writeblk(system_utsname_addr+65*4, "x86_64", 7);
	test = fwmem_readblk(utsname->machine, system_utsname_addr+65*4, 65);
	fprintf(stderr,"Attached to machine '%s'\n", test);

	fwmem_writechar(system_utsname_addr+65*4, 'x');
	fwmem_writechar(system_utsname_addr+65*4+1, '8');
	test = fwmem_readblk(utsname->machine, system_utsname_addr+65*4, 65);
	fprintf(stderr,"Attached to machine '%s'\n", test);
#endif
	free(utsname);
}

static int
hex (unsigned char ch)
{
  if ((ch >= 'a') && (ch <= 'f'))
    return (ch - 'a' + 10);
  if ((ch >= '0') && (ch <= '9'))
    return (ch - '0');
  if ((ch >= 'A') && (ch <= 'F'))
    return (ch - 'A' + 10);
  return (-1);
}

/*
 * scan for the sequence $<data>#<checksum>
 */
static int eval_inbyte(char ch)
{
	static unsigned char checksum;
	static unsigned char xmitcsum;
	static int count = BUFMAX; // need '$' before receiving
	static unsigned char buffer[BUFMAX+1];
	static int checksum_nibble;
	
		
	if ((ch & 0x7f) == '$') {
		checksum_nibble = -1; // no checksum_nibble received
		checksum = count = 0; // allow receiving
		return 0;
	}

	if (count >= BUFMAX) {
		if (ch != '+')
			fprintf(logfile, "received '%c' but not receiving\n", ch);
		return 0;
	}

	if (ch == '#') {
		checksum_nibble = 0; // next byte is 1st checksum_nibble byte
		return 0;
	}

	switch (checksum_nibble) {
		case -1:
			checksum = checksum + ch;
			buffer[count++] = ch;
			return 0;
		case 0:
			xmitcsum = hex(ch & 0x7f) << 4;
			checksum_nibble = 1; // next byte is 2nd checksum_nibble byte
			return 0;
        }
	xmitcsum |= hex(ch & 0x7f);

	if (checksum != xmitcsum) {
		fprintf(logfile, "checksum error: %x != %x, string:%s\n",
				checksum, xmitcsum, buffer);
		putDebugChar('-');	/* failed checksum */
		checksum = 0;
		count = BUFMAX;
		return 0;
	}
	putDebugChar('+'); /* send ACK for successful transfer */
	fflush(outfile);

	buffer[count] = 0;
	count = BUFMAX;

	if (checksum == xmitcsum)  {
		//fprintf(logfile, "command received: %s\n", buffer);
		handle_command(buffer);
	}
	return (checksum == xmitcsum);
}
/*
 * scan for the sequence $<data>#<checksum>
 */
static void getpacket(int infd)
{
	int count = 0;
	static unsigned char readbuffer[BUFMAX+1];
	ssize_t retval;

	do
		retval = read(infd, &readbuffer, BUFMAX);
	while (retval == -EAGAIN || retval == -EINTR);

	if (retval == 0) {
		fprintf(logfile, "connection closed by peer\n");
		exit(2);
	}
	if (retval < 0) {
			perror("getpacket");
			exit(2);
	}
	
	if (verbose) {
		for (count = 0; count < retval;) {
			unsigned char ch = readbuffer[count++];
			if (ch == '$')
				putchar (' ');
			else
				putchar (ch);
		}
		putchar ('\n');
	}

	for (count = 0; count < retval;)
		eval_inbyte (readbuffer[count++]);
}

void process_console_input(int consoleinfd)
{
        static unsigned char readbuffer[BUFMAX];
        ssize_t retval;

        do
                retval = read(consoleinfd, &readbuffer, BUFMAX);
        while (retval == -EAGAIN || retval == -EINTR);

	if (*readbuffer == 'v')
		verbose = ~verbose;
	fprintf(logfile, "verbose %s\n", verbose ? "on" : "off");
	// dummy code, does not work right yet:
	if (*readbuffer == 'd') {
		FILE * dumpfile = fopen("fwmem.dump", "w");
		void * readret;
		unsigned long addr;
		for (addr=0; addr < 1024*1024*1024; addr += BUFMAX) {
			memset(readbuffer, 0, BUFMAX);
			readret = fwmem_readblk(readbuffer, addr, BUFMAX);
			size_t wret = fwrite(readbuffer, BUFMAX, 1, dumpfile);
			if (readret) {
				printf("read error\n");
				break;
			}
			if (wret != BUFMAX) {
				perror("fwrite");
				break;
			}
		}
		printf("closing dumpfile\n");
		fclose(dumpfile);
	}
}
#define test_gccbug 0
#if test_gccbug
void rx(nodeaddr_t addr, int len)
{
	printf("readblx(%x - size=%u)\n", addr, len);
	sleep(9);
	exit(6);
}
#endif
int main(int argc, char** argv)
{
	int fd, rc;

#if test_gccbug
	rx(6, 8);
#endif
	outfile = stdout;

	if (setup_handle(0) != 0)
		return 0;
	fd = raw1394_get_fd(g_handle);

	if (argc < 2) {
			printf("%s: If you have, please pass the System.map to fireproxy\n", argv[0]);
	}
	if (argc >= 2) {
		int smfd;
		/* Don't use stat because that doesn't work on /proc/kallsyms */
		g_sysmap_size = 4 << 20; 
	again:
		smfd = open(argv[1], O_RDONLY);
		if (smfd < 0) {
			printf("Can't load system.map <%s>", argv[1]);
			exit(5);
		} else {
			g_sysmap = malloc(g_sysmap_size);
			long n = read(smfd, g_sysmap, g_sysmap_size);
			close(smfd);
			if (n == g_sysmap_size)  {
				if (g_sysmap_size > 128<<20)
					exit(1);
				g_sysmap_size <<= 1;
				goto again;
			}
			g_sysmap_size = n;
		}
		printf("Loaded system.map <%s> <%lu> bytes\n", argv[1], g_sysmap_size);
	}

	select_target();
	attach_kernel();
	status();

	logfile = fdopen(5, "a");
	if (!logfile) {
		logfile = stderr;
	}

	// For now, we accept a connect on TCP port 4:
        remote_open(4);
        remote_accept();

	printf("fireproxy console ready, press 'v<ENTER>' to toggle verbose\n");

	do {
#define NUM_POLLFDS 3
		struct pollfd pfds[NUM_POLLFDS];
		int loop = 0;

		memset(&pfds, 0, sizeof(pfds));
		pfds[0].fd = STDIN_FILENO;
		pfds[0].events = POLLIN;
		pfds[1].fd = fd;
		pfds[1].events = POLLIN|POLLPRI|POLLERR|POLLHUP;
		pfds[2].fd = remote_desc;
		pfds[2].events = POLLIN|POLLPRI|POLLERR|POLLHUP;

		int timeout = -1;

		rc = poll(pfds, NUM_POLLFDS, timeout);
		if (rc < 0) {
			rc = errno;
			if (rc != EINTR) {
				printf("poll error %d, exiting...\n", rc);
				goto bail;
			}
		} else if (rc > 0) {
			// For giving commands in stdin, left there for debugging:
			if (pfds[0].revents != 0)
				process_console_input(pfds[0].fd);
			// Data from network (gdb) received:
			if (pfds[2].revents != 0) {
				outfile = fdopen(pfds[2].fd, "r+");
				getpacket(pfds[2].fd);
			}
			// Data from raw1394 device to read, call raw1394_loop_iterate:
			if (pfds[1].revents != 0)
				loop = 1;
			// For allowing (re)connect later in loop(later):
			//if (pfds[3].revents != 0) remote_accept();
		}
		if (loop)
			raw1394_loop_iterate(g_handle);
	} while(1);	

bail:
	printf("Exiting...\n");
	raw1394_destroy_handle(g_handle);
	free(g_sysmap);
	return 0;
}
