/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Original Copyright (c) 2005 Covalent Technologies
 *
 * FTP Protocol module for Apache 2.0
 */

#include "mod_ftp.h"
#include "ftp_internal.h"

#include "ap_mpm.h"             /* For MPM query interface */
#include "apr_dbm.h"
#if !(defined(WIN32) || defined(NETWARE))
#include "unixd.h"
#endif

#define FTP_SERVER_LIMIT_KEY "FireballXL5OnDVD"
#define FTP_DB_FILE_MODE ( APR_UREAD | APR_UWRITE | APR_GREAD | APR_WREAD )

/*
 * We also use the below as a state variable. Ugly.
 */
static apr_global_mutex_t *ftp_lock = NULL;

static apr_status_t ftp_mutex_init(server_rec *s, apr_pool_t *p)
{
    ftp_server_config *fsc = ftp_get_module_config(s->module_config);

    if (fsc->limit_perip || fsc->limit_peruser || fsc->limit_perserver)
        return apr_global_mutex_create(&ftp_lock,
                            apr_pstrcat(p, fsc->limitdbfile, ".LoCK", NULL),
                                       APR_LOCK_DEFAULT, p);
    else
        return APR_SUCCESS;
}

static apr_status_t ftp_mutex_on(void)
{
    return apr_global_mutex_lock(ftp_lock);
}

static apr_status_t ftp_mutex_off(void)
{
    return apr_global_mutex_unlock(ftp_lock);
}

static apr_status_t ftp_db_init(server_rec *s, apr_pool_t *p)
{
    apr_status_t rv;
    apr_dbm_t *dbf;
    ftp_server_config *fsc = ftp_get_module_config(s->module_config);

    /*
     * Noop in cases where we have not
     * setup or enabled the actual mutex. We know
     * that this is (still) NULL if ftp_mutex_init() determined
     * it didn't need to bother with login limits.
     */
    if (!ftp_lock)
        return APR_SUCCESS;

    ftp_mutex_on();
    if ((rv = apr_dbm_open(&dbf, fsc->limitdbfile,
                   APR_DBM_RWCREATE, FTP_DB_FILE_MODE, p)) != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, s,
                     "Cannot create FTPLimitDBFile file `%s'",
                     fsc->limitdbfile);
        ftp_mutex_off();
        return rv;
    }
    apr_dbm_close(dbf);
#if !defined(OS2) && !defined(WIN32) && !defined(BEOS) && !defined(NETWARE)
    if (geteuid() == 0) {
        int ign;
#if MODULE_MAGIC_NUMBER_MAJOR < 20081201
#define ap_unixd_config unixd_config
#endif
        ign = chown(fsc->limitdbfile, ap_unixd_config.user_id, -1);
        ign = chown(apr_pstrcat(p, fsc->limitdbfile, ".LoCK", NULL),
                    ap_unixd_config.user_id, -1);
        ign = chown(apr_pstrcat(p, fsc->limitdbfile, ".db", NULL),
                    ap_unixd_config.user_id, -1);
        ign = chown(apr_pstrcat(p, fsc->limitdbfile, ".dir", NULL),
                    ap_unixd_config.user_id, -1);
        ign = chown(apr_pstrcat(p, fsc->limitdbfile, ".pag", NULL),
                    ap_unixd_config.user_id, -1);
    }
#endif
    ftp_mutex_off();

    return APR_SUCCESS;
}

apr_status_t ftp_mutexdb_init(server_rec *s, apr_pool_t *p)
{
    apr_status_t rv;
    if ((rv = ftp_mutex_init(s, p)) != APR_SUCCESS)
        return rv;
    return ftp_db_init(s, p);
}

apr_status_t ftp_mutexdb_child_init(server_rec *s, apr_pool_t *p)
{
    ftp_server_config *fsc = ftp_get_module_config(s->module_config);

    if (!ftp_lock)
        return APR_SUCCESS;
    else
        return apr_global_mutex_child_init(&ftp_lock, fsc->limitdbfile, p);
}

apr_status_t ftp_mutexdb_cleanup(void *data)
{
    server_rec *s = data;
    ftp_server_config *fsc = ftp_get_module_config(s->module_config);
    apr_pool_t *p;

    if (ftp_lock) {
        apr_global_mutex_destroy(ftp_lock);
        apr_pool_create_ex(&p, s->process->pool, NULL, NULL);
        apr_pool_tag(p, "ftp_mutex");
        if (p) {
            unlink(apr_pstrcat(p, fsc->limitdbfile, ".db", NULL));
            unlink(apr_pstrcat(p, fsc->limitdbfile, ".dir", NULL));
            unlink(apr_pstrcat(p, fsc->limitdbfile, ".pag", NULL));
            unlink(apr_pstrcat(p, fsc->limitdbfile, ".LoCK", NULL));
            unlink(fsc->limitdbfile);
            apr_pool_destroy(p);
        }
        ftp_lock = NULL;
    }

    return APR_SUCCESS;
}

#define MYMIN(a,b) ( (a) < (b) ? (a) : (b) )

ftp_loginlimit_t ftp_limitlogin_check(const char *user, request_rec *r)
{
    apr_status_t rv;
    conn_rec *c = r->connection;
    apr_datum_t ukey;
    apr_datum_t ikey;
    apr_datum_t skey;
    apr_datum_t val;
    apr_dbm_t *dbf;
    char temp[10];              /* Note: This means only values <=
                                 * 999,999,999 */
    int uval = 0;
    int ival = 0;
    int sval = 0;
    char *tkey;
    char *sname = (r->server->server_hostname
                   ? r->server->server_hostname
                   : "unknown");
    ftp_server_config *fsc =
    ftp_get_module_config(r->server->module_config);

    if (!ftp_lock)
        return FTP_LIMIT_OK;

    ftp_mutex_on();
    if ((rv = apr_dbm_open(&dbf, fsc->limitdbfile,
             APR_DBM_RWCREATE, FTP_DB_FILE_MODE, r->pool)) != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server,
                     "Cannot open FTPLimitDBFile file `%s' for login check",
                     fsc->limitdbfile);
        ftp_mutex_off();
        return FTP_LIMIT_ERROR;
    }
    /*
     * First we check the user settings.
     * This is a safe cast, this is a lookup key.
     */
    tkey = apr_psprintf(r->pool, "%s-%s", sname, user);
    ukey.dptr = (char *) tkey;
    ukey.dsize = strlen(tkey);
    rv = apr_dbm_fetch(dbf, ukey, &val);        /* error for non-existant? */
    if (val.dptr != NULL && val.dsize > 0) {
        apr_cpystrn(temp, val.dptr, MYMIN(sizeof(temp), val.dsize + 1));
        uval = atoi(temp);
    }
    if (fsc->limit_peruser && uval >= fsc->limit_peruser) {
        ftp_mutex_off();
        return FTP_LIMIT_HIT_PERUSER;
    }
    /*
     * Now we check the IP settings.
     * This is a safe cast, this is a lookup key.
     */
    tkey = apr_psprintf(r->pool, "%s-%s", sname, c->remote_ip);
    ikey.dptr = (char *) tkey;
    ikey.dsize = strlen(tkey);
    rv = apr_dbm_fetch(dbf, ikey, &val);        /* error for non-existant? */
    if (val.dptr != NULL && val.dsize > 0) {
        apr_cpystrn(temp, val.dptr, MYMIN(sizeof(temp), val.dsize + 1));
        ival = atoi(temp);
    }
    if (fsc->limit_perip && ival >= fsc->limit_perip) {
        ftp_mutex_off();
        return FTP_LIMIT_HIT_PERIP;
    }
    /*
     * OK, so we're not up against the per user or IP limit,
     * we need to check the perserver limit then
     */

    tkey = apr_psprintf(r->pool, "%s-%s", sname, FTP_SERVER_LIMIT_KEY);
    skey.dptr = (char *) tkey;
    skey.dsize = strlen(tkey);
    rv = apr_dbm_fetch(dbf, skey, &val);        /* error for non-existant? */
    if (val.dptr != NULL && val.dsize > 0) {
        apr_cpystrn(temp, val.dptr, MYMIN(sizeof(temp), val.dsize + 1));
        sval = atoi(temp);
    }
    if (fsc->limit_perserver && sval >= fsc->limit_perserver) {
        ftp_mutex_off();
        return FTP_LIMIT_HIT_PERSERVER;
    }

    /*
     * Oh joy. Oh rapture. We have room for this person. We
     * now go ahead and update these values in the DB "atomically".
     * If not (that is, if we check and then, if OK, we *then*
     * update), we hit a race condition.
     */
    sval++;
    uval++;
    ival++;
    apr_snprintf(temp, sizeof(temp), "%d", uval);
    val.dptr = temp;
    val.dsize = strlen(temp);
    rv = apr_dbm_store(dbf, ukey, val);

    apr_snprintf(temp, sizeof(temp), "%d", ival);
    val.dptr = temp;
    val.dsize = strlen(temp);
    rv = apr_dbm_store(dbf, ikey, val);

    apr_snprintf(temp, sizeof(temp), "%d", sval);
    val.dptr = temp;
    val.dsize = strlen(temp);
    rv = apr_dbm_store(dbf, skey, val);

    apr_dbm_close(dbf);

    ftp_mutex_off();

    return FTP_LIMIT_OK;
}

int ftp_limitlogin_loggedout(conn_rec *c)
{
    ftp_connection *fc = ftp_get_module_config(c->conn_config);
    apr_status_t rv;
    apr_datum_t ukey;
    apr_datum_t ikey;
    apr_datum_t skey;
    apr_datum_t val;
    apr_dbm_t *dbf;
    char temp[10];              /* Note: This means only values <=
                                 * 999,999,999 */
    int uval = 0;
    int ival = 0;
    int sval = 0;
    char *tkey;
    char *sname = (fc->orig_server->server_hostname
                   ? fc->orig_server->server_hostname
                   : "unknown");
    ftp_server_config *fsc =
    ftp_get_module_config(fc->orig_server->module_config);

    if (!ftp_lock)
        return 0;

    ftp_mutex_on();
    if ((rv = apr_dbm_open(&dbf, fsc->limitdbfile,
                        APR_DBM_RWCREATE, FTP_DB_FILE_MODE, fc->login_pool))
        != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, fc->orig_server,
               "Cannot open FTPLimitDBFile file `%s' for logged out update",
                     fsc->limitdbfile);
        ftp_mutex_off();
        return rv;
    }
    /*
     * This is a safe cast, it's a lookup key
     */
    tkey = apr_psprintf(c->pool, "%s-%s", sname, fc->user);
    ukey.dptr = (char *) tkey;
    ukey.dsize = strlen(tkey);
    rv = apr_dbm_fetch(dbf, ukey, &val);        /* error for non-existant? */
    if (val.dptr != NULL && val.dsize > 0) {
        apr_cpystrn(temp, val.dptr, MYMIN(sizeof(temp), val.dsize + 1));
        uval = atoi(temp);
    }

    tkey = apr_psprintf(c->pool, "%s-%s", sname, c->remote_ip);
    ikey.dptr = (char *) tkey;
    ikey.dsize = strlen(tkey);
    rv = apr_dbm_fetch(dbf, ikey, &val);        /* error for non-existant? */
    if (val.dptr != NULL && val.dsize > 0) {
        apr_cpystrn(temp, val.dptr, MYMIN(sizeof(temp), val.dsize + 1));
        ival = atoi(temp);
    }

    tkey = apr_psprintf(c->pool, "%s-%s", sname, FTP_SERVER_LIMIT_KEY);
    skey.dptr = tkey;
    skey.dsize = strlen(tkey);
    rv = apr_dbm_fetch(dbf, skey, &val);        /* error for non-existant? */
    if (val.dptr != NULL && val.dsize > 0) {
        apr_cpystrn(temp, val.dptr, MYMIN(sizeof(temp), val.dsize + 1));
        sval = atoi(temp);
    }
    sval--;
    uval--;
    ival--;
    if (sval < 0)
        sval = 0;
    if (uval < 0)
        uval = 0;
    if (ival < 0)
        ival = 0;

    apr_snprintf(temp, sizeof(temp), "%d", uval);
    val.dptr = temp;
    val.dsize = strlen(temp);
    rv = apr_dbm_store(dbf, ukey, val);

    apr_snprintf(temp, sizeof(temp), "%d", ival);
    val.dptr = temp;
    val.dsize = strlen(temp);
    rv = apr_dbm_store(dbf, ikey, val);

    apr_snprintf(temp, sizeof(temp), "%d", sval);
    val.dptr = temp;
    val.dsize = strlen(temp);
    rv = apr_dbm_store(dbf, skey, val);

    apr_dbm_close(dbf);

    ftp_mutex_off();

    return 0;
}
