/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package com.sun.electric.database.hierarchy

import com.sun.electric.database.CellBackup
import com.sun.electric.database.EditingPreferences
import com.sun.electric.database.ImmutableArcInst
import com.sun.electric.database.ImmutableExport
import com.sun.electric.database.ImmutableNodeInst
import com.sun.electric.database.Snapshot
import com.sun.electric.database.geometry.EPoint
import com.sun.electric.database.geometry.Poly
import com.sun.electric.database.hierarchy.BatchChanges.NodeReplacement
import com.sun.electric.database.id.CellId
import com.sun.electric.database.id.ExportId
import com.sun.electric.database.id.NodeProtoId
import com.sun.electric.database.id.PortProtoId
import com.sun.electric.database.id.PrimitiveNodeId
import com.sun.electric.database.id.PrimitivePortId
import com.sun.electric.database.text.Name
import com.sun.electric.database.topology.NodeInst
import com.sun.electric.technology.PrimitiveNode
import com.sun.electric.util.math.FixpTransform
import com.sun.electric.util.math.GenMath
import com.sun.electric.util.math.Orientation

import org.slf4j.LoggerFactory
import scala.collection.JavaConverters._
import scala.collection.immutable.TreeMap
import scala.collection.mutable.ArrayBuffer

/**
 * This class replaces nodes in immutable database according to Node replacement requests.
 * @param oldSnapshot Snapshot to transform
 * @param ep EditingPreferences
 */
class ReplaceBuilder(oldSnapshot: Snapshot, ep: EditingPreferences) {
  val logger = LoggerFactory.getLogger(classOf[ReplaceBuilder])
  // oldSnapshot
  val oldTool = oldSnapshot.tool
  val oldEnvironment = oldSnapshot.environment
  val oldTechPool = oldEnvironment.techPool
  val oldCellTrees = oldSnapshot.cellTrees
  
  /**
   * @param replacements an Iterable of Node replacement requests.
   */
  def update(replacements: Iterable[NodeReplacement]): Snapshot = {
    // Split node replacement requests by CellIds
    val map = scala.collection.mutable.Map[CellId,scala.collection.mutable.Map[Int,NodeReplacement]]()
    for (r <- replacements) {
      val cellId = r.cellId
      val nodeId = r.nodeId
      val map1 = map.getOrElseUpdate(cellId, scala.collection.mutable.Map[Int,NodeReplacement]())
      map1.put(nodeId, r)
    }
    // Traverse cells fron bottom to top and replace nodes there 
    val curCellBackups = new Array[CellBackup](oldCellTrees.size)
    for (cellId <- oldSnapshot.getCellsDownTop.asScala) {
      val oldCellBackup = oldSnapshot.getCell(cellId)
      curCellBackups(cellId.cellIndex) = map.get(cellId) match {
        case Some(m) => // there are replacement requests in this Cell
          val cb = new CellBuilder(cellId)
          cb.update(m)
        case _ => // leave this Cell unchanged
          oldCellBackup
      }
    }
    // Gather updated CellBackups in a new Snapshot
    val newSnapshot = oldSnapshot.`with`(oldTool, oldEnvironment, curCellBackups, null)
    logger.debug("Snapshot is ready")
    newSnapshot
  }
  
  /**
   * Class to perform node replacements in a Cell
   * @param cellId CellId of the Cell
   */
  class CellBuilder(cellId: CellId) {
    val oldCellTree = oldSnapshot.getCellTree(cellId)
    val oldCellBackup = oldCellTree.top
    val oldCellRevision = oldCellBackup.cellRevision
    val oldCell = oldCellRevision.d
    // nodes
    val deletedNodeInds = scala.collection.mutable.BitSet()
    var curNodesCount = oldCellRevision.nodes.size
    var lastNodeId = oldCellRevision.getMaxNodeId
    var maxNodeSuffixesOrdered = TreeMap[Name,MaxNodeSuffix]()
    val replacedNodes = scala.collection.mutable.Map[Int,ImmutableNodeInst]()
    // arcs
    val deletedArcInds = scala.collection.mutable.BitSet()
    var curArcsCount = oldCellRevision.arcs.size
    var lastArcId = oldCellRevision.getMaxArcId
    val arcInsertionPoint = searchArcInsertionPoint(ImmutableArcInst.BASENAME.toString)
    var maxArcSuffix = -1
    if (arcInsertionPoint > 0) {
      val name = oldCellRevision.arcs.get(arcInsertionPoint - 1).name
      if (name.isTempname) {
        assert(name.getBasename == ImmutableArcInst.BASENAME)
        maxArcSuffix = name.getNumSuffix
      }
    }
    val addedArcs = ArrayBuffer[ImmutableArcInst]()
    val replacedArcs = scala.collection.mutable.Map[Int,ImmutableArcInst]()
    // exports
    val deletedExportInds = scala.collection.mutable.BitSet()
    var curExportsCount = oldCellRevision.exports.size
    val replacedExports = scala.collection.mutable.Map[ExportId,ImmutableExport]()
    
    /**
     * Update CellBackup according to a Map from nodeId to NodeReplacement
     * @param replacements a Map from nodeId to NodeReplacement
     * @return updated CellBackup
     */
    def update(replacements: scala.collection.mutable.Map[Int,NodeReplacement]): CellBackup = {
      // replace nodes
      val nit = oldCellRevision.nodes.iterator
      var nodeInd = 0
      while (nit.hasNext) {
        val n = nit.next
        val r = replacements.getOrElse(n.nodeId, null)
        if (r != null)
          replaceNode(nodeInd, n, r)
        nodeInd += 1
      }
      
      // replace arcs
      val ait = oldCellRevision.arcs.iterator
      var arcInd = 0
      while (ait.hasNext) {
        val a = ait.next
        val newTailPortId = portMap(replacements, a.tailNodeId, a.tailPortId)
        val newHeadPortId = portMap(replacements, a.headNodeId, a.headPortId)
        if (newTailPortId == null || newHeadPortId == null) {
          deletedArcInds += arcInd
          curArcsCount -= 1
        } else if (newTailPortId != a.tailPortId || newHeadPortId != a.headPortId) {
          val newA = replaceArc(a, newTailPortId, newHeadPortId)
          replacedArcs.put(a.arcId, newA)
        }
        arcInd += 1
      }
      
      // replace exports
      val eit = oldCellRevision.exports.iterator
      var exportInd = 0
      while (eit.hasNext) {
        val e = eit.next
        val newOriginalPortId = portMap(replacements, e.originalNodeId, e.originalPortId)
        if (newOriginalPortId == null) {
          deletedExportInds += exportInd
          curExportsCount -= 1
        } else if (newOriginalPortId != e.originalPortId) {
          val newE = e.withOriginalPort(e.originalNodeId, newOriginalPortId)
          replacedExports.put(e.exportId, newE)
        }
        exportInd += 1
      }
      
      // Create new CellBackup from accumulated changes
      commit
    }
    
    /**
     * Get base name of temporary name of node
     * @param protoId NodeProtoId proto of node
     * @param function Function of node
     * @return base name
     */
    def getBaseName(protoId: NodeProtoId, function: PrimitiveNode.Function): Name = protoId match {
      case cellId: CellId => cellId.cellName.getBasename
      case pnId: PrimitiveNodeId =>
        val pn = oldTechPool.getPrimitiveNode(pnId)
        val newTechBits = pn.getPrimitiveFunctionBits(function)
        oldTechPool.getPrimitiveNode(pnId).getPrimitiveFunction(newTechBits).getBasename
    }
    
    /**
     * Get storage of nodes with temporary names
     * @param protoId NodeProtoId proto of node
     * @param function Function of node
     * @return storage of nodes with temporary names
     */
    def getMaxSuffix(protoId: NodeProtoId, function: PrimitiveNode.Function): MaxNodeSuffix = {
      val basename = getBaseName(protoId, function)
      maxNodeSuffixesOrdered.get(basename) match {
        case Some(ms) => ms
        case None =>
          val ms = new MaxNodeSuffix(this, basename)
          maxNodeSuffixesOrdered = maxNodeSuffixesOrdered.insert(basename, ms)
          ms
      }
    }
    
    /**
     * Replace a node
     * @param nodeInd index in alphanumeric order
     * @param n old node
     * @param r request to replacement
     */
    def replaceNode(nodeInd: Int, n: ImmutableNodeInst, r: NodeReplacement) {
      val newProtoId = r.newProtoId
      def newPrimNode: PrimitiveNode = oldTechPool.getPrimitiveNode(newProtoId.asInstanceOf[PrimitiveNodeId])
      var maxSuffix: MaxNodeSuffix = null
      val newName = if (!n.name.isTempname) n.name else {
        val baseName = getBaseName(newProtoId, r.newFunction)
        if (baseName == n.name.getBasename) n.name else {
          maxSuffix = getMaxSuffix(newProtoId, r.newFunction)
          maxSuffix.getNextName()
        }
      }
      val newSize = (n.protoId,newProtoId) match {
        case (oldPnId: PrimitiveNodeId, newPnId: PrimitiveNodeId) =>
          val oldBaseRect = oldTechPool.getPrimitiveNode(oldPnId.asInstanceOf[PrimitiveNodeId]).getBaseRectangle
          val newBaseRect = newPrimNode.getBaseRectangle
          EPoint.fromGrid(
            n.size.getGridX + oldBaseRect.getGridWidth - newBaseRect.getGridWidth,
            n.size.getGridY + oldBaseRect.getGridHeight - newBaseRect.getGridHeight)
        case (_, newPnId: PrimitiveNodeId) => newPrimNode.getDefSize(ep)
        case _ => EPoint.ORIGIN
      } 
      val newTechBits = newProtoId match {
        case pnId: PrimitiveNodeId if r.newFunction != null => newPrimNode.getPrimitiveFunctionBits(r.newFunction)
        case _ => n.techBits
      }
      var newN = ImmutableNodeInst.newInstance(
        n.nodeId, r.newProtoId,
        newName, n.nameDescriptor,
        n.orient, n.anchor, n.size,
        n.flags, newTechBits, n.protoDescriptor)
      val vit = n.getVariables
      while (vit.hasNext) {
        val v = vit.next
        if (v.getKey == NodeInst.TRACE) {
          newN = newN.withTrace(v.getObject.asInstanceOf[Array[EPoint]], n.anchor)
        } else {
          newN = newN.withVariable(v)
        }
      }
      if (newName != n.name) {
        maxSuffix.add(newN)
        deletedNodeInds += nodeInd
      }
      replacedNodes.put(n.nodeId,newN)
    }
    
    /**
     * Replace arc
     * @param a old arc
     * @param newTailPortId new tail PortProtoId
     * @param newHeadPortId new head PortProtoId
     * @return ImmutableArcInst with replaced PortProtoIds and Locations
     */
    def replaceArc(a: ImmutableArcInst, newTailPortId: PortProtoId, newHeadPortId: PortProtoId): ImmutableArcInst = {
      /**
       * Function to get shape of port
       * @param n node
       * @param portId port proto
       * @return shape of port
       */
      def getShapeOfPort(n: ImmutableNodeInst, portId: PortProtoId): Poly = n.protoId match {
        case cellId: CellId => null
          val subCell = oldSnapshot.getCellRevision(cellId)
          val export = subCell.getExport(portId.asInstanceOf[ExportId])
          val origN = subCell.getNodeById(export.originalNodeId)
          val poly = getShapeOfPort(origN, export.originalPortId)
          poly.transform(new FixpTransform(n.anchor, n.orient))
          poly
        case pnId: PrimitiveNodeId =>
          val polyBuilder = Poly.newLambdaBuilder
          polyBuilder.getShape(n, oldTechPool.getPrimitivePort(portId.asInstanceOf[PrimitivePortId]));
      }
      
      // compute end points after nodes on ends were modified
      var newTailPoint: EPoint = if (newTailPortId == a.tailPortId) a.tailLocation else {
        assert (newTailPortId != null)
        val n = replacedNodes(a.tailNodeId)
        val poly = getShapeOfPort(n, newTailPortId)
        if (poly.isInside(a.tailLocation)) a.tailLocation else poly.getCenter
      }
      var newHeadPoint: EPoint = if (newHeadPortId == a.headPortId) a.headLocation else {
        assert (newHeadPortId != null)
        val n = replacedNodes(a.headNodeId)
        val poly = getShapeOfPort(n, newHeadPortId)
        if (poly.isInside(a.headLocation)) a.headLocation else poly.getCenter
      }
      
      // see if a bend must be made in the wire
      val zigzag = a.isFixedAngle &&
      (newTailPoint.getGridX != newHeadPoint.getGridX || newTailPoint.getGridY != newHeadPoint.getGridY) &&
      GenMath.figureAngle(newTailPoint, newHeadPoint) % 1800 != a.getDefinedAngle % 1800

      // see if a bend can be a straight by some simple manipulations
      if (zigzag && !a.isRigid && (a.getDefinedAngle % 900) == 0 &&
          ((newTailPortId == a.tailPortId) || (newHeadPortId == a.headPortId))) {
        // find the node at the other end
        val otherIsHead = newHeadPortId == a.headPortId
        val otherNodeId = if (otherIsHead) a.headNodeId else a.tailNodeId
        val adjustThisNode = oldCellRevision.getNodeById(otherNodeId)
        if (!replacedNodes.contains(otherNodeId) && !oldCellRevision.hasExportsOnNode(adjustThisNode)) {
          // other end not exported, see if all arcs can be adjusted
          val headEnds = new java.util.BitSet
          val connList = oldCellRevision.getConnections(headEnds, adjustThisNode, null)
          // test other arcs on adjustThisNode
          val adjustable = connList.asScala.forall(oa =>
            (oa eq a) || // skip test for the arc being replaced
            (oa.tailNodeId != adjustThisNode.nodeId || oa.headNodeId != adjustThisNode.nodeId) && // only one end on adjustThisNode
            !replacedArcs.contains(oa.arcId) && // not replaced yet
            !oa.isRigid && // not rigid
            oa.getDefinedAngle % 900 == 0 && // manhattan
            ((a.getDefinedAngle / 900) & 1) != ((oa.getDefinedAngle / 900) & 1) // orthogonal
          )
          if (adjustable) {
            val (newPoint: EPoint, dX: Long, dY: Long) = if (a.getDefinedAngle % 1800 == 0) {
              // horizontal arc: move the other node vertically
              if (otherIsHead) {
                (EPoint.fromGrid(newHeadPoint.getGridX, newTailPoint.getGridY),
                 0L, newTailPoint.getGridY - newHeadPoint.getGridY)
              } else {
                (EPoint.fromGrid(newTailPoint.getGridX, newHeadPoint.getGridY),
                 0L, newHeadPoint.getGridY - newTailPoint.getGridY)
              }
//              dY = newPoint[1 - otherEnd].getY() - newPoint[otherEnd].getY();
//              newPoint[otherEnd] = EPoint.fromLambda(newPoint[otherEnd].getX(), newPoint[1 - otherEnd].getY());
            } else {
              // vertical arc: move the other node horizontally
              if (otherIsHead) {
                (EPoint.fromGrid(newTailPoint.getGridX, newHeadPoint.getGridY),
                 newTailPoint.getGridX - newHeadPoint.getGridX, 0L)
              } else {
                (EPoint.fromGrid(newHeadPoint.getGridX, newTailPoint.getGridY),
                 newHeadPoint.getGridX - newTailPoint.getGridX, 0L)
              }
//              dX = newPoint[1 - otherEnd].getX() - newPoint[otherEnd].getX();
//              newPoint[otherEnd] = EPoint.fromLambda(newPoint[1 - otherEnd].getX(), newPoint[otherEnd].getY());
            }
            if (otherIsHead) {
              newHeadPoint = newPoint
            } else {
              newTailPoint = newPoint
            }

            // special case where the old arc must be deleted first so that the other node can move
            assert(!replacedNodes.contains(adjustThisNode.nodeId))
            val oldAnchor = adjustThisNode.anchor
            val newAnchor = EPoint.fromGrid(oldAnchor.getGridX + dX, oldAnchor.getGridY + dY)
            replacedNodes.put(adjustThisNode.nodeId, adjustThisNode.withAnchor(newAnchor))
            var connInd = 0
            while (connInd < connList.size) {
              val oa = connList.get(connInd)
              assert(!replacedArcs.contains(oa.arcId))
              if (oa != a) {
                val newOA = if (headEnds.get(connInd)) oa.withLocations(oa.tailLocation, newPoint) else oa.withLocations(newPoint, oa.headLocation)
                replacedArcs.put(oa.arcId, newOA)
              }
              connInd += 1
            }
            return a.withConnections(a.tailNodeId, newTailPortId, a.headNodeId, newHeadPortId).withLocations(newTailPoint, newHeadPoint)
          }
        }
      }
       
      if (zigzag) {
        // make that two wires
        val c = EPoint.fromGrid(newTailPoint.getGridX, newHeadPoint.getGridY)
        val ap = oldTechPool.getArcProto(a.protoId)
        val pinNp = ap.findOverridablePinProto(ep)
        val pinPpId = pinNp.getPort(0).getId
        val psx = pinNp.getDefWidth(ep)
        val psy = pinNp.getDefHeight(ep)
                
        // create a pin
        lastNodeId += 1
        val pinId = lastNodeId
        val maxSuffix = getMaxSuffix(pinNp.getId, PrimitiveNode.Function.UNKNOWN)
        val name = maxSuffix.getNextName
        val nameTd = ep.getNodeTextDescriptor
        val orient = Orientation.IDENT
        val anchor = c
        val size = pinNp.getDefSize(ep)
        val flags = 0
        val techBits = 0
        val protoTd = ep.getInstanceTextDescriptor
        val pinN = ImmutableNodeInst.newInstance(pinId, pinNp.getId, name, nameTd,
                                                 orient, anchor, size, flags, techBits, protoTd)
        maxSuffix.add(pinN)
        curNodesCount += 1
                
        // create two arcs
        val newAi1 = a.withConnections(pinId, pinPpId, a.headNodeId, newHeadPortId).withLocations(c, newHeadPoint)
//                val newAi = ArcInst.newInstanceBase(ai.getProto(), ep, ai.getLambdaBaseWidth(), newPortInst[ArcInst.HEADEND], pinPi, newPoint[ArcInst.HEADEND],
//                        new Point2D.Double(cX, cY), null, ArcInst.DEFAULTANGLE);
        val newAi2 = a.withConnections(pinId, pinPpId, a.tailNodeId, newTailPortId).withLocations(c, newTailPoint)
//                ArcInst newAi2 = ArcInst.newInstanceBase(ai.getProto(), ep, ai.getLambdaBaseWidth(), pinPi, newPortInst[ArcInst.TAILEND], new Point2D.Double(cX, cY),
//                        newPoint[ArcInst.TAILEND], null, ArcInst.DEFAULTANGLE);
                        
        // the arc connected to node will save its name, the other arc will obtain temporary name
        lastArcId += 1
        val arcId = lastArcId
        assert(maxArcSuffix < Int.MaxValue)
        maxArcSuffix += 1
        val arcName = ImmutableArcInst.BASENAME.findSuffixed(maxArcSuffix)
        if (newTailPortId eq a.tailPortId) {
          addedArcs += newAi1.withArcId(arcId).withName(arcName)
//                  assert !curArcs.get(arcId);
//                  curArcs.set(arcId);
          curArcsCount += 1
          newAi2
        } else {
          addedArcs += newAi2.withArcId(arcId).withName(arcName)
//                  assert !curArcs.get(arcId);
//                  curArcs.set(arcId);
          curArcsCount += 1
          newAi1
        }
      } else {
        // replace the arc with another arc
        a.withConnections(a.tailNodeId, newTailPortId, a.headNodeId, newHeadPortId).withLocations(newTailPoint, newHeadPoint)
      }
    }
    
    def commit(): CellBackup = {
      // commit nodes
      val newNodes = new Array[ImmutableNodeInst](curNodesCount)
      val nit = oldCellRevision.nodes.iterator
      var oldNodeIndex = 0
      var newNodeIndex = 0
      for (maxSuffix <- maxNodeSuffixesOrdered.values) {
        while (oldNodeIndex < maxSuffix.insertionPoint) {
          val n = nit.next
          if (!deletedNodeInds(oldNodeIndex)) {
            val newN = replacedNodes.getOrElse(n.nodeId, null)
            newNodes(newNodeIndex) =  if (newN != null) newN else n
            newNodeIndex += 1
          }
          oldNodeIndex += 1
        }
        for (n <- maxSuffix.addedNodes) {
//          assert(curNodes(n.nodeId))
          newNodes(newNodeIndex) = n
          newNodeIndex += 1
        }
      }
      while (oldNodeIndex < oldCellRevision.nodes.size) {
        val n = nit.next
        if (!deletedNodeInds(oldNodeIndex)) {
          val newN = replacedNodes.getOrElse(n.nodeId, null)
          newNodes(newNodeIndex) =  if (newN != null) newN else n
          newNodeIndex += 1
        }
        oldNodeIndex += 1
      }
      assert(!nit.hasNext)
      assert(newNodeIndex == newNodes.size)
      
      // commit arcs
      val newArcs = new Array[ImmutableArcInst](curArcsCount)
      val ait = oldCellRevision.arcs.iterator
      var oldArcIndex = 0
      var newArcIndex = 0
      while (ait.hasNext && oldArcIndex < arcInsertionPoint) {
        val a = ait.next
        if (!deletedArcInds(oldArcIndex)) {
          val newA = replacedArcs.getOrElse(a.arcId, null)
          newArcs(newArcIndex) = if (newA != null) newA else a
          newArcIndex += 1
        }
        oldArcIndex += 1
      }
      for (a <- addedArcs) {
//        assert(curArcs(a.arcId))
        newArcs(newArcIndex) = a
        newArcIndex += 1
      }
      assert(oldArcIndex == arcInsertionPoint)
      while(oldArcIndex < oldCellRevision.arcs.size) {
        val a = ait.next
        if (!deletedArcInds(oldArcIndex)) {
          val newA = replacedArcs.getOrElse(a.arcId, null)
          newArcs(newArcIndex) = if (newA != null) newA else a
          newArcIndex += 1
        }
        oldArcIndex += 1
      }
      assert(!ait.hasNext)
      assert(newArcIndex == newArcs.size)
      
      // commit exports
      val newExports = new Array[ImmutableExport](curExportsCount)
      val eit = oldCellRevision.exports.iterator
      var oldExportIndex = 0
      var newExportIndex = 0
      while (oldExportIndex < oldCellRevision.exports.size) {
        val e = eit.next
        if (!deletedExportInds(oldExportIndex)) {
          val newE = replacedExports.getOrElse(e.exportId, null)
          newExports(newExportIndex) = if (newE != null) newE else e
          newExportIndex += 1
        }
        oldExportIndex += 1
      }
      assert(!eit.hasNext)
      assert(newExportIndex == newExports.size)
            
      oldCellBackup.`with`(oldCell, newNodes, newArcs, newExports, oldTechPool)
    }
    
    /**
     * Find new immutable PortInst
     * @param replacements map from nodeId to NodeReplacement
     * @param nodeId nodeId of node
     * @param ppId PortProtoId of old PortInst
     * @reutn PortProtoId of new PortInst
     */
    def portMap(replacements: scala.collection.mutable.Map[Int,NodeReplacement], nodeId: Int, ppId: PortProtoId): PortProtoId = {
      val r = replacements.getOrElse(nodeId, null)
      if (r == null) ppId else {
        val oldN = oldCellRevision.getNodeById(nodeId)
        assert(ppId.parentId eq oldN.protoId)
        val portIndex: Int = ppId match {
          case exportId: ExportId => oldSnapshot.getCellRevision(exportId.getParentId).getExportIndexByExportId(exportId)
          case ppId: PrimitivePortId => oldTechPool.getPrimitivePort(ppId).getPortIndex
        }
        r.assoc(portIndex)
      }
    }
    
    /**
     * Search a split index in original alphanumerical list of nodes
     * where to put temporary nodes with specified basename
     * @param basename speicified basename
     * @return split index
     */
    def searchNodeInsertionPoint(basename: String): Int = {
      assert(basename.endsWith("@0"))
      val nextChar = (basename.charAt(basename.length - 2) + 1).toChar
      val nextName = basename.substring(0, basename.length() - 2) + nextChar
      val index = oldCellRevision.nodes.searchByName(nextName)
      if (index >= 0) index else -(index + 1)
    }

    /**
     * Search a split index in original alphanumerical list of arcs
     * where to put temporary nodes with specified basename
     * @param basename speicified basename
     * @return split index
     */
    def searchArcInsertionPoint(basename: String): Int = {
      assert(basename.endsWith("@0"))
      val nextChar = (basename.charAt(basename.length - 2) + 1).toChar
      val nextName = basename.substring(0, basename.length - 2) + nextChar
      val index = oldCellRevision.arcs.searchByName(nextName)
      if (index >= 0) index else -(index + 1)
    }
  }
}

/**
 * Class stores temp nodes with specified basename
 * @param b CellBuilder where temp nodes are inserted
 * @param basename specified basenaem
 */
class MaxNodeSuffix(b: ReplaceBuilder#CellBuilder, val basename: Name) {
  val insertionPoint: Int = b.searchNodeInsertionPoint(basename.toString)
  var maxSuffix: Int = -1
  val addedNodes = ArrayBuffer[ImmutableNodeInst]()

  if (insertionPoint > 0) {
    val name = b.oldCellRevision.nodes.get(insertionPoint - 1).name;
    if (name.isTempname && name.getBasename == basename) {
      maxSuffix = name.getNumSuffix
    }
  }

  /**
   * get name of next temporary node
   */
  def getNextName(): Name = basename.findSuffixed(maxSuffix + 1)

  /**
   * insert temporary node and adjust name of next node
   * @param n tempory node
   */
  def add(n: ImmutableNodeInst) {
    maxSuffix += 1;
    assert(n.name.getBasename == basename)
    assert(n.name.getNumSuffix == maxSuffix)
    addedNodes += n
  }
}


