#! /bin/sh
# Copyright (C) 2011-2013 Free Software Foundation, Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# As a special exception to the GNU General Public License, if you
# distribute this file as part of a program that contains a
# configuration script generated by Autoconf, you may include it under
# the same distribution terms that you use for the rest of that program.

# This file is maintained in Automake, please report
# bugs to <bug-automake@gnu.org> or send patches to
# <automake-patches@gnu.org>.

scriptversion=2013-12-23.17; # UTC

# Make unconditional expansion of undefined variables an error.  This
# helps a lot in preventing typo-related bugs.
set -u

me=tap-driver.sh

fatal ()
{
  echo "$me: fatal: $*" >&2
  exit 1
}

usage_error ()
{
  echo "$me: $*" >&2
  print_usage >&2
  exit 2
}

print_usage ()
{
  cat <<END
Usage:
  tap-driver.sh --test-name=NAME --log-file=PATH --trs-file=PATH
                [--expect-failure={yes|no}] [--color-tests={yes|no}]
                [--enable-hard-errors={yes|no}] [--ignore-exit]
                [--diagnostic-string=STRING] [--merge|--no-merge]
                [--comments|--no-comments] [--] TEST-COMMAND
The '--test-name', '-log-file' and '--trs-file' options are mandatory.
END
}

# TODO: better error handling in option parsing (in particular, ensure
# TODO: $log_file, $trs_file and $test_name are defined).
test_name= # Used for reporting.
log_file=  # Where to save the result and output of the test script.
trs_file=  # Where to save the metadata of the test run.
expect_failure=0
color_tests=0
merge=0
ignore_exit=0
comments=0
diag_string='#'

while test $# -gt 0; do
  case $1 in
  --help) print_usage; exit $?;;
  --version) echo "$me $scriptversion"; exit $?;;
  --test-name) test_name=$2; shift;;
  --log-file) log_file=$2; shift;;
  --trs-file) trs_file=$2; shift;;
  --color-tests) color_tests=$2; shift;;
  --expect-failure) expect_failure=$2; shift;;
  --enable-hard-errors) shift;; # No-op.
  --merge) merge=1;;
  --no-merge) merge=0;;
  --ignore-exit) ignore_exit=1;;
  --comments) comments=1;;
  --no-comments) comments=0;;
  --diagnostic-string) diag_string=$2; shift;;
  --) shift; break;;
  -*) usage_error "invalid option: '$1'";;
  esac
  shift
done

test $# -gt 0 || usage_error "missing test argument"


case $expect_failure in
  yes) expect_failure=1;;
    *) expect_failure=0;;
esac

init_colors=''
if test $color_tests = yes; then
  init_colors='
    color_map["red"]="[0;31m" # Red.
    color_map["grn"]="[0;32m" # Green.
    color_map["lgn"]="[1;32m" # Light green.
    color_map["blu"]="[1;34m" # Blue.
    color_map["mgn"]="[0;35m" # Magenta.
    color_map["std"]="[m"     # No color.
    color_for_result["ERROR"] = "mgn"
    color_for_result["PASS"]  = "grn"
    color_for_result["XPASS"] = "red"
    color_for_result["FAIL"]  = "red"
    color_for_result["XFAIL"] = "lgn"
    color_for_result["SKIP"]  = "blu"'
fi

# :; is there to work around a bug in bash 3.2 (and earlier) which
# does not always set '$?' properly on redirection failure.
# See the Autoconf manual for more details.
:;{
  (
    # Ignore common signals (in this subshell only!), to avoid potential
    # problems with Korn shells.  Some Korn shells are known to propagate
    # to themselves signals that have killed a child process they were
    # waiting for; this is done at least for SIGINT (and usually only for
    # it, in truth).  Without the `trap' below, such a behaviour could
    # cause a premature exit in the current subshell, e.g., in case the
    # test command it runs gets terminated by a SIGINT.  Thus, the awk
    # script we are piping into would never see the exit status it
    # expects on its last input line (which is displayed below by the
    # last `echo $?' statement), and would thus die reporting an internal
    # error.
    # For more information, see the Autoconf manual and the threads:
    # <http://lists.gnu.org/archive/html/bug-autoconf/2011-09/msg00004.html>
    # <http://mail.opensolaris.org/pipermail/ksh93-integration-discuss/2009-February/004121.html>
    trap : 1 3 2 13 15
    if test $merge -gt 0; then
      exec 2>&1
    else
      exec 2>&3
    fi
    
    # Execute the test in tap-driver.sh [options] test
    "$@"
#    echo $?
  ) | LC_ALL=C ${AM_TAP_AWK-awk} \
        -v me="$me" \
        -v test_script_name="$test_name" \
        -v log_file="$log_file" \
        -v trs_file="$trs_file" \
        -v expect_failure="$expect_failure" \
        -v merge="$merge" \
        -v ignore_exit="$ignore_exit" \
        -v comments="$comments" \
        -v diag_string="$diag_string" \
'
# TODO: the usages of "cat >&3" below could be optimized when using
#       GNU awk, and/on on systems that supports /dev/fd/.

# NOTES:
#   There are two gawk issues which affect the way that arguments are used
#   and the passage of information from a function back to its caller. 
#   1: gawk promotes the specification of local variables to the function
#      parameter list. Any parameter not specifically passed by the caller
#      is considered a local variable. 
#      
#      Parameters and local variables are specified in the function statement
#      by separating the local variables by 3 spaces from the parameters, as 
#      in:
#      
#        function fnc(P1, P2,   V1, V2) or
#        function fnc(   V1, V2)
#      
#      where the parameters are P1 and P2, and the local variables are V1 and 
#      V2.
#      
#      In each use the caller must use all of the parameters. In descriptions
#      of the function, only the parameters will be listed. This means that
#      the caller will always use:
#      
#         fnc(P1, P2) and descriptions will use
#         fnc(P1, P2) without exception.
#      
#   2: Parameters are always IN and never INOUT or OUT. Passage of date from
#      a function to the caller is either by return value or by a global or
#      both. All globals are consumed by a caller before reuse. Each function
#      passing back data will have a side affect of altering a global. This
#      will be indicated in the function preamble with the annotation:
#      
#        sideeffect[OUT] or
#        sideeffect[INOUT]
#

# -------------- function listing --------------
# function decorate_result(result)
# function handle_bail_out(line)
# function handle_diagnostics()
# function handle_not_ok(line)
# function handle_ok(line)
# function handle_test_plan(line)
# function handle_yaml()
# function input_check(st)
# function parse_test_line(line)
# function post_processing() {
# function preamble()
# function write_error(msg)
# function write_fail_test_results()
# function write_fatal_test_results()
# function write_log(str)
# function write_log_summary()
# function write_options()
# function write_pass_test_results()
# function write_skip_test_results()
# function write_result(result, str)
# ----------------------------------------------

## ----------- ##
##  FUNCTIONS  ##
## ----------- ##

#
# @fn decorate_result(result)
# @brief Determine a message color to use for console output.
# <p>Console output can be colorized. This function determines
#    what color to use for the given message type.</p>
# <p>The associatION of color to message type is done on script
#    entry in the shell script, not in awk.</p>
# <p>The result values are:</p>
# <ul>
#    <li>"ERROR" A parsing or I/O error was found</li>
#    <li>"PASS"  TAP ok</li>
#    <li>"XPASS" TAP ok with expect_failure = yes</li>
#    <li>"FAIL"  TAP not ok</li>
#    <li>"XFAIL" TAP not ok with expect_failure = yes</li>
#    <li>"SKIP"  TAP ok # SKIP</li>
# </ul>
# <p>If the output is not to be colorized then the input
#    result is returned.</p>
#
# side effects: none
#
# @param[in] result one of the message types
# @return either a colorized result or a non-colorized result
#
function decorate_result(result)
{
  color_name = color_for_result[result]
  if (color_name)
    return color_map[color_name] "" result "" color_map["std"]
    
  # If we are not using colorized output, or if we do not know how
  # to colorize the given result, we should return it unchanged.
  return result
}

#
# @fn handle_bail_out(line)
# @brief Process Bail Out!.
# <p>Bail Out! is a fatal error. The assumed test case intent is
#    to terminate all processing because of some unexpected and
#    fatal condition. As an Automake extension Diagnostics and
#    comments following a Bail Out! are accepted. The first test
#    line is flagged as an error (one error per test case) and
#    all test lines are processed to determine if there are any
#    TAP syntax errors. The test results is discarded and the
#    output .trs file will not contain .trs metadata to represent
#    these test lines.</p>
# <p>If this is not the first Bail Out! seen post a message.</p>
# <p>There are two phases to the processing:</p>
# <ul>
#    <li>Phase I: This is the first time that a Bail Out! message
#        has been seen. Issue a message and then continue stripping
#        out all Diagnostics and YAML.</li>
#        <ul>
#           <li>If a test line, test plan is found, issue a one
#               time message and return to the main parsing loop.</li>
#           <li>If a Bail Out! is seend issue a message and
#               return to the main parsing loop.</li>
#           <li>Process all Diagnostics.</li>
#           <li>Process all YAML comments.</li>
#           <li>If Bail Out! is seen, issue a diagnostic message
#               and return to the main parsing loop.
#        </ul>
#    <li>Phase II: Issue a diagnostic message and return to the
#        main parsing loop.</li>
# </ul>
# <p>The read loop assumes that a line has been input. This
#    assumption affects the main parsing loop. When a return
#    from this function is made, the next line of input has
#    already been input. Therefore, the st = getline statement
#    in the main parsing loop has to be bypassed. This is 
#    all to say that this jewel of a code fragment is sneaky 
#    pete code.</p>
#
# sideeffect[inout] bailed_out value changed to 1 + testno
# sideeffect[OUT]   testno     +1 if test line command found
#
# @param[inout] line input line - destroyed during processing
# @return st status of I/O on input pipe
# 
function handle_bail_out(line,   bailout_message, st)
{
    if (DEBUG) print "handle_bail_out(\"" line "\")" > log_file

    if (bailed_out) 
    {
       write_error("Multiple Bail Out!")
    }
    else 
    {
       bailed_out = 1 + testno
       # Strip leading and trailing whitespace.
       sub("^[ \t]*Bail out!", "", line);         # leading whitespace
       sub("[ \t]*$", "", line);                  # trailing whitespace
       
       # Format the error message 
       bailout_message = "Bail out! " line
       write_result("ERROR", bailout_message)
       
       st = getline
       while(st > 0) 
       { 
          write_log($0)
          if ($0 ~ /^not ok/) 
          {
              testno++
              handle_not_ok($0)
              write_error("Illegal test line after Bail Out!")
          }
          # TAP test line ok.
          else if ($0 ~ /^ok/) 
          {
              testno++
              handle_ok($0)
              write_error("Illegal test line after Bail Out!")
          }
          # TAP test plan 1..N
          else if ($0 ~ /^1/) 
          {
              handle_test_plan($0)
              write_error("Illegal test plan after Bail Out!")
            }
          else if ($0 ~ /^[ \t]*Bail out!/) 
          {
              write_error("Multiple Bail Out!")
          }
          # Diagnostics.
          else if ($0 ~ /^#/) 
          {
              if (handle_diagnostics() > 0) continue
              else break;
          }
          else if (!($0 ~ /^[ \t]*$/))            # not a blank line
          {    # check for YAML & if not then ERROR
             if (DEBUG) print "check for YAML" > log_file
             if ($0 ~ /[ \t]+---[ \t]*$/)        # YAML start --- line
             {
                 if (handle_yaml() > 0) 
                    continue
                 else 
                 {
                     write_error("Missing  YAML end")
                     break
                 }
             }
             else if ($0 ~ /^[ \t]+\.\.\.[ \t]*$/) # YAML stop ... line
                 write_error("Missing  YAML begin")
                 
             else
                 write_error("Undefined TAP statement")

            }
          st = getline
          if (DEBUG) print "test_suite_status " test_suite_status > log_file
       } 
    }
    
    if (DEBUG) print "handle_bail_out: bailed_out " bailed_out " testno " testno > log_file

    return st
}

#
# @fn handle_diagnostics()
# @brief Process Diagnostics
# <p>Where one Diagnnostic exist there may be many. Continue
#    inputting lines from the input test case pipe until 
#    either a non-Diagnostic if found, an EOF is found,
#    or there is an I/O error.</p>
# <p>The read loop assumes that a line has been input. This
#    assumption affects the main parsing loop. When a return
#    from this function is made, the next line of input has
#    already been input. Therefore, the st = getline statement
#    in the main parsing loop has to be bypassed. This is 
#    all to say that this jewel of a code fragment is sneaky 
#    pete code.</p>
#
# @return st status of input test case pipe
#
function handle_diagnostics(   st) 
{
    if (DEBUG) print "handle_diagnostics:" > log_file

    st = getline
    while(st > 0) {
        if ($0 ~ /^#/)  write_log($0)
        else break
        st = getline
    }
    return st
}

#
# @fn handle_not_ok(line)
# @brief Process TAP not ok test line.
# <p>A test line is parsed and the result analyzed for
#    correctness.  The default result is given in COOKED-FAIL.
#    COOKED_FAIL values are "FAIL" or "XFAIL" if a failure
#    is expected.</p>
#
# sideeffect[OUT]   test_suite_status reflects global status
# sideeffect[IN]    N either testno or the parsed  statement test no
# sideeffect[IN]    comment the parsed statement comment
#
# @param[inout] line the line to be processed. Processing destroys the line.
# @return 0 for successful, 1 for failure.
#
function handle_not_ok(line,   result, parse) 
{
    if (DEBUG) print "handle_not_ok(\"" line "\")  COOKED_FAIL " COOKED_FAIL > log_file
    
    result  = COOKED_FAIL
    sub(/^not ok/, "", line)
    if (line ~ /^[:graph:]/)       # check if not EOL or [ \t]
    {
       result = "ERROR"
    }
    else                           # [:blank:] or EOL
    {
       parse = parse_test_line(line)
       switch(parse)
       {
          case "":
          case "TODO":
              if (DEBUG) print "handle_not_ok N(" N ") " testno > log_file
              if (N != testno)
              {
                  test_suite_status = or(test_suite_status, ERROR)
                  result = "ERROR"
                  write_error("Test line number wrong " N " != " testno)
              }
              else if (result == "XFAIL")
                  test_suite_status = or(test_suite_status, PASS)
              else
                  test_suite_status = or(test_suite_status, FAIL)
              break
              if (parse == "TODO") result = COOKED_PASS
          case "SKIP":
              write_error("SKIP not allowed")
          case "ERROR":
              test_suite_status = ERROR
              result = "ERROR"
              break
       }
    }
    
    write_result(result, comment)
    return (result == "ERROR")? 1: 0
}

#
# @fn handle_ok(line)
# @brief Process TAP ok test line.
# <p>A test line is parsed and the result analyzed for
#    correctness.  The default result is given in COOKED_PASS.
#    COOKED_PASS values are "PASS" and "XPASS" if a PASS is
#    not expected expected.</p>
#
# sideeffect[OUT]   test_suite_status reflects global status
# sideeffect[IN]    N either testno or the parsed  statement test no
# sideeffect[IN]    comment the parsed statement comment
#
# @param[inout] line the line to be processed. Processing destroys the line.
# return 0 for successful, 1 for failure.
#
function handle_ok(line,   result, parse) 
{    
    if (DEBUG) print "handle_ok(\"" line "\")" > log_file

    result  = COOKED_PASS
    sub(/^ok/, "", line)
    if (line ~ /^[:graph:]/)       # check if EOL or not [:blank:]
    {
       result = "ERROR"
    }
    else                           # [:blank:] or EOL
    {
       parse = parse_test_line(line)
       switch(parse)
       {
          case "":
          case "TODO":
              if (DEBUG) print "handle_ok N(" N ") " testno > log_file
              if (N != testno)
              {
                  test_suite_status = ERROR
                  result = "ERROR"
                  write_error("Test line number wrong " N " != " testno)
              }
              else if ((result == "XFAIL") || (result == "PASS"))
                  test_suite_status = or(test_suite_status, PASS)
              else
                  test_suite_status = or(test_suite_status, FAIL)
              break
          case "SKIP":
              result = "SKIP"
              test_suite_status = or(test_suite_status, SKIP)
              break
          case "ERROR":
              test_suite_status = ERROR
              result = "ERROR"
              break
       }
    }

    write_result(result, comment)
    return (result == "ERROR")? 1: 0
}

#
# @fn handle_test_plan(line)
# @brief Parse and process test plans
# <p>Test plans have the following format:</p>
# <tt><pre>
#    1..N [# [comment]]
#    1..0 [# [comment]]
# </pre></tt>
#
# <p>where N is a number >= 0 and</p>
# <ul>
#    <li><b>1..0</b> signifies that the test suite has been skipped.
#        Any test lines of any description are errors and will cause
#        the test suite to be in ERROR.</li>
#    <li><b>1..N</b> signifies that the maximum number of test lines
#        in the test suite are N. If there are k > N test lines the
#        test suite will be in ERROR. If there are k < N test lines
#        in the test suite, the the missing k+1..N lines will be 
#        considered as SKIPed.</li>
# </ul>
# <p>Parsing consists of recognizing the correct format of the input
#    statement and recording N or 0 in the global planned_tests variable. 
#    Comment lines are discarded.</p>
# <p>Successful processing of the test plan depends on knowing
#    whether the current plan is the first or the last or if the
#    plan precedes all test lines that have been processed. These
#    determinations are made with the global variables:</p>
#    </br>
#    plan_seen which indicates the status of the test plan.
#    </br>
#    testno which counts the test lines seen.
#    <p>The values for plan_seen are:</p>
# <ul>
#    <li><b>NO_PLAN</b>[0]    the plan has not been seen.</li>
#    <li><b>EARLY_PLAN</b>[1] the plan is at the top.</li>
#    <li><b>LATE_PLAN</b>[>1] the plan is at the bottom</li>
# </ul>
# <p>At termination, after and EOF or I/O failure, it is possible
#    to determine the status of the test plan. The Algorithm is:</p>
#  <ul>
#     <li>test_plan == 0 A test plan has not been seen. This 
#         violates Automake standards and is an error.</li>
#     <li>test_plan == 1 A test plan is an EARLY_PLAN. This
#         accords to the Automake standard and is accepted.</li>
#     <li>test_plan >  2 A test plan can either be a LATE_PLAN
#         or can be intermingled with test lines. If the test
#         plan is a LATE_PLAN then this violates Automake 
#         standards. If the test plan is intermingled, it is
#         illegal. To distinguish between the two cases:
#         <p>if ((plan_seen - 1) == testno) then it is a LATE_PLAN</p>
#         </br>
#         else it is intermingled and illegal.
#     </li>
#  </ul>
#
# sideeffect[OUT]   test_suite_status reflects global status
# sideeffect[OUT]   plan_seen         1 + testno if plan ok
# sideeffect[OUT]   planned_tests     N in 1..N or MAX_TESTS for ERROR
#
# @param[inout] line the input line - the value is destroyed during parsing
# @return [BOOL] true if the plan is successfully validated.
#
function handle_test_plan(line)
{
    if (DEBUG) printf("handle_test_plan(\"%s\"\n                plan_seen %d testno %d\n", line, plan_seen, testno) > log_file
    
    if (line ~ /^1..[[:digit:]]+/)
    {
       if (plan_seen)
       {
          test_suite_status = ERROR
          write_error("Test plan duplicate.")
       }
       else 
       {
          if (testno)
          {
             test_suite_status = ERROR
             write_error("Test plan follows first test.")
          }
          sub(/^1../, "", line)
          match(line, /^[[:digit:]]+/)
          planned_tests = strtonum(substr(line, RSTART, RLENGTH))
          plan_seen = 1 + testno
          sub(/^[[:digit:]]+/, "", line)
          sub(/^[[:blank:]]*/, "", line)
          if (planned_tests < 0) 
          {
              planned_test = MAX_TESTS
              test_suite_status = ERROR
              write_error("Test plan number wrong")
          }
          else if ((length(line) > 0) && !(line ~ /^#/))
          {
              test_suite_status = ERROR
              write_error("Test plan syntax wrong. Comment starts with \"#\".")
          }
          else if (planned_tests == 0)
          {
              test_suite_status = SKIP
              write_result("SKIP", "")
          }
       }
    }
    else
    {
       test_suite_status = ERROR
       write_error("Test plan syntax wrong. Should be \"1..digit\".")
    }
    if (DEBUG) print "handle_test_plan: planned_tests " planned_tests > log_file
}

#
# @fn handle_yaml()
# @brief Process YAML comments.
# <p>YAML comments are part of th Test Results File interfae.
#    There is no way to return YAML to the Automake Test Harnes
#    all YAML comments are disgarded.</p>
# <p>This function searches for the YAML comment terminator.</p>
# <p>The read loop assumes that a line has been input and the
#    YAML lead (---) has been found. This assumption affects the 
#    main parsing loop. When a return from this function is made, 
#    the next line of input has already been input. Therefore, 
#    the st = getline statement in the main parsing loop has to
#    be bypassed. This is all to say that this jewel of a code 
#    fragment is sneaky pete code.</p>
#
# @return st  status of test case input file
#
function handle_yaml()
{
    if (DEBUG) print "handle_yaml(" line ")" > log_file

    st = getline
    while(st > 0) 
    {
       write_log($0)
       if ($0 ~ /^[ \t]+\.\.\.[ \t]*$/) break;
       st = getline
    }
    return st
}

#
# @fn input_check(st)
# @brief Determine if an I/O error occurred.
# <p>Use the input status value to determine whether the 
#    error was fatal (< 0) or and End Of File (== 0).</p>
#
# @param[in] st I/O status 
#
function input_check(st)
{
    if(st <= 0) 
    {
       write_fatal_test_results()
       if (st < 0) # I/O error.
          write_error("I/O error in input stream")
       else if (st == 0) # End-of-input
          write_error("No input lines")
    }
    return st
}

#
# @fn parse_test_line(line)
# @brief Parse test line.
# <p>The test line is parsed.</p>
# <p>test line has the following formats</p>
# <tt><pre>
#     ok     [N] [comment] [# Directive [d_comment]]
#     not ok [N] [comment] [# Directive [d_comment]]
# </pre></tt>
#
# <p>There must be at least on <ws> separator between elements of
#    the test-line except that not ok must have a single blank
#    separator between not and ok.</p>
# <p>where</p>
# <ul>
#   <li>N[current test line number] Optional test line number.</li>
#   <li>comment[""] Optional test list comment. The comment can 
#        not contain a #.</li>
#   <li>Directive[""] Optional TAP directive. The Directives are not
#       case sensitive. If a Directive is found it is treated as a 
#       SKIP.The legal directives are:</li>
#      <ul>
#         <li>SKIP </li>
#         <li>TODO </li>
#      </ul>
#   <li>d_comment[""] Optional directive comment</li>
# </ul>
#
# <p>The trs :test-result: comment is:</p>
# <tt><pre>
#    [comment] [# Directive [d_comment]]
# </pre></tt>
# <p>And the :test-result: Test Results File output will look like:</p>
# <tt><pre>
#    :test-result: xxxxx [comment] [# Directive [d_comment]]
# </pre></pre>
# <p>Where xxxxx is the result left justified into a 5 character field
#    separated by a blank, from the comments.</p>
#
# sideeffect[OUT]    N either testno or the parsed  statement test no
# sideeffect[OUT]    comment the parsed statement comment
#
# @param[inout] line TAP test-line irremedially modified
# @return processing result {"", "SKIP", "ERROR"}
#
function parse_test_line(line,   d_loc, d_name, d_sep, directive)
{
    N         = testno
    comment   = ""
    directive = ""
    if (line ~ /^[[:blank:]]+[[:digit:]]/)      # check for N
    {
        sub(/^[[:blank:]]+/, "", line)          # remove leading <ws>
        match(line, /^[[:digit:]]+/)            # find N
        N = substr(line, RSTART, RLENGTH) + 0   # N
        sub(/^[[:digit:]]+/, "", line)         
    }
    comment = substr(line, 1)
    d_loc = match(line, /[[:blank:]]#[[:blank:]]/) # search for Directive separator
    if (DEBUG) print "parse_test_line; d_loc " d_loc "  N (" N ") \"" comment "\"" > log_file
    if (d_loc)                                  # d_loc > 0 => found #
    {
        sub(/^.*#[[:blank:]]*/, "", line)       # remove .*#<ws>
        if (DEBUG) print "parse_test_line; \"" line "\"" > log_file
        directive = "SKIP"                      # assume SKIP
        d_name    = toupper(substr(line, 1, 4))
        if (!((d_name == "SKIP") ||             # not SKIP or TODO
              (d_name == "TODO")))
        {
            directive = "ERROR"
            write_error("Directive Unknown (" d_name ")")
        } 
        else if ((length(line) > 4) && !(substr(line, 5, 1) ~ /[[:blank:]]/))
        {
            directive = "ERROR"
            write_error("SKIP or TODO not followed by whitespace \"" line "\"")
        }
        else if (d_name == "TODO")
            directive = "TODO"
        
    }
    else 
    {
       d_loc = index(line, "#")
       if (DEBUG) print "parse_test_line d_loc " d_loc " \"" comment "\""> log_file
       if (d_loc)
       { # is this wrong I ask you
           directive = "ERROR"
           d_sep     = substr(line, d_loc-1, 3)
           write_error("Directive Ill-formed (" d_sep ")")
       }
    }
    
    if (DEBUG) print "parse_test_line directive (" directive ") N (" N ") \"" comment "\"" > log_file
    
    return directive
}

#
# @fn post_processing()
# @brief Processing after the main parsing loop terminates.
#
# sideeffect[IN]    bailed_out        Bail Out! status
# sideeffect[IN]    plan_seen         test plan status
# sideeffect[OUT]   test_suite_status global test result
# sideeffect[IN]    testno            last test line numver processed
#
function post_processing() 
{
   # A "Bail out!" directive should cause us to ignore any following TAP
   # error, as well as a non-zero exit status from the TAP producer.
    if (DEBUG) {
       print "post_processing: testno:            " testno            > log_file
       print "                 bailed_out:        " bailed_out        > log_file
       print "                 plan_seen:         " plan_seen - 1     > log_file
       print "                 planned_tests:     " planned_tests     > log_file
    }
    if (bailed_out) {
      test_suite_status = ERROR
      if (bailed_out == testno)
         write_error("Illegal test line after Bail Out!")
      else if (bailed_out < testno)
         write_error("Illegal test lines after Bail Out!")
    }
    else if (planned_tests != testno)
    {
       if (planned_tests == MAX_TESTS) 
       {
          test_suite_status = ERROR
          write_error("Test plan missing")
       }
       else if (testno < planned_tests)     # if k < N then k+1 .. N are SKIPs
       {
          for(; testno < planned_tests; testno++)
             write_result("SKIP", "")
       }
       else if (planned_tests != MAX_TESTS)
       {
          test_suite_status = ERROR
          write_error("Test plan number wrong")
       }
    }
    if (((plan_seen) <= testno) && (plan_seen > 1)) 
    {
      test_suite_status = ERROR
      write_error("Test plan found between two test lines.")
    }
    
    #
    # Output .trs test suite metadata 
    #
        if (DEBUG) print "                 test_suite_status: " test_suite_status > log_file

    switch(test_suite_status) 
    {
        case 0:      # PASS
            write_pass_test_results()
            break
        case 1:      # FAIL
        case 3:      # FAIL | SKIP
            write_fail_test_results()
            break
        case 2:      # SKIP
            write_skip_test_results()
            break;
        case 4:      # ERROR
        case 5:      # ERROR | FAIL
        case 6:      # ERROR | SKIP
        case 7:      # ERROR | FAIL | SKIP
            write_fatal_test_results()
            break
    }
    
    close(trs_file)
    
#   if (!ignore_exit) {
#        # Fetch exit status from the last line.
#        exit_message = get_test_exit_message(nextline)
#        if (exit_message)
#          testsuite_error(exit_message)
#    }
    write_log_summary()
}

#
# @fn preamble()
# @brief Check and log Diagnostics preceding the test.
# <p>This is an Automake extension of the TAP Standard.</p>
# <p>An unlimited number of Diagnostics and YAML comments 
#    can precede the first statement in the TAP test. The 
#    first statement of the test is any one of a test plan, 
#    test line, or Bail Out!. </p>
#
# @return I/O status of last input
#
function preamble(   equal) {
    
    #
    # output executable name
    #
    len = length(test_script_name)
    equal = "## "
    while(len--) equal = equal "="
    equal = equal " ##"
    printf "\n%s\n## %s ##\n%s\n\n", equal, test_script_name, equal > log_file

    write_options()
    
    if (DEBUG) print "preamble:" > log_file
    
    #
    # .log file header
    print "lineno  statement" > log_file

    return getline
}

#
# @fn write_error(msg)
# @brief Output diagnostic messages.
# @param[in] msg error message
#
function write_error(msg) {
    print test_script_name ": ERROR  " msg > log_file
    if ($comments)  print test_script_name ": " decorate_result("ERROR") "  " msg;
}

#
# @fn write_fail_test_results()
# @brief Output .trs file metadata if the test suite status is FAIL.
#
function write_fail_test_results() {
    print ":global-test-result: FAIL"  > trs_file
    print ":recheck: yes"              > trs_file
    print ":copy-in-global-log: yes"   > trs_file
}

#
# @fn write_fatal_test_results()
# @brief Output .trs file metadat if the test suite status is ERROR.
#
function write_fatal_test_results() {
    print ":global-test-result: ERROR" > trs_file
    print ":recheck: yes"              > trs_file
    print ":copy-in-global-log: no"    > trs_file
}

#
# @fn write_log(str)
# @brief Write to log file and update line counter.
#
# sideeffect[OUT} lineno  current input line number (updated)
# @param[in] str output string
function write_log(str)
{
    lineno++
    printf "%6d  %s\n", lineno, str > log_file
}

#
# @fn write_log_summary()
# @brief Write log summary statistics
# <p>Not implemented. No idea of what to output</p>
#
function write_log_summary() {

}

#
# @fn write_options()
# @brief Pretty-print input options
#
function write_options(   comments_Option, expect_failure_Option, ignore_exit_Option, merge_Option) 
{
    comments_Option       = (comments)? "yes": "no"
    expect_failure_Option = (expect_failure)? "yes": "no"
    ignore_exit_Option    = (ignore_exit)? "yes": "no"
    merge_Option          = (merge)? "yes": "no"
    
    print "Options      Custom TAP driver = " me                     > log_file
    print "             test_script_name  = " test_script_name        > log_file
    print "             log_file          = " log_file                > log_file
    print "             trs_file          = " trs_file                > log_file
    print "             comments          = " comments_Option         > log_file
    print "             expect_failure    = " expect_failure_Option   > log_file
    print "             ignore_exit       = " ignore_exit_Option      > log_file
    print "             merge             = " merge_Option            > log_file
    print "             diag_string       = " diag_string             > log_file
    print "---------------------------------------------\n"           > log_file

}

#
# @fn write_pass_test_results()
# @brief Output .trs file metadat if the test suite status is PASS.
#
function write_pass_test_results() {
    print ":global-test-result: PASS"  > trs_file
    print ":recheck: no"               > trs_file
    print ":copy-in-global-log: no"    > trs_file
}

#
# @fn write_skip_test_results()
# @brief Output .trs file metadat if the test suite status is SKIP.
#
function write_skip_test_results() {
    print ":global-test-result: PASS"  > trs_file
    print ":recheck: yes"              > trs_file
    print ":copy-in-global-log: no"    > trs_file
}

#
# @fn write_result(result, str)
# @brief Output TRS data to the trs file and the monitor
# <p>The results of a test are output to both the trs file
#    and to the users screen. The output format is:</p>
# </br>
# test_file_name: result: str
# <p>Where the result can be any one of:</p>
# <ul>
#     <li><b>PASS</b> Test Passed</li>
#     <li><b>FAIL</b> Test Failed</li>
#     <li><b>XFAIL</b> Test Failure expected</li>
#     <li><b>XPASS</b> Test PASS unexpected</li>
#     <li><b>SKIP</b> Test not run</li>
#     <li><b>ERROR</b> Test not run because of an error</li>
# </ul>
# <p>The result is colorized as required.</p>
#
# @param[in] result test execution result
# @param[in] comment test comment
function write_result(result, comment) {
   if (DEBUG) print "write_result(" result ", " comment ")" > log_file
   print ":test-result: " result " " comment > trs_file
   print decorate_result(result) ":" comment 
   if ($merge) print ":test-result: " result " " comment > log_file;
}

## ----------- ##
##   CONTROL   ##
## ----------- ##

#
# Some prefatory comments.
# This gawk script translates TAP formatted messages into the
# Automake Test Results File (.trs) format. This format has
# metadata tags which support reporting the global result of
# executing all tests and local results of executing each
# individual task. The TAP Diagnostics and YAML structured
# comments are discarded, and the test line comments and
# Directive comments are concatenated.
# 
# Terminology
# o Each Automake Test Case returns TAP Formatted data.
# o A Test Case: A script or compiled program which executes 
#   individual tests. There may be zero or more tests within 
#   a Test Case.
# o Test: A TAP test line. 
# o Test Result: The result contained in a TAP test line 
#   is any one of PASS, FAIL, XPASS, XFAIL, or SKIP or ERROR.
# o Test Suite Status: The cumulative result of executing 
#   all tests within a Test Case. This status
#   is one of PASS, FAIL, SKIP, or ERROR.
# o ERROR:  syntax or semantic error. Automake treats it 
#   as an external event causing a Test Case to fail or 
#   to not be executed.
# o Test Results File(.trs): Interface file to Automake 
#   containing Test Results File metadata.
# o .log file: Contains all TAP data from the Tests in a
#   Test Case plus diagnostic messages and optionally the
#   generated .trs data.
# 
# GENERAL TAP SYNTAX
#    test plan    1..0 [# comment]
#    test plan    1..N [# comment]
#    test line    ok     [comment] [# SKIP [comment]]
#    test line    ok     [comment] [# TODO [comment]]
#    test line    not ok [comment] [# TODO [comment]]
#    Bail Out!    Bail Out! [# comment]
#    Diagnostic   # [comment]
#    YAML comment    --- 
#    YAML comment  o o o
#    YAML comment    ...
#
# Automake Test Results File metadata tags
#    :copy-in-global-log:  copy .log file to test_suite_log file
#    :recheck:             execute Test Case on make check recheck
#    :test-global-result:  Test Suite Status
#    :test-result:         test line result

BEGIN {

## ------- ##
##  SETUP  ##
## ------- ##

DEBUG = 0 # DEBUG = 1 output to log_file

#
# This little bit of ugly code is a legacy item. It
# terminates the gawk program (the firs apostrophe)
# then expands the definition of init_colors in the
# sorrounding shell script, then reenters the gawk
# program (second apostrophe). The result is that the
# init_colors definition is inserted in this place.
#
# The code is perverse!
#
'"$init_colors"'


COOKED_PASS = expect_failure ? "XPASS": "PASS";
COOKED_FAIL = expect_failure ? "XFAIL": "FAIL";

# 
# test plan processing uses the test-plan variable to pass
# information concerning test plan status. The value of 
# the variable is: test_plan =
# 1: 0    no test plan found
# 2: 1    test plan if first non-diagnostic line (EARLY_PLAN)
# 3: > 1  test plan is last test line seen       (LATE_PLAN)
#
# See handle_test_plan for a desciption of the algorithm
#
NO_PLAN = 0
plan_seen = NO_PLAN

#
# The initial maximum test range is set to all test lines to
# be processed without error, until a test plan is seen. At
# the time a test plan is processed appropriate diagnostic
# messages are output (if test lines have been seend) and
# the planned_tests value is set to N in the 1..N test plan.
#
MAX_TESTS     = 20000
planned_tests = MAX_TESTS # Maximum number of test lines w/o a test plan

# Enumeration constants for test suite status.
# Test suite status is calculated as:
#    test_suite_status = or(test_suite_status, status)
#
# With the proviso that:
#    test_suite_status = ERROR
#
# whenever an ERROR is seen
#
# For test_suite_status =
#   0: EXIT_PASS
#   1: EXIT_FAIL
#   2: EXIT_HARD
#   3: EXIT_HARD
# 
#
PASS  = 0        # test line pass or XFAIL found
FAIL  = 1        # test line fail or XPASS found
SKIP  = 2        # test line is ok # SKIP
ERROR = 4        # syntax or processing error
test_suite_status = PASS

# Exit enumerated constants
# 
EXIT_PASS = 0    # no errors
EXIT_FAIL = 1    # FAIL or XPASS found
EXIT_SKIP = 77   # TODO or SKIP  found
EXIT_HARD = 99   # I/O error, input not TAP compliant, other
exit_value= EXIT_PASS

#
# bailed_out is either 0 or 1 + the testno of the last 
# test line seen before the Bail Out! TAP statement. During
# post-processing if the current testno is greater than
# the testno when Bail Out was processed, then a diagnostic
# is issued stating that there was a failure to communicate.
#
bailed_out = 0 # Whether a "Bail out!" directive has been seen.
skip_no    = 0 # Number of test lines that are skipped
testno     = 0 # Number of test lines seen so far.
lineno     = 0 # Number of input lines

## --------- ##
##  PARSING  ##
## --------- ##

#
# Process inital Diagnostics 
# Exit if there is an I/O error or the test
# is not test.
#
if (preamble() <= 0) exit EXIT_HARD

## ----------------- ##
## MAIN PARSING LOOP ##
## ----------------- ##

do 
  {
    write_log($0)       # copy input into log file
  
    # TAP test line not ok.
    if ($0 ~ /^not ok/) {
        testno += 1
        handle_not_ok($0)
    }    
    # TAP test line ok.
    else if ($0 ~ /^ok/) {
        testno += 1
        handle_ok($0)
    }    
    # TAP plan 1..N
    else if ($0 ~ /^1/)
        handle_test_plan($0)
        
    # "Bail out!" magic.
    # Older versions of prove and TAP::Harness (e.g., 3.17) did not
    # recognize a "Bail out!" directive when preceded by leading
    # whitespace, but more modern versions (e.g., 3.23) do.  So we
    # emulate the latter, "more modern" behaviour.
    else if ($0 ~ /^[ \t]*Bail out!/)
        if (handle_bail_out($0) <= 0) break
    # Diagnostics.
    else if ($0 ~ /^#/)  {
        if (handle_diagnostics() > 0) continue
        else break;
    }
    else if (!($0 ~ /^[ \t]*$/))            # not a blank line
    {    # check for YAML & if not then ERROR
       if (DEBUG) print "check for YAML" > log_file
       if ($0 ~ /[ \t]+---[ \t]*$/)        # YAML start --- line
       {
           if (handle_yaml() > 0) 
              continue
           else 
           {
               write_error("Missing  YAML end")
               break
           }
       }
       else if ($0 ~ /^[ \t]+\.\.\.[ \t]*$/) # YAML stop ... line
           write_error("Missing  YAML begin")
           
       else
           write_error("Undefined TAP statement")

      }
    st = getline
    if (DEBUG) print "test_suite_status " test_suite_status > log_file
  } while(st > 0)
  
    if (st < 0) # I/O error.
      fatal("I/O error while reading from input stream")

   if (DEBUG) print "exit input loop. lineno(" lineno ") testno(" testno ")" > log_file
   
## -------- ##
##  FINISH  ##
## -------- ##

    post_processing()

# switch(test_suite_status) {
#     case 0:    # PASS
#         exit EXIT_PASS
#     case 1:    # FAIL
#         exit EXIT_FAIL
#     case 2:    # ERROR
#     case 3:    # ERROR
#         exit EXIT_ERROR
# }
    exit PASS
} # End of "BEGIN" block.
'

# TODO: document that we consume the file descriptor 3 :-(
} 3>"$log_file"

# test $? -eq 0 || fatal "I/O or internal error"

# Local Variables:
# mode: shell-script
# sh-indentation: 4
# eval: (add-hook "write-file-hooks "time-stamp)
# time-stamp-start: "scriptversion="
# time-stamp-format: "%:y-%02m-%02d.%02H"
# time-stamp-time-zone: "UTC"
# time-stamp-end: "; # UTC"
# End:

