/***************************************************************************
                          threeddlg.h  -  description
                             -------------------
    begin                : Thu Apr 25 2002
    copyright            : (C) 2003 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef THREEDDLG_H
#define THREEDDLG_H

#include <kdialogbase.h>

class QLineEdit;
class QCheckBox;
class QButtonGroup;
class QVBoxLayout;
class QGroupBox;
class QGridLayout;
class QPushButton;
class KColorButton;
class KURL;
class KplDoubleEdit;
class KplSpinBox;
class KplDoc;
class ThreeDItem;
class Fun3DItem;
class Array3DItem;

/**
  * 3D item settings dialog class. Provides a base dialog for the settings
  * dialogs for 3D arrays and 3D functions.
  * @author Werner Stille
  */
class ThreeDDlg : public KDialogBase
{
  Q_OBJECT
public:
  /**
   * Constructor.
   * @param parent pointer to parent widget.
   * @param model pointer to KplDoc model object.
   * @param td pointer to 3D item object.
   * @param caption window caption.
   * @param anchor help anchor string.
   * @param page1 header for first dialog page.
   */
  ThreeDDlg(QWidget* parent, KplDoc* model, ThreeDItem* td,
            const QString& caption, const QString anchor,
            const QString& page1);
  /** Destructor. */
  ~ThreeDDlg();

protected:
  /**
   * Initializes range widgets.
   * @param grid pointer to grid layout of dialog page.
   * @param row row index for x range widgets.
   */
  void initRange(QGridLayout* grid, int row = 0);
  /** Copies settings to 3D item object. */
  void getValues();
  /**
   * Performs automatic scaling.
   * @param xmin minimum x value.
   * @param xmax maximum x value.
   * @param ymin minimum y value.
   * @param ymax maximum y value.
   * @param zmin minimum z value.
   * @param zmax maximum z value.
   */
  void autoScale(double xmin, double xmax, double ymin, double ymax,
                 double zmin, double zmax);
  KplDoc* m;
  ThreeDItem* td;
  QFrame* frame1;
  QVBoxLayout* vbox1;
  QGroupBox* g1;
  KplDoubleEdit *ex0, *exl, *ey0, *eyl, *exMin, *exMax, *exStep, *eyMin,
                *eyMax, *ezMin, *ezMax, *efx, *ezStep, *efy, *efz, *exTic,
                *eyTic, *ezTic, *ePhi, *eTheta, *eXref, *eYref, *eBoxx,
                *eBoxy, *eBoxz;
  KplSpinBox *emTicx, *emTicy, *emTicz, *enDigx, *enDigy, *enDigz, *ex, *ey,
             *ez, *eRelSiz;
  KColorButton *colFrame, *colGrid, *colData;
  QButtonGroup *bg1, *bg2;
  QCheckBox *modeFrame, *logX, *logY, *logZ, *autoNorm;
  QLineEdit *eLetX, *eLetY, *eLetZ, *eLetH;
  QPushButton* bAuto;

protected slots:
  /**
   * Updates dialog according to logarithmic axis option.
   * @param state state of log checkbox.
   */
  void slotLogToggled(bool state);
  /**
   * Updates dialog according to autonormalization option.
   * @param state state of autonormalization checkbox.
   */
  void slotNormToggled(bool state);
  /** Displays dialog to edit the x axis label. */
  void slotEditLetX();
  /** Displays dialog to edit the y axis label. */
  void slotEditLetY();
  /** Displays dialog to edit the z axis label. */
  void slotEditLetZ();
  /** Displays dialog to edit the heading. */
  void slotEditLetH();
};

/**
  * 3D function settings dialog class. Provides a dialog for the settings
  * dialog of a 3D function.
  * @author Werner Stille
  */
class Fun3DDlg : public ThreeDDlg
{
  Q_OBJECT
public:
  /**
   * Constructor.
   * @param parent pointer to parent widget.
   * @param model pointer to KplDoc model object.
   * @param fd pointer to 3D function object.
   */
  Fun3DDlg(QWidget* parent, KplDoc* model, Fun3DItem *fd);
  /** Destructor. */
  ~Fun3DDlg();

protected:
  /**
   * Copies settings to temporary function object.
   * @return false for error.
   */
  bool getTemp();
  /**
   * Copies settings to 3D function object.
   * @param accept true for leaving the dialog, false for applying.
   */
  void getValues(bool accept);
  QLineEdit *fileName, *func;
  Fun3DItem *fd, *fdt;

protected slots:
  /** Displays the dialog for selection of a library file to load. */
  void slotFile();
  /** Displays the dialog for selection of a function. */
  void slotFunc();
  /** Displays the dialog for selection of a parameter file to read. */
  void slotGetParameter();
  /** Displays the dialog for parameter editing. */
  void slotEditParameter();
  /** Displays the dialog for selection of a parameter file to save. */
  void slotSaveParameter();
  /** Saves function value table into a data file. */
  void slotExportValues();
  /** Performs automatic scaling. */
  void slotAutoScale();
  /** Applies settings and leaves the dialog. */
  void slotOk();
  /** Applies settings without leaving the dialog. */
  void slotApply();
  /** Saves function value table into a data file. */
  virtual void slotUser1();
};

/**
  * 3D array settings dialog class. Provides a dialog for the settings
  * dialog of a 3D array.
  * @author Werner Stille
  */
class Array3DDlg : public ThreeDDlg
{
  Q_OBJECT
public:
  /**
   * Constructor.
   * @param parent pointer to parent widget.
   * @param model pointer to KplDoc model object.
   * @param ad pointer to 3D array object.
   */
  Array3DDlg(QWidget* parent, KplDoc* model, Array3DItem *ad);
  /** Destructor. */
  ~Array3DDlg();

protected:
  /**
   * Updates ranges of spin boxes.
   * @param ad pointer to 3D array object.
   */
  void setSpinBoxes(Array3DItem* ad);
  /**
   * Loads file.
   * @return true when when succeeded.
   */
  bool load(KURL &url);
  /**
   * Copies settings to 3D array object.
   * @param accept true for leaving the dialog, false for applying.
   */
  void getValues(bool accept);
  /**
   * Reads edited data into 3D array object.
   * @param fromBuffer read editor buffer instead of file when true.
   * @return true when when succeeded.
   */
  bool readData(bool fromBuffer = false);
  QString buf;
  QLabel* fileName;
  QPushButton *bReload, *edit;
  QCheckBox *err, *internal;
  KplSpinBox *ix, *iy, *iz, *ie, *iStart, *n;
  KplDoubleEdit *eFactor;
  Array3DItem *ad, *adt;

protected slots:
  /** Adjusts the Number of points spinbox when start index is changed. */
  void slotStartChanged(int);
  /** Adjusts the Start index spinbox when number of points is changed. */
  void slotNChanged(int);
  /** Displays the dialog for selection of a data file. */
  void slotFile();
  /** Reloads file. */
  void slotReload();
  /** Toggles internal/external data. */
  void slotInternal(bool);
  /** Edits internal data. */
  void slotEdit();
  /** Applies edited array data. */
  void slotEditApply();
  /** Performs automatic scaling. */
  void slotAutoScale();
  /** Applies settings and leaves the dialog. */
  void slotOk();
  /** Applies settings without leaving the dialog. */
  void slotApply();
  /** Saves interpolation value table into a data file. */
  virtual void slotUser1();
};

#endif
