/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#ifndef CATALOG_H
#define CATALOG_H


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qobject.h>
#include <qlist.h>
#include <kurl.h>

#include "settings.h"
#include "catalogitem.h"
#include "msgfmt.h"

class QString;
class QTextStream;
class KBabelView;
class HeaderEditor;
class KFileInfo;
class QTextCodec;
class QFile;


struct PoInfo
{
   int total;
   int fuzzy;
   int untranslated;

   QString lastTranslator;
   QString languageTeam;
   QString revision;
   QString project;
};


/**
* This class represents a catalog, saved in a po-file.
* It has the ability to load from and save to a po-file.
* Also it defines all necessary functions to set and get the entries
*
* @short Class, that represents a translation catalog(po-file)
* @author Matthias Kiefer <matthias.kiefer@gmx.de>
*/
class Catalog : public QObject
{
    Q_OBJECT

public:
   enum IOStatus{OK,OS_ERROR,NO_PERMISSIONS,PARSE_ERROR,NO_FILE};

   static IOStatus info(const QString url,PoInfo& info);
   static PoInfo headerInfo(const CatalogItem);

   /**
   * reads header information from the file and searches for charset
   * information.
   *
   * @return Codec for found charset or 0, if no information has been found
   */
   static QTextCodec* codecForFile(QFile& file);

   Catalog(QObject* parent=0, const char* name=0);
   virtual ~Catalog();

   QString msgid(uint index) const;
   QString msgstr(uint index) const;
   QString comment(uint index) const;
   CatalogItem header() const;

   /**
   * replaces msgstr in catalog item at index index with msgstr
   * returns true, if untranslated status changed
   */
   bool setMsgstr(uint index,QString msgstr);
   /**
   * replaces comment in catalog item at index index with comment
   * returns true, if fuzzy status changed
   */
   bool setComment(uint index,QString comment);
   /**
   * replaces header with given String
   * @return false if the header is not valid
   */
   bool setHeader(CatalogItem header);

   /** removes string ", fuzzy" from comment */
   void removeFuzzyStatus(uint index);

   /**
   * used for session managment when modified file is
   * saved in a temp file
   */
   void setCurrentURL(QString url);
   QString currentURL() const;
   QString packageName() const;
   QString packageDir() const;

   /**
   * opens file url by using KDE's network downlad and calls
   * openFile with a local filename
   */
   IOStatus openURL(QString url);

   /**
   * opens file openURL by using KDE's network downlad and calls
   * openFile with a local filename
   * sets current URL to saveURL
   */
   IOStatus openURL(QString openURL,QString saveURL);

   /** save the file under the old filename */
   IOStatus saveFile();
   /** saves the file under a new filename */
   IOStatus saveFileAs(QString url,bool overwrite=false);

   /**
   * saves the current catalog in a temporary file and
   * returns the name and path of the file.
   */
   QString saveTempFile();

   Msgfmt::Status checkSyntax(QString& output);

   /** closes the file and deletes all entries */
   void clear();

   bool isModified() const{return _modified;}
   /** sets modified flag */
   void setModified(bool flag);

   bool isReadOnly() const{return _readOnly;}
   uint numberOfEntries() const;
   uint numberOfFuzzies() const;
   uint numberOfUntranslated() const;

   bool hasFuzzyInFront(uint index) const;
   bool hasFuzzyAfterwards(uint index) const;
   bool hasUntranslatedInFront(uint index) const;
   bool hasUntranslatedAfterwards(uint index) const;
   bool hasErrorInFront(uint index) const;
   bool hasErrorAfterwards(uint index) const;
   bool isFuzzy(uint index) const;
   bool isUntranslated(uint index) const;
   bool hasError(uint index) const;

   /**
   * returns index of next fuzzy entry behind startIndex
   * -1 if there is no entry behind
   */
   int nextFuzzy(uint startIndex) const;
   /**
   * returns index of previous fuzzy entry before startIndex
   * -1 if there is no entry before
   */
   int prevFuzzy(uint startIndex) const;
   /**
   * returns index of next untranslated entry behind startIndex
   * -1 if there is no entry behind
   */
   int nextUntranslated(uint startIndex) const;
   /**
   * returns index of previous untranslated entry before startIndex
   * -1 if there is no entry before
   */
   int prevUntranslated(uint startIndex) const;
   /**
   * returns index of next error entry behind startIndex
   * -1 if there is no entry behind
   */
   int nextError(uint startIndex) const;
   /**
   * returns index of previous error entry before startIndex
   * -1 if there is no entry before
   */
   int prevError(uint startIndex) const;



   /**
   * returns the header with Information updated, but does _not_ change
   * the header in the catalog
   */
   CatalogItem updatedHeader(CatalogItem oldHeader, bool usePrefs=true) const;

   /** read settings from the apps config file*/
   void readPreferences();
   /** save settings to the apps config file*/
   void savePreferences();

   SaveSettings saveSettings() const {return _saveSettings;}
   IdentitySettings identitySettings() const {return _identitySettings;}

   /**
   * reads the header from QTextstream and puts it in header
   * I made it static to be able to use this function for
   * the search in po-files. This way, I can easily find the first
   * catalog entry in the textstream
   */
   static IOStatus readHeader(QTextStream& stream,CatalogItem& header);


   void registerView(KBabelView* view);
   void removeView(KBabelView* view);
   void updateViews(int index,KBabelView* view2exclude=0);
   bool hasView() const;
   bool isLastView() const;

   HeaderEditor* headerEditor();

public slots:
   void setSettings(SaveSettings settings);
   void setSettings(IdentitySettings settings);


private:
   IOStatus writeFile(QString localfile, bool overwrite=false);
   /**
   * closes the current catalog,opens file from a local file and create
   * new CatalogItems
   */
   IOStatus openFile(QString filename);

   /**
   * generates  lists that contain indexes of all fuzzy and untranslated entries
   */
   void generateIndexLists();

   /**
   * returns value in list that is lower than parameter index
   * or -1 if there is none
   */
   int findPrevInList(const QValueList<uint>& list,uint index) const;
   /**
   * returns value in list that is bigger than parameter index
   * or -1 if there is none
   */
   int findNextInList(const QValueList<uint>& list,uint index) const;

   /** returns the current date and time in the format of the users choice */
   QString dateTime() const;

private:
   /** url of the po-file, that belongs to this catalog */
   KURL _url;

   /** holds the entries in the catalog */
   QValueList<CatalogItem> _entries;
   /** The header of the po-file. */
   CatalogItem _header;

   bool _modified;
   bool _readOnly;

   QValueList<uint> _fuzzyIndex;
   QValueList<uint> _untransIndex;
   QValueList<uint> _errorIndex;

   QList<KBabelView> _views;
   HeaderEditor* _headerEditor;

   SaveSettings _saveSettings;
   IdentitySettings _identitySettings;

signals:
   void signalError(QString);
   void signalResetProgressBar(QString,int);
   void signalProgress(int);
   void signalClearProgressBar();

   void signalModified(bool);
   /** emitted when a file was opened or saved under another name */
   void signalFileOpened(bool readOnly);

   void signalNumberOfFuzziesChanged(uint number);
   void signalNumberOfUntranslatedChanged(uint number);
   void signalTotalNumberChanged(uint number);

   void signalSettingsChanged(SaveSettings);
   void signalSettingsChanged(IdentitySettings);

};

#endif //CATALOG_H
