/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#include "prefwidgets.h"
#include "resources.h"
#include "cmdedit.h"

#include <klocale.h>
#include <kdialog.h>
#include <kfiledialog.h>
#include <klined.h> // for KLineEdit
#include <kurlcompletion.h>

#include <qlayout.h>
#include <qlabel.h>
#include <qvbox.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qbuttongroup.h>
#include <qpushbutton.h>
#include <qcombobox.h>
#include <qradiobutton.h>
#include <qtextcodec.h>
#include <qwhatsthis.h>



SavePreferences::SavePreferences(QWidget *parent)
    : QVBox(parent)
{
    setSpacing(KDialog::spacingHint());
    setMargin(KDialog::marginHint());

    QGroupBox* box=new QGroupBox(1,Qt::Horizontal,this);

    box->setMargin(KDialog::marginHint());
    _updateButton = new QCheckBox(i18n("&Update header"),box);

    QGroupBox* gridBox = new QGroupBox(2,Qt::Horizontal,i18n("Fields to update:"),this);
    gridBox->setMargin(KDialog::marginHint());

    _revisionButton = new QCheckBox(i18n("&Revision-Date"),gridBox);
    _lastButton = new QCheckBox(i18n("Last-&Translator"),gridBox);
    _languageButton = new QCheckBox(i18n("&Language"),gridBox);
    _charsetButton = new QCheckBox(i18n("&Charset"),gridBox);
    _encodingButton = new QCheckBox(i18n("&Encoding"),gridBox);

    QGroupBox* dateBox = new QButtonGroup(2,Qt::Horizontal,i18n("Format of Revision-Date:"),this);
    box->setMargin(KDialog::marginHint());

    _defaultDateButton = new QRadioButton( i18n("&Default date format"),dateBox );
    _localDateButton = new QRadioButton( i18n("L&ocal date format"),dateBox );
    _customDateButton = new QRadioButton( i18n("Cu&stom date format:"),dateBox );
    _dateFormatEdit = new QLineEdit(dateBox);
    _dateFormatEdit->setEnabled(false);

    connect( _customDateButton, SIGNAL(toggled(bool)), this, SLOT( customDateActivated(bool) ) );


    QGroupBox* encodingBox = new QGroupBox(2,Qt::Horizontal,this);
    box->setMargin(KDialog::marginHint());

    QLabel* tempLabel=new QLabel(i18n("&Encoding:"),encodingBox);
    _encodingBox = new QComboBox(encodingBox);

    QString defaultName=QTextCodec::codecForLocale()->name();
    defaultName+=" "+i18n("(default)");
    QString utf8Name;
    QString utf16Name;

    _encodingBox->insertItem(defaultName,(int)Locale);

    QTextCodec* codec;
    codec = QTextCodec::codecForName("UTF-8");
    if(codec)
    {
       utf8Name=codec->name();
       _encodingBox->insertItem(utf8Name,(int)UTF8);
    }
    codec = QTextCodec::codecForName("UTF-16");
    if(codec)
    {
       utf16Name=codec->name();
       _encodingBox->insertItem(utf16Name,(int)UTF16);
    }
    tempLabel->setBuddy(_encodingBox);

    box=new QGroupBox(1,Qt::Horizontal,this);
    box->setMargin(KDialog::marginHint());
    _autoCheckButton = new QCheckBox(i18n("&Check syntax of file when saving"),box);


    defaults();


    QWhatsThis::add(_updateButton,
    i18n("<qt><p><b>Update Header</b></p>\n\
<p>Check this button, to update the header information of the file \
everytime when it is saved.</p>\n\
<p>The header normally keeps information about the date and time the file was last\n\
updated,the last translator etc.</p>\n\
<p>You can choose which information you want to update from the checkboxes below.\n\
Fields, that do not exist are added to the header.\n\
If you want to add additional fields to the header you can edit the header manually by choosing\n\
<b>Edit->Edit Header</b> in the editor window.</p></qt>"));

    QWhatsThis::add(gridBox,i18n("<qt><p><b>Fields to update</b></p>\n\
<p>Choose which fields in the header you want to have updated when saving.\n\
If a field does not exist, it is appended to the header.</p>\n\
<p>If you want to add other information to the header, you have to edit the header manually\n\
by choosing <b>Edit->Edit Header</b> in the editor window.</p>\n\
<p>Deactivate <b>Update Header</b> above if you don't want to have the header\n\
updated when saving.</p></qt>"));

    QWhatsThis::add(encodingBox,i18n("<qt><p><b>Encoding</b></p>\n\
<p>Choose how to encode characters when saving to a file.</p>\n\
<ul><li><b>%1</b>: this is the encoding most suitable for your charset.</li>\n\
<li><b>%2</b>: uses unicode(utf8) encoding.</li>\n\
<li><b>%3</b>: uses unicode(utf16) encoding.\n\
This is not recommended at the moment, because gettext might have some trouble reading this.</li></ul></qt>").arg(defaultName).arg(utf8Name).arg(utf16Name) );

    QWhatsThis::add(_autoCheckButton,i18n("<qt><p><b>Check syntax of file when saving</b></p>\n\
<p>Check this to automatically check syntax of file with \"msgfmt --statistics\"\n\
when saving a file. You will only get a message, if an error occured.</p></qt>"));


    QWhatsThis::add(dateBox, i18n("<qt><p><b>Format of Revision-Date</b></p>\
<p>Choose in which format the date and time of the header field\n\
<i>PO-Revision-Date</i> is saved: <ul>\n\
<li><b>Default</b> is the format normally used in PO-files.</li>\n\
<li><b>Local</b> is the format specific to your country.\n\
It can be configured in KDE's Control Center.</li>\n\
<li><b>Custom</b> let you define your own format, where you can use the following strings:\n\
<pre>\n\
Year\n\
%y     year                     - 00 to 99\n\
%Y     year                     - 0001 to 9999\n\
Month\n\
%m     month of year            - 01 to 12\n\
%f     month of year            - 1 to 12\n\
%b,%h  month abbreviation       - Jan to Dec\n\
Day\n\
%j     day of the year          - 001 to 366\n\
%d     day of month             - 01 to 31\n\
%e     day of month             - 1 to 31\n\
%a     weekday abbreviation     - Sun to Sat\n\
Hour\n\
%H     hour                     - 00 to 23\n\
%k     hour                     - 0 to 23\n\
%i     hour                     - 1 to 12\n\
%I     hour                     - 01 to 12\n\
%p     AM or PM\n\
Minute, Second, Timezone\n\
%M     minute                   - 00 to 59\n\
%S     second                   - 00 to 59\n\
%Z     timezone                 - as given in\n\
                                  identity settings\n\
</pre></li></ul></p></qt>") );



    setMinimumSize(sizeHint());
}

QSize SavePreferences::sizeHint () const
{
  //
  // The size is computed by adding the sizeHint().height() of all
  // widget children and taking the width of the widest child and adding
  // layout()->margin() and layout()->spacing()
  //

  kDebug("save");
  QSize size;

  int numChild = 0;
  QObjectList *l = (QObjectList*)children();

  for( uint i=0; i < l->count(); i++ )
  {
    QObject *o = l->at(i);
    if( o->isWidgetType() )
    {
      numChild += 1;
      QWidget *w=((QWidget*)o);

      QSize s = w->sizeHint();
      if( s.isEmpty() == true )
      {
	 s = QSize( 50, 100 ); // Default size
      }
      size.setHeight( size.height() + s.height() );
      if( s.width() > size.width() ) { size.setWidth( s.width() ); }
    }
  }

  if( numChild > 0 )
  {
    size.setHeight( size.height() + layout()->spacing()*(numChild-1) );
    size += QSize( layout()->margin()*2, layout()->margin()*2 + 1 );
  }
  else
  {
    size = QSize( 1, 1 );
  }

  return( size );
}

QSize SavePreferences::minimumSizeHint () const
{
   return sizeHint();
}


void SavePreferences::setSettings(const SaveSettings& settings)
{
   _updateButton->setChecked(settings.autoUpdate);

   _lastButton->setChecked(settings.updateLastTranslator);
   _revisionButton->setChecked(settings.updateRevisionDate);
   _languageButton->setChecked(settings.updateLanguageTeam);
   _charsetButton->setChecked(settings.updateCharset);
   _encodingButton->setChecked(settings.updateEncoding);

   _encodingBox->setCurrentItem((int)settings.encoding);

   _autoCheckButton->setChecked(settings.autoSyntaxCheck);

   _dateFormatEdit->setText(settings.customDateFormat);

   switch(settings.dateFormat)
   {
      case Default:
         _defaultDateButton->setChecked(true);
         break;
      case Local:
         _localDateButton->setChecked(true);
         break;
      case Custom:
         _customDateButton->setChecked(true);
         break;
   }

}

SaveSettings SavePreferences::settings() const
{
   SaveSettings settings;

   settings.updateLastTranslator=_lastButton->isChecked();
   settings.updateRevisionDate=_revisionButton->isChecked();
   settings.updateLanguageTeam=_languageButton->isChecked();
   settings.updateCharset=_charsetButton->isChecked();
   settings.updateEncoding=_encodingButton->isChecked();
   settings.autoUpdate=_updateButton->isChecked();

   settings.encoding=(Encoding)_encodingBox->currentItem();

   settings.autoSyntaxCheck = _autoCheckButton->isChecked();


   settings.customDateFormat=_dateFormatEdit->text();

   if(_defaultDateButton->isChecked())
   {
      settings.dateFormat=Default;
   }
   else if(_localDateButton->isChecked())
   {
      settings.dateFormat=Local;
   }
   else
   {
      settings.dateFormat=Custom;
   }


   return settings;
}

void SavePreferences::defaults()
{
   _updateButton->setChecked(Defaults::Save::autoUpdate);

   _lastButton->setChecked(Defaults::Save::updateLastTranslator);
   _revisionButton->setChecked(Defaults::Save::updateRevisionDate);
   _languageButton->setChecked(Defaults::Save::updateLanguageTeam);
   _charsetButton->setChecked(Defaults::Save::updateCharset);
   _encodingButton->setChecked(Defaults::Save::updateEncoding);

   _encodingBox->setCurrentItem((int)Defaults::Save::encoding);

   _autoCheckButton->setChecked(Defaults::Save::autoSyntaxCheck);

   _dateFormatEdit->setText(Defaults::Save::customDateFormat);

   switch(Defaults::Save::dateFormat)
   {
      case Default:
         _defaultDateButton->setChecked(true);
         break;
      case Local:
         _localDateButton->setChecked(true);
         break;
      case Custom:
         _customDateButton->setChecked(true);
         break;
   }

}


void SavePreferences::customDateActivated(bool on)
{
   _dateFormatEdit->setEnabled(on);
   _dateFormatEdit->setFocus();
}






IdentityPreferences::IdentityPreferences(QWidget* parent)
         : QVBox(parent)
{

    setSpacing(KDialog::spacingHint());
    setMargin(KDialog::marginHint());

    QGroupBox* group = new QGroupBox(1,Qt::Horizontal,this);
    group->setMargin(KDialog::marginHint());

    QLabel* tempLabel=new QLabel(i18n("&Name:"),group);
    _nameEdit = new QLineEdit(group);
    _nameEdit->setMinimumSize(250,_nameEdit->sizeHint().height());
    tempLabel->setBuddy(_nameEdit);


    tempLabel=new QLabel(i18n("E-&Mail:"),group);
    _mailEdit = new QLineEdit(group);
    _mailEdit->setMinimumSize(250,_mailEdit->sizeHint().height());
    tempLabel->setBuddy(_mailEdit);


    tempLabel=new QLabel(i18n("&Full language name:"),group);
    _langEdit = new QLineEdit(group);
    _langEdit->setMinimumSize(250,_langEdit->sizeHint().height());
    tempLabel->setBuddy(_langEdit);


    tempLabel=new QLabel(i18n("&Language mailinglist:"),group);
    _listEdit = new QLineEdit(group);
    _listEdit->setMinimumSize(250,_listEdit->sizeHint().height());
    tempLabel->setBuddy(_listEdit);



    QString whatsThisMsg=i18n("<qt><p><b>Identity</b></p>\n\
<p>Fill in information about you and your translation team.\n\
This information is used when updating the header of a file.</p>\n\
<p>You can find the options if and what fields in the header should be updated\n\
on page <b>Save</b> in this dialog.</p></qt>");

    QWhatsThis::add(group,whatsThisMsg);

    group = new QGroupBox(2,Qt::Horizontal,this);
    group->setMargin(KDialog::marginHint());
    tempLabel=new QLabel(i18n("&Timezone:"),group);
    _timezoneEdit = new QLineEdit(group);
    _timezoneEdit->setMinimumSize(100,_timezoneEdit->sizeHint().height());
    tempLabel->setBuddy(_timezoneEdit);


    QWhatsThis::add(group,whatsThisMsg);

    defaults();

    setMinimumSize(sizeHint());
}

void IdentityPreferences::setSettings(const IdentitySettings& settings)
{
   _nameEdit->setText(settings.authorName);
   _mailEdit->setText(settings.authorEmail);
   _langEdit->setText(settings.languageName);
   _listEdit->setText(settings.mailingList);

   _timezoneEdit->setText(settings.timeZone);
}

IdentitySettings IdentityPreferences::settings() const
{
    IdentitySettings settings;

    settings.authorName=_nameEdit->text();
    settings.authorEmail=_mailEdit->text();
    settings.languageName=_langEdit->text();
    settings.mailingList=_listEdit->text();

    settings.timeZone=_timezoneEdit->text();

    return settings;
}


void IdentityPreferences::defaults()
{
   _nameEdit->setText(Defaults::Identity::authorName());
   _mailEdit->setText(Defaults::Identity::authorEmail());
   _langEdit->setText(Defaults::Identity::languageName());
   _listEdit->setText(Defaults::Identity::mailingList());
   _timezoneEdit->setText(Defaults::Identity::timezone());
}


QSize IdentityPreferences::sizeHint () const
{
  //
  // The size is computed by adding the sizeHint().height() of all
  // widget children and taking the width of the widest child and adding
  // layout()->margin() and layout()->spacing()
  //

  kDebug("identity");


  QSize size;

  int numChild = 0;
  QObjectList *l = (QObjectList*)children();

  for( uint i=0; i < l->count(); i++ )
  {
    QObject *o = l->at(i);
    if( o->isWidgetType() )
    {
      numChild += 1;
      QWidget *w=((QWidget*)o);

      QSize s = w->sizeHint();
      if( s.isEmpty() == true )
      {
	 s = QSize( 50, 100 ); // Default size
      }
      size.setHeight( size.height() + s.height() );
      if( s.width() > size.width() ) { size.setWidth( s.width() ); }
    }
  }

  if( numChild > 0 )
  {
    size.setHeight( size.height() + layout()->spacing()*(numChild-1) );
    size += QSize( layout()->margin()*2, layout()->margin()*2 + 1 );
  }
  else
  {
    size = QSize( 1, 1 );
  }

  return( size );
}

QSize IdentityPreferences::minimumSizeHint () const
{
   return sizeHint();
}







EditorPreferences::EditorPreferences(QWidget* parent)
         : QVBox(parent)
{
    setSpacing(KDialog::spacingHint());
    setMargin(KDialog::marginHint());

    QGroupBox* box=new QGroupBox(1,Qt::Horizontal,this);
    box->setMargin(KDialog::marginHint());
    _fuzzyButton = new QCheckBox(i18n("A&utomatically unset fuzzy status"),box);


    QWhatsThis::add(box,i18n("<qt><p><b>Automatically unset fuzzy status</b></p>\n\
<p>If this is activated and you are editing a fuzzy entry, the fuzzy status is automatically\n\
unset (this means the string <i>, fuzzy</i>\n\
is removed from the entry's comment).</p></qt>"));

    defaults();

    setMinimumSize(sizeHint());
}



void EditorPreferences::setSettings(const EditorSettings& settings)
{
    _fuzzyButton->setChecked(settings.autoUnsetFuzzy);
}


EditorSettings EditorPreferences::settings() const
{
    EditorSettings settings;

    settings.autoUnsetFuzzy=_fuzzyButton->isChecked();

    return settings;
}

void EditorPreferences::defaults()
{
    _fuzzyButton->setChecked(Defaults::Editor::autoUnsetFuzzy);
}



QSize EditorPreferences::sizeHint () const
{
  //
  // The size is computed by adding the sizeHint().height() of all
  // widget children and taking the width of the widest child and adding
  // layout()->margin() and layout()->spacing()
  //

  kDebug("editor");

  QSize size;

  int numChild = 0;
  QObjectList *l = (QObjectList*)children();

  for( uint i=0; i < l->count(); i++ )
  {
    QObject *o = l->at(i);
    if( o->isWidgetType() )
    {
      numChild += 1;
      QWidget *w=((QWidget*)o);

      QSize s = w->sizeHint();
      if( s.isEmpty() == true )
      {
	 s = QSize( 50, 100 ); // Default size
      }
      size.setHeight( size.height() + s.height() );
      if( s.width() > size.width() ) { size.setWidth( s.width() ); }
    }
  }

  if( numChild > 0 )
  {
    size.setHeight( size.height() + layout()->spacing()*(numChild-1) );
    size += QSize( layout()->margin()*2, layout()->margin()*2 + 1 );
  }
  else
  {
    size = QSize( 1, 1 );
  }

  return( size );
}

QSize EditorPreferences::minimumSizeHint () const
{
   return sizeHint();
}






SearchPreferences::SearchPreferences(QWidget* parent)
         : KTabCtl(parent)
{
    // first page
    QWidget* page= new QWidget(this);
    QVBoxLayout* layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    QGroupBox* box=new QGroupBox(1,Qt::Horizontal,page);
    layout->addWidget(box);
    box->setMargin(KDialog::marginHint());
    _searchButton = new QCheckBox(i18n("Au&tomatically start search"),box);

    QWhatsThis::add(_searchButton,i18n("<qt><p><b>Automatically start search</b></p>\n\
<p>If this is activated, the search is automatically started whenever\n\
you switch to another entry in the editor. You can choose where to search\n\
in section <b>Default Search</b>.</p>\n\
<p>You can also start searching manually by choosing an entry in the popup menu\n\
that appears either when clicking at <b>Tools->Search</b> or keeping the\n\
search button in the toolbar pressed for a while.</p></qt>"));


    QButtonGroup* bBox=new QButtonGroup(1,Qt::Horizontal,i18n("Default Search:"),page);
    layout->addWidget(bBox);
    _inCatsButton = new QRadioButton(i18n("&Search in MO-files"),bBox);

    _inCatsButton->setEnabled(false);

    _inCompendiumButton = new QRadioButton(i18n("S&earch in a compendium file"),bBox);
    _inAuxiliaryButton = new QRadioButton(i18n("Sea&rch in a auxiliary file"),bBox);
    // test if at least one button is one
    if(!_inCatsButton->isChecked() && !_inCompendiumButton->isChecked() && ! _inAuxiliaryButton->isChecked())
    {
       bBox->setButton(0);
    }


    QWhatsThis::add(bBox,i18n("<qt><p><b>Default Search</b></p>\n\
<p>Choose here where to search as default. This setting is used when searching is started\n\
automatically or when pressing the search button in the toolbar.</p>\n\
<p><ul><li><b>MO-file</b>: MO-files are compiled message files, that are prepared\n\
for fast finding a translation. Choosing this you can easily search for similiar translations\n\
in message catalogs of a package without having to download the source of this package.\n\
<br/><b>This is not implemented yet!</b></li>\n\
<li><b>Compendium</b>: The compendium is a normal PO-File, which should contain a list\n\
of standard translation from your translation team. If you don't have one, you can also use\n\
a file that contains all used translations of your team (e.g. the $lang.messages file in the\n\
KDE Project).</li>\n\
<li><b>Auxiliary</b>: The auxiliary should help you finding the context of a translation\n\
by looking up the same message in a message catalog of the same package but translated to another\n\
language. This way you can have a look how this message is translated in another language.</li></ul></p>\n\
<p>You can define where to find the needed files on page <b>Files</b>.</p></qt>"));


    layout->addStretch(1);

    page->setMinimumSize(page->sizeHint());

    addTab(page,i18n("&General"));



    // second page
    page=new QWidget(this);
    layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());


    box=new QGroupBox(2,Qt::Horizontal,page);
    layout->addWidget(box);
    box->setMargin(KDialog::marginHint());
    _caseButton=new QCheckBox(i18n("Case Sensiti&ve"),box);
    _ignoreAmpButton=new QCheckBox(i18n("&Ignore Ampersand (&&)"),box);
    _ignoreNewlineButton=new QCheckBox(i18n("I&gnore Newlines (\\n)"),box);
    _ignoreDotsButton=new QCheckBox(i18n("Ig&nore Dots (.)"),box);
    _ignoreOtherButton=new QCheckBox(i18n("Ign&ore Other:"),box);
    _otherEdit = new QLineEdit(box);
    _otherEdit->setMinimumSize(50,_otherEdit->sizeHint().height());


    QWhatsThis::add(box,i18n("<qt><p><b>Parameters</b></p>\n\
<p>You can choose here which characters has to be ignored when comparing two strings or if you\n\
want to search case sensitive.</p>\n\
<p>If you want ignore additional characters, activate option <b>Ignore Other</b>\n\
and type in the field behind this button the characters, that has to be ignored.\n\
E.g.: If you type in <i>?!_</i> all of this three characters will be ignored.</p>\n\
<p>Since searching in a auxiliary is intended to find exactly the same message these\n\
options are not used, when searching in the auxiliary.</p></qt>"));


    box = new QGroupBox(1,Qt::Horizontal,page);
    layout->addWidget(box);
    box->setMargin(KDialog::marginHint());
    QLabel* label= new QLabel(i18n("A found text matches if it..."),box);
    _exactButton = new QCheckBox(i18n("is equal to original message"),box);
    _containButton = new QCheckBox(i18n("contains original message"),box);
    _isContainedButton = new QCheckBox(i18n("is contained in original message"),box);


    QWhatsThis::add(box,i18n("<qt><p><b>Comparison Options</b></p>\n\
<p>Choose here what messages you want to have treated as a matching message:</p>\n\
<ul><li><b>is equal to original message</b>: In this case messages will be displayed, that\n\
are equal to the original message.</li>\n\
<li><b>contains original message</b>: displays messages, that contain the original message as a\n\
whole. <br/>E.g.: If the original message is \"setup\", you will get messages displayed like\n\
\"Click to setup\".</li>\n\
<li><b>is contained in original message</b>: displays messages, that are contained as a whole in the\n\
original message. <br/>E.g.: If the original message is \"Click to setup\", you will get messages\n\
displayed like \"setup\".</li></ul>\n\
<p>Comparison will always ignore the characters, you can configure above.</p>\n\
<p>Since searching in a auxiliary is intended to find exactly the same message these\n\
options are not used, when searching in the auxiliary.</p></qt>"));

    // test if at least one button is one
    if(!_exactButton->isChecked() && !_containButton->isChecked() && ! _isContainedButton->isChecked())
    {
       _exactButton->setChecked(true);
    }

    connect(_exactButton,SIGNAL(toggled(bool)),
            this,SLOT(exactButtonToggled(bool)));
    connect(_containButton,SIGNAL(toggled(bool)),
            this,SLOT(containButtonToggled(bool)));
    connect(_isContainedButton,SIGNAL(toggled(bool)),
            this,SLOT(isContainedButtonToggled(bool)));

    layout->addStretch(1);

    page->setMinimumSize(layout->sizeHint());
    addTab(page,i18n("&Parameters"));


    // third page
    page=new QWidget(this);
    layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());


    box=new QGroupBox(1,Qt::Horizontal,page);
    layout->addWidget(box);
    box->setMargin(KDialog::marginHint());

    QHBox* hbox= new QHBox(box);
    label=new QLabel(i18n("&Language:"),hbox);
    _langEdit = new QLineEdit(hbox);
    _langEdit->setMinimumSize(30,_langEdit->sizeHint().height());
    label->setBuddy(_langEdit);


    QWhatsThis::add(box,i18n("<qt><p><b>Language</b></p>\n\
<p>You can type in here the language code for the language you are translating in.\n\
This is at the moment just useful, if you translate for more than one language and don't\n\
want to always have to change the pathes of the files in section <b>Locations</b>.\n\
In this case you can use the string <b>@LANG@</b> in the path to a file, which will then\n\
replaced by your language code.</p></qt>"));

    box=new QGroupBox(1,Qt::Horizontal,i18n("Locations:"),page);
    layout->addWidget(box);
    box->setMargin(KDialog::marginHint());


    QString explanationOfVariables=i18n("<p>If you insert the variables <b>@PACKAGE@</b> or\n\
<b>@LANG@</b> in the path, these will be replaced with either the language code or the\n\
name of the package you have currently opened (this means the name of the file without extension).\n\
<br/>E.g.: giving the path <i>/home/translations/@PACKAGE@/@LANG@.messages</i> and editing the file\n\
<i>/usr/local/src/po/kbabel.po</i>, will result in <i>/home/translations/kbabel/de.messages</i>\n\
(assumed you have entered \"de\" in the language field above).</p>");

     QString explanationOfRelativePath=i18n("<p>You can also use a relative path, which will\n\
then be prepended by the directory you have opened the file in.</p>");

    QWhatsThis::add(box,"<qt>"+i18n("<p><b>Locations</b></p>\n\
<p>Configure here what files should be used for searching and where this files can be found.</p>")+
    explanationOfVariables+explanationOfRelativePath+"</qt>");

    label=new QLabel(i18n("&Path to MO-files:"),box);
    hbox = new QHBox(box);
    hbox->setSpacing(KDialog::spacingHint());
    _moPathEdit = new KLineEdit(hbox);

#warning FIXME: completions
    //_moPathEdit->setCompletionObject( new KURLCompletion() );
    _moPathEdit->setMinimumSize(250,_moPathEdit->sizeHint().height());
    label->setBuddy(_moPathEdit);
    QPushButton* browse = new QPushButton(i18n("Browse..."),hbox);
    connect(browse,SIGNAL(clicked()),this,SLOT(browseMO()));

    hbox->setEnabled(false);

    QWhatsThis::add(hbox,i18n("<qt><p><b>This is not implemented yet!</b></p></qt>"));
    QWhatsThis::add(label,i18n("<qt><p><b>This is not implemented yet!</b></p></qt>"));


    label=new QLabel(i18n("Path to co&mpendium file:"),box);
    hbox = new QHBox(box);
    hbox->setSpacing(KDialog::spacingHint());
    _compendiumEdit = new KLineEdit(hbox);
    //_compendiumEdit->setCompletionObject( new KURLCompletion() );
    _compendiumEdit->setMinimumSize(250,_compendiumEdit->sizeHint().height());
    label->setBuddy(_compendiumEdit);
    browse = new QPushButton(i18n("Browse..."),hbox);
    connect(browse,SIGNAL(clicked()),this,SLOT(browseCompendium()));



    QString msg="<qt>"+i18n("<p><b>Path to compendium file</b></p>\n\
<p>Configure what file has to be used when activating search in compendium.</p>\n\
<p>The compendium is a normal PO-File, which should contain a list\n\
of standard translation from your translation team. If you don't have one, you can also use\n\
a file that contains all used translations of your team (e.g. the $lang.messages file in the\n\
KDE Project).</p>")+explanationOfVariables+explanationOfRelativePath+"</qt>";
    QWhatsThis::add(hbox,msg);
    QWhatsThis::add(label,msg);


    label=new QLabel(i18n("Path to au&xiliary file:"),box);
    hbox = new QHBox(box);
    hbox->setSpacing(KDialog::spacingHint());
    _auxiliaryEdit = new KLineEdit(hbox);
    //_auxiliaryEdit->setCompletionObject( new KURLCompletion() );
    _auxiliaryEdit->setMinimumSize(250,_auxiliaryEdit->sizeHint().height());
    label->setBuddy(_auxiliaryEdit);
    browse = new QPushButton(i18n("Browse..."),hbox);
    connect(browse,SIGNAL(clicked()),this,SLOT(browseAuxiliary()));


    msg="<qt>"+i18n("<p><b>Path to auxiliary file</b></p>\n\
<p>Configure what file has to be used when activating search in auxiliary.</p>\n\
<p>The auxiliary should help you finding the context of a translation\n\
by looking up the same message in a message catalog of the same package but translated to another\n\
language. This way you can have a look how this message is translated in another language.</p>")
    +explanationOfVariables+explanationOfRelativePath+"</qt>";

    QWhatsThis::add(hbox,msg);
    QWhatsThis::add(label,msg);


    label=new QLabel(i18n("You can use @LANG@ for your language or\n\
@PACKAGE@ for the name of the open file (without extension)."),box);

    QWhatsThis::add(label,"<qt>"+i18n("<p><b>Using variables</b></p>")+explanationOfVariables+"</qt>");

    layout->addStretch(1);

    page->setMinimumSize(layout->sizeHint());
    addTab(page,i18n("&Files"));

    defaults();


    setMinimumSize(sizeHint());
}


SearchPreferences::~SearchPreferences()
{
}


void SearchPreferences::setSettings(const SearchSettings& settings)
{
    _langEdit->setText(settings.language);

    _caseButton->setChecked(settings.caseSensitive);
    _ignoreAmpButton->setChecked(settings.ignoreAmpersand);
    _ignoreNewlineButton->setChecked(settings.ignoreNewline);
    _ignoreDotsButton->setChecked(settings.ignoreDots);
    _ignoreOtherButton->setChecked(settings.ignoreOther);

    QString other=settings.otherIgnores;

    // guess what this does: replace two "\" with one "\"    ;-)
    other.replace(QRegExp("\\\\\\\\"),"\\");
    other.replace(QRegExp("\\\\-"),"-");
    other.replace(QRegExp("\\\\^"),"^");
    other.replace(QRegExp("\\\\]"),"]");
    _otherEdit->setText(other);

    _exactButton->setChecked(settings.exact);
    _containButton->setChecked(settings.contains);
    _isContainedButton->setChecked(settings.isContained);

    _searchButton->setChecked(settings.autoSearch);

    SearchMode mode=settings.defaultMode;

    switch(mode)
    {
       case CompendiumSearch:
          _inCompendiumButton->setChecked(true);
          break;
       case AuxiliarySearch:
          _inAuxiliaryButton->setChecked(true);
          break;
       default:
          _inCatsButton->setChecked(true);
          break;
    }

    _moPathEdit->setText(settings.moPath);
    _compendiumEdit->setText(settings.compendium);
    _auxiliaryEdit->setText(settings.auxiliary);
}


SearchSettings SearchPreferences::settings() const
{
    SearchSettings settings;

    settings.language=_langEdit->text();

    settings.caseSensitive=_caseButton->isChecked();
    settings.ignoreAmpersand=_ignoreAmpButton->isChecked();
    settings.ignoreNewline=_ignoreNewlineButton->isChecked();
    settings.ignoreDots=_ignoreDotsButton->isChecked();
    settings.ignoreOther=_ignoreOtherButton->isChecked();

    QString other=_otherEdit->text();
    other.replace(QRegExp("\\"),"\\\\");
    other.replace(QRegExp("-"),"\\-");
    other.replace(QRegExp("\\^"),"\\^");
    other.replace(QRegExp("]"),"\\]");
    settings.otherIgnores=other;

    settings.exact=_exactButton->isChecked();
    settings.contains=_containButton->isChecked();
    settings.isContained=_isContainedButton->isChecked();


    settings.autoSearch=_searchButton->isChecked();

    SearchMode mode=CatalogSearch;
    if(_inCatsButton->isChecked())
       mode=CatalogSearch;
    else if(_inCompendiumButton->isChecked())
       mode=CompendiumSearch;
    else if(_inAuxiliaryButton->isChecked())
       mode=AuxiliarySearch;

    settings.defaultMode=mode;


    settings.moPath=_moPathEdit->text();
    settings.compendium=_compendiumEdit->text();
    settings.auxiliary=_auxiliaryEdit->text();

    return settings;
}



void SearchPreferences::defaults()
{
    _caseButton->setChecked(Defaults::Search::caseSensitive);
    _ignoreAmpButton->setChecked(Defaults::Search::ignoreAmpersand);
    _ignoreNewlineButton->setChecked(Defaults::Search::ignoreNewline);
    _ignoreDotsButton->setChecked(Defaults::Search::ignoreDots);
    _ignoreOtherButton->setChecked(Defaults::Search::ignoreOther);

    _otherEdit->setText(Defaults::Search::otherIgnores);
    _langEdit->setText(Defaults::Search::defaultLanguage());

    _exactButton->setChecked(Defaults::Search::exact);
    _containButton->setChecked(Defaults::Search::contains);
    _isContainedButton->setChecked(Defaults::Search::isContained);


    _searchButton->setChecked(Defaults::Search::autoSearch);


    switch(Defaults::Search::defaultMode)
    {
       case CompendiumSearch:
          _inCompendiumButton->setChecked(true);
          break;
       case AuxiliarySearch:
          _inAuxiliaryButton->setChecked(true);
          break;
       default:
          _inCatsButton->setChecked(true);
          break;
    }


    _moPathEdit->setText(Defaults::Search::moPath);
    _compendiumEdit->setText(Defaults::Search::compendium);
    _auxiliaryEdit->setText(Defaults::Search::auxiliary);

}


void SearchPreferences::exactButtonToggled(bool on)
{
   if(!on)
   {
      if(!_containButton->isChecked() && !_isContainedButton->isChecked())
      {
         _exactButton->setChecked(true);
      }
   }
}

void SearchPreferences::containButtonToggled(bool on)
{
   if(!on)
   {
      if(!_exactButton->isChecked() && !_isContainedButton->isChecked())
      {
         _containButton->setChecked(true);
      }
   }
}

void SearchPreferences::isContainedButtonToggled(bool on)
{
   if(!on)
   {
      if(!_containButton->isChecked() && !_exactButton->isChecked())
      {
         _isContainedButton->setChecked(true);
      }
   }
}


QSize SearchPreferences::sizeHint () const
{
#warning FIXME: sizeHint
   //QSize s=KTabCtl::sizeHint();
   QSize s(400,340);

   return s;
}

QSize SearchPreferences::minimumSizeHint () const
{
   return sizeHint();
}





void SearchPreferences::browseMO()
{
    QString name=KFileDialog::getOpenFileName();
    if(!name)
    {
       return;
    }

    _moPathEdit->setText(name);
}

void SearchPreferences::browseCompendium()
{
    QString name=KFileDialog::getOpenFileName();
    if(!name)
    {
       return;
    }

    _compendiumEdit->setText(name);
}

void SearchPreferences::browseAuxiliary()
{
    QString name=KFileDialog::getOpenFileName();
    if(!name)
    {
       return;
    }

    _auxiliaryEdit->setText(name);
}




CatManPreferences::CatManPreferences(QWidget* parent)
         : KTabCtl(parent)
{
    QWidget* page = new QWidget(this);

    QVBoxLayout* layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    QGroupBox* box=new QGroupBox(1,Qt::Horizontal,page);
    box->setMargin(KDialog::marginHint());
    layout->addWidget(box);

    QLabel* label=new QLabel(i18n("&Base directory of PO-Files:"),box);
    QHBox* hbox = new QHBox(box);
    hbox->setSpacing(KDialog::spacingHint());

    _poDirEdit = new KLineEdit(hbox);
#warning FIXME: completion
    //_poDirEdit->setCompletionObject( new KURLCompletion() );
    _poDirEdit->setMinimumSize(250,_poDirEdit->sizeHint().height());
    label->setBuddy(_poDirEdit);
    QPushButton* browse = new QPushButton(i18n("Browse..."),hbox);
    connect(browse,SIGNAL(clicked()),this,SLOT(browsePO()));


    label=new QLabel(i18n("&Base directory of POT-Files:"),box);
    hbox = new QHBox(box);
    hbox->setSpacing(KDialog::spacingHint());

    _potDirEdit = new KLineEdit(hbox);
    //_potDirEdit->setCompletionObject( new KURLCompletion() );
    _potDirEdit->setMinimumSize(250,_potDirEdit->sizeHint().height());
    label->setBuddy(_potDirEdit);
    browse = new QPushButton(i18n("Browse..."),hbox);
    connect(browse,SIGNAL(clicked()),this,SLOT(browsePOT()));



    QWhatsThis::add(box,i18n("<qt><p><b>Base directories</b></p>\n\
<p>Type in the directories which contains all your PO- and POT-files.\n\
The files and the directories in these directories will then be merged to one\n\
tree.</p></qt>"));


    box=new QGroupBox(1,Qt::Horizontal,page);
    box->setMargin(KDialog::marginHint());
    layout->addWidget(box);

    _openWindowButton = new QCheckBox(i18n("Open files in new window"),box);


    QWhatsThis::add(_openWindowButton,i18n("<qt><p><b>Open files in new window</b></p>\n\
<p>If this is activated all files that are opened from the Catalog Manager are opened\n\
in a new window.</p></qt>"));

    _killButton = new QCheckBox( i18n("&Kill processes on exit") , box );

    QWhatsThis::add( _killButton , i18n("<qt><p><b>Kill processes on exit</b></p>\n\
<p>If you check this, KBabel tries to kill the processes, that are not exited already when KBabel exits,\n\
by sending a kill signal to them.</p>\n\
<p>NOTE: It's not guaranteed, that the processes are killed.</p></qt>") );


    layout->addStretch(1);

    page->setMinimumSize(layout->sizeHint());

    addTab( page , i18n("&General") );



    page = new QWidget( this );
    layout = new QVBoxLayout( page );
    layout->setSpacing( KDialog::spacingHint() );
    layout->setMargin( KDialog::marginHint() );

    box = new QGroupBox( 1 , Qt::Horizontal , i18n("Commands for directories") , page );
    box->setMargin( KDialog::marginHint() );
    layout->addWidget( box );

    _dirCmdEdit = new CmdEdit( box );
    label = new QLabel( i18n("Replaceables:\n@PACKAGE@, @PODIR@, @POTDIR@"), box);

    QWhatsThis::add( box , i18n("<qt><p><b>Commands for directories</b></p>\n\
<p>Insert here the commands you want to execute in directories from the Catalog Manager.\n\
The commands are then shown in the submenu <b>Commands</b> in the Catalog Manager's context menu.</p>\n\
<p>Insert in field <b>Name</b> the name of the command. The name can be chosen freely and is \n\
only used to be displayed in the menu. In field <b>Command</b> insert the command, you want to\n\
have executed when selecting the corresponding menu item. Then press <b>Add</b> to add the\n\
command to your available commands. To edit a command, select it, press <b>Edit</b>\n\
and press <b>Add</b> after you have finished.</p>\n\
<p>The command is executed through your default shell, so you can execute multiple commands at\n\
once by separating them with a semicolon, and you can set environment variables, if you need.\n\
The commands are executed in the (PO-file) directory, you have selected in the Catalog Manager.</p>\n\
<p>The following strings will be replaced in a command:<ul>\n\
<li>@PACKAGE@: The name of the directory without path</li>\n\
<li>@PODIR@: The name of the PO-directory with path</li>\n\
<li>@POTDIR@: The name of the template directory with path</li></ul></p>\n\
<p>E.g.: If you want to execute \"<tt>make</tt>\" and then \"<tt>make install</tt>\" you could insert\n\
in field <b>Name</b> \"<tt>Make install</tt>\" and in field <b>Command</b> \"<tt>make; make install</tt>\".\n\
If you then select <b>Commands->Make install</b> from the context menu of a directory,\n\
the commands \"<tt>make</tt>\" and \"<tt>make install</tt>\" will be executed in this directory.</p></qt>") );



    layout->addStretch(1);

    addTab( page , i18n("D&irectory Commands") );


    page = new QWidget( this );
    layout = new QVBoxLayout( page );
    layout->setSpacing( KDialog::spacingHint() );
    layout->setMargin( KDialog::marginHint() );

    box=new QGroupBox( 1 , Qt::Horizontal , i18n("Commands for files") , page );
    box->setMargin( KDialog::marginHint() );
    layout->addWidget( box );

    _fileCmdEdit = new CmdEdit( box );
    label = new QLabel( i18n("Replaceables:\n\
@PACKAGE@, @POFILE@,@POTFILE@,\n@PODIR@, @POTDIR@"), box);



    QWhatsThis::add( box , i18n("<qt><p><b>Commands for files</b></p>\n\
<p>Insert here the commands you want to execute on files from the Catalog Manager.\n\
The commands are then shown in the submenu <b>Commands</b> in the Catalog Manager's context menu.</p>\n\
<p>Insert in field <b>Name</b> the name of the command. The name can be chosen freely and is \n\
only used to be displayed in the menu. In field <b>Command</b> insert the command, you want to\n\
have executed when selecting the corresponding menu item. Then press <b>Add</b> to add the\n\
command to your available commands. To edit a command, select it, press <b>Edit</b>\n\
and press <b>Add</b> after you have finished.</p>\n\
<p>The command is executed through your default shell, so you can execute multiple commands at\n\
once by separating them with a semicolon, and you can set environment variables, if you need.\n\
The commands are executed in the (PO-file) directory, in which the file, you have selected in the\n\
Catalog Manager, is.</p>\n\
<p>The following strings will be replaced in a command:<ul>\n\
<li>@PACKAGE@: The name of the file without path and extension</li>\n\
<li>@POFILE@: The name of the PO-file with path and extension</li>\n\
<li>@POTFILE@: The name of the corresponding template file with path and extension</li>\n\
<li>@PODIR@: The name of the directory, the PO-file is in, with path</li>\n\
<li>@POTDIR@: The name of the directory, the template file is in, with path</li></ul></p>\n\
<p>E.g.: If you want to merge the template file into your PO-file you could insert in field\n\
<b>Name</b> \"<tt>Merge</tt>\" and in field <b>Command</b>\n\
\"<tt>msgmerge @POFILE@ @POTFILE@ > @PACKAGE@.new && mv @PACKAGE@.new @PACKAGE@.po</tt>\".\n\
If you then select <b>Commands->Merge</b> from a file's context menu,\n\
the po-file will be merged with its template file.</p></qt>") );



    layout->addStretch(1);

    addTab( page , i18n("&File Commands") );



    defaults();

    setMinimumSize(sizeHint());
}


CatManPreferences::~CatManPreferences()
{
}


void CatManPreferences::setSettings(const CatManSettings& settings)
{
   _poDirEdit->setText(settings.poBaseDir);
   _potDirEdit->setText(settings.potBaseDir);

   _openWindowButton->setChecked(settings.openWindow);

   _dirCmdEdit->setCommands( settings.dirCommands , settings.dirCommandNames );
   _fileCmdEdit->setCommands( settings.fileCommands , settings.fileCommandNames );

   _killButton->setChecked( settings.killCmdOnExit );
}


CatManSettings CatManPreferences::settings() const
{
    CatManSettings settings;

    settings.poBaseDir=_poDirEdit->text();
    settings.potBaseDir=_potDirEdit->text();
    settings.openWindow=_openWindowButton->isChecked();

    _dirCmdEdit->commands( settings.dirCommands , settings.dirCommandNames );
    _fileCmdEdit->commands( settings.fileCommands , settings.fileCommandNames );

    settings.killCmdOnExit = _killButton->isChecked() ;

    return settings;
}

void CatManPreferences::defaults()
{
   _poDirEdit->setText(Defaults::CatalogManager::poBaseDir());
   _potDirEdit->setText(Defaults::CatalogManager::potBaseDir());

   _openWindowButton->setChecked(Defaults::CatalogManager::openWindow);

   _dirCmdEdit->setCommands( Defaults::CatalogManager::dirCommands()
                            , Defaults::CatalogManager::dirCommandNames() );
   _fileCmdEdit->setCommands( Defaults::CatalogManager::fileCommands()
                            , Defaults::CatalogManager::fileCommandNames() );

   _killButton->setChecked( Defaults::CatalogManager::killCmdOnExit );
}


void CatManPreferences::browsePO()
{
    QString name=KFileDialog::getExistingDirectory(_poDirEdit->text(),this);
    if(!name)
    {
       return;
    }

    _poDirEdit->setText(name);
}


void CatManPreferences::browsePOT()
{
    QString name=KFileDialog::getExistingDirectory(_potDirEdit->text(),this);
    if(!name)
    {
       return;
    }

    _potDirEdit->setText(name);
}


QSize CatManPreferences::sizeHint () const
{
#warning FIXME: sizeHint
   //QSize s=KTabCtl::sizeHint();

   QSize s(400,310);

   return s;
}

QSize CatManPreferences::minimumSizeHint () const
{
   return sizeHint();
}


