/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%               DDDD   IIIII  SSSSS  TTTTT   OOO   RRRR   TTTTT               %
%               D   D    I    SS       T    O   O  R   R    T                 %
%               D   D    I     SSS     T    O   O  RRRR     T                 %
%               D   D    I       SS    T    O   O  R R      T                 %
%               DDDD   IIIII  SSSSS    T     OOO   R  R     T                 %
%                                                                             %
%                                                                             %
%                     ImageMagick Image Distortion Methods.                   %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                                 June 2007                                   %
%                                                                             %
%                                                                             %
%  Copyright 1999-2007 ImageMagick Studio LLC, a non-profit organization      %
%  dedicated to making software imaging solutions freely available.           %
%                                                                             %
%  You may not use this file except in compliance with the License.  You may  %
%  obtain a copy of the License at                                            %
%                                                                             %
%    http://www.imagemagick.org/script/license.php                            %
%                                                                             %
%  Unless required by applicable law or agreed to in writing, software        %
%  distributed under the License is distributed on an "AS IS" BASIS,          %
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   %
%  See the License for the specific language governing permissions and        %
%  limitations under the License.                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/cache-view.h"
#include "magick/distort.h"
#include "magick/exception.h"
#include "magick/exception-private.h"
#include "magick/gem.h"
#include "magick/hashmap.h"
#include "magick/image.h"
#include "magick/list.h"
#include "magick/memory_.h"
#include "magick/pixel.h"
#include "magick/pixel-private.h"
#include "magick/registry.h"
#include "magick/semaphore.h"
#include "magick/splay-tree.h"
#include "magick/string_.h"

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   D i s t o r t I m a g e                                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DistortImage() distorts an image using various distortion methods, by
%  mapping color lookups of the source image to a new destination image
%  generally of the same size.
%
%  The format of the DistortImage() method is:
%
%      Image *DistortImage(const Image *image,const DistortImageMethod method,
%        const unsigned long number_arguments,const double *arguments,
%        ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image: The image to be distorted.
%
%    o method: The method of image distortion.
%
%    o number_arguments: The number of arguments given for this distortion method.
%
%    o arguments: The arguments for this distortion method.
%
%    o exception: Return any errors or warnings in this structure
%
*/

static MagickBooleanType GaussJordanElimination(double **matrix,
  const unsigned long rank,double *vector)
{
#define GaussJordanSwap(x,y) \
{ \
  if ((x) != (y)) \
    { \
      (x)+=(y); \
      (y)=(x)-(y); \
      (x)=(x)-(y); \
    } \
}

  double
    max,
    scale;

  long
    column,
    *columns,
    *pivots,
    row,
    *rows;

  register long
    i,
    j,
    k;

  columns=(long *) AcquireMagickMemory(rank*sizeof(*columns));
  rows=(long *) AcquireMagickMemory(rank*sizeof(*rows));
  pivots=(long *) AcquireMagickMemory(rank*sizeof(*pivots));
  if ((rows == (long *) NULL) || (columns == (long *) NULL) ||
      (pivots == (long *) NULL))
    {
      if (pivots != (long *) NULL)
        pivots=(long *) RelinquishMagickMemory(pivots);
      if (columns != (long *) NULL)
        columns=(long *) RelinquishMagickMemory(columns);
      if (rows != (long *) NULL)
        rows=(long *) RelinquishMagickMemory(rows);
      return(MagickFalse);
    }
  (void) ResetMagickMemory(columns,0,rank*sizeof(*columns));
  (void) ResetMagickMemory(rows,0,rank*sizeof(*rows));
  (void) ResetMagickMemory(pivots,0,rank*sizeof(*pivots));
  column=0;
  row=0;
  for (i=0; i < (long) rank; i++)
  {
    max=0.0;
    for (j=0; j < (long) rank; j++)
      if (pivots[j] != 1)
        {
          for (k=0; k < (long) rank; k++)
            if (pivots[k] != 0)
              {
                if (pivots[k] > 1)
                  return(MagickFalse);
              }
            else
              if (fabs(matrix[j][k]) >= max)
                {
                  max=fabs(matrix[j][k]);
                  row=j;
                  column=k;
                }
        }
    pivots[column]++;
    if (row != column)
      {
        for (k=0; k < (long) rank; k++)
          GaussJordanSwap(matrix[row][k],matrix[column][k]);
        GaussJordanSwap(vector[row],vector[column]);
      }
    rows[i]=row;
    columns[i]=column;
    if (matrix[column][column] == 0.0)
      return(MagickFalse);  /* sigularity */
    scale=1.0/matrix[column][column];
    matrix[column][column]=1.0;
    for (j=0; j < (long) rank; j++)
      matrix[column][j]*=scale;
    vector[column]*=scale;
    for (j=0; j < (long) rank; j++)
      if (j != column)
        {
          scale=matrix[j][column];
          matrix[j][column]=0.0;
          for (k=0; k < (long) rank; k++)
            matrix[j][k]-=scale*matrix[column][k];
          vector[j]-=scale*vector[column];
        }
  }
  for (j=(long) rank-1; j >= 0; j--)
    if (columns[j] != rows[j])
      for (i=0; i < (long) rank; i++)
        GaussJordanSwap(matrix[i][rows[j]],matrix[i][columns[j]]);
  pivots=(long *) RelinquishMagickMemory(pivots);
  rows=(long *) RelinquishMagickMemory(rows);
  columns=(long *) RelinquishMagickMemory(columns);
  return(MagickTrue);
}

static MagickBooleanType SolveAffineDistortion(
  const unsigned long number_points,const PointInfo *points,double **matrix,
  double *vector)
{
  /*
    Given the coordinates of two triangles
      u0,v0, ... u2,v2,   x3,y3, ... x5,y5

    Solve for the 6 cofficients c0..c6 for a affine distortion:
      u = c0*x + c2*y + c4
      v = c1*x + c3*y + c5
  */
  if (number_points < 6)
    return(MagickFalse);
  matrix[0][0]=points[3].x;
  matrix[0][2]=points[3].y;
  matrix[0][4]=1.0;
  matrix[1][1]=points[3].x;
  matrix[1][3]=points[3].y;
  matrix[1][5]=1.0;
  matrix[2][0]=points[4].x;
  matrix[2][2]=points[4].y;
  matrix[2][4]=1.0;
  matrix[3][1]=points[4].x;
  matrix[3][3]=points[4].y;
  matrix[3][5]=1.0;
  matrix[4][0]=points[5].x;
  matrix[4][2]=points[5].y;
  matrix[4][4]=1.0;
  matrix[5][1]=points[5].x;
  matrix[5][3]=points[5].y;
  matrix[5][5]=1.0;
  vector[0]=points[0].x;
  vector[1]=points[0].y;
  vector[2]=points[1].x;
  vector[3]=points[1].y;
  vector[4]=points[2].x;
  vector[5]=points[2].y;
  return(GaussJordanElimination(matrix,6,vector));
}

static MagickBooleanType SolveBilinearDistortion(
  const unsigned long number_points,const PointInfo *points,double **matrix,
  double *vector)
{
  /*
    Given the coordinates of two quadrilaterals:
      u0,v0, ... u3,v3,   x4,y4, ... x7,y7

    Solve for the 8 coeffecients c0..c7 for a bilinear distortion:
      u = c0*x + c1*y + c2*x*y + c3
      v = c4*x + c5*y + c6*x*y + c7
  */
  if (number_points < 8)
    return(MagickFalse);
  matrix[0][0]=points[4].x;
  matrix[0][1]=points[4].y;
  matrix[0][2]=points[4].x*points[4].y;
  matrix[0][3]=1.0;
  matrix[1][4]=points[4].x;
  matrix[1][5]=points[4].y;
  matrix[1][6]=points[4].x*points[4].y;
  matrix[1][7]=1.0;
  matrix[2][0]=points[5].x;
  matrix[2][1]=points[5].y;
  matrix[2][2]=points[5].x*points[5].y;
  matrix[2][3]=1.0;
  matrix[3][4]=points[5].x;
  matrix[3][5]=points[5].y;
  matrix[3][6]=points[5].x*points[5].y;
  matrix[3][7]=1.0;
  matrix[4][0]=points[6].x;
  matrix[4][1]=points[6].y;
  matrix[4][2]=points[6].x*points[6].y;
  matrix[4][3]=1.0;
  matrix[5][4]=points[6].x;
  matrix[5][5]=points[6].y;
  matrix[5][6]=points[6].x*points[6].y;
  matrix[5][7]=1.0;
  matrix[6][0]=points[7].x;
  matrix[6][1]=points[7].y;
  matrix[6][2]=points[7].x*points[7].y;
  matrix[6][3]=1.0;
  matrix[7][4]=points[7].x;
  matrix[7][5]=points[7].y;
  matrix[7][6]=points[7].x*points[7].y;
  matrix[7][7]=1.0;
  vector[0]=points[0].x;
  vector[1]=points[0].y;
  vector[2]=points[1].x;
  vector[3]=points[1].y;
  vector[4]=points[2].x;
  vector[5]=points[2].y;
  vector[6]=points[3].x;
  vector[7]=points[3].y;
  return(GaussJordanElimination(matrix,8,vector));
}

static MagickBooleanType SolvePerspectiveDistortion(
  const unsigned long number_points,const PointInfo *points,double **matrix,
  double *vector)
{
  /*
    Given the coordinates of two quadrilaterals:
      u0,v0, ... u3,v3,   x4,y4, ... x7,y7

    Solve for the 8 coefficients c0..c7 for a perspective distortion:
       u = ( c0*x + c1*y + c2 ) / ( c7*x + c8*y + 1 )
       v = ( c3*x + c4*y + c5 ) / ( c7*x + c8*y + 1 )
   */
  if (number_points < 8)
    return(MagickFalse);
  matrix[0][0]=points[4].x;
  matrix[0][1]=points[4].y;
  matrix[0][2]=1.0;
  matrix[0][6]=(-points[4].x*points[0].x);
  matrix[0][7]=(-points[4].y*points[0].x);
  matrix[1][3]=points[4].x;
  matrix[1][4]=points[4].y;
  matrix[1][5]=1.0;
  matrix[1][6]=(-points[4].x*points[0].y);
  matrix[1][7]=(-points[4].y*points[0].y);
  matrix[2][0]=points[5].x;
  matrix[2][1]=points[5].y;
  matrix[2][2]=1.0;
  matrix[2][6]=(-points[5].x*points[1].x);
  matrix[2][7]=(-points[5].y*points[1].x);
  matrix[3][3]=points[5].x;
  matrix[3][4]=points[5].y;
  matrix[3][5]=1.0;
  matrix[3][6]=(-points[5].x*points[1].y);
  matrix[3][7]=(-points[5].y*points[1].y);
  matrix[4][0]=points[6].x;
  matrix[4][1]=points[6].y;
  matrix[4][2]=1.0;
  matrix[4][6]=(-points[6].x*points[2].x);
  matrix[4][7]=(-points[6].y*points[2].x);
  matrix[5][3]=points[6].x;
  matrix[5][4]=points[6].y;
  matrix[5][5]=1.0;
  matrix[5][6]=(-points[6].x*points[2].y);
  matrix[5][7]=(-points[6].y*points[2].y);
  matrix[6][0]=points[7].x;
  matrix[6][1]=points[7].y;
  matrix[6][2]=1.0;
  matrix[6][6]=(-points[7].x*points[3].x);
  matrix[6][7]=(-points[7].y*points[3].x);
  matrix[7][3]=points[7].x;
  matrix[7][4]=points[7].y;
  matrix[7][5]=1.0;
  matrix[7][6]=(-points[7].x*points[3].y);
  matrix[7][7]=(-points[7].y*points[3].y);
  vector[0]=points[0].x;
  vector[1]=points[0].y;
  vector[2]=points[1].x;
  vector[3]=points[1].y;
  vector[4]=points[2].x;
  vector[5]=points[2].y;
  vector[6]=points[3].x;
  vector[7]=points[3].y;
  return(GaussJordanElimination(matrix,8,vector));
}

MagickExport Image *DistortImage(Image *image,const DistortImageMethod method,
  const unsigned long number_arguments,const double *arguments,
  ExceptionInfo *exception)
{
#define DistortImageTag  "Distort/Image"

  double
    coefficients[8],
    scale;

  Image
    *distort_image;

  long
    y;

  MagickBooleanType
    status;

  MagickPixelPacket
    pixel;

  PointInfo
    point;

  register IndexPacket
    *indexes;

  register long
    i,
    x;

  register PixelPacket
    *q;

  ViewInfo
    *image_view,
    *distort_view;

  /*
    Convert input arguments into appropriate arguments for distortion
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"%s",image->filename);
  assert(exception != (ExceptionInfo *) NULL);
  assert(exception->signature == MagickSignature);
  (void) ResetMagickMemory(coefficients,0,sizeof(coefficients));
  switch (method)
  {
    case AffineDistortion:
    {
      double
        *matrix[6];

      PointInfo
        *points;

      points=(PointInfo *) AcquireMagickMemory(((number_arguments+1)/2)*
        sizeof(*points));
      if (points == (PointInfo *) NULL)
        ThrowImageException(ResourceLimitError,"MemoryAllocationFailed");
      for (i=0; i < (long) number_arguments; i++)
        if ((i % 2 ) == 0)
          points[i/2].x=arguments[i];
        else
          points[i/2].y=arguments[i];
      for (i=0; i < 6; i++)
      {
        matrix[i]=(double *) AcquireMagickMemory(6*sizeof(*matrix[i]));
        if (matrix[i] == (double *) NULL)
          {
            for (x=0; x < i; x++)
              matrix[i]=(double *) RelinquishMagickMemory(matrix[i]);
            points=(PointInfo *) RelinquishMagickMemory(points);
            ThrowImageException(ResourceLimitError,"MemoryAllocationFailed");
          }
        (void) ResetMagickMemory(matrix[i],0,6*sizeof(*matrix[i]));
      }
      status=SolveAffineDistortion((number_arguments+1)/2,points,matrix,
        coefficients);
      for (i=0; i < 6; i++)
        matrix[i]=(double *) RelinquishMagickMemory(matrix[i]);
      points=(PointInfo *) RelinquishMagickMemory(points);
      break;
    }
    case AffineProjectionDistortion:
    {
      double
        determinant;

      /*
        Invert the given affine matrix: sx,rx,ry,sy,tx,ty.
      */
      if (number_arguments < 6)
        return((Image *) NULL);
      determinant=1.0/(arguments[0]*arguments[3]-arguments[1]*arguments[2]);
      coefficients[0]=determinant*arguments[3];
      coefficients[1]=determinant*(-arguments[1]);
      coefficients[2]=determinant*(-arguments[2]);
      coefficients[3]=determinant*arguments[0];
      coefficients[4]=(-arguments[4])*coefficients[0]-arguments[5]*
        coefficients[2];
      coefficients[5]=(-arguments[4])*coefficients[1]-arguments[5]*
        coefficients[3];
      break;
    }
    case BilinearDistortion:
    {
      double
        *matrix[8];

      PointInfo
        *points;

      points=(PointInfo *) AcquireMagickMemory(((number_arguments+1)/2)*
        sizeof(*points));
      if (points == (PointInfo *) NULL)
        ThrowImageException(ResourceLimitError,"MemoryAllocationFailed");
      for (i=0; i < (long) number_arguments; i++)
        if ((i % 2 ) == 0)
          points[i/2].x=arguments[i];
        else
          points[i/2].y=arguments[i];
      for (i=0; i < 8; i++)
      {
        matrix[i]=(double *) AcquireMagickMemory(8*sizeof(*matrix[i]));
        if (matrix[i] == (double *) NULL)
          {
            for (x=0; x < i; x++)
              matrix[i]=(double *) RelinquishMagickMemory(matrix[i]);
            points=(PointInfo *) RelinquishMagickMemory(points);
            ThrowImageException(ResourceLimitError,"MemoryAllocationFailed");
          }
        (void) ResetMagickMemory(matrix[i],0,8*sizeof(*matrix[i]));
      }
      status=SolveBilinearDistortion((number_arguments+1)/2,points,matrix,
        coefficients);
      for (i=0; i < 8; i++)
        matrix[i]=(double *) RelinquishMagickMemory(matrix[i]);
      points=(PointInfo *) RelinquishMagickMemory(points);
      break;
    }
    case PerspectiveDistortion:
    {
      double
        *matrix[8];

      PointInfo
        *points;

      points=(PointInfo *) AcquireMagickMemory(((number_arguments+1)/2)*
        sizeof(*points));
      if (points == (PointInfo *) NULL)
        ThrowImageException(ResourceLimitError,"MemoryAllocationFailed");
      for (i=0; i < (long) number_arguments; i++)
        if ((i % 2 ) == 0)
          points[i/2].x=arguments[i];
        else
          points[i/2].y=arguments[i];
      for (i=0; i < 8; i++)
      {
        matrix[i]=(double *) AcquireMagickMemory(8*sizeof(*matrix[i]));
        if (matrix[i] == (double *) NULL)
          {
            for (x=0; x < i; x++)
              matrix[i]=(double *) RelinquishMagickMemory(matrix[i]);
            points=(PointInfo *) RelinquishMagickMemory(points);
            ThrowImageException(ResourceLimitError,"MemoryAllocationFailed");
          }
        (void) ResetMagickMemory(matrix[i],0,8*sizeof(*matrix[i]));
      }
      status=SolvePerspectiveDistortion((number_arguments+1)/2,points,matrix,
        coefficients);
      for (i=0; i < 8; i++)
        matrix[i]=(double *) RelinquishMagickMemory(matrix[i]);
      points=(PointInfo *) RelinquishMagickMemory(points);
      break;
    }
    case ScaleRotateTranslateDistortion:
    {
      double
        cosine, sine,
        x,y,sx,sy,a,nx,ny;

      /*
         Argument options, by number of arguments given:
            a
            s, a
            x,y, a
            x,y, s, a
            x,y, sx,sy, a
            x,y, s, a, nx,ny
            x,y, sx,sy, a, nx,ny
         Where actions are (in order of application)
            x,y     'center' of transforms     (default = image center)
            sx,sy   scale image by this amount (default = 1)
            a       angle of rotation          (argument required)
            nx,ny   move 'center' here         (default = no movement)
         And convert to affine equation
      */
      x = nx = ((double)image->rows/2);
      y = ny = ((double)image->columns/2);
      sx = sy = 1.0;
      switch ( number_arguments ) {
      case 0:
        return((Image *) NULL);
      case 1:
        a = arguments[0];
        break;
      case 2:
        sx = sy = arguments[0];
        a = arguments[1];
      default:
        x = nx = arguments[0];
        y = ny = arguments[1];
        switch ( number_arguments ) {
        case 3:
          a = arguments[2];
          break;
        case 4:
          sx = sy = arguments[2];
          a = arguments[3];
          break;
        case 5:
          sx = arguments[2];
          sy = arguments[3];
          a = arguments[4];
          break;
        case 6:
          sx = sy = arguments[2];
          a = arguments[3];
          nx = arguments[4];
          ny = arguments[5];
          break;
        case 7:
          sx = arguments[2];
          sy = arguments[3];
          a = arguments[4];
          nx = arguments[5];
          ny = arguments[6];
          break;
        default:
          return((Image *) NULL);
        }
        break;
      }
      a=DegreesToRadians(a);
      cosine=cos(a);
      sine=sin(a);
      coefficients[0]=cosine/sx;
      coefficients[1]=(-sine)/sy;
      coefficients[2]=sine/sx;
      coefficients[3]=cosine/sy;
      coefficients[4]=x-nx*coefficients[0]-ny*coefficients[2];
      coefficients[5]=y-nx*coefficients[1]-ny*coefficients[3];
    }
    default:
      break;
  }
  /*
    Initialize the distort image attributes.
  */
  distort_image=CloneImage(image,image->columns,(unsigned long) image->rows,
    MagickTrue,exception);
  if (distort_image == (Image *) NULL)
    return((Image *) NULL);
  if (SetImageStorageClass(distort_image,DirectClass) == MagickFalse)
    {
      InheritException(exception,&distort_image->exception);
      distort_image=DestroyImage(distort_image);
      return((Image *) NULL);
    }
  if (distort_image->background_color.opacity != OpaqueOpacity)
    distort_image->matte=MagickTrue;
  /*
    Map the source image colors to each pixel in the distort image.
  */
  GetMagickPixelPacket(distort_image,&pixel);
  image_view=OpenCacheView(image);
  distort_view=OpenCacheView(distort_image);
  for (y=0; y < (long) distort_image->rows; y++)
  {
    q=SetCacheView(distort_view,0,y,distort_image->columns,1);
    if (q == (PixelPacket *) NULL)
      break;
    indexes=GetIndexes(distort_image);
    for (x=0; x < (long) distort_image->columns; x++)
    {
      switch (method)
      {
        case AffineDistortion:
        case AffineProjectionDistortion:
        case ScaleRotateTranslateDistortion:
        {
          point.x=coefficients[0]*x+coefficients[2]*y+coefficients[4];
          point.y=coefficients[1]*x+coefficients[3]*y+coefficients[5];
          break;
        }
        case BilinearDistortion:
        {
          point.x=coefficients[0]*x+coefficients[1]*y+coefficients[2]*x*y+
            coefficients[3];
          point.y=coefficients[4]*x+coefficients[5]*y+coefficients[6]*x*y+
            coefficients[7];
          break;
        }
        case PerspectiveDistortion:
        {
          scale=coefficients[6]*x+coefficients[7]*y+1.0;
          scale=1.0/(fabs(scale) <= MagickEpsilon ? 1.0 : scale);
          point.x=scale*(coefficients[0]*x+coefficients[1]*y+coefficients[2]);
          point.y=scale*(coefficients[3]*x+coefficients[4]*y+coefficients[5]);
          break;
        }
        default:
        {
          /*
            Noop distortion (failsafe).
          */
          point.x=(double) x;
          point.y=(double) y;
          break;
        }
      }
      pixel=InterpolatePixelColor(image,image_view,image->interpolate,
        point.x,point.y,exception);
      SetPixelPacket(distort_image,&pixel,q,indexes+x);
      q++;
    }
    if (SyncImagePixels(distort_image) == MagickFalse)
      break;
    if ((image->progress_monitor != (MagickProgressMonitor) NULL) &&
        (QuantumTick(y,image->rows) != MagickFalse))
      {
        status=image->progress_monitor(DistortImageTag,y,image->rows,
          image->client_data);
        if (status == MagickFalse)
          break;
      }
  }
  distort_view=CloseCacheView(distort_view);
  image_view=CloseCacheView(image_view);
  return(distort_image);
}
