/*************************************************************************
 *
 *  $RCSfile: conditn.c,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: jl $ $Date: 2001/03/14 09:48:10 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <premac.h>
#include <Threads.h>
#include "TimerImp.h"
#include <postmac.h>

#ifndef _OSL_TIME_H_
#include <osl/time.h>
#endif


#include <osl/conditn.h>
#include <osl/types.h>

typedef struct TThreadElem TThreadElem;

struct TThreadElem {
	TThreadElem	*fNext, *fPrev;
	ThreadID	fThreadID;
	int			result;
};

typedef struct TThreadQueue {
	TThreadElem	*fHead, *fTail;
} TThreadQueue;

typedef struct TCondition {
	TThreadQueue	fQueue;
	sal_Bool			fSet;
	long			timer;
} TCondition;

enum {
	CONDITION_UNSET		= 0,
	CONDITION_SET		= 1,
	CONDITION_TIMEOUT	= -1,
	CONDITION_CANCELED	= -2};

int QueueError = 0;

static sal_Bool IsThreadValid(ThreadID id)
{
	ThreadState	state;
	return (sal_Bool)(GetThreadState(id, &state) == noErr);
}

void CheckThreadQueue(TThreadQueue *queue)
{
	int error = QueueError;

	if ( queue->fHead )
	{
		if (queue->fHead->fPrev)
			error++;

		if (queue->fTail->fNext)
			error++;

		if (queue->fHead->fNext)
		{
			if (queue->fTail->fPrev == NULL)
				error++;
		}
		else
		{
			if (queue->fTail->fPrev)
				error++;

			if (queue->fTail != queue->fHead)
				error++;
		}
	}
	else
	{
	// if head is 0, tail must be too
		if (queue->fTail)
			error++;
	}
	if (error)
		QueueError = error;
}

void InitThreadQueue(TThreadQueue *queue)
{
	queue->fHead = queue->fTail = NULL;
	CheckThreadQueue(queue);
}

void AppendThread(TThreadQueue *queue, TThreadElem *elem)
{
	CheckThreadQueue(queue);
	if (queue->fTail)
	{
		elem->fNext = queue->fTail->fNext; /* should be NULL */
		queue->fTail->fNext = elem;
	}
	else
	{
		elem->fNext = NULL;
		queue->fHead = elem;
	}
	elem->fPrev = queue->fTail;
	queue->fTail = elem;
	CheckThreadQueue(queue);
}

void RemoveThread(TThreadQueue *queue, TThreadElem *elem)
{
	CheckThreadQueue(queue);
	if (elem->fPrev)
		elem->fPrev->fNext = elem->fNext;
	else	/* elem is the head */
		queue->fHead = elem->fNext;
	if (elem->fNext)
		elem->fNext->fPrev = elem->fPrev;
	else /* elem is the tail */
		queue->fTail = elem->fPrev;
	CheckThreadQueue(queue);
}

/*****************************************************************************/
/* osl_createCondition */
/*****************************************************************************/
oslCondition SAL_CALL osl_createCondition()
{
	TCondition	*c = malloc(sizeof(TCondition));
	if (c)
	{
		c->fSet = sal_False;
		c->timer = 0;
		InitThreadQueue(&c->fQueue);
	}
	return (oslCondition)c;
}

/*****************************************************************************/
/* osl_destroyCondition */
/*****************************************************************************/
void SAL_CALL osl_destroyCondition(oslCondition Condition)
{
	TCondition *c = (TCondition *)Condition;
	if (c)
	{
		TThreadElem *pElem = c->fQueue.fHead;
		if (c->timer)
		{
			CancelTimerTask(c->timer);
			DestroyTimerTask(c->timer);
			c->timer = 0;
		}
		while (pElem)
		{
			TThreadElem *pNext = pElem->fNext;
			if (SetThreadState(pElem->fThreadID, kReadyThreadState, kNoThreadID) != noErr)
			{
				RemoveThread(&c->fQueue, pElem);
				free(pElem);
			}
			pElem = pNext;
		}
		free(c);
	}}

/*****************************************************************************/
/* osl_setCondition */
/*****************************************************************************/
sal_Bool SAL_CALL osl_setCondition(oslCondition Condition)
{
	TCondition *c = (TCondition *)Condition;
	if (c)
	{
		TThreadElem *pElem = c->fQueue.fHead;
		if (c->timer)
		{
			CancelTimerTask(c->timer);
			DestroyTimerTask(c->timer);
			c->timer = 0;
		}
		c->fSet = sal_True;

		while (pElem)
		{
			TThreadElem *pNext = pElem->fNext;
			if (SetThreadState(pElem->fThreadID, kReadyThreadState, kNoThreadID) == noErr)
				pElem->result = CONDITION_SET;
			 else
			 {
				RemoveThread(&c->fQueue, pElem);
				free(pElem);
			}
			pElem = pNext;
		}
		return sal_True;
	}
	else
		return sal_False;
}

/*****************************************************************************/
/* osl_resetCondition */
/*****************************************************************************/
sal_Bool SAL_CALL osl_resetCondition(oslCondition Condition)
{
	TCondition *c = (TCondition *)Condition;
	if (c)
	{
		if (c->timer)
		{
			CancelTimerTask( c->timer );
			DestroyTimerTask( c->timer );
			c->timer = 0;
		}
		c->fSet = sal_False;
		return sal_True;
	}
	else
    	return sal_False;
}

/*****************************************************************************/
/* osl_waitCondition */
/*****************************************************************************/
static void pascal ConditionTimerTask(void *param)
{
	TThreadElem *pElem = (TThreadElem *)param;
	if (SetThreadState(pElem->fThreadID, kReadyThreadState, kNoThreadID) == noErr)
		pElem->result = CONDITION_TIMEOUT;
}

oslConditionResult SAL_CALL osl_waitCondition(oslCondition Condition, const TimeValue * pTimeout )
{
	TCondition *c = (TCondition *)Condition;
	if (c)
	{
//		pTimeout = NULL;

		if (c->fSet)
	        return osl_cond_result_ok;
		else
		{
			TThreadElem *pElem = malloc(sizeof(TThreadElem));
			long	millisecs;
			int		result;
			if (pElem == NULL)
				return osl_cond_result_error;
			if (c->timer)
			{
				CancelTimerTask(c->timer);
				DestroyTimerTask(c->timer);
				c->timer = 0;
			}
			GetCurrentThread(&pElem->fThreadID);
			pElem->result = CONDITION_UNSET;
			AppendThread(&c->fQueue, pElem);

			if (pTimeout)
			{
				millisecs = pTimeout->Seconds * 1000L + pTimeout->Nanosec / 1000000L;
				c->timer = CreateTimerTask(ConditionTimerTask, pElem);
				ScheduleTimerTask(c->timer, millisecs);
			}
			SetThreadState(pElem->fThreadID, kStoppedThreadState, kNoThreadID);
			if (c->timer)
			{
				CancelTimerTask(c->timer);
				DestroyTimerTask(c->timer);
				c->timer = 0;
			}
			result = pElem->result;
			RemoveThread(&c->fQueue, pElem);
			free(pElem);
			switch (result)
			{
				case CONDITION_SET:
					return osl_cond_result_ok;
				case CONDITION_TIMEOUT:
					return osl_cond_result_timeout;
				case CONDITION_UNSET:
				default:
					return osl_cond_result_error;
			}
		}
	}
	else
        return osl_cond_result_error;
}

/*****************************************************************************/
/* osl_checkCondition */
/*****************************************************************************/
sal_Bool SAL_CALL osl_checkCondition(oslCondition Condition)
{
	TCondition *c = (TCondition *)Condition;
	return (sal_Bool)(c && c->fSet);
}

