#! /bin/sh
#########################################################################
# Filename       : install.sh
# Creation Date  : 07/17/2001
# Author         : Keith Marshall
# Purpose        : Install the S1W
# Outputs        : written to stdout,
#                  1 = error
#                  0 = successful install
# Notes:         : stderr/stdout are logged to a file in /tmp
#
# Update History : 07/17/01 Keith Marshall
#                           Initial coding.
#                  08/30/01 Kay Koll
#                           Add verifyDBTables.sh
#                  04/04/02 Thomas Pfohe
#                           redesign to One-Click-Installer (OCI)
#
#########################################################################

#set -x

ScriptsDirectory="script"
SOWVersion=1.1m1
SolarisVersion=`uname -r | cut -d "." -f2`
StartTime=`date`
UniqueTempDir=/tmp/`echo $StartTime | sed s/" "/_/g | sed s/:/_/g`
LogFile="${UniqueTempDir}.log"
UlimitRequired="1024"

AWK="nawk"

if [ -d $0 ]
then
    sd_basename=`basename $0`
    sd_script=`ls -l $0 | sed "s/.*${sd_basename} -> //g"`
else
    sd_script="$0"
fi

cd "`dirname "$sd_script"`"

# save the current directory
InitialWorkingDirectory=`pwd`

if [ -r ./setup.ini ]
then
    SourceSetupDir=`grep path ./setup.ini | cut -d "=" -f2`
else
    SourceSetupDir="-"
fi

#------------------------------------------------------------------------
# Function  : verifyUserIsRoot
# Purpose   : Verifies that the user is running as 'root'
# Inputs    : NONE
# Returned  : 0 - if the user is 'root'
#             1 - exits if not 'root' and the -a flag is not set
#------------------------------------------------------------------------
verifyUserIsRoot()
{
    UserIDName=`id |cut -d "(" -f2 |cut -d ")" -f1"`
    
    echo " User name: ${UserIDName}"
    
    if [ ${UserIDName} != "root" ] && [ ${Flag_AllUsers} = "off" ]
    then
        echo
        echo " The install program requires the User ID to be 'root'"
        echo " Please relogin as 'root' and rerun setup"
        exit 1
    fi
    return 0
}

#------------------------------------------------------------------------
# Function  : verifyOS
# Purpose   : Checks for the required kernel version number.
# Inputs    : NONE
# Returned  : 0 - if the OS is Solaris 7/8
#             1 - exits if not Solaris 7/8
#------------------------------------------------------------------------
verifyOS()
{
    SolarisVersion7="7"
    SolarisVersion8="8"
    
    echo " Host:      `hostname`"
    echo " System:    `uname -a`"
    
    if [ $SolarisVersion != $SolarisVersion8 ] && [ $SolarisVersion != $SolarisVersion7 ]
    then
        echo
        echo " The operating system must be either Solaris 7 or Solaris 8."
        echo " Please try the install on a Solaris 7 or 8 machine."
        echo
        exit 1
    fi

    return 0
}

#------------------------------------------------------------------------
# Purpose   : Extracts the appended original setup script and executes it
#------------------------------------------------------------------------
detachOriginalSetupScript()
{
    # determine begin of file attached to this file by finding the # 4711 delimiter
    Delim1=`grep -n 4711_delim_1 $0 | tail -1`
    Delim1Line=`echo $Delim1 | cut -d ":" -f1`
    Delim1Line=`expr $Delim1Line + 1`
    # write attached file into file which name is given as 2nd field in the # 4711 delimiter line
    DelimFile=`echo $Delim1 | cut -d " " -f3`
    if [ ! -r  ./$DelimFile ]
    then
        tail +${Delim1Line} $0 > $DelimFile
        chmod 755 $DelimFile
    fi
}

#------------------------------------------------------------------------
# Purpose   : Holds the main install steps so
#             that we can tee the output
#------------------------------------------------------------------------
main()
{
    echo
    echo " Sun ONE Webtop ${SOWVersion} - One Click Installer"
    echo
    echo " Log file:  ${LogFile}"

    verifyUserIsRoot

    verifyOS
    
    if [ -d $SourceSetupDir/../$ScriptsDirectory ] && [ ${Flag_AllUsers} = "off" ]
    then
        cd $SourceSetupDir/../$ScriptsDirectory

        # verify and/or correct the ulimit requirement
        if [ ${Flag_uLimits} = "on" ] && [ -x  ./installUlimits.sh ]
        then
            ./installUlimits.sh ${UlimitRequired}
        fi
    
        # verify and/or correct the Solaris patches requirement
        if [ ${Flag_SolarisPatches} = "on" ] && [ -x  ./installSolarisPatches.sh ]
        then
            mkdir -p ${UniqueTempDir} > /dev/null
            
            for PatchZip in ../../../patches/solaris${SolarisVersion}/*.zip
            do
                unzip ${PatchZip} -d ${UniqueTempDir}
            done
            
            echo
            echo "Verifying Solaris system patches"
    
            ./installSolarisPatches.sh ${UniqueTempDir}
            
            rm -rf ${UniqueTempDir}
        fi
    
        # verify and/or correct the java requirement
        if [ ${Flag_Java} = "on" ] && [ -x  ./installJava.sh ]
        then
            mkdir -p ${UniqueTempDir} > /dev/null
    
            ./installJava.sh ${UniqueTempDir}
    
            rm -rf ${UniqueTempDir}
        fi
    
        # verify and/or correct the anonymous FTP server requirement
        if [ ${Flag_FTP} = "on" ] && [ -x  ./installFTP.sh ]
        then
            ./installFTP.sh 1
        fi
    
        # verify and/or correct the ghostscript requirement
        if [ ${Flag_Ghostscript} = "on" ] && [ -x  ./installGhostscript.sh ]
        then
            ./installGhostscript.sh
        fi
    
        # verify and/or correct the HylaFax requirement
        if [ ${Flag_HylaFAX} = "on" ] && [ -x  ./installHylafax.sh ]
        then
            ./installHylafax.sh
        fi
    
        cd $InitialWorkingDirectory
    fi

    # verify and/or correct the Sun ONE Webtop requirement
    if [ ${Flag_Webtop} = "on" ] && [ -f  ./installWebtop.sh ]
    then
        detachOriginalSetupScript
        chmod 755 ./installWebtop.sh
        ./installWebtop.sh ${InstallDirectory} ${UserDirectory} ${MachineName} ${PortRange}
    fi

    if [ -d $SourceSetupDir/../$ScriptsDirectory ]
    then
        cd $SourceSetupDir/../$ScriptsDirectory

        # verify and/or correct the anonymous FTP server requirement
        if [ ${Flag_FTP} = "on" ] && [ -x  ./installFTP.sh ] && [ ${Flag_AllUsers} = "off" ]
        then
            ./installFTP.sh 2 $SourceSetupDir $InstallDirectory
        fi
    fi    

    # output the final message
    echo
    echo " Your Install of Sun ONE Webtop is complete"
    echo " Please log out of the system and log back in OR"
    echo " type '. /etc/profile' from the command prompt"
    echo " so the changes to you global environment can take effect."
    echo
    echo " Install started at [ ${StartTime} ]"
    echo "       and ended at [ `date` ]"
    echo
    echo
}

#------------------------------------------------------------------------
# Purpose   : Displays usage
#------------------------------------------------------------------------
displayUsage()
{
    echo "usage:   setup [-r:] [-clsjfgxwa] [-n portrange] [-m machine] [-i installdir] [-u userdir]"
    echo "         setup [-r:] -0 [-r responsefile]"
    echo "         setup [-r:] -1 -r responsefile -p profile"
    echo "         setup [-r:] -2 -r responsefile"
    echo "         setup -h"
    echo
    echo "params:  -r: do not use X"
    echo "         -c  OneClickIntaller mode"
    echo "         -l  install ulimits (with CD only)"
    echo "         -s  install solaris patches (with CD only)"
    echo "         -j  install Java (with CD only)"
    echo "         -f  install FTP (with CD only)"
    echo "         -g  install Ghostscript (with CD only)"
    echo "         -x  install HylaFAX (with CD only)"
    echo "         -w  install Webtop"
    echo "         -a  allow installation even if not root"
    echo "         -n  change default ports"
    echo "         -m  use machine name"
    echo "         -n  use install dir"
    echo "         -n  use user dir"
    echo "         -0  start rspa step"
    echo "         -1  start rsp1 step"
    echo "         -2  start rsp2 step"
    echo "         -r  responsefile"
    echo "         -p profile"
    echo "         -h this screen"
    echo
    echo "example: setup -r: -cwa -n 23xxx -m margherita -i /usr/opt/webtop -u /usr/opt/users"
}

#------------------------------------------------------------------------

# use original setup script (OneClickInstall logic is not needed) if
#   - called in install directory
#   - the old parameters rspa, rsp1, rsp2 are used
echo $@ | grep rsp > /dev/null

if [ $? = 0 ] || [ "$SourceSetupDir" = "-" ]
then
    detachOriginalSetupScript
    exec "./setupsetup" $@
    exit $?
fi

# parse parameters
OriginalParams=$@

# filter "-r:" which has might been added to disable X interface
# but use the commandline interface of the setup
OriginalParams=`echo "$OriginalParams" | sed s/"-r: "//g`
OriginalParams=`echo "$OriginalParams" | sed s/r://g`

Flag_oneClickInstallerMode="off"
Flag_uLimits="off"
Flag_SolarisPatches="off"
Flag_Java="off"
Flag_FTP="off"
Flag_Ghostscript="off"
Flag_HylaFAX="off"
Flag_Webtop="off"
Flag_PortRange="off"
Flag_MachineName="off"
Flag_InstallDir="off"
Flag_UserDir="off"
Flag_AllUsers="off"
flag0="off"
flag1="off"
flag2="off"
Flag_Responsefile="off"
Flag_Profile="off"

InstallDirectory="-"
UserDirectory="-"
PortRange="-"
MachineName="-"
Profile="-"
Responsefile="-"

set -- $OriginalParams
set -- `getopt clsjfgxwn:m:i:u:ha012r:p: "$@"`

while [ $# -gt 0 ]
do
    case "$1" in
        -c) Flag_oneClickInstallerMode="on";;
        -l) Flag_uLimits="on";;
        -s) Flag_SolarisPatches="on";;
        -j) Flag_Java="on";;
        -f) Flag_FTP="on";;
        -g) Flag_Ghostscript="on";;
        -x) Flag_HylaFAX="on";;
        -w) Flag_Webtop="on";;
        -n) Flag_PortRange="on"; PortRange="$2"; shift;;
        -m) Flag_MachineName="on"; MachineName="$2"; shift;;
        -i) Flag_InstallDir="on"; InstallDirectory="$2"; shift;;
        -u) Flag_UserDir="on"; UserDirectory="$2"; shift;;
        -h) displayUsage; exit 1;;
        -a) Flag_AllUsers="on";;
        -0) flag0="on";;
        -1) flag1="on";;
        -2) flag2="on";;
        -r) Flag_Responsefile="on"; Responsefile="$2"; shift ;;
        -p) Flag_Profile="on"; Profile="$2"; shift;;
        --) shift; break;;
        -*) displayUsage; exit 1;;
        *)  break;;         # terminate while loop
    esac
    shift
done

# escape / for use with sed
EscapedResponsefile=`echo ${Responsefile} | cut -d "/" -f1-100 | ${AWK} '{gsub("/","\\\\/"); print $0}'`

if [ ${Flag_oneClickInstallerMode} = "on" ]
then
    # call main and tee output/error in logfile&console
    main 2>&1 | tee ${LogFile}
elif [ ${flag0} = "on" ] 
then
    if [ ${Flag_Responsefile} = "on" ]
    then
        TransformedParams=`echo $OriginalParams | sed s/-r" "//g`
        TransformedParams=`echo $TransformedParams | sed s/${EscapedResponsefile}//g`
        TransformedParams=`echo $TransformedParams | sed s/-0/-rspa:${EscapedResponsefile}/g`
    else
        TransformedParams=`echo $OriginalParams | sed s/-0/-rspa/g`
    fi
    detachOriginalSetupScript
    exec "./setupsetup" ${TransformedParams}
elif [ ${flag1} = "on" ] 
then
    if [ ${Flag_Responsefile} = "on" ]
    then
        TransformedParams=`echo $OriginalParams | sed s/-r" "//g`
        TransformedParams=`echo $TransformedParams | sed s/${EscapedResponsefile}//g`
        TransformedParams=`echo $TransformedParams | sed s/-1/-rsp1:${EscapedResponsefile}/g`
    else
        echo " ERROR: -1 needs parameter -r"
        displayUsage
        exit 1
    fi
    if [ ${Flag_Profile} = "on" ]
    then
        TransformedParams=`echo $OriginalParams | sed s/${Profile}//g`
        TransformedParams=`echo $TransformedParams | sed s/-p/-mset:${Profile}/g`
    else
        echo " ERROR: -1 needs parameter -p"
        displayUsage
        exit 1
    fi
    detachOriginalSetupScript
    exec "./setupsetup" ${TransformedParams}
elif [ ${flag2} = "on" ] 
then
    if [ ${Flag_Responsefile} = "on" ]
    then
        TransformedParams=`echo $OriginalParams | sed s/-r" "//g`
        TransformedParams=`echo $TransformedParams | sed s/${EscapedResponsefile}//g`
        TransformedParams=`echo $TransformedParams | sed s/-2/-rsp2:${EscapedResponsefile}/g`
    else
        echo " ERROR: -1 needs parameter -r"
        displayUsage
        exit 1
    fi
    detachOriginalSetupScript
    exec "./setupsetup" ${TransformedParams}
else
    detachOriginalSetupScript
    exec "./setupsetup" ${OriginalParams}
fi

# restore original start directory
cd ${InitialWorkingDirectory}

# set a successful return code
exit 0

# 4711_delim_1 setupsetup
