/*************************************************************************
 *
 *  $RCSfile: registryw9x.cxx,v $
 *
 *  $Revision: 1.7.98.1 $
 *
 *  last change: $Author: hr $ $Date: 2004/01/09 18:52:36 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

//---------------------------------------
//
//---------------------------------------

#ifndef _REGISTRYW9X_HXX_
#include "registryw9x.hxx"
#endif

#include <windows.h>
#include <malloc.h>

#ifndef _OSL_DIAGNOSE_H_
#include <osl/diagnose.h>
#endif

#ifndef _REGISTRYVALUEIMPL_HXX_
#include "registryvalueimpl.hxx"
#endif

#ifndef _REGISTRYEXCEPTION_HXX_
#include "registryexception.hxx"
#endif

#ifndef _STRINGCONVERTER_HXX_
#include "stringconverter.hxx"
#endif

#pragma warning(disable : 4786)

//---------------------------------------
//
//---------------------------------------

const size_t MAX_TMP_BUFF_SIZE = 1024 * sizeof(wchar_t);


//############################################
// Creation
// only possible through WindowsRegistry class
//############################################


//-----------------------------------------------------
/** Create instance and open the specified Registry key
*/
RegistryKeyImplWin9x::RegistryKeyImplWin9x(HKEY RootKey, const std::wstring& KeyName) :
	RegistryKeyImpl(RootKey, KeyName)
{
}

//-----------------------------------------------------
/** Create instance and open the specified Registry key
*/
RegistryKeyImplWin9x::RegistryKeyImplWin9x(HKEY RootKey) :
	RegistryKeyImpl(RootKey)
{
}

//-----------------------------------------------------
/** Create an instances of the specified Registry key,
	the key is assumed to be already opened.
*/
RegistryKeyImplWin9x::RegistryKeyImplWin9x(HKEY RootKey, HKEY SubKey, const std::wstring& KeyName, bool Writeable) :
	RegistryKeyImpl(RootKey, SubKey, KeyName, Writeable)
{
}


//############################################
// Queries
//############################################


//-----------------------------------------------------
/** The number of sub values of the key at hand

	@precond IsOpen = true

	@throws
*/
size_t RegistryKeyImplWin9x::GetSubValueCount() const
{
	OSL_PRECOND(IsOpen(), "Registry key is not open");

	DWORD nSubValues = 0;

	LONG rc = RegQueryInfoKeyA(
		m_hSubKey,
		0, 0, 0, 0, 0, 0, &nSubValues, 0, 0, 0, 0);

	if (ERROR_INVALID_HANDLE == rc)
		throw RegistryIOException(rc);
	else if (ERROR_SUCCESS != rc)
		throw RegistryException(rc);

	return nSubValues;
}

//-----------------------------------------------------
/** The number of sub-keys of the key at hand

	@precond IsOpen = true

	@throws
*/
size_t RegistryKeyImplWin9x::GetSubKeyCount() const
{
	OSL_PRECOND(IsOpen(), "Registry key is not open");

	DWORD nSubKeys = 0;

	LONG rc = RegQueryInfoKeyA(
		m_hSubKey,
		0, 0, 0, &nSubKeys, 0, 0, 0, 0, 0, 0, 0);

	if (ERROR_INVALID_HANDLE == rc)
		throw RegistryIOException(rc);
	else if (ERROR_SUCCESS != rc)
		throw RegistryException(rc);

	return nSubKeys;
}

//-----------------------------------------------------
/**
*/
StringListPtr RegistryKeyImplWin9x::GetSubKeyNames() const
{
	OSL_PRECOND(IsOpen(), "Registry key is not open");

	char        buff[1024];
	DWORD   buff_size = sizeof(buff);
	FILETIME ftime;

	StringList* key_names = new StringList();

	LONG rc = ERROR_SUCCESS;

	for (size_t i = 0; /* left empty */; i++)
	{
		rc = RegEnumKeyExA(
			m_hSubKey, i, buff, &buff_size,
			0, 0, 0, &ftime);

		if (ERROR_SUCCESS != rc &&
			ERROR_MORE_DATA != rc)
			break;

		buff_size = sizeof(buff);

		key_names->push_back(AnsiToUnicodeString(buff));
	}

	if (ERROR_INVALID_HANDLE == rc)
		throw RegistryIOException(rc);
	else if (ERROR_NO_MORE_ITEMS != rc && ERROR_SUCCESS != rc)
		throw RegistryException(rc);

	return (StringListPtr) key_names;
}

//-----------------------------------------------------
/**
*/
StringListPtr RegistryKeyImplWin9x::GetSubValueNames() const
{
	OSL_PRECOND(IsOpen(), "Registry key is not open");

	char        buff[1024];
	DWORD   buff_size = sizeof(buff);

	StringList* value_names = new StringList();

	LONG rc = ERROR_SUCCESS;

	for (size_t i = 0; /* left empty */; i++)
	{
        rc = RegEnumValueA(
			m_hSubKey, i, buff, &buff_size,
			0, 0, 0, 0);

		if (ERROR_SUCCESS != rc &&
			ERROR_MORE_DATA != rc)
			break;

		buff_size = sizeof(buff);

		value_names->push_back(AnsiToUnicodeString(buff));
	}

	if (ERROR_INVALID_HANDLE == rc)
		throw RegistryIOException(rc);
	else if (ERROR_NO_MORE_ITEMS != rc && ERROR_SUCCESS != rc)
		throw RegistryException(rc);

	return (StringListPtr) value_names;
}

//-----------------------------------------------------
/** Get the specified registry value

	@precond IsOpen = true
*/
RegistryValue RegistryKeyImplWin9x::GetValue(const std::wstring& Name) const
{
	OSL_PRECOND(IsOpen(), "Registry key is not open");

	DWORD Type;
	char  buff[MAX_TMP_BUFF_SIZE];
	DWORD size = sizeof(buff);

	LONG rc = RegQueryValueExA(
		m_hSubKey,
		UnicodeToAnsiString(Name).c_str(),
		0,
		&Type,
		reinterpret_cast<LPBYTE>(buff),
		&size);

	if (ERROR_FILE_NOT_FOUND == rc)
		throw RegistryValueNotFoundException(rc);
	else if (ERROR_ACCESS_DENIED == rc)
		throw RegistryAccessDeniedException(rc);
	else if (ERROR_SUCCESS != rc)
		throw RegistryException(rc);

	RegistryValue regval;

	if (REG_DWORD == Type)
    {
		regval = RegistryValue(new RegistryValueImpl(Name, *(reinterpret_cast<int*>(buff))));
    }
	else if (REG_SZ == Type || REG_EXPAND_SZ == Type || REG_MULTI_SZ == Type)
    {
        if (size > 0)
            regval = RegistryValue(new RegistryValueImpl(Name, std::string(reinterpret_cast<char*>(buff))));
        else
            regval = RegistryValue(new RegistryValueImpl(Name, std::string()));
    }
	else
    {
		OSL_ENSURE(false, "Registry value type not supported");
    }

	return regval;
}

//-----------------------------------------------------
/** Get the specified registry value, return the given
	default value if value not found

	@precond IsOpen = true
*/
RegistryValue RegistryKeyImplWin9x::GetValue(const std::wstring& Name, const RegistryValue& Default) const
{
	OSL_PRECOND(IsOpen(), "Registry key is not open");

	DWORD Type;
	char  buff[MAX_TMP_BUFF_SIZE];
	DWORD size = sizeof(buff);

	LONG rc = RegQueryValueExA(
		m_hSubKey,
		UnicodeToAnsiString(Name).c_str(),
		0,
		&Type,
		reinterpret_cast<LPBYTE>(buff),
		&size);

	if (ERROR_FILE_NOT_FOUND == rc)
	{
		#if (_MSC_VER < 1300)
		return Default;
		#else
		RegistryValue regval_ptr;
		regval_ptr = RegistryValue(new RegistryValueImpl(*Default));
		return regval_ptr;
		#endif
		}

	if (ERROR_ACCESS_DENIED == rc)
		throw RegistryAccessDeniedException(rc);
	else if (ERROR_SUCCESS != rc)
		throw RegistryException(rc);

	RegistryValue regval;

	if (REG_DWORD == Type)
		regval = RegistryValue(new RegistryValueImpl(Name, *reinterpret_cast<int*>(buff)));
	else if (REG_SZ == Type || REG_EXPAND_SZ == Type || REG_MULTI_SZ == Type)
		regval = RegistryValue(new RegistryValueImpl(Name, std::string(reinterpret_cast<char*>(buff))));
	else
		OSL_ENSURE(false, "Registry value type not supported");

	return regval;
}


//############################################
// Commands
//############################################


//-----------------------------------------------------
/** Open the registry key, has no effect if
	the key is already open

	@precond IsOpen = false

	@throws RegistryKeyNotFoundException
			RegistryWriteAccessDenyException
			RegistryAccessDenyException
*/
void RegistryKeyImplWin9x::Open(bool Writeable)
{
	OSL_PRECOND(!IsOpen(), "Registry key is already open - Resource leak!");

	REGSAM regsam = KEY_READ;

	if (Writeable)
		regsam |= KEY_WRITE;

	LONG rc = RegOpenKeyExA(
		m_hRootKey,
		UnicodeToAnsiString(m_KeyName).c_str(),
		0,
		regsam,
		&m_hSubKey);

	if (ERROR_FILE_NOT_FOUND == rc)
		throw RegistryKeyNotFoundException(rc);
	else if (ERROR_ACCESS_DENIED == rc)
		throw RegistryAccessDeniedException(rc);
	else if (ERROR_SUCCESS != rc)
		throw RegistryException(rc);

	m_IsWriteable = Writeable;

	OSL_POSTCOND(IsOpen(), "Could not open Registry key");
}

//-----------------------------------------------------
/** Open the specified sub-key of the registry key
	at hand

	@precond IsOpen = true
			 HasSubKey(Name) = true

	@throws RegistryIOException
			RegistryKeyNotFoundException
			RegistryAccessDeniedException
*/
RegistryKey RegistryKeyImplWin9x::OpenSubKey(const std::wstring& Name, bool Writeable)
{
	RegistryKey regkey(new RegistryKeyImplWin9x(m_hSubKey, Name));
	regkey->Open(Writeable);
	return regkey;
}

//-----------------------------------------------------
/** Creates a new sub-key below the key at hand

	@precond IsOpen = true
			 IsWriteable = true

	@throws  RegistryIOException
			 RegistryWriteAccessDenyException
*/

RegistryKey RegistryKeyImplWin9x::CreateSubKey(const std::wstring& Name)
{
	OSL_PRECOND(IsOpen(), "Registry key is not open");
	OSL_PRECOND(IsWriteable(), "Registry key is readonly");

	HKEY hRoot = IsRootKey() ? m_hRootKey : m_hSubKey;

	HKEY hKey;

	LONG rc = RegCreateKeyExA(
		hRoot,
		UnicodeToAnsiString(Name).c_str(),
		0,
		0,
		REG_OPTION_NON_VOLATILE,
		KEY_READ | KEY_WRITE,
		0,
		&hKey,
		0);

	if (ERROR_INVALID_HANDLE == rc)
		throw RegistryIOException(rc);
	else if (ERROR_ACCESS_DENIED == rc)
		throw RegistryAccessDeniedException(rc);
	else if (ERROR_SUCCESS != rc)
		throw RegistryException(rc);

	return RegistryKey(new RegistryKeyImplWin9x(hRoot, hKey, Name));
}

//-----------------------------------------------------
/** Deletes a sub-key below the key at hand, the
	key must not have sub-keys

	@precond IsOpen = true
			 IsWriteable = true

	@throws  RegistryIOException
			 RegistryWriteAccessDenyException
*/
void RegistryKeyImplWin9x::DeleteSubKey(const std::wstring& Name)
{
	OSL_PRECOND(IsOpen(), "Registry key is not open");
	OSL_PRECOND(IsWriteable(), "Registry key is readonly");
	OSL_PRECOND(HasSubKey(Name), "Subkey not found");

	RegistryKey SubKey = OpenSubKey(Name);

	size_t nSubKeyCount = SubKey->GetSubKeyCount();

	OSL_PRECOND(0 == nSubKeyCount, "Subkey is not empty");

	if (nSubKeyCount)
		throw RegistryInvalidOperationException(ERROR_NOT_SUPPORTED);

	LONG rc = RegDeleteKeyA(m_hSubKey, UnicodeToAnsiString(Name).c_str());

	if (ERROR_INVALID_HANDLE == rc)
		throw RegistryIOException(rc);
	else if (ERROR_ACCESS_DENIED == rc)
		throw RegistryAccessDeniedException(rc);
	else if (ERROR_SUCCESS != rc)
		throw RegistryException(rc);		
}

//-----------------------------------------------------
/** Deletes a sub-key below the key at hand with all
	its sub-keys

	@precond IsOpen = true
			 IsWriteable = true;

	@throws  RegistryIOException
			 RegistryWriteAccessDenyException
*/
void RegistryKeyImplWin9x::DeleteSubKeyTree(const std::wstring& Name)
{
	LONG rc = RegDeleteKeyA(m_hSubKey, UnicodeToAnsiString(Name).c_str());

	if (ERROR_INVALID_HANDLE == rc)
		throw RegistryIOException(rc);
	else if (ERROR_ACCESS_DENIED == rc)
		throw RegistryAccessDeniedException(rc);
	else if (ERROR_SUCCESS != rc)
		throw RegistryException(rc);
}

//-----------------------------------------------------
/** Delete the specified value

		@precond IsOpen = true
				 IsWriteable = true
				 HasValue(Name) = true

		@throws	RegistryIOException
				RegistryWriteAccessDeniedException
				RegistryValueNotFoundException
*/
void RegistryKeyImplWin9x::DeleteValue(const std::wstring& Name)
{
	OSL_PRECOND(IsOpen(), "Registry key is not open");
	OSL_PRECOND(HasValue(Name), "Sub value not found");
	OSL_PRECOND(IsWriteable(), "Registry key is readonly");

	LONG rc = RegDeleteValueA(
		m_hSubKey,
		UnicodeToAnsiString(Name).c_str());

	if (ERROR_INVALID_HANDLE == rc)
		throw RegistryIOException(rc);
	else if (ERROR_ACCESS_DENIED == rc)
		throw RegistryNoWriteAccessException(rc);
	else if (ERROR_FILE_NOT_FOUND == rc)
		throw RegistryValueNotFoundException(rc);
	else if (ERROR_SUCCESS != rc)
		throw RegistryException(rc);
}

//-----------------------------------------------------
/** Set the specified registry value

	@precond IsOpen = true
			 IsWriteable = true

	@throws  RegistryIOException
			 RegistryWriteAccessDenyException
*/
void RegistryKeyImplWin9x::SetValue(const RegistryValue& Value)
{
	OSL_PRECOND(IsOpen(), "Registry key is not open");
	OSL_PRECOND(IsWriteable(),"Registry key is readonly");

	LONG rc = ERROR_SUCCESS;

	if (REG_SZ == Value->GetType())
	{
		std::string AnsiStr = Value->GetDataAsAnsiString();

		rc = RegSetValueExA(
			m_hSubKey,
			UnicodeToAnsiString(Value->GetName()).c_str(),
			0,
			Value->GetType(),
			reinterpret_cast<const unsigned char*>(AnsiStr.c_str()),
			(AnsiStr.length() + 1));
	}
	else
	{
		rc = RegSetValueExA(
			m_hSubKey,
			UnicodeToAnsiString(Value->GetName()).c_str(),
			0,
			Value->GetType(),
			reinterpret_cast<const unsigned char*>(Value->GetDataBuffer()),
			Value->GetDataSize());
	}

	if (ERROR_INVALID_HANDLE == rc)
		throw RegistryIOException(rc);
	else if (ERROR_ACCESS_DENIED == rc)
		throw RegistryAccessDeniedException(rc);
	else if (ERROR_SUCCESS != rc)
		throw RegistryException(rc);
}

