/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: strmmac.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 14:34:28 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include <mac_start.h>
#include "Memory.h"
#include "Files.h"
#include "Strings.h"
#include <errors.h>
#include <string.h>
#include <script.h>
#include <mac_end.h>

#include <debug.hxx>
#include <string.hxx>
#include <fsys.hxx>
#include <stream.hxx>
#include <fastfsys.hxx>

// -----------------------------------------------------------------------

// --------------
// - StreamData -
// --------------

class StreamData
{
public:
    String  aFilename;
    long    nLogicalPos;
    short   aRefNum;
    BOOL    bDirty;
    BOOL    bFiller;

            StreamData();
};

StreamData::StreamData()
{
    bDirty      = FALSE;
    bFiller     = FALSE;
    aRefNum     = 0;
    nLogicalPos = 0;
}

// -----------------------------------------------------------------------

#define STANDARD_CREATOR    ((long)'SDsv')
#define STANDARD_TYPE       ((long)'DATA')

ULONG GetSvError( OSErr nMacError )
{
    static struct { OSErr mac; ULONG sv; } errArr[] =
    {
        { noErr,            SVSTREAM_OK },
        { posErr,           SVSTREAM_SEEK_ERROR },
        { fLckdErr,         SVSTREAM_LOCKING_VIOLATION },
        { vLckdErr,         SVSTREAM_LOCKING_VIOLATION },
        { wPrErr,           SVSTREAM_LOCKING_VIOLATION },
        { paramErr,         SVSTREAM_INVALID_PARAMETER },
        { rfNumErr,         SVSTREAM_INVALID_HANDLE },
        { afpAccessDenied,  SVSTREAM_ACCESS_DENIED },
        { dskFulErr,        SVSTREAM_DISK_FULL },
        { dirFulErr,        SVSTREAM_DISK_FULL },

        { afpDenyConflict,  SVSTREAM_ACCESS_DENIED },
        { afpRangeOverlap,  SVSTREAM_ACCESS_DENIED },

        { fnfErr,           SVSTREAM_FILE_NOT_FOUND },
        { nsvErr,           SVSTREAM_PATH_NOT_FOUND },
        { bdNamErr,         SVSTREAM_FILE_NOT_FOUND },
        { tmfoErr,          SVSTREAM_TOO_MANY_OPEN_FILES },
        { permErr,          SVSTREAM_LOCKING_VIOLATION },
        { dirNFErr,         SVSTREAM_PATH_NOT_FOUND },
        { afpObjectTypeErr, SVSTREAM_INVALID_PARAMETER },

        /* hier neue Fehlercodes einfuegen */

        { (OSErr)0xFFFF, SVSTREAM_GENERALERROR }
    };

    ULONG nRetVal = SVSTREAM_GENERALERROR;    // Standardfehler
    int i=0;
    do
    {
        if ( errArr[i].mac == nMacError )
        {
            nRetVal = errArr[i].sv;
            break;
        }
        i++;
    }
    while( errArr[i].mac != (OSErr)0xFFFF );
    return nRetVal;
}

/*************************************************************************
|*
|*    SvFileStream::SvFileStream()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 13.07.94
|*    Letzte Aenderung  OV 13.07.94
|*
*************************************************************************/

SvFileStream::SvFileStream( const String& rFileName,StreamMode nOpenMode )
{
    bIsOpen             = FALSE;
    nLockCounter        = 0;
    bIsWritable         = FALSE;
    pInstanceData       = new StreamData;
    SetBufferSize( 1024 );
    Open( rFileName, nOpenMode );
}

/*************************************************************************
|*
|*    SvFileStream::SvFileStream()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 22.11.94
|*    Letzte Aenderung  OV 22.11.94
|*
*************************************************************************/

SvFileStream::SvFileStream()
{
    bIsOpen             = FALSE;
    nLockCounter        = 0;
    bIsWritable         = FALSE;
    pInstanceData       = new StreamData;
    SetBufferSize( 1024 );
}

/*************************************************************************
|*
|*    SvFileStream::~SvFileStream()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 22.11.94
|*    Letzte Aenderung  OV 22.11.94
|*
*************************************************************************/

SvFileStream::~SvFileStream()
{
    Close();
    if ( pInstanceData )
        delete pInstanceData;
}

/*************************************************************************
|*
|*    SvFileStream::GetFileHandle()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 22.11.94
|*    Letzte Aenderung  OV 22.11.94
|*
*************************************************************************/

ULONG SvFileStream::GetFileHandle() const
{
    return (ULONG)pInstanceData->aRefNum;
}

/*************************************************************************
|*
|*    SvFileStream::IsA()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 14.06.94
|*    Letzte Aenderung  OV 14.06.94
|*
*************************************************************************/

USHORT SvFileStream::IsA() const
{
    return ID_FILESTREAM;
}

/*************************************************************************
|*
|*    SvFileStream::GetData()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 13.07.94
|*    Letzte Aenderung  OV 13.07.94
|*
*************************************************************************/

ULONG SvFileStream::GetData( char* pData, ULONG nSize )
{
#ifdef DBG_UTIL
    String aTraceStr( "SvFileStream::GetData(): " );
    aTraceStr += nSize;
    aTraceStr += " Bytes from ";
    aTraceStr += aFilename;
    DBG_TRACE( aTraceStr );
#endif

    if ( !IsOpen() )
        return 0L;

    OSErr nResult;
    long  nRealEof;

    // Wenn noch nicht geschen, dann jetzt richtig auf EOF positionieren
    nResult = GetEOF(pInstanceData->aRefNum, &nRealEof);
    if (nRealEof <= pInstanceData->nLogicalPos)
        return 0;   // EOF ist keine Fehler !

    if (nResult == noErr)
        nResult = FSRead( pInstanceData->aRefNum, (long*)&nSize, (MAC_Ptr)pData );

    // Logische Position mitfuehren
    pInstanceData->nLogicalPos += nSize;

    if ( nResult != noErr && nResult != eofErr)
    {
        // bei EOF kann trotzden noch was gelesen worden sein ...
        SetError(::GetSvError(nResult) );
        nSize = 0;
    }
    return nSize;
}

/*************************************************************************
|*
|*    SvFileStream::PutData()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 13.07.94
|*    Letzte Aenderung  OV 13.07.94
|*
*************************************************************************/

ULONG SvFileStream::PutData( const char* pData, ULONG nSize )
{
#ifdef DBG_UTIL
    String aTraceStr( "SvFileStrean::PutData: " );
    aTraceStr += nSize;
    aTraceStr += " Bytes to ";
    aTraceStr += aFilename;
    DBG_TRACE( aTraceStr );
#endif

    OSErr nResult;
    long  nRealEof;

    if ( !IsOpen() )
        return 0L;

    // Wenn noch nicht geschehen, dann jetzt richtig auf EOF positionieren
    nResult = GetEOF(pInstanceData->aRefNum, &nRealEof);
    if (nRealEof < pInstanceData->nLogicalPos)
    {
        // Datei jetzt (erst) verlaengern !
        nResult = SetEOF(pInstanceData->aRefNum, pInstanceData->nLogicalPos);
        if (nResult == noErr)
            nResult= SetFPos( pInstanceData->aRefNum,fsFromStart, pInstanceData->nLogicalPos);
    }

    nResult = FSWrite( pInstanceData->aRefNum, (long*)&nSize, (MAC_Ptr)pData );
    pInstanceData->nLogicalPos += nSize;

    if ( nResult )
        SetError(::GetSvError(nResult) );
    else
        pInstanceData->bDirty = TRUE;
    return nSize;
}

/*************************************************************************
|*
|*    SvFileStream::SeekPos()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 13.07.94
|*    Letzte Aenderung  OV 13.07.94
|*
*************************************************************************/

ULONG SvFileStream::SeekPos( ULONG nPos )
{
    ULONG nNewPos = 0L;
    if ( IsOpen() )
    {
        OSErr nResult = noErr;

        if ( nPos != STREAM_SEEK_TO_END )
        {
            long nRealEof;

            nResult = GetEOF(pInstanceData->aRefNum, &nRealEof);
            // Nur Positionieren wenn nich hinter EOF. Beim Schreiben wird
            // dann eventuell richtig positioniert ...

            if (nResult == noErr)
            {
                nNewPos = nPos;
                if (nPos <= nRealEof)
                    nResult= SetFPos( pInstanceData->aRefNum,fsFromStart,(long)nPos);
            }

        }
        else
        {
            nResult= SetFPos( pInstanceData->aRefNum,fsFromLEOF, 0L );
            if (nResult == noErr)
                nResult = GetFPos( pInstanceData->aRefNum, (long*)&nNewPos );
        }

        if ( nResult != noErr && nResult != eofErr)
            SetError(::GetSvError(nResult) );

        // Logische Position richtig setzten ...
        pInstanceData->nLogicalPos = (long) nNewPos;
    }
    else
        SetError( SVSTREAM_GENERALERROR );
    return nNewPos;
}

/*************************************************************************
|*
|*    SvFileStream::FlushData()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 13.07.94
|*    Letzte Aenderung  OV 13.07.94
|*
*************************************************************************/

void SvFileStream::FlushData()
{
    if ( IsOpen() && pInstanceData->bDirty )
    {
        OSErr nResult;
        ParamBlockRec aParam;

        memset( (void*)&aParam, 0, sizeof( aParam ) );
        // keine Callback-Fkt, da synchrone Ausfuehrung
        aParam.ioParam.ioCompletion = 0;
        aParam.ioParam.ioRefNum = pInstanceData->aRefNum;

        nResult = PBFlushFileSync( (ParmBlkPtr)&aParam );
        if ( nResult )
            SetError(::GetSvError(nResult) );
    }
}

/*************************************************************************
|*
|*    SvFileStream::LockRange()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 13.07.94
|*    Letzte Aenderung  OV 13.07.94
|*
*************************************************************************/

BOOL SvFileStream::LockRange( ULONG nByteOffset, ULONG nBytes )
{
    BOOL bRetVal = FALSE;
    if ( IsOpen() )
    {
        OSErr nResult;
        ParamBlockRec aParam;

        memset( (void*)&aParam, 0, sizeof( aParam ) );
        // keine Callback-Fkt, da synchrone Ausfuehrung
        aParam.ioParam.ioRefNum = pInstanceData->aRefNum;
        aParam.ioParam.ioReqCount = (long)nBytes;
        aParam.ioParam.ioPosMode = fsFromStart;
        aParam.ioParam.ioPosOffset = (long)nByteOffset;

        nResult = PBLockRangeSync( &aParam );
        if ( nResult )
            SetError(::GetSvError(nResult) );
        else
            bRetVal = TRUE;
    }
    return bRetVal;
}

/*************************************************************************
|*
|*    SvFileStream::UnlockRange()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 13.07.94
|*    Letzte Aenderung  OV 13.07.94
|*
*************************************************************************/

BOOL SvFileStream::UnlockRange( ULONG nByteOffset, ULONG nBytes )
{
    BOOL bRetVal = FALSE;
    if ( IsOpen() )
    {
        OSErr nResult;
        ParamBlockRec aParam;

        memset( (void*)&aParam, 0, sizeof( aParam ) );
        // keine Callback-Fkt, da synchrone Ausfuehrung
        aParam.ioParam.ioRefNum = pInstanceData->aRefNum;
        aParam.ioParam.ioReqCount = (long)nBytes;
        aParam.ioParam.ioPosMode = fsFromStart;
        aParam.ioParam.ioPosOffset = (long)nByteOffset;

        nResult = PBUnlockRangeSync( &aParam );
        if ( nResult )
            SetError(::GetSvError(nResult) );
        else
            bRetVal = TRUE;
    }
    return bRetVal;
}


/*************************************************************************
|*
|*    SvFileStream::LockFile()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 13.07.94
|*    Letzte Aenderung  OV 13.07.94
|*
*************************************************************************/

BOOL SvFileStream::LockFile()
{
    BOOL bRetVal = FALSE;
    if ( !IsOpen() )
        return FALSE;

    if ( !nLockCounter )
    {
        FSSpec  aSpec;
        OSErr   nResult;
        nResult=FSMakeFSSpec(0,0,pInstanceData->aFilename.GetPascalStr(),&aSpec);
        if ( !nResult )
        {
            nResult = FSpSetFLock( &aSpec );
            if ( !nResult )
            {
                nLockCounter = 1;
                bRetVal = TRUE;
            }
        }

        if ( nResult )
            SetError(::GetSvError(nResult));
    }
    else
    {
        nLockCounter++;
        bRetVal = TRUE;
    }
    return bRetVal;
}

/*************************************************************************
|*
|*    SvFileStream::UnlockFile()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 13.07.94
|*    Letzte Aenderung  OV 13.07.94
|*
*************************************************************************/

BOOL SvFileStream::UnlockFile()
{
    BOOL bRetVal = FALSE;
    if ( !IsOpen() )
        return FALSE;

    if ( nLockCounter > 0)
    {
        if ( nLockCounter == 1)
        {
            FSSpec  aSpec;
            OSErr   nResult;
            nResult=FSMakeFSSpec(0,0,pInstanceData->aFilename.GetPascalStr(),&aSpec);
            if ( !nResult )
            {
                nResult = FSpRstFLock( &aSpec );
                if ( !nResult )
                {
                    nLockCounter = 0;
                    bRetVal = TRUE;
                }
            }

            if ( nResult )
                SetError(::GetSvError(nResult));
        }
        else
        {
            nLockCounter--;
            bRetVal = TRUE;
        }
    }
    return bRetVal;
}

/*************************************************************************
|*
|*    SvFileStream::Open()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 13.07.94
|*    Letzte Aenderung  OV 13.07.94
|*
*************************************************************************/

void SvFileStream::Open(const String& rFilename, StreamMode nOpenMode)
{
	String aParsedFilename;

	if	( Folder::IsAvailable() && (rFilename.Search("{") < 9) )
	{
		String		aVirtualPart;
		String		aRealPart;
		String		aVirtualPath;
		ItemIDPath	aVirtualURL;
		ULONG		nDivider = 0;

		String		aVirtualString(rFilename);

		for (int x=aVirtualString.Len(); x>0; x--)
		{
			if (aVirtualString.Copy(x,1).Compare("}")==COMPARE_EQUAL)
			{
				nDivider = x;
				break;
			}
		}

		aVirtualPart = aVirtualString.Copy(0,nDivider+1);
		aRealPart = aVirtualString.Copy(nDivider+2);

		aVirtualURL  = aVirtualPart;
		aVirtualPath = aVirtualURL.GetHostNotationPath();

		DirEntry aTempDirEntry(aVirtualPath);

		aTempDirEntry += aRealPart;

		aParsedFilename = aTempDirEntry.GetFull();
	}
	else
	{
		aParsedFilename = rFilename;
	}

    Close();
    SvStream::ClearBuffer();

//    !!! NoOp: Ansonsten ToAbs() verwendern
//    !!! DirEntry aDirEntry( rFilename );
//    !!! aFilename = aDirEntry.GetFull();
    aFilename = aParsedFilename ;
	FSysRedirector::DoRedirect( aFilename );
    pInstanceData->aFilename = rFilename;

    eStreamMode = nOpenMode;
    eStreamMode &= ~STREAM_TRUNC; // beim ReOpen nicht cutten

#ifdef DBG_UTIL
    String aTraceStr( "SvFileStream::Open(): " );
    aTraceStr += aFilename;
    DBG_TRACE( aTraceStr );
#endif

    SignedByte      aPermission;
    USHORT          aNewHFSDenyMode;
    FSSpec          aSpec;
    OSErr           nResult;
    HParamBlockRec  ahParam;
    ParamBlockRec   aParam;

    BOOL bReadOnly = !(nOpenMode & STREAM_WRITE);

    if ( nOpenMode & STREAM_SHARE_DENYALL )
        nOpenMode |= ( STREAM_SHARE_DENYREAD | STREAM_SHARE_DENYWRITE );

    // Permission fuer alte HFS-Calls
    // STREAM_SHARE_DENYREAD muss nicht extra behandelt werden,
    // ein normales fsRdPerm schtzt gene ffnen mit fsWrPerm
    if (nOpenMode & STREAM_SHARE_DENYWRITE )
    {
        aPermission = fsRdWrPerm;
        bReadOnly = FALSE;
    }
    else
    {
        if ( bReadOnly )
		{
            aPermission = fsRdPerm;
			nOpenMode |= STREAM_NOCREATE;
        }
		else
            aPermission = fsWrPerm;
    }

    // Deny/RdWr mode fuer neue HFS-Calls
    aNewHFSDenyMode = 0;
    if ( nOpenMode & STREAM_SHARE_DENYWRITE )
        aNewHFSDenyMode |= (USHORT)32;  // Bit 5
    if ( nOpenMode & STREAM_SHARE_DENYREAD )
        aNewHFSDenyMode |= (USHORT)16;  // Bit 4
    if ( nOpenMode & STREAM_WRITE )
        aNewHFSDenyMode |= (USHORT)2;   // Bit 1
    if ( nOpenMode & STREAM_READ )
        aNewHFSDenyMode |= (USHORT)1;   // Bit 0

    // Filespec erzeugen
    nResult = FSMakeFSSpec( 0, 0,aFilename.GetPascalStr(), &aSpec );

    // Wenn die Datei nicht existiert und nicht das Flag STREAM_NOCREATE gesetzt
	// ist, dann wird die Datei erzeugt.
	if ( ( nResult == fnfErr ) && ( !( nOpenMode & STREAM_NOCREATE ) ) )
	{
        nResult=FSpCreate( &aSpec,STANDARD_CREATOR,STANDARD_TYPE,smSystemScript );
	}

    if ( nResult != noErr )
    {
        SetError(::GetSvError(nResult));
        return;
    }

    // Datei oeffnen

    // erstmal den neuen Call ausprobieren
    memset( (void*)&ahParam,0,sizeof(HParamBlockRec));
    ahParam.accessParam.ioNamePtr  = (MAC_StringPtr) aFilename.GetPascalStr();
    ahParam.accessParam.ioDenyModes = aNewHFSDenyMode;
    nResult = PBHOpenDenySync( &ahParam );
    pInstanceData->aRefNum = ahParam.ioParam.ioRefNum;
    if ( nResult == paramErr )
        // alter Call
        nResult = FSpOpenDF( &aSpec, aPermission, &pInstanceData->aRefNum );

    if ( nResult && !bReadOnly )
    {
        bReadOnly = TRUE;
        aPermission = fsRdPerm;
        nResult = FSpOpenDF( &aSpec, aPermission, &pInstanceData->aRefNum );
    }

    if ( !nResult )
    {
        bIsOpen = TRUE;
        if ( !bReadOnly )
        {
            MAC_Ptr nNewPos;
            memset( (void*)&aParam,0,sizeof(aParam));
            aParam.ioParam.ioRefNum = pInstanceData->aRefNum;
            aParam.ioParam.ioMisc = 0;
            if (!(nOpenMode & STREAM_TRUNC) )
                // wenn kein Truncate, dann das EOF
                // auf die Dateigroesse setzen. Hiermit wird
                // geprueft, ob die Datei beschreibbar ist
                nResult = PBGetEOFSync(&aParam );
			//KH: gewollter Nebeneffekt: STREAM_TRUNC fuehrt zu nNewPos=0
			nNewPos = aParam.ioParam.ioMisc;
            if( !nResult )
            {
                memset( (void*)&aParam,0,sizeof(aParam));
                aParam.ioParam.ioRefNum = pInstanceData->aRefNum;
                aParam.ioParam.ioMisc = nNewPos;
                nResult = PBSetEOFSync( &aParam );
                if ( !nResult )
                    bIsWritable = TRUE;
            }
        }
    }
    else
        SetError(::GetSvError(nResult));
}

/*************************************************************************
|*
|*    SvFileStream::ReOpen()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 15.06.94
|*    Letzte Aenderung  OV 15.06.94
|*
*************************************************************************/

void SvFileStream::ReOpen()
{
    if ( !bIsOpen && aFilename.Len() )
        Open( aFilename, eStreamMode );
}

/*************************************************************************
|*
|*    SvFileStream::Close()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 13.07.94
|*    Letzte Aenderung  OV 13.07.94
|*
*************************************************************************/

void SvFileStream::Close()
{
    if ( IsOpen() )
    {
#ifdef DBG_UTIL
        String aTraceStr( "SvFileStream::Close(): " );
        aTraceStr += aFilename;
        DBG_TRACE( aTraceStr );
#endif

        if ( nLockCounter )
        {
            nLockCounter = 1;
            UnlockFile();
        }
        Flush();   // !!! WICHTIG !!!

        OSErr           nErr;
        HParamBlockRec  aInfoRec;

        // FSClose setzt das Datum um, das wollen wir aber nicht immer !
        if (!pInstanceData->bDirty)
        {
            aInfoRec.fileParam.ioCompletion = NULL;
            aInfoRec.fileParam.ioNamePtr    =
                (MAC_StringPtr) pInstanceData->aFilename.GetPascalStr();
            aInfoRec.fileParam.ioVRefNum    = 0;
            aInfoRec.fileParam.ioFDirIndex  = 0;
            aInfoRec.fileParam.ioDirID      = 0;
            nErr = PBHGetFInfoSync(&aInfoRec);
        }

        FSClose( pInstanceData->aRefNum );

        if (!pInstanceData->bDirty && !nErr)
            nErr = PBHSetFInfoSync( &aInfoRec);

        // Ueberlegen: Volume flushen ?
    }

    pInstanceData->bDirty   = FALSE;
    pInstanceData->aRefNum  = 0;
    pInstanceData->nLogicalPos  = 0;
    // eStreamMode              = 0;
    bIsOpen                 = FALSE;
    nLockCounter            = 0;
    bIsWritable             = FALSE;
    SvStream::ClearError();
}


/*************************************************************************
|*
|*    SvFileStream::ResetError()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 13.07.94
|*    Letzte Aenderung  OV 13.07.94
|*
*************************************************************************/

void SvFileStream::ResetError()
{
    SvStream::ClearError();
}

/*************************************************************************
|*
|*    SvFileStream::SetSize()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 13.07.94
|*    Letzte Aenderung  OV 13.07.94
|*
*************************************************************************/

void SvFileStream::SetSize( ULONG nSize )
{
	OSErr nErr = SetEOF(pInstanceData->aRefNum, nSize);
    if ( nErr )
        SetError(::GetSvError(nErr) );
}

/*************************************************************************
|*
|*    SvSharedMemoryStream::SvSharedMemoryStream()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    CL 05.05.95
|*    Letzte Aenderung  CL 05.05.95
|*
*************************************************************************/

SvSharedMemoryStream::SvSharedMemoryStream( void* pBuffer, ULONG nBufSize,
                                            StreamMode eMode ) :
            SvMemoryStream( NULL, nBufSize, eMode )
{
    aHandle 	= pBuffer;
    bOwnsData   = FALSE;
    nResize     = 4096;
    DBG_ASSERT(!pBuffer ||  (HandleZone( (Handle) pBuffer ) == SystemZone()),
            "SvSharedMemoryStream::SvSharedMemoryStream : Handle not in Systemheap");

    // Uebergebene nSize wird effektiv ignoriert !
    nSize       = GetHandleSize((Handle) pBuffer);
}

/*************************************************************************
|*
|*    SvSharedMemoryStream::SvSharedMemoryStream()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    CL 05.05.95
|*    Letzte Aenderung  DV 03.05.96
|*
*************************************************************************/

SvSharedMemoryStream::SvSharedMemoryStream( ULONG nInitSize,
                                            ULONG nResizeOffset) :
            SvMemoryStream( (void*)NULL )
{
	aHandle 	= 0;
    bIsWritable = TRUE;
    bOwnsData   = TRUE;
    nEndOfData  = 0L;
    nResize     = nResizeOffset;
	nPos		= 0;
	pBuf		= 0;

    if( nResize != 0 && nResize < 16 )
        nResize = 16;

	if( nInitSize && !AllocateMemory( nInitSize ) )
	{
        SetError( SVSTREAM_OUTOFMEMORY );
		nSize = 0;
	}
	else
		nSize = nInitSize;

	SetBufferSize( 64 );
}

/*************************************************************************
|*
|*    SvSharedMemoryStream::~SvSharedMemoryStream()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    CL 05.05.95
|*    Letzte Aenderung  DV 03.05.96
|*
*************************************************************************/
SvSharedMemoryStream::~SvSharedMemoryStream()
{
    if ( bOwnsData )
    {
        FreeMemory();
        pBuf = NULL;    // Damit SvMemoryStream nicht auf dumme Gedanken kommt
    }
    else
        Flush();
}

/*************************************************************************
|*
|*    SvSharedMemoryStream::IsA()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    CL 05.05.95
|*    Letzte Aenderung  CL 05.05.95
|*
*************************************************************************/

USHORT SvSharedMemoryStream::IsA() const
{
    return (USHORT)ID_SHAREDMEMORYSTREAM;
}

/*************************************************************************
|*
|*    SvSharedMemoryStream::AllocateMemory()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    KH 13.06.95
|*    Letzte Aenderung  KH 16.06.95
|*
*************************************************************************/

BOOL SvSharedMemoryStream::AllocateMemory( ULONG nNewSize )
{
    aHandle = (char*) NewHandleSys(nNewSize);
    return (aHandle != 0);
}

/*************************************************************************
|*
|*    SvSharedMemoryStream::ReAllocateMemory()
|*
|*    Beschreibung      STREAM.SDW      (Bozo-Algorithmus)
|*    Ersterstellung    CL 05.05.95
|*    Letzte Aenderung  CL 05.05.95
|*
*************************************************************************/

BOOL SvSharedMemoryStream::ReAllocateMemory( long nDiff )
{
    ULONG nNewSize  = nSize + nDiff;

    SetHandleSize((Handle) aHandle,nNewSize);
    nSize = nNewSize;
    if ( MemError() != noErr )
        return FALSE;
	//
	// HIER KLAFFT NOCH EINE GROSSE LUECKE (beim Verkleinern!)
	//
	return TRUE;
}

/*************************************************************************
|*
|*    SvSharedMemoryStream::FreeMemory()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    CL 05.05.95
|*    Letzte Aenderung  DV 03.05.96
|*
*************************************************************************/

void SvSharedMemoryStream::FreeMemory()
{
	DBG_ASSERT(( pBuf == NULL ), "SvSharedMemoryStream::FreeMemory : pBuf not NULL");

    if( aHandle )
    {
        DBG_ASSERT( (HandleZone( (Handle) aHandle ) == SystemZone() ),
                  "SvSharedMemoryStream::FreeMemory : Handle not in Systemheap");

        DisposeHandle( (Handle) aHandle );
        aHandle = 0;
    }
}

/*************************************************************************
|*
|*    SvSharedMemoryStream::SwitchBuffer()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    KH 16.06.95
|*    Letzte Aenderung  KH 16.06.95
|*
*************************************************************************/

void* SvSharedMemoryStream::SwitchBuffer( ULONG nInitSize, ULONG nResizeOffset )
{
    Flush();
    if ( !bOwnsData )
        return NULL;
	Seek( STREAM_SEEK_TO_BEGIN );
    Handle hRetVal 	= (Handle) aHandle;
    aHandle 		= 0;
    nEndOfData  	= 0L;
	nPos			= 0L;

    if ( nResizeOffset != 0 && nResizeOffset < 16 )
        nResizeOffset = 16;
    nResize = nResizeOffset;

	ResetError();

	if( nInitSize && !AllocateMemory( nInitSize ) )
	{
        SetError( SVSTREAM_OUTOFMEMORY );
		nSize = 0L;
	}
	else
		nSize = nInitSize;

	SetBufferSize( 64 );
    return (void*) hRetVal;
}

/*************************************************************************
|*
|*    SvSharedMemoryStream::SetBuffer()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    KH 16.06.95
|*    Letzte Aenderung  KH 16.06.95
|*
*************************************************************************/

void* SvSharedMemoryStream::SetBuffer( void* pParamBuf, ULONG nSize,
                                       BOOL bOwnsTheData, ULONG nEOF )
{
    DBG_ASSERT(!pParamBuf || (HandleZone( (Handle) pParamBuf ) == SystemZone()),
               "SvSharedMemoryStream::SetBuffer : Handle not in Systemheap");

    Flush();
    Handle hRetVal = (Handle) aHandle;
    if ( bOwnsData && pParamBuf != aHandle)
    {
        DisposeHandle((Handle) aHandle);
        hRetVal = NULL;
    }
    bOwnsData = bOwnsTheData;
    aHandle = (char*) pParamBuf;
    if( nEOF > nSize )
        nEOF = nSize;
    nEndOfData  = nEOF;
    nResize     = 4096;
    ResetError();
    if ( ! pParamBuf )
        SetError( SVSTREAM_OUTOFMEMORY );
    SetBufferSize( 64 );

    return (void*) hRetVal;
}

/*************************************************************************
|*
|*    SvSharedMemoryStream::GetData()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    KH 08.09.94
|*    Letzte Aenderung  KH 08.09.94
|*
*************************************************************************/

ULONG SvSharedMemoryStream::GetData( char* pData, ULONG nCount )
{
    ULONG nMaxCount = nEndOfData-nPos;
    if ( nCount > nMaxCount )
        nCount = nMaxCount;
    memcpy( pData, (* ((Handle)aHandle))+nPos, (size_t)nCount );
    nPos += nCount;
    return nCount;
}

/*************************************************************************
|*
|*    SvSharedMemoryStream::PutData()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    KH 08.09.94
|*    Letzte Aenderung  KH 08.09.94
|*
*************************************************************************/

ULONG SvSharedMemoryStream::PutData( const char* pData, ULONG nCount )
{
    if ( GetError() )
        return 0L;

    ULONG nMaxCount = nSize-nPos;

    // auf Ueberlauf testen
    if ( nCount > nMaxCount )
    {
        if ( nResize == 0 )
        {
            nCount = nMaxCount;
            SetError( SVSTREAM_OUTOFMEMORY );
        }
        else
        {
            if ( (nCount-nMaxCount) < nResize )
            {
                // fehlender Speicher ist kleiner als Resize-Offset,
                // deshalb um Resize-Offset vergroessern
                if ( ! ReAllocateMemory( nResize ) )
                {
                    nCount = 0;
                    SetError( SVSTREAM_WRITE_ERROR );
                }
            }
            else
            {
                // fehlender Speicher ist groesser als Resize-Offset
                // deshalb um Differenz+ResizeOffset vergroessern
                if ( ! ReAllocateMemory( nCount-nMaxCount+nResize) )
                {
                    nCount = 0;
                    SetError( SVSTREAM_WRITE_ERROR );
                }
            }
        }
    }
    memcpy( (* ((Handle)aHandle))+nPos, pData, (size_t)nCount);

    nPos += nCount;
    if ( nPos > nEndOfData )
        nEndOfData = nPos;
    return nCount;
}

/*************************************************************************
|*
|*    SvSharedMemoryStream::GetData()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 08.06.94
|*    Letzte Aenderung  KH 03.08.95
|*
*************************************************************************/

const char* SvSharedMemoryStream::GetData()
{
    Flush();
    return *((Handle)aHandle);
}

/*************************************************************************
|*
|*    SvSharedMemoryStream::operator const char*()
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 08.06.94
|*    Letzte Aenderung  KH 03.08.95
|*
*************************************************************************/

SvSharedMemoryStream::operator const char*()
{
    Flush();
    return *((Handle)aHandle);
}

/*************************************************************************
|*
|*    SvSharedMemoryStream::SetHandle
|*
|*    Beschreibung      STREAM.SDW
|*    Ersterstellung    OV 08.06.94
|*    Letzte Aenderung  KH 03.08.95
|*
*************************************************************************/

void* SvSharedMemoryStream::SetHandle( void* aHandle, ULONG nNewSize,
                                       BOOL bOwnsTheData, ULONG nEOF)
{
    return SetBuffer(aHandle,nNewSize,bOwnsTheData, nEOF);
}
