/*
 * Copyright © 2013 Jonas Ådahl
 * Copyright © 2013-2015 Red Hat, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#ifndef LIBINPUT_H
#define LIBINPUT_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdlib.h>
#include <stdint.h>
#include <stdarg.h>
#include <libudev.h>

#define LIBINPUT_ATTRIBUTE_PRINTF(_format, _args) \
	__attribute__ ((format (printf, _format, _args)))
#define LIBINPUT_ATTRIBUTE_DEPRECATED __attribute__ ((deprecated))

/**
 * @ingroup base
 * @struct libinput
 *
 * A handle for accessing libinput. This struct is refcounted, use
 * libinput_ref() and libinput_unref().
 */
struct libinput;

/**
 * @ingroup device
 * @struct libinput_device
 *
 * A base handle for accessing libinput devices. This struct is
 * refcounted, use libinput_device_ref() and libinput_device_unref().
 */
struct libinput_device;

/**
 * @ingroup device
 * @struct libinput_device_group
 *
 * A base handle for accessing libinput device groups. This struct is
 * refcounted, use libinput_device_group_ref() and
 * libinput_device_group_unref().
 */
struct libinput_device_group;

/**
 * @ingroup seat
 * @struct libinput_seat
 *
 * The base handle for accessing libinput seats. This struct is
 * refcounted, use libinput_seat_ref() and libinput_seat_unref().
 */
struct libinput_seat;

/**
 * @ingroup device
 * @struct libinput_tablet_tool
 *
 * An object representing a tool being used by a device with the @ref
 * LIBINPUT_DEVICE_CAP_TABLET_TOOL capability.
 *
 * Tablet events generated by such a device are bound to a specific tool
 * rather than coming from the device directly. Depending on the hardware it
 * is possible to track the same physical tool across multiple
 * struct libinput_device devices.
 * See libinput_tablet_tool_get_serial() for more details.
 *
 * This struct is refcounted, use libinput_tablet_tool_ref() and
 * libinput_tablet_tool_unref().
 *
 * @since 1.2
 */
struct libinput_tablet_tool;

/**
 * @ingroup event
 * @struct libinput_event
 *
 * The base event type. Use libinput_event_get_pointer_event() or similar to
 * get the actual event type.
 *
 * @warning Unlike other structs events are considered transient and
 * <b>not</b> refcounted.
 */
struct libinput_event;

/**
 * @ingroup event
 * @struct libinput_event_device_notify
 *
 * An event notifying the caller of a device being added or removed.
 */
struct libinput_event_device_notify;

/**
 * @ingroup event_keyboard
 * @struct libinput_event_keyboard
 *
 * A keyboard event representing a key press/release.
 */
struct libinput_event_keyboard;

/**
 * @ingroup event_pointer
 * @struct libinput_event_pointer
 *
 * A pointer event representing relative or absolute pointer movement,
 * a button press/release or scroll axis events.
 */
struct libinput_event_pointer;

/**
 * @ingroup event_touch
 * @struct libinput_event_touch
 *
 * Touch event representing a touch down, move or up, as well as a touch
 * cancel and touch frame events. Valid event types for this event are @ref
 * LIBINPUT_EVENT_TOUCH_DOWN, @ref LIBINPUT_EVENT_TOUCH_MOTION, @ref
 * LIBINPUT_EVENT_TOUCH_UP, @ref LIBINPUT_EVENT_TOUCH_CANCEL and @ref
 * LIBINPUT_EVENT_TOUCH_FRAME.
 */
struct libinput_event_touch;

/**
 * @ingroup event_tablet
 * @struct libinput_event_tablet_tool
 *
 * Tablet tool event representing an axis update, button press, or tool
 * update. Valid event types for this event are @ref
 * LIBINPUT_EVENT_TABLET_TOOL_AXIS, @ref
 * LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY and @ref
 * LIBINPUT_EVENT_TABLET_TOOL_BUTTON.
 *
 * @since 1.2
 */
struct libinput_event_tablet_tool;

/**
 * @ingroup event_tablet_pad
 * @struct libinput_event_tablet_pad
 *
 * Tablet pad event representing a button press, or ring/strip update on
 * the tablet pad itself. Valid event types for this event are @ref
 * LIBINPUT_EVENT_TABLET_PAD_BUTTON, @ref LIBINPUT_EVENT_TABLET_PAD_RING and
 * @ref LIBINPUT_EVENT_TABLET_PAD_STRIP.
 *
 * @since 1.3
 */
struct libinput_event_tablet_pad;

/**
 * @ingroup base
 *
 * Log priority for internal logging messages.
 */
enum libinput_log_priority {
	LIBINPUT_LOG_PRIORITY_DEBUG = 10,
	LIBINPUT_LOG_PRIORITY_INFO = 20,
	LIBINPUT_LOG_PRIORITY_ERROR = 30,
};

/**
 * @ingroup device
 *
 * Capabilities on a device. A device may have one or more capabilities
 * at a time, capabilities remain static for the lifetime of the device.
 */
enum libinput_device_capability {
	LIBINPUT_DEVICE_CAP_KEYBOARD = 0,
	LIBINPUT_DEVICE_CAP_POINTER = 1,
	LIBINPUT_DEVICE_CAP_TOUCH = 2,
	LIBINPUT_DEVICE_CAP_TABLET_TOOL = 3,
	LIBINPUT_DEVICE_CAP_TABLET_PAD = 4,
	LIBINPUT_DEVICE_CAP_GESTURE = 5,
	LIBINPUT_DEVICE_CAP_SWITCH = 6,
};

/**
 * @ingroup device
 *
 * Logical state of a key. Note that the logical state may not represent
 * the physical state of the key.
 */
enum libinput_key_state {
	LIBINPUT_KEY_STATE_RELEASED = 0,
	LIBINPUT_KEY_STATE_PRESSED = 1
};

/**
 * @ingroup device
 *
 * Mask reflecting LEDs on a device.
 */
enum libinput_led {
	LIBINPUT_LED_NUM_LOCK = (1 << 0),
	LIBINPUT_LED_CAPS_LOCK = (1 << 1),
	LIBINPUT_LED_SCROLL_LOCK = (1 << 2)
};

/**
 * @ingroup device
 *
 * Logical state of a physical button. Note that the logical state may not
 * represent the physical state of the button.
 */
enum libinput_button_state {
	LIBINPUT_BUTTON_STATE_RELEASED = 0,
	LIBINPUT_BUTTON_STATE_PRESSED = 1
};

/**
 * @ingroup device
 *
 * Axes on a device with the capability @ref LIBINPUT_DEVICE_CAP_POINTER
 * that are not x or y coordinates.
 *
 * The two scroll axes @ref LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL and
 * @ref LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL are engaged separately,
 * depending on the device. libinput provides some scroll direction locking
 * but it is up to the caller to determine which axis is needed and
 * appropriate in the current interaction
 */
enum libinput_pointer_axis {
	LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL = 0,
	LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL = 1,
};

/**
 * @ingroup device
 *
 * The source for a libinput_pointer_axis event. See
 * libinput_event_pointer_get_axis_source() for details.
 *
 * @note Pointer axis sources are deprecated, the source is now encoded in
 * the event types
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_WHEEL,
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_FINGER, and
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS.
 */
enum libinput_pointer_axis_source {
	/**
	 * The event is caused by the rotation of a wheel.
	 */
	LIBINPUT_POINTER_AXIS_SOURCE_WHEEL = 1,
	/**
	 * The event is caused by the movement of one or more fingers on a
	 * device.
	 */
	LIBINPUT_POINTER_AXIS_SOURCE_FINGER,
	/**
	 * The event is caused by the motion of some device.
	 */
	LIBINPUT_POINTER_AXIS_SOURCE_CONTINUOUS,
	/**
	 * The event is caused by the tilting of a mouse wheel rather than
	 * its rotation. This method is commonly used on mice without
	 * separate horizontal scroll wheels.
	 *
	 * @deprecated This axis source is deprecated as of libinput 1.16.
	 * It was never used by any device before libinput 1.16. All wheel
	 * tilt devices use @ref LIBINPUT_POINTER_AXIS_SOURCE_WHEEL instead.
	 */
	LIBINPUT_POINTER_AXIS_SOURCE_WHEEL_TILT,
};

/**
 * @ingroup event_tablet_pad
 *
 * The source for a @ref LIBINPUT_EVENT_TABLET_PAD_RING event. See
 * libinput_event_tablet_pad_get_ring_source() for details.
 *
 * @since 1.3
 */
enum libinput_tablet_pad_ring_axis_source {
	LIBINPUT_TABLET_PAD_RING_SOURCE_UNKNOWN = 1,
	/**
	 * The event is caused by the movement of one or more fingers on
	 * the ring.
	 */
	LIBINPUT_TABLET_PAD_RING_SOURCE_FINGER,
};

/**
 * @ingroup event_tablet_pad
 *
 * The source for a @ref LIBINPUT_EVENT_TABLET_PAD_STRIP event. See
 * libinput_event_tablet_pad_get_strip_source() for details.
 *
 * @since 1.3
 */
enum libinput_tablet_pad_strip_axis_source {
	LIBINPUT_TABLET_PAD_STRIP_SOURCE_UNKNOWN = 1,
	/**
	 * The event is caused by the movement of one or more fingers on
	 * the strip.
	 */
	LIBINPUT_TABLET_PAD_STRIP_SOURCE_FINGER,
};

/**
 * @ingroup device
 *
 * Available tool types for a device with the @ref
 * LIBINPUT_DEVICE_CAP_TABLET_TOOL capability. The tool type defines the default
 * usage of the tool as advertised by the manufacturer. Multiple different
 * physical tools may share the same tool type, e.g. a Wacom Classic Pen,
 * Wacom Pro Pen and a Wacom Grip Pen are all of type @ref
 * LIBINPUT_TABLET_TOOL_TYPE_PEN.
 * Use libinput_tablet_tool_get_tool_id() to get a specific model where applicable.
 *
 * Note that on some device, the eraser tool is on the tail end of a pen
 * device. On other devices, e.g. MS Surface 3, the eraser is the pen tip
 * while a button is held down.
 *
 * @note The @ref libinput_tablet_tool_type can only describe the default physical
 * type of the device. For devices with adjustable physical properties
 * the tool type remains the same, i.e. putting a Wacom stroke nib into a
 * classic pen leaves the tool type as @ref LIBINPUT_TABLET_TOOL_TYPE_PEN.
 *
 * @since 1.2
 */
enum libinput_tablet_tool_type {
	LIBINPUT_TABLET_TOOL_TYPE_PEN = 1,	/**< A generic pen */
	LIBINPUT_TABLET_TOOL_TYPE_ERASER,	/**< Eraser */
	LIBINPUT_TABLET_TOOL_TYPE_BRUSH,	/**< A paintbrush-like tool */
	LIBINPUT_TABLET_TOOL_TYPE_PENCIL,	/**< Physical drawing tool, e.g.
					             Wacom Inking Pen */
	LIBINPUT_TABLET_TOOL_TYPE_AIRBRUSH,	/**< An airbrush-like tool */
	LIBINPUT_TABLET_TOOL_TYPE_MOUSE,	/**< A mouse bound to the tablet */
	LIBINPUT_TABLET_TOOL_TYPE_LENS,		/**< A mouse tool with a lens */
	LIBINPUT_TABLET_TOOL_TYPE_TOTEM,	/**< A rotary device with
						     positional and rotation
						     data */
};

/**
 * @ingroup device
 *
 * The state of proximity for a tool on a device. The device must have the @ref
 * LIBINPUT_DEVICE_CAP_TABLET_TOOL capability.
 *
 * The proximity of a tool is a binary state signalling whether the tool is
 * within a detectable distance of the tablet device. A tool that is out of
 * proximity cannot generate events.
 *
 * On some hardware a tool goes out of proximity when it ceases to touch the
 * surface. On other hardware, the tool is still detectable within a short
 * distance (a few cm) off the surface.
 *
 * @since 1.2
 */
enum libinput_tablet_tool_proximity_state {
	LIBINPUT_TABLET_TOOL_PROXIMITY_STATE_OUT = 0,
	LIBINPUT_TABLET_TOOL_PROXIMITY_STATE_IN = 1,
};

/**
 * @ingroup device
 *
 * The tip contact state for a tool on a device. The device must have
 * the @ref LIBINPUT_DEVICE_CAP_TABLET_TOOL capability.
 *
 * The tip contact state of a tool is a binary state signalling whether the tool is
 * touching the surface of the tablet device.
 *
 * @since 1.2
 */
enum libinput_tablet_tool_tip_state {
	LIBINPUT_TABLET_TOOL_TIP_UP = 0,
	LIBINPUT_TABLET_TOOL_TIP_DOWN = 1,
};

/**
 * @defgroup tablet_pad_modes Tablet pad modes
 *
 * Handling the virtual mode groups of buttons, strips and rings on tablet
 * pad devices. See the libinput documentation for more details.
 */

/**
 * @ingroup tablet_pad_modes
 * @struct libinput_tablet_pad_mode_group
 *
 * A mode on a tablet pad is a virtual grouping of functionality, usually
 * based on some visual feedback like LEDs on the pad. The set of buttons,
 * rings and strips that share the same mode are a "mode group". Whenever
 * the mode changes, all buttons, rings and strips within this mode group
 * are affected.
 *
 * Most tablets only have a single mode group, some tablets provide multiple
 * mode groups through independent banks of LEDs (e.g. the Wacom Cintiq
 * 24HD). libinput guarantees that at least one mode group is always
 * available.
 *
 * This struct is refcounted, use libinput_tablet_pad_mode_group_ref() and
 * libinput_tablet_pad_mode_group_unref().
 *
 * @since 1.4
 */
struct libinput_tablet_pad_mode_group;

/**
 * @ingroup tablet_pad_modes
 *
 * Most devices only provide a single mode group, however devices such as
 * the Wacom Cintiq 22HD provide two mode groups. If multiple mode groups
 * are available, a caller should use
 * libinput_tablet_pad_mode_group_has_button(),
 * libinput_tablet_pad_mode_group_has_ring() and
 * libinput_tablet_pad_mode_group_has_strip() to associate each button,
 * ring and strip with the correct mode group.
 *
 * @return the number of mode groups available on this device
 *
 * @since 1.4
 */
int
libinput_device_tablet_pad_get_num_mode_groups(struct libinput_device *device);

/**
 * @ingroup tablet_pad_modes
 *
 * The returned mode group is not refcounted and may become invalid after
 * the next call to libinput. Use libinput_tablet_pad_mode_group_ref() and
 * libinput_tablet_pad_mode_group_unref() to continue using the handle
 * outside of the immediate scope.
 *
 * While at least one reference is kept by the caller, the returned mode
 * group will be identical for each subsequent call of this function with
 * the same index and that same struct is returned from
 * libinput_event_tablet_pad_get_mode_group(), provided the event was
 * generated by this mode group.
 *
 * @param device A device with the @ref LIBINPUT_DEVICE_CAP_TABLET_PAD
 * capability
 * @param index A mode group index
 * @return the mode group with the given index or NULL if an invalid index
 * is given.
 *
 * @since 1.4
 */
struct libinput_tablet_pad_mode_group*
libinput_device_tablet_pad_get_mode_group(struct libinput_device *device,
					  unsigned int index);

/**
 * @ingroup tablet_pad_modes
 *
 * The returned number is the same index as passed to
 * libinput_device_tablet_pad_get_mode_group(). For tablets with only one
 * mode this number is always 0.
 *
 * @param group A previously obtained mode group
 * @return the numeric index this mode group represents, starting at 0
 *
 * @since 1.4
 */
unsigned int
libinput_tablet_pad_mode_group_get_index(struct libinput_tablet_pad_mode_group *group);

/**
 * @ingroup tablet_pad_modes
 *
 * Query the mode group for the number of available modes. The number of
 * modes is usually decided by the number of physical LEDs available on the
 * device. Different mode groups may have a different number of modes. Use
 * libinput_tablet_pad_mode_group_get_mode() to get the currently active
 * mode.
 *
 * libinput guarantees that at least one mode is available. A device without
 * mode switching capability has a single mode group and a single mode.
 *
 * @param group A previously obtained mode group
 * @return the number of modes available in this mode group
 *
 * @since 1.4
 */
unsigned int
libinput_tablet_pad_mode_group_get_num_modes(struct libinput_tablet_pad_mode_group *group);

/**
 * @ingroup tablet_pad_modes
 *
 * Return the current mode this mode group is in. Note that the returned
 * mode is the mode valid as of completing the last libinput_dispatch().
 * The returned mode may thus be different than the mode returned by
 * libinput_event_tablet_pad_get_mode().
 *
 * For example, if the mode was toggled three times between the call to
 * libinput_dispatch(), this function returns the third mode but the events
 * in the event queue will return the modes 1, 2 and 3, respectively.
 *
 * @param group A previously obtained mode group
 * @return the numeric index of the current mode in this group, starting at 0
 *
 * @see libinput_event_tablet_pad_get_mode
 *
 * @since 1.4
 */
unsigned int
libinput_tablet_pad_mode_group_get_mode(struct libinput_tablet_pad_mode_group *group);

/**
 * @ingroup tablet_pad_modes
 *
 * Devices without mode switching capabilities return true for every button.
 *
 * @param group A previously obtained mode group
 * @param button A button index, starting at 0
 * @return true if the given button index is part of this mode group or
 * false otherwise
 *
 * @since 1.4
 */
int
libinput_tablet_pad_mode_group_has_button(struct libinput_tablet_pad_mode_group *group,
					  unsigned int button);

/**
 * @ingroup tablet_pad_modes
 *
 * Devices without mode switching capabilities return true for every ring.
 *
 * @param group A previously obtained mode group
 * @param ring A ring index, starting at 0
 * @return true if the given ring index is part of this mode group or
 * false otherwise
 *
 * @since 1.4
 */
int
libinput_tablet_pad_mode_group_has_ring(struct libinput_tablet_pad_mode_group *group,
					  unsigned int ring);

/**
 * @ingroup tablet_pad_modes
 *
 * Devices without mode switching capabilities return true for every strip.
 *
 * @param group A previously obtained mode group
 * @param strip A strip index, starting at 0
 * @return true if the given strip index is part of this mode group or
 * false otherwise
 *
 * @since 1.4
 */
int
libinput_tablet_pad_mode_group_has_strip(struct libinput_tablet_pad_mode_group *group,
					  unsigned int strip);

/**
 * @ingroup tablet_pad_modes
 *
 * The toggle button in a mode group is the button assigned to cycle to or
 * directly assign a new mode when pressed. Not all devices have a toggle
 * button and some devices may have more than one toggle button. For
 * example, the Wacom Cintiq 24HD has six toggle buttons in two groups, each
 * directly selecting one of the three modes per group.
 *
 * Devices without mode switching capabilities return false for every button.
 *
 * @param group A previously obtained mode group
 * @param button A button index, starting at 0
 * @retval non-zero if the button is a mode toggle button for this group, or
 * zero otherwise
 *
 * @since 1.4
 */
int
libinput_tablet_pad_mode_group_button_is_toggle(struct libinput_tablet_pad_mode_group *group,
						unsigned int button);

/**
 * @ingroup tablet_pad_modes
 *
 * Increase the refcount of the mode group. A mode group will be
 * freed whenever the refcount reaches 0.
 *
 * @param group A previously obtained mode group
 * @return The passed mode group
 *
 * @since 1.4
 */
struct libinput_tablet_pad_mode_group *
libinput_tablet_pad_mode_group_ref(
			struct libinput_tablet_pad_mode_group *group);

/**
 * @ingroup tablet_pad_modes
 *
 * Decrease the refcount of the mode group. A mode group will be
 * freed whenever the refcount reaches 0.
 *
 * @param group A previously obtained mode group
 * @return NULL if the group was destroyed, otherwise the passed mode group
 *
 * @since 1.4
 */
struct libinput_tablet_pad_mode_group *
libinput_tablet_pad_mode_group_unref(
			struct libinput_tablet_pad_mode_group *group);

/**
 * @ingroup tablet_pad_modes
 *
 * Set caller-specific data associated with this mode group. libinput does
 * not manage, look at, or modify this data. The caller must ensure the
 * data is valid.
 *
 * @param group A previously obtained mode group
 * @param user_data Caller-specific data pointer
 * @see libinput_tablet_pad_mode_group_get_user_data
 *
 * @since 1.4
 */
void
libinput_tablet_pad_mode_group_set_user_data(
			struct libinput_tablet_pad_mode_group *group,
			void *user_data);

/**
 * @ingroup tablet_pad_modes
 *
 * Get the caller-specific data associated with this mode group, if any.
 *
 * @param group A previously obtained mode group
 * @return Caller-specific data pointer or NULL if none was set
 * @see libinput_tablet_pad_mode_group_set_user_data
 *
 * @since 1.4
 */
void *
libinput_tablet_pad_mode_group_get_user_data(
			struct libinput_tablet_pad_mode_group *group);

/**
 * @ingroup device
 *
 * The state of a switch. The default state of a switch is @ref
 * LIBINPUT_SWITCH_STATE_OFF and no event is sent to confirm a switch in the
 * off position. If a switch is logically on during initialization, libinput
 * sends an event of type @ref LIBINPUT_EVENT_SWITCH_TOGGLE with a state
 * @ref LIBINPUT_SWITCH_STATE_ON.
 *
 * @since 1.7
 */
enum libinput_switch_state {
	LIBINPUT_SWITCH_STATE_OFF = 0,
	LIBINPUT_SWITCH_STATE_ON = 1,
};

/**
 * @ingroup device
 *
 * The type of a switch.
 *
 * @since 1.7
 */
enum libinput_switch {
	/**
	 * The laptop lid was closed when the switch state is @ref
	 * LIBINPUT_SWITCH_STATE_ON, or was opened when it is @ref
	 * LIBINPUT_SWITCH_STATE_OFF.
	 */
	LIBINPUT_SWITCH_LID = 1,

	/**
	 * This switch indicates whether the device is in normal laptop mode
	 * or behaves like a tablet-like device where the primary
	 * interaction is usually a touch screen. When in tablet mode, the
	 * keyboard and touchpad are usually inaccessible.
	 *
	 * If the switch is in state @ref LIBINPUT_SWITCH_STATE_OFF, the
	 * device is in laptop mode. If the switch is in state @ref
	 * LIBINPUT_SWITCH_STATE_ON, the device is in tablet mode and the
	 * keyboard or touchpad may not be  accessible.
	 *
	 * It is up to the caller to identify which devices are inaccessible
	 * in tablet mode.
	 */
	LIBINPUT_SWITCH_TABLET_MODE,
};

/**
 * @ingroup event_switch
 * @struct libinput_event_switch
 *
 * A switch event representing a changed state in a switch.
 *
 * @since 1.7
 */
struct libinput_event_switch;

/**
 * @ingroup base
 *
 * Event type for events returned by libinput_get_event().
 */
enum libinput_event_type {
	/**
	 * This is not a real event type, and is only used to tell the user that
	 * no new event is available in the queue. See
	 * libinput_next_event_type().
	 */
	LIBINPUT_EVENT_NONE = 0,

	/**
	 * Signals that a device has been added to the context. The device will
	 * not be read until the next time the user calls libinput_dispatch()
	 * and data is available.
	 *
	 * This allows setting up initial device configuration before any events
	 * are created.
	 */
	LIBINPUT_EVENT_DEVICE_ADDED,

	/**
	 * Signals that a device has been removed. No more events from the
	 * associated device will be in the queue or be queued after this event.
	 */
	LIBINPUT_EVENT_DEVICE_REMOVED,

	LIBINPUT_EVENT_KEYBOARD_KEY = 300,

	LIBINPUT_EVENT_POINTER_MOTION = 400,
	LIBINPUT_EVENT_POINTER_MOTION_ABSOLUTE,
	LIBINPUT_EVENT_POINTER_BUTTON,
	/**
	 * A scroll event from various sources.
	 *
	 * This event is deprecated as of libinput 1.19. Use
	 * @ref LIBINPUT_EVENT_POINTER_SCROLL_WHEEL,
	 * @ref LIBINPUT_EVENT_POINTER_SCROLL_FINGER, and
	 * @ref LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS instead.
	 *
	 * Use libinput_event_pointer_get_axis_source() to determine the
	 * source of a scroll event. For libinput versions 1.19 and later,
	 * the source is encoded in the event type.
	 *
	 * This event is sent **in addition** to events of type
	 * @ref LIBINPUT_EVENT_POINTER_SCROLL_WHEEL,
	 * @ref LIBINPUT_EVENT_POINTER_SCROLL_FINGER, and
	 * @ref LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS.
	 * Do not mix and match, either use the old event or the new events.
	 * libinput makes no guarantee about the relation between
	 * @ref LIBINPUT_EVENT_POINTER_AXIS and the new event types
	 * @ref LIBINPUT_EVENT_POINTER_SCROLL_WHEEL,
	 * @ref LIBINPUT_EVENT_POINTER_SCROLL_FINGER, and
	 * @ref LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS. You may receive
	 * multiple zero, one or more new events per legacy event.
	 *
	 * @warning Ignore this event if you are processing
	 * @ref LIBINPUT_EVENT_POINTER_SCROLL_WHEEL,
	 * @ref LIBINPUT_EVENT_POINTER_SCROLL_FINGER, and
	 * @ref LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS.
	 */
	LIBINPUT_EVENT_POINTER_AXIS,

	/**
	 * A scroll event from a wheel. This event is sent is sent **in
	 * addition** to the @ref LIBINPUT_EVENT_POINTER_AXIS
	 * event for all events with a
	 * libinput_event_pointer_get_axis_source() of @ref
	 * LIBINPUT_POINTER_AXIS_SOURCE_WHEEL. Ignore @ref
	 * LIBINPUT_EVENT_POINTER_AXIS if you are processing this event.
	 *
	 * See the libinput documentation for details.
	 *
	 * @since 1.19
	 */
	LIBINPUT_EVENT_POINTER_SCROLL_WHEEL,

	/**
	 * A scroll event caused by the movement of one or more fingers on a
	 * device. This event is sent is sent **in addition** to the @ref
	 * LIBINPUT_EVENT_POINTER_AXIS event for all events with a
	 * libinput_event_pointer_get_axis_source() of @ref
	 * LIBINPUT_POINTER_AXIS_SOURCE_FINGER. Ignore @ref
	 * LIBINPUT_EVENT_POINTER_AXIS if you are processing this event.
	 *
	 * See the libinput documentation for details.
	 *
	 * @since 1.19
	 */
	LIBINPUT_EVENT_POINTER_SCROLL_FINGER,

	/**
	 * A scroll event from a continuous scroll source, e.g. button
	 * scrolling. This event is sent is sent **in
	 * addition** to the @ref LIBINPUT_EVENT_POINTER_AXIS
	 * event for all events with a
	 * libinput_event_pointer_get_axis_source() of @ref
	 * LIBINPUT_POINTER_AXIS_SOURCE_CONTINUOUS. Ignore @ref
	 * LIBINPUT_EVENT_POINTER_AXIS if you are processing this event.
	 *
	 * See the libinput documentation for details.
	 *
	 * @since 1.19
	 */
	LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS,

	LIBINPUT_EVENT_TOUCH_DOWN = 500,
	LIBINPUT_EVENT_TOUCH_UP,
	LIBINPUT_EVENT_TOUCH_MOTION,
	LIBINPUT_EVENT_TOUCH_CANCEL,
	/**
	 * Signals the end of a set of touchpoints at one device sample
	 * time. This event has no coordinate information attached.
	 */
	LIBINPUT_EVENT_TOUCH_FRAME,

	/**
	 * One or more axes have changed state on a device with the @ref
	 * LIBINPUT_DEVICE_CAP_TABLET_TOOL capability. This event is only sent
	 * when the tool is in proximity, see @ref
	 * LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY for details.
	 *
	 * The proximity event contains the initial state of the axis as the
	 * tool comes into proximity. An event of type @ref
	 * LIBINPUT_EVENT_TABLET_TOOL_AXIS is only sent when an axis value
	 * changes from this initial state. It is possible for a tool to
	 * enter and leave proximity without sending an event of type @ref
	 * LIBINPUT_EVENT_TABLET_TOOL_AXIS.
	 *
	 * An event of type @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS is sent
	 * when the tip state does not change. See the documentation for
	 * @ref LIBINPUT_EVENT_TABLET_TOOL_TIP for more details.
	 *
	 * @since 1.2
	 */
	LIBINPUT_EVENT_TABLET_TOOL_AXIS = 600,
	/**
	 * Signals that a tool has come in or out of proximity of a device with
	 * the @ref LIBINPUT_DEVICE_CAP_TABLET_TOOL capability.
	 *
	 * Proximity events contain each of the current values for each axis,
	 * and these values may be extracted from them in the same way they are
	 * with @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS events.
	 *
	 * Some tools may always be in proximity. For these tools, events of
	 * type @ref LIBINPUT_TABLET_TOOL_PROXIMITY_STATE_IN are sent only once after @ref
	 * LIBINPUT_EVENT_DEVICE_ADDED, and events of type @ref
	 * LIBINPUT_TABLET_TOOL_PROXIMITY_STATE_OUT are sent only once before @ref
	 * LIBINPUT_EVENT_DEVICE_REMOVED.
	 *
	 * If the tool that comes into proximity supports x/y coordinates,
	 * libinput guarantees that both x and y are set in the proximity
	 * event.
	 *
	 * When a tool goes out of proximity, the value of every axis should be
	 * assumed to have an undefined state and any buttons that are currently held
	 * down on the stylus are marked as released. Button release events for
	 * each button that was held down on the stylus are sent before the
	 * proximity out event.
	 *
	 * @since 1.2
	 */
	LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY,
	/**
	 * Signals that a tool has come in contact with the surface of a
	 * device with the @ref LIBINPUT_DEVICE_CAP_TABLET_TOOL capability.
	 *
	 * On devices without distance proximity detection, the @ref
	 * LIBINPUT_EVENT_TABLET_TOOL_TIP is sent immediately after @ref
	 * LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY for the tip down event, and
	 * immediately before for the tip up event.
	 *
	 * The decision when a tip touches the surface is device-dependent
	 * and may be derived from pressure data or other means. If the tip
	 * state is changed by axes changing state, the
	 * @ref LIBINPUT_EVENT_TABLET_TOOL_TIP event includes the changed
	 * axes and no additional axis event is sent for this state change.
	 * In other words, a caller must look at both @ref
	 * LIBINPUT_EVENT_TABLET_TOOL_AXIS and @ref
	 * LIBINPUT_EVENT_TABLET_TOOL_TIP events to know the current state
	 * of the axes.
	 *
	 * If a button state change occurs at the same time as a tip state
	 * change, the order of events is device-dependent.
	 *
	 * @since 1.2
	 */
	LIBINPUT_EVENT_TABLET_TOOL_TIP,
	/**
	 * Signals that a tool has changed a logical button state on a
	 * device with the @ref LIBINPUT_DEVICE_CAP_TABLET_TOOL capability.
	 *
	 * Button state changes occur on their own and do not include axis
	 * state changes. If button and axis state changes occur within the
	 * same logical hardware event, the order of the @ref
	 * LIBINPUT_EVENT_TABLET_TOOL_BUTTON and @ref
	 * LIBINPUT_EVENT_TABLET_TOOL_AXIS event is device-specific.
	 *
	 * This event is not to be confused with the button events emitted
	 * by the tablet pad. See @ref LIBINPUT_EVENT_TABLET_PAD_BUTTON.
	 *
	 * @see LIBINPUT_EVENT_TABLET_PAD_BUTTON
	 *
	 * @since 1.2
	 */
	LIBINPUT_EVENT_TABLET_TOOL_BUTTON,

	/**
	 * A button pressed on a device with the @ref
	 * LIBINPUT_DEVICE_CAP_TABLET_PAD capability.
	 *
	 * A button differs from @ref LIBINPUT_EVENT_TABLET_PAD_KEY in that
	 * buttons are sequentially indexed from 0 and do not carry any
	 * other information.  Keys have a specific functionality assigned
	 * to them. The key code thus carries a semantic meaning, a button
	 * number does not.
	 *
	 * This event is not to be confused with the button events emitted
	 * by tools on a tablet (@ref LIBINPUT_EVENT_TABLET_TOOL_BUTTON).
	 *
	 * @since 1.3
	 */
	LIBINPUT_EVENT_TABLET_PAD_BUTTON = 700,
	/**
	 * A status change on a tablet ring with the @ref
	 * LIBINPUT_DEVICE_CAP_TABLET_PAD capability.
	 *
	 * @since 1.3
	 */
	LIBINPUT_EVENT_TABLET_PAD_RING,

	/**
	 * A status change on a strip on a device with the @ref
	 * LIBINPUT_DEVICE_CAP_TABLET_PAD capability.
	 *
	 * @since 1.3
	 */
	LIBINPUT_EVENT_TABLET_PAD_STRIP,

	/**
	 * A key pressed on a device with the @ref
	 * LIBINPUT_DEVICE_CAP_TABLET_PAD capability.
	 *
	 * A key differs from @ref LIBINPUT_EVENT_TABLET_PAD_BUTTON in that
	 * keys have a specific functionality assigned to them (buttons are
	 * sequentially ordered). The key code thus carries a semantic
	 * meaning, a button number does not.
	 *
	 * @since 1.15
	 */
	LIBINPUT_EVENT_TABLET_PAD_KEY,

	LIBINPUT_EVENT_GESTURE_SWIPE_BEGIN = 800,
	LIBINPUT_EVENT_GESTURE_SWIPE_UPDATE,
	LIBINPUT_EVENT_GESTURE_SWIPE_END,
	LIBINPUT_EVENT_GESTURE_PINCH_BEGIN,
	LIBINPUT_EVENT_GESTURE_PINCH_UPDATE,
	LIBINPUT_EVENT_GESTURE_PINCH_END,
	/**
	 * @since 1.19
	 */
	LIBINPUT_EVENT_GESTURE_HOLD_BEGIN,
	LIBINPUT_EVENT_GESTURE_HOLD_END,

	/**
	 * @since 1.7
	 */
	LIBINPUT_EVENT_SWITCH_TOGGLE = 900,
};

/**
 * @defgroup event Accessing and destruction of events
 */

/**
 * @ingroup event
 *
 * Destroy the event, freeing all associated resources. Resources obtained
 * from this event must be considered invalid after this call.
 *
 * @warning Unlike other structs events are considered transient and
 * <b>not</b> refcounted. Calling libinput_event_destroy() <b>will</b>
 * destroy the event.
 *
 * @param event An event retrieved by libinput_get_event().
 */
void
libinput_event_destroy(struct libinput_event *event);

/**
 * @ingroup event
 *
 * Get the type of the event.
 *
 * @param event An event retrieved by libinput_get_event().
 */
enum libinput_event_type
libinput_event_get_type(struct libinput_event *event);

/**
 * @ingroup event
 *
 * Get the libinput context from the event.
 *
 * @param event The libinput event
 * @return The libinput context for this event.
 */
struct libinput *
libinput_event_get_context(struct libinput_event *event);

/**
 * @ingroup event
 *
 * Return the device associated with this event. For device added/removed
 * events this is the device added or removed. For all other device events,
 * this is the device that generated the event.
 *
 * This device is not refcounted and its lifetime is that of the event. Use
 * libinput_device_ref() before using the device outside of this scope.
 *
 * @return The device associated with this event
 */

struct libinput_device *
libinput_event_get_device(struct libinput_event *event);

/**
 * @ingroup event
 *
 * Return the pointer event that is this input event. If the event type does
 * not match the pointer event types, this function returns NULL.
 *
 * The inverse of this function is libinput_event_pointer_get_base_event().
 *
 * @return A pointer event, or NULL for other events
 */
struct libinput_event_pointer *
libinput_event_get_pointer_event(struct libinput_event *event);

/**
 * @ingroup event
 *
 * Return the keyboard event that is this input event. If the event type does
 * not match the keyboard event types, this function returns NULL.
 *
 * The inverse of this function is libinput_event_keyboard_get_base_event().
 *
 * @return A keyboard event, or NULL for other events
 */
struct libinput_event_keyboard *
libinput_event_get_keyboard_event(struct libinput_event *event);

/**
 * @ingroup event
 *
 * Return the touch event that is this input event. If the event type does
 * not match the touch event types, this function returns NULL.
 *
 * The inverse of this function is libinput_event_touch_get_base_event().
 *
 * @return A touch event, or NULL for other events
 */
struct libinput_event_touch *
libinput_event_get_touch_event(struct libinput_event *event);

/**
 * @ingroup event
 *
 * Return the gesture event that is this input event. If the event type does
 * not match the gesture event types, this function returns NULL.
 *
 * A gesture's lifetime has three distinct stages: begin, update and end, each
 * with their own event types. Begin is sent when the fingers are first set
 * down or libinput decides that the gesture begins. For @ref
 * LIBINPUT_EVENT_GESTURE_PINCH_BEGIN this sets the initial scale. Any
 * events changing properties of the gesture are sent as update events. On
 * termination of the gesture, an end event is sent.
 *
 * The inverse of this function is libinput_event_gesture_get_base_event().
 *
 * @return A gesture event, or NULL for other events
 */
struct libinput_event_gesture *
libinput_event_get_gesture_event(struct libinput_event *event);

/**
 * @ingroup event
 *
 * Return the tablet tool event that is this input event. If the event type
 * does not match the tablet tool event types, this function returns NULL.
 *
 * The inverse of this function is libinput_event_tablet_tool_get_base_event().
 *
 * @return A tablet tool event, or NULL for other events
 *
 * @since 1.2
 */
struct libinput_event_tablet_tool *
libinput_event_get_tablet_tool_event(struct libinput_event *event);

/**
 * @ingroup event
 *
 * Return the tablet pad event that is this input event. If the event type does not
 * match the tablet pad event types, this function returns NULL.
 *
 * The inverse of this function is libinput_event_tablet_pad_get_base_event().
 *
 * @return A tablet pad event, or NULL for other events
 */
struct libinput_event_tablet_pad *
libinput_event_get_tablet_pad_event(struct libinput_event *event);

/**
 * @ingroup event
 *
 * Return the switch event that is this input event. If the event type does
 * not match the switch event types, this function returns NULL.
 *
 * The inverse of this function is libinput_event_switch_get_base_event().
 *
 * @return A switch event, or NULL for other events
 *
 * @since 1.7
 */
struct libinput_event_switch *
libinput_event_get_switch_event(struct libinput_event *event);

/**
 * @ingroup event
 *
 * Return the device event that is this input event. If the event type does
 * not match the device event types, this function returns NULL.
 *
 * The inverse of this function is
 * libinput_event_device_notify_get_base_event().
 *
 * @return A device event, or NULL for other events
 */
struct libinput_event_device_notify *
libinput_event_get_device_notify_event(struct libinput_event *event);

/**
 * @ingroup event
 *
 * @return The generic libinput_event of this event
 */
struct libinput_event *
libinput_event_device_notify_get_base_event(struct libinput_event_device_notify *event);

/**
 * @defgroup event_keyboard Keyboard events
 *
 * Key events are generated when a key changes its logical state, usually by
 * being pressed or released.
 */

/**
 * @ingroup event_keyboard
 *
 * @note Timestamps may not always increase. See the libinput documentation
 * for more details.
 *
 * @return The event time for this event
 */
uint32_t
libinput_event_keyboard_get_time(struct libinput_event_keyboard *event);

/**
 * @ingroup event_keyboard
 *
 * @note Timestamps may not always increase. See the libinput documentation
 * for more details.
 *
 * @return The event time for this event in microseconds
 */
uint64_t
libinput_event_keyboard_get_time_usec(struct libinput_event_keyboard *event);

/**
 * @ingroup event_keyboard
 *
 * @return The keycode that triggered this key event
 */
uint32_t
libinput_event_keyboard_get_key(struct libinput_event_keyboard *event);

/**
 * @ingroup event_keyboard
 *
 * @return The state change of the key
 */
enum libinput_key_state
libinput_event_keyboard_get_key_state(struct libinput_event_keyboard *event);

/**
 * @ingroup event_keyboard
 *
 * @return The generic libinput_event of this event
 */
struct libinput_event *
libinput_event_keyboard_get_base_event(struct libinput_event_keyboard *event);

/**
 * @ingroup event_keyboard
 *
 * For the key of a @ref LIBINPUT_EVENT_KEYBOARD_KEY event, return the total number
 * of keys pressed on all devices on the associated seat after the event was
 * triggered.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_KEYBOARD_KEY. For other events, this function returns 0.
 *
 * @return The seat wide pressed key count for the key of this event
 */
uint32_t
libinput_event_keyboard_get_seat_key_count(
	struct libinput_event_keyboard *event);

/**
 * @defgroup event_pointer Pointer events
 *
 * Pointer events reflect motion, button and scroll events, as well as
 * events from other axes.
 */

/**
 * @ingroup event_pointer
 *
 * @note Timestamps may not always increase. See the libinput documentation
 * for more details.
 *
 * @return The event time for this event
 */
uint32_t
libinput_event_pointer_get_time(struct libinput_event_pointer *event);

/**
 * @ingroup event_pointer
 *
 * @note Timestamps may not always increase. See the libinput documentation
 * for more details.
 *
 * @return The event time for this event in microseconds
 */
uint64_t
libinput_event_pointer_get_time_usec(struct libinput_event_pointer *event);

/**
 * @ingroup event_pointer
 *
 * Return the delta between the last event and the current event. For pointer
 * events that are not of type @ref LIBINPUT_EVENT_POINTER_MOTION, this
 * function returns 0.
 *
 * If a device employs pointer acceleration, the delta returned by this
 * function is the accelerated delta.
 *
 * Relative motion deltas are to be interpreted as pixel movement of a
 * standardized mouse. See the libinput documentation for more details.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_MOTION.
 *
 * @return The relative x movement since the last event
 */
double
libinput_event_pointer_get_dx(struct libinput_event_pointer *event);

/**
 * @ingroup event_pointer
 *
 * Return the delta between the last event and the current event. For pointer
 * events that are not of type @ref LIBINPUT_EVENT_POINTER_MOTION, this
 * function returns 0.
 *
 * If a device employs pointer acceleration, the delta returned by this
 * function is the accelerated delta.
 *
 * Relative motion deltas are to be interpreted as pixel movement of a
 * standardized mouse. See the libinput documentation for more details.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_MOTION.
 *
 * @return The relative y movement since the last event
 */
double
libinput_event_pointer_get_dy(struct libinput_event_pointer *event);

/**
 * @ingroup event_pointer
 *
 * Return the relative delta of the unaccelerated motion vector of the
 * current event. For pointer events that are not of type @ref
 * LIBINPUT_EVENT_POINTER_MOTION, this function returns 0.
 *
 * Relative unaccelerated motion deltas are raw device coordinates.
 * Note that these coordinates are subject to the device's native
 * resolution. Touchpad coordinates represent raw device coordinates in the
 * X resolution of the touchpad. See the libinput documentation for more
 * details.
 *
 * Any rotation applied to the device also applies to unaccelerated motion
 * (see libinput_device_config_rotation_set_angle()).
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_MOTION.
 *
 * @return The unaccelerated relative x movement since the last event
 */
double
libinput_event_pointer_get_dx_unaccelerated(
	struct libinput_event_pointer *event);

/**
 * @ingroup event_pointer
 *
 * Return the relative delta of the unaccelerated motion vector of the
 * current event. For pointer events that are not of type @ref
 * LIBINPUT_EVENT_POINTER_MOTION, this function returns 0.
 *
 * Relative unaccelerated motion deltas are raw device coordinates.
 * Note that these coordinates are subject to the device's native
 * resolution. Touchpad coordinates represent raw device coordinates in the
 * X resolution of the touchpad. See the libinput documentation for more
 * details.
 *
 * Any rotation applied to the device also applies to unaccelerated motion
 * (see libinput_device_config_rotation_set_angle()).
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_MOTION.
 *
 * @return The unaccelerated relative y movement since the last event
 */
double
libinput_event_pointer_get_dy_unaccelerated(
	struct libinput_event_pointer *event);

/**
 * @ingroup event_pointer
 *
 * Return the current absolute x coordinate of the pointer event, in mm from
 * the top left corner of the device. To get the corresponding output screen
 * coordinate, use libinput_event_pointer_get_absolute_x_transformed().
 *
 * For pointer events that are not of type
 * @ref LIBINPUT_EVENT_POINTER_MOTION_ABSOLUTE, this function returns 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_MOTION_ABSOLUTE.
 *
 * @return The current absolute x coordinate
 */
double
libinput_event_pointer_get_absolute_x(struct libinput_event_pointer *event);

/**
 * @ingroup event_pointer
 *
 * Return the current absolute y coordinate of the pointer event, in mm from
 * the top left corner of the device. To get the corresponding output screen
 * coordinate, use libinput_event_pointer_get_absolute_y_transformed().
 *
 * For pointer events that are not of type
 * @ref LIBINPUT_EVENT_POINTER_MOTION_ABSOLUTE, this function returns 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_MOTION_ABSOLUTE.
 *
 * @return The current absolute y coordinate
 */
double
libinput_event_pointer_get_absolute_y(struct libinput_event_pointer *event);

/**
 * @ingroup event_pointer
 *
 * Return the current absolute x coordinate of the pointer event, transformed to
 * screen coordinates.
 *
 * For pointer events that are not of type
 * @ref LIBINPUT_EVENT_POINTER_MOTION_ABSOLUTE, the return value of this
 * function is undefined.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_MOTION_ABSOLUTE.
 *
 * @param event The libinput pointer event
 * @param width The current output screen width
 * @return The current absolute x coordinate transformed to a screen coordinate
 */
double
libinput_event_pointer_get_absolute_x_transformed(
	struct libinput_event_pointer *event,
	uint32_t width);

/**
 * @ingroup event_pointer
 *
 * Return the current absolute y coordinate of the pointer event, transformed to
 * screen coordinates.
 *
 * For pointer events that are not of type
 * @ref LIBINPUT_EVENT_POINTER_MOTION_ABSOLUTE, the return value of this function is
 * undefined.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_MOTION_ABSOLUTE.
 *
 * @param event The libinput pointer event
 * @param height The current output screen height
 * @return The current absolute y coordinate transformed to a screen coordinate
 */
double
libinput_event_pointer_get_absolute_y_transformed(
	struct libinput_event_pointer *event,
	uint32_t height);

/**
 * @ingroup event_pointer
 *
 * Return the button that triggered this event.
 * For pointer events that are not of type @ref
 * LIBINPUT_EVENT_POINTER_BUTTON, this function returns 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_BUTTON.
 *
 * @return The button triggering this event
 */
uint32_t
libinput_event_pointer_get_button(struct libinput_event_pointer *event);

/**
 * @ingroup event_pointer
 *
 * Return the button state that triggered this event.
 * For pointer events that are not of type @ref
 * LIBINPUT_EVENT_POINTER_BUTTON, this function returns 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_BUTTON.
 *
 * @return The button state triggering this event
 */
enum libinput_button_state
libinput_event_pointer_get_button_state(struct libinput_event_pointer *event);

/**
 * @ingroup event_pointer
 *
 * For the button of a @ref LIBINPUT_EVENT_POINTER_BUTTON event, return the
 * total number of buttons pressed on all devices on the associated seat
 * after the event was triggered.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_BUTTON. For other events, this function
 * returns 0.
 *
 * @return The seat wide pressed button count for the key of this event
 */
uint32_t
libinput_event_pointer_get_seat_button_count(
	struct libinput_event_pointer *event);

/**
 * @ingroup event_pointer
 *
 * Check if the event has a valid value for the given axis.
 *
 * If this function returns non-zero for an axis and
 * libinput_event_pointer_get_axis_value() returns a value of 0, the event
 * is a scroll stop event.
 *
 * For pointer events that are not of type @ref LIBINPUT_EVENT_POINTER_AXIS,
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_WHEEL, @ref
 * LIBINPUT_EVENT_POINTER_SCROLL_FINGER, or @ref
 * LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS this function returns 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_AXIS,
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_WHEEL,
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_FINGER, or
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS.
 *
 * @return Non-zero if this event contains a value for this axis
 */
int
libinput_event_pointer_has_axis(struct libinput_event_pointer *event,
				enum libinput_pointer_axis axis);

/**
 * @ingroup event_pointer
 *
 * Return the axis value of the given axis. The interpretation of the value
 * depends on the axis. For the two scrolling axes
 * @ref LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL and
 * @ref LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL, the value of the event is in
 * relative scroll units, with the positive direction being down or right,
 * respectively. For the interpretation of the value, see
 * libinput_event_pointer_get_axis_source().
 *
 * If libinput_event_pointer_has_axis() returns 0 for an axis, this function
 * returns 0 for that axis.
 *
 * For pointer events that are not of type @ref LIBINPUT_EVENT_POINTER_AXIS,
 * this function returns 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_AXIS.
 *
 * @return The axis value of this event
 *
 * @see libinput_event_pointer_get_axis_value_discrete
 */
double
libinput_event_pointer_get_axis_value(struct libinput_event_pointer *event,
				      enum libinput_pointer_axis axis);

/**
 * @ingroup event_pointer
 *
 * Return the source for a given axis event. Axis events (scroll events) can
 * be caused by a hardware item such as a scroll wheel or emulated from
 * other input sources, such as two-finger or edge scrolling on a
 * touchpad.
 *
 * If the source is @ref LIBINPUT_POINTER_AXIS_SOURCE_FINGER, libinput
 * guarantees that a scroll sequence is terminated with a scroll value of 0.
 * A caller may use this information to decide on whether kinetic scrolling
 * should be triggered on this scroll sequence.
 * The coordinate system is identical to the cursor movement, i.e. a
 * scroll value of 1 represents the equivalent relative motion of 1.
 *
 * If the source is @ref LIBINPUT_POINTER_AXIS_SOURCE_WHEEL, no terminating
 * event is guaranteed (though it may happen).
 * Scrolling is in discrete steps, the value is the angle the wheel moved
 * in degrees. The default is 15 degrees per wheel click, but some mice may
 * have differently grained wheels. It is up to the caller how to interpret
 * such different step sizes. Callers should use
 * libinput_event_pointer_get_scroll_value_v120() for a simpler API of
 * handling scroll wheel events of different step sizes.
 *
 * If the source is @ref LIBINPUT_POINTER_AXIS_SOURCE_CONTINUOUS, no
 * terminating event is guaranteed (though it may happen).
 * The coordinate system is identical to the cursor movement, i.e. a
 * scroll value of 1 represents the equivalent relative motion of 1.
 *
 * @deprecated The source @ref LIBINPUT_POINTER_AXIS_SOURCE_WHEEL_TILT is
 * deprecated as of libinput 1.16. No device has ever sent this source.
 *
 * For pointer events that are not of type @ref LIBINPUT_EVENT_POINTER_AXIS,
 * this function returns 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_AXIS.
 *
 * @note This function is superfluous as of libinput 1.19. The event
 * codes for @ref LIBINPUT_EVENT_POINTER_SCROLL_WHEEL, @ref
 * LIBINPUT_EVENT_POINTER_SCROLL_FINGER and @ref
 * LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS have the axis source encoded in
 * the event type.
 *
 * @return The source for this axis event
 */
enum libinput_pointer_axis_source
libinput_event_pointer_get_axis_source(struct libinput_event_pointer *event);

/**
 * @ingroup event_pointer
 *
 * Return the axis value in discrete steps for a given axis event. How a
 * value translates into a discrete step depends on the source.
 *
 * @note This function does not support high-resolution mouse wheels and
 * should be considered deprecated as of libinput 1.19. Callers should use
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_WHEEL and
 * libinput_event_pointer_get_scroll_value_v120() instead.
 *
 * If the event is not of type @ref LIBINPUT_EVENT_POINTER_AXIS, this
 * function returns 0.
 *
 * If the source is @ref LIBINPUT_POINTER_AXIS_SOURCE_WHEEL, the discrete
 * value correspond to the number of physical mouse wheel clicks.
 *
 * If the source is @ref LIBINPUT_POINTER_AXIS_SOURCE_CONTINUOUS or @ref
 * LIBINPUT_POINTER_AXIS_SOURCE_FINGER, the discrete value is always 0.
 *
 * @return The discrete value for the given event.
 *
 * @see libinput_event_pointer_get_axis_value
 * @see libinput_event_pointer_get_scroll_value_v120
 */
double
libinput_event_pointer_get_axis_value_discrete(struct libinput_event_pointer *event,
					       enum libinput_pointer_axis axis);

/**
 * @ingroup event_pointer
 *
 * Return the axis value of the given axis. The interpretation of the value
 * depends on the axis. For the two scrolling axes
 * @ref LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL and
 * @ref LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL, the value of the event is in
 * relative scroll units, with the positive direction being down or right,
 * respectively. If libinput_event_pointer_has_axis() returns 0 for an axis,
 * this function returns 0 for that axis.
 *
 * If the event is @ref LIBINPUT_EVENT_POINTER_SCROLL_FINGER, libinput
 * guarantees that a scroll sequence is terminated with a scroll value of 0.
 * A caller may use this information to decide on whether kinetic scrolling
 * should be triggered on this scroll sequence.
 * The coordinate system is identical to the cursor movement, i.e. a
 * scroll value of 1 represents the equivalent relative motion of 1.
 *
 * If the event is @ref LIBINPUT_EVENT_POINTER_SCROLL_WHEEL, no terminating
 * event is guaranteed (though it may happen).
 * Scrolling is in discrete steps, the value is the angle the wheel moved
 * in degrees. The default is 15 degrees per wheel click, but some mice may
 * have differently grained wheels. It is up to the caller how to interpret
 * such different step sizes. Callers should use
 * libinput_event_pointer_get_scroll_value_v120() for a simpler API of
 * handling scroll wheel events of different step sizes.
 *
 * If the event is @ref LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS, no
 * terminating event is guaranteed (though it may happen).
 * The coordinate system is identical to the cursor movement, i.e. a
 * scroll value of 1 represents the equivalent relative motion of 1.
 *
 * For pointer events that are not of type
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_WHEEL,
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_FINGER, or
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS, this function returns zero.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_WHEEL,
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_FINGER, or
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS.
 *
 * @return The axis value of this event
 *
 * @see libinput_event_pointer_get_scroll_value_v120
 *
 * @since 1.19
 */
double
libinput_event_pointer_get_scroll_value(struct libinput_event_pointer *event,
					enum libinput_pointer_axis axis);

/**
 * @ingroup event_pointer
 *
 * For events of type @ref LIBINPUT_EVENT_POINTER_SCROLL_WHEEL
 * the v120-normalized value represents the movement in logical mouse wheel
 * clicks, normalized to the -120..+120 range.
 *
 * A value that is a fraction of ±120 indicates a wheel movement less than
 * one logical click, a caller should either scroll by the respective
 * fraction of the normal scroll distance or accumulate that value until a
 * multiple of 120 is reached.
 *
 * For most callers, this is the preferred way of handling high-resolution
 * scroll events.
 *
 * The normalized v120 value does not take device-specific physical angles
 * or distances into account, i.e. a wheel with a click angle of 20 degrees
 * produces only 18 logical clicks per 360 degree rotation, a wheel with a
 * click angle of 15 degrees produces 24 logical clicks per 360 degree
 * rotation. Where the physical angle matters, use
 * libinput_event_pointer_get_axis_value() instead.
 *
 * The magic number 120 originates from the <a
 * href="http://download.microsoft.com/download/b/d/1/bd1f7ef4-7d72-419e-bc5c-9f79ad7bb66e/wheel.docx">
 * Windows Vista Mouse Wheel design document</a>.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_POINTER_SCROLL_WHEEL.
 *
 * @return A value normalized to the 0-±120 range
 *
 * @see libinput_event_pointer_get_axis_value
 * @see libinput_event_pointer_get_axis_value_discrete
 *
 * @since 1.19
 */
double
libinput_event_pointer_get_scroll_value_v120(struct libinput_event_pointer *event,
					     enum libinput_pointer_axis axis);

/**
 * @ingroup event_pointer
 *
 * @return The generic libinput_event of this event
 */
struct libinput_event *
libinput_event_pointer_get_base_event(struct libinput_event_pointer *event);

/**
 * @defgroup event_touch Touch events
 *
 * Events from absolute touch devices.
 */

/**
 * @ingroup event_touch
 *
 * @note Timestamps may not always increase. See the libinput documentation
 * for more details.
 *
 * @return The event time for this event
 */
uint32_t
libinput_event_touch_get_time(struct libinput_event_touch *event);

/**
 * @ingroup event_touch
 *
 * @note Timestamps may not always increase. See the libinput documentation
 * for more details.
 *
 * @return The event time for this event in microseconds
 */
uint64_t
libinput_event_touch_get_time_usec(struct libinput_event_touch *event);

/**
 * @ingroup event_touch
 *
 * Get the slot of this touch event. See the kernel's multitouch
 * protocol B documentation for more information.
 *
 * If the touch event has no assigned slot, for example if it is from a
 * single touch device, this function returns -1.
 *
 * For events not of type @ref LIBINPUT_EVENT_TOUCH_DOWN, @ref
 * LIBINPUT_EVENT_TOUCH_UP, @ref LIBINPUT_EVENT_TOUCH_MOTION or @ref
 * LIBINPUT_EVENT_TOUCH_CANCEL, this function returns 0.
 *
 * @note It is an application bug to call this function for events of type
 * other than @ref LIBINPUT_EVENT_TOUCH_DOWN, @ref LIBINPUT_EVENT_TOUCH_UP,
 * @ref LIBINPUT_EVENT_TOUCH_MOTION or @ref LIBINPUT_EVENT_TOUCH_CANCEL.
 *
 * @return The slot of this touch event
 */
int32_t
libinput_event_touch_get_slot(struct libinput_event_touch *event);

/**
 * @ingroup event_touch
 *
 * Get the seat slot of the touch event. A seat slot is a non-negative seat
 * wide unique identifier of an active touch point.
 *
 * Events from single touch devices will be represented as one individual
 * touch point per device.
 *
 * For events not of type @ref LIBINPUT_EVENT_TOUCH_DOWN, @ref
 * LIBINPUT_EVENT_TOUCH_UP, @ref LIBINPUT_EVENT_TOUCH_MOTION or @ref
 * LIBINPUT_EVENT_TOUCH_CANCEL, this function returns 0.
 *
 * @note It is an application bug to call this function for events of type
 * other than @ref LIBINPUT_EVENT_TOUCH_DOWN, @ref LIBINPUT_EVENT_TOUCH_UP,
 * @ref LIBINPUT_EVENT_TOUCH_MOTION or @ref LIBINPUT_EVENT_TOUCH_CANCEL.
 *
 * @return The seat slot of the touch event
 */
int32_t
libinput_event_touch_get_seat_slot(struct libinput_event_touch *event);

/**
 * @ingroup event_touch
 *
 * Return the current absolute x coordinate of the touch event, in mm from
 * the top left corner of the device. To get the corresponding output screen
 * coordinate, use libinput_event_touch_get_x_transformed().
 *
 * For events not of type @ref LIBINPUT_EVENT_TOUCH_DOWN, @ref
 * LIBINPUT_EVENT_TOUCH_MOTION, this function returns 0.
 *
 * @note It is an application bug to call this function for events of type
 * other than @ref LIBINPUT_EVENT_TOUCH_DOWN or @ref
 * LIBINPUT_EVENT_TOUCH_MOTION.
 *
 * @param event The libinput touch event
 * @return The current absolute x coordinate
 */
double
libinput_event_touch_get_x(struct libinput_event_touch *event);

/**
 * @ingroup event_touch
 *
 * Return the current absolute y coordinate of the touch event, in mm from
 * the top left corner of the device. To get the corresponding output screen
 * coordinate, use libinput_event_touch_get_y_transformed().
 *
 * For events not of type @ref LIBINPUT_EVENT_TOUCH_DOWN, @ref
 * LIBINPUT_EVENT_TOUCH_MOTION, this function returns 0.
 *
 * @note It is an application bug to call this function for events of type
 * other than @ref LIBINPUT_EVENT_TOUCH_DOWN or @ref
 * LIBINPUT_EVENT_TOUCH_MOTION.
 *
 * @param event The libinput touch event
 * @return The current absolute y coordinate
 */
double
libinput_event_touch_get_y(struct libinput_event_touch *event);

/**
 * @ingroup event_touch
 *
 * Return the current absolute x coordinate of the touch event, transformed to
 * screen coordinates.
 *
 * For events not of type @ref LIBINPUT_EVENT_TOUCH_DOWN, @ref
 * LIBINPUT_EVENT_TOUCH_MOTION, this function returns 0.
 *
 * @note It is an application bug to call this function for events of type
 * other than @ref LIBINPUT_EVENT_TOUCH_DOWN or @ref
 * LIBINPUT_EVENT_TOUCH_MOTION.
 *
 * @param event The libinput touch event
 * @param width The current output screen width
 * @return The current absolute x coordinate transformed to a screen coordinate
 */
double
libinput_event_touch_get_x_transformed(struct libinput_event_touch *event,
				       uint32_t width);

/**
 * @ingroup event_touch
 *
 * Return the current absolute y coordinate of the touch event, transformed to
 * screen coordinates.
 *
 * For events not of type @ref LIBINPUT_EVENT_TOUCH_DOWN, @ref
 * LIBINPUT_EVENT_TOUCH_MOTION, this function returns 0.
 *
 * @note It is an application bug to call this function for events of type
 * other than @ref LIBINPUT_EVENT_TOUCH_DOWN or @ref
 * LIBINPUT_EVENT_TOUCH_MOTION.
 *
 * @param event The libinput touch event
 * @param height The current output screen height
 * @return The current absolute y coordinate transformed to a screen coordinate
 */
double
libinput_event_touch_get_y_transformed(struct libinput_event_touch *event,
				       uint32_t height);

/**
 * @ingroup event_touch
 *
 * @return The generic libinput_event of this event
 */
struct libinput_event *
libinput_event_touch_get_base_event(struct libinput_event_touch *event);

/**
 * @defgroup event_gesture Gesture events
 *
 * Gesture events are generated when a gesture is recognized on a touchpad.
 *
 * Gesture sequences always start with a LIBINPUT_EVENT_GESTURE_FOO_START
 * event. All following gesture events will be of the
 * LIBINPUT_EVENT_GESTURE_FOO_UPDATE type until a
 * LIBINPUT_EVENT_GESTURE_FOO_END is generated which signals the end of the
 * gesture.
 *
 * See the libinput documentation for details on gesture handling.
 */

/**
 * @ingroup event_gesture
 *
 * @note Timestamps may not always increase. See the libinput documentation
 * for more details.
 *
 * @return The event time for this event
 */
uint32_t
libinput_event_gesture_get_time(struct libinput_event_gesture *event);

/**
 * @ingroup event_gesture
 *
 * @note Timestamps may not always increase. See the libinput documentation
 * for more details.
 *
 * @return The event time for this event in microseconds
 */
uint64_t
libinput_event_gesture_get_time_usec(struct libinput_event_gesture *event);

/**
 * @ingroup event_gesture
 *
 * @return The generic libinput_event of this event
 */
struct libinput_event *
libinput_event_gesture_get_base_event(struct libinput_event_gesture *event);

/**
 * @ingroup event_gesture
 *
 * Return the number of fingers used for a gesture. This can be used e.g.
 * to differentiate between 3 or 4 finger swipes.
 *
 * This function can be called on all gesture events and the returned finger
 * count value remains the same for the lifetime of a gesture. Thus, if a
 * user puts down a fourth finger during a three-finger swipe gesture,
 * libinput will end the three-finger gesture and, if applicable, start a
 * four-finger swipe gesture. A caller may decide that those gestures are
 * semantically identical and continue the two gestures as one single gesture.
 *
 * @return the number of fingers used for a gesture
 */
int
libinput_event_gesture_get_finger_count(struct libinput_event_gesture *event);

/**
 * @ingroup event_gesture
 *
 * Return if the gesture ended normally, or if it was cancelled.
 * For gesture events that are not of type
 * @ref LIBINPUT_EVENT_GESTURE_SWIPE_END or
 * @ref LIBINPUT_EVENT_GESTURE_PINCH_END, this function returns 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_GESTURE_SWIPE_END or
 * @ref LIBINPUT_EVENT_GESTURE_PINCH_END.
 *
 * @return 0 or 1, with 1 indicating that the gesture was cancelled.
 */
int
libinput_event_gesture_get_cancelled(struct libinput_event_gesture *event);

/**
 * @ingroup event_gesture
 *
 * Return the delta between the last event and the current event. For gesture
 * events that are not of type @ref LIBINPUT_EVENT_GESTURE_SWIPE_UPDATE or
 * @ref LIBINPUT_EVENT_GESTURE_PINCH_UPDATE, this function returns 0.
 *
 * If a device employs pointer acceleration, the delta returned by this
 * function is the accelerated delta.
 *
 * Relative motion deltas are normalized to represent those of a device with
 * 1000dpi resolution. See the libinput documentation for more details.
 *
 * @return the relative x movement since the last event
 */
double
libinput_event_gesture_get_dx(struct libinput_event_gesture *event);

/**
 * @ingroup event_gesture
 *
 * Return the delta between the last event and the current event. For gesture
 * events that are not of type @ref LIBINPUT_EVENT_GESTURE_SWIPE_UPDATE or
 * @ref LIBINPUT_EVENT_GESTURE_PINCH_UPDATE, this function returns 0.
 *
 * If a device employs pointer acceleration, the delta returned by this
 * function is the accelerated delta.
 *
 * Relative motion deltas are normalized to represent those of a device with
 * 1000dpi resolution. See the libinput documentation for more details.
 *
 * @return the relative y movement since the last event
 */
double
libinput_event_gesture_get_dy(struct libinput_event_gesture *event);

/**
 * @ingroup event_gesture
 *
 * Return the relative delta of the unaccelerated motion vector of the
 * current event. For gesture events that are not of type
 * @ref LIBINPUT_EVENT_GESTURE_SWIPE_UPDATE or
 * @ref LIBINPUT_EVENT_GESTURE_PINCH_UPDATE, this function returns 0.
 *
 * Relative unaccelerated motion deltas are normalized to represent those of a
 * device with 1000dpi resolution. See the libinput documentation for more
 * details. Note that unaccelerated events are not equivalent to 'raw' events
 * as read from the device.
 *
 * Any rotation applied to the device also applies to gesture motion
 * (see libinput_device_config_rotation_set_angle()).
 *
 * @return the unaccelerated relative x movement since the last event
 */
double
libinput_event_gesture_get_dx_unaccelerated(
	struct libinput_event_gesture *event);

/**
 * @ingroup event_gesture
 *
 * Return the relative delta of the unaccelerated motion vector of the
 * current event. For gesture events that are not of type
 * @ref LIBINPUT_EVENT_GESTURE_SWIPE_UPDATE or
 * @ref LIBINPUT_EVENT_GESTURE_PINCH_UPDATE, this function returns 0.
 *
 * Relative unaccelerated motion deltas are normalized to represent those of a
 * device with 1000dpi resolution. See the libinput documentation for more
 * details. Note that unaccelerated events are not equivalent to 'raw' events
 * as read from the device.
 *
 * Any rotation applied to the device also applies to gesture motion
 * (see libinput_device_config_rotation_set_angle()).
 *
 * @return the unaccelerated relative y movement since the last event
 */
double
libinput_event_gesture_get_dy_unaccelerated(
	struct libinput_event_gesture *event);

/**
 * @ingroup event_gesture
 *
 * Return the absolute scale of a pinch gesture, the scale is the division
 * of the current distance between the fingers and the distance at the start
 * of the gesture. The scale begins at 1.0, and if e.g. the fingers moved
 * together by 50% then the scale will become 0.5, if they move twice as far
 * apart as initially the scale becomes 2.0, etc.
 *
 * For gesture events that are of type @ref
 * LIBINPUT_EVENT_GESTURE_PINCH_BEGIN, this function returns 1.0.
 *
 * For gesture events that are of type @ref
 * LIBINPUT_EVENT_GESTURE_PINCH_END, this function returns the scale value
 * of the most recent @ref LIBINPUT_EVENT_GESTURE_PINCH_UPDATE event (if
 * any) or 1.0 otherwise.
 *
 * For all other events this function returns 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_GESTURE_PINCH_BEGIN, @ref
 * LIBINPUT_EVENT_GESTURE_PINCH_END or
 * @ref LIBINPUT_EVENT_GESTURE_PINCH_UPDATE.
 *
 * @return the absolute scale of a pinch gesture
 */
double
libinput_event_gesture_get_scale(struct libinput_event_gesture *event);

/**
 * @ingroup event_gesture
 *
 * Return the angle delta in degrees between the last and the current @ref
 * LIBINPUT_EVENT_GESTURE_PINCH_UPDATE event. For gesture events that
 * are not of type @ref LIBINPUT_EVENT_GESTURE_PINCH_UPDATE, this
 * function returns 0.
 *
 * The angle delta is defined as the change in angle of the line formed by
 * the 2 fingers of a pinch gesture. Clockwise rotation is represented
 * by a positive delta, counter-clockwise by a negative delta. If e.g. the
 * fingers are on the 12 and 6 location of a clock face plate and they move
 * to the 1 resp. 7 location in a single event then the angle delta is
 * 30 degrees.
 *
 * If more than two fingers are present, the angle represents the rotation
 * around the center of gravity. The calculation of the center of gravity is
 * implementation-dependent.
 *
 * @return the angle delta since the last event
 */
double
libinput_event_gesture_get_angle_delta(struct libinput_event_gesture *event);

/**
 * @defgroup event_tablet Tablet events
 *
 * Events that come from tools on tablet devices. For events from the pad,
 * see @ref event_tablet_pad.
 *
 * Events from tablet devices are exposed by two interfaces, tools and pads.
 * Tool events originate (usually) from a stylus-like device, pad events
 * reflect any events originating from the physical tablet itself.
 *
 * Note that many tablets support touch events. These are exposed through
 * the @ref LIBINPUT_DEVICE_CAP_POINTER interface (for external touchpad-like
 * devices such as the Wacom Intuos series) or @ref
 * LIBINPUT_DEVICE_CAP_TOUCH interface (for built-in touchscreen-like
 * devices such as the Wacom Cintiq series).
 */

/**
 * @ingroup event_tablet
 *
 * @return The generic libinput_event of this event
 *
 * @since 1.2
 */
struct libinput_event *
libinput_event_tablet_tool_get_base_event(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Check if the x axis was updated in this event.
 * For events that are not of type @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS,
 * @ref LIBINPUT_EVENT_TABLET_TOOL_TIP, or
 * @ref LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, this function returns 0.
 *
 * @note It is an application bug to call this function for events other
 * than @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS, @ref
 * LIBINPUT_EVENT_TABLET_TOOL_TIP, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_BUTTON.
 *
 * @param event The libinput tablet tool event
 * @return 1 if the axis was updated or 0 otherwise
 *
 * @since 1.2
 */
int
libinput_event_tablet_tool_x_has_changed(
				struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Check if the y axis was updated in this event.
 * For events that are not of type @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS,
 * @ref LIBINPUT_EVENT_TABLET_TOOL_TIP, or
 * @ref LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, this function returns 0.
 *
 * @note It is an application bug to call this function for events other
 * than @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS, @ref
 * LIBINPUT_EVENT_TABLET_TOOL_TIP, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_BUTTON.
 *
 * @param event The libinput tablet tool event
 * @return 1 if the axis was updated or 0 otherwise
 *
 * @since 1.2
 */
int
libinput_event_tablet_tool_y_has_changed(
				struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Check if the pressure axis was updated in this event.
 * For events that are not of type @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS,
 * @ref LIBINPUT_EVENT_TABLET_TOOL_TIP, or
 * @ref LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, this function returns 0.
 *
 * @note It is an application bug to call this function for events other
 * than @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS, @ref
 * LIBINPUT_EVENT_TABLET_TOOL_TIP, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_BUTTON.
 *
 * @param event The libinput tablet tool event
 * @return 1 if the axis was updated or 0 otherwise
 *
 * @since 1.2
 */
int
libinput_event_tablet_tool_pressure_has_changed(
				struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Check if the distance axis was updated in this event.
 * For events that are not of type @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS,
 * @ref LIBINPUT_EVENT_TABLET_TOOL_TIP, or
 * @ref LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, this function returns 0.
 * For tablet tool events of type @ref LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY,
 * this function always returns 1.
 *
 * @note It is an application bug to call this function for events other
 * than @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS, @ref
 * LIBINPUT_EVENT_TABLET_TOOL_TIP, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_BUTTON.
 *
 * @param event The libinput tablet tool event
 * @return 1 if the axis was updated or 0 otherwise
 *
 * @since 1.2
 */
int
libinput_event_tablet_tool_distance_has_changed(
				struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Check if the tilt x axis was updated in this event.
 * For events that are not of type @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS,
 * @ref LIBINPUT_EVENT_TABLET_TOOL_TIP, or
 * @ref LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, this function returns 0.
 *
 * @note It is an application bug to call this function for events other
 * than @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS, @ref
 * LIBINPUT_EVENT_TABLET_TOOL_TIP, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_BUTTON.
 *
 * @param event The libinput tablet tool event
 * @return 1 if the axis was updated or 0 otherwise
 *
 * @since 1.2
 */
int
libinput_event_tablet_tool_tilt_x_has_changed(
				struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Check if the tilt y axis was updated in this event.
 * For events that are not of type @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS,
 * @ref LIBINPUT_EVENT_TABLET_TOOL_TIP, or
 * @ref LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, this function returns 0.
 *
 * @note It is an application bug to call this function for events other
 * than @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS, @ref
 * LIBINPUT_EVENT_TABLET_TOOL_TIP, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_BUTTON.
 *
 * @param event The libinput tablet tool event
 * @return 1 if the axis was updated or 0 otherwise
 *
 * @since 1.2
 */
int
libinput_event_tablet_tool_tilt_y_has_changed(
				struct libinput_event_tablet_tool *event);
/**
 * @ingroup event_tablet
 *
 * Check if the z-rotation axis was updated in this event.
 * For events that are not of type @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS,
 * @ref LIBINPUT_EVENT_TABLET_TOOL_TIP, or
 * @ref LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, this function returns 0.
 *
 * @note It is an application bug to call this function for events other
 * than @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS, @ref
 * LIBINPUT_EVENT_TABLET_TOOL_TIP, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_BUTTON.
 *
 * @param event The libinput tablet tool event
 * @return 1 if the axis was updated or 0 otherwise
 *
 * @since 1.2
 */
int
libinput_event_tablet_tool_rotation_has_changed(
				struct libinput_event_tablet_tool *event);
/**
 * @ingroup event_tablet
 *
 * Check if the slider axis was updated in this event.
 * For events that are not of type @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS,
 * @ref LIBINPUT_EVENT_TABLET_TOOL_TIP, or
 * @ref LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, this function returns 0.
 *
 * @note It is an application bug to call this function for events other
 * than @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS, @ref
 * LIBINPUT_EVENT_TABLET_TOOL_TIP, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_BUTTON.
 *
 * @param event The libinput tablet tool event
 * @return 1 if the axis was updated or 0 otherwise
 *
 * @since 1.2
 */
int
libinput_event_tablet_tool_slider_has_changed(
				struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Check if the size major axis was updated in this event.
 * For events that are not of type @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS,
 * @ref LIBINPUT_EVENT_TABLET_TOOL_TIP, or
 * @ref LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, this function returns 0.
 *
 * @note It is an application bug to call this function for events other
 * than @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS, @ref
 * LIBINPUT_EVENT_TABLET_TOOL_TIP, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_BUTTON.
 *
 * @param event The libinput tablet tool event
 * @return 1 if the axis was updated or 0 otherwise
 */
int
libinput_event_tablet_tool_size_major_has_changed(
				struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Check if the size minor axis was updated in this event.
 * For events that are not of type @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS,
 * @ref LIBINPUT_EVENT_TABLET_TOOL_TIP, or
 * @ref LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, this function returns 0.
 *
 * @note It is an application bug to call this function for events other
 * than @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS, @ref
 * LIBINPUT_EVENT_TABLET_TOOL_TIP, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_BUTTON.
 *
 * @param event The libinput tablet tool event
 * @return 1 if the axis was updated or 0 otherwise
 */
int
libinput_event_tablet_tool_size_minor_has_changed(
				struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Check if the wheel axis was updated in this event.
 * For events that are not of type @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS,
 * @ref LIBINPUT_EVENT_TABLET_TOOL_TIP, or
 * @ref LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, this function returns 0.
 *
 * @note It is an application bug to call this function for events other
 * than @ref LIBINPUT_EVENT_TABLET_TOOL_AXIS, @ref
 * LIBINPUT_EVENT_TABLET_TOOL_TIP, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, or @ref
 * LIBINPUT_EVENT_TABLET_TOOL_BUTTON.
 *
 * @param event The libinput tablet tool event
 * @return 1 if the axis was updated or 0 otherwise
 *
 * @since 1.2
 */
int
libinput_event_tablet_tool_wheel_has_changed(
				struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Returns the X coordinate of the tablet tool, in mm from the top left
 * corner of the tablet in its current logical orientation. Use
 * libinput_event_tablet_tool_get_x_transformed() for transforming the axis
 * value into a different coordinate space.
 *
 * @note On some devices, returned value may be negative or larger than the
 * width of the device. See the libinput documentation for more details.
 *
 * @param event The libinput tablet tool event
 * @return The current value of the the axis
 *
 * @since 1.2
 */
double
libinput_event_tablet_tool_get_x(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Returns the Y coordinate of the tablet tool, in mm from the top left
 * corner of the tablet in its current logical orientation. Use
 * libinput_event_tablet_tool_get_y_transformed() for transforming the axis
 * value into a different coordinate space.
 *
 * @note On some devices, returned value may be negative or larger than the
 * width of the device. See the libinput documentation for more details.
 *
 * @param event The libinput tablet tool event
 * @return The current value of the the axis
 *
 * @since 1.2
 */
double
libinput_event_tablet_tool_get_y(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Return the delta between the last event and the current event.
 * If the tool employs pointer acceleration, the delta returned by this
 * function is the accelerated delta.
 *
 * This value is in screen coordinate space, the delta is to be interpreted
 * like the return value of libinput_event_pointer_get_dx().
 * See the libinput documentation for more details.
 *
 * @param event The libinput tablet event
 * @return The relative x movement since the last event
 *
 * @since 1.2
 */
double
libinput_event_tablet_tool_get_dx(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Return the delta between the last event and the current event.
 * If the tool employs pointer acceleration, the delta returned by this
 * function is the accelerated delta.
 *
 * This value is in screen coordinate space, the delta is to be interpreted
 * like the return value of libinput_event_pointer_get_dx().
 * See the libinput documentation for more details.
 *
 * @param event The libinput tablet event
 * @return The relative y movement since the last event
 *
 * @since 1.2
 */
double
libinput_event_tablet_tool_get_dy(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Returns the current pressure being applied on the tool in use, normalized
 * to the range [0, 1].
 *
 * If this axis does not exist on the current tool, this function returns 0.
 *
 * @param event The libinput tablet tool event
 * @return The current value of the the axis
 *
 * @since 1.2
 */
double
libinput_event_tablet_tool_get_pressure(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Returns the current distance from the tablet's sensor, normalized to the
 * range [0, 1].
 *
 * If this axis does not exist on the current tool, this function returns 0.
 *
 * @param event The libinput tablet tool event
 * @return The current value of the the axis
 *
 * @since 1.2
 */
double
libinput_event_tablet_tool_get_distance(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Returns the current tilt along the X axis of the tablet's current logical
 * orientation, in degrees off the tablet's z axis. That is, if the tool is
 * perfectly orthogonal to the tablet, the tilt angle is 0. When the top
 * tilts towards the logical top/left of the tablet, the x/y tilt angles are
 * negative, if the top tilts towards the logical bottom/right of the
 * tablet, the x/y tilt angles are positive.
 *
 * If this axis does not exist on the current tool, this function returns 0.
 *
 * @param event The libinput tablet tool event
 * @return The current value of the axis in degrees
 *
 * @since 1.2
 */
double
libinput_event_tablet_tool_get_tilt_x(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Returns the current tilt along the Y axis of the tablet's current logical
 * orientation, in degrees off the tablet's z axis. That is, if the tool is
 * perfectly orthogonal to the tablet, the tilt angle is 0. When the top
 * tilts towards the logical top/left of the tablet, the x/y tilt angles are
 * negative, if the top tilts towards the logical bottom/right of the
 * tablet, the x/y tilt angles are positive.
 *
 * If this axis does not exist on the current tool, this function returns 0.
 *
 * @param event The libinput tablet tool event
 * @return The current value of the the axis in degrees
 *
 * @since 1.2
 */
double
libinput_event_tablet_tool_get_tilt_y(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Returns the current z rotation of the tool in degrees, clockwise from the
 * tool's logical neutral position.
 *
 * For tools of type @ref LIBINPUT_TABLET_TOOL_TYPE_MOUSE and @ref
 * LIBINPUT_TABLET_TOOL_TYPE_LENS the logical neutral position is
 * pointing to the current logical north of the tablet. For tools of type @ref
 * LIBINPUT_TABLET_TOOL_TYPE_BRUSH, the logical neutral position is with the
 * buttons pointing up.
 *
 * If this axis does not exist on the current tool, this function returns 0.
 *
 * @param event The libinput tablet tool event
 * @return The current value of the the axis
 *
 * @since 1.2
 */
double
libinput_event_tablet_tool_get_rotation(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Returns the current position of the slider on the tool, normalized to the
 * range [-1, 1]. The logical zero is the neutral position of the slider, or
 * the logical center of the axis. This axis is available on e.g. the Wacom
 * Airbrush.
 *
 * If this axis does not exist on the current tool, this function returns 0.
 *
 * @param event The libinput tablet tool event
 * @return The current value of the the axis
 *
 * @since 1.2
 */
double
libinput_event_tablet_tool_get_slider_position(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Returns the current size in mm along the major axis of the touching
 * ellipse. This axis is not necessarily aligned with either x or y, the
 * rotation must be taken into account.
 *
 * Where no rotation is available on a tool, or where rotation is zero, the
 * major axis aligns with the y axis and the minor axis with the x axis.
 *
 * If this axis does not exist on the current tool, this function returns 0.
 *
 * @param event The libinput tablet tool event
 * @return The current value of the axis major in mm
 */
double
libinput_event_tablet_tool_get_size_major(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Returns the current size in mm along the minor axis of the touching
 * ellipse. This axis is not necessarily aligned with either x or y, the
 * rotation must be taken into account.
 *
 * Where no rotation is available on a tool, or where rotation is zero, the
 * minor axis aligns with the y axis and the minor axis with the x axis.
 *
 * If this axis does not exist on the current tool, this function returns 0.
 *
 * @param event The libinput tablet tool event
 * @return The current value of the axis minor in mm
 */
double
libinput_event_tablet_tool_get_size_minor(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Return the delta for the wheel in degrees.
 *
 * @param event The libinput tablet tool event
 * @return The delta of the wheel, in degrees, compared to the last event
 *
 * @see libinput_event_tablet_tool_get_wheel_delta_discrete
 */
double
libinput_event_tablet_tool_get_wheel_delta(
				   struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Return the delta for the wheel in discrete steps (e.g. wheel clicks).

 * @param event The libinput tablet tool event
 * @return The delta of the wheel, in discrete steps, compared to the last event
 *
 * @see libinput_event_tablet_tool_get_wheel_delta_discrete
 *
 * @since 1.2
 */
int
libinput_event_tablet_tool_get_wheel_delta_discrete(
				    struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Return the current absolute x coordinate of the tablet tool event,
 * transformed to screen coordinates.
 *
 * @note This function may be called for a specific axis even if
 * libinput_event_tablet_tool_*_has_changed() returns 0 for that axis.
 * libinput always includes all device axes in the event.
 *
 * @note On some devices, returned value may be negative or larger than the
 * width of the device. See the libinput documentation for more details.
 *
 * @param event The libinput tablet tool event
 * @param width The current output screen width
 * @return the current absolute x coordinate transformed to a screen coordinate
 *
 * @since 1.2
 */
double
libinput_event_tablet_tool_get_x_transformed(struct libinput_event_tablet_tool *event,
					     uint32_t width);

/**
 * @ingroup event_tablet
 *
 * Return the current absolute y coordinate of the tablet tool event,
 * transformed to screen coordinates.
 *
 * @note This function may be called for a specific axis even if
 * libinput_event_tablet_tool_*_has_changed() returns 0 for that axis.
 * libinput always includes all device axes in the event.
 *
 * @note On some devices, returned value may be negative or larger than the
 * width of the device. See the libinput documentation for more details.
 *
 * @param event The libinput tablet tool event
 * @param height The current output screen height
 * @return the current absolute y coordinate transformed to a screen coordinate
 *
 * @since 1.2
 */
double
libinput_event_tablet_tool_get_y_transformed(struct libinput_event_tablet_tool *event,
					     uint32_t height);

/**
 * @ingroup event_tablet
 *
 * Returns the tool that was in use during this event.
 *
 * The returned tablet tool is not refcounted and may become invalid after
 * the next call to libinput. Use libinput_tablet_tool_ref() and
 * libinput_tablet_tool_unref() to continue using the handle outside of the
 * immediate scope.
 *
 * If the caller holds at least one reference, this struct is used
 * whenever the tools enters proximity again.
  *
 * @note Physical tool tracking requires hardware support. If unavailable,
 * libinput creates one tool per type per tablet. See
 * libinput_tablet_tool_get_serial() for more details.
 *
 * @param event The libinput tablet tool event
 * @return The new tool triggering this event
 *
 * @since 1.2
 */
struct libinput_tablet_tool *
libinput_event_tablet_tool_get_tool(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Returns the new proximity state of a tool from a proximity event.
 * Used to check whether or not a tool came in or out of proximity during an
 * event of type @ref LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY.
 *
 * The proximity state represents the logical proximity state which does not
 * necessarily match when a tool comes into sensor range or leaves the
 * sensor range. On some tools this range does not represent the physical
 * range but a reduced tool-specific logical range. If the range is reduced,
 * this is done transparent to the caller.
 *
 * For example, the Wacom mouse and lens cursor tools are usually
 * used in relative mode, lying flat on the tablet. Movement typically follows
 * the interaction normal mouse movements have, i.e. slightly lift the tool and
 * place it in a separate location. The proximity detection on Wacom
 * tablets however extends further than the user may lift the mouse, i.e. the
 * tool may not be lifted out of physical proximity. For such tools, libinput
 * provides software-emulated proximity.
 *
 * @param event The libinput tablet tool event
 * @return The new proximity state of the tool from the event.
 *
 * @since 1.2
 */
enum libinput_tablet_tool_proximity_state
libinput_event_tablet_tool_get_proximity_state(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Returns the new tip state of a tool from a tip event.
 * Used to check whether or not a tool came in contact with the tablet
 * surface or left contact with the tablet surface during an
 * event of type @ref LIBINPUT_EVENT_TABLET_TOOL_TIP.
 *
 * @param event The libinput tablet tool event
 * @return The new tip state of the tool from the event.
 *
 * @since 1.2
 */
enum libinput_tablet_tool_tip_state
libinput_event_tablet_tool_get_tip_state(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Return the button that triggered this event.  For events that are not of
 * type @ref LIBINPUT_EVENT_TABLET_TOOL_BUTTON, this function returns 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_TABLET_TOOL_BUTTON.
 *
 * @param event The libinput tablet tool event
 * @return the button triggering this event
 *
 * @since 1.2
 */
uint32_t
libinput_event_tablet_tool_get_button(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Return the button state of the event.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_TABLET_TOOL_BUTTON.
 *
 * @param event The libinput tablet tool event
 * @return the button state triggering this event
 *
 * @since 1.2
 */
enum libinput_button_state
libinput_event_tablet_tool_get_button_state(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * For the button of a @ref LIBINPUT_EVENT_TABLET_TOOL_BUTTON event, return the total
 * number of buttons pressed on all devices on the associated seat after the
 * the event was triggered.
 *
 " @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_TABLET_TOOL_BUTTON. For other events, this function returns 0.
 *
 * @param event The libinput tablet tool event
 * @return the seat wide pressed button count for the key of this event
 *
 * @since 1.2
 */
uint32_t
libinput_event_tablet_tool_get_seat_button_count(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * @note Timestamps may not always increase. See the libinput documentation
 * for more details.
 *
 * @param event The libinput tablet tool event
 * @return The event time for this event
 *
 * @since 1.2
 */
uint32_t
libinput_event_tablet_tool_get_time(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * @note Timestamps may not always increase. See the libinput documentation
 * for more details.
 *
 * @param event The libinput tablet tool event
 * @return The event time for this event in microseconds
 *
 * @since 1.2
 */
uint64_t
libinput_event_tablet_tool_get_time_usec(struct libinput_event_tablet_tool *event);

/**
 * @ingroup event_tablet
 *
 * Return the high-level tool type for a tool object.
 *
 * The high level tool describes general interaction expected with the tool.
 * For example, a user would expect a tool of type @ref
 * LIBINPUT_TABLET_TOOL_TYPE_PEN to interact with a graphics application
 * taking pressure and tilt into account. The default virtual tool assigned
 * should be a drawing tool, e.g. a virtual pen or brush.
 * A tool of type @ref LIBINPUT_TABLET_TOOL_TYPE_ERASER would normally be
 * mapped to an eraser-like virtual tool.
 *
 * If supported by the hardware, a more specific tool id is always
 * available, see libinput_tablet_tool_get_tool_id().
 *
 * @param tool The libinput tool
 * @return The tool type for this tool object
 *
 * @see libinput_tablet_tool_get_tool_id
 *
 * @since 1.2
 */
enum libinput_tablet_tool_type
libinput_tablet_tool_get_type(struct libinput_tablet_tool *tool);

/**
 * @ingroup event_tablet
 *
 * Return the tool ID for a tool object. If nonzero, this number identifies
 * the specific type of the tool with more precision than the type returned in
 * libinput_tablet_tool_get_type(). Not all tablets support a tool ID.
 *
 * Tablets known to support tool IDs include the Wacom Intuos 3, 4, 5, Wacom
 * Cintiq and Wacom Intuos Pro series. The tool ID can be used to
 * distinguish between e.g. a Wacom Classic Pen or a Wacom Pro Pen.  It is
 * the caller's responsibility to interpret the tool ID.
 *
 * @param tool The libinput tool
 * @return The tool ID for this tool object or 0 if none is provided
 *
 * @see libinput_tablet_tool_get_type
 *
 * @since 1.2
 */
uint64_t
libinput_tablet_tool_get_tool_id(struct libinput_tablet_tool *tool);

/**
 * @ingroup event_tablet
 *
 * Increment the reference count of the tool by one. A tool is destroyed
 * whenever the reference count reaches 0. See libinput_tablet_tool_unref().
 *
 * @param tool The tool to increment the ref count of
 * @return The passed tool
 *
 * @see libinput_tablet_tool_unref
 *
 * @since 1.2
 */
struct libinput_tablet_tool *
libinput_tablet_tool_ref(struct libinput_tablet_tool *tool);

/**
 * @ingroup event_tablet
 *
 * Decrement the reference count of the tool by one. When the reference
 * count of the tool reaches 0, the memory allocated for the tool will be
 * freed.
 *
 * @param tool The tool to decrement the ref count of
 * @return NULL if the tool was destroyed otherwise the passed tool
 *
 * @see libinput_tablet_tool_ref
 *
 * @since 1.2
 */
struct libinput_tablet_tool *
libinput_tablet_tool_unref(struct libinput_tablet_tool *tool);

/**
 * @ingroup event_tablet
 *
 * Return whether the tablet tool supports pressure.
 *
 * @param tool The tool to check the axis capabilities of
 * @return Nonzero if the axis is available, zero otherwise.
 *
 * @since 1.2
 */
int
libinput_tablet_tool_has_pressure(struct libinput_tablet_tool *tool);

/**
 * @ingroup event_tablet
 *
 * Return whether the tablet tool supports distance.
 *
 * @param tool The tool to check the axis capabilities of
 * @return Nonzero if the axis is available, zero otherwise.
 *
 * @since 1.2
 */
int
libinput_tablet_tool_has_distance(struct libinput_tablet_tool *tool);

/**
 * @ingroup event_tablet
 *
 * Return whether the tablet tool supports tilt.
 *
 * @param tool The tool to check the axis capabilities of
 * @return Nonzero if the axis is available, zero otherwise.
 *
 * @since 1.2
 */
int
libinput_tablet_tool_has_tilt(struct libinput_tablet_tool *tool);

/**
 * @ingroup event_tablet
 *
 * Return whether the tablet tool supports z-rotation.
 *
 * @param tool The tool to check the axis capabilities of
 * @return Nonzero if the axis is available, zero otherwise.
 *
 * @since 1.2
 */
int
libinput_tablet_tool_has_rotation(struct libinput_tablet_tool *tool);

/**
 * @ingroup event_tablet
 *
 * Return whether the tablet tool has a slider axis.
 *
 * @param tool The tool to check the axis capabilities of
 * @return Nonzero if the axis is available, zero otherwise.
 *
 * @since 1.2
 */
int
libinput_tablet_tool_has_slider(struct libinput_tablet_tool *tool);

/**
 * @ingroup event_tablet
 *
 * Return whether the tablet tool has a ellipsis major and minor.
 * Where the underlying hardware only supports one of either major or minor,
 * libinput emulates the other axis as a circular contact, i.e. major ==
 * minor for all values of major.
 *
 * @param tool The tool to check the axis capabilities of
 * @return Nonzero if the axis is available, zero otherwise.
 */
int
libinput_tablet_tool_has_size(struct libinput_tablet_tool *tool);

/**
 * @ingroup event_tablet
 *
 * Return whether the tablet tool has a relative wheel.
 *
 * @param tool The tool to check the axis capabilities of
 * @return Nonzero if the axis is available, zero otherwise.
 *
 * @since 1.2
 */
int
libinput_tablet_tool_has_wheel(struct libinput_tablet_tool *tool);

/**
 * @ingroup event_tablet
 *
 * Check if a tablet tool has a button with the
 * passed-in code (see linux/input.h).
 *
 * @param tool A tablet tool
 * @param code button code to check for
 *
 * @return 1 if the tool supports this button code, 0 if it does not
 *
 * @since 1.2
 */
int
libinput_tablet_tool_has_button(struct libinput_tablet_tool *tool,
				uint32_t code);

/**
 * @ingroup event_tablet
 *
 * Return nonzero if the physical tool can be uniquely identified by
 * libinput, or nonzero otherwise. If a tool can be uniquely identified,
 * keeping a reference to the tool allows tracking the tool across
 * proximity out sequences and across compatible tablets.
 * See libinput_tablet_tool_get_serial() for more details.
 *
 * @param tool A tablet tool
 * @return 1 if the tool can be uniquely identified, 0 otherwise.
 *
 * @see libinput_tablet_tool_get_serial
 *
 * @since 1.2
 */
int
libinput_tablet_tool_is_unique(struct libinput_tablet_tool *tool);

/**
 * @ingroup event_tablet
 *
 * Return the serial number of a tool. If the tool does not report a serial
 * number, this function returns zero.
 *
 * Some tools provide hardware information that enables libinput to uniquely
 * identify the physical device. For example, tools compatible with the
 * Wacom Intuos 4, Intuos 5, Intuos Pro and Cintiq series are uniquely
 * identifiable through a serial number. libinput does not specify how a
 * tool can be identified uniquely, a caller should use
 * libinput_tablet_tool_is_unique() to check if the tool is unique.
 *
 * libinput creates a struct @ref libinput_tablet_tool on the first
 * proximity in of this tool. By default, this struct is destroyed on
 * proximity out and re-initialized on the next proximity in. If a caller
 * keeps a reference to the tool by using libinput_tablet_tool_ref()
 * libinput re-uses this struct whenever that same physical tool comes into
 * proximity on any tablet
 * recognized by libinput. It is possible to attach tool-specific virtual
 * state to the tool. For example, a graphics program such as the GIMP may
 * assign a specific color to each tool, allowing the artist to use the
 * tools like physical pens of different color. In multi-tablet setups it is
 * also possible to track the tool across devices.
 *
 * If the tool does not have a unique identifier, libinput creates a single
 * struct @ref libinput_tablet_tool per tool type on each tablet the tool is
 * used on.
 *
 * @param tool The libinput tool
 * @return The tool serial number
 *
 * @see libinput_tablet_tool_is_unique
 *
 * @since 1.2
 */
uint64_t
libinput_tablet_tool_get_serial(struct libinput_tablet_tool *tool);

/**
 * @ingroup event_tablet
 *
 * Return the user data associated with a tool object. libinput does
 * not manage, look at, or modify this data. The caller must ensure the
 * data is valid.
 *
 * @param tool The libinput tool
 * @return The user data associated with the tool object
 *
 * @since 1.2
 */
void *
libinput_tablet_tool_get_user_data(struct libinput_tablet_tool *tool);

/**
 * @ingroup event_tablet
 *
 * Set the user data associated with a tool object, if any.
 *
 * @param tool The libinput tool
 * @param user_data The user data to associate with the tool object
 *
 * @since 1.2
 */
void
libinput_tablet_tool_set_user_data(struct libinput_tablet_tool *tool,
				   void *user_data);

/**
 * @defgroup event_tablet_pad Tablet pad events
 *
 * Events that come from the pad of tablet devices.  For events from the
 * tablet tools, see @ref event_tablet.
 *
 * @since 1.3
 */

/**
 * @ingroup event_tablet_pad
 *
 * @return The generic libinput_event of this event
 *
 * @since 1.3
 */
struct libinput_event *
libinput_event_tablet_pad_get_base_event(struct libinput_event_tablet_pad *event);

/**
 * @ingroup event_tablet_pad
 *
 * Returns the current position of the ring, in degrees counterclockwise
 * from the northern-most point of the ring in the tablet's current logical
 * orientation.
 *
 * If the source is @ref LIBINPUT_TABLET_PAD_RING_SOURCE_FINGER,
 * libinput sends a terminating event with a ring value of -1 when the
 * finger is lifted from the ring. A caller may use this information to e.g.
 * determine if kinetic scrolling should be triggered.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_TABLET_PAD_RING.  For other events, this function
 * returns 0.
 *
 * @param event The libinput tablet pad event
 * @return The current value of the the axis
 * @retval -1 The finger was lifted
 *
 * @since 1.3
 */
double
libinput_event_tablet_pad_get_ring_position(struct libinput_event_tablet_pad *event);

/**
 * @ingroup event_tablet_pad
 *
 * Returns the number of the ring that has changed state, with 0 being the
 * first ring. On tablets with only one ring, this function always returns
 * 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_TABLET_PAD_RING.  For other events, this function
 * returns 0.
 *
 * @param event The libinput tablet pad event
 * @return The index of the ring that changed state
 *
 * @since 1.3
 */
unsigned int
libinput_event_tablet_pad_get_ring_number(struct libinput_event_tablet_pad *event);

/**
 * @ingroup event_tablet_pad
 *
 * Returns the source of the interaction with the ring. If the source is
 * @ref LIBINPUT_TABLET_PAD_RING_SOURCE_FINGER, libinput sends a ring
 * position value of -1 to terminate the current interaction.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_TABLET_PAD_RING.  For other events, this function
 * returns 0.
 *
 * @param event The libinput tablet pad event
 * @return The source of the ring interaction
 *
 * @since 1.3
 */
enum libinput_tablet_pad_ring_axis_source
libinput_event_tablet_pad_get_ring_source(struct libinput_event_tablet_pad *event);

/**
 * @ingroup event_tablet_pad
 *
 * Returns the current position of the strip, normalized to the range
 * [0, 1], with 0 being the top/left-most point in the tablet's current
 * logical orientation.
 *
 * If the source is @ref LIBINPUT_TABLET_PAD_STRIP_SOURCE_FINGER,
 * libinput sends a terminating event with a ring value of -1 when the
 * finger is lifted from the ring. A caller may use this information to e.g.
 * determine if kinetic scrolling should be triggered.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_TABLET_PAD_STRIP.  For other events, this function
 * returns 0.
 *
 * @param event The libinput tablet pad event
 * @return The current value of the the axis
 * @retval -1 The finger was lifted
 *
 * @since 1.3
 */
double
libinput_event_tablet_pad_get_strip_position(struct libinput_event_tablet_pad *event);

/**
 * @ingroup event_tablet_pad
 *
 * Returns the number of the strip that has changed state, with 0 being the
 * first strip. On tablets with only one strip, this function always returns
 * 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_TABLET_PAD_STRIP.  For other events, this function
 * returns 0.
 *
 * @param event The libinput tablet pad event
 * @return The index of the strip that changed state
 *
 * @since 1.3
 */
unsigned int
libinput_event_tablet_pad_get_strip_number(struct libinput_event_tablet_pad *event);

/**
 * @ingroup event_tablet_pad
 *
 * Returns the source of the interaction with the strip. If the source is
 * @ref LIBINPUT_TABLET_PAD_STRIP_SOURCE_FINGER, libinput sends a strip
 * position value of -1 to terminate the current interaction.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_TABLET_PAD_STRIP.  For other events, this function
 * returns 0.
 *
 * @param event The libinput tablet pad event
 * @return The source of the strip interaction
 *
 * @since 1.3
 */
enum libinput_tablet_pad_strip_axis_source
libinput_event_tablet_pad_get_strip_source(struct libinput_event_tablet_pad *event);

/**
 * @ingroup event_tablet_pad
 *
 * Return the button number that triggered this event, starting at 0.
 * For events that are not of type @ref LIBINPUT_EVENT_TABLET_PAD_BUTTON,
 * this function returns 0.
 *
 * Note that the number returned is a generic sequential button number and
 * not a semantic button code as defined in linux/input.h.
 * See the libinput documentation for more details.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_TABLET_PAD_BUTTON. For other events, this function
 * returns 0.
 *
 * @param event The libinput tablet pad event
 * @return the button triggering this event
 *
 * @since 1.3
 */
uint32_t
libinput_event_tablet_pad_get_button_number(struct libinput_event_tablet_pad *event);

/**
 * @ingroup event_tablet_pad
 *
 * Return the button state of the event.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_TABLET_PAD_BUTTON. For other events, this function
 * returns 0.
 *
 * @param event The libinput tablet pad event
 * @return the button state triggering this event
 *
 * @since 1.3
 */
enum libinput_button_state
libinput_event_tablet_pad_get_button_state(struct libinput_event_tablet_pad *event);

/**
 * @ingroup event_tablet_pad
 *
 * Return the key code that triggered this event, e.g. KEY_CONTROLPANEL. The
 * list of key codes is defined in linux/input-event-codes.h.
 *
 * For events that are not of type @ref LIBINPUT_EVENT_TABLET_PAD_KEY,
 * this function returns 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_TABLET_PAD_KEY. For other events, this function
 * returns 0.
 *
 * @param event The libinput tablet pad event
 * @return the key code triggering this event
 *
 * @since 1.15
 */
uint32_t
libinput_event_tablet_pad_get_key(struct libinput_event_tablet_pad *event);

/**
 * @ingroup event_tablet_pad
 *
 * Return the key state of the event.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_TABLET_PAD_KEY. For other events, this function
 * returns 0.
 *
 * @param event The libinput tablet pad event
 * @return the key state triggering this event
 *
 * @since 1.15
 */
enum libinput_key_state
libinput_event_tablet_pad_get_key_state(struct libinput_event_tablet_pad *event);

/**
 * @ingroup event_tablet_pad
 *
 * Returns the mode the button, ring, or strip that triggered this event is
 * in, at the time of the event.
 *
 * The mode is a virtual grouping of functionality, usually based on some
 * visual feedback like LEDs on the pad. Mode indices start at 0, a device
 * that does not support modes always returns 0.
 *
 * @note Pad keys are not part of a mode group. It is an application bug to
 * call this function for @ref LIBINPUT_EVENT_TABLET_PAD_KEY.
 *
 * Mode switching is controlled by libinput and more than one mode may exist
 * on the tablet. This function returns the mode that this event's button,
 * ring or strip is logically in. If the button is a mode toggle button
 * and the button event caused a new mode to be toggled, the mode returned
 * is the new mode the button is in.
 *
 * Note that the returned mode is the mode valid as of the time of the
 * event. The returned mode may thus be different to the mode returned by
 * libinput_tablet_pad_mode_group_get_mode(). See
 * libinput_tablet_pad_mode_group_get_mode() for details.
 *
 * @param event The libinput tablet pad event
 * @return the 0-indexed mode of this button, ring or strip at the time of
 * the event
 *
 * @see libinput_tablet_pad_mode_group_get_mode
 *
 * @since 1.4
 */
unsigned int
libinput_event_tablet_pad_get_mode(struct libinput_event_tablet_pad *event);

/**
 * @ingroup event_tablet_pad
 *
 * Returns the mode group that the button, ring, or strip that triggered
 * this event is considered in. The mode is a virtual grouping of
 * functionality, usually based on some visual feedback like LEDs on the
 * pad.
 *
 * @note Pad keys are not part of a mode group. It is an application bug to
 * call this function for @ref LIBINPUT_EVENT_TABLET_PAD_KEY.
 *
 * The returned mode group is not refcounted and may become invalid after
 * the next call to libinput. Use libinput_tablet_pad_mode_group_ref() and
 * libinput_tablet_pad_mode_group_unref() to continue using the handle
 * outside of the immediate scope.
 *
 * @param event The libinput tablet pad event
 * @return the mode group of the button, ring or strip that caused this event
 *
 * @see libinput_device_tablet_pad_get_mode_group
 *
 * @since 1.4
 */
struct libinput_tablet_pad_mode_group *
libinput_event_tablet_pad_get_mode_group(struct libinput_event_tablet_pad *event);

/**
 * @ingroup event_tablet_pad
 *
 * @note Timestamps may not always increase. See the libinput documentation
 * for more details.
 *
 * @param event The libinput tablet pad event
 * @return The event time for this event
 *
 * @since 1.3
 */
uint32_t
libinput_event_tablet_pad_get_time(struct libinput_event_tablet_pad *event);

/**
 * @ingroup event_tablet_pad
 *
 * @note Timestamps may not always increase. See the libinput documentation
 * for more details.
 *
 * @param event The libinput tablet pad event
 * @return The event time for this event in microseconds
 *
 * @since 1.3
 */
uint64_t
libinput_event_tablet_pad_get_time_usec(struct libinput_event_tablet_pad *event);

/**
 * @defgroup event_switch Switch events
 *
 * Events that come from switch devices.
 */

/**
 * @ingroup event_switch
 *
 * Return the switch that triggered this event.
 * For pointer events that are not of type @ref
 * LIBINPUT_EVENT_SWITCH_TOGGLE, this function returns 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_SWITCH_TOGGLE.
 *
 * @param event The libinput switch event
 * @return The switch triggering this event
 *
 * @since 1.7
 */
enum libinput_switch
libinput_event_switch_get_switch(struct libinput_event_switch *event);

/**
 * @ingroup event_switch
 *
 * Return the switch state that triggered this event.
 * For switch events that are not of type @ref
 * LIBINPUT_EVENT_SWITCH_TOGGLE, this function returns 0.
 *
 * @note It is an application bug to call this function for events other than
 * @ref LIBINPUT_EVENT_SWITCH_TOGGLE.
 *
 * @param event The libinput switch event
 * @return The switch state triggering this event
 *
 * @since 1.7
 */
enum libinput_switch_state
libinput_event_switch_get_switch_state(struct libinput_event_switch *event);

/**
 * @ingroup event_switch
 *
 * @return The generic libinput_event of this event
 *
 * @since 1.7
 */
struct libinput_event *
libinput_event_switch_get_base_event(struct libinput_event_switch *event);

/**
 * @ingroup event_switch
 *
 * @note Timestamps may not always increase. See the libinput documentation
 * for more details.
 *
 * @param event The libinput switch event
 * @return The event time for this event
 *
 * @since 1.7
 */
uint32_t
libinput_event_switch_get_time(struct libinput_event_switch *event);

/**
 * @ingroup event_switch
 *
 * @note Timestamps may not always increase. See the libinput documentation
 * for more details.
 *
 * @param event The libinput switch event
 * @return The event time for this event in microseconds
 *
 * @since 1.7
 */
uint64_t
libinput_event_switch_get_time_usec(struct libinput_event_switch *event);

/**
 * @defgroup base Initialization and manipulation of libinput contexts
 */

/**
 * @ingroup base
 * @struct libinput_interface
 *
 * libinput does not open file descriptors to devices directly, instead
 * open_restricted() and close_restricted() are called for each path that
 * must be opened.
 *
 * @see libinput_udev_create_context
 * @see libinput_path_create_context
 */
struct libinput_interface {
	/**
	 * Open the device at the given path with the flags provided and
	 * return the fd.
	 *
	 * @param path The device path to open
	 * @param flags Flags as defined by open(2)
	 * @param user_data The user_data provided in
	 * libinput_udev_create_context()
	 *
	 * @return The file descriptor, or a negative errno on failure.
	 */
	int (*open_restricted)(const char *path, int flags, void *user_data);
	/**
	 * Close the file descriptor.
	 *
	 * @param fd The file descriptor to close
	 * @param user_data The user_data provided in
	 * libinput_udev_create_context()
	 */
	void (*close_restricted)(int fd, void *user_data);
};

/**
 * @ingroup base
 *
 * Create a new libinput context from udev. This context is inactive until
 * assigned a seat ID with libinput_udev_assign_seat().
 *
 * @param interface The callback interface
 * @param user_data Caller-specific data passed to the various callback
 * interfaces.
 * @param udev An already initialized udev context
 *
 * @return An initialized, but inactive libinput context or NULL on error
 */
struct libinput *
libinput_udev_create_context(const struct libinput_interface *interface,
			     void *user_data,
			     struct udev *udev);

/**
 * @ingroup base
 *
 * Assign a seat to this libinput context. New devices or the removal of
 * existing devices will appear as events during libinput_dispatch().
 *
 * libinput_udev_assign_seat() succeeds even if no input devices are currently
 * available on this seat, or if devices are available but fail to open in
 * @ref libinput_interface::open_restricted. Devices that do not have the
 * minimum capabilities to be recognized as pointer, keyboard or touch
 * device are ignored. Such devices and those that failed to open
 * ignored until the next call to libinput_resume().
 *
 * This function may only be called once per context.
 *
 * @param libinput A libinput context initialized with
 * libinput_udev_create_context()
 * @param seat_id A seat identifier. This string must not be NULL.
 *
 * @return 0 on success or -1 on failure.
 */
int
libinput_udev_assign_seat(struct libinput *libinput,
			  const char *seat_id);

/**
 * @ingroup base
 *
 * Create a new libinput context that requires the caller to manually add or
 * remove devices with libinput_path_add_device() and
 * libinput_path_remove_device().
 *
 * The context is fully initialized but will not generate events until at
 * least one device has been added.
 *
 * The reference count of the context is initialized to 1. See @ref
 * libinput_unref.
 *
 * @param interface The callback interface
 * @param user_data Caller-specific data passed to the various callback
 * interfaces.
 *
 * @return An initialized, empty libinput context.
 */
struct libinput *
libinput_path_create_context(const struct libinput_interface *interface,
			     void *user_data);

/**
 * @ingroup base
 *
 * Add a device to a libinput context initialized with
 * libinput_path_create_context(). If successful, the device will be
 * added to the internal list and re-opened on libinput_resume(). The device
 * can be removed with libinput_path_remove_device().
 *
 * If the device was successfully initialized, it is returned in the device
 * argument. The lifetime of the returned device pointer is limited until
 * the next libinput_dispatch(), use libinput_device_ref() to keep a permanent
 * reference.
 *
 * @param libinput A previously initialized libinput context
 * @param path Path to an input device
 * @return The newly initiated device on success, or NULL on failure.
 *
 * @note It is an application bug to call this function on a libinput
 * context initialized with libinput_udev_create_context().
 */
struct libinput_device *
libinput_path_add_device(struct libinput *libinput,
			 const char *path);

/**
 * @ingroup base
 *
 * Remove a device from a libinput context initialized with
 * libinput_path_create_context() or added to such a context with
 * libinput_path_add_device().
 *
 * Events already processed from this input device are kept in the queue,
 * the @ref LIBINPUT_EVENT_DEVICE_REMOVED event marks the end of events for
 * this device.
 *
 * If no matching device exists, this function does nothing.
 *
 * @param device A libinput device
 *
 * @note It is an application bug to call this function on a libinput
 * context initialized with libinput_udev_create_context().
 */
void
libinput_path_remove_device(struct libinput_device *device);

/**
 * @ingroup base
 *
 * libinput keeps a single file descriptor for all events. Call into
 * libinput_dispatch() if any events become available on this fd.
 *
 * @return The file descriptor used to notify of pending events.
 */
int
libinput_get_fd(struct libinput *libinput);

/**
 * @ingroup base
 *
 * Main event dispatchment function. Reads events of the file descriptors
 * and processes them internally. Use libinput_get_event() to retrieve the
 * events.
 *
 * Dispatching does not necessarily queue libinput events. This function
 * should be called immediately once data is available on the file
 * descriptor returned by libinput_get_fd(). libinput has a number of
 * timing-sensitive features (e.g. tap-to-click), any delay in calling
 * libinput_dispatch() may prevent these features from working correctly.
 *
 * @param libinput A previously initialized libinput context
 *
 * @return 0 on success, or a negative errno on failure
 */
int
libinput_dispatch(struct libinput *libinput);

/**
 * @ingroup base
 *
 * Retrieve the next event from libinput's internal event queue.
 *
 * After handling the retrieved event, the caller must destroy it using
 * libinput_event_destroy().
 *
 * @param libinput A previously initialized libinput context
 * @return The next available event, or NULL if no event is available.
 */
struct libinput_event *
libinput_get_event(struct libinput *libinput);

/**
 * @ingroup base
 *
 * Return the type of the next event in the internal queue. This function
 * does not pop the event off the queue and the next call to
 * libinput_get_event() returns that event.
 *
 * @param libinput A previously initialized libinput context
 * @return The event type of the next available event or @ref
 * LIBINPUT_EVENT_NONE if no event is available.
 */
enum libinput_event_type
libinput_next_event_type(struct libinput *libinput);

/**
 * @ingroup base
 *
 * Set caller-specific data associated with this context. libinput does
 * not manage, look at, or modify this data. The caller must ensure the
 * data is valid.
 *
 * @param libinput A previously initialized libinput context
 * @param user_data Caller-specific data passed to the various callback
 * interfaces.
 */
void
libinput_set_user_data(struct libinput *libinput,
		       void *user_data);

/**
 * @ingroup base
 *
 * Get the caller-specific data associated with this context, if any.
 *
 * @param libinput A previously initialized libinput context
 * @return The caller-specific data previously assigned in
 * libinput_set_user_data(), libinput_path_create_context() or
 * libinput_udev_create_context().
 */
void *
libinput_get_user_data(struct libinput *libinput);

/**
 * @ingroup base
 *
 * Resume a suspended libinput context. This re-enables device
 * monitoring and adds existing devices.
 *
 * @param libinput A previously initialized libinput context
 * @see libinput_suspend
 *
 * @return 0 on success or -1 on failure
 */
int
libinput_resume(struct libinput *libinput);

/**
 * @ingroup base
 *
 * Suspend monitoring for new devices and close existing devices.
 * This all but terminates libinput but does keep the context
 * valid to be resumed with libinput_resume().
 *
 * @param libinput A previously initialized libinput context
 */
void
libinput_suspend(struct libinput *libinput);

/**
 * @ingroup base
 *
 * Add a reference to the context. A context is destroyed whenever the
 * reference count reaches 0. See @ref libinput_unref.
 *
 * @param libinput A previously initialized valid libinput context
 * @return The passed libinput context
 */
struct libinput *
libinput_ref(struct libinput *libinput);

/**
 * @ingroup base
 *
 * Dereference the libinput context. After this, the context may have been
 * destroyed, if the last reference was dereferenced. If so, the context is
 * invalid and may not be interacted with.
 *
 * @bug When the refcount reaches zero, libinput_unref() releases resources
 * even if a caller still holds refcounted references to related resources
 * (e.g. a libinput_device). When libinput_unref() returns
 * NULL, the caller must consider any resources related to that context
 * invalid. See https://bugs.freedesktop.org/show_bug.cgi?id=91872.
 *
 * Example code:
 * @code
 * li = libinput_path_create_context(&interface, NULL);
 * device = libinput_path_add_device(li, "/dev/input/event0");
 * // get extra reference to device
 * libinput_device_ref(device);
 *
 * // refcount reaches 0, so *all* resources are cleaned up,
 * // including device
 * libinput_unref(li);
 *
 * // INCORRECT: device has been cleaned up and must not be used
 * // li = libinput_device_get_context(device);
 * @endcode
 *
 * @param libinput A previously initialized libinput context
 * @return NULL if context was destroyed otherwise the passed context
 */
struct libinput *
libinput_unref(struct libinput *libinput);

/**
 * @ingroup base
 *
 * Set the log priority for the libinput context. Messages with priorities
 * equal to or higher than the argument will be printed to the context's
 * log handler.
 *
 * The default log priority is @ref LIBINPUT_LOG_PRIORITY_ERROR.
 *
 * @param libinput A previously initialized libinput context
 * @param priority The minimum priority of log messages to print.
 *
 * @see libinput_log_set_handler
 * @see libinput_log_get_priority
 */
void
libinput_log_set_priority(struct libinput *libinput,
			  enum libinput_log_priority priority);

/**
 * @ingroup base
 *
 * Get the context's log priority. Messages with priorities equal to or
 * higher than the argument will be printed to the current log handler.
 *
 * The default log priority is @ref LIBINPUT_LOG_PRIORITY_ERROR.
 *
 * @param libinput A previously initialized libinput context
 * @return The minimum priority of log messages to print.
 *
 * @see libinput_log_set_handler
 * @see libinput_log_set_priority
 */
enum libinput_log_priority
libinput_log_get_priority(const struct libinput *libinput);

/**
 * @ingroup base
 *
 * Log handler type for custom logging.
 *
 * @param libinput The libinput context
 * @param priority The priority of the current message
 * @param format Message format in printf-style
 * @param args Message arguments
 *
 * @see libinput_log_set_priority
 * @see libinput_log_get_priority
 * @see libinput_log_set_handler
 */
typedef void (*libinput_log_handler)(struct libinput *libinput,
				     enum libinput_log_priority priority,
				     const char *format, va_list args)
	   LIBINPUT_ATTRIBUTE_PRINTF(3, 0);

/**
 * @ingroup base
 *
 * Set the context's log handler. Messages with priorities equal to or
 * higher than the context's log priority will be passed to the given
 * log handler.
 *
 * The default log handler prints to stderr.
 *
 * @param libinput A previously initialized libinput context
 * @param log_handler The log handler for library messages.
 *
 * @see libinput_log_set_priority
 * @see libinput_log_get_priority
 */
void
libinput_log_set_handler(struct libinput *libinput,
			 libinput_log_handler log_handler);

/**
 * @defgroup seat Initialization and manipulation of seats
 *
 * A seat has two identifiers, the physical name and the logical name. A
 * device is always assigned to exactly one seat. It may change to a
 * different logical seat but it cannot change physical seats.
 *
 * See the libinput documentation for more information on seats.
 */

/**
 * @ingroup seat
 *
 * Increase the refcount of the seat. A seat will be freed whenever the
 * refcount reaches 0. This may happen during libinput_dispatch() if the
 * seat was removed from the system. A caller must ensure to reference
 * the seat correctly to avoid dangling pointers.
 *
 * @param seat A previously obtained seat
 * @return The passed seat
 */
struct libinput_seat *
libinput_seat_ref(struct libinput_seat *seat);

/**
 * @ingroup seat
 *
 * Decrease the refcount of the seat. A seat will be freed whenever the
 * refcount reaches 0. This may happen during libinput_dispatch() if the
 * seat was removed from the system. A caller must ensure to reference
 * the seat correctly to avoid dangling pointers.
 *
 * @param seat A previously obtained seat
 * @return NULL if seat was destroyed, otherwise the passed seat
 */
struct libinput_seat *
libinput_seat_unref(struct libinput_seat *seat);

/**
 * @ingroup seat
 *
 * Set caller-specific data associated with this seat. libinput does
 * not manage, look at, or modify this data. The caller must ensure the
 * data is valid.
 *
 * @param seat A previously obtained seat
 * @param user_data Caller-specific data pointer
 * @see libinput_seat_get_user_data
 */
void
libinput_seat_set_user_data(struct libinput_seat *seat, void *user_data);

/**
 * @ingroup seat
 *
 * Get the caller-specific data associated with this seat, if any.
 *
 * @param seat A previously obtained seat
 * @return Caller-specific data pointer or NULL if none was set
 * @see libinput_seat_set_user_data
 */
void *
libinput_seat_get_user_data(struct libinput_seat *seat);

/**
 * @ingroup seat
 *
 * Get the libinput context from the seat.
 *
 * @param seat A previously obtained seat
 * @return The libinput context for this seat.
 */
struct libinput *
libinput_seat_get_context(struct libinput_seat *seat);

/**
 * @ingroup seat
 *
 * Return the physical name of the seat. For libinput contexts created from
 * udev, this is always the same value as passed into
 * libinput_udev_assign_seat() and all seats from that context will have
 * the same physical name.
 *
 * The physical name of the seat is one that is usually set by the system or
 * lower levels of the stack. In most cases, this is the base filter for
 * devices - devices assigned to seats outside the current seat will not
 * be available to the caller.
 *
 * @param seat A previously obtained seat
 * @return The physical name of this seat
 */
const char *
libinput_seat_get_physical_name(struct libinput_seat *seat);

/**
 * @ingroup seat
 *
 * Return the logical name of the seat. This is an identifier to group sets
 * of devices within the compositor.
 *
 * @param seat A previously obtained seat
 * @return The logical name of this seat
 */
const char *
libinput_seat_get_logical_name(struct libinput_seat *seat);

/**
 * @defgroup device Initialization and manipulation of input devices
 */

/**
 * @ingroup device
 *
 * Increase the refcount of the input device. An input device will be freed
 * whenever the refcount reaches 0. This may happen during
 * libinput_dispatch() if the device was removed from the system. A caller
 * must ensure to reference the device correctly to avoid dangling pointers.
 *
 * @param device A previously obtained device
 * @return The passed device
 */
struct libinput_device *
libinput_device_ref(struct libinput_device *device);

/**
 * @ingroup device
 *
 * Decrease the refcount of the input device. An input device will be freed
 * whenever the refcount reaches 0. This may happen during libinput_dispatch
 * if the device was removed from the system. A caller must ensure to
 * reference the device correctly to avoid dangling pointers.
 *
 * @param device A previously obtained device
 * @return NULL if the device was destroyed, otherwise the passed device
 */
struct libinput_device *
libinput_device_unref(struct libinput_device *device);

/**
 * @ingroup device
 *
 * Set caller-specific data associated with this input device. libinput does
 * not manage, look at, or modify this data. The caller must ensure the
 * data is valid.
 *
 * @param device A previously obtained device
 * @param user_data Caller-specific data pointer
 * @see libinput_device_get_user_data
 */
void
libinput_device_set_user_data(struct libinput_device *device, void *user_data);

/**
 * @ingroup device
 *
 * Get the caller-specific data associated with this input device, if any.
 *
 * @param device A previously obtained device
 * @return Caller-specific data pointer or NULL if none was set
 * @see libinput_device_set_user_data
 */
void *
libinput_device_get_user_data(struct libinput_device *device);

/**
 * @ingroup device
 *
 * Get the libinput context from the device.
 *
 * @param device A previously obtained device
 * @return The libinput context for this device.
 */
struct libinput *
libinput_device_get_context(struct libinput_device *device);

/**
 * @ingroup device
 *
 * Get the device group this device is assigned to. Some physical
 * devices like graphics tablets are represented by multiple kernel
 * devices and thus by multiple struct @ref libinput_device.
 *
 * libinput assigns these devices to the same @ref libinput_device_group
 * allowing the caller to identify such devices and adjust configuration
 * settings accordingly. For example, setting a tablet to left-handed often
 * means turning it upside down. A touch device on the same tablet would
 * need to be turned upside down too to work correctly.
 *
 * All devices are part of a device group though for most devices the group
 * will be a singleton. A device is assigned to a device group on @ref
 * LIBINPUT_EVENT_DEVICE_ADDED and removed from that group on @ref
 * LIBINPUT_EVENT_DEVICE_REMOVED. It is up to the caller to track how many
 * devices are in each device group.
 *
 * @dot
 * digraph groups_libinput {
 *   rankdir="TB";
 *   node [
 *     shape="box";
 *   ]
 *
 *   mouse [ label="mouse"; URL="\ref libinput_device"];
 *   kbd [ label="keyboard"; URL="\ref libinput_device"];
 *
 *   pen [ label="tablet pen"; URL="\ref libinput_device"];
 *   touch [ label="tablet touch"; URL="\ref libinput_device"];
 *   pad [ label="tablet pad"; URL="\ref libinput_device"];
 *
 *   group1 [ label="group 1"; URL="\ref libinput_device_group"];
 *   group2 [ label="group 2"; URL="\ref libinput_device_group"];
 *   group3 [ label="group 3"; URL="\ref libinput_device_group"];
 *
 *   mouse -> group1
 *   kbd -> group2
 *
 *   pen -> group3;
 *   touch -> group3;
 *   pad -> group3;
 * }
 * @enddot
 *
 * Device groups do not get re-used once the last device in the group was
 * removed, i.e. unplugging and re-plugging a physical device with grouped
 * devices will return a different device group after every unplug.
 *
 * The returned device group is not refcounted and may become invalid after
 * the next call to libinput. Use libinput_device_group_ref() and
 * libinput_device_group_unref() to continue using the handle outside of the
 * immediate scope.
 *
 * Device groups are assigned based on the <b>LIBINPUT_DEVICE_GROUP</b> udev
 * property, see the libinput documentation for more details.
 *
 * @return The device group this device belongs to
 */
struct libinput_device_group *
libinput_device_get_device_group(struct libinput_device *device);

/**
 * @ingroup device
 *
 * Get the system name of the device.
 *
 * To get the descriptive device name, use libinput_device_get_name().
 *
 * @param device A previously obtained device
 * @return System name of the device
 *
 */
const char *
libinput_device_get_sysname(struct libinput_device *device);

/**
 * @ingroup device
 *
 * The descriptive device name as advertised by the kernel and/or the
 * hardware itself. To get the sysname for this device, use
 * libinput_device_get_sysname().
 *
 * The lifetime of the returned string is tied to the struct
 * libinput_device. The string may be the empty string but is never NULL.
 *
 * @param device A previously obtained device
 * @return The device name
 */
const char *
libinput_device_get_name(struct libinput_device *device);

/**
 * @ingroup device
 *
 * Get the product ID for this device.
 *
 * @param device A previously obtained device
 * @return The product ID of this device
 */
unsigned int
libinput_device_get_id_product(struct libinput_device *device);

/**
 * @ingroup device
 *
 * Get the vendor ID for this device.
 *
 * @param device A previously obtained device
 * @return The vendor ID of this device
 */
unsigned int
libinput_device_get_id_vendor(struct libinput_device *device);

/**
 * @ingroup device
 *
 * A device may be mapped to a single output, or all available outputs. If a
 * device is mapped to a single output only, a relative device may not move
 * beyond the boundaries of this output. An absolute device has its input
 * coordinates mapped to the extents of this output.
 *
 * @note <b>Use of this function is discouraged.</b> Its return value is not
 * precisely defined and may not be understood by the caller or may be
 * insufficient to map the device. Instead, the system configuration could
 * set a udev property the caller understands and interprets correctly. The
 * caller could then obtain device with libinput_device_get_udev_device()
 * and query it for this property. For more complex cases, the caller
 * must implement monitor-to-device association heuristics.
 *
 * @return The name of the output this device is mapped to, or NULL if no
 * output is set
 */
const char *
libinput_device_get_output_name(struct libinput_device *device);

/**
 * @ingroup device
 *
 * Get the seat associated with this input device.
 *
 * A seat can be uniquely identified by the physical and logical seat name.
 * There will ever be only one seat instance with a given physical and logical
 * seat name pair at any given time, but if no external reference is kept, it
 * may be destroyed if no device belonging to it is left.
 *
 * The returned seat is not refcounted and may become invalid after
 * the next call to libinput. Use libinput_seat_ref() and
 * libinput_seat_unref() to continue using the handle outside of the
 * immediate scope.
 *
 * See the libinput documentation for more information on seats.
 *
 * @param device A previously obtained device
 * @return The seat this input device belongs to
 */
struct libinput_seat *
libinput_device_get_seat(struct libinput_device *device);

/**
 * @ingroup device
 *
 * Change the logical seat associated with this device by removing the
 * device and adding it to the new seat.
 *
 * This command is identical to physically unplugging the device, then
 * re-plugging it as a member of the new seat. libinput will generate a
 * @ref LIBINPUT_EVENT_DEVICE_REMOVED event and this @ref libinput_device is
 * considered removed from the context; it will not generate further events
 * and will be freed when the refcount reaches zero.
 * A @ref LIBINPUT_EVENT_DEVICE_ADDED event is generated with a new @ref
 * libinput_device handle. It is the caller's responsibility to update
 * references to the new device accordingly.
 *
 * If the logical seat name already exists in the device's physical seat,
 * the device is added to this seat. Otherwise, a new seat is created.
 *
 * @note This change applies to this device until removal or @ref
 * libinput_suspend(), whichever happens earlier.
 *
 * @param device A previously obtained device
 * @param name The new logical seat name
 * @return 0 on success, non-zero on error
 */
int
libinput_device_set_seat_logical_name(struct libinput_device *device,
				      const char *name);

/**
 * @ingroup device
 *
 * Return a udev handle to the device that is this libinput device, if any.
 * The returned handle has a refcount of at least 1, the caller must call
 * <i>udev_device_unref()</i> once to release the associated resources.
 * See the [libudev documentation]
 * (http://www.freedesktop.org/software/systemd/libudev/) for details.
 *
 * Some devices may not have a udev device, or the udev device may be
 * unobtainable. This function returns NULL if no udev device was available.
 *
 * Calling this function multiple times for the same device may not
 * return the same udev handle each time.
 *
 * @param device A previously obtained device
 * @return A udev handle to the device with a refcount of >= 1 or NULL.
 * @retval NULL This device is not represented by a udev device
 */
struct udev_device *
libinput_device_get_udev_device(struct libinput_device *device);

/**
 * @ingroup device
 *
 * Update the LEDs on the device, if any. If the device does not have
 * LEDs, or does not have one or more of the LEDs given in the mask, this
 * function does nothing.
 *
 * @param device A previously obtained device
 * @param leds A mask of the LEDs to set, or unset.
 */
void
libinput_device_led_update(struct libinput_device *device,
			   enum libinput_led leds);

/**
 * @ingroup device
 *
 * Check if the given device has the specified capability
 *
 * @return Non-zero if the given device has the capability or zero otherwise
 */
int
libinput_device_has_capability(struct libinput_device *device,
			       enum libinput_device_capability capability);

/**
 * @ingroup device
 *
 * Get the physical size of a device in mm, where meaningful. This function
 * only succeeds on devices with the required data, i.e. tablets, touchpads
 * and touchscreens.
 *
 * If this function returns nonzero, width and height are unmodified.
 *
 * @param device The device
 * @param width Set to the width of the device
 * @param height Set to the height of the device
 * @return 0 on success, or nonzero otherwise
 */
int
libinput_device_get_size(struct libinput_device *device,
			 double *width,
			 double *height);

/**
 * @ingroup device
 *
 * Check if a @ref LIBINPUT_DEVICE_CAP_POINTER device has a button with the
 * given code (see linux/input-event-codes.h).
 *
 * @param device A current input device
 * @param code Button code to check for, e.g. <i>BTN_LEFT</i>
 *
 * @return 1 if the device supports this button code, 0 if it does not, -1
 * on error.
 */
int
libinput_device_pointer_has_button(struct libinput_device *device, uint32_t code);

/**
 * @ingroup device
 *
 * Check if a @ref LIBINPUT_DEVICE_CAP_KEYBOARD device has a key with the
 * given code (see linux/input-event-codes.h).
 *
 * @param device A current input device
 * @param code Key code to check for, e.g. <i>KEY_ESC</i>
 *
 * @return 1 if the device supports this key code, 0 if it does not, -1
 * on error.
 */
int
libinput_device_keyboard_has_key(struct libinput_device *device,
				 uint32_t code);

/**
 * @ingroup device
 *
 * Check how many touches a @ref LIBINPUT_DEVICE_CAP_TOUCH device supports
 * simultaneously.
 *
 * @param device A current input device
 *
 * @return The number of simultaneous touches or 0 if unknown, -1
 * on error.
 *
 * @since 1.11
 */
int
libinput_device_touch_get_touch_count(struct libinput_device *device);

/**
 * @ingroup device
 *
 * Check if a @ref LIBINPUT_DEVICE_CAP_SWITCH device has a switch of the
 * given type.
 *
 * @param device A current input device
 * @param sw Switch to check for
 *
 * @return 1 if the device supports this switch, 0 if it does not, -1
 * on error.
 *
 * @since 1.9
 */
int
libinput_device_switch_has_switch(struct libinput_device *device,
				  enum libinput_switch sw);

/**
 * @ingroup device
 *
 * Return the number of buttons on a device with the
 * @ref LIBINPUT_DEVICE_CAP_TABLET_PAD capability.
 * Buttons on a pad device are numbered sequentially, see the
 * libinput documentation for details.
 *
 * @param device A current input device
 *
 * @return The number of buttons supported by the device.
 *
 * @since 1.3
 */
int
libinput_device_tablet_pad_get_num_buttons(struct libinput_device *device);

/**
 * @ingroup device
 *
 * Return the number of rings a device with the @ref
 * LIBINPUT_DEVICE_CAP_TABLET_PAD capability provides.
 *
 * @param device A current input device
 *
 * @return The number of rings or 0 if the device has no rings.
 *
 * @see libinput_event_tablet_pad_get_ring_number
 *
 * @since 1.3
 */
int
libinput_device_tablet_pad_get_num_rings(struct libinput_device *device);

/**
 * @ingroup device
 *
 * Return the number of strips a device with the @ref
 * LIBINPUT_DEVICE_CAP_TABLET_PAD capability provides.
 *
 * @param device A current input device
 *
 * @return The number of strips or 0 if the device has no strips.
 *
 * @see libinput_event_tablet_pad_get_strip_number
 *
 * @since 1.3
 */
int
libinput_device_tablet_pad_get_num_strips(struct libinput_device *device);

/**
 * @ingroup device
 *
 * Check if a @ref LIBINPUT_DEVICE_CAP_TABLET_PAD device has a key with the
 * given code (see linux/input-event-codes.h).
 *
 * @param device A current input device
 * @param code Key code to check for, e.g. <i>KEY_ESC</i>
 *
 * @return 1 if the device supports this key code, 0 if it does not, -1
 * on error.
 *
 * @since 1.15
 */
int
libinput_device_tablet_pad_has_key(struct libinput_device *device,
				   uint32_t code);

/**
 * @ingroup device
 *
 * Increase the refcount of the device group. A device group will be freed
 * whenever the refcount reaches 0. This may happen during
 * libinput_dispatch() if all devices of this group were removed from the
 * system. A caller must ensure to reference the device group correctly to
 * avoid dangling pointers.
 *
 * @param group A previously obtained device group
 * @return The passed device group
 */
struct libinput_device_group *
libinput_device_group_ref(struct libinput_device_group *group);

/**
 * @ingroup device
 *
 * Decrease the refcount of the device group. A device group will be freed
 * whenever the refcount reaches 0. This may happen during
 * libinput_dispatch() if all devices of this group were removed from the
 * system. A caller must ensure to reference the device group correctly to
 * avoid dangling pointers.
 *
 * @param group A previously obtained device group
 * @return NULL if the device group was destroyed, otherwise the passed
 * device group
 */
struct libinput_device_group *
libinput_device_group_unref(struct libinput_device_group *group);

/**
 * @ingroup device
 *
 * Set caller-specific data associated with this device group. libinput does
 * not manage, look at, or modify this data. The caller must ensure the
 * data is valid.
 *
 * @param group A previously obtained device group
 * @param user_data Caller-specific data pointer
 * @see libinput_device_group_get_user_data
 */
void
libinput_device_group_set_user_data(struct libinput_device_group *group,
				    void *user_data);

/**
 * @ingroup device
 *
 * Get the caller-specific data associated with this input device group, if
 * any.
 *
 * @param group A previously obtained group
 * @return Caller-specific data pointer or NULL if none was set
 * @see libinput_device_group_set_user_data
 */
void *
libinput_device_group_get_user_data(struct libinput_device_group *group);

/**
 * @defgroup config Device configuration
 *
 * Enable, disable, change and/or check for device-specific features. For
 * all features, libinput assigns a default based on the hardware
 * configuration. This default can be obtained with the respective
 * get_default call.
 *
 * Configuration options are device dependent and not all options are
 * supported on all devices. For all configuration options, libinput
 * provides a call to check if a configuration option is available on a
 * device (e.g. libinput_device_config_calibration_has_matrix())
 *
 * Some configuration option may be dependent on or mutually exclusive with
 * with other options. The behavior in those cases is
 * implementation-dependent, the caller must ensure that the options are set
 * in the right order.
 *
 * Below is a general grouping of configuration options according to device
 * type. Note that this is a guide only and not indicative of any specific
 * device.
 * - Touchpad:
 *    - libinput_device_config_tap_set_enabled()
 *    - libinput_device_config_tap_set_drag_enabled()
 *    - libinput_device_config_tap_set_drag_lock_enabled()
 *    - libinput_device_config_click_set_method()
 *    - libinput_device_config_scroll_set_method()
 *    - libinput_device_config_dwt_set_enabled()
 * - Touchscreens:
 *    - libinput_device_config_calibration_set_matrix()
 * - Pointer devices (mice, trackballs, touchpads):
 *    - libinput_device_config_accel_set_speed()
 *    - libinput_device_config_accel_set_profile()
 *    - libinput_device_config_scroll_set_natural_scroll_enabled()
 *    - libinput_device_config_left_handed_set()
 *    - libinput_device_config_middle_emulation_set_enabled()
 *    - libinput_device_config_rotation_set_angle()
 * - All devices:
 *    - libinput_device_config_send_events_set_mode()
 */

/**
 * @ingroup config
 *
 * Status codes returned when applying configuration settings.
 */
enum libinput_config_status {
	LIBINPUT_CONFIG_STATUS_SUCCESS = 0,	/**< Config applied successfully */
	LIBINPUT_CONFIG_STATUS_UNSUPPORTED,	/**< Configuration not available on
						     this device */
	LIBINPUT_CONFIG_STATUS_INVALID,		/**< Invalid parameter range */
};

/**
 * @ingroup config
 *
 * Return a string describing the error.
 *
 * @param status The status to translate to a string
 * @return A human-readable string representing the error or NULL for an
 * invalid status.
 */
const char *
libinput_config_status_to_str(enum libinput_config_status status);

/**
 * @ingroup config
 */
enum libinput_config_tap_state {
	LIBINPUT_CONFIG_TAP_DISABLED, /**< Tapping is to be disabled, or is
					currently disabled */
	LIBINPUT_CONFIG_TAP_ENABLED, /**< Tapping is to be enabled, or is
				       currently enabled */
};

/**
 * @ingroup config
 *
 * Check if the device supports tap-to-click and how many fingers can be
 * used for tapping. See
 * libinput_device_config_tap_set_enabled() for more information.
 *
 * @param device The device to configure
 * @return The number of fingers that can generate a tap event, or 0 if the
 * device does not support tapping.
 *
 * @see libinput_device_config_tap_set_enabled
 * @see libinput_device_config_tap_get_enabled
 * @see libinput_device_config_tap_get_default_enabled
 */
int
libinput_device_config_tap_get_finger_count(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Enable or disable tap-to-click on this device, with a default mapping of
 * 1, 2, 3 finger tap mapping to left, right, middle click, respectively.
 * Tapping is limited by the number of simultaneous touches
 * supported by the device, see
 * libinput_device_config_tap_get_finger_count().
 *
 * @param device The device to configure
 * @param enable @ref LIBINPUT_CONFIG_TAP_ENABLED to enable tapping or @ref
 * LIBINPUT_CONFIG_TAP_DISABLED to disable tapping
 *
 * @return A config status code. Disabling tapping on a device that does not
 * support tapping always succeeds.
 *
 * @see libinput_device_config_tap_get_finger_count
 * @see libinput_device_config_tap_get_enabled
 * @see libinput_device_config_tap_get_default_enabled
 */
enum libinput_config_status
libinput_device_config_tap_set_enabled(struct libinput_device *device,
				       enum libinput_config_tap_state enable);

/**
 * @ingroup config
 *
 * Check if tap-to-click is enabled on this device. If the device does not
 * support tapping, this function always returns @ref
 * LIBINPUT_CONFIG_TAP_DISABLED.
 *
 * @param device The device to configure
 *
 * @retval LIBINPUT_CONFIG_TAP_ENABLED If tapping is currently enabled
 * @retval LIBINPUT_CONFIG_TAP_DISABLED If tapping is currently disabled
 *
 * @see libinput_device_config_tap_get_finger_count
 * @see libinput_device_config_tap_set_enabled
 * @see libinput_device_config_tap_get_default_enabled
 */
enum libinput_config_tap_state
libinput_device_config_tap_get_enabled(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Return the default setting for whether tap-to-click is enabled on this
 * device.
 *
 * @param device The device to configure
 * @retval LIBINPUT_CONFIG_TAP_ENABLED If tapping is enabled by default
 * @retval LIBINPUT_CONFIG_TAP_DISABLED If tapping Is disabled by default
 *
 * @see libinput_device_config_tap_get_finger_count
 * @see libinput_device_config_tap_set_enabled
 * @see libinput_device_config_tap_get_enabled
 */
enum libinput_config_tap_state
libinput_device_config_tap_get_default_enabled(struct libinput_device *device);

/**
 * @ingroup config
 *
 * @since 1.5
 */
enum libinput_config_tap_button_map {
	/** 1/2/3 finger tap maps to left/right/middle */
	LIBINPUT_CONFIG_TAP_MAP_LRM,
	/** 1/2/3 finger tap maps to left/middle/right*/
	LIBINPUT_CONFIG_TAP_MAP_LMR,
};

/**
 * @ingroup config
 *
 * Set the finger number to button number mapping for tap-to-click. The
 * default mapping on most devices is to have a 1, 2 and 3 finger tap to map
 * to the left, right and middle button, respectively.
 * A device may permit changing the button mapping but disallow specific
 * maps. In this case @ref LIBINPUT_CONFIG_STATUS_UNSUPPORTED is returned,
 * the caller is expected to handle this case correctly.
 *
 * Changing the button mapping may not take effect immediately,
 * the device may wait until it is in a neutral state before applying any
 * changes.
 *
 * The mapping may be changed when tap-to-click is disabled. The new mapping
 * takes effect when tap-to-click is enabled in the future.
 *
 * @note It is an application bug to call this function for devices where
 * libinput_device_config_tap_get_finger_count() returns 0.
 *
 * @param device The device to configure
 * @param map The new finger-to-button number mapping
 * @return A config status code. Changing the order on a device that does not
 * support tapping always fails with @ref LIBINPUT_CONFIG_STATUS_UNSUPPORTED.
 *
 * @see libinput_device_config_tap_get_button_map
 * @see libinput_device_config_tap_get_default_button_map
 *
 * @since 1.5
 */
enum libinput_config_status
libinput_device_config_tap_set_button_map(struct libinput_device *device,
					    enum libinput_config_tap_button_map map);

/**
 * @ingroup config
 *
 * Get the finger number to button number mapping for tap-to-click.
 *
 * The return value for a device that does not support tapping is always
 * @ref LIBINPUT_CONFIG_TAP_MAP_LRM.
 *
 * @note It is an application bug to call this function for devices where
 * libinput_device_config_tap_get_finger_count() returns 0.
 *
 * @param device The device to configure
 * @return The current finger-to-button number mapping
 *
 * @see libinput_device_config_tap_set_button_map
 * @see libinput_device_config_tap_get_default_button_map
 *
 * @since 1.5
 */
enum libinput_config_tap_button_map
libinput_device_config_tap_get_button_map(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Get the default finger number to button number mapping for tap-to-click.
 *
 * The return value for a device that does not support tapping is always
 * @ref LIBINPUT_CONFIG_TAP_MAP_LRM.
 *
 * @note It is an application bug to call this function for devices where
 * libinput_device_config_tap_get_finger_count() returns 0.
 *
 * @param device The device to configure
 * @return The current finger-to-button number mapping
 *
 * @see libinput_device_config_tap_set_button_map
 * @see libinput_device_config_tap_get_default_button_map
 *
 * @since 1.5
 */
enum libinput_config_tap_button_map
libinput_device_config_tap_get_default_button_map(struct libinput_device *device);

/**
 * @ingroup config
 *
 * A config status to distinguish or set dragging on a device. Currently
 * implemented for tap-and-drag only, see
 * libinput_device_config_tap_set_drag_enabled()
 *
 * @since 1.2
 */
enum libinput_config_drag_state {
	/**
	 * Drag is to be disabled, or is
	 * currently disabled.
	 */
	LIBINPUT_CONFIG_DRAG_DISABLED,
	/**
	 * Drag is to be enabled, or is
	 * currently enabled
	 */
	LIBINPUT_CONFIG_DRAG_ENABLED,
};

/**
 * @ingroup config
 *
 * Enable or disable tap-and-drag on this device. When enabled, a
 * tap immediately followed by a finger down results in a button down event,
 * subsequent finger motion thus triggers a drag. The button is released
 * on finger up. See the libinput documentation for more details.
 *
 * @param device The device to configure
 * @param enable @ref LIBINPUT_CONFIG_DRAG_ENABLED to enable, @ref
 * LIBINPUT_CONFIG_DRAG_DISABLED to disable tap-and-drag
 *
 * @see libinput_device_config_tap_drag_get_enabled
 * @see libinput_device_config_tap_drag_get_default_enabled
 *
 * @since 1.2
 */
enum libinput_config_status
libinput_device_config_tap_set_drag_enabled(struct libinput_device *device,
					    enum libinput_config_drag_state enable);

/**
 * @ingroup config
 *
 * Return whether tap-and-drag is enabled or disabled on this device.
 *
 * @param device The device to check
 * @retval LIBINPUT_CONFIG_DRAG_ENABLED if tap-and-drag is enabled
 * @retval LIBINPUT_CONFIG_DRAG_DISABLED if tap-and-drag is
 * disabled
 *
 * @see libinput_device_config_tap_drag_set_enabled
 * @see libinput_device_config_tap_drag_get_default_enabled
 *
 * @since 1.2
 */
enum libinput_config_drag_state
libinput_device_config_tap_get_drag_enabled(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Return whether tap-and-drag is enabled or disabled by default on this
 * device.
 *
 * @param device The device to check
 * @retval LIBINPUT_CONFIG_DRAG_ENABLED if tap-and-drag is enabled by
 * default
 * @retval LIBINPUT_CONFIG_DRAG_DISABLED if tap-and-drag is
 * disabled by default
 *
 * @see libinput_device_config_tap_drag_set_enabled
 * @see libinput_device_config_tap_drag_get_enabled
 *
 * @since 1.2
 */
enum libinput_config_drag_state
libinput_device_config_tap_get_default_drag_enabled(struct libinput_device *device);

/**
 * @ingroup config
 */
enum libinput_config_drag_lock_state {
	/** Drag lock is to be disabled, or is currently disabled */
	LIBINPUT_CONFIG_DRAG_LOCK_DISABLED,
	/** Drag lock is to be enabled, or is currently disabled */
	LIBINPUT_CONFIG_DRAG_LOCK_ENABLED,
};

/**
 * @ingroup config
 *
 * Enable or disable drag-lock during tapping on this device. When enabled,
 * a finger may be lifted and put back on the touchpad within a timeout and
 * the drag process continues. When disabled, lifting the finger during a
 * tap-and-drag will immediately stop the drag. See the libinput
 * documentation for more details.
 *
 * Enabling drag lock on a device that has tapping disabled is permitted,
 * but has no effect until tapping is enabled.
 *
 * @param device The device to configure
 * @param enable @ref LIBINPUT_CONFIG_DRAG_LOCK_ENABLED to enable drag lock
 * or @ref LIBINPUT_CONFIG_DRAG_LOCK_DISABLED to disable drag lock
 *
 * @return A config status code. Disabling drag lock on a device that does not
 * support tapping always succeeds.
 *
 * @see libinput_device_config_tap_get_drag_lock_enabled
 * @see libinput_device_config_tap_get_default_drag_lock_enabled
 */
enum libinput_config_status
libinput_device_config_tap_set_drag_lock_enabled(struct libinput_device *device,
						 enum libinput_config_drag_lock_state enable);

/**
 * @ingroup config
 *
 * Check if drag-lock during tapping is enabled on this device. If the
 * device does not support tapping, this function always returns
 * @ref LIBINPUT_CONFIG_DRAG_LOCK_DISABLED.
 *
 * Drag lock may be enabled even when tapping is disabled.
 *
 * @param device The device to configure
 *
 * @retval LIBINPUT_CONFIG_DRAG_LOCK_ENABLED If drag lock is currently enabled
 * @retval LIBINPUT_CONFIG_DRAG_LOCK_DISABLED If drag lock is currently disabled
 *
 * @see libinput_device_config_tap_set_drag_lock_enabled
 * @see libinput_device_config_tap_get_default_drag_lock_enabled
 */
enum libinput_config_drag_lock_state
libinput_device_config_tap_get_drag_lock_enabled(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Check if drag-lock during tapping is enabled by default on this device.
 * If the device does not support tapping, this function always returns
 * @ref LIBINPUT_CONFIG_DRAG_LOCK_DISABLED.
 *
 * Drag lock may be enabled by default even when tapping is disabled by
 * default.
 *
 * @param device The device to configure
 *
 * @retval LIBINPUT_CONFIG_DRAG_LOCK_ENABLED If drag lock is enabled by
 * default
 * @retval LIBINPUT_CONFIG_DRAG_LOCK_DISABLED If drag lock is disabled by
 * default
 *
 * @see libinput_device_config_tap_set_drag_lock_enabled
 * @see libinput_device_config_tap_get_drag_lock_enabled
 */
enum libinput_config_drag_lock_state
libinput_device_config_tap_get_default_drag_lock_enabled(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Check if the device can be calibrated via a calibration matrix.
 *
 * @param device The device to check
 * @return Non-zero if the device can be calibrated, zero otherwise.
 *
 * @see libinput_device_config_calibration_set_matrix
 * @see libinput_device_config_calibration_get_matrix
 * @see libinput_device_config_calibration_get_default_matrix
 */
int
libinput_device_config_calibration_has_matrix(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Apply the 3x3 transformation matrix to absolute device coordinates. This
 * matrix has no effect on relative events.
 *
 * Given a 6-element array [a, b, c, d, e, f], the matrix is applied as
 * @code
 * [ a  b  c ]   [ x ]
 * [ d  e  f ] * [ y ]
 * [ 0  0  1 ]   [ 1 ]
 * @endcode
 *
 * The translation component (c, f) is expected to be normalized to the
 * device coordinate range. For example, the matrix
 * @code
 * [ 1 0  1 ]
 * [ 0 1 -1 ]
 * [ 0 0  1 ]
 * @endcode
 * moves all coordinates by 1 device-width to the right and 1 device-height
 * up.
 *
 * The rotation matrix for rotation around the origin is defined as
 * @code
 * [ cos(a) -sin(a) 0 ]
 * [ sin(a)  cos(a) 0 ]
 * [   0      0     1 ]
 * @endcode
 * Note that any rotation requires an additional translation component to
 * translate the rotated coordinates back into the original device space.
 * The rotation matrixes for 90, 180 and 270 degrees clockwise are:
 * @code
 * 90 deg cw:		180 deg cw:		270 deg cw:
 * [ 0 -1 1]		[ -1  0 1]		[  0 1 0 ]
 * [ 1  0 0]		[  0 -1 1]		[ -1 0 1 ]
 * [ 0  0 1]		[  0  0 1]		[  0 0 1 ]
 * @endcode
 *
 * @param device The device to configure
 * @param matrix An array representing the first two rows of a 3x3 matrix as
 * described above.
 *
 * @return A config status code.
 *
 * @see libinput_device_config_calibration_has_matrix
 * @see libinput_device_config_calibration_get_matrix
 * @see libinput_device_config_calibration_get_default_matrix
 */
enum libinput_config_status
libinput_device_config_calibration_set_matrix(struct libinput_device *device,
					      const float matrix[6]);

/**
 * @ingroup config
 *
 * Return the current calibration matrix for this device.
 *
 * @param device The device to configure
 * @param matrix Set to the array representing the first two rows of a 3x3 matrix as
 * described in libinput_device_config_calibration_set_matrix().
 *
 * @return 0 if no calibration is set and the returned matrix is the
 * identity matrix, 1 otherwise
 *
 * @see libinput_device_config_calibration_has_matrix
 * @see libinput_device_config_calibration_set_matrix
 * @see libinput_device_config_calibration_get_default_matrix
 */
int
libinput_device_config_calibration_get_matrix(struct libinput_device *device,
					      float matrix[6]);

/**
 * @ingroup config
 *
 * Return the default calibration matrix for this device. On most devices,
 * this is the identity matrix. If the udev property
 * <b>LIBINPUT_CALIBRATION_MATRIX</b> is set on the respective udev device,
 * that property's value becomes the default matrix, see the libinput
 * documentation for more details.
 *
 * @param device The device to configure
 * @param matrix Set to the array representing the first two rows of a 3x3 matrix as
 * described in libinput_device_config_calibration_set_matrix().
 *
 * @return 0 if no calibration is set and the returned matrix is the
 * identity matrix, 1 otherwise
 *
 * @see libinput_device_config_calibration_has_matrix
 * @see libinput_device_config_calibration_set_matrix
 * @see libinput_device_config_calibration_get_matrix
 */
int
libinput_device_config_calibration_get_default_matrix(struct libinput_device *device,
						      float matrix[6]);

/**
 * @ingroup config
 *
 * The send-event mode of a device defines when a device may generate events
 * and pass those events to the caller.
 */
enum libinput_config_send_events_mode {
	/**
	 * Send events from this device normally. This is a placeholder
	 * mode only, any device detected by libinput can be enabled. Do not
	 * test for this value as bitmask.
	 */
	LIBINPUT_CONFIG_SEND_EVENTS_ENABLED = 0,
	/**
	 * Do not send events through this device. Depending on the device,
	 * this may close all file descriptors on the device or it may leave
	 * the file descriptors open and route events through a different
	 * device.
	 *
	 * If this bit field is set, other disable modes may be
	 * ignored. For example, if both @ref
	 * LIBINPUT_CONFIG_SEND_EVENTS_DISABLED and @ref
	 * LIBINPUT_CONFIG_SEND_EVENTS_DISABLED_ON_EXTERNAL_MOUSE are set,
	 * the device remains disabled when all external pointer devices are
	 * unplugged.
	 */
	LIBINPUT_CONFIG_SEND_EVENTS_DISABLED = (1 << 0),
	/**
	 * If an external pointer device is plugged in, do not send events
	 * from this device. This option may be available on built-in
	 * touchpads.
	 */
	LIBINPUT_CONFIG_SEND_EVENTS_DISABLED_ON_EXTERNAL_MOUSE = (1 << 1),
};

/**
 * @ingroup config
 *
 * Return the possible send-event modes for this device. These modes define
 * when a device may process and send events.
 *
 * @param device The device to configure
 *
 * @return A bitmask of possible modes.
 *
 * @see libinput_device_config_send_events_set_mode
 * @see libinput_device_config_send_events_get_mode
 * @see libinput_device_config_send_events_get_default_mode
 */
uint32_t
libinput_device_config_send_events_get_modes(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Set the send-event mode for this device. The mode defines when the device
 * processes and sends events to the caller.
 *
 * The selected mode may not take effect immediately. Events already
 * received and processed from this device are unaffected and will be passed
 * to the caller on the next call to libinput_get_event().
 *
 * If the mode is a bitmask of @ref libinput_config_send_events_mode,
 * the device may wait for or generate events until it is in a neutral
 * state. For example, this may include waiting for or generating button
 * release events.
 *
 * If the device is already suspended, this function does nothing and
 * returns success. Changing the send-event mode on a device that has been
 * removed is permitted.
 *
 * @param device The device to configure
 * @param mode A bitmask of send-events modes
 *
 * @return A config status code.
 *
 * @see libinput_device_config_send_events_get_modes
 * @see libinput_device_config_send_events_get_mode
 * @see libinput_device_config_send_events_get_default_mode
 */
enum libinput_config_status
libinput_device_config_send_events_set_mode(struct libinput_device *device,
					    uint32_t mode);

/**
 * @ingroup config
 *
 * Get the send-event mode for this device. The mode defines when the device
 * processes and sends events to the caller.
 *
 * If a caller enables the bits for multiple modes, some of which are
 * subsets of another mode libinput may drop the bits that are subsets. In
 * other words, don't expect libinput_device_config_send_events_get_mode()
 * to always return exactly the same bitmask as passed into
 * libinput_device_config_send_events_set_mode().
 *
 * @param device The device to configure
 * @return The current bitmask of the send-event mode for this device.
 *
 * @see libinput_device_config_send_events_get_modes
 * @see libinput_device_config_send_events_set_mode
 * @see libinput_device_config_send_events_get_default_mode
 */
uint32_t
libinput_device_config_send_events_get_mode(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Get the default send-event mode for this device. The mode defines when
 * the device processes and sends events to the caller.
 *
 * @param device The device to configure
 * @return The bitmask of the send-event mode for this device.
 *
 * @see libinput_device_config_send_events_get_modes
 * @see libinput_device_config_send_events_set_mode
 * @see libinput_device_config_send_events_get_mode
 */
uint32_t
libinput_device_config_send_events_get_default_mode(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Check if a device uses libinput-internal pointer-acceleration.
 *
 * @param device The device to configure
 *
 * @return 0 if the device is not accelerated, nonzero if it is accelerated
 *
 * @see libinput_device_config_accel_set_speed
 * @see libinput_device_config_accel_get_speed
 * @see libinput_device_config_accel_get_default_speed
 */
int
libinput_device_config_accel_is_available(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Set the pointer acceleration speed of this pointer device within a range
 * of [-1, 1], where 0 is the default acceleration for this device, -1 is
 * the slowest acceleration and 1 is the maximum acceleration available on
 * this device. The actual pointer acceleration mechanism is
 * implementation-dependent, as is the number of steps available within the
 * range. libinput picks the semantically closest acceleration step if the
 * requested value does not match a discrete setting.
 *
 * @param device The device to configure
 * @param speed The normalized speed, in a range of [-1, 1]
 *
 * @return A config status code
 *
 * @see libinput_device_config_accel_is_available
 * @see libinput_device_config_accel_get_speed
 * @see libinput_device_config_accel_get_default_speed
 */
enum libinput_config_status
libinput_device_config_accel_set_speed(struct libinput_device *device,
				       double speed);

/**
 * @ingroup config
 *
 * Get the current pointer acceleration setting for this pointer device. The
 * returned value is normalized to a range of [-1, 1].
 * See libinput_device_config_accel_set_speed() for details.
 *
 * @param device The device to configure
 *
 * @return The current speed, range -1 to 1
 *
 * @see libinput_device_config_accel_is_available
 * @see libinput_device_config_accel_set_speed
 * @see libinput_device_config_accel_get_default_speed
 */
double
libinput_device_config_accel_get_speed(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Return the default speed setting for this device, normalized to a range
 * of [-1, 1].
 * See libinput_device_config_accel_set_speed() for details.
 *
 * @param device The device to configure
 * @return The default speed setting for this device.
 *
 * @see libinput_device_config_accel_is_available
 * @see libinput_device_config_accel_set_speed
 * @see libinput_device_config_accel_get_speed
 */
double
libinput_device_config_accel_get_default_speed(struct libinput_device *device);

/**
 * @ingroup config
 *
 * @since 1.1
 */
enum libinput_config_accel_profile {
	/**
	 * Placeholder for devices that don't have a configurable pointer
	 * acceleration profile.
	 */
	LIBINPUT_CONFIG_ACCEL_PROFILE_NONE = 0,
	/**
	 * A flat acceleration profile. Pointer motion is accelerated by a
	 * constant (device-specific) factor, depending on the current
	 * speed.
	 *
	 * @see libinput_device_config_accel_set_speed
	 */
	LIBINPUT_CONFIG_ACCEL_PROFILE_FLAT = (1 << 0),

	/**
	 * An adaptive acceleration profile. Pointer acceleration depends
	 * on the input speed. This is the default profile for most devices.
	 */
	LIBINPUT_CONFIG_ACCEL_PROFILE_ADAPTIVE = (1 << 1),
};

/**
 * @ingroup config
 *
 * Returns a bitmask of the configurable acceleration modes available on
 * this device.
 *
 * @param device The device to configure
 *
 * @return A bitmask of all configurable modes available on this device.
 *
 * @since 1.1
 */
uint32_t
libinput_device_config_accel_get_profiles(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Set the pointer acceleration profile of this pointer device to the given
 * mode.
 *
 * @param device The device to configure
 * @param profile The profile to set the device to.
 *
 * @return A config status code
 *
 * @since 1.1
 */
enum libinput_config_status
libinput_device_config_accel_set_profile(struct libinput_device *device,
					 enum libinput_config_accel_profile profile);

/**
 * @ingroup config
 *
 * Get the current pointer acceleration profile for this pointer device.
 *
 * @param device The device to configure
 *
 * @return The currently configured pointer acceleration profile.
 *
 * @since 1.1
 */
enum libinput_config_accel_profile
libinput_device_config_accel_get_profile(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Return the default pointer acceleration profile for this pointer device.
 *
 * @param device The device to configure
 *
 * @return The default acceleration profile for this device.
 *
 * @since 1.1
 */
enum libinput_config_accel_profile
libinput_device_config_accel_get_default_profile(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Return non-zero if the device supports "natural scrolling".
 *
 * In traditional scroll mode, the movement of fingers on a touchpad when
 * scrolling matches the movement of the scroll bars. When the fingers move
 * down, the scroll bar moves down, a line of text on the screen moves
 * towards the upper end of the screen. This also matches scroll wheels on
 * mice (wheel down, content moves up).
 *
 * Natural scrolling is the term coined by Apple for inverted scrolling.
 * In this mode, the effect of scrolling movement of fingers on a touchpad
 * resemble physical manipulation of paper. When the fingers move down, a
 * line of text on the screen moves down (scrollbars move up). This is the
 * opposite of scroll wheels on mice.
 *
 * A device supporting natural scrolling can be switched between traditional
 * scroll mode and natural scroll mode.
 *
 * @param device The device to configure
 *
 * @return Zero if natural scrolling is not supported, non-zero if natural
 * scrolling is supported by this device
 *
 * @see libinput_device_config_scroll_set_natural_scroll_enabled
 * @see libinput_device_config_scroll_get_natural_scroll_enabled
 * @see libinput_device_config_scroll_get_default_natural_scroll_enabled
 */
int
libinput_device_config_scroll_has_natural_scroll(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Enable or disable natural scrolling on the device.
 *
 * @param device The device to configure
 * @param enable non-zero to enable, zero to disable natural scrolling
 *
 * @return A config status code
 *
 * @see libinput_device_config_scroll_has_natural_scroll
 * @see libinput_device_config_scroll_get_natural_scroll_enabled
 * @see libinput_device_config_scroll_get_default_natural_scroll_enabled
 */
enum libinput_config_status
libinput_device_config_scroll_set_natural_scroll_enabled(struct libinput_device *device,
							 int enable);
/**
 * @ingroup config
 *
 * Get the current mode for scrolling on this device
 *
 * @param device The device to configure
 *
 * @return Zero if natural scrolling is disabled, non-zero if enabled
 *
 * @see libinput_device_config_scroll_has_natural_scroll
 * @see libinput_device_config_scroll_set_natural_scroll_enabled
 * @see libinput_device_config_scroll_get_default_natural_scroll_enabled
 */
int
libinput_device_config_scroll_get_natural_scroll_enabled(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Get the default mode for scrolling on this device
 *
 * @param device The device to configure
 *
 * @return Zero if natural scrolling is disabled by default, non-zero if enabled
 *
 * @see libinput_device_config_scroll_has_natural_scroll
 * @see libinput_device_config_scroll_set_natural_scroll_enabled
 * @see libinput_device_config_scroll_get_natural_scroll_enabled
 */
int
libinput_device_config_scroll_get_default_natural_scroll_enabled(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Check if a device has a configuration that supports left-handed usage.
 *
 * @param device The device to configure
 * @return Non-zero if the device can be set to left-handed, or zero
 * otherwise
 *
 * @see libinput_device_config_left_handed_set
 * @see libinput_device_config_left_handed_get
 * @see libinput_device_config_left_handed_get_default
 */
int
libinput_device_config_left_handed_is_available(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Set the left-handed configuration of the device.
 *
 * The exact behavior is device-dependent. On a mouse and most pointing
 * devices, left and right buttons are swapped but the middle button is
 * unmodified. On a touchpad, physical buttons (if present) are swapped. On a
 * clickpad, the top and bottom software-emulated buttons are swapped where
 * present, the main area of the touchpad remains a left button. Tapping and
 * clickfinger behavior is not affected by this setting.
 *
 * Changing the left-handed configuration of a device may not take effect
 * until all buttons have been logically released.
 *
 * @param device The device to configure
 * @param left_handed Zero to disable, non-zero to enable left-handed mode
 * @return A configuration status code
 *
 * @see libinput_device_config_left_handed_is_available
 * @see libinput_device_config_left_handed_get
 * @see libinput_device_config_left_handed_get_default
 */
enum libinput_config_status
libinput_device_config_left_handed_set(struct libinput_device *device,
				       int left_handed);

/**
 * @ingroup config
 *
 * Get the current left-handed configuration of the device.
 *
 * @param device The device to configure
 * @return Zero if the device is in right-handed mode, non-zero if the
 * device is in left-handed mode
 *
 * @see libinput_device_config_left_handed_is_available
 * @see libinput_device_config_left_handed_set
 * @see libinput_device_config_left_handed_get_default
 */
int
libinput_device_config_left_handed_get(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Get the default left-handed configuration of the device.
 *
 * @param device The device to configure
 * @return Zero if the device is in right-handed mode by default, or non-zero
 * if the device is in left-handed mode by default
 *
 * @see libinput_device_config_left_handed_is_available
 * @see libinput_device_config_left_handed_set
 * @see libinput_device_config_left_handed_get
 */
int
libinput_device_config_left_handed_get_default(struct libinput_device *device);

/**
 * @ingroup config
 *
 * The click method defines when to generate software-emulated
 * buttons, usually on a device that does not have a specific physical
 * button available.
 */
enum libinput_config_click_method {
	/**
	 * Do not send software-emulated button events. This has no effect
	 * on events generated by physical buttons.
	 */
	LIBINPUT_CONFIG_CLICK_METHOD_NONE = 0,
	/**
	 * Use software-button areas to generate button events.
	 */
	LIBINPUT_CONFIG_CLICK_METHOD_BUTTON_AREAS = (1 << 0),
	/**
	 * The number of fingers decides which button press to generate.
	 */
	LIBINPUT_CONFIG_CLICK_METHOD_CLICKFINGER = (1 << 1),
};

/**
 * @ingroup config
 *
 * Check which button click methods a device supports. The button click
 * method defines when to generate software-emulated buttons, usually on a
 * device that does not have a specific physical button available.
 *
 * @param device The device to configure
 *
 * @return A bitmask of possible methods.
 *
 * @see libinput_device_config_click_get_methods
 * @see libinput_device_config_click_set_method
 * @see libinput_device_config_click_get_method
 */
uint32_t
libinput_device_config_click_get_methods(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Set the button click method for this device. The button click
 * method defines when to generate software-emulated buttons, usually on a
 * device that does not have a specific physical button available.
 *
 * @note The selected click method may not take effect immediately. The
 * device may require changing to a neutral state first before activating
 * the new method.
 *
 * @param device The device to configure
 * @param method The button click method
 *
 * @return A config status code
 *
 * @see libinput_device_config_click_get_methods
 * @see libinput_device_config_click_get_method
 * @see libinput_device_config_click_get_default_method
 */
enum libinput_config_status
libinput_device_config_click_set_method(struct libinput_device *device,
					enum libinput_config_click_method method);
/**
 * @ingroup config
 *
 * Get the button click method for this device. The button click
 * method defines when to generate software-emulated buttons, usually on a
 * device that does not have a specific physical button available.
 *
 * @param device The device to configure
 *
 * @return The current button click method for this device
 *
 * @see libinput_device_config_click_get_methods
 * @see libinput_device_config_click_set_method
 * @see libinput_device_config_click_get_default_method
 */
enum libinput_config_click_method
libinput_device_config_click_get_method(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Get the default button click method for this device. The button click
 * method defines when to generate software-emulated buttons, usually on a
 * device that does not have a specific physical button available.
 *
 * @param device The device to configure
 *
 * @return The default button click method for this device
 *
 * @see libinput_device_config_click_get_methods
 * @see libinput_device_config_click_set_method
 * @see libinput_device_config_click_get_method
 */
enum libinput_config_click_method
libinput_device_config_click_get_default_method(struct libinput_device *device);

/**
 * @ingroup config
 */
enum libinput_config_middle_emulation_state {
	/**
	 * Middle mouse button emulation is to be disabled, or
	 * is currently disabled.
	 */
	LIBINPUT_CONFIG_MIDDLE_EMULATION_DISABLED,
	/**
	 * Middle mouse button emulation is to be enabled, or
	 * is currently enabled.
	 */
	LIBINPUT_CONFIG_MIDDLE_EMULATION_ENABLED,
};

/**
 * @ingroup config
 *
 * Check if middle mouse button emulation configuration is available on this
 * device. See libinput_device_config_middle_emulation_set_enabled() for
 * more details.
 *
 * @note Some devices provide middle mouse button emulation but do not allow
 * enabling/disabling that emulation. These devices return zero in
 * libinput_device_config_middle_emulation_is_available().
 *
 * @param device The device to query
 *
 * @return Non-zero if middle mouse button emulation is available and can be
 * configured, zero otherwise.
 *
 * @see libinput_device_config_middle_emulation_set_enabled
 * @see libinput_device_config_middle_emulation_get_enabled
 * @see libinput_device_config_middle_emulation_get_default_enabled
 */
int
libinput_device_config_middle_emulation_is_available(
		struct libinput_device *device);

/**
 * @ingroup config
 *
 * Enable or disable middle button emulation on this device. When enabled, a
 * simultaneous press of the left and right button generates a middle mouse
 * button event. Releasing the buttons generates a middle mouse button
 * release, the left and right button events are discarded otherwise.
 *
 * See the libinput documentation for more details.
 *
 * @param device The device to configure
 * @param enable @ref LIBINPUT_CONFIG_MIDDLE_EMULATION_DISABLED to
 * disable, @ref LIBINPUT_CONFIG_MIDDLE_EMULATION_ENABLED To enable
 * middle button emulation.
 *
 * @return A config status code. Disabling middle button emulation on a
 * device that does not support middle button emulation always succeeds.
 *
 * @see libinput_device_config_middle_emulation_is_available
 * @see libinput_device_config_middle_emulation_get_enabled
 * @see libinput_device_config_middle_emulation_get_default_enabled
 */
enum libinput_config_status
libinput_device_config_middle_emulation_set_enabled(
		struct libinput_device *device,
		enum libinput_config_middle_emulation_state enable);

/**
 * @ingroup config
 *
 * Check if configurable middle button emulation is enabled on this device.
 * See libinput_device_config_middle_emulation_set_enabled() for more
 * details.
 *
 * If the device does not have configurable middle button emulation, this
 * function returns @ref LIBINPUT_CONFIG_MIDDLE_EMULATION_DISABLED.
 *
 * @note Some devices provide middle mouse button emulation but do not allow
 * enabling/disabling that emulation. These devices always return @ref
 * LIBINPUT_CONFIG_MIDDLE_EMULATION_DISABLED.
 *
 * @param device The device to configure
 * @return @ref LIBINPUT_CONFIG_MIDDLE_EMULATION_DISABLED if disabled
 * or not available/configurable, @ref
 * LIBINPUT_CONFIG_MIDDLE_EMULATION_ENABLED If enabled.
 *
 * @see libinput_device_config_middle_emulation_is_available
 * @see libinput_device_config_middle_emulation_set_enabled
 * @see libinput_device_config_middle_emulation_get_default_enabled
 */
enum libinput_config_middle_emulation_state
libinput_device_config_middle_emulation_get_enabled(
		struct libinput_device *device);

/**
 * @ingroup config
 *
 * Check if configurable middle button emulation is enabled by default on
 * this device. See libinput_device_config_middle_emulation_set_enabled()
 * for more details.
 *
 * If the device does not have configurable middle button
 * emulation, this function returns @ref
 * LIBINPUT_CONFIG_MIDDLE_EMULATION_DISABLED.
 *
 * @note Some devices provide middle mouse button emulation but do not allow
 * enabling/disabling that emulation. These devices always return @ref
 * LIBINPUT_CONFIG_MIDDLE_EMULATION_DISABLED.
 *
 * @param device The device to configure
 * @return @ref LIBINPUT_CONFIG_MIDDLE_EMULATION_DISABLED If disabled
 * or not available, @ref LIBINPUT_CONFIG_MIDDLE_EMULATION_ENABLED if
 * enabled.
 *
 * @see libinput_device_config_middle_emulation_is_available
 * @see libinput_device_config_middle_emulation_set_enabled
 * @see libinput_device_config_middle_emulation_get_enabled
 */
enum libinput_config_middle_emulation_state
libinput_device_config_middle_emulation_get_default_enabled(
		struct libinput_device *device);

/**
 * @ingroup config
 *
 * The scroll method of a device selects when to generate scroll axis events
 * instead of pointer motion events.
 */
enum libinput_config_scroll_method {
	/**
	 * Never send scroll events instead of pointer motion events.
	 * This has no effect on events generated by scroll wheels.
	 */
	LIBINPUT_CONFIG_SCROLL_NO_SCROLL = 0,
	/**
	 * Send scroll events when two fingers are logically down on the
	 * device.
	 */
	LIBINPUT_CONFIG_SCROLL_2FG = (1 << 0),
	/**
	 * Send scroll events when a finger moves along the bottom or
	 * right edge of a device.
	 */
	LIBINPUT_CONFIG_SCROLL_EDGE = (1 << 1),
	/**
	 * Send scroll events when a button is down and the device moves
	 * along a scroll-capable axis.
	 */
	LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN = (1 << 2),
};

/**
 * @ingroup config
 *
 * Check which scroll methods a device supports. The method defines when to
 * generate scroll axis events instead of pointer motion events.
 *
 * @param device The device to configure
 *
 * @return A bitmask of possible methods.
 *
 * @see libinput_device_config_scroll_set_method
 * @see libinput_device_config_scroll_get_method
 * @see libinput_device_config_scroll_get_default_method
 * @see libinput_device_config_scroll_set_button
 * @see libinput_device_config_scroll_get_button
 * @see libinput_device_config_scroll_get_default_button
 */
uint32_t
libinput_device_config_scroll_get_methods(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Set the scroll method for this device. The method defines when to
 * generate scroll axis events instead of pointer motion events.
 *
 * @note Setting @ref LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN enables
 * the scroll method, but scrolling is only activated when the configured
 * button is held down. If no button is set, i.e.
 * libinput_device_config_scroll_get_button() returns 0, scrolling
 * cannot activate.
 *
 * @param device The device to configure
 * @param method The scroll method for this device.
 *
 * @return A config status code.
 *
 * @see libinput_device_config_scroll_get_methods
 * @see libinput_device_config_scroll_get_method
 * @see libinput_device_config_scroll_get_default_method
 * @see libinput_device_config_scroll_set_button
 * @see libinput_device_config_scroll_get_button
 * @see libinput_device_config_scroll_get_default_button
 */
enum libinput_config_status
libinput_device_config_scroll_set_method(struct libinput_device *device,
					 enum libinput_config_scroll_method method);

/**
 * @ingroup config
 *
 * Get the scroll method for this device. The method defines when to
 * generate scroll axis events instead of pointer motion events.
 *
 * @param device The device to configure
 * @return The current scroll method for this device.
 *
 * @see libinput_device_config_scroll_get_methods
 * @see libinput_device_config_scroll_set_method
 * @see libinput_device_config_scroll_get_default_method
 * @see libinput_device_config_scroll_set_button
 * @see libinput_device_config_scroll_get_button
 * @see libinput_device_config_scroll_get_default_button
 */
enum libinput_config_scroll_method
libinput_device_config_scroll_get_method(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Get the default scroll method for this device. The method defines when to
 * generate scroll axis events instead of pointer motion events.
 *
 * @param device The device to configure
 * @return The default scroll method for this device.
 *
 * @see libinput_device_config_scroll_get_methods
 * @see libinput_device_config_scroll_set_method
 * @see libinput_device_config_scroll_get_method
 * @see libinput_device_config_scroll_set_button
 * @see libinput_device_config_scroll_get_button
 * @see libinput_device_config_scroll_get_default_button
 */
enum libinput_config_scroll_method
libinput_device_config_scroll_get_default_method(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Set the button for the @ref LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN method
 * for this device.
 *
 * When the current scroll method is set to @ref
 * LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN, no button press/release events
 * will be send for the configured button.
 *
 * When the configured button is pressed, any motion events along a
 * scroll-capable axis are turned into scroll axis events.
 *
 * @note Setting the button does not change the scroll method. To change the
 * scroll method call libinput_device_config_scroll_set_method().
 *
 * If the button is 0, button scrolling is effectively disabled.
 *
 * @param device The device to configure
 * @param button The button which when pressed switches to sending scroll events
 *
 * @return A config status code
 * @retval LIBINPUT_CONFIG_STATUS_SUCCESS On success
 * @retval LIBINPUT_CONFIG_STATUS_UNSUPPORTED If @ref
 * LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN is not supported
 * @retval LIBINPUT_CONFIG_STATUS_INVALID The given button does not
 * exist on this device
 *
 * @see libinput_device_config_scroll_get_methods
 * @see libinput_device_config_scroll_set_method
 * @see libinput_device_config_scroll_get_method
 * @see libinput_device_config_scroll_get_default_method
 * @see libinput_device_config_scroll_get_button
 * @see libinput_device_config_scroll_get_default_button
 */
enum libinput_config_status
libinput_device_config_scroll_set_button(struct libinput_device *device,
					 uint32_t button);

/**
 * @ingroup config
 *
 * Get the button for the @ref LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN method
 * for this device.
 *
 * If @ref LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN scroll method is not
 * supported, or no button is set, this function returns 0.
 *
 * @note The return value is independent of the currently selected
 * scroll-method. For button scrolling to activate, a device must have the
 * @ref LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN method enabled, and a non-zero
 * button set as scroll button.
 *
 * @param device The device to configure
 * @return The button which when pressed switches to sending scroll events
 *
 * @see libinput_device_config_scroll_get_methods
 * @see libinput_device_config_scroll_set_method
 * @see libinput_device_config_scroll_get_method
 * @see libinput_device_config_scroll_get_default_method
 * @see libinput_device_config_scroll_set_button
 * @see libinput_device_config_scroll_get_default_button
 */
uint32_t
libinput_device_config_scroll_get_button(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Get the default button for the @ref LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN
 * method for this device.
 *
 * If @ref LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN scroll method is not supported,
 * or no default button is set, this function returns 0.
 *
 * @param device The device to configure
 * @return The default button for the @ref
 * LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN method
 *
 * @see libinput_device_config_scroll_get_methods
 * @see libinput_device_config_scroll_set_method
 * @see libinput_device_config_scroll_get_method
 * @see libinput_device_config_scroll_get_default_method
 * @see libinput_device_config_scroll_set_button
 * @see libinput_device_config_scroll_get_button
 */
uint32_t
libinput_device_config_scroll_get_default_button(struct libinput_device *device);

enum libinput_config_scroll_button_lock_state {
	LIBINPUT_CONFIG_SCROLL_BUTTON_LOCK_DISABLED,
	LIBINPUT_CONFIG_SCROLL_BUTTON_LOCK_ENABLED,
};

/**
 * @ingroup config
 *
 * Set the scroll button lock. If the state is
 * @ref LIBINPUT_CONFIG_SCROLL_BUTTON_LOCK_DISABLED, the button must
 * physically be held down for button scrolling to work.
 * If the state is
 * @ref LIBINPUT_CONFIG_SCROLL_BUTTON_LOCK_ENABLED, the button is considered
 * logically down after the first press and release sequence, and logically
 * up after the second press and release sequence.
 *
 * @param device The device to configure
 * @param state The state to set the scroll button lock to
 *
 * @return A config status code. Disabling the scroll button lock on
 * device that does not support button scrolling always succeeds.
 *
 * @see libinput_device_config_scroll_set_button
 * @see libinput_device_config_scroll_get_button
 * @see libinput_device_config_scroll_get_default_button
 */
enum libinput_config_status
libinput_device_config_scroll_set_button_lock(struct libinput_device *device,
					      enum libinput_config_scroll_button_lock_state state);

/**
 * @ingroup config
 *
 * Get the current scroll button lock state.
 *
 * If @ref LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN scroll method is not
 * supported, or no button is set, this function returns @ref
 * LIBINPUT_CONFIG_SCROLL_BUTTON_LOCK_DISABLED.
 *
 * @note The return value is independent of the currently selected
 * scroll-method. For the scroll button lock to activate, a device must have
 * the @ref LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN method enabled, and a
 * non-zero button set as scroll button.
 *
 * @param device The device to configure
 * @return The scroll button lock state
 *
 * @see libinput_device_config_scroll_set_button
 * @see libinput_device_config_scroll_set_button_lock
 * @see libinput_device_config_scroll_get_button_lock
 * @see libinput_device_config_scroll_get_default_button_lock
 */
enum libinput_config_scroll_button_lock_state
libinput_device_config_scroll_get_button_lock(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Get the default scroll button lock state.
 *
 * If @ref LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN scroll method is not
 * supported, or no button is set, this function returns @ref
 * LIBINPUT_CONFIG_SCROLL_BUTTON_LOCK_DISABLED.
 *
 * @param device The device to configure
 * @return The default scroll button lock state
 *
 * @see libinput_device_config_scroll_set_button
 * @see libinput_device_config_scroll_set_button_lock
 * @see libinput_device_config_scroll_get_button_lock
 * @see libinput_device_config_scroll_get_default_button_lock
 */
enum libinput_config_scroll_button_lock_state
libinput_device_config_scroll_get_default_button_lock(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Possible states for the disable-while-typing feature.
 */
enum libinput_config_dwt_state {
	LIBINPUT_CONFIG_DWT_DISABLED,
	LIBINPUT_CONFIG_DWT_ENABLED,
};

/**
 * @ingroup config
 *
 * Check if this device supports configurable disable-while-typing feature.
 * This feature is usually available on built-in touchpads and disables the
 * touchpad while typing. See the libinput documentation for details.
 *
 * @param device The device to configure
 * @return 0 if this device does not support disable-while-typing, or 1
 * otherwise.
 *
 * @see libinput_device_config_dwt_set_enabled
 * @see libinput_device_config_dwt_get_enabled
 * @see libinput_device_config_dwt_get_default_enabled
 */
int
libinput_device_config_dwt_is_available(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Enable or disable the disable-while-typing feature. When enabled, the
 * device will be disabled while typing and for a short period after. See
 * the libinput documentation for details.
 *
 * @note Enabling or disabling disable-while-typing may not take effect
 * immediately.
 *
 * @param device The device to configure
 * @param enable @ref LIBINPUT_CONFIG_DWT_DISABLED to disable
 * disable-while-typing, @ref LIBINPUT_CONFIG_DWT_ENABLED to enable
 *
 * @return A config status code. Disabling disable-while-typing on a
 * device that does not support the feature always succeeds.
 *
 * @see libinput_device_config_dwt_is_available
 * @see libinput_device_config_dwt_get_enabled
 * @see libinput_device_config_dwt_get_default_enabled
 */
enum libinput_config_status
libinput_device_config_dwt_set_enabled(struct libinput_device *device,
				       enum libinput_config_dwt_state enable);

/**
 * @ingroup config
 *
 * Check if the disable-while typing feature is currently enabled on this
 * device. If the device does not support disable-while-typing, this
 * function returns @ref LIBINPUT_CONFIG_DWT_DISABLED.
 *
 * @param device The device to configure
 * @return @ref LIBINPUT_CONFIG_DWT_DISABLED if disabled, @ref
 * LIBINPUT_CONFIG_DWT_ENABLED if enabled.
 *
 * @see libinput_device_config_dwt_is_available
 * @see libinput_device_config_dwt_set_enabled
 * @see libinput_device_config_dwt_get_default_enabled
 */
enum libinput_config_dwt_state
libinput_device_config_dwt_get_enabled(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Check if the disable-while typing feature is enabled on this device by
 * default. If the device does not support disable-while-typing, this
 * function returns @ref LIBINPUT_CONFIG_DWT_DISABLED.
 *
 * @param device The device to configure
 * @return @ref LIBINPUT_CONFIG_DWT_DISABLED if disabled, @ref
 * LIBINPUT_CONFIG_DWT_ENABLED if enabled.
 *
 * @see libinput_device_config_dwt_is_available
 * @see libinput_device_config_dwt_set_enabled
 * @see libinput_device_config_dwt_get_enabled
 */
enum libinput_config_dwt_state
libinput_device_config_dwt_get_default_enabled(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Check whether a device can have a custom rotation applied.
 *
 * @param device The device to configure
 * @return Non-zero if a device can be rotated, zero otherwise.
 *
 * @see libinput_device_config_rotation_set_angle
 * @see libinput_device_config_rotation_get_angle
 * @see libinput_device_config_rotation_get_default_angle
 *
 * @since 1.4
 */
int
libinput_device_config_rotation_is_available(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Set the rotation of a device in degrees clockwise off the logical neutral
 * position. Any subsequent motion events are adjusted according to the
 * given angle.
 *
 * The angle has to be in the range of [0, 360[ degrees, otherwise this
 * function returns LIBINPUT_CONFIG_STATUS_INVALID. If the angle is a
 * multiple of 360 or negative, the caller must ensure the correct ranging
 * before calling this function.
 *
 * libinput guarantees that this function accepts multiples of 90 degrees.
 * If a value is within the [0, 360[ range but not a multiple of 90 degrees,
 * this function may return LIBINPUT_CONFIG_STATUS_INVALID if the underlying
 * device or implementation does not support finer-grained rotation angles.
 *
 * The rotation angle is applied to all motion events emitted by the device.
 * Thus, rotating the device also changes the angle required or presented by
 * scrolling, gestures, etc.
 *
 * @param device The device to configure
 * @param degrees_cw The angle in degrees clockwise
 * @return A config status code. Setting a rotation of 0 degrees on a
 * device that does not support rotation always succeeds.
 *
 * @see libinput_device_config_rotation_is_available
 * @see libinput_device_config_rotation_get_angle
 * @see libinput_device_config_rotation_get_default_angle
 *
 * @since 1.4
 */
enum libinput_config_status
libinput_device_config_rotation_set_angle(struct libinput_device *device,
					  unsigned int degrees_cw);

/**
 * @ingroup config
 *
 * Get the current rotation of a device in degrees clockwise off the logical
 * neutral position. If this device does not support rotation, the return
 * value is always 0.
 *
 * @param device The device to configure
 * @return The angle in degrees clockwise
 *
 * @see libinput_device_config_rotation_is_available
 * @see libinput_device_config_rotation_set_angle
 * @see libinput_device_config_rotation_get_default_angle
 *
 * @since 1.4
 */
unsigned int
libinput_device_config_rotation_get_angle(struct libinput_device *device);

/**
 * @ingroup config
 *
 * Get the default rotation of a device in degrees clockwise off the logical
 * neutral position. If this device does not support rotation, the return
 * value is always 0.
 *
 * @param device The device to configure
 * @return The default angle in degrees clockwise
 *
 * @see libinput_device_config_rotation_is_available
 * @see libinput_device_config_rotation_set_angle
 * @see libinput_device_config_rotation_get_angle
 *
 * @since 1.4
 */
unsigned int
libinput_device_config_rotation_get_default_angle(struct libinput_device *device);

#ifdef __cplusplus
}
#endif
#endif /* LIBINPUT_H */
