/* -*- Mode: javascript; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 ; js-indent-level : 2 ; js-curly-indent-offset: 0 -*- */
/* vim: set ts=2 et sw=2 tw=80: */

// Mandelbrot using SIMD
// Author: Peter Jensen, Intel Corporation

load(libdir + "asm.js");
if (!isSimdAvailable() || typeof SIMD === 'undefined') {
    print("won't run tests as simd extensions aren't activated yet");
    quit(0);
}

// global variables
const MAX_ITERATIONS = 10;
const DRAW_ITERATIONS = 10;

const CANVAS_WIDTH = 20;
const CANVAS_HEIGHT = 20;

const LIMIT_SHOW = 20 * 20 * 4;

// Asm.js module buffer.
var buffer = new ArrayBuffer(16 * 1024 * 1024);
var view = new Uint8Array(buffer);

var moduleCode = `
  "use asm"
  var b8 = new global.Uint8Array(buffer);
  var toF = global.Math.fround;
  var i4 = global.SIMD.int32x4;
  var f4 = global.SIMD.float32x4;
  var i4add = i4.add;
  var i4and = i4.and;
  var f4add = f4.add;
  var f4sub = f4.sub;
  var f4mul = f4.mul;
  var f4lessThanOrEqual = f4.lessThanOrEqual;
  var f4splat = f4.splat;
  var imul = global.Math.imul;
  const one4 = i4(1,1,1,1), two4 = f4(2,2,2,2), four4 = f4(4,4,4,4);

  const mk0 = 0x007fffff;

  function declareHeapLength() {
    b8[0x00ffffff] = 0;
  }

  function mapColorAndSetPixel (x, y, width, value, max_iterations) {
    x = x | 0;
    y = y | 0;
    width = width | 0;
    value = value | 0;
    max_iterations = max_iterations | 0;

    var rgb = 0, r = 0, g = 0, b = 0, index = 0;

    index = (((imul((width >>> 0), (y >>> 0)) + x) | 0) * 4) | 0;
    if ((value | 0) == (max_iterations | 0)) {
      r = 0;
      g = 0;
      b = 0;
    } else {
      rgb = ~~toF(toF(toF(toF(value >>> 0) * toF(0xffff)) / toF(max_iterations >>> 0)) * toF(0xff));
      r = rgb & 0xff;
      g = (rgb >>> 8) & 0xff;
      b = (rgb >>> 16) & 0xff;
    }
    b8[(index & mk0) >> 0] = r;
    b8[(index & mk0) + 1 >> 0] = g;
    b8[(index & mk0) + 2 >> 0] = b;
    b8[(index & mk0) + 3 >> 0] = 255;
  }

  function mandelPixelX4 (xf, yf, yd, max_iterations) {
    xf = toF(xf);
    yf = toF(yf);
    yd = toF(yd);
    max_iterations = max_iterations | 0;
    var c_re4  = f4(0,0,0,0), c_im4  = f4(0,0,0,0);
    var z_re4  = f4(0,0,0,0), z_im4  = f4(0,0,0,0);
    var count4 = i4(0,0,0,0);
    var z_re24 = f4(0,0,0,0), z_im24 = f4(0,0,0,0);
    var new_re4 = f4(0,0,0,0), new_im4 = f4(0,0,0,0);
    var i = 0;
    var mi4 = i4(0,0,0,0);

    c_re4 = f4splat(xf);
    c_im4 = f4(yf, toF(yd + yf), toF(yd + toF(yd + yf)), toF(yd + toF(yd + toF(yd + yf))));

    z_re4  = c_re4;
    z_im4  = c_im4;

    for (i = 0; (i | 0) < (max_iterations | 0); i = (i + 1) | 0) {
      z_re24 = f4mul(z_re4, z_re4);
      z_im24 = f4mul(z_im4, z_im4);
      mi4 = f4lessThanOrEqual(f4add(z_re24, z_im24), four4);
      // If all 4 values are greater than 4.0, there's no reason to continue.
      if ((mi4.signMask | 0) == 0x00)
        break;

      new_re4 = f4sub(z_re24, z_im24);
      new_im4 = f4mul(f4mul(two4, z_re4), z_im4);
      z_re4   = f4add(c_re4, new_re4);
      z_im4   = f4add(c_im4, new_im4);
      count4  = i4add(count4, i4and(mi4, one4));
    }
    return i4(count4);
  }

  function mandelColumnX4 (x, width, height, xf, yf, yd, max_iterations) {
    x = x | 0;
    width = width | 0;
    height = height | 0;
    xf = toF(xf);
    yf = toF(yf);
    yd = toF(yd);
    max_iterations = max_iterations | 0;

    var y = 0;
    var ydx4 = toF(0);
    var m4 = i4(0,0,0,0);

    ydx4 = toF(yd * toF(4));
    for (y = 0; (y | 0) < (height | 0); y = (y + 4) | 0) {
      m4   = i4(mandelPixelX4(toF(xf), toF(yf), toF(yd), max_iterations));
      mapColorAndSetPixel(x | 0, y | 0,   width, m4.x, max_iterations);
      mapColorAndSetPixel(x | 0, (y + 1) | 0, width, m4.y, max_iterations);
      mapColorAndSetPixel(x | 0, (y + 2) | 0, width, m4.z, max_iterations);
      mapColorAndSetPixel(x | 0, (y + 3) | 0, width, m4.w, max_iterations);
      yf = toF(yf + ydx4);
    }
  }

  function mandel (width, height, xc, yc, scale, max_iterations) {
    width = width | 0;
    height = height | 0;
    xc = toF(xc);
    yc = toF(yc);
    scale = toF(scale);
    max_iterations = max_iterations | 0;

    var x0 = toF(0), y0 = toF(0);
    var xd = toF(0), yd = toF(0);
    var xf = toF(0);
    var x = 0;

    x0 = toF(xc - toF(scale * toF(1.5)));
    y0 = toF(yc - scale);
    xd = toF(toF(scale * toF(3)) / toF(width >>> 0));
    yd = toF(toF(scale * toF(2)) / toF(height >>> 0));
    xf = x0;

    for (x = 0; (x | 0) < (width | 0); x = (x + 1) | 0) {
      mandelColumnX4(x, width, height, xf, y0, yd, max_iterations);
      xf = toF(xf + xd);
    }
  }

  return mandel;
`;

var FFI = {};
var mandelbro = asmLink(asmCompile('global', 'ffi', 'buffer', moduleCode), this, FFI, buffer);

function animateMandelbrot () {
  var scale_start = 1.0;
  var scale_end   = 0.0005;
  var xc_start    = -0.5;
  var yc_start    = 0.0;
  var xc_end      = 0.0;
  var yc_end      = 0.75;
  var steps       = 200.0;
  var scale_step  = (scale_end - scale_start)/steps;
  var xc_step     = (xc_end - xc_start)/steps;
  var yc_step     = (yc_end - yc_start)/steps;
  var scale       = scale_start;
  var xc          = xc_start;
  var yc          = yc_start;
  var i           = 0;
  var now         = dateNow();

  function draw1 () {
    mandelbro(CANVAS_WIDTH, CANVAS_HEIGHT, xc, yc, scale, MAX_ITERATIONS);
    if (scale < scale_end || scale > scale_start) {
      scale_step = -scale_step;
      xc_step = -xc_step;
      yc_step = -yc_step;
    }
    scale += scale_step;
    xc += xc_step;
    yc += yc_step;
    i++;
  }

  var b = dateNow();
  for (var j = DRAW_ITERATIONS; j --> 0;)
    draw1();
  print(dateNow() - b);
}

animateMandelbrot();

assertEq(view[0], 0, "0th value should be 0");
assertEq(view[1], 0, "1th value should be 0");
assertEq(view[2], 0, "2th value should be 0");
assertEq(view[3], 255, "3th value should be 255");
assertEq(view[4], 230, "4th value should be 230");
assertEq(view[5], 127, "5th value should be 127");
assertEq(view[6], 25, "6th value should be 25");
assertEq(view[7], 255, "7th value should be 255");
assertEq(view[8], 230, "8th value should be 230");
assertEq(view[9], 127, "9th value should be 127");
assertEq(view[10], 25, "10th value should be 25");
assertEq(view[11], 255, "11th value should be 255");
assertEq(view[12], 205, "12th value should be 205");
assertEq(view[13], 255, "13th value should be 255");
assertEq(view[14], 50, "14th value should be 50");
assertEq(view[15], 255, "15th value should be 255");
assertEq(view[16], 205, "16th value should be 205");
assertEq(view[17], 255, "17th value should be 255");
assertEq(view[18], 50, "18th value should be 50");
assertEq(view[19], 255, "19th value should be 255");
assertEq(view[20], 205, "20th value should be 205");
assertEq(view[21], 255, "21th value should be 255");
assertEq(view[22], 50, "22th value should be 50");
assertEq(view[23], 255, "23th value should be 255");
assertEq(view[24], 205, "24th value should be 205");
assertEq(view[25], 255, "25th value should be 255");
assertEq(view[26], 50, "26th value should be 50");
assertEq(view[27], 255, "27th value should be 255");
assertEq(view[28], 205, "28th value should be 205");
assertEq(view[29], 255, "29th value should be 255");
assertEq(view[30], 50, "30th value should be 50");
assertEq(view[31], 255, "31th value should be 255");
assertEq(view[32], 179, "32th value should be 179");
assertEq(view[33], 127, "33th value should be 127");
assertEq(view[34], 76, "34th value should be 76");
assertEq(view[35], 255, "35th value should be 255");
assertEq(view[36], 179, "36th value should be 179");
assertEq(view[37], 127, "37th value should be 127");
assertEq(view[38], 76, "38th value should be 76");
assertEq(view[39], 255, "39th value should be 255");
assertEq(view[40], 179, "40th value should be 179");
assertEq(view[41], 127, "41th value should be 127");
assertEq(view[42], 76, "42th value should be 76");
assertEq(view[43], 255, "43th value should be 255");
assertEq(view[44], 154, "44th value should be 154");
assertEq(view[45], 255, "45th value should be 255");
assertEq(view[46], 101, "46th value should be 101");
assertEq(view[47], 255, "47th value should be 255");
assertEq(view[48], 78, "48th value should be 78");
assertEq(view[49], 127, "49th value should be 127");
assertEq(view[50], 178, "50th value should be 178");
assertEq(view[51], 255, "51th value should be 255");
assertEq(view[52], 52, "52th value should be 52");
assertEq(view[53], 255, "53th value should be 255");
assertEq(view[54], 203, "54th value should be 203");
assertEq(view[55], 255, "55th value should be 255");
assertEq(view[56], 154, "56th value should be 154");
assertEq(view[57], 255, "57th value should be 255");
assertEq(view[58], 101, "58th value should be 101");
assertEq(view[59], 255, "59th value should be 255");
assertEq(view[60], 179, "60th value should be 179");
assertEq(view[61], 127, "61th value should be 127");
assertEq(view[62], 76, "62th value should be 76");
assertEq(view[63], 255, "63th value should be 255");
assertEq(view[64], 205, "64th value should be 205");
assertEq(view[65], 255, "65th value should be 255");
assertEq(view[66], 50, "66th value should be 50");
assertEq(view[67], 255, "67th value should be 255");
assertEq(view[68], 205, "68th value should be 205");
assertEq(view[69], 255, "69th value should be 255");
assertEq(view[70], 50, "70th value should be 50");
assertEq(view[71], 255, "71th value should be 255");
assertEq(view[72], 230, "72th value should be 230");
assertEq(view[73], 127, "73th value should be 127");
assertEq(view[74], 25, "74th value should be 25");
assertEq(view[75], 255, "75th value should be 255");
assertEq(view[76], 230, "76th value should be 230");
assertEq(view[77], 127, "77th value should be 127");
assertEq(view[78], 25, "78th value should be 25");
assertEq(view[79], 255, "79th value should be 255");
assertEq(view[80], 0, "80th value should be 0");
assertEq(view[81], 0, "81th value should be 0");
assertEq(view[82], 0, "82th value should be 0");
assertEq(view[83], 255, "83th value should be 255");
assertEq(view[84], 230, "84th value should be 230");
assertEq(view[85], 127, "85th value should be 127");
assertEq(view[86], 25, "86th value should be 25");
assertEq(view[87], 255, "87th value should be 255");
assertEq(view[88], 205, "88th value should be 205");
assertEq(view[89], 255, "89th value should be 255");
assertEq(view[90], 50, "90th value should be 50");
assertEq(view[91], 255, "91th value should be 255");
assertEq(view[92], 205, "92th value should be 205");
assertEq(view[93], 255, "93th value should be 255");
assertEq(view[94], 50, "94th value should be 50");
assertEq(view[95], 255, "95th value should be 255");
assertEq(view[96], 205, "96th value should be 205");
assertEq(view[97], 255, "97th value should be 255");
assertEq(view[98], 50, "98th value should be 50");
assertEq(view[99], 255, "99th value should be 255");
assertEq(view[100], 205, "100th value should be 205");
assertEq(view[101], 255, "101th value should be 255");
assertEq(view[102], 50, "102th value should be 50");
assertEq(view[103], 255, "103th value should be 255");
assertEq(view[104], 205, "104th value should be 205");
assertEq(view[105], 255, "105th value should be 255");
assertEq(view[106], 50, "106th value should be 50");
assertEq(view[107], 255, "107th value should be 255");
assertEq(view[108], 205, "108th value should be 205");
assertEq(view[109], 255, "109th value should be 255");
assertEq(view[110], 50, "110th value should be 50");
assertEq(view[111], 255, "111th value should be 255");
assertEq(view[112], 179, "112th value should be 179");
assertEq(view[113], 127, "113th value should be 127");
assertEq(view[114], 76, "114th value should be 76");
assertEq(view[115], 255, "115th value should be 255");
assertEq(view[116], 179, "116th value should be 179");
assertEq(view[117], 127, "117th value should be 127");
assertEq(view[118], 76, "118th value should be 76");
assertEq(view[119], 255, "119th value should be 255");
assertEq(view[120], 154, "120th value should be 154");
assertEq(view[121], 255, "121th value should be 255");
assertEq(view[122], 101, "122th value should be 101");
assertEq(view[123], 255, "123th value should be 255");
assertEq(view[124], 103, "124th value should be 103");
assertEq(view[125], 255, "125th value should be 255");
assertEq(view[126], 152, "126th value should be 152");
assertEq(view[127], 255, "127th value should be 255");
assertEq(view[128], 0, "128th value should be 0");
assertEq(view[129], 0, "129th value should be 0");
assertEq(view[130], 0, "130th value should be 0");
assertEq(view[131], 255, "131th value should be 255");
assertEq(view[132], 0, "132th value should be 0");
assertEq(view[133], 0, "133th value should be 0");
assertEq(view[134], 0, "134th value should be 0");
assertEq(view[135], 255, "135th value should be 255");
assertEq(view[136], 128, "136th value should be 128");
assertEq(view[137], 127, "137th value should be 127");
assertEq(view[138], 127, "138th value should be 127");
assertEq(view[139], 255, "139th value should be 255");
assertEq(view[140], 154, "140th value should be 154");
assertEq(view[141], 255, "141th value should be 255");
assertEq(view[142], 101, "142th value should be 101");
assertEq(view[143], 255, "143th value should be 255");
assertEq(view[144], 179, "144th value should be 179");
assertEq(view[145], 127, "145th value should be 127");
assertEq(view[146], 76, "146th value should be 76");
assertEq(view[147], 255, "147th value should be 255");
assertEq(view[148], 205, "148th value should be 205");
assertEq(view[149], 255, "149th value should be 255");
assertEq(view[150], 50, "150th value should be 50");
assertEq(view[151], 255, "151th value should be 255");
assertEq(view[152], 205, "152th value should be 205");
assertEq(view[153], 255, "153th value should be 255");
assertEq(view[154], 50, "154th value should be 50");
assertEq(view[155], 255, "155th value should be 255");
assertEq(view[156], 230, "156th value should be 230");
assertEq(view[157], 127, "157th value should be 127");
assertEq(view[158], 25, "158th value should be 25");
assertEq(view[159], 255, "159th value should be 255");
assertEq(view[160], 0, "160th value should be 0");
assertEq(view[161], 0, "161th value should be 0");
assertEq(view[162], 0, "162th value should be 0");
assertEq(view[163], 255, "163th value should be 255");
assertEq(view[164], 230, "164th value should be 230");
assertEq(view[165], 127, "165th value should be 127");
assertEq(view[166], 25, "166th value should be 25");
assertEq(view[167], 255, "167th value should be 255");
assertEq(view[168], 205, "168th value should be 205");
assertEq(view[169], 255, "169th value should be 255");
assertEq(view[170], 50, "170th value should be 50");
assertEq(view[171], 255, "171th value should be 255");
assertEq(view[172], 205, "172th value should be 205");
assertEq(view[173], 255, "173th value should be 255");
assertEq(view[174], 50, "174th value should be 50");
assertEq(view[175], 255, "175th value should be 255");
assertEq(view[176], 205, "176th value should be 205");
assertEq(view[177], 255, "177th value should be 255");
assertEq(view[178], 50, "178th value should be 50");
assertEq(view[179], 255, "179th value should be 255");
assertEq(view[180], 205, "180th value should be 205");
assertEq(view[181], 255, "181th value should be 255");
assertEq(view[182], 50, "182th value should be 50");
assertEq(view[183], 255, "183th value should be 255");
assertEq(view[184], 205, "184th value should be 205");
assertEq(view[185], 255, "185th value should be 255");
assertEq(view[186], 50, "186th value should be 50");
assertEq(view[187], 255, "187th value should be 255");
assertEq(view[188], 179, "188th value should be 179");
assertEq(view[189], 127, "189th value should be 127");
assertEq(view[190], 76, "190th value should be 76");
assertEq(view[191], 255, "191th value should be 255");
assertEq(view[192], 179, "192th value should be 179");
assertEq(view[193], 127, "193th value should be 127");
assertEq(view[194], 76, "194th value should be 76");
assertEq(view[195], 255, "195th value should be 255");
assertEq(view[196], 154, "196th value should be 154");
assertEq(view[197], 255, "197th value should be 255");
assertEq(view[198], 101, "198th value should be 101");
assertEq(view[199], 255, "199th value should be 255");
assertEq(view[200], 103, "200th value should be 103");
assertEq(view[201], 255, "201th value should be 255");
assertEq(view[202], 152, "202th value should be 152");
assertEq(view[203], 255, "203th value should be 255");
assertEq(view[204], 78, "204th value should be 78");
assertEq(view[205], 127, "205th value should be 127");
assertEq(view[206], 178, "206th value should be 178");
assertEq(view[207], 255, "207th value should be 255");
assertEq(view[208], 0, "208th value should be 0");
assertEq(view[209], 0, "209th value should be 0");
assertEq(view[210], 0, "210th value should be 0");
assertEq(view[211], 255, "211th value should be 255");
assertEq(view[212], 0, "212th value should be 0");
assertEq(view[213], 0, "213th value should be 0");
assertEq(view[214], 0, "214th value should be 0");
assertEq(view[215], 255, "215th value should be 255");
assertEq(view[216], 78, "216th value should be 78");
assertEq(view[217], 127, "217th value should be 127");
assertEq(view[218], 178, "218th value should be 178");
assertEq(view[219], 255, "219th value should be 255");
assertEq(view[220], 128, "220th value should be 128");
assertEq(view[221], 127, "221th value should be 127");
assertEq(view[222], 127, "222th value should be 127");
assertEq(view[223], 255, "223th value should be 255");
assertEq(view[224], 154, "224th value should be 154");
assertEq(view[225], 255, "225th value should be 255");
assertEq(view[226], 101, "226th value should be 101");
assertEq(view[227], 255, "227th value should be 255");
assertEq(view[228], 205, "228th value should be 205");
assertEq(view[229], 255, "229th value should be 255");
assertEq(view[230], 50, "230th value should be 50");
assertEq(view[231], 255, "231th value should be 255");
assertEq(view[232], 205, "232th value should be 205");
assertEq(view[233], 255, "233th value should be 255");
assertEq(view[234], 50, "234th value should be 50");
assertEq(view[235], 255, "235th value should be 255");
assertEq(view[236], 230, "236th value should be 230");
assertEq(view[237], 127, "237th value should be 127");
assertEq(view[238], 25, "238th value should be 25");
assertEq(view[239], 255, "239th value should be 255");
assertEq(view[240], 0, "240th value should be 0");
assertEq(view[241], 0, "241th value should be 0");
assertEq(view[242], 0, "242th value should be 0");
assertEq(view[243], 255, "243th value should be 255");
assertEq(view[244], 205, "244th value should be 205");
assertEq(view[245], 255, "245th value should be 255");
assertEq(view[246], 50, "246th value should be 50");
assertEq(view[247], 255, "247th value should be 255");
assertEq(view[248], 205, "248th value should be 205");
assertEq(view[249], 255, "249th value should be 255");
assertEq(view[250], 50, "250th value should be 50");
assertEq(view[251], 255, "251th value should be 255");
assertEq(view[252], 205, "252th value should be 205");
assertEq(view[253], 255, "253th value should be 255");
assertEq(view[254], 50, "254th value should be 50");
assertEq(view[255], 255, "255th value should be 255");
assertEq(view[256], 205, "256th value should be 205");
assertEq(view[257], 255, "257th value should be 255");
assertEq(view[258], 50, "258th value should be 50");
assertEq(view[259], 255, "259th value should be 255");
assertEq(view[260], 205, "260th value should be 205");
assertEq(view[261], 255, "261th value should be 255");
assertEq(view[262], 50, "262th value should be 50");
assertEq(view[263], 255, "263th value should be 255");
assertEq(view[264], 179, "264th value should be 179");
assertEq(view[265], 127, "265th value should be 127");
assertEq(view[266], 76, "266th value should be 76");
assertEq(view[267], 255, "267th value should be 255");
assertEq(view[268], 179, "268th value should be 179");
assertEq(view[269], 127, "269th value should be 127");
assertEq(view[270], 76, "270th value should be 76");
assertEq(view[271], 255, "271th value should be 255");
assertEq(view[272], 154, "272th value should be 154");
assertEq(view[273], 255, "273th value should be 255");
assertEq(view[274], 101, "274th value should be 101");
assertEq(view[275], 255, "275th value should be 255");
assertEq(view[276], 52, "276th value should be 52");
assertEq(view[277], 255, "277th value should be 255");
assertEq(view[278], 203, "278th value should be 203");
assertEq(view[279], 255, "279th value should be 255");
assertEq(view[280], 0, "280th value should be 0");
assertEq(view[281], 0, "281th value should be 0");
assertEq(view[282], 0, "282th value should be 0");
assertEq(view[283], 255, "283th value should be 255");
assertEq(view[284], 0, "284th value should be 0");
assertEq(view[285], 0, "285th value should be 0");
assertEq(view[286], 0, "286th value should be 0");
assertEq(view[287], 255, "287th value should be 255");
assertEq(view[288], 0, "288th value should be 0");
assertEq(view[289], 0, "289th value should be 0");
assertEq(view[290], 0, "290th value should be 0");
assertEq(view[291], 255, "291th value should be 255");
assertEq(view[292], 0, "292th value should be 0");
assertEq(view[293], 0, "293th value should be 0");
assertEq(view[294], 0, "294th value should be 0");
assertEq(view[295], 255, "295th value should be 255");
assertEq(view[296], 0, "296th value should be 0");
assertEq(view[297], 0, "297th value should be 0");
assertEq(view[298], 0, "298th value should be 0");
assertEq(view[299], 255, "299th value should be 255");
assertEq(view[300], 52, "300th value should be 52");
assertEq(view[301], 255, "301th value should be 255");
assertEq(view[302], 203, "302th value should be 203");
assertEq(view[303], 255, "303th value should be 255");
assertEq(view[304], 52, "304th value should be 52");
assertEq(view[305], 255, "305th value should be 255");
assertEq(view[306], 203, "306th value should be 203");
assertEq(view[307], 255, "307th value should be 255");
assertEq(view[308], 179, "308th value should be 179");
assertEq(view[309], 127, "309th value should be 127");
assertEq(view[310], 76, "310th value should be 76");
assertEq(view[311], 255, "311th value should be 255");
assertEq(view[312], 205, "312th value should be 205");
assertEq(view[313], 255, "313th value should be 255");
assertEq(view[314], 50, "314th value should be 50");
assertEq(view[315], 255, "315th value should be 255");
assertEq(view[316], 205, "316th value should be 205");
assertEq(view[317], 255, "317th value should be 255");
assertEq(view[318], 50, "318th value should be 50");
assertEq(view[319], 255, "319th value should be 255");
assertEq(view[320], 230, "320th value should be 230");
assertEq(view[321], 127, "321th value should be 127");
assertEq(view[322], 25, "322th value should be 25");
assertEq(view[323], 255, "323th value should be 255");
assertEq(view[324], 205, "324th value should be 205");
assertEq(view[325], 255, "325th value should be 255");
assertEq(view[326], 50, "326th value should be 50");
assertEq(view[327], 255, "327th value should be 255");
assertEq(view[328], 205, "328th value should be 205");
assertEq(view[329], 255, "329th value should be 255");
assertEq(view[330], 50, "330th value should be 50");
assertEq(view[331], 255, "331th value should be 255");
assertEq(view[332], 205, "332th value should be 205");
assertEq(view[333], 255, "333th value should be 255");
assertEq(view[334], 50, "334th value should be 50");
assertEq(view[335], 255, "335th value should be 255");
assertEq(view[336], 205, "336th value should be 205");
assertEq(view[337], 255, "337th value should be 255");
assertEq(view[338], 50, "338th value should be 50");
assertEq(view[339], 255, "339th value should be 255");
assertEq(view[340], 179, "340th value should be 179");
assertEq(view[341], 127, "341th value should be 127");
assertEq(view[342], 76, "342th value should be 76");
assertEq(view[343], 255, "343th value should be 255");
assertEq(view[344], 154, "344th value should be 154");
assertEq(view[345], 255, "345th value should be 255");
assertEq(view[346], 101, "346th value should be 101");
assertEq(view[347], 255, "347th value should be 255");
assertEq(view[348], 154, "348th value should be 154");
assertEq(view[349], 255, "349th value should be 255");
assertEq(view[350], 101, "350th value should be 101");
assertEq(view[351], 255, "351th value should be 255");
assertEq(view[352], 128, "352th value should be 128");
assertEq(view[353], 127, "353th value should be 127");
assertEq(view[354], 127, "354th value should be 127");
assertEq(view[355], 255, "355th value should be 255");
assertEq(view[356], 52, "356th value should be 52");
assertEq(view[357], 255, "357th value should be 255");
assertEq(view[358], 203, "358th value should be 203");
assertEq(view[359], 255, "359th value should be 255");
assertEq(view[360], 0, "360th value should be 0");
assertEq(view[361], 0, "361th value should be 0");
assertEq(view[362], 0, "362th value should be 0");
assertEq(view[363], 255, "363th value should be 255");
assertEq(view[364], 0, "364th value should be 0");
assertEq(view[365], 0, "365th value should be 0");
assertEq(view[366], 0, "366th value should be 0");
assertEq(view[367], 255, "367th value should be 255");
assertEq(view[368], 0, "368th value should be 0");
assertEq(view[369], 0, "369th value should be 0");
assertEq(view[370], 0, "370th value should be 0");
assertEq(view[371], 255, "371th value should be 255");
assertEq(view[372], 0, "372th value should be 0");
assertEq(view[373], 0, "373th value should be 0");
assertEq(view[374], 0, "374th value should be 0");
assertEq(view[375], 255, "375th value should be 255");
assertEq(view[376], 0, "376th value should be 0");
assertEq(view[377], 0, "377th value should be 0");
assertEq(view[378], 0, "378th value should be 0");
assertEq(view[379], 255, "379th value should be 255");
assertEq(view[380], 0, "380th value should be 0");
assertEq(view[381], 0, "381th value should be 0");
assertEq(view[382], 0, "382th value should be 0");
assertEq(view[383], 255, "383th value should be 255");
assertEq(view[384], 52, "384th value should be 52");
assertEq(view[385], 255, "385th value should be 255");
assertEq(view[386], 203, "386th value should be 203");
assertEq(view[387], 255, "387th value should be 255");
assertEq(view[388], 179, "388th value should be 179");
assertEq(view[389], 127, "389th value should be 127");
assertEq(view[390], 76, "390th value should be 76");
assertEq(view[391], 255, "391th value should be 255");
assertEq(view[392], 205, "392th value should be 205");
assertEq(view[393], 255, "393th value should be 255");
assertEq(view[394], 50, "394th value should be 50");
assertEq(view[395], 255, "395th value should be 255");
assertEq(view[396], 205, "396th value should be 205");
assertEq(view[397], 255, "397th value should be 255");
assertEq(view[398], 50, "398th value should be 50");
assertEq(view[399], 255, "399th value should be 255");
assertEq(view[400], 205, "400th value should be 205");
assertEq(view[401], 255, "401th value should be 255");
assertEq(view[402], 50, "402th value should be 50");
assertEq(view[403], 255, "403th value should be 255");
assertEq(view[404], 205, "404th value should be 205");
assertEq(view[405], 255, "405th value should be 255");
assertEq(view[406], 50, "406th value should be 50");
assertEq(view[407], 255, "407th value should be 255");
assertEq(view[408], 205, "408th value should be 205");
assertEq(view[409], 255, "409th value should be 255");
assertEq(view[410], 50, "410th value should be 50");
assertEq(view[411], 255, "411th value should be 255");
assertEq(view[412], 205, "412th value should be 205");
assertEq(view[413], 255, "413th value should be 255");
assertEq(view[414], 50, "414th value should be 50");
assertEq(view[415], 255, "415th value should be 255");
assertEq(view[416], 154, "416th value should be 154");
assertEq(view[417], 255, "417th value should be 255");
assertEq(view[418], 101, "418th value should be 101");
assertEq(view[419], 255, "419th value should be 255");
assertEq(view[420], 128, "420th value should be 128");
assertEq(view[421], 127, "421th value should be 127");
assertEq(view[422], 127, "422th value should be 127");
assertEq(view[423], 255, "423th value should be 255");
assertEq(view[424], 154, "424th value should be 154");
assertEq(view[425], 255, "425th value should be 255");
assertEq(view[426], 101, "426th value should be 101");
assertEq(view[427], 255, "427th value should be 255");
assertEq(view[428], 128, "428th value should be 128");
assertEq(view[429], 127, "429th value should be 127");
assertEq(view[430], 127, "430th value should be 127");
assertEq(view[431], 255, "431th value should be 255");
assertEq(view[432], 103, "432th value should be 103");
assertEq(view[433], 255, "433th value should be 255");
assertEq(view[434], 152, "434th value should be 152");
assertEq(view[435], 255, "435th value should be 255");
assertEq(view[436], 0, "436th value should be 0");
assertEq(view[437], 0, "437th value should be 0");
assertEq(view[438], 0, "438th value should be 0");
assertEq(view[439], 255, "439th value should be 255");
assertEq(view[440], 0, "440th value should be 0");
assertEq(view[441], 0, "441th value should be 0");
assertEq(view[442], 0, "442th value should be 0");
assertEq(view[443], 255, "443th value should be 255");
assertEq(view[444], 0, "444th value should be 0");
assertEq(view[445], 0, "445th value should be 0");
assertEq(view[446], 0, "446th value should be 0");
assertEq(view[447], 255, "447th value should be 255");
assertEq(view[448], 0, "448th value should be 0");
assertEq(view[449], 0, "449th value should be 0");
assertEq(view[450], 0, "450th value should be 0");
assertEq(view[451], 255, "451th value should be 255");
assertEq(view[452], 0, "452th value should be 0");
assertEq(view[453], 0, "453th value should be 0");
assertEq(view[454], 0, "454th value should be 0");
assertEq(view[455], 255, "455th value should be 255");
assertEq(view[456], 0, "456th value should be 0");
assertEq(view[457], 0, "457th value should be 0");
assertEq(view[458], 0, "458th value should be 0");
assertEq(view[459], 255, "459th value should be 255");
assertEq(view[460], 0, "460th value should be 0");
assertEq(view[461], 0, "461th value should be 0");
assertEq(view[462], 0, "462th value should be 0");
assertEq(view[463], 255, "463th value should be 255");
assertEq(view[464], 78, "464th value should be 78");
assertEq(view[465], 127, "465th value should be 127");
assertEq(view[466], 178, "466th value should be 178");
assertEq(view[467], 255, "467th value should be 255");
assertEq(view[468], 154, "468th value should be 154");
assertEq(view[469], 255, "469th value should be 255");
assertEq(view[470], 101, "470th value should be 101");
assertEq(view[471], 255, "471th value should be 255");
assertEq(view[472], 205, "472th value should be 205");
assertEq(view[473], 255, "473th value should be 255");
assertEq(view[474], 50, "474th value should be 50");
assertEq(view[475], 255, "475th value should be 255");
assertEq(view[476], 205, "476th value should be 205");
assertEq(view[477], 255, "477th value should be 255");
assertEq(view[478], 50, "478th value should be 50");
assertEq(view[479], 255, "479th value should be 255");
assertEq(view[480], 205, "480th value should be 205");
assertEq(view[481], 255, "481th value should be 255");
assertEq(view[482], 50, "482th value should be 50");
assertEq(view[483], 255, "483th value should be 255");
assertEq(view[484], 205, "484th value should be 205");
assertEq(view[485], 255, "485th value should be 255");
assertEq(view[486], 50, "486th value should be 50");
assertEq(view[487], 255, "487th value should be 255");
assertEq(view[488], 179, "488th value should be 179");
assertEq(view[489], 127, "489th value should be 127");
assertEq(view[490], 76, "490th value should be 76");
assertEq(view[491], 255, "491th value should be 255");
assertEq(view[492], 179, "492th value should be 179");
assertEq(view[493], 127, "493th value should be 127");
assertEq(view[494], 76, "494th value should be 76");
assertEq(view[495], 255, "495th value should be 255");
assertEq(view[496], 128, "496th value should be 128");
assertEq(view[497], 127, "497th value should be 127");
assertEq(view[498], 127, "498th value should be 127");
assertEq(view[499], 255, "499th value should be 255");
assertEq(view[500], 52, "500th value should be 52");
assertEq(view[501], 255, "501th value should be 255");
assertEq(view[502], 203, "502th value should be 203");
assertEq(view[503], 255, "503th value should be 255");
assertEq(view[504], 0, "504th value should be 0");
assertEq(view[505], 0, "505th value should be 0");
assertEq(view[506], 0, "506th value should be 0");
assertEq(view[507], 255, "507th value should be 255");
assertEq(view[508], 78, "508th value should be 78");
assertEq(view[509], 127, "509th value should be 127");
assertEq(view[510], 178, "510th value should be 178");
assertEq(view[511], 255, "511th value should be 255");
assertEq(view[512], 52, "512th value should be 52");
assertEq(view[513], 255, "513th value should be 255");
assertEq(view[514], 203, "514th value should be 203");
assertEq(view[515], 255, "515th value should be 255");
assertEq(view[516], 0, "516th value should be 0");
assertEq(view[517], 0, "517th value should be 0");
assertEq(view[518], 0, "518th value should be 0");
assertEq(view[519], 255, "519th value should be 255");
assertEq(view[520], 0, "520th value should be 0");
assertEq(view[521], 0, "521th value should be 0");
assertEq(view[522], 0, "522th value should be 0");
assertEq(view[523], 255, "523th value should be 255");
assertEq(view[524], 0, "524th value should be 0");
assertEq(view[525], 0, "525th value should be 0");
assertEq(view[526], 0, "526th value should be 0");
assertEq(view[527], 255, "527th value should be 255");
assertEq(view[528], 0, "528th value should be 0");
assertEq(view[529], 0, "529th value should be 0");
assertEq(view[530], 0, "530th value should be 0");
assertEq(view[531], 255, "531th value should be 255");
assertEq(view[532], 0, "532th value should be 0");
assertEq(view[533], 0, "533th value should be 0");
assertEq(view[534], 0, "534th value should be 0");
assertEq(view[535], 255, "535th value should be 255");
assertEq(view[536], 0, "536th value should be 0");
assertEq(view[537], 0, "537th value should be 0");
assertEq(view[538], 0, "538th value should be 0");
assertEq(view[539], 255, "539th value should be 255");
assertEq(view[540], 0, "540th value should be 0");
assertEq(view[541], 0, "541th value should be 0");
assertEq(view[542], 0, "542th value should be 0");
assertEq(view[543], 255, "543th value should be 255");
assertEq(view[544], 0, "544th value should be 0");
assertEq(view[545], 0, "545th value should be 0");
assertEq(view[546], 0, "546th value should be 0");
assertEq(view[547], 255, "547th value should be 255");
assertEq(view[548], 154, "548th value should be 154");
assertEq(view[549], 255, "549th value should be 255");
assertEq(view[550], 101, "550th value should be 101");
assertEq(view[551], 255, "551th value should be 255");
assertEq(view[552], 205, "552th value should be 205");
assertEq(view[553], 255, "553th value should be 255");
assertEq(view[554], 50, "554th value should be 50");
assertEq(view[555], 255, "555th value should be 255");
assertEq(view[556], 205, "556th value should be 205");
assertEq(view[557], 255, "557th value should be 255");
assertEq(view[558], 50, "558th value should be 50");
assertEq(view[559], 255, "559th value should be 255");
assertEq(view[560], 205, "560th value should be 205");
assertEq(view[561], 255, "561th value should be 255");
assertEq(view[562], 50, "562th value should be 50");
assertEq(view[563], 255, "563th value should be 255");
assertEq(view[564], 179, "564th value should be 179");
assertEq(view[565], 127, "565th value should be 127");
assertEq(view[566], 76, "566th value should be 76");
assertEq(view[567], 255, "567th value should be 255");
assertEq(view[568], 179, "568th value should be 179");
assertEq(view[569], 127, "569th value should be 127");
assertEq(view[570], 76, "570th value should be 76");
assertEq(view[571], 255, "571th value should be 255");
assertEq(view[572], 154, "572th value should be 154");
assertEq(view[573], 255, "573th value should be 255");
assertEq(view[574], 101, "574th value should be 101");
assertEq(view[575], 255, "575th value should be 255");
assertEq(view[576], 103, "576th value should be 103");
assertEq(view[577], 255, "577th value should be 255");
assertEq(view[578], 152, "578th value should be 152");
assertEq(view[579], 255, "579th value should be 255");
assertEq(view[580], 0, "580th value should be 0");
assertEq(view[581], 0, "581th value should be 0");
assertEq(view[582], 0, "582th value should be 0");
assertEq(view[583], 255, "583th value should be 255");
assertEq(view[584], 0, "584th value should be 0");
assertEq(view[585], 0, "585th value should be 0");
assertEq(view[586], 0, "586th value should be 0");
assertEq(view[587], 255, "587th value should be 255");
assertEq(view[588], 0, "588th value should be 0");
assertEq(view[589], 0, "589th value should be 0");
assertEq(view[590], 0, "590th value should be 0");
assertEq(view[591], 255, "591th value should be 255");
assertEq(view[592], 0, "592th value should be 0");
assertEq(view[593], 0, "593th value should be 0");
assertEq(view[594], 0, "594th value should be 0");
assertEq(view[595], 255, "595th value should be 255");
assertEq(view[596], 0, "596th value should be 0");
assertEq(view[597], 0, "597th value should be 0");
assertEq(view[598], 0, "598th value should be 0");
assertEq(view[599], 255, "599th value should be 255");
assertEq(view[600], 0, "600th value should be 0");
assertEq(view[601], 0, "601th value should be 0");
assertEq(view[602], 0, "602th value should be 0");
assertEq(view[603], 255, "603th value should be 255");
assertEq(view[604], 0, "604th value should be 0");
assertEq(view[605], 0, "605th value should be 0");
assertEq(view[606], 0, "606th value should be 0");
assertEq(view[607], 255, "607th value should be 255");
assertEq(view[608], 0, "608th value should be 0");
assertEq(view[609], 0, "609th value should be 0");
assertEq(view[610], 0, "610th value should be 0");
assertEq(view[611], 255, "611th value should be 255");
assertEq(view[612], 0, "612th value should be 0");
assertEq(view[613], 0, "613th value should be 0");
assertEq(view[614], 0, "614th value should be 0");
assertEq(view[615], 255, "615th value should be 255");
assertEq(view[616], 0, "616th value should be 0");
assertEq(view[617], 0, "617th value should be 0");
assertEq(view[618], 0, "618th value should be 0");
assertEq(view[619], 255, "619th value should be 255");
assertEq(view[620], 0, "620th value should be 0");
assertEq(view[621], 0, "621th value should be 0");
assertEq(view[622], 0, "622th value should be 0");
assertEq(view[623], 255, "623th value should be 255");
assertEq(view[624], 0, "624th value should be 0");
assertEq(view[625], 0, "625th value should be 0");
assertEq(view[626], 0, "626th value should be 0");
assertEq(view[627], 255, "627th value should be 255");
assertEq(view[628], 154, "628th value should be 154");
assertEq(view[629], 255, "629th value should be 255");
assertEq(view[630], 101, "630th value should be 101");
assertEq(view[631], 255, "631th value should be 255");
assertEq(view[632], 205, "632th value should be 205");
assertEq(view[633], 255, "633th value should be 255");
assertEq(view[634], 50, "634th value should be 50");
assertEq(view[635], 255, "635th value should be 255");
assertEq(view[636], 205, "636th value should be 205");
assertEq(view[637], 255, "637th value should be 255");
assertEq(view[638], 50, "638th value should be 50");
assertEq(view[639], 255, "639th value should be 255");
assertEq(view[640], 179, "640th value should be 179");
assertEq(view[641], 127, "641th value should be 127");
assertEq(view[642], 76, "642th value should be 76");
assertEq(view[643], 255, "643th value should be 255");
assertEq(view[644], 179, "644th value should be 179");
assertEq(view[645], 127, "645th value should be 127");
assertEq(view[646], 76, "646th value should be 76");
assertEq(view[647], 255, "647th value should be 255");
assertEq(view[648], 154, "648th value should be 154");
assertEq(view[649], 255, "649th value should be 255");
assertEq(view[650], 101, "650th value should be 101");
assertEq(view[651], 255, "651th value should be 255");
assertEq(view[652], 128, "652th value should be 128");
assertEq(view[653], 127, "653th value should be 127");
assertEq(view[654], 127, "654th value should be 127");
assertEq(view[655], 255, "655th value should be 255");
assertEq(view[656], 52, "656th value should be 52");
assertEq(view[657], 255, "657th value should be 255");
assertEq(view[658], 203, "658th value should be 203");
assertEq(view[659], 255, "659th value should be 255");
assertEq(view[660], 0, "660th value should be 0");
assertEq(view[661], 0, "661th value should be 0");
assertEq(view[662], 0, "662th value should be 0");
assertEq(view[663], 255, "663th value should be 255");
assertEq(view[664], 0, "664th value should be 0");
assertEq(view[665], 0, "665th value should be 0");
assertEq(view[666], 0, "666th value should be 0");
assertEq(view[667], 255, "667th value should be 255");
assertEq(view[668], 0, "668th value should be 0");
assertEq(view[669], 0, "669th value should be 0");
assertEq(view[670], 0, "670th value should be 0");
assertEq(view[671], 255, "671th value should be 255");
assertEq(view[672], 0, "672th value should be 0");
assertEq(view[673], 0, "673th value should be 0");
assertEq(view[674], 0, "674th value should be 0");
assertEq(view[675], 255, "675th value should be 255");
assertEq(view[676], 0, "676th value should be 0");
assertEq(view[677], 0, "677th value should be 0");
assertEq(view[678], 0, "678th value should be 0");
assertEq(view[679], 255, "679th value should be 255");
assertEq(view[680], 0, "680th value should be 0");
assertEq(view[681], 0, "681th value should be 0");
assertEq(view[682], 0, "682th value should be 0");
assertEq(view[683], 255, "683th value should be 255");
assertEq(view[684], 0, "684th value should be 0");
assertEq(view[685], 0, "685th value should be 0");
assertEq(view[686], 0, "686th value should be 0");
assertEq(view[687], 255, "687th value should be 255");
assertEq(view[688], 0, "688th value should be 0");
assertEq(view[689], 0, "689th value should be 0");
assertEq(view[690], 0, "690th value should be 0");
assertEq(view[691], 255, "691th value should be 255");
assertEq(view[692], 0, "692th value should be 0");
assertEq(view[693], 0, "693th value should be 0");
assertEq(view[694], 0, "694th value should be 0");
assertEq(view[695], 255, "695th value should be 255");
assertEq(view[696], 0, "696th value should be 0");
assertEq(view[697], 0, "697th value should be 0");
assertEq(view[698], 0, "698th value should be 0");
assertEq(view[699], 255, "699th value should be 255");
assertEq(view[700], 0, "700th value should be 0");
assertEq(view[701], 0, "701th value should be 0");
assertEq(view[702], 0, "702th value should be 0");
assertEq(view[703], 255, "703th value should be 255");
assertEq(view[704], 0, "704th value should be 0");
assertEq(view[705], 0, "705th value should be 0");
assertEq(view[706], 0, "706th value should be 0");
assertEq(view[707], 255, "707th value should be 255");
assertEq(view[708], 154, "708th value should be 154");
assertEq(view[709], 255, "709th value should be 255");
assertEq(view[710], 101, "710th value should be 101");
assertEq(view[711], 255, "711th value should be 255");
assertEq(view[712], 179, "712th value should be 179");
assertEq(view[713], 127, "713th value should be 127");
assertEq(view[714], 76, "714th value should be 76");
assertEq(view[715], 255, "715th value should be 255");
assertEq(view[716], 205, "716th value should be 205");
assertEq(view[717], 255, "717th value should be 255");
assertEq(view[718], 50, "718th value should be 50");
assertEq(view[719], 255, "719th value should be 255");
assertEq(view[720], 154, "720th value should be 154");
assertEq(view[721], 255, "721th value should be 255");
assertEq(view[722], 101, "722th value should be 101");
assertEq(view[723], 255, "723th value should be 255");
assertEq(view[724], 52, "724th value should be 52");
assertEq(view[725], 255, "725th value should be 255");
assertEq(view[726], 203, "726th value should be 203");
assertEq(view[727], 255, "727th value should be 255");
assertEq(view[728], 128, "728th value should be 128");
assertEq(view[729], 127, "729th value should be 127");
assertEq(view[730], 127, "730th value should be 127");
assertEq(view[731], 255, "731th value should be 255");
assertEq(view[732], 78, "732th value should be 78");
assertEq(view[733], 127, "733th value should be 127");
assertEq(view[734], 178, "734th value should be 178");
assertEq(view[735], 255, "735th value should be 255");
assertEq(view[736], 0, "736th value should be 0");
assertEq(view[737], 0, "737th value should be 0");
assertEq(view[738], 0, "738th value should be 0");
assertEq(view[739], 255, "739th value should be 255");
assertEq(view[740], 0, "740th value should be 0");
assertEq(view[741], 0, "741th value should be 0");
assertEq(view[742], 0, "742th value should be 0");
assertEq(view[743], 255, "743th value should be 255");
assertEq(view[744], 0, "744th value should be 0");
assertEq(view[745], 0, "745th value should be 0");
assertEq(view[746], 0, "746th value should be 0");
assertEq(view[747], 255, "747th value should be 255");
assertEq(view[748], 0, "748th value should be 0");
assertEq(view[749], 0, "749th value should be 0");
assertEq(view[750], 0, "750th value should be 0");
assertEq(view[751], 255, "751th value should be 255");
assertEq(view[752], 0, "752th value should be 0");
assertEq(view[753], 0, "753th value should be 0");
assertEq(view[754], 0, "754th value should be 0");
assertEq(view[755], 255, "755th value should be 255");
assertEq(view[756], 0, "756th value should be 0");
assertEq(view[757], 0, "757th value should be 0");
assertEq(view[758], 0, "758th value should be 0");
assertEq(view[759], 255, "759th value should be 255");
assertEq(view[760], 0, "760th value should be 0");
assertEq(view[761], 0, "761th value should be 0");
assertEq(view[762], 0, "762th value should be 0");
assertEq(view[763], 255, "763th value should be 255");
assertEq(view[764], 0, "764th value should be 0");
assertEq(view[765], 0, "765th value should be 0");
assertEq(view[766], 0, "766th value should be 0");
assertEq(view[767], 255, "767th value should be 255");
assertEq(view[768], 0, "768th value should be 0");
assertEq(view[769], 0, "769th value should be 0");
assertEq(view[770], 0, "770th value should be 0");
assertEq(view[771], 255, "771th value should be 255");
assertEq(view[772], 0, "772th value should be 0");
assertEq(view[773], 0, "773th value should be 0");
assertEq(view[774], 0, "774th value should be 0");
assertEq(view[775], 255, "775th value should be 255");
assertEq(view[776], 0, "776th value should be 0");
assertEq(view[777], 0, "777th value should be 0");
assertEq(view[778], 0, "778th value should be 0");
assertEq(view[779], 255, "779th value should be 255");
assertEq(view[780], 0, "780th value should be 0");
assertEq(view[781], 0, "781th value should be 0");
assertEq(view[782], 0, "782th value should be 0");
assertEq(view[783], 255, "783th value should be 255");
assertEq(view[784], 78, "784th value should be 78");
assertEq(view[785], 127, "785th value should be 127");
assertEq(view[786], 178, "786th value should be 178");
assertEq(view[787], 255, "787th value should be 255");
assertEq(view[788], 154, "788th value should be 154");
assertEq(view[789], 255, "789th value should be 255");
assertEq(view[790], 101, "790th value should be 101");
assertEq(view[791], 255, "791th value should be 255");
assertEq(view[792], 179, "792th value should be 179");
assertEq(view[793], 127, "793th value should be 127");
assertEq(view[794], 76, "794th value should be 76");
assertEq(view[795], 255, "795th value should be 255");
assertEq(view[796], 205, "796th value should be 205");
assertEq(view[797], 255, "797th value should be 255");
assertEq(view[798], 50, "798th value should be 50");
assertEq(view[799], 255, "799th value should be 255");
assertEq(view[800], 128, "800th value should be 128");
assertEq(view[801], 127, "801th value should be 127");
assertEq(view[802], 127, "802th value should be 127");
assertEq(view[803], 255, "803th value should be 255");
assertEq(view[804], 0, "804th value should be 0");
assertEq(view[805], 0, "805th value should be 0");
assertEq(view[806], 0, "806th value should be 0");
assertEq(view[807], 255, "807th value should be 255");
assertEq(view[808], 26, "808th value should be 26");
assertEq(view[809], 127, "809th value should be 127");
assertEq(view[810], 229, "810th value should be 229");
assertEq(view[811], 255, "811th value should be 255");
assertEq(view[812], 0, "812th value should be 0");
assertEq(view[813], 0, "813th value should be 0");
assertEq(view[814], 0, "814th value should be 0");
assertEq(view[815], 255, "815th value should be 255");
assertEq(view[816], 0, "816th value should be 0");
assertEq(view[817], 0, "817th value should be 0");
assertEq(view[818], 0, "818th value should be 0");
assertEq(view[819], 255, "819th value should be 255");
assertEq(view[820], 0, "820th value should be 0");
assertEq(view[821], 0, "821th value should be 0");
assertEq(view[822], 0, "822th value should be 0");
assertEq(view[823], 255, "823th value should be 255");
assertEq(view[824], 0, "824th value should be 0");
assertEq(view[825], 0, "825th value should be 0");
assertEq(view[826], 0, "826th value should be 0");
assertEq(view[827], 255, "827th value should be 255");
assertEq(view[828], 0, "828th value should be 0");
assertEq(view[829], 0, "829th value should be 0");
assertEq(view[830], 0, "830th value should be 0");
assertEq(view[831], 255, "831th value should be 255");
assertEq(view[832], 0, "832th value should be 0");
assertEq(view[833], 0, "833th value should be 0");
assertEq(view[834], 0, "834th value should be 0");
assertEq(view[835], 255, "835th value should be 255");
assertEq(view[836], 0, "836th value should be 0");
assertEq(view[837], 0, "837th value should be 0");
assertEq(view[838], 0, "838th value should be 0");
assertEq(view[839], 255, "839th value should be 255");
assertEq(view[840], 0, "840th value should be 0");
assertEq(view[841], 0, "841th value should be 0");
assertEq(view[842], 0, "842th value should be 0");
assertEq(view[843], 255, "843th value should be 255");
assertEq(view[844], 0, "844th value should be 0");
assertEq(view[845], 0, "845th value should be 0");
assertEq(view[846], 0, "846th value should be 0");
assertEq(view[847], 255, "847th value should be 255");
assertEq(view[848], 0, "848th value should be 0");
assertEq(view[849], 0, "849th value should be 0");
assertEq(view[850], 0, "850th value should be 0");
assertEq(view[851], 255, "851th value should be 255");
assertEq(view[852], 0, "852th value should be 0");
assertEq(view[853], 0, "853th value should be 0");
assertEq(view[854], 0, "854th value should be 0");
assertEq(view[855], 255, "855th value should be 255");
assertEq(view[856], 0, "856th value should be 0");
assertEq(view[857], 0, "857th value should be 0");
assertEq(view[858], 0, "858th value should be 0");
assertEq(view[859], 255, "859th value should be 255");
assertEq(view[860], 0, "860th value should be 0");
assertEq(view[861], 0, "861th value should be 0");
assertEq(view[862], 0, "862th value should be 0");
assertEq(view[863], 255, "863th value should be 255");
assertEq(view[864], 103, "864th value should be 103");
assertEq(view[865], 255, "865th value should be 255");
assertEq(view[866], 152, "866th value should be 152");
assertEq(view[867], 255, "867th value should be 255");
assertEq(view[868], 154, "868th value should be 154");
assertEq(view[869], 255, "869th value should be 255");
assertEq(view[870], 101, "870th value should be 101");
assertEq(view[871], 255, "871th value should be 255");
assertEq(view[872], 179, "872th value should be 179");
assertEq(view[873], 127, "873th value should be 127");
assertEq(view[874], 76, "874th value should be 76");
assertEq(view[875], 255, "875th value should be 255");
assertEq(view[876], 205, "876th value should be 205");
assertEq(view[877], 255, "877th value should be 255");
assertEq(view[878], 50, "878th value should be 50");
assertEq(view[879], 255, "879th value should be 255");
assertEq(view[880], 179, "880th value should be 179");
assertEq(view[881], 127, "881th value should be 127");
assertEq(view[882], 76, "882th value should be 76");
assertEq(view[883], 255, "883th value should be 255");
assertEq(view[884], 179, "884th value should be 179");
assertEq(view[885], 127, "885th value should be 127");
assertEq(view[886], 76, "886th value should be 76");
assertEq(view[887], 255, "887th value should be 255");
assertEq(view[888], 128, "888th value should be 128");
assertEq(view[889], 127, "889th value should be 127");
assertEq(view[890], 127, "890th value should be 127");
assertEq(view[891], 255, "891th value should be 255");
assertEq(view[892], 103, "892th value should be 103");
assertEq(view[893], 255, "893th value should be 255");
assertEq(view[894], 152, "894th value should be 152");
assertEq(view[895], 255, "895th value should be 255");
assertEq(view[896], 26, "896th value should be 26");
assertEq(view[897], 127, "897th value should be 127");
assertEq(view[898], 229, "898th value should be 229");
assertEq(view[899], 255, "899th value should be 255");
assertEq(view[900], 0, "900th value should be 0");
assertEq(view[901], 0, "901th value should be 0");
assertEq(view[902], 0, "902th value should be 0");
assertEq(view[903], 255, "903th value should be 255");
assertEq(view[904], 0, "904th value should be 0");
assertEq(view[905], 0, "905th value should be 0");
assertEq(view[906], 0, "906th value should be 0");
assertEq(view[907], 255, "907th value should be 255");
assertEq(view[908], 0, "908th value should be 0");
assertEq(view[909], 0, "909th value should be 0");
assertEq(view[910], 0, "910th value should be 0");
assertEq(view[911], 255, "911th value should be 255");
assertEq(view[912], 0, "912th value should be 0");
assertEq(view[913], 0, "913th value should be 0");
assertEq(view[914], 0, "914th value should be 0");
assertEq(view[915], 255, "915th value should be 255");
assertEq(view[916], 0, "916th value should be 0");
assertEq(view[917], 0, "917th value should be 0");
assertEq(view[918], 0, "918th value should be 0");
assertEq(view[919], 255, "919th value should be 255");
assertEq(view[920], 0, "920th value should be 0");
assertEq(view[921], 0, "921th value should be 0");
assertEq(view[922], 0, "922th value should be 0");
assertEq(view[923], 255, "923th value should be 255");
assertEq(view[924], 0, "924th value should be 0");
assertEq(view[925], 0, "925th value should be 0");
assertEq(view[926], 0, "926th value should be 0");
assertEq(view[927], 255, "927th value should be 255");
assertEq(view[928], 0, "928th value should be 0");
assertEq(view[929], 0, "929th value should be 0");
assertEq(view[930], 0, "930th value should be 0");
assertEq(view[931], 255, "931th value should be 255");
assertEq(view[932], 0, "932th value should be 0");
assertEq(view[933], 0, "933th value should be 0");
assertEq(view[934], 0, "934th value should be 0");
assertEq(view[935], 255, "935th value should be 255");
assertEq(view[936], 0, "936th value should be 0");
assertEq(view[937], 0, "937th value should be 0");
assertEq(view[938], 0, "938th value should be 0");
assertEq(view[939], 255, "939th value should be 255");
assertEq(view[940], 0, "940th value should be 0");
assertEq(view[941], 0, "941th value should be 0");
assertEq(view[942], 0, "942th value should be 0");
assertEq(view[943], 255, "943th value should be 255");
assertEq(view[944], 0, "944th value should be 0");
assertEq(view[945], 0, "945th value should be 0");
assertEq(view[946], 0, "946th value should be 0");
assertEq(view[947], 255, "947th value should be 255");
assertEq(view[948], 154, "948th value should be 154");
assertEq(view[949], 255, "949th value should be 255");
assertEq(view[950], 101, "950th value should be 101");
assertEq(view[951], 255, "951th value should be 255");
assertEq(view[952], 179, "952th value should be 179");
assertEq(view[953], 127, "953th value should be 127");
assertEq(view[954], 76, "954th value should be 76");
assertEq(view[955], 255, "955th value should be 255");
assertEq(view[956], 205, "956th value should be 205");
assertEq(view[957], 255, "957th value should be 255");
assertEq(view[958], 50, "958th value should be 50");
assertEq(view[959], 255, "959th value should be 255");
assertEq(view[960], 179, "960th value should be 179");
assertEq(view[961], 127, "961th value should be 127");
assertEq(view[962], 76, "962th value should be 76");
assertEq(view[963], 255, "963th value should be 255");
assertEq(view[964], 179, "964th value should be 179");
assertEq(view[965], 127, "965th value should be 127");
assertEq(view[966], 76, "966th value should be 76");
assertEq(view[967], 255, "967th value should be 255");
assertEq(view[968], 179, "968th value should be 179");
assertEq(view[969], 127, "969th value should be 127");
assertEq(view[970], 76, "970th value should be 76");
assertEq(view[971], 255, "971th value should be 255");
assertEq(view[972], 154, "972th value should be 154");
assertEq(view[973], 255, "973th value should be 255");
assertEq(view[974], 101, "974th value should be 101");
assertEq(view[975], 255, "975th value should be 255");
assertEq(view[976], 103, "976th value should be 103");
assertEq(view[977], 255, "977th value should be 255");
assertEq(view[978], 152, "978th value should be 152");
assertEq(view[979], 255, "979th value should be 255");
assertEq(view[980], 0, "980th value should be 0");
assertEq(view[981], 0, "981th value should be 0");
assertEq(view[982], 0, "982th value should be 0");
assertEq(view[983], 255, "983th value should be 255");
assertEq(view[984], 0, "984th value should be 0");
assertEq(view[985], 0, "985th value should be 0");
assertEq(view[986], 0, "986th value should be 0");
assertEq(view[987], 255, "987th value should be 255");
assertEq(view[988], 0, "988th value should be 0");
assertEq(view[989], 0, "989th value should be 0");
assertEq(view[990], 0, "990th value should be 0");
assertEq(view[991], 255, "991th value should be 255");
assertEq(view[992], 0, "992th value should be 0");
assertEq(view[993], 0, "993th value should be 0");
assertEq(view[994], 0, "994th value should be 0");
assertEq(view[995], 255, "995th value should be 255");
assertEq(view[996], 0, "996th value should be 0");
assertEq(view[997], 0, "997th value should be 0");
assertEq(view[998], 0, "998th value should be 0");
assertEq(view[999], 255, "999th value should be 255");
assertEq(view[1000], 0, "1000th value should be 0");
assertEq(view[1001], 0, "1001th value should be 0");
assertEq(view[1002], 0, "1002th value should be 0");
assertEq(view[1003], 255, "1003th value should be 255");
assertEq(view[1004], 0, "1004th value should be 0");
assertEq(view[1005], 0, "1005th value should be 0");
assertEq(view[1006], 0, "1006th value should be 0");
assertEq(view[1007], 255, "1007th value should be 255");
assertEq(view[1008], 0, "1008th value should be 0");
assertEq(view[1009], 0, "1009th value should be 0");
assertEq(view[1010], 0, "1010th value should be 0");
assertEq(view[1011], 255, "1011th value should be 255");
assertEq(view[1012], 0, "1012th value should be 0");
assertEq(view[1013], 0, "1013th value should be 0");
assertEq(view[1014], 0, "1014th value should be 0");
assertEq(view[1015], 255, "1015th value should be 255");
assertEq(view[1016], 0, "1016th value should be 0");
assertEq(view[1017], 0, "1017th value should be 0");
assertEq(view[1018], 0, "1018th value should be 0");
assertEq(view[1019], 255, "1019th value should be 255");
assertEq(view[1020], 0, "1020th value should be 0");
assertEq(view[1021], 0, "1021th value should be 0");
assertEq(view[1022], 0, "1022th value should be 0");
assertEq(view[1023], 255, "1023th value should be 255");
assertEq(view[1024], 0, "1024th value should be 0");
assertEq(view[1025], 0, "1025th value should be 0");
assertEq(view[1026], 0, "1026th value should be 0");
assertEq(view[1027], 255, "1027th value should be 255");
assertEq(view[1028], 154, "1028th value should be 154");
assertEq(view[1029], 255, "1029th value should be 255");
assertEq(view[1030], 101, "1030th value should be 101");
assertEq(view[1031], 255, "1031th value should be 255");
assertEq(view[1032], 205, "1032th value should be 205");
assertEq(view[1033], 255, "1033th value should be 255");
assertEq(view[1034], 50, "1034th value should be 50");
assertEq(view[1035], 255, "1035th value should be 255");
assertEq(view[1036], 205, "1036th value should be 205");
assertEq(view[1037], 255, "1037th value should be 255");
assertEq(view[1038], 50, "1038th value should be 50");
assertEq(view[1039], 255, "1039th value should be 255");
assertEq(view[1040], 205, "1040th value should be 205");
assertEq(view[1041], 255, "1041th value should be 255");
assertEq(view[1042], 50, "1042th value should be 50");
assertEq(view[1043], 255, "1043th value should be 255");
assertEq(view[1044], 179, "1044th value should be 179");
assertEq(view[1045], 127, "1045th value should be 127");
assertEq(view[1046], 76, "1046th value should be 76");
assertEq(view[1047], 255, "1047th value should be 255");
assertEq(view[1048], 179, "1048th value should be 179");
assertEq(view[1049], 127, "1049th value should be 127");
assertEq(view[1050], 76, "1050th value should be 76");
assertEq(view[1051], 255, "1051th value should be 255");
assertEq(view[1052], 154, "1052th value should be 154");
assertEq(view[1053], 255, "1053th value should be 255");
assertEq(view[1054], 101, "1054th value should be 101");
assertEq(view[1055], 255, "1055th value should be 255");
assertEq(view[1056], 128, "1056th value should be 128");
assertEq(view[1057], 127, "1057th value should be 127");
assertEq(view[1058], 127, "1058th value should be 127");
assertEq(view[1059], 255, "1059th value should be 255");
assertEq(view[1060], 0, "1060th value should be 0");
assertEq(view[1061], 0, "1061th value should be 0");
assertEq(view[1062], 0, "1062th value should be 0");
assertEq(view[1063], 255, "1063th value should be 255");
assertEq(view[1064], 0, "1064th value should be 0");
assertEq(view[1065], 0, "1065th value should be 0");
assertEq(view[1066], 0, "1066th value should be 0");
assertEq(view[1067], 255, "1067th value should be 255");
assertEq(view[1068], 26, "1068th value should be 26");
assertEq(view[1069], 127, "1069th value should be 127");
assertEq(view[1070], 229, "1070th value should be 229");
assertEq(view[1071], 255, "1071th value should be 255");
assertEq(view[1072], 26, "1072th value should be 26");
assertEq(view[1073], 127, "1073th value should be 127");
assertEq(view[1074], 229, "1074th value should be 229");
assertEq(view[1075], 255, "1075th value should be 255");
assertEq(view[1076], 0, "1076th value should be 0");
assertEq(view[1077], 0, "1077th value should be 0");
assertEq(view[1078], 0, "1078th value should be 0");
assertEq(view[1079], 255, "1079th value should be 255");
assertEq(view[1080], 0, "1080th value should be 0");
assertEq(view[1081], 0, "1081th value should be 0");
assertEq(view[1082], 0, "1082th value should be 0");
assertEq(view[1083], 255, "1083th value should be 255");
assertEq(view[1084], 0, "1084th value should be 0");
assertEq(view[1085], 0, "1085th value should be 0");
assertEq(view[1086], 0, "1086th value should be 0");
assertEq(view[1087], 255, "1087th value should be 255");
assertEq(view[1088], 0, "1088th value should be 0");
assertEq(view[1089], 0, "1089th value should be 0");
assertEq(view[1090], 0, "1090th value should be 0");
assertEq(view[1091], 255, "1091th value should be 255");
assertEq(view[1092], 0, "1092th value should be 0");
assertEq(view[1093], 0, "1093th value should be 0");
assertEq(view[1094], 0, "1094th value should be 0");
assertEq(view[1095], 255, "1095th value should be 255");
assertEq(view[1096], 0, "1096th value should be 0");
assertEq(view[1097], 0, "1097th value should be 0");
assertEq(view[1098], 0, "1098th value should be 0");
assertEq(view[1099], 255, "1099th value should be 255");
assertEq(view[1100], 0, "1100th value should be 0");
assertEq(view[1101], 0, "1101th value should be 0");
assertEq(view[1102], 0, "1102th value should be 0");
assertEq(view[1103], 255, "1103th value should be 255");
assertEq(view[1104], 0, "1104th value should be 0");
assertEq(view[1105], 0, "1105th value should be 0");
assertEq(view[1106], 0, "1106th value should be 0");
assertEq(view[1107], 255, "1107th value should be 255");
assertEq(view[1108], 154, "1108th value should be 154");
assertEq(view[1109], 255, "1109th value should be 255");
assertEq(view[1110], 101, "1110th value should be 101");
assertEq(view[1111], 255, "1111th value should be 255");
assertEq(view[1112], 205, "1112th value should be 205");
assertEq(view[1113], 255, "1113th value should be 255");
assertEq(view[1114], 50, "1114th value should be 50");
assertEq(view[1115], 255, "1115th value should be 255");
assertEq(view[1116], 205, "1116th value should be 205");
assertEq(view[1117], 255, "1117th value should be 255");
assertEq(view[1118], 50, "1118th value should be 50");
assertEq(view[1119], 255, "1119th value should be 255");
assertEq(view[1120], 205, "1120th value should be 205");
assertEq(view[1121], 255, "1121th value should be 255");
assertEq(view[1122], 50, "1122th value should be 50");
assertEq(view[1123], 255, "1123th value should be 255");
assertEq(view[1124], 205, "1124th value should be 205");
assertEq(view[1125], 255, "1125th value should be 255");
assertEq(view[1126], 50, "1126th value should be 50");
assertEq(view[1127], 255, "1127th value should be 255");
assertEq(view[1128], 205, "1128th value should be 205");
assertEq(view[1129], 255, "1129th value should be 255");
assertEq(view[1130], 50, "1130th value should be 50");
assertEq(view[1131], 255, "1131th value should be 255");
assertEq(view[1132], 179, "1132th value should be 179");
assertEq(view[1133], 127, "1133th value should be 127");
assertEq(view[1134], 76, "1134th value should be 76");
assertEq(view[1135], 255, "1135th value should be 255");
assertEq(view[1136], 154, "1136th value should be 154");
assertEq(view[1137], 255, "1137th value should be 255");
assertEq(view[1138], 101, "1138th value should be 101");
assertEq(view[1139], 255, "1139th value should be 255");
assertEq(view[1140], 128, "1140th value should be 128");
assertEq(view[1141], 127, "1141th value should be 127");
assertEq(view[1142], 127, "1142th value should be 127");
assertEq(view[1143], 255, "1143th value should be 255");
assertEq(view[1144], 128, "1144th value should be 128");
assertEq(view[1145], 127, "1145th value should be 127");
assertEq(view[1146], 127, "1146th value should be 127");
assertEq(view[1147], 255, "1147th value should be 255");
assertEq(view[1148], 103, "1148th value should be 103");
assertEq(view[1149], 255, "1149th value should be 255");
assertEq(view[1150], 152, "1150th value should be 152");
assertEq(view[1151], 255, "1151th value should be 255");
assertEq(view[1152], 78, "1152th value should be 78");
assertEq(view[1153], 127, "1153th value should be 127");
assertEq(view[1154], 178, "1154th value should be 178");
assertEq(view[1155], 255, "1155th value should be 255");
assertEq(view[1156], 0, "1156th value should be 0");
assertEq(view[1157], 0, "1157th value should be 0");
assertEq(view[1158], 0, "1158th value should be 0");
assertEq(view[1159], 255, "1159th value should be 255");
assertEq(view[1160], 0, "1160th value should be 0");
assertEq(view[1161], 0, "1161th value should be 0");
assertEq(view[1162], 0, "1162th value should be 0");
assertEq(view[1163], 255, "1163th value should be 255");
assertEq(view[1164], 0, "1164th value should be 0");
assertEq(view[1165], 0, "1165th value should be 0");
assertEq(view[1166], 0, "1166th value should be 0");
assertEq(view[1167], 255, "1167th value should be 255");
assertEq(view[1168], 0, "1168th value should be 0");
assertEq(view[1169], 0, "1169th value should be 0");
assertEq(view[1170], 0, "1170th value should be 0");
assertEq(view[1171], 255, "1171th value should be 255");
assertEq(view[1172], 0, "1172th value should be 0");
assertEq(view[1173], 0, "1173th value should be 0");
assertEq(view[1174], 0, "1174th value should be 0");
assertEq(view[1175], 255, "1175th value should be 255");
assertEq(view[1176], 0, "1176th value should be 0");
assertEq(view[1177], 0, "1177th value should be 0");
assertEq(view[1178], 0, "1178th value should be 0");
assertEq(view[1179], 255, "1179th value should be 255");
assertEq(view[1180], 0, "1180th value should be 0");
assertEq(view[1181], 0, "1181th value should be 0");
assertEq(view[1182], 0, "1182th value should be 0");
assertEq(view[1183], 255, "1183th value should be 255");
assertEq(view[1184], 26, "1184th value should be 26");
assertEq(view[1185], 127, "1185th value should be 127");
assertEq(view[1186], 229, "1186th value should be 229");
assertEq(view[1187], 255, "1187th value should be 255");
assertEq(view[1188], 154, "1188th value should be 154");
assertEq(view[1189], 255, "1189th value should be 255");
assertEq(view[1190], 101, "1190th value should be 101");
assertEq(view[1191], 255, "1191th value should be 255");
assertEq(view[1192], 205, "1192th value should be 205");
assertEq(view[1193], 255, "1193th value should be 255");
assertEq(view[1194], 50, "1194th value should be 50");
assertEq(view[1195], 255, "1195th value should be 255");
assertEq(view[1196], 205, "1196th value should be 205");
assertEq(view[1197], 255, "1197th value should be 255");
assertEq(view[1198], 50, "1198th value should be 50");
assertEq(view[1199], 255, "1199th value should be 255");
assertEq(view[1200], 230, "1200th value should be 230");
assertEq(view[1201], 127, "1201th value should be 127");
assertEq(view[1202], 25, "1202th value should be 25");
assertEq(view[1203], 255, "1203th value should be 255");
assertEq(view[1204], 205, "1204th value should be 205");
assertEq(view[1205], 255, "1205th value should be 255");
assertEq(view[1206], 50, "1206th value should be 50");
assertEq(view[1207], 255, "1207th value should be 255");
assertEq(view[1208], 205, "1208th value should be 205");
assertEq(view[1209], 255, "1209th value should be 255");
assertEq(view[1210], 50, "1210th value should be 50");
assertEq(view[1211], 255, "1211th value should be 255");
assertEq(view[1212], 205, "1212th value should be 205");
assertEq(view[1213], 255, "1213th value should be 255");
assertEq(view[1214], 50, "1214th value should be 50");
assertEq(view[1215], 255, "1215th value should be 255");
assertEq(view[1216], 205, "1216th value should be 205");
assertEq(view[1217], 255, "1217th value should be 255");
assertEq(view[1218], 50, "1218th value should be 50");
assertEq(view[1219], 255, "1219th value should be 255");
assertEq(view[1220], 154, "1220th value should be 154");
assertEq(view[1221], 255, "1221th value should be 255");
assertEq(view[1222], 101, "1222th value should be 101");
assertEq(view[1223], 255, "1223th value should be 255");
assertEq(view[1224], 154, "1224th value should be 154");
assertEq(view[1225], 255, "1225th value should be 255");
assertEq(view[1226], 101, "1226th value should be 101");
assertEq(view[1227], 255, "1227th value should be 255");
assertEq(view[1228], 154, "1228th value should be 154");
assertEq(view[1229], 255, "1229th value should be 255");
assertEq(view[1230], 101, "1230th value should be 101");
assertEq(view[1231], 255, "1231th value should be 255");
assertEq(view[1232], 128, "1232th value should be 128");
assertEq(view[1233], 127, "1233th value should be 127");
assertEq(view[1234], 127, "1234th value should be 127");
assertEq(view[1235], 255, "1235th value should be 255");
assertEq(view[1236], 26, "1236th value should be 26");
assertEq(view[1237], 127, "1237th value should be 127");
assertEq(view[1238], 229, "1238th value should be 229");
assertEq(view[1239], 255, "1239th value should be 255");
assertEq(view[1240], 0, "1240th value should be 0");
assertEq(view[1241], 0, "1241th value should be 0");
assertEq(view[1242], 0, "1242th value should be 0");
assertEq(view[1243], 255, "1243th value should be 255");
assertEq(view[1244], 0, "1244th value should be 0");
assertEq(view[1245], 0, "1245th value should be 0");
assertEq(view[1246], 0, "1246th value should be 0");
assertEq(view[1247], 255, "1247th value should be 255");
assertEq(view[1248], 0, "1248th value should be 0");
assertEq(view[1249], 0, "1249th value should be 0");
assertEq(view[1250], 0, "1250th value should be 0");
assertEq(view[1251], 255, "1251th value should be 255");
assertEq(view[1252], 0, "1252th value should be 0");
assertEq(view[1253], 0, "1253th value should be 0");
assertEq(view[1254], 0, "1254th value should be 0");
assertEq(view[1255], 255, "1255th value should be 255");
assertEq(view[1256], 0, "1256th value should be 0");
assertEq(view[1257], 0, "1257th value should be 0");
assertEq(view[1258], 0, "1258th value should be 0");
assertEq(view[1259], 255, "1259th value should be 255");
assertEq(view[1260], 0, "1260th value should be 0");
assertEq(view[1261], 0, "1261th value should be 0");
assertEq(view[1262], 0, "1262th value should be 0");
assertEq(view[1263], 255, "1263th value should be 255");
assertEq(view[1264], 78, "1264th value should be 78");
assertEq(view[1265], 127, "1265th value should be 127");
assertEq(view[1266], 178, "1266th value should be 178");
assertEq(view[1267], 255, "1267th value should be 255");
assertEq(view[1268], 179, "1268th value should be 179");
assertEq(view[1269], 127, "1269th value should be 127");
assertEq(view[1270], 76, "1270th value should be 76");
assertEq(view[1271], 255, "1271th value should be 255");
assertEq(view[1272], 205, "1272th value should be 205");
assertEq(view[1273], 255, "1273th value should be 255");
assertEq(view[1274], 50, "1274th value should be 50");
assertEq(view[1275], 255, "1275th value should be 255");
assertEq(view[1276], 205, "1276th value should be 205");
assertEq(view[1277], 255, "1277th value should be 255");
assertEq(view[1278], 50, "1278th value should be 50");
assertEq(view[1279], 255, "1279th value should be 255");
assertEq(view[1280], 0, "1280th value should be 0");
assertEq(view[1281], 0, "1281th value should be 0");
assertEq(view[1282], 0, "1282th value should be 0");
assertEq(view[1283], 255, "1283th value should be 255");
assertEq(view[1284], 205, "1284th value should be 205");
assertEq(view[1285], 255, "1285th value should be 255");
assertEq(view[1286], 50, "1286th value should be 50");
assertEq(view[1287], 255, "1287th value should be 255");
assertEq(view[1288], 205, "1288th value should be 205");
assertEq(view[1289], 255, "1289th value should be 255");
assertEq(view[1290], 50, "1290th value should be 50");
assertEq(view[1291], 255, "1291th value should be 255");
assertEq(view[1292], 205, "1292th value should be 205");
assertEq(view[1293], 255, "1293th value should be 255");
assertEq(view[1294], 50, "1294th value should be 50");
assertEq(view[1295], 255, "1295th value should be 255");
assertEq(view[1296], 205, "1296th value should be 205");
assertEq(view[1297], 255, "1297th value should be 255");
assertEq(view[1298], 50, "1298th value should be 50");
assertEq(view[1299], 255, "1299th value should be 255");
assertEq(view[1300], 205, "1300th value should be 205");
assertEq(view[1301], 255, "1301th value should be 255");
assertEq(view[1302], 50, "1302th value should be 50");
assertEq(view[1303], 255, "1303th value should be 255");
assertEq(view[1304], 179, "1304th value should be 179");
assertEq(view[1305], 127, "1305th value should be 127");
assertEq(view[1306], 76, "1306th value should be 76");
assertEq(view[1307], 255, "1307th value should be 255");
assertEq(view[1308], 154, "1308th value should be 154");
assertEq(view[1309], 255, "1309th value should be 255");
assertEq(view[1310], 101, "1310th value should be 101");
assertEq(view[1311], 255, "1311th value should be 255");
assertEq(view[1312], 154, "1312th value should be 154");
assertEq(view[1313], 255, "1313th value should be 255");
assertEq(view[1314], 101, "1314th value should be 101");
assertEq(view[1315], 255, "1315th value should be 255");
assertEq(view[1316], 0, "1316th value should be 0");
assertEq(view[1317], 0, "1317th value should be 0");
assertEq(view[1318], 0, "1318th value should be 0");
assertEq(view[1319], 255, "1319th value should be 255");
assertEq(view[1320], 0, "1320th value should be 0");
assertEq(view[1321], 0, "1321th value should be 0");
assertEq(view[1322], 0, "1322th value should be 0");
assertEq(view[1323], 255, "1323th value should be 255");
assertEq(view[1324], 0, "1324th value should be 0");
assertEq(view[1325], 0, "1325th value should be 0");
assertEq(view[1326], 0, "1326th value should be 0");
assertEq(view[1327], 255, "1327th value should be 255");
assertEq(view[1328], 0, "1328th value should be 0");
assertEq(view[1329], 0, "1329th value should be 0");
assertEq(view[1330], 0, "1330th value should be 0");
assertEq(view[1331], 255, "1331th value should be 255");
assertEq(view[1332], 0, "1332th value should be 0");
assertEq(view[1333], 0, "1333th value should be 0");
assertEq(view[1334], 0, "1334th value should be 0");
assertEq(view[1335], 255, "1335th value should be 255");
assertEq(view[1336], 0, "1336th value should be 0");
assertEq(view[1337], 0, "1337th value should be 0");
assertEq(view[1338], 0, "1338th value should be 0");
assertEq(view[1339], 255, "1339th value should be 255");
assertEq(view[1340], 0, "1340th value should be 0");
assertEq(view[1341], 0, "1341th value should be 0");
assertEq(view[1342], 0, "1342th value should be 0");
assertEq(view[1343], 255, "1343th value should be 255");
assertEq(view[1344], 0, "1344th value should be 0");
assertEq(view[1345], 0, "1345th value should be 0");
assertEq(view[1346], 0, "1346th value should be 0");
assertEq(view[1347], 255, "1347th value should be 255");
assertEq(view[1348], 179, "1348th value should be 179");
assertEq(view[1349], 127, "1349th value should be 127");
assertEq(view[1350], 76, "1350th value should be 76");
assertEq(view[1351], 255, "1351th value should be 255");
assertEq(view[1352], 205, "1352th value should be 205");
assertEq(view[1353], 255, "1353th value should be 255");
assertEq(view[1354], 50, "1354th value should be 50");
assertEq(view[1355], 255, "1355th value should be 255");
assertEq(view[1356], 205, "1356th value should be 205");
assertEq(view[1357], 255, "1357th value should be 255");
assertEq(view[1358], 50, "1358th value should be 50");
assertEq(view[1359], 255, "1359th value should be 255");
assertEq(view[1360], 0, "1360th value should be 0");
assertEq(view[1361], 0, "1361th value should be 0");
assertEq(view[1362], 0, "1362th value should be 0");
assertEq(view[1363], 255, "1363th value should be 255");
assertEq(view[1364], 205, "1364th value should be 205");
assertEq(view[1365], 255, "1365th value should be 255");
assertEq(view[1366], 50, "1366th value should be 50");
assertEq(view[1367], 255, "1367th value should be 255");
assertEq(view[1368], 205, "1368th value should be 205");
assertEq(view[1369], 255, "1369th value should be 255");
assertEq(view[1370], 50, "1370th value should be 50");
assertEq(view[1371], 255, "1371th value should be 255");
assertEq(view[1372], 205, "1372th value should be 205");
assertEq(view[1373], 255, "1373th value should be 255");
assertEq(view[1374], 50, "1374th value should be 50");
assertEq(view[1375], 255, "1375th value should be 255");
assertEq(view[1376], 205, "1376th value should be 205");
assertEq(view[1377], 255, "1377th value should be 255");
assertEq(view[1378], 50, "1378th value should be 50");
assertEq(view[1379], 255, "1379th value should be 255");
assertEq(view[1380], 205, "1380th value should be 205");
assertEq(view[1381], 255, "1381th value should be 255");
assertEq(view[1382], 50, "1382th value should be 50");
assertEq(view[1383], 255, "1383th value should be 255");
assertEq(view[1384], 205, "1384th value should be 205");
assertEq(view[1385], 255, "1385th value should be 255");
assertEq(view[1386], 50, "1386th value should be 50");
assertEq(view[1387], 255, "1387th value should be 255");
assertEq(view[1388], 179, "1388th value should be 179");
assertEq(view[1389], 127, "1389th value should be 127");
assertEq(view[1390], 76, "1390th value should be 76");
assertEq(view[1391], 255, "1391th value should be 255");
assertEq(view[1392], 179, "1392th value should be 179");
assertEq(view[1393], 127, "1393th value should be 127");
assertEq(view[1394], 76, "1394th value should be 76");
assertEq(view[1395], 255, "1395th value should be 255");
assertEq(view[1396], 103, "1396th value should be 103");
assertEq(view[1397], 255, "1397th value should be 255");
assertEq(view[1398], 152, "1398th value should be 152");
assertEq(view[1399], 255, "1399th value should be 255");
assertEq(view[1400], 78, "1400th value should be 78");
assertEq(view[1401], 127, "1401th value should be 127");
assertEq(view[1402], 178, "1402th value should be 178");
assertEq(view[1403], 255, "1403th value should be 255");
assertEq(view[1404], 52, "1404th value should be 52");
assertEq(view[1405], 255, "1405th value should be 255");
assertEq(view[1406], 203, "1406th value should be 203");
assertEq(view[1407], 255, "1407th value should be 255");
assertEq(view[1408], 0, "1408th value should be 0");
assertEq(view[1409], 0, "1409th value should be 0");
assertEq(view[1410], 0, "1410th value should be 0");
assertEq(view[1411], 255, "1411th value should be 255");
assertEq(view[1412], 0, "1412th value should be 0");
assertEq(view[1413], 0, "1413th value should be 0");
assertEq(view[1414], 0, "1414th value should be 0");
assertEq(view[1415], 255, "1415th value should be 255");
assertEq(view[1416], 52, "1416th value should be 52");
assertEq(view[1417], 255, "1417th value should be 255");
assertEq(view[1418], 203, "1418th value should be 203");
assertEq(view[1419], 255, "1419th value should be 255");
assertEq(view[1420], 128, "1420th value should be 128");
assertEq(view[1421], 127, "1421th value should be 127");
assertEq(view[1422], 127, "1422th value should be 127");
assertEq(view[1423], 255, "1423th value should be 255");
assertEq(view[1424], 128, "1424th value should be 128");
assertEq(view[1425], 127, "1425th value should be 127");
assertEq(view[1426], 127, "1426th value should be 127");
assertEq(view[1427], 255, "1427th value should be 255");
assertEq(view[1428], 205, "1428th value should be 205");
assertEq(view[1429], 255, "1429th value should be 255");
assertEq(view[1430], 50, "1430th value should be 50");
assertEq(view[1431], 255, "1431th value should be 255");
assertEq(view[1432], 205, "1432th value should be 205");
assertEq(view[1433], 255, "1433th value should be 255");
assertEq(view[1434], 50, "1434th value should be 50");
assertEq(view[1435], 255, "1435th value should be 255");
assertEq(view[1436], 230, "1436th value should be 230");
assertEq(view[1437], 127, "1437th value should be 127");
assertEq(view[1438], 25, "1438th value should be 25");
assertEq(view[1439], 255, "1439th value should be 255");
assertEq(view[1440], 0, "1440th value should be 0");
assertEq(view[1441], 0, "1441th value should be 0");
assertEq(view[1442], 0, "1442th value should be 0");
assertEq(view[1443], 255, "1443th value should be 255");
assertEq(view[1444], 230, "1444th value should be 230");
assertEq(view[1445], 127, "1445th value should be 127");
assertEq(view[1446], 25, "1446th value should be 25");
assertEq(view[1447], 255, "1447th value should be 255");
assertEq(view[1448], 205, "1448th value should be 205");
assertEq(view[1449], 255, "1449th value should be 255");
assertEq(view[1450], 50, "1450th value should be 50");
assertEq(view[1451], 255, "1451th value should be 255");
assertEq(view[1452], 205, "1452th value should be 205");
assertEq(view[1453], 255, "1453th value should be 255");
assertEq(view[1454], 50, "1454th value should be 50");
assertEq(view[1455], 255, "1455th value should be 255");
assertEq(view[1456], 205, "1456th value should be 205");
assertEq(view[1457], 255, "1457th value should be 255");
assertEq(view[1458], 50, "1458th value should be 50");
assertEq(view[1459], 255, "1459th value should be 255");
assertEq(view[1460], 205, "1460th value should be 205");
assertEq(view[1461], 255, "1461th value should be 255");
assertEq(view[1462], 50, "1462th value should be 50");
assertEq(view[1463], 255, "1463th value should be 255");
assertEq(view[1464], 205, "1464th value should be 205");
assertEq(view[1465], 255, "1465th value should be 255");
assertEq(view[1466], 50, "1466th value should be 50");
assertEq(view[1467], 255, "1467th value should be 255");
assertEq(view[1468], 179, "1468th value should be 179");
assertEq(view[1469], 127, "1469th value should be 127");
assertEq(view[1470], 76, "1470th value should be 76");
assertEq(view[1471], 255, "1471th value should be 255");
assertEq(view[1472], 179, "1472th value should be 179");
assertEq(view[1473], 127, "1473th value should be 127");
assertEq(view[1474], 76, "1474th value should be 76");
assertEq(view[1475], 255, "1475th value should be 255");
assertEq(view[1476], 179, "1476th value should be 179");
assertEq(view[1477], 127, "1477th value should be 127");
assertEq(view[1478], 76, "1478th value should be 76");
assertEq(view[1479], 255, "1479th value should be 255");
assertEq(view[1480], 128, "1480th value should be 128");
assertEq(view[1481], 127, "1481th value should be 127");
assertEq(view[1482], 127, "1482th value should be 127");
assertEq(view[1483], 255, "1483th value should be 255");
assertEq(view[1484], 103, "1484th value should be 103");
assertEq(view[1485], 255, "1485th value should be 255");
assertEq(view[1486], 152, "1486th value should be 152");
assertEq(view[1487], 255, "1487th value should be 255");
assertEq(view[1488], 0, "1488th value should be 0");
assertEq(view[1489], 0, "1489th value should be 0");
assertEq(view[1490], 0, "1490th value should be 0");
assertEq(view[1491], 255, "1491th value should be 255");
assertEq(view[1492], 0, "1492th value should be 0");
assertEq(view[1493], 0, "1493th value should be 0");
assertEq(view[1494], 0, "1494th value should be 0");
assertEq(view[1495], 255, "1495th value should be 255");
assertEq(view[1496], 128, "1496th value should be 128");
assertEq(view[1497], 127, "1497th value should be 127");
assertEq(view[1498], 127, "1498th value should be 127");
assertEq(view[1499], 255, "1499th value should be 255");
assertEq(view[1500], 154, "1500th value should be 154");
assertEq(view[1501], 255, "1501th value should be 255");
assertEq(view[1502], 101, "1502th value should be 101");
assertEq(view[1503], 255, "1503th value should be 255");
assertEq(view[1504], 179, "1504th value should be 179");
assertEq(view[1505], 127, "1505th value should be 127");
assertEq(view[1506], 76, "1506th value should be 76");
assertEq(view[1507], 255, "1507th value should be 255");
assertEq(view[1508], 205, "1508th value should be 205");
assertEq(view[1509], 255, "1509th value should be 255");
assertEq(view[1510], 50, "1510th value should be 50");
assertEq(view[1511], 255, "1511th value should be 255");
assertEq(view[1512], 205, "1512th value should be 205");
assertEq(view[1513], 255, "1513th value should be 255");
assertEq(view[1514], 50, "1514th value should be 50");
assertEq(view[1515], 255, "1515th value should be 255");
assertEq(view[1516], 230, "1516th value should be 230");
assertEq(view[1517], 127, "1517th value should be 127");
assertEq(view[1518], 25, "1518th value should be 25");
assertEq(view[1519], 255, "1519th value should be 255");
assertEq(view[1520], 0, "1520th value should be 0");
assertEq(view[1521], 0, "1521th value should be 0");
assertEq(view[1522], 0, "1522th value should be 0");
assertEq(view[1523], 255, "1523th value should be 255");
assertEq(view[1524], 230, "1524th value should be 230");
assertEq(view[1525], 127, "1525th value should be 127");
assertEq(view[1526], 25, "1526th value should be 25");
assertEq(view[1527], 255, "1527th value should be 255");
assertEq(view[1528], 230, "1528th value should be 230");
assertEq(view[1529], 127, "1529th value should be 127");
assertEq(view[1530], 25, "1530th value should be 25");
assertEq(view[1531], 255, "1531th value should be 255");
assertEq(view[1532], 205, "1532th value should be 205");
assertEq(view[1533], 255, "1533th value should be 255");
assertEq(view[1534], 50, "1534th value should be 50");
assertEq(view[1535], 255, "1535th value should be 255");
assertEq(view[1536], 205, "1536th value should be 205");
assertEq(view[1537], 255, "1537th value should be 255");
assertEq(view[1538], 50, "1538th value should be 50");
assertEq(view[1539], 255, "1539th value should be 255");
assertEq(view[1540], 205, "1540th value should be 205");
assertEq(view[1541], 255, "1541th value should be 255");
assertEq(view[1542], 50, "1542th value should be 50");
assertEq(view[1543], 255, "1543th value should be 255");
assertEq(view[1544], 205, "1544th value should be 205");
assertEq(view[1545], 255, "1545th value should be 255");
assertEq(view[1546], 50, "1546th value should be 50");
assertEq(view[1547], 255, "1547th value should be 255");
assertEq(view[1548], 205, "1548th value should be 205");
assertEq(view[1549], 255, "1549th value should be 255");
assertEq(view[1550], 50, "1550th value should be 50");
assertEq(view[1551], 255, "1551th value should be 255");
assertEq(view[1552], 179, "1552th value should be 179");
assertEq(view[1553], 127, "1553th value should be 127");
assertEq(view[1554], 76, "1554th value should be 76");
assertEq(view[1555], 255, "1555th value should be 255");
assertEq(view[1556], 179, "1556th value should be 179");
assertEq(view[1557], 127, "1557th value should be 127");
assertEq(view[1558], 76, "1558th value should be 76");
assertEq(view[1559], 255, "1559th value should be 255");
assertEq(view[1560], 179, "1560th value should be 179");
assertEq(view[1561], 127, "1561th value should be 127");
assertEq(view[1562], 76, "1562th value should be 76");
assertEq(view[1563], 255, "1563th value should be 255");
assertEq(view[1564], 154, "1564th value should be 154");
assertEq(view[1565], 255, "1565th value should be 255");
assertEq(view[1566], 101, "1566th value should be 101");
assertEq(view[1567], 255, "1567th value should be 255");
assertEq(view[1568], 26, "1568th value should be 26");
assertEq(view[1569], 127, "1569th value should be 127");
assertEq(view[1570], 229, "1570th value should be 229");
assertEq(view[1571], 255, "1571th value should be 255");
assertEq(view[1572], 0, "1572th value should be 0");
assertEq(view[1573], 0, "1573th value should be 0");
assertEq(view[1574], 0, "1574th value should be 0");
assertEq(view[1575], 255, "1575th value should be 255");
assertEq(view[1576], 154, "1576th value should be 154");
assertEq(view[1577], 255, "1577th value should be 255");
assertEq(view[1578], 101, "1578th value should be 101");
assertEq(view[1579], 255, "1579th value should be 255");
assertEq(view[1580], 179, "1580th value should be 179");
assertEq(view[1581], 127, "1581th value should be 127");
assertEq(view[1582], 76, "1582th value should be 76");
assertEq(view[1583], 255, "1583th value should be 255");
assertEq(view[1584], 205, "1584th value should be 205");
assertEq(view[1585], 255, "1585th value should be 255");
assertEq(view[1586], 50, "1586th value should be 50");
assertEq(view[1587], 255, "1587th value should be 255");
assertEq(view[1588], 205, "1588th value should be 205");
assertEq(view[1589], 255, "1589th value should be 255");
assertEq(view[1590], 50, "1590th value should be 50");
assertEq(view[1591], 255, "1591th value should be 255");
assertEq(view[1592], 230, "1592th value should be 230");
assertEq(view[1593], 127, "1593th value should be 127");
assertEq(view[1594], 25, "1594th value should be 25");
assertEq(view[1595], 255, "1595th value should be 255");
assertEq(view[1596], 230, "1596th value should be 230");
assertEq(view[1597], 127, "1597th value should be 127");
assertEq(view[1598], 25, "1598th value should be 25");
assertEq(view[1599], 255, "1599th value should be 255");

// Code used to generate the assertEq list above.
function generateAssertList() {
  function template(i, x) {
    return 'assertEq(view[' + i + '], ' + x + ', "' + i + 'th value should be ' + x + '");\n';
  }
  var buf = ''
  for (var i = 0; i < LIMIT_SHOW; i++)
      buf += template(i, view[i]);
  print(buf);
}
//generateAssertList();
