/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef MOZILLA_SOURCEBUFFERDECODER_H_
#define MOZILLA_SOURCEBUFFERDECODER_H_

#include "AbstractMediaDecoder.h"
#include "MediaDecoderReader.h"
#include "SourceBufferResource.h"
#include "mozilla/Attributes.h"
#ifdef MOZ_EME
#include "mozilla/CDMProxy.h"
#endif
#include "mozilla/ReentrantMonitor.h"

namespace mozilla {

class MediaResource;
class MediaDecoderReader;

namespace dom {

class TimeRanges;

} // namespace dom

class SourceBufferDecoder MOZ_FINAL : public AbstractMediaDecoder
{
public:
  // This class holds a weak pointer to MediaResource.  It's the responsibility
  // of the caller to manage the memory of the MediaResource object.
  SourceBufferDecoder(MediaResource* aResource, AbstractMediaDecoder* aParentDecoder,
                      int64_t aTimestampOffset /* microseconds */);

  NS_DECL_THREADSAFE_ISUPPORTS

  virtual bool IsMediaSeekable() MOZ_FINAL MOZ_OVERRIDE;
  virtual bool IsShutdown() const MOZ_FINAL MOZ_OVERRIDE;
  virtual bool IsTransportSeekable() MOZ_FINAL MOZ_OVERRIDE;
  virtual bool OnDecodeThread() const MOZ_FINAL MOZ_OVERRIDE;
  virtual bool OnStateMachineThread() const MOZ_FINAL MOZ_OVERRIDE;
  virtual int64_t GetMediaDuration() MOZ_FINAL MOZ_OVERRIDE;
  virtual layers::ImageContainer* GetImageContainer() MOZ_FINAL MOZ_OVERRIDE;
  virtual MediaDecoderOwner* GetOwner() MOZ_FINAL MOZ_OVERRIDE;
  virtual SourceBufferResource* GetResource() const MOZ_FINAL MOZ_OVERRIDE;
  virtual ReentrantMonitor& GetReentrantMonitor() MOZ_FINAL MOZ_OVERRIDE;
  virtual VideoFrameContainer* GetVideoFrameContainer() MOZ_FINAL MOZ_OVERRIDE;
  virtual void MetadataLoaded(nsAutoPtr<MediaInfo> aInfo, nsAutoPtr<MetadataTags> aTags, bool aRestoredFromDromant) MOZ_FINAL MOZ_OVERRIDE;
  virtual void FirstFrameLoaded(nsAutoPtr<MediaInfo> aInfo, bool aRestoredFromDromant) MOZ_FINAL MOZ_OVERRIDE;
  virtual void NotifyBytesConsumed(int64_t aBytes, int64_t aOffset) MOZ_FINAL MOZ_OVERRIDE;
  virtual void NotifyDataArrived(const char* aBuffer, uint32_t aLength, int64_t aOffset) MOZ_FINAL MOZ_OVERRIDE;
  virtual void NotifyDecodedFrames(uint32_t aParsed, uint32_t aDecoded, uint32_t aDropped) MOZ_FINAL MOZ_OVERRIDE;
  virtual void NotifyWaitingForResourcesStatusChanged() MOZ_FINAL MOZ_OVERRIDE;
  virtual void OnReadMetadataCompleted() MOZ_FINAL MOZ_OVERRIDE;
  virtual void QueueMetadata(int64_t aTime, nsAutoPtr<MediaInfo> aInfo, nsAutoPtr<MetadataTags> aTags) MOZ_FINAL MOZ_OVERRIDE;
  virtual void RemoveMediaTracks() MOZ_FINAL MOZ_OVERRIDE;
  virtual void SetMediaDuration(int64_t aDuration) MOZ_FINAL MOZ_OVERRIDE;
  virtual void SetMediaEndTime(int64_t aTime) MOZ_FINAL MOZ_OVERRIDE;
  virtual void SetMediaSeekable(bool aMediaSeekable) MOZ_FINAL MOZ_OVERRIDE;
  virtual void UpdateEstimatedMediaDuration(int64_t aDuration) MOZ_FINAL MOZ_OVERRIDE;
  virtual void UpdatePlaybackPosition(int64_t aTime) MOZ_FINAL MOZ_OVERRIDE;
  virtual bool HasInitializationData() MOZ_FINAL MOZ_OVERRIDE;

  // SourceBufferResource specific interface below.
  int64_t GetTimestampOffset() const { return mTimestampOffset; }
  void SetTimestampOffset(int64_t aOffset)  { mTimestampOffset = aOffset; }

  // Warning: this mirrors GetBuffered in MediaDecoder, but this class's base is
  // AbstractMediaDecoder, which does not supply this interface.
  nsresult GetBuffered(dom::TimeRanges* aBuffered);

  void SetReader(MediaDecoderReader* aReader)
  {
    MOZ_ASSERT(!mReader);
    mReader = aReader;
  }

  MediaDecoderReader* GetReader() const
  {
    return mReader;
  }

  void SetTaskQueue(MediaTaskQueue* aTaskQueue)
  {
    MOZ_ASSERT((!mTaskQueue && aTaskQueue) || (mTaskQueue && !aTaskQueue));
    mTaskQueue = aTaskQueue;
  }

  void BreakCycles()
  {
    if (mReader) {
      mReader->BreakCycles();
      mReader = nullptr;
    }
    mTaskQueue = nullptr;
#ifdef MOZ_EME
    mCDMProxy = nullptr;
#endif
  }

#ifdef MOZ_EME
  virtual nsresult SetCDMProxy(CDMProxy* aProxy) MOZ_OVERRIDE
  {
    MOZ_ASSERT(NS_IsMainThread());
    ReentrantMonitorAutoEnter mon(GetReentrantMonitor());
    mCDMProxy = aProxy;
    return NS_OK;
  }

  virtual CDMProxy* GetCDMProxy() MOZ_OVERRIDE
  {
    MOZ_ASSERT(OnDecodeThread() || NS_IsMainThread());
    ReentrantMonitorAutoEnter mon(GetReentrantMonitor());
    return mCDMProxy;
  }
#endif

  // Given a time convert it into an approximate byte offset from the
  // cached data. Returns -1 if no such value is computable.
  int64_t ConvertToByteOffset(double aTime);

  // All durations are in usecs.

  // We can't at this stage, accurately remove coded frames.
  // Trim is a work around that hides data located after a given time by
  // preventing playback beyond the trim point.
  // No data is actually removed.
  // aDuration is were data will be trimmed from.
  void Trim(int64_t aDuration);
  bool WasTrimmed()
  {
    return mTrimmedOffset >= 0;
  }

  // returns the real duration of the resource, including trimmed data.
  void SetRealMediaDuration(int64_t aDuration);
  int64_t GetRealMediaDuration()
  {
    return mRealMediaDuration;
  }

private:
  virtual ~SourceBufferDecoder();

  // Our TrackBuffer's task queue, this is only non-null during initialization.
  RefPtr<MediaTaskQueue> mTaskQueue;

  nsRefPtr<MediaResource> mResource;

  AbstractMediaDecoder* mParentDecoder;
  nsRefPtr<MediaDecoderReader> mReader;
  // in microseconds
  int64_t mTimestampOffset;
  // mMediaDuration contains the apparent buffer duration, excluding trimmed data.
  int64_t mMediaDuration;
  // mRealMediaDuration contains the real buffer duration, including trimmed data.
  int64_t mRealMediaDuration;
  // in seconds
  double mTrimmedOffset;

#ifdef MOZ_EME
  nsRefPtr<CDMProxy> mCDMProxy;
#endif
};

} // namespace mozilla

#endif /* MOZILLA_SOURCEBUFFERDECODER_H_ */
