/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef mozilla_dom_PostMessageEvent_h
#define mozilla_dom_PostMessageEvent_h

#include "js/StructuredClone.h"
#include "nsCOMPtr.h"
#include "nsRefPtr.h"
#include "nsTArray.h"

class nsGlobalWindow;
class nsIPrincipal;

namespace mozilla {
namespace dom {

class MessagePortBase;
class MessagePortIdentifier;

/**
 * Class used to represent events generated by calls to Window.postMessage,
 * which asynchronously creates and dispatches events.
 */
class PostMessageEvent final : public nsRunnable
{
public:
  NS_DECL_NSIRUNNABLE

  PostMessageEvent(nsGlobalWindow* aSource,
                   const nsAString& aCallerOrigin,
                   nsGlobalWindow* aTargetWindow,
                   nsIPrincipal* aProvidedPrincipal,
                   bool aTrustedCaller);

  bool Write(JSContext* aCx, JS::Handle<JS::Value> aMessage,
             JS::Handle<JS::Value> aTransfer, bool aSubsumes,
             nsPIDOMWindow* aWindow);

private:
  ~PostMessageEvent();

  const MessagePortIdentifier& GetPortIdentifier(uint64_t aId);

  MessagePortIdentifier* NewPortIdentifier(uint64_t* aPosition);

  bool StoreISupports(nsISupports* aSupports)
  {
    mSupportsArray.AppendElement(aSupports);
    return true;
  }

  static JSObject*
  ReadStructuredClone(JSContext* cx,
                      JSStructuredCloneReader* reader,
                      uint32_t tag,
                      uint32_t data,
                      void* closure);

  static bool
  WriteStructuredClone(JSContext* cx,
                       JSStructuredCloneWriter* writer,
                       JS::Handle<JSObject*> obj,
                       void *closure);

  static bool
  ReadTransferStructuredClone(JSContext* aCx,
                              JSStructuredCloneReader* reader,
                              uint32_t tag, void* aData,
                              uint64_t aExtraData,
                              void* aClosure,
                              JS::MutableHandle<JSObject*> returnObject);

  static bool
  TransferStructuredClone(JSContext* aCx,
                          JS::Handle<JSObject*> aObj,
                          void* aClosure,
                          uint32_t* aTag,
                          JS::TransferableOwnership* aOwnership,
                          void** aContent,
                          uint64_t* aExtraData);

  static void
  FreeTransferStructuredClone(uint32_t aTag,
                              JS::TransferableOwnership aOwnership,
                              void *aContent,
                              uint64_t aExtraData,
                              void* aClosure);

  static const JSStructuredCloneCallbacks sPostMessageCallbacks;

  JSAutoStructuredCloneBuffer mBuffer;
  nsRefPtr<nsGlobalWindow> mSource;
  nsString mCallerOrigin;
  nsRefPtr<nsGlobalWindow> mTargetWindow;
  nsCOMPtr<nsIPrincipal> mProvidedPrincipal;
  bool mTrustedCaller;
  nsTArray<nsCOMPtr<nsISupports>> mSupportsArray;
  nsTArray<MessagePortIdentifier> mPortIdentifiers;
};

} // dom namespace
} // mozilla namespace

#endif // mozilla_dom_PostMessageEvent_h
