/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
* License, v. 2.0. If a copy of the MPL was not distributed with this
* file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "gtest/gtest.h"
#include <algorithm>
#include <stdint.h>
#include <vector>

#include "../ClearKeyBase64.cpp"
#include "../ClearKeyCencParser.cpp"
#include "../ArrayUtils.h"


using namespace std;

struct B64Test {
  const char* b64;
  uint8_t raw[16];
  bool shouldPass;
};

B64Test tests[] = {
  {
    "AAAAADk4AU4AAAAAAAAAAA",
    { 0x0, 0x0, 0x0, 0x0, 0x39, 0x38, 0x1, 0x4e, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0 },
    true
  },
  {
    "h2mqp1zAJjDIC34YXEXBxA==",
    { 0x87, 0x69, 0xaa, 0xa7, 0x5c, 0xc0, 0x26, 0x30, 0xc8, 0xb, 0x7e, 0x18, 0x5c, 0x45, 0xc1, 0xc4 },
    true
  },
  {
    "flcdA35XHQN-Vx0DflcdAw",
    { 0x7e, 0x57, 0x1d, 0x3, 0x7e, 0x57, 0x1d, 0x3, 0x7e, 0x57, 0x1d, 0x3, 0x7e, 0x57, 0x1d, 0x3 },
    true
  },
  {
    "flczM35XMzN-VzMzflczMw",
    { 0x7e, 0x57, 0x33, 0x33, 0x7e, 0x57, 0x33, 0x33, 0x7e, 0x57, 0x33, 0x33, 0x7e, 0x57, 0x33, 0x33 },
    true
  },
  {
    "flcdBH5XHQR-Vx0EflcdBA",
    { 0x7e, 0x57, 0x1d, 0x4, 0x7e, 0x57, 0x1d, 0x4, 0x7e, 0x57, 0x1d, 0x4, 0x7e, 0x57, 0x1d, 0x4 },
    true
  },
  {
    "fldERH5XRER-V0REfldERA",
    { 0x7e, 0x57, 0x44, 0x44, 0x7e, 0x57, 0x44, 0x44, 0x7e, 0x57, 0x44, 0x44, 0x7e, 0x57, 0x44, 0x44 },
    true
  },
  // Failure tests
  { "", { 0 }, false }, // empty
  { "fuzzbiz", { 0 }, false }, // Too short
  { "fuzzbizfuzzbizfuzzbizfuzzbizfuzzbizfuzzbizfuzzbizfuzzbiz", { 0 }, false }, // too long

};

TEST(ClearKey, DecodeBase64KeyOrId) {
  for (size_t i = 0; i < MOZ_ARRAY_LENGTH(tests); i++) {
    vector<uint8_t> v;
    const B64Test& test = tests[i];
    bool rv = DecodeBase64KeyOrId(string(test.b64), v);
    EXPECT_EQ(rv, test.shouldPass);
    if (test.shouldPass) {
      EXPECT_EQ(v.size(), 16u);
      for (size_t k = 0; k < 16; k++) {
        EXPECT_EQ(v[k], test.raw[k]);
      }
    }
  }
}

// This is the CENC initData from Google's web-platform tests.
// https://github.com/w3c/web-platform-tests/blob/master/encrypted-media/Google/encrypted-media-utils.js#L50
const uint8_t gGoogleWPTCencInitData[] = {
  0x00, 0x00, 0x00, 0x00,                          // size = 0
  0x70, 0x73, 0x73, 0x68,                          // 'pssh'
  0x01,                                            // version = 1
  0x00, 0x00, 0x00,                                // flags
  0x10, 0x77, 0xEF, 0xEC, 0xC0, 0xB2, 0x4D, 0x02,  // Common SystemID
  0xAC, 0xE3, 0x3C, 0x1E, 0x52, 0xE2, 0xFB, 0x4B,
  0x00, 0x00, 0x00, 0x01,                          // key count
  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,  // key
  0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
  0x00, 0x00, 0x00, 0x00                           // datasize
};

// Example CENC initData from the EME spec format registry:
// https://w3c.github.io/encrypted-media/format-registry/initdata/cenc.html
const uint8_t gW3SpecExampleCencInitData[] = {
  0x00, 0x00, 0x00, 0x4c, 0x70, 0x73, 0x73, 0x68, // BMFF box header (76 bytes, 'pssh')
  0x01, 0x00, 0x00, 0x00,                         // Full box header (version = 1, flags = 0)
  0x10, 0x77, 0xef, 0xec, 0xc0, 0xb2, 0x4d, 0x02, // SystemID
  0xac, 0xe3, 0x3c, 0x1e, 0x52, 0xe2, 0xfb, 0x4b,
  0x00, 0x00, 0x00, 0x02,                         // KID_count (2)
  0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, // First KID ("0123456789012345")
  0x38, 0x39, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35,
  0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, // Second KID ("ABCDEFGHIJKLMNOP")
  0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50,
  0x00, 0x00, 0x00, 0x00                         // Size of Data (0)
};

// Invalid box size, would overflow if used.
const uint8_t gOverflowBoxSize[] = {
  0xff, 0xff, 0xff, 0xff,                          // size = UINT32_MAX
};

// Invalid box size, but retrievable data.
const uint8_t gMalformedCencInitData[] = {
  0x00, 0x00, 0xff, 0xff,                          // size = too big a number
  0x70, 0x73, 0x73, 0x68,                          // 'pssh'
  0x01,                                            // version = 1
  0xff, 0xff, 0xff,                                // flags
  0x10, 0x77, 0xEF, 0xEC, 0xC0, 0xB2, 0x4D, 0x02,  // Common SystemID
  0xAC, 0xE3, 0x3C, 0x1E, 0x52, 0xE2, 0xFB, 0x4B,
  0xff, 0xff, 0xff, 0xff,                          // key count = UINT32_MAX
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,  // key
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff                           // datasize
};

// Non PSSH box, followed by Non common SystemID PSSH, followed by common SystemID PSSH box.
const uint8_t gLeadingNonCommonCencInitData[] = {
  0x00, 0x00, 0x00, 0x09,                          // size = 9
  0xff, 0xff, 0xff, 0xff,                          // something other than 'pssh'
  0xff,                                            // odd number of bytes of garbage to throw off the parser

  0x00, 0x00, 0x00, 0x5c,                          // size = 92
  0x70, 0x73, 0x73, 0x68,                          // 'pssh'
  0x01,                                            // version = 1
  0x00, 0x00, 0x00,                                // flags
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,  // Invalid SystemID
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,  // Some data to pad out the box.
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,

  // gW3SpecExampleCencInitData
  0x00, 0x00, 0x00, 0x4c, 0x70, 0x73, 0x73, 0x68, // BMFF box header (76 bytes, 'pssh')
  0x01, 0x00, 0x00, 0x00,                         // Full box header (version = 1, flags = 0)
  0x10, 0x77, 0xef, 0xec, 0xc0, 0xb2, 0x4d, 0x02, // SystemID
  0xac, 0xe3, 0x3c, 0x1e, 0x52, 0xe2, 0xfb, 0x4b,
  0x00, 0x00, 0x00, 0x02,                         // KID_count (2)
  0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, // First KID ("0123456789012345")
  0x38, 0x39, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35,
  0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, // Second KID ("ABCDEFGHIJKLMNOP")
  0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50,
  0x00, 0x00, 0x00, 0x00                         // Size of Data (0)
};

const uint8_t gNonPSSHBoxZeroSize[] = {
  0x00, 0x00, 0x00, 0x00,                          // size = 0
  0xff, 0xff, 0xff, 0xff,                          // something other than 'pssh'
};

// Two lots of the google init data. To ensure we handle
// multiple boxes with size 0.
const uint8_t g2xGoogleWPTCencInitData[] = {
  0x00, 0x00, 0x00, 0x00,                          // size = 0
  0x70, 0x73, 0x73, 0x68,                          // 'pssh'
  0x01,                                            // version = 1
  0x00, 0x00, 0x00,                                // flags
  0x10, 0x77, 0xEF, 0xEC, 0xC0, 0xB2, 0x4D, 0x02,  // Common SystemID
  0xAC, 0xE3, 0x3C, 0x1E, 0x52, 0xE2, 0xFB, 0x4B,
  0x00, 0x00, 0x00, 0x01,                          // key count
  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,  // key
  0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
  0x00, 0x00, 0x00, 0x00,                          // datasize

  0x00, 0x00, 0x00, 0x00,                          // size = 0
  0x70, 0x73, 0x73, 0x68,                          // 'pssh'
  0x01,                                            // version = 1
  0x00, 0x00, 0x00,                                // flags
  0x10, 0x77, 0xEF, 0xEC, 0xC0, 0xB2, 0x4D, 0x02,  // Common SystemID
  0xAC, 0xE3, 0x3C, 0x1E, 0x52, 0xE2, 0xFB, 0x4B,
  0x00, 0x00, 0x00, 0x01,                          // key count
  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,  // key
  0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
  0x00, 0x00, 0x00, 0x00                           // datasize  
};

TEST(ClearKey, ParseCencInitData) {
  std::vector<std::vector<uint8_t>> keyIds;

  ParseCENCInitData(gGoogleWPTCencInitData, MOZ_ARRAY_LENGTH(gGoogleWPTCencInitData), keyIds);
  EXPECT_EQ(keyIds.size(), 1u);
  EXPECT_EQ(keyIds[0].size(), 16u);
  EXPECT_EQ(memcmp(&keyIds[0].front(), &gGoogleWPTCencInitData[32], 16), 0);

  keyIds.clear();
  ParseCENCInitData(gW3SpecExampleCencInitData, MOZ_ARRAY_LENGTH(gW3SpecExampleCencInitData), keyIds);
  EXPECT_EQ(keyIds.size(), 2u);
  EXPECT_EQ(keyIds[0].size(), 16u);
  EXPECT_EQ(memcmp(&keyIds[0].front(), &gW3SpecExampleCencInitData[32], 16), 0);
  EXPECT_EQ(memcmp(&keyIds[1].front(), &gW3SpecExampleCencInitData[48], 16), 0);

  keyIds.clear();
  ParseCENCInitData(gOverflowBoxSize, MOZ_ARRAY_LENGTH(gOverflowBoxSize), keyIds);
  EXPECT_EQ(keyIds.size(), 0u);

  keyIds.clear();
  ParseCENCInitData(gMalformedCencInitData, MOZ_ARRAY_LENGTH(gMalformedCencInitData), keyIds);
  EXPECT_EQ(keyIds.size(), 1u);
  EXPECT_EQ(keyIds[0].size(), 16u);
  EXPECT_EQ(memcmp(&keyIds[0].front(), &gMalformedCencInitData[32], 16), 0);

  keyIds.clear();
  ParseCENCInitData(gLeadingNonCommonCencInitData, MOZ_ARRAY_LENGTH(gLeadingNonCommonCencInitData), keyIds);
  EXPECT_EQ(keyIds.size(), 2u);
  EXPECT_EQ(keyIds[0].size(), 16u);
  EXPECT_EQ(memcmp(&keyIds[0].front(), &gW3SpecExampleCencInitData[32], 16), 0);
  EXPECT_EQ(memcmp(&keyIds[1].front(), &gW3SpecExampleCencInitData[48], 16), 0);

  keyIds.clear();
  ParseCENCInitData(gNonPSSHBoxZeroSize, MOZ_ARRAY_LENGTH(gNonPSSHBoxZeroSize), keyIds);
  EXPECT_EQ(keyIds.size(), 0u);

  keyIds.clear();
  ParseCENCInitData(g2xGoogleWPTCencInitData, MOZ_ARRAY_LENGTH(g2xGoogleWPTCencInitData), keyIds);
  EXPECT_EQ(keyIds.size(), 2u);
  EXPECT_EQ(keyIds[0].size(), 16u);
  EXPECT_EQ(keyIds[1].size(), 16u);
  EXPECT_EQ(memcmp(&keyIds[0].front(), &g2xGoogleWPTCencInitData[32], 16), 0);
  EXPECT_EQ(memcmp(&keyIds[1].front(), &g2xGoogleWPTCencInitData[84], 16), 0);

}
