/*
 * Copyright (c) 2016, Alliance for Open Media. All rights reserved
 *
 * This source code is subject to the terms of the BSD 2 Clause License and
 * the Alliance for Open Media Patent License 1.0. If the BSD 2 Clause License
 * was not distributed with this source code in the LICENSE file, you can
 * obtain it at www.aomedia.org/license/software. If the Alliance for Open
 * Media Patent License 1.0 was not distributed with this source code in the
 * PATENTS file, you can obtain it at www.aomedia.org/license/patent.
 */

#include "av1/common/common.h"
#include "av1/common/onyxc_int.h"
#include "av1/common/entropy.h"
#include "av1/common/quant_common.h"
#include "av1/common/seg_common.h"
#include "av1/common/blockd.h"

#if CONFIG_NEW_QUANT
// Bin widths expressed as a fraction over 128 of the quant stepsize,
// for the quantization bins 0-4.
// So a value x indicates the bin is actually factor x/128 of the
// nominal quantization step.  For the zero bin, the width is only
// for one side of zero, so the actual width is twice that.
//
// Functions with nuq correspond to "non uniform quantization"
// TODO(sarahparker, debargha): Optimize these tables

typedef struct {
  uint8_t knots[NUQ_KNOTS];  // offsets
  uint8_t doff;              // dequantization
} qprofile_type;

static const qprofile_type nuq[QUANT_PROFILES][COEF_BANDS] = {
  {
      // lossless
      { { 64, 128, 128 }, 0 },  // dc, band 0
      { { 64, 128, 128 }, 0 },  // band 1
      { { 64, 128, 128 }, 0 },  // band 2
      { { 64, 128, 128 }, 0 },  // band 3
      { { 64, 128, 128 }, 0 },  // band 4
      { { 64, 128, 128 }, 0 },  // band 5
  },
  {
      { { 64, 128, 128 }, 4 },   // dc, band 0
      { { 64, 128, 128 }, 6 },   // band 1
      { { 64, 128, 128 }, 8 },   // band 2
      { { 64, 128, 128 }, 10 },  // band 3
      { { 72, 128, 128 }, 12 },  // band 4
      { { 80, 128, 128 }, 14 }   // band 5
  },
  {
      { { 64, 128, 128 }, 6 },   // dc, band 0
      { { 64, 128, 128 }, 8 },   // band 1
      { { 64, 128, 128 }, 10 },  // band 2
      { { 64, 128, 128 }, 12 },  // band 3
      { { 72, 128, 128 }, 14 },  // band 4
      { { 80, 128, 128 }, 16 }   // band 5
  },
  {
      { { 64, 128, 128 }, 8 },   // dc, band 0
      { { 64, 128, 128 }, 10 },  // band 1
      { { 64, 128, 128 }, 12 },  // band 2
      { { 72, 128, 128 }, 14 },  // band 3
      { { 76, 128, 128 }, 16 },  // band 4
      { { 80, 128, 128 }, 18 }   // band 5
  }
};

static const uint8_t *get_nuq_knots(int band, int q_profile) {
  return nuq[q_profile][band].knots;
}

static INLINE int16_t quant_to_doff_fixed(int band, int q_profile) {
  return nuq[q_profile][band].doff;
}

// get cumulative bins
static INLINE void get_cuml_bins_nuq(int q, int band, tran_low_t *cuml_bins,
                                     int q_profile) {
  const uint8_t *knots = get_nuq_knots(band, q_profile);
  int16_t cuml_knots[NUQ_KNOTS];
  int i;
  cuml_knots[0] = knots[0];
  for (i = 1; i < NUQ_KNOTS; ++i) cuml_knots[i] = cuml_knots[i - 1] + knots[i];
  for (i = 0; i < NUQ_KNOTS; ++i)
    cuml_bins[i] = ROUND_POWER_OF_TWO(cuml_knots[i] * q, 7);
}

void av1_get_dequant_val_nuq(int q, int band, tran_low_t *dq,
                             tran_low_t *cuml_bins, int q_profile) {
  const uint8_t *knots = get_nuq_knots(band, q_profile);
  tran_low_t cuml_bins_[NUQ_KNOTS], *cuml_bins_ptr;
  tran_low_t doff;
  int i;
  cuml_bins_ptr = (cuml_bins ? cuml_bins : cuml_bins_);
  get_cuml_bins_nuq(q, band, cuml_bins_ptr, q_profile);
  dq[0] = 0;
  for (i = 1; i < NUQ_KNOTS; ++i) {
    doff = quant_to_doff_fixed(band, q_profile);
    doff = ROUND_POWER_OF_TWO(doff * knots[i], 7);
    dq[i] =
        cuml_bins_ptr[i - 1] + ROUND_POWER_OF_TWO((knots[i] - doff * 2) * q, 8);
  }
  doff = quant_to_doff_fixed(band, q_profile);
  dq[NUQ_KNOTS] =
      cuml_bins_ptr[NUQ_KNOTS - 1] + ROUND_POWER_OF_TWO((64 - doff) * q, 7);
}

tran_low_t av1_dequant_abscoeff_nuq(int v, int q, const tran_low_t *dq) {
  if (v <= NUQ_KNOTS)
    return dq[v];
  else
    return dq[NUQ_KNOTS] + (v - NUQ_KNOTS) * q;
}

tran_low_t av1_dequant_coeff_nuq(int v, int q, const tran_low_t *dq) {
  tran_low_t dqmag = av1_dequant_abscoeff_nuq(abs(v), q, dq);
  return (v < 0 ? -dqmag : dqmag);
}
#endif  // CONFIG_NEW_QUANT

static const int16_t dc_qlookup[QINDEX_RANGE] = {
  4,    8,    8,    9,    10,  11,  12,  12,  13,  14,  15,   16,   17,   18,
  19,   19,   20,   21,   22,  23,  24,  25,  26,  26,  27,   28,   29,   30,
  31,   32,   32,   33,   34,  35,  36,  37,  38,  38,  39,   40,   41,   42,
  43,   43,   44,   45,   46,  47,  48,  48,  49,  50,  51,   52,   53,   53,
  54,   55,   56,   57,   57,  58,  59,  60,  61,  62,  62,   63,   64,   65,
  66,   66,   67,   68,   69,  70,  70,  71,  72,  73,  74,   74,   75,   76,
  77,   78,   78,   79,   80,  81,  81,  82,  83,  84,  85,   85,   87,   88,
  90,   92,   93,   95,   96,  98,  99,  101, 102, 104, 105,  107,  108,  110,
  111,  113,  114,  116,  117, 118, 120, 121, 123, 125, 127,  129,  131,  134,
  136,  138,  140,  142,  144, 146, 148, 150, 152, 154, 156,  158,  161,  164,
  166,  169,  172,  174,  177, 180, 182, 185, 187, 190, 192,  195,  199,  202,
  205,  208,  211,  214,  217, 220, 223, 226, 230, 233, 237,  240,  243,  247,
  250,  253,  257,  261,  265, 269, 272, 276, 280, 284, 288,  292,  296,  300,
  304,  309,  313,  317,  322, 326, 330, 335, 340, 344, 349,  354,  359,  364,
  369,  374,  379,  384,  389, 395, 400, 406, 411, 417, 423,  429,  435,  441,
  447,  454,  461,  467,  475, 482, 489, 497, 505, 513, 522,  530,  539,  549,
  559,  569,  579,  590,  602, 614, 626, 640, 654, 668, 684,  700,  717,  736,
  755,  775,  796,  819,  843, 869, 896, 925, 955, 988, 1022, 1058, 1098, 1139,
  1184, 1232, 1282, 1336,
};

#if CONFIG_HIGHBITDEPTH
static const int16_t dc_qlookup_10[QINDEX_RANGE] = {
  4,    9,    10,   13,   15,   17,   20,   22,   25,   28,   31,   34,   37,
  40,   43,   47,   50,   53,   57,   60,   64,   68,   71,   75,   78,   82,
  86,   90,   93,   97,   101,  105,  109,  113,  116,  120,  124,  128,  132,
  136,  140,  143,  147,  151,  155,  159,  163,  166,  170,  174,  178,  182,
  185,  189,  193,  197,  200,  204,  208,  212,  215,  219,  223,  226,  230,
  233,  237,  241,  244,  248,  251,  255,  259,  262,  266,  269,  273,  276,
  280,  283,  287,  290,  293,  297,  300,  304,  307,  310,  314,  317,  321,
  324,  327,  331,  334,  337,  343,  350,  356,  362,  369,  375,  381,  387,
  394,  400,  406,  412,  418,  424,  430,  436,  442,  448,  454,  460,  466,
  472,  478,  484,  490,  499,  507,  516,  525,  533,  542,  550,  559,  567,
  576,  584,  592,  601,  609,  617,  625,  634,  644,  655,  666,  676,  687,
  698,  708,  718,  729,  739,  749,  759,  770,  782,  795,  807,  819,  831,
  844,  856,  868,  880,  891,  906,  920,  933,  947,  961,  975,  988,  1001,
  1015, 1030, 1045, 1061, 1076, 1090, 1105, 1120, 1137, 1153, 1170, 1186, 1202,
  1218, 1236, 1253, 1271, 1288, 1306, 1323, 1342, 1361, 1379, 1398, 1416, 1436,
  1456, 1476, 1496, 1516, 1537, 1559, 1580, 1601, 1624, 1647, 1670, 1692, 1717,
  1741, 1766, 1791, 1817, 1844, 1871, 1900, 1929, 1958, 1990, 2021, 2054, 2088,
  2123, 2159, 2197, 2236, 2276, 2319, 2363, 2410, 2458, 2508, 2561, 2616, 2675,
  2737, 2802, 2871, 2944, 3020, 3102, 3188, 3280, 3375, 3478, 3586, 3702, 3823,
  3953, 4089, 4236, 4394, 4559, 4737, 4929, 5130, 5347,
};

static const int16_t dc_qlookup_12[QINDEX_RANGE] = {
  4,     12,    18,    25,    33,    41,    50,    60,    70,    80,    91,
  103,   115,   127,   140,   153,   166,   180,   194,   208,   222,   237,
  251,   266,   281,   296,   312,   327,   343,   358,   374,   390,   405,
  421,   437,   453,   469,   484,   500,   516,   532,   548,   564,   580,
  596,   611,   627,   643,   659,   674,   690,   706,   721,   737,   752,
  768,   783,   798,   814,   829,   844,   859,   874,   889,   904,   919,
  934,   949,   964,   978,   993,   1008,  1022,  1037,  1051,  1065,  1080,
  1094,  1108,  1122,  1136,  1151,  1165,  1179,  1192,  1206,  1220,  1234,
  1248,  1261,  1275,  1288,  1302,  1315,  1329,  1342,  1368,  1393,  1419,
  1444,  1469,  1494,  1519,  1544,  1569,  1594,  1618,  1643,  1668,  1692,
  1717,  1741,  1765,  1789,  1814,  1838,  1862,  1885,  1909,  1933,  1957,
  1992,  2027,  2061,  2096,  2130,  2165,  2199,  2233,  2267,  2300,  2334,
  2367,  2400,  2434,  2467,  2499,  2532,  2575,  2618,  2661,  2704,  2746,
  2788,  2830,  2872,  2913,  2954,  2995,  3036,  3076,  3127,  3177,  3226,
  3275,  3324,  3373,  3421,  3469,  3517,  3565,  3621,  3677,  3733,  3788,
  3843,  3897,  3951,  4005,  4058,  4119,  4181,  4241,  4301,  4361,  4420,
  4479,  4546,  4612,  4677,  4742,  4807,  4871,  4942,  5013,  5083,  5153,
  5222,  5291,  5367,  5442,  5517,  5591,  5665,  5745,  5825,  5905,  5984,
  6063,  6149,  6234,  6319,  6404,  6495,  6587,  6678,  6769,  6867,  6966,
  7064,  7163,  7269,  7376,  7483,  7599,  7715,  7832,  7958,  8085,  8214,
  8352,  8492,  8635,  8788,  8945,  9104,  9275,  9450,  9639,  9832,  10031,
  10245, 10465, 10702, 10946, 11210, 11482, 11776, 12081, 12409, 12750, 13118,
  13501, 13913, 14343, 14807, 15290, 15812, 16356, 16943, 17575, 18237, 18949,
  19718, 20521, 21387,
};
#endif

static const int16_t ac_qlookup[QINDEX_RANGE] = {
  4,    8,    9,    10,   11,   12,   13,   14,   15,   16,   17,   18,   19,
  20,   21,   22,   23,   24,   25,   26,   27,   28,   29,   30,   31,   32,
  33,   34,   35,   36,   37,   38,   39,   40,   41,   42,   43,   44,   45,
  46,   47,   48,   49,   50,   51,   52,   53,   54,   55,   56,   57,   58,
  59,   60,   61,   62,   63,   64,   65,   66,   67,   68,   69,   70,   71,
  72,   73,   74,   75,   76,   77,   78,   79,   80,   81,   82,   83,   84,
  85,   86,   87,   88,   89,   90,   91,   92,   93,   94,   95,   96,   97,
  98,   99,   100,  101,  102,  104,  106,  108,  110,  112,  114,  116,  118,
  120,  122,  124,  126,  128,  130,  132,  134,  136,  138,  140,  142,  144,
  146,  148,  150,  152,  155,  158,  161,  164,  167,  170,  173,  176,  179,
  182,  185,  188,  191,  194,  197,  200,  203,  207,  211,  215,  219,  223,
  227,  231,  235,  239,  243,  247,  251,  255,  260,  265,  270,  275,  280,
  285,  290,  295,  300,  305,  311,  317,  323,  329,  335,  341,  347,  353,
  359,  366,  373,  380,  387,  394,  401,  408,  416,  424,  432,  440,  448,
  456,  465,  474,  483,  492,  501,  510,  520,  530,  540,  550,  560,  571,
  582,  593,  604,  615,  627,  639,  651,  663,  676,  689,  702,  715,  729,
  743,  757,  771,  786,  801,  816,  832,  848,  864,  881,  898,  915,  933,
  951,  969,  988,  1007, 1026, 1046, 1066, 1087, 1108, 1129, 1151, 1173, 1196,
  1219, 1243, 1267, 1292, 1317, 1343, 1369, 1396, 1423, 1451, 1479, 1508, 1537,
  1567, 1597, 1628, 1660, 1692, 1725, 1759, 1793, 1828,
};

#if CONFIG_HIGHBITDEPTH
static const int16_t ac_qlookup_10[QINDEX_RANGE] = {
  4,    9,    11,   13,   16,   18,   21,   24,   27,   30,   33,   37,   40,
  44,   48,   51,   55,   59,   63,   67,   71,   75,   79,   83,   88,   92,
  96,   100,  105,  109,  114,  118,  122,  127,  131,  136,  140,  145,  149,
  154,  158,  163,  168,  172,  177,  181,  186,  190,  195,  199,  204,  208,
  213,  217,  222,  226,  231,  235,  240,  244,  249,  253,  258,  262,  267,
  271,  275,  280,  284,  289,  293,  297,  302,  306,  311,  315,  319,  324,
  328,  332,  337,  341,  345,  349,  354,  358,  362,  367,  371,  375,  379,
  384,  388,  392,  396,  401,  409,  417,  425,  433,  441,  449,  458,  466,
  474,  482,  490,  498,  506,  514,  523,  531,  539,  547,  555,  563,  571,
  579,  588,  596,  604,  616,  628,  640,  652,  664,  676,  688,  700,  713,
  725,  737,  749,  761,  773,  785,  797,  809,  825,  841,  857,  873,  889,
  905,  922,  938,  954,  970,  986,  1002, 1018, 1038, 1058, 1078, 1098, 1118,
  1138, 1158, 1178, 1198, 1218, 1242, 1266, 1290, 1314, 1338, 1362, 1386, 1411,
  1435, 1463, 1491, 1519, 1547, 1575, 1603, 1631, 1663, 1695, 1727, 1759, 1791,
  1823, 1859, 1895, 1931, 1967, 2003, 2039, 2079, 2119, 2159, 2199, 2239, 2283,
  2327, 2371, 2415, 2459, 2507, 2555, 2603, 2651, 2703, 2755, 2807, 2859, 2915,
  2971, 3027, 3083, 3143, 3203, 3263, 3327, 3391, 3455, 3523, 3591, 3659, 3731,
  3803, 3876, 3952, 4028, 4104, 4184, 4264, 4348, 4432, 4516, 4604, 4692, 4784,
  4876, 4972, 5068, 5168, 5268, 5372, 5476, 5584, 5692, 5804, 5916, 6032, 6148,
  6268, 6388, 6512, 6640, 6768, 6900, 7036, 7172, 7312,
};

static const int16_t ac_qlookup_12[QINDEX_RANGE] = {
  4,     13,    19,    27,    35,    44,    54,    64,    75,    87,    99,
  112,   126,   139,   154,   168,   183,   199,   214,   230,   247,   263,
  280,   297,   314,   331,   349,   366,   384,   402,   420,   438,   456,
  475,   493,   511,   530,   548,   567,   586,   604,   623,   642,   660,
  679,   698,   716,   735,   753,   772,   791,   809,   828,   846,   865,
  884,   902,   920,   939,   957,   976,   994,   1012,  1030,  1049,  1067,
  1085,  1103,  1121,  1139,  1157,  1175,  1193,  1211,  1229,  1246,  1264,
  1282,  1299,  1317,  1335,  1352,  1370,  1387,  1405,  1422,  1440,  1457,
  1474,  1491,  1509,  1526,  1543,  1560,  1577,  1595,  1627,  1660,  1693,
  1725,  1758,  1791,  1824,  1856,  1889,  1922,  1954,  1987,  2020,  2052,
  2085,  2118,  2150,  2183,  2216,  2248,  2281,  2313,  2346,  2378,  2411,
  2459,  2508,  2556,  2605,  2653,  2701,  2750,  2798,  2847,  2895,  2943,
  2992,  3040,  3088,  3137,  3185,  3234,  3298,  3362,  3426,  3491,  3555,
  3619,  3684,  3748,  3812,  3876,  3941,  4005,  4069,  4149,  4230,  4310,
  4390,  4470,  4550,  4631,  4711,  4791,  4871,  4967,  5064,  5160,  5256,
  5352,  5448,  5544,  5641,  5737,  5849,  5961,  6073,  6185,  6297,  6410,
  6522,  6650,  6778,  6906,  7034,  7162,  7290,  7435,  7579,  7723,  7867,
  8011,  8155,  8315,  8475,  8635,  8795,  8956,  9132,  9308,  9484,  9660,
  9836,  10028, 10220, 10412, 10604, 10812, 11020, 11228, 11437, 11661, 11885,
  12109, 12333, 12573, 12813, 13053, 13309, 13565, 13821, 14093, 14365, 14637,
  14925, 15213, 15502, 15806, 16110, 16414, 16734, 17054, 17390, 17726, 18062,
  18414, 18766, 19134, 19502, 19886, 20270, 20670, 21070, 21486, 21902, 22334,
  22766, 23214, 23662, 24126, 24590, 25070, 25551, 26047, 26559, 27071, 27599,
  28143, 28687, 29247,
};
#endif

int16_t av1_dc_quant(int qindex, int delta, aom_bit_depth_t bit_depth) {
#if CONFIG_HIGHBITDEPTH
  switch (bit_depth) {
    case AOM_BITS_8: return dc_qlookup[clamp(qindex + delta, 0, MAXQ)];
    case AOM_BITS_10: return dc_qlookup_10[clamp(qindex + delta, 0, MAXQ)];
    case AOM_BITS_12: return dc_qlookup_12[clamp(qindex + delta, 0, MAXQ)];
    default:
      assert(0 && "bit_depth should be AOM_BITS_8, AOM_BITS_10 or AOM_BITS_12");
      return -1;
  }
#else
  (void)bit_depth;
  return dc_qlookup[clamp(qindex + delta, 0, MAXQ)];
#endif
}

int16_t av1_ac_quant(int qindex, int delta, aom_bit_depth_t bit_depth) {
#if CONFIG_HIGHBITDEPTH
  switch (bit_depth) {
    case AOM_BITS_8: return ac_qlookup[clamp(qindex + delta, 0, MAXQ)];
    case AOM_BITS_10: return ac_qlookup_10[clamp(qindex + delta, 0, MAXQ)];
    case AOM_BITS_12: return ac_qlookup_12[clamp(qindex + delta, 0, MAXQ)];
    default:
      assert(0 && "bit_depth should be AOM_BITS_8, AOM_BITS_10 or AOM_BITS_12");
      return -1;
  }
#else
  (void)bit_depth;
  return ac_qlookup[clamp(qindex + delta, 0, MAXQ)];
#endif
}

int16_t av1_qindex_from_ac(int ac, aom_bit_depth_t bit_depth) {
  int i;
  const int16_t *tab = ac_qlookup;
  ac *= 4;
#if CONFIG_HIGHBITDEPTH
  switch (bit_depth) {
    case AOM_BITS_10: {
      tab = ac_qlookup_10;
      ac *= 4;
      break;
    }
    case AOM_BITS_12: {
      tab = ac_qlookup_12;
      ac *= 16;
      break;
    }
    default:
      assert(0 && "bit_depth should be AOM_BITS_8, AOM_BITS_10 or AOM_BITS_12");
      return -1;
  }
#endif
  (void)bit_depth;
  for (i = 0; i < QINDEX_RANGE; i++) {
    if (ac <= tab[i]) return i;
  }
  return QINDEX_RANGE - 1;
}

int av1_get_qindex(const struct segmentation *seg, int segment_id,
                   int base_qindex) {
  if (segfeature_active(seg, segment_id, SEG_LVL_ALT_Q)) {
    const int data = get_segdata(seg, segment_id, SEG_LVL_ALT_Q);
    const int seg_qindex =
        seg->abs_delta == SEGMENT_ABSDATA ? data : base_qindex + data;
    return clamp(seg_qindex, 0, MAXQ);
  } else {
    return base_qindex;
  }
}

#if CONFIG_AOM_QM
qm_val_t *aom_iqmatrix(AV1_COMMON *cm, int qmlevel, int is_chroma,
                       TX_SIZE tx_size, int is_intra) {
  return &cm->giqmatrix[qmlevel][!!is_chroma][!!is_intra][tx_size][0];
}
qm_val_t *aom_qmatrix(AV1_COMMON *cm, int qmlevel, int is_chroma,
                      TX_SIZE tx_size, int is_intra) {
  return &cm->gqmatrix[qmlevel][!!is_chroma][!!is_intra][tx_size][0];
}

#if CONFIG_CB4X4
#define QM_TOTAL_SIZE 3348
#else
#define QM_TOTAL_SIZE 3344
#endif
static uint16_t wt_matrix_ref[NUM_QM_LEVELS][2][QM_TOTAL_SIZE];
static uint16_t iwt_matrix_ref[NUM_QM_LEVELS][2][QM_TOTAL_SIZE];

void aom_qm_init(AV1_COMMON *cm) {
  int q, c, f, t, size;
  int current;
  for (q = 0; q < NUM_QM_LEVELS; ++q) {
    for (c = 0; c < 2; ++c) {
      for (f = 0; f < 2; ++f) {
        current = 0;
        for (t = 0; t < TX_SIZES_ALL; ++t) {
          size = tx_size_2d[t];
          cm->gqmatrix[q][c][f][t] = &wt_matrix_ref[AOMMIN(
              NUM_QM_LEVELS - 1, f == 0 ? q + DEFAULT_QM_INTER_OFFSET : q)][c]
                                                   [current];
          cm->giqmatrix[q][c][f][t] = &iwt_matrix_ref[AOMMIN(
              NUM_QM_LEVELS - 1, f == 0 ? q + DEFAULT_QM_INTER_OFFSET : q)][c]
                                                     [current];
          current += size;
        }
      }
    }
  }
}

/* Provide 16 sets of quantization matrices for chroma and luma
   and each TX size. Matrices for different TX sizes are in fact
   sub-sampled from the 32x32 and 16x16 sizes, but explicitly
   defined here for convenience. Intra and inter matrix sets are the
   same but changing DEFAULT_QM_INTER_OFFSET from zero allows
   for different matrices for inter and intra blocks in the same
   frame.
   Matrices for different QM levels have been rescaled in the
   frequency domain according to different nominal viewing
   distances.
 */
static uint16_t iwt_matrix_ref[NUM_QM_LEVELS][2][QM_TOTAL_SIZE] = {
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        86, 173, 173, 334,
#endif
        /* Size 4x4 */
        65, 86, 146, 202, 86, 134, 189, 239, 146, 189, 274, 270, 202, 239, 270,
        405,
        /* Size 8x8 */
        64, 65, 77, 103, 136, 168, 196, 230, 65, 71, 81, 99, 126, 152, 178, 212,
        77, 81, 108, 130, 156, 182, 200, 211, 103, 99, 130, 165, 195, 223, 222,
        228, 136, 126, 156, 195, 235, 269, 290, 277, 168, 152, 182, 223, 269,
        304, 296, 366, 196, 178, 200, 222, 290, 296, 368, 485, 230, 212, 211,
        228, 277, 366, 485, 446,
        /* Size 16x16 */
        64, 63, 63, 68, 72, 88, 97, 118, 131, 160, 166, 184, 199, 216, 234, 253,
        63, 65, 65, 67, 69, 82, 89, 108, 119, 145, 150, 167, 182, 198, 215, 233,
        63, 65, 67, 71, 73, 84, 91, 108, 118, 143, 148, 163, 176, 189, 205, 223,
        68, 67, 71, 79, 85, 95, 102, 117, 126, 148, 153, 165, 170, 181, 192,
        209, 72, 69, 73, 85, 97, 108, 114, 128, 136, 158, 162, 177, 190, 191,
        225, 239, 88, 82, 84, 95, 108, 127, 134, 151, 159, 181, 185, 189, 210,
        199, 232, 252, 97, 89, 91, 102, 114, 134, 142, 161, 171, 193, 198, 218,
        209, 232, 231, 272, 118, 108, 108, 117, 128, 151, 161, 184, 196, 221,
        226, 226, 227, 263, 231, 296, 131, 119, 118, 126, 136, 159, 171, 196,
        210, 237, 242, 261, 273, 289, 245, 319, 160, 145, 143, 148, 158, 181,
        193, 221, 237, 268, 274, 278, 290, 272, 260, 359, 166, 150, 148, 153,
        162, 185, 198, 226, 242, 274, 280, 295, 286, 282, 297, 376, 184, 167,
        163, 165, 177, 189, 218, 226, 261, 278, 295, 319, 319, 329, 347, 429,
        199, 182, 176, 170, 190, 210, 209, 227, 273, 290, 286, 319, 351, 351,
        430, 448, 216, 198, 189, 181, 191, 199, 232, 263, 289, 272, 282, 329,
        351, 386, 385, 528, 234, 215, 205, 192, 225, 232, 231, 231, 245, 260,
        297, 347, 430, 385, 425, 418, 253, 233, 223, 209, 239, 252, 272, 296,
        319, 359, 376, 429, 448, 528, 418, 468,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 64, 68, 70, 72, 79, 88, 92, 97, 108, 118, 124, 131,
        143, 160, 163, 166, 177, 184, 192, 199, 207, 216, 225, 234, 243, 253,
        263, 63, 64, 64, 64, 64, 65, 68, 69, 70, 77, 85, 89, 93, 103, 113, 118,
        125, 136, 152, 154, 157, 168, 175, 182, 189, 196, 204, 213, 221, 230,
        239, 246, 63, 64, 65, 65, 65, 65, 67, 68, 69, 75, 82, 85, 89, 99, 108,
        113, 119, 130, 145, 147, 150, 160, 167, 174, 182, 190, 198, 206, 215,
        224, 233, 249, 63, 64, 65, 65, 66, 67, 69, 70, 71, 77, 83, 86, 90, 99,
        108, 113, 119, 129, 144, 146, 149, 159, 165, 173, 180, 187, 195, 202,
        211, 219, 228, 229, 63, 64, 65, 66, 67, 68, 71, 72, 73, 78, 84, 88, 91,
        100, 108, 113, 118, 129, 143, 145, 148, 157, 163, 169, 176, 181, 189,
        197, 205, 214, 223, 237, 64, 65, 65, 67, 68, 71, 74, 75, 77, 81, 85, 88,
        92, 99, 107, 111, 116, 126, 139, 141, 144, 152, 157, 163, 170, 178, 189,
        196, 204, 212, 220, 223, 68, 68, 67, 69, 71, 74, 79, 82, 85, 90, 95, 99,
        102, 109, 117, 121, 126, 136, 148, 151, 153, 161, 165, 166, 170, 176,
        181, 184, 192, 200, 209, 228, 70, 69, 68, 70, 72, 75, 82, 86, 91, 95,
        101, 104, 107, 114, 122, 126, 131, 141, 153, 155, 158, 166, 167, 173,
        186, 193, 198, 191, 196, 203, 210, 208, 72, 70, 69, 71, 73, 77, 85, 91,
        97, 101, 108, 111, 114, 120, 128, 132, 136, 146, 158, 160, 162, 171,
        177, 186, 190, 185, 191, 204, 225, 232, 239, 218, 79, 77, 75, 77, 78,
        81, 90, 95, 101, 108, 116, 119, 122, 130, 138, 142, 147, 156, 168, 171,
        173, 182, 188, 186, 187, 200, 213, 219, 204, 211, 218, 262, 88, 85, 82,
        83, 84, 85, 95, 101, 108, 116, 127, 130, 134, 143, 151, 155, 159, 169,
        181, 183, 185, 192, 189, 192, 210, 213, 199, 205, 232, 245, 252, 226,
        92, 89, 85, 86, 88, 88, 99, 104, 111, 119, 130, 134, 138, 148, 156, 160,
        165, 175, 187, 189, 191, 197, 198, 213, 207, 203, 226, 239, 227, 218,
        225, 273, 97, 93, 89, 90, 91, 92, 102, 107, 114, 122, 134, 138, 142,
        153, 161, 166, 171, 181, 193, 195, 198, 206, 218, 210, 209, 237, 232,
        215, 231, 266, 272, 232, 108, 103, 99, 99, 100, 99, 109, 114, 120, 130,
        143, 148, 153, 165, 174, 179, 185, 195, 208, 210, 213, 223, 221, 216,
        239, 222, 220, 263, 264, 228, 234, 297, 118, 113, 108, 108, 108, 107,
        117, 122, 128, 138, 151, 156, 161, 174, 184, 190, 196, 207, 221, 223,
        226, 232, 226, 248, 227, 235, 263, 232, 231, 292, 296, 241, 124, 118,
        113, 113, 113, 111, 121, 126, 132, 142, 155, 160, 166, 179, 190, 196,
        203, 214, 229, 231, 234, 238, 248, 241, 245, 271, 233, 263, 290, 241,
        244, 324, 131, 125, 119, 119, 118, 116, 126, 131, 136, 147, 159, 165,
        171, 185, 196, 203, 210, 222, 237, 239, 242, 251, 261, 246, 273, 242,
        289, 282, 245, 301, 319, 250, 143, 136, 130, 129, 129, 126, 136, 141,
        146, 156, 169, 175, 181, 195, 207, 214, 222, 235, 251, 253, 256, 269,
        257, 286, 252, 290, 272, 255, 327, 277, 257, 349, 160, 152, 145, 144,
        143, 139, 148, 153, 158, 168, 181, 187, 193, 208, 221, 229, 237, 251,
        268, 271, 274, 280, 278, 276, 290, 286, 272, 320, 260, 310, 359, 264,
        163, 154, 147, 146, 145, 141, 151, 155, 160, 171, 183, 189, 195, 210,
        223, 231, 239, 253, 271, 274, 277, 286, 305, 280, 307, 279, 335, 270,
        343, 311, 272, 394, 166, 157, 150, 149, 148, 144, 153, 158, 162, 173,
        185, 191, 198, 213, 226, 234, 242, 256, 274, 277, 280, 292, 295, 331,
        286, 336, 282, 366, 297, 309, 376, 278, 177, 168, 160, 159, 157, 152,
        161, 166, 171, 182, 192, 197, 206, 223, 232, 238, 251, 269, 280, 286,
        292, 304, 300, 306, 346, 296, 357, 301, 348, 366, 289, 409, 184, 175,
        167, 165, 163, 157, 165, 167, 177, 188, 189, 198, 218, 221, 226, 248,
        261, 257, 278, 305, 295, 300, 319, 307, 319, 351, 329, 350, 347, 316,
        429, 296, 192, 182, 174, 173, 169, 163, 166, 173, 186, 186, 192, 213,
        210, 216, 248, 241, 246, 286, 276, 280, 331, 306, 307, 334, 313, 334,
        351, 373, 337, 408, 309, 469, 199, 189, 182, 180, 176, 170, 170, 186,
        190, 187, 210, 207, 209, 239, 227, 245, 273, 252, 290, 307, 286, 346,
        319, 313, 351, 321, 351, 345, 430, 321, 448, 316, 207, 196, 190, 187,
        181, 178, 176, 193, 185, 200, 213, 203, 237, 222, 235, 271, 242, 290,
        286, 279, 336, 296, 351, 334, 321, 368, 329, 369, 337, 485, 328, 484,
        216, 204, 198, 195, 189, 189, 181, 198, 191, 213, 199, 226, 232, 220,
        263, 233, 289, 272, 272, 335, 282, 357, 329, 351, 351, 329, 386, 337,
        385, 345, 528, 336, 225, 213, 206, 202, 197, 196, 184, 191, 204, 219,
        205, 239, 215, 263, 232, 263, 282, 255, 320, 270, 366, 301, 350, 373,
        345, 369, 337, 405, 345, 401, 353, 574, 234, 221, 215, 211, 205, 204,
        192, 196, 225, 204, 232, 227, 231, 264, 231, 290, 245, 327, 260, 343,
        297, 348, 347, 337, 430, 337, 385, 345, 425, 353, 418, 361, 243, 230,
        224, 219, 214, 212, 200, 203, 232, 211, 245, 218, 266, 228, 292, 241,
        301, 277, 310, 311, 309, 366, 316, 408, 321, 485, 345, 401, 353, 446,
        362, 436, 253, 239, 233, 228, 223, 220, 209, 210, 239, 218, 252, 225,
        272, 234, 296, 244, 319, 257, 359, 272, 376, 289, 429, 309, 448, 328,
        528, 353, 418, 362, 468, 370, 263, 246, 249, 229, 237, 223, 228, 208,
        218, 262, 226, 273, 232, 297, 241, 324, 250, 349, 264, 394, 278, 409,
        296, 469, 316, 484, 336, 574, 361, 436, 370, 491,
        /* Size 4x8 */
        64, 85, 151, 202, 66, 85, 138, 188, 75, 116, 168, 205, 99, 142, 207,
        256, 131, 169, 250, 260, 161, 193, 285, 301, 188, 214, 299, 358, 219,
        218, 266, 343,
        /* Size 8x4 */
        64, 66, 75, 99, 131, 161, 188, 219, 85, 85, 116, 142, 169, 193, 214,
        218, 151, 138, 168, 207, 250, 285, 299, 266, 202, 188, 205, 256, 260,
        301, 358, 343,
        /* Size 8x16 */
        64, 64, 72, 107, 131, 174, 199, 229, 63, 66, 69, 99, 119, 157, 182, 211,
        64, 68, 73, 100, 118, 154, 174, 202, 68, 74, 85, 109, 126, 159, 170,
        190, 72, 77, 96, 120, 136, 168, 179, 221, 88, 86, 107, 143, 159, 191,
        205, 235, 96, 93, 113, 152, 171, 204, 232, 256, 117, 108, 127, 174, 196,
        232, 231, 282, 130, 117, 136, 185, 210, 249, 237, 292, 159, 140, 158,
        208, 236, 282, 285, 309, 165, 145, 162, 212, 242, 289, 331, 310, 183,
        160, 176, 218, 259, 294, 333, 325, 198, 174, 187, 209, 273, 344, 314,
        327, 215, 188, 210, 229, 248, 281, 350, 337, 232, 205, 222, 261, 319,
        363, 364, 342, 252, 223, 237, 272, 320, 378, 451, 431,
        /* Size 16x8 */
        64, 63, 64, 68, 72, 88, 96, 117, 130, 159, 165, 183, 198, 215, 232, 252,
        64, 66, 68, 74, 77, 86, 93, 108, 117, 140, 145, 160, 174, 188, 205, 223,
        72, 69, 73, 85, 96, 107, 113, 127, 136, 158, 162, 176, 187, 210, 222,
        237, 107, 99, 100, 109, 120, 143, 152, 174, 185, 208, 212, 218, 209,
        229, 261, 272, 131, 119, 118, 126, 136, 159, 171, 196, 210, 236, 242,
        259, 273, 248, 319, 320, 174, 157, 154, 159, 168, 191, 204, 232, 249,
        282, 289, 294, 344, 281, 363, 378, 199, 182, 174, 170, 179, 205, 232,
        231, 237, 285, 331, 333, 314, 350, 364, 451, 229, 211, 202, 190, 221,
        235, 256, 282, 292, 309, 310, 325, 327, 337, 342, 431,
        /* Size 16x32 */
        64, 63, 64, 68, 72, 88, 107, 118, 131, 158, 174, 186, 199, 214, 229,
        245, 63, 64, 65, 68, 71, 85, 103, 113, 124, 151, 165, 176, 189, 202,
        216, 229, 63, 65, 66, 67, 69, 82, 99, 108, 119, 144, 157, 169, 182, 196,
        211, 232, 63, 65, 67, 69, 71, 83, 99, 108, 119, 143, 156, 168, 180, 193,
        206, 213, 64, 65, 68, 71, 73, 84, 100, 108, 118, 142, 154, 165, 174,
        188, 202, 221, 64, 66, 71, 74, 77, 85, 99, 107, 116, 138, 150, 159, 172,
        188, 200, 210, 68, 68, 74, 79, 85, 96, 109, 117, 126, 147, 159, 163,
        170, 176, 190, 214, 70, 69, 75, 82, 90, 101, 114, 122, 131, 152, 164,
        169, 187, 184, 192, 195, 72, 69, 77, 86, 96, 108, 120, 128, 136, 157,
        168, 183, 179, 200, 221, 205, 79, 75, 81, 90, 101, 116, 130, 138, 147,
        168, 179, 183, 195, 205, 201, 248, 88, 82, 86, 96, 107, 126, 143, 151,
        159, 180, 191, 189, 205, 200, 235, 213, 92, 86, 89, 99, 110, 130, 147,
        156, 165, 186, 197, 211, 198, 231, 209, 260, 96, 90, 93, 102, 113, 134,
        152, 161, 171, 192, 204, 210, 232, 208, 256, 220, 107, 99, 100, 109,
        120, 142, 164, 174, 185, 207, 219, 213, 220, 256, 218, 284, 117, 108,
        108, 117, 127, 150, 174, 184, 196, 220, 232, 242, 231, 228, 282, 230,
        123, 114, 112, 121, 132, 155, 179, 190, 203, 228, 240, 242, 266, 257,
        234, 311, 130, 120, 117, 126, 136, 159, 185, 196, 210, 236, 249, 242,
        237, 269, 292, 239, 142, 131, 127, 136, 146, 169, 195, 207, 222, 250,
        264, 276, 276, 260, 264, 335, 159, 145, 140, 148, 158, 181, 208, 221,
        236, 267, 282, 279, 285, 309, 309, 253, 162, 148, 142, 151, 160, 183,
        210, 224, 239, 270, 285, 275, 275, 263, 293, 380, 165, 151, 145, 153,
        162, 185, 212, 226, 242, 273, 289, 314, 331, 348, 310, 268, 176, 161,
        154, 161, 171, 193, 219, 232, 251, 285, 298, 305, 306, 301, 339, 393,
        183, 167, 160, 165, 176, 203, 218, 226, 259, 293, 294, 301, 333, 337,
        325, 285, 191, 174, 167, 167, 182, 208, 210, 231, 276, 275, 298, 327,
        330, 365, 375, 451, 198, 181, 174, 171, 187, 210, 209, 259, 273, 269,
        344, 318, 314, 340, 327, 304, 206, 188, 181, 177, 205, 214, 214, 264,
        242, 299, 337, 312, 358, 358, 454, 465, 215, 195, 188, 184, 210, 199,
        229, 264, 248, 325, 281, 372, 350, 328, 337, 324, 223, 203, 196, 191,
        216, 205, 257, 232, 272, 321, 288, 394, 327, 392, 388, 550, 232, 211,
        205, 198, 222, 212, 261, 232, 319, 260, 363, 340, 364, 390, 342, 348,
        242, 219, 214, 205, 229, 218, 266, 238, 319, 266, 382, 301, 462, 343,
        430, 420, 252, 228, 223, 212, 237, 225, 272, 244, 320, 272, 378, 308,
        451, 351, 431, 356, 262, 232, 237, 214, 220, 270, 232, 318, 250, 385,
        278, 457, 315, 543, 359, 471,
        /* Size 32x16 */
        64, 63, 63, 63, 64, 64, 68, 70, 72, 79, 88, 92, 96, 107, 117, 123, 130,
        142, 159, 162, 165, 176, 183, 191, 198, 206, 215, 223, 232, 242, 252,
        262, 63, 64, 65, 65, 65, 66, 68, 69, 69, 75, 82, 86, 90, 99, 108, 114,
        120, 131, 145, 148, 151, 161, 167, 174, 181, 188, 195, 203, 211, 219,
        228, 232, 64, 65, 66, 67, 68, 71, 74, 75, 77, 81, 86, 89, 93, 100, 108,
        112, 117, 127, 140, 142, 145, 154, 160, 167, 174, 181, 188, 196, 205,
        214, 223, 237, 68, 68, 67, 69, 71, 74, 79, 82, 86, 90, 96, 99, 102, 109,
        117, 121, 126, 136, 148, 151, 153, 161, 165, 167, 171, 177, 184, 191,
        198, 205, 212, 214, 72, 71, 69, 71, 73, 77, 85, 90, 96, 101, 107, 110,
        113, 120, 127, 132, 136, 146, 158, 160, 162, 171, 176, 182, 187, 205,
        210, 216, 222, 229, 237, 220, 88, 85, 82, 83, 84, 85, 96, 101, 108, 116,
        126, 130, 134, 142, 150, 155, 159, 169, 181, 183, 185, 193, 203, 208,
        210, 214, 199, 205, 212, 218, 225, 270, 107, 103, 99, 99, 100, 99, 109,
        114, 120, 130, 143, 147, 152, 164, 174, 179, 185, 195, 208, 210, 212,
        219, 218, 210, 209, 214, 229, 257, 261, 266, 272, 232, 118, 113, 108,
        108, 108, 107, 117, 122, 128, 138, 151, 156, 161, 174, 184, 190, 196,
        207, 221, 224, 226, 232, 226, 231, 259, 264, 264, 232, 232, 238, 244,
        318, 131, 124, 119, 119, 118, 116, 126, 131, 136, 147, 159, 165, 171,
        185, 196, 203, 210, 222, 236, 239, 242, 251, 259, 276, 273, 242, 248,
        272, 319, 319, 320, 250, 158, 151, 144, 143, 142, 138, 147, 152, 157,
        168, 180, 186, 192, 207, 220, 228, 236, 250, 267, 270, 273, 285, 293,
        275, 269, 299, 325, 321, 260, 266, 272, 385, 174, 165, 157, 156, 154,
        150, 159, 164, 168, 179, 191, 197, 204, 219, 232, 240, 249, 264, 282,
        285, 289, 298, 294, 298, 344, 337, 281, 288, 363, 382, 378, 278, 186,
        176, 169, 168, 165, 159, 163, 169, 183, 183, 189, 211, 210, 213, 242,
        242, 242, 276, 279, 275, 314, 305, 301, 327, 318, 312, 372, 394, 340,
        301, 308, 457, 199, 189, 182, 180, 174, 172, 170, 187, 179, 195, 205,
        198, 232, 220, 231, 266, 237, 276, 285, 275, 331, 306, 333, 330, 314,
        358, 350, 327, 364, 462, 451, 315, 214, 202, 196, 193, 188, 188, 176,
        184, 200, 205, 200, 231, 208, 256, 228, 257, 269, 260, 309, 263, 348,
        301, 337, 365, 340, 358, 328, 392, 390, 343, 351, 543, 229, 216, 211,
        206, 202, 200, 190, 192, 221, 201, 235, 209, 256, 218, 282, 234, 292,
        264, 309, 293, 310, 339, 325, 375, 327, 454, 337, 388, 342, 430, 431,
        359, 245, 229, 232, 213, 221, 210, 214, 195, 205, 248, 213, 260, 220,
        284, 230, 311, 239, 335, 253, 380, 268, 393, 285, 451, 304, 465, 324,
        550, 348, 420, 356, 471,
        /* Size 4x16 */
        63, 88, 158, 214, 65, 82, 144, 196, 65, 84, 142, 188, 68, 96, 147, 176,
        69, 108, 157, 200, 82, 126, 180, 200, 90, 134, 192, 208, 108, 150, 220,
        228, 120, 159, 236, 269, 145, 181, 267, 309, 151, 185, 273, 348, 167,
        203, 293, 337, 181, 210, 269, 340, 195, 199, 325, 328, 211, 212, 260,
        390, 228, 225, 272, 351,
        /* Size 16x4 */
        63, 65, 65, 68, 69, 82, 90, 108, 120, 145, 151, 167, 181, 195, 211, 228,
        88, 82, 84, 96, 108, 126, 134, 150, 159, 181, 185, 203, 210, 199, 212,
        225, 158, 144, 142, 147, 157, 180, 192, 220, 236, 267, 273, 293, 269,
        325, 260, 272, 214, 196, 188, 176, 200, 200, 208, 228, 269, 309, 348,
        337, 340, 328, 390, 351,
        /* Size 8x32 */
        64, 64, 72, 107, 131, 174, 199, 229, 63, 65, 71, 103, 124, 165, 189,
        216, 63, 66, 69, 99, 119, 157, 182, 211, 63, 67, 71, 99, 119, 156, 180,
        206, 64, 68, 73, 100, 118, 154, 174, 202, 64, 71, 77, 99, 116, 150, 172,
        200, 68, 74, 85, 109, 126, 159, 170, 190, 70, 75, 90, 114, 131, 164,
        187, 192, 72, 77, 96, 120, 136, 168, 179, 221, 79, 81, 101, 130, 147,
        179, 195, 201, 88, 86, 107, 143, 159, 191, 205, 235, 92, 89, 110, 147,
        165, 197, 198, 209, 96, 93, 113, 152, 171, 204, 232, 256, 107, 100, 120,
        164, 185, 219, 220, 218, 117, 108, 127, 174, 196, 232, 231, 282, 123,
        112, 132, 179, 203, 240, 266, 234, 130, 117, 136, 185, 210, 249, 237,
        292, 142, 127, 146, 195, 222, 264, 276, 264, 159, 140, 158, 208, 236,
        282, 285, 309, 162, 142, 160, 210, 239, 285, 275, 293, 165, 145, 162,
        212, 242, 289, 331, 310, 176, 154, 171, 219, 251, 298, 306, 339, 183,
        160, 176, 218, 259, 294, 333, 325, 191, 167, 182, 210, 276, 298, 330,
        375, 198, 174, 187, 209, 273, 344, 314, 327, 206, 181, 205, 214, 242,
        337, 358, 454, 215, 188, 210, 229, 248, 281, 350, 337, 223, 196, 216,
        257, 272, 288, 327, 388, 232, 205, 222, 261, 319, 363, 364, 342, 242,
        214, 229, 266, 319, 382, 462, 430, 252, 223, 237, 272, 320, 378, 451,
        431, 262, 237, 220, 232, 250, 278, 315, 359,
        /* Size 32x8 */
        64, 63, 63, 63, 64, 64, 68, 70, 72, 79, 88, 92, 96, 107, 117, 123, 130,
        142, 159, 162, 165, 176, 183, 191, 198, 206, 215, 223, 232, 242, 252,
        262, 64, 65, 66, 67, 68, 71, 74, 75, 77, 81, 86, 89, 93, 100, 108, 112,
        117, 127, 140, 142, 145, 154, 160, 167, 174, 181, 188, 196, 205, 214,
        223, 237, 72, 71, 69, 71, 73, 77, 85, 90, 96, 101, 107, 110, 113, 120,
        127, 132, 136, 146, 158, 160, 162, 171, 176, 182, 187, 205, 210, 216,
        222, 229, 237, 220, 107, 103, 99, 99, 100, 99, 109, 114, 120, 130, 143,
        147, 152, 164, 174, 179, 185, 195, 208, 210, 212, 219, 218, 210, 209,
        214, 229, 257, 261, 266, 272, 232, 131, 124, 119, 119, 118, 116, 126,
        131, 136, 147, 159, 165, 171, 185, 196, 203, 210, 222, 236, 239, 242,
        251, 259, 276, 273, 242, 248, 272, 319, 319, 320, 250, 174, 165, 157,
        156, 154, 150, 159, 164, 168, 179, 191, 197, 204, 219, 232, 240, 249,
        264, 282, 285, 289, 298, 294, 298, 344, 337, 281, 288, 363, 382, 378,
        278, 199, 189, 182, 180, 174, 172, 170, 187, 179, 195, 205, 198, 232,
        220, 231, 266, 237, 276, 285, 275, 331, 306, 333, 330, 314, 358, 350,
        327, 364, 462, 451, 315, 229, 216, 211, 206, 202, 200, 190, 192, 221,
        201, 235, 209, 256, 218, 282, 234, 292, 264, 309, 293, 310, 339, 325,
        375, 327, 454, 337, 388, 342, 430, 431, 359 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        101, 125, 125, 205,
#endif
        /* Size 4x4 */
        71, 92, 115, 138, 92, 120, 139, 155, 115, 139, 180, 173, 138, 155, 173,
        232,
        /* Size 8x8 */
        63, 77, 94, 100, 114, 128, 139, 151, 77, 95, 93, 94, 104, 115, 125, 139,
        94, 93, 109, 115, 123, 132, 138, 138, 100, 94, 115, 133, 145, 155, 150,
        147, 114, 104, 123, 145, 164, 178, 184, 173, 128, 115, 132, 155, 178,
        193, 188, 214, 139, 125, 138, 150, 184, 188, 218, 261, 151, 139, 138,
        147, 173, 214, 261, 247,
        /* Size 16x16 */
        64, 61, 67, 83, 98, 98, 101, 109, 114, 127, 130, 138, 144, 150, 157,
        163, 61, 64, 71, 85, 93, 90, 93, 99, 104, 115, 117, 125, 131, 137, 144,
        150, 67, 71, 78, 90, 95, 91, 93, 99, 102, 113, 115, 121, 126, 130, 136,
        143, 83, 85, 90, 97, 101, 98, 100, 104, 107, 115, 117, 121, 121, 125,
        127, 134, 98, 93, 95, 101, 106, 107, 108, 111, 113, 121, 123, 128, 134,
        131, 147, 151, 98, 90, 91, 98, 107, 117, 120, 125, 127, 135, 136, 135,
        145, 136, 151, 158, 101, 93, 93, 100, 108, 120, 123, 130, 134, 142, 143,
        151, 144, 154, 150, 168, 109, 99, 99, 104, 111, 125, 130, 142, 147, 156,
        158, 156, 154, 170, 151, 180, 114, 104, 102, 107, 113, 127, 134, 147,
        153, 164, 166, 173, 177, 183, 158, 191, 127, 115, 113, 115, 121, 135,
        142, 156, 164, 178, 180, 181, 186, 175, 166, 209, 130, 117, 115, 117,
        123, 136, 143, 158, 166, 180, 183, 189, 184, 180, 185, 216, 138, 125,
        121, 121, 128, 135, 151, 156, 173, 181, 189, 199, 199, 202, 207, 239,
        144, 131, 126, 121, 134, 145, 144, 154, 177, 186, 184, 199, 212, 211,
        241, 246, 150, 137, 130, 125, 131, 136, 154, 170, 183, 175, 180, 202,
        211, 225, 224, 276, 157, 144, 136, 127, 147, 151, 150, 151, 158, 166,
        185, 207, 241, 224, 240, 236, 163, 150, 143, 134, 151, 158, 168, 180,
        191, 209, 216, 239, 246, 276, 236, 255,
        /* Size 32x32 */
        64, 62, 61, 64, 67, 73, 83, 90, 98, 97, 98, 100, 101, 104, 109, 112,
        114, 120, 127, 128, 130, 135, 138, 141, 144, 147, 150, 153, 157, 160,
        163, 167, 62, 63, 62, 66, 69, 77, 85, 90, 95, 94, 94, 95, 97, 100, 104,
        106, 109, 114, 121, 122, 123, 128, 130, 133, 136, 139, 142, 145, 148,
        151, 154, 156, 61, 62, 64, 67, 71, 80, 85, 89, 93, 91, 90, 91, 93, 95,
        99, 102, 104, 109, 115, 116, 117, 122, 125, 128, 131, 134, 137, 140,
        144, 147, 150, 158, 64, 66, 67, 71, 74, 83, 87, 90, 94, 92, 90, 92, 93,
        95, 99, 101, 103, 108, 114, 115, 116, 120, 123, 126, 130, 132, 135, 138,
        141, 144, 147, 145, 67, 69, 71, 74, 78, 87, 90, 92, 95, 93, 91, 92, 93,
        95, 99, 100, 102, 107, 113, 114, 115, 119, 121, 123, 126, 127, 130, 133,
        136, 140, 143, 150, 73, 77, 80, 83, 87, 95, 95, 95, 96, 93, 91, 92, 92,
        94, 97, 98, 100, 104, 109, 110, 111, 115, 116, 119, 122, 125, 131, 134,
        136, 139, 142, 140, 83, 85, 85, 87, 90, 95, 97, 99, 101, 99, 98, 99,
        100, 101, 104, 105, 107, 111, 115, 116, 117, 120, 121, 120, 121, 123,
        125, 124, 127, 131, 134, 144, 90, 90, 89, 90, 92, 95, 99, 101, 103, 103,
        103, 103, 104, 105, 107, 109, 110, 114, 118, 119, 120, 123, 123, 125,
        131, 134, 135, 130, 131, 133, 136, 130, 98, 95, 93, 94, 95, 96, 101,
        103, 106, 106, 107, 108, 108, 109, 111, 112, 113, 117, 121, 122, 123,
        126, 128, 133, 134, 129, 131, 137, 147, 149, 151, 138, 97, 94, 91, 92,
        93, 93, 99, 103, 106, 109, 111, 112, 113, 115, 117, 119, 120, 123, 127,
        128, 129, 132, 135, 132, 132, 138, 144, 145, 136, 138, 140, 162, 98, 94,
        90, 90, 91, 91, 98, 103, 107, 111, 117, 118, 120, 123, 125, 126, 127,
        131, 135, 135, 136, 138, 135, 136, 145, 145, 136, 138, 151, 156, 158,
        143, 100, 95, 91, 92, 92, 92, 99, 103, 108, 112, 118, 120, 121, 125,
        128, 129, 131, 134, 138, 139, 140, 141, 141, 148, 143, 140, 150, 155,
        148, 142, 144, 167, 101, 97, 93, 93, 93, 92, 100, 104, 108, 113, 120,
        121, 123, 127, 130, 132, 134, 137, 142, 142, 143, 146, 151, 146, 144,
        158, 154, 143, 150, 167, 168, 146, 104, 100, 95, 95, 95, 94, 101, 105,
        109, 115, 123, 125, 127, 133, 137, 139, 141, 145, 150, 150, 151, 155,
        153, 149, 160, 150, 148, 168, 167, 147, 149, 178, 109, 104, 99, 99, 99,
        97, 104, 107, 111, 117, 125, 128, 130, 137, 142, 144, 147, 151, 156,
        157, 158, 160, 156, 166, 154, 157, 170, 153, 151, 179, 180, 151, 112,
        106, 102, 101, 100, 98, 105, 109, 112, 119, 126, 129, 132, 139, 144,
        147, 150, 154, 160, 161, 162, 163, 167, 162, 163, 175, 155, 168, 180,
        155, 155, 191, 114, 109, 104, 103, 102, 100, 107, 110, 113, 120, 127,
        131, 134, 141, 147, 150, 153, 158, 164, 165, 166, 169, 173, 165, 177,
        161, 183, 178, 158, 184, 191, 157, 120, 114, 109, 108, 107, 104, 111,
        114, 117, 123, 131, 134, 137, 145, 151, 154, 158, 164, 170, 171, 172,
        178, 171, 184, 167, 184, 174, 165, 198, 173, 162, 203, 127, 121, 115,
        114, 113, 109, 115, 118, 121, 127, 135, 138, 142, 150, 156, 160, 164,
        170, 178, 179, 180, 183, 181, 180, 186, 183, 175, 195, 166, 189, 209,
        164, 128, 122, 116, 115, 114, 110, 116, 119, 122, 128, 135, 139, 142,
        150, 157, 161, 165, 171, 179, 180, 182, 185, 193, 182, 192, 180, 204,
        173, 205, 189, 170, 223, 130, 123, 117, 116, 115, 111, 117, 120, 123,
        129, 136, 140, 143, 151, 158, 162, 166, 172, 180, 182, 183, 188, 189,
        204, 184, 204, 180, 216, 185, 189, 216, 172, 135, 128, 122, 120, 119,
        115, 120, 123, 126, 132, 138, 141, 146, 155, 160, 163, 169, 178, 183,
        185, 188, 193, 191, 193, 210, 188, 213, 188, 208, 214, 179, 229, 138,
        130, 125, 123, 121, 116, 121, 123, 128, 135, 135, 141, 151, 153, 156,
        167, 173, 171, 181, 193, 189, 191, 199, 194, 199, 212, 202, 209, 207,
        193, 239, 181, 141, 133, 128, 126, 123, 119, 120, 125, 133, 132, 136,
        148, 146, 149, 166, 162, 165, 184, 180, 182, 204, 193, 194, 205, 196,
        205, 211, 219, 204, 231, 189, 254, 144, 136, 131, 130, 126, 122, 121,
        131, 134, 132, 145, 143, 144, 160, 154, 163, 177, 167, 186, 192, 184,
        210, 199, 196, 212, 199, 211, 209, 241, 197, 246, 192, 147, 139, 134,
        132, 127, 125, 123, 134, 129, 138, 145, 140, 158, 150, 157, 175, 161,
        184, 183, 180, 204, 188, 212, 205, 199, 218, 203, 218, 205, 261, 199,
        259, 150, 142, 137, 135, 130, 131, 125, 135, 131, 144, 136, 150, 154,
        148, 170, 155, 183, 174, 175, 204, 180, 213, 202, 211, 211, 203, 225,
        206, 224, 208, 276, 202, 153, 145, 140, 138, 133, 134, 124, 130, 137,
        145, 138, 155, 143, 168, 153, 168, 178, 165, 195, 173, 216, 188, 209,
        219, 209, 218, 206, 232, 209, 230, 211, 292, 157, 148, 144, 141, 136,
        136, 127, 131, 147, 136, 151, 148, 150, 167, 151, 180, 158, 198, 166,
        205, 185, 208, 207, 204, 241, 205, 224, 209, 240, 212, 236, 214, 160,
        151, 147, 144, 140, 139, 131, 133, 149, 138, 156, 142, 167, 147, 179,
        155, 184, 173, 189, 189, 189, 214, 193, 231, 197, 261, 208, 230, 212,
        247, 215, 243, 163, 154, 150, 147, 143, 142, 134, 136, 151, 140, 158,
        144, 168, 149, 180, 155, 191, 162, 209, 170, 216, 179, 239, 189, 246,
        199, 276, 211, 236, 215, 255, 218, 167, 156, 158, 145, 150, 140, 144,
        130, 138, 162, 143, 167, 146, 178, 151, 191, 157, 203, 164, 223, 172,
        229, 181, 254, 192, 259, 202, 292, 214, 243, 218, 263,
        /* Size 4x8 */
        63, 94, 120, 141, 80, 91, 109, 131, 92, 112, 128, 140, 96, 122, 150,
        167, 109, 131, 170, 169, 122, 139, 185, 189, 133, 145, 189, 214, 144,
        142, 169, 208,
        /* Size 8x4 */
        63, 80, 92, 96, 109, 122, 133, 144, 94, 91, 112, 122, 131, 139, 145,
        142, 120, 109, 128, 150, 170, 185, 189, 169, 141, 131, 140, 167, 169,
        189, 214, 208,
        /* Size 8x16 */
        64, 74, 96, 105, 115, 133, 144, 155, 61, 80, 92, 96, 104, 121, 131, 143,
        67, 86, 94, 95, 103, 118, 125, 136, 84, 95, 100, 101, 107, 120, 121,
        127, 98, 96, 106, 109, 114, 125, 127, 146, 98, 92, 107, 123, 128, 139,
        142, 153, 101, 93, 108, 128, 134, 146, 157, 164, 109, 98, 111, 137, 147,
        161, 156, 177, 114, 101, 113, 141, 153, 169, 160, 182, 127, 110, 121,
        150, 164, 184, 183, 190, 129, 112, 123, 151, 166, 187, 203, 191, 137,
        119, 128, 152, 173, 189, 205, 198, 143, 124, 132, 144, 177, 209, 197,
        200, 149, 130, 142, 153, 163, 180, 211, 206, 156, 136, 146, 166, 194,
        214, 216, 208, 163, 143, 150, 168, 191, 218, 247, 242,
        /* Size 16x8 */
        64, 61, 67, 84, 98, 98, 101, 109, 114, 127, 129, 137, 143, 149, 156,
        163, 74, 80, 86, 95, 96, 92, 93, 98, 101, 110, 112, 119, 124, 130, 136,
        143, 96, 92, 94, 100, 106, 107, 108, 111, 113, 121, 123, 128, 132, 142,
        146, 150, 105, 96, 95, 101, 109, 123, 128, 137, 141, 150, 151, 152, 144,
        153, 166, 168, 115, 104, 103, 107, 114, 128, 134, 147, 153, 164, 166,
        173, 177, 163, 194, 191, 133, 121, 118, 120, 125, 139, 146, 161, 169,
        184, 187, 189, 209, 180, 214, 218, 144, 131, 125, 121, 127, 142, 157,
        156, 160, 183, 203, 205, 197, 211, 216, 247, 155, 143, 136, 127, 146,
        153, 164, 177, 182, 190, 191, 198, 200, 206, 208, 242,
        /* Size 16x32 */
        64, 62, 74, 84, 96, 98, 105, 109, 115, 127, 133, 139, 144, 149, 155,
        161, 63, 63, 77, 85, 94, 94, 100, 104, 109, 120, 126, 131, 136, 141,
        147, 150, 61, 64, 80, 85, 92, 91, 96, 100, 104, 115, 121, 126, 131, 137,
        143, 152, 64, 68, 83, 88, 93, 91, 96, 99, 103, 114, 119, 125, 130, 135,
        140, 140, 67, 72, 86, 90, 94, 92, 95, 99, 103, 112, 118, 122, 125, 130,
        136, 145, 74, 80, 94, 95, 95, 91, 94, 97, 100, 109, 114, 117, 123, 131,
        135, 136, 84, 86, 95, 97, 100, 99, 101, 104, 107, 115, 120, 120, 121,
        122, 127, 140, 90, 89, 95, 99, 103, 103, 105, 108, 110, 118, 123, 123,
        132, 127, 130, 127, 98, 93, 96, 101, 106, 107, 109, 111, 114, 121, 125,
        132, 127, 137, 146, 134, 97, 92, 94, 100, 106, 112, 115, 118, 120, 128,
        132, 132, 137, 140, 135, 158, 98, 91, 92, 99, 107, 117, 123, 125, 128,
        135, 139, 135, 142, 137, 153, 139, 99, 92, 92, 99, 107, 118, 125, 128,
        131, 138, 142, 147, 138, 154, 139, 164, 101, 93, 93, 100, 108, 119, 128,
        131, 134, 142, 146, 147, 157, 141, 164, 143, 104, 96, 95, 101, 109, 122,
        133, 137, 142, 150, 154, 149, 150, 167, 145, 175, 109, 100, 98, 104,
        111, 125, 137, 142, 147, 156, 161, 164, 156, 152, 177, 148, 111, 102,
        99, 105, 112, 126, 139, 144, 150, 160, 165, 164, 174, 167, 153, 188,
        114, 104, 101, 107, 113, 128, 141, 147, 153, 164, 169, 164, 160, 173,
        182, 154, 120, 109, 105, 111, 117, 131, 145, 151, 158, 170, 176, 180,
        179, 169, 169, 200, 127, 115, 110, 116, 121, 135, 150, 157, 164, 178,
        184, 182, 183, 192, 190, 161, 128, 117, 111, 116, 122, 136, 151, 157,
        165, 179, 185, 180, 179, 171, 183, 220, 129, 118, 112, 117, 123, 136,
        151, 158, 166, 180, 187, 197, 203, 210, 191, 169, 134, 122, 116, 120,
        126, 139, 153, 160, 169, 185, 191, 194, 193, 189, 204, 225, 137, 125,
        119, 122, 128, 144, 152, 156, 173, 188, 189, 192, 205, 205, 198, 178,
        140, 127, 121, 121, 130, 145, 146, 158, 180, 179, 190, 202, 203, 217,
        219, 249, 143, 130, 124, 122, 132, 145, 144, 170, 177, 176, 209, 198,
        197, 207, 200, 188, 146, 133, 127, 125, 141, 145, 146, 172, 161, 189,
        205, 196, 215, 214, 250, 253, 149, 136, 130, 127, 142, 136, 153, 170,
        163, 199, 180, 220, 211, 202, 206, 198, 152, 138, 133, 129, 144, 138,
        166, 153, 174, 196, 183, 227, 202, 228, 226, 284, 156, 141, 136, 132,
        146, 140, 166, 151, 194, 167, 214, 205, 216, 226, 208, 210, 159, 144,
        140, 134, 148, 142, 167, 153, 193, 169, 221, 187, 253, 208, 242, 237,
        163, 147, 143, 137, 150, 144, 168, 155, 191, 171, 218, 190, 247, 211,
        242, 214, 166, 147, 150, 134, 139, 166, 147, 189, 157, 219, 172, 250,
        192, 281, 214, 257,
        /* Size 32x16 */
        64, 63, 61, 64, 67, 74, 84, 90, 98, 97, 98, 99, 101, 104, 109, 111, 114,
        120, 127, 128, 129, 134, 137, 140, 143, 146, 149, 152, 156, 159, 163,
        166, 62, 63, 64, 68, 72, 80, 86, 89, 93, 92, 91, 92, 93, 96, 100, 102,
        104, 109, 115, 117, 118, 122, 125, 127, 130, 133, 136, 138, 141, 144,
        147, 147, 74, 77, 80, 83, 86, 94, 95, 95, 96, 94, 92, 92, 93, 95, 98,
        99, 101, 105, 110, 111, 112, 116, 119, 121, 124, 127, 130, 133, 136,
        140, 143, 150, 84, 85, 85, 88, 90, 95, 97, 99, 101, 100, 99, 99, 100,
        101, 104, 105, 107, 111, 116, 116, 117, 120, 122, 121, 122, 125, 127,
        129, 132, 134, 137, 134, 96, 94, 92, 93, 94, 95, 100, 103, 106, 106,
        107, 107, 108, 109, 111, 112, 113, 117, 121, 122, 123, 126, 128, 130,
        132, 141, 142, 144, 146, 148, 150, 139, 98, 94, 91, 91, 92, 91, 99, 103,
        107, 112, 117, 118, 119, 122, 125, 126, 128, 131, 135, 136, 136, 139,
        144, 145, 145, 145, 136, 138, 140, 142, 144, 166, 105, 100, 96, 96, 95,
        94, 101, 105, 109, 115, 123, 125, 128, 133, 137, 139, 141, 145, 150,
        151, 151, 153, 152, 146, 144, 146, 153, 166, 166, 167, 168, 147, 109,
        104, 100, 99, 99, 97, 104, 108, 111, 118, 125, 128, 131, 137, 142, 144,
        147, 151, 157, 157, 158, 160, 156, 158, 170, 172, 170, 153, 151, 153,
        155, 189, 115, 109, 104, 103, 103, 100, 107, 110, 114, 120, 128, 131,
        134, 142, 147, 150, 153, 158, 164, 165, 166, 169, 173, 180, 177, 161,
        163, 174, 194, 193, 191, 157, 127, 120, 115, 114, 112, 109, 115, 118,
        121, 128, 135, 138, 142, 150, 156, 160, 164, 170, 178, 179, 180, 185,
        188, 179, 176, 189, 199, 196, 167, 169, 171, 219, 133, 126, 121, 119,
        118, 114, 120, 123, 125, 132, 139, 142, 146, 154, 161, 165, 169, 176,
        184, 185, 187, 191, 189, 190, 209, 205, 180, 183, 214, 221, 218, 172,
        139, 131, 126, 125, 122, 117, 120, 123, 132, 132, 135, 147, 147, 149,
        164, 164, 164, 180, 182, 180, 197, 194, 192, 202, 198, 196, 220, 227,
        205, 187, 190, 250, 144, 136, 131, 130, 125, 123, 121, 132, 127, 137,
        142, 138, 157, 150, 156, 174, 160, 179, 183, 179, 203, 193, 205, 203,
        197, 215, 211, 202, 216, 253, 247, 192, 149, 141, 137, 135, 130, 131,
        122, 127, 137, 140, 137, 154, 141, 167, 152, 167, 173, 169, 192, 171,
        210, 189, 205, 217, 207, 214, 202, 228, 226, 208, 211, 281, 155, 147,
        143, 140, 136, 135, 127, 130, 146, 135, 153, 139, 164, 145, 177, 153,
        182, 169, 190, 183, 191, 204, 198, 219, 200, 250, 206, 226, 208, 242,
        242, 214, 161, 150, 152, 140, 145, 136, 140, 127, 134, 158, 139, 164,
        143, 175, 148, 188, 154, 200, 161, 220, 169, 225, 178, 249, 188, 253,
        198, 284, 210, 237, 214, 257,
        /* Size 4x16 */
        62, 98, 127, 149, 64, 91, 115, 137, 72, 92, 112, 130, 86, 99, 115, 122,
        93, 107, 121, 137, 91, 117, 135, 137, 93, 119, 142, 141, 100, 125, 156,
        152, 104, 128, 164, 173, 115, 135, 178, 192, 118, 136, 180, 210, 125,
        144, 188, 205, 130, 145, 176, 207, 136, 136, 199, 202, 141, 140, 167,
        226, 147, 144, 171, 211,
        /* Size 16x4 */
        62, 64, 72, 86, 93, 91, 93, 100, 104, 115, 118, 125, 130, 136, 141, 147,
        98, 91, 92, 99, 107, 117, 119, 125, 128, 135, 136, 144, 145, 136, 140,
        144, 127, 115, 112, 115, 121, 135, 142, 156, 164, 178, 180, 188, 176,
        199, 167, 171, 149, 137, 130, 122, 137, 137, 141, 152, 173, 192, 210,
        205, 207, 202, 226, 211,
        /* Size 8x32 */
        64, 74, 96, 105, 115, 133, 144, 155, 63, 77, 94, 100, 109, 126, 136,
        147, 61, 80, 92, 96, 104, 121, 131, 143, 64, 83, 93, 96, 103, 119, 130,
        140, 67, 86, 94, 95, 103, 118, 125, 136, 74, 94, 95, 94, 100, 114, 123,
        135, 84, 95, 100, 101, 107, 120, 121, 127, 90, 95, 103, 105, 110, 123,
        132, 130, 98, 96, 106, 109, 114, 125, 127, 146, 97, 94, 106, 115, 120,
        132, 137, 135, 98, 92, 107, 123, 128, 139, 142, 153, 99, 92, 107, 125,
        131, 142, 138, 139, 101, 93, 108, 128, 134, 146, 157, 164, 104, 95, 109,
        133, 142, 154, 150, 145, 109, 98, 111, 137, 147, 161, 156, 177, 111, 99,
        112, 139, 150, 165, 174, 153, 114, 101, 113, 141, 153, 169, 160, 182,
        120, 105, 117, 145, 158, 176, 179, 169, 127, 110, 121, 150, 164, 184,
        183, 190, 128, 111, 122, 151, 165, 185, 179, 183, 129, 112, 123, 151,
        166, 187, 203, 191, 134, 116, 126, 153, 169, 191, 193, 204, 137, 119,
        128, 152, 173, 189, 205, 198, 140, 121, 130, 146, 180, 190, 203, 219,
        143, 124, 132, 144, 177, 209, 197, 200, 146, 127, 141, 146, 161, 205,
        215, 250, 149, 130, 142, 153, 163, 180, 211, 206, 152, 133, 144, 166,
        174, 183, 202, 226, 156, 136, 146, 166, 194, 214, 216, 208, 159, 140,
        148, 167, 193, 221, 253, 242, 163, 143, 150, 168, 191, 218, 247, 242,
        166, 150, 139, 147, 157, 172, 192, 214,
        /* Size 32x8 */
        64, 63, 61, 64, 67, 74, 84, 90, 98, 97, 98, 99, 101, 104, 109, 111, 114,
        120, 127, 128, 129, 134, 137, 140, 143, 146, 149, 152, 156, 159, 163,
        166, 74, 77, 80, 83, 86, 94, 95, 95, 96, 94, 92, 92, 93, 95, 98, 99,
        101, 105, 110, 111, 112, 116, 119, 121, 124, 127, 130, 133, 136, 140,
        143, 150, 96, 94, 92, 93, 94, 95, 100, 103, 106, 106, 107, 107, 108,
        109, 111, 112, 113, 117, 121, 122, 123, 126, 128, 130, 132, 141, 142,
        144, 146, 148, 150, 139, 105, 100, 96, 96, 95, 94, 101, 105, 109, 115,
        123, 125, 128, 133, 137, 139, 141, 145, 150, 151, 151, 153, 152, 146,
        144, 146, 153, 166, 166, 167, 168, 147, 115, 109, 104, 103, 103, 100,
        107, 110, 114, 120, 128, 131, 134, 142, 147, 150, 153, 158, 164, 165,
        166, 169, 173, 180, 177, 161, 163, 174, 194, 193, 191, 157, 133, 126,
        121, 119, 118, 114, 120, 123, 125, 132, 139, 142, 146, 154, 161, 165,
        169, 176, 184, 185, 187, 191, 189, 190, 209, 205, 180, 183, 214, 221,
        218, 172, 144, 136, 131, 130, 125, 123, 121, 132, 127, 137, 142, 138,
        157, 150, 156, 174, 160, 179, 183, 179, 203, 193, 205, 203, 197, 215,
        211, 202, 216, 253, 247, 192, 155, 147, 143, 140, 136, 135, 127, 130,
        146, 135, 153, 139, 164, 145, 177, 153, 182, 169, 190, 183, 191, 204,
        198, 219, 200, 250, 206, 226, 208, 242, 242, 214 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        79, 164, 164, 311,
#endif
        /* Size 4x4 */
        65, 82, 138, 189, 82, 127, 176, 219, 138, 176, 254, 259, 189, 219, 259,
        371,
        /* Size 8x8 */
        64, 65, 74, 95, 125, 157, 184, 213, 65, 71, 79, 93, 116, 144, 168, 197,
        74, 79, 103, 120, 142, 169, 188, 198, 95, 93, 120, 146, 174, 201, 210,
        215, 125, 116, 142, 174, 210, 242, 266, 258, 157, 144, 169, 201, 242,
        280, 284, 332, 184, 168, 188, 210, 266, 284, 340, 426, 213, 197, 198,
        215, 258, 332, 426, 406,
        /* Size 16x16 */
        64, 63, 63, 65, 72, 79, 95, 108, 123, 143, 160, 173, 187, 202, 217, 234,
        63, 64, 65, 66, 69, 75, 88, 100, 113, 131, 146, 158, 172, 186, 200, 216,
        63, 65, 67, 69, 73, 78, 90, 100, 112, 129, 143, 154, 166, 177, 191, 207,
        65, 66, 69, 73, 80, 84, 94, 103, 114, 130, 142, 153, 160, 170, 181, 196,
        72, 69, 73, 80, 97, 101, 112, 120, 131, 146, 158, 168, 177, 180, 206,
        217, 79, 75, 78, 84, 101, 108, 121, 130, 141, 156, 168, 178, 194, 189,
        214, 230, 95, 88, 90, 94, 112, 121, 139, 151, 163, 178, 190, 200, 200,
        215, 216, 247, 108, 100, 100, 103, 120, 130, 151, 165, 178, 195, 208,
        218, 216, 241, 221, 268, 123, 113, 112, 114, 131, 141, 163, 178, 194,
        213, 227, 238, 250, 264, 234, 288, 143, 131, 129, 130, 146, 156, 178,
        195, 213, 235, 251, 263, 270, 259, 249, 321, 160, 146, 143, 142, 158,
        168, 190, 208, 227, 251, 268, 281, 275, 271, 280, 338, 173, 158, 154,
        153, 168, 178, 200, 218, 238, 263, 281, 295, 303, 308, 321, 380, 187,
        172, 166, 160, 177, 194, 200, 216, 250, 270, 275, 303, 325, 329, 384,
        399, 202, 186, 177, 170, 180, 189, 215, 241, 264, 259, 271, 308, 329,
        355, 357, 461, 217, 200, 191, 181, 206, 214, 216, 221, 234, 249, 280,
        321, 384, 357, 389, 387, 234, 216, 207, 196, 217, 230, 247, 268, 288,
        321, 338, 380, 399, 461, 387, 425,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 64, 65, 68, 72, 76, 79, 88, 95, 99, 108, 118, 123,
        131, 143, 153, 160, 166, 173, 180, 187, 194, 202, 209, 217, 225, 234,
        243, 63, 64, 64, 64, 64, 65, 66, 68, 70, 74, 77, 85, 91, 95, 103, 113,
        117, 125, 136, 145, 152, 157, 164, 171, 177, 184, 191, 198, 205, 213,
        221, 227, 63, 64, 64, 65, 65, 65, 66, 67, 69, 73, 75, 82, 88, 92, 100,
        109, 113, 120, 131, 140, 146, 152, 158, 165, 172, 178, 186, 193, 200,
        208, 216, 229, 63, 64, 65, 65, 65, 66, 67, 68, 70, 73, 76, 82, 88, 91,
        99, 108, 112, 119, 130, 138, 144, 150, 156, 163, 169, 175, 182, 189,
        196, 203, 211, 212, 63, 64, 65, 65, 67, 68, 69, 71, 73, 76, 78, 84, 90,
        93, 100, 108, 112, 118, 129, 137, 143, 148, 154, 160, 166, 171, 177,
        184, 191, 199, 207, 218, 64, 65, 65, 66, 68, 71, 72, 74, 77, 79, 81, 85,
        90, 93, 99, 107, 110, 116, 126, 133, 139, 144, 149, 156, 162, 168, 177,
        184, 190, 197, 204, 208, 65, 66, 66, 67, 69, 72, 73, 76, 80, 82, 84, 89,
        94, 97, 103, 111, 114, 120, 130, 137, 142, 147, 153, 156, 160, 166, 170,
        174, 181, 188, 196, 211, 68, 68, 67, 68, 71, 74, 76, 79, 85, 88, 90, 95,
        100, 103, 109, 117, 120, 126, 136, 143, 148, 153, 159, 164, 173, 178,
        183, 179, 183, 189, 196, 195, 72, 70, 69, 70, 73, 77, 80, 85, 97, 100,
        101, 108, 112, 115, 120, 128, 131, 136, 146, 153, 158, 162, 168, 173,
        177, 175, 180, 191, 206, 211, 217, 203, 76, 74, 73, 73, 76, 79, 82, 88,
        100, 103, 105, 112, 117, 120, 126, 134, 137, 142, 152, 159, 164, 169,
        174, 175, 178, 188, 197, 201, 192, 198, 204, 237, 79, 77, 75, 76, 78,
        81, 84, 90, 101, 105, 108, 116, 121, 124, 130, 138, 141, 147, 156, 163,
        168, 173, 178, 184, 194, 197, 189, 194, 214, 224, 230, 211, 88, 85, 82,
        82, 84, 85, 89, 95, 108, 112, 116, 127, 133, 136, 143, 151, 154, 159,
        169, 176, 181, 185, 190, 197, 195, 194, 210, 219, 211, 206, 211, 248,
        95, 91, 88, 88, 90, 90, 94, 100, 112, 117, 121, 133, 139, 142, 151, 159,
        163, 168, 178, 185, 190, 195, 200, 199, 200, 218, 215, 204, 216, 242,
        247, 218, 99, 95, 92, 91, 93, 93, 97, 103, 115, 120, 124, 136, 142, 146,
        155, 163, 167, 174, 184, 191, 196, 201, 206, 208, 220, 210, 210, 240,
        241, 215, 221, 268, 108, 103, 100, 99, 100, 99, 103, 109, 120, 126, 130,
        143, 151, 155, 165, 174, 178, 185, 195, 203, 208, 213, 218, 227, 216,
        223, 241, 220, 221, 264, 268, 227, 118, 113, 109, 108, 108, 107, 111,
        117, 128, 134, 138, 151, 159, 163, 174, 184, 189, 196, 207, 215, 221,
        226, 232, 228, 232, 247, 223, 245, 263, 229, 231, 291, 123, 117, 113,
        112, 112, 110, 114, 120, 131, 137, 141, 154, 163, 167, 178, 189, 194,
        202, 213, 221, 227, 232, 238, 237, 250, 233, 264, 259, 234, 275, 288,
        237, 131, 125, 120, 119, 118, 116, 120, 126, 136, 142, 147, 159, 168,
        174, 185, 196, 202, 210, 222, 231, 237, 242, 248, 260, 242, 266, 255,
        244, 294, 258, 245, 314, 143, 136, 131, 130, 129, 126, 130, 136, 146,
        152, 156, 169, 178, 184, 195, 207, 213, 222, 235, 244, 251, 256, 263,
        261, 270, 267, 259, 291, 249, 286, 321, 251, 153, 145, 140, 138, 137,
        133, 137, 143, 153, 159, 163, 176, 185, 191, 203, 215, 221, 231, 244,
        254, 261, 267, 273, 270, 282, 267, 302, 259, 311, 288, 260, 351, 160,
        152, 146, 144, 143, 139, 142, 148, 158, 164, 168, 181, 190, 196, 208,
        221, 227, 237, 251, 261, 268, 274, 281, 298, 275, 305, 271, 328, 280,
        289, 338, 265, 166, 157, 152, 150, 148, 144, 147, 153, 162, 169, 173,
        185, 195, 201, 213, 226, 232, 242, 256, 267, 274, 280, 288, 292, 314,
        284, 324, 286, 320, 332, 276, 365, 173, 164, 158, 156, 154, 149, 153,
        159, 168, 174, 178, 190, 200, 206, 218, 232, 238, 248, 263, 273, 281,
        288, 295, 295, 303, 323, 308, 323, 321, 299, 380, 282, 180, 171, 165,
        163, 160, 156, 156, 164, 173, 175, 184, 197, 199, 208, 227, 228, 237,
        260, 261, 270, 298, 292, 295, 311, 302, 315, 327, 341, 317, 367, 295,
        414, 187, 177, 172, 169, 166, 162, 160, 173, 177, 178, 194, 195, 200,
        220, 216, 232, 250, 242, 270, 282, 275, 314, 303, 302, 325, 308, 329,
        326, 384, 307, 399, 301, 194, 184, 178, 175, 171, 168, 166, 178, 175,
        188, 197, 194, 218, 210, 223, 247, 233, 266, 267, 267, 305, 284, 323,
        315, 308, 340, 315, 344, 322, 426, 313, 429, 202, 191, 186, 182, 177,
        177, 170, 183, 180, 197, 189, 210, 215, 210, 241, 223, 264, 255, 259,
        302, 271, 324, 308, 327, 329, 315, 355, 323, 357, 329, 461, 320, 209,
        198, 193, 189, 184, 184, 174, 179, 191, 201, 194, 219, 204, 240, 220,
        245, 259, 244, 291, 259, 328, 286, 323, 341, 326, 344, 323, 371, 330,
        372, 337, 498, 217, 205, 200, 196, 191, 190, 181, 183, 206, 192, 214,
        211, 216, 241, 221, 263, 234, 294, 249, 311, 280, 320, 321, 317, 384,
        322, 357, 330, 389, 337, 387, 344, 225, 213, 208, 203, 199, 197, 188,
        189, 211, 198, 224, 206, 242, 215, 264, 229, 275, 258, 286, 288, 289,
        332, 299, 367, 307, 426, 329, 372, 337, 406, 345, 402, 234, 221, 216,
        211, 207, 204, 196, 196, 217, 204, 230, 211, 247, 221, 268, 231, 288,
        245, 321, 260, 338, 276, 380, 295, 399, 313, 461, 337, 387, 345, 425,
        353, 243, 227, 229, 212, 218, 208, 211, 195, 203, 237, 211, 248, 218,
        268, 227, 291, 237, 314, 251, 351, 265, 365, 282, 414, 301, 429, 320,
        498, 344, 402, 353, 444,
        /* Size 4x8 */
        64, 85, 139, 191, 66, 85, 128, 177, 73, 112, 154, 192, 92, 135, 186,
        235, 120, 159, 225, 247, 151, 185, 260, 286, 176, 197, 279, 336, 203,
        206, 254, 328,
        /* Size 8x4 */
        64, 66, 73, 92, 120, 151, 176, 203, 85, 85, 112, 135, 159, 185, 197,
        206, 139, 128, 154, 186, 225, 260, 279, 254, 191, 177, 192, 235, 247,
        286, 336, 328,
        /* Size 8x16 */
        64, 64, 72, 95, 131, 158, 189, 215, 63, 65, 70, 88, 120, 145, 173, 199,
        64, 68, 73, 90, 118, 142, 166, 190, 65, 71, 80, 94, 120, 142, 161, 180,
        72, 75, 96, 112, 136, 157, 171, 203, 79, 80, 101, 121, 147, 168, 192,
        217, 94, 91, 112, 139, 168, 190, 215, 235, 107, 100, 120, 150, 185, 207,
        220, 258, 122, 112, 131, 163, 201, 226, 230, 268, 142, 128, 146, 178,
        222, 250, 266, 285, 159, 141, 158, 190, 236, 267, 302, 290, 172, 152,
        168, 200, 248, 280, 312, 305, 186, 164, 178, 200, 251, 311, 304, 311,
        200, 177, 194, 216, 238, 270, 328, 323, 216, 191, 204, 239, 288, 330,
        341, 328, 232, 207, 216, 247, 289, 339, 401, 396,
        /* Size 16x8 */
        64, 63, 64, 65, 72, 79, 94, 107, 122, 142, 159, 172, 186, 200, 216, 232,
        64, 65, 68, 71, 75, 80, 91, 100, 112, 128, 141, 152, 164, 177, 191, 207,
        72, 70, 73, 80, 96, 101, 112, 120, 131, 146, 158, 168, 178, 194, 204,
        216, 95, 88, 90, 94, 112, 121, 139, 150, 163, 178, 190, 200, 200, 216,
        239, 247, 131, 120, 118, 120, 136, 147, 168, 185, 201, 222, 236, 248,
        251, 238, 288, 289, 158, 145, 142, 142, 157, 168, 190, 207, 226, 250,
        267, 280, 311, 270, 330, 339, 189, 173, 166, 161, 171, 192, 215, 220,
        230, 266, 302, 312, 304, 328, 341, 401, 215, 199, 190, 180, 203, 217,
        235, 258, 268, 285, 290, 305, 311, 323, 328, 396,
        /* Size 16x32 */
        64, 63, 64, 65, 72, 88, 95, 107, 131, 146, 158, 177, 189, 201, 215, 229,
        63, 64, 65, 66, 71, 85, 91, 103, 124, 139, 151, 168, 179, 191, 203, 214,
        63, 65, 65, 66, 70, 82, 88, 99, 120, 134, 145, 162, 173, 186, 199, 216,
        63, 65, 66, 67, 70, 82, 88, 99, 119, 132, 143, 160, 171, 182, 194, 201,
        64, 65, 68, 69, 73, 84, 90, 100, 118, 131, 142, 157, 166, 178, 190, 206,
        64, 66, 70, 72, 77, 85, 90, 99, 116, 128, 138, 153, 164, 177, 188, 197,
        65, 66, 71, 73, 80, 89, 94, 103, 120, 132, 142, 154, 161, 168, 180, 200,
        68, 68, 72, 76, 85, 96, 101, 109, 126, 138, 147, 162, 174, 173, 181,
        185, 72, 69, 75, 80, 96, 108, 112, 120, 136, 148, 157, 171, 171, 187,
        203, 192, 76, 73, 78, 83, 99, 112, 117, 126, 142, 154, 163, 174, 184,
        192, 191, 227, 79, 75, 80, 85, 101, 116, 121, 130, 147, 159, 168, 181,
        192, 190, 217, 201, 88, 82, 85, 90, 107, 126, 132, 143, 159, 171, 180,
        196, 190, 213, 198, 239, 94, 88, 91, 95, 112, 132, 139, 150, 168, 181,
        190, 199, 215, 199, 235, 209, 98, 92, 94, 97, 114, 135, 142, 155, 173,
        186, 195, 206, 209, 235, 209, 259, 107, 99, 100, 103, 120, 142, 150,
        164, 185, 198, 207, 225, 220, 217, 258, 219, 117, 108, 108, 111, 127,
        150, 159, 174, 196, 210, 220, 229, 244, 240, 224, 282, 122, 113, 112,
        115, 131, 154, 163, 178, 201, 215, 226, 235, 230, 250, 268, 229, 130,
        120, 118, 121, 136, 159, 168, 185, 210, 225, 236, 256, 258, 247, 249,
        304, 142, 131, 128, 131, 146, 169, 178, 195, 222, 238, 250, 262, 266,
        284, 285, 242, 152, 139, 136, 138, 153, 176, 185, 202, 230, 247, 260,
        266, 264, 253, 275, 340, 159, 145, 141, 143, 158, 181, 190, 208, 236,
        254, 267, 290, 302, 316, 290, 257, 165, 151, 146, 148, 162, 185, 195,
        212, 242, 260, 273, 291, 290, 286, 313, 354, 172, 157, 152, 154, 168,
        190, 200, 218, 248, 266, 280, 292, 312, 314, 305, 273, 179, 164, 158,
        158, 173, 193, 199, 223, 253, 260, 287, 306, 312, 335, 344, 400, 186,
        170, 164, 162, 178, 195, 200, 239, 251, 259, 311, 302, 304, 322, 311,
        292, 193, 176, 171, 167, 190, 197, 205, 241, 233, 279, 305, 300, 333,
        336, 405, 415, 200, 182, 177, 173, 194, 189, 216, 241, 238, 295, 270,
        340, 328, 316, 323, 311, 208, 189, 184, 179, 199, 194, 236, 220, 256,
        292, 276, 353, 313, 362, 362, 481, 216, 196, 191, 185, 204, 200, 239,
        221, 288, 249, 330, 316, 341, 361, 328, 333, 224, 203, 199, 191, 210,
        206, 243, 226, 288, 254, 343, 288, 410, 328, 395, 389, 232, 211, 207,
        198, 216, 212, 247, 232, 289, 260, 339, 295, 401, 335, 396, 342, 241,
        215, 219, 200, 204, 246, 218, 287, 237, 344, 265, 405, 301, 476, 342,
        430,
        /* Size 32x16 */
        64, 63, 63, 63, 64, 64, 65, 68, 72, 76, 79, 88, 94, 98, 107, 117, 122,
        130, 142, 152, 159, 165, 172, 179, 186, 193, 200, 208, 216, 224, 232,
        241, 63, 64, 65, 65, 65, 66, 66, 68, 69, 73, 75, 82, 88, 92, 99, 108,
        113, 120, 131, 139, 145, 151, 157, 164, 170, 176, 182, 189, 196, 203,
        211, 215, 64, 65, 65, 66, 68, 70, 71, 72, 75, 78, 80, 85, 91, 94, 100,
        108, 112, 118, 128, 136, 141, 146, 152, 158, 164, 171, 177, 184, 191,
        199, 207, 219, 65, 66, 66, 67, 69, 72, 73, 76, 80, 83, 85, 90, 95, 97,
        103, 111, 115, 121, 131, 138, 143, 148, 154, 158, 162, 167, 173, 179,
        185, 191, 198, 200, 72, 71, 70, 70, 73, 77, 80, 85, 96, 99, 101, 107,
        112, 114, 120, 127, 131, 136, 146, 153, 158, 162, 168, 173, 178, 190,
        194, 199, 204, 210, 216, 204, 88, 85, 82, 82, 84, 85, 89, 96, 108, 112,
        116, 126, 132, 135, 142, 150, 154, 159, 169, 176, 181, 185, 190, 193,
        195, 197, 189, 194, 200, 206, 212, 246, 95, 91, 88, 88, 90, 90, 94, 101,
        112, 117, 121, 132, 139, 142, 150, 159, 163, 168, 178, 185, 190, 195,
        200, 199, 200, 205, 216, 236, 239, 243, 247, 218, 107, 103, 99, 99, 100,
        99, 103, 109, 120, 126, 130, 143, 150, 155, 164, 174, 178, 185, 195,
        202, 208, 212, 218, 223, 239, 241, 241, 220, 221, 226, 232, 287, 131,
        124, 120, 119, 118, 116, 120, 126, 136, 142, 147, 159, 168, 173, 185,
        196, 201, 210, 222, 230, 236, 242, 248, 253, 251, 233, 238, 256, 288,
        288, 289, 237, 146, 139, 134, 132, 131, 128, 132, 138, 148, 154, 159,
        171, 181, 186, 198, 210, 215, 225, 238, 247, 254, 260, 266, 260, 259,
        279, 295, 292, 249, 254, 260, 344, 158, 151, 145, 143, 142, 138, 142,
        147, 157, 163, 168, 180, 190, 195, 207, 220, 226, 236, 250, 260, 267,
        273, 280, 287, 311, 305, 270, 276, 330, 343, 339, 265, 177, 168, 162,
        160, 157, 153, 154, 162, 171, 174, 181, 196, 199, 206, 225, 229, 235,
        256, 262, 266, 290, 291, 292, 306, 302, 300, 340, 353, 316, 288, 295,
        405, 189, 179, 173, 171, 166, 164, 161, 174, 171, 184, 192, 190, 215,
        209, 220, 244, 230, 258, 266, 264, 302, 290, 312, 312, 304, 333, 328,
        313, 341, 410, 401, 301, 201, 191, 186, 182, 178, 177, 168, 173, 187,
        192, 190, 213, 199, 235, 217, 240, 250, 247, 284, 253, 316, 286, 314,
        335, 322, 336, 316, 362, 361, 328, 335, 476, 215, 203, 199, 194, 190,
        188, 180, 181, 203, 191, 217, 198, 235, 209, 258, 224, 268, 249, 285,
        275, 290, 313, 305, 344, 311, 405, 323, 362, 328, 395, 396, 342, 229,
        214, 216, 201, 206, 197, 200, 185, 192, 227, 201, 239, 209, 259, 219,
        282, 229, 304, 242, 340, 257, 354, 273, 400, 292, 415, 311, 481, 333,
        389, 342, 430,
        /* Size 4x16 */
        63, 88, 146, 201, 65, 82, 134, 186, 65, 84, 131, 178, 66, 89, 132, 168,
        69, 108, 148, 187, 75, 116, 159, 190, 88, 132, 181, 199, 99, 142, 198,
        217, 113, 154, 215, 250, 131, 169, 238, 284, 145, 181, 254, 316, 157,
        190, 266, 314, 170, 195, 259, 322, 182, 189, 295, 316, 196, 200, 249,
        361, 211, 212, 260, 335,
        /* Size 16x4 */
        63, 65, 65, 66, 69, 75, 88, 99, 113, 131, 145, 157, 170, 182, 196, 211,
        88, 82, 84, 89, 108, 116, 132, 142, 154, 169, 181, 190, 195, 189, 200,
        212, 146, 134, 131, 132, 148, 159, 181, 198, 215, 238, 254, 266, 259,
        295, 249, 260, 201, 186, 178, 168, 187, 190, 199, 217, 250, 284, 316,
        314, 322, 316, 361, 335,
        /* Size 8x32 */
        64, 64, 72, 95, 131, 158, 189, 215, 63, 65, 71, 91, 124, 151, 179, 203,
        63, 65, 70, 88, 120, 145, 173, 199, 63, 66, 70, 88, 119, 143, 171, 194,
        64, 68, 73, 90, 118, 142, 166, 190, 64, 70, 77, 90, 116, 138, 164, 188,
        65, 71, 80, 94, 120, 142, 161, 180, 68, 72, 85, 101, 126, 147, 174, 181,
        72, 75, 96, 112, 136, 157, 171, 203, 76, 78, 99, 117, 142, 163, 184,
        191, 79, 80, 101, 121, 147, 168, 192, 217, 88, 85, 107, 132, 159, 180,
        190, 198, 94, 91, 112, 139, 168, 190, 215, 235, 98, 94, 114, 142, 173,
        195, 209, 209, 107, 100, 120, 150, 185, 207, 220, 258, 117, 108, 127,
        159, 196, 220, 244, 224, 122, 112, 131, 163, 201, 226, 230, 268, 130,
        118, 136, 168, 210, 236, 258, 249, 142, 128, 146, 178, 222, 250, 266,
        285, 152, 136, 153, 185, 230, 260, 264, 275, 159, 141, 158, 190, 236,
        267, 302, 290, 165, 146, 162, 195, 242, 273, 290, 313, 172, 152, 168,
        200, 248, 280, 312, 305, 179, 158, 173, 199, 253, 287, 312, 344, 186,
        164, 178, 200, 251, 311, 304, 311, 193, 171, 190, 205, 233, 305, 333,
        405, 200, 177, 194, 216, 238, 270, 328, 323, 208, 184, 199, 236, 256,
        276, 313, 362, 216, 191, 204, 239, 288, 330, 341, 328, 224, 199, 210,
        243, 288, 343, 410, 395, 232, 207, 216, 247, 289, 339, 401, 396, 241,
        219, 204, 218, 237, 265, 301, 342,
        /* Size 32x8 */
        64, 63, 63, 63, 64, 64, 65, 68, 72, 76, 79, 88, 94, 98, 107, 117, 122,
        130, 142, 152, 159, 165, 172, 179, 186, 193, 200, 208, 216, 224, 232,
        241, 64, 65, 65, 66, 68, 70, 71, 72, 75, 78, 80, 85, 91, 94, 100, 108,
        112, 118, 128, 136, 141, 146, 152, 158, 164, 171, 177, 184, 191, 199,
        207, 219, 72, 71, 70, 70, 73, 77, 80, 85, 96, 99, 101, 107, 112, 114,
        120, 127, 131, 136, 146, 153, 158, 162, 168, 173, 178, 190, 194, 199,
        204, 210, 216, 204, 95, 91, 88, 88, 90, 90, 94, 101, 112, 117, 121, 132,
        139, 142, 150, 159, 163, 168, 178, 185, 190, 195, 200, 199, 200, 205,
        216, 236, 239, 243, 247, 218, 131, 124, 120, 119, 118, 116, 120, 126,
        136, 142, 147, 159, 168, 173, 185, 196, 201, 210, 222, 230, 236, 242,
        248, 253, 251, 233, 238, 256, 288, 288, 289, 237, 158, 151, 145, 143,
        142, 138, 142, 147, 157, 163, 168, 180, 190, 195, 207, 220, 226, 236,
        250, 260, 267, 273, 280, 287, 311, 305, 270, 276, 330, 343, 339, 265,
        189, 179, 173, 171, 166, 164, 161, 174, 171, 184, 192, 190, 215, 209,
        220, 244, 230, 258, 266, 264, 302, 290, 312, 312, 304, 333, 328, 313,
        341, 410, 401, 301, 215, 203, 199, 194, 190, 188, 180, 181, 203, 191,
        217, 198, 235, 209, 258, 224, 268, 249, 285, 275, 290, 313, 305, 344,
        311, 405, 323, 362, 328, 395, 396, 342 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        97, 122, 122, 196,
#endif
        /* Size 4x4 */
        66, 90, 112, 133, 90, 117, 133, 148, 112, 133, 172, 169, 133, 148, 169,
        220,
        /* Size 8x8 */
        63, 77, 95, 97, 109, 123, 134, 145, 77, 95, 94, 93, 100, 111, 122, 134,
        95, 94, 107, 111, 117, 127, 133, 134, 97, 93, 111, 125, 135, 145, 146,
        144, 109, 100, 117, 135, 153, 166, 175, 166, 123, 111, 127, 145, 166,
        183, 183, 201, 134, 122, 133, 146, 175, 183, 207, 240, 145, 134, 134,
        144, 166, 201, 240, 233,
        /* Size 16x16 */
        64, 61, 67, 77, 98, 97, 101, 104, 111, 120, 127, 133, 139, 144, 150,
        157, 61, 63, 71, 82, 93, 92, 93, 96, 102, 110, 116, 121, 127, 133, 139,
        145, 67, 71, 78, 88, 95, 93, 93, 95, 100, 107, 113, 117, 122, 126, 132,
        138, 77, 82, 88, 95, 98, 96, 95, 96, 101, 107, 111, 116, 118, 121, 124,
        130, 98, 93, 95, 98, 106, 106, 108, 109, 112, 117, 121, 125, 128, 127,
        139, 143, 97, 92, 93, 96, 106, 109, 113, 115, 118, 123, 127, 131, 138,
        132, 144, 149, 101, 93, 93, 95, 108, 113, 122, 126, 131, 136, 140, 144,
        141, 147, 145, 158, 104, 96, 95, 96, 109, 115, 126, 133, 139, 145, 150,
        153, 150, 161, 148, 169, 111, 102, 100, 101, 112, 118, 131, 139, 146,
        154, 159, 163, 168, 173, 155, 179, 120, 110, 107, 107, 117, 123, 136,
        145, 154, 164, 170, 175, 177, 170, 163, 195, 127, 116, 113, 111, 121,
        127, 140, 150, 159, 170, 178, 183, 180, 176, 179, 202, 133, 121, 117,
        116, 125, 131, 144, 153, 163, 175, 183, 189, 192, 193, 197, 221, 139,
        127, 122, 118, 128, 138, 141, 150, 168, 177, 180, 192, 201, 203, 224,
        228, 144, 133, 126, 121, 127, 132, 147, 161, 173, 170, 176, 193, 203,
        213, 214, 253, 150, 139, 132, 124, 139, 144, 145, 148, 155, 163, 179,
        197, 224, 214, 226, 225, 157, 145, 138, 130, 143, 149, 158, 169, 179,
        195, 202, 221, 228, 253, 225, 240,
        /* Size 32x32 */
        64, 62, 61, 62, 67, 73, 77, 83, 98, 98, 97, 98, 101, 102, 104, 109, 111,
        114, 120, 124, 127, 130, 133, 136, 139, 142, 144, 147, 150, 153, 157,
        160, 62, 63, 63, 64, 69, 77, 80, 85, 95, 95, 94, 94, 96, 97, 100, 104,
        106, 109, 114, 118, 121, 123, 126, 129, 131, 134, 137, 139, 142, 145,
        148, 149, 61, 63, 63, 65, 71, 79, 82, 85, 93, 92, 92, 91, 93, 94, 96,
        100, 102, 105, 110, 114, 116, 118, 121, 124, 127, 130, 133, 136, 139,
        142, 145, 151, 62, 64, 65, 66, 72, 81, 83, 86, 93, 92, 91, 90, 92, 93,
        95, 99, 101, 103, 109, 112, 114, 117, 119, 123, 125, 128, 130, 133, 135,
        138, 141, 140, 67, 69, 71, 72, 78, 87, 88, 90, 95, 93, 93, 91, 93, 94,
        95, 99, 100, 102, 107, 110, 113, 115, 117, 120, 122, 123, 126, 129, 132,
        135, 138, 144, 73, 77, 79, 81, 87, 95, 95, 95, 96, 94, 93, 91, 92, 93,
        94, 97, 98, 100, 104, 107, 109, 111, 113, 116, 119, 122, 127, 129, 131,
        134, 136, 136, 77, 80, 82, 83, 88, 95, 95, 96, 98, 96, 96, 94, 95, 95,
        96, 99, 101, 102, 107, 109, 111, 113, 116, 116, 118, 120, 121, 121, 124,
        127, 130, 139, 83, 85, 85, 86, 90, 95, 96, 97, 101, 100, 99, 98, 100,
        100, 101, 104, 105, 107, 111, 113, 115, 117, 119, 122, 126, 127, 129,
        125, 126, 128, 131, 127, 98, 95, 93, 93, 95, 96, 98, 101, 106, 106, 106,
        107, 108, 108, 109, 111, 112, 113, 117, 119, 121, 123, 125, 127, 128,
        125, 127, 132, 139, 141, 143, 133, 98, 95, 92, 92, 93, 94, 96, 100, 106,
        107, 108, 110, 111, 111, 112, 115, 116, 117, 121, 123, 125, 127, 129,
        128, 129, 133, 137, 138, 131, 134, 136, 152, 97, 94, 92, 91, 93, 93, 96,
        99, 106, 108, 109, 111, 113, 113, 115, 117, 118, 120, 123, 126, 127,
        129, 131, 133, 138, 138, 132, 134, 144, 148, 149, 138, 98, 94, 91, 90,
        91, 91, 94, 98, 107, 110, 111, 117, 119, 120, 123, 125, 126, 127, 131,
        133, 135, 136, 138, 141, 138, 136, 144, 148, 142, 138, 140, 158, 101,
        96, 93, 92, 93, 92, 95, 100, 108, 111, 113, 119, 122, 123, 126, 129,
        131, 133, 136, 138, 140, 142, 144, 142, 141, 150, 147, 140, 145, 157,
        158, 142, 102, 97, 94, 93, 94, 93, 95, 100, 108, 111, 113, 120, 123,
        125, 129, 132, 133, 135, 139, 141, 143, 145, 147, 147, 152, 146, 145,
        159, 158, 144, 145, 168, 104, 100, 96, 95, 95, 94, 96, 101, 109, 112,
        115, 123, 126, 129, 133, 137, 139, 141, 145, 148, 150, 151, 153, 157,
        150, 152, 161, 149, 148, 169, 169, 147, 109, 104, 100, 99, 99, 97, 99,
        104, 111, 115, 117, 125, 129, 132, 137, 142, 144, 147, 151, 154, 156,
        158, 160, 157, 159, 165, 152, 162, 170, 151, 151, 179, 111, 106, 102,
        101, 100, 98, 101, 105, 112, 116, 118, 126, 131, 133, 139, 144, 146,
        149, 154, 157, 159, 161, 163, 162, 168, 158, 173, 169, 155, 174, 179,
        153, 114, 109, 105, 103, 102, 100, 102, 107, 113, 117, 120, 127, 133,
        135, 141, 147, 149, 153, 158, 161, 164, 166, 168, 173, 164, 175, 168,
        162, 185, 166, 158, 190, 120, 114, 110, 109, 107, 104, 107, 111, 117,
        121, 123, 131, 136, 139, 145, 151, 154, 158, 164, 168, 170, 172, 175,
        173, 177, 175, 170, 184, 163, 180, 195, 160, 124, 118, 114, 112, 110,
        107, 109, 113, 119, 123, 126, 133, 138, 141, 148, 154, 157, 161, 168,
        172, 175, 177, 180, 178, 183, 175, 190, 169, 192, 181, 166, 207, 127,
        121, 116, 114, 113, 109, 111, 115, 121, 125, 127, 135, 140, 143, 150,
        156, 159, 164, 170, 175, 178, 180, 183, 190, 180, 192, 176, 201, 179,
        182, 202, 168, 130, 123, 118, 117, 115, 111, 113, 117, 123, 127, 129,
        136, 142, 145, 151, 158, 161, 166, 172, 177, 180, 183, 186, 188, 197,
        183, 200, 183, 197, 201, 175, 213, 133, 126, 121, 119, 117, 113, 116,
        119, 125, 129, 131, 138, 144, 147, 153, 160, 163, 168, 175, 180, 183,
        186, 189, 189, 192, 200, 193, 199, 197, 187, 221, 177, 136, 129, 124,
        123, 120, 116, 116, 122, 127, 128, 133, 141, 142, 147, 157, 157, 162,
        173, 173, 178, 190, 188, 189, 196, 192, 197, 202, 207, 196, 216, 185,
        234, 139, 131, 127, 125, 122, 119, 118, 126, 128, 129, 138, 138, 141,
        152, 150, 159, 168, 164, 177, 183, 180, 197, 192, 192, 201, 194, 203,
        201, 224, 191, 228, 187, 142, 134, 130, 128, 123, 122, 120, 127, 125,
        133, 138, 136, 150, 146, 152, 165, 158, 175, 175, 175, 192, 183, 200,
        197, 194, 207, 197, 208, 199, 240, 194, 239, 144, 137, 133, 130, 126,
        127, 121, 129, 127, 137, 132, 144, 147, 145, 161, 152, 173, 168, 170,
        190, 176, 200, 193, 202, 203, 197, 213, 200, 214, 202, 253, 196, 147,
        139, 136, 133, 129, 129, 121, 125, 132, 138, 134, 148, 140, 159, 149,
        162, 169, 162, 184, 169, 201, 183, 199, 207, 201, 208, 200, 220, 203,
        219, 205, 266, 150, 142, 139, 135, 132, 131, 124, 126, 139, 131, 144,
        142, 145, 158, 148, 170, 155, 185, 163, 192, 179, 197, 197, 196, 224,
        199, 214, 203, 226, 206, 225, 208, 153, 145, 142, 138, 135, 134, 127,
        128, 141, 134, 148, 138, 157, 144, 169, 151, 174, 166, 180, 181, 182,
        201, 187, 216, 191, 240, 202, 219, 206, 233, 209, 230, 157, 148, 145,
        141, 138, 136, 130, 131, 143, 136, 149, 140, 158, 145, 169, 151, 179,
        158, 195, 166, 202, 175, 221, 185, 228, 194, 253, 205, 225, 209, 240,
        212, 160, 149, 151, 140, 144, 136, 139, 127, 133, 152, 138, 158, 142,
        168, 147, 179, 153, 190, 160, 207, 168, 213, 177, 234, 187, 239, 196,
        266, 208, 230, 212, 247,
        /* Size 4x8 */
        63, 94, 115, 137, 80, 91, 105, 127, 92, 110, 122, 134, 94, 120, 140,
        158, 104, 128, 159, 164, 118, 136, 174, 183, 128, 138, 181, 206, 139,
        138, 165, 202,
        /* Size 8x4 */
        63, 80, 92, 94, 104, 118, 128, 139, 94, 91, 110, 120, 128, 136, 138,
        138, 115, 105, 122, 140, 159, 174, 181, 165, 137, 127, 134, 158, 164,
        183, 206, 202,
        /* Size 8x16 */
        64, 71, 96, 101, 115, 127, 140, 150, 61, 76, 92, 93, 105, 116, 128, 139,
        67, 83, 94, 93, 103, 112, 122, 132, 78, 92, 97, 95, 103, 111, 118, 125,
        98, 96, 106, 108, 114, 121, 124, 139, 97, 93, 106, 113, 120, 128, 136,
        146, 100, 93, 108, 122, 133, 140, 149, 156, 104, 95, 109, 127, 142, 150,
        152, 167, 111, 99, 112, 131, 149, 159, 157, 173, 120, 106, 117, 136,
        158, 170, 175, 181, 127, 111, 121, 140, 164, 178, 192, 184, 132, 116,
        125, 144, 168, 183, 196, 191, 138, 121, 129, 142, 168, 196, 193, 194,
        144, 126, 136, 148, 160, 177, 202, 200, 150, 132, 138, 157, 182, 202,
        207, 203, 156, 138, 142, 159, 180, 203, 229, 229,
        /* Size 16x8 */
        64, 61, 67, 78, 98, 97, 100, 104, 111, 120, 127, 132, 138, 144, 150,
        156, 71, 76, 83, 92, 96, 93, 93, 95, 99, 106, 111, 116, 121, 126, 132,
        138, 96, 92, 94, 97, 106, 106, 108, 109, 112, 117, 121, 125, 129, 136,
        138, 142, 101, 93, 93, 95, 108, 113, 122, 127, 131, 136, 140, 144, 142,
        148, 157, 159, 115, 105, 103, 103, 114, 120, 133, 142, 149, 158, 164,
        168, 168, 160, 182, 180, 127, 116, 112, 111, 121, 128, 140, 150, 159,
        170, 178, 183, 196, 177, 202, 203, 140, 128, 122, 118, 124, 136, 149,
        152, 157, 175, 192, 196, 193, 202, 207, 229, 150, 139, 132, 125, 139,
        146, 156, 167, 173, 181, 184, 191, 194, 200, 203, 229,
        /* Size 16x32 */
        64, 62, 71, 77, 96, 98, 101, 105, 115, 122, 127, 135, 140, 145, 150,
        155, 63, 63, 74, 80, 94, 94, 97, 100, 109, 115, 120, 128, 132, 137, 142,
        145, 61, 64, 76, 81, 92, 91, 93, 97, 105, 111, 116, 123, 128, 133, 139,
        147, 62, 66, 77, 83, 92, 91, 93, 96, 104, 110, 114, 121, 126, 130, 135,
        136, 67, 72, 83, 88, 94, 92, 93, 95, 103, 108, 112, 119, 122, 127, 132,
        140, 74, 80, 91, 95, 95, 91, 92, 94, 100, 105, 109, 115, 121, 127, 131,
        132, 78, 82, 92, 95, 97, 94, 95, 97, 103, 108, 111, 116, 118, 120, 125,
        135, 84, 86, 93, 96, 100, 99, 100, 101, 107, 112, 115, 121, 126, 123,
        126, 124, 98, 93, 96, 98, 106, 107, 108, 109, 114, 118, 121, 127, 124,
        132, 139, 130, 97, 92, 94, 97, 106, 110, 111, 113, 117, 122, 125, 128,
        132, 134, 131, 149, 97, 92, 93, 96, 106, 112, 113, 115, 120, 124, 128,
        133, 136, 134, 146, 135, 98, 91, 91, 94, 107, 117, 119, 123, 128, 132,
        135, 141, 135, 146, 136, 155, 100, 93, 93, 96, 108, 119, 122, 127, 133,
        137, 140, 143, 149, 139, 156, 139, 102, 94, 94, 96, 108, 120, 123, 129,
        136, 140, 143, 146, 146, 158, 142, 165, 104, 96, 95, 97, 109, 122, 127,
        133, 142, 146, 150, 156, 152, 149, 167, 145, 109, 100, 98, 100, 111,
        125, 130, 137, 147, 152, 156, 159, 164, 161, 150, 177, 111, 102, 99,
        101, 112, 126, 131, 139, 149, 155, 159, 161, 157, 166, 173, 151, 114,
        104, 101, 103, 113, 128, 133, 141, 153, 159, 164, 172, 171, 164, 163,
        187, 120, 109, 106, 107, 117, 131, 136, 145, 158, 165, 170, 175, 175,
        182, 181, 158, 124, 113, 109, 110, 120, 133, 139, 148, 162, 169, 175,
        177, 175, 168, 176, 204, 127, 115, 111, 112, 121, 135, 140, 150, 164,
        172, 178, 187, 192, 197, 184, 166, 129, 118, 113, 114, 123, 136, 142,
        151, 166, 174, 180, 188, 187, 183, 194, 210, 132, 121, 116, 116, 125,
        138, 144, 153, 168, 177, 183, 188, 196, 196, 191, 175, 135, 123, 118,
        117, 127, 138, 142, 155, 170, 173, 186, 194, 196, 205, 207, 230, 138,
        126, 121, 119, 129, 138, 142, 162, 168, 172, 196, 192, 193, 200, 194,
        184, 141, 128, 124, 121, 134, 138, 143, 162, 158, 181, 193, 191, 205,
        206, 232, 235, 144, 131, 126, 123, 136, 133, 148, 161, 160, 187, 177,
        207, 202, 198, 200, 193, 147, 133, 129, 125, 137, 135, 157, 149, 168,
        185, 179, 212, 196, 216, 216, 261, 150, 136, 132, 127, 138, 136, 157,
        148, 182, 163, 202, 196, 207, 215, 203, 204, 153, 139, 135, 130, 140,
        138, 158, 150, 181, 165, 206, 183, 234, 202, 229, 226, 156, 141, 138,
        132, 142, 140, 159, 151, 180, 167, 203, 185, 229, 205, 229, 208, 159,
        142, 144, 130, 134, 157, 142, 178, 153, 204, 169, 231, 187, 258, 207,
        242,
        /* Size 32x16 */
        64, 63, 61, 62, 67, 74, 78, 84, 98, 97, 97, 98, 100, 102, 104, 109, 111,
        114, 120, 124, 127, 129, 132, 135, 138, 141, 144, 147, 150, 153, 156,
        159, 62, 63, 64, 66, 72, 80, 82, 86, 93, 92, 92, 91, 93, 94, 96, 100,
        102, 104, 109, 113, 115, 118, 121, 123, 126, 128, 131, 133, 136, 139,
        141, 142, 71, 74, 76, 77, 83, 91, 92, 93, 96, 94, 93, 91, 93, 94, 95,
        98, 99, 101, 106, 109, 111, 113, 116, 118, 121, 124, 126, 129, 132, 135,
        138, 144, 77, 80, 81, 83, 88, 95, 95, 96, 98, 97, 96, 94, 96, 96, 97,
        100, 101, 103, 107, 110, 112, 114, 116, 117, 119, 121, 123, 125, 127,
        130, 132, 130, 96, 94, 92, 92, 94, 95, 97, 100, 106, 106, 106, 107, 108,
        108, 109, 111, 112, 113, 117, 120, 121, 123, 125, 127, 129, 134, 136,
        137, 138, 140, 142, 134, 98, 94, 91, 91, 92, 91, 94, 99, 107, 110, 112,
        117, 119, 120, 122, 125, 126, 128, 131, 133, 135, 136, 138, 138, 138,
        138, 133, 135, 136, 138, 140, 157, 101, 97, 93, 93, 93, 92, 95, 100,
        108, 111, 113, 119, 122, 123, 127, 130, 131, 133, 136, 139, 140, 142,
        144, 142, 142, 143, 148, 157, 157, 158, 159, 142, 105, 100, 97, 96, 95,
        94, 97, 101, 109, 113, 115, 123, 127, 129, 133, 137, 139, 141, 145, 148,
        150, 151, 153, 155, 162, 162, 161, 149, 148, 150, 151, 178, 115, 109,
        105, 104, 103, 100, 103, 107, 114, 117, 120, 128, 133, 136, 142, 147,
        149, 153, 158, 162, 164, 166, 168, 170, 168, 158, 160, 168, 182, 181,
        180, 153, 122, 115, 111, 110, 108, 105, 108, 112, 118, 122, 124, 132,
        137, 140, 146, 152, 155, 159, 165, 169, 172, 174, 177, 173, 172, 181,
        187, 185, 163, 165, 167, 204, 127, 120, 116, 114, 112, 109, 111, 115,
        121, 125, 128, 135, 140, 143, 150, 156, 159, 164, 170, 175, 178, 180,
        183, 186, 196, 193, 177, 179, 202, 206, 203, 169, 135, 128, 123, 121,
        119, 115, 116, 121, 127, 128, 133, 141, 143, 146, 156, 159, 161, 172,
        175, 177, 187, 188, 188, 194, 192, 191, 207, 212, 196, 183, 185, 231,
        140, 132, 128, 126, 122, 121, 118, 126, 124, 132, 136, 135, 149, 146,
        152, 164, 157, 171, 175, 175, 192, 187, 196, 196, 193, 205, 202, 196,
        207, 234, 229, 187, 145, 137, 133, 130, 127, 127, 120, 123, 132, 134,
        134, 146, 139, 158, 149, 161, 166, 164, 182, 168, 197, 183, 196, 205,
        200, 206, 198, 216, 215, 202, 205, 258, 150, 142, 139, 135, 132, 131,
        125, 126, 139, 131, 146, 136, 156, 142, 167, 150, 173, 163, 181, 176,
        184, 194, 191, 207, 194, 232, 200, 216, 203, 229, 229, 207, 155, 145,
        147, 136, 140, 132, 135, 124, 130, 149, 135, 155, 139, 165, 145, 177,
        151, 187, 158, 204, 166, 210, 175, 230, 184, 235, 193, 261, 204, 226,
        208, 242,
        /* Size 4x16 */
        62, 98, 122, 145, 64, 91, 111, 133, 72, 92, 108, 127, 82, 94, 108, 120,
        93, 107, 118, 132, 92, 112, 124, 134, 93, 119, 137, 139, 96, 122, 146,
        149, 102, 126, 155, 166, 109, 131, 165, 182, 115, 135, 172, 197, 121,
        138, 177, 196, 126, 138, 172, 200, 131, 133, 187, 198, 136, 136, 163,
        215, 141, 140, 167, 205,
        /* Size 16x4 */
        62, 64, 72, 82, 93, 92, 93, 96, 102, 109, 115, 121, 126, 131, 136, 141,
        98, 91, 92, 94, 107, 112, 119, 122, 126, 131, 135, 138, 138, 133, 136,
        140, 122, 111, 108, 108, 118, 124, 137, 146, 155, 165, 172, 177, 172,
        187, 163, 167, 145, 133, 127, 120, 132, 134, 139, 149, 166, 182, 197,
        196, 200, 198, 215, 205,
        /* Size 8x32 */
        64, 71, 96, 101, 115, 127, 140, 150, 63, 74, 94, 97, 109, 120, 132, 142,
        61, 76, 92, 93, 105, 116, 128, 139, 62, 77, 92, 93, 104, 114, 126, 135,
        67, 83, 94, 93, 103, 112, 122, 132, 74, 91, 95, 92, 100, 109, 121, 131,
        78, 92, 97, 95, 103, 111, 118, 125, 84, 93, 100, 100, 107, 115, 126,
        126, 98, 96, 106, 108, 114, 121, 124, 139, 97, 94, 106, 111, 117, 125,
        132, 131, 97, 93, 106, 113, 120, 128, 136, 146, 98, 91, 107, 119, 128,
        135, 135, 136, 100, 93, 108, 122, 133, 140, 149, 156, 102, 94, 108, 123,
        136, 143, 146, 142, 104, 95, 109, 127, 142, 150, 152, 167, 109, 98, 111,
        130, 147, 156, 164, 150, 111, 99, 112, 131, 149, 159, 157, 173, 114,
        101, 113, 133, 153, 164, 171, 163, 120, 106, 117, 136, 158, 170, 175,
        181, 124, 109, 120, 139, 162, 175, 175, 176, 127, 111, 121, 140, 164,
        178, 192, 184, 129, 113, 123, 142, 166, 180, 187, 194, 132, 116, 125,
        144, 168, 183, 196, 191, 135, 118, 127, 142, 170, 186, 196, 207, 138,
        121, 129, 142, 168, 196, 193, 194, 141, 124, 134, 143, 158, 193, 205,
        232, 144, 126, 136, 148, 160, 177, 202, 200, 147, 129, 137, 157, 168,
        179, 196, 216, 150, 132, 138, 157, 182, 202, 207, 203, 153, 135, 140,
        158, 181, 206, 234, 229, 156, 138, 142, 159, 180, 203, 229, 229, 159,
        144, 134, 142, 153, 169, 187, 207,
        /* Size 32x8 */
        64, 63, 61, 62, 67, 74, 78, 84, 98, 97, 97, 98, 100, 102, 104, 109, 111,
        114, 120, 124, 127, 129, 132, 135, 138, 141, 144, 147, 150, 153, 156,
        159, 71, 74, 76, 77, 83, 91, 92, 93, 96, 94, 93, 91, 93, 94, 95, 98, 99,
        101, 106, 109, 111, 113, 116, 118, 121, 124, 126, 129, 132, 135, 138,
        144, 96, 94, 92, 92, 94, 95, 97, 100, 106, 106, 106, 107, 108, 108, 109,
        111, 112, 113, 117, 120, 121, 123, 125, 127, 129, 134, 136, 137, 138,
        140, 142, 134, 101, 97, 93, 93, 93, 92, 95, 100, 108, 111, 113, 119,
        122, 123, 127, 130, 131, 133, 136, 139, 140, 142, 144, 142, 142, 143,
        148, 157, 157, 158, 159, 142, 115, 109, 105, 104, 103, 100, 103, 107,
        114, 117, 120, 128, 133, 136, 142, 147, 149, 153, 158, 162, 164, 166,
        168, 170, 168, 158, 160, 168, 182, 181, 180, 153, 127, 120, 116, 114,
        112, 109, 111, 115, 121, 125, 128, 135, 140, 143, 150, 156, 159, 164,
        170, 175, 178, 180, 183, 186, 196, 193, 177, 179, 202, 206, 203, 169,
        140, 132, 128, 126, 122, 121, 118, 126, 124, 132, 136, 135, 149, 146,
        152, 164, 157, 171, 175, 175, 192, 187, 196, 196, 193, 205, 202, 196,
        207, 234, 229, 187, 150, 142, 139, 135, 132, 131, 125, 126, 139, 131,
        146, 136, 156, 142, 167, 150, 173, 163, 181, 176, 184, 194, 191, 207,
        194, 232, 200, 216, 203, 229, 229, 207 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        79, 153, 153, 280,
#endif
        /* Size 4x4 */
        65, 77, 127, 176, 77, 113, 157, 200, 127, 157, 227, 248, 176, 200, 248,
        341,
        /* Size 8x8 */
        64, 65, 70, 92, 114, 152, 172, 198, 65, 68, 75, 91, 108, 141, 159, 183,
        70, 75, 97, 113, 128, 158, 176, 186, 92, 91, 113, 141, 161, 192, 199,
        204, 114, 108, 128, 161, 186, 222, 245, 242, 152, 141, 158, 192, 222,
        268, 272, 301, 172, 159, 176, 199, 245, 272, 314, 375, 198, 183, 186,
        204, 242, 301, 375, 370,
        /* Size 16x16 */
        64, 63, 63, 64, 69, 79, 88, 99, 116, 131, 143, 163, 176, 188, 202, 216,
        63, 64, 65, 65, 68, 76, 83, 93, 108, 121, 132, 151, 162, 174, 187, 201,
        63, 65, 67, 68, 72, 78, 84, 93, 106, 118, 129, 146, 156, 167, 179, 192,
        64, 65, 68, 71, 75, 81, 85, 93, 105, 116, 126, 142, 151, 161, 171, 184,
        69, 68, 72, 75, 85, 94, 100, 107, 119, 130, 140, 155, 164, 170, 188,
        197, 79, 76, 78, 81, 94, 108, 116, 124, 136, 147, 156, 171, 180, 179,
        197, 209, 88, 83, 84, 85, 100, 116, 127, 136, 149, 159, 169, 183, 192,
        199, 203, 224, 99, 93, 93, 93, 107, 124, 136, 146, 162, 174, 184, 199,
        206, 220, 210, 242, 116, 108, 106, 105, 119, 136, 149, 162, 180, 194,
        205, 221, 230, 242, 223, 260, 131, 121, 118, 116, 130, 147, 159, 174,
        194, 210, 222, 240, 251, 246, 238, 287, 143, 132, 129, 126, 140, 156,
        169, 184, 205, 222, 235, 254, 265, 260, 264, 303, 163, 151, 146, 142,
        155, 171, 183, 199, 221, 240, 254, 275, 287, 288, 296, 337, 176, 162,
        156, 151, 164, 180, 192, 206, 230, 251, 265, 287, 301, 308, 342, 356,
        188, 174, 167, 161, 170, 179, 199, 220, 242, 246, 260, 288, 308, 327,
        332, 402, 202, 187, 179, 171, 188, 197, 203, 210, 223, 238, 264, 296,
        342, 332, 355, 357, 216, 201, 192, 184, 197, 209, 224, 242, 260, 287,
        303, 337, 356, 402, 357, 386,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 64, 64, 68, 69, 72, 79, 82, 88, 96, 99, 108, 116,
        119, 131, 139, 143, 160, 163, 166, 176, 182, 188, 195, 202, 209, 216,
        223, 63, 64, 64, 64, 64, 65, 65, 68, 69, 70, 77, 79, 85, 92, 95, 103,
        111, 114, 125, 132, 136, 152, 155, 157, 167, 172, 178, 185, 191, 198,
        204, 210, 63, 64, 64, 64, 65, 65, 65, 67, 68, 69, 76, 78, 83, 90, 93,
        100, 108, 111, 121, 129, 132, 147, 151, 153, 162, 168, 174, 181, 187,
        194, 201, 211, 63, 64, 64, 65, 65, 65, 65, 67, 68, 69, 75, 77, 82, 88,
        91, 99, 106, 109, 119, 127, 130, 145, 148, 150, 159, 164, 170, 176, 182,
        188, 195, 197, 63, 64, 65, 65, 67, 67, 68, 71, 72, 73, 78, 80, 84, 90,
        93, 100, 106, 109, 118, 126, 129, 143, 146, 148, 156, 161, 167, 173,
        179, 185, 192, 201, 64, 65, 65, 65, 67, 68, 69, 72, 73, 75, 80, 81, 85,
        91, 93, 99, 106, 108, 117, 124, 127, 141, 144, 146, 154, 159, 166, 172,
        177, 183, 189, 193, 64, 65, 65, 65, 68, 69, 71, 74, 75, 77, 81, 82, 85,
        91, 93, 99, 105, 108, 116, 123, 126, 139, 142, 144, 151, 156, 161, 165,
        171, 177, 184, 196, 68, 68, 67, 67, 71, 72, 74, 79, 82, 85, 90, 92, 95,
        101, 103, 109, 115, 118, 126, 133, 136, 148, 151, 153, 161, 165, 168,
        167, 171, 177, 182, 183, 69, 69, 68, 68, 72, 73, 75, 82, 85, 90, 94, 96,
        100, 106, 107, 113, 119, 122, 130, 137, 140, 152, 155, 157, 164, 165,
        170, 178, 188, 193, 197, 188, 72, 70, 69, 69, 73, 75, 77, 85, 90, 97,
        101, 103, 108, 113, 115, 120, 126, 128, 136, 143, 146, 158, 160, 162,
        170, 176, 182, 185, 180, 186, 191, 215, 79, 77, 76, 75, 78, 80, 81, 90,
        94, 101, 108, 110, 116, 122, 124, 130, 136, 139, 147, 153, 156, 168,
        171, 173, 180, 182, 179, 184, 197, 205, 209, 197, 82, 79, 78, 77, 80,
        81, 82, 92, 96, 103, 110, 113, 119, 125, 127, 134, 140, 142, 150, 157,
        160, 172, 175, 176, 183, 185, 195, 200, 197, 194, 199, 225, 88, 85, 83,
        82, 84, 85, 85, 95, 100, 108, 116, 119, 127, 133, 136, 143, 149, 151,
        159, 166, 169, 181, 183, 185, 192, 200, 199, 194, 203, 221, 224, 204,
        96, 92, 90, 88, 90, 91, 91, 101, 106, 113, 122, 125, 133, 141, 143, 152,
        158, 161, 170, 176, 180, 192, 194, 196, 203, 199, 201, 219, 220, 204,
        209, 243, 99, 95, 93, 91, 93, 93, 93, 103, 107, 115, 124, 127, 136, 143,
        146, 155, 162, 164, 174, 180, 184, 196, 199, 201, 206, 211, 220, 209,
        210, 240, 242, 214, 108, 103, 100, 99, 100, 99, 99, 109, 113, 120, 130,
        134, 143, 152, 155, 165, 172, 175, 185, 192, 195, 208, 211, 213, 220,
        225, 214, 228, 239, 217, 220, 262, 116, 111, 108, 106, 106, 106, 105,
        115, 119, 126, 136, 140, 149, 158, 162, 172, 180, 183, 194, 201, 205,
        218, 221, 223, 230, 224, 242, 238, 223, 251, 260, 225, 119, 114, 111,
        109, 109, 108, 108, 118, 122, 128, 139, 142, 151, 161, 164, 175, 183,
        186, 198, 205, 209, 222, 225, 227, 234, 245, 239, 234, 265, 242, 233,
        282, 131, 125, 121, 119, 118, 117, 116, 126, 130, 136, 147, 150, 159,
        170, 174, 185, 194, 198, 210, 218, 222, 237, 240, 242, 251, 248, 246,
        264, 238, 264, 287, 238, 139, 132, 129, 127, 126, 124, 123, 133, 137,
        143, 153, 157, 166, 176, 180, 192, 201, 205, 218, 227, 231, 246, 250,
        252, 259, 256, 272, 248, 282, 267, 248, 312, 143, 136, 132, 130, 129,
        127, 126, 136, 140, 146, 156, 160, 169, 180, 184, 195, 205, 209, 222,
        231, 235, 251, 254, 256, 265, 277, 260, 294, 264, 271, 303, 253, 160,
        152, 147, 145, 143, 141, 139, 148, 152, 158, 168, 172, 181, 192, 196,
        208, 218, 222, 237, 246, 251, 268, 272, 274, 285, 272, 294, 271, 294,
        301, 264, 327, 163, 155, 151, 148, 146, 144, 142, 151, 155, 160, 171,
        175, 183, 194, 199, 211, 221, 225, 240, 250, 254, 272, 275, 278, 287,
        297, 288, 298, 296, 283, 337, 269, 166, 157, 153, 150, 148, 146, 144,
        153, 157, 162, 173, 176, 185, 196, 201, 213, 223, 227, 242, 252, 256,
        274, 278, 280, 290, 297, 305, 311, 298, 330, 282, 364, 176, 167, 162,
        159, 156, 154, 151, 161, 164, 170, 180, 183, 192, 203, 206, 220, 230,
        234, 251, 259, 265, 285, 287, 290, 301, 296, 308, 308, 342, 293, 356,
        287, 182, 172, 168, 164, 161, 159, 156, 165, 165, 176, 182, 185, 200,
        199, 211, 225, 224, 245, 248, 256, 277, 272, 297, 297, 296, 314, 303,
        320, 308, 375, 299, 380, 188, 178, 174, 170, 167, 166, 161, 168, 170,
        182, 179, 195, 199, 201, 220, 214, 242, 239, 246, 272, 260, 294, 288,
        305, 308, 303, 327, 309, 332, 314, 402, 305, 195, 185, 181, 176, 173,
        172, 165, 167, 178, 185, 184, 200, 194, 219, 209, 228, 238, 234, 264,
        248, 294, 271, 298, 311, 308, 320, 309, 341, 315, 344, 321, 433, 202,
        191, 187, 182, 179, 177, 171, 171, 188, 180, 197, 197, 203, 220, 210,
        239, 223, 265, 238, 282, 264, 294, 296, 298, 342, 308, 332, 315, 355,
        322, 357, 328, 209, 198, 194, 188, 185, 183, 177, 177, 193, 186, 205,
        194, 221, 204, 240, 217, 251, 242, 264, 267, 271, 301, 283, 330, 293,
        375, 314, 344, 322, 370, 329, 370, 216, 204, 201, 195, 192, 189, 184,
        182, 197, 191, 209, 199, 224, 209, 242, 220, 260, 233, 287, 248, 303,
        264, 337, 282, 356, 299, 402, 321, 357, 329, 386, 336, 223, 210, 211,
        197, 201, 193, 196, 183, 188, 215, 197, 225, 204, 243, 214, 262, 225,
        282, 238, 312, 253, 327, 269, 364, 287, 380, 305, 433, 328, 370, 336,
        402,
        /* Size 4x8 */
        64, 77, 124, 180, 65, 80, 117, 168, 69, 102, 136, 179, 89, 122, 170,
        216, 109, 139, 197, 235, 145, 168, 236, 271, 165, 182, 260, 315, 189,
        194, 243, 313,
        /* Size 8x4 */
        64, 65, 69, 89, 109, 145, 165, 189, 77, 80, 102, 122, 139, 168, 182,
        194, 124, 117, 136, 170, 197, 236, 260, 243, 180, 168, 179, 216, 235,
        271, 315, 313,
        /* Size 8x16 */
        64, 64, 72, 88, 116, 158, 179, 201, 63, 65, 70, 83, 108, 146, 165, 187,
        64, 67, 73, 84, 106, 142, 158, 179, 64, 69, 77, 85, 105, 138, 154, 171,
        69, 72, 89, 100, 119, 151, 163, 187, 79, 79, 101, 116, 136, 168, 179,
        200, 88, 85, 107, 126, 149, 180, 199, 216, 98, 93, 114, 135, 162, 195,
        209, 235, 115, 106, 126, 149, 180, 217, 222, 247, 130, 118, 136, 159,
        194, 236, 249, 263, 142, 129, 146, 169, 205, 250, 276, 271, 162, 145,
        161, 183, 221, 271, 292, 287, 175, 155, 170, 192, 230, 281, 294, 296,
        187, 167, 179, 204, 229, 260, 307, 310, 200, 179, 187, 218, 261, 300,
        320, 316, 215, 192, 197, 224, 261, 305, 357, 363,
        /* Size 16x8 */
        64, 63, 64, 64, 69, 79, 88, 98, 115, 130, 142, 162, 175, 187, 200, 215,
        64, 65, 67, 69, 72, 79, 85, 93, 106, 118, 129, 145, 155, 167, 179, 192,
        72, 70, 73, 77, 89, 101, 107, 114, 126, 136, 146, 161, 170, 179, 187,
        197, 88, 83, 84, 85, 100, 116, 126, 135, 149, 159, 169, 183, 192, 204,
        218, 224, 116, 108, 106, 105, 119, 136, 149, 162, 180, 194, 205, 221,
        230, 229, 261, 261, 158, 146, 142, 138, 151, 168, 180, 195, 217, 236,
        250, 271, 281, 260, 300, 305, 179, 165, 158, 154, 163, 179, 199, 209,
        222, 249, 276, 292, 294, 307, 320, 357, 201, 187, 179, 171, 187, 200,
        216, 235, 247, 263, 271, 287, 296, 310, 316, 363,
        /* Size 16x32 */
        64, 63, 64, 64, 72, 79, 88, 107, 116, 131, 158, 163, 179, 190, 201, 214,
        63, 64, 65, 65, 71, 77, 85, 103, 111, 124, 151, 155, 169, 180, 191, 201,
        63, 64, 65, 65, 70, 76, 83, 100, 108, 121, 146, 150, 165, 176, 187, 202,
        63, 65, 65, 66, 69, 75, 82, 99, 106, 119, 144, 148, 162, 171, 182, 189,
        64, 65, 67, 68, 73, 78, 84, 100, 106, 118, 142, 145, 158, 168, 179, 193,
        64, 65, 68, 69, 75, 80, 85, 99, 106, 117, 140, 143, 157, 168, 177, 185,
        64, 66, 69, 71, 77, 81, 85, 99, 105, 116, 138, 141, 154, 161, 171, 188,
        68, 68, 71, 74, 85, 90, 96, 109, 115, 126, 147, 151, 163, 164, 171, 176,
        69, 68, 72, 75, 89, 94, 100, 113, 119, 130, 151, 154, 163, 175, 187,
        181, 72, 69, 74, 77, 96, 102, 108, 120, 126, 136, 157, 160, 173, 179,
        180, 208, 79, 75, 79, 81, 101, 108, 116, 130, 136, 147, 168, 171, 179,
        181, 200, 190, 81, 77, 81, 83, 103, 110, 119, 134, 140, 150, 171, 174,
        183, 197, 189, 219, 88, 82, 85, 86, 107, 116, 126, 143, 149, 159, 180,
        183, 199, 191, 216, 198, 95, 89, 91, 92, 112, 122, 133, 151, 158, 170,
        191, 194, 199, 216, 199, 236, 98, 92, 93, 94, 114, 124, 135, 155, 162,
        173, 195, 198, 209, 207, 235, 208, 107, 99, 100, 100, 120, 130, 142,
        164, 172, 185, 207, 210, 224, 225, 213, 256, 115, 106, 106, 106, 126,
        136, 149, 172, 180, 194, 217, 221, 222, 233, 247, 219, 118, 109, 109,
        109, 128, 139, 151, 175, 183, 197, 222, 225, 240, 235, 235, 275, 130,
        120, 118, 117, 136, 147, 159, 185, 194, 210, 236, 239, 249, 260, 263,
        232, 138, 127, 125, 124, 143, 153, 166, 192, 201, 218, 245, 249, 254,
        245, 258, 304, 142, 131, 129, 127, 146, 156, 169, 195, 205, 222, 250,
        254, 276, 287, 271, 247, 159, 145, 142, 140, 158, 168, 181, 208, 218,
        236, 267, 271, 275, 271, 288, 318, 162, 148, 145, 143, 161, 171, 183,
        210, 221, 240, 271, 275, 292, 292, 287, 263, 165, 151, 147, 145, 162,
        173, 185, 212, 223, 242, 273, 278, 295, 309, 315, 355, 175, 159, 155,
        153, 170, 180, 192, 220, 230, 250, 281, 286, 294, 305, 296, 281, 181,
        165, 161, 158, 176, 182, 197, 221, 224, 260, 277, 288, 309, 315, 361,
        371, 187, 171, 167, 162, 179, 179, 204, 221, 229, 267, 260, 311, 307,
        304, 310, 298, 194, 176, 173, 168, 183, 184, 216, 209, 241, 265, 265,
        317, 300, 335, 338, 421, 200, 182, 179, 173, 187, 189, 218, 210, 261,
        238, 300, 293, 320, 334, 316, 320, 207, 189, 186, 178, 192, 194, 221,
        215, 260, 243, 307, 276, 363, 313, 362, 361, 215, 195, 192, 184, 197,
        199, 224, 220, 261, 248, 305, 281, 357, 319, 363, 328, 222, 199, 202,
        186, 190, 224, 205, 260, 225, 307, 253, 359, 287, 418, 326, 392,
        /* Size 32x16 */
        64, 63, 63, 63, 64, 64, 64, 68, 69, 72, 79, 81, 88, 95, 98, 107, 115,
        118, 130, 138, 142, 159, 162, 165, 175, 181, 187, 194, 200, 207, 215,
        222, 63, 64, 64, 65, 65, 65, 66, 68, 68, 69, 75, 77, 82, 89, 92, 99,
        106, 109, 120, 127, 131, 145, 148, 151, 159, 165, 171, 176, 182, 189,
        195, 199, 64, 65, 65, 65, 67, 68, 69, 71, 72, 74, 79, 81, 85, 91, 93,
        100, 106, 109, 118, 125, 129, 142, 145, 147, 155, 161, 167, 173, 179,
        186, 192, 202, 64, 65, 65, 66, 68, 69, 71, 74, 75, 77, 81, 83, 86, 92,
        94, 100, 106, 109, 117, 124, 127, 140, 143, 145, 153, 158, 162, 168,
        173, 178, 184, 186, 72, 71, 70, 69, 73, 75, 77, 85, 89, 96, 101, 103,
        107, 112, 114, 120, 126, 128, 136, 143, 146, 158, 161, 162, 170, 176,
        179, 183, 187, 192, 197, 190, 79, 77, 76, 75, 78, 80, 81, 90, 94, 102,
        108, 110, 116, 122, 124, 130, 136, 139, 147, 153, 156, 168, 171, 173,
        180, 182, 179, 184, 189, 194, 199, 224, 88, 85, 83, 82, 84, 85, 85, 96,
        100, 108, 116, 119, 126, 133, 135, 142, 149, 151, 159, 166, 169, 181,
        183, 185, 192, 197, 204, 216, 218, 221, 224, 205, 107, 103, 100, 99,
        100, 99, 99, 109, 113, 120, 130, 134, 143, 151, 155, 164, 172, 175, 185,
        192, 195, 208, 210, 212, 220, 221, 221, 209, 210, 215, 220, 260, 116,
        111, 108, 106, 106, 106, 105, 115, 119, 126, 136, 140, 149, 158, 162,
        172, 180, 183, 194, 201, 205, 218, 221, 223, 230, 224, 229, 241, 261,
        260, 261, 225, 131, 124, 121, 119, 118, 117, 116, 126, 130, 136, 147,
        150, 159, 170, 173, 185, 194, 197, 210, 218, 222, 236, 240, 242, 250,
        260, 267, 265, 238, 243, 248, 307, 158, 151, 146, 144, 142, 140, 138,
        147, 151, 157, 168, 171, 180, 191, 195, 207, 217, 222, 236, 245, 250,
        267, 271, 273, 281, 277, 260, 265, 300, 307, 305, 253, 163, 155, 150,
        148, 145, 143, 141, 151, 154, 160, 171, 174, 183, 194, 198, 210, 221,
        225, 239, 249, 254, 271, 275, 278, 286, 288, 311, 317, 293, 276, 281,
        359, 179, 169, 165, 162, 158, 157, 154, 163, 163, 173, 179, 183, 199,
        199, 209, 224, 222, 240, 249, 254, 276, 275, 292, 295, 294, 309, 307,
        300, 320, 363, 357, 287, 190, 180, 176, 171, 168, 168, 161, 164, 175,
        179, 181, 197, 191, 216, 207, 225, 233, 235, 260, 245, 287, 271, 292,
        309, 305, 315, 304, 335, 334, 313, 319, 418, 201, 191, 187, 182, 179,
        177, 171, 171, 187, 180, 200, 189, 216, 199, 235, 213, 247, 235, 263,
        258, 271, 288, 287, 315, 296, 361, 310, 338, 316, 362, 363, 326, 214,
        201, 202, 189, 193, 185, 188, 176, 181, 208, 190, 219, 198, 236, 208,
        256, 219, 275, 232, 304, 247, 318, 263, 355, 281, 371, 298, 421, 320,
        361, 328, 392,
        /* Size 4x16 */
        63, 79, 131, 190, 64, 76, 121, 176, 65, 78, 118, 168, 66, 81, 116, 161,
        68, 94, 130, 175, 75, 108, 147, 181, 82, 116, 159, 191, 92, 124, 173,
        207, 106, 136, 194, 233, 120, 147, 210, 260, 131, 156, 222, 287, 148,
        171, 240, 292, 159, 180, 250, 305, 171, 179, 267, 304, 182, 189, 238,
        334, 195, 199, 248, 319,
        /* Size 16x4 */
        63, 64, 65, 66, 68, 75, 82, 92, 106, 120, 131, 148, 159, 171, 182, 195,
        79, 76, 78, 81, 94, 108, 116, 124, 136, 147, 156, 171, 180, 179, 189,
        199, 131, 121, 118, 116, 130, 147, 159, 173, 194, 210, 222, 240, 250,
        267, 238, 248, 190, 176, 168, 161, 175, 181, 191, 207, 233, 260, 287,
        292, 305, 304, 334, 319,
        /* Size 8x32 */
        64, 64, 72, 88, 116, 158, 179, 201, 63, 65, 71, 85, 111, 151, 169, 191,
        63, 65, 70, 83, 108, 146, 165, 187, 63, 65, 69, 82, 106, 144, 162, 182,
        64, 67, 73, 84, 106, 142, 158, 179, 64, 68, 75, 85, 106, 140, 157, 177,
        64, 69, 77, 85, 105, 138, 154, 171, 68, 71, 85, 96, 115, 147, 163, 171,
        69, 72, 89, 100, 119, 151, 163, 187, 72, 74, 96, 108, 126, 157, 173,
        180, 79, 79, 101, 116, 136, 168, 179, 200, 81, 81, 103, 119, 140, 171,
        183, 189, 88, 85, 107, 126, 149, 180, 199, 216, 95, 91, 112, 133, 158,
        191, 199, 199, 98, 93, 114, 135, 162, 195, 209, 235, 107, 100, 120, 142,
        172, 207, 224, 213, 115, 106, 126, 149, 180, 217, 222, 247, 118, 109,
        128, 151, 183, 222, 240, 235, 130, 118, 136, 159, 194, 236, 249, 263,
        138, 125, 143, 166, 201, 245, 254, 258, 142, 129, 146, 169, 205, 250,
        276, 271, 159, 142, 158, 181, 218, 267, 275, 288, 162, 145, 161, 183,
        221, 271, 292, 287, 165, 147, 162, 185, 223, 273, 295, 315, 175, 155,
        170, 192, 230, 281, 294, 296, 181, 161, 176, 197, 224, 277, 309, 361,
        187, 167, 179, 204, 229, 260, 307, 310, 194, 173, 183, 216, 241, 265,
        300, 338, 200, 179, 187, 218, 261, 300, 320, 316, 207, 186, 192, 221,
        260, 307, 363, 362, 215, 192, 197, 224, 261, 305, 357, 363, 222, 202,
        190, 205, 225, 253, 287, 326,
        /* Size 32x8 */
        64, 63, 63, 63, 64, 64, 64, 68, 69, 72, 79, 81, 88, 95, 98, 107, 115,
        118, 130, 138, 142, 159, 162, 165, 175, 181, 187, 194, 200, 207, 215,
        222, 64, 65, 65, 65, 67, 68, 69, 71, 72, 74, 79, 81, 85, 91, 93, 100,
        106, 109, 118, 125, 129, 142, 145, 147, 155, 161, 167, 173, 179, 186,
        192, 202, 72, 71, 70, 69, 73, 75, 77, 85, 89, 96, 101, 103, 107, 112,
        114, 120, 126, 128, 136, 143, 146, 158, 161, 162, 170, 176, 179, 183,
        187, 192, 197, 190, 88, 85, 83, 82, 84, 85, 85, 96, 100, 108, 116, 119,
        126, 133, 135, 142, 149, 151, 159, 166, 169, 181, 183, 185, 192, 197,
        204, 216, 218, 221, 224, 205, 116, 111, 108, 106, 106, 106, 105, 115,
        119, 126, 136, 140, 149, 158, 162, 172, 180, 183, 194, 201, 205, 218,
        221, 223, 230, 224, 229, 241, 261, 260, 261, 225, 158, 151, 146, 144,
        142, 140, 138, 147, 151, 157, 168, 171, 180, 191, 195, 207, 217, 222,
        236, 245, 250, 267, 271, 273, 281, 277, 260, 265, 300, 307, 305, 253,
        179, 169, 165, 162, 158, 157, 154, 163, 163, 173, 179, 183, 199, 199,
        209, 224, 222, 240, 249, 254, 276, 275, 292, 295, 294, 309, 307, 300,
        320, 363, 357, 287, 201, 191, 187, 182, 179, 177, 171, 171, 187, 180,
        200, 189, 216, 199, 235, 213, 247, 235, 263, 258, 271, 288, 287, 315,
        296, 361, 310, 338, 316, 362, 363, 326 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        97, 117, 117, 183,
#endif
        /* Size 4x4 */
        64, 91, 107, 128, 91, 110, 125, 140, 107, 125, 160, 166, 128, 140, 166,
        208,
        /* Size 8x8 */
        63, 73, 95, 97, 104, 121, 129, 139, 73, 86, 95, 93, 98, 111, 119, 129,
        95, 95, 106, 108, 111, 121, 128, 129, 97, 93, 108, 122, 130, 141, 142,
        140, 104, 98, 111, 130, 143, 157, 166, 160, 121, 111, 121, 141, 157,
        178, 179, 189, 129, 119, 128, 142, 166, 179, 197, 221, 139, 129, 129,
        140, 160, 189, 221, 219,
        /* Size 16x16 */
        64, 61, 67, 73, 89, 97, 98, 102, 108, 114, 120, 129, 134, 139, 144, 150,
        61, 63, 70, 79, 88, 92, 92, 95, 100, 106, 111, 119, 123, 129, 134, 140,
        67, 70, 78, 87, 92, 93, 91, 94, 98, 102, 107, 114, 118, 122, 127, 133,
        73, 79, 87, 95, 95, 93, 91, 93, 96, 100, 104, 110, 114, 118, 121, 126,
        89, 88, 92, 95, 100, 102, 102, 103, 106, 109, 113, 119, 122, 124, 132,
        135, 97, 92, 93, 93, 102, 109, 111, 113, 117, 120, 123, 129, 132, 129,
        137, 141, 98, 92, 91, 91, 102, 111, 117, 120, 124, 127, 131, 136, 139,
        140, 140, 149, 102, 95, 94, 93, 103, 113, 120, 125, 131, 135, 139, 144,
        146, 152, 145, 159, 108, 100, 98, 96, 106, 117, 124, 131, 140, 146, 150,
        156, 159, 163, 152, 168, 114, 106, 102, 100, 109, 120, 127, 135, 146,
        153, 158, 165, 169, 165, 160, 181, 120, 111, 107, 104, 113, 123, 131,
        139, 150, 158, 164, 172, 176, 173, 173, 189, 129, 119, 114, 110, 119,
        129, 136, 144, 156, 165, 172, 181, 186, 186, 188, 205, 134, 123, 118,
        114, 122, 132, 139, 146, 159, 169, 176, 186, 192, 194, 208, 212, 139,
        129, 122, 118, 124, 129, 140, 152, 163, 165, 173, 186, 194, 202, 204,
        231, 144, 134, 127, 121, 132, 137, 140, 145, 152, 160, 173, 188, 208,
        204, 213, 214, 150, 140, 133, 126, 135, 141, 149, 159, 168, 181, 189,
        205, 212, 231, 214, 225,
        /* Size 32x32 */
        64, 62, 61, 61, 67, 70, 73, 83, 89, 98, 97, 97, 98, 101, 102, 104, 108,
        110, 114, 118, 120, 127, 129, 130, 134, 137, 139, 142, 144, 147, 150,
        153, 62, 63, 63, 62, 69, 73, 77, 85, 89, 95, 94, 94, 94, 97, 97, 100,
        103, 104, 109, 112, 114, 121, 122, 123, 127, 129, 132, 134, 137, 139,
        142, 143, 61, 63, 63, 63, 70, 74, 79, 85, 88, 94, 92, 92, 92, 94, 95,
        97, 100, 102, 106, 109, 111, 117, 119, 120, 123, 126, 129, 131, 134,
        137, 140, 144, 61, 62, 63, 64, 71, 75, 80, 85, 88, 93, 91, 91, 90, 93,
        93, 95, 99, 100, 104, 107, 109, 115, 116, 117, 121, 123, 126, 128, 130,
        133, 135, 135, 67, 69, 70, 71, 78, 82, 87, 90, 92, 95, 93, 92, 91, 93,
        94, 95, 98, 99, 102, 106, 107, 113, 114, 115, 118, 120, 122, 125, 127,
        130, 133, 138, 70, 73, 74, 75, 82, 86, 91, 92, 93, 95, 93, 92, 91, 93,
        93, 94, 97, 98, 101, 104, 106, 111, 112, 113, 116, 119, 122, 124, 127,
        129, 131, 131, 73, 77, 79, 80, 87, 91, 95, 95, 95, 96, 93, 92, 91, 92,
        93, 94, 96, 97, 100, 103, 104, 109, 110, 111, 114, 116, 118, 119, 121,
        124, 126, 133, 83, 85, 85, 85, 90, 92, 95, 97, 99, 101, 99, 99, 98, 100,
        100, 101, 103, 104, 107, 109, 111, 115, 116, 117, 120, 121, 122, 121,
        122, 124, 126, 124, 89, 89, 88, 88, 92, 93, 95, 99, 100, 103, 102, 102,
        102, 103, 103, 104, 106, 107, 109, 112, 113, 118, 119, 119, 122, 122,
        124, 127, 132, 133, 135, 128, 98, 95, 94, 93, 95, 95, 96, 101, 103, 106,
        106, 107, 107, 108, 108, 109, 111, 111, 113, 116, 117, 121, 122, 123,
        126, 128, 131, 131, 127, 129, 131, 143, 97, 94, 92, 91, 93, 93, 93, 99,
        102, 106, 109, 109, 111, 113, 113, 115, 117, 118, 120, 122, 123, 127,
        129, 129, 132, 132, 129, 131, 137, 140, 141, 133, 97, 94, 92, 91, 92,
        92, 92, 99, 102, 107, 109, 110, 113, 115, 115, 117, 119, 120, 122, 125,
        126, 130, 131, 131, 133, 133, 138, 140, 137, 134, 136, 149, 98, 94, 92,
        90, 91, 91, 91, 98, 102, 107, 111, 113, 117, 119, 120, 123, 124, 125,
        127, 130, 131, 135, 136, 136, 139, 142, 140, 137, 140, 149, 149, 138,
        101, 97, 94, 93, 93, 93, 92, 100, 103, 108, 113, 115, 119, 122, 124,
        127, 129, 130, 133, 136, 137, 141, 142, 143, 145, 142, 142, 150, 149,
        140, 141, 158, 102, 97, 95, 93, 94, 93, 93, 100, 103, 108, 113, 115,
        120, 124, 125, 129, 131, 132, 135, 138, 139, 143, 144, 145, 146, 148,
        152, 145, 145, 159, 159, 143, 104, 100, 97, 95, 95, 94, 94, 101, 104,
        109, 115, 117, 123, 127, 129, 133, 136, 137, 141, 144, 145, 150, 151,
        151, 154, 156, 149, 155, 160, 147, 148, 168, 108, 103, 100, 99, 98, 97,
        96, 103, 106, 111, 117, 119, 124, 129, 131, 136, 140, 141, 146, 149,
        150, 155, 156, 157, 159, 155, 163, 160, 152, 165, 168, 149, 110, 104,
        102, 100, 99, 98, 97, 104, 107, 111, 118, 120, 125, 130, 132, 137, 141,
        143, 147, 150, 152, 157, 158, 159, 161, 166, 162, 159, 173, 160, 155,
        178, 114, 109, 106, 104, 102, 101, 100, 107, 109, 113, 120, 122, 127,
        133, 135, 141, 146, 147, 153, 156, 158, 164, 165, 166, 169, 167, 165,
        173, 160, 172, 181, 156, 118, 112, 109, 107, 106, 104, 103, 109, 112,
        116, 122, 125, 130, 136, 138, 144, 149, 150, 156, 160, 162, 168, 170,
        170, 173, 171, 178, 166, 181, 173, 163, 192, 120, 114, 111, 109, 107,
        106, 104, 111, 113, 117, 123, 126, 131, 137, 139, 145, 150, 152, 158,
        162, 164, 170, 172, 172, 176, 181, 173, 188, 173, 175, 189, 164, 127,
        121, 117, 115, 113, 111, 109, 115, 118, 121, 127, 130, 135, 141, 143,
        150, 155, 157, 164, 168, 170, 178, 179, 180, 185, 179, 188, 177, 187,
        189, 171, 199, 129, 122, 119, 116, 114, 112, 110, 116, 119, 122, 129,
        131, 136, 142, 144, 151, 156, 158, 165, 170, 172, 179, 181, 182, 186,
        190, 186, 189, 188, 182, 205, 173, 130, 123, 120, 117, 115, 113, 111,
        117, 119, 123, 129, 131, 136, 143, 145, 151, 157, 159, 166, 170, 172,
        180, 182, 183, 187, 190, 193, 195, 189, 202, 180, 216, 134, 127, 123,
        121, 118, 116, 114, 120, 122, 126, 132, 133, 139, 145, 146, 154, 159,
        161, 169, 173, 176, 185, 186, 187, 192, 190, 194, 194, 208, 187, 212,
        182, 137, 129, 126, 123, 120, 119, 116, 121, 122, 128, 132, 133, 142,
        142, 148, 156, 155, 166, 167, 171, 181, 179, 190, 190, 190, 197, 192,
        199, 194, 221, 189, 222, 139, 132, 129, 126, 122, 122, 118, 122, 124,
        131, 129, 138, 140, 142, 152, 149, 163, 162, 165, 178, 173, 188, 186,
        193, 194, 192, 202, 195, 204, 196, 231, 191, 142, 134, 131, 128, 125,
        124, 119, 121, 127, 131, 131, 140, 137, 150, 145, 155, 160, 159, 173,
        166, 188, 177, 189, 195, 194, 199, 195, 208, 197, 209, 199, 243, 144,
        137, 134, 130, 127, 127, 121, 122, 132, 127, 137, 137, 140, 149, 145,
        160, 152, 173, 160, 181, 173, 187, 188, 189, 208, 194, 204, 197, 213,
        200, 214, 202, 147, 139, 137, 133, 130, 129, 124, 124, 133, 129, 140,
        134, 149, 140, 159, 147, 165, 160, 172, 173, 175, 189, 182, 202, 187,
        221, 196, 209, 200, 219, 203, 219, 150, 142, 140, 135, 133, 131, 126,
        126, 135, 131, 141, 136, 149, 141, 159, 148, 168, 155, 181, 163, 189,
        171, 205, 180, 212, 189, 231, 199, 214, 203, 225, 205, 153, 143, 144,
        135, 138, 131, 133, 124, 128, 143, 133, 149, 138, 158, 143, 168, 149,
        178, 156, 192, 164, 199, 173, 216, 182, 222, 191, 243, 202, 219, 205,
        231,
        /* Size 4x8 */
        63, 94, 109, 133, 76, 93, 101, 123, 93, 107, 114, 129, 93, 113, 133,
        150, 100, 118, 148, 160, 115, 128, 164, 178, 124, 132, 173, 198, 133,
        134, 162, 196,
        /* Size 8x4 */
        63, 76, 93, 93, 100, 115, 124, 133, 94, 93, 107, 113, 118, 128, 132,
        134, 109, 101, 114, 133, 148, 164, 173, 162, 133, 123, 129, 150, 160,
        178, 198, 196,
        /* Size 8x16 */
        64, 68, 96, 98, 109, 127, 135, 145, 62, 72, 93, 92, 101, 117, 125, 134,
        67, 80, 94, 92, 98, 112, 119, 128, 74, 88, 95, 91, 96, 109, 115, 122,
        89, 92, 102, 102, 106, 118, 121, 132, 97, 93, 106, 112, 117, 128, 131,
        139, 98, 91, 107, 117, 125, 135, 142, 148, 102, 94, 108, 120, 131, 143,
        148, 158, 108, 98, 111, 124, 140, 155, 155, 164, 114, 102, 113, 128,
        146, 164, 168, 172, 120, 107, 117, 131, 150, 170, 181, 176, 128, 114,
        122, 136, 156, 179, 188, 184, 133, 118, 126, 139, 159, 183, 189, 188,
        139, 123, 129, 143, 157, 173, 194, 195, 144, 128, 132, 149, 171, 190,
        199, 198, 149, 133, 134, 150, 169, 190, 213, 216,
        /* Size 16x8 */
        64, 62, 67, 74, 89, 97, 98, 102, 108, 114, 120, 128, 133, 139, 144, 149,
        68, 72, 80, 88, 92, 93, 91, 94, 98, 102, 107, 114, 118, 123, 128, 133,
        96, 93, 94, 95, 102, 106, 107, 108, 111, 113, 117, 122, 126, 129, 132,
        134, 98, 92, 92, 91, 102, 112, 117, 120, 124, 128, 131, 136, 139, 143,
        149, 150, 109, 101, 98, 96, 106, 117, 125, 131, 140, 146, 150, 156, 159,
        157, 171, 169, 127, 117, 112, 109, 118, 128, 135, 143, 155, 164, 170,
        179, 183, 173, 190, 190, 135, 125, 119, 115, 121, 131, 142, 148, 155,
        168, 181, 188, 189, 194, 199, 213, 145, 134, 128, 122, 132, 139, 148,
        158, 164, 172, 176, 184, 188, 195, 198, 216,
        /* Size 16x32 */
        64, 62, 68, 74, 96, 97, 98, 105, 109, 115, 127, 129, 135, 140, 145, 149,
        63, 63, 71, 77, 94, 94, 94, 100, 103, 109, 120, 122, 128, 133, 137, 140,
        62, 64, 72, 79, 93, 92, 92, 97, 101, 106, 117, 119, 125, 130, 134, 141,
        61, 64, 73, 80, 92, 91, 91, 96, 99, 104, 115, 116, 122, 126, 130, 132,
        67, 72, 80, 86, 94, 93, 92, 95, 98, 103, 112, 114, 119, 124, 128, 135,
        71, 76, 84, 90, 95, 93, 91, 95, 97, 101, 111, 112, 118, 123, 127, 129,
        74, 80, 88, 94, 95, 93, 91, 94, 96, 100, 109, 110, 115, 117, 122, 131,
        84, 86, 91, 95, 100, 100, 99, 101, 103, 107, 115, 117, 121, 120, 122,
        121, 89, 89, 92, 95, 102, 102, 102, 104, 106, 109, 118, 119, 121, 127,
        132, 126, 98, 93, 95, 96, 106, 107, 107, 109, 111, 114, 121, 122, 128,
        129, 128, 141, 97, 92, 93, 94, 106, 109, 112, 115, 117, 120, 128, 129,
        131, 130, 139, 131, 97, 91, 92, 93, 106, 110, 113, 117, 119, 122, 130,
        131, 133, 139, 133, 147, 98, 91, 91, 92, 107, 112, 117, 123, 125, 128,
        135, 136, 142, 136, 148, 136, 101, 93, 93, 93, 108, 113, 119, 127, 130,
        133, 141, 142, 142, 150, 139, 156, 102, 94, 94, 94, 108, 114, 120, 129,
        131, 136, 143, 144, 148, 145, 158, 141, 104, 96, 95, 95, 109, 115, 122,
        133, 137, 142, 150, 151, 156, 154, 147, 167, 108, 99, 98, 97, 111, 117,
        124, 136, 140, 146, 155, 156, 155, 159, 164, 148, 109, 100, 99, 98, 111,
        118, 125, 137, 141, 148, 157, 158, 164, 160, 158, 176, 114, 104, 102,
        101, 113, 120, 128, 141, 146, 153, 164, 165, 168, 172, 172, 155, 118,
        108, 105, 104, 116, 123, 130, 144, 149, 157, 168, 170, 171, 165, 170,
        190, 120, 109, 107, 105, 117, 124, 131, 145, 150, 158, 170, 172, 181,
        185, 176, 163, 127, 115, 112, 110, 121, 128, 135, 150, 155, 164, 178,
        180, 181, 178, 185, 196, 128, 117, 114, 111, 122, 129, 136, 151, 156,
        165, 179, 181, 188, 187, 184, 171, 129, 118, 114, 112, 123, 130, 136,
        151, 157, 166, 180, 182, 189, 195, 196, 213, 133, 122, 118, 115, 126,
        132, 139, 154, 159, 169, 183, 186, 189, 193, 188, 180, 136, 124, 120,
        117, 129, 132, 140, 154, 155, 173, 181, 186, 195, 198, 216, 218, 139,
        126, 123, 119, 129, 129, 143, 152, 157, 176, 173, 196, 194, 193, 195,
        189, 141, 128, 125, 121, 130, 131, 149, 145, 162, 174, 175, 198, 191,
        206, 207, 239, 144, 131, 128, 123, 132, 133, 149, 145, 171, 160, 190,
        187, 199, 205, 198, 199, 146, 133, 130, 125, 133, 134, 149, 146, 170,
        162, 192, 178, 216, 196, 216, 216, 149, 136, 133, 127, 134, 136, 150,
        148, 169, 163, 190, 180, 213, 199, 216, 202, 152, 136, 138, 126, 129,
        148, 138, 167, 149, 190, 165, 213, 182, 237, 201, 228,
        /* Size 32x16 */
        64, 63, 62, 61, 67, 71, 74, 84, 89, 98, 97, 97, 98, 101, 102, 104, 108,
        109, 114, 118, 120, 127, 128, 129, 133, 136, 139, 141, 144, 146, 149,
        152, 62, 63, 64, 64, 72, 76, 80, 86, 89, 93, 92, 91, 91, 93, 94, 96, 99,
        100, 104, 108, 109, 115, 117, 118, 122, 124, 126, 128, 131, 133, 136,
        136, 68, 71, 72, 73, 80, 84, 88, 91, 92, 95, 93, 92, 91, 93, 94, 95, 98,
        99, 102, 105, 107, 112, 114, 114, 118, 120, 123, 125, 128, 130, 133,
        138, 74, 77, 79, 80, 86, 90, 94, 95, 95, 96, 94, 93, 92, 93, 94, 95, 97,
        98, 101, 104, 105, 110, 111, 112, 115, 117, 119, 121, 123, 125, 127,
        126, 96, 94, 93, 92, 94, 95, 95, 100, 102, 106, 106, 106, 107, 108, 108,
        109, 111, 111, 113, 116, 117, 121, 122, 123, 126, 129, 129, 130, 132,
        133, 134, 129, 97, 94, 92, 91, 93, 93, 93, 100, 102, 107, 109, 110, 112,
        113, 114, 115, 117, 118, 120, 123, 124, 128, 129, 130, 132, 132, 129,
        131, 133, 134, 136, 148, 98, 94, 92, 91, 92, 91, 91, 99, 102, 107, 112,
        113, 117, 119, 120, 122, 124, 125, 128, 130, 131, 135, 136, 136, 139,
        140, 143, 149, 149, 149, 150, 138, 105, 100, 97, 96, 95, 95, 94, 101,
        104, 109, 115, 117, 123, 127, 129, 133, 136, 137, 141, 144, 145, 150,
        151, 151, 154, 154, 152, 145, 145, 146, 148, 167, 109, 103, 101, 99, 98,
        97, 96, 103, 106, 111, 117, 119, 125, 130, 131, 137, 140, 141, 146, 149,
        150, 155, 156, 157, 159, 155, 157, 162, 171, 170, 169, 149, 115, 109,
        106, 104, 103, 101, 100, 107, 109, 114, 120, 122, 128, 133, 136, 142,
        146, 148, 153, 157, 158, 164, 165, 166, 169, 173, 176, 174, 160, 162,
        163, 190, 127, 120, 117, 115, 112, 111, 109, 115, 118, 121, 128, 130,
        135, 141, 143, 150, 155, 157, 164, 168, 170, 178, 179, 180, 183, 181,
        173, 175, 190, 192, 190, 165, 129, 122, 119, 116, 114, 112, 110, 117,
        119, 122, 129, 131, 136, 142, 144, 151, 156, 158, 165, 170, 172, 180,
        181, 182, 186, 186, 196, 198, 187, 178, 180, 213, 135, 128, 125, 122,
        119, 118, 115, 121, 121, 128, 131, 133, 142, 142, 148, 156, 155, 164,
        168, 171, 181, 181, 188, 189, 189, 195, 194, 191, 199, 216, 213, 182,
        140, 133, 130, 126, 124, 123, 117, 120, 127, 129, 130, 139, 136, 150,
        145, 154, 159, 160, 172, 165, 185, 178, 187, 195, 193, 198, 193, 206,
        205, 196, 199, 237, 145, 137, 134, 130, 128, 127, 122, 122, 132, 128,
        139, 133, 148, 139, 158, 147, 164, 158, 172, 170, 176, 185, 184, 196,
        188, 216, 195, 207, 198, 216, 216, 201, 149, 140, 141, 132, 135, 129,
        131, 121, 126, 141, 131, 147, 136, 156, 141, 167, 148, 176, 155, 190,
        163, 196, 171, 213, 180, 218, 189, 239, 199, 216, 202, 228,
        /* Size 4x16 */
        62, 97, 115, 140, 64, 92, 106, 130, 72, 93, 103, 124, 80, 93, 100, 117,
        89, 102, 109, 127, 92, 109, 120, 130, 91, 112, 128, 136, 94, 114, 136,
        145, 99, 117, 146, 159, 104, 120, 153, 172, 109, 124, 158, 185, 117,
        129, 165, 187, 122, 132, 169, 193, 126, 129, 176, 193, 131, 133, 160,
        205, 136, 136, 163, 199,
        /* Size 16x4 */
        62, 64, 72, 80, 89, 92, 91, 94, 99, 104, 109, 117, 122, 126, 131, 136,
        97, 92, 93, 93, 102, 109, 112, 114, 117, 120, 124, 129, 132, 129, 133,
        136, 115, 106, 103, 100, 109, 120, 128, 136, 146, 153, 158, 165, 169,
        176, 160, 163, 140, 130, 124, 117, 127, 130, 136, 145, 159, 172, 185,
        187, 193, 193, 205, 199,
        /* Size 8x32 */
        64, 68, 96, 98, 109, 127, 135, 145, 63, 71, 94, 94, 103, 120, 128, 137,
        62, 72, 93, 92, 101, 117, 125, 134, 61, 73, 92, 91, 99, 115, 122, 130,
        67, 80, 94, 92, 98, 112, 119, 128, 71, 84, 95, 91, 97, 111, 118, 127,
        74, 88, 95, 91, 96, 109, 115, 122, 84, 91, 100, 99, 103, 115, 121, 122,
        89, 92, 102, 102, 106, 118, 121, 132, 98, 95, 106, 107, 111, 121, 128,
        128, 97, 93, 106, 112, 117, 128, 131, 139, 97, 92, 106, 113, 119, 130,
        133, 133, 98, 91, 107, 117, 125, 135, 142, 148, 101, 93, 108, 119, 130,
        141, 142, 139, 102, 94, 108, 120, 131, 143, 148, 158, 104, 95, 109, 122,
        137, 150, 156, 147, 108, 98, 111, 124, 140, 155, 155, 164, 109, 99, 111,
        125, 141, 157, 164, 158, 114, 102, 113, 128, 146, 164, 168, 172, 118,
        105, 116, 130, 149, 168, 171, 170, 120, 107, 117, 131, 150, 170, 181,
        176, 127, 112, 121, 135, 155, 178, 181, 185, 128, 114, 122, 136, 156,
        179, 188, 184, 129, 114, 123, 136, 157, 180, 189, 196, 133, 118, 126,
        139, 159, 183, 189, 188, 136, 120, 129, 140, 155, 181, 195, 216, 139,
        123, 129, 143, 157, 173, 194, 195, 141, 125, 130, 149, 162, 175, 191,
        207, 144, 128, 132, 149, 171, 190, 199, 198, 146, 130, 133, 149, 170,
        192, 216, 216, 149, 133, 134, 150, 169, 190, 213, 216, 152, 138, 129,
        138, 149, 165, 182, 201,
        /* Size 32x8 */
        64, 63, 62, 61, 67, 71, 74, 84, 89, 98, 97, 97, 98, 101, 102, 104, 108,
        109, 114, 118, 120, 127, 128, 129, 133, 136, 139, 141, 144, 146, 149,
        152, 68, 71, 72, 73, 80, 84, 88, 91, 92, 95, 93, 92, 91, 93, 94, 95, 98,
        99, 102, 105, 107, 112, 114, 114, 118, 120, 123, 125, 128, 130, 133,
        138, 96, 94, 93, 92, 94, 95, 95, 100, 102, 106, 106, 106, 107, 108, 108,
        109, 111, 111, 113, 116, 117, 121, 122, 123, 126, 129, 129, 130, 132,
        133, 134, 129, 98, 94, 92, 91, 92, 91, 91, 99, 102, 107, 112, 113, 117,
        119, 120, 122, 124, 125, 128, 130, 131, 135, 136, 136, 139, 140, 143,
        149, 149, 149, 150, 138, 109, 103, 101, 99, 98, 97, 96, 103, 106, 111,
        117, 119, 125, 130, 131, 137, 140, 141, 146, 149, 150, 155, 156, 157,
        159, 155, 157, 162, 171, 170, 169, 149, 127, 120, 117, 115, 112, 111,
        109, 115, 118, 121, 128, 130, 135, 141, 143, 150, 155, 157, 164, 168,
        170, 178, 179, 180, 183, 181, 173, 175, 190, 192, 190, 165, 135, 128,
        125, 122, 119, 118, 115, 121, 121, 128, 131, 133, 142, 142, 148, 156,
        155, 164, 168, 171, 181, 181, 188, 189, 189, 195, 194, 191, 199, 216,
        213, 182, 145, 137, 134, 130, 128, 127, 122, 122, 132, 128, 139, 133,
        148, 139, 158, 147, 164, 158, 172, 170, 176, 185, 184, 196, 188, 216,
        195, 207, 198, 216, 216, 201 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        73, 142, 142, 268,
#endif
        /* Size 4x4 */
        65, 75, 117, 164, 75, 108, 145, 184, 117, 145, 204, 238, 164, 184, 238,
        313,
        /* Size 8x8 */
        64, 64, 70, 85, 107, 136, 157, 183, 64, 67, 72, 84, 103, 129, 148, 171,
        70, 72, 92, 105, 121, 144, 160, 174, 85, 84, 105, 127, 146, 169, 185,
        193, 107, 103, 121, 146, 172, 200, 218, 226, 136, 129, 144, 169, 200,
        235, 256, 273, 157, 148, 160, 185, 218, 256, 280, 329, 183, 171, 174,
        193, 226, 273, 329, 338,
        /* Size 16x16 */
        64, 63, 63, 64, 68, 72, 82, 95, 108, 118, 131, 149, 164, 176, 187, 199,
        63, 64, 65, 65, 68, 70, 79, 90, 101, 111, 122, 139, 153, 164, 175, 187,
        63, 65, 66, 67, 70, 72, 80, 89, 99, 108, 119, 134, 147, 157, 167, 179,
        64, 65, 67, 71, 74, 77, 83, 90, 99, 107, 116, 131, 143, 151, 161, 172,
        68, 68, 70, 74, 79, 85, 92, 100, 109, 117, 126, 141, 152, 161, 172, 179,
        72, 70, 72, 77, 85, 97, 104, 112, 120, 128, 136, 150, 161, 170, 182,
        190, 82, 79, 80, 83, 92, 104, 114, 125, 135, 143, 151, 166, 177, 184,
        191, 203, 95, 90, 89, 90, 100, 112, 125, 139, 151, 159, 168, 183, 194,
        202, 200, 219, 108, 101, 99, 99, 109, 120, 135, 151, 165, 174, 185, 200,
        212, 221, 213, 235, 118, 111, 108, 107, 117, 128, 143, 159, 174, 184,
        196, 213, 225, 234, 228, 257, 131, 122, 119, 116, 126, 136, 151, 168,
        185, 196, 210, 228, 241, 250, 248, 273, 149, 139, 134, 131, 141, 150,
        166, 183, 200, 213, 228, 247, 262, 270, 274, 299, 164, 153, 147, 143,
        152, 161, 177, 194, 212, 225, 241, 262, 278, 288, 305, 317, 176, 164,
        157, 151, 161, 170, 184, 202, 221, 234, 250, 270, 288, 301, 308, 351,
        187, 175, 167, 161, 172, 182, 191, 200, 213, 228, 248, 274, 305, 308,
        325, 330, 199, 187, 179, 172, 179, 190, 203, 219, 235, 257, 273, 299,
        317, 351, 330, 351,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 63, 64, 65, 68, 71, 72, 79, 82, 88, 95, 97, 108,
        112, 118, 128, 131, 143, 149, 160, 164, 166, 176, 181, 187, 193, 199,
        206, 63, 64, 64, 64, 64, 64, 65, 66, 68, 70, 70, 77, 80, 85, 91, 93,
        103, 107, 113, 122, 125, 136, 142, 152, 156, 157, 167, 172, 177, 183,
        189, 194, 63, 64, 64, 64, 65, 65, 65, 66, 68, 69, 70, 76, 79, 84, 90,
        91, 101, 105, 111, 120, 122, 134, 139, 149, 153, 154, 164, 169, 175,
        181, 187, 195, 63, 64, 64, 65, 65, 65, 65, 66, 67, 68, 69, 75, 77, 82,
        88, 89, 99, 102, 108, 117, 119, 130, 136, 145, 149, 150, 159, 164, 169,
        175, 180, 183, 63, 64, 65, 65, 66, 66, 67, 68, 70, 72, 72, 78, 80, 84,
        89, 91, 99, 103, 108, 116, 119, 129, 134, 143, 147, 148, 157, 162, 167,
        173, 179, 186, 63, 64, 65, 65, 66, 67, 68, 69, 71, 72, 73, 78, 81, 84,
        90, 91, 100, 103, 108, 116, 118, 129, 134, 143, 147, 148, 156, 161, 166,
        171, 176, 180, 64, 65, 65, 65, 67, 68, 71, 72, 74, 76, 77, 81, 83, 85,
        90, 92, 99, 102, 107, 114, 116, 126, 131, 139, 143, 144, 151, 156, 161,
        166, 172, 181, 65, 66, 66, 66, 68, 69, 72, 73, 76, 79, 80, 84, 86, 89,
        94, 95, 103, 106, 111, 118, 120, 130, 135, 142, 146, 147, 155, 156, 160,
        165, 170, 171, 68, 68, 68, 67, 70, 71, 74, 76, 79, 84, 85, 90, 92, 95,
        100, 102, 109, 112, 117, 124, 126, 136, 141, 148, 152, 153, 161, 166,
        172, 176, 179, 175, 71, 70, 69, 68, 72, 72, 76, 79, 84, 92, 94, 99, 101,
        105, 110, 111, 118, 121, 125, 132, 134, 144, 148, 156, 159, 160, 168,
        170, 170, 174, 179, 195, 72, 70, 70, 69, 72, 73, 77, 80, 85, 94, 97,
        101, 104, 108, 112, 114, 120, 123, 128, 134, 136, 146, 150, 158, 161,
        162, 170, 174, 182, 187, 190, 184, 79, 77, 76, 75, 78, 78, 81, 84, 90,
        99, 101, 108, 111, 116, 121, 122, 130, 133, 138, 145, 147, 156, 161,
        168, 172, 173, 181, 184, 183, 183, 187, 205, 82, 80, 79, 77, 80, 81, 83,
        86, 92, 101, 104, 111, 114, 120, 125, 127, 135, 138, 143, 150, 151, 161,
        166, 173, 177, 178, 184, 184, 191, 201, 203, 192, 88, 85, 84, 82, 84,
        84, 85, 89, 95, 105, 108, 116, 120, 127, 133, 134, 143, 146, 151, 158,
        159, 169, 173, 181, 184, 185, 192, 200, 200, 193, 197, 220, 95, 91, 90,
        88, 89, 90, 90, 94, 100, 110, 112, 121, 125, 133, 139, 141, 151, 154,
        159, 166, 168, 178, 183, 190, 194, 195, 202, 198, 200, 217, 219, 202,
        97, 93, 91, 89, 91, 91, 92, 95, 102, 111, 114, 122, 127, 134, 141, 142,
        153, 156, 161, 169, 171, 181, 186, 193, 197, 198, 205, 212, 218, 206,
        209, 236, 108, 103, 101, 99, 99, 100, 99, 103, 109, 118, 120, 130, 135,
        143, 151, 153, 165, 168, 174, 183, 185, 195, 200, 208, 212, 213, 221,
        219, 213, 230, 235, 213, 112, 107, 105, 102, 103, 103, 102, 106, 112,
        121, 123, 133, 138, 146, 154, 156, 168, 172, 178, 187, 189, 200, 205,
        213, 217, 218, 224, 224, 239, 226, 222, 253, 118, 113, 111, 108, 108,
        108, 107, 111, 117, 125, 128, 138, 143, 151, 159, 161, 174, 178, 184,
        194, 196, 207, 213, 221, 225, 226, 234, 240, 228, 244, 257, 226, 128,
        122, 120, 117, 116, 116, 114, 118, 124, 132, 134, 145, 150, 158, 166,
        169, 183, 187, 194, 204, 207, 219, 224, 233, 237, 238, 245, 238, 255,
        247, 237, 277, 131, 125, 122, 119, 119, 118, 116, 120, 126, 134, 136,
        147, 151, 159, 168, 171, 185, 189, 196, 207, 210, 222, 228, 237, 241,
        242, 250, 263, 248, 254, 273, 241, 143, 136, 134, 130, 129, 129, 126,
        130, 136, 144, 146, 156, 161, 169, 178, 181, 195, 200, 207, 219, 222,
        235, 241, 251, 255, 256, 266, 258, 270, 273, 252, 292, 149, 142, 139,
        136, 134, 134, 131, 135, 141, 148, 150, 161, 166, 173, 183, 186, 200,
        205, 213, 224, 228, 241, 247, 257, 262, 263, 270, 276, 274, 268, 299,
        256, 160, 152, 149, 145, 143, 143, 139, 142, 148, 156, 158, 168, 173,
        181, 190, 193, 208, 213, 221, 233, 237, 251, 257, 268, 273, 274, 285,
        285, 280, 297, 269, 321, 164, 156, 153, 149, 147, 147, 143, 146, 152,
        159, 161, 172, 177, 184, 194, 197, 212, 217, 225, 237, 241, 255, 262,
        273, 278, 279, 288, 291, 305, 280, 317, 274, 166, 157, 154, 150, 148,
        148, 144, 147, 153, 160, 162, 173, 178, 185, 195, 198, 213, 218, 226,
        238, 242, 256, 263, 274, 279, 280, 290, 298, 295, 329, 286, 337, 176,
        167, 164, 159, 157, 156, 151, 155, 161, 168, 170, 181, 184, 192, 202,
        205, 221, 224, 234, 245, 250, 266, 270, 285, 288, 290, 301, 296, 308,
        300, 351, 291, 181, 172, 169, 164, 162, 161, 156, 156, 166, 170, 174,
        184, 184, 200, 198, 212, 219, 224, 240, 238, 263, 258, 276, 285, 291,
        298, 296, 313, 302, 319, 306, 376, 187, 177, 175, 169, 167, 166, 161,
        160, 172, 170, 182, 183, 191, 200, 200, 218, 213, 239, 228, 255, 248,
        270, 274, 280, 305, 295, 308, 302, 325, 308, 330, 312, 193, 183, 181,
        175, 173, 171, 166, 165, 176, 174, 187, 183, 201, 193, 217, 206, 230,
        226, 244, 247, 254, 273, 268, 297, 280, 329, 300, 319, 308, 338, 313,
        341, 199, 189, 187, 180, 179, 176, 172, 170, 179, 179, 190, 187, 203,
        197, 219, 209, 235, 222, 257, 237, 273, 252, 299, 269, 317, 286, 351,
        306, 330, 313, 351, 319, 206, 194, 195, 183, 186, 180, 181, 171, 175,
        195, 184, 205, 192, 220, 202, 236, 213, 253, 226, 277, 241, 292, 256,
        321, 274, 337, 291, 376, 312, 341, 319, 364,
        /* Size 4x8 */
        64, 71, 119, 169, 65, 73, 114, 158, 69, 94, 130, 168, 82, 107, 156, 199,
        103, 123, 185, 224, 131, 146, 216, 258, 151, 162, 235, 296, 175, 183,
        232, 299,
        /* Size 8x4 */
        64, 65, 69, 82, 103, 131, 151, 175, 71, 73, 94, 107, 123, 146, 162, 183,
        119, 114, 130, 156, 185, 216, 235, 232, 169, 158, 168, 199, 224, 258,
        296, 299,
        /* Size 8x16 */
        64, 63, 70, 88, 107, 131, 164, 189, 63, 65, 69, 83, 101, 122, 153, 177,
        63, 66, 71, 84, 99, 118, 147, 169, 64, 68, 75, 85, 99, 116, 142, 163,
        68, 70, 83, 96, 109, 126, 152, 172, 72, 72, 92, 108, 120, 136, 161, 184,
        82, 80, 99, 120, 135, 152, 177, 198, 94, 89, 107, 132, 150, 168, 194,
        215, 107, 100, 115, 142, 164, 185, 212, 227, 117, 108, 123, 150, 174,
        196, 225, 242, 130, 119, 132, 159, 185, 210, 241, 253, 149, 134, 146,
        173, 200, 227, 262, 269, 164, 147, 158, 184, 211, 241, 278, 281, 175,
        157, 166, 192, 220, 250, 288, 298, 186, 167, 172, 199, 236, 272, 299,
        304, 198, 179, 180, 204, 235, 273, 317, 334,
        /* Size 16x8 */
        64, 63, 63, 64, 68, 72, 82, 94, 107, 117, 130, 149, 164, 175, 186, 198,
        63, 65, 66, 68, 70, 72, 80, 89, 100, 108, 119, 134, 147, 157, 167, 179,
        70, 69, 71, 75, 83, 92, 99, 107, 115, 123, 132, 146, 158, 166, 172, 180,
        88, 83, 84, 85, 96, 108, 120, 132, 142, 150, 159, 173, 184, 192, 199,
        204, 107, 101, 99, 99, 109, 120, 135, 150, 164, 174, 185, 200, 211, 220,
        236, 235, 131, 122, 118, 116, 126, 136, 152, 168, 185, 196, 210, 227,
        241, 250, 272, 273, 164, 153, 147, 142, 152, 161, 177, 194, 212, 225,
        241, 262, 278, 288, 299, 317, 189, 177, 169, 163, 172, 184, 198, 215,
        227, 242, 253, 269, 281, 298, 304, 334,
        /* Size 16x32 */
        64, 63, 63, 64, 70, 72, 88, 95, 107, 125, 131, 158, 164, 179, 189, 199,
        63, 64, 64, 65, 70, 71, 85, 91, 103, 119, 124, 151, 156, 169, 179, 188,
        63, 64, 65, 65, 69, 70, 83, 90, 101, 117, 122, 148, 153, 167, 177, 189,
        63, 65, 65, 66, 68, 69, 82, 88, 99, 114, 119, 144, 149, 162, 171, 178,
        63, 65, 66, 68, 71, 72, 84, 89, 99, 114, 118, 142, 147, 159, 169, 180,
        64, 65, 66, 68, 72, 73, 84, 90, 100, 114, 118, 142, 146, 158, 167, 174,
        64, 66, 68, 71, 75, 77, 85, 90, 99, 112, 116, 138, 142, 153, 163, 176,
        65, 66, 69, 72, 78, 80, 89, 94, 103, 116, 120, 142, 146, 155, 161, 167,
        68, 68, 70, 74, 83, 85, 96, 101, 109, 122, 126, 147, 152, 164, 172, 170,
        71, 69, 72, 76, 90, 94, 105, 110, 118, 130, 134, 155, 159, 168, 171,
        190, 72, 69, 72, 77, 92, 96, 108, 112, 120, 133, 136, 157, 161, 172,
        184, 179, 79, 75, 78, 81, 96, 101, 116, 121, 130, 143, 147, 168, 172,
        182, 180, 201, 82, 78, 80, 83, 99, 103, 120, 125, 135, 148, 152, 172,
        177, 183, 198, 187, 88, 82, 84, 86, 102, 107, 126, 132, 143, 156, 159,
        180, 184, 199, 190, 215, 94, 88, 89, 91, 107, 112, 132, 139, 150, 164,
        168, 190, 194, 197, 215, 198, 96, 90, 91, 93, 108, 113, 134, 140, 152,
        167, 171, 192, 197, 210, 204, 232, 107, 99, 100, 100, 115, 120, 142,
        150, 164, 180, 185, 207, 212, 217, 227, 209, 111, 103, 103, 103, 118,
        123, 145, 154, 168, 185, 189, 212, 217, 224, 222, 249, 117, 108, 108,
        108, 123, 127, 150, 159, 174, 191, 196, 220, 225, 238, 242, 222, 127,
        117, 116, 115, 130, 134, 157, 166, 182, 201, 207, 232, 237, 236, 242,
        273, 130, 120, 119, 117, 132, 136, 159, 168, 185, 204, 210, 236, 241,
        260, 253, 237, 142, 131, 129, 127, 142, 146, 169, 178, 195, 216, 222,
        250, 255, 258, 266, 287, 149, 136, 134, 132, 146, 150, 173, 183, 200,
        221, 227, 256, 262, 272, 269, 252, 159, 145, 143, 140, 154, 158, 181,
        190, 208, 230, 236, 267, 273, 284, 289, 315, 164, 150, 147, 144, 158,
        162, 184, 194, 211, 234, 241, 272, 278, 289, 281, 269, 165, 151, 148,
        145, 159, 162, 185, 195, 212, 235, 242, 273, 279, 296, 322, 331, 175,
        159, 157, 153, 166, 170, 192, 202, 220, 242, 250, 284, 288, 294, 298,
        286, 180, 165, 162, 157, 169, 174, 198, 198, 226, 240, 254, 284, 288,
        309, 315, 369, 186, 170, 167, 162, 172, 178, 199, 200, 236, 228, 272,
        272, 299, 309, 304, 307, 192, 175, 173, 166, 176, 183, 201, 205, 235,
        232, 276, 264, 322, 299, 333, 335, 198, 181, 179, 171, 180, 187, 204,
        209, 235, 237, 273, 269, 317, 305, 334, 314, 205, 185, 186, 174, 176,
        204, 192, 235, 213, 274, 241, 318, 274, 366, 311, 358,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 64, 64, 65, 68, 71, 72, 79, 82, 88, 94, 96, 107,
        111, 117, 127, 130, 142, 149, 159, 164, 165, 175, 180, 186, 192, 198,
        205, 63, 64, 64, 65, 65, 65, 66, 66, 68, 69, 69, 75, 78, 82, 88, 90, 99,
        103, 108, 117, 120, 131, 136, 145, 150, 151, 159, 165, 170, 175, 181,
        185, 63, 64, 65, 65, 66, 66, 68, 69, 70, 72, 72, 78, 80, 84, 89, 91,
        100, 103, 108, 116, 119, 129, 134, 143, 147, 148, 157, 162, 167, 173,
        179, 186, 64, 65, 65, 66, 68, 68, 71, 72, 74, 76, 77, 81, 83, 86, 91,
        93, 100, 103, 108, 115, 117, 127, 132, 140, 144, 145, 153, 157, 162,
        166, 171, 174, 70, 70, 69, 68, 71, 72, 75, 78, 83, 90, 92, 96, 99, 102,
        107, 108, 115, 118, 123, 130, 132, 142, 146, 154, 158, 159, 166, 169,
        172, 176, 180, 176, 72, 71, 70, 69, 72, 73, 77, 80, 85, 94, 96, 101,
        103, 107, 112, 113, 120, 123, 127, 134, 136, 146, 150, 158, 162, 162,
        170, 174, 178, 183, 187, 204, 88, 85, 83, 82, 84, 84, 85, 89, 96, 105,
        108, 116, 120, 126, 132, 134, 142, 145, 150, 157, 159, 169, 173, 181,
        184, 185, 192, 198, 199, 201, 204, 192, 95, 91, 90, 88, 89, 90, 90, 94,
        101, 110, 112, 121, 125, 132, 139, 140, 150, 154, 159, 166, 168, 178,
        183, 190, 194, 195, 202, 198, 200, 205, 209, 235, 107, 103, 101, 99, 99,
        100, 99, 103, 109, 118, 120, 130, 135, 143, 150, 152, 164, 168, 174,
        182, 185, 195, 200, 208, 211, 212, 220, 226, 236, 235, 235, 213, 125,
        119, 117, 114, 114, 114, 112, 116, 122, 130, 133, 143, 148, 156, 164,
        167, 180, 185, 191, 201, 204, 216, 221, 230, 234, 235, 242, 240, 228,
        232, 237, 274, 131, 124, 122, 119, 118, 118, 116, 120, 126, 134, 136,
        147, 152, 159, 168, 171, 185, 189, 196, 207, 210, 222, 227, 236, 241,
        242, 250, 254, 272, 276, 273, 241, 158, 151, 148, 144, 142, 142, 138,
        142, 147, 155, 157, 168, 172, 180, 190, 192, 207, 212, 220, 232, 236,
        250, 256, 267, 272, 273, 284, 284, 272, 264, 269, 318, 164, 156, 153,
        149, 147, 146, 142, 146, 152, 159, 161, 172, 177, 184, 194, 197, 212,
        217, 225, 237, 241, 255, 262, 273, 278, 279, 288, 288, 299, 322, 317,
        274, 179, 169, 167, 162, 159, 158, 153, 155, 164, 168, 172, 182, 183,
        199, 197, 210, 217, 224, 238, 236, 260, 258, 272, 284, 289, 296, 294,
        309, 309, 299, 305, 366, 189, 179, 177, 171, 169, 167, 163, 161, 172,
        171, 184, 180, 198, 190, 215, 204, 227, 222, 242, 242, 253, 266, 269,
        289, 281, 322, 298, 315, 304, 333, 334, 311, 199, 188, 189, 178, 180,
        174, 176, 167, 170, 190, 179, 201, 187, 215, 198, 232, 209, 249, 222,
        273, 237, 287, 252, 315, 269, 331, 286, 369, 307, 335, 314, 358,
        /* Size 4x16 */
        63, 72, 125, 179, 64, 70, 117, 167, 65, 72, 114, 159, 66, 77, 112, 153,
        68, 85, 122, 164, 69, 96, 133, 172, 78, 103, 148, 183, 88, 112, 164,
        197, 99, 120, 180, 217, 108, 127, 191, 238, 120, 136, 204, 260, 136,
        150, 221, 272, 150, 162, 234, 289, 159, 170, 242, 294, 170, 178, 228,
        309, 181, 187, 237, 305,
        /* Size 16x4 */
        63, 64, 65, 66, 68, 69, 78, 88, 99, 108, 120, 136, 150, 159, 170, 181,
        72, 70, 72, 77, 85, 96, 103, 112, 120, 127, 136, 150, 162, 170, 178,
        187, 125, 117, 114, 112, 122, 133, 148, 164, 180, 191, 204, 221, 234,
        242, 228, 237, 179, 167, 159, 153, 164, 172, 183, 197, 217, 238, 260,
        272, 289, 294, 309, 305,
        /* Size 8x32 */
        64, 63, 70, 88, 107, 131, 164, 189, 63, 64, 70, 85, 103, 124, 156, 179,
        63, 65, 69, 83, 101, 122, 153, 177, 63, 65, 68, 82, 99, 119, 149, 171,
        63, 66, 71, 84, 99, 118, 147, 169, 64, 66, 72, 84, 100, 118, 146, 167,
        64, 68, 75, 85, 99, 116, 142, 163, 65, 69, 78, 89, 103, 120, 146, 161,
        68, 70, 83, 96, 109, 126, 152, 172, 71, 72, 90, 105, 118, 134, 159, 171,
        72, 72, 92, 108, 120, 136, 161, 184, 79, 78, 96, 116, 130, 147, 172,
        180, 82, 80, 99, 120, 135, 152, 177, 198, 88, 84, 102, 126, 143, 159,
        184, 190, 94, 89, 107, 132, 150, 168, 194, 215, 96, 91, 108, 134, 152,
        171, 197, 204, 107, 100, 115, 142, 164, 185, 212, 227, 111, 103, 118,
        145, 168, 189, 217, 222, 117, 108, 123, 150, 174, 196, 225, 242, 127,
        116, 130, 157, 182, 207, 237, 242, 130, 119, 132, 159, 185, 210, 241,
        253, 142, 129, 142, 169, 195, 222, 255, 266, 149, 134, 146, 173, 200,
        227, 262, 269, 159, 143, 154, 181, 208, 236, 273, 289, 164, 147, 158,
        184, 211, 241, 278, 281, 165, 148, 159, 185, 212, 242, 279, 322, 175,
        157, 166, 192, 220, 250, 288, 298, 180, 162, 169, 198, 226, 254, 288,
        315, 186, 167, 172, 199, 236, 272, 299, 304, 192, 173, 176, 201, 235,
        276, 322, 333, 198, 179, 180, 204, 235, 273, 317, 334, 205, 186, 176,
        192, 213, 241, 274, 311,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 64, 64, 65, 68, 71, 72, 79, 82, 88, 94, 96, 107,
        111, 117, 127, 130, 142, 149, 159, 164, 165, 175, 180, 186, 192, 198,
        205, 63, 64, 65, 65, 66, 66, 68, 69, 70, 72, 72, 78, 80, 84, 89, 91,
        100, 103, 108, 116, 119, 129, 134, 143, 147, 148, 157, 162, 167, 173,
        179, 186, 70, 70, 69, 68, 71, 72, 75, 78, 83, 90, 92, 96, 99, 102, 107,
        108, 115, 118, 123, 130, 132, 142, 146, 154, 158, 159, 166, 169, 172,
        176, 180, 176, 88, 85, 83, 82, 84, 84, 85, 89, 96, 105, 108, 116, 120,
        126, 132, 134, 142, 145, 150, 157, 159, 169, 173, 181, 184, 185, 192,
        198, 199, 201, 204, 192, 107, 103, 101, 99, 99, 100, 99, 103, 109, 118,
        120, 130, 135, 143, 150, 152, 164, 168, 174, 182, 185, 195, 200, 208,
        211, 212, 220, 226, 236, 235, 235, 213, 131, 124, 122, 119, 118, 118,
        116, 120, 126, 134, 136, 147, 152, 159, 168, 171, 185, 189, 196, 207,
        210, 222, 227, 236, 241, 242, 250, 254, 272, 276, 273, 241, 164, 156,
        153, 149, 147, 146, 142, 146, 152, 159, 161, 172, 177, 184, 194, 197,
        212, 217, 225, 237, 241, 255, 262, 273, 278, 279, 288, 288, 299, 322,
        317, 274, 189, 179, 177, 171, 169, 167, 163, 161, 172, 171, 184, 180,
        198, 190, 215, 204, 227, 222, 242, 242, 253, 266, 269, 289, 281, 322,
        298, 315, 304, 333, 334, 311 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        95, 111, 111, 178,
#endif
        /* Size 4x4 */
        64, 91, 103, 123, 91, 109, 119, 133, 103, 119, 150, 162, 123, 133, 162,
        196,
        /* Size 8x8 */
        63, 69, 93, 94, 101, 114, 123, 134, 69, 78, 94, 91, 96, 107, 115, 124,
        93, 94, 104, 105, 108, 116, 122, 125, 94, 91, 105, 117, 124, 131, 136,
        136, 101, 96, 108, 124, 136, 147, 154, 154, 114, 107, 116, 131, 147,
        164, 172, 178, 123, 115, 122, 136, 154, 172, 183, 203, 134, 124, 125,
        136, 154, 178, 203, 206,
        /* Size 16x16 */
        64, 62, 66, 73, 83, 98, 98, 101, 104, 109, 114, 123, 129, 134, 139, 144,
        62, 63, 68, 78, 85, 94, 93, 95, 98, 102, 107, 114, 120, 125, 129, 134,
        66, 68, 75, 85, 89, 94, 92, 93, 95, 99, 103, 110, 115, 119, 123, 128,
        73, 78, 85, 95, 95, 96, 92, 92, 94, 97, 100, 106, 111, 114, 118, 123,
        83, 85, 89, 95, 97, 101, 99, 100, 101, 104, 107, 112, 117, 120, 125,
        127, 98, 94, 94, 96, 101, 106, 107, 108, 109, 111, 113, 119, 123, 126,
        131, 134, 98, 93, 92, 92, 99, 107, 111, 115, 118, 120, 123, 128, 132,
        134, 136, 141, 101, 95, 93, 92, 100, 108, 115, 122, 126, 129, 133, 138,
        141, 144, 141, 149, 104, 98, 95, 94, 101, 109, 118, 126, 133, 137, 141,
        147, 151, 154, 148, 158, 109, 102, 99, 97, 104, 111, 120, 129, 137, 142,
        147, 153, 158, 161, 156, 169, 114, 107, 103, 100, 107, 113, 123, 133,
        141, 147, 153, 160, 165, 169, 167, 177, 123, 114, 110, 106, 112, 119,
        128, 138, 147, 153, 160, 169, 175, 178, 179, 189, 129, 120, 115, 111,
        117, 123, 132, 141, 151, 158, 165, 175, 182, 186, 193, 197, 134, 125,
        119, 114, 120, 126, 134, 144, 154, 161, 169, 178, 186, 192, 194, 212,
        139, 129, 123, 118, 125, 131, 136, 141, 148, 156, 167, 179, 193, 194,
        201, 203, 144, 134, 128, 123, 127, 134, 141, 149, 158, 169, 177, 189,
        197, 212, 203, 212,
        /* Size 32x32 */
        64, 62, 62, 61, 66, 67, 73, 77, 83, 95, 98, 97, 98, 98, 101, 101, 104,
        106, 109, 113, 114, 120, 123, 127, 129, 130, 134, 136, 139, 141, 144,
        146, 62, 63, 63, 62, 68, 69, 77, 80, 85, 93, 95, 94, 94, 94, 96, 97,
        100, 101, 104, 108, 109, 114, 117, 121, 123, 123, 127, 129, 131, 134,
        136, 138, 62, 63, 63, 63, 68, 70, 78, 81, 85, 92, 94, 93, 93, 92, 95,
        95, 98, 100, 102, 106, 107, 112, 114, 118, 120, 121, 125, 127, 129, 132,
        134, 138, 61, 62, 63, 64, 69, 71, 80, 82, 85, 91, 93, 91, 91, 90, 92,
        93, 95, 97, 99, 103, 104, 109, 111, 115, 117, 117, 121, 123, 125, 127,
        130, 130, 66, 68, 68, 69, 75, 77, 85, 87, 89, 93, 94, 92, 92, 91, 93,
        93, 95, 97, 99, 102, 103, 107, 110, 113, 115, 115, 119, 121, 123, 126,
        128, 132, 67, 69, 70, 71, 77, 78, 87, 88, 90, 94, 95, 93, 92, 91, 93,
        93, 95, 96, 99, 102, 102, 107, 109, 113, 114, 115, 118, 120, 122, 124,
        126, 126, 73, 77, 78, 80, 85, 87, 95, 95, 95, 95, 96, 93, 92, 91, 92,
        92, 94, 95, 97, 99, 100, 104, 106, 109, 111, 111, 114, 116, 118, 120,
        123, 128, 77, 80, 81, 82, 87, 88, 95, 95, 96, 97, 98, 96, 95, 94, 95,
        95, 96, 98, 99, 102, 102, 107, 108, 111, 113, 113, 116, 116, 118, 119,
        121, 120, 83, 85, 85, 85, 89, 90, 95, 96, 97, 100, 101, 99, 99, 98, 100,
        100, 101, 102, 104, 106, 107, 111, 112, 115, 117, 117, 120, 122, 125,
        126, 127, 123, 95, 93, 92, 91, 93, 94, 95, 97, 100, 104, 105, 105, 105,
        105, 106, 106, 107, 108, 110, 111, 112, 116, 117, 120, 121, 122, 124,
        125, 123, 125, 127, 135, 98, 95, 94, 93, 94, 95, 96, 98, 101, 105, 106,
        106, 107, 107, 108, 108, 109, 110, 111, 113, 113, 117, 119, 121, 123,
        123, 126, 127, 131, 133, 134, 129, 97, 94, 93, 91, 92, 93, 93, 96, 99,
        105, 106, 109, 110, 111, 113, 113, 115, 116, 117, 119, 120, 123, 125,
        127, 129, 129, 132, 133, 131, 130, 132, 141, 98, 94, 93, 91, 92, 92, 92,
        95, 99, 105, 107, 110, 111, 114, 115, 116, 118, 119, 120, 122, 123, 126,
        128, 130, 132, 132, 134, 133, 136, 141, 141, 133, 98, 94, 92, 90, 91,
        91, 91, 94, 98, 105, 107, 111, 114, 117, 119, 120, 123, 124, 125, 127,
        127, 131, 132, 135, 136, 136, 139, 142, 141, 136, 138, 149, 101, 96, 95,
        92, 93, 93, 92, 95, 100, 106, 108, 113, 115, 119, 122, 122, 126, 128,
        129, 132, 133, 136, 138, 140, 141, 142, 144, 141, 141, 150, 149, 139,
        101, 97, 95, 93, 93, 93, 92, 95, 100, 106, 108, 113, 116, 120, 122, 123,
        127, 129, 130, 133, 134, 137, 139, 142, 143, 143, 146, 149, 151, 143,
        144, 158, 104, 100, 98, 95, 95, 95, 94, 96, 101, 107, 109, 115, 118,
        123, 126, 127, 133, 135, 137, 140, 141, 145, 147, 150, 151, 151, 154,
        152, 148, 156, 158, 145, 106, 101, 100, 97, 97, 96, 95, 98, 102, 108,
        110, 116, 119, 124, 128, 129, 135, 136, 139, 143, 143, 147, 149, 152,
        154, 154, 156, 155, 162, 154, 151, 166, 109, 104, 102, 99, 99, 99, 97,
        99, 104, 110, 111, 117, 120, 125, 129, 130, 137, 139, 142, 146, 147,
        151, 153, 156, 158, 158, 161, 163, 156, 164, 169, 153, 113, 108, 106,
        103, 102, 102, 99, 102, 106, 111, 113, 119, 122, 127, 132, 133, 140,
        143, 146, 150, 152, 157, 159, 162, 164, 164, 167, 162, 170, 165, 159,
        178, 114, 109, 107, 104, 103, 102, 100, 102, 107, 112, 113, 120, 123,
        127, 133, 134, 141, 143, 147, 152, 153, 158, 160, 164, 165, 166, 169,
        175, 167, 169, 177, 161, 120, 114, 112, 109, 107, 107, 104, 107, 111,
        116, 117, 123, 126, 131, 136, 137, 145, 147, 151, 157, 158, 164, 166,
        170, 172, 172, 176, 172, 177, 178, 167, 185, 123, 117, 114, 111, 110,
        109, 106, 108, 112, 117, 119, 125, 128, 132, 138, 139, 147, 149, 153,
        159, 160, 166, 169, 173, 175, 176, 178, 180, 179, 176, 189, 169, 127,
        121, 118, 115, 113, 113, 109, 111, 115, 120, 121, 127, 130, 135, 140,
        142, 150, 152, 156, 162, 164, 170, 173, 178, 180, 180, 185, 184, 182,
        189, 176, 199, 129, 123, 120, 117, 115, 114, 111, 113, 117, 121, 123,
        129, 132, 136, 141, 143, 151, 154, 158, 164, 165, 172, 175, 180, 182,
        183, 186, 187, 193, 182, 197, 178, 130, 123, 121, 117, 115, 115, 111,
        113, 117, 122, 123, 129, 132, 136, 142, 143, 151, 154, 158, 164, 166,
        172, 176, 180, 183, 183, 187, 190, 189, 203, 184, 205, 134, 127, 125,
        121, 119, 118, 114, 116, 120, 124, 126, 132, 134, 139, 144, 146, 154,
        156, 161, 167, 169, 176, 178, 185, 186, 187, 192, 189, 194, 191, 212,
        186, 136, 129, 127, 123, 121, 120, 116, 116, 122, 125, 127, 133, 133,
        142, 141, 149, 152, 155, 163, 162, 175, 172, 180, 184, 187, 190, 189,
        196, 192, 199, 193, 221, 139, 131, 129, 125, 123, 122, 118, 118, 125,
        123, 131, 131, 136, 141, 141, 151, 148, 162, 156, 170, 167, 177, 179,
        182, 193, 189, 194, 192, 201, 194, 203, 196, 141, 134, 132, 127, 126,
        124, 120, 119, 126, 125, 133, 130, 141, 136, 150, 143, 156, 154, 164,
        165, 169, 178, 176, 189, 182, 203, 191, 199, 194, 206, 196, 207, 144,
        136, 134, 130, 128, 126, 123, 121, 127, 127, 134, 132, 141, 138, 149,
        144, 158, 151, 169, 159, 177, 167, 189, 176, 197, 184, 212, 193, 203,
        196, 212, 199, 146, 138, 138, 130, 132, 126, 128, 120, 123, 135, 129,
        141, 133, 149, 139, 158, 145, 166, 153, 178, 161, 185, 169, 199, 178,
        205, 186, 221, 196, 207, 199, 217,
        /* Size 4x8 */
        63, 94, 107, 128, 72, 94, 101, 119, 92, 105, 111, 124, 91, 107, 127,
        142, 98, 110, 142, 156, 109, 117, 155, 173, 118, 123, 163, 190, 128,
        131, 158, 191,
        /* Size 8x4 */
        63, 72, 92, 91, 98, 109, 118, 128, 94, 94, 105, 107, 110, 117, 123, 131,
        107, 101, 111, 127, 142, 155, 163, 158, 128, 119, 124, 142, 156, 173,
        190, 191,
        /* Size 8x16 */
        64, 66, 91, 98, 105, 115, 129, 140, 62, 69, 90, 93, 98, 107, 120, 131,
        66, 75, 92, 91, 95, 103, 115, 124, 74, 86, 95, 91, 94, 100, 111, 119,
        84, 89, 99, 99, 101, 107, 117, 125, 98, 94, 104, 107, 109, 114, 123,
        132, 97, 92, 103, 114, 118, 123, 132, 140, 100, 93, 104, 119, 127, 133,
        142, 149, 104, 95, 106, 122, 133, 142, 151, 156, 109, 99, 108, 125, 137,
        147, 158, 164, 114, 103, 111, 128, 141, 153, 166, 169, 122, 110, 116,
        133, 147, 160, 175, 177, 129, 115, 120, 136, 151, 166, 182, 183, 133,
        119, 124, 139, 154, 169, 186, 190, 138, 124, 125, 141, 160, 179, 191,
        193, 143, 128, 127, 141, 158, 177, 198, 205,
        /* Size 16x8 */
        64, 62, 66, 74, 84, 98, 97, 100, 104, 109, 114, 122, 129, 133, 138, 143,
        66, 69, 75, 86, 89, 94, 92, 93, 95, 99, 103, 110, 115, 119, 124, 128,
        91, 90, 92, 95, 99, 104, 103, 104, 106, 108, 111, 116, 120, 124, 125,
        127, 98, 93, 91, 91, 99, 107, 114, 119, 122, 125, 128, 133, 136, 139,
        141, 141, 105, 98, 95, 94, 101, 109, 118, 127, 133, 137, 141, 147, 151,
        154, 160, 158, 115, 107, 103, 100, 107, 114, 123, 133, 142, 147, 153,
        160, 166, 169, 179, 177, 129, 120, 115, 111, 117, 123, 132, 142, 151,
        158, 166, 175, 182, 186, 191, 198, 140, 131, 124, 119, 125, 132, 140,
        149, 156, 164, 169, 177, 183, 190, 193, 205,
        /* Size 16x32 */
        64, 62, 66, 74, 91, 96, 98, 101, 105, 112, 115, 127, 129, 135, 140, 144,
        63, 63, 68, 77, 90, 94, 94, 97, 100, 107, 109, 120, 123, 128, 132, 136,
        62, 64, 69, 78, 90, 93, 93, 95, 98, 105, 107, 118, 120, 126, 131, 136,
        61, 64, 70, 80, 89, 92, 91, 93, 96, 102, 104, 115, 117, 122, 126, 128,
        66, 70, 75, 85, 92, 94, 91, 93, 95, 101, 103, 113, 115, 120, 124, 130,
        67, 72, 77, 86, 92, 94, 92, 93, 95, 101, 103, 112, 114, 119, 123, 125,
        74, 80, 86, 94, 95, 95, 91, 92, 94, 99, 100, 109, 111, 115, 119, 126,
        78, 82, 87, 95, 97, 97, 94, 95, 97, 101, 103, 111, 113, 116, 118, 119,
        84, 86, 89, 95, 99, 100, 99, 100, 101, 106, 107, 115, 117, 122, 125,
        122, 95, 92, 93, 96, 103, 105, 106, 106, 107, 111, 112, 120, 122, 124,
        124, 134, 98, 93, 94, 96, 104, 106, 107, 108, 109, 113, 114, 121, 123,
        127, 132, 127, 97, 92, 93, 94, 103, 106, 112, 113, 115, 119, 120, 128,
        129, 133, 129, 140, 97, 91, 92, 93, 103, 106, 114, 115, 118, 122, 123,
        130, 132, 133, 140, 132, 98, 91, 91, 92, 103, 107, 117, 119, 123, 127,
        128, 135, 136, 142, 135, 148, 100, 93, 93, 93, 104, 108, 119, 122, 127,
        131, 133, 140, 142, 141, 149, 138, 101, 93, 93, 93, 105, 108, 119, 123,
        128, 133, 134, 142, 143, 148, 143, 157, 104, 96, 95, 95, 106, 109, 122,
        127, 133, 140, 142, 150, 151, 152, 156, 144, 106, 98, 97, 96, 107, 110,
        123, 128, 135, 142, 144, 152, 154, 156, 153, 165, 109, 100, 99, 98, 108,
        111, 125, 130, 137, 145, 147, 156, 158, 163, 164, 152, 113, 103, 102,
        100, 110, 113, 127, 132, 140, 149, 152, 162, 164, 162, 164, 177, 114,
        104, 103, 101, 111, 113, 128, 133, 141, 151, 153, 164, 166, 174, 169,
        160, 120, 109, 108, 105, 115, 117, 131, 136, 145, 155, 158, 170, 172,
        173, 175, 184, 122, 112, 110, 107, 116, 119, 133, 138, 147, 158, 160,
        173, 175, 179, 177, 167, 127, 115, 113, 110, 119, 121, 135, 140, 150,
        161, 164, 178, 180, 184, 186, 197, 129, 117, 115, 112, 120, 123, 136,
        142, 151, 162, 166, 180, 182, 187, 183, 176, 129, 118, 116, 112, 121,
        123, 136, 142, 151, 163, 166, 180, 183, 190, 200, 203, 133, 122, 119,
        115, 124, 126, 139, 144, 154, 165, 169, 185, 186, 189, 190, 184, 136,
        124, 121, 117, 124, 127, 141, 141, 156, 163, 171, 184, 186, 195, 198,
        219, 138, 126, 124, 119, 125, 129, 141, 142, 160, 157, 179, 179, 191,
        195, 193, 194, 141, 128, 126, 121, 126, 131, 141, 143, 159, 158, 179,
        174, 200, 191, 205, 205, 143, 130, 128, 122, 127, 132, 141, 144, 158,
        159, 177, 176, 198, 193, 205, 197, 145, 131, 132, 122, 124, 140, 134,
        157, 146, 177, 161, 197, 178, 218, 195, 215,
        /* Size 32x16 */
        64, 63, 62, 61, 66, 67, 74, 78, 84, 95, 98, 97, 97, 98, 100, 101, 104,
        106, 109, 113, 114, 120, 122, 127, 129, 129, 133, 136, 138, 141, 143,
        145, 62, 63, 64, 64, 70, 72, 80, 82, 86, 92, 93, 92, 91, 91, 93, 93, 96,
        98, 100, 103, 104, 109, 112, 115, 117, 118, 122, 124, 126, 128, 130,
        131, 66, 68, 69, 70, 75, 77, 86, 87, 89, 93, 94, 93, 92, 91, 93, 93, 95,
        97, 99, 102, 103, 108, 110, 113, 115, 116, 119, 121, 124, 126, 128, 132,
        74, 77, 78, 80, 85, 86, 94, 95, 95, 96, 96, 94, 93, 92, 93, 93, 95, 96,
        98, 100, 101, 105, 107, 110, 112, 112, 115, 117, 119, 121, 122, 122, 91,
        90, 90, 89, 92, 92, 95, 97, 99, 103, 104, 103, 103, 103, 104, 105, 106,
        107, 108, 110, 111, 115, 116, 119, 120, 121, 124, 124, 125, 126, 127,
        124, 96, 94, 93, 92, 94, 94, 95, 97, 100, 105, 106, 106, 106, 107, 108,
        108, 109, 110, 111, 113, 113, 117, 119, 121, 123, 123, 126, 127, 129,
        131, 132, 140, 98, 94, 93, 91, 91, 92, 91, 94, 99, 106, 107, 112, 114,
        117, 119, 119, 122, 123, 125, 127, 128, 131, 133, 135, 136, 136, 139,
        141, 141, 141, 141, 134, 101, 97, 95, 93, 93, 93, 92, 95, 100, 106, 108,
        113, 115, 119, 122, 123, 127, 128, 130, 132, 133, 136, 138, 140, 142,
        142, 144, 141, 142, 143, 144, 157, 105, 100, 98, 96, 95, 95, 94, 97,
        101, 107, 109, 115, 118, 123, 127, 128, 133, 135, 137, 140, 141, 145,
        147, 150, 151, 151, 154, 156, 160, 159, 158, 146, 112, 107, 105, 102,
        101, 101, 99, 101, 106, 111, 113, 119, 122, 127, 131, 133, 140, 142,
        145, 149, 151, 155, 158, 161, 162, 163, 165, 163, 157, 158, 159, 177,
        115, 109, 107, 104, 103, 103, 100, 103, 107, 112, 114, 120, 123, 128,
        133, 134, 142, 144, 147, 152, 153, 158, 160, 164, 166, 166, 169, 171,
        179, 179, 177, 161, 127, 120, 118, 115, 113, 112, 109, 111, 115, 120,
        121, 128, 130, 135, 140, 142, 150, 152, 156, 162, 164, 170, 173, 178,
        180, 180, 185, 184, 179, 174, 176, 197, 129, 123, 120, 117, 115, 114,
        111, 113, 117, 122, 123, 129, 132, 136, 142, 143, 151, 154, 158, 164,
        166, 172, 175, 180, 182, 183, 186, 186, 191, 200, 198, 178, 135, 128,
        126, 122, 120, 119, 115, 116, 122, 124, 127, 133, 133, 142, 141, 148,
        152, 156, 163, 162, 174, 173, 179, 184, 187, 190, 189, 195, 195, 191,
        193, 218, 140, 132, 131, 126, 124, 123, 119, 118, 125, 124, 132, 129,
        140, 135, 149, 143, 156, 153, 164, 164, 169, 175, 177, 186, 183, 200,
        190, 198, 193, 205, 205, 195, 144, 136, 136, 128, 130, 125, 126, 119,
        122, 134, 127, 140, 132, 148, 138, 157, 144, 165, 152, 177, 160, 184,
        167, 197, 176, 203, 184, 219, 194, 205, 197, 215,
        /* Size 4x16 */
        62, 96, 112, 135, 64, 93, 105, 126, 70, 94, 101, 120, 80, 95, 99, 115,
        86, 100, 106, 122, 93, 106, 113, 127, 91, 106, 122, 133, 93, 108, 131,
        141, 96, 109, 140, 152, 100, 111, 145, 163, 104, 113, 151, 174, 112,
        119, 158, 179, 117, 123, 162, 187, 122, 126, 165, 189, 126, 129, 157,
        195, 130, 132, 159, 193,
        /* Size 16x4 */
        62, 64, 70, 80, 86, 93, 91, 93, 96, 100, 104, 112, 117, 122, 126, 130,
        96, 93, 94, 95, 100, 106, 106, 108, 109, 111, 113, 119, 123, 126, 129,
        132, 112, 105, 101, 99, 106, 113, 122, 131, 140, 145, 151, 158, 162,
        165, 157, 159, 135, 126, 120, 115, 122, 127, 133, 141, 152, 163, 174,
        179, 187, 189, 195, 193,
        /* Size 8x32 */
        64, 66, 91, 98, 105, 115, 129, 140, 63, 68, 90, 94, 100, 109, 123, 132,
        62, 69, 90, 93, 98, 107, 120, 131, 61, 70, 89, 91, 96, 104, 117, 126,
        66, 75, 92, 91, 95, 103, 115, 124, 67, 77, 92, 92, 95, 103, 114, 123,
        74, 86, 95, 91, 94, 100, 111, 119, 78, 87, 97, 94, 97, 103, 113, 118,
        84, 89, 99, 99, 101, 107, 117, 125, 95, 93, 103, 106, 107, 112, 122,
        124, 98, 94, 104, 107, 109, 114, 123, 132, 97, 93, 103, 112, 115, 120,
        129, 129, 97, 92, 103, 114, 118, 123, 132, 140, 98, 91, 103, 117, 123,
        128, 136, 135, 100, 93, 104, 119, 127, 133, 142, 149, 101, 93, 105, 119,
        128, 134, 143, 143, 104, 95, 106, 122, 133, 142, 151, 156, 106, 97, 107,
        123, 135, 144, 154, 153, 109, 99, 108, 125, 137, 147, 158, 164, 113,
        102, 110, 127, 140, 152, 164, 164, 114, 103, 111, 128, 141, 153, 166,
        169, 120, 108, 115, 131, 145, 158, 172, 175, 122, 110, 116, 133, 147,
        160, 175, 177, 127, 113, 119, 135, 150, 164, 180, 186, 129, 115, 120,
        136, 151, 166, 182, 183, 129, 116, 121, 136, 151, 166, 183, 200, 133,
        119, 124, 139, 154, 169, 186, 190, 136, 121, 124, 141, 156, 171, 186,
        198, 138, 124, 125, 141, 160, 179, 191, 193, 141, 126, 126, 141, 159,
        179, 200, 205, 143, 128, 127, 141, 158, 177, 198, 205, 145, 132, 124,
        134, 146, 161, 178, 195,
        /* Size 32x8 */
        64, 63, 62, 61, 66, 67, 74, 78, 84, 95, 98, 97, 97, 98, 100, 101, 104,
        106, 109, 113, 114, 120, 122, 127, 129, 129, 133, 136, 138, 141, 143,
        145, 66, 68, 69, 70, 75, 77, 86, 87, 89, 93, 94, 93, 92, 91, 93, 93, 95,
        97, 99, 102, 103, 108, 110, 113, 115, 116, 119, 121, 124, 126, 128, 132,
        91, 90, 90, 89, 92, 92, 95, 97, 99, 103, 104, 103, 103, 103, 104, 105,
        106, 107, 108, 110, 111, 115, 116, 119, 120, 121, 124, 124, 125, 126,
        127, 124, 98, 94, 93, 91, 91, 92, 91, 94, 99, 106, 107, 112, 114, 117,
        119, 119, 122, 123, 125, 127, 128, 131, 133, 135, 136, 136, 139, 141,
        141, 141, 141, 134, 105, 100, 98, 96, 95, 95, 94, 97, 101, 107, 109,
        115, 118, 123, 127, 128, 133, 135, 137, 140, 141, 145, 147, 150, 151,
        151, 154, 156, 160, 159, 158, 146, 115, 109, 107, 104, 103, 103, 100,
        103, 107, 112, 114, 120, 123, 128, 133, 134, 142, 144, 147, 152, 153,
        158, 160, 164, 166, 166, 169, 171, 179, 179, 177, 161, 129, 123, 120,
        117, 115, 114, 111, 113, 117, 122, 123, 129, 132, 136, 142, 143, 151,
        154, 158, 164, 166, 172, 175, 180, 182, 183, 186, 186, 191, 200, 198,
        178, 140, 132, 131, 126, 124, 123, 119, 118, 125, 124, 132, 129, 140,
        135, 149, 143, 156, 153, 164, 164, 169, 175, 177, 186, 183, 200, 190,
        198, 193, 205, 205, 195 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        71, 126, 126, 235,
#endif
        /* Size 4x4 */
        65, 69, 107, 150, 69, 98, 128, 163, 107, 128, 182, 225, 150, 163, 225,
        280,
        /* Size 8x8 */
        64, 64, 68, 79, 100, 125, 152, 170, 64, 67, 71, 80, 97, 118, 143, 159,
        68, 71, 79, 92, 107, 126, 148, 163, 79, 80, 92, 113, 131, 150, 172, 183,
        100, 97, 107, 131, 157, 180, 203, 211, 125, 118, 126, 150, 180, 210,
        237, 247, 152, 143, 148, 172, 203, 237, 268, 289, 170, 159, 163, 183,
        211, 247, 289, 308,
        /* Size 16x16 */
        64, 63, 63, 64, 67, 72, 79, 88, 97, 108, 118, 133, 149, 163, 173, 184,
        63, 64, 64, 65, 67, 70, 77, 84, 92, 102, 112, 126, 141, 154, 162, 174,
        63, 64, 65, 66, 68, 70, 76, 83, 90, 99, 108, 121, 135, 147, 155, 166,
        64, 65, 66, 69, 72, 75, 80, 85, 91, 99, 107, 119, 132, 143, 151, 161,
        67, 67, 68, 72, 77, 84, 88, 93, 100, 107, 115, 126, 139, 149, 157, 163,
        72, 70, 70, 75, 84, 97, 101, 108, 114, 120, 128, 138, 150, 160, 168,
        173, 79, 77, 76, 80, 88, 101, 108, 116, 122, 130, 138, 148, 161, 171,
        178, 185, 88, 84, 83, 85, 93, 108, 116, 127, 134, 143, 151, 161, 173,
        183, 190, 198, 97, 92, 90, 91, 100, 114, 122, 134, 142, 153, 161, 173,
        186, 196, 203, 212, 108, 102, 99, 99, 107, 120, 130, 143, 153, 165, 174,
        187, 200, 211, 218, 229, 118, 112, 108, 107, 115, 128, 138, 151, 161,
        174, 184, 198, 213, 224, 232, 245, 133, 126, 121, 119, 126, 138, 148,
        161, 173, 187, 198, 215, 230, 242, 251, 265, 149, 141, 135, 132, 139,
        150, 161, 173, 186, 200, 213, 230, 247, 261, 270, 282, 163, 154, 147,
        143, 149, 160, 171, 183, 196, 211, 224, 242, 261, 275, 285, 307, 173,
        162, 155, 151, 157, 168, 178, 190, 203, 218, 232, 251, 270, 285, 295,
        305, 184, 174, 166, 161, 163, 173, 185, 198, 212, 229, 245, 265, 282,
        307, 305, 319,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 63, 64, 64, 67, 68, 72, 72, 79, 82, 88, 92, 97, 104,
        108, 117, 118, 131, 133, 143, 149, 160, 163, 166, 173, 179, 184, 190,
        63, 64, 64, 64, 64, 64, 65, 65, 67, 68, 70, 71, 77, 79, 85, 89, 93, 100,
        103, 112, 113, 125, 127, 136, 142, 152, 155, 157, 164, 170, 175, 179,
        63, 64, 64, 64, 64, 65, 65, 65, 67, 68, 70, 71, 77, 79, 84, 88, 92, 99,
        102, 111, 112, 123, 126, 135, 141, 150, 154, 156, 162, 168, 174, 180,
        63, 64, 64, 65, 65, 65, 65, 65, 67, 67, 69, 69, 75, 77, 82, 85, 89, 96,
        99, 107, 108, 119, 121, 130, 136, 145, 148, 150, 156, 162, 167, 170, 63,
        64, 64, 65, 65, 66, 66, 66, 68, 69, 70, 71, 76, 78, 83, 86, 90, 96, 99,
        107, 108, 119, 121, 130, 135, 144, 147, 149, 155, 161, 166, 171, 63, 64,
        65, 65, 66, 67, 67, 68, 70, 71, 73, 73, 78, 80, 84, 88, 91, 97, 100,
        107, 108, 118, 120, 129, 134, 143, 146, 148, 154, 159, 163, 167, 64, 65,
        65, 65, 66, 67, 69, 70, 72, 72, 75, 76, 80, 81, 85, 88, 91, 97, 99, 106,
        107, 117, 119, 127, 132, 140, 143, 145, 151, 156, 161, 168, 64, 65, 65,
        65, 66, 68, 70, 71, 73, 74, 77, 77, 81, 82, 85, 88, 92, 97, 99, 106,
        107, 116, 118, 126, 131, 139, 142, 144, 149, 154, 158, 161, 67, 67, 67,
        67, 68, 70, 72, 73, 77, 78, 84, 84, 88, 90, 93, 96, 100, 105, 107, 114,
        115, 124, 126, 134, 139, 146, 149, 151, 157, 160, 163, 162, 68, 68, 68,
        67, 69, 71, 72, 74, 78, 79, 85, 86, 90, 92, 95, 99, 102, 107, 109, 116,
        117, 126, 128, 136, 141, 148, 151, 153, 159, 163, 167, 177, 72, 70, 70,
        69, 70, 73, 75, 77, 84, 85, 97, 97, 101, 103, 108, 111, 114, 118, 120,
        127, 128, 136, 138, 146, 150, 158, 160, 162, 168, 171, 173, 171, 72, 71,
        71, 69, 71, 73, 76, 77, 84, 86, 97, 98, 102, 104, 108, 111, 114, 119,
        121, 128, 129, 137, 139, 147, 151, 159, 161, 163, 169, 172, 176, 186,
        79, 77, 77, 75, 76, 78, 80, 81, 88, 90, 101, 102, 108, 110, 116, 119,
        122, 128, 130, 137, 138, 147, 148, 156, 161, 168, 171, 173, 178, 183,
        185, 180, 82, 79, 79, 77, 78, 80, 81, 82, 90, 92, 103, 104, 110, 113,
        119, 122, 126, 131, 134, 141, 141, 150, 152, 160, 164, 172, 175, 176,
        182, 183, 187, 199, 88, 85, 84, 82, 83, 84, 85, 85, 93, 95, 108, 108,
        116, 119, 127, 130, 134, 140, 143, 150, 151, 159, 161, 169, 173, 181,
        183, 185, 190, 197, 198, 191, 92, 89, 88, 85, 86, 88, 88, 88, 96, 99,
        111, 111, 119, 122, 130, 134, 138, 145, 148, 155, 156, 165, 167, 175,
        179, 187, 189, 191, 196, 195, 198, 213, 97, 93, 92, 89, 90, 91, 91, 92,
        100, 102, 114, 114, 122, 126, 134, 138, 142, 149, 153, 160, 161, 171,
        173, 181, 186, 193, 196, 198, 203, 210, 212, 202, 104, 100, 99, 96, 96,
        97, 97, 97, 105, 107, 118, 119, 128, 131, 140, 145, 149, 157, 161, 169,
        170, 180, 182, 191, 195, 203, 206, 208, 213, 211, 211, 227, 108, 103,
        102, 99, 99, 100, 99, 99, 107, 109, 120, 121, 130, 134, 143, 148, 153,
        161, 165, 173, 174, 185, 187, 195, 200, 208, 211, 213, 218, 225, 229,
        215, 117, 112, 111, 107, 107, 107, 106, 106, 114, 116, 127, 128, 137,
        141, 150, 155, 160, 169, 173, 182, 183, 195, 197, 206, 211, 220, 223,
        225, 230, 229, 226, 247, 118, 113, 112, 108, 108, 108, 107, 107, 115,
        117, 128, 129, 138, 141, 151, 156, 161, 170, 174, 183, 184, 196, 198,
        207, 213, 221, 224, 226, 232, 238, 245, 230, 131, 125, 123, 119, 119,
        118, 117, 116, 124, 126, 136, 137, 147, 150, 159, 165, 171, 180, 185,
        195, 196, 210, 212, 222, 228, 237, 240, 242, 248, 247, 240, 261, 133,
        127, 126, 121, 121, 120, 119, 118, 126, 128, 138, 139, 148, 152, 161,
        167, 173, 182, 187, 197, 198, 212, 215, 224, 230, 239, 242, 245, 251,
        254, 265, 244, 143, 136, 135, 130, 130, 129, 127, 126, 134, 136, 146,
        147, 156, 160, 169, 175, 181, 191, 195, 206, 207, 222, 224, 235, 241,
        251, 254, 256, 263, 267, 257, 283, 149, 142, 141, 136, 135, 134, 132,
        131, 139, 141, 150, 151, 161, 164, 173, 179, 186, 195, 200, 211, 213,
        228, 230, 241, 247, 257, 261, 263, 270, 268, 282, 261, 160, 152, 150,
        145, 144, 143, 140, 139, 146, 148, 158, 159, 168, 172, 181, 187, 193,
        203, 208, 220, 221, 237, 239, 251, 257, 268, 272, 274, 281, 289, 273,
        298, 163, 155, 154, 148, 147, 146, 143, 142, 149, 151, 160, 161, 171,
        175, 183, 189, 196, 206, 211, 223, 224, 240, 242, 254, 261, 272, 275,
        278, 285, 287, 307, 277, 166, 157, 156, 150, 149, 148, 145, 144, 151,
        153, 162, 163, 173, 176, 185, 191, 198, 208, 213, 225, 226, 242, 245,
        256, 263, 274, 278, 280, 288, 296, 292, 326, 173, 164, 162, 156, 155,
        154, 151, 149, 157, 159, 168, 169, 178, 182, 190, 196, 203, 213, 218,
        230, 232, 248, 251, 263, 270, 281, 285, 288, 295, 294, 305, 297, 179,
        170, 168, 162, 161, 159, 156, 154, 160, 163, 171, 172, 183, 183, 197,
        195, 210, 211, 225, 229, 238, 247, 254, 267, 268, 289, 287, 296, 294,
        308, 299, 314, 184, 175, 174, 167, 166, 163, 161, 158, 163, 167, 173,
        176, 185, 187, 198, 198, 212, 211, 229, 226, 245, 240, 265, 257, 282,
        273, 307, 292, 305, 299, 319, 304, 190, 179, 180, 170, 171, 167, 168,
        161, 162, 177, 171, 186, 180, 199, 191, 213, 202, 227, 215, 247, 230,
        261, 244, 283, 261, 298, 277, 326, 297, 314, 304, 330,
        /* Size 4x8 */
        64, 71, 103, 155, 65, 73, 100, 145, 68, 85, 109, 151, 77, 103, 134, 174,
        96, 118, 161, 206, 120, 136, 185, 239, 145, 158, 208, 271, 162, 172,
        222, 286,
        /* Size 8x4 */
        64, 65, 68, 77, 96, 120, 145, 162, 71, 73, 85, 103, 118, 136, 158, 172,
        103, 100, 109, 134, 161, 185, 208, 222, 155, 145, 151, 174, 206, 239,
        271, 286,
        /* Size 8x16 */
        64, 63, 67, 81, 103, 131, 158, 177, 63, 64, 67, 78, 98, 123, 149, 166,
        63, 65, 68, 77, 95, 119, 143, 159, 64, 66, 72, 81, 96, 117, 139, 155,
        67, 68, 77, 89, 104, 124, 145, 159, 72, 71, 84, 103, 117, 136, 157, 170,
        79, 77, 88, 109, 127, 147, 168, 182, 88, 83, 93, 118, 139, 159, 180,
        196, 96, 90, 100, 125, 148, 171, 192, 209, 107, 99, 107, 132, 159, 185,
        207, 224, 117, 108, 115, 140, 168, 196, 220, 237, 132, 121, 126, 151,
        181, 212, 238, 253, 149, 135, 139, 163, 194, 227, 256, 267, 162, 147,
        150, 173, 204, 240, 271, 285, 172, 156, 157, 181, 212, 248, 280, 292,
        183, 166, 164, 185, 212, 245, 282, 306,
        /* Size 16x8 */
        64, 63, 63, 64, 67, 72, 79, 88, 96, 107, 117, 132, 149, 162, 172, 183,
        63, 64, 65, 66, 68, 71, 77, 83, 90, 99, 108, 121, 135, 147, 156, 166,
        67, 67, 68, 72, 77, 84, 88, 93, 100, 107, 115, 126, 139, 150, 157, 164,
        81, 78, 77, 81, 89, 103, 109, 118, 125, 132, 140, 151, 163, 173, 181,
        185, 103, 98, 95, 96, 104, 117, 127, 139, 148, 159, 168, 181, 194, 204,
        212, 212, 131, 123, 119, 117, 124, 136, 147, 159, 171, 185, 196, 212,
        227, 240, 248, 245, 158, 149, 143, 139, 145, 157, 168, 180, 192, 207,
        220, 238, 256, 271, 280, 282, 177, 166, 159, 155, 159, 170, 182, 196,
        209, 224, 237, 253, 267, 285, 292, 306,
        /* Size 16x32 */
        64, 63, 63, 64, 67, 72, 81, 88, 103, 107, 131, 133, 158, 163, 177, 186,
        63, 64, 64, 65, 67, 71, 79, 85, 99, 103, 124, 127, 151, 155, 168, 176,
        63, 64, 64, 65, 67, 70, 78, 84, 98, 102, 123, 125, 149, 153, 166, 176,
        63, 65, 65, 66, 67, 69, 76, 82, 95, 99, 119, 121, 144, 148, 160, 167,
        63, 65, 65, 67, 68, 71, 77, 83, 95, 99, 119, 121, 143, 147, 159, 168,
        64, 65, 66, 68, 70, 73, 79, 84, 96, 100, 118, 120, 142, 145, 157, 164,
        64, 65, 66, 70, 72, 75, 81, 85, 96, 99, 117, 119, 139, 143, 155, 165,
        64, 66, 67, 71, 73, 77, 82, 85, 96, 99, 116, 118, 138, 141, 152, 158,
        67, 67, 68, 73, 77, 83, 89, 93, 104, 107, 124, 126, 145, 149, 159, 159,
        68, 68, 69, 74, 78, 85, 91, 96, 106, 109, 126, 128, 147, 151, 162, 174,
        72, 69, 71, 77, 84, 96, 103, 108, 117, 120, 136, 138, 157, 160, 170,
        169, 72, 70, 71, 77, 84, 96, 103, 108, 118, 121, 137, 139, 158, 161,
        171, 184, 79, 75, 77, 81, 88, 101, 109, 116, 127, 130, 147, 149, 168,
        171, 182, 178, 81, 77, 78, 83, 90, 103, 112, 119, 130, 134, 150, 152,
        171, 174, 181, 196, 88, 82, 83, 86, 93, 107, 118, 126, 139, 143, 159,
        161, 180, 183, 196, 188, 92, 86, 87, 89, 97, 110, 121, 130, 144, 147,
        165, 167, 186, 189, 194, 211, 96, 90, 90, 93, 100, 113, 125, 134, 148,
        152, 171, 173, 192, 196, 209, 200, 104, 96, 97, 98, 105, 118, 130, 140,
        156, 161, 180, 182, 202, 206, 210, 225, 107, 99, 99, 100, 107, 120, 132,
        142, 159, 164, 185, 187, 207, 210, 224, 213, 116, 107, 107, 107, 114,
        127, 139, 149, 167, 173, 195, 197, 219, 222, 227, 244, 117, 108, 108,
        108, 115, 127, 140, 150, 168, 174, 196, 198, 220, 224, 237, 228, 130,
        120, 119, 117, 124, 136, 149, 159, 179, 185, 210, 212, 236, 239, 246,
        258, 132, 122, 121, 119, 126, 138, 151, 161, 181, 187, 212, 214, 238,
        242, 253, 242, 142, 131, 130, 127, 134, 146, 159, 169, 189, 195, 222,
        224, 250, 254, 265, 280, 149, 136, 135, 132, 139, 150, 163, 173, 194,
        200, 227, 230, 256, 260, 267, 259, 159, 145, 144, 140, 147, 158, 171,
        181, 202, 208, 236, 239, 267, 271, 287, 295, 162, 148, 147, 143, 150,
        161, 173, 183, 204, 210, 240, 242, 271, 275, 285, 275, 165, 151, 149,
        145, 151, 162, 175, 185, 206, 212, 242, 244, 273, 278, 294, 322, 172,
        157, 156, 151, 157, 168, 181, 190, 212, 218, 248, 250, 280, 285, 292,
        294, 178, 162, 161, 155, 161, 172, 183, 195, 212, 222, 247, 253, 286,
        286, 305, 311, 183, 167, 166, 159, 164, 176, 185, 198, 212, 226, 245,
        257, 282, 291, 306, 301, 189, 171, 172, 163, 164, 186, 180, 212, 202,
        245, 230, 281, 261, 321, 296, 327,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 64, 64, 64, 67, 68, 72, 72, 79, 81, 88, 92, 96, 104,
        107, 116, 117, 130, 132, 142, 149, 159, 162, 165, 172, 178, 183, 189,
        63, 64, 64, 65, 65, 65, 65, 66, 67, 68, 69, 70, 75, 77, 82, 86, 90, 96,
        99, 107, 108, 120, 122, 131, 136, 145, 148, 151, 157, 162, 167, 171, 63,
        64, 64, 65, 65, 66, 66, 67, 68, 69, 71, 71, 77, 78, 83, 87, 90, 97, 99,
        107, 108, 119, 121, 130, 135, 144, 147, 149, 156, 161, 166, 172, 64, 65,
        65, 66, 67, 68, 70, 71, 73, 74, 77, 77, 81, 83, 86, 89, 93, 98, 100,
        107, 108, 117, 119, 127, 132, 140, 143, 145, 151, 155, 159, 163, 67, 67,
        67, 67, 68, 70, 72, 73, 77, 78, 84, 84, 88, 90, 93, 97, 100, 105, 107,
        114, 115, 124, 126, 134, 139, 147, 150, 151, 157, 161, 164, 164, 72, 71,
        70, 69, 71, 73, 75, 77, 83, 85, 96, 96, 101, 103, 107, 110, 113, 118,
        120, 127, 127, 136, 138, 146, 150, 158, 161, 162, 168, 172, 176, 186,
        81, 79, 78, 76, 77, 79, 81, 82, 89, 91, 103, 103, 109, 112, 118, 121,
        125, 130, 132, 139, 140, 149, 151, 159, 163, 171, 173, 175, 181, 183,
        185, 180, 88, 85, 84, 82, 83, 84, 85, 85, 93, 96, 108, 108, 116, 119,
        126, 130, 134, 140, 142, 149, 150, 159, 161, 169, 173, 181, 183, 185,
        190, 195, 198, 212, 103, 99, 98, 95, 95, 96, 96, 96, 104, 106, 117, 118,
        127, 130, 139, 144, 148, 156, 159, 167, 168, 179, 181, 189, 194, 202,
        204, 206, 212, 212, 212, 202, 107, 103, 102, 99, 99, 100, 99, 99, 107,
        109, 120, 121, 130, 134, 143, 147, 152, 161, 164, 173, 174, 185, 187,
        195, 200, 208, 210, 212, 218, 222, 226, 245, 131, 124, 123, 119, 119,
        118, 117, 116, 124, 126, 136, 137, 147, 150, 159, 165, 171, 180, 185,
        195, 196, 210, 212, 222, 227, 236, 240, 242, 248, 247, 245, 230, 133,
        127, 125, 121, 121, 120, 119, 118, 126, 128, 138, 139, 149, 152, 161,
        167, 173, 182, 187, 197, 198, 212, 214, 224, 230, 239, 242, 244, 250,
        253, 257, 281, 158, 151, 149, 144, 143, 142, 139, 138, 145, 147, 157,
        158, 168, 171, 180, 186, 192, 202, 207, 219, 220, 236, 238, 250, 256,
        267, 271, 273, 280, 286, 282, 261, 163, 155, 153, 148, 147, 145, 143,
        141, 149, 151, 160, 161, 171, 174, 183, 189, 196, 206, 210, 222, 224,
        239, 242, 254, 260, 271, 275, 278, 285, 286, 291, 321, 177, 168, 166,
        160, 159, 157, 155, 152, 159, 162, 170, 171, 182, 181, 196, 194, 209,
        210, 224, 227, 237, 246, 253, 265, 267, 287, 285, 294, 292, 305, 306,
        296, 186, 176, 176, 167, 168, 164, 165, 158, 159, 174, 169, 184, 178,
        196, 188, 211, 200, 225, 213, 244, 228, 258, 242, 280, 259, 295, 275,
        322, 294, 311, 301, 327,
        /* Size 4x16 */
        63, 72, 107, 163, 64, 70, 102, 153, 65, 71, 99, 147, 65, 75, 99, 143,
        67, 83, 107, 149, 69, 96, 120, 160, 75, 101, 130, 171, 82, 107, 143,
        183, 90, 113, 152, 196, 99, 120, 164, 210, 108, 127, 174, 224, 122, 138,
        187, 242, 136, 150, 200, 260, 148, 161, 210, 275, 157, 168, 218, 285,
        167, 176, 226, 291,
        /* Size 16x4 */
        63, 64, 65, 65, 67, 69, 75, 82, 90, 99, 108, 122, 136, 148, 157, 167,
        72, 70, 71, 75, 83, 96, 101, 107, 113, 120, 127, 138, 150, 161, 168,
        176, 107, 102, 99, 99, 107, 120, 130, 143, 152, 164, 174, 187, 200, 210,
        218, 226, 163, 153, 147, 143, 149, 160, 171, 183, 196, 210, 224, 242,
        260, 275, 285, 291,
        /* Size 8x32 */
        64, 63, 67, 81, 103, 131, 158, 177, 63, 64, 67, 79, 99, 124, 151, 168,
        63, 64, 67, 78, 98, 123, 149, 166, 63, 65, 67, 76, 95, 119, 144, 160,
        63, 65, 68, 77, 95, 119, 143, 159, 64, 66, 70, 79, 96, 118, 142, 157,
        64, 66, 72, 81, 96, 117, 139, 155, 64, 67, 73, 82, 96, 116, 138, 152,
        67, 68, 77, 89, 104, 124, 145, 159, 68, 69, 78, 91, 106, 126, 147, 162,
        72, 71, 84, 103, 117, 136, 157, 170, 72, 71, 84, 103, 118, 137, 158,
        171, 79, 77, 88, 109, 127, 147, 168, 182, 81, 78, 90, 112, 130, 150,
        171, 181, 88, 83, 93, 118, 139, 159, 180, 196, 92, 87, 97, 121, 144,
        165, 186, 194, 96, 90, 100, 125, 148, 171, 192, 209, 104, 97, 105, 130,
        156, 180, 202, 210, 107, 99, 107, 132, 159, 185, 207, 224, 116, 107,
        114, 139, 167, 195, 219, 227, 117, 108, 115, 140, 168, 196, 220, 237,
        130, 119, 124, 149, 179, 210, 236, 246, 132, 121, 126, 151, 181, 212,
        238, 253, 142, 130, 134, 159, 189, 222, 250, 265, 149, 135, 139, 163,
        194, 227, 256, 267, 159, 144, 147, 171, 202, 236, 267, 287, 162, 147,
        150, 173, 204, 240, 271, 285, 165, 149, 151, 175, 206, 242, 273, 294,
        172, 156, 157, 181, 212, 248, 280, 292, 178, 161, 161, 183, 212, 247,
        286, 305, 183, 166, 164, 185, 212, 245, 282, 306, 189, 172, 164, 180,
        202, 230, 261, 296,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 64, 64, 64, 67, 68, 72, 72, 79, 81, 88, 92, 96, 104,
        107, 116, 117, 130, 132, 142, 149, 159, 162, 165, 172, 178, 183, 189,
        63, 64, 64, 65, 65, 66, 66, 67, 68, 69, 71, 71, 77, 78, 83, 87, 90, 97,
        99, 107, 108, 119, 121, 130, 135, 144, 147, 149, 156, 161, 166, 172, 67,
        67, 67, 67, 68, 70, 72, 73, 77, 78, 84, 84, 88, 90, 93, 97, 100, 105,
        107, 114, 115, 124, 126, 134, 139, 147, 150, 151, 157, 161, 164, 164,
        81, 79, 78, 76, 77, 79, 81, 82, 89, 91, 103, 103, 109, 112, 118, 121,
        125, 130, 132, 139, 140, 149, 151, 159, 163, 171, 173, 175, 181, 183,
        185, 180, 103, 99, 98, 95, 95, 96, 96, 96, 104, 106, 117, 118, 127, 130,
        139, 144, 148, 156, 159, 167, 168, 179, 181, 189, 194, 202, 204, 206,
        212, 212, 212, 202, 131, 124, 123, 119, 119, 118, 117, 116, 124, 126,
        136, 137, 147, 150, 159, 165, 171, 180, 185, 195, 196, 210, 212, 222,
        227, 236, 240, 242, 248, 247, 245, 230, 158, 151, 149, 144, 143, 142,
        139, 138, 145, 147, 157, 158, 168, 171, 180, 186, 192, 202, 207, 219,
        220, 236, 238, 250, 256, 267, 271, 273, 280, 286, 282, 261, 177, 168,
        166, 160, 159, 157, 155, 152, 159, 162, 170, 171, 182, 181, 196, 194,
        209, 210, 224, 227, 237, 246, 253, 265, 267, 287, 285, 294, 292, 305,
        306, 296 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        95, 104, 104, 164,
#endif
        /* Size 4x4 */
        64, 93, 99, 117, 93, 106, 111, 124, 99, 111, 141, 157, 117, 124, 157,
        183,
        /* Size 8x8 */
        63, 69, 85, 94, 99, 109, 121, 128, 69, 78, 90, 92, 95, 102, 113, 119,
        85, 90, 97, 99, 101, 107, 115, 121, 94, 92, 99, 110, 116, 122, 130, 133,
        99, 95, 101, 116, 130, 139, 147, 149, 109, 102, 107, 122, 139, 153, 164,
        167, 121, 113, 115, 130, 147, 164, 178, 186, 128, 119, 121, 133, 149,
        167, 186, 194,
        /* Size 16x16 */
        64, 62, 63, 71, 81, 98, 97, 98, 101, 104, 109, 115, 123, 129, 133, 138,
        62, 63, 65, 74, 83, 95, 94, 93, 96, 99, 103, 109, 115, 121, 125, 129,
        63, 65, 69, 79, 86, 93, 92, 90, 93, 95, 99, 104, 110, 115, 119, 123, 71,
        74, 79, 88, 92, 95, 93, 91, 93, 94, 97, 102, 107, 112, 115, 119, 81, 83,
        86, 92, 96, 100, 98, 97, 98, 99, 102, 106, 111, 115, 118, 120, 98, 95,
        93, 95, 100, 106, 106, 107, 108, 109, 111, 114, 119, 122, 125, 126, 97,
        94, 92, 93, 98, 106, 109, 111, 113, 115, 117, 121, 125, 129, 131, 133,
        98, 93, 90, 91, 97, 107, 111, 117, 120, 123, 125, 128, 132, 136, 138,
        140, 101, 96, 93, 93, 98, 108, 113, 120, 123, 127, 130, 135, 139, 143,
        145, 148, 104, 99, 95, 94, 99, 109, 115, 123, 127, 133, 137, 142, 147,
        151, 153, 157, 109, 103, 99, 97, 102, 111, 117, 125, 130, 137, 142, 148,
        153, 157, 160, 165, 115, 109, 104, 102, 106, 114, 121, 128, 135, 142,
        148, 155, 161, 166, 169, 175, 123, 115, 110, 107, 111, 119, 125, 132,
        139, 147, 153, 161, 169, 175, 178, 183, 129, 121, 115, 112, 115, 122,
        129, 136, 143, 151, 157, 166, 175, 181, 185, 194, 133, 125, 119, 115,
        118, 125, 131, 138, 145, 153, 160, 169, 178, 185, 189, 193, 138, 129,
        123, 119, 120, 126, 133, 140, 148, 157, 165, 175, 183, 194, 193, 199,
        /* Size 32x32 */
        64, 62, 62, 61, 63, 67, 71, 73, 81, 83, 98, 98, 97, 97, 98, 100, 101,
        104, 104, 109, 109, 114, 115, 120, 123, 127, 129, 130, 133, 135, 138,
        140, 62, 63, 63, 62, 65, 69, 74, 77, 83, 85, 95, 95, 94, 94, 94, 95, 97,
        99, 100, 104, 104, 109, 110, 114, 117, 121, 122, 123, 126, 128, 130,
        132, 62, 63, 63, 63, 65, 70, 74, 78, 83, 85, 95, 95, 94, 93, 93, 95, 96,
        98, 99, 103, 103, 108, 109, 113, 115, 119, 121, 122, 125, 127, 129, 132,
        61, 62, 63, 64, 66, 71, 76, 80, 84, 85, 93, 93, 91, 91, 90, 91, 93, 95,
        95, 99, 99, 104, 105, 109, 111, 115, 116, 117, 120, 122, 124, 125, 63,
        65, 65, 66, 69, 74, 79, 83, 86, 87, 93, 93, 92, 91, 90, 92, 93, 95, 95,
        99, 99, 103, 104, 108, 110, 114, 115, 116, 119, 121, 123, 126, 67, 69,
        70, 71, 74, 78, 83, 87, 89, 90, 95, 94, 93, 92, 91, 92, 93, 95, 95, 98,
        99, 102, 103, 107, 109, 113, 114, 115, 117, 119, 121, 122, 71, 74, 74,
        76, 79, 83, 88, 92, 92, 93, 95, 95, 93, 92, 91, 92, 93, 94, 94, 97, 97,
        101, 102, 105, 107, 110, 112, 113, 115, 117, 119, 123, 73, 77, 78, 80,
        83, 87, 92, 95, 95, 95, 96, 95, 93, 92, 91, 92, 92, 93, 94, 96, 97, 100,
        101, 104, 106, 109, 110, 111, 113, 115, 117, 117, 81, 83, 83, 84, 86,
        89, 92, 95, 96, 97, 100, 99, 98, 98, 97, 98, 98, 99, 99, 102, 102, 105,
        106, 109, 111, 114, 115, 116, 118, 119, 120, 118, 83, 85, 85, 85, 87,
        90, 93, 95, 97, 97, 101, 100, 99, 99, 98, 99, 100, 101, 101, 104, 104,
        107, 107, 111, 112, 115, 116, 117, 119, 121, 123, 127, 98, 95, 95, 93,
        93, 95, 95, 96, 100, 101, 106, 106, 106, 107, 107, 108, 108, 109, 109,
        111, 111, 113, 114, 117, 119, 121, 122, 123, 125, 126, 126, 124, 98, 95,
        95, 93, 93, 94, 95, 95, 99, 100, 106, 106, 107, 107, 108, 108, 108, 109,
        109, 111, 112, 114, 115, 118, 119, 122, 123, 124, 126, 127, 128, 133,
        97, 94, 94, 91, 92, 93, 93, 93, 98, 99, 106, 107, 109, 109, 111, 112,
        113, 114, 115, 117, 117, 120, 121, 123, 125, 127, 129, 129, 131, 133,
        133, 129, 97, 94, 93, 91, 91, 92, 92, 92, 98, 99, 107, 107, 109, 110,
        113, 114, 115, 116, 117, 119, 120, 122, 123, 126, 127, 130, 131, 131,
        133, 133, 134, 140, 98, 94, 93, 90, 90, 91, 91, 91, 97, 98, 107, 108,
        111, 113, 117, 118, 120, 122, 123, 125, 125, 127, 128, 131, 132, 135,
        136, 136, 138, 141, 140, 135, 100, 95, 95, 91, 92, 92, 92, 92, 98, 99,
        108, 108, 112, 114, 118, 120, 121, 124, 125, 127, 128, 131, 131, 134,
        136, 138, 139, 140, 141, 140, 141, 148, 101, 97, 96, 93, 93, 93, 93, 92,
        98, 100, 108, 108, 113, 115, 120, 121, 123, 126, 127, 130, 130, 134,
        135, 137, 139, 142, 143, 143, 145, 148, 148, 142, 104, 99, 98, 95, 95,
        95, 94, 93, 99, 101, 109, 109, 114, 116, 122, 124, 126, 130, 131, 135,
        135, 139, 140, 143, 144, 147, 148, 149, 151, 149, 148, 156, 104, 100,
        99, 95, 95, 95, 94, 94, 99, 101, 109, 109, 115, 117, 123, 125, 127, 131,
        133, 137, 137, 141, 142, 145, 147, 150, 151, 151, 153, 156, 157, 149,
        109, 104, 103, 99, 99, 98, 97, 96, 102, 104, 111, 111, 117, 119, 125,
        127, 130, 135, 137, 141, 141, 146, 147, 150, 152, 156, 157, 157, 159,
        158, 156, 165, 109, 104, 103, 99, 99, 99, 97, 97, 102, 104, 111, 112,
        117, 120, 125, 128, 130, 135, 137, 141, 142, 147, 148, 151, 153, 156,
        157, 158, 160, 162, 165, 157, 114, 109, 108, 104, 103, 102, 101, 100,
        105, 107, 113, 114, 120, 122, 127, 131, 134, 139, 141, 146, 147, 153,
        154, 158, 160, 164, 165, 166, 168, 167, 163, 172, 115, 110, 109, 105,
        104, 103, 102, 101, 106, 107, 114, 115, 121, 123, 128, 131, 135, 140,
        142, 147, 148, 154, 155, 159, 161, 165, 166, 167, 169, 170, 175, 164,
        120, 114, 113, 109, 108, 107, 105, 104, 109, 111, 117, 118, 123, 126,
        131, 134, 137, 143, 145, 150, 151, 158, 159, 164, 166, 170, 172, 172,
        175, 176, 171, 183, 123, 117, 115, 111, 110, 109, 107, 106, 111, 112,
        119, 119, 125, 127, 132, 136, 139, 144, 147, 152, 153, 160, 161, 166,
        169, 173, 175, 176, 178, 177, 183, 173, 127, 121, 119, 115, 114, 113,
        110, 109, 114, 115, 121, 122, 127, 130, 135, 138, 142, 147, 150, 156,
        156, 164, 165, 170, 173, 178, 179, 180, 183, 186, 179, 190, 129, 122,
        121, 116, 115, 114, 112, 110, 115, 116, 122, 123, 129, 131, 136, 139,
        143, 148, 151, 157, 157, 165, 166, 172, 175, 179, 181, 182, 185, 186,
        194, 181, 130, 123, 122, 117, 116, 115, 113, 111, 116, 117, 123, 124,
        129, 131, 136, 140, 143, 149, 151, 157, 158, 166, 167, 172, 176, 180,
        182, 183, 186, 189, 188, 202, 133, 126, 125, 120, 119, 117, 115, 113,
        118, 119, 125, 126, 131, 133, 138, 141, 145, 151, 153, 159, 160, 168,
        169, 175, 178, 183, 185, 186, 189, 188, 193, 190, 135, 128, 127, 122,
        121, 119, 117, 115, 119, 121, 126, 127, 133, 133, 141, 140, 148, 149,
        156, 158, 162, 167, 170, 176, 177, 186, 186, 189, 188, 194, 191, 197,
        138, 130, 129, 124, 123, 121, 119, 117, 120, 123, 126, 128, 133, 134,
        140, 141, 148, 148, 157, 156, 165, 163, 175, 171, 183, 179, 194, 188,
        193, 191, 199, 193, 140, 132, 132, 125, 126, 122, 123, 117, 118, 127,
        124, 133, 129, 140, 135, 148, 142, 156, 149, 165, 157, 172, 164, 183,
        173, 190, 181, 202, 190, 197, 193, 203,
        /* Size 4x8 */
        63, 94, 100, 122, 72, 94, 95, 114, 86, 100, 101, 117, 91, 106, 117, 131,
        95, 108, 132, 148, 104, 113, 141, 165, 115, 121, 150, 180, 123, 127,
        155, 185,
        /* Size 8x4 */
        63, 72, 86, 91, 95, 104, 115, 123, 94, 94, 100, 106, 108, 113, 121, 127,
        100, 95, 101, 117, 132, 141, 150, 155, 122, 114, 117, 131, 148, 165,
        180, 185,
        /* Size 8x16 */
        64, 64, 81, 98, 103, 115, 127, 135, 62, 66, 83, 94, 98, 108, 119, 127,
        63, 70, 86, 92, 95, 103, 114, 121, 71, 79, 92, 93, 94, 101, 110, 117,
        82, 86, 96, 98, 99, 105, 114, 119, 98, 94, 100, 107, 109, 114, 121, 126,
        97, 92, 98, 109, 114, 120, 128, 133, 98, 91, 97, 113, 122, 128, 135,
        141, 101, 93, 99, 115, 126, 134, 142, 148, 104, 96, 100, 117, 131, 142,
        150, 156, 109, 100, 103, 119, 134, 147, 156, 162, 115, 105, 106, 122,
        139, 154, 165, 171, 122, 111, 112, 127, 144, 160, 173, 177, 128, 116,
        116, 130, 147, 165, 179, 185, 132, 119, 119, 133, 150, 168, 183, 188,
        137, 124, 121, 133, 149, 166, 183, 194,
        /* Size 16x8 */
        64, 62, 63, 71, 82, 98, 97, 98, 101, 104, 109, 115, 122, 128, 132, 137,
        64, 66, 70, 79, 86, 94, 92, 91, 93, 96, 100, 105, 111, 116, 119, 124,
        81, 83, 86, 92, 96, 100, 98, 97, 99, 100, 103, 106, 112, 116, 119, 121,
        98, 94, 92, 93, 98, 107, 109, 113, 115, 117, 119, 122, 127, 130, 133,
        133, 103, 98, 95, 94, 99, 109, 114, 122, 126, 131, 134, 139, 144, 147,
        150, 149, 115, 108, 103, 101, 105, 114, 120, 128, 134, 142, 147, 154,
        160, 165, 168, 166, 127, 119, 114, 110, 114, 121, 128, 135, 142, 150,
        156, 165, 173, 179, 183, 183, 135, 127, 121, 117, 119, 126, 133, 141,
        148, 156, 162, 171, 177, 185, 188, 194,
        /* Size 16x32 */
        64, 62, 64, 74, 81, 96, 98, 98, 103, 105, 115, 116, 127, 129, 135, 139,
        63, 63, 66, 77, 83, 94, 94, 94, 99, 100, 109, 110, 120, 122, 128, 131,
        62, 63, 66, 77, 83, 94, 94, 94, 98, 99, 108, 109, 119, 121, 127, 131,
        61, 64, 67, 80, 84, 92, 91, 91, 95, 96, 104, 105, 115, 116, 122, 124,
        63, 67, 70, 82, 86, 93, 92, 91, 95, 96, 103, 104, 114, 115, 121, 125,
        67, 72, 74, 86, 89, 94, 93, 92, 95, 95, 103, 104, 112, 114, 119, 121,
        71, 77, 79, 91, 92, 95, 93, 91, 94, 95, 101, 102, 110, 112, 117, 122,
        74, 80, 83, 94, 95, 95, 93, 91, 93, 94, 100, 101, 109, 110, 115, 116,
        82, 85, 86, 95, 96, 99, 98, 97, 99, 100, 105, 106, 114, 115, 119, 118,
        84, 86, 87, 95, 97, 100, 99, 99, 101, 101, 107, 108, 115, 117, 121, 127,
        98, 93, 94, 96, 100, 106, 107, 107, 109, 109, 114, 114, 121, 122, 126,
        123, 98, 93, 94, 96, 100, 106, 107, 108, 109, 110, 114, 115, 122, 123,
        126, 132, 97, 92, 92, 94, 98, 106, 109, 112, 114, 115, 120, 121, 128,
        129, 133, 129, 97, 91, 92, 93, 98, 106, 110, 113, 116, 117, 122, 123,
        130, 131, 132, 139, 98, 91, 91, 92, 97, 107, 113, 117, 122, 123, 128,
        128, 135, 136, 141, 135, 99, 92, 92, 92, 98, 107, 114, 118, 124, 125,
        131, 132, 138, 139, 140, 147, 101, 93, 93, 93, 99, 108, 115, 119, 126,
        128, 134, 135, 142, 143, 148, 141, 103, 95, 95, 94, 99, 108, 116, 122,
        129, 132, 139, 140, 147, 148, 149, 155, 104, 96, 96, 95, 100, 109, 117,
        122, 131, 133, 142, 142, 150, 151, 156, 149, 108, 100, 99, 97, 102, 111,
        119, 125, 134, 137, 146, 147, 155, 157, 158, 165, 109, 100, 100, 98,
        103, 111, 119, 125, 134, 137, 147, 148, 156, 157, 162, 157, 114, 104,
        104, 101, 105, 113, 121, 128, 138, 141, 153, 154, 164, 165, 167, 172,
        115, 105, 105, 102, 106, 114, 122, 128, 139, 142, 154, 155, 165, 167,
        171, 164, 120, 109, 109, 105, 110, 117, 125, 131, 142, 145, 158, 159,
        170, 172, 176, 182, 122, 112, 111, 107, 112, 119, 127, 133, 144, 147,
        160, 162, 173, 175, 177, 172, 127, 115, 114, 110, 114, 121, 129, 135,
        146, 150, 164, 165, 178, 180, 186, 189, 128, 117, 116, 111, 116, 122,
        130, 136, 147, 151, 165, 167, 179, 181, 185, 180, 129, 118, 117, 112,
        116, 123, 131, 136, 148, 151, 166, 167, 180, 182, 189, 201, 132, 121,
        119, 114, 119, 125, 133, 138, 150, 153, 168, 170, 183, 185, 188, 189,
        135, 123, 122, 116, 120, 127, 133, 140, 149, 155, 167, 170, 185, 185,
        194, 196, 137, 125, 124, 118, 121, 128, 133, 141, 149, 156, 166, 172,
        183, 187, 194, 192, 139, 126, 127, 119, 120, 133, 130, 148, 142, 165,
        157, 183, 173, 200, 189, 202,
        /* Size 32x16 */
        64, 63, 62, 61, 63, 67, 71, 74, 82, 84, 98, 98, 97, 97, 98, 99, 101,
        103, 104, 108, 109, 114, 115, 120, 122, 127, 128, 129, 132, 135, 137,
        139, 62, 63, 63, 64, 67, 72, 77, 80, 85, 86, 93, 93, 92, 91, 91, 92, 93,
        95, 96, 100, 100, 104, 105, 109, 112, 115, 117, 118, 121, 123, 125, 126,
        64, 66, 66, 67, 70, 74, 79, 83, 86, 87, 94, 94, 92, 92, 91, 92, 93, 95,
        96, 99, 100, 104, 105, 109, 111, 114, 116, 117, 119, 122, 124, 127, 74,
        77, 77, 80, 82, 86, 91, 94, 95, 95, 96, 96, 94, 93, 92, 92, 93, 94, 95,
        97, 98, 101, 102, 105, 107, 110, 111, 112, 114, 116, 118, 119, 81, 83,
        83, 84, 86, 89, 92, 95, 96, 97, 100, 100, 98, 98, 97, 98, 99, 99, 100,
        102, 103, 105, 106, 110, 112, 114, 116, 116, 119, 120, 121, 120, 96, 94,
        94, 92, 93, 94, 95, 95, 99, 100, 106, 106, 106, 106, 107, 107, 108, 108,
        109, 111, 111, 113, 114, 117, 119, 121, 122, 123, 125, 127, 128, 133,
        98, 94, 94, 91, 92, 93, 93, 93, 98, 99, 107, 107, 109, 110, 113, 114,
        115, 116, 117, 119, 119, 121, 122, 125, 127, 129, 130, 131, 133, 133,
        133, 130, 98, 94, 94, 91, 91, 92, 91, 91, 97, 99, 107, 108, 112, 113,
        117, 118, 119, 122, 122, 125, 125, 128, 128, 131, 133, 135, 136, 136,
        138, 140, 141, 148, 103, 99, 98, 95, 95, 95, 94, 93, 99, 101, 109, 109,
        114, 116, 122, 124, 126, 129, 131, 134, 134, 138, 139, 142, 144, 146,
        147, 148, 150, 149, 149, 142, 105, 100, 99, 96, 96, 95, 95, 94, 100,
        101, 109, 110, 115, 117, 123, 125, 128, 132, 133, 137, 137, 141, 142,
        145, 147, 150, 151, 151, 153, 155, 156, 165, 115, 109, 108, 104, 103,
        103, 101, 100, 105, 107, 114, 114, 120, 122, 128, 131, 134, 139, 142,
        146, 147, 153, 154, 158, 160, 164, 165, 166, 168, 167, 166, 157, 116,
        110, 109, 105, 104, 104, 102, 101, 106, 108, 114, 115, 121, 123, 128,
        132, 135, 140, 142, 147, 148, 154, 155, 159, 162, 165, 167, 167, 170,
        170, 172, 183, 127, 120, 119, 115, 114, 112, 110, 109, 114, 115, 121,
        122, 128, 130, 135, 138, 142, 147, 150, 155, 156, 164, 165, 170, 173,
        178, 179, 180, 183, 185, 183, 173, 129, 122, 121, 116, 115, 114, 112,
        110, 115, 117, 122, 123, 129, 131, 136, 139, 143, 148, 151, 157, 157,
        165, 167, 172, 175, 180, 181, 182, 185, 185, 187, 200, 135, 128, 127,
        122, 121, 119, 117, 115, 119, 121, 126, 126, 133, 132, 141, 140, 148,
        149, 156, 158, 162, 167, 171, 176, 177, 186, 185, 189, 188, 194, 194,
        189, 139, 131, 131, 124, 125, 121, 122, 116, 118, 127, 123, 132, 129,
        139, 135, 147, 141, 155, 149, 165, 157, 172, 164, 182, 172, 189, 180,
        201, 189, 196, 192, 202,
        /* Size 4x16 */
        62, 96, 105, 129, 63, 94, 99, 121, 67, 93, 96, 115, 77, 95, 95, 112, 85,
        99, 100, 115, 93, 106, 109, 122, 92, 106, 115, 129, 91, 107, 123, 136,
        93, 108, 128, 143, 96, 109, 133, 151, 100, 111, 137, 157, 105, 114, 142,
        167, 112, 119, 147, 175, 117, 122, 151, 181, 121, 125, 153, 185, 125,
        128, 156, 187,
        /* Size 16x4 */
        62, 63, 67, 77, 85, 93, 92, 91, 93, 96, 100, 105, 112, 117, 121, 125,
        96, 94, 93, 95, 99, 106, 106, 107, 108, 109, 111, 114, 119, 122, 125,
        128, 105, 99, 96, 95, 100, 109, 115, 123, 128, 133, 137, 142, 147, 151,
        153, 156, 129, 121, 115, 112, 115, 122, 129, 136, 143, 151, 157, 167,
        175, 181, 185, 187,
        /* Size 8x32 */
        64, 64, 81, 98, 103, 115, 127, 135, 63, 66, 83, 94, 99, 109, 120, 128,
        62, 66, 83, 94, 98, 108, 119, 127, 61, 67, 84, 91, 95, 104, 115, 122,
        63, 70, 86, 92, 95, 103, 114, 121, 67, 74, 89, 93, 95, 103, 112, 119,
        71, 79, 92, 93, 94, 101, 110, 117, 74, 83, 95, 93, 93, 100, 109, 115,
        82, 86, 96, 98, 99, 105, 114, 119, 84, 87, 97, 99, 101, 107, 115, 121,
        98, 94, 100, 107, 109, 114, 121, 126, 98, 94, 100, 107, 109, 114, 122,
        126, 97, 92, 98, 109, 114, 120, 128, 133, 97, 92, 98, 110, 116, 122,
        130, 132, 98, 91, 97, 113, 122, 128, 135, 141, 99, 92, 98, 114, 124,
        131, 138, 140, 101, 93, 99, 115, 126, 134, 142, 148, 103, 95, 99, 116,
        129, 139, 147, 149, 104, 96, 100, 117, 131, 142, 150, 156, 108, 99, 102,
        119, 134, 146, 155, 158, 109, 100, 103, 119, 134, 147, 156, 162, 114,
        104, 105, 121, 138, 153, 164, 167, 115, 105, 106, 122, 139, 154, 165,
        171, 120, 109, 110, 125, 142, 158, 170, 176, 122, 111, 112, 127, 144,
        160, 173, 177, 127, 114, 114, 129, 146, 164, 178, 186, 128, 116, 116,
        130, 147, 165, 179, 185, 129, 117, 116, 131, 148, 166, 180, 189, 132,
        119, 119, 133, 150, 168, 183, 188, 135, 122, 120, 133, 149, 167, 185,
        194, 137, 124, 121, 133, 149, 166, 183, 194, 139, 127, 120, 130, 142,
        157, 173, 189,
        /* Size 32x8 */
        64, 63, 62, 61, 63, 67, 71, 74, 82, 84, 98, 98, 97, 97, 98, 99, 101,
        103, 104, 108, 109, 114, 115, 120, 122, 127, 128, 129, 132, 135, 137,
        139, 64, 66, 66, 67, 70, 74, 79, 83, 86, 87, 94, 94, 92, 92, 91, 92, 93,
        95, 96, 99, 100, 104, 105, 109, 111, 114, 116, 117, 119, 122, 124, 127,
        81, 83, 83, 84, 86, 89, 92, 95, 96, 97, 100, 100, 98, 98, 97, 98, 99,
        99, 100, 102, 103, 105, 106, 110, 112, 114, 116, 116, 119, 120, 121,
        120, 98, 94, 94, 91, 92, 93, 93, 93, 98, 99, 107, 107, 109, 110, 113,
        114, 115, 116, 117, 119, 119, 121, 122, 125, 127, 129, 130, 131, 133,
        133, 133, 130, 103, 99, 98, 95, 95, 95, 94, 93, 99, 101, 109, 109, 114,
        116, 122, 124, 126, 129, 131, 134, 134, 138, 139, 142, 144, 146, 147,
        148, 150, 149, 149, 142, 115, 109, 108, 104, 103, 103, 101, 100, 105,
        107, 114, 114, 120, 122, 128, 131, 134, 139, 142, 146, 147, 153, 154,
        158, 160, 164, 165, 166, 168, 167, 166, 157, 127, 120, 119, 115, 114,
        112, 110, 109, 114, 115, 121, 122, 128, 130, 135, 138, 142, 147, 150,
        155, 156, 164, 165, 170, 173, 178, 179, 180, 183, 185, 183, 173, 135,
        128, 127, 122, 121, 119, 117, 115, 119, 121, 126, 126, 133, 132, 141,
        140, 148, 149, 156, 158, 162, 167, 171, 176, 177, 186, 185, 189, 188,
        194, 194, 189 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        71, 116, 116, 210,
#endif
        /* Size 4x4 */
        65, 69, 99, 145, 69, 97, 120, 158, 99, 120, 165, 208, 145, 158, 208,
        268,
        /* Size 8x8 */
        64, 64, 68, 77, 93, 113, 136, 157, 64, 67, 71, 78, 91, 108, 129, 148,
        68, 71, 79, 90, 102, 117, 136, 153, 77, 78, 90, 108, 122, 138, 156, 173,
        93, 91, 102, 122, 142, 161, 181, 198, 113, 108, 117, 138, 161, 184, 207,
        226, 136, 129, 136, 156, 181, 207, 235, 256, 157, 148, 153, 173, 198,
        226, 256, 280,
        /* Size 16x16 */
        64, 63, 63, 63, 64, 68, 72, 79, 88, 97, 108, 118, 131, 143, 160, 166,
        63, 64, 64, 64, 65, 68, 70, 77, 85, 93, 103, 113, 125, 136, 152, 157,
        63, 64, 65, 65, 65, 67, 69, 75, 82, 89, 99, 108, 119, 130, 145, 150, 63,
        64, 65, 67, 68, 71, 73, 78, 84, 91, 100, 108, 118, 129, 143, 148, 64,
        65, 65, 68, 71, 74, 77, 81, 85, 92, 99, 107, 116, 126, 139, 144, 68, 68,
        67, 71, 74, 79, 85, 90, 95, 102, 109, 117, 126, 136, 148, 153, 72, 70,
        69, 73, 77, 85, 97, 101, 108, 114, 120, 128, 136, 146, 158, 162, 79, 77,
        75, 78, 81, 90, 101, 108, 116, 122, 130, 138, 147, 156, 168, 173, 88,
        85, 82, 84, 85, 95, 108, 116, 127, 134, 143, 151, 159, 169, 181, 185,
        97, 93, 89, 91, 92, 102, 114, 122, 134, 142, 153, 161, 171, 181, 193,
        198, 108, 103, 99, 100, 99, 109, 120, 130, 143, 153, 165, 174, 185, 195,
        208, 213, 118, 113, 108, 108, 107, 117, 128, 138, 151, 161, 174, 184,
        196, 207, 221, 226, 131, 125, 119, 118, 116, 126, 136, 147, 159, 171,
        185, 196, 210, 222, 237, 242, 143, 136, 130, 129, 126, 136, 146, 156,
        169, 181, 195, 207, 222, 235, 251, 256, 160, 152, 145, 143, 139, 148,
        158, 168, 181, 193, 208, 221, 237, 251, 268, 274, 166, 157, 150, 148,
        144, 153, 162, 173, 185, 198, 213, 226, 242, 256, 274, 280,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 63, 63, 64, 64, 68, 68, 72, 72, 79, 79, 88, 88, 97,
        97, 108, 108, 118, 118, 131, 131, 143, 143, 160, 160, 166, 166, 175, 63,
        64, 64, 64, 64, 64, 64, 65, 65, 68, 68, 70, 70, 77, 77, 85, 85, 93, 93,
        103, 103, 113, 113, 125, 125, 136, 136, 152, 152, 157, 157, 166, 63, 64,
        64, 64, 64, 64, 64, 65, 65, 68, 68, 70, 70, 77, 77, 85, 85, 93, 93, 103,
        103, 113, 113, 125, 125, 136, 136, 152, 152, 157, 157, 166, 63, 64, 64,
        65, 65, 65, 65, 65, 65, 67, 67, 69, 69, 75, 75, 82, 82, 89, 89, 99, 99,
        108, 108, 119, 119, 130, 130, 145, 145, 150, 150, 158, 63, 64, 64, 65,
        65, 65, 65, 65, 65, 67, 67, 69, 69, 75, 75, 82, 82, 89, 89, 99, 99, 108,
        108, 119, 119, 130, 130, 145, 145, 150, 150, 158, 63, 64, 64, 65, 65,
        67, 67, 68, 68, 71, 71, 73, 73, 78, 78, 84, 84, 91, 91, 100, 100, 108,
        108, 118, 118, 129, 129, 143, 143, 148, 148, 155, 63, 64, 64, 65, 65,
        67, 67, 68, 68, 71, 71, 73, 73, 78, 78, 84, 84, 91, 91, 100, 100, 108,
        108, 118, 118, 129, 129, 143, 143, 148, 148, 155, 64, 65, 65, 65, 65,
        68, 68, 71, 71, 74, 74, 77, 77, 81, 81, 85, 85, 92, 92, 99, 99, 107,
        107, 116, 116, 126, 126, 139, 139, 144, 144, 151, 64, 65, 65, 65, 65,
        68, 68, 71, 71, 74, 74, 77, 77, 81, 81, 85, 85, 92, 92, 99, 99, 107,
        107, 116, 116, 126, 126, 139, 139, 144, 144, 151, 68, 68, 68, 67, 67,
        71, 71, 74, 74, 79, 79, 85, 85, 90, 90, 95, 95, 102, 102, 109, 109, 117,
        117, 126, 126, 136, 136, 148, 148, 153, 153, 160, 68, 68, 68, 67, 67,
        71, 71, 74, 74, 79, 79, 85, 85, 90, 90, 95, 95, 102, 102, 109, 109, 117,
        117, 126, 126, 136, 136, 148, 148, 153, 153, 160, 72, 70, 70, 69, 69,
        73, 73, 77, 77, 85, 85, 97, 97, 101, 101, 108, 108, 114, 114, 120, 120,
        128, 128, 136, 136, 146, 146, 158, 158, 162, 162, 169, 72, 70, 70, 69,
        69, 73, 73, 77, 77, 85, 85, 97, 97, 101, 101, 108, 108, 114, 114, 120,
        120, 128, 128, 136, 136, 146, 146, 158, 158, 162, 162, 169, 79, 77, 77,
        75, 75, 78, 78, 81, 81, 90, 90, 101, 101, 108, 108, 116, 116, 122, 122,
        130, 130, 138, 138, 147, 147, 156, 156, 168, 168, 173, 173, 180, 79, 77,
        77, 75, 75, 78, 78, 81, 81, 90, 90, 101, 101, 108, 108, 116, 116, 122,
        122, 130, 130, 138, 138, 147, 147, 156, 156, 168, 168, 173, 173, 180,
        88, 85, 85, 82, 82, 84, 84, 85, 85, 95, 95, 108, 108, 116, 116, 127,
        127, 134, 134, 143, 143, 151, 151, 159, 159, 169, 169, 181, 181, 185,
        185, 192, 88, 85, 85, 82, 82, 84, 84, 85, 85, 95, 95, 108, 108, 116,
        116, 127, 127, 134, 134, 143, 143, 151, 151, 159, 159, 169, 169, 181,
        181, 185, 185, 192, 97, 93, 93, 89, 89, 91, 91, 92, 92, 102, 102, 114,
        114, 122, 122, 134, 134, 142, 142, 153, 153, 161, 161, 171, 171, 181,
        181, 193, 193, 198, 198, 204, 97, 93, 93, 89, 89, 91, 91, 92, 92, 102,
        102, 114, 114, 122, 122, 134, 134, 142, 142, 153, 153, 161, 161, 171,
        171, 181, 181, 193, 193, 198, 198, 204, 108, 103, 103, 99, 99, 100, 100,
        99, 99, 109, 109, 120, 120, 130, 130, 143, 143, 153, 153, 165, 165, 174,
        174, 185, 185, 195, 195, 208, 208, 213, 213, 219, 108, 103, 103, 99, 99,
        100, 100, 99, 99, 109, 109, 120, 120, 130, 130, 143, 143, 153, 153, 165,
        165, 174, 174, 185, 185, 195, 195, 208, 208, 213, 213, 219, 118, 113,
        113, 108, 108, 108, 108, 107, 107, 117, 117, 128, 128, 138, 138, 151,
        151, 161, 161, 174, 174, 184, 184, 196, 196, 207, 207, 221, 221, 226,
        226, 233, 118, 113, 113, 108, 108, 108, 108, 107, 107, 117, 117, 128,
        128, 138, 138, 151, 151, 161, 161, 174, 174, 184, 184, 196, 196, 207,
        207, 221, 221, 226, 226, 233, 131, 125, 125, 119, 119, 118, 118, 116,
        116, 126, 126, 136, 136, 147, 147, 159, 159, 171, 171, 185, 185, 196,
        196, 210, 210, 222, 222, 237, 237, 242, 242, 249, 131, 125, 125, 119,
        119, 118, 118, 116, 116, 126, 126, 136, 136, 147, 147, 159, 159, 171,
        171, 185, 185, 196, 196, 210, 210, 222, 222, 237, 237, 242, 242, 249,
        143, 136, 136, 130, 130, 129, 129, 126, 126, 136, 136, 146, 146, 156,
        156, 169, 169, 181, 181, 195, 195, 207, 207, 222, 222, 235, 235, 251,
        251, 256, 256, 264, 143, 136, 136, 130, 130, 129, 129, 126, 126, 136,
        136, 146, 146, 156, 156, 169, 169, 181, 181, 195, 195, 207, 207, 222,
        222, 235, 235, 251, 251, 256, 256, 264, 160, 152, 152, 145, 145, 143,
        143, 139, 139, 148, 148, 158, 158, 168, 168, 181, 181, 193, 193, 208,
        208, 221, 221, 237, 237, 251, 251, 268, 268, 274, 274, 283, 160, 152,
        152, 145, 145, 143, 143, 139, 139, 148, 148, 158, 158, 168, 168, 181,
        181, 193, 193, 208, 208, 221, 221, 237, 237, 251, 251, 268, 268, 274,
        274, 283, 166, 157, 157, 150, 150, 148, 148, 144, 144, 153, 153, 162,
        162, 173, 173, 185, 185, 198, 198, 213, 213, 226, 226, 242, 242, 256,
        256, 274, 274, 280, 280, 289, 166, 157, 157, 150, 150, 148, 148, 144,
        144, 153, 153, 162, 162, 173, 173, 185, 185, 198, 198, 213, 213, 226,
        226, 242, 242, 256, 256, 274, 274, 280, 280, 289, 175, 166, 166, 158,
        158, 155, 155, 151, 151, 160, 160, 169, 169, 180, 180, 192, 192, 204,
        204, 219, 219, 233, 233, 249, 249, 264, 264, 283, 283, 289, 289, 299,
        /* Size 4x8 */
        64, 71, 103, 151, 65, 73, 100, 142, 68, 85, 109, 147, 75, 101, 130, 168,
        90, 113, 152, 192, 108, 127, 174, 220, 131, 146, 195, 250, 151, 162,
        212, 273,
        /* Size 8x4 */
        64, 65, 68, 75, 90, 108, 131, 151, 71, 73, 85, 101, 113, 127, 146, 162,
        103, 100, 109, 130, 152, 174, 195, 212, 151, 142, 147, 168, 192, 220,
        250, 273,
        /* Size 8x16 */
        64, 63, 64, 72, 88, 107, 131, 158, 63, 64, 65, 71, 85, 103, 124, 151,
        63, 65, 66, 69, 82, 99, 119, 144, 64, 65, 68, 73, 84, 100, 118, 142, 64,
        66, 71, 77, 85, 99, 116, 138, 68, 68, 74, 85, 96, 109, 126, 147, 72, 69,
        77, 96, 108, 120, 136, 157, 79, 75, 81, 101, 116, 130, 147, 168, 88, 82,
        86, 107, 126, 143, 159, 180, 96, 90, 93, 113, 134, 152, 171, 192, 107,
        99, 100, 120, 142, 164, 185, 207, 117, 108, 108, 127, 150, 174, 196,
        220, 130, 120, 117, 136, 159, 185, 210, 236, 142, 131, 127, 146, 169,
        195, 222, 250, 159, 145, 140, 158, 181, 208, 236, 267, 165, 151, 145,
        162, 185, 212, 242, 273,
        /* Size 16x8 */
        64, 63, 63, 64, 64, 68, 72, 79, 88, 96, 107, 117, 130, 142, 159, 165,
        63, 64, 65, 65, 66, 68, 69, 75, 82, 90, 99, 108, 120, 131, 145, 151, 64,
        65, 66, 68, 71, 74, 77, 81, 86, 93, 100, 108, 117, 127, 140, 145, 72,
        71, 69, 73, 77, 85, 96, 101, 107, 113, 120, 127, 136, 146, 158, 162, 88,
        85, 82, 84, 85, 96, 108, 116, 126, 134, 142, 150, 159, 169, 181, 185,
        107, 103, 99, 100, 99, 109, 120, 130, 143, 152, 164, 174, 185, 195, 208,
        212, 131, 124, 119, 118, 116, 126, 136, 147, 159, 171, 185, 196, 210,
        222, 236, 242, 158, 151, 144, 142, 138, 147, 157, 168, 180, 192, 207,
        220, 236, 250, 267, 273,
        /* Size 16x32 */
        64, 63, 63, 64, 64, 72, 72, 88, 88, 107, 107, 131, 131, 158, 158, 174,
        63, 64, 64, 65, 65, 71, 71, 85, 85, 103, 103, 124, 124, 151, 151, 165,
        63, 64, 64, 65, 65, 71, 71, 85, 85, 103, 103, 124, 124, 151, 151, 165,
        63, 65, 65, 66, 66, 69, 69, 82, 82, 99, 99, 119, 119, 144, 144, 157, 63,
        65, 65, 66, 66, 69, 69, 82, 82, 99, 99, 119, 119, 144, 144, 157, 64, 65,
        65, 68, 68, 73, 73, 84, 84, 100, 100, 118, 118, 142, 142, 154, 64, 65,
        65, 68, 68, 73, 73, 84, 84, 100, 100, 118, 118, 142, 142, 154, 64, 66,
        66, 71, 71, 77, 77, 85, 85, 99, 99, 116, 116, 138, 138, 150, 64, 66, 66,
        71, 71, 77, 77, 85, 85, 99, 99, 116, 116, 138, 138, 150, 68, 68, 68, 74,
        74, 85, 85, 96, 96, 109, 109, 126, 126, 147, 147, 159, 68, 68, 68, 74,
        74, 85, 85, 96, 96, 109, 109, 126, 126, 147, 147, 159, 72, 69, 69, 77,
        77, 96, 96, 108, 108, 120, 120, 136, 136, 157, 157, 168, 72, 69, 69, 77,
        77, 96, 96, 108, 108, 120, 120, 136, 136, 157, 157, 168, 79, 75, 75, 81,
        81, 101, 101, 116, 116, 130, 130, 147, 147, 168, 168, 179, 79, 75, 75,
        81, 81, 101, 101, 116, 116, 130, 130, 147, 147, 168, 168, 179, 88, 82,
        82, 86, 86, 107, 107, 126, 126, 143, 143, 159, 159, 180, 180, 191, 88,
        82, 82, 86, 86, 107, 107, 126, 126, 143, 143, 159, 159, 180, 180, 191,
        96, 90, 90, 93, 93, 113, 113, 134, 134, 152, 152, 171, 171, 192, 192,
        204, 96, 90, 90, 93, 93, 113, 113, 134, 134, 152, 152, 171, 171, 192,
        192, 204, 107, 99, 99, 100, 100, 120, 120, 142, 142, 164, 164, 185, 185,
        207, 207, 219, 107, 99, 99, 100, 100, 120, 120, 142, 142, 164, 164, 185,
        185, 207, 207, 219, 117, 108, 108, 108, 108, 127, 127, 150, 150, 174,
        174, 196, 196, 220, 220, 232, 117, 108, 108, 108, 108, 127, 127, 150,
        150, 174, 174, 196, 196, 220, 220, 232, 130, 120, 120, 117, 117, 136,
        136, 159, 159, 185, 185, 210, 210, 236, 236, 249, 130, 120, 120, 117,
        117, 136, 136, 159, 159, 185, 185, 210, 210, 236, 236, 249, 142, 131,
        131, 127, 127, 146, 146, 169, 169, 195, 195, 222, 222, 250, 250, 264,
        142, 131, 131, 127, 127, 146, 146, 169, 169, 195, 195, 222, 222, 250,
        250, 264, 159, 145, 145, 140, 140, 158, 158, 181, 181, 208, 208, 236,
        236, 267, 267, 282, 159, 145, 145, 140, 140, 158, 158, 181, 181, 208,
        208, 236, 236, 267, 267, 282, 165, 151, 151, 145, 145, 162, 162, 185,
        185, 212, 212, 242, 242, 273, 273, 289, 165, 151, 151, 145, 145, 162,
        162, 185, 185, 212, 212, 242, 242, 273, 273, 289, 174, 159, 159, 152,
        152, 169, 169, 192, 192, 219, 219, 249, 249, 282, 282, 298,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 64, 64, 64, 64, 68, 68, 72, 72, 79, 79, 88, 88, 96,
        96, 107, 107, 117, 117, 130, 130, 142, 142, 159, 159, 165, 165, 174, 63,
        64, 64, 65, 65, 65, 65, 66, 66, 68, 68, 69, 69, 75, 75, 82, 82, 90, 90,
        99, 99, 108, 108, 120, 120, 131, 131, 145, 145, 151, 151, 159, 63, 64,
        64, 65, 65, 65, 65, 66, 66, 68, 68, 69, 69, 75, 75, 82, 82, 90, 90, 99,
        99, 108, 108, 120, 120, 131, 131, 145, 145, 151, 151, 159, 64, 65, 65,
        66, 66, 68, 68, 71, 71, 74, 74, 77, 77, 81, 81, 86, 86, 93, 93, 100,
        100, 108, 108, 117, 117, 127, 127, 140, 140, 145, 145, 152, 64, 65, 65,
        66, 66, 68, 68, 71, 71, 74, 74, 77, 77, 81, 81, 86, 86, 93, 93, 100,
        100, 108, 108, 117, 117, 127, 127, 140, 140, 145, 145, 152, 72, 71, 71,
        69, 69, 73, 73, 77, 77, 85, 85, 96, 96, 101, 101, 107, 107, 113, 113,
        120, 120, 127, 127, 136, 136, 146, 146, 158, 158, 162, 162, 169, 72, 71,
        71, 69, 69, 73, 73, 77, 77, 85, 85, 96, 96, 101, 101, 107, 107, 113,
        113, 120, 120, 127, 127, 136, 136, 146, 146, 158, 158, 162, 162, 169,
        88, 85, 85, 82, 82, 84, 84, 85, 85, 96, 96, 108, 108, 116, 116, 126,
        126, 134, 134, 142, 142, 150, 150, 159, 159, 169, 169, 181, 181, 185,
        185, 192, 88, 85, 85, 82, 82, 84, 84, 85, 85, 96, 96, 108, 108, 116,
        116, 126, 126, 134, 134, 142, 142, 150, 150, 159, 159, 169, 169, 181,
        181, 185, 185, 192, 107, 103, 103, 99, 99, 100, 100, 99, 99, 109, 109,
        120, 120, 130, 130, 143, 143, 152, 152, 164, 164, 174, 174, 185, 185,
        195, 195, 208, 208, 212, 212, 219, 107, 103, 103, 99, 99, 100, 100, 99,
        99, 109, 109, 120, 120, 130, 130, 143, 143, 152, 152, 164, 164, 174,
        174, 185, 185, 195, 195, 208, 208, 212, 212, 219, 131, 124, 124, 119,
        119, 118, 118, 116, 116, 126, 126, 136, 136, 147, 147, 159, 159, 171,
        171, 185, 185, 196, 196, 210, 210, 222, 222, 236, 236, 242, 242, 249,
        131, 124, 124, 119, 119, 118, 118, 116, 116, 126, 126, 136, 136, 147,
        147, 159, 159, 171, 171, 185, 185, 196, 196, 210, 210, 222, 222, 236,
        236, 242, 242, 249, 158, 151, 151, 144, 144, 142, 142, 138, 138, 147,
        147, 157, 157, 168, 168, 180, 180, 192, 192, 207, 207, 220, 220, 236,
        236, 250, 250, 267, 267, 273, 273, 282, 158, 151, 151, 144, 144, 142,
        142, 138, 138, 147, 147, 157, 157, 168, 168, 180, 180, 192, 192, 207,
        207, 220, 220, 236, 236, 250, 250, 267, 267, 273, 273, 282, 174, 165,
        165, 157, 157, 154, 154, 150, 150, 159, 159, 168, 168, 179, 179, 191,
        191, 204, 204, 219, 219, 232, 232, 249, 249, 264, 264, 282, 282, 289,
        289, 298,
        /* Size 4x16 */
        63, 72, 107, 158, 64, 71, 103, 151, 65, 69, 99, 144, 65, 73, 100, 142,
        66, 77, 99, 138, 68, 85, 109, 147, 69, 96, 120, 157, 75, 101, 130, 168,
        82, 107, 143, 180, 90, 113, 152, 192, 99, 120, 164, 207, 108, 127, 174,
        220, 120, 136, 185, 236, 131, 146, 195, 250, 145, 158, 208, 267, 151,
        162, 212, 273,
        /* Size 16x4 */
        63, 64, 65, 65, 66, 68, 69, 75, 82, 90, 99, 108, 120, 131, 145, 151, 72,
        71, 69, 73, 77, 85, 96, 101, 107, 113, 120, 127, 136, 146, 158, 162,
        107, 103, 99, 100, 99, 109, 120, 130, 143, 152, 164, 174, 185, 195, 208,
        212, 158, 151, 144, 142, 138, 147, 157, 168, 180, 192, 207, 220, 236,
        250, 267, 273,
        /* Size 8x32 */
        64, 63, 64, 72, 88, 107, 131, 158, 63, 64, 65, 71, 85, 103, 124, 151,
        63, 64, 65, 71, 85, 103, 124, 151, 63, 65, 66, 69, 82, 99, 119, 144, 63,
        65, 66, 69, 82, 99, 119, 144, 64, 65, 68, 73, 84, 100, 118, 142, 64, 65,
        68, 73, 84, 100, 118, 142, 64, 66, 71, 77, 85, 99, 116, 138, 64, 66, 71,
        77, 85, 99, 116, 138, 68, 68, 74, 85, 96, 109, 126, 147, 68, 68, 74, 85,
        96, 109, 126, 147, 72, 69, 77, 96, 108, 120, 136, 157, 72, 69, 77, 96,
        108, 120, 136, 157, 79, 75, 81, 101, 116, 130, 147, 168, 79, 75, 81,
        101, 116, 130, 147, 168, 88, 82, 86, 107, 126, 143, 159, 180, 88, 82,
        86, 107, 126, 143, 159, 180, 96, 90, 93, 113, 134, 152, 171, 192, 96,
        90, 93, 113, 134, 152, 171, 192, 107, 99, 100, 120, 142, 164, 185, 207,
        107, 99, 100, 120, 142, 164, 185, 207, 117, 108, 108, 127, 150, 174,
        196, 220, 117, 108, 108, 127, 150, 174, 196, 220, 130, 120, 117, 136,
        159, 185, 210, 236, 130, 120, 117, 136, 159, 185, 210, 236, 142, 131,
        127, 146, 169, 195, 222, 250, 142, 131, 127, 146, 169, 195, 222, 250,
        159, 145, 140, 158, 181, 208, 236, 267, 159, 145, 140, 158, 181, 208,
        236, 267, 165, 151, 145, 162, 185, 212, 242, 273, 165, 151, 145, 162,
        185, 212, 242, 273, 174, 159, 152, 169, 192, 219, 249, 282,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 64, 64, 64, 64, 68, 68, 72, 72, 79, 79, 88, 88, 96,
        96, 107, 107, 117, 117, 130, 130, 142, 142, 159, 159, 165, 165, 174, 63,
        64, 64, 65, 65, 65, 65, 66, 66, 68, 68, 69, 69, 75, 75, 82, 82, 90, 90,
        99, 99, 108, 108, 120, 120, 131, 131, 145, 145, 151, 151, 159, 64, 65,
        65, 66, 66, 68, 68, 71, 71, 74, 74, 77, 77, 81, 81, 86, 86, 93, 93, 100,
        100, 108, 108, 117, 117, 127, 127, 140, 140, 145, 145, 152, 72, 71, 71,
        69, 69, 73, 73, 77, 77, 85, 85, 96, 96, 101, 101, 107, 107, 113, 113,
        120, 120, 127, 127, 136, 136, 146, 146, 158, 158, 162, 162, 169, 88, 85,
        85, 82, 82, 84, 84, 85, 85, 96, 96, 108, 108, 116, 116, 126, 126, 134,
        134, 142, 142, 150, 150, 159, 159, 169, 169, 181, 181, 185, 185, 192,
        107, 103, 103, 99, 99, 100, 100, 99, 99, 109, 109, 120, 120, 130, 130,
        143, 143, 152, 152, 164, 164, 174, 174, 185, 185, 195, 195, 208, 208,
        212, 212, 219, 131, 124, 124, 119, 119, 118, 118, 116, 116, 126, 126,
        136, 136, 147, 147, 159, 159, 171, 171, 185, 185, 196, 196, 210, 210,
        222, 222, 236, 236, 242, 242, 249, 158, 151, 151, 144, 144, 142, 142,
        138, 138, 147, 147, 157, 157, 168, 168, 180, 180, 192, 192, 207, 207,
        220, 220, 236, 236, 250, 250, 267, 267, 273, 273, 282 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        95, 100, 100, 153,
#endif
        /* Size 4x4 */
        64, 93, 95, 115, 93, 106, 109, 121, 95, 109, 133, 150, 115, 121, 150,
        178,
        /* Size 8x8 */
        63, 69, 85, 94, 97, 104, 114, 123, 69, 78, 90, 93, 93, 99, 107, 115, 85,
        90, 97, 99, 100, 104, 111, 117, 94, 93, 99, 109, 113, 117, 123, 129, 97,
        93, 100, 113, 123, 130, 137, 143, 104, 99, 104, 117, 130, 142, 151, 158,
        114, 107, 111, 123, 137, 151, 164, 172, 123, 115, 117, 129, 143, 158,
        172, 183,
        /* Size 16x16 */
        64, 62, 61, 67, 73, 83, 98, 97, 98, 101, 104, 109, 114, 120, 127, 130,
        62, 63, 62, 69, 77, 85, 95, 94, 94, 97, 100, 104, 109, 114, 121, 123,
        61, 62, 64, 71, 80, 85, 93, 91, 90, 93, 95, 99, 104, 109, 115, 117, 67,
        69, 71, 78, 87, 90, 95, 93, 91, 93, 95, 99, 102, 107, 113, 115, 73, 77,
        80, 87, 95, 95, 96, 93, 91, 92, 94, 97, 100, 104, 109, 111, 83, 85, 85,
        90, 95, 97, 101, 99, 98, 100, 101, 104, 107, 111, 115, 117, 98, 95, 93,
        95, 96, 101, 106, 106, 107, 108, 109, 111, 113, 117, 121, 123, 97, 94,
        91, 93, 93, 99, 106, 109, 111, 113, 115, 117, 120, 123, 127, 129, 98,
        94, 90, 91, 91, 98, 107, 111, 117, 120, 123, 125, 127, 131, 135, 136,
        101, 97, 93, 93, 92, 100, 108, 113, 120, 123, 127, 130, 134, 137, 142,
        143, 104, 100, 95, 95, 94, 101, 109, 115, 123, 127, 133, 137, 141, 145,
        150, 151, 109, 104, 99, 99, 97, 104, 111, 117, 125, 130, 137, 142, 147,
        151, 156, 158, 114, 109, 104, 102, 100, 107, 113, 120, 127, 134, 141,
        147, 153, 158, 164, 166, 120, 114, 109, 107, 104, 111, 117, 123, 131,
        137, 145, 151, 158, 164, 170, 172, 127, 121, 115, 113, 109, 115, 121,
        127, 135, 142, 150, 156, 164, 170, 178, 180, 130, 123, 117, 115, 111,
        117, 123, 129, 136, 143, 151, 158, 166, 172, 180, 183,
        /* Size 32x32 */
        64, 62, 62, 61, 61, 67, 67, 73, 73, 83, 83, 98, 98, 97, 97, 98, 98, 101,
        101, 104, 104, 109, 109, 114, 114, 120, 120, 127, 127, 130, 130, 134,
        62, 63, 63, 62, 62, 69, 69, 77, 77, 85, 85, 95, 95, 94, 94, 94, 94, 97,
        97, 100, 100, 104, 104, 109, 109, 114, 114, 121, 121, 123, 123, 127, 62,
        63, 63, 62, 62, 69, 69, 77, 77, 85, 85, 95, 95, 94, 94, 94, 94, 97, 97,
        100, 100, 104, 104, 109, 109, 114, 114, 121, 121, 123, 123, 127, 61, 62,
        62, 64, 64, 71, 71, 80, 80, 85, 85, 93, 93, 91, 91, 90, 90, 93, 93, 95,
        95, 99, 99, 104, 104, 109, 109, 115, 115, 117, 117, 121, 61, 62, 62, 64,
        64, 71, 71, 80, 80, 85, 85, 93, 93, 91, 91, 90, 90, 93, 93, 95, 95, 99,
        99, 104, 104, 109, 109, 115, 115, 117, 117, 121, 67, 69, 69, 71, 71, 78,
        78, 87, 87, 90, 90, 95, 95, 93, 93, 91, 91, 93, 93, 95, 95, 99, 99, 102,
        102, 107, 107, 113, 113, 115, 115, 118, 67, 69, 69, 71, 71, 78, 78, 87,
        87, 90, 90, 95, 95, 93, 93, 91, 91, 93, 93, 95, 95, 99, 99, 102, 102,
        107, 107, 113, 113, 115, 115, 118, 73, 77, 77, 80, 80, 87, 87, 95, 95,
        95, 95, 96, 96, 93, 93, 91, 91, 92, 92, 94, 94, 97, 97, 100, 100, 104,
        104, 109, 109, 111, 111, 114, 73, 77, 77, 80, 80, 87, 87, 95, 95, 95,
        95, 96, 96, 93, 93, 91, 91, 92, 92, 94, 94, 97, 97, 100, 100, 104, 104,
        109, 109, 111, 111, 114, 83, 85, 85, 85, 85, 90, 90, 95, 95, 97, 97,
        101, 101, 99, 99, 98, 98, 100, 100, 101, 101, 104, 104, 107, 107, 111,
        111, 115, 115, 117, 117, 120, 83, 85, 85, 85, 85, 90, 90, 95, 95, 97,
        97, 101, 101, 99, 99, 98, 98, 100, 100, 101, 101, 104, 104, 107, 107,
        111, 111, 115, 115, 117, 117, 120, 98, 95, 95, 93, 93, 95, 95, 96, 96,
        101, 101, 106, 106, 106, 106, 107, 107, 108, 108, 109, 109, 111, 111,
        113, 113, 117, 117, 121, 121, 123, 123, 125, 98, 95, 95, 93, 93, 95, 95,
        96, 96, 101, 101, 106, 106, 106, 106, 107, 107, 108, 108, 109, 109, 111,
        111, 113, 113, 117, 117, 121, 121, 123, 123, 125, 97, 94, 94, 91, 91,
        93, 93, 93, 93, 99, 99, 106, 106, 109, 109, 111, 111, 113, 113, 115,
        115, 117, 117, 120, 120, 123, 123, 127, 127, 129, 129, 132, 97, 94, 94,
        91, 91, 93, 93, 93, 93, 99, 99, 106, 106, 109, 109, 111, 111, 113, 113,
        115, 115, 117, 117, 120, 120, 123, 123, 127, 127, 129, 129, 132, 98, 94,
        94, 90, 90, 91, 91, 91, 91, 98, 98, 107, 107, 111, 111, 117, 117, 120,
        120, 123, 123, 125, 125, 127, 127, 131, 131, 135, 135, 136, 136, 138,
        98, 94, 94, 90, 90, 91, 91, 91, 91, 98, 98, 107, 107, 111, 111, 117,
        117, 120, 120, 123, 123, 125, 125, 127, 127, 131, 131, 135, 135, 136,
        136, 138, 101, 97, 97, 93, 93, 93, 93, 92, 92, 100, 100, 108, 108, 113,
        113, 120, 120, 123, 123, 127, 127, 130, 130, 134, 134, 137, 137, 142,
        142, 143, 143, 145, 101, 97, 97, 93, 93, 93, 93, 92, 92, 100, 100, 108,
        108, 113, 113, 120, 120, 123, 123, 127, 127, 130, 130, 134, 134, 137,
        137, 142, 142, 143, 143, 145, 104, 100, 100, 95, 95, 95, 95, 94, 94,
        101, 101, 109, 109, 115, 115, 123, 123, 127, 127, 133, 133, 137, 137,
        141, 141, 145, 145, 150, 150, 151, 151, 153, 104, 100, 100, 95, 95, 95,
        95, 94, 94, 101, 101, 109, 109, 115, 115, 123, 123, 127, 127, 133, 133,
        137, 137, 141, 141, 145, 145, 150, 150, 151, 151, 153, 109, 104, 104,
        99, 99, 99, 99, 97, 97, 104, 104, 111, 111, 117, 117, 125, 125, 130,
        130, 137, 137, 142, 142, 147, 147, 151, 151, 156, 156, 158, 158, 161,
        109, 104, 104, 99, 99, 99, 99, 97, 97, 104, 104, 111, 111, 117, 117,
        125, 125, 130, 130, 137, 137, 142, 142, 147, 147, 151, 151, 156, 156,
        158, 158, 161, 114, 109, 109, 104, 104, 102, 102, 100, 100, 107, 107,
        113, 113, 120, 120, 127, 127, 134, 134, 141, 141, 147, 147, 153, 153,
        158, 158, 164, 164, 166, 166, 169, 114, 109, 109, 104, 104, 102, 102,
        100, 100, 107, 107, 113, 113, 120, 120, 127, 127, 134, 134, 141, 141,
        147, 147, 153, 153, 158, 158, 164, 164, 166, 166, 169, 120, 114, 114,
        109, 109, 107, 107, 104, 104, 111, 111, 117, 117, 123, 123, 131, 131,
        137, 137, 145, 145, 151, 151, 158, 158, 164, 164, 170, 170, 172, 172,
        176, 120, 114, 114, 109, 109, 107, 107, 104, 104, 111, 111, 117, 117,
        123, 123, 131, 131, 137, 137, 145, 145, 151, 151, 158, 158, 164, 164,
        170, 170, 172, 172, 176, 127, 121, 121, 115, 115, 113, 113, 109, 109,
        115, 115, 121, 121, 127, 127, 135, 135, 142, 142, 150, 150, 156, 156,
        164, 164, 170, 170, 178, 178, 180, 180, 184, 127, 121, 121, 115, 115,
        113, 113, 109, 109, 115, 115, 121, 121, 127, 127, 135, 135, 142, 142,
        150, 150, 156, 156, 164, 164, 170, 170, 178, 178, 180, 180, 184, 130,
        123, 123, 117, 117, 115, 115, 111, 111, 117, 117, 123, 123, 129, 129,
        136, 136, 143, 143, 151, 151, 158, 158, 166, 166, 172, 172, 180, 180,
        183, 183, 187, 130, 123, 123, 117, 117, 115, 115, 111, 111, 117, 117,
        123, 123, 129, 129, 136, 136, 143, 143, 151, 151, 158, 158, 166, 166,
        172, 172, 180, 180, 183, 183, 187, 134, 127, 127, 121, 121, 118, 118,
        114, 114, 120, 120, 125, 125, 132, 132, 138, 138, 145, 145, 153, 153,
        161, 161, 169, 169, 176, 176, 184, 184, 187, 187, 191,
        /* Size 4x8 */
        63, 94, 100, 120, 72, 94, 95, 112, 86, 100, 101, 115, 92, 106, 115, 128,
        93, 108, 128, 142, 100, 111, 137, 156, 109, 117, 145, 170, 118, 123,
        151, 180,
        /* Size 8x4 */
        63, 72, 86, 92, 93, 100, 109, 118, 94, 94, 100, 106, 108, 111, 117, 123,
        100, 95, 101, 115, 128, 137, 145, 151, 120, 112, 115, 128, 142, 156,
        170, 180,
        /* Size 8x16 */
        64, 62, 74, 96, 98, 105, 115, 127, 63, 63, 77, 94, 94, 100, 109, 120,
        61, 64, 80, 92, 91, 96, 104, 115, 67, 72, 86, 94, 92, 95, 103, 112, 74,
        80, 94, 95, 91, 94, 100, 109, 84, 86, 95, 100, 99, 101, 107, 115, 98,
        93, 96, 106, 107, 109, 114, 121, 97, 92, 94, 106, 112, 115, 120, 128,
        98, 91, 92, 107, 117, 123, 128, 135, 101, 93, 93, 108, 119, 128, 134,
        142, 104, 96, 95, 109, 122, 133, 142, 150, 109, 100, 98, 111, 125, 137,
        147, 156, 114, 104, 101, 113, 128, 141, 153, 164, 120, 109, 105, 117,
        131, 145, 158, 170, 127, 115, 110, 121, 135, 150, 164, 178, 129, 118,
        112, 123, 136, 151, 166, 180,
        /* Size 16x8 */
        64, 63, 61, 67, 74, 84, 98, 97, 98, 101, 104, 109, 114, 120, 127, 129,
        62, 63, 64, 72, 80, 86, 93, 92, 91, 93, 96, 100, 104, 109, 115, 118, 74,
        77, 80, 86, 94, 95, 96, 94, 92, 93, 95, 98, 101, 105, 110, 112, 96, 94,
        92, 94, 95, 100, 106, 106, 107, 108, 109, 111, 113, 117, 121, 123, 98,
        94, 91, 92, 91, 99, 107, 112, 117, 119, 122, 125, 128, 131, 135, 136,
        105, 100, 96, 95, 94, 101, 109, 115, 123, 128, 133, 137, 141, 145, 150,
        151, 115, 109, 104, 103, 100, 107, 114, 120, 128, 134, 142, 147, 153,
        158, 164, 166, 127, 120, 115, 112, 109, 115, 121, 128, 135, 142, 150,
        156, 164, 170, 178, 180,
        /* Size 16x32 */
        64, 62, 62, 74, 74, 96, 96, 98, 98, 105, 105, 115, 115, 127, 127, 133,
        63, 63, 63, 77, 77, 94, 94, 94, 94, 100, 100, 109, 109, 120, 120, 126,
        63, 63, 63, 77, 77, 94, 94, 94, 94, 100, 100, 109, 109, 120, 120, 126,
        61, 64, 64, 80, 80, 92, 92, 91, 91, 96, 96, 104, 104, 115, 115, 121, 61,
        64, 64, 80, 80, 92, 92, 91, 91, 96, 96, 104, 104, 115, 115, 121, 67, 72,
        72, 86, 86, 94, 94, 92, 92, 95, 95, 103, 103, 112, 112, 118, 67, 72, 72,
        86, 86, 94, 94, 92, 92, 95, 95, 103, 103, 112, 112, 118, 74, 80, 80, 94,
        94, 95, 95, 91, 91, 94, 94, 100, 100, 109, 109, 114, 74, 80, 80, 94, 94,
        95, 95, 91, 91, 94, 94, 100, 100, 109, 109, 114, 84, 86, 86, 95, 95,
        100, 100, 99, 99, 101, 101, 107, 107, 115, 115, 120, 84, 86, 86, 95, 95,
        100, 100, 99, 99, 101, 101, 107, 107, 115, 115, 120, 98, 93, 93, 96, 96,
        106, 106, 107, 107, 109, 109, 114, 114, 121, 121, 125, 98, 93, 93, 96,
        96, 106, 106, 107, 107, 109, 109, 114, 114, 121, 121, 125, 97, 92, 92,
        94, 94, 106, 106, 112, 112, 115, 115, 120, 120, 128, 128, 132, 97, 92,
        92, 94, 94, 106, 106, 112, 112, 115, 115, 120, 120, 128, 128, 132, 98,
        91, 91, 92, 92, 107, 107, 117, 117, 123, 123, 128, 128, 135, 135, 139,
        98, 91, 91, 92, 92, 107, 107, 117, 117, 123, 123, 128, 128, 135, 135,
        139, 101, 93, 93, 93, 93, 108, 108, 119, 119, 128, 128, 134, 134, 142,
        142, 146, 101, 93, 93, 93, 93, 108, 108, 119, 119, 128, 128, 134, 134,
        142, 142, 146, 104, 96, 96, 95, 95, 109, 109, 122, 122, 133, 133, 142,
        142, 150, 150, 154, 104, 96, 96, 95, 95, 109, 109, 122, 122, 133, 133,
        142, 142, 150, 150, 154, 109, 100, 100, 98, 98, 111, 111, 125, 125, 137,
        137, 147, 147, 156, 156, 161, 109, 100, 100, 98, 98, 111, 111, 125, 125,
        137, 137, 147, 147, 156, 156, 161, 114, 104, 104, 101, 101, 113, 113,
        128, 128, 141, 141, 153, 153, 164, 164, 169, 114, 104, 104, 101, 101,
        113, 113, 128, 128, 141, 141, 153, 153, 164, 164, 169, 120, 109, 109,
        105, 105, 117, 117, 131, 131, 145, 145, 158, 158, 170, 170, 176, 120,
        109, 109, 105, 105, 117, 117, 131, 131, 145, 145, 158, 158, 170, 170,
        176, 127, 115, 115, 110, 110, 121, 121, 135, 135, 150, 150, 164, 164,
        178, 178, 184, 127, 115, 115, 110, 110, 121, 121, 135, 135, 150, 150,
        164, 164, 178, 178, 184, 129, 118, 118, 112, 112, 123, 123, 136, 136,
        151, 151, 166, 166, 180, 180, 187, 129, 118, 118, 112, 112, 123, 123,
        136, 136, 151, 151, 166, 166, 180, 180, 187, 133, 121, 121, 115, 115,
        126, 126, 139, 139, 154, 154, 169, 169, 184, 184, 191,
        /* Size 32x16 */
        64, 63, 63, 61, 61, 67, 67, 74, 74, 84, 84, 98, 98, 97, 97, 98, 98, 101,
        101, 104, 104, 109, 109, 114, 114, 120, 120, 127, 127, 129, 129, 133,
        62, 63, 63, 64, 64, 72, 72, 80, 80, 86, 86, 93, 93, 92, 92, 91, 91, 93,
        93, 96, 96, 100, 100, 104, 104, 109, 109, 115, 115, 118, 118, 121, 62,
        63, 63, 64, 64, 72, 72, 80, 80, 86, 86, 93, 93, 92, 92, 91, 91, 93, 93,
        96, 96, 100, 100, 104, 104, 109, 109, 115, 115, 118, 118, 121, 74, 77,
        77, 80, 80, 86, 86, 94, 94, 95, 95, 96, 96, 94, 94, 92, 92, 93, 93, 95,
        95, 98, 98, 101, 101, 105, 105, 110, 110, 112, 112, 115, 74, 77, 77, 80,
        80, 86, 86, 94, 94, 95, 95, 96, 96, 94, 94, 92, 92, 93, 93, 95, 95, 98,
        98, 101, 101, 105, 105, 110, 110, 112, 112, 115, 96, 94, 94, 92, 92, 94,
        94, 95, 95, 100, 100, 106, 106, 106, 106, 107, 107, 108, 108, 109, 109,
        111, 111, 113, 113, 117, 117, 121, 121, 123, 123, 126, 96, 94, 94, 92,
        92, 94, 94, 95, 95, 100, 100, 106, 106, 106, 106, 107, 107, 108, 108,
        109, 109, 111, 111, 113, 113, 117, 117, 121, 121, 123, 123, 126, 98, 94,
        94, 91, 91, 92, 92, 91, 91, 99, 99, 107, 107, 112, 112, 117, 117, 119,
        119, 122, 122, 125, 125, 128, 128, 131, 131, 135, 135, 136, 136, 139,
        98, 94, 94, 91, 91, 92, 92, 91, 91, 99, 99, 107, 107, 112, 112, 117,
        117, 119, 119, 122, 122, 125, 125, 128, 128, 131, 131, 135, 135, 136,
        136, 139, 105, 100, 100, 96, 96, 95, 95, 94, 94, 101, 101, 109, 109,
        115, 115, 123, 123, 128, 128, 133, 133, 137, 137, 141, 141, 145, 145,
        150, 150, 151, 151, 154, 105, 100, 100, 96, 96, 95, 95, 94, 94, 101,
        101, 109, 109, 115, 115, 123, 123, 128, 128, 133, 133, 137, 137, 141,
        141, 145, 145, 150, 150, 151, 151, 154, 115, 109, 109, 104, 104, 103,
        103, 100, 100, 107, 107, 114, 114, 120, 120, 128, 128, 134, 134, 142,
        142, 147, 147, 153, 153, 158, 158, 164, 164, 166, 166, 169, 115, 109,
        109, 104, 104, 103, 103, 100, 100, 107, 107, 114, 114, 120, 120, 128,
        128, 134, 134, 142, 142, 147, 147, 153, 153, 158, 158, 164, 164, 166,
        166, 169, 127, 120, 120, 115, 115, 112, 112, 109, 109, 115, 115, 121,
        121, 128, 128, 135, 135, 142, 142, 150, 150, 156, 156, 164, 164, 170,
        170, 178, 178, 180, 180, 184, 127, 120, 120, 115, 115, 112, 112, 109,
        109, 115, 115, 121, 121, 128, 128, 135, 135, 142, 142, 150, 150, 156,
        156, 164, 164, 170, 170, 178, 178, 180, 180, 184, 133, 126, 126, 121,
        121, 118, 118, 114, 114, 120, 120, 125, 125, 132, 132, 139, 139, 146,
        146, 154, 154, 161, 161, 169, 169, 176, 176, 184, 184, 187, 187, 191,
        /* Size 4x16 */
        62, 96, 105, 127, 63, 94, 100, 120, 64, 92, 96, 115, 72, 94, 95, 112,
        80, 95, 94, 109, 86, 100, 101, 115, 93, 106, 109, 121, 92, 106, 115,
        128, 91, 107, 123, 135, 93, 108, 128, 142, 96, 109, 133, 150, 100, 111,
        137, 156, 104, 113, 141, 164, 109, 117, 145, 170, 115, 121, 150, 178,
        118, 123, 151, 180,
        /* Size 16x4 */
        62, 63, 64, 72, 80, 86, 93, 92, 91, 93, 96, 100, 104, 109, 115, 118, 96,
        94, 92, 94, 95, 100, 106, 106, 107, 108, 109, 111, 113, 117, 121, 123,
        105, 100, 96, 95, 94, 101, 109, 115, 123, 128, 133, 137, 141, 145, 150,
        151, 127, 120, 115, 112, 109, 115, 121, 128, 135, 142, 150, 156, 164,
        170, 178, 180,
        /* Size 8x32 */
        64, 62, 74, 96, 98, 105, 115, 127, 63, 63, 77, 94, 94, 100, 109, 120,
        63, 63, 77, 94, 94, 100, 109, 120, 61, 64, 80, 92, 91, 96, 104, 115, 61,
        64, 80, 92, 91, 96, 104, 115, 67, 72, 86, 94, 92, 95, 103, 112, 67, 72,
        86, 94, 92, 95, 103, 112, 74, 80, 94, 95, 91, 94, 100, 109, 74, 80, 94,
        95, 91, 94, 100, 109, 84, 86, 95, 100, 99, 101, 107, 115, 84, 86, 95,
        100, 99, 101, 107, 115, 98, 93, 96, 106, 107, 109, 114, 121, 98, 93, 96,
        106, 107, 109, 114, 121, 97, 92, 94, 106, 112, 115, 120, 128, 97, 92,
        94, 106, 112, 115, 120, 128, 98, 91, 92, 107, 117, 123, 128, 135, 98,
        91, 92, 107, 117, 123, 128, 135, 101, 93, 93, 108, 119, 128, 134, 142,
        101, 93, 93, 108, 119, 128, 134, 142, 104, 96, 95, 109, 122, 133, 142,
        150, 104, 96, 95, 109, 122, 133, 142, 150, 109, 100, 98, 111, 125, 137,
        147, 156, 109, 100, 98, 111, 125, 137, 147, 156, 114, 104, 101, 113,
        128, 141, 153, 164, 114, 104, 101, 113, 128, 141, 153, 164, 120, 109,
        105, 117, 131, 145, 158, 170, 120, 109, 105, 117, 131, 145, 158, 170,
        127, 115, 110, 121, 135, 150, 164, 178, 127, 115, 110, 121, 135, 150,
        164, 178, 129, 118, 112, 123, 136, 151, 166, 180, 129, 118, 112, 123,
        136, 151, 166, 180, 133, 121, 115, 126, 139, 154, 169, 184,
        /* Size 32x8 */
        64, 63, 63, 61, 61, 67, 67, 74, 74, 84, 84, 98, 98, 97, 97, 98, 98, 101,
        101, 104, 104, 109, 109, 114, 114, 120, 120, 127, 127, 129, 129, 133,
        62, 63, 63, 64, 64, 72, 72, 80, 80, 86, 86, 93, 93, 92, 92, 91, 91, 93,
        93, 96, 96, 100, 100, 104, 104, 109, 109, 115, 115, 118, 118, 121, 74,
        77, 77, 80, 80, 86, 86, 94, 94, 95, 95, 96, 96, 94, 94, 92, 92, 93, 93,
        95, 95, 98, 98, 101, 101, 105, 105, 110, 110, 112, 112, 115, 96, 94, 94,
        92, 92, 94, 94, 95, 95, 100, 100, 106, 106, 106, 106, 107, 107, 108,
        108, 109, 109, 111, 111, 113, 113, 117, 117, 121, 121, 123, 123, 126,
        98, 94, 94, 91, 91, 92, 92, 91, 91, 99, 99, 107, 107, 112, 112, 117,
        117, 119, 119, 122, 122, 125, 125, 128, 128, 131, 131, 135, 135, 136,
        136, 139, 105, 100, 100, 96, 96, 95, 95, 94, 94, 101, 101, 109, 109,
        115, 115, 123, 123, 128, 128, 133, 133, 137, 137, 141, 141, 145, 145,
        150, 150, 151, 151, 154, 115, 109, 109, 104, 104, 103, 103, 100, 100,
        107, 107, 114, 114, 120, 120, 128, 128, 134, 134, 142, 142, 147, 147,
        153, 153, 158, 158, 164, 164, 166, 166, 169, 127, 120, 120, 115, 115,
        112, 112, 109, 109, 115, 115, 121, 121, 128, 128, 135, 135, 142, 142,
        150, 150, 156, 156, 164, 164, 170, 170, 178, 178, 180, 180, 184 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        68, 105, 105, 178,
#endif
        /* Size 4x4 */
        64, 67, 90, 124, 67, 79, 102, 129, 90, 102, 142, 174, 124, 129, 174,
        217,
        /* Size 8x8 */
        63, 64, 65, 71, 85, 102, 119, 138, 64, 65, 67, 71, 83, 98, 113, 131, 65,
        67, 71, 77, 86, 99, 112, 128, 71, 71, 77, 97, 108, 119, 132, 147, 85,
        83, 86, 108, 127, 142, 155, 170, 102, 98, 99, 119, 142, 162, 178, 195,
        119, 113, 112, 132, 155, 178, 196, 216, 138, 131, 128, 147, 170, 195,
        216, 238,
        /* Size 16x16 */
        64, 63, 63, 63, 64, 68, 71, 76, 82, 90, 97, 108, 118, 131, 143, 160, 63,
        64, 64, 64, 65, 68, 70, 74, 80, 86, 93, 103, 113, 125, 136, 152, 63, 64,
        65, 65, 65, 67, 68, 72, 77, 83, 89, 99, 108, 119, 130, 145, 63, 64, 65,
        67, 68, 71, 72, 76, 81, 85, 91, 100, 108, 118, 129, 143, 64, 65, 65, 68,
        71, 74, 76, 79, 83, 87, 92, 99, 107, 116, 126, 139, 68, 68, 67, 71, 74,
        79, 84, 88, 92, 97, 102, 109, 117, 126, 136, 148, 71, 70, 68, 72, 76,
        84, 92, 97, 101, 106, 111, 118, 125, 134, 144, 156, 76, 74, 72, 76, 79,
        88, 97, 103, 108, 114, 119, 126, 134, 142, 152, 164, 82, 80, 77, 81, 83,
        92, 101, 108, 114, 121, 127, 135, 143, 151, 161, 173, 90, 86, 83, 85,
        87, 97, 106, 114, 121, 130, 136, 145, 153, 162, 171, 183, 97, 93, 89,
        91, 92, 102, 111, 119, 127, 136, 142, 153, 161, 171, 181, 193, 108, 103,
        99, 100, 99, 109, 118, 126, 135, 145, 153, 165, 174, 185, 195, 208, 118,
        113, 108, 108, 107, 117, 125, 134, 143, 153, 161, 174, 184, 196, 207,
        221, 131, 125, 119, 118, 116, 126, 134, 142, 151, 162, 171, 185, 196,
        210, 222, 237, 143, 136, 130, 129, 126, 136, 144, 152, 161, 171, 181,
        195, 207, 222, 235, 251, 160, 152, 145, 143, 139, 148, 156, 164, 173,
        183, 193, 208, 221, 237, 251, 268,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 63, 63, 64, 64, 64, 68, 68, 71, 72, 76, 79, 82, 88,
        90, 97, 97, 107, 108, 115, 118, 124, 131, 134, 143, 145, 160, 160, 63,
        63, 64, 64, 64, 64, 64, 64, 65, 65, 68, 68, 70, 71, 74, 77, 80, 85, 87,
        93, 93, 102, 104, 110, 113, 119, 125, 128, 137, 138, 152, 152, 63, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 68, 68, 70, 70, 74, 77, 80, 85, 86, 93,
        93, 102, 103, 110, 113, 118, 125, 128, 136, 138, 152, 152, 63, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 67, 67, 69, 69, 73, 76, 78, 83, 84, 90, 90,
        99, 100, 106, 109, 115, 121, 124, 132, 133, 147, 147, 63, 64, 64, 64,
        65, 65, 65, 65, 65, 66, 67, 67, 68, 69, 72, 75, 77, 82, 83, 89, 89, 98,
        99, 105, 108, 113, 119, 122, 130, 131, 145, 145, 63, 64, 64, 65, 65, 65,
        66, 66, 67, 67, 69, 69, 70, 71, 74, 77, 79, 83, 84, 90, 90, 98, 99, 105,
        108, 113, 119, 122, 129, 131, 144, 144, 63, 64, 64, 65, 65, 66, 67, 67,
        68, 68, 71, 71, 72, 73, 76, 78, 81, 84, 85, 91, 91, 99, 100, 105, 108,
        113, 118, 121, 129, 130, 143, 143, 64, 64, 64, 65, 65, 66, 67, 68, 69,
        69, 71, 71, 73, 74, 77, 79, 81, 84, 86, 91, 91, 99, 99, 105, 108, 113,
        118, 121, 128, 129, 141, 141, 64, 65, 65, 65, 65, 67, 68, 69, 71, 71,
        74, 74, 76, 77, 79, 81, 83, 85, 87, 92, 92, 98, 99, 104, 107, 111, 116,
        119, 126, 127, 139, 139, 64, 65, 65, 65, 66, 67, 68, 69, 71, 71, 74, 74,
        77, 77, 80, 82, 83, 86, 87, 93, 93, 99, 100, 105, 108, 112, 117, 120,
        127, 128, 140, 140, 68, 68, 68, 67, 67, 69, 71, 71, 74, 74, 79, 79, 84,
        85, 88, 90, 92, 95, 97, 102, 102, 108, 109, 114, 117, 121, 126, 129,
        136, 137, 148, 148, 68, 68, 68, 67, 67, 69, 71, 71, 74, 74, 79, 79, 84,
        85, 88, 90, 92, 95, 97, 102, 102, 108, 109, 114, 117, 121, 126, 129,
        136, 137, 148, 148, 71, 70, 70, 69, 68, 70, 72, 73, 76, 77, 84, 84, 92,
        94, 97, 99, 101, 105, 106, 111, 111, 117, 118, 123, 125, 129, 134, 137,
        144, 145, 156, 156, 72, 71, 70, 69, 69, 71, 73, 74, 77, 77, 85, 85, 94,
        97, 100, 101, 104, 108, 109, 114, 114, 119, 120, 125, 128, 132, 136,
        139, 146, 147, 158, 158, 76, 74, 74, 73, 72, 74, 76, 77, 79, 80, 88, 88,
        97, 100, 103, 105, 108, 112, 114, 119, 119, 125, 126, 131, 134, 138,
        142, 145, 152, 153, 164, 164, 79, 77, 77, 76, 75, 77, 78, 79, 81, 82,
        90, 90, 99, 101, 105, 108, 111, 116, 117, 122, 122, 129, 130, 135, 138,
        142, 147, 149, 156, 157, 168, 168, 82, 80, 80, 78, 77, 79, 81, 81, 83,
        83, 92, 92, 101, 104, 108, 111, 114, 120, 121, 127, 127, 134, 135, 140,
        143, 147, 151, 154, 161, 162, 173, 173, 88, 85, 85, 83, 82, 83, 84, 84,
        85, 86, 95, 95, 105, 108, 112, 116, 120, 127, 128, 134, 134, 142, 143,
        148, 151, 155, 159, 162, 169, 170, 181, 181, 90, 87, 86, 84, 83, 84, 85,
        86, 87, 87, 97, 97, 106, 109, 114, 117, 121, 128, 130, 136, 136, 144,
        145, 150, 153, 157, 162, 164, 171, 172, 183, 183, 97, 93, 93, 90, 89,
        90, 91, 91, 92, 93, 102, 102, 111, 114, 119, 122, 127, 134, 136, 142,
        142, 151, 153, 158, 161, 166, 171, 174, 181, 182, 193, 193, 97, 93, 93,
        90, 89, 90, 91, 91, 92, 93, 102, 102, 111, 114, 119, 122, 127, 134, 136,
        142, 142, 151, 153, 158, 161, 166, 171, 174, 181, 182, 193, 193, 107,
        102, 102, 99, 98, 98, 99, 99, 98, 99, 108, 108, 117, 119, 125, 129, 134,
        142, 144, 151, 151, 162, 163, 170, 173, 178, 183, 186, 194, 195, 206,
        206, 108, 104, 103, 100, 99, 99, 100, 99, 99, 100, 109, 109, 118, 120,
        126, 130, 135, 143, 145, 153, 153, 163, 165, 171, 174, 179, 185, 188,
        195, 196, 208, 208, 115, 110, 110, 106, 105, 105, 105, 105, 104, 105,
        114, 114, 123, 125, 131, 135, 140, 148, 150, 158, 158, 170, 171, 178,
        181, 187, 193, 196, 204, 205, 217, 217, 118, 113, 113, 109, 108, 108,
        108, 108, 107, 108, 117, 117, 125, 128, 134, 138, 143, 151, 153, 161,
        161, 173, 174, 181, 184, 190, 196, 199, 207, 209, 221, 221, 124, 119,
        118, 115, 113, 113, 113, 113, 111, 112, 121, 121, 129, 132, 138, 142,
        147, 155, 157, 166, 166, 178, 179, 187, 190, 196, 203, 206, 214, 216,
        229, 229, 131, 125, 125, 121, 119, 119, 118, 118, 116, 117, 126, 126,
        134, 136, 142, 147, 151, 159, 162, 171, 171, 183, 185, 193, 196, 203,
        210, 213, 222, 223, 237, 237, 134, 128, 128, 124, 122, 122, 121, 121,
        119, 120, 129, 129, 137, 139, 145, 149, 154, 162, 164, 174, 174, 186,
        188, 196, 199, 206, 213, 217, 226, 227, 241, 241, 143, 137, 136, 132,
        130, 129, 129, 128, 126, 127, 136, 136, 144, 146, 152, 156, 161, 169,
        171, 181, 181, 194, 195, 204, 207, 214, 222, 226, 235, 236, 251, 251,
        145, 138, 138, 133, 131, 131, 130, 129, 127, 128, 137, 137, 145, 147,
        153, 157, 162, 170, 172, 182, 182, 195, 196, 205, 209, 216, 223, 227,
        236, 238, 252, 252, 160, 152, 152, 147, 145, 144, 143, 141, 139, 140,
        148, 148, 156, 158, 164, 168, 173, 181, 183, 193, 193, 206, 208, 217,
        221, 229, 237, 241, 251, 252, 268, 268, 160, 152, 152, 147, 145, 144,
        143, 141, 139, 140, 148, 148, 156, 158, 164, 168, 173, 181, 183, 193,
        193, 206, 208, 217, 221, 229, 237, 241, 251, 252, 268, 268,
        /* Size 4x8 */
        64, 68, 87, 125, 65, 69, 84, 119, 66, 74, 88, 117, 70, 86, 109, 136, 83,
        96, 128, 159, 99, 108, 143, 183, 115, 121, 157, 203, 133, 137, 172, 223,
        /* Size 8x4 */
        64, 65, 66, 70, 83, 99, 115, 133, 68, 69, 74, 86, 96, 108, 121, 137, 87,
        84, 88, 109, 128, 143, 157, 172, 125, 119, 117, 136, 159, 183, 203, 223,
        /* Size 8x16 */
        64, 63, 64, 72, 88, 107, 125, 146, 63, 64, 65, 71, 85, 103, 119, 139,
        63, 65, 66, 69, 82, 99, 114, 133, 64, 65, 68, 73, 84, 100, 114, 131, 64,
        66, 71, 77, 85, 99, 112, 128, 68, 68, 74, 85, 96, 109, 122, 138, 71, 69,
        76, 94, 105, 118, 130, 146, 76, 73, 80, 99, 112, 126, 139, 154, 82, 78,
        83, 103, 120, 135, 148, 163, 89, 84, 87, 108, 128, 145, 158, 173, 96,
        90, 93, 113, 134, 152, 167, 183, 107, 99, 100, 120, 142, 164, 180, 198,
        117, 108, 108, 127, 150, 174, 191, 210, 130, 120, 117, 136, 159, 185,
        204, 225, 142, 131, 127, 146, 169, 195, 216, 238, 159, 145, 140, 158,
        181, 208, 230, 254,
        /* Size 16x8 */
        64, 63, 63, 64, 64, 68, 71, 76, 82, 89, 96, 107, 117, 130, 142, 159, 63,
        64, 65, 65, 66, 68, 69, 73, 78, 84, 90, 99, 108, 120, 131, 145, 64, 65,
        66, 68, 71, 74, 76, 80, 83, 87, 93, 100, 108, 117, 127, 140, 72, 71, 69,
        73, 77, 85, 94, 99, 103, 108, 113, 120, 127, 136, 146, 158, 88, 85, 82,
        84, 85, 96, 105, 112, 120, 128, 134, 142, 150, 159, 169, 181, 107, 103,
        99, 100, 99, 109, 118, 126, 135, 145, 152, 164, 174, 185, 195, 208, 125,
        119, 114, 114, 112, 122, 130, 139, 148, 158, 167, 180, 191, 204, 216,
        230, 146, 139, 133, 131, 128, 138, 146, 154, 163, 173, 183, 198, 210,
        225, 238, 254,
        /* Size 16x32 */
        64, 63, 63, 64, 64, 68, 72, 76, 88, 89, 107, 107, 125, 131, 146, 158,
        63, 64, 64, 65, 65, 68, 71, 75, 85, 87, 103, 103, 120, 125, 140, 151,
        63, 64, 64, 65, 65, 68, 71, 74, 85, 86, 103, 103, 119, 124, 139, 151,
        63, 64, 64, 65, 65, 67, 70, 73, 83, 84, 100, 100, 116, 121, 134, 146,
        63, 65, 65, 65, 66, 67, 69, 73, 82, 83, 99, 99, 114, 119, 133, 144, 63,
        65, 65, 66, 67, 69, 71, 74, 83, 84, 99, 99, 114, 119, 132, 143, 64, 65,
        65, 67, 68, 71, 73, 76, 84, 86, 100, 100, 114, 118, 131, 142, 64, 65,
        65, 68, 69, 71, 74, 77, 85, 86, 99, 99, 113, 118, 130, 140, 64, 65, 66,
        69, 71, 74, 77, 79, 85, 87, 99, 99, 112, 116, 128, 138, 64, 66, 66, 69,
        71, 74, 78, 80, 86, 88, 100, 100, 113, 117, 129, 139, 68, 68, 68, 72,
        74, 79, 85, 88, 96, 97, 109, 109, 122, 126, 138, 147, 68, 68, 68, 72,
        74, 79, 85, 88, 96, 97, 109, 109, 122, 126, 138, 147, 71, 69, 69, 74,
        76, 84, 94, 97, 105, 106, 118, 118, 130, 134, 146, 155, 72, 70, 69, 75,
        77, 86, 96, 99, 108, 109, 120, 120, 133, 136, 148, 157, 76, 73, 73, 77,
        80, 88, 99, 103, 112, 114, 126, 126, 139, 142, 154, 163, 79, 76, 75, 80,
        81, 90, 101, 105, 116, 117, 130, 130, 143, 147, 159, 168, 82, 78, 78,
        82, 83, 92, 103, 108, 120, 121, 135, 135, 148, 152, 163, 172, 88, 83,
        82, 85, 86, 96, 107, 112, 126, 128, 143, 143, 156, 159, 171, 180, 89,
        84, 84, 86, 87, 97, 108, 113, 128, 129, 145, 145, 158, 162, 173, 182,
        96, 90, 90, 92, 93, 102, 113, 119, 134, 135, 152, 152, 167, 171, 183,
        192, 96, 90, 90, 92, 93, 102, 113, 119, 134, 135, 152, 152, 167, 171,
        183, 192, 106, 99, 98, 99, 99, 108, 119, 125, 142, 143, 163, 163, 179,
        183, 196, 206, 107, 100, 99, 100, 100, 109, 120, 126, 142, 144, 164,
        164, 180, 185, 198, 207, 114, 106, 105, 105, 105, 114, 125, 131, 148,
        150, 171, 171, 188, 193, 206, 216, 117, 109, 108, 108, 108, 117, 127,
        134, 150, 152, 174, 174, 191, 196, 210, 220, 123, 115, 114, 113, 112,
        121, 132, 138, 155, 157, 179, 179, 197, 203, 217, 228, 130, 121, 120,
        118, 117, 126, 136, 142, 159, 161, 185, 185, 204, 210, 225, 236, 134,
        124, 123, 121, 120, 129, 139, 145, 162, 164, 188, 188, 207, 213, 228,
        240, 142, 132, 131, 128, 127, 136, 146, 152, 169, 171, 195, 195, 216,
        222, 238, 250, 144, 133, 132, 129, 128, 137, 147, 153, 170, 172, 196,
        196, 217, 223, 239, 251, 159, 147, 145, 142, 140, 148, 158, 164, 181,
        183, 208, 208, 230, 236, 254, 267, 159, 147, 145, 142, 140, 148, 158,
        164, 181, 183, 208, 208, 230, 236, 254, 267,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 63, 64, 64, 64, 64, 68, 68, 71, 72, 76, 79, 82, 88,
        89, 96, 96, 106, 107, 114, 117, 123, 130, 134, 142, 144, 159, 159, 63,
        64, 64, 64, 65, 65, 65, 65, 65, 66, 68, 68, 69, 70, 73, 76, 78, 83, 84,
        90, 90, 99, 100, 106, 109, 115, 121, 124, 132, 133, 147, 147, 63, 64,
        64, 64, 65, 65, 65, 65, 66, 66, 68, 68, 69, 69, 73, 75, 78, 82, 84, 90,
        90, 98, 99, 105, 108, 114, 120, 123, 131, 132, 145, 145, 64, 65, 65, 65,
        65, 66, 67, 68, 69, 69, 72, 72, 74, 75, 77, 80, 82, 85, 86, 92, 92, 99,
        100, 105, 108, 113, 118, 121, 128, 129, 142, 142, 64, 65, 65, 65, 66,
        67, 68, 69, 71, 71, 74, 74, 76, 77, 80, 81, 83, 86, 87, 93, 93, 99, 100,
        105, 108, 112, 117, 120, 127, 128, 140, 140, 68, 68, 68, 67, 67, 69, 71,
        71, 74, 74, 79, 79, 84, 86, 88, 90, 92, 96, 97, 102, 102, 108, 109, 114,
        117, 121, 126, 129, 136, 137, 148, 148, 72, 71, 71, 70, 69, 71, 73, 74,
        77, 78, 85, 85, 94, 96, 99, 101, 103, 107, 108, 113, 113, 119, 120, 125,
        127, 132, 136, 139, 146, 147, 158, 158, 76, 75, 74, 73, 73, 74, 76, 77,
        79, 80, 88, 88, 97, 99, 103, 105, 108, 112, 113, 119, 119, 125, 126,
        131, 134, 138, 142, 145, 152, 153, 164, 164, 88, 85, 85, 83, 82, 83, 84,
        85, 85, 86, 96, 96, 105, 108, 112, 116, 120, 126, 128, 134, 134, 142,
        142, 148, 150, 155, 159, 162, 169, 170, 181, 181, 89, 87, 86, 84, 83,
        84, 86, 86, 87, 88, 97, 97, 106, 109, 114, 117, 121, 128, 129, 135, 135,
        143, 144, 150, 152, 157, 161, 164, 171, 172, 183, 183, 107, 103, 103,
        100, 99, 99, 100, 99, 99, 100, 109, 109, 118, 120, 126, 130, 135, 143,
        145, 152, 152, 163, 164, 171, 174, 179, 185, 188, 195, 196, 208, 208,
        107, 103, 103, 100, 99, 99, 100, 99, 99, 100, 109, 109, 118, 120, 126,
        130, 135, 143, 145, 152, 152, 163, 164, 171, 174, 179, 185, 188, 195,
        196, 208, 208, 125, 120, 119, 116, 114, 114, 114, 113, 112, 113, 122,
        122, 130, 133, 139, 143, 148, 156, 158, 167, 167, 179, 180, 188, 191,
        197, 204, 207, 216, 217, 230, 230, 131, 125, 124, 121, 119, 119, 118,
        118, 116, 117, 126, 126, 134, 136, 142, 147, 152, 159, 162, 171, 171,
        183, 185, 193, 196, 203, 210, 213, 222, 223, 236, 236, 146, 140, 139,
        134, 133, 132, 131, 130, 128, 129, 138, 138, 146, 148, 154, 159, 163,
        171, 173, 183, 183, 196, 198, 206, 210, 217, 225, 228, 238, 239, 254,
        254, 158, 151, 151, 146, 144, 143, 142, 140, 138, 139, 147, 147, 155,
        157, 163, 168, 172, 180, 182, 192, 192, 206, 207, 216, 220, 228, 236,
        240, 250, 251, 267, 267,
        /* Size 4x16 */
        63, 68, 89, 131, 64, 68, 86, 124, 65, 67, 83, 119, 65, 71, 86, 118, 65,
        74, 87, 116, 68, 79, 97, 126, 69, 84, 106, 134, 73, 88, 114, 142, 78,
        92, 121, 152, 84, 97, 129, 162, 90, 102, 135, 171, 100, 109, 144, 185,
        109, 117, 152, 196, 121, 126, 161, 210, 132, 136, 171, 222, 147, 148,
        183, 236,
        /* Size 16x4 */
        63, 64, 65, 65, 65, 68, 69, 73, 78, 84, 90, 100, 109, 121, 132, 147, 68,
        68, 67, 71, 74, 79, 84, 88, 92, 97, 102, 109, 117, 126, 136, 148, 89,
        86, 83, 86, 87, 97, 106, 114, 121, 129, 135, 144, 152, 161, 171, 183,
        131, 124, 119, 118, 116, 126, 134, 142, 152, 162, 171, 185, 196, 210,
        222, 236,
        /* Size 8x32 */
        64, 63, 64, 72, 88, 107, 125, 146, 63, 64, 65, 71, 85, 103, 120, 140,
        63, 64, 65, 71, 85, 103, 119, 139, 63, 64, 65, 70, 83, 100, 116, 134,
        63, 65, 66, 69, 82, 99, 114, 133, 63, 65, 67, 71, 83, 99, 114, 132, 64,
        65, 68, 73, 84, 100, 114, 131, 64, 65, 69, 74, 85, 99, 113, 130, 64, 66,
        71, 77, 85, 99, 112, 128, 64, 66, 71, 78, 86, 100, 113, 129, 68, 68, 74,
        85, 96, 109, 122, 138, 68, 68, 74, 85, 96, 109, 122, 138, 71, 69, 76,
        94, 105, 118, 130, 146, 72, 69, 77, 96, 108, 120, 133, 148, 76, 73, 80,
        99, 112, 126, 139, 154, 79, 75, 81, 101, 116, 130, 143, 159, 82, 78, 83,
        103, 120, 135, 148, 163, 88, 82, 86, 107, 126, 143, 156, 171, 89, 84,
        87, 108, 128, 145, 158, 173, 96, 90, 93, 113, 134, 152, 167, 183, 96,
        90, 93, 113, 134, 152, 167, 183, 106, 98, 99, 119, 142, 163, 179, 196,
        107, 99, 100, 120, 142, 164, 180, 198, 114, 105, 105, 125, 148, 171,
        188, 206, 117, 108, 108, 127, 150, 174, 191, 210, 123, 114, 112, 132,
        155, 179, 197, 217, 130, 120, 117, 136, 159, 185, 204, 225, 134, 123,
        120, 139, 162, 188, 207, 228, 142, 131, 127, 146, 169, 195, 216, 238,
        144, 132, 128, 147, 170, 196, 217, 239, 159, 145, 140, 158, 181, 208,
        230, 254, 159, 145, 140, 158, 181, 208, 230, 254,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 63, 64, 64, 64, 64, 68, 68, 71, 72, 76, 79, 82, 88,
        89, 96, 96, 106, 107, 114, 117, 123, 130, 134, 142, 144, 159, 159, 63,
        64, 64, 64, 65, 65, 65, 65, 66, 66, 68, 68, 69, 69, 73, 75, 78, 82, 84,
        90, 90, 98, 99, 105, 108, 114, 120, 123, 131, 132, 145, 145, 64, 65, 65,
        65, 66, 67, 68, 69, 71, 71, 74, 74, 76, 77, 80, 81, 83, 86, 87, 93, 93,
        99, 100, 105, 108, 112, 117, 120, 127, 128, 140, 140, 72, 71, 71, 70,
        69, 71, 73, 74, 77, 78, 85, 85, 94, 96, 99, 101, 103, 107, 108, 113,
        113, 119, 120, 125, 127, 132, 136, 139, 146, 147, 158, 158, 88, 85, 85,
        83, 82, 83, 84, 85, 85, 86, 96, 96, 105, 108, 112, 116, 120, 126, 128,
        134, 134, 142, 142, 148, 150, 155, 159, 162, 169, 170, 181, 181, 107,
        103, 103, 100, 99, 99, 100, 99, 99, 100, 109, 109, 118, 120, 126, 130,
        135, 143, 145, 152, 152, 163, 164, 171, 174, 179, 185, 188, 195, 196,
        208, 208, 125, 120, 119, 116, 114, 114, 114, 113, 112, 113, 122, 122,
        130, 133, 139, 143, 148, 156, 158, 167, 167, 179, 180, 188, 191, 197,
        204, 207, 216, 217, 230, 230, 146, 140, 139, 134, 133, 132, 131, 130,
        128, 129, 138, 138, 146, 148, 154, 159, 163, 171, 173, 183, 183, 196,
        198, 206, 210, 217, 225, 228, 238, 239, 254, 254 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        83, 97, 97, 139,
#endif
        /* Size 4x4 */
        63, 85, 94, 107, 85, 97, 100, 108, 94, 100, 123, 135, 107, 108, 135,
        156,
        /* Size 8x8 */
        63, 65, 77, 96, 94, 100, 107, 115, 65, 71, 84, 94, 90, 95, 101, 109, 77,
        84, 95, 96, 91, 94, 99, 105, 96, 94, 96, 106, 107, 109, 112, 117, 94,
        90, 91, 107, 117, 122, 126, 131, 100, 95, 94, 109, 122, 132, 138, 145,
        107, 101, 99, 112, 126, 138, 147, 155, 115, 109, 105, 117, 131, 145,
        155, 165,
        /* Size 16x16 */
        64, 62, 61, 67, 73, 83, 95, 98, 98, 99, 101, 104, 109, 114, 120, 127,
        62, 63, 62, 69, 77, 85, 93, 95, 94, 94, 97, 100, 104, 109, 114, 121, 61,
        62, 64, 71, 80, 85, 91, 92, 91, 91, 93, 95, 99, 104, 109, 115, 67, 69,
        71, 78, 87, 90, 94, 93, 92, 91, 93, 95, 99, 102, 107, 113, 73, 77, 80,
        87, 95, 95, 95, 94, 92, 91, 92, 94, 97, 100, 104, 109, 83, 85, 85, 90,
        95, 97, 100, 100, 99, 99, 100, 101, 104, 107, 111, 115, 95, 93, 91, 94,
        95, 100, 104, 105, 105, 105, 106, 107, 110, 112, 116, 120, 98, 95, 92,
        93, 94, 100, 105, 107, 108, 110, 111, 112, 115, 117, 121, 125, 98, 94,
        91, 92, 92, 99, 105, 108, 111, 114, 116, 118, 120, 123, 126, 130, 99,
        94, 91, 91, 91, 99, 105, 110, 114, 118, 120, 123, 126, 129, 132, 136,
        101, 97, 93, 93, 92, 100, 106, 111, 116, 120, 123, 127, 130, 134, 137,
        142, 104, 100, 95, 95, 94, 101, 107, 112, 118, 123, 127, 133, 137, 141,
        145, 150, 109, 104, 99, 99, 97, 104, 110, 115, 120, 126, 130, 137, 142,
        147, 151, 156, 114, 109, 104, 102, 100, 107, 112, 117, 123, 129, 134,
        141, 147, 153, 158, 164, 120, 114, 109, 107, 104, 111, 116, 121, 126,
        132, 137, 145, 151, 158, 164, 170, 127, 121, 115, 113, 109, 115, 120,
        125, 130, 136, 142, 150, 156, 164, 170, 178,
        /* Size 32x32 */
        64, 62, 62, 61, 61, 64, 67, 69, 73, 74, 83, 83, 95, 98, 98, 97, 98, 98,
        99, 101, 101, 104, 104, 108, 109, 112, 114, 116, 120, 121, 127, 127, 62,
        63, 63, 62, 62, 65, 69, 71, 77, 77, 84, 84, 93, 96, 95, 94, 94, 94, 95,
        97, 97, 100, 100, 103, 105, 107, 109, 111, 115, 115, 121, 121, 62, 63,
        63, 63, 62, 66, 69, 71, 77, 78, 85, 85, 93, 95, 95, 94, 94, 94, 94, 97,
        97, 99, 100, 103, 104, 106, 109, 110, 114, 115, 121, 121, 61, 62, 63,
        63, 63, 67, 70, 73, 79, 80, 85, 85, 92, 94, 93, 92, 92, 91, 92, 94, 94,
        96, 97, 100, 101, 103, 105, 107, 110, 111, 117, 117, 61, 62, 62, 63, 64,
        67, 71, 73, 80, 80, 85, 85, 91, 93, 92, 91, 91, 90, 91, 93, 93, 95, 95,
        98, 99, 102, 104, 105, 109, 109, 115, 115, 64, 65, 66, 67, 67, 71, 74,
        77, 83, 84, 87, 87, 92, 94, 93, 92, 91, 90, 91, 93, 93, 95, 95, 98, 99,
        101, 103, 104, 108, 109, 114, 114, 67, 69, 69, 70, 71, 74, 78, 80, 87,
        87, 90, 90, 94, 95, 93, 93, 92, 91, 91, 93, 93, 95, 95, 98, 99, 100,
        102, 104, 107, 108, 113, 113, 69, 71, 71, 73, 73, 77, 80, 83, 89, 89,
        91, 91, 94, 95, 94, 93, 92, 91, 91, 93, 93, 94, 95, 97, 98, 100, 102,
        103, 106, 107, 111, 111, 73, 77, 77, 79, 80, 83, 87, 89, 95, 95, 95, 95,
        95, 96, 94, 93, 92, 91, 91, 92, 92, 93, 94, 96, 97, 98, 100, 101, 104,
        105, 109, 109, 74, 77, 78, 80, 80, 84, 87, 89, 95, 95, 95, 95, 96, 96,
        95, 94, 93, 91, 92, 93, 93, 94, 94, 96, 97, 99, 100, 102, 105, 105, 110,
        110, 83, 84, 85, 85, 85, 87, 90, 91, 95, 95, 97, 97, 100, 101, 100, 99,
        99, 98, 99, 100, 100, 101, 101, 103, 104, 105, 107, 108, 111, 111, 115,
        115, 83, 84, 85, 85, 85, 87, 90, 91, 95, 95, 97, 97, 100, 101, 100, 99,
        99, 98, 99, 100, 100, 101, 101, 103, 104, 105, 107, 108, 111, 111, 115,
        115, 95, 93, 93, 92, 91, 92, 94, 94, 95, 96, 100, 100, 104, 105, 105,
        105, 105, 105, 105, 106, 106, 107, 107, 109, 110, 111, 112, 113, 116,
        116, 120, 120, 98, 96, 95, 94, 93, 94, 95, 95, 96, 96, 101, 101, 105,
        106, 106, 106, 107, 107, 107, 108, 108, 109, 109, 110, 111, 112, 113,
        114, 117, 117, 121, 121, 98, 95, 95, 93, 92, 93, 93, 94, 94, 95, 100,
        100, 105, 106, 107, 108, 108, 110, 110, 111, 111, 112, 112, 114, 115,
        116, 117, 118, 121, 121, 125, 125, 97, 94, 94, 92, 91, 92, 93, 93, 93,
        94, 99, 99, 105, 106, 108, 109, 110, 111, 112, 113, 113, 115, 115, 117,
        117, 119, 120, 121, 123, 124, 127, 127, 98, 94, 94, 92, 91, 91, 92, 92,
        92, 93, 99, 99, 105, 107, 108, 110, 111, 114, 114, 116, 116, 118, 118,
        120, 120, 122, 123, 124, 126, 127, 130, 130, 98, 94, 94, 91, 90, 90, 91,
        91, 91, 91, 98, 98, 105, 107, 110, 111, 114, 117, 118, 120, 120, 122,
        123, 124, 125, 126, 127, 128, 131, 131, 135, 135, 99, 95, 94, 92, 91,
        91, 91, 91, 91, 92, 99, 99, 105, 107, 110, 112, 114, 118, 118, 120, 120,
        123, 123, 125, 126, 127, 129, 130, 132, 133, 136, 136, 101, 97, 97, 94,
        93, 93, 93, 93, 92, 93, 100, 100, 106, 108, 111, 113, 116, 120, 120,
        123, 123, 127, 127, 130, 130, 132, 134, 135, 137, 138, 142, 142, 101,
        97, 97, 94, 93, 93, 93, 93, 92, 93, 100, 100, 106, 108, 111, 113, 116,
        120, 120, 123, 123, 127, 127, 130, 130, 132, 134, 135, 137, 138, 142,
        142, 104, 100, 99, 96, 95, 95, 95, 94, 93, 94, 101, 101, 107, 109, 112,
        115, 118, 122, 123, 127, 127, 132, 133, 135, 136, 138, 141, 142, 144,
        145, 149, 149, 104, 100, 100, 97, 95, 95, 95, 95, 94, 94, 101, 101, 107,
        109, 112, 115, 118, 123, 123, 127, 127, 133, 133, 136, 137, 139, 141,
        142, 145, 146, 150, 150, 108, 103, 103, 100, 98, 98, 98, 97, 96, 96,
        103, 103, 109, 110, 114, 117, 120, 124, 125, 130, 130, 135, 136, 139,
        140, 143, 145, 146, 149, 150, 154, 154, 109, 105, 104, 101, 99, 99, 99,
        98, 97, 97, 104, 104, 110, 111, 115, 117, 120, 125, 126, 130, 130, 136,
        137, 140, 142, 144, 147, 148, 151, 152, 156, 156, 112, 107, 106, 103,
        102, 101, 100, 100, 98, 99, 105, 105, 111, 112, 116, 119, 122, 126, 127,
        132, 132, 138, 139, 143, 144, 147, 150, 151, 154, 155, 160, 160, 114,
        109, 109, 105, 104, 103, 102, 102, 100, 100, 107, 107, 112, 113, 117,
        120, 123, 127, 129, 134, 134, 141, 141, 145, 147, 150, 153, 155, 158,
        159, 164, 164, 116, 111, 110, 107, 105, 104, 104, 103, 101, 102, 108,
        108, 113, 114, 118, 121, 124, 128, 130, 135, 135, 142, 142, 146, 148,
        151, 155, 156, 160, 160, 166, 166, 120, 115, 114, 110, 109, 108, 107,
        106, 104, 105, 111, 111, 116, 117, 121, 123, 126, 131, 132, 137, 137,
        144, 145, 149, 151, 154, 158, 160, 164, 164, 170, 170, 121, 115, 115,
        111, 109, 109, 108, 107, 105, 105, 111, 111, 116, 117, 121, 124, 127,
        131, 133, 138, 138, 145, 146, 150, 152, 155, 159, 160, 164, 165, 171,
        171, 127, 121, 121, 117, 115, 114, 113, 111, 109, 110, 115, 115, 120,
        121, 125, 127, 130, 135, 136, 142, 142, 149, 150, 154, 156, 160, 164,
        166, 170, 171, 178, 178, 127, 121, 121, 117, 115, 114, 113, 111, 109,
        110, 115, 115, 120, 121, 125, 127, 130, 135, 136, 142, 142, 149, 150,
        154, 156, 160, 164, 166, 170, 171, 178, 178,
        /* Size 4x8 */
        63, 85, 95, 109, 67, 88, 91, 103, 80, 95, 92, 101, 94, 101, 108, 114,
        91, 99, 118, 128, 96, 101, 123, 141, 103, 105, 127, 150, 111, 111, 133,
        159,
        /* Size 8x4 */
        63, 67, 80, 94, 91, 96, 103, 111, 85, 88, 95, 101, 99, 101, 105, 111,
        95, 91, 92, 108, 118, 123, 127, 133, 109, 103, 101, 114, 128, 141, 150,
        159,
        /* Size 8x16 */
        64, 62, 74, 96, 98, 105, 112, 122, 63, 63, 77, 94, 94, 100, 107, 115,
        61, 64, 80, 92, 91, 96, 102, 110, 67, 72, 86, 94, 92, 95, 101, 108, 74,
        80, 94, 95, 91, 94, 99, 105, 84, 86, 95, 100, 99, 101, 106, 112, 95, 92,
        96, 105, 106, 107, 111, 117, 97, 92, 95, 106, 110, 113, 116, 122, 97,
        91, 93, 106, 114, 118, 122, 127, 98, 91, 92, 107, 117, 124, 128, 133,
        101, 93, 93, 108, 119, 128, 133, 139, 104, 96, 95, 109, 122, 133, 140,
        146, 109, 100, 98, 111, 125, 137, 145, 152, 114, 104, 101, 113, 128,
        141, 151, 159, 120, 109, 105, 117, 131, 145, 155, 165, 127, 115, 110,
        121, 135, 150, 161, 172,
        /* Size 16x8 */
        64, 63, 61, 67, 74, 84, 95, 97, 97, 98, 101, 104, 109, 114, 120, 127,
        62, 63, 64, 72, 80, 86, 92, 92, 91, 91, 93, 96, 100, 104, 109, 115, 74,
        77, 80, 86, 94, 95, 96, 95, 93, 92, 93, 95, 98, 101, 105, 110, 96, 94,
        92, 94, 95, 100, 105, 106, 106, 107, 108, 109, 111, 113, 117, 121, 98,
        94, 91, 92, 91, 99, 106, 110, 114, 117, 119, 122, 125, 128, 131, 135,
        105, 100, 96, 95, 94, 101, 107, 113, 118, 124, 128, 133, 137, 141, 145,
        150, 112, 107, 102, 101, 99, 106, 111, 116, 122, 128, 133, 140, 145,
        151, 155, 161, 122, 115, 110, 108, 105, 112, 117, 122, 127, 133, 139,
        146, 152, 159, 165, 172,
        /* Size 16x32 */
        64, 62, 62, 70, 74, 84, 96, 97, 98, 99, 105, 105, 112, 115, 122, 127,
        63, 63, 63, 72, 77, 85, 94, 95, 95, 95, 100, 100, 108, 109, 116, 121,
        63, 63, 63, 72, 77, 85, 94, 94, 94, 95, 100, 100, 107, 109, 115, 120,
        61, 64, 64, 74, 79, 85, 93, 92, 92, 92, 97, 97, 104, 105, 112, 116, 61,
        64, 64, 74, 80, 85, 92, 92, 91, 91, 96, 96, 102, 104, 110, 115, 64, 67,
        68, 78, 83, 88, 93, 92, 91, 91, 96, 96, 102, 103, 109, 114, 67, 71, 72,
        81, 86, 90, 94, 93, 92, 92, 95, 95, 101, 103, 108, 112, 69, 74, 74, 84,
        89, 91, 94, 94, 91, 92, 95, 95, 100, 102, 107, 111, 74, 80, 80, 90, 94,
        95, 95, 94, 91, 92, 94, 94, 99, 100, 105, 109, 75, 80, 81, 90, 94, 95,
        96, 95, 92, 92, 95, 95, 99, 101, 106, 110, 84, 86, 86, 92, 95, 97, 100,
        100, 99, 99, 101, 101, 106, 107, 112, 115, 84, 86, 86, 92, 95, 97, 100,
        100, 99, 99, 101, 101, 106, 107, 112, 115, 95, 92, 92, 95, 96, 100, 105,
        105, 106, 106, 107, 107, 111, 112, 117, 120, 98, 94, 93, 95, 96, 101,
        106, 106, 107, 108, 109, 109, 113, 114, 118, 121, 97, 93, 92, 94, 95,
        100, 106, 107, 110, 110, 113, 113, 116, 117, 122, 125, 97, 92, 92, 93,
        94, 100, 106, 108, 112, 112, 115, 115, 119, 120, 124, 128, 97, 92, 91,
        92, 93, 99, 106, 108, 114, 114, 118, 118, 122, 123, 127, 130, 98, 91,
        91, 91, 92, 99, 107, 110, 117, 118, 123, 123, 127, 128, 132, 135, 98,
        92, 91, 92, 92, 99, 107, 110, 117, 118, 124, 124, 128, 129, 133, 136,
        101, 94, 93, 93, 93, 100, 108, 111, 119, 120, 128, 128, 133, 134, 139,
        142, 101, 94, 93, 93, 93, 100, 108, 111, 119, 120, 128, 128, 133, 134,
        139, 142, 104, 96, 96, 95, 94, 101, 109, 112, 122, 123, 133, 133, 139,
        141, 145, 149, 104, 97, 96, 95, 95, 101, 109, 112, 122, 123, 133, 133,
        140, 142, 146, 150, 107, 100, 99, 97, 97, 103, 110, 114, 124, 125, 136,
        136, 143, 145, 150, 154, 109, 101, 100, 98, 98, 104, 111, 115, 125, 126,
        137, 137, 145, 147, 152, 156, 111, 103, 102, 100, 99, 105, 112, 116,
        126, 127, 139, 139, 148, 150, 156, 160, 114, 105, 104, 102, 101, 107,
        113, 117, 128, 129, 141, 141, 151, 153, 159, 164, 116, 107, 106, 103,
        102, 108, 114, 118, 129, 130, 142, 142, 152, 155, 161, 166, 120, 110,
        109, 106, 105, 111, 117, 121, 131, 132, 145, 145, 155, 158, 165, 170,
        120, 111, 110, 107, 106, 111, 118, 121, 131, 133, 146, 146, 156, 159,
        166, 171, 127, 117, 115, 112, 110, 116, 121, 125, 135, 136, 150, 150,
        161, 164, 172, 178, 127, 117, 115, 112, 110, 116, 121, 125, 135, 136,
        150, 150, 161, 164, 172, 178,
        /* Size 32x16 */
        64, 63, 63, 61, 61, 64, 67, 69, 74, 75, 84, 84, 95, 98, 97, 97, 97, 98,
        98, 101, 101, 104, 104, 107, 109, 111, 114, 116, 120, 120, 127, 127, 62,
        63, 63, 64, 64, 67, 71, 74, 80, 80, 86, 86, 92, 94, 93, 92, 92, 91, 92,
        94, 94, 96, 97, 100, 101, 103, 105, 107, 110, 111, 117, 117, 62, 63, 63,
        64, 64, 68, 72, 74, 80, 81, 86, 86, 92, 93, 92, 92, 91, 91, 91, 93, 93,
        96, 96, 99, 100, 102, 104, 106, 109, 110, 115, 115, 70, 72, 72, 74, 74,
        78, 81, 84, 90, 90, 92, 92, 95, 95, 94, 93, 92, 91, 92, 93, 93, 95, 95,
        97, 98, 100, 102, 103, 106, 107, 112, 112, 74, 77, 77, 79, 80, 83, 86,
        89, 94, 94, 95, 95, 96, 96, 95, 94, 93, 92, 92, 93, 93, 94, 95, 97, 98,
        99, 101, 102, 105, 106, 110, 110, 84, 85, 85, 85, 85, 88, 90, 91, 95,
        95, 97, 97, 100, 101, 100, 100, 99, 99, 99, 100, 100, 101, 101, 103,
        104, 105, 107, 108, 111, 111, 116, 116, 96, 94, 94, 93, 92, 93, 94, 94,
        95, 96, 100, 100, 105, 106, 106, 106, 106, 107, 107, 108, 108, 109, 109,
        110, 111, 112, 113, 114, 117, 118, 121, 121, 97, 95, 94, 92, 92, 92, 93,
        94, 94, 95, 100, 100, 105, 106, 107, 108, 108, 110, 110, 111, 111, 112,
        112, 114, 115, 116, 117, 118, 121, 121, 125, 125, 98, 95, 94, 92, 91,
        91, 92, 91, 91, 92, 99, 99, 106, 107, 110, 112, 114, 117, 117, 119, 119,
        122, 122, 124, 125, 126, 128, 129, 131, 131, 135, 135, 99, 95, 95, 92,
        91, 91, 92, 92, 92, 92, 99, 99, 106, 108, 110, 112, 114, 118, 118, 120,
        120, 123, 123, 125, 126, 127, 129, 130, 132, 133, 136, 136, 105, 100,
        100, 97, 96, 96, 95, 95, 94, 95, 101, 101, 107, 109, 113, 115, 118, 123,
        124, 128, 128, 133, 133, 136, 137, 139, 141, 142, 145, 146, 150, 150,
        105, 100, 100, 97, 96, 96, 95, 95, 94, 95, 101, 101, 107, 109, 113, 115,
        118, 123, 124, 128, 128, 133, 133, 136, 137, 139, 141, 142, 145, 146,
        150, 150, 112, 108, 107, 104, 102, 102, 101, 100, 99, 99, 106, 106, 111,
        113, 116, 119, 122, 127, 128, 133, 133, 139, 140, 143, 145, 148, 151,
        152, 155, 156, 161, 161, 115, 109, 109, 105, 104, 103, 103, 102, 100,
        101, 107, 107, 112, 114, 117, 120, 123, 128, 129, 134, 134, 141, 142,
        145, 147, 150, 153, 155, 158, 159, 164, 164, 122, 116, 115, 112, 110,
        109, 108, 107, 105, 106, 112, 112, 117, 118, 122, 124, 127, 132, 133,
        139, 139, 145, 146, 150, 152, 156, 159, 161, 165, 166, 172, 172, 127,
        121, 120, 116, 115, 114, 112, 111, 109, 110, 115, 115, 120, 121, 125,
        128, 130, 135, 136, 142, 142, 149, 150, 154, 156, 160, 164, 166, 170,
        171, 178, 178,
        /* Size 4x16 */
        62, 84, 99, 115, 63, 85, 95, 109, 64, 85, 91, 104, 71, 90, 92, 103, 80,
        95, 92, 100, 86, 97, 99, 107, 92, 100, 106, 112, 93, 100, 110, 117, 92,
        99, 114, 123, 92, 99, 118, 129, 94, 100, 120, 134, 97, 101, 123, 142,
        101, 104, 126, 147, 105, 107, 129, 153, 110, 111, 132, 158, 117, 116,
        136, 164,
        /* Size 16x4 */
        62, 63, 64, 71, 80, 86, 92, 93, 92, 92, 94, 97, 101, 105, 110, 117, 84,
        85, 85, 90, 95, 97, 100, 100, 99, 99, 100, 101, 104, 107, 111, 116, 99,
        95, 91, 92, 92, 99, 106, 110, 114, 118, 120, 123, 126, 129, 132, 136,
        115, 109, 104, 103, 100, 107, 112, 117, 123, 129, 134, 142, 147, 153,
        158, 164,
        /* Size 8x32 */
        64, 62, 74, 96, 98, 105, 112, 122, 63, 63, 77, 94, 95, 100, 108, 116,
        63, 63, 77, 94, 94, 100, 107, 115, 61, 64, 79, 93, 92, 97, 104, 112, 61,
        64, 80, 92, 91, 96, 102, 110, 64, 68, 83, 93, 91, 96, 102, 109, 67, 72,
        86, 94, 92, 95, 101, 108, 69, 74, 89, 94, 91, 95, 100, 107, 74, 80, 94,
        95, 91, 94, 99, 105, 75, 81, 94, 96, 92, 95, 99, 106, 84, 86, 95, 100,
        99, 101, 106, 112, 84, 86, 95, 100, 99, 101, 106, 112, 95, 92, 96, 105,
        106, 107, 111, 117, 98, 93, 96, 106, 107, 109, 113, 118, 97, 92, 95,
        106, 110, 113, 116, 122, 97, 92, 94, 106, 112, 115, 119, 124, 97, 91,
        93, 106, 114, 118, 122, 127, 98, 91, 92, 107, 117, 123, 127, 132, 98,
        91, 92, 107, 117, 124, 128, 133, 101, 93, 93, 108, 119, 128, 133, 139,
        101, 93, 93, 108, 119, 128, 133, 139, 104, 96, 94, 109, 122, 133, 139,
        145, 104, 96, 95, 109, 122, 133, 140, 146, 107, 99, 97, 110, 124, 136,
        143, 150, 109, 100, 98, 111, 125, 137, 145, 152, 111, 102, 99, 112, 126,
        139, 148, 156, 114, 104, 101, 113, 128, 141, 151, 159, 116, 106, 102,
        114, 129, 142, 152, 161, 120, 109, 105, 117, 131, 145, 155, 165, 120,
        110, 106, 118, 131, 146, 156, 166, 127, 115, 110, 121, 135, 150, 161,
        172, 127, 115, 110, 121, 135, 150, 161, 172,
        /* Size 32x8 */
        64, 63, 63, 61, 61, 64, 67, 69, 74, 75, 84, 84, 95, 98, 97, 97, 97, 98,
        98, 101, 101, 104, 104, 107, 109, 111, 114, 116, 120, 120, 127, 127, 62,
        63, 63, 64, 64, 68, 72, 74, 80, 81, 86, 86, 92, 93, 92, 92, 91, 91, 91,
        93, 93, 96, 96, 99, 100, 102, 104, 106, 109, 110, 115, 115, 74, 77, 77,
        79, 80, 83, 86, 89, 94, 94, 95, 95, 96, 96, 95, 94, 93, 92, 92, 93, 93,
        94, 95, 97, 98, 99, 101, 102, 105, 106, 110, 110, 96, 94, 94, 93, 92,
        93, 94, 94, 95, 96, 100, 100, 105, 106, 106, 106, 106, 107, 107, 108,
        108, 109, 109, 110, 111, 112, 113, 114, 117, 118, 121, 121, 98, 95, 94,
        92, 91, 91, 92, 91, 91, 92, 99, 99, 106, 107, 110, 112, 114, 117, 117,
        119, 119, 122, 122, 124, 125, 126, 128, 129, 131, 131, 135, 135, 105,
        100, 100, 97, 96, 96, 95, 95, 94, 95, 101, 101, 107, 109, 113, 115, 118,
        123, 124, 128, 128, 133, 133, 136, 137, 139, 141, 142, 145, 146, 150,
        150, 112, 108, 107, 104, 102, 102, 101, 100, 99, 99, 106, 106, 111, 113,
        116, 119, 122, 127, 128, 133, 133, 139, 140, 143, 145, 148, 151, 152,
        155, 156, 161, 161, 122, 116, 115, 112, 110, 109, 108, 107, 105, 106,
        112, 112, 117, 118, 122, 124, 127, 132, 133, 139, 139, 145, 146, 150,
        152, 156, 159, 161, 165, 166, 172, 172 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        67, 94, 94, 150,
#endif
        /* Size 4x4 */
        64, 67, 84, 111, 67, 77, 93, 115, 84, 93, 127, 151, 111, 115, 151, 184,
        /* Size 8x8 */
        63, 64, 64, 69, 77, 92, 104, 126, 64, 65, 65, 68, 75, 88, 99, 119, 64,
        65, 71, 75, 81, 90, 99, 116, 69, 68, 75, 85, 94, 105, 113, 130, 77, 75,
        81, 94, 108, 121, 130, 147, 92, 88, 90, 105, 121, 139, 151, 168, 104,
        99, 99, 113, 130, 151, 165, 185, 126, 119, 116, 130, 147, 168, 185, 210,
        /* Size 16x16 */
        64, 63, 63, 63, 64, 64, 68, 72, 77, 82, 88, 97, 108, 116, 123, 131, 63,
        64, 64, 64, 64, 65, 68, 70, 76, 80, 85, 93, 103, 111, 117, 125, 63, 64,
        65, 65, 65, 65, 67, 69, 74, 77, 82, 89, 99, 106, 112, 119, 63, 64, 65,
        66, 67, 67, 70, 72, 76, 80, 84, 91, 99, 106, 112, 119, 64, 64, 65, 67,
        68, 69, 72, 74, 78, 81, 85, 91, 99, 106, 111, 118, 64, 65, 65, 67, 69,
        71, 74, 77, 80, 83, 85, 92, 99, 105, 110, 116, 68, 68, 67, 70, 72, 74,
        79, 85, 89, 92, 95, 102, 109, 115, 120, 126, 72, 70, 69, 72, 74, 77, 85,
        97, 101, 104, 108, 114, 120, 126, 131, 136, 77, 76, 74, 76, 78, 80, 89,
        101, 105, 109, 114, 121, 128, 134, 139, 144, 82, 80, 77, 80, 81, 83, 92,
        104, 109, 114, 120, 127, 135, 141, 146, 151, 88, 85, 82, 84, 85, 85, 95,
        108, 114, 120, 127, 134, 143, 149, 154, 159, 97, 93, 89, 91, 91, 92,
        102, 114, 121, 127, 134, 142, 153, 159, 165, 171, 108, 103, 99, 99, 99,
        99, 109, 120, 128, 135, 143, 153, 165, 172, 178, 185, 116, 111, 106,
        106, 106, 105, 115, 126, 134, 141, 149, 159, 172, 180, 187, 194, 123,
        117, 112, 112, 111, 110, 120, 131, 139, 146, 154, 165, 178, 187, 194,
        202, 131, 125, 119, 119, 118, 116, 126, 136, 144, 151, 159, 171, 185,
        194, 202, 210,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 67, 68, 69, 72, 72, 77, 79,
        82, 88, 88, 95, 97, 101, 108, 108, 116, 118, 123, 131, 131, 140, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 67, 68, 69, 71, 71, 76, 77, 80, 85,
        85, 92, 94, 98, 104, 104, 112, 114, 118, 126, 126, 135, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 65, 65, 67, 68, 69, 70, 70, 76, 77, 80, 85, 85, 91,
        93, 97, 103, 103, 111, 113, 117, 125, 125, 134, 63, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 67, 68, 68, 70, 70, 75, 76, 79, 84, 84, 90, 91, 95,
        101, 101, 109, 111, 115, 122, 122, 131, 63, 64, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 67, 67, 68, 69, 69, 74, 75, 77, 82, 82, 88, 89, 93, 99, 99,
        106, 108, 112, 119, 119, 128, 63, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 67, 67, 68, 69, 69, 74, 75, 77, 82, 82, 88, 89, 93, 99, 99, 106,
        108, 112, 119, 119, 128, 63, 64, 64, 65, 65, 65, 66, 66, 67, 67, 67, 69,
        70, 71, 72, 72, 76, 78, 80, 84, 84, 89, 91, 94, 99, 99, 106, 108, 112,
        119, 119, 127, 63, 64, 64, 65, 65, 65, 66, 67, 67, 68, 68, 70, 71, 72,
        73, 73, 77, 78, 81, 84, 84, 90, 91, 94, 100, 100, 106, 108, 112, 118,
        118, 127, 64, 64, 64, 65, 65, 65, 67, 67, 68, 69, 69, 71, 72, 73, 74,
        74, 78, 79, 81, 85, 85, 90, 91, 94, 99, 99, 106, 108, 111, 118, 118,
        126, 64, 64, 65, 65, 65, 65, 67, 68, 69, 71, 71, 73, 74, 75, 77, 77, 80,
        81, 83, 85, 85, 90, 92, 94, 99, 99, 105, 107, 110, 116, 116, 124, 64,
        64, 65, 65, 65, 65, 67, 68, 69, 71, 71, 73, 74, 75, 77, 77, 80, 81, 83,
        85, 85, 90, 92, 94, 99, 99, 105, 107, 110, 116, 116, 124, 67, 67, 67,
        67, 67, 67, 69, 70, 71, 73, 73, 77, 78, 80, 84, 84, 87, 88, 90, 93, 93,
        98, 100, 102, 107, 107, 113, 115, 118, 124, 124, 132, 68, 68, 68, 68,
        67, 67, 70, 71, 72, 74, 74, 78, 79, 82, 85, 85, 89, 90, 92, 95, 95, 100,
        102, 105, 109, 109, 115, 117, 120, 126, 126, 134, 69, 69, 69, 68, 68,
        68, 71, 72, 73, 75, 75, 80, 82, 85, 90, 90, 93, 94, 97, 100, 100, 105,
        106, 109, 113, 113, 119, 121, 124, 130, 130, 138, 72, 71, 70, 70, 69,
        69, 72, 73, 74, 77, 77, 84, 85, 90, 97, 97, 101, 101, 104, 108, 108,
        112, 114, 116, 120, 120, 126, 128, 131, 136, 136, 144, 72, 71, 70, 70,
        69, 69, 72, 73, 74, 77, 77, 84, 85, 90, 97, 97, 101, 101, 104, 108, 108,
        112, 114, 116, 120, 120, 126, 128, 131, 136, 136, 144, 77, 76, 76, 75,
        74, 74, 76, 77, 78, 80, 80, 87, 89, 93, 101, 101, 105, 106, 109, 114,
        114, 119, 121, 123, 128, 128, 134, 136, 139, 144, 144, 152, 79, 77, 77,
        76, 75, 75, 78, 78, 79, 81, 81, 88, 90, 94, 101, 101, 106, 108, 111,
        116, 116, 121, 122, 125, 130, 130, 136, 138, 141, 147, 147, 154, 82, 80,
        80, 79, 77, 77, 80, 81, 81, 83, 83, 90, 92, 97, 104, 104, 109, 111, 114,
        120, 120, 125, 127, 130, 135, 135, 141, 143, 146, 151, 151, 159, 88, 85,
        85, 84, 82, 82, 84, 84, 85, 85, 85, 93, 95, 100, 108, 108, 114, 116,
        120, 127, 127, 133, 134, 137, 143, 143, 149, 151, 154, 159, 159, 167,
        88, 85, 85, 84, 82, 82, 84, 84, 85, 85, 85, 93, 95, 100, 108, 108, 114,
        116, 120, 127, 127, 133, 134, 137, 143, 143, 149, 151, 154, 159, 159,
        167, 95, 92, 91, 90, 88, 88, 89, 90, 90, 90, 90, 98, 100, 105, 112, 112,
        119, 121, 125, 133, 133, 139, 141, 144, 151, 151, 157, 159, 163, 168,
        168, 176, 97, 94, 93, 91, 89, 89, 91, 91, 91, 92, 92, 100, 102, 106,
        114, 114, 121, 122, 127, 134, 134, 141, 142, 146, 153, 153, 159, 161,
        165, 171, 171, 179, 101, 98, 97, 95, 93, 93, 94, 94, 94, 94, 94, 102,
        105, 109, 116, 116, 123, 125, 130, 137, 137, 144, 146, 150, 157, 157,
        164, 166, 170, 176, 176, 184, 108, 104, 103, 101, 99, 99, 99, 100, 99,
        99, 99, 107, 109, 113, 120, 120, 128, 130, 135, 143, 143, 151, 153, 157,
        165, 165, 172, 174, 178, 185, 185, 193, 108, 104, 103, 101, 99, 99, 99,
        100, 99, 99, 99, 107, 109, 113, 120, 120, 128, 130, 135, 143, 143, 151,
        153, 157, 165, 165, 172, 174, 178, 185, 185, 193, 116, 112, 111, 109,
        106, 106, 106, 106, 106, 105, 105, 113, 115, 119, 126, 126, 134, 136,
        141, 149, 149, 157, 159, 164, 172, 172, 180, 182, 187, 194, 194, 203,
        118, 114, 113, 111, 108, 108, 108, 108, 108, 107, 107, 115, 117, 121,
        128, 128, 136, 138, 143, 151, 151, 159, 161, 166, 174, 174, 182, 184,
        189, 196, 196, 205, 123, 118, 117, 115, 112, 112, 112, 112, 111, 110,
        110, 118, 120, 124, 131, 131, 139, 141, 146, 154, 154, 163, 165, 170,
        178, 178, 187, 189, 194, 202, 202, 211, 131, 126, 125, 122, 119, 119,
        119, 118, 118, 116, 116, 124, 126, 130, 136, 136, 144, 147, 151, 159,
        159, 168, 171, 176, 185, 185, 194, 196, 202, 210, 210, 219, 131, 126,
        125, 122, 119, 119, 119, 118, 118, 116, 116, 124, 126, 130, 136, 136,
        144, 147, 151, 159, 159, 168, 171, 176, 185, 185, 194, 196, 202, 210,
        210, 219, 140, 135, 134, 131, 128, 128, 127, 127, 126, 124, 124, 132,
        134, 138, 144, 144, 152, 154, 159, 167, 167, 176, 179, 184, 193, 193,
        203, 205, 211, 219, 219, 229,
        /* Size 4x8 */
        64, 65, 85, 112, 64, 66, 82, 106, 65, 71, 85, 105, 69, 75, 100, 119, 76,
        81, 116, 136, 89, 91, 132, 157, 101, 100, 142, 172, 122, 117, 159, 194,
        /* Size 8x4 */
        64, 64, 65, 69, 76, 89, 101, 122, 65, 66, 71, 75, 81, 91, 100, 117, 85,
        82, 85, 100, 116, 132, 142, 159, 112, 106, 105, 119, 136, 157, 172, 194,
        /* Size 8x16 */
        64, 63, 64, 70, 78, 88, 107, 131, 63, 64, 65, 70, 76, 85, 103, 124, 63,
        65, 66, 68, 74, 82, 99, 119, 63, 65, 68, 71, 77, 84, 99, 118, 64, 65,
        69, 73, 78, 85, 99, 117, 64, 66, 71, 75, 80, 85, 99, 116, 68, 68, 74,
        83, 89, 96, 109, 126, 72, 69, 77, 92, 100, 108, 120, 136, 77, 74, 81,
        95, 105, 114, 128, 144, 82, 78, 83, 99, 109, 120, 135, 152, 88, 82, 86,
        102, 114, 126, 143, 159, 96, 90, 93, 108, 120, 134, 152, 171, 107, 99,
        100, 115, 128, 142, 164, 185, 115, 106, 106, 121, 134, 149, 172, 194,
        122, 113, 112, 126, 139, 154, 178, 201, 130, 120, 117, 132, 144, 159,
        185, 210,
        /* Size 16x8 */
        64, 63, 63, 63, 64, 64, 68, 72, 77, 82, 88, 96, 107, 115, 122, 130, 63,
        64, 65, 65, 65, 66, 68, 69, 74, 78, 82, 90, 99, 106, 113, 120, 64, 65,
        66, 68, 69, 71, 74, 77, 81, 83, 86, 93, 100, 106, 112, 117, 70, 70, 68,
        71, 73, 75, 83, 92, 95, 99, 102, 108, 115, 121, 126, 132, 78, 76, 74,
        77, 78, 80, 89, 100, 105, 109, 114, 120, 128, 134, 139, 144, 88, 85, 82,
        84, 85, 85, 96, 108, 114, 120, 126, 134, 142, 149, 154, 159, 107, 103,
        99, 99, 99, 99, 109, 120, 128, 135, 143, 152, 164, 172, 178, 185, 131,
        124, 119, 118, 117, 116, 126, 136, 144, 152, 159, 171, 185, 194, 201,
        210,
        /* Size 16x32 */
        64, 63, 63, 63, 64, 64, 70, 72, 78, 88, 88, 103, 107, 116, 131, 131, 63,
        64, 64, 64, 65, 65, 70, 71, 76, 85, 85, 99, 104, 112, 126, 126, 63, 64,
        64, 64, 65, 65, 70, 71, 76, 85, 85, 99, 103, 111, 124, 124, 63, 64, 64,
        65, 65, 65, 69, 70, 75, 83, 83, 97, 101, 109, 122, 122, 63, 64, 65, 65,
        66, 66, 68, 69, 74, 82, 82, 95, 99, 106, 119, 119, 63, 64, 65, 65, 66,
        66, 68, 69, 74, 82, 82, 95, 99, 106, 119, 119, 63, 65, 65, 66, 68, 68,
        71, 72, 77, 84, 84, 96, 99, 106, 118, 118, 64, 65, 65, 66, 68, 68, 72,
        73, 77, 84, 84, 96, 100, 106, 118, 118, 64, 65, 65, 67, 69, 69, 73, 75,
        78, 85, 85, 96, 99, 106, 117, 117, 64, 65, 66, 68, 71, 71, 75, 77, 80,
        85, 85, 96, 99, 105, 116, 116, 64, 65, 66, 68, 71, 71, 75, 77, 80, 85,
        85, 96, 99, 105, 116, 116, 67, 67, 67, 70, 73, 73, 81, 83, 87, 93, 93,
        104, 107, 113, 124, 124, 68, 68, 68, 70, 74, 74, 83, 85, 89, 96, 96,
        106, 109, 115, 126, 126, 69, 69, 68, 71, 75, 75, 86, 89, 93, 100, 100,
        110, 113, 119, 130, 130, 72, 70, 69, 72, 77, 77, 92, 96, 100, 108, 108,
        117, 120, 126, 136, 136, 72, 70, 69, 72, 77, 77, 92, 96, 100, 108, 108,
        117, 120, 126, 136, 136, 77, 75, 74, 77, 81, 81, 95, 100, 105, 114, 114,
        125, 128, 134, 144, 144, 79, 76, 75, 78, 81, 81, 96, 101, 106, 116, 116,
        127, 130, 136, 147, 147, 82, 79, 78, 80, 83, 83, 99, 103, 109, 120, 120,
        132, 135, 141, 152, 152, 88, 83, 82, 84, 86, 86, 102, 107, 114, 126,
        126, 139, 143, 149, 159, 159, 88, 83, 82, 84, 86, 86, 102, 107, 114,
        126, 126, 139, 143, 149, 159, 159, 94, 89, 88, 89, 91, 91, 107, 112,
        119, 132, 132, 146, 150, 157, 168, 168, 96, 91, 90, 91, 93, 93, 108,
        113, 120, 134, 134, 148, 152, 159, 171, 171, 100, 95, 93, 94, 95, 95,
        111, 116, 123, 137, 137, 152, 157, 164, 176, 176, 107, 101, 99, 100,
        100, 100, 115, 120, 128, 142, 142, 159, 164, 172, 185, 185, 107, 101,
        99, 100, 100, 100, 115, 120, 128, 142, 142, 159, 164, 172, 185, 185,
        115, 108, 106, 106, 106, 106, 121, 126, 134, 149, 149, 166, 172, 180,
        194, 194, 117, 110, 108, 108, 108, 108, 123, 127, 136, 150, 150, 168,
        174, 182, 196, 196, 122, 114, 113, 112, 112, 112, 126, 131, 139, 154,
        154, 172, 178, 187, 201, 201, 130, 122, 120, 119, 117, 117, 132, 136,
        144, 159, 159, 179, 185, 194, 210, 210, 130, 122, 120, 119, 117, 117,
        132, 136, 144, 159, 159, 179, 185, 194, 210, 210, 140, 130, 128, 127,
        125, 125, 140, 144, 152, 167, 167, 187, 193, 203, 219, 219,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 67, 68, 69, 72, 72, 77, 79,
        82, 88, 88, 94, 96, 100, 107, 107, 115, 117, 122, 130, 130, 140, 63, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 65, 67, 68, 69, 70, 70, 75, 76, 79, 83,
        83, 89, 91, 95, 101, 101, 108, 110, 114, 122, 122, 130, 63, 64, 64, 64,
        65, 65, 65, 65, 65, 66, 66, 67, 68, 68, 69, 69, 74, 75, 78, 82, 82, 88,
        90, 93, 99, 99, 106, 108, 113, 120, 120, 128, 63, 64, 64, 65, 65, 65,
        66, 66, 67, 68, 68, 70, 70, 71, 72, 72, 77, 78, 80, 84, 84, 89, 91, 94,
        100, 100, 106, 108, 112, 119, 119, 127, 64, 65, 65, 65, 66, 66, 68, 68,
        69, 71, 71, 73, 74, 75, 77, 77, 81, 81, 83, 86, 86, 91, 93, 95, 100,
        100, 106, 108, 112, 117, 117, 125, 64, 65, 65, 65, 66, 66, 68, 68, 69,
        71, 71, 73, 74, 75, 77, 77, 81, 81, 83, 86, 86, 91, 93, 95, 100, 100,
        106, 108, 112, 117, 117, 125, 70, 70, 70, 69, 68, 68, 71, 72, 73, 75,
        75, 81, 83, 86, 92, 92, 95, 96, 99, 102, 102, 107, 108, 111, 115, 115,
        121, 123, 126, 132, 132, 140, 72, 71, 71, 70, 69, 69, 72, 73, 75, 77,
        77, 83, 85, 89, 96, 96, 100, 101, 103, 107, 107, 112, 113, 116, 120,
        120, 126, 127, 131, 136, 136, 144, 78, 76, 76, 75, 74, 74, 77, 77, 78,
        80, 80, 87, 89, 93, 100, 100, 105, 106, 109, 114, 114, 119, 120, 123,
        128, 128, 134, 136, 139, 144, 144, 152, 88, 85, 85, 83, 82, 82, 84, 84,
        85, 85, 85, 93, 96, 100, 108, 108, 114, 116, 120, 126, 126, 132, 134,
        137, 142, 142, 149, 150, 154, 159, 159, 167, 88, 85, 85, 83, 82, 82, 84,
        84, 85, 85, 85, 93, 96, 100, 108, 108, 114, 116, 120, 126, 126, 132,
        134, 137, 142, 142, 149, 150, 154, 159, 159, 167, 103, 99, 99, 97, 95,
        95, 96, 96, 96, 96, 96, 104, 106, 110, 117, 117, 125, 127, 132, 139,
        139, 146, 148, 152, 159, 159, 166, 168, 172, 179, 179, 187, 107, 104,
        103, 101, 99, 99, 99, 100, 99, 99, 99, 107, 109, 113, 120, 120, 128,
        130, 135, 143, 143, 150, 152, 157, 164, 164, 172, 174, 178, 185, 185,
        193, 116, 112, 111, 109, 106, 106, 106, 106, 106, 105, 105, 113, 115,
        119, 126, 126, 134, 136, 141, 149, 149, 157, 159, 164, 172, 172, 180,
        182, 187, 194, 194, 203, 131, 126, 124, 122, 119, 119, 118, 118, 117,
        116, 116, 124, 126, 130, 136, 136, 144, 147, 152, 159, 159, 168, 171,
        176, 185, 185, 194, 196, 201, 210, 210, 219, 131, 126, 124, 122, 119,
        119, 118, 118, 117, 116, 116, 124, 126, 130, 136, 136, 144, 147, 152,
        159, 159, 168, 171, 176, 185, 185, 194, 196, 201, 210, 210, 219,
        /* Size 4x16 */
        63, 64, 88, 116, 64, 65, 85, 111, 64, 66, 82, 106, 65, 68, 84, 106, 65,
        69, 85, 106, 65, 71, 85, 105, 68, 74, 96, 115, 70, 77, 108, 126, 75, 81,
        114, 134, 79, 83, 120, 141, 83, 86, 126, 149, 91, 93, 134, 159, 101,
        100, 142, 172, 108, 106, 149, 180, 114, 112, 154, 187, 122, 117, 159,
        194,
        /* Size 16x4 */
        63, 64, 64, 65, 65, 65, 68, 70, 75, 79, 83, 91, 101, 108, 114, 122, 64,
        65, 66, 68, 69, 71, 74, 77, 81, 83, 86, 93, 100, 106, 112, 117, 88, 85,
        82, 84, 85, 85, 96, 108, 114, 120, 126, 134, 142, 149, 154, 159, 116,
        111, 106, 106, 106, 105, 115, 126, 134, 141, 149, 159, 172, 180, 187,
        194,
        /* Size 8x32 */
        64, 63, 64, 70, 78, 88, 107, 131, 63, 64, 65, 70, 76, 85, 104, 126, 63,
        64, 65, 70, 76, 85, 103, 124, 63, 64, 65, 69, 75, 83, 101, 122, 63, 65,
        66, 68, 74, 82, 99, 119, 63, 65, 66, 68, 74, 82, 99, 119, 63, 65, 68,
        71, 77, 84, 99, 118, 64, 65, 68, 72, 77, 84, 100, 118, 64, 65, 69, 73,
        78, 85, 99, 117, 64, 66, 71, 75, 80, 85, 99, 116, 64, 66, 71, 75, 80,
        85, 99, 116, 67, 67, 73, 81, 87, 93, 107, 124, 68, 68, 74, 83, 89, 96,
        109, 126, 69, 68, 75, 86, 93, 100, 113, 130, 72, 69, 77, 92, 100, 108,
        120, 136, 72, 69, 77, 92, 100, 108, 120, 136, 77, 74, 81, 95, 105, 114,
        128, 144, 79, 75, 81, 96, 106, 116, 130, 147, 82, 78, 83, 99, 109, 120,
        135, 152, 88, 82, 86, 102, 114, 126, 143, 159, 88, 82, 86, 102, 114,
        126, 143, 159, 94, 88, 91, 107, 119, 132, 150, 168, 96, 90, 93, 108,
        120, 134, 152, 171, 100, 93, 95, 111, 123, 137, 157, 176, 107, 99, 100,
        115, 128, 142, 164, 185, 107, 99, 100, 115, 128, 142, 164, 185, 115,
        106, 106, 121, 134, 149, 172, 194, 117, 108, 108, 123, 136, 150, 174,
        196, 122, 113, 112, 126, 139, 154, 178, 201, 130, 120, 117, 132, 144,
        159, 185, 210, 130, 120, 117, 132, 144, 159, 185, 210, 140, 128, 125,
        140, 152, 167, 193, 219,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 67, 68, 69, 72, 72, 77, 79,
        82, 88, 88, 94, 96, 100, 107, 107, 115, 117, 122, 130, 130, 140, 63, 64,
        64, 64, 65, 65, 65, 65, 65, 66, 66, 67, 68, 68, 69, 69, 74, 75, 78, 82,
        82, 88, 90, 93, 99, 99, 106, 108, 113, 120, 120, 128, 64, 65, 65, 65,
        66, 66, 68, 68, 69, 71, 71, 73, 74, 75, 77, 77, 81, 81, 83, 86, 86, 91,
        93, 95, 100, 100, 106, 108, 112, 117, 117, 125, 70, 70, 70, 69, 68, 68,
        71, 72, 73, 75, 75, 81, 83, 86, 92, 92, 95, 96, 99, 102, 102, 107, 108,
        111, 115, 115, 121, 123, 126, 132, 132, 140, 78, 76, 76, 75, 74, 74, 77,
        77, 78, 80, 80, 87, 89, 93, 100, 100, 105, 106, 109, 114, 114, 119, 120,
        123, 128, 128, 134, 136, 139, 144, 144, 152, 88, 85, 85, 83, 82, 82, 84,
        84, 85, 85, 85, 93, 96, 100, 108, 108, 114, 116, 120, 126, 126, 132,
        134, 137, 142, 142, 149, 150, 154, 159, 159, 167, 107, 104, 103, 101,
        99, 99, 99, 100, 99, 99, 99, 107, 109, 113, 120, 120, 128, 130, 135,
        143, 143, 150, 152, 157, 164, 164, 172, 174, 178, 185, 185, 193, 131,
        126, 124, 122, 119, 119, 118, 118, 117, 116, 116, 124, 126, 130, 136,
        136, 144, 147, 152, 159, 159, 168, 171, 176, 185, 185, 194, 196, 201,
        210, 210, 219 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        78, 94, 94, 127,
#endif
        /* Size 4x4 */
        63, 83, 92, 102, 83, 96, 97, 102, 92, 97, 117, 125, 102, 102, 125, 142,
        /* Size 8x8 */
        63, 62, 76, 89, 95, 97, 101, 110, 62, 64, 80, 88, 91, 92, 95, 104, 76,
        80, 95, 95, 93, 92, 94, 100, 89, 88, 95, 100, 102, 103, 104, 109, 95,
        91, 93, 102, 109, 113, 115, 120, 97, 92, 92, 103, 113, 122, 126, 133,
        101, 95, 94, 104, 115, 126, 133, 141, 110, 104, 100, 109, 120, 133, 141,
        153,
        /* Size 16x16 */
        64, 62, 61, 66, 69, 73, 83, 98, 97, 98, 98, 101, 104, 108, 111, 114, 62,
        63, 62, 68, 72, 77, 85, 95, 94, 94, 94, 97, 100, 103, 106, 109, 61, 62,
        64, 69, 74, 80, 85, 93, 91, 91, 90, 93, 95, 99, 101, 104, 66, 68, 69,
        75, 80, 85, 89, 94, 93, 92, 91, 93, 95, 98, 100, 103, 69, 72, 74, 80,
        85, 90, 92, 95, 93, 92, 91, 93, 94, 97, 99, 101, 73, 77, 80, 85, 90, 95,
        95, 96, 94, 92, 91, 92, 94, 96, 98, 100, 83, 85, 85, 89, 92, 95, 97,
        101, 100, 99, 98, 100, 101, 103, 105, 107, 98, 95, 93, 94, 95, 96, 101,
        106, 106, 107, 107, 108, 109, 111, 112, 113, 97, 94, 91, 93, 93, 94,
        100, 106, 108, 109, 111, 112, 114, 116, 117, 118, 98, 94, 91, 92, 92,
        92, 99, 107, 109, 111, 114, 116, 118, 120, 121, 123, 98, 94, 90, 91, 91,
        91, 98, 107, 111, 114, 117, 120, 123, 124, 126, 127, 101, 97, 93, 93,
        93, 92, 100, 108, 112, 116, 120, 123, 127, 130, 132, 134, 104, 100, 95,
        95, 94, 94, 101, 109, 114, 118, 123, 127, 133, 136, 139, 141, 108, 103,
        99, 98, 97, 96, 103, 111, 116, 120, 124, 130, 136, 140, 143, 146, 111,
        106, 101, 100, 99, 98, 105, 112, 117, 121, 126, 132, 139, 143, 146, 149,
        114, 109, 104, 103, 101, 100, 107, 113, 118, 123, 127, 134, 141, 146,
        149, 153,
        /* Size 32x32 */
        64, 63, 62, 62, 61, 61, 66, 67, 69, 73, 73, 81, 83, 89, 98, 98, 97, 97,
        98, 98, 98, 101, 101, 103, 104, 104, 108, 109, 111, 114, 114, 119, 63,
        63, 63, 62, 62, 62, 67, 69, 72, 76, 76, 83, 84, 89, 96, 96, 95, 95, 95,
        95, 95, 97, 98, 99, 101, 101, 104, 105, 107, 110, 110, 114, 62, 63, 63,
        63, 62, 62, 68, 69, 72, 77, 77, 83, 85, 89, 95, 95, 94, 94, 94, 94, 94,
        96, 97, 98, 100, 100, 103, 104, 106, 109, 109, 113, 62, 62, 63, 63, 63,
        63, 68, 70, 73, 78, 78, 83, 85, 88, 94, 94, 93, 93, 93, 92, 92, 95, 95,
        96, 98, 98, 101, 102, 104, 107, 107, 111, 61, 62, 62, 63, 64, 64, 69,
        71, 74, 80, 80, 84, 85, 88, 93, 93, 91, 91, 91, 90, 90, 92, 93, 94, 95,
        95, 99, 99, 101, 104, 104, 108, 61, 62, 62, 63, 64, 64, 69, 71, 74, 80,
        80, 84, 85, 88, 93, 93, 91, 91, 91, 90, 90, 92, 93, 94, 95, 95, 99, 99,
        101, 104, 104, 108, 66, 67, 68, 68, 69, 69, 75, 77, 80, 85, 85, 88, 89,
        91, 94, 94, 93, 92, 92, 91, 91, 93, 93, 94, 95, 95, 98, 99, 100, 103,
        103, 106, 67, 69, 69, 70, 71, 71, 77, 78, 81, 87, 87, 89, 90, 92, 95,
        95, 93, 93, 92, 91, 91, 93, 93, 94, 95, 95, 98, 99, 100, 102, 102, 106,
        69, 72, 72, 73, 74, 74, 80, 81, 85, 90, 90, 91, 92, 93, 95, 95, 93, 93,
        92, 91, 91, 92, 93, 94, 94, 94, 97, 98, 99, 101, 101, 105, 73, 76, 77,
        78, 80, 80, 85, 87, 90, 95, 95, 95, 95, 95, 96, 96, 94, 93, 92, 91, 91,
        92, 92, 93, 94, 94, 96, 97, 98, 100, 100, 103, 73, 76, 77, 78, 80, 80,
        85, 87, 90, 95, 95, 95, 95, 95, 96, 96, 94, 93, 92, 91, 91, 92, 92, 93,
        94, 94, 96, 97, 98, 100, 100, 103, 81, 83, 83, 83, 84, 84, 88, 89, 91,
        95, 95, 96, 97, 98, 100, 100, 98, 98, 98, 97, 97, 98, 98, 99, 99, 99,
        102, 102, 103, 105, 105, 108, 83, 84, 85, 85, 85, 85, 89, 90, 92, 95,
        95, 97, 97, 99, 101, 101, 100, 99, 99, 98, 98, 100, 100, 100, 101, 101,
        103, 104, 105, 107, 107, 110, 89, 89, 89, 88, 88, 88, 91, 92, 93, 95,
        95, 98, 99, 100, 103, 103, 102, 102, 102, 102, 102, 103, 103, 103, 104,
        104, 106, 107, 108, 109, 109, 112, 98, 96, 95, 94, 93, 93, 94, 95, 95,
        96, 96, 100, 101, 103, 106, 106, 106, 106, 107, 107, 107, 108, 108, 108,
        109, 109, 111, 111, 112, 113, 113, 116, 98, 96, 95, 94, 93, 93, 94, 95,
        95, 96, 96, 100, 101, 103, 106, 106, 106, 106, 107, 107, 107, 108, 108,
        108, 109, 109, 111, 111, 112, 113, 113, 116, 97, 95, 94, 93, 91, 91, 93,
        93, 93, 94, 94, 98, 100, 102, 106, 106, 108, 108, 109, 111, 111, 112,
        112, 113, 114, 114, 116, 116, 117, 118, 118, 121, 97, 95, 94, 93, 91,
        91, 92, 93, 93, 93, 93, 98, 99, 102, 106, 106, 108, 109, 110, 111, 111,
        113, 113, 114, 115, 115, 117, 117, 118, 120, 120, 123, 98, 95, 94, 93,
        91, 91, 92, 92, 92, 92, 92, 98, 99, 102, 107, 107, 109, 110, 111, 114,
        114, 115, 116, 116, 118, 118, 120, 120, 121, 123, 123, 126, 98, 95, 94,
        92, 90, 90, 91, 91, 91, 91, 91, 97, 98, 102, 107, 107, 111, 111, 114,
        117, 117, 119, 120, 121, 123, 123, 124, 125, 126, 127, 127, 130, 98, 95,
        94, 92, 90, 90, 91, 91, 91, 91, 91, 97, 98, 102, 107, 107, 111, 111,
        114, 117, 117, 119, 120, 121, 123, 123, 124, 125, 126, 127, 127, 130,
        101, 97, 96, 95, 92, 92, 93, 93, 92, 92, 92, 98, 100, 103, 108, 108,
        112, 113, 115, 119, 119, 122, 122, 124, 126, 126, 129, 129, 131, 133,
        133, 135, 101, 98, 97, 95, 93, 93, 93, 93, 93, 92, 92, 98, 100, 103,
        108, 108, 112, 113, 116, 120, 120, 122, 123, 125, 127, 127, 130, 130,
        132, 134, 134, 137, 103, 99, 98, 96, 94, 94, 94, 94, 94, 93, 93, 99,
        100, 103, 108, 108, 113, 114, 116, 121, 121, 124, 125, 127, 130, 130,
        132, 133, 134, 137, 137, 140, 104, 101, 100, 98, 95, 95, 95, 95, 94, 94,
        94, 99, 101, 104, 109, 109, 114, 115, 118, 123, 123, 126, 127, 130, 133,
        133, 136, 137, 139, 141, 141, 144, 104, 101, 100, 98, 95, 95, 95, 95,
        94, 94, 94, 99, 101, 104, 109, 109, 114, 115, 118, 123, 123, 126, 127,
        130, 133, 133, 136, 137, 139, 141, 141, 144, 108, 104, 103, 101, 99, 99,
        98, 98, 97, 96, 96, 102, 103, 106, 111, 111, 116, 117, 120, 124, 124,
        129, 130, 132, 136, 136, 140, 141, 143, 146, 146, 149, 109, 105, 104,
        102, 99, 99, 99, 99, 98, 97, 97, 102, 104, 107, 111, 111, 116, 117, 120,
        125, 125, 129, 130, 133, 137, 137, 141, 142, 144, 147, 147, 150, 111,
        107, 106, 104, 101, 101, 100, 100, 99, 98, 98, 103, 105, 108, 112, 112,
        117, 118, 121, 126, 126, 131, 132, 134, 139, 139, 143, 144, 146, 149,
        149, 153, 114, 110, 109, 107, 104, 104, 103, 102, 101, 100, 100, 105,
        107, 109, 113, 113, 118, 120, 123, 127, 127, 133, 134, 137, 141, 141,
        146, 147, 149, 153, 153, 157, 114, 110, 109, 107, 104, 104, 103, 102,
        101, 100, 100, 105, 107, 109, 113, 113, 118, 120, 123, 127, 127, 133,
        134, 137, 141, 141, 146, 147, 149, 153, 153, 157, 119, 114, 113, 111,
        108, 108, 106, 106, 105, 103, 103, 108, 110, 112, 116, 116, 121, 123,
        126, 130, 130, 135, 137, 140, 144, 144, 149, 150, 153, 157, 157, 161,
        /* Size 4x8 */
        63, 76, 95, 104, 64, 80, 91, 99, 79, 94, 91, 96, 89, 95, 102, 106, 93,
        94, 112, 117, 94, 93, 119, 129, 97, 95, 122, 137, 106, 101, 128, 146,
        /* Size 8x4 */
        63, 64, 79, 89, 93, 94, 97, 106, 76, 80, 94, 95, 94, 93, 95, 101, 95,
        91, 91, 102, 112, 119, 122, 128, 104, 99, 96, 106, 117, 129, 137, 146,
        /* Size 8x16 */
        64, 62, 74, 91, 97, 98, 105, 115, 63, 63, 77, 90, 94, 94, 100, 109, 61,
        64, 80, 89, 91, 91, 96, 104, 66, 70, 85, 92, 93, 91, 95, 103, 70, 75,
        89, 93, 93, 91, 95, 102, 74, 80, 94, 95, 94, 91, 94, 100, 84, 86, 95,
        99, 100, 99, 101, 107, 98, 93, 96, 104, 106, 107, 109, 114, 97, 92, 94,
        103, 108, 111, 114, 119, 97, 91, 93, 103, 109, 114, 118, 123, 98, 91,
        92, 103, 111, 117, 123, 128, 101, 93, 93, 105, 112, 119, 128, 134, 104,
        96, 95, 106, 114, 122, 133, 142, 108, 99, 97, 108, 116, 124, 136, 146,
        111, 102, 99, 109, 117, 126, 139, 149, 114, 104, 101, 111, 119, 128,
        141, 153,
        /* Size 16x8 */
        64, 63, 61, 66, 70, 74, 84, 98, 97, 97, 98, 101, 104, 108, 111, 114, 62,
        63, 64, 70, 75, 80, 86, 93, 92, 91, 91, 93, 96, 99, 102, 104, 74, 77,
        80, 85, 89, 94, 95, 96, 94, 93, 92, 93, 95, 97, 99, 101, 91, 90, 89, 92,
        93, 95, 99, 104, 103, 103, 103, 105, 106, 108, 109, 111, 97, 94, 91, 93,
        93, 94, 100, 106, 108, 109, 111, 112, 114, 116, 117, 119, 98, 94, 91,
        91, 91, 91, 99, 107, 111, 114, 117, 119, 122, 124, 126, 128, 105, 100,
        96, 95, 95, 94, 101, 109, 114, 118, 123, 128, 133, 136, 139, 141, 115,
        109, 104, 103, 102, 100, 107, 114, 119, 123, 128, 134, 142, 146, 149,
        153,
        /* Size 16x32 */
        64, 62, 62, 66, 74, 74, 91, 96, 97, 98, 98, 103, 105, 109, 115, 115, 63,
        63, 63, 68, 76, 76, 90, 95, 95, 95, 95, 100, 101, 104, 110, 110, 63, 63,
        63, 68, 77, 77, 90, 94, 94, 94, 94, 99, 100, 103, 109, 109, 62, 63, 64,
        69, 78, 78, 90, 93, 93, 93, 93, 97, 98, 102, 107, 107, 61, 64, 64, 70,
        80, 80, 89, 92, 91, 91, 91, 95, 96, 99, 104, 104, 61, 64, 64, 70, 80,
        80, 89, 92, 91, 91, 91, 95, 96, 99, 104, 104, 66, 69, 70, 75, 85, 85,
        92, 94, 93, 91, 91, 95, 95, 98, 103, 103, 67, 71, 72, 77, 86, 86, 92,
        94, 93, 92, 92, 95, 95, 98, 103, 103, 70, 74, 75, 80, 89, 89, 93, 95,
        93, 91, 91, 94, 95, 97, 102, 102, 74, 79, 80, 86, 94, 94, 95, 95, 94,
        91, 91, 93, 94, 96, 100, 100, 74, 79, 80, 86, 94, 94, 95, 95, 94, 91,
        91, 93, 94, 96, 100, 100, 82, 84, 85, 88, 95, 95, 98, 99, 98, 97, 97,
        99, 100, 102, 105, 105, 84, 85, 86, 89, 95, 95, 99, 100, 100, 99, 99,
        101, 101, 103, 107, 107, 89, 89, 89, 91, 95, 95, 101, 102, 102, 102,
        102, 104, 104, 106, 109, 109, 98, 94, 93, 94, 96, 96, 104, 106, 106,
        107, 107, 109, 109, 111, 114, 114, 98, 94, 93, 94, 96, 96, 104, 106,
        106, 107, 107, 109, 109, 111, 114, 114, 97, 93, 92, 93, 94, 94, 103,
        106, 108, 111, 111, 113, 114, 116, 119, 119, 97, 93, 92, 93, 94, 94,
        103, 106, 108, 112, 112, 114, 115, 117, 120, 120, 97, 92, 91, 92, 93,
        93, 103, 106, 109, 114, 114, 117, 118, 120, 123, 123, 98, 92, 91, 91,
        92, 92, 103, 107, 111, 117, 117, 122, 123, 125, 128, 128, 98, 92, 91,
        91, 92, 92, 103, 107, 111, 117, 117, 122, 123, 125, 128, 128, 100, 94,
        93, 93, 93, 93, 104, 108, 112, 119, 119, 125, 127, 129, 133, 133, 101,
        95, 93, 93, 93, 93, 105, 108, 112, 119, 119, 126, 128, 130, 134, 134,
        102, 96, 94, 94, 94, 94, 105, 108, 113, 121, 121, 128, 130, 133, 137,
        137, 104, 97, 96, 95, 95, 95, 106, 109, 114, 122, 122, 131, 133, 137,
        142, 142, 104, 97, 96, 95, 95, 95, 106, 109, 114, 122, 122, 131, 133,
        137, 142, 142, 108, 101, 99, 98, 97, 97, 108, 111, 116, 124, 124, 134,
        136, 140, 146, 146, 109, 102, 100, 99, 98, 98, 108, 111, 116, 125, 125,
        134, 137, 141, 147, 147, 111, 103, 102, 101, 99, 99, 109, 112, 117, 126,
        126, 136, 139, 143, 149, 149, 114, 106, 104, 103, 101, 101, 111, 113,
        119, 128, 128, 138, 141, 146, 153, 153, 114, 106, 104, 103, 101, 101,
        111, 113, 119, 128, 128, 138, 141, 146, 153, 153, 118, 110, 108, 107,
        104, 104, 114, 116, 122, 130, 130, 141, 144, 149, 157, 157,
        /* Size 32x16 */
        64, 63, 63, 62, 61, 61, 66, 67, 70, 74, 74, 82, 84, 89, 98, 98, 97, 97,
        97, 98, 98, 100, 101, 102, 104, 104, 108, 109, 111, 114, 114, 118, 62,
        63, 63, 63, 64, 64, 69, 71, 74, 79, 79, 84, 85, 89, 94, 94, 93, 93, 92,
        92, 92, 94, 95, 96, 97, 97, 101, 102, 103, 106, 106, 110, 62, 63, 63,
        64, 64, 64, 70, 72, 75, 80, 80, 85, 86, 89, 93, 93, 92, 92, 91, 91, 91,
        93, 93, 94, 96, 96, 99, 100, 102, 104, 104, 108, 66, 68, 68, 69, 70, 70,
        75, 77, 80, 86, 86, 88, 89, 91, 94, 94, 93, 93, 92, 91, 91, 93, 93, 94,
        95, 95, 98, 99, 101, 103, 103, 107, 74, 76, 77, 78, 80, 80, 85, 86, 89,
        94, 94, 95, 95, 95, 96, 96, 94, 94, 93, 92, 92, 93, 93, 94, 95, 95, 97,
        98, 99, 101, 101, 104, 74, 76, 77, 78, 80, 80, 85, 86, 89, 94, 94, 95,
        95, 95, 96, 96, 94, 94, 93, 92, 92, 93, 93, 94, 95, 95, 97, 98, 99, 101,
        101, 104, 91, 90, 90, 90, 89, 89, 92, 92, 93, 95, 95, 98, 99, 101, 104,
        104, 103, 103, 103, 103, 103, 104, 105, 105, 106, 106, 108, 108, 109,
        111, 111, 114, 96, 95, 94, 93, 92, 92, 94, 94, 95, 95, 95, 99, 100, 102,
        106, 106, 106, 106, 106, 107, 107, 108, 108, 108, 109, 109, 111, 111,
        112, 113, 113, 116, 97, 95, 94, 93, 91, 91, 93, 93, 93, 94, 94, 98, 100,
        102, 106, 106, 108, 108, 109, 111, 111, 112, 112, 113, 114, 114, 116,
        116, 117, 119, 119, 122, 98, 95, 94, 93, 91, 91, 91, 92, 91, 91, 91, 97,
        99, 102, 107, 107, 111, 112, 114, 117, 117, 119, 119, 121, 122, 122,
        124, 125, 126, 128, 128, 130, 98, 95, 94, 93, 91, 91, 91, 92, 91, 91,
        91, 97, 99, 102, 107, 107, 111, 112, 114, 117, 117, 119, 119, 121, 122,
        122, 124, 125, 126, 128, 128, 130, 103, 100, 99, 97, 95, 95, 95, 95, 94,
        93, 93, 99, 101, 104, 109, 109, 113, 114, 117, 122, 122, 125, 126, 128,
        131, 131, 134, 134, 136, 138, 138, 141, 105, 101, 100, 98, 96, 96, 95,
        95, 95, 94, 94, 100, 101, 104, 109, 109, 114, 115, 118, 123, 123, 127,
        128, 130, 133, 133, 136, 137, 139, 141, 141, 144, 109, 104, 103, 102,
        99, 99, 98, 98, 97, 96, 96, 102, 103, 106, 111, 111, 116, 117, 120, 125,
        125, 129, 130, 133, 137, 137, 140, 141, 143, 146, 146, 149, 115, 110,
        109, 107, 104, 104, 103, 103, 102, 100, 100, 105, 107, 109, 114, 114,
        119, 120, 123, 128, 128, 133, 134, 137, 142, 142, 146, 147, 149, 153,
        153, 157, 115, 110, 109, 107, 104, 104, 103, 103, 102, 100, 100, 105,
        107, 109, 114, 114, 119, 120, 123, 128, 128, 133, 134, 137, 142, 142,
        146, 147, 149, 153, 153, 157,
        /* Size 4x16 */
        62, 74, 98, 109, 63, 77, 94, 103, 64, 80, 91, 99, 69, 85, 91, 98, 74,
        89, 91, 97, 79, 94, 91, 96, 85, 95, 99, 103, 94, 96, 107, 111, 93, 94,
        111, 116, 92, 93, 114, 120, 92, 92, 117, 125, 95, 93, 119, 130, 97, 95,
        122, 137, 101, 97, 124, 140, 103, 99, 126, 143, 106, 101, 128, 146,
        /* Size 16x4 */
        62, 63, 64, 69, 74, 79, 85, 94, 93, 92, 92, 95, 97, 101, 103, 106, 74,
        77, 80, 85, 89, 94, 95, 96, 94, 93, 92, 93, 95, 97, 99, 101, 98, 94, 91,
        91, 91, 91, 99, 107, 111, 114, 117, 119, 122, 124, 126, 128, 109, 103,
        99, 98, 97, 96, 103, 111, 116, 120, 125, 130, 137, 140, 143, 146,
        /* Size 8x32 */
        64, 62, 74, 91, 97, 98, 105, 115, 63, 63, 76, 90, 95, 95, 101, 110, 63,
        63, 77, 90, 94, 94, 100, 109, 62, 64, 78, 90, 93, 93, 98, 107, 61, 64,
        80, 89, 91, 91, 96, 104, 61, 64, 80, 89, 91, 91, 96, 104, 66, 70, 85,
        92, 93, 91, 95, 103, 67, 72, 86, 92, 93, 92, 95, 103, 70, 75, 89, 93,
        93, 91, 95, 102, 74, 80, 94, 95, 94, 91, 94, 100, 74, 80, 94, 95, 94,
        91, 94, 100, 82, 85, 95, 98, 98, 97, 100, 105, 84, 86, 95, 99, 100, 99,
        101, 107, 89, 89, 95, 101, 102, 102, 104, 109, 98, 93, 96, 104, 106,
        107, 109, 114, 98, 93, 96, 104, 106, 107, 109, 114, 97, 92, 94, 103,
        108, 111, 114, 119, 97, 92, 94, 103, 108, 112, 115, 120, 97, 91, 93,
        103, 109, 114, 118, 123, 98, 91, 92, 103, 111, 117, 123, 128, 98, 91,
        92, 103, 111, 117, 123, 128, 100, 93, 93, 104, 112, 119, 127, 133, 101,
        93, 93, 105, 112, 119, 128, 134, 102, 94, 94, 105, 113, 121, 130, 137,
        104, 96, 95, 106, 114, 122, 133, 142, 104, 96, 95, 106, 114, 122, 133,
        142, 108, 99, 97, 108, 116, 124, 136, 146, 109, 100, 98, 108, 116, 125,
        137, 147, 111, 102, 99, 109, 117, 126, 139, 149, 114, 104, 101, 111,
        119, 128, 141, 153, 114, 104, 101, 111, 119, 128, 141, 153, 118, 108,
        104, 114, 122, 130, 144, 157,
        /* Size 32x8 */
        64, 63, 63, 62, 61, 61, 66, 67, 70, 74, 74, 82, 84, 89, 98, 98, 97, 97,
        97, 98, 98, 100, 101, 102, 104, 104, 108, 109, 111, 114, 114, 118, 62,
        63, 63, 64, 64, 64, 70, 72, 75, 80, 80, 85, 86, 89, 93, 93, 92, 92, 91,
        91, 91, 93, 93, 94, 96, 96, 99, 100, 102, 104, 104, 108, 74, 76, 77, 78,
        80, 80, 85, 86, 89, 94, 94, 95, 95, 95, 96, 96, 94, 94, 93, 92, 92, 93,
        93, 94, 95, 95, 97, 98, 99, 101, 101, 104, 91, 90, 90, 90, 89, 89, 92,
        92, 93, 95, 95, 98, 99, 101, 104, 104, 103, 103, 103, 103, 103, 104,
        105, 105, 106, 106, 108, 108, 109, 111, 111, 114, 97, 95, 94, 93, 91,
        91, 93, 93, 93, 94, 94, 98, 100, 102, 106, 106, 108, 108, 109, 111, 111,
        112, 112, 113, 114, 114, 116, 116, 117, 119, 119, 122, 98, 95, 94, 93,
        91, 91, 91, 92, 91, 91, 91, 97, 99, 102, 107, 107, 111, 112, 114, 117,
        117, 119, 119, 121, 122, 122, 124, 125, 126, 128, 128, 130, 105, 101,
        100, 98, 96, 96, 95, 95, 95, 94, 94, 100, 101, 104, 109, 109, 114, 115,
        118, 123, 123, 127, 128, 130, 133, 133, 136, 137, 139, 141, 141, 144,
        115, 110, 109, 107, 104, 104, 103, 103, 102, 100, 100, 105, 107, 109,
        114, 114, 119, 120, 123, 128, 128, 133, 134, 137, 142, 142, 146, 147,
        149, 153, 153, 157 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        66, 85, 85, 128,
#endif
        /* Size 4x4 */
        64, 65, 77, 102, 65, 71, 81, 98, 77, 81, 108, 129, 102, 98, 129, 162,
        /* Size 8x8 */
        63, 64, 64, 68, 71, 83, 94, 107, 64, 65, 65, 67, 69, 80, 89, 101, 64,
        65, 68, 71, 74, 83, 91, 102, 68, 67, 71, 79, 85, 94, 102, 111, 71, 69,
        74, 85, 97, 106, 114, 122, 83, 80, 83, 94, 106, 120, 130, 141, 94, 89,
        91, 102, 114, 130, 142, 155, 107, 101, 102, 111, 122, 141, 155, 170,
        /* Size 16x16 */
        64, 63, 63, 63, 63, 64, 65, 68, 72, 76, 79, 88, 95, 99, 108, 118, 63,
        64, 64, 64, 64, 65, 66, 68, 70, 74, 77, 85, 91, 95, 103, 113, 63, 64,
        64, 65, 65, 65, 66, 67, 69, 73, 75, 82, 88, 92, 100, 109, 63, 64, 65,
        65, 65, 66, 67, 68, 70, 73, 76, 82, 88, 91, 99, 108, 63, 64, 65, 65, 67,
        68, 69, 71, 73, 76, 78, 84, 90, 93, 100, 108, 64, 65, 65, 66, 68, 71,
        72, 74, 77, 79, 81, 85, 90, 93, 99, 107, 65, 66, 66, 67, 69, 72, 73, 76,
        80, 82, 84, 89, 94, 97, 103, 111, 68, 68, 67, 68, 71, 74, 76, 79, 85,
        88, 90, 95, 100, 103, 109, 117, 72, 70, 69, 70, 73, 77, 80, 85, 97, 100,
        101, 108, 112, 115, 120, 128, 76, 74, 73, 73, 76, 79, 82, 88, 100, 103,
        105, 112, 117, 120, 126, 134, 79, 77, 75, 76, 78, 81, 84, 90, 101, 105,
        108, 116, 121, 124, 130, 138, 88, 85, 82, 82, 84, 85, 89, 95, 108, 112,
        116, 127, 133, 136, 143, 151, 95, 91, 88, 88, 90, 90, 94, 100, 112, 117,
        121, 133, 139, 142, 151, 159, 99, 95, 92, 91, 93, 93, 97, 103, 115, 120,
        124, 136, 142, 146, 155, 163, 108, 103, 100, 99, 100, 99, 103, 109, 120,
        126, 130, 143, 151, 155, 165, 174, 118, 113, 109, 108, 108, 107, 111,
        117, 128, 134, 138, 151, 159, 163, 174, 184,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 65, 68, 68, 70, 72, 72,
        76, 79, 79, 85, 88, 89, 95, 97, 99, 107, 108, 111, 118, 118, 63, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 66, 68, 68, 69, 71, 71, 75, 78, 78,
        83, 86, 87, 92, 94, 96, 103, 104, 107, 114, 114, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 66, 68, 68, 69, 70, 70, 74, 77, 77, 82, 85, 86,
        91, 93, 95, 102, 103, 106, 113, 113, 63, 63, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 66, 68, 68, 69, 70, 70, 74, 77, 77, 82, 84, 85, 91, 93, 94,
        102, 103, 105, 112, 112, 63, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65,
        66, 67, 67, 68, 69, 69, 73, 75, 75, 80, 82, 83, 88, 90, 92, 99, 100,
        102, 109, 109, 63, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 66, 67,
        67, 68, 69, 69, 72, 75, 75, 80, 82, 82, 88, 89, 91, 98, 99, 101, 108,
        108, 63, 64, 64, 64, 65, 65, 65, 65, 65, 66, 66, 66, 67, 68, 68, 69, 70,
        70, 73, 76, 76, 80, 82, 83, 88, 90, 91, 98, 99, 101, 108, 108, 63, 64,
        64, 64, 65, 65, 65, 66, 66, 67, 68, 68, 69, 70, 70, 71, 73, 73, 76, 78,
        78, 82, 84, 85, 89, 91, 93, 99, 100, 102, 108, 108, 63, 64, 64, 64, 65,
        65, 65, 66, 67, 67, 68, 68, 69, 71, 71, 72, 73, 73, 76, 78, 78, 82, 84,
        85, 90, 91, 93, 99, 100, 102, 108, 108, 64, 64, 64, 65, 65, 65, 66, 67,
        67, 68, 69, 69, 70, 71, 71, 73, 74, 74, 77, 79, 79, 83, 84, 85, 90, 91,
        93, 99, 99, 102, 108, 108, 64, 64, 65, 65, 65, 65, 66, 68, 68, 69, 71,
        71, 72, 74, 74, 75, 77, 77, 79, 81, 81, 84, 85, 86, 90, 92, 93, 98, 99,
        101, 107, 107, 64, 64, 65, 65, 65, 65, 66, 68, 68, 69, 71, 71, 72, 74,
        74, 75, 77, 77, 79, 81, 81, 84, 85, 86, 90, 92, 93, 98, 99, 101, 107,
        107, 65, 66, 66, 66, 66, 66, 67, 69, 69, 70, 72, 72, 73, 76, 76, 78, 80,
        80, 82, 84, 84, 88, 89, 90, 94, 95, 97, 102, 103, 105, 111, 111, 68, 68,
        68, 68, 67, 67, 68, 70, 71, 71, 74, 74, 76, 79, 79, 82, 85, 85, 88, 90,
        90, 94, 95, 96, 100, 102, 103, 108, 109, 111, 117, 117, 68, 68, 68, 68,
        67, 67, 68, 70, 71, 71, 74, 74, 76, 79, 79, 82, 85, 85, 88, 90, 90, 94,
        95, 96, 100, 102, 103, 108, 109, 111, 117, 117, 70, 69, 69, 69, 68, 68,
        69, 71, 72, 73, 75, 75, 78, 82, 82, 86, 91, 91, 94, 95, 95, 99, 101,
        102, 106, 107, 109, 114, 114, 116, 122, 122, 72, 71, 70, 70, 69, 69, 70,
        73, 73, 74, 77, 77, 80, 85, 85, 91, 97, 97, 100, 101, 101, 106, 108,
        108, 112, 114, 115, 119, 120, 122, 128, 128, 72, 71, 70, 70, 69, 69, 70,
        73, 73, 74, 77, 77, 80, 85, 85, 91, 97, 97, 100, 101, 101, 106, 108,
        108, 112, 114, 115, 119, 120, 122, 128, 128, 76, 75, 74, 74, 73, 72, 73,
        76, 76, 77, 79, 79, 82, 88, 88, 94, 100, 100, 103, 105, 105, 110, 112,
        113, 117, 119, 120, 125, 126, 128, 134, 134, 79, 78, 77, 77, 75, 75, 76,
        78, 78, 79, 81, 81, 84, 90, 90, 95, 101, 101, 105, 108, 108, 113, 116,
        117, 121, 122, 124, 129, 130, 132, 138, 138, 79, 78, 77, 77, 75, 75, 76,
        78, 78, 79, 81, 81, 84, 90, 90, 95, 101, 101, 105, 108, 108, 113, 116,
        117, 121, 122, 124, 129, 130, 132, 138, 138, 85, 83, 82, 82, 80, 80, 80,
        82, 82, 83, 84, 84, 88, 94, 94, 99, 106, 106, 110, 113, 113, 120, 123,
        124, 129, 130, 132, 138, 139, 141, 147, 147, 88, 86, 85, 84, 82, 82, 82,
        84, 84, 84, 85, 85, 89, 95, 95, 101, 108, 108, 112, 116, 116, 123, 127,
        128, 133, 134, 136, 142, 143, 145, 151, 151, 89, 87, 86, 85, 83, 82, 83,
        85, 85, 85, 86, 86, 90, 96, 96, 102, 108, 108, 113, 117, 117, 124, 128,
        128, 133, 135, 137, 143, 144, 146, 152, 152, 95, 92, 91, 91, 88, 88, 88,
        89, 90, 90, 90, 90, 94, 100, 100, 106, 112, 112, 117, 121, 121, 129,
        133, 133, 139, 141, 142, 149, 151, 153, 159, 159, 97, 94, 93, 93, 90,
        89, 90, 91, 91, 91, 92, 92, 95, 102, 102, 107, 114, 114, 119, 122, 122,
        130, 134, 135, 141, 142, 144, 151, 153, 155, 161, 161, 99, 96, 95, 94,
        92, 91, 91, 93, 93, 93, 93, 93, 97, 103, 103, 109, 115, 115, 120, 124,
        124, 132, 136, 137, 142, 144, 146, 154, 155, 157, 163, 163, 107, 103,
        102, 102, 99, 98, 98, 99, 99, 99, 98, 98, 102, 108, 108, 114, 119, 119,
        125, 129, 129, 138, 142, 143, 149, 151, 154, 162, 163, 166, 173, 173,
        108, 104, 103, 103, 100, 99, 99, 100, 100, 99, 99, 99, 103, 109, 109,
        114, 120, 120, 126, 130, 130, 139, 143, 144, 151, 153, 155, 163, 165,
        167, 174, 174, 111, 107, 106, 105, 102, 101, 101, 102, 102, 102, 101,
        101, 105, 111, 111, 116, 122, 122, 128, 132, 132, 141, 145, 146, 153,
        155, 157, 166, 167, 170, 177, 177, 118, 114, 113, 112, 109, 108, 108,
        108, 108, 108, 107, 107, 111, 117, 117, 122, 128, 128, 134, 138, 138,
        147, 151, 152, 159, 161, 163, 173, 174, 177, 184, 184, 118, 114, 113,
        112, 109, 108, 108, 108, 108, 108, 107, 107, 111, 117, 117, 122, 128,
        128, 134, 138, 138, 147, 151, 152, 159, 161, 163, 173, 174, 177, 184,
        184,
        /* Size 4x8 */
        64, 65, 75, 104, 64, 66, 73, 99, 65, 69, 77, 99, 68, 74, 88, 109, 70,
        77, 99, 120, 81, 85, 110, 139, 92, 93, 119, 152, 104, 102, 128, 167,
        /* Size 8x4 */
        64, 64, 65, 68, 70, 81, 92, 104, 65, 66, 69, 74, 77, 85, 93, 102, 75,
        73, 77, 88, 99, 110, 119, 128, 104, 99, 99, 109, 120, 139, 152, 167,
        /* Size 8x16 */
        64, 63, 64, 65, 72, 88, 95, 107, 63, 64, 65, 66, 71, 85, 91, 103, 63,
        65, 65, 66, 70, 82, 88, 99, 63, 65, 66, 67, 70, 82, 88, 99, 64, 65, 68,
        69, 73, 84, 90, 100, 64, 66, 70, 72, 77, 85, 90, 99, 65, 66, 71, 73, 80,
        89, 94, 103, 68, 68, 72, 76, 85, 96, 101, 109, 72, 69, 75, 80, 96, 108,
        112, 120, 76, 73, 78, 83, 99, 112, 117, 126, 79, 75, 80, 85, 101, 116,
        121, 130, 88, 82, 85, 90, 107, 126, 132, 143, 94, 88, 91, 95, 112, 132,
        139, 150, 98, 92, 94, 97, 114, 135, 142, 155, 107, 99, 100, 103, 120,
        142, 150, 164, 117, 108, 108, 111, 127, 150, 159, 174,
        /* Size 16x8 */
        64, 63, 63, 63, 64, 64, 65, 68, 72, 76, 79, 88, 94, 98, 107, 117, 63,
        64, 65, 65, 65, 66, 66, 68, 69, 73, 75, 82, 88, 92, 99, 108, 64, 65, 65,
        66, 68, 70, 71, 72, 75, 78, 80, 85, 91, 94, 100, 108, 65, 66, 66, 67,
        69, 72, 73, 76, 80, 83, 85, 90, 95, 97, 103, 111, 72, 71, 70, 70, 73,
        77, 80, 85, 96, 99, 101, 107, 112, 114, 120, 127, 88, 85, 82, 82, 84,
        85, 89, 96, 108, 112, 116, 126, 132, 135, 142, 150, 95, 91, 88, 88, 90,
        90, 94, 101, 112, 117, 121, 132, 139, 142, 150, 159, 107, 103, 99, 99,
        100, 99, 103, 109, 120, 126, 130, 143, 150, 155, 164, 174,
        /* Size 16x32 */
        64, 63, 63, 63, 64, 64, 65, 71, 72, 76, 88, 88, 95, 107, 107, 118, 63,
        64, 64, 64, 64, 65, 66, 70, 71, 75, 86, 86, 92, 104, 104, 114, 63, 64,
        64, 64, 65, 65, 66, 70, 71, 74, 85, 85, 91, 103, 103, 113, 63, 64, 64,
        64, 65, 65, 66, 70, 71, 74, 84, 84, 91, 102, 102, 112, 63, 64, 65, 65,
        65, 65, 66, 69, 70, 73, 82, 82, 88, 99, 99, 109, 63, 64, 65, 65, 65, 66,
        66, 69, 69, 73, 82, 82, 88, 99, 99, 108, 63, 64, 65, 65, 66, 66, 67, 70,
        70, 73, 82, 82, 88, 99, 99, 108, 64, 65, 65, 65, 67, 68, 69, 72, 73, 76,
        84, 84, 90, 99, 99, 108, 64, 65, 65, 66, 68, 68, 69, 73, 73, 76, 84, 84,
        90, 100, 100, 108, 64, 65, 65, 66, 68, 69, 70, 74, 74, 77, 85, 85, 90,
        99, 99, 108, 64, 65, 66, 66, 70, 71, 72, 76, 77, 79, 85, 85, 90, 99, 99,
        107, 64, 65, 66, 66, 70, 71, 72, 76, 77, 79, 85, 85, 90, 99, 99, 107,
        65, 66, 66, 67, 71, 72, 73, 79, 80, 83, 89, 89, 94, 103, 103, 111, 68,
        68, 68, 68, 72, 74, 76, 84, 85, 88, 96, 96, 101, 109, 109, 117, 68, 68,
        68, 68, 72, 74, 76, 84, 85, 88, 96, 96, 101, 109, 109, 117, 70, 69, 69,
        69, 74, 75, 78, 89, 90, 93, 101, 101, 106, 114, 114, 122, 72, 70, 69,
        70, 75, 77, 80, 94, 96, 99, 108, 108, 112, 120, 120, 128, 72, 70, 69,
        70, 75, 77, 80, 94, 96, 99, 108, 108, 112, 120, 120, 128, 76, 74, 73,
        74, 78, 80, 83, 97, 99, 103, 112, 112, 117, 126, 126, 134, 79, 76, 75,
        76, 80, 81, 85, 98, 101, 105, 116, 116, 121, 130, 130, 138, 79, 76, 75,
        76, 80, 81, 85, 98, 101, 105, 116, 116, 121, 130, 130, 138, 85, 81, 80,
        81, 84, 85, 88, 103, 105, 110, 123, 123, 129, 139, 139, 147, 88, 84, 82,
        83, 85, 86, 90, 105, 107, 112, 126, 126, 132, 143, 143, 151, 88, 85, 83,
        83, 86, 87, 90, 105, 108, 113, 127, 127, 133, 144, 144, 152, 94, 90, 88,
        89, 91, 91, 95, 109, 112, 117, 132, 132, 139, 150, 150, 159, 96, 92, 90,
        90, 92, 93, 96, 111, 113, 119, 134, 134, 140, 152, 152, 161, 98, 94, 92,
        92, 94, 94, 97, 112, 114, 120, 135, 135, 142, 155, 155, 163, 106, 100,
        98, 98, 99, 99, 103, 117, 119, 125, 142, 142, 149, 163, 163, 173, 107,
        102, 99, 99, 100, 100, 103, 118, 120, 126, 142, 142, 150, 164, 164, 174,
        110, 104, 102, 102, 102, 102, 106, 120, 122, 128, 145, 145, 153, 167,
        167, 177, 117, 111, 108, 108, 108, 108, 111, 125, 127, 134, 150, 150,
        159, 174, 174, 184, 117, 111, 108, 108, 108, 108, 111, 125, 127, 134,
        150, 150, 159, 174, 174, 184,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 65, 68, 68, 70, 72, 72,
        76, 79, 79, 85, 88, 88, 94, 96, 98, 106, 107, 110, 117, 117, 63, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 65, 66, 68, 68, 69, 70, 70, 74, 76, 76,
        81, 84, 85, 90, 92, 94, 100, 102, 104, 111, 111, 63, 64, 64, 64, 65, 65,
        65, 65, 65, 65, 66, 66, 66, 68, 68, 69, 69, 69, 73, 75, 75, 80, 82, 83,
        88, 90, 92, 98, 99, 102, 108, 108, 63, 64, 64, 64, 65, 65, 65, 65, 66,
        66, 66, 66, 67, 68, 68, 69, 70, 70, 74, 76, 76, 81, 83, 83, 89, 90, 92,
        98, 99, 102, 108, 108, 64, 64, 65, 65, 65, 65, 66, 67, 68, 68, 70, 70,
        71, 72, 72, 74, 75, 75, 78, 80, 80, 84, 85, 86, 91, 92, 94, 99, 100,
        102, 108, 108, 64, 65, 65, 65, 65, 66, 66, 68, 68, 69, 71, 71, 72, 74,
        74, 75, 77, 77, 80, 81, 81, 85, 86, 87, 91, 93, 94, 99, 100, 102, 108,
        108, 65, 66, 66, 66, 66, 66, 67, 69, 69, 70, 72, 72, 73, 76, 76, 78, 80,
        80, 83, 85, 85, 88, 90, 90, 95, 96, 97, 103, 103, 106, 111, 111, 71, 70,
        70, 70, 69, 69, 70, 72, 73, 74, 76, 76, 79, 84, 84, 89, 94, 94, 97, 98,
        98, 103, 105, 105, 109, 111, 112, 117, 118, 120, 125, 125, 72, 71, 71,
        71, 70, 69, 70, 73, 73, 74, 77, 77, 80, 85, 85, 90, 96, 96, 99, 101,
        101, 105, 107, 108, 112, 113, 114, 119, 120, 122, 127, 127, 76, 75, 74,
        74, 73, 73, 73, 76, 76, 77, 79, 79, 83, 88, 88, 93, 99, 99, 103, 105,
        105, 110, 112, 113, 117, 119, 120, 125, 126, 128, 134, 134, 88, 86, 85,
        84, 82, 82, 82, 84, 84, 85, 85, 85, 89, 96, 96, 101, 108, 108, 112, 116,
        116, 123, 126, 127, 132, 134, 135, 142, 142, 145, 150, 150, 88, 86, 85,
        84, 82, 82, 82, 84, 84, 85, 85, 85, 89, 96, 96, 101, 108, 108, 112, 116,
        116, 123, 126, 127, 132, 134, 135, 142, 142, 145, 150, 150, 95, 92, 91,
        91, 88, 88, 88, 90, 90, 90, 90, 90, 94, 101, 101, 106, 112, 112, 117,
        121, 121, 129, 132, 133, 139, 140, 142, 149, 150, 153, 159, 159, 107,
        104, 103, 102, 99, 99, 99, 99, 100, 99, 99, 99, 103, 109, 109, 114, 120,
        120, 126, 130, 130, 139, 143, 144, 150, 152, 155, 163, 164, 167, 174,
        174, 107, 104, 103, 102, 99, 99, 99, 99, 100, 99, 99, 99, 103, 109, 109,
        114, 120, 120, 126, 130, 130, 139, 143, 144, 150, 152, 155, 163, 164,
        167, 174, 174, 118, 114, 113, 112, 109, 108, 108, 108, 108, 108, 107,
        107, 111, 117, 117, 122, 128, 128, 134, 138, 138, 147, 151, 152, 159,
        161, 163, 173, 174, 177, 184, 184,
        /* Size 4x16 */
        63, 64, 76, 107, 64, 65, 74, 103, 64, 65, 73, 99, 64, 66, 73, 99, 65,
        68, 76, 100, 65, 71, 79, 99, 66, 72, 83, 103, 68, 74, 88, 109, 70, 77,
        99, 120, 74, 80, 103, 126, 76, 81, 105, 130, 84, 86, 112, 143, 90, 91,
        117, 150, 94, 94, 120, 155, 102, 100, 126, 164, 111, 108, 134, 174,
        /* Size 16x4 */
        63, 64, 64, 64, 65, 65, 66, 68, 70, 74, 76, 84, 90, 94, 102, 111, 64,
        65, 65, 66, 68, 71, 72, 74, 77, 80, 81, 86, 91, 94, 100, 108, 76, 74,
        73, 73, 76, 79, 83, 88, 99, 103, 105, 112, 117, 120, 126, 134, 107, 103,
        99, 99, 100, 99, 103, 109, 120, 126, 130, 143, 150, 155, 164, 174,
        /* Size 8x32 */
        64, 63, 64, 65, 72, 88, 95, 107, 63, 64, 64, 66, 71, 86, 92, 104, 63,
        64, 65, 66, 71, 85, 91, 103, 63, 64, 65, 66, 71, 84, 91, 102, 63, 65,
        65, 66, 70, 82, 88, 99, 63, 65, 65, 66, 69, 82, 88, 99, 63, 65, 66, 67,
        70, 82, 88, 99, 64, 65, 67, 69, 73, 84, 90, 99, 64, 65, 68, 69, 73, 84,
        90, 100, 64, 65, 68, 70, 74, 85, 90, 99, 64, 66, 70, 72, 77, 85, 90, 99,
        64, 66, 70, 72, 77, 85, 90, 99, 65, 66, 71, 73, 80, 89, 94, 103, 68, 68,
        72, 76, 85, 96, 101, 109, 68, 68, 72, 76, 85, 96, 101, 109, 70, 69, 74,
        78, 90, 101, 106, 114, 72, 69, 75, 80, 96, 108, 112, 120, 72, 69, 75,
        80, 96, 108, 112, 120, 76, 73, 78, 83, 99, 112, 117, 126, 79, 75, 80,
        85, 101, 116, 121, 130, 79, 75, 80, 85, 101, 116, 121, 130, 85, 80, 84,
        88, 105, 123, 129, 139, 88, 82, 85, 90, 107, 126, 132, 143, 88, 83, 86,
        90, 108, 127, 133, 144, 94, 88, 91, 95, 112, 132, 139, 150, 96, 90, 92,
        96, 113, 134, 140, 152, 98, 92, 94, 97, 114, 135, 142, 155, 106, 98, 99,
        103, 119, 142, 149, 163, 107, 99, 100, 103, 120, 142, 150, 164, 110,
        102, 102, 106, 122, 145, 153, 167, 117, 108, 108, 111, 127, 150, 159,
        174, 117, 108, 108, 111, 127, 150, 159, 174,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 65, 68, 68, 70, 72, 72,
        76, 79, 79, 85, 88, 88, 94, 96, 98, 106, 107, 110, 117, 117, 63, 64, 64,
        64, 65, 65, 65, 65, 65, 65, 66, 66, 66, 68, 68, 69, 69, 69, 73, 75, 75,
        80, 82, 83, 88, 90, 92, 98, 99, 102, 108, 108, 64, 64, 65, 65, 65, 65,
        66, 67, 68, 68, 70, 70, 71, 72, 72, 74, 75, 75, 78, 80, 80, 84, 85, 86,
        91, 92, 94, 99, 100, 102, 108, 108, 65, 66, 66, 66, 66, 66, 67, 69, 69,
        70, 72, 72, 73, 76, 76, 78, 80, 80, 83, 85, 85, 88, 90, 90, 95, 96, 97,
        103, 103, 106, 111, 111, 72, 71, 71, 71, 70, 69, 70, 73, 73, 74, 77, 77,
        80, 85, 85, 90, 96, 96, 99, 101, 101, 105, 107, 108, 112, 113, 114, 119,
        120, 122, 127, 127, 88, 86, 85, 84, 82, 82, 82, 84, 84, 85, 85, 85, 89,
        96, 96, 101, 108, 108, 112, 116, 116, 123, 126, 127, 132, 134, 135, 142,
        142, 145, 150, 150, 95, 92, 91, 91, 88, 88, 88, 90, 90, 90, 90, 90, 94,
        101, 101, 106, 112, 112, 117, 121, 121, 129, 132, 133, 139, 140, 142,
        149, 150, 153, 159, 159, 107, 104, 103, 102, 99, 99, 99, 99, 100, 99,
        99, 99, 103, 109, 109, 114, 120, 120, 126, 130, 130, 139, 143, 144, 150,
        152, 155, 163, 164, 167, 174, 174 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        77, 91, 91, 118,
#endif
        /* Size 4x4 */
        63, 77, 94, 99, 77, 95, 93, 93, 94, 93, 109, 115, 99, 93, 115, 132,
        /* Size 8x8 */
        63, 62, 71, 84, 96, 95, 98, 102, 62, 64, 73, 85, 93, 90, 93, 97, 71, 73,
        83, 91, 95, 91, 93, 96, 84, 85, 91, 97, 101, 99, 100, 102, 96, 93, 95,
        101, 106, 107, 108, 109, 95, 90, 91, 99, 107, 114, 118, 121, 98, 93, 93,
        100, 108, 118, 123, 128, 102, 97, 96, 102, 109, 121, 128, 136,
        /* Size 16x16 */
        64, 62, 61, 62, 67, 73, 77, 83, 98, 98, 97, 98, 101, 102, 104, 109, 62,
        63, 63, 64, 69, 77, 80, 85, 95, 95, 94, 94, 96, 97, 100, 104, 61, 63,
        63, 65, 71, 79, 82, 85, 93, 92, 92, 91, 93, 94, 96, 100, 62, 64, 65, 66,
        72, 81, 83, 86, 93, 92, 91, 90, 92, 93, 95, 99, 67, 69, 71, 72, 78, 87,
        88, 90, 95, 93, 93, 91, 93, 94, 95, 99, 73, 77, 79, 81, 87, 95, 95, 95,
        96, 94, 93, 91, 92, 93, 94, 97, 77, 80, 82, 83, 88, 95, 95, 96, 98, 96,
        96, 94, 95, 95, 96, 99, 83, 85, 85, 86, 90, 95, 96, 97, 101, 100, 99,
        98, 100, 100, 101, 104, 98, 95, 93, 93, 95, 96, 98, 101, 106, 106, 106,
        107, 108, 108, 109, 111, 98, 95, 92, 92, 93, 94, 96, 100, 106, 107, 108,
        110, 111, 111, 112, 115, 97, 94, 92, 91, 93, 93, 96, 99, 106, 108, 109,
        111, 113, 113, 115, 117, 98, 94, 91, 90, 91, 91, 94, 98, 107, 110, 111,
        117, 119, 120, 123, 125, 101, 96, 93, 92, 93, 92, 95, 100, 108, 111,
        113, 119, 122, 123, 126, 129, 102, 97, 94, 93, 94, 93, 95, 100, 108,
        111, 113, 120, 123, 125, 129, 132, 104, 100, 96, 95, 95, 94, 96, 101,
        109, 112, 115, 123, 126, 129, 133, 137, 109, 104, 100, 99, 99, 97, 99,
        104, 111, 115, 117, 125, 129, 132, 137, 142,
        /* Size 32x32 */
        64, 63, 62, 62, 61, 61, 62, 66, 67, 69, 73, 73, 77, 83, 83, 90, 98, 98,
        98, 97, 97, 98, 98, 98, 101, 101, 102, 104, 104, 106, 109, 109, 63, 63,
        63, 63, 62, 62, 63, 68, 69, 71, 76, 76, 79, 84, 84, 90, 96, 96, 95, 95,
        95, 95, 95, 95, 98, 98, 99, 101, 101, 102, 106, 106, 62, 63, 63, 63, 63,
        62, 64, 69, 69, 71, 77, 77, 80, 85, 85, 90, 95, 95, 95, 94, 94, 94, 94,
        94, 96, 97, 97, 99, 100, 101, 104, 104, 62, 63, 63, 63, 63, 63, 64, 69,
        69, 72, 77, 77, 80, 85, 85, 90, 95, 95, 94, 94, 94, 94, 93, 94, 96, 96,
        97, 99, 99, 101, 104, 104, 61, 62, 63, 63, 63, 63, 65, 70, 71, 73, 79,
        79, 82, 85, 85, 89, 93, 93, 92, 92, 92, 91, 91, 91, 93, 94, 94, 96, 96,
        97, 100, 100, 61, 62, 62, 63, 63, 64, 65, 70, 71, 73, 80, 80, 82, 85,
        85, 89, 93, 93, 92, 91, 91, 90, 90, 90, 92, 93, 93, 95, 95, 97, 99, 99,
        62, 63, 64, 64, 65, 65, 66, 71, 72, 75, 81, 81, 83, 86, 86, 89, 93, 93,
        92, 91, 91, 91, 90, 90, 92, 93, 93, 95, 95, 96, 99, 99, 66, 68, 69, 69,
        70, 70, 71, 77, 77, 80, 86, 86, 87, 89, 89, 92, 94, 94, 93, 92, 92, 91,
        91, 91, 93, 93, 94, 95, 95, 96, 99, 99, 67, 69, 69, 69, 71, 71, 72, 77,
        78, 80, 87, 87, 88, 90, 90, 92, 95, 95, 93, 93, 93, 91, 91, 91, 93, 93,
        94, 95, 95, 96, 99, 99, 69, 71, 71, 72, 73, 73, 75, 80, 80, 83, 89, 89,
        90, 91, 91, 93, 95, 95, 94, 93, 93, 91, 91, 91, 93, 93, 93, 94, 95, 96,
        98, 98, 73, 76, 77, 77, 79, 80, 81, 86, 87, 89, 95, 95, 95, 95, 95, 95,
        96, 96, 94, 93, 93, 91, 91, 91, 92, 92, 93, 93, 94, 94, 97, 97, 73, 76,
        77, 77, 79, 80, 81, 86, 87, 89, 95, 95, 95, 95, 95, 95, 96, 96, 94, 93,
        93, 91, 91, 91, 92, 92, 93, 93, 94, 94, 97, 97, 77, 79, 80, 80, 82, 82,
        83, 87, 88, 90, 95, 95, 95, 96, 96, 97, 98, 98, 96, 96, 96, 94, 94, 94,
        95, 95, 95, 96, 96, 97, 99, 99, 83, 84, 85, 85, 85, 85, 86, 89, 90, 91,
        95, 95, 96, 97, 97, 99, 101, 101, 100, 99, 99, 99, 98, 99, 100, 100,
        100, 101, 101, 102, 104, 104, 83, 84, 85, 85, 85, 85, 86, 89, 90, 91,
        95, 95, 96, 97, 97, 99, 101, 101, 100, 99, 99, 99, 98, 99, 100, 100,
        100, 101, 101, 102, 104, 104, 90, 90, 90, 90, 89, 89, 89, 92, 92, 93,
        95, 95, 97, 99, 99, 101, 103, 103, 103, 103, 103, 103, 103, 103, 104,
        104, 104, 105, 105, 105, 107, 107, 98, 96, 95, 95, 93, 93, 93, 94, 95,
        95, 96, 96, 98, 101, 101, 103, 106, 106, 106, 106, 106, 107, 107, 107,
        108, 108, 108, 109, 109, 109, 111, 111, 98, 96, 95, 95, 93, 93, 93, 94,
        95, 95, 96, 96, 98, 101, 101, 103, 106, 106, 106, 106, 106, 107, 107,
        107, 108, 108, 108, 109, 109, 109, 111, 111, 98, 95, 95, 94, 92, 92, 92,
        93, 93, 94, 94, 94, 96, 100, 100, 103, 106, 106, 107, 108, 108, 109,
        110, 110, 111, 111, 111, 112, 112, 113, 115, 115, 97, 95, 94, 94, 92,
        91, 91, 92, 93, 93, 93, 93, 96, 99, 99, 103, 106, 106, 108, 109, 109,
        111, 111, 112, 113, 113, 113, 115, 115, 116, 117, 117, 97, 95, 94, 94,
        92, 91, 91, 92, 93, 93, 93, 93, 96, 99, 99, 103, 106, 106, 108, 109,
        109, 111, 111, 112, 113, 113, 113, 115, 115, 116, 117, 117, 98, 95, 94,
        94, 91, 90, 91, 91, 91, 91, 91, 91, 94, 99, 99, 103, 107, 107, 109, 111,
        111, 114, 115, 116, 117, 118, 118, 120, 120, 121, 123, 123, 98, 95, 94,
        93, 91, 90, 90, 91, 91, 91, 91, 91, 94, 98, 98, 103, 107, 107, 110, 111,
        111, 115, 117, 117, 119, 120, 120, 122, 123, 123, 125, 125, 98, 95, 94,
        94, 91, 90, 90, 91, 91, 91, 91, 91, 94, 99, 99, 103, 107, 107, 110, 112,
        112, 116, 117, 118, 119, 120, 121, 123, 123, 124, 126, 126, 101, 98, 96,
        96, 93, 92, 92, 93, 93, 93, 92, 92, 95, 100, 100, 104, 108, 108, 111,
        113, 113, 117, 119, 119, 122, 122, 123, 126, 126, 127, 129, 129, 101,
        98, 97, 96, 94, 93, 93, 93, 93, 93, 92, 92, 95, 100, 100, 104, 108, 108,
        111, 113, 113, 118, 120, 120, 122, 123, 124, 127, 127, 128, 130, 130,
        102, 99, 97, 97, 94, 93, 93, 94, 94, 93, 93, 93, 95, 100, 100, 104, 108,
        108, 111, 113, 113, 118, 120, 121, 123, 124, 125, 128, 129, 129, 132,
        132, 104, 101, 99, 99, 96, 95, 95, 95, 95, 94, 93, 93, 96, 101, 101,
        105, 109, 109, 112, 115, 115, 120, 122, 123, 126, 127, 128, 132, 133,
        134, 136, 136, 104, 101, 100, 99, 96, 95, 95, 95, 95, 95, 94, 94, 96,
        101, 101, 105, 109, 109, 112, 115, 115, 120, 123, 123, 126, 127, 129,
        133, 133, 134, 137, 137, 106, 102, 101, 101, 97, 97, 96, 96, 96, 96, 94,
        94, 97, 102, 102, 105, 109, 109, 113, 116, 116, 121, 123, 124, 127, 128,
        129, 134, 134, 136, 138, 138, 109, 106, 104, 104, 100, 99, 99, 99, 99,
        98, 97, 97, 99, 104, 104, 107, 111, 111, 115, 117, 117, 123, 125, 126,
        129, 130, 132, 136, 137, 138, 142, 142, 109, 106, 104, 104, 100, 99, 99,
        99, 99, 98, 97, 97, 99, 104, 104, 107, 111, 111, 115, 117, 117, 123,
        125, 126, 129, 130, 132, 136, 137, 138, 142, 142,
        /* Size 4x8 */
        63, 76, 95, 101, 63, 80, 92, 96, 73, 89, 94, 95, 85, 95, 100, 101, 95,
        96, 106, 109, 93, 92, 109, 120, 96, 93, 111, 128, 100, 96, 113, 134,
        /* Size 8x4 */
        63, 63, 73, 85, 95, 93, 96, 100, 76, 80, 89, 95, 96, 92, 93, 96, 95, 92,
        94, 100, 106, 109, 111, 113, 101, 96, 95, 101, 109, 120, 128, 134,
        /* Size 8x16 */
        64, 62, 71, 77, 96, 98, 101, 105, 63, 63, 74, 80, 94, 94, 97, 100, 61,
        64, 76, 81, 92, 91, 93, 97, 62, 66, 77, 83, 92, 91, 93, 96, 67, 72, 83,
        88, 94, 92, 93, 95, 74, 80, 91, 95, 95, 91, 92, 94, 78, 82, 92, 95, 97,
        94, 95, 97, 84, 86, 93, 96, 100, 99, 100, 101, 98, 93, 96, 98, 106, 107,
        108, 109, 97, 92, 94, 97, 106, 110, 111, 113, 97, 92, 93, 96, 106, 112,
        113, 115, 98, 91, 91, 94, 107, 117, 119, 123, 100, 93, 93, 96, 108, 119,
        122, 127, 102, 94, 94, 96, 108, 120, 123, 129, 104, 96, 95, 97, 109,
        122, 127, 133, 109, 100, 98, 100, 111, 125, 130, 137,
        /* Size 16x8 */
        64, 63, 61, 62, 67, 74, 78, 84, 98, 97, 97, 98, 100, 102, 104, 109, 62,
        63, 64, 66, 72, 80, 82, 86, 93, 92, 92, 91, 93, 94, 96, 100, 71, 74, 76,
        77, 83, 91, 92, 93, 96, 94, 93, 91, 93, 94, 95, 98, 77, 80, 81, 83, 88,
        95, 95, 96, 98, 97, 96, 94, 96, 96, 97, 100, 96, 94, 92, 92, 94, 95, 97,
        100, 106, 106, 106, 107, 108, 108, 109, 111, 98, 94, 91, 91, 92, 91, 94,
        99, 107, 110, 112, 117, 119, 120, 122, 125, 101, 97, 93, 93, 93, 92, 95,
        100, 108, 111, 113, 119, 122, 123, 127, 130, 105, 100, 97, 96, 95, 94,
        97, 101, 109, 113, 115, 123, 127, 129, 133, 137,
        /* Size 16x32 */
        64, 62, 62, 63, 71, 74, 77, 94, 96, 97, 98, 98, 101, 105, 105, 109, 63,
        63, 63, 64, 73, 76, 79, 93, 95, 95, 96, 96, 98, 101, 101, 106, 63, 63,
        63, 64, 74, 77, 80, 92, 94, 94, 94, 94, 97, 100, 100, 104, 62, 63, 63,
        64, 74, 77, 80, 92, 94, 94, 94, 94, 96, 100, 100, 104, 61, 63, 64, 65,
        76, 79, 81, 91, 92, 92, 91, 91, 93, 97, 97, 101, 61, 63, 64, 66, 76, 80,
        82, 91, 92, 92, 91, 91, 93, 96, 96, 100, 62, 65, 66, 67, 77, 81, 83, 91,
        92, 92, 91, 91, 93, 96, 96, 100, 67, 70, 71, 72, 82, 86, 87, 93, 94, 93,
        91, 91, 93, 95, 95, 99, 67, 70, 72, 73, 83, 86, 88, 93, 94, 93, 92, 92,
        93, 95, 95, 99, 69, 73, 74, 75, 85, 89, 90, 94, 94, 94, 91, 91, 93, 95,
        95, 98, 74, 78, 80, 82, 91, 94, 95, 95, 95, 94, 91, 91, 92, 94, 94, 97,
        74, 78, 80, 82, 91, 94, 95, 95, 95, 94, 91, 91, 92, 94, 94, 97, 78, 81,
        82, 84, 92, 95, 95, 97, 97, 96, 94, 94, 95, 97, 97, 100, 84, 85, 86, 87,
        93, 95, 96, 100, 100, 100, 99, 99, 100, 101, 101, 104, 84, 85, 86, 87,
        93, 95, 96, 100, 100, 100, 99, 99, 100, 101, 101, 104, 90, 90, 89, 90,
        94, 95, 97, 102, 103, 103, 103, 103, 104, 105, 105, 108, 98, 95, 93, 94,
        96, 96, 98, 105, 106, 106, 107, 107, 108, 109, 109, 111, 98, 95, 93, 94,
        96, 96, 98, 105, 106, 106, 107, 107, 108, 109, 109, 111, 97, 94, 92, 93,
        94, 95, 97, 105, 106, 107, 110, 110, 111, 113, 113, 115, 97, 93, 92, 92,
        93, 94, 96, 105, 106, 108, 112, 112, 113, 115, 115, 118, 97, 93, 92, 92,
        93, 94, 96, 105, 106, 108, 112, 112, 113, 115, 115, 118, 98, 93, 91, 91,
        92, 92, 95, 105, 107, 109, 115, 115, 117, 120, 120, 123, 98, 93, 91, 91,
        91, 92, 94, 105, 107, 110, 117, 117, 119, 123, 123, 125, 98, 93, 91, 91,
        92, 92, 94, 105, 107, 110, 117, 117, 120, 123, 123, 126, 100, 95, 93,
        93, 93, 93, 96, 106, 108, 111, 119, 119, 122, 127, 127, 130, 101, 96,
        93, 93, 93, 93, 96, 106, 108, 111, 119, 119, 123, 128, 128, 131, 102,
        96, 94, 94, 94, 94, 96, 106, 108, 111, 120, 120, 123, 129, 129, 132,
        104, 98, 96, 96, 95, 94, 97, 107, 109, 112, 122, 122, 126, 133, 133,
        137, 104, 98, 96, 96, 95, 95, 97, 107, 109, 112, 122, 122, 127, 133,
        133, 137, 106, 100, 97, 97, 96, 96, 98, 108, 109, 113, 123, 123, 127,
        134, 134, 139, 109, 103, 100, 100, 98, 98, 100, 110, 111, 115, 125, 125,
        130, 137, 137, 142, 109, 103, 100, 100, 98, 98, 100, 110, 111, 115, 125,
        125, 130, 137, 137, 142,
        /* Size 32x16 */
        64, 63, 63, 62, 61, 61, 62, 67, 67, 69, 74, 74, 78, 84, 84, 90, 98, 98,
        97, 97, 97, 98, 98, 98, 100, 101, 102, 104, 104, 106, 109, 109, 62, 63,
        63, 63, 63, 63, 65, 70, 70, 73, 78, 78, 81, 85, 85, 90, 95, 95, 94, 93,
        93, 93, 93, 93, 95, 96, 96, 98, 98, 100, 103, 103, 62, 63, 63, 63, 64,
        64, 66, 71, 72, 74, 80, 80, 82, 86, 86, 89, 93, 93, 92, 92, 92, 91, 91,
        91, 93, 93, 94, 96, 96, 97, 100, 100, 63, 64, 64, 64, 65, 66, 67, 72,
        73, 75, 82, 82, 84, 87, 87, 90, 94, 94, 93, 92, 92, 91, 91, 91, 93, 93,
        94, 96, 96, 97, 100, 100, 71, 73, 74, 74, 76, 76, 77, 82, 83, 85, 91,
        91, 92, 93, 93, 94, 96, 96, 94, 93, 93, 92, 91, 92, 93, 93, 94, 95, 95,
        96, 98, 98, 74, 76, 77, 77, 79, 80, 81, 86, 86, 89, 94, 94, 95, 95, 95,
        95, 96, 96, 95, 94, 94, 92, 92, 92, 93, 93, 94, 94, 95, 96, 98, 98, 77,
        79, 80, 80, 81, 82, 83, 87, 88, 90, 95, 95, 95, 96, 96, 97, 98, 98, 97,
        96, 96, 95, 94, 94, 96, 96, 96, 97, 97, 98, 100, 100, 94, 93, 92, 92,
        91, 91, 91, 93, 93, 94, 95, 95, 97, 100, 100, 102, 105, 105, 105, 105,
        105, 105, 105, 105, 106, 106, 106, 107, 107, 108, 110, 110, 96, 95, 94,
        94, 92, 92, 92, 94, 94, 94, 95, 95, 97, 100, 100, 103, 106, 106, 106,
        106, 106, 107, 107, 107, 108, 108, 108, 109, 109, 109, 111, 111, 97, 95,
        94, 94, 92, 92, 92, 93, 93, 94, 94, 94, 96, 100, 100, 103, 106, 106,
        107, 108, 108, 109, 110, 110, 111, 111, 111, 112, 112, 113, 115, 115,
        98, 96, 94, 94, 91, 91, 91, 91, 92, 91, 91, 91, 94, 99, 99, 103, 107,
        107, 110, 112, 112, 115, 117, 117, 119, 119, 120, 122, 122, 123, 125,
        125, 98, 96, 94, 94, 91, 91, 91, 91, 92, 91, 91, 91, 94, 99, 99, 103,
        107, 107, 110, 112, 112, 115, 117, 117, 119, 119, 120, 122, 122, 123,
        125, 125, 101, 98, 97, 96, 93, 93, 93, 93, 93, 93, 92, 92, 95, 100, 100,
        104, 108, 108, 111, 113, 113, 117, 119, 120, 122, 123, 123, 126, 127,
        127, 130, 130, 105, 101, 100, 100, 97, 96, 96, 95, 95, 95, 94, 94, 97,
        101, 101, 105, 109, 109, 113, 115, 115, 120, 123, 123, 127, 128, 129,
        133, 133, 134, 137, 137, 105, 101, 100, 100, 97, 96, 96, 95, 95, 95, 94,
        94, 97, 101, 101, 105, 109, 109, 113, 115, 115, 120, 123, 123, 127, 128,
        129, 133, 133, 134, 137, 137, 109, 106, 104, 104, 101, 100, 100, 99, 99,
        98, 97, 97, 100, 104, 104, 108, 111, 111, 115, 118, 118, 123, 125, 126,
        130, 131, 132, 137, 137, 139, 142, 142,
        /* Size 4x16 */
        62, 74, 97, 105, 63, 77, 94, 100, 63, 79, 92, 97, 65, 81, 92, 96, 70,
        86, 93, 95, 78, 94, 94, 94, 81, 95, 96, 97, 85, 95, 100, 101, 95, 96,
        106, 109, 94, 95, 107, 113, 93, 94, 108, 115, 93, 92, 110, 123, 95, 93,
        111, 127, 96, 94, 111, 129, 98, 95, 112, 133, 103, 98, 115, 137,
        /* Size 16x4 */
        62, 63, 63, 65, 70, 78, 81, 85, 95, 94, 93, 93, 95, 96, 98, 103, 74, 77,
        79, 81, 86, 94, 95, 95, 96, 95, 94, 92, 93, 94, 95, 98, 97, 94, 92, 92,
        93, 94, 96, 100, 106, 107, 108, 110, 111, 111, 112, 115, 105, 100, 97,
        96, 95, 94, 97, 101, 109, 113, 115, 123, 127, 129, 133, 137,
        /* Size 8x32 */
        64, 62, 71, 77, 96, 98, 101, 105, 63, 63, 73, 79, 95, 96, 98, 101, 63,
        63, 74, 80, 94, 94, 97, 100, 62, 63, 74, 80, 94, 94, 96, 100, 61, 64,
        76, 81, 92, 91, 93, 97, 61, 64, 76, 82, 92, 91, 93, 96, 62, 66, 77, 83,
        92, 91, 93, 96, 67, 71, 82, 87, 94, 91, 93, 95, 67, 72, 83, 88, 94, 92,
        93, 95, 69, 74, 85, 90, 94, 91, 93, 95, 74, 80, 91, 95, 95, 91, 92, 94,
        74, 80, 91, 95, 95, 91, 92, 94, 78, 82, 92, 95, 97, 94, 95, 97, 84, 86,
        93, 96, 100, 99, 100, 101, 84, 86, 93, 96, 100, 99, 100, 101, 90, 89,
        94, 97, 103, 103, 104, 105, 98, 93, 96, 98, 106, 107, 108, 109, 98, 93,
        96, 98, 106, 107, 108, 109, 97, 92, 94, 97, 106, 110, 111, 113, 97, 92,
        93, 96, 106, 112, 113, 115, 97, 92, 93, 96, 106, 112, 113, 115, 98, 91,
        92, 95, 107, 115, 117, 120, 98, 91, 91, 94, 107, 117, 119, 123, 98, 91,
        92, 94, 107, 117, 120, 123, 100, 93, 93, 96, 108, 119, 122, 127, 101,
        93, 93, 96, 108, 119, 123, 128, 102, 94, 94, 96, 108, 120, 123, 129,
        104, 96, 95, 97, 109, 122, 126, 133, 104, 96, 95, 97, 109, 122, 127,
        133, 106, 97, 96, 98, 109, 123, 127, 134, 109, 100, 98, 100, 111, 125,
        130, 137, 109, 100, 98, 100, 111, 125, 130, 137,
        /* Size 32x8 */
        64, 63, 63, 62, 61, 61, 62, 67, 67, 69, 74, 74, 78, 84, 84, 90, 98, 98,
        97, 97, 97, 98, 98, 98, 100, 101, 102, 104, 104, 106, 109, 109, 62, 63,
        63, 63, 64, 64, 66, 71, 72, 74, 80, 80, 82, 86, 86, 89, 93, 93, 92, 92,
        92, 91, 91, 91, 93, 93, 94, 96, 96, 97, 100, 100, 71, 73, 74, 74, 76,
        76, 77, 82, 83, 85, 91, 91, 92, 93, 93, 94, 96, 96, 94, 93, 93, 92, 91,
        92, 93, 93, 94, 95, 95, 96, 98, 98, 77, 79, 80, 80, 81, 82, 83, 87, 88,
        90, 95, 95, 95, 96, 96, 97, 98, 98, 97, 96, 96, 95, 94, 94, 96, 96, 96,
        97, 97, 98, 100, 100, 96, 95, 94, 94, 92, 92, 92, 94, 94, 94, 95, 95,
        97, 100, 100, 103, 106, 106, 106, 106, 106, 107, 107, 107, 108, 108,
        108, 109, 109, 109, 111, 111, 98, 96, 94, 94, 91, 91, 91, 91, 92, 91,
        91, 91, 94, 99, 99, 103, 107, 107, 110, 112, 112, 115, 117, 117, 119,
        119, 120, 122, 122, 123, 125, 125, 101, 98, 97, 96, 93, 93, 93, 93, 93,
        93, 92, 92, 95, 100, 100, 104, 108, 108, 111, 113, 113, 117, 119, 120,
        122, 123, 123, 126, 127, 127, 130, 130, 105, 101, 100, 100, 97, 96, 96,
        95, 95, 95, 94, 94, 97, 101, 101, 105, 109, 109, 113, 115, 115, 120,
        123, 123, 127, 128, 129, 133, 133, 134, 137, 137 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        65, 76, 76, 108,
#endif
        /* Size 4x4 */
        64, 65, 70, 86, 65, 69, 75, 86, 70, 75, 97, 109, 86, 86, 109, 130,
        /* Size 8x8 */
        63, 63, 64, 64, 68, 75, 86, 94, 63, 65, 65, 65, 68, 72, 82, 89, 64, 65,
        67, 68, 71, 76, 84, 91, 64, 65, 68, 71, 74, 79, 85, 92, 68, 68, 71, 74,
        82, 90, 98, 104, 75, 72, 76, 79, 90, 103, 112, 119, 86, 82, 84, 85, 98,
        112, 127, 134, 94, 89, 91, 92, 104, 119, 134, 142,
        /* Size 16x16 */
        64, 63, 63, 63, 63, 63, 64, 65, 68, 71, 72, 79, 82, 88, 95, 97, 63, 64,
        64, 64, 64, 64, 65, 66, 68, 70, 70, 77, 80, 85, 91, 93, 63, 64, 64, 64,
        65, 65, 65, 66, 68, 69, 70, 76, 79, 84, 90, 91, 63, 64, 64, 65, 65, 65,
        65, 66, 67, 68, 69, 75, 77, 82, 88, 89, 63, 64, 65, 65, 66, 66, 67, 68,
        70, 72, 72, 78, 80, 84, 89, 91, 63, 64, 65, 65, 66, 67, 68, 69, 71, 72,
        73, 78, 81, 84, 90, 91, 64, 65, 65, 65, 67, 68, 71, 72, 74, 76, 77, 81,
        83, 85, 90, 92, 65, 66, 66, 66, 68, 69, 72, 73, 76, 79, 80, 84, 86, 89,
        94, 95, 68, 68, 68, 67, 70, 71, 74, 76, 79, 84, 85, 90, 92, 95, 100,
        102, 71, 70, 69, 68, 72, 72, 76, 79, 84, 92, 94, 99, 101, 105, 110, 111,
        72, 70, 70, 69, 72, 73, 77, 80, 85, 94, 97, 101, 104, 108, 112, 114, 79,
        77, 76, 75, 78, 78, 81, 84, 90, 99, 101, 108, 111, 116, 121, 122, 82,
        80, 79, 77, 80, 81, 83, 86, 92, 101, 104, 111, 114, 120, 125, 127, 88,
        85, 84, 82, 84, 84, 85, 89, 95, 105, 108, 116, 120, 127, 133, 134, 95,
        91, 90, 88, 89, 90, 90, 94, 100, 110, 112, 121, 125, 133, 139, 141, 97,
        93, 91, 89, 91, 91, 92, 95, 102, 111, 114, 122, 127, 134, 141, 142,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 65, 68, 68, 68,
        71, 72, 72, 76, 79, 79, 82, 88, 88, 90, 95, 97, 97, 103, 63, 63, 63, 63,
        63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 66, 68, 68, 68, 70, 71, 71, 75,
        78, 78, 81, 86, 86, 88, 93, 94, 94, 100, 63, 63, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 65, 65, 65, 66, 68, 68, 68, 70, 70, 70, 74, 77, 77, 80, 85,
        85, 86, 91, 93, 93, 99, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65,
        65, 65, 66, 68, 68, 68, 70, 70, 70, 74, 77, 77, 80, 85, 85, 86, 91, 93,
        93, 99, 63, 63, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 66, 68,
        68, 68, 69, 70, 70, 73, 76, 76, 79, 84, 84, 85, 90, 91, 91, 97, 63, 63,
        64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 67, 67, 68, 68, 69,
        69, 72, 75, 75, 77, 82, 82, 83, 88, 89, 89, 95, 63, 63, 64, 64, 64, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 66, 67, 67, 68, 68, 69, 69, 72, 75, 75,
        77, 82, 82, 83, 88, 89, 89, 95, 63, 64, 64, 64, 64, 65, 65, 65, 65, 65,
        65, 66, 66, 66, 67, 68, 68, 68, 69, 70, 70, 73, 76, 76, 78, 82, 82, 84,
        88, 90, 90, 95, 63, 64, 64, 64, 65, 65, 65, 65, 66, 66, 66, 67, 67, 67,
        68, 70, 70, 70, 72, 72, 72, 75, 78, 78, 80, 84, 84, 85, 89, 91, 91, 96,
        63, 64, 64, 64, 65, 65, 65, 65, 66, 67, 67, 67, 68, 68, 69, 71, 71, 71,
        72, 73, 73, 76, 78, 78, 81, 84, 84, 85, 90, 91, 91, 96, 63, 64, 64, 64,
        65, 65, 65, 65, 66, 67, 67, 67, 68, 68, 69, 71, 71, 71, 72, 73, 73, 76,
        78, 78, 81, 84, 84, 85, 90, 91, 91, 96, 64, 64, 65, 65, 65, 65, 65, 66,
        67, 67, 67, 69, 70, 70, 71, 72, 72, 73, 75, 75, 75, 78, 80, 80, 82, 85,
        85, 86, 90, 91, 91, 96, 64, 64, 65, 65, 65, 65, 65, 66, 67, 68, 68, 70,
        71, 71, 72, 74, 74, 74, 76, 77, 77, 79, 81, 81, 83, 85, 85, 87, 90, 92,
        92, 96, 64, 64, 65, 65, 65, 65, 65, 66, 67, 68, 68, 70, 71, 71, 72, 74,
        74, 74, 76, 77, 77, 79, 81, 81, 83, 85, 85, 87, 90, 92, 92, 96, 65, 66,
        66, 66, 66, 66, 66, 67, 68, 69, 69, 71, 72, 72, 73, 76, 76, 76, 79, 80,
        80, 82, 84, 84, 86, 89, 89, 90, 94, 95, 95, 100, 68, 68, 68, 68, 68, 67,
        67, 68, 70, 71, 71, 72, 74, 74, 76, 79, 79, 80, 84, 85, 85, 88, 90, 90,
        92, 95, 95, 97, 100, 102, 102, 106, 68, 68, 68, 68, 68, 67, 67, 68, 70,
        71, 71, 72, 74, 74, 76, 79, 79, 80, 84, 85, 85, 88, 90, 90, 92, 95, 95,
        97, 100, 102, 102, 106, 68, 68, 68, 68, 68, 68, 68, 68, 70, 71, 71, 73,
        74, 74, 76, 80, 80, 82, 86, 88, 88, 90, 92, 92, 94, 98, 98, 99, 103,
        104, 104, 108, 71, 70, 70, 70, 69, 68, 68, 69, 72, 72, 72, 75, 76, 76,
        79, 84, 84, 86, 92, 94, 94, 97, 99, 99, 101, 105, 105, 106, 110, 111,
        111, 115, 72, 71, 70, 70, 70, 69, 69, 70, 72, 73, 73, 75, 77, 77, 80,
        85, 85, 88, 94, 97, 97, 100, 101, 101, 104, 108, 108, 109, 112, 114,
        114, 117, 72, 71, 70, 70, 70, 69, 69, 70, 72, 73, 73, 75, 77, 77, 80,
        85, 85, 88, 94, 97, 97, 100, 101, 101, 104, 108, 108, 109, 112, 114,
        114, 117, 76, 75, 74, 74, 73, 72, 72, 73, 75, 76, 76, 78, 79, 79, 82,
        88, 88, 90, 97, 100, 100, 103, 105, 105, 108, 112, 112, 114, 117, 119,
        119, 123, 79, 78, 77, 77, 76, 75, 75, 76, 78, 78, 78, 80, 81, 81, 84,
        90, 90, 92, 99, 101, 101, 105, 108, 108, 111, 116, 116, 117, 121, 122,
        122, 127, 79, 78, 77, 77, 76, 75, 75, 76, 78, 78, 78, 80, 81, 81, 84,
        90, 90, 92, 99, 101, 101, 105, 108, 108, 111, 116, 116, 117, 121, 122,
        122, 127, 82, 81, 80, 80, 79, 77, 77, 78, 80, 81, 81, 82, 83, 83, 86,
        92, 92, 94, 101, 104, 104, 108, 111, 111, 114, 120, 120, 121, 125, 127,
        127, 132, 88, 86, 85, 85, 84, 82, 82, 82, 84, 84, 84, 85, 85, 85, 89,
        95, 95, 98, 105, 108, 108, 112, 116, 116, 120, 127, 127, 128, 133, 134,
        134, 139, 88, 86, 85, 85, 84, 82, 82, 82, 84, 84, 84, 85, 85, 85, 89,
        95, 95, 98, 105, 108, 108, 112, 116, 116, 120, 127, 127, 128, 133, 134,
        134, 139, 90, 88, 86, 86, 85, 83, 83, 84, 85, 85, 85, 86, 87, 87, 90,
        97, 97, 99, 106, 109, 109, 114, 117, 117, 121, 128, 128, 130, 134, 136,
        136, 141, 95, 93, 91, 91, 90, 88, 88, 88, 89, 90, 90, 90, 90, 90, 94,
        100, 100, 103, 110, 112, 112, 117, 121, 121, 125, 133, 133, 134, 139,
        141, 141, 146, 97, 94, 93, 93, 91, 89, 89, 90, 91, 91, 91, 91, 92, 92,
        95, 102, 102, 104, 111, 114, 114, 119, 122, 122, 127, 134, 134, 136,
        141, 142, 142, 148, 97, 94, 93, 93, 91, 89, 89, 90, 91, 91, 91, 91, 92,
        92, 95, 102, 102, 104, 111, 114, 114, 119, 122, 122, 127, 134, 134, 136,
        141, 142, 142, 148, 103, 100, 99, 99, 97, 95, 95, 95, 96, 96, 96, 96,
        96, 96, 100, 106, 106, 108, 115, 117, 117, 123, 127, 127, 132, 139, 139,
        141, 146, 148, 148, 155,
        /* Size 4x8 */
        63, 65, 71, 86, 64, 66, 69, 82, 65, 68, 73, 84, 65, 71, 77, 85, 68, 74,
        87, 98, 74, 80, 99, 112, 84, 86, 107, 126, 92, 93, 113, 134,
        /* Size 8x4 */
        63, 64, 65, 65, 68, 74, 84, 92, 65, 66, 68, 71, 74, 80, 86, 93, 71, 69,
        73, 77, 87, 99, 107, 113, 86, 82, 84, 85, 98, 112, 126, 134,
        /* Size 8x16 */
        64, 63, 63, 64, 70, 72, 88, 95, 63, 64, 64, 65, 70, 71, 85, 91, 63, 64,
        65, 65, 69, 70, 83, 90, 63, 65, 65, 66, 68, 69, 82, 88, 63, 65, 66, 68,
        71, 72, 84, 89, 64, 65, 66, 68, 72, 73, 84, 90, 64, 66, 68, 71, 75, 77,
        85, 90, 65, 66, 69, 72, 78, 80, 89, 94, 68, 68, 70, 74, 83, 85, 96, 101,
        71, 69, 72, 76, 90, 94, 105, 110, 72, 69, 72, 77, 92, 96, 108, 112, 79,
        75, 78, 81, 96, 101, 116, 121, 82, 78, 80, 83, 99, 103, 120, 125, 88,
        82, 84, 86, 102, 107, 126, 132, 94, 88, 89, 91, 107, 112, 132, 139, 96,
        90, 91, 93, 108, 113, 134, 140,
        /* Size 16x8 */
        64, 63, 63, 63, 63, 64, 64, 65, 68, 71, 72, 79, 82, 88, 94, 96, 63, 64,
        64, 65, 65, 65, 66, 66, 68, 69, 69, 75, 78, 82, 88, 90, 63, 64, 65, 65,
        66, 66, 68, 69, 70, 72, 72, 78, 80, 84, 89, 91, 64, 65, 65, 66, 68, 68,
        71, 72, 74, 76, 77, 81, 83, 86, 91, 93, 70, 70, 69, 68, 71, 72, 75, 78,
        83, 90, 92, 96, 99, 102, 107, 108, 72, 71, 70, 69, 72, 73, 77, 80, 85,
        94, 96, 101, 103, 107, 112, 113, 88, 85, 83, 82, 84, 84, 85, 89, 96,
        105, 108, 116, 120, 126, 132, 134, 95, 91, 90, 88, 89, 90, 90, 94, 101,
        110, 112, 121, 125, 132, 139, 140,
        /* Size 16x32 */
        64, 63, 63, 63, 63, 64, 64, 65, 70, 72, 72, 81, 88, 88, 95, 107, 63, 63,
        64, 64, 64, 65, 65, 66, 70, 71, 71, 79, 86, 86, 93, 105, 63, 64, 64, 64,
        64, 65, 65, 66, 70, 71, 71, 79, 85, 85, 91, 103, 63, 64, 64, 64, 64, 65,
        65, 66, 70, 71, 71, 79, 85, 85, 91, 103, 63, 64, 64, 64, 65, 65, 65, 66,
        69, 70, 70, 78, 83, 83, 90, 101, 63, 64, 65, 65, 65, 66, 66, 66, 68, 69,
        69, 76, 82, 82, 88, 99, 63, 64, 65, 65, 65, 66, 66, 66, 68, 69, 69, 76,
        82, 82, 88, 99, 63, 64, 65, 65, 65, 66, 66, 67, 69, 70, 70, 77, 82, 82,
        88, 99, 63, 64, 65, 65, 66, 68, 68, 69, 71, 72, 72, 79, 84, 84, 89, 99,
        64, 65, 65, 65, 66, 68, 68, 69, 72, 73, 73, 79, 84, 84, 90, 100, 64, 65,
        65, 65, 66, 68, 68, 69, 72, 73, 73, 79, 84, 84, 90, 100, 64, 65, 65, 65,
        67, 70, 70, 71, 74, 75, 75, 81, 85, 85, 90, 99, 64, 65, 66, 66, 68, 71,
        71, 72, 75, 77, 77, 82, 85, 85, 90, 99, 64, 65, 66, 66, 68, 71, 71, 72,
        75, 77, 77, 82, 85, 85, 90, 99, 65, 66, 66, 66, 69, 72, 72, 73, 78, 80,
        80, 85, 89, 89, 94, 103, 68, 68, 68, 68, 70, 74, 74, 76, 83, 85, 85, 91,
        96, 96, 101, 109, 68, 68, 68, 68, 70, 74, 74, 76, 83, 85, 85, 91, 96,
        96, 101, 109, 68, 68, 68, 68, 70, 74, 74, 77, 84, 87, 87, 93, 98, 98,
        103, 111, 71, 70, 69, 69, 72, 76, 76, 79, 90, 94, 94, 100, 105, 105,
        110, 118, 72, 70, 69, 69, 72, 77, 77, 80, 92, 96, 96, 103, 108, 108,
        112, 120, 72, 70, 69, 69, 72, 77, 77, 80, 92, 96, 96, 103, 108, 108,
        112, 120, 76, 74, 73, 73, 75, 80, 80, 83, 94, 99, 99, 107, 112, 112,
        117, 126, 79, 77, 75, 75, 78, 81, 81, 85, 96, 101, 101, 109, 116, 116,
        121, 130, 79, 77, 75, 75, 78, 81, 81, 85, 96, 101, 101, 109, 116, 116,
        121, 130, 82, 80, 78, 78, 80, 83, 83, 87, 99, 103, 103, 113, 120, 120,
        125, 135, 88, 84, 82, 82, 84, 86, 86, 90, 102, 107, 107, 118, 126, 126,
        132, 143, 88, 84, 82, 82, 84, 86, 86, 90, 102, 107, 107, 118, 126, 126,
        132, 143, 89, 86, 84, 84, 85, 87, 87, 91, 103, 108, 108, 119, 128, 128,
        134, 145, 94, 91, 88, 88, 89, 91, 91, 95, 107, 112, 112, 123, 132, 132,
        139, 150, 96, 92, 90, 90, 91, 93, 93, 96, 108, 113, 113, 125, 134, 134,
        140, 152, 96, 92, 90, 90, 91, 93, 93, 96, 108, 113, 113, 125, 134, 134,
        140, 152, 103, 98, 95, 95, 96, 97, 97, 100, 112, 117, 117, 129, 139,
        139, 146, 159,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 65, 68, 68, 68,
        71, 72, 72, 76, 79, 79, 82, 88, 88, 89, 94, 96, 96, 103, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 66, 68, 68, 68, 70, 70, 70, 74,
        77, 77, 80, 84, 84, 86, 91, 92, 92, 98, 63, 64, 64, 64, 64, 65, 65, 65,
        65, 65, 65, 65, 66, 66, 66, 68, 68, 68, 69, 69, 69, 73, 75, 75, 78, 82,
        82, 84, 88, 90, 90, 95, 63, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65,
        66, 66, 66, 68, 68, 68, 69, 69, 69, 73, 75, 75, 78, 82, 82, 84, 88, 90,
        90, 95, 63, 64, 64, 64, 65, 65, 65, 65, 66, 66, 66, 67, 68, 68, 69, 70,
        70, 70, 72, 72, 72, 75, 78, 78, 80, 84, 84, 85, 89, 91, 91, 96, 64, 65,
        65, 65, 65, 66, 66, 66, 68, 68, 68, 70, 71, 71, 72, 74, 74, 74, 76, 77,
        77, 80, 81, 81, 83, 86, 86, 87, 91, 93, 93, 97, 64, 65, 65, 65, 65, 66,
        66, 66, 68, 68, 68, 70, 71, 71, 72, 74, 74, 74, 76, 77, 77, 80, 81, 81,
        83, 86, 86, 87, 91, 93, 93, 97, 65, 66, 66, 66, 66, 66, 66, 67, 69, 69,
        69, 71, 72, 72, 73, 76, 76, 77, 79, 80, 80, 83, 85, 85, 87, 90, 90, 91,
        95, 96, 96, 100, 70, 70, 70, 70, 69, 68, 68, 69, 71, 72, 72, 74, 75, 75,
        78, 83, 83, 84, 90, 92, 92, 94, 96, 96, 99, 102, 102, 103, 107, 108,
        108, 112, 72, 71, 71, 71, 70, 69, 69, 70, 72, 73, 73, 75, 77, 77, 80,
        85, 85, 87, 94, 96, 96, 99, 101, 101, 103, 107, 107, 108, 112, 113, 113,
        117, 72, 71, 71, 71, 70, 69, 69, 70, 72, 73, 73, 75, 77, 77, 80, 85, 85,
        87, 94, 96, 96, 99, 101, 101, 103, 107, 107, 108, 112, 113, 113, 117,
        81, 79, 79, 79, 78, 76, 76, 77, 79, 79, 79, 81, 82, 82, 85, 91, 91, 93,
        100, 103, 103, 107, 109, 109, 113, 118, 118, 119, 123, 125, 125, 129,
        88, 86, 85, 85, 83, 82, 82, 82, 84, 84, 84, 85, 85, 85, 89, 96, 96, 98,
        105, 108, 108, 112, 116, 116, 120, 126, 126, 128, 132, 134, 134, 139,
        88, 86, 85, 85, 83, 82, 82, 82, 84, 84, 84, 85, 85, 85, 89, 96, 96, 98,
        105, 108, 108, 112, 116, 116, 120, 126, 126, 128, 132, 134, 134, 139,
        95, 93, 91, 91, 90, 88, 88, 88, 89, 90, 90, 90, 90, 90, 94, 101, 101,
        103, 110, 112, 112, 117, 121, 121, 125, 132, 132, 134, 139, 140, 140,
        146, 107, 105, 103, 103, 101, 99, 99, 99, 99, 100, 100, 99, 99, 99, 103,
        109, 109, 111, 118, 120, 120, 126, 130, 130, 135, 143, 143, 145, 150,
        152, 152, 159,
        /* Size 4x16 */
        63, 64, 72, 88, 64, 65, 71, 85, 64, 65, 70, 83, 64, 66, 69, 82, 64, 68,
        72, 84, 65, 68, 73, 84, 65, 71, 77, 85, 66, 72, 80, 89, 68, 74, 85, 96,
        70, 76, 94, 105, 70, 77, 96, 108, 77, 81, 101, 116, 80, 83, 103, 120,
        84, 86, 107, 126, 91, 91, 112, 132, 92, 93, 113, 134,
        /* Size 16x4 */
        63, 64, 64, 64, 64, 65, 65, 66, 68, 70, 70, 77, 80, 84, 91, 92, 64, 65,
        65, 66, 68, 68, 71, 72, 74, 76, 77, 81, 83, 86, 91, 93, 72, 71, 70, 69,
        72, 73, 77, 80, 85, 94, 96, 101, 103, 107, 112, 113, 88, 85, 83, 82, 84,
        84, 85, 89, 96, 105, 108, 116, 120, 126, 132, 134,
        /* Size 8x32 */
        64, 63, 63, 64, 70, 72, 88, 95, 63, 64, 64, 65, 70, 71, 86, 93, 63, 64,
        64, 65, 70, 71, 85, 91, 63, 64, 64, 65, 70, 71, 85, 91, 63, 64, 65, 65,
        69, 70, 83, 90, 63, 65, 65, 66, 68, 69, 82, 88, 63, 65, 65, 66, 68, 69,
        82, 88, 63, 65, 65, 66, 69, 70, 82, 88, 63, 65, 66, 68, 71, 72, 84, 89,
        64, 65, 66, 68, 72, 73, 84, 90, 64, 65, 66, 68, 72, 73, 84, 90, 64, 65,
        67, 70, 74, 75, 85, 90, 64, 66, 68, 71, 75, 77, 85, 90, 64, 66, 68, 71,
        75, 77, 85, 90, 65, 66, 69, 72, 78, 80, 89, 94, 68, 68, 70, 74, 83, 85,
        96, 101, 68, 68, 70, 74, 83, 85, 96, 101, 68, 68, 70, 74, 84, 87, 98,
        103, 71, 69, 72, 76, 90, 94, 105, 110, 72, 69, 72, 77, 92, 96, 108, 112,
        72, 69, 72, 77, 92, 96, 108, 112, 76, 73, 75, 80, 94, 99, 112, 117, 79,
        75, 78, 81, 96, 101, 116, 121, 79, 75, 78, 81, 96, 101, 116, 121, 82,
        78, 80, 83, 99, 103, 120, 125, 88, 82, 84, 86, 102, 107, 126, 132, 88,
        82, 84, 86, 102, 107, 126, 132, 89, 84, 85, 87, 103, 108, 128, 134, 94,
        88, 89, 91, 107, 112, 132, 139, 96, 90, 91, 93, 108, 113, 134, 140, 96,
        90, 91, 93, 108, 113, 134, 140, 103, 95, 96, 97, 112, 117, 139, 146,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 65, 68, 68, 68,
        71, 72, 72, 76, 79, 79, 82, 88, 88, 89, 94, 96, 96, 103, 63, 64, 64, 64,
        64, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 68, 68, 68, 69, 69, 69, 73,
        75, 75, 78, 82, 82, 84, 88, 90, 90, 95, 63, 64, 64, 64, 65, 65, 65, 65,
        66, 66, 66, 67, 68, 68, 69, 70, 70, 70, 72, 72, 72, 75, 78, 78, 80, 84,
        84, 85, 89, 91, 91, 96, 64, 65, 65, 65, 65, 66, 66, 66, 68, 68, 68, 70,
        71, 71, 72, 74, 74, 74, 76, 77, 77, 80, 81, 81, 83, 86, 86, 87, 91, 93,
        93, 97, 70, 70, 70, 70, 69, 68, 68, 69, 71, 72, 72, 74, 75, 75, 78, 83,
        83, 84, 90, 92, 92, 94, 96, 96, 99, 102, 102, 103, 107, 108, 108, 112,
        72, 71, 71, 71, 70, 69, 69, 70, 72, 73, 73, 75, 77, 77, 80, 85, 85, 87,
        94, 96, 96, 99, 101, 101, 103, 107, 107, 108, 112, 113, 113, 117, 88,
        86, 85, 85, 83, 82, 82, 82, 84, 84, 84, 85, 85, 85, 89, 96, 96, 98, 105,
        108, 108, 112, 116, 116, 120, 126, 126, 128, 132, 134, 134, 139, 95, 93,
        91, 91, 90, 88, 88, 88, 89, 90, 90, 90, 90, 90, 94, 101, 101, 103, 110,
        112, 112, 117, 121, 121, 125, 132, 132, 134, 139, 140, 140, 146 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        66, 91, 91, 109,
#endif
        /* Size 4x4 */
        63, 74, 95, 94, 74, 88, 95, 91, 95, 95, 106, 107, 94, 91, 107, 118,
        /* Size 8x8 */
        63, 62, 68, 75, 86, 96, 95, 99, 62, 64, 71, 80, 87, 92, 90, 93, 68, 71,
        78, 87, 91, 93, 91, 93, 75, 80, 87, 95, 95, 94, 91, 92, 86, 87, 91, 95,
        99, 101, 100, 101, 96, 92, 93, 94, 101, 107, 110, 111, 95, 90, 91, 91,
        100, 110, 117, 120, 99, 93, 93, 92, 101, 111, 120, 123,
        /* Size 16x16 */
        64, 62, 62, 61, 66, 67, 73, 77, 83, 95, 98, 97, 98, 98, 101, 101, 62,
        63, 63, 62, 68, 69, 77, 80, 85, 93, 95, 94, 94, 94, 96, 97, 62, 63, 63,
        63, 68, 70, 78, 81, 85, 92, 94, 93, 93, 92, 95, 95, 61, 62, 63, 64, 69,
        71, 80, 82, 85, 91, 93, 91, 91, 90, 92, 93, 66, 68, 68, 69, 75, 77, 85,
        87, 89, 93, 94, 92, 92, 91, 93, 93, 67, 69, 70, 71, 77, 78, 87, 88, 90,
        94, 95, 93, 92, 91, 93, 93, 73, 77, 78, 80, 85, 87, 95, 95, 95, 95, 96,
        93, 92, 91, 92, 92, 77, 80, 81, 82, 87, 88, 95, 95, 96, 97, 98, 96, 95,
        94, 95, 95, 83, 85, 85, 85, 89, 90, 95, 96, 97, 100, 101, 99, 99, 98,
        100, 100, 95, 93, 92, 91, 93, 94, 95, 97, 100, 104, 105, 105, 105, 105,
        106, 106, 98, 95, 94, 93, 94, 95, 96, 98, 101, 105, 106, 106, 107, 107,
        108, 108, 97, 94, 93, 91, 92, 93, 93, 96, 99, 105, 106, 109, 110, 111,
        113, 113, 98, 94, 93, 91, 92, 92, 92, 95, 99, 105, 107, 110, 111, 114,
        115, 116, 98, 94, 92, 90, 91, 91, 91, 94, 98, 105, 107, 111, 114, 117,
        119, 120, 101, 96, 95, 92, 93, 93, 92, 95, 100, 106, 108, 113, 115, 119,
        122, 122, 101, 97, 95, 93, 93, 93, 92, 95, 100, 106, 108, 113, 116, 120,
        122, 123,
        /* Size 32x32 */
        64, 63, 62, 62, 62, 61, 61, 62, 66, 67, 67, 71, 73, 73, 77, 83, 83, 86,
        95, 98, 98, 98, 97, 97, 98, 98, 98, 99, 101, 101, 101, 103, 63, 63, 63,
        63, 62, 62, 62, 63, 67, 68, 68, 72, 75, 75, 79, 84, 84, 86, 94, 96, 96,
        96, 95, 95, 95, 95, 95, 96, 98, 99, 99, 100, 62, 63, 63, 63, 63, 62, 62,
        64, 68, 69, 69, 74, 77, 77, 80, 85, 85, 87, 93, 95, 95, 95, 94, 94, 94,
        94, 94, 94, 96, 97, 97, 99, 62, 63, 63, 63, 63, 62, 62, 64, 68, 69, 69,
        74, 77, 77, 80, 85, 85, 87, 93, 95, 95, 95, 94, 94, 94, 94, 94, 94, 96,
        97, 97, 99, 62, 62, 63, 63, 63, 63, 63, 64, 68, 70, 70, 75, 78, 78, 81,
        85, 85, 87, 92, 94, 94, 93, 93, 93, 93, 92, 92, 93, 95, 95, 95, 97, 61,
        62, 62, 62, 63, 64, 64, 65, 69, 71, 71, 76, 80, 80, 82, 85, 85, 87, 91,
        93, 93, 92, 91, 91, 91, 90, 90, 91, 92, 93, 93, 94, 61, 62, 62, 62, 63,
        64, 64, 65, 69, 71, 71, 76, 80, 80, 82, 85, 85, 87, 91, 93, 93, 92, 91,
        91, 91, 90, 90, 91, 92, 93, 93, 94, 62, 63, 64, 64, 64, 65, 65, 66, 71,
        72, 72, 77, 81, 81, 83, 86, 86, 87, 92, 93, 93, 92, 91, 91, 91, 90, 90,
        91, 92, 93, 93, 94, 66, 67, 68, 68, 68, 69, 69, 71, 75, 77, 77, 82, 85,
        85, 87, 89, 89, 90, 93, 94, 94, 93, 92, 92, 92, 91, 91, 91, 93, 93, 93,
        94, 67, 68, 69, 69, 70, 71, 71, 72, 77, 78, 78, 83, 87, 87, 88, 90, 90,
        91, 94, 95, 95, 93, 93, 93, 92, 91, 91, 91, 93, 93, 93, 94, 67, 68, 69,
        69, 70, 71, 71, 72, 77, 78, 78, 83, 87, 87, 88, 90, 90, 91, 94, 95, 95,
        93, 93, 93, 92, 91, 91, 91, 93, 93, 93, 94, 71, 72, 74, 74, 75, 76, 76,
        77, 82, 83, 83, 88, 92, 92, 92, 93, 93, 93, 95, 95, 95, 94, 93, 93, 92,
        91, 91, 91, 92, 93, 93, 94, 73, 75, 77, 77, 78, 80, 80, 81, 85, 87, 87,
        92, 95, 95, 95, 95, 95, 95, 95, 96, 96, 94, 93, 93, 92, 91, 91, 91, 92,
        92, 92, 93, 73, 75, 77, 77, 78, 80, 80, 81, 85, 87, 87, 92, 95, 95, 95,
        95, 95, 95, 95, 96, 96, 94, 93, 93, 92, 91, 91, 91, 92, 92, 92, 93, 77,
        79, 80, 80, 81, 82, 82, 83, 87, 88, 88, 92, 95, 95, 95, 96, 96, 96, 97,
        98, 98, 96, 96, 96, 95, 94, 94, 94, 95, 95, 95, 96, 83, 84, 85, 85, 85,
        85, 85, 86, 89, 90, 90, 93, 95, 95, 96, 97, 97, 98, 100, 101, 101, 100,
        99, 99, 99, 98, 98, 99, 100, 100, 100, 101, 83, 84, 85, 85, 85, 85, 85,
        86, 89, 90, 90, 93, 95, 95, 96, 97, 97, 98, 100, 101, 101, 100, 99, 99,
        99, 98, 98, 99, 100, 100, 100, 101, 86, 86, 87, 87, 87, 87, 87, 87, 90,
        91, 91, 93, 95, 95, 96, 98, 98, 99, 101, 102, 102, 101, 101, 101, 100,
        100, 100, 100, 101, 101, 101, 102, 95, 94, 93, 93, 92, 91, 91, 92, 93,
        94, 94, 95, 95, 95, 97, 100, 100, 101, 104, 105, 105, 105, 105, 105,
        105, 105, 105, 105, 106, 106, 106, 107, 98, 96, 95, 95, 94, 93, 93, 93,
        94, 95, 95, 95, 96, 96, 98, 101, 101, 102, 105, 106, 106, 106, 106, 106,
        107, 107, 107, 107, 108, 108, 108, 108, 98, 96, 95, 95, 94, 93, 93, 93,
        94, 95, 95, 95, 96, 96, 98, 101, 101, 102, 105, 106, 106, 106, 106, 106,
        107, 107, 107, 107, 108, 108, 108, 108, 98, 96, 95, 95, 93, 92, 92, 92,
        93, 93, 93, 94, 94, 94, 96, 100, 100, 101, 105, 106, 106, 107, 108, 108,
        108, 110, 110, 110, 111, 111, 111, 112, 97, 95, 94, 94, 93, 91, 91, 91,
        92, 93, 93, 93, 93, 93, 96, 99, 99, 101, 105, 106, 106, 108, 109, 109,
        110, 111, 111, 112, 113, 113, 113, 114, 97, 95, 94, 94, 93, 91, 91, 91,
        92, 93, 93, 93, 93, 93, 96, 99, 99, 101, 105, 106, 106, 108, 109, 109,
        110, 111, 111, 112, 113, 113, 113, 114, 98, 95, 94, 94, 93, 91, 91, 91,
        92, 92, 92, 92, 92, 92, 95, 99, 99, 100, 105, 107, 107, 108, 110, 110,
        111, 114, 114, 114, 115, 116, 116, 117, 98, 95, 94, 94, 92, 90, 90, 90,
        91, 91, 91, 91, 91, 91, 94, 98, 98, 100, 105, 107, 107, 110, 111, 111,
        114, 117, 117, 118, 119, 120, 120, 121, 98, 95, 94, 94, 92, 90, 90, 90,
        91, 91, 91, 91, 91, 91, 94, 98, 98, 100, 105, 107, 107, 110, 111, 111,
        114, 117, 117, 118, 119, 120, 120, 121, 99, 96, 94, 94, 93, 91, 91, 91,
        91, 91, 91, 91, 91, 91, 94, 99, 99, 100, 105, 107, 107, 110, 112, 112,
        114, 118, 118, 118, 120, 120, 120, 122, 101, 98, 96, 96, 95, 92, 92, 92,
        93, 93, 93, 92, 92, 92, 95, 100, 100, 101, 106, 108, 108, 111, 113, 113,
        115, 119, 119, 120, 122, 122, 122, 125, 101, 99, 97, 97, 95, 93, 93, 93,
        93, 93, 93, 93, 92, 92, 95, 100, 100, 101, 106, 108, 108, 111, 113, 113,
        116, 120, 120, 120, 122, 123, 123, 126, 101, 99, 97, 97, 95, 93, 93, 93,
        93, 93, 93, 93, 92, 92, 95, 100, 100, 101, 106, 108, 108, 111, 113, 113,
        116, 120, 120, 120, 122, 123, 123, 126, 103, 100, 99, 99, 97, 94, 94,
        94, 94, 94, 94, 94, 93, 93, 96, 101, 101, 102, 107, 108, 108, 112, 114,
        114, 117, 121, 121, 122, 125, 126, 126, 129,
        /* Size 4x8 */
        63, 76, 95, 96, 63, 80, 92, 91, 70, 86, 94, 92, 78, 94, 95, 91, 87, 95,
        101, 100, 94, 95, 106, 110, 93, 92, 107, 117, 96, 93, 108, 119,
        /* Size 8x4 */
        63, 63, 70, 78, 87, 94, 93, 96, 76, 80, 86, 94, 95, 95, 92, 93, 95, 92,
        94, 95, 101, 106, 107, 108, 96, 91, 92, 91, 100, 110, 117, 119,
        /* Size 8x16 */
        64, 62, 66, 74, 91, 96, 98, 101, 63, 63, 68, 77, 90, 94, 94, 97, 62, 64,
        69, 78, 90, 93, 93, 95, 61, 64, 70, 80, 89, 92, 91, 93, 66, 70, 75, 85,
        92, 94, 91, 93, 67, 72, 77, 86, 92, 94, 92, 93, 74, 80, 86, 94, 95, 95,
        91, 92, 78, 82, 87, 95, 97, 97, 94, 95, 84, 86, 89, 95, 99, 100, 99,
        100, 95, 92, 93, 96, 103, 105, 106, 106, 98, 93, 94, 96, 104, 106, 107,
        108, 97, 92, 93, 94, 103, 106, 112, 113, 97, 91, 92, 93, 103, 106, 114,
        115, 98, 91, 91, 92, 103, 107, 117, 119, 100, 93, 93, 93, 104, 108, 119,
        122, 101, 93, 93, 93, 105, 108, 119, 123,
        /* Size 16x8 */
        64, 63, 62, 61, 66, 67, 74, 78, 84, 95, 98, 97, 97, 98, 100, 101, 62,
        63, 64, 64, 70, 72, 80, 82, 86, 92, 93, 92, 91, 91, 93, 93, 66, 68, 69,
        70, 75, 77, 86, 87, 89, 93, 94, 93, 92, 91, 93, 93, 74, 77, 78, 80, 85,
        86, 94, 95, 95, 96, 96, 94, 93, 92, 93, 93, 91, 90, 90, 89, 92, 92, 95,
        97, 99, 103, 104, 103, 103, 103, 104, 105, 96, 94, 93, 92, 94, 94, 95,
        97, 100, 105, 106, 106, 106, 107, 108, 108, 98, 94, 93, 91, 91, 92, 91,
        94, 99, 106, 107, 112, 114, 117, 119, 119, 101, 97, 95, 93, 93, 93, 92,
        95, 100, 106, 108, 113, 115, 119, 122, 123,
        /* Size 16x32 */
        64, 62, 62, 62, 66, 74, 74, 77, 91, 96, 96, 98, 98, 98, 101, 105, 63,
        63, 63, 63, 67, 76, 76, 79, 90, 95, 95, 96, 96, 96, 98, 102, 63, 63, 63,
        63, 68, 77, 77, 80, 90, 94, 94, 94, 94, 94, 97, 100, 63, 63, 63, 63, 68,
        77, 77, 80, 90, 94, 94, 94, 94, 94, 97, 100, 62, 63, 64, 64, 69, 78, 78,
        81, 90, 93, 93, 93, 93, 93, 95, 98, 61, 63, 64, 64, 70, 80, 80, 82, 89,
        92, 92, 91, 91, 91, 93, 96, 61, 63, 64, 64, 70, 80, 80, 82, 89, 92, 92,
        91, 91, 91, 93, 96, 62, 64, 66, 66, 71, 81, 81, 83, 90, 92, 92, 91, 91,
        91, 93, 96, 66, 68, 70, 70, 75, 85, 85, 86, 92, 94, 94, 92, 91, 91, 93,
        95, 67, 70, 72, 72, 77, 86, 86, 88, 92, 94, 94, 93, 92, 92, 93, 95, 67,
        70, 72, 72, 77, 86, 86, 88, 92, 94, 94, 93, 92, 92, 93, 95, 71, 74, 77,
        77, 82, 91, 91, 92, 94, 95, 95, 93, 91, 91, 93, 95, 74, 78, 80, 80, 86,
        94, 94, 95, 95, 95, 95, 93, 91, 91, 92, 94, 74, 78, 80, 80, 86, 94, 94,
        95, 95, 95, 95, 93, 91, 91, 92, 94, 78, 80, 82, 82, 87, 95, 95, 95, 97,
        97, 97, 95, 94, 94, 95, 97, 84, 85, 86, 86, 89, 95, 95, 96, 99, 100,
        100, 99, 99, 99, 100, 101, 84, 85, 86, 86, 89, 95, 95, 96, 99, 100, 100,
        99, 99, 99, 100, 101, 86, 87, 87, 87, 90, 95, 95, 96, 100, 101, 101,
        101, 100, 100, 101, 103, 95, 93, 92, 92, 93, 96, 96, 97, 103, 105, 105,
        105, 106, 106, 106, 107, 98, 95, 93, 93, 94, 96, 96, 98, 104, 106, 106,
        107, 107, 107, 108, 109, 98, 95, 93, 93, 94, 96, 96, 98, 104, 106, 106,
        107, 107, 107, 108, 109, 97, 94, 92, 92, 93, 95, 95, 97, 104, 106, 106,
        108, 110, 110, 111, 113, 97, 94, 92, 92, 93, 94, 94, 96, 103, 106, 106,
        109, 112, 112, 113, 115, 97, 94, 92, 92, 93, 94, 94, 96, 103, 106, 106,
        109, 112, 112, 113, 115, 97, 94, 91, 91, 92, 93, 93, 95, 103, 106, 106,
        111, 114, 114, 115, 118, 98, 93, 91, 91, 91, 92, 92, 94, 103, 107, 107,
        113, 117, 117, 119, 123, 98, 93, 91, 91, 91, 92, 92, 94, 103, 107, 107,
        113, 117, 117, 119, 123, 98, 94, 91, 91, 91, 92, 92, 95, 104, 107, 107,
        113, 117, 117, 120, 124, 100, 96, 93, 93, 93, 93, 93, 96, 104, 108, 108,
        114, 119, 119, 122, 127, 101, 96, 93, 93, 93, 93, 93, 96, 105, 108, 108,
        115, 119, 119, 123, 128, 101, 96, 93, 93, 93, 93, 93, 96, 105, 108, 108,
        115, 119, 119, 123, 128, 103, 98, 95, 95, 95, 94, 94, 97, 105, 108, 108,
        116, 121, 121, 125, 131,
        /* Size 32x16 */
        64, 63, 63, 63, 62, 61, 61, 62, 66, 67, 67, 71, 74, 74, 78, 84, 84, 86,
        95, 98, 98, 97, 97, 97, 97, 98, 98, 98, 100, 101, 101, 103, 62, 63, 63,
        63, 63, 63, 63, 64, 68, 70, 70, 74, 78, 78, 80, 85, 85, 87, 93, 95, 95,
        94, 94, 94, 94, 93, 93, 94, 96, 96, 96, 98, 62, 63, 63, 63, 64, 64, 64,
        66, 70, 72, 72, 77, 80, 80, 82, 86, 86, 87, 92, 93, 93, 92, 92, 92, 91,
        91, 91, 91, 93, 93, 93, 95, 62, 63, 63, 63, 64, 64, 64, 66, 70, 72, 72,
        77, 80, 80, 82, 86, 86, 87, 92, 93, 93, 92, 92, 92, 91, 91, 91, 91, 93,
        93, 93, 95, 66, 67, 68, 68, 69, 70, 70, 71, 75, 77, 77, 82, 86, 86, 87,
        89, 89, 90, 93, 94, 94, 93, 93, 93, 92, 91, 91, 91, 93, 93, 93, 95, 74,
        76, 77, 77, 78, 80, 80, 81, 85, 86, 86, 91, 94, 94, 95, 95, 95, 95, 96,
        96, 96, 95, 94, 94, 93, 92, 92, 92, 93, 93, 93, 94, 74, 76, 77, 77, 78,
        80, 80, 81, 85, 86, 86, 91, 94, 94, 95, 95, 95, 95, 96, 96, 96, 95, 94,
        94, 93, 92, 92, 92, 93, 93, 93, 94, 77, 79, 80, 80, 81, 82, 82, 83, 86,
        88, 88, 92, 95, 95, 95, 96, 96, 96, 97, 98, 98, 97, 96, 96, 95, 94, 94,
        95, 96, 96, 96, 97, 91, 90, 90, 90, 90, 89, 89, 90, 92, 92, 92, 94, 95,
        95, 97, 99, 99, 100, 103, 104, 104, 104, 103, 103, 103, 103, 103, 104,
        104, 105, 105, 105, 96, 95, 94, 94, 93, 92, 92, 92, 94, 94, 94, 95, 95,
        95, 97, 100, 100, 101, 105, 106, 106, 106, 106, 106, 106, 107, 107, 107,
        108, 108, 108, 108, 96, 95, 94, 94, 93, 92, 92, 92, 94, 94, 94, 95, 95,
        95, 97, 100, 100, 101, 105, 106, 106, 106, 106, 106, 106, 107, 107, 107,
        108, 108, 108, 108, 98, 96, 94, 94, 93, 91, 91, 91, 92, 93, 93, 93, 93,
        93, 95, 99, 99, 101, 105, 107, 107, 108, 109, 109, 111, 113, 113, 113,
        114, 115, 115, 116, 98, 96, 94, 94, 93, 91, 91, 91, 91, 92, 92, 91, 91,
        91, 94, 99, 99, 100, 106, 107, 107, 110, 112, 112, 114, 117, 117, 117,
        119, 119, 119, 121, 98, 96, 94, 94, 93, 91, 91, 91, 91, 92, 92, 91, 91,
        91, 94, 99, 99, 100, 106, 107, 107, 110, 112, 112, 114, 117, 117, 117,
        119, 119, 119, 121, 101, 98, 97, 97, 95, 93, 93, 93, 93, 93, 93, 93, 92,
        92, 95, 100, 100, 101, 106, 108, 108, 111, 113, 113, 115, 119, 119, 120,
        122, 123, 123, 125, 105, 102, 100, 100, 98, 96, 96, 96, 95, 95, 95, 95,
        94, 94, 97, 101, 101, 103, 107, 109, 109, 113, 115, 115, 118, 123, 123,
        124, 127, 128, 128, 131,
        /* Size 4x16 */
        62, 74, 96, 98, 63, 77, 94, 94, 63, 78, 93, 93, 63, 80, 92, 91, 68, 85,
        94, 91, 70, 86, 94, 92, 78, 94, 95, 91, 80, 95, 97, 94, 85, 95, 100, 99,
        93, 96, 105, 106, 95, 96, 106, 107, 94, 94, 106, 112, 94, 93, 106, 114,
        93, 92, 107, 117, 96, 93, 108, 119, 96, 93, 108, 119,
        /* Size 16x4 */
        62, 63, 63, 63, 68, 70, 78, 80, 85, 93, 95, 94, 94, 93, 96, 96, 74, 77,
        78, 80, 85, 86, 94, 95, 95, 96, 96, 94, 93, 92, 93, 93, 96, 94, 93, 92,
        94, 94, 95, 97, 100, 105, 106, 106, 106, 107, 108, 108, 98, 94, 93, 91,
        91, 92, 91, 94, 99, 106, 107, 112, 114, 117, 119, 119,
        /* Size 8x32 */
        64, 62, 66, 74, 91, 96, 98, 101, 63, 63, 67, 76, 90, 95, 96, 98, 63, 63,
        68, 77, 90, 94, 94, 97, 63, 63, 68, 77, 90, 94, 94, 97, 62, 64, 69, 78,
        90, 93, 93, 95, 61, 64, 70, 80, 89, 92, 91, 93, 61, 64, 70, 80, 89, 92,
        91, 93, 62, 66, 71, 81, 90, 92, 91, 93, 66, 70, 75, 85, 92, 94, 91, 93,
        67, 72, 77, 86, 92, 94, 92, 93, 67, 72, 77, 86, 92, 94, 92, 93, 71, 77,
        82, 91, 94, 95, 91, 93, 74, 80, 86, 94, 95, 95, 91, 92, 74, 80, 86, 94,
        95, 95, 91, 92, 78, 82, 87, 95, 97, 97, 94, 95, 84, 86, 89, 95, 99, 100,
        99, 100, 84, 86, 89, 95, 99, 100, 99, 100, 86, 87, 90, 95, 100, 101,
        100, 101, 95, 92, 93, 96, 103, 105, 106, 106, 98, 93, 94, 96, 104, 106,
        107, 108, 98, 93, 94, 96, 104, 106, 107, 108, 97, 92, 93, 95, 104, 106,
        110, 111, 97, 92, 93, 94, 103, 106, 112, 113, 97, 92, 93, 94, 103, 106,
        112, 113, 97, 91, 92, 93, 103, 106, 114, 115, 98, 91, 91, 92, 103, 107,
        117, 119, 98, 91, 91, 92, 103, 107, 117, 119, 98, 91, 91, 92, 104, 107,
        117, 120, 100, 93, 93, 93, 104, 108, 119, 122, 101, 93, 93, 93, 105,
        108, 119, 123, 101, 93, 93, 93, 105, 108, 119, 123, 103, 95, 95, 94,
        105, 108, 121, 125,
        /* Size 32x8 */
        64, 63, 63, 63, 62, 61, 61, 62, 66, 67, 67, 71, 74, 74, 78, 84, 84, 86,
        95, 98, 98, 97, 97, 97, 97, 98, 98, 98, 100, 101, 101, 103, 62, 63, 63,
        63, 64, 64, 64, 66, 70, 72, 72, 77, 80, 80, 82, 86, 86, 87, 92, 93, 93,
        92, 92, 92, 91, 91, 91, 91, 93, 93, 93, 95, 66, 67, 68, 68, 69, 70, 70,
        71, 75, 77, 77, 82, 86, 86, 87, 89, 89, 90, 93, 94, 94, 93, 93, 93, 92,
        91, 91, 91, 93, 93, 93, 95, 74, 76, 77, 77, 78, 80, 80, 81, 85, 86, 86,
        91, 94, 94, 95, 95, 95, 95, 96, 96, 96, 95, 94, 94, 93, 92, 92, 92, 93,
        93, 93, 94, 91, 90, 90, 90, 90, 89, 89, 90, 92, 92, 92, 94, 95, 95, 97,
        99, 99, 100, 103, 104, 104, 104, 103, 103, 103, 103, 103, 104, 104, 105,
        105, 105, 96, 95, 94, 94, 93, 92, 92, 92, 94, 94, 94, 95, 95, 95, 97,
        100, 100, 101, 105, 106, 106, 106, 106, 106, 106, 107, 107, 107, 108,
        108, 108, 108, 98, 96, 94, 94, 93, 91, 91, 91, 91, 92, 92, 91, 91, 91,
        94, 99, 99, 100, 106, 107, 107, 110, 112, 112, 114, 117, 117, 117, 119,
        119, 119, 121, 101, 98, 97, 97, 95, 93, 93, 93, 93, 93, 93, 93, 92, 92,
        95, 100, 100, 101, 106, 108, 108, 111, 113, 113, 115, 119, 119, 120,
        122, 123, 123, 125 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        65, 69, 69, 97,
#endif
        /* Size 4x4 */
        64, 64, 68, 77, 64, 67, 71, 78, 68, 71, 79, 90, 77, 78, 90, 108,
        /* Size 8x8 */
        63, 63, 64, 64, 66, 69, 74, 82, 63, 64, 65, 65, 66, 69, 73, 79, 64, 65,
        65, 66, 68, 70, 74, 80, 64, 65, 66, 68, 71, 73, 77, 82, 66, 66, 68, 71,
        74, 78, 83, 88, 69, 69, 70, 73, 78, 86, 93, 98, 74, 73, 74, 77, 83, 93,
        102, 108, 82, 79, 80, 82, 88, 98, 108, 116,
        /* Size 16x16 */
        64, 63, 63, 63, 63, 63, 63, 64, 64, 68, 68, 72, 72, 79, 79, 88, 63, 64,
        64, 64, 64, 64, 64, 65, 65, 68, 68, 70, 70, 77, 77, 85, 63, 64, 64, 64,
        64, 64, 64, 65, 65, 68, 68, 70, 70, 77, 77, 85, 63, 64, 64, 65, 65, 65,
        65, 65, 65, 67, 67, 69, 69, 75, 75, 82, 63, 64, 64, 65, 65, 65, 65, 65,
        65, 67, 67, 69, 69, 75, 75, 82, 63, 64, 64, 65, 65, 67, 67, 68, 68, 71,
        71, 73, 73, 78, 78, 84, 63, 64, 64, 65, 65, 67, 67, 68, 68, 71, 71, 73,
        73, 78, 78, 84, 64, 65, 65, 65, 65, 68, 68, 71, 71, 74, 74, 77, 77, 81,
        81, 85, 64, 65, 65, 65, 65, 68, 68, 71, 71, 74, 74, 77, 77, 81, 81, 85,
        68, 68, 68, 67, 67, 71, 71, 74, 74, 79, 79, 85, 85, 90, 90, 95, 68, 68,
        68, 67, 67, 71, 71, 74, 74, 79, 79, 85, 85, 90, 90, 95, 72, 70, 70, 69,
        69, 73, 73, 77, 77, 85, 85, 97, 97, 101, 101, 108, 72, 70, 70, 69, 69,
        73, 73, 77, 77, 85, 85, 97, 97, 101, 101, 108, 79, 77, 77, 75, 75, 78,
        78, 81, 81, 90, 90, 101, 101, 108, 108, 116, 79, 77, 77, 75, 75, 78, 78,
        81, 81, 90, 90, 101, 101, 108, 108, 116, 88, 85, 85, 82, 82, 84, 84, 85,
        85, 95, 95, 108, 108, 116, 116, 127,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 66,
        68, 68, 68, 70, 72, 72, 72, 75, 79, 79, 79, 83, 88, 88, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 66, 68, 68, 68, 69,
        71, 71, 71, 74, 78, 78, 78, 82, 86, 86, 63, 63, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 65, 65, 65, 65, 66, 68, 68, 68, 69, 70, 70, 70, 74,
        77, 77, 77, 81, 85, 85, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 65, 65, 65, 65, 66, 68, 68, 68, 69, 70, 70, 70, 74, 77, 77, 77, 81,
        85, 85, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65,
        65, 66, 68, 68, 68, 69, 70, 70, 70, 74, 77, 77, 77, 81, 85, 85, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 66, 68, 68,
        68, 69, 70, 70, 70, 73, 76, 76, 76, 79, 83, 83, 63, 63, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 67, 67, 67, 68, 69, 69,
        69, 72, 75, 75, 75, 78, 82, 82, 63, 63, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 66, 67, 67, 67, 68, 69, 69, 69, 72, 75, 75,
        75, 78, 82, 82, 63, 63, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 66, 67, 67, 67, 68, 69, 69, 69, 72, 75, 75, 75, 78, 82, 82,
        63, 64, 64, 64, 64, 65, 65, 65, 65, 65, 66, 66, 66, 66, 67, 67, 67, 68,
        69, 69, 69, 70, 71, 71, 71, 74, 77, 77, 77, 80, 83, 83, 63, 64, 64, 64,
        64, 65, 65, 65, 65, 66, 67, 67, 67, 67, 68, 68, 68, 69, 71, 71, 71, 72,
        73, 73, 73, 76, 78, 78, 78, 81, 84, 84, 63, 64, 64, 64, 64, 65, 65, 65,
        65, 66, 67, 67, 67, 67, 68, 68, 68, 69, 71, 71, 71, 72, 73, 73, 73, 76,
        78, 78, 78, 81, 84, 84, 63, 64, 64, 64, 64, 65, 65, 65, 65, 66, 67, 67,
        67, 67, 68, 68, 68, 69, 71, 71, 71, 72, 73, 73, 73, 76, 78, 78, 78, 81,
        84, 84, 64, 64, 65, 65, 65, 65, 65, 65, 65, 66, 67, 67, 67, 68, 69, 69,
        69, 71, 72, 72, 72, 73, 75, 75, 75, 77, 80, 80, 80, 82, 85, 85, 64, 64,
        65, 65, 65, 65, 65, 65, 65, 67, 68, 68, 68, 69, 71, 71, 71, 72, 74, 74,
        74, 75, 77, 77, 77, 79, 81, 81, 81, 83, 85, 85, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 67, 68, 68, 68, 69, 71, 71, 71, 72, 74, 74, 74, 75, 77, 77,
        77, 79, 81, 81, 81, 83, 85, 85, 64, 64, 65, 65, 65, 65, 65, 65, 65, 67,
        68, 68, 68, 69, 71, 71, 71, 72, 74, 74, 74, 75, 77, 77, 77, 79, 81, 81,
        81, 83, 85, 85, 66, 66, 66, 66, 66, 66, 66, 66, 66, 68, 69, 69, 69, 71,
        72, 72, 72, 74, 76, 76, 76, 78, 81, 81, 81, 83, 85, 85, 85, 88, 90, 90,
        68, 68, 68, 68, 68, 68, 67, 67, 67, 69, 71, 71, 71, 72, 74, 74, 74, 76,
        79, 79, 79, 82, 85, 85, 85, 88, 90, 90, 90, 93, 95, 95, 68, 68, 68, 68,
        68, 68, 67, 67, 67, 69, 71, 71, 71, 72, 74, 74, 74, 76, 79, 79, 79, 82,
        85, 85, 85, 88, 90, 90, 90, 93, 95, 95, 68, 68, 68, 68, 68, 68, 67, 67,
        67, 69, 71, 71, 71, 72, 74, 74, 74, 76, 79, 79, 79, 82, 85, 85, 85, 88,
        90, 90, 90, 93, 95, 95, 70, 69, 69, 69, 69, 69, 68, 68, 68, 70, 72, 72,
        72, 73, 75, 75, 75, 78, 82, 82, 82, 86, 91, 91, 91, 93, 95, 95, 95, 98,
        101, 101, 72, 71, 70, 70, 70, 70, 69, 69, 69, 71, 73, 73, 73, 75, 77,
        77, 77, 81, 85, 85, 85, 91, 97, 97, 97, 99, 101, 101, 101, 104, 108,
        108, 72, 71, 70, 70, 70, 70, 69, 69, 69, 71, 73, 73, 73, 75, 77, 77, 77,
        81, 85, 85, 85, 91, 97, 97, 97, 99, 101, 101, 101, 104, 108, 108, 72,
        71, 70, 70, 70, 70, 69, 69, 69, 71, 73, 73, 73, 75, 77, 77, 77, 81, 85,
        85, 85, 91, 97, 97, 97, 99, 101, 101, 101, 104, 108, 108, 75, 74, 74,
        74, 74, 73, 72, 72, 72, 74, 76, 76, 76, 77, 79, 79, 79, 83, 88, 88, 88,
        93, 99, 99, 99, 102, 104, 104, 104, 108, 112, 112, 79, 78, 77, 77, 77,
        76, 75, 75, 75, 77, 78, 78, 78, 80, 81, 81, 81, 85, 90, 90, 90, 95, 101,
        101, 101, 104, 108, 108, 108, 112, 116, 116, 79, 78, 77, 77, 77, 76, 75,
        75, 75, 77, 78, 78, 78, 80, 81, 81, 81, 85, 90, 90, 90, 95, 101, 101,
        101, 104, 108, 108, 108, 112, 116, 116, 79, 78, 77, 77, 77, 76, 75, 75,
        75, 77, 78, 78, 78, 80, 81, 81, 81, 85, 90, 90, 90, 95, 101, 101, 101,
        104, 108, 108, 108, 112, 116, 116, 83, 82, 81, 81, 81, 79, 78, 78, 78,
        80, 81, 81, 81, 82, 83, 83, 83, 88, 93, 93, 93, 98, 104, 104, 104, 108,
        112, 112, 112, 116, 121, 121, 88, 86, 85, 85, 85, 83, 82, 82, 82, 83,
        84, 84, 84, 85, 85, 85, 85, 90, 95, 95, 95, 101, 108, 108, 108, 112,
        116, 116, 116, 121, 127, 127, 88, 86, 85, 85, 85, 83, 82, 82, 82, 83,
        84, 84, 84, 85, 85, 85, 85, 90, 95, 95, 95, 101, 108, 108, 108, 112,
        116, 116, 116, 121, 127, 127,
        /* Size 4x8 */
        63, 64, 68, 78, 64, 65, 68, 76, 64, 66, 69, 77, 65, 67, 72, 80, 66, 69,
        76, 85, 69, 72, 82, 95, 74, 76, 88, 105, 81, 81, 93, 112,
        /* Size 8x4 */
        63, 64, 64, 65, 66, 69, 74, 81, 64, 65, 66, 67, 69, 72, 76, 81, 68, 68,
        69, 72, 76, 82, 88, 93, 78, 76, 77, 80, 85, 95, 105, 112,
        /* Size 8x16 */
        64, 63, 63, 64, 64, 72, 72, 88, 63, 64, 64, 65, 65, 71, 71, 85, 63, 64,
        64, 65, 65, 71, 71, 85, 63, 65, 65, 66, 66, 69, 69, 82, 63, 65, 65, 66,
        66, 69, 69, 82, 64, 65, 65, 68, 68, 73, 73, 84, 64, 65, 65, 68, 68, 73,
        73, 84, 64, 66, 66, 71, 71, 77, 77, 85, 64, 66, 66, 71, 71, 77, 77, 85,
        68, 68, 68, 74, 74, 85, 85, 96, 68, 68, 68, 74, 74, 85, 85, 96, 72, 69,
        69, 77, 77, 96, 96, 108, 72, 69, 69, 77, 77, 96, 96, 108, 79, 75, 75,
        81, 81, 101, 101, 116, 79, 75, 75, 81, 81, 101, 101, 116, 88, 82, 82,
        86, 86, 107, 107, 126,
        /* Size 16x8 */
        64, 63, 63, 63, 63, 64, 64, 64, 64, 68, 68, 72, 72, 79, 79, 88, 63, 64,
        64, 65, 65, 65, 65, 66, 66, 68, 68, 69, 69, 75, 75, 82, 63, 64, 64, 65,
        65, 65, 65, 66, 66, 68, 68, 69, 69, 75, 75, 82, 64, 65, 65, 66, 66, 68,
        68, 71, 71, 74, 74, 77, 77, 81, 81, 86, 64, 65, 65, 66, 66, 68, 68, 71,
        71, 74, 74, 77, 77, 81, 81, 86, 72, 71, 71, 69, 69, 73, 73, 77, 77, 85,
        85, 96, 96, 101, 101, 107, 72, 71, 71, 69, 69, 73, 73, 77, 77, 85, 85,
        96, 96, 101, 101, 107, 88, 85, 85, 82, 82, 84, 84, 85, 85, 96, 96, 108,
        108, 116, 116, 126,
        /* Size 16x32 */
        64, 63, 63, 63, 63, 64, 64, 64, 64, 68, 72, 72, 72, 79, 88, 88, 63, 63,
        63, 63, 63, 64, 64, 64, 64, 68, 71, 71, 71, 78, 86, 86, 63, 64, 64, 64,
        64, 64, 65, 65, 65, 68, 71, 71, 71, 77, 85, 85, 63, 64, 64, 64, 64, 64,
        65, 65, 65, 68, 71, 71, 71, 77, 85, 85, 63, 64, 64, 64, 64, 64, 65, 65,
        65, 68, 71, 71, 71, 77, 85, 85, 63, 64, 64, 64, 64, 65, 65, 65, 65, 68,
        70, 70, 70, 76, 83, 83, 63, 64, 65, 65, 65, 65, 66, 66, 66, 67, 69, 69,
        69, 75, 82, 82, 63, 64, 65, 65, 65, 65, 66, 66, 66, 67, 69, 69, 69, 75,
        82, 82, 63, 64, 65, 65, 65, 65, 66, 66, 66, 67, 69, 69, 69, 75, 82, 82,
        63, 64, 65, 65, 65, 66, 67, 67, 67, 69, 71, 71, 71, 77, 83, 83, 64, 64,
        65, 65, 65, 67, 68, 68, 68, 71, 73, 73, 73, 78, 84, 84, 64, 64, 65, 65,
        65, 67, 68, 68, 68, 71, 73, 73, 73, 78, 84, 84, 64, 64, 65, 65, 65, 67,
        68, 68, 68, 71, 73, 73, 73, 78, 84, 84, 64, 65, 65, 65, 65, 67, 69, 69,
        69, 72, 75, 75, 75, 80, 85, 85, 64, 65, 66, 66, 66, 68, 71, 71, 71, 74,
        77, 77, 77, 81, 85, 85, 64, 65, 66, 66, 66, 68, 71, 71, 71, 74, 77, 77,
        77, 81, 85, 85, 64, 65, 66, 66, 66, 68, 71, 71, 71, 74, 77, 77, 77, 81,
        85, 85, 66, 66, 67, 67, 67, 69, 72, 72, 72, 76, 81, 81, 81, 85, 90, 90,
        68, 68, 68, 68, 68, 71, 74, 74, 74, 79, 85, 85, 85, 90, 96, 96, 68, 68,
        68, 68, 68, 71, 74, 74, 74, 79, 85, 85, 85, 90, 96, 96, 68, 68, 68, 68,
        68, 71, 74, 74, 74, 79, 85, 85, 85, 90, 96, 96, 70, 69, 69, 69, 69, 72,
        75, 75, 75, 82, 90, 90, 90, 95, 101, 101, 72, 70, 69, 69, 69, 73, 77,
        77, 77, 86, 96, 96, 96, 102, 108, 108, 72, 70, 69, 69, 69, 73, 77, 77,
        77, 86, 96, 96, 96, 102, 108, 108, 72, 70, 69, 69, 69, 73, 77, 77, 77,
        86, 96, 96, 96, 102, 108, 108, 75, 74, 72, 72, 72, 76, 79, 79, 79, 88,
        98, 98, 98, 105, 112, 112, 79, 77, 75, 75, 75, 78, 81, 81, 81, 90, 101,
        101, 101, 108, 116, 116, 79, 77, 75, 75, 75, 78, 81, 81, 81, 90, 101,
        101, 101, 108, 116, 116, 79, 77, 75, 75, 75, 78, 81, 81, 81, 90, 101,
        101, 101, 108, 116, 116, 83, 81, 79, 79, 79, 81, 84, 84, 84, 93, 104,
        104, 104, 112, 121, 121, 88, 85, 82, 82, 82, 84, 86, 86, 86, 96, 107,
        107, 107, 116, 126, 126, 88, 85, 82, 82, 82, 84, 86, 86, 86, 96, 107,
        107, 107, 116, 126, 126,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 66,
        68, 68, 68, 70, 72, 72, 72, 75, 79, 79, 79, 83, 88, 88, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 66, 68, 68, 68, 69,
        70, 70, 70, 74, 77, 77, 77, 81, 85, 85, 63, 63, 64, 64, 64, 64, 65, 65,
        65, 65, 65, 65, 65, 65, 66, 66, 66, 67, 68, 68, 68, 69, 69, 69, 69, 72,
        75, 75, 75, 79, 82, 82, 63, 63, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 65, 66, 66, 66, 67, 68, 68, 68, 69, 69, 69, 69, 72, 75, 75, 75, 79,
        82, 82, 63, 63, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66,
        66, 67, 68, 68, 68, 69, 69, 69, 69, 72, 75, 75, 75, 79, 82, 82, 64, 64,
        64, 64, 64, 65, 65, 65, 65, 66, 67, 67, 67, 67, 68, 68, 68, 69, 71, 71,
        71, 72, 73, 73, 73, 76, 78, 78, 78, 81, 84, 84, 64, 64, 65, 65, 65, 65,
        66, 66, 66, 67, 68, 68, 68, 69, 71, 71, 71, 72, 74, 74, 74, 75, 77, 77,
        77, 79, 81, 81, 81, 84, 86, 86, 64, 64, 65, 65, 65, 65, 66, 66, 66, 67,
        68, 68, 68, 69, 71, 71, 71, 72, 74, 74, 74, 75, 77, 77, 77, 79, 81, 81,
        81, 84, 86, 86, 64, 64, 65, 65, 65, 65, 66, 66, 66, 67, 68, 68, 68, 69,
        71, 71, 71, 72, 74, 74, 74, 75, 77, 77, 77, 79, 81, 81, 81, 84, 86, 86,
        68, 68, 68, 68, 68, 68, 67, 67, 67, 69, 71, 71, 71, 72, 74, 74, 74, 76,
        79, 79, 79, 82, 86, 86, 86, 88, 90, 90, 90, 93, 96, 96, 72, 71, 71, 71,
        71, 70, 69, 69, 69, 71, 73, 73, 73, 75, 77, 77, 77, 81, 85, 85, 85, 90,
        96, 96, 96, 98, 101, 101, 101, 104, 107, 107, 72, 71, 71, 71, 71, 70,
        69, 69, 69, 71, 73, 73, 73, 75, 77, 77, 77, 81, 85, 85, 85, 90, 96, 96,
        96, 98, 101, 101, 101, 104, 107, 107, 72, 71, 71, 71, 71, 70, 69, 69,
        69, 71, 73, 73, 73, 75, 77, 77, 77, 81, 85, 85, 85, 90, 96, 96, 96, 98,
        101, 101, 101, 104, 107, 107, 79, 78, 77, 77, 77, 76, 75, 75, 75, 77,
        78, 78, 78, 80, 81, 81, 81, 85, 90, 90, 90, 95, 102, 102, 102, 105, 108,
        108, 108, 112, 116, 116, 88, 86, 85, 85, 85, 83, 82, 82, 82, 83, 84, 84,
        84, 85, 85, 85, 85, 90, 96, 96, 96, 101, 108, 108, 108, 112, 116, 116,
        116, 121, 126, 126, 88, 86, 85, 85, 85, 83, 82, 82, 82, 83, 84, 84, 84,
        85, 85, 85, 85, 90, 96, 96, 96, 101, 108, 108, 108, 112, 116, 116, 116,
        121, 126, 126,
        /* Size 4x16 */
        63, 64, 68, 79, 64, 64, 68, 77, 64, 64, 68, 77, 64, 65, 67, 75, 64, 65,
        67, 75, 64, 67, 71, 78, 64, 67, 71, 78, 65, 68, 74, 81, 65, 68, 74, 81,
        68, 71, 79, 90, 68, 71, 79, 90, 70, 73, 86, 102, 70, 73, 86, 102, 77,
        78, 90, 108, 77, 78, 90, 108, 85, 84, 96, 116,
        /* Size 16x4 */
        63, 64, 64, 64, 64, 64, 64, 65, 65, 68, 68, 70, 70, 77, 77, 85, 64, 64,
        64, 65, 65, 67, 67, 68, 68, 71, 71, 73, 73, 78, 78, 84, 68, 68, 68, 67,
        67, 71, 71, 74, 74, 79, 79, 86, 86, 90, 90, 96, 79, 77, 77, 75, 75, 78,
        78, 81, 81, 90, 90, 102, 102, 108, 108, 116,
        /* Size 8x32 */
        64, 63, 63, 64, 64, 72, 72, 88, 63, 63, 63, 64, 64, 71, 71, 86, 63, 64,
        64, 65, 65, 71, 71, 85, 63, 64, 64, 65, 65, 71, 71, 85, 63, 64, 64, 65,
        65, 71, 71, 85, 63, 64, 64, 65, 65, 70, 70, 83, 63, 65, 65, 66, 66, 69,
        69, 82, 63, 65, 65, 66, 66, 69, 69, 82, 63, 65, 65, 66, 66, 69, 69, 82,
        63, 65, 65, 67, 67, 71, 71, 83, 64, 65, 65, 68, 68, 73, 73, 84, 64, 65,
        65, 68, 68, 73, 73, 84, 64, 65, 65, 68, 68, 73, 73, 84, 64, 65, 65, 69,
        69, 75, 75, 85, 64, 66, 66, 71, 71, 77, 77, 85, 64, 66, 66, 71, 71, 77,
        77, 85, 64, 66, 66, 71, 71, 77, 77, 85, 66, 67, 67, 72, 72, 81, 81, 90,
        68, 68, 68, 74, 74, 85, 85, 96, 68, 68, 68, 74, 74, 85, 85, 96, 68, 68,
        68, 74, 74, 85, 85, 96, 70, 69, 69, 75, 75, 90, 90, 101, 72, 69, 69, 77,
        77, 96, 96, 108, 72, 69, 69, 77, 77, 96, 96, 108, 72, 69, 69, 77, 77,
        96, 96, 108, 75, 72, 72, 79, 79, 98, 98, 112, 79, 75, 75, 81, 81, 101,
        101, 116, 79, 75, 75, 81, 81, 101, 101, 116, 79, 75, 75, 81, 81, 101,
        101, 116, 83, 79, 79, 84, 84, 104, 104, 121, 88, 82, 82, 86, 86, 107,
        107, 126, 88, 82, 82, 86, 86, 107, 107, 126,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 66,
        68, 68, 68, 70, 72, 72, 72, 75, 79, 79, 79, 83, 88, 88, 63, 63, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 67, 68, 68, 68, 69,
        69, 69, 69, 72, 75, 75, 75, 79, 82, 82, 63, 63, 64, 64, 64, 64, 65, 65,
        65, 65, 65, 65, 65, 65, 66, 66, 66, 67, 68, 68, 68, 69, 69, 69, 69, 72,
        75, 75, 75, 79, 82, 82, 64, 64, 65, 65, 65, 65, 66, 66, 66, 67, 68, 68,
        68, 69, 71, 71, 71, 72, 74, 74, 74, 75, 77, 77, 77, 79, 81, 81, 81, 84,
        86, 86, 64, 64, 65, 65, 65, 65, 66, 66, 66, 67, 68, 68, 68, 69, 71, 71,
        71, 72, 74, 74, 74, 75, 77, 77, 77, 79, 81, 81, 81, 84, 86, 86, 72, 71,
        71, 71, 71, 70, 69, 69, 69, 71, 73, 73, 73, 75, 77, 77, 77, 81, 85, 85,
        85, 90, 96, 96, 96, 98, 101, 101, 101, 104, 107, 107, 72, 71, 71, 71,
        71, 70, 69, 69, 69, 71, 73, 73, 73, 75, 77, 77, 77, 81, 85, 85, 85, 90,
        96, 96, 96, 98, 101, 101, 101, 104, 107, 107, 88, 86, 85, 85, 85, 83,
        82, 82, 82, 83, 84, 84, 84, 85, 85, 85, 85, 90, 96, 96, 96, 101, 108,
        108, 108, 112, 116, 116, 116, 121, 126, 126 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        64, 93, 93, 106,
#endif
        /* Size 4x4 */
        63, 69, 85, 94, 69, 78, 90, 93, 85, 90, 97, 99, 94, 93, 99, 109,
        /* Size 8x8 */
        63, 62, 65, 71, 79, 90, 96, 96, 62, 63, 66, 74, 82, 89, 93, 92, 65, 66,
        71, 79, 85, 90, 93, 91, 71, 74, 79, 86, 91, 94, 94, 92, 79, 82, 85, 91,
        95, 97, 97, 95, 90, 89, 90, 94, 97, 101, 103, 103, 96, 93, 93, 94, 97,
        103, 107, 108, 96, 92, 91, 92, 95, 103, 108, 112,
        /* Size 16x16 */
        64, 62, 62, 61, 61, 67, 67, 73, 73, 83, 83, 98, 98, 97, 97, 98, 62, 63,
        63, 62, 62, 69, 69, 77, 77, 85, 85, 95, 95, 94, 94, 94, 62, 63, 63, 62,
        62, 69, 69, 77, 77, 85, 85, 95, 95, 94, 94, 94, 61, 62, 62, 64, 64, 71,
        71, 80, 80, 85, 85, 93, 93, 91, 91, 90, 61, 62, 62, 64, 64, 71, 71, 80,
        80, 85, 85, 93, 93, 91, 91, 90, 67, 69, 69, 71, 71, 78, 78, 87, 87, 90,
        90, 95, 95, 93, 93, 91, 67, 69, 69, 71, 71, 78, 78, 87, 87, 90, 90, 95,
        95, 93, 93, 91, 73, 77, 77, 80, 80, 87, 87, 95, 95, 95, 95, 96, 96, 93,
        93, 91, 73, 77, 77, 80, 80, 87, 87, 95, 95, 95, 95, 96, 96, 93, 93, 91,
        83, 85, 85, 85, 85, 90, 90, 95, 95, 97, 97, 101, 101, 99, 99, 98, 83,
        85, 85, 85, 85, 90, 90, 95, 95, 97, 97, 101, 101, 99, 99, 98, 98, 95,
        95, 93, 93, 95, 95, 96, 96, 101, 101, 106, 106, 106, 106, 107, 98, 95,
        95, 93, 93, 95, 95, 96, 96, 101, 101, 106, 106, 106, 106, 107, 97, 94,
        94, 91, 91, 93, 93, 93, 93, 99, 99, 106, 106, 109, 109, 111, 97, 94, 94,
        91, 91, 93, 93, 93, 93, 99, 99, 106, 106, 109, 109, 111, 98, 94, 94, 90,
        90, 91, 91, 91, 91, 98, 98, 107, 107, 111, 111, 117,
        /* Size 32x32 */
        64, 63, 62, 62, 62, 62, 61, 61, 61, 64, 67, 67, 67, 70, 73, 73, 73, 78,
        83, 83, 83, 90, 98, 98, 98, 98, 97, 97, 97, 98, 98, 98, 63, 63, 63, 63,
        63, 62, 62, 62, 62, 65, 68, 68, 68, 71, 75, 75, 75, 79, 84, 84, 84, 90,
        97, 97, 97, 96, 96, 96, 96, 96, 96, 96, 62, 63, 63, 63, 63, 63, 62, 62,
        62, 66, 69, 69, 69, 73, 77, 77, 77, 81, 85, 85, 85, 90, 95, 95, 95, 95,
        94, 94, 94, 94, 94, 94, 62, 63, 63, 63, 63, 63, 62, 62, 62, 66, 69, 69,
        69, 73, 77, 77, 77, 81, 85, 85, 85, 90, 95, 95, 95, 95, 94, 94, 94, 94,
        94, 94, 62, 63, 63, 63, 63, 63, 62, 62, 62, 66, 69, 69, 69, 73, 77, 77,
        77, 81, 85, 85, 85, 90, 95, 95, 95, 95, 94, 94, 94, 94, 94, 94, 62, 62,
        63, 63, 63, 63, 63, 63, 63, 66, 70, 70, 70, 74, 78, 78, 78, 82, 85, 85,
        85, 89, 94, 94, 94, 93, 93, 93, 93, 92, 92, 92, 61, 62, 62, 62, 62, 63,
        64, 64, 64, 67, 71, 71, 71, 75, 80, 80, 80, 83, 85, 85, 85, 89, 93, 93,
        93, 92, 91, 91, 91, 91, 90, 90, 61, 62, 62, 62, 62, 63, 64, 64, 64, 67,
        71, 71, 71, 75, 80, 80, 80, 83, 85, 85, 85, 89, 93, 93, 93, 92, 91, 91,
        91, 91, 90, 90, 61, 62, 62, 62, 62, 63, 64, 64, 64, 67, 71, 71, 71, 75,
        80, 80, 80, 83, 85, 85, 85, 89, 93, 93, 93, 92, 91, 91, 91, 91, 90, 90,
        64, 65, 66, 66, 66, 66, 67, 67, 67, 71, 74, 74, 74, 79, 83, 83, 83, 85,
        87, 87, 87, 90, 94, 94, 94, 93, 92, 92, 92, 91, 90, 90, 67, 68, 69, 69,
        69, 70, 71, 71, 71, 74, 78, 78, 78, 82, 87, 87, 87, 88, 90, 90, 90, 92,
        95, 95, 95, 94, 93, 93, 93, 92, 91, 91, 67, 68, 69, 69, 69, 70, 71, 71,
        71, 74, 78, 78, 78, 82, 87, 87, 87, 88, 90, 90, 90, 92, 95, 95, 95, 94,
        93, 93, 93, 92, 91, 91, 67, 68, 69, 69, 69, 70, 71, 71, 71, 74, 78, 78,
        78, 82, 87, 87, 87, 88, 90, 90, 90, 92, 95, 95, 95, 94, 93, 93, 93, 92,
        91, 91, 70, 71, 73, 73, 73, 74, 75, 75, 75, 79, 82, 82, 82, 86, 91, 91,
        91, 91, 92, 92, 92, 94, 95, 95, 95, 94, 93, 93, 93, 92, 91, 91, 73, 75,
        77, 77, 77, 78, 80, 80, 80, 83, 87, 87, 87, 91, 95, 95, 95, 95, 95, 95,
        95, 95, 96, 96, 96, 94, 93, 93, 93, 92, 91, 91, 73, 75, 77, 77, 77, 78,
        80, 80, 80, 83, 87, 87, 87, 91, 95, 95, 95, 95, 95, 95, 95, 95, 96, 96,
        96, 94, 93, 93, 93, 92, 91, 91, 73, 75, 77, 77, 77, 78, 80, 80, 80, 83,
        87, 87, 87, 91, 95, 95, 95, 95, 95, 95, 95, 95, 96, 96, 96, 94, 93, 93,
        93, 92, 91, 91, 78, 79, 81, 81, 81, 82, 83, 83, 83, 85, 88, 88, 88, 91,
        95, 95, 95, 95, 96, 96, 96, 97, 98, 98, 98, 97, 96, 96, 96, 95, 94, 94,
        83, 84, 85, 85, 85, 85, 85, 85, 85, 87, 90, 90, 90, 92, 95, 95, 95, 96,
        97, 97, 97, 99, 101, 101, 101, 100, 99, 99, 99, 99, 98, 98, 83, 84, 85,
        85, 85, 85, 85, 85, 85, 87, 90, 90, 90, 92, 95, 95, 95, 96, 97, 97, 97,
        99, 101, 101, 101, 100, 99, 99, 99, 99, 98, 98, 83, 84, 85, 85, 85, 85,
        85, 85, 85, 87, 90, 90, 90, 92, 95, 95, 95, 96, 97, 97, 97, 99, 101,
        101, 101, 100, 99, 99, 99, 99, 98, 98, 90, 90, 90, 90, 90, 89, 89, 89,
        89, 90, 92, 92, 92, 94, 95, 95, 95, 97, 99, 99, 99, 101, 103, 103, 103,
        103, 103, 103, 103, 103, 103, 103, 98, 97, 95, 95, 95, 94, 93, 93, 93,
        94, 95, 95, 95, 95, 96, 96, 96, 98, 101, 101, 101, 103, 106, 106, 106,
        106, 106, 106, 106, 107, 107, 107, 98, 97, 95, 95, 95, 94, 93, 93, 93,
        94, 95, 95, 95, 95, 96, 96, 96, 98, 101, 101, 101, 103, 106, 106, 106,
        106, 106, 106, 106, 107, 107, 107, 98, 97, 95, 95, 95, 94, 93, 93, 93,
        94, 95, 95, 95, 95, 96, 96, 96, 98, 101, 101, 101, 103, 106, 106, 106,
        106, 106, 106, 106, 107, 107, 107, 98, 96, 95, 95, 95, 93, 92, 92, 92,
        93, 94, 94, 94, 94, 94, 94, 94, 97, 100, 100, 100, 103, 106, 106, 106,
        107, 107, 107, 107, 108, 109, 109, 97, 96, 94, 94, 94, 93, 91, 91, 91,
        92, 93, 93, 93, 93, 93, 93, 93, 96, 99, 99, 99, 103, 106, 106, 106, 107,
        109, 109, 109, 110, 111, 111, 97, 96, 94, 94, 94, 93, 91, 91, 91, 92,
        93, 93, 93, 93, 93, 93, 93, 96, 99, 99, 99, 103, 106, 106, 106, 107,
        109, 109, 109, 110, 111, 111, 97, 96, 94, 94, 94, 93, 91, 91, 91, 92,
        93, 93, 93, 93, 93, 93, 93, 96, 99, 99, 99, 103, 106, 106, 106, 107,
        109, 109, 109, 110, 111, 111, 98, 96, 94, 94, 94, 92, 91, 91, 91, 91,
        92, 92, 92, 92, 92, 92, 92, 95, 99, 99, 99, 103, 107, 107, 107, 108,
        110, 110, 110, 112, 114, 114, 98, 96, 94, 94, 94, 92, 90, 90, 90, 90,
        91, 91, 91, 91, 91, 91, 91, 94, 98, 98, 98, 103, 107, 107, 107, 109,
        111, 111, 111, 114, 117, 117, 98, 96, 94, 94, 94, 92, 90, 90, 90, 90,
        91, 91, 91, 91, 91, 91, 91, 94, 98, 98, 98, 103, 107, 107, 107, 109,
        111, 111, 111, 114, 117, 117,
        /* Size 4x8 */
        63, 68, 84, 96, 63, 70, 85, 93, 66, 75, 88, 92, 73, 82, 92, 93, 81, 88,
        96, 96, 90, 92, 99, 103, 95, 94, 100, 108, 94, 92, 99, 110,
        /* Size 8x4 */
        63, 63, 66, 73, 81, 90, 95, 94, 68, 70, 75, 82, 88, 92, 94, 92, 84, 85,
        88, 92, 96, 99, 100, 99, 96, 93, 92, 93, 96, 103, 108, 110,
        /* Size 8x16 */
        64, 62, 62, 74, 74, 96, 96, 98, 63, 63, 63, 77, 77, 94, 94, 94, 63, 63,
        63, 77, 77, 94, 94, 94, 61, 64, 64, 80, 80, 92, 92, 91, 61, 64, 64, 80,
        80, 92, 92, 91, 67, 72, 72, 86, 86, 94, 94, 92, 67, 72, 72, 86, 86, 94,
        94, 92, 74, 80, 80, 94, 94, 95, 95, 91, 74, 80, 80, 94, 94, 95, 95, 91,
        84, 86, 86, 95, 95, 100, 100, 99, 84, 86, 86, 95, 95, 100, 100, 99, 98,
        93, 93, 96, 96, 106, 106, 107, 98, 93, 93, 96, 96, 106, 106, 107, 97,
        92, 92, 94, 94, 106, 106, 112, 97, 92, 92, 94, 94, 106, 106, 112, 98,
        91, 91, 92, 92, 107, 107, 117,
        /* Size 16x8 */
        64, 63, 63, 61, 61, 67, 67, 74, 74, 84, 84, 98, 98, 97, 97, 98, 62, 63,
        63, 64, 64, 72, 72, 80, 80, 86, 86, 93, 93, 92, 92, 91, 62, 63, 63, 64,
        64, 72, 72, 80, 80, 86, 86, 93, 93, 92, 92, 91, 74, 77, 77, 80, 80, 86,
        86, 94, 94, 95, 95, 96, 96, 94, 94, 92, 74, 77, 77, 80, 80, 86, 86, 94,
        94, 95, 95, 96, 96, 94, 94, 92, 96, 94, 94, 92, 92, 94, 94, 95, 95, 100,
        100, 106, 106, 106, 106, 107, 96, 94, 94, 92, 92, 94, 94, 95, 95, 100,
        100, 106, 106, 106, 106, 107, 98, 94, 94, 91, 91, 92, 92, 91, 91, 99,
        99, 107, 107, 112, 112, 117,
        /* Size 16x32 */
        64, 63, 62, 62, 62, 67, 74, 74, 74, 84, 96, 96, 96, 97, 98, 98, 63, 63,
        62, 62, 62, 68, 75, 75, 75, 84, 95, 95, 95, 96, 96, 96, 63, 63, 63, 63,
        63, 69, 77, 77, 77, 85, 94, 94, 94, 94, 94, 94, 63, 63, 63, 63, 63, 69,
        77, 77, 77, 85, 94, 94, 94, 94, 94, 94, 63, 63, 63, 63, 63, 69, 77, 77,
        77, 85, 94, 94, 94, 94, 94, 94, 62, 63, 64, 64, 64, 70, 78, 78, 78, 85,
        93, 93, 93, 93, 92, 92, 61, 63, 64, 64, 64, 71, 80, 80, 80, 85, 92, 92,
        92, 91, 91, 91, 61, 63, 64, 64, 64, 71, 80, 80, 80, 85, 92, 92, 92, 91,
        91, 91, 61, 63, 64, 64, 64, 71, 80, 80, 80, 85, 92, 92, 92, 91, 91, 91,
        64, 66, 68, 68, 68, 75, 83, 83, 83, 88, 93, 93, 93, 92, 91, 91, 67, 69,
        72, 72, 72, 78, 86, 86, 86, 90, 94, 94, 94, 93, 92, 92, 67, 69, 72, 72,
        72, 78, 86, 86, 86, 90, 94, 94, 94, 93, 92, 92, 67, 69, 72, 72, 72, 78,
        86, 86, 86, 90, 94, 94, 94, 93, 92, 92, 71, 73, 76, 76, 76, 82, 90, 90,
        90, 92, 95, 95, 95, 93, 91, 91, 74, 77, 80, 80, 80, 87, 94, 94, 94, 95,
        95, 95, 95, 93, 91, 91, 74, 77, 80, 80, 80, 87, 94, 94, 94, 95, 95, 95,
        95, 93, 91, 91, 74, 77, 80, 80, 80, 87, 94, 94, 94, 95, 95, 95, 95, 93,
        91, 91, 79, 81, 83, 83, 83, 88, 95, 95, 95, 96, 98, 98, 98, 96, 95, 95,
        84, 85, 86, 86, 86, 90, 95, 95, 95, 97, 100, 100, 100, 100, 99, 99, 84,
        85, 86, 86, 86, 90, 95, 95, 95, 97, 100, 100, 100, 100, 99, 99, 84, 85,
        86, 86, 86, 90, 95, 95, 95, 97, 100, 100, 100, 100, 99, 99, 90, 90, 89,
        89, 89, 92, 95, 95, 95, 99, 103, 103, 103, 103, 103, 103, 98, 96, 93,
        93, 93, 95, 96, 96, 96, 101, 106, 106, 106, 107, 107, 107, 98, 96, 93,
        93, 93, 95, 96, 96, 96, 101, 106, 106, 106, 107, 107, 107, 98, 96, 93,
        93, 93, 95, 96, 96, 96, 101, 106, 106, 106, 107, 107, 107, 97, 95, 93,
        93, 93, 94, 95, 95, 95, 100, 106, 106, 106, 108, 109, 109, 97, 94, 92,
        92, 92, 93, 94, 94, 94, 100, 106, 106, 106, 109, 112, 112, 97, 94, 92,
        92, 92, 93, 94, 94, 94, 100, 106, 106, 106, 109, 112, 112, 97, 94, 92,
        92, 92, 93, 94, 94, 94, 100, 106, 106, 106, 109, 112, 112, 97, 94, 91,
        91, 91, 92, 93, 93, 93, 99, 106, 106, 106, 110, 114, 114, 98, 94, 91,
        91, 91, 91, 92, 92, 92, 99, 107, 107, 107, 112, 117, 117, 98, 94, 91,
        91, 91, 91, 92, 92, 92, 99, 107, 107, 107, 112, 117, 117,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 62, 61, 61, 61, 64, 67, 67, 67, 71, 74, 74, 74, 79,
        84, 84, 84, 90, 98, 98, 98, 97, 97, 97, 97, 97, 98, 98, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 66, 69, 69, 69, 73, 77, 77, 77, 81, 85, 85, 85, 90,
        96, 96, 96, 95, 94, 94, 94, 94, 94, 94, 62, 62, 63, 63, 63, 64, 64, 64,
        64, 68, 72, 72, 72, 76, 80, 80, 80, 83, 86, 86, 86, 89, 93, 93, 93, 93,
        92, 92, 92, 91, 91, 91, 62, 62, 63, 63, 63, 64, 64, 64, 64, 68, 72, 72,
        72, 76, 80, 80, 80, 83, 86, 86, 86, 89, 93, 93, 93, 93, 92, 92, 92, 91,
        91, 91, 62, 62, 63, 63, 63, 64, 64, 64, 64, 68, 72, 72, 72, 76, 80, 80,
        80, 83, 86, 86, 86, 89, 93, 93, 93, 93, 92, 92, 92, 91, 91, 91, 67, 68,
        69, 69, 69, 70, 71, 71, 71, 75, 78, 78, 78, 82, 87, 87, 87, 88, 90, 90,
        90, 92, 95, 95, 95, 94, 93, 93, 93, 92, 91, 91, 74, 75, 77, 77, 77, 78,
        80, 80, 80, 83, 86, 86, 86, 90, 94, 94, 94, 95, 95, 95, 95, 95, 96, 96,
        96, 95, 94, 94, 94, 93, 92, 92, 74, 75, 77, 77, 77, 78, 80, 80, 80, 83,
        86, 86, 86, 90, 94, 94, 94, 95, 95, 95, 95, 95, 96, 96, 96, 95, 94, 94,
        94, 93, 92, 92, 74, 75, 77, 77, 77, 78, 80, 80, 80, 83, 86, 86, 86, 90,
        94, 94, 94, 95, 95, 95, 95, 95, 96, 96, 96, 95, 94, 94, 94, 93, 92, 92,
        84, 84, 85, 85, 85, 85, 85, 85, 85, 88, 90, 90, 90, 92, 95, 95, 95, 96,
        97, 97, 97, 99, 101, 101, 101, 100, 100, 100, 100, 99, 99, 99, 96, 95,
        94, 94, 94, 93, 92, 92, 92, 93, 94, 94, 94, 95, 95, 95, 95, 98, 100,
        100, 100, 103, 106, 106, 106, 106, 106, 106, 106, 106, 107, 107, 96, 95,
        94, 94, 94, 93, 92, 92, 92, 93, 94, 94, 94, 95, 95, 95, 95, 98, 100,
        100, 100, 103, 106, 106, 106, 106, 106, 106, 106, 106, 107, 107, 96, 95,
        94, 94, 94, 93, 92, 92, 92, 93, 94, 94, 94, 95, 95, 95, 95, 98, 100,
        100, 100, 103, 106, 106, 106, 106, 106, 106, 106, 106, 107, 107, 97, 96,
        94, 94, 94, 93, 91, 91, 91, 92, 93, 93, 93, 93, 93, 93, 93, 96, 100,
        100, 100, 103, 107, 107, 107, 108, 109, 109, 109, 110, 112, 112, 98, 96,
        94, 94, 94, 92, 91, 91, 91, 91, 92, 92, 92, 91, 91, 91, 91, 95, 99, 99,
        99, 103, 107, 107, 107, 109, 112, 112, 112, 114, 117, 117, 98, 96, 94,
        94, 94, 92, 91, 91, 91, 91, 92, 92, 92, 91, 91, 91, 91, 95, 99, 99, 99,
        103, 107, 107, 107, 109, 112, 112, 112, 114, 117, 117,
        /* Size 4x16 */
        63, 67, 84, 97, 63, 69, 85, 94, 63, 69, 85, 94, 63, 71, 85, 91, 63, 71,
        85, 91, 69, 78, 90, 93, 69, 78, 90, 93, 77, 87, 95, 93, 77, 87, 95, 93,
        85, 90, 97, 100, 85, 90, 97, 100, 96, 95, 101, 107, 96, 95, 101, 107,
        94, 93, 100, 109, 94, 93, 100, 109, 94, 91, 99, 112,
        /* Size 16x4 */
        63, 63, 63, 63, 63, 69, 69, 77, 77, 85, 85, 96, 96, 94, 94, 94, 67, 69,
        69, 71, 71, 78, 78, 87, 87, 90, 90, 95, 95, 93, 93, 91, 84, 85, 85, 85,
        85, 90, 90, 95, 95, 97, 97, 101, 101, 100, 100, 99, 97, 94, 94, 91, 91,
        93, 93, 93, 93, 100, 100, 107, 107, 109, 109, 112,
        /* Size 8x32 */
        64, 62, 62, 74, 74, 96, 96, 98, 63, 62, 62, 75, 75, 95, 95, 96, 63, 63,
        63, 77, 77, 94, 94, 94, 63, 63, 63, 77, 77, 94, 94, 94, 63, 63, 63, 77,
        77, 94, 94, 94, 62, 64, 64, 78, 78, 93, 93, 92, 61, 64, 64, 80, 80, 92,
        92, 91, 61, 64, 64, 80, 80, 92, 92, 91, 61, 64, 64, 80, 80, 92, 92, 91,
        64, 68, 68, 83, 83, 93, 93, 91, 67, 72, 72, 86, 86, 94, 94, 92, 67, 72,
        72, 86, 86, 94, 94, 92, 67, 72, 72, 86, 86, 94, 94, 92, 71, 76, 76, 90,
        90, 95, 95, 91, 74, 80, 80, 94, 94, 95, 95, 91, 74, 80, 80, 94, 94, 95,
        95, 91, 74, 80, 80, 94, 94, 95, 95, 91, 79, 83, 83, 95, 95, 98, 98, 95,
        84, 86, 86, 95, 95, 100, 100, 99, 84, 86, 86, 95, 95, 100, 100, 99, 84,
        86, 86, 95, 95, 100, 100, 99, 90, 89, 89, 95, 95, 103, 103, 103, 98, 93,
        93, 96, 96, 106, 106, 107, 98, 93, 93, 96, 96, 106, 106, 107, 98, 93,
        93, 96, 96, 106, 106, 107, 97, 93, 93, 95, 95, 106, 106, 109, 97, 92,
        92, 94, 94, 106, 106, 112, 97, 92, 92, 94, 94, 106, 106, 112, 97, 92,
        92, 94, 94, 106, 106, 112, 97, 91, 91, 93, 93, 106, 106, 114, 98, 91,
        91, 92, 92, 107, 107, 117, 98, 91, 91, 92, 92, 107, 107, 117,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 62, 61, 61, 61, 64, 67, 67, 67, 71, 74, 74, 74, 79,
        84, 84, 84, 90, 98, 98, 98, 97, 97, 97, 97, 97, 98, 98, 62, 62, 63, 63,
        63, 64, 64, 64, 64, 68, 72, 72, 72, 76, 80, 80, 80, 83, 86, 86, 86, 89,
        93, 93, 93, 93, 92, 92, 92, 91, 91, 91, 62, 62, 63, 63, 63, 64, 64, 64,
        64, 68, 72, 72, 72, 76, 80, 80, 80, 83, 86, 86, 86, 89, 93, 93, 93, 93,
        92, 92, 92, 91, 91, 91, 74, 75, 77, 77, 77, 78, 80, 80, 80, 83, 86, 86,
        86, 90, 94, 94, 94, 95, 95, 95, 95, 95, 96, 96, 96, 95, 94, 94, 94, 93,
        92, 92, 74, 75, 77, 77, 77, 78, 80, 80, 80, 83, 86, 86, 86, 90, 94, 94,
        94, 95, 95, 95, 95, 95, 96, 96, 96, 95, 94, 94, 94, 93, 92, 92, 96, 95,
        94, 94, 94, 93, 92, 92, 92, 93, 94, 94, 94, 95, 95, 95, 95, 98, 100,
        100, 100, 103, 106, 106, 106, 106, 106, 106, 106, 106, 107, 107, 96, 95,
        94, 94, 94, 93, 92, 92, 92, 93, 94, 94, 94, 95, 95, 95, 95, 98, 100,
        100, 100, 103, 106, 106, 106, 106, 106, 106, 106, 106, 107, 107, 98, 96,
        94, 94, 94, 92, 91, 91, 91, 91, 92, 92, 92, 91, 91, 91, 91, 95, 99, 99,
        99, 103, 107, 107, 107, 109, 112, 112, 112, 114, 117, 117 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        64, 67, 67, 79,
#endif
        /* Size 4x4 */
        64, 64, 65, 70, 64, 65, 66, 70, 65, 66, 71, 76, 70, 70, 76, 92,
        /* Size 8x8 */
        63, 63, 63, 64, 64, 65, 68, 71, 63, 64, 64, 64, 65, 66, 68, 70, 63, 64,
        65, 65, 65, 66, 67, 69, 64, 64, 65, 67, 68, 69, 71, 73, 64, 65, 65, 68,
        70, 71, 73, 76, 65, 66, 66, 69, 71, 73, 76, 80, 68, 68, 67, 71, 73, 76,
        79, 85, 71, 70, 69, 73, 76, 80, 85, 97,
        /* Size 16x16 */
        64, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 67, 68, 69, 72, 72, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 67, 68, 69, 71, 71, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 65, 65, 67, 68, 69, 70, 70, 63, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 67, 68, 68, 70, 70, 63, 64, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 67, 67, 68, 69, 69, 63, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 67, 67, 68, 69, 69, 63, 64, 64, 65, 65, 65, 66, 66, 67, 67, 67, 69,
        70, 71, 72, 72, 63, 64, 64, 65, 65, 65, 66, 67, 67, 68, 68, 70, 71, 72,
        73, 73, 64, 64, 64, 65, 65, 65, 67, 67, 68, 69, 69, 71, 72, 73, 74, 74,
        64, 64, 65, 65, 65, 65, 67, 68, 69, 71, 71, 73, 74, 75, 77, 77, 64, 64,
        65, 65, 65, 65, 67, 68, 69, 71, 71, 73, 74, 75, 77, 77, 67, 67, 67, 67,
        67, 67, 69, 70, 71, 73, 73, 77, 78, 80, 84, 84, 68, 68, 68, 68, 67, 67,
        70, 71, 72, 74, 74, 78, 79, 82, 85, 85, 69, 69, 69, 68, 68, 68, 71, 72,
        73, 75, 75, 80, 82, 85, 90, 90, 72, 71, 70, 70, 69, 69, 72, 73, 74, 77,
        77, 84, 85, 90, 97, 97, 72, 71, 70, 70, 69, 69, 72, 73, 74, 77, 77, 84,
        85, 90, 97, 97,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64,
        64, 64, 64, 65, 67, 68, 68, 68, 69, 71, 72, 72, 72, 74, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65,
        67, 68, 68, 68, 69, 70, 71, 71, 71, 74, 63, 63, 63, 63, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 66, 67, 68, 68, 68,
        69, 70, 71, 71, 71, 73, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 65, 65, 65, 65, 66, 67, 68, 68, 68, 69, 70, 70, 70,
        70, 73, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 65, 65, 65, 65, 66, 67, 68, 68, 68, 69, 70, 70, 70, 70, 73, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65,
        65, 66, 67, 68, 68, 68, 69, 70, 70, 70, 70, 73, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 67, 68,
        68, 68, 68, 69, 70, 70, 70, 72, 63, 63, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 67, 67, 67, 67, 68, 69,
        69, 69, 69, 71, 63, 63, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 66, 67, 67, 67, 67, 68, 68, 69, 69, 69, 71,
        63, 63, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 66, 67, 67, 67, 67, 68, 68, 69, 69, 69, 71, 63, 63, 64, 64,
        64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66,
        67, 67, 67, 67, 68, 68, 69, 69, 69, 71, 63, 63, 64, 64, 64, 64, 64, 65,
        65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 66, 67, 68, 69, 69, 69,
        69, 70, 70, 70, 70, 73, 63, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 66,
        66, 66, 66, 66, 67, 67, 67, 67, 67, 68, 69, 70, 70, 70, 71, 72, 72, 72,
        72, 74, 63, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 66, 66, 67, 67, 67,
        67, 68, 68, 68, 68, 69, 70, 71, 71, 71, 72, 72, 73, 73, 73, 75, 63, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 65, 66, 66, 67, 67, 67, 67, 68, 68, 68,
        68, 69, 70, 71, 71, 71, 72, 72, 73, 73, 73, 75, 63, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 66, 66, 67, 67, 67, 67, 68, 68, 68, 68, 69, 70, 71,
        71, 71, 72, 72, 73, 73, 73, 75, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 66, 67, 67, 67, 67, 68, 69, 69, 69, 69, 70, 71, 72, 72, 72, 73, 74,
        74, 74, 74, 76, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 66, 67, 68,
        68, 68, 69, 70, 70, 70, 70, 71, 72, 73, 73, 73, 74, 75, 76, 76, 76, 78,
        64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 66, 67, 68, 68, 68, 69, 70,
        71, 71, 71, 72, 73, 74, 74, 74, 75, 76, 77, 77, 77, 78, 64, 64, 64, 65,
        65, 65, 65, 65, 65, 65, 65, 66, 67, 68, 68, 68, 69, 70, 71, 71, 71, 72,
        73, 74, 74, 74, 75, 76, 77, 77, 77, 78, 64, 64, 64, 65, 65, 65, 65, 65,
        65, 65, 65, 66, 67, 68, 68, 68, 69, 70, 71, 71, 71, 72, 73, 74, 74, 74,
        75, 76, 77, 77, 77, 78, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 66, 67,
        68, 69, 69, 69, 70, 71, 72, 72, 72, 73, 75, 76, 76, 76, 77, 79, 80, 80,
        80, 82, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 68, 69, 70, 70, 70,
        71, 72, 73, 73, 73, 75, 77, 78, 78, 78, 80, 82, 84, 84, 84, 85, 68, 68,
        68, 68, 68, 68, 68, 67, 67, 67, 67, 69, 70, 71, 71, 71, 72, 73, 74, 74,
        74, 76, 78, 79, 79, 79, 82, 84, 85, 85, 85, 87, 68, 68, 68, 68, 68, 68,
        68, 67, 67, 67, 67, 69, 70, 71, 71, 71, 72, 73, 74, 74, 74, 76, 78, 79,
        79, 79, 82, 84, 85, 85, 85, 87, 68, 68, 68, 68, 68, 68, 68, 67, 67, 67,
        67, 69, 70, 71, 71, 71, 72, 73, 74, 74, 74, 76, 78, 79, 79, 79, 82, 84,
        85, 85, 85, 87, 69, 69, 69, 69, 69, 69, 68, 68, 68, 68, 68, 69, 71, 72,
        72, 72, 73, 74, 75, 75, 75, 77, 80, 82, 82, 82, 85, 88, 90, 90, 90, 91,
        71, 70, 70, 70, 70, 70, 69, 69, 68, 68, 68, 70, 72, 72, 72, 72, 74, 75,
        76, 76, 76, 79, 82, 84, 84, 84, 88, 92, 94, 94, 94, 96, 72, 71, 71, 70,
        70, 70, 70, 69, 69, 69, 69, 70, 72, 73, 73, 73, 74, 76, 77, 77, 77, 80,
        84, 85, 85, 85, 90, 94, 97, 97, 97, 99, 72, 71, 71, 70, 70, 70, 70, 69,
        69, 69, 69, 70, 72, 73, 73, 73, 74, 76, 77, 77, 77, 80, 84, 85, 85, 85,
        90, 94, 97, 97, 97, 99, 72, 71, 71, 70, 70, 70, 70, 69, 69, 69, 69, 70,
        72, 73, 73, 73, 74, 76, 77, 77, 77, 80, 84, 85, 85, 85, 90, 94, 97, 97,
        97, 99, 74, 74, 73, 73, 73, 73, 72, 71, 71, 71, 71, 73, 74, 75, 75, 75,
        76, 78, 78, 78, 78, 82, 85, 87, 87, 87, 91, 96, 99, 99, 99, 101,
        /* Size 4x8 */
        63, 63, 64, 71, 64, 64, 65, 71, 64, 65, 66, 69, 64, 65, 68, 73, 64, 66,
        70, 76, 66, 66, 72, 80, 68, 68, 74, 85, 71, 69, 77, 96,
        /* Size 8x4 */
        63, 64, 64, 64, 64, 66, 68, 71, 63, 64, 65, 65, 66, 66, 68, 69, 64, 65,
        66, 68, 70, 72, 74, 77, 71, 71, 69, 73, 76, 80, 85, 96,
        /* Size 8x16 */
        64, 63, 63, 63, 64, 64, 70, 72, 63, 64, 64, 64, 65, 65, 70, 71, 63, 64,
        64, 64, 65, 65, 70, 71, 63, 64, 64, 65, 65, 65, 69, 70, 63, 64, 65, 65,
        66, 66, 68, 69, 63, 64, 65, 65, 66, 66, 68, 69, 63, 65, 65, 66, 68, 68,
        71, 72, 64, 65, 65, 66, 68, 68, 72, 73, 64, 65, 65, 67, 69, 69, 73, 75,
        64, 65, 66, 68, 71, 71, 75, 77, 64, 65, 66, 68, 71, 71, 75, 77, 67, 67,
        67, 70, 73, 73, 81, 83, 68, 68, 68, 70, 74, 74, 83, 85, 69, 69, 68, 71,
        75, 75, 86, 89, 72, 70, 69, 72, 77, 77, 92, 96, 72, 70, 69, 72, 77, 77,
        92, 96,
        /* Size 16x8 */
        64, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 67, 68, 69, 72, 72, 63, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 65, 67, 68, 69, 70, 70, 63, 64, 64, 64,
        65, 65, 65, 65, 65, 66, 66, 67, 68, 68, 69, 69, 63, 64, 64, 65, 65, 65,
        66, 66, 67, 68, 68, 70, 70, 71, 72, 72, 64, 65, 65, 65, 66, 66, 68, 68,
        69, 71, 71, 73, 74, 75, 77, 77, 64, 65, 65, 65, 66, 66, 68, 68, 69, 71,
        71, 73, 74, 75, 77, 77, 70, 70, 70, 69, 68, 68, 71, 72, 73, 75, 75, 81,
        83, 86, 92, 92, 72, 71, 71, 70, 69, 69, 72, 73, 75, 77, 77, 83, 85, 89,
        96, 96,
        /* Size 16x32 */
        64, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 67, 70, 72, 72, 72, 63, 63,
        63, 63, 63, 63, 64, 64, 64, 64, 64, 67, 70, 71, 71, 71, 63, 63, 64, 64,
        64, 64, 64, 65, 65, 65, 65, 67, 70, 71, 71, 71, 63, 64, 64, 64, 64, 64,
        64, 65, 65, 65, 65, 67, 70, 71, 71, 71, 63, 64, 64, 64, 64, 64, 64, 65,
        65, 65, 65, 67, 70, 71, 71, 71, 63, 64, 64, 64, 64, 64, 64, 65, 65, 65,
        65, 67, 70, 71, 71, 71, 63, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 67,
        69, 70, 70, 70, 63, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 67, 69, 70,
        70, 70, 63, 64, 64, 65, 65, 65, 65, 65, 66, 66, 66, 67, 68, 69, 69, 69,
        63, 64, 64, 65, 65, 65, 65, 65, 66, 66, 66, 67, 68, 69, 69, 69, 63, 64,
        64, 65, 65, 65, 65, 65, 66, 66, 66, 67, 68, 69, 69, 69, 63, 64, 65, 65,
        65, 65, 66, 66, 67, 67, 67, 68, 70, 71, 71, 71, 63, 64, 65, 65, 65, 65,
        66, 67, 68, 68, 68, 69, 71, 72, 72, 72, 64, 64, 65, 65, 65, 65, 66, 68,
        68, 68, 68, 70, 72, 73, 73, 73, 64, 64, 65, 65, 65, 65, 66, 68, 68, 68,
        68, 70, 72, 73, 73, 73, 64, 64, 65, 65, 65, 65, 66, 68, 68, 68, 68, 70,
        72, 73, 73, 73, 64, 64, 65, 65, 65, 65, 67, 68, 69, 69, 69, 71, 73, 75,
        75, 75, 64, 64, 65, 66, 66, 66, 67, 69, 70, 70, 70, 72, 75, 76, 76, 76,
        64, 64, 65, 66, 66, 66, 68, 70, 71, 71, 71, 73, 75, 77, 77, 77, 64, 64,
        65, 66, 66, 66, 68, 70, 71, 71, 71, 73, 75, 77, 77, 77, 64, 64, 65, 66,
        66, 66, 68, 70, 71, 71, 71, 73, 75, 77, 77, 77, 65, 66, 66, 66, 66, 66,
        69, 71, 72, 72, 72, 75, 78, 80, 80, 80, 67, 67, 67, 67, 67, 67, 70, 72,
        73, 73, 73, 77, 81, 83, 83, 83, 68, 68, 68, 68, 68, 68, 70, 72, 74, 74,
        74, 78, 83, 85, 85, 85, 68, 68, 68, 68, 68, 68, 70, 72, 74, 74, 74, 78,
        83, 85, 85, 85, 68, 68, 68, 68, 68, 68, 70, 72, 74, 74, 74, 78, 83, 85,
        85, 85, 69, 69, 69, 68, 68, 68, 71, 74, 75, 75, 75, 80, 86, 89, 89, 89,
        71, 70, 69, 69, 69, 69, 72, 75, 76, 76, 76, 82, 90, 94, 94, 94, 72, 71,
        70, 69, 69, 69, 72, 75, 77, 77, 77, 84, 92, 96, 96, 96, 72, 71, 70, 69,
        69, 69, 72, 75, 77, 77, 77, 84, 92, 96, 96, 96, 72, 71, 70, 69, 69, 69,
        72, 75, 77, 77, 77, 84, 92, 96, 96, 96, 74, 73, 72, 72, 72, 72, 74, 77,
        79, 79, 79, 85, 93, 98, 98, 98,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 65, 67, 68, 68, 68, 69, 71, 72, 72, 72, 74, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 66,
        67, 68, 68, 68, 69, 70, 71, 71, 71, 73, 63, 63, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 67, 68, 68, 68,
        69, 69, 70, 70, 70, 72, 63, 63, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 67, 68, 68, 68, 68, 69, 69, 69,
        69, 72, 63, 63, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 66, 66, 66, 66, 66, 67, 68, 68, 68, 68, 69, 69, 69, 69, 72, 63, 63,
        64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66,
        66, 66, 67, 68, 68, 68, 68, 69, 69, 69, 69, 72, 63, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 67, 67, 68, 68, 68, 69, 70, 70,
        70, 70, 71, 72, 72, 72, 72, 74, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 66, 67, 68, 68, 68, 68, 69, 70, 70, 70, 71, 72, 72, 72, 72, 74, 75,
        75, 75, 75, 77, 64, 64, 65, 65, 65, 65, 65, 65, 66, 66, 66, 67, 68, 68,
        68, 68, 69, 70, 71, 71, 71, 72, 73, 74, 74, 74, 75, 76, 77, 77, 77, 79,
        64, 64, 65, 65, 65, 65, 65, 65, 66, 66, 66, 67, 68, 68, 68, 68, 69, 70,
        71, 71, 71, 72, 73, 74, 74, 74, 75, 76, 77, 77, 77, 79, 64, 64, 65, 65,
        65, 65, 65, 65, 66, 66, 66, 67, 68, 68, 68, 68, 69, 70, 71, 71, 71, 72,
        73, 74, 74, 74, 75, 76, 77, 77, 77, 79, 67, 67, 67, 67, 67, 67, 67, 67,
        67, 67, 67, 68, 69, 70, 70, 70, 71, 72, 73, 73, 73, 75, 77, 78, 78, 78,
        80, 82, 84, 84, 84, 85, 70, 70, 70, 70, 70, 70, 69, 69, 68, 68, 68, 70,
        71, 72, 72, 72, 73, 75, 75, 75, 75, 78, 81, 83, 83, 83, 86, 90, 92, 92,
        92, 93, 72, 71, 71, 71, 71, 71, 70, 70, 69, 69, 69, 71, 72, 73, 73, 73,
        75, 76, 77, 77, 77, 80, 83, 85, 85, 85, 89, 94, 96, 96, 96, 98, 72, 71,
        71, 71, 71, 71, 70, 70, 69, 69, 69, 71, 72, 73, 73, 73, 75, 76, 77, 77,
        77, 80, 83, 85, 85, 85, 89, 94, 96, 96, 96, 98, 72, 71, 71, 71, 71, 71,
        70, 70, 69, 69, 69, 71, 72, 73, 73, 73, 75, 76, 77, 77, 77, 80, 83, 85,
        85, 85, 89, 94, 96, 96, 96, 98,
        /* Size 4x16 */
        63, 63, 64, 72, 63, 64, 65, 71, 64, 64, 65, 71, 64, 64, 65, 70, 64, 65,
        66, 69, 64, 65, 66, 69, 64, 65, 68, 72, 64, 65, 68, 73, 64, 65, 69, 75,
        64, 66, 71, 77, 64, 66, 71, 77, 67, 67, 73, 83, 68, 68, 74, 85, 69, 68,
        75, 89, 71, 69, 77, 96, 71, 69, 77, 96,
        /* Size 16x4 */
        63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 67, 68, 69, 71, 71, 63, 64,
        64, 64, 65, 65, 65, 65, 65, 66, 66, 67, 68, 68, 69, 69, 64, 65, 65, 65,
        66, 66, 68, 68, 69, 71, 71, 73, 74, 75, 77, 77, 72, 71, 71, 70, 69, 69,
        72, 73, 75, 77, 77, 83, 85, 89, 96, 96,
        /* Size 8x32 */
        64, 63, 63, 63, 64, 64, 70, 72, 63, 63, 63, 64, 64, 64, 70, 71, 63, 64,
        64, 64, 65, 65, 70, 71, 63, 64, 64, 64, 65, 65, 70, 71, 63, 64, 64, 64,
        65, 65, 70, 71, 63, 64, 64, 64, 65, 65, 70, 71, 63, 64, 64, 65, 65, 65,
        69, 70, 63, 64, 65, 65, 65, 65, 69, 70, 63, 64, 65, 65, 66, 66, 68, 69,
        63, 64, 65, 65, 66, 66, 68, 69, 63, 64, 65, 65, 66, 66, 68, 69, 63, 65,
        65, 66, 67, 67, 70, 71, 63, 65, 65, 66, 68, 68, 71, 72, 64, 65, 65, 66,
        68, 68, 72, 73, 64, 65, 65, 66, 68, 68, 72, 73, 64, 65, 65, 66, 68, 68,
        72, 73, 64, 65, 65, 67, 69, 69, 73, 75, 64, 65, 66, 67, 70, 70, 75, 76,
        64, 65, 66, 68, 71, 71, 75, 77, 64, 65, 66, 68, 71, 71, 75, 77, 64, 65,
        66, 68, 71, 71, 75, 77, 65, 66, 66, 69, 72, 72, 78, 80, 67, 67, 67, 70,
        73, 73, 81, 83, 68, 68, 68, 70, 74, 74, 83, 85, 68, 68, 68, 70, 74, 74,
        83, 85, 68, 68, 68, 70, 74, 74, 83, 85, 69, 69, 68, 71, 75, 75, 86, 89,
        71, 69, 69, 72, 76, 76, 90, 94, 72, 70, 69, 72, 77, 77, 92, 96, 72, 70,
        69, 72, 77, 77, 92, 96, 72, 70, 69, 72, 77, 77, 92, 96, 74, 72, 72, 74,
        79, 79, 93, 98,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 65, 67, 68, 68, 68, 69, 71, 72, 72, 72, 74, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66,
        67, 68, 68, 68, 69, 69, 70, 70, 70, 72, 63, 63, 64, 64, 64, 64, 64, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 67, 68, 68, 68,
        68, 69, 69, 69, 69, 72, 63, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 66,
        66, 66, 66, 66, 67, 67, 68, 68, 68, 69, 70, 70, 70, 70, 71, 72, 72, 72,
        72, 74, 64, 64, 65, 65, 65, 65, 65, 65, 66, 66, 66, 67, 68, 68, 68, 68,
        69, 70, 71, 71, 71, 72, 73, 74, 74, 74, 75, 76, 77, 77, 77, 79, 64, 64,
        65, 65, 65, 65, 65, 65, 66, 66, 66, 67, 68, 68, 68, 68, 69, 70, 71, 71,
        71, 72, 73, 74, 74, 74, 75, 76, 77, 77, 77, 79, 70, 70, 70, 70, 70, 70,
        69, 69, 68, 68, 68, 70, 71, 72, 72, 72, 73, 75, 75, 75, 75, 78, 81, 83,
        83, 83, 86, 90, 92, 92, 92, 93, 72, 71, 71, 71, 71, 71, 70, 70, 69, 69,
        69, 71, 72, 73, 73, 73, 75, 76, 77, 77, 77, 80, 83, 85, 85, 85, 89, 94,
        96, 96, 96, 98 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        63, 85, 85, 97,
#endif
        /* Size 4x4 */
        63, 65, 77, 93, 65, 69, 83, 92, 77, 83, 95, 95, 93, 92, 95, 104,
        /* Size 8x8 */
        63, 63, 61, 68, 73, 78, 84, 97, 63, 63, 62, 69, 75, 80, 85, 95, 61, 62,
        64, 71, 78, 82, 85, 93, 68, 69, 71, 78, 85, 88, 90, 95, 73, 75, 78, 85,
        92, 94, 94, 95, 78, 80, 82, 88, 94, 95, 96, 98, 84, 85, 85, 90, 94, 96,
        97, 101, 97, 95, 93, 95, 95, 98, 101, 106,
        /* Size 16x16 */
        64, 63, 62, 62, 61, 61, 66, 67, 69, 73, 73, 81, 83, 89, 98, 98, 63, 63,
        63, 62, 62, 62, 67, 69, 72, 76, 76, 83, 84, 89, 96, 96, 62, 63, 63, 63,
        62, 62, 68, 69, 72, 77, 77, 83, 85, 89, 95, 95, 62, 62, 63, 63, 63, 63,
        68, 70, 73, 78, 78, 83, 85, 88, 94, 94, 61, 62, 62, 63, 64, 64, 69, 71,
        74, 80, 80, 84, 85, 88, 93, 93, 61, 62, 62, 63, 64, 64, 69, 71, 74, 80,
        80, 84, 85, 88, 93, 93, 66, 67, 68, 68, 69, 69, 75, 77, 80, 85, 85, 88,
        89, 91, 94, 94, 67, 69, 69, 70, 71, 71, 77, 78, 81, 87, 87, 89, 90, 92,
        95, 95, 69, 72, 72, 73, 74, 74, 80, 81, 85, 90, 90, 91, 92, 93, 95, 95,
        73, 76, 77, 78, 80, 80, 85, 87, 90, 95, 95, 95, 95, 95, 96, 96, 73, 76,
        77, 78, 80, 80, 85, 87, 90, 95, 95, 95, 95, 95, 96, 96, 81, 83, 83, 83,
        84, 84, 88, 89, 91, 95, 95, 96, 97, 98, 100, 100, 83, 84, 85, 85, 85,
        85, 89, 90, 92, 95, 95, 97, 97, 99, 101, 101, 89, 89, 89, 88, 88, 88,
        91, 92, 93, 95, 95, 98, 99, 100, 103, 103, 98, 96, 95, 94, 93, 93, 94,
        95, 95, 96, 96, 100, 101, 103, 106, 106, 98, 96, 95, 94, 93, 93, 94, 95,
        95, 96, 96, 100, 101, 103, 106, 106,
        /* Size 32x32 */
        64, 63, 63, 62, 62, 62, 62, 61, 61, 61, 61, 63, 66, 67, 67, 67, 69, 72,
        73, 73, 73, 77, 81, 83, 83, 83, 89, 95, 98, 98, 98, 98, 63, 63, 63, 63,
        63, 63, 62, 62, 61, 61, 61, 64, 66, 68, 68, 68, 70, 73, 75, 75, 75, 78,
        82, 84, 84, 84, 89, 94, 97, 97, 97, 97, 63, 63, 63, 63, 63, 63, 62, 62,
        62, 62, 62, 65, 67, 69, 69, 69, 72, 75, 76, 76, 76, 79, 83, 84, 84, 84,
        89, 93, 96, 96, 96, 95, 62, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 65,
        68, 69, 69, 69, 72, 75, 77, 77, 77, 80, 83, 85, 85, 85, 89, 93, 95, 95,
        95, 95, 62, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 65, 68, 69, 69, 69,
        72, 75, 77, 77, 77, 80, 83, 85, 85, 85, 89, 93, 95, 95, 95, 95, 62, 63,
        63, 63, 63, 63, 63, 63, 62, 62, 62, 65, 68, 69, 69, 69, 72, 75, 77, 77,
        77, 80, 83, 85, 85, 85, 89, 93, 95, 95, 95, 95, 62, 62, 62, 63, 63, 63,
        63, 63, 63, 63, 63, 66, 68, 70, 70, 70, 73, 76, 78, 78, 78, 81, 83, 85,
        85, 85, 88, 92, 94, 94, 94, 94, 61, 62, 62, 63, 63, 63, 63, 63, 63, 63,
        63, 66, 69, 71, 71, 71, 74, 77, 79, 79, 79, 82, 84, 85, 85, 85, 88, 92,
        93, 93, 93, 93, 61, 61, 62, 62, 62, 62, 63, 63, 64, 64, 64, 66, 69, 71,
        71, 71, 74, 78, 80, 80, 80, 82, 84, 85, 85, 85, 88, 91, 93, 93, 93, 92,
        61, 61, 62, 62, 62, 62, 63, 63, 64, 64, 64, 66, 69, 71, 71, 71, 74, 78,
        80, 80, 80, 82, 84, 85, 85, 85, 88, 91, 93, 93, 93, 92, 61, 61, 62, 62,
        62, 62, 63, 63, 64, 64, 64, 66, 69, 71, 71, 71, 74, 78, 80, 80, 80, 82,
        84, 85, 85, 85, 88, 91, 93, 93, 93, 92, 63, 64, 65, 65, 65, 65, 66, 66,
        66, 66, 66, 69, 72, 74, 74, 74, 77, 81, 83, 83, 83, 84, 86, 87, 87, 87,
        89, 92, 93, 93, 93, 93, 66, 66, 67, 68, 68, 68, 68, 69, 69, 69, 69, 72,
        75, 77, 77, 77, 80, 83, 85, 85, 85, 87, 88, 89, 89, 89, 91, 93, 94, 94,
        94, 93, 67, 68, 69, 69, 69, 69, 70, 71, 71, 71, 71, 74, 77, 78, 78, 78,
        81, 85, 87, 87, 87, 88, 89, 90, 90, 90, 92, 94, 95, 95, 95, 94, 67, 68,
        69, 69, 69, 69, 70, 71, 71, 71, 71, 74, 77, 78, 78, 78, 81, 85, 87, 87,
        87, 88, 89, 90, 90, 90, 92, 94, 95, 95, 95, 94, 67, 68, 69, 69, 69, 69,
        70, 71, 71, 71, 71, 74, 77, 78, 78, 78, 81, 85, 87, 87, 87, 88, 89, 90,
        90, 90, 92, 94, 95, 95, 95, 94, 69, 70, 72, 72, 72, 72, 73, 74, 74, 74,
        74, 77, 80, 81, 81, 81, 85, 88, 90, 90, 90, 91, 91, 92, 92, 92, 93, 94,
        95, 95, 95, 94, 72, 73, 75, 75, 75, 75, 76, 77, 78, 78, 78, 81, 83, 85,
        85, 85, 88, 92, 93, 93, 93, 94, 94, 94, 94, 94, 94, 95, 95, 95, 95, 94,
        73, 75, 76, 77, 77, 77, 78, 79, 80, 80, 80, 83, 85, 87, 87, 87, 90, 93,
        95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 96, 96, 96, 95, 73, 75, 76, 77,
        77, 77, 78, 79, 80, 80, 80, 83, 85, 87, 87, 87, 90, 93, 95, 95, 95, 95,
        95, 95, 95, 95, 95, 95, 96, 96, 96, 95, 73, 75, 76, 77, 77, 77, 78, 79,
        80, 80, 80, 83, 85, 87, 87, 87, 90, 93, 95, 95, 95, 95, 95, 95, 95, 95,
        95, 95, 96, 96, 96, 95, 77, 78, 79, 80, 80, 80, 81, 82, 82, 82, 82, 84,
        87, 88, 88, 88, 91, 94, 95, 95, 95, 95, 96, 96, 96, 96, 96, 97, 98, 98,
        98, 97, 81, 82, 83, 83, 83, 83, 83, 84, 84, 84, 84, 86, 88, 89, 89, 89,
        91, 94, 95, 95, 95, 96, 96, 97, 97, 97, 98, 99, 100, 100, 100, 99, 83,
        84, 84, 85, 85, 85, 85, 85, 85, 85, 85, 87, 89, 90, 90, 90, 92, 94, 95,
        95, 95, 96, 97, 97, 97, 97, 99, 100, 101, 101, 101, 100, 83, 84, 84, 85,
        85, 85, 85, 85, 85, 85, 85, 87, 89, 90, 90, 90, 92, 94, 95, 95, 95, 96,
        97, 97, 97, 97, 99, 100, 101, 101, 101, 100, 83, 84, 84, 85, 85, 85, 85,
        85, 85, 85, 85, 87, 89, 90, 90, 90, 92, 94, 95, 95, 95, 96, 97, 97, 97,
        97, 99, 100, 101, 101, 101, 100, 89, 89, 89, 89, 89, 89, 88, 88, 88, 88,
        88, 89, 91, 92, 92, 92, 93, 94, 95, 95, 95, 96, 98, 99, 99, 99, 100,
        102, 103, 103, 103, 102, 95, 94, 93, 93, 93, 93, 92, 92, 91, 91, 91, 92,
        93, 94, 94, 94, 94, 95, 95, 95, 95, 97, 99, 100, 100, 100, 102, 104,
        105, 105, 105, 105, 98, 97, 96, 95, 95, 95, 94, 93, 93, 93, 93, 93, 94,
        95, 95, 95, 95, 95, 96, 96, 96, 98, 100, 101, 101, 101, 103, 105, 106,
        106, 106, 106, 98, 97, 96, 95, 95, 95, 94, 93, 93, 93, 93, 93, 94, 95,
        95, 95, 95, 95, 96, 96, 96, 98, 100, 101, 101, 101, 103, 105, 106, 106,
        106, 106, 98, 97, 96, 95, 95, 95, 94, 93, 93, 93, 93, 93, 94, 95, 95,
        95, 95, 95, 96, 96, 96, 98, 100, 101, 101, 101, 103, 105, 106, 106, 106,
        106, 98, 97, 95, 95, 95, 95, 94, 93, 92, 92, 92, 93, 93, 94, 94, 94, 94,
        94, 95, 95, 95, 97, 99, 100, 100, 100, 102, 105, 106, 106, 106, 107,
        /* Size 4x8 */
        63, 62, 75, 96, 63, 63, 77, 94, 62, 64, 80, 92, 69, 72, 86, 94, 75, 79,
        93, 95, 79, 82, 95, 97, 85, 86, 95, 100, 96, 93, 96, 106,
        /* Size 8x4 */
        63, 63, 62, 69, 75, 79, 85, 96, 62, 63, 64, 72, 79, 82, 86, 93, 75, 77,
        80, 86, 93, 95, 95, 96, 96, 94, 92, 94, 95, 97, 100, 106,
        /* Size 8x16 */
        64, 62, 62, 66, 74, 74, 91, 96, 63, 63, 63, 68, 76, 76, 90, 95, 63, 63,
        63, 68, 77, 77, 90, 94, 62, 63, 64, 69, 78, 78, 90, 93, 61, 64, 64, 70,
        80, 80, 89, 92, 61, 64, 64, 70, 80, 80, 89, 92, 66, 69, 70, 75, 85, 85,
        92, 94, 67, 71, 72, 77, 86, 86, 92, 94, 70, 74, 75, 80, 89, 89, 93, 95,
        74, 79, 80, 86, 94, 94, 95, 95, 74, 79, 80, 86, 94, 94, 95, 95, 82, 84,
        85, 88, 95, 95, 98, 99, 84, 85, 86, 89, 95, 95, 99, 100, 89, 89, 89, 91,
        95, 95, 101, 102, 98, 94, 93, 94, 96, 96, 104, 106, 98, 94, 93, 94, 96,
        96, 104, 106,
        /* Size 16x8 */
        64, 63, 63, 62, 61, 61, 66, 67, 70, 74, 74, 82, 84, 89, 98, 98, 62, 63,
        63, 63, 64, 64, 69, 71, 74, 79, 79, 84, 85, 89, 94, 94, 62, 63, 63, 64,
        64, 64, 70, 72, 75, 80, 80, 85, 86, 89, 93, 93, 66, 68, 68, 69, 70, 70,
        75, 77, 80, 86, 86, 88, 89, 91, 94, 94, 74, 76, 77, 78, 80, 80, 85, 86,
        89, 94, 94, 95, 95, 95, 96, 96, 74, 76, 77, 78, 80, 80, 85, 86, 89, 94,
        94, 95, 95, 95, 96, 96, 91, 90, 90, 90, 89, 89, 92, 92, 93, 95, 95, 98,
        99, 101, 104, 104, 96, 95, 94, 93, 92, 92, 94, 94, 95, 95, 95, 99, 100,
        102, 106, 106,
        /* Size 16x32 */
        64, 63, 62, 62, 62, 62, 66, 71, 74, 74, 74, 81, 91, 96, 96, 96, 63, 63,
        62, 62, 62, 62, 67, 72, 75, 75, 75, 82, 91, 96, 96, 96, 63, 63, 63, 63,
        63, 63, 68, 73, 76, 76, 76, 83, 90, 95, 95, 95, 63, 63, 63, 63, 63, 63,
        68, 74, 77, 77, 77, 83, 90, 94, 94, 94, 63, 63, 63, 63, 63, 63, 68, 74,
        77, 77, 77, 83, 90, 94, 94, 94, 63, 63, 63, 63, 63, 63, 68, 74, 77, 77,
        77, 83, 90, 94, 94, 94, 62, 63, 63, 64, 64, 64, 69, 75, 78, 78, 78, 83,
        90, 93, 93, 93, 61, 62, 63, 64, 64, 64, 69, 76, 79, 79, 79, 84, 89, 92,
        92, 92, 61, 62, 64, 64, 64, 64, 70, 76, 80, 80, 80, 84, 89, 92, 92, 92,
        61, 62, 64, 64, 64, 64, 70, 76, 80, 80, 80, 84, 89, 92, 92, 92, 61, 62,
        64, 64, 64, 64, 70, 76, 80, 80, 80, 84, 89, 92, 92, 92, 63, 65, 66, 67,
        67, 67, 72, 79, 82, 82, 82, 86, 90, 93, 93, 93, 66, 68, 69, 70, 70, 70,
        75, 81, 85, 85, 85, 88, 92, 94, 94, 94, 67, 69, 71, 72, 72, 72, 77, 83,
        86, 86, 86, 89, 92, 94, 94, 94, 67, 69, 71, 72, 72, 72, 77, 83, 86, 86,
        86, 89, 92, 94, 94, 94, 67, 69, 71, 72, 72, 72, 77, 83, 86, 86, 86, 89,
        92, 94, 94, 94, 70, 72, 74, 75, 75, 75, 80, 86, 89, 89, 89, 91, 93, 95,
        95, 95, 73, 75, 77, 79, 79, 79, 84, 89, 93, 93, 93, 94, 95, 95, 95, 95,
        74, 76, 79, 80, 80, 80, 86, 91, 94, 94, 94, 95, 95, 95, 95, 95, 74, 76,
        79, 80, 80, 80, 86, 91, 94, 94, 94, 95, 95, 95, 95, 95, 74, 76, 79, 80,
        80, 80, 86, 91, 94, 94, 94, 95, 95, 95, 95, 95, 78, 79, 81, 82, 82, 82,
        87, 92, 95, 95, 95, 96, 97, 97, 97, 97, 82, 83, 84, 85, 85, 85, 88, 92,
        95, 95, 95, 96, 98, 99, 99, 99, 84, 85, 85, 86, 86, 86, 89, 93, 95, 95,
        95, 97, 99, 100, 100, 100, 84, 85, 85, 86, 86, 86, 89, 93, 95, 95, 95,
        97, 99, 100, 100, 100, 84, 85, 85, 86, 86, 86, 89, 93, 95, 95, 95, 97,
        99, 100, 100, 100, 89, 89, 89, 89, 89, 89, 91, 94, 95, 95, 95, 98, 101,
        102, 102, 102, 95, 93, 92, 92, 92, 92, 93, 95, 96, 96, 96, 99, 103, 105,
        105, 105, 98, 96, 94, 93, 93, 93, 94, 96, 96, 96, 96, 100, 104, 106,
        106, 106, 98, 96, 94, 93, 93, 93, 94, 96, 96, 96, 96, 100, 104, 106,
        106, 106, 98, 96, 94, 93, 93, 93, 94, 96, 96, 96, 96, 100, 104, 106,
        106, 106, 98, 96, 94, 93, 93, 93, 94, 95, 95, 95, 95, 99, 104, 106, 106,
        106,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 63, 62, 61, 61, 61, 61, 63, 66, 67, 67, 67, 70, 73,
        74, 74, 74, 78, 82, 84, 84, 84, 89, 95, 98, 98, 98, 98, 63, 63, 63, 63,
        63, 63, 63, 62, 62, 62, 62, 65, 68, 69, 69, 69, 72, 75, 76, 76, 76, 79,
        83, 85, 85, 85, 89, 93, 96, 96, 96, 96, 62, 62, 63, 63, 63, 63, 63, 63,
        64, 64, 64, 66, 69, 71, 71, 71, 74, 77, 79, 79, 79, 81, 84, 85, 85, 85,
        89, 92, 94, 94, 94, 94, 62, 62, 63, 63, 63, 63, 64, 64, 64, 64, 64, 67,
        70, 72, 72, 72, 75, 79, 80, 80, 80, 82, 85, 86, 86, 86, 89, 92, 93, 93,
        93, 93, 62, 62, 63, 63, 63, 63, 64, 64, 64, 64, 64, 67, 70, 72, 72, 72,
        75, 79, 80, 80, 80, 82, 85, 86, 86, 86, 89, 92, 93, 93, 93, 93, 62, 62,
        63, 63, 63, 63, 64, 64, 64, 64, 64, 67, 70, 72, 72, 72, 75, 79, 80, 80,
        80, 82, 85, 86, 86, 86, 89, 92, 93, 93, 93, 93, 66, 67, 68, 68, 68, 68,
        69, 69, 70, 70, 70, 72, 75, 77, 77, 77, 80, 84, 86, 86, 86, 87, 88, 89,
        89, 89, 91, 93, 94, 94, 94, 94, 71, 72, 73, 74, 74, 74, 75, 76, 76, 76,
        76, 79, 81, 83, 83, 83, 86, 89, 91, 91, 91, 92, 92, 93, 93, 93, 94, 95,
        96, 96, 96, 95, 74, 75, 76, 77, 77, 77, 78, 79, 80, 80, 80, 82, 85, 86,
        86, 86, 89, 93, 94, 94, 94, 95, 95, 95, 95, 95, 95, 96, 96, 96, 96, 95,
        74, 75, 76, 77, 77, 77, 78, 79, 80, 80, 80, 82, 85, 86, 86, 86, 89, 93,
        94, 94, 94, 95, 95, 95, 95, 95, 95, 96, 96, 96, 96, 95, 74, 75, 76, 77,
        77, 77, 78, 79, 80, 80, 80, 82, 85, 86, 86, 86, 89, 93, 94, 94, 94, 95,
        95, 95, 95, 95, 95, 96, 96, 96, 96, 95, 81, 82, 83, 83, 83, 83, 83, 84,
        84, 84, 84, 86, 88, 89, 89, 89, 91, 94, 95, 95, 95, 96, 96, 97, 97, 97,
        98, 99, 100, 100, 100, 99, 91, 91, 90, 90, 90, 90, 90, 89, 89, 89, 89,
        90, 92, 92, 92, 92, 93, 95, 95, 95, 95, 97, 98, 99, 99, 99, 101, 103,
        104, 104, 104, 104, 96, 96, 95, 94, 94, 94, 93, 92, 92, 92, 92, 93, 94,
        94, 94, 94, 95, 95, 95, 95, 95, 97, 99, 100, 100, 100, 102, 105, 106,
        106, 106, 106, 96, 96, 95, 94, 94, 94, 93, 92, 92, 92, 92, 93, 94, 94,
        94, 94, 95, 95, 95, 95, 95, 97, 99, 100, 100, 100, 102, 105, 106, 106,
        106, 106, 96, 96, 95, 94, 94, 94, 93, 92, 92, 92, 92, 93, 94, 94, 94,
        94, 95, 95, 95, 95, 95, 97, 99, 100, 100, 100, 102, 105, 106, 106, 106,
        106,
        /* Size 4x16 */
        63, 62, 74, 96, 63, 63, 76, 95, 63, 63, 77, 94, 63, 64, 78, 93, 62, 64,
        80, 92, 62, 64, 80, 92, 68, 70, 85, 94, 69, 72, 86, 94, 72, 75, 89, 95,
        76, 80, 94, 95, 76, 80, 94, 95, 83, 85, 95, 99, 85, 86, 95, 100, 89, 89,
        95, 102, 96, 93, 96, 106, 96, 93, 96, 106,
        /* Size 16x4 */
        63, 63, 63, 63, 62, 62, 68, 69, 72, 76, 76, 83, 85, 89, 96, 96, 62, 63,
        63, 64, 64, 64, 70, 72, 75, 80, 80, 85, 86, 89, 93, 93, 74, 76, 77, 78,
        80, 80, 85, 86, 89, 94, 94, 95, 95, 95, 96, 96, 96, 95, 94, 93, 92, 92,
        94, 94, 95, 95, 95, 99, 100, 102, 106, 106,
        /* Size 8x32 */
        64, 62, 62, 66, 74, 74, 91, 96, 63, 62, 62, 67, 75, 75, 91, 96, 63, 63,
        63, 68, 76, 76, 90, 95, 63, 63, 63, 68, 77, 77, 90, 94, 63, 63, 63, 68,
        77, 77, 90, 94, 63, 63, 63, 68, 77, 77, 90, 94, 62, 63, 64, 69, 78, 78,
        90, 93, 61, 63, 64, 69, 79, 79, 89, 92, 61, 64, 64, 70, 80, 80, 89, 92,
        61, 64, 64, 70, 80, 80, 89, 92, 61, 64, 64, 70, 80, 80, 89, 92, 63, 66,
        67, 72, 82, 82, 90, 93, 66, 69, 70, 75, 85, 85, 92, 94, 67, 71, 72, 77,
        86, 86, 92, 94, 67, 71, 72, 77, 86, 86, 92, 94, 67, 71, 72, 77, 86, 86,
        92, 94, 70, 74, 75, 80, 89, 89, 93, 95, 73, 77, 79, 84, 93, 93, 95, 95,
        74, 79, 80, 86, 94, 94, 95, 95, 74, 79, 80, 86, 94, 94, 95, 95, 74, 79,
        80, 86, 94, 94, 95, 95, 78, 81, 82, 87, 95, 95, 97, 97, 82, 84, 85, 88,
        95, 95, 98, 99, 84, 85, 86, 89, 95, 95, 99, 100, 84, 85, 86, 89, 95, 95,
        99, 100, 84, 85, 86, 89, 95, 95, 99, 100, 89, 89, 89, 91, 95, 95, 101,
        102, 95, 92, 92, 93, 96, 96, 103, 105, 98, 94, 93, 94, 96, 96, 104, 106,
        98, 94, 93, 94, 96, 96, 104, 106, 98, 94, 93, 94, 96, 96, 104, 106, 98,
        94, 93, 94, 95, 95, 104, 106,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 63, 62, 61, 61, 61, 61, 63, 66, 67, 67, 67, 70, 73,
        74, 74, 74, 78, 82, 84, 84, 84, 89, 95, 98, 98, 98, 98, 62, 62, 63, 63,
        63, 63, 63, 63, 64, 64, 64, 66, 69, 71, 71, 71, 74, 77, 79, 79, 79, 81,
        84, 85, 85, 85, 89, 92, 94, 94, 94, 94, 62, 62, 63, 63, 63, 63, 64, 64,
        64, 64, 64, 67, 70, 72, 72, 72, 75, 79, 80, 80, 80, 82, 85, 86, 86, 86,
        89, 92, 93, 93, 93, 93, 66, 67, 68, 68, 68, 68, 69, 69, 70, 70, 70, 72,
        75, 77, 77, 77, 80, 84, 86, 86, 86, 87, 88, 89, 89, 89, 91, 93, 94, 94,
        94, 94, 74, 75, 76, 77, 77, 77, 78, 79, 80, 80, 80, 82, 85, 86, 86, 86,
        89, 93, 94, 94, 94, 95, 95, 95, 95, 95, 95, 96, 96, 96, 96, 95, 74, 75,
        76, 77, 77, 77, 78, 79, 80, 80, 80, 82, 85, 86, 86, 86, 89, 93, 94, 94,
        94, 95, 95, 95, 95, 95, 95, 96, 96, 96, 96, 95, 91, 91, 90, 90, 90, 90,
        90, 89, 89, 89, 89, 90, 92, 92, 92, 92, 93, 95, 95, 95, 95, 97, 98, 99,
        99, 99, 101, 103, 104, 104, 104, 104, 96, 96, 95, 94, 94, 94, 93, 92,
        92, 92, 92, 93, 94, 94, 94, 94, 95, 95, 95, 95, 95, 97, 99, 100, 100,
        100, 102, 105, 106, 106, 106, 106 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        64, 65, 65, 70,
#endif
        /* Size 4x4 */
        63, 64, 64, 65, 64, 65, 65, 66, 64, 65, 67, 68, 65, 66, 68, 71,
        /* Size 8x8 */
        63, 63, 63, 63, 64, 64, 64, 67, 63, 64, 64, 64, 64, 64, 65, 67, 63, 64,
        64, 64, 65, 65, 65, 67, 63, 64, 64, 65, 65, 65, 65, 67, 64, 64, 65, 65,
        67, 67, 68, 70, 64, 64, 65, 65, 67, 68, 69, 71, 64, 65, 65, 65, 68, 69,
        71, 73, 67, 67, 67, 67, 70, 71, 73, 76,
        /* Size 16x16 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 65, 68, 63, 63,
        63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 66, 68, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 66, 68, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 65, 65, 65, 66, 68, 63, 63, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 65, 66, 68, 63, 63, 64, 64, 64, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 66, 67, 63, 63, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 66, 67, 63, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 66, 66, 66,
        67, 68, 63, 64, 64, 64, 65, 65, 65, 65, 66, 66, 66, 67, 67, 67, 68, 70,
        63, 64, 64, 64, 65, 65, 65, 65, 66, 67, 67, 67, 68, 68, 69, 71, 63, 64,
        64, 64, 65, 65, 65, 65, 66, 67, 67, 67, 68, 68, 69, 71, 64, 64, 65, 65,
        65, 65, 65, 66, 67, 67, 67, 69, 70, 70, 71, 72, 64, 64, 65, 65, 65, 65,
        65, 66, 67, 68, 68, 70, 71, 71, 72, 74, 64, 64, 65, 65, 65, 65, 65, 66,
        67, 68, 68, 70, 71, 71, 72, 74, 65, 66, 66, 66, 66, 66, 66, 67, 68, 69,
        69, 71, 72, 72, 73, 76, 68, 68, 68, 68, 68, 67, 67, 68, 70, 71, 71, 72,
        74, 74, 76, 79,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 65, 66, 68, 68, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 67, 68, 68, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 65, 66, 67, 68, 68, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 66, 67,
        68, 68, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 66, 67, 68, 68, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 65, 66, 67, 68, 68, 63, 63, 63, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 65, 65, 66, 67, 68, 68, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65,
        66, 67, 68, 68, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 67, 68, 68,
        63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 67, 67, 67, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 66, 66, 67, 67, 67, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 66, 66, 67, 67, 67, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 67,
        67, 67, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 67, 67, 67, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 66, 66, 66, 66, 66, 66, 66, 67, 67, 68, 68, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 67,
        67, 67, 67, 67, 68, 68, 69, 69, 63, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 67, 67, 67, 67, 67, 67, 68,
        68, 69, 70, 70, 63, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 66, 66, 67, 67, 67, 67, 67, 67, 68, 68, 68, 68, 68, 69, 70, 71, 71,
        63, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 66, 66, 67,
        67, 67, 67, 67, 67, 68, 68, 68, 68, 68, 69, 70, 71, 71, 63, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 66, 66, 67, 67, 67, 67, 67,
        67, 68, 68, 68, 68, 68, 69, 70, 71, 71, 63, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 65, 65, 66, 66, 67, 67, 67, 67, 67, 67, 68, 68, 68,
        68, 68, 69, 70, 71, 71, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65,
        65, 65, 66, 66, 67, 67, 67, 67, 67, 68, 68, 69, 69, 69, 69, 69, 70, 71,
        71, 71, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66,
        67, 67, 67, 67, 67, 68, 69, 69, 70, 70, 70, 70, 71, 71, 72, 72, 64, 64,
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 67, 67, 68, 68, 68,
        68, 69, 69, 70, 70, 70, 70, 71, 71, 72, 73, 73, 64, 64, 64, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 66, 67, 67, 68, 68, 68, 68, 69, 70, 70,
        71, 71, 71, 71, 72, 73, 74, 74, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 66, 67, 67, 68, 68, 68, 68, 69, 70, 70, 71, 71, 71, 71,
        72, 73, 74, 74, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        66, 67, 67, 68, 68, 68, 68, 69, 70, 70, 71, 71, 71, 71, 72, 73, 74, 74,
        64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 67, 68, 68,
        68, 68, 68, 69, 70, 71, 71, 71, 71, 71, 72, 73, 74, 74, 65, 65, 66, 66,
        66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 67, 68, 68, 69, 69, 69, 69, 70,
        71, 71, 72, 72, 72, 72, 73, 74, 76, 76, 66, 67, 67, 67, 67, 67, 67, 67,
        67, 67, 67, 67, 67, 67, 67, 68, 69, 70, 70, 70, 70, 71, 71, 72, 73, 73,
        73, 73, 74, 76, 77, 77, 68, 68, 68, 68, 68, 68, 68, 68, 68, 67, 67, 67,
        67, 67, 68, 69, 70, 71, 71, 71, 71, 71, 72, 73, 74, 74, 74, 74, 76, 77,
        79, 79, 68, 68, 68, 68, 68, 68, 68, 68, 68, 67, 67, 67, 67, 67, 68, 69,
        70, 71, 71, 71, 71, 71, 72, 73, 74, 74, 74, 74, 76, 77, 79, 79,
        /* Size 4x8 */
        63, 63, 64, 64, 63, 64, 65, 65, 64, 64, 65, 65, 64, 65, 65, 66, 64, 65,
        67, 68, 64, 65, 68, 69, 64, 66, 69, 71, 67, 67, 71, 73,
        /* Size 8x4 */
        63, 63, 64, 64, 64, 64, 64, 67, 63, 64, 64, 65, 65, 65, 66, 67, 64, 65,
        65, 65, 67, 68, 69, 71, 64, 65, 65, 66, 68, 69, 71, 73,
        /* Size 8x16 */
        64, 63, 63, 63, 63, 64, 64, 65, 63, 63, 64, 64, 64, 65, 65, 66, 63, 64,
        64, 64, 64, 65, 65, 66, 63, 64, 64, 64, 64, 65, 65, 66, 63, 64, 64, 64,
        65, 65, 65, 66, 63, 64, 65, 65, 65, 66, 66, 66, 63, 64, 65, 65, 65, 66,
        66, 66, 63, 64, 65, 65, 65, 66, 66, 67, 63, 64, 65, 65, 66, 68, 68, 69,
        64, 65, 65, 65, 66, 68, 68, 69, 64, 65, 65, 65, 66, 68, 68, 69, 64, 65,
        65, 65, 67, 70, 70, 71, 64, 65, 66, 66, 68, 71, 71, 72, 64, 65, 66, 66,
        68, 71, 71, 72, 65, 66, 66, 66, 69, 72, 72, 73, 68, 68, 68, 68, 70, 74,
        74, 76,
        /* Size 16x8 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 65, 68, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 66, 68, 63, 64, 64, 64,
        64, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 68, 63, 64, 64, 64, 64, 65,
        65, 65, 65, 65, 65, 65, 66, 66, 66, 68, 63, 64, 64, 64, 65, 65, 65, 65,
        66, 66, 66, 67, 68, 68, 69, 70, 64, 65, 65, 65, 65, 66, 66, 66, 68, 68,
        68, 70, 71, 71, 72, 74, 64, 65, 65, 65, 65, 66, 66, 66, 68, 68, 68, 70,
        71, 71, 72, 74, 65, 66, 66, 66, 66, 66, 66, 67, 69, 69, 69, 71, 72, 72,
        73, 76,
        /* Size 16x32 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 65, 68, 63, 63,
        63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 66, 68, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 66, 68, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 65, 65, 65, 65, 65, 66, 68, 63, 63, 64, 64, 64, 64, 64, 64,
        64, 65, 65, 65, 65, 65, 66, 68, 63, 63, 64, 64, 64, 64, 64, 64, 64, 65,
        65, 65, 65, 65, 66, 68, 63, 63, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65,
        65, 65, 66, 68, 63, 63, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65,
        66, 68, 63, 63, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 66, 68,
        63, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 66, 67, 63, 64,
        64, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 67, 63, 64, 64, 65,
        65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 67, 63, 64, 64, 65, 65, 65,
        65, 65, 65, 65, 66, 66, 66, 66, 66, 67, 63, 64, 64, 65, 65, 65, 65, 65,
        65, 65, 66, 66, 66, 66, 66, 67, 63, 64, 64, 65, 65, 65, 65, 65, 65, 66,
        66, 66, 66, 66, 67, 68, 63, 64, 64, 65, 65, 65, 65, 65, 66, 66, 67, 67,
        67, 67, 68, 69, 63, 64, 64, 65, 65, 65, 65, 65, 66, 67, 68, 68, 68, 68,
        69, 70, 64, 64, 65, 65, 65, 65, 65, 66, 66, 67, 68, 68, 68, 68, 69, 71,
        64, 64, 65, 65, 65, 65, 65, 66, 66, 67, 68, 68, 68, 68, 69, 71, 64, 64,
        65, 65, 65, 65, 65, 66, 66, 67, 68, 68, 68, 68, 69, 71, 64, 64, 65, 65,
        65, 65, 65, 66, 66, 67, 68, 68, 68, 68, 69, 71, 64, 64, 65, 65, 65, 65,
        65, 66, 67, 68, 69, 69, 69, 69, 70, 71, 64, 64, 65, 65, 65, 65, 65, 66,
        67, 68, 70, 70, 70, 70, 71, 72, 64, 64, 65, 65, 66, 66, 66, 66, 67, 69,
        70, 70, 70, 70, 72, 73, 64, 64, 65, 65, 66, 66, 66, 66, 68, 69, 71, 71,
        71, 71, 72, 74, 64, 64, 65, 65, 66, 66, 66, 66, 68, 69, 71, 71, 71, 71,
        72, 74, 64, 64, 65, 65, 66, 66, 66, 66, 68, 69, 71, 71, 71, 71, 72, 74,
        64, 65, 65, 66, 66, 66, 66, 66, 68, 69, 71, 71, 71, 71, 72, 74, 65, 66,
        66, 66, 66, 66, 66, 67, 69, 70, 72, 72, 72, 72, 73, 76, 67, 67, 67, 67,
        67, 67, 67, 68, 69, 71, 73, 73, 73, 73, 75, 77, 68, 68, 68, 68, 68, 68,
        68, 68, 70, 72, 74, 74, 74, 74, 76, 79, 68, 68, 68, 68, 68, 68, 68, 68,
        70, 72, 74, 74, 74, 74, 76, 79,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 67, 68, 68, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 65, 66, 67, 68, 68, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 66, 67, 68, 68, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 67,
        68, 68, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 67, 68, 68, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 66, 66, 66, 66, 66, 66, 67, 68, 68, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66,
        66, 66, 66, 66, 66, 67, 68, 68, 63, 63, 64, 64, 64, 64, 64, 64, 64, 65,
        65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66,
        67, 68, 68, 68, 63, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 66, 66, 66, 66, 66, 66, 67, 67, 67, 68, 68, 68, 68, 69, 69, 70, 70,
        64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 67, 67,
        67, 67, 67, 68, 68, 69, 69, 69, 69, 69, 70, 71, 72, 72, 64, 64, 65, 65,
        65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 67, 68, 68, 68, 68, 68, 69,
        70, 70, 71, 71, 71, 71, 72, 73, 74, 74, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 65, 66, 66, 66, 66, 66, 67, 68, 68, 68, 68, 68, 69, 70, 70, 71, 71,
        71, 71, 72, 73, 74, 74, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66,
        66, 66, 66, 67, 68, 68, 68, 68, 68, 69, 70, 70, 71, 71, 71, 71, 72, 73,
        74, 74, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 67,
        68, 68, 68, 68, 68, 69, 70, 70, 71, 71, 71, 71, 72, 73, 74, 74, 65, 66,
        66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 67, 68, 69, 69, 69, 69,
        69, 70, 71, 72, 72, 72, 72, 72, 73, 75, 76, 76, 68, 68, 68, 68, 68, 68,
        68, 68, 68, 67, 67, 67, 67, 67, 68, 69, 70, 71, 71, 71, 71, 71, 72, 73,
        74, 74, 74, 74, 76, 77, 79, 79,
        /* Size 4x16 */
        63, 63, 64, 64, 63, 64, 64, 65, 63, 64, 65, 65, 63, 64, 65, 65, 63, 64,
        65, 65, 64, 65, 65, 66, 64, 65, 65, 66, 64, 65, 66, 66, 64, 65, 67, 68,
        64, 65, 67, 68, 64, 65, 67, 68, 64, 65, 68, 70, 64, 66, 69, 71, 64, 66,
        69, 71, 66, 66, 70, 72, 68, 68, 72, 74,
        /* Size 16x4 */
        63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 66, 68, 63, 64,
        64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 68, 64, 64, 65, 65,
        65, 65, 65, 66, 67, 67, 67, 68, 69, 69, 70, 72, 64, 65, 65, 65, 65, 66,
        66, 66, 68, 68, 68, 70, 71, 71, 72, 74,
        /* Size 8x32 */
        64, 63, 63, 63, 63, 64, 64, 65, 63, 63, 63, 63, 64, 64, 64, 66, 63, 63,
        64, 64, 64, 65, 65, 66, 63, 64, 64, 64, 64, 65, 65, 66, 63, 64, 64, 64,
        64, 65, 65, 66, 63, 64, 64, 64, 64, 65, 65, 66, 63, 64, 64, 64, 64, 65,
        65, 66, 63, 64, 64, 64, 64, 65, 65, 66, 63, 64, 64, 64, 65, 65, 65, 66,
        63, 64, 64, 64, 65, 65, 65, 66, 63, 64, 65, 65, 65, 66, 66, 66, 63, 64,
        65, 65, 65, 66, 66, 66, 63, 64, 65, 65, 65, 66, 66, 66, 63, 64, 65, 65,
        65, 66, 66, 66, 63, 64, 65, 65, 65, 66, 66, 67, 63, 64, 65, 65, 66, 67,
        67, 68, 63, 64, 65, 65, 66, 68, 68, 69, 64, 65, 65, 65, 66, 68, 68, 69,
        64, 65, 65, 65, 66, 68, 68, 69, 64, 65, 65, 65, 66, 68, 68, 69, 64, 65,
        65, 65, 66, 68, 68, 69, 64, 65, 65, 65, 67, 69, 69, 70, 64, 65, 65, 65,
        67, 70, 70, 71, 64, 65, 66, 66, 67, 70, 70, 72, 64, 65, 66, 66, 68, 71,
        71, 72, 64, 65, 66, 66, 68, 71, 71, 72, 64, 65, 66, 66, 68, 71, 71, 72,
        64, 65, 66, 66, 68, 71, 71, 72, 65, 66, 66, 66, 69, 72, 72, 73, 67, 67,
        67, 67, 69, 73, 73, 75, 68, 68, 68, 68, 70, 74, 74, 76, 68, 68, 68, 68,
        70, 74, 74, 76,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 67, 68, 68, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 66, 67, 68, 68, 63, 63, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66,
        66, 66, 66, 67, 68, 68, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 67,
        68, 68, 63, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 66,
        66, 66, 66, 66, 66, 67, 67, 67, 68, 68, 68, 68, 69, 69, 70, 70, 64, 64,
        65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 67, 68, 68, 68, 68,
        68, 69, 70, 70, 71, 71, 71, 71, 72, 73, 74, 74, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 65, 66, 66, 66, 66, 66, 67, 68, 68, 68, 68, 68, 69, 70, 70,
        71, 71, 71, 71, 72, 73, 74, 74, 65, 66, 66, 66, 66, 66, 66, 66, 66, 66,
        66, 66, 66, 66, 67, 68, 69, 69, 69, 69, 69, 70, 71, 72, 72, 72, 72, 72,
        73, 75, 76, 76 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        63, 76, 76, 93,
#endif
        /* Size 4x4 */
        63, 62, 69, 77, 62, 64, 71, 80, 69, 71, 78, 87, 77, 80, 87, 95,
        /* Size 8x8 */
        63, 62, 62, 61, 68, 70, 74, 81, 62, 63, 63, 62, 69, 71, 77, 82, 62, 63,
        63, 63, 70, 73, 79, 83, 61, 62, 63, 64, 71, 73, 80, 84, 68, 69, 70, 71,
        78, 80, 87, 89, 70, 71, 73, 73, 80, 83, 89, 91, 74, 77, 79, 80, 87, 89,
        95, 95, 81, 82, 83, 84, 89, 91, 95, 96,
        /* Size 16x16 */
        64, 63, 62, 62, 62, 61, 61, 62, 66, 67, 67, 71, 73, 73, 77, 83, 63, 63,
        63, 63, 62, 62, 62, 63, 67, 68, 68, 72, 75, 75, 79, 84, 62, 63, 63, 63,
        63, 62, 62, 64, 68, 69, 69, 74, 77, 77, 80, 85, 62, 63, 63, 63, 63, 62,
        62, 64, 68, 69, 69, 74, 77, 77, 80, 85, 62, 62, 63, 63, 63, 63, 63, 64,
        68, 70, 70, 75, 78, 78, 81, 85, 61, 62, 62, 62, 63, 64, 64, 65, 69, 71,
        71, 76, 80, 80, 82, 85, 61, 62, 62, 62, 63, 64, 64, 65, 69, 71, 71, 76,
        80, 80, 82, 85, 62, 63, 64, 64, 64, 65, 65, 66, 71, 72, 72, 77, 81, 81,
        83, 86, 66, 67, 68, 68, 68, 69, 69, 71, 75, 77, 77, 82, 85, 85, 87, 89,
        67, 68, 69, 69, 70, 71, 71, 72, 77, 78, 78, 83, 87, 87, 88, 90, 67, 68,
        69, 69, 70, 71, 71, 72, 77, 78, 78, 83, 87, 87, 88, 90, 71, 72, 74, 74,
        75, 76, 76, 77, 82, 83, 83, 88, 92, 92, 92, 93, 73, 75, 77, 77, 78, 80,
        80, 81, 85, 87, 87, 92, 95, 95, 95, 95, 73, 75, 77, 77, 78, 80, 80, 81,
        85, 87, 87, 92, 95, 95, 95, 95, 77, 79, 80, 80, 81, 82, 82, 83, 87, 88,
        88, 92, 95, 95, 95, 96, 83, 84, 85, 85, 85, 85, 85, 86, 89, 90, 90, 93,
        95, 95, 96, 97,
        /* Size 32x32 */
        64, 63, 63, 62, 62, 62, 62, 62, 62, 61, 61, 61, 61, 61, 62, 64, 66, 67,
        67, 67, 67, 69, 71, 73, 73, 73, 73, 74, 77, 80, 83, 83, 63, 63, 63, 63,
        62, 62, 62, 62, 62, 62, 61, 61, 61, 61, 62, 64, 66, 68, 68, 68, 68, 70,
        72, 74, 74, 74, 74, 75, 78, 81, 84, 84, 63, 63, 63, 63, 63, 63, 63, 63,
        62, 62, 62, 62, 62, 62, 63, 65, 67, 68, 68, 68, 68, 70, 72, 75, 75, 75,
        75, 76, 79, 81, 84, 84, 62, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62,
        62, 62, 64, 65, 68, 69, 69, 69, 69, 71, 73, 76, 77, 77, 77, 77, 80, 82,
        84, 84, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 64, 66,
        68, 69, 69, 69, 69, 71, 74, 76, 77, 77, 77, 78, 80, 82, 85, 85, 62, 62,
        63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 64, 66, 68, 69, 69, 69,
        69, 71, 74, 76, 77, 77, 77, 78, 80, 82, 85, 85, 62, 62, 63, 63, 63, 63,
        63, 63, 63, 63, 62, 62, 62, 62, 64, 66, 68, 69, 69, 69, 69, 71, 74, 76,
        77, 77, 77, 78, 80, 82, 85, 85, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 64, 66, 68, 69, 69, 69, 69, 72, 74, 76, 77, 77, 77, 78,
        80, 82, 85, 85, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        64, 66, 68, 70, 70, 70, 70, 72, 75, 77, 78, 78, 78, 79, 81, 83, 85, 85,
        61, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 67, 69, 70,
        70, 70, 70, 73, 75, 78, 79, 79, 79, 80, 81, 83, 85, 85, 61, 61, 62, 62,
        62, 62, 62, 63, 63, 63, 64, 64, 64, 64, 65, 67, 69, 71, 71, 71, 71, 73,
        76, 79, 80, 80, 80, 80, 82, 84, 85, 85, 61, 61, 62, 62, 62, 62, 62, 63,
        63, 63, 64, 64, 64, 64, 65, 67, 69, 71, 71, 71, 71, 73, 76, 79, 80, 80,
        80, 80, 82, 84, 85, 85, 61, 61, 62, 62, 62, 62, 62, 63, 63, 63, 64, 64,
        64, 64, 65, 67, 69, 71, 71, 71, 71, 73, 76, 79, 80, 80, 80, 80, 82, 84,
        85, 85, 61, 61, 62, 62, 62, 62, 62, 63, 63, 63, 64, 64, 64, 64, 65, 67,
        69, 71, 71, 71, 71, 73, 76, 79, 80, 80, 80, 80, 82, 84, 85, 85, 62, 62,
        63, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 66, 68, 71, 72, 72, 72,
        72, 75, 77, 80, 81, 81, 81, 82, 83, 85, 86, 86, 64, 64, 65, 65, 66, 66,
        66, 66, 66, 67, 67, 67, 67, 67, 68, 71, 73, 74, 74, 74, 74, 77, 79, 82,
        83, 83, 83, 84, 85, 86, 87, 87, 66, 66, 67, 68, 68, 68, 68, 68, 68, 69,
        69, 69, 69, 69, 71, 73, 75, 77, 77, 77, 77, 79, 82, 84, 85, 85, 85, 86,
        87, 88, 89, 89, 67, 68, 68, 69, 69, 69, 69, 69, 70, 70, 71, 71, 71, 71,
        72, 74, 77, 78, 78, 78, 78, 80, 83, 86, 87, 87, 87, 87, 88, 89, 90, 90,
        67, 68, 68, 69, 69, 69, 69, 69, 70, 70, 71, 71, 71, 71, 72, 74, 77, 78,
        78, 78, 78, 80, 83, 86, 87, 87, 87, 87, 88, 89, 90, 90, 67, 68, 68, 69,
        69, 69, 69, 69, 70, 70, 71, 71, 71, 71, 72, 74, 77, 78, 78, 78, 78, 80,
        83, 86, 87, 87, 87, 87, 88, 89, 90, 90, 67, 68, 68, 69, 69, 69, 69, 69,
        70, 70, 71, 71, 71, 71, 72, 74, 77, 78, 78, 78, 78, 80, 83, 86, 87, 87,
        87, 87, 88, 89, 90, 90, 69, 70, 70, 71, 71, 71, 71, 72, 72, 73, 73, 73,
        73, 73, 75, 77, 79, 80, 80, 80, 80, 83, 85, 88, 89, 89, 89, 89, 90, 91,
        91, 91, 71, 72, 72, 73, 74, 74, 74, 74, 75, 75, 76, 76, 76, 76, 77, 79,
        82, 83, 83, 83, 83, 85, 88, 91, 92, 92, 92, 92, 92, 92, 93, 93, 73, 74,
        75, 76, 76, 76, 76, 76, 77, 78, 79, 79, 79, 79, 80, 82, 84, 86, 86, 86,
        86, 88, 91, 93, 94, 94, 94, 94, 94, 94, 94, 94, 73, 74, 75, 77, 77, 77,
        77, 77, 78, 79, 80, 80, 80, 80, 81, 83, 85, 87, 87, 87, 87, 89, 92, 94,
        95, 95, 95, 95, 95, 95, 95, 95, 73, 74, 75, 77, 77, 77, 77, 77, 78, 79,
        80, 80, 80, 80, 81, 83, 85, 87, 87, 87, 87, 89, 92, 94, 95, 95, 95, 95,
        95, 95, 95, 95, 73, 74, 75, 77, 77, 77, 77, 77, 78, 79, 80, 80, 80, 80,
        81, 83, 85, 87, 87, 87, 87, 89, 92, 94, 95, 95, 95, 95, 95, 95, 95, 95,
        74, 75, 76, 77, 78, 78, 78, 78, 79, 80, 80, 80, 80, 80, 82, 84, 86, 87,
        87, 87, 87, 89, 92, 94, 95, 95, 95, 95, 95, 95, 95, 95, 77, 78, 79, 80,
        80, 80, 80, 80, 81, 81, 82, 82, 82, 82, 83, 85, 87, 88, 88, 88, 88, 90,
        92, 94, 95, 95, 95, 95, 95, 96, 96, 96, 80, 81, 81, 82, 82, 82, 82, 82,
        83, 83, 84, 84, 84, 84, 85, 86, 88, 89, 89, 89, 89, 91, 92, 94, 95, 95,
        95, 95, 96, 96, 96, 96, 83, 84, 84, 84, 85, 85, 85, 85, 85, 85, 85, 85,
        85, 85, 86, 87, 89, 90, 90, 90, 90, 91, 93, 94, 95, 95, 95, 95, 96, 96,
        97, 97, 83, 84, 84, 84, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 86, 87,
        89, 90, 90, 90, 90, 91, 93, 94, 95, 95, 95, 95, 96, 96, 97, 97,
        /* Size 4x8 */
        63, 62, 70, 75, 63, 63, 72, 77, 62, 64, 74, 79, 62, 64, 74, 80, 69, 72,
        81, 86, 71, 74, 84, 89, 76, 80, 90, 94, 82, 84, 91, 95,
        /* Size 8x4 */
        63, 63, 62, 62, 69, 71, 76, 82, 62, 63, 64, 64, 72, 74, 80, 84, 70, 72,
        74, 74, 81, 84, 90, 91, 75, 77, 79, 80, 86, 89, 94, 95,
        /* Size 8x16 */
        64, 62, 62, 62, 66, 74, 74, 77, 63, 63, 63, 63, 67, 76, 76, 79, 63, 63,
        63, 63, 68, 77, 77, 80, 63, 63, 63, 63, 68, 77, 77, 80, 62, 63, 64, 64,
        69, 78, 78, 81, 61, 63, 64, 64, 70, 80, 80, 82, 61, 63, 64, 64, 70, 80,
        80, 82, 62, 64, 66, 66, 71, 81, 81, 83, 66, 68, 70, 70, 75, 85, 85, 86,
        67, 70, 72, 72, 77, 86, 86, 88, 67, 70, 72, 72, 77, 86, 86, 88, 71, 74,
        77, 77, 82, 91, 91, 92, 74, 78, 80, 80, 86, 94, 94, 95, 74, 78, 80, 80,
        86, 94, 94, 95, 78, 80, 82, 82, 87, 95, 95, 95, 84, 85, 86, 86, 89, 95,
        95, 96,
        /* Size 16x8 */
        64, 63, 63, 63, 62, 61, 61, 62, 66, 67, 67, 71, 74, 74, 78, 84, 62, 63,
        63, 63, 63, 63, 63, 64, 68, 70, 70, 74, 78, 78, 80, 85, 62, 63, 63, 63,
        64, 64, 64, 66, 70, 72, 72, 77, 80, 80, 82, 86, 62, 63, 63, 63, 64, 64,
        64, 66, 70, 72, 72, 77, 80, 80, 82, 86, 66, 67, 68, 68, 69, 70, 70, 71,
        75, 77, 77, 82, 86, 86, 87, 89, 74, 76, 77, 77, 78, 80, 80, 81, 85, 86,
        86, 91, 94, 94, 95, 95, 74, 76, 77, 77, 78, 80, 80, 81, 85, 86, 86, 91,
        94, 94, 95, 95, 77, 79, 80, 80, 81, 82, 82, 83, 86, 88, 88, 92, 95, 95,
        95, 96,
        /* Size 16x32 */
        64, 63, 62, 62, 62, 62, 62, 63, 66, 70, 74, 74, 74, 74, 77, 84, 63, 63,
        63, 62, 62, 62, 62, 63, 67, 70, 75, 75, 75, 75, 78, 84, 63, 63, 63, 63,
        63, 63, 63, 64, 67, 71, 76, 76, 76, 76, 79, 84, 63, 63, 63, 63, 63, 63,
        63, 64, 68, 72, 77, 77, 77, 77, 80, 85, 63, 63, 63, 63, 63, 63, 63, 64,
        68, 72, 77, 77, 77, 77, 80, 85, 63, 63, 63, 63, 63, 63, 63, 64, 68, 72,
        77, 77, 77, 77, 80, 85, 63, 63, 63, 63, 63, 63, 63, 64, 68, 72, 77, 77,
        77, 77, 80, 85, 62, 63, 63, 63, 63, 63, 63, 64, 68, 72, 77, 77, 77, 77,
        80, 85, 62, 62, 63, 63, 64, 64, 64, 65, 69, 73, 78, 78, 78, 78, 81, 85,
        61, 62, 63, 64, 64, 64, 64, 65, 69, 74, 79, 79, 79, 79, 81, 85, 61, 62,
        63, 64, 64, 64, 64, 66, 70, 74, 80, 80, 80, 80, 82, 85, 61, 62, 63, 64,
        64, 64, 64, 66, 70, 74, 80, 80, 80, 80, 82, 85, 61, 62, 63, 64, 64, 64,
        64, 66, 70, 74, 80, 80, 80, 80, 82, 85, 61, 62, 63, 64, 64, 64, 64, 66,
        70, 74, 80, 80, 80, 80, 82, 85, 62, 63, 64, 65, 66, 66, 66, 67, 71, 76,
        81, 81, 81, 81, 83, 86, 64, 65, 66, 67, 68, 68, 68, 69, 73, 78, 83, 83,
        83, 83, 85, 88, 66, 67, 68, 70, 70, 70, 70, 71, 75, 80, 85, 85, 85, 85,
        86, 89, 67, 69, 70, 71, 72, 72, 72, 73, 77, 81, 86, 86, 86, 86, 88, 90,
        67, 69, 70, 71, 72, 72, 72, 73, 77, 81, 86, 86, 86, 86, 88, 90, 67, 69,
        70, 71, 72, 72, 72, 73, 77, 81, 86, 86, 86, 86, 88, 90, 67, 69, 70, 71,
        72, 72, 72, 73, 77, 81, 86, 86, 86, 86, 88, 90, 69, 71, 72, 74, 74, 74,
        74, 75, 79, 84, 89, 89, 89, 89, 90, 91, 71, 73, 74, 76, 77, 77, 77, 78,
        82, 86, 91, 91, 91, 91, 92, 93, 73, 75, 77, 79, 79, 79, 79, 81, 85, 89,
        94, 94, 94, 94, 94, 94, 74, 76, 78, 80, 80, 80, 80, 82, 86, 90, 94, 94,
        94, 94, 95, 95, 74, 76, 78, 80, 80, 80, 80, 82, 86, 90, 94, 94, 94, 94,
        95, 95, 74, 76, 78, 80, 80, 80, 80, 82, 86, 90, 94, 94, 94, 94, 95, 95,
        75, 77, 78, 80, 81, 81, 81, 82, 86, 90, 94, 94, 94, 94, 95, 95, 78, 79,
        80, 82, 82, 82, 82, 84, 87, 91, 95, 95, 95, 95, 95, 96, 81, 82, 83, 84,
        84, 84, 84, 85, 88, 91, 95, 95, 95, 95, 95, 97, 84, 84, 85, 86, 86, 86,
        86, 87, 89, 92, 95, 95, 95, 95, 96, 97, 84, 84, 85, 86, 86, 86, 86, 87,
        89, 92, 95, 95, 95, 95, 96, 97,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 63, 63, 62, 62, 61, 61, 61, 61, 61, 62, 64, 66, 67,
        67, 67, 67, 69, 71, 73, 74, 74, 74, 75, 78, 81, 84, 84, 63, 63, 63, 63,
        63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 63, 65, 67, 69, 69, 69, 69, 71,
        73, 75, 76, 76, 76, 77, 79, 82, 84, 84, 62, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 64, 66, 68, 70, 70, 70, 70, 72, 74, 77, 78, 78,
        78, 78, 80, 83, 85, 85, 62, 62, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64,
        64, 64, 65, 67, 70, 71, 71, 71, 71, 74, 76, 79, 80, 80, 80, 80, 82, 84,
        86, 86, 62, 62, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 66, 68,
        70, 72, 72, 72, 72, 74, 77, 79, 80, 80, 80, 81, 82, 84, 86, 86, 62, 62,
        63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 66, 68, 70, 72, 72, 72,
        72, 74, 77, 79, 80, 80, 80, 81, 82, 84, 86, 86, 62, 62, 63, 63, 63, 63,
        63, 63, 64, 64, 64, 64, 64, 64, 66, 68, 70, 72, 72, 72, 72, 74, 77, 79,
        80, 80, 80, 81, 82, 84, 86, 86, 63, 63, 64, 64, 64, 64, 64, 64, 65, 65,
        66, 66, 66, 66, 67, 69, 71, 73, 73, 73, 73, 75, 78, 81, 82, 82, 82, 82,
        84, 85, 87, 87, 66, 67, 67, 68, 68, 68, 68, 68, 69, 69, 70, 70, 70, 70,
        71, 73, 75, 77, 77, 77, 77, 79, 82, 85, 86, 86, 86, 86, 87, 88, 89, 89,
        70, 70, 71, 72, 72, 72, 72, 72, 73, 74, 74, 74, 74, 74, 76, 78, 80, 81,
        81, 81, 81, 84, 86, 89, 90, 90, 90, 90, 91, 91, 92, 92, 74, 75, 76, 77,
        77, 77, 77, 77, 78, 79, 80, 80, 80, 80, 81, 83, 85, 86, 86, 86, 86, 89,
        91, 94, 94, 94, 94, 94, 95, 95, 95, 95, 74, 75, 76, 77, 77, 77, 77, 77,
        78, 79, 80, 80, 80, 80, 81, 83, 85, 86, 86, 86, 86, 89, 91, 94, 94, 94,
        94, 94, 95, 95, 95, 95, 74, 75, 76, 77, 77, 77, 77, 77, 78, 79, 80, 80,
        80, 80, 81, 83, 85, 86, 86, 86, 86, 89, 91, 94, 94, 94, 94, 94, 95, 95,
        95, 95, 74, 75, 76, 77, 77, 77, 77, 77, 78, 79, 80, 80, 80, 80, 81, 83,
        85, 86, 86, 86, 86, 89, 91, 94, 94, 94, 94, 94, 95, 95, 95, 95, 77, 78,
        79, 80, 80, 80, 80, 80, 81, 81, 82, 82, 82, 82, 83, 85, 86, 88, 88, 88,
        88, 90, 92, 94, 95, 95, 95, 95, 95, 95, 96, 96, 84, 84, 84, 85, 85, 85,
        85, 85, 85, 85, 85, 85, 85, 85, 86, 88, 89, 90, 90, 90, 90, 91, 93, 94,
        95, 95, 95, 95, 96, 97, 97, 97,
        /* Size 4x16 */
        63, 62, 70, 74, 63, 63, 71, 76, 63, 63, 72, 77, 63, 63, 72, 77, 62, 64,
        73, 78, 62, 64, 74, 80, 62, 64, 74, 80, 63, 66, 76, 81, 67, 70, 80, 85,
        69, 72, 81, 86, 69, 72, 81, 86, 73, 77, 86, 91, 76, 80, 90, 94, 76, 80,
        90, 94, 79, 82, 91, 95, 84, 86, 92, 95,
        /* Size 16x4 */
        63, 63, 63, 63, 62, 62, 62, 63, 67, 69, 69, 73, 76, 76, 79, 84, 62, 63,
        63, 63, 64, 64, 64, 66, 70, 72, 72, 77, 80, 80, 82, 86, 70, 71, 72, 72,
        73, 74, 74, 76, 80, 81, 81, 86, 90, 90, 91, 92, 74, 76, 77, 77, 78, 80,
        80, 81, 85, 86, 86, 91, 94, 94, 95, 95,
        /* Size 8x32 */
        64, 62, 62, 62, 66, 74, 74, 77, 63, 63, 62, 62, 67, 75, 75, 78, 63, 63,
        63, 63, 67, 76, 76, 79, 63, 63, 63, 63, 68, 77, 77, 80, 63, 63, 63, 63,
        68, 77, 77, 80, 63, 63, 63, 63, 68, 77, 77, 80, 63, 63, 63, 63, 68, 77,
        77, 80, 62, 63, 63, 63, 68, 77, 77, 80, 62, 63, 64, 64, 69, 78, 78, 81,
        61, 63, 64, 64, 69, 79, 79, 81, 61, 63, 64, 64, 70, 80, 80, 82, 61, 63,
        64, 64, 70, 80, 80, 82, 61, 63, 64, 64, 70, 80, 80, 82, 61, 63, 64, 64,
        70, 80, 80, 82, 62, 64, 66, 66, 71, 81, 81, 83, 64, 66, 68, 68, 73, 83,
        83, 85, 66, 68, 70, 70, 75, 85, 85, 86, 67, 70, 72, 72, 77, 86, 86, 88,
        67, 70, 72, 72, 77, 86, 86, 88, 67, 70, 72, 72, 77, 86, 86, 88, 67, 70,
        72, 72, 77, 86, 86, 88, 69, 72, 74, 74, 79, 89, 89, 90, 71, 74, 77, 77,
        82, 91, 91, 92, 73, 77, 79, 79, 85, 94, 94, 94, 74, 78, 80, 80, 86, 94,
        94, 95, 74, 78, 80, 80, 86, 94, 94, 95, 74, 78, 80, 80, 86, 94, 94, 95,
        75, 78, 81, 81, 86, 94, 94, 95, 78, 80, 82, 82, 87, 95, 95, 95, 81, 83,
        84, 84, 88, 95, 95, 95, 84, 85, 86, 86, 89, 95, 95, 96, 84, 85, 86, 86,
        89, 95, 95, 96,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 63, 63, 62, 62, 61, 61, 61, 61, 61, 62, 64, 66, 67,
        67, 67, 67, 69, 71, 73, 74, 74, 74, 75, 78, 81, 84, 84, 62, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 66, 68, 70, 70, 70, 70, 72,
        74, 77, 78, 78, 78, 78, 80, 83, 85, 85, 62, 62, 63, 63, 63, 63, 63, 63,
        64, 64, 64, 64, 64, 64, 66, 68, 70, 72, 72, 72, 72, 74, 77, 79, 80, 80,
        80, 81, 82, 84, 86, 86, 62, 62, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64,
        64, 64, 66, 68, 70, 72, 72, 72, 72, 74, 77, 79, 80, 80, 80, 81, 82, 84,
        86, 86, 66, 67, 67, 68, 68, 68, 68, 68, 69, 69, 70, 70, 70, 70, 71, 73,
        75, 77, 77, 77, 77, 79, 82, 85, 86, 86, 86, 86, 87, 88, 89, 89, 74, 75,
        76, 77, 77, 77, 77, 77, 78, 79, 80, 80, 80, 80, 81, 83, 85, 86, 86, 86,
        86, 89, 91, 94, 94, 94, 94, 94, 95, 95, 95, 95, 74, 75, 76, 77, 77, 77,
        77, 77, 78, 79, 80, 80, 80, 80, 81, 83, 85, 86, 86, 86, 86, 89, 91, 94,
        94, 94, 94, 94, 95, 95, 95, 95, 77, 78, 79, 80, 80, 80, 80, 80, 81, 81,
        82, 82, 82, 82, 83, 85, 86, 88, 88, 88, 88, 90, 92, 94, 95, 95, 95, 95,
        95, 95, 96, 96 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        64, 64, 64, 66,
#endif
        /* Size 4x4 */
        63, 63, 63, 64, 63, 64, 64, 65, 63, 64, 65, 65, 64, 65, 65, 67,
        /* Size 8x8 */
        63, 63, 63, 63, 63, 63, 64, 64, 63, 64, 64, 64, 64, 64, 64, 64, 63, 64,
        64, 64, 64, 64, 64, 64, 63, 64, 64, 64, 64, 64, 65, 65, 63, 64, 64, 64,
        65, 65, 65, 65, 63, 64, 64, 64, 65, 65, 65, 65, 64, 64, 64, 65, 65, 65,
        67, 67, 64, 64, 64, 65, 65, 65, 67, 67,
        /* Size 16x16 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 63, 63, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 65, 63, 63, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        63, 63, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 63,
        64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 63, 64, 64,
        64, 64, 64, 65, 65, 65, 65, 65, 66, 66, 66, 66, 63, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 63, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 65, 66, 66, 67, 67, 67, 63, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 66, 66, 67, 67, 67, 63, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 66,
        66, 67, 67, 67,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65,
        65, 65, 65, 65, 65, 65, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 65, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 63,
        63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 63, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 63, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 63, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 66, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 66, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66,
        66, 66, 66, 66, 66, 66, 66, 67, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 67, 67, 67,
        67, 67, 67, 67, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 67, 67, 67, 67, 67, 67, 67,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 66, 66, 66, 67, 67, 67, 67, 67, 67, 67, 63, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        66, 66, 66, 67, 67, 67, 67, 67, 67, 67, 63, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 67,
        67, 67, 67, 67, 67, 67, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 67, 67, 67, 67, 67,
        67, 67, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 66, 66, 66, 67, 67, 67, 67, 67, 67, 67, 67,
        /* Size 4x8 */
        63, 63, 63, 64, 63, 64, 64, 65, 63, 64, 64, 65, 63, 64, 64, 65, 63, 65,
        65, 65, 63, 65, 65, 66, 64, 65, 65, 67, 64, 65, 65, 67,
        /* Size 8x4 */
        63, 63, 63, 63, 63, 63, 64, 64, 63, 64, 64, 64, 65, 65, 65, 65, 63, 64,
        64, 64, 65, 65, 65, 65, 64, 65, 65, 65, 65, 66, 67, 67,
        /* Size 8x16 */
        64, 63, 63, 63, 63, 63, 63, 64, 63, 63, 63, 63, 63, 63, 64, 64, 63, 63,
        64, 64, 64, 64, 64, 65, 63, 64, 64, 64, 64, 64, 64, 65, 63, 64, 64, 64,
        64, 64, 64, 65, 63, 64, 64, 64, 64, 64, 64, 65, 63, 64, 64, 64, 64, 64,
        65, 65, 63, 64, 64, 65, 65, 65, 65, 65, 63, 64, 64, 65, 65, 65, 65, 65,
        63, 64, 64, 65, 65, 65, 65, 65, 63, 64, 64, 65, 65, 65, 65, 65, 63, 64,
        65, 65, 65, 65, 66, 66, 63, 64, 65, 65, 65, 65, 66, 67, 64, 64, 65, 65,
        65, 65, 66, 68, 64, 64, 65, 65, 65, 65, 66, 68, 64, 64, 65, 65, 65, 65,
        66, 68,
        /* Size 16x8 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 63, 63, 64, 64, 64, 64,
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 63, 64, 64, 64, 64, 64, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 63, 63, 64, 64, 64, 64, 64, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 63, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 66,
        66, 66, 66, 66, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 67, 68,
        68, 68,
        /* Size 16x32 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 65, 65, 65, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 65, 65, 65, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65,
        65, 65, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65,
        63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 63, 63, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 65, 65, 65, 65, 65, 63, 63, 64, 64, 64, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 63, 63, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 66, 63, 63, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 66, 63, 63, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66,
        63, 63, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 63, 63,
        64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 63, 63, 64, 64,
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 63, 63, 64, 64, 64, 65,
        65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 63, 64, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 65, 66, 66, 66, 67, 63, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 65, 66, 66, 67, 67, 63, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 66,
        66, 67, 67, 68, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 67,
        68, 68, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 67, 68, 68,
        64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 67, 68, 68, 64, 64,
        64, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 67, 68, 68, 64, 64, 64, 65,
        65, 65, 65, 65, 65, 65, 65, 66, 66, 67, 68, 68, 64, 64, 64, 65, 65, 65,
        65, 65, 65, 65, 65, 66, 66, 67, 68, 68, 64, 64, 64, 65, 65, 65, 65, 65,
        65, 65, 65, 66, 67, 67, 68, 69,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65,
        65, 65, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 63, 63, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 63, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66,
        66, 66, 66, 66, 66, 66, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66,
        66, 67, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 66, 66, 66, 67, 67, 67, 67, 67, 67, 67, 67, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 66, 66, 67, 67, 68, 68, 68, 68, 68, 68, 68, 64, 64, 64, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 67, 67,
        68, 68, 68, 68, 68, 68, 68, 69,
        /* Size 4x16 */
        63, 63, 63, 64, 63, 63, 63, 64, 63, 64, 64, 64, 63, 64, 64, 65, 63, 64,
        64, 65, 63, 64, 64, 65, 63, 64, 64, 65, 63, 65, 65, 65, 63, 65, 65, 65,
        63, 65, 65, 65, 63, 65, 65, 65, 64, 65, 65, 66, 64, 65, 65, 67, 64, 65,
        65, 67, 64, 65, 65, 67, 64, 65, 65, 67,
        /* Size 16x4 */
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 63, 63,
        64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 63, 64, 64,
        64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 65, 65, 65,
        65, 65, 65, 65, 65, 66, 67, 67, 67, 67,
        /* Size 8x32 */
        64, 63, 63, 63, 63, 63, 63, 64, 63, 63, 63, 63, 63, 63, 64, 64, 63, 63,
        63, 63, 63, 63, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64,
        64, 64, 64, 65, 63, 64, 64, 64, 64, 64, 64, 65, 63, 64, 64, 64, 64, 64,
        64, 65, 63, 64, 64, 64, 64, 64, 64, 65, 63, 64, 64, 64, 64, 64, 64, 65,
        63, 64, 64, 64, 64, 64, 64, 65, 63, 64, 64, 64, 64, 64, 64, 65, 63, 64,
        64, 64, 64, 64, 64, 65, 63, 64, 64, 64, 64, 64, 65, 65, 63, 64, 64, 64,
        64, 64, 65, 65, 63, 64, 64, 65, 65, 65, 65, 65, 63, 64, 64, 65, 65, 65,
        65, 65, 63, 64, 64, 65, 65, 65, 65, 65, 63, 64, 64, 65, 65, 65, 65, 65,
        63, 64, 64, 65, 65, 65, 65, 65, 63, 64, 64, 65, 65, 65, 65, 65, 63, 64,
        64, 65, 65, 65, 65, 65, 63, 64, 64, 65, 65, 65, 65, 66, 63, 64, 65, 65,
        65, 65, 66, 66, 63, 64, 65, 65, 65, 65, 66, 67, 63, 64, 65, 65, 65, 65,
        66, 67, 64, 64, 65, 65, 65, 65, 66, 68, 64, 64, 65, 65, 65, 65, 66, 68,
        64, 64, 65, 65, 65, 65, 66, 68, 64, 64, 65, 65, 65, 65, 66, 68, 64, 64,
        65, 65, 65, 65, 66, 68, 64, 64, 65, 65, 65, 65, 66, 68, 64, 64, 65, 65,
        65, 65, 67, 68,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66,
        66, 66, 66, 66, 66, 66, 66, 67, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 67, 67, 68, 68, 68,
        68, 68, 68, 68 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        63, 66, 66, 72,
#endif
        /* Size 4x4 */
        63, 62, 62, 68, 62, 63, 63, 70, 62, 63, 64, 71, 68, 70, 71, 78,
        /* Size 8x8 */
        63, 62, 62, 62, 61, 62, 67, 67, 62, 63, 63, 63, 62, 64, 69, 69, 62, 63,
        63, 63, 62, 64, 69, 69, 62, 63, 63, 63, 63, 64, 70, 70, 61, 62, 62, 63,
        64, 65, 71, 71, 62, 64, 64, 64, 65, 66, 72, 72, 67, 69, 69, 70, 71, 72,
        78, 78, 67, 69, 69, 70, 71, 72, 78, 78,
        /* Size 16x16 */
        64, 63, 63, 62, 62, 62, 62, 61, 61, 61, 61, 63, 66, 67, 67, 67, 63, 63,
        63, 63, 63, 63, 62, 62, 61, 61, 61, 64, 66, 68, 68, 68, 63, 63, 63, 63,
        63, 63, 62, 62, 62, 62, 62, 65, 67, 69, 69, 69, 62, 63, 63, 63, 63, 63,
        63, 63, 62, 62, 62, 65, 68, 69, 69, 69, 62, 63, 63, 63, 63, 63, 63, 63,
        62, 62, 62, 65, 68, 69, 69, 69, 62, 63, 63, 63, 63, 63, 63, 63, 62, 62,
        62, 65, 68, 69, 69, 69, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 66,
        68, 70, 70, 70, 61, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 66, 69, 71,
        71, 71, 61, 61, 62, 62, 62, 62, 63, 63, 64, 64, 64, 66, 69, 71, 71, 71,
        61, 61, 62, 62, 62, 62, 63, 63, 64, 64, 64, 66, 69, 71, 71, 71, 61, 61,
        62, 62, 62, 62, 63, 63, 64, 64, 64, 66, 69, 71, 71, 71, 63, 64, 65, 65,
        65, 65, 66, 66, 66, 66, 66, 69, 72, 74, 74, 74, 66, 66, 67, 68, 68, 68,
        68, 69, 69, 69, 69, 72, 75, 77, 77, 77, 67, 68, 69, 69, 69, 69, 70, 71,
        71, 71, 71, 74, 77, 78, 78, 78, 67, 68, 69, 69, 69, 69, 70, 71, 71, 71,
        71, 74, 77, 78, 78, 78, 67, 68, 69, 69, 69, 69, 70, 71, 71, 71, 71, 74,
        77, 78, 78, 78,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 61, 61, 61, 61, 61,
        61, 61, 61, 62, 63, 64, 66, 67, 67, 67, 67, 67, 67, 68, 63, 63, 63, 63,
        63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 61, 61, 61, 61, 61, 61, 61, 62,
        63, 65, 66, 67, 67, 67, 67, 67, 67, 69, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 62, 62, 62, 62, 61, 61, 61, 61, 61, 61, 63, 64, 65, 66, 68,
        68, 68, 68, 68, 68, 69, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62,
        62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 64, 66, 67, 68, 68, 68, 68, 68,
        68, 70, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62,
        62, 62, 62, 62, 62, 63, 65, 66, 67, 69, 69, 69, 69, 69, 69, 70, 62, 62,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62,
        62, 64, 65, 66, 68, 69, 69, 69, 69, 69, 69, 71, 62, 62, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 64, 65, 66,
        68, 69, 69, 69, 69, 69, 69, 71, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 64, 65, 66, 68, 69, 69, 69,
        69, 69, 69, 71, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 62, 62, 62, 62, 62, 62, 64, 65, 66, 68, 69, 69, 69, 69, 69, 69, 71,
        62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62,
        62, 62, 62, 64, 65, 66, 68, 69, 69, 69, 69, 69, 69, 71, 62, 62, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 64,
        65, 66, 68, 69, 69, 69, 69, 69, 69, 71, 62, 62, 62, 62, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 65, 67, 68, 70,
        70, 70, 70, 70, 70, 71, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 66, 67, 68, 70, 70, 70, 70, 70,
        70, 71, 61, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 64, 66, 67, 69, 70, 70, 70, 70, 70, 70, 72, 61, 61,
        62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 65, 66, 68, 69, 71, 71, 71, 71, 71, 71, 72, 61, 61, 61, 62, 62, 62,
        62, 62, 62, 62, 62, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 65, 66, 68,
        69, 71, 71, 71, 71, 71, 71, 73, 61, 61, 61, 62, 62, 62, 62, 62, 62, 62,
        62, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 65, 66, 68, 69, 71, 71, 71,
        71, 71, 71, 73, 61, 61, 61, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 65, 66, 68, 69, 71, 71, 71, 71, 71, 71, 73,
        61, 61, 61, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 64, 64, 64,
        64, 64, 64, 65, 66, 68, 69, 71, 71, 71, 71, 71, 71, 73, 61, 61, 61, 62,
        62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 65,
        66, 68, 69, 71, 71, 71, 71, 71, 71, 73, 61, 61, 61, 62, 62, 62, 62, 62,
        62, 62, 62, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 65, 66, 68, 69, 71,
        71, 71, 71, 71, 71, 73, 62, 62, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 65, 65, 66, 68, 69, 71, 72, 72, 72, 72, 72,
        72, 74, 63, 63, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66,
        66, 66, 66, 66, 66, 68, 69, 71, 72, 74, 74, 74, 74, 74, 74, 75, 64, 65,
        65, 66, 66, 66, 66, 66, 66, 66, 66, 67, 67, 67, 68, 68, 68, 68, 68, 68,
        68, 69, 71, 72, 74, 75, 75, 75, 75, 75, 75, 77, 66, 66, 66, 67, 67, 68,
        68, 68, 68, 68, 68, 68, 68, 69, 69, 69, 69, 69, 69, 69, 69, 71, 72, 74,
        75, 77, 77, 77, 77, 77, 77, 78, 67, 67, 68, 68, 69, 69, 69, 69, 69, 69,
        69, 70, 70, 70, 71, 71, 71, 71, 71, 71, 71, 72, 74, 75, 77, 78, 78, 78,
        78, 78, 78, 80, 67, 67, 68, 68, 69, 69, 69, 69, 69, 69, 69, 70, 70, 70,
        71, 71, 71, 71, 71, 71, 71, 72, 74, 75, 77, 78, 78, 78, 78, 78, 78, 80,
        67, 67, 68, 68, 69, 69, 69, 69, 69, 69, 69, 70, 70, 70, 71, 71, 71, 71,
        71, 71, 71, 72, 74, 75, 77, 78, 78, 78, 78, 78, 78, 80, 67, 67, 68, 68,
        69, 69, 69, 69, 69, 69, 69, 70, 70, 70, 71, 71, 71, 71, 71, 71, 71, 72,
        74, 75, 77, 78, 78, 78, 78, 78, 78, 80, 67, 67, 68, 68, 69, 69, 69, 69,
        69, 69, 69, 70, 70, 70, 71, 71, 71, 71, 71, 71, 71, 72, 74, 75, 77, 78,
        78, 78, 78, 78, 78, 80, 67, 67, 68, 68, 69, 69, 69, 69, 69, 69, 69, 70,
        70, 70, 71, 71, 71, 71, 71, 71, 71, 72, 74, 75, 77, 78, 78, 78, 78, 78,
        78, 80, 68, 69, 69, 70, 70, 71, 71, 71, 71, 71, 71, 71, 71, 72, 72, 73,
        73, 73, 73, 73, 73, 74, 75, 77, 78, 80, 80, 80, 80, 80, 80, 81,
        /* Size 4x8 */
        63, 62, 62, 69, 63, 63, 63, 71, 63, 63, 63, 71, 62, 64, 64, 72, 62, 64,
        64, 73, 63, 66, 66, 74, 68, 72, 72, 80, 68, 72, 72, 80,
        /* Size 8x4 */
        63, 63, 63, 62, 62, 63, 68, 68, 62, 63, 63, 64, 64, 66, 72, 72, 62, 63,
        63, 64, 64, 66, 72, 72, 69, 71, 71, 72, 73, 74, 80, 80,
        /* Size 8x16 */
        64, 63, 62, 62, 62, 62, 66, 71, 63, 63, 62, 62, 62, 62, 67, 72, 63, 63,
        63, 63, 63, 63, 68, 73, 63, 63, 63, 63, 63, 63, 68, 74, 63, 63, 63, 63,
        63, 63, 68, 74, 63, 63, 63, 63, 63, 63, 68, 74, 62, 63, 63, 64, 64, 64,
        69, 75, 61, 62, 63, 64, 64, 64, 69, 76, 61, 62, 64, 64, 64, 64, 70, 76,
        61, 62, 64, 64, 64, 64, 70, 76, 61, 62, 64, 64, 64, 64, 70, 76, 63, 65,
        66, 67, 67, 67, 72, 79, 66, 68, 69, 70, 70, 70, 75, 81, 67, 69, 71, 72,
        72, 72, 77, 83, 67, 69, 71, 72, 72, 72, 77, 83, 67, 69, 71, 72, 72, 72,
        77, 83,
        /* Size 16x8 */
        64, 63, 63, 63, 63, 63, 62, 61, 61, 61, 61, 63, 66, 67, 67, 67, 63, 63,
        63, 63, 63, 63, 63, 62, 62, 62, 62, 65, 68, 69, 69, 69, 62, 62, 63, 63,
        63, 63, 63, 63, 64, 64, 64, 66, 69, 71, 71, 71, 62, 62, 63, 63, 63, 63,
        64, 64, 64, 64, 64, 67, 70, 72, 72, 72, 62, 62, 63, 63, 63, 63, 64, 64,
        64, 64, 64, 67, 70, 72, 72, 72, 62, 62, 63, 63, 63, 63, 64, 64, 64, 64,
        64, 67, 70, 72, 72, 72, 66, 67, 68, 68, 68, 68, 69, 69, 70, 70, 70, 72,
        75, 77, 77, 77, 71, 72, 73, 74, 74, 74, 75, 76, 76, 76, 76, 79, 81, 83,
        83, 83,
        /* Size 16x32 */
        64, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 64, 66, 68, 71, 74, 63, 63,
        63, 62, 62, 62, 62, 62, 62, 62, 62, 64, 66, 69, 71, 74, 63, 63, 63, 63,
        62, 62, 62, 62, 62, 62, 62, 64, 67, 69, 72, 75, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 65, 67, 70, 73, 76, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 65, 68, 70, 73, 76, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 66, 68, 71, 74, 77, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 66,
        68, 71, 74, 77, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 66, 68, 71,
        74, 77, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 66, 68, 71, 74, 77,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 66, 68, 71, 74, 77, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 66, 68, 71, 74, 77, 62, 62, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 66, 68, 71, 74, 77, 62, 62, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 66, 69, 72, 75, 78, 62, 62, 62, 63, 63, 64, 64, 64,
        64, 64, 64, 66, 69, 72, 75, 79, 61, 62, 62, 63, 63, 64, 64, 64, 64, 64,
        64, 67, 69, 72, 76, 79, 61, 62, 62, 63, 64, 64, 64, 64, 64, 64, 64, 67,
        70, 73, 76, 80, 61, 62, 62, 63, 64, 64, 64, 64, 64, 64, 64, 67, 70, 73,
        76, 80, 61, 62, 62, 63, 64, 64, 64, 64, 64, 64, 64, 67, 70, 73, 76, 80,
        61, 62, 62, 63, 64, 64, 64, 64, 64, 64, 64, 67, 70, 73, 76, 80, 61, 62,
        62, 63, 64, 64, 64, 64, 64, 64, 64, 67, 70, 73, 76, 80, 61, 62, 62, 63,
        64, 64, 64, 64, 64, 64, 64, 67, 70, 73, 76, 80, 62, 63, 63, 64, 65, 66,
        66, 66, 66, 66, 66, 68, 71, 74, 77, 81, 63, 64, 65, 66, 66, 67, 67, 67,
        67, 67, 67, 70, 72, 75, 79, 82, 65, 65, 66, 67, 68, 68, 68, 68, 68, 68,
        68, 71, 74, 77, 80, 83, 66, 67, 68, 68, 69, 70, 70, 70, 70, 70, 70, 73,
        75, 78, 81, 85, 67, 68, 69, 70, 71, 72, 72, 72, 72, 72, 72, 74, 77, 80,
        83, 86, 67, 68, 69, 70, 71, 72, 72, 72, 72, 72, 72, 74, 77, 80, 83, 86,
        67, 68, 69, 70, 71, 72, 72, 72, 72, 72, 72, 74, 77, 80, 83, 86, 67, 68,
        69, 70, 71, 72, 72, 72, 72, 72, 72, 74, 77, 80, 83, 86, 67, 68, 69, 70,
        71, 72, 72, 72, 72, 72, 72, 74, 77, 80, 83, 86, 67, 68, 69, 70, 71, 72,
        72, 72, 72, 72, 72, 74, 77, 80, 83, 86, 69, 69, 70, 71, 72, 73, 73, 73,
        73, 73, 73, 76, 78, 81, 84, 88,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 61, 61, 61, 61,
        61, 61, 61, 62, 63, 65, 66, 67, 67, 67, 67, 67, 67, 69, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 63,
        64, 65, 67, 68, 68, 68, 68, 68, 68, 69, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 63, 65, 66, 68, 69,
        69, 69, 69, 69, 69, 70, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 66, 67, 68, 70, 70, 70, 70, 70,
        70, 71, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 65, 66, 68, 69, 71, 71, 71, 71, 71, 71, 72, 62, 62,
        62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 66, 67, 68, 70, 72, 72, 72, 72, 72, 72, 73, 62, 62, 62, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 66, 67, 68,
        70, 72, 72, 72, 72, 72, 72, 73, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 66, 67, 68, 70, 72, 72, 72,
        72, 72, 72, 73, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 66, 67, 68, 70, 72, 72, 72, 72, 72, 72, 73,
        62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 66, 67, 68, 70, 72, 72, 72, 72, 72, 72, 73, 62, 62, 62, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 66,
        67, 68, 70, 72, 72, 72, 72, 72, 72, 73, 64, 64, 64, 65, 65, 66, 66, 66,
        66, 66, 66, 66, 66, 66, 67, 67, 67, 67, 67, 67, 67, 68, 70, 71, 73, 74,
        74, 74, 74, 74, 74, 76, 66, 66, 67, 67, 68, 68, 68, 68, 68, 68, 68, 68,
        69, 69, 69, 70, 70, 70, 70, 70, 70, 71, 72, 74, 75, 77, 77, 77, 77, 77,
        77, 78, 68, 69, 69, 70, 70, 71, 71, 71, 71, 71, 71, 71, 72, 72, 72, 73,
        73, 73, 73, 73, 73, 74, 75, 77, 78, 80, 80, 80, 80, 80, 80, 81, 71, 71,
        72, 73, 73, 74, 74, 74, 74, 74, 74, 74, 75, 75, 76, 76, 76, 76, 76, 76,
        76, 77, 79, 80, 81, 83, 83, 83, 83, 83, 83, 84, 74, 74, 75, 76, 76, 77,
        77, 77, 77, 77, 77, 77, 78, 79, 79, 80, 80, 80, 80, 80, 80, 81, 82, 83,
        85, 86, 86, 86, 86, 86, 86, 88,
        /* Size 4x16 */
        63, 62, 62, 68, 63, 62, 62, 69, 63, 63, 63, 70, 63, 63, 63, 71, 63, 63,
        63, 71, 63, 63, 63, 71, 62, 64, 64, 72, 62, 64, 64, 72, 62, 64, 64, 73,
        62, 64, 64, 73, 62, 64, 64, 73, 64, 67, 67, 75, 67, 70, 70, 78, 68, 72,
        72, 80, 68, 72, 72, 80, 68, 72, 72, 80,
        /* Size 16x4 */
        63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 64, 67, 68, 68, 68, 62, 62,
        63, 63, 63, 63, 64, 64, 64, 64, 64, 67, 70, 72, 72, 72, 62, 62, 63, 63,
        63, 63, 64, 64, 64, 64, 64, 67, 70, 72, 72, 72, 68, 69, 70, 71, 71, 71,
        72, 72, 73, 73, 73, 75, 78, 80, 80, 80,
        /* Size 8x32 */
        64, 63, 62, 62, 62, 62, 66, 71, 63, 63, 62, 62, 62, 62, 66, 71, 63, 63,
        62, 62, 62, 62, 67, 72, 63, 63, 63, 63, 63, 63, 67, 73, 63, 63, 63, 63,
        63, 63, 68, 73, 63, 63, 63, 63, 63, 63, 68, 74, 63, 63, 63, 63, 63, 63,
        68, 74, 63, 63, 63, 63, 63, 63, 68, 74, 63, 63, 63, 63, 63, 63, 68, 74,
        63, 63, 63, 63, 63, 63, 68, 74, 63, 63, 63, 63, 63, 63, 68, 74, 62, 63,
        63, 63, 63, 63, 68, 74, 62, 63, 63, 64, 64, 64, 69, 75, 62, 62, 63, 64,
        64, 64, 69, 75, 61, 62, 63, 64, 64, 64, 69, 76, 61, 62, 64, 64, 64, 64,
        70, 76, 61, 62, 64, 64, 64, 64, 70, 76, 61, 62, 64, 64, 64, 64, 70, 76,
        61, 62, 64, 64, 64, 64, 70, 76, 61, 62, 64, 64, 64, 64, 70, 76, 61, 62,
        64, 64, 64, 64, 70, 76, 62, 63, 65, 66, 66, 66, 71, 77, 63, 65, 66, 67,
        67, 67, 72, 79, 65, 66, 68, 68, 68, 68, 74, 80, 66, 68, 69, 70, 70, 70,
        75, 81, 67, 69, 71, 72, 72, 72, 77, 83, 67, 69, 71, 72, 72, 72, 77, 83,
        67, 69, 71, 72, 72, 72, 77, 83, 67, 69, 71, 72, 72, 72, 77, 83, 67, 69,
        71, 72, 72, 72, 77, 83, 67, 69, 71, 72, 72, 72, 77, 83, 69, 70, 72, 73,
        73, 73, 78, 84,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 61, 61, 61, 61,
        61, 61, 61, 62, 63, 65, 66, 67, 67, 67, 67, 67, 67, 69, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 63,
        65, 66, 68, 69, 69, 69, 69, 69, 69, 70, 62, 62, 62, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 65, 66, 68, 69, 71,
        71, 71, 71, 71, 71, 72, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 66, 67, 68, 70, 72, 72, 72, 72, 72,
        72, 73, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 66, 67, 68, 70, 72, 72, 72, 72, 72, 72, 73, 62, 62,
        62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 66, 67, 68, 70, 72, 72, 72, 72, 72, 72, 73, 66, 66, 67, 67, 68, 68,
        68, 68, 68, 68, 68, 68, 69, 69, 69, 70, 70, 70, 70, 70, 70, 71, 72, 74,
        75, 77, 77, 77, 77, 77, 77, 78, 71, 71, 72, 73, 73, 74, 74, 74, 74, 74,
        74, 74, 75, 75, 76, 76, 76, 76, 76, 76, 76, 77, 79, 80, 81, 83, 83, 83,
        83, 83, 83, 84 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        63, 63, 63, 64,
#endif
        /* Size 4x4 */
        63, 63, 63, 63, 63, 64, 64, 64, 63, 64, 64, 64, 63, 64, 64, 64,
        /* Size 8x8 */
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64,
        64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64,
        64, 64, 63, 63, 64, 64, 64, 64, 64, 65,
        /* Size 16x16 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 65, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 63, 63, 63, 63, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 65, 65, 65, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        /* Size 4x8 */
        63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 63, 64, 64, 64, 63, 64,
        64, 64, 63, 64, 64, 64, 63, 64, 64, 64, 63, 64, 64, 65,
        /* Size 8x4 */
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63,
        64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 65,
        /* Size 8x16 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 63, 63, 63, 64,
        64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64,
        64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64,
        63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63,
        64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64,
        64, 64, 64, 64, 63, 63, 64, 64, 64, 65, 65, 65, 63, 63, 64, 64, 64, 65,
        65, 65,
        /* Size 16x8 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        65, 65,
        /* Size 16x32 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 63, 63, 63, 64, 64, 64,
        64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 65, 65, 65, 65, 65, 65, 65,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 63, 63, 63, 63,
        63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 63, 63, 63, 63, 63, 63, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 63, 63,
        63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 63, 63, 63, 63, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65,
        /* Size 4x16 */
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 63, 64,
        64, 64, 63, 64, 64, 64, 63, 64, 64, 64, 63, 64, 64, 64, 63, 64, 64, 64,
        63, 64, 64, 64, 63, 64, 64, 64, 63, 64, 64, 64, 63, 64, 64, 64, 63, 64,
        64, 64, 63, 64, 64, 65, 63, 64, 65, 65,
        /* Size 16x4 */
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 63, 63, 63, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        /* Size 8x32 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64,
        64, 64, 63, 63, 63, 64, 64, 64, 64, 64, 63, 63, 63, 64, 64, 64, 64, 64,
        63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63,
        64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64,
        64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64,
        64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64,
        63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63,
        64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64,
        64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64,
        64, 64, 63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 64, 64, 64,
        63, 63, 64, 64, 64, 64, 64, 64, 63, 63, 64, 64, 64, 65, 65, 65, 63, 63,
        64, 64, 64, 65, 65, 65, 63, 63, 64, 64, 64, 65, 65, 65, 63, 63, 64, 64,
        64, 65, 65, 65,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 63, 63, 63, 63, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 65 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        63, 62, 62, 63,
#endif
        /* Size 4x4 */
        63, 62, 62, 62, 62, 63, 63, 63, 62, 63, 63, 63, 62, 63, 63, 63,
        /* Size 8x8 */
        63, 63, 63, 62, 62, 62, 62, 61, 63, 63, 63, 63, 63, 62, 62, 62, 63, 63,
        63, 63, 63, 63, 62, 62, 62, 63, 63, 63, 63, 63, 63, 62, 62, 63, 63, 63,
        63, 63, 63, 62, 62, 62, 63, 63, 63, 63, 63, 63, 62, 62, 62, 63, 63, 63,
        63, 63, 61, 62, 62, 62, 62, 63, 63, 63,
        /* Size 16x16 */
        64, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 61, 61, 61, 63, 63,
        63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 61, 61, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 61, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62,
        62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 61, 62, 62, 62, 62, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 61, 61, 62, 62, 62, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 61, 61, 61, 62, 62, 62, 62, 62, 62, 62, 62, 63,
        63, 63, 63, 64,
        /* Size 32x32 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62,
        62, 62, 62, 62, 62, 62, 62, 62, 61, 61, 61, 61, 61, 61, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62,
        62, 62, 62, 62, 61, 61, 61, 61, 61, 61, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62,
        62, 61, 61, 61, 61, 61, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62,
        62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 61, 61,
        61, 61, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 61, 61, 61, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62,
        62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62,
        62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62,
        62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62,
        62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62,
        62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62,
        62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62,
        62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62,
        62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 61, 61, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        61, 61, 61, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 61, 61, 61, 61,
        62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 61, 61, 61, 61, 61, 62, 62, 62,
        62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 61, 61, 61, 61, 61, 62, 62, 62, 62, 62, 62, 62,
        62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        64, 64, 61, 61, 61, 61, 61, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62,
        62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64,
        /* Size 4x8 */
        63, 63, 62, 62, 63, 63, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 62, 63, 63, 63, 62, 63, 64, 64, 61, 63, 64, 64,
        /* Size 8x4 */
        63, 63, 63, 63, 63, 62, 62, 61, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62,
        63, 63, 63, 63, 64, 64, 62, 62, 63, 63, 63, 63, 64, 64,
        /* Size 8x16 */
        64, 63, 63, 62, 62, 62, 62, 62, 63, 63, 63, 62, 62, 62, 62, 62, 63, 63,
        63, 63, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62,
        63, 63, 63, 63, 63, 63, 62, 62, 63, 63, 63, 64, 64, 64, 62, 62, 62, 63,
        63, 64, 64, 64, 61, 62, 62, 63, 63, 64, 64, 64, 61, 62, 62, 63, 64, 64,
        64, 64,
        /* Size 16x8 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 61, 61, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 64, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 64, 64, 64, 64, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        64, 64, 64, 64, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64,
        64, 64,
        /* Size 16x32 */
        64, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63,
        63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63,
        63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63,
        63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 62,
        62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62,
        62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63,
        64, 64, 64, 64, 64, 64, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 64, 64,
        64, 64, 64, 64, 62, 62, 62, 62, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 62, 62, 62, 62, 62, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64,
        61, 62, 62, 62, 62, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 61, 62,
        62, 62, 62, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 61, 61, 62, 62,
        62, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 61, 61, 62, 62, 62, 63,
        63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 61, 61, 62, 62, 62, 63, 63, 63,
        64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 32x16 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 62, 62, 62, 62, 62, 62, 61, 61, 61, 61, 61, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62,
        62, 62, 62, 62, 62, 62, 62, 61, 61, 61, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62,
        62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62,
        62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64,
        62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 62, 62, 62, 62,
        62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62, 62, 62, 62, 62, 62, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62, 62,
        62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62, 62, 62, 62, 62, 62,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 4x16 */
        63, 63, 62, 62, 63, 63, 62, 62, 63, 63, 62, 62, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 62, 63, 63, 63, 62, 63, 63, 64, 62, 63,
        64, 64, 62, 63, 64, 64, 61, 63, 64, 64,
        /* Size 16x4 */
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 61, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 62, 62, 62, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 64, 64, 64, 64,
        /* Size 8x32 */
        64, 63, 63, 62, 62, 62, 62, 62, 63, 63, 63, 62, 62, 62, 62, 62, 63, 63,
        63, 62, 62, 62, 62, 62, 63, 63, 63, 63, 62, 62, 62, 62, 63, 63, 63, 63,
        62, 62, 62, 62, 63, 63, 63, 63, 63, 62, 62, 62, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 62, 63, 63, 63, 63, 63, 63, 63, 62, 62, 63, 63,
        63, 63, 63, 63, 62, 62, 63, 63, 63, 64, 64, 64, 62, 62, 63, 63, 63, 64,
        64, 64, 62, 62, 63, 63, 63, 64, 64, 64, 62, 62, 62, 63, 63, 64, 64, 64,
        61, 62, 62, 63, 63, 64, 64, 64, 61, 62, 62, 63, 63, 64, 64, 64, 61, 62,
        62, 63, 64, 64, 64, 64, 61, 62, 62, 63, 64, 64, 64, 64, 61, 62, 62, 63,
        64, 64, 64, 64,
        /* Size 32x8 */
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 62, 62, 62, 62, 62, 62, 61, 61, 61, 61, 61, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 62, 62,
        62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62, 62, 62, 62, 62, 62,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64,
        64, 64, 64, 64 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        64, 64, 64, 64,
#endif
        /* Size 4x4 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 8x8 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 16x16 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64,
        /* Size 32x32 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 4x8 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 8x4 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 8x16 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64,
        /* Size 16x8 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64,
        /* Size 16x32 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 32x16 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 4x16 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 16x4 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 8x32 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64,
        /* Size 32x8 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        64, 64, 64, 64,
#endif
        /* Size 4x4 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 8x8 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 16x16 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64,
        /* Size 32x32 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 4x8 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 8x4 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 8x16 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64,
        /* Size 16x8 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64,
        /* Size 16x32 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 32x16 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 4x16 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 16x4 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        /* Size 8x32 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64,
        /* Size 32x8 */
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64 },
  },
};

static uint16_t wt_matrix_ref[NUM_QM_LEVELS][2][QM_TOTAL_SIZE] = {
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        48, 24, 24, 12,
#endif
        /* Size 4x4 */
        63, 48, 28, 20, 48, 31, 22, 17, 28, 22, 15, 15, 20, 17, 15, 10,
        /* Size 8x8 */
        64, 63, 53, 40, 30, 24, 21, 18, 63, 58, 51, 41, 33, 27, 23, 19, 53, 51,
        38, 32, 26, 23, 20, 19, 40, 41, 32, 25, 21, 18, 18, 18, 30, 33, 26, 21,
        17, 15, 14, 15, 24, 27, 23, 18, 15, 13, 14, 11, 21, 23, 20, 18, 14, 14,
        11, 8, 18, 19, 19, 18, 15, 11, 8, 9,
        /* Size 16x16 */
        64, 65, 65, 60, 57, 47, 42, 35, 31, 26, 25, 22, 21, 19, 18, 16, 65, 63,
        63, 61, 59, 50, 46, 38, 34, 28, 27, 25, 23, 21, 19, 18, 65, 63, 61, 58,
        56, 49, 45, 38, 35, 29, 28, 25, 23, 22, 20, 18, 60, 61, 58, 52, 48, 43,
        40, 35, 33, 28, 27, 25, 24, 23, 21, 20, 57, 59, 56, 48, 42, 38, 36, 32,
        30, 26, 25, 23, 22, 21, 18, 17, 47, 50, 49, 43, 38, 32, 31, 27, 26, 23,
        22, 22, 20, 21, 18, 16, 42, 46, 45, 40, 36, 31, 29, 25, 24, 21, 21, 19,
        20, 18, 18, 15, 35, 38, 38, 35, 32, 27, 25, 22, 21, 19, 18, 18, 18, 16,
        18, 14, 31, 34, 35, 33, 30, 26, 24, 21, 20, 17, 17, 16, 15, 14, 17, 13,
        26, 28, 29, 28, 26, 23, 21, 19, 17, 15, 15, 15, 14, 15, 16, 11, 25, 27,
        28, 27, 25, 22, 21, 18, 17, 15, 15, 14, 14, 15, 14, 11, 22, 25, 25, 25,
        23, 22, 19, 18, 16, 15, 14, 13, 13, 12, 12, 10, 21, 23, 23, 24, 22, 20,
        20, 18, 15, 14, 14, 13, 12, 12, 10, 9, 19, 21, 22, 23, 21, 21, 18, 16,
        14, 15, 15, 12, 12, 11, 11, 8, 18, 19, 20, 21, 18, 18, 18, 18, 17, 16,
        14, 12, 10, 11, 10, 10, 16, 18, 18, 20, 17, 16, 15, 14, 13, 11, 11, 10,
        9, 8, 10, 9,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 64, 60, 59, 57, 52, 47, 45, 42, 38, 35, 33, 31, 29,
        26, 25, 25, 23, 22, 21, 21, 20, 19, 18, 18, 17, 16, 16, 65, 64, 64, 64,
        64, 63, 60, 59, 59, 53, 48, 46, 44, 40, 36, 35, 33, 30, 27, 27, 26, 24,
        23, 23, 22, 21, 20, 19, 19, 18, 17, 17, 65, 64, 63, 63, 63, 63, 61, 60,
        59, 55, 50, 48, 46, 41, 38, 36, 34, 32, 28, 28, 27, 26, 25, 24, 23, 22,
        21, 20, 19, 18, 18, 16, 65, 64, 63, 63, 62, 61, 59, 59, 58, 53, 49, 48,
        46, 41, 38, 36, 34, 32, 28, 28, 27, 26, 25, 24, 23, 22, 21, 20, 19, 19,
        18, 18, 65, 64, 63, 62, 61, 60, 58, 57, 56, 53, 49, 47, 45, 41, 38, 36,
        35, 32, 29, 28, 28, 26, 25, 24, 23, 23, 22, 21, 20, 19, 18, 17, 64, 63,
        63, 61, 60, 58, 55, 55, 53, 51, 48, 47, 45, 41, 38, 37, 35, 33, 29, 29,
        28, 27, 26, 25, 24, 23, 22, 21, 20, 19, 19, 18, 60, 60, 61, 59, 58, 55,
        52, 50, 48, 46, 43, 41, 40, 38, 35, 34, 33, 30, 28, 27, 27, 25, 25, 25,
        24, 23, 23, 22, 21, 20, 20, 18, 59, 59, 60, 59, 57, 55, 50, 48, 45, 43,
        41, 39, 38, 36, 34, 33, 31, 29, 27, 26, 26, 25, 25, 24, 22, 21, 21, 21,
        21, 20, 20, 20, 57, 59, 59, 58, 56, 53, 48, 45, 42, 41, 38, 37, 36, 34,
        32, 31, 30, 28, 26, 26, 25, 24, 23, 22, 22, 22, 21, 20, 18, 18, 17, 19,
        52, 53, 55, 53, 53, 51, 46, 43, 41, 38, 35, 34, 34, 32, 30, 29, 28, 26,
        24, 24, 24, 23, 22, 22, 22, 20, 19, 19, 20, 19, 19, 16, 47, 48, 50, 49,
        49, 48, 43, 41, 38, 35, 32, 32, 31, 29, 27, 26, 26, 24, 23, 22, 22, 21,
        22, 21, 20, 19, 21, 20, 18, 17, 16, 18, 45, 46, 48, 48, 47, 47, 41, 39,
        37, 34, 32, 31, 30, 28, 26, 26, 25, 23, 22, 22, 21, 21, 21, 19, 20, 20,
        18, 17, 18, 19, 18, 15, 42, 44, 46, 46, 45, 45, 40, 38, 36, 34, 31, 30,
        29, 27, 25, 25, 24, 23, 21, 21, 21, 20, 19, 20, 20, 17, 18, 19, 18, 15,
        15, 18, 38, 40, 41, 41, 41, 41, 38, 36, 34, 32, 29, 28, 27, 25, 24, 23,
        22, 21, 20, 20, 19, 18, 19, 19, 17, 18, 19, 16, 16, 18, 18, 14, 35, 36,
        38, 38, 38, 38, 35, 34, 32, 30, 27, 26, 25, 24, 22, 22, 21, 20, 19, 18,
        18, 18, 18, 17, 18, 17, 16, 18, 18, 14, 14, 17, 33, 35, 36, 36, 36, 37,
        34, 33, 31, 29, 26, 26, 25, 23, 22, 21, 20, 19, 18, 18, 18, 17, 17, 17,
        17, 15, 18, 16, 14, 17, 17, 13, 31, 33, 34, 34, 35, 35, 33, 31, 30, 28,
        26, 25, 24, 22, 21, 20, 20, 18, 17, 17, 17, 16, 16, 17, 15, 17, 14, 15,
        17, 14, 13, 16, 29, 30, 32, 32, 32, 33, 30, 29, 28, 26, 24, 23, 23, 21,
        20, 19, 18, 17, 16, 16, 16, 15, 16, 14, 16, 14, 15, 16, 13, 15, 16, 12,
        26, 27, 28, 28, 29, 29, 28, 27, 26, 24, 23, 22, 21, 20, 19, 18, 17, 16,
        15, 15, 15, 15, 15, 15, 14, 14, 15, 13, 16, 13, 11, 16, 25, 27, 28, 28,
        28, 29, 27, 26, 26, 24, 22, 22, 21, 20, 18, 18, 17, 16, 15, 15, 15, 14,
        13, 15, 13, 15, 12, 15, 12, 13, 15, 10, 25, 26, 27, 27, 28, 28, 27, 26,
        25, 24, 22, 21, 21, 19, 18, 18, 17, 16, 15, 15, 15, 14, 14, 12, 14, 12,
        15, 11, 14, 13, 11, 15, 23, 24, 26, 26, 26, 27, 25, 25, 24, 23, 21, 21,
        20, 18, 18, 17, 16, 15, 15, 14, 14, 13, 14, 13, 12, 14, 11, 14, 12, 11,
        14, 10, 22, 23, 25, 25, 25, 26, 25, 25, 23, 22, 22, 21, 19, 19, 18, 17,
        16, 16, 15, 13, 14, 14, 13, 13, 13, 12, 12, 12, 12, 13, 10, 14, 21, 23,
        24, 24, 24, 25, 25, 24, 22, 22, 21, 19, 20, 19, 17, 17, 17, 14, 15, 15,
        12, 13, 13, 12, 13, 12, 12, 11, 12, 10, 13, 9, 21, 22, 23, 23, 23, 24,
        24, 22, 22, 22, 20, 20, 20, 17, 18, 17, 15, 16, 14, 13, 14, 12, 13, 13,
        12, 13, 12, 12, 10, 13, 9, 13, 20, 21, 22, 22, 23, 23, 23, 21, 22, 20,
        19, 20, 17, 18, 17, 15, 17, 14, 14, 15, 12, 14, 12, 12, 13, 11, 12, 11,
        12, 8, 12, 8, 19, 20, 21, 21, 22, 22, 23, 21, 21, 19, 21, 18, 18, 19,
        16, 18, 14, 15, 15, 12, 15, 11, 12, 12, 12, 12, 11, 12, 11, 12, 8, 12,
        18, 19, 20, 20, 21, 21, 22, 21, 20, 19, 20, 17, 19, 16, 18, 16, 15, 16,
        13, 15, 11, 14, 12, 11, 12, 11, 12, 10, 12, 10, 12, 7, 18, 19, 19, 19,
        20, 20, 21, 21, 18, 20, 18, 18, 18, 16, 18, 14, 17, 13, 16, 12, 14, 12,
        12, 12, 10, 12, 11, 12, 10, 12, 10, 11, 17, 18, 18, 19, 19, 19, 20, 20,
        18, 19, 17, 19, 15, 18, 14, 17, 14, 15, 13, 13, 13, 11, 13, 10, 13, 8,
        12, 10, 12, 9, 11, 9, 16, 17, 18, 18, 18, 19, 20, 20, 17, 19, 16, 18,
        15, 18, 14, 17, 13, 16, 11, 15, 11, 14, 10, 13, 9, 12, 8, 12, 10, 11, 9,
        11, 16, 17, 16, 18, 17, 18, 18, 20, 19, 16, 18, 15, 18, 14, 17, 13, 16,
        12, 16, 10, 15, 10, 14, 9, 13, 8, 12, 7, 11, 9, 11, 8,
        /* Size 4x8 */
        64, 48, 27, 20, 62, 48, 30, 22, 55, 35, 24, 20, 41, 29, 20, 16, 31, 24,
        16, 16, 25, 21, 14, 14, 22, 19, 14, 11, 19, 19, 15, 12,
        /* Size 8x4 */
        64, 62, 55, 41, 31, 25, 22, 19, 48, 48, 35, 29, 24, 21, 19, 19, 27, 30,
        24, 20, 16, 14, 14, 15, 20, 22, 20, 16, 16, 14, 11, 12,
        /* Size 8x16 */
        64, 64, 57, 38, 31, 24, 21, 18, 65, 62, 59, 41, 34, 26, 23, 19, 64, 60,
        56, 41, 35, 27, 24, 20, 60, 55, 48, 38, 33, 26, 24, 22, 57, 53, 43, 34,
        30, 24, 23, 19, 47, 48, 38, 29, 26, 21, 20, 17, 43, 44, 36, 27, 24, 20,
        18, 16, 35, 38, 32, 24, 21, 18, 18, 15, 32, 35, 30, 22, 20, 16, 17, 14,
        26, 29, 26, 20, 17, 15, 14, 13, 25, 28, 25, 19, 17, 14, 12, 13, 22, 26,
        23, 19, 16, 14, 12, 13, 21, 24, 22, 20, 15, 12, 13, 13, 19, 22, 20, 18,
        17, 15, 12, 12, 18, 20, 18, 16, 13, 11, 11, 12, 16, 18, 17, 15, 13, 11,
        9, 10,
        /* Size 16x8 */
        64, 65, 64, 60, 57, 47, 43, 35, 32, 26, 25, 22, 21, 19, 18, 16, 64, 62,
        60, 55, 53, 48, 44, 38, 35, 29, 28, 26, 24, 22, 20, 18, 57, 59, 56, 48,
        43, 38, 36, 32, 30, 26, 25, 23, 22, 20, 18, 17, 38, 41, 41, 38, 34, 29,
        27, 24, 22, 20, 19, 19, 20, 18, 16, 15, 31, 34, 35, 33, 30, 26, 24, 21,
        20, 17, 17, 16, 15, 17, 13, 13, 24, 26, 27, 26, 24, 21, 20, 18, 16, 15,
        14, 14, 12, 15, 11, 11, 21, 23, 24, 24, 23, 20, 18, 18, 17, 14, 12, 12,
        13, 12, 11, 9, 18, 19, 20, 22, 19, 17, 16, 15, 14, 13, 13, 13, 13, 12,
        12, 10,
        /* Size 16x32 */
        64, 65, 64, 60, 57, 47, 38, 35, 31, 26, 24, 22, 21, 19, 18, 17, 65, 64,
        63, 60, 58, 48, 40, 36, 33, 27, 25, 23, 22, 20, 19, 18, 65, 63, 62, 61,
        59, 50, 41, 38, 34, 28, 26, 24, 23, 21, 19, 18, 65, 63, 61, 59, 58, 49,
        41, 38, 34, 29, 26, 24, 23, 21, 20, 19, 64, 63, 60, 58, 56, 49, 41, 38,
        35, 29, 27, 25, 24, 22, 20, 19, 64, 62, 58, 55, 53, 48, 41, 38, 35, 30,
        27, 26, 24, 22, 20, 20, 60, 60, 55, 52, 48, 43, 38, 35, 33, 28, 26, 25,
        24, 23, 22, 19, 59, 59, 55, 50, 46, 41, 36, 34, 31, 27, 25, 24, 22, 22,
        21, 21, 57, 59, 53, 48, 43, 38, 34, 32, 30, 26, 24, 22, 23, 20, 19, 20,
        52, 55, 51, 46, 41, 35, 32, 30, 28, 24, 23, 22, 21, 20, 20, 17, 47, 50,
        48, 43, 38, 33, 29, 27, 26, 23, 21, 22, 20, 20, 17, 19, 45, 48, 46, 41,
        37, 32, 28, 26, 25, 22, 21, 19, 21, 18, 20, 16, 43, 46, 44, 40, 36, 31,
        27, 25, 24, 21, 20, 20, 18, 20, 16, 19, 38, 41, 41, 38, 34, 29, 25, 24,
        22, 20, 19, 19, 19, 16, 19, 14, 35, 38, 38, 35, 32, 27, 24, 22, 21, 19,
        18, 17, 18, 18, 15, 18, 33, 36, 37, 34, 31, 26, 23, 22, 20, 18, 17, 17,
        15, 16, 18, 13, 32, 34, 35, 33, 30, 26, 22, 21, 20, 17, 16, 17, 17, 15,
        14, 17, 29, 31, 32, 30, 28, 24, 21, 20, 18, 16, 16, 15, 15, 16, 16, 12,
        26, 28, 29, 28, 26, 23, 20, 19, 17, 15, 15, 15, 14, 13, 13, 16, 25, 28,
        29, 27, 26, 22, 20, 18, 17, 15, 14, 15, 15, 16, 14, 11, 25, 27, 28, 27,
        25, 22, 19, 18, 17, 15, 14, 13, 12, 12, 13, 15, 23, 25, 27, 25, 24, 21,
        19, 18, 16, 14, 14, 13, 13, 14, 12, 10, 22, 25, 26, 25, 23, 20, 19, 18,
        16, 14, 14, 14, 12, 12, 13, 14, 21, 24, 25, 25, 23, 20, 20, 18, 15, 15,
        14, 13, 12, 11, 11, 9, 21, 23, 24, 24, 22, 20, 20, 16, 15, 15, 12, 13,
        13, 12, 13, 13, 20, 22, 23, 23, 20, 19, 19, 16, 17, 14, 12, 13, 11, 11,
        9, 9, 19, 21, 22, 22, 20, 21, 18, 16, 17, 13, 15, 11, 12, 12, 12, 13,
        18, 20, 21, 21, 19, 20, 16, 18, 15, 13, 14, 10, 13, 10, 11, 7, 18, 19,
        20, 21, 18, 19, 16, 18, 13, 16, 11, 12, 11, 11, 12, 12, 17, 19, 19, 20,
        18, 19, 15, 17, 13, 15, 11, 14, 9, 12, 10, 10, 16, 18, 18, 19, 17, 18,
        15, 17, 13, 15, 11, 13, 9, 12, 10, 12, 16, 18, 17, 19, 19, 15, 18, 13,
        16, 11, 15, 9, 13, 8, 11, 9,
        /* Size 32x16 */
        64, 65, 65, 65, 64, 64, 60, 59, 57, 52, 47, 45, 43, 38, 35, 33, 32, 29,
        26, 25, 25, 23, 22, 21, 21, 20, 19, 18, 18, 17, 16, 16, 65, 64, 63, 63,
        63, 62, 60, 59, 59, 55, 50, 48, 46, 41, 38, 36, 34, 31, 28, 28, 27, 25,
        25, 24, 23, 22, 21, 20, 19, 19, 18, 18, 64, 63, 62, 61, 60, 58, 55, 55,
        53, 51, 48, 46, 44, 41, 38, 37, 35, 32, 29, 29, 28, 27, 26, 25, 24, 23,
        22, 21, 20, 19, 18, 17, 60, 60, 61, 59, 58, 55, 52, 50, 48, 46, 43, 41,
        40, 38, 35, 34, 33, 30, 28, 27, 27, 25, 25, 25, 24, 23, 22, 21, 21, 20,
        19, 19, 57, 58, 59, 58, 56, 53, 48, 46, 43, 41, 38, 37, 36, 34, 32, 31,
        30, 28, 26, 26, 25, 24, 23, 23, 22, 20, 20, 19, 18, 18, 17, 19, 47, 48,
        50, 49, 49, 48, 43, 41, 38, 35, 33, 32, 31, 29, 27, 26, 26, 24, 23, 22,
        22, 21, 20, 20, 20, 19, 21, 20, 19, 19, 18, 15, 38, 40, 41, 41, 41, 41,
        38, 36, 34, 32, 29, 28, 27, 25, 24, 23, 22, 21, 20, 20, 19, 19, 19, 20,
        20, 19, 18, 16, 16, 15, 15, 18, 35, 36, 38, 38, 38, 38, 35, 34, 32, 30,
        27, 26, 25, 24, 22, 22, 21, 20, 19, 18, 18, 18, 18, 18, 16, 16, 16, 18,
        18, 17, 17, 13, 31, 33, 34, 34, 35, 35, 33, 31, 30, 28, 26, 25, 24, 22,
        21, 20, 20, 18, 17, 17, 17, 16, 16, 15, 15, 17, 17, 15, 13, 13, 13, 16,
        26, 27, 28, 29, 29, 30, 28, 27, 26, 24, 23, 22, 21, 20, 19, 18, 17, 16,
        15, 15, 15, 14, 14, 15, 15, 14, 13, 13, 16, 15, 15, 11, 24, 25, 26, 26,
        27, 27, 26, 25, 24, 23, 21, 21, 20, 19, 18, 17, 16, 16, 15, 14, 14, 14,
        14, 14, 12, 12, 15, 14, 11, 11, 11, 15, 22, 23, 24, 24, 25, 26, 25, 24,
        22, 22, 22, 19, 20, 19, 17, 17, 17, 15, 15, 15, 13, 13, 14, 13, 13, 13,
        11, 10, 12, 14, 13, 9, 21, 22, 23, 23, 24, 24, 24, 22, 23, 21, 20, 21,
        18, 19, 18, 15, 17, 15, 14, 15, 12, 13, 12, 12, 13, 11, 12, 13, 11, 9,
        9, 13, 19, 20, 21, 21, 22, 22, 23, 22, 20, 20, 20, 18, 20, 16, 18, 16,
        15, 16, 13, 16, 12, 14, 12, 11, 12, 11, 12, 10, 11, 12, 12, 8, 18, 19,
        19, 20, 20, 20, 22, 21, 19, 20, 17, 20, 16, 19, 15, 18, 14, 16, 13, 14,
        13, 12, 13, 11, 13, 9, 12, 11, 12, 10, 10, 11, 17, 18, 18, 19, 19, 20,
        19, 21, 20, 17, 19, 16, 19, 14, 18, 13, 17, 12, 16, 11, 15, 10, 14, 9,
        13, 9, 13, 7, 12, 10, 12, 9,
        /* Size 4x16 */
        65, 47, 26, 19, 63, 50, 28, 21, 63, 49, 29, 22, 60, 43, 28, 23, 59, 38,
        26, 20, 50, 33, 23, 20, 46, 31, 21, 20, 38, 27, 19, 18, 34, 26, 17, 15,
        28, 23, 15, 13, 27, 22, 15, 12, 25, 20, 14, 12, 23, 20, 15, 12, 21, 21,
        13, 12, 19, 19, 16, 11, 18, 18, 15, 12,
        /* Size 16x4 */
        65, 63, 63, 60, 59, 50, 46, 38, 34, 28, 27, 25, 23, 21, 19, 18, 47, 50,
        49, 43, 38, 33, 31, 27, 26, 23, 22, 20, 20, 21, 19, 18, 26, 28, 29, 28,
        26, 23, 21, 19, 17, 15, 15, 14, 15, 13, 16, 15, 19, 21, 22, 23, 20, 20,
        20, 18, 15, 13, 12, 12, 12, 12, 11, 12,
        /* Size 8x32 */
        64, 64, 57, 38, 31, 24, 21, 18, 65, 63, 58, 40, 33, 25, 22, 19, 65, 62,
        59, 41, 34, 26, 23, 19, 65, 61, 58, 41, 34, 26, 23, 20, 64, 60, 56, 41,
        35, 27, 24, 20, 64, 58, 53, 41, 35, 27, 24, 20, 60, 55, 48, 38, 33, 26,
        24, 22, 59, 55, 46, 36, 31, 25, 22, 21, 57, 53, 43, 34, 30, 24, 23, 19,
        52, 51, 41, 32, 28, 23, 21, 20, 47, 48, 38, 29, 26, 21, 20, 17, 45, 46,
        37, 28, 25, 21, 21, 20, 43, 44, 36, 27, 24, 20, 18, 16, 38, 41, 34, 25,
        22, 19, 19, 19, 35, 38, 32, 24, 21, 18, 18, 15, 33, 37, 31, 23, 20, 17,
        15, 18, 32, 35, 30, 22, 20, 16, 17, 14, 29, 32, 28, 21, 18, 16, 15, 16,
        26, 29, 26, 20, 17, 15, 14, 13, 25, 29, 26, 20, 17, 14, 15, 14, 25, 28,
        25, 19, 17, 14, 12, 13, 23, 27, 24, 19, 16, 14, 13, 12, 22, 26, 23, 19,
        16, 14, 12, 13, 21, 25, 23, 20, 15, 14, 12, 11, 21, 24, 22, 20, 15, 12,
        13, 13, 20, 23, 20, 19, 17, 12, 11, 9, 19, 22, 20, 18, 17, 15, 12, 12,
        18, 21, 19, 16, 15, 14, 13, 11, 18, 20, 18, 16, 13, 11, 11, 12, 17, 19,
        18, 15, 13, 11, 9, 10, 16, 18, 17, 15, 13, 11, 9, 10, 16, 17, 19, 18,
        16, 15, 13, 11,
        /* Size 32x8 */
        64, 65, 65, 65, 64, 64, 60, 59, 57, 52, 47, 45, 43, 38, 35, 33, 32, 29,
        26, 25, 25, 23, 22, 21, 21, 20, 19, 18, 18, 17, 16, 16, 64, 63, 62, 61,
        60, 58, 55, 55, 53, 51, 48, 46, 44, 41, 38, 37, 35, 32, 29, 29, 28, 27,
        26, 25, 24, 23, 22, 21, 20, 19, 18, 17, 57, 58, 59, 58, 56, 53, 48, 46,
        43, 41, 38, 37, 36, 34, 32, 31, 30, 28, 26, 26, 25, 24, 23, 23, 22, 20,
        20, 19, 18, 18, 17, 19, 38, 40, 41, 41, 41, 41, 38, 36, 34, 32, 29, 28,
        27, 25, 24, 23, 22, 21, 20, 20, 19, 19, 19, 20, 20, 19, 18, 16, 16, 15,
        15, 18, 31, 33, 34, 34, 35, 35, 33, 31, 30, 28, 26, 25, 24, 22, 21, 20,
        20, 18, 17, 17, 17, 16, 16, 15, 15, 17, 17, 15, 13, 13, 13, 16, 24, 25,
        26, 26, 27, 27, 26, 25, 24, 23, 21, 21, 20, 19, 18, 17, 16, 16, 15, 14,
        14, 14, 14, 14, 12, 12, 15, 14, 11, 11, 11, 15, 21, 22, 23, 23, 24, 24,
        24, 22, 23, 21, 20, 21, 18, 19, 18, 15, 17, 15, 14, 15, 12, 13, 12, 12,
        13, 11, 12, 13, 11, 9, 9, 13, 18, 19, 19, 20, 20, 20, 22, 21, 19, 20,
        17, 20, 16, 19, 15, 18, 14, 16, 13, 14, 13, 12, 13, 11, 13, 9, 12, 11,
        12, 10, 10, 11 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        41, 33, 33, 20,
#endif
        /* Size 4x4 */
        58, 45, 36, 30, 45, 34, 29, 26, 36, 29, 23, 24, 30, 26, 24, 18,
        /* Size 8x8 */
        65, 53, 44, 41, 36, 32, 29, 27, 53, 43, 44, 44, 39, 36, 33, 29, 44, 44,
        38, 36, 33, 31, 30, 30, 41, 44, 36, 31, 28, 26, 27, 28, 36, 39, 33, 28,
        25, 23, 22, 24, 32, 36, 31, 26, 23, 21, 22, 19, 29, 33, 30, 27, 22, 22,
        19, 16, 27, 29, 30, 28, 24, 19, 16, 17,
        /* Size 16x16 */
        64, 67, 61, 49, 42, 42, 41, 38, 36, 32, 32, 30, 28, 27, 26, 25, 67, 64,
        58, 48, 44, 46, 44, 41, 39, 36, 35, 33, 31, 30, 28, 27, 61, 58, 53, 46,
        43, 45, 44, 41, 40, 36, 36, 34, 33, 32, 30, 29, 49, 48, 46, 42, 41, 42,
        41, 39, 38, 36, 35, 34, 34, 33, 32, 31, 42, 44, 43, 41, 39, 38, 38, 37,
        36, 34, 33, 32, 31, 31, 28, 27, 42, 46, 45, 42, 38, 35, 34, 33, 32, 30,
        30, 30, 28, 30, 27, 26, 41, 44, 44, 41, 38, 34, 33, 32, 31, 29, 29, 27,
        28, 27, 27, 24, 38, 41, 41, 39, 37, 33, 32, 29, 28, 26, 26, 26, 27, 24,
        27, 23, 36, 39, 40, 38, 36, 32, 31, 28, 27, 25, 25, 24, 23, 22, 26, 21,
        32, 36, 36, 36, 34, 30, 29, 26, 25, 23, 23, 23, 22, 23, 25, 20, 32, 35,
        36, 35, 33, 30, 29, 26, 25, 23, 22, 22, 22, 23, 22, 19, 30, 33, 34, 34,
        32, 30, 27, 26, 24, 23, 22, 21, 21, 20, 20, 17, 28, 31, 33, 34, 31, 28,
        28, 27, 23, 22, 22, 21, 19, 19, 17, 17, 27, 30, 32, 33, 31, 30, 27, 24,
        22, 23, 23, 20, 19, 18, 18, 15, 26, 28, 30, 32, 28, 27, 27, 27, 26, 25,
        22, 20, 17, 18, 17, 17, 25, 27, 29, 31, 27, 26, 24, 23, 21, 20, 19, 17,
        17, 15, 17, 16,
        /* Size 32x32 */
        64, 66, 67, 64, 61, 56, 49, 46, 42, 42, 42, 41, 41, 39, 38, 37, 36, 34,
        32, 32, 32, 30, 30, 29, 28, 28, 27, 27, 26, 26, 25, 25, 66, 65, 66, 62,
        59, 53, 48, 46, 43, 44, 44, 43, 42, 41, 39, 39, 38, 36, 34, 34, 33, 32,
        32, 31, 30, 29, 29, 28, 28, 27, 27, 26, 67, 66, 64, 61, 58, 51, 48, 46,
        44, 45, 46, 45, 44, 43, 41, 40, 39, 38, 36, 35, 35, 34, 33, 32, 31, 31,
        30, 29, 28, 28, 27, 26, 64, 62, 61, 58, 55, 49, 47, 46, 44, 45, 46, 45,
        44, 43, 41, 41, 40, 38, 36, 36, 35, 34, 33, 33, 32, 31, 30, 30, 29, 28,
        28, 28, 61, 59, 58, 55, 53, 47, 46, 45, 43, 44, 45, 45, 44, 43, 41, 41,
        40, 38, 36, 36, 36, 34, 34, 33, 33, 32, 32, 31, 30, 29, 29, 27, 56, 53,
        51, 49, 47, 43, 43, 43, 43, 44, 45, 45, 45, 44, 42, 42, 41, 39, 38, 37,
        37, 36, 35, 34, 34, 33, 31, 31, 30, 29, 29, 29, 49, 48, 48, 47, 46, 43,
        42, 41, 41, 41, 42, 41, 41, 41, 39, 39, 38, 37, 36, 35, 35, 34, 34, 34,
        34, 33, 33, 33, 32, 31, 31, 28, 46, 46, 46, 46, 45, 43, 41, 41, 40, 40,
        40, 40, 39, 39, 38, 38, 37, 36, 35, 34, 34, 33, 33, 33, 31, 31, 30, 32,
        31, 31, 30, 32, 42, 43, 44, 44, 43, 43, 41, 40, 39, 39, 38, 38, 38, 38,
        37, 37, 36, 35, 34, 34, 33, 33, 32, 31, 31, 32, 31, 30, 28, 27, 27, 30,
        42, 44, 45, 45, 44, 44, 41, 40, 39, 38, 37, 37, 36, 36, 35, 34, 34, 33,
        32, 32, 32, 31, 30, 31, 31, 30, 28, 28, 30, 30, 29, 25, 42, 44, 46, 46,
        45, 45, 42, 40, 38, 37, 35, 35, 34, 33, 33, 33, 32, 31, 30, 30, 30, 30,
        30, 30, 28, 28, 30, 30, 27, 26, 26, 29, 41, 43, 45, 45, 45, 45, 41, 40,
        38, 37, 35, 34, 34, 33, 32, 32, 31, 31, 30, 29, 29, 29, 29, 28, 29, 29,
        27, 26, 28, 29, 28, 25, 41, 42, 44, 44, 44, 45, 41, 39, 38, 36, 34, 34,
        33, 32, 32, 31, 31, 30, 29, 29, 29, 28, 27, 28, 28, 26, 27, 29, 27, 25,
        24, 28, 39, 41, 43, 43, 43, 44, 41, 39, 38, 36, 33, 33, 32, 31, 30, 29,
        29, 28, 27, 27, 27, 26, 27, 27, 26, 27, 28, 24, 25, 28, 27, 23, 38, 39,
        41, 41, 41, 42, 39, 38, 37, 35, 33, 32, 32, 30, 29, 28, 28, 27, 26, 26,
        26, 26, 26, 25, 27, 26, 24, 27, 27, 23, 23, 27, 37, 39, 40, 41, 41, 42,
        39, 38, 37, 34, 33, 32, 31, 29, 28, 28, 27, 27, 26, 25, 25, 25, 25, 25,
        25, 23, 26, 24, 23, 26, 26, 21, 36, 38, 39, 40, 40, 41, 38, 37, 36, 34,
        32, 31, 31, 29, 28, 27, 27, 26, 25, 25, 25, 24, 24, 25, 23, 25, 22, 23,
        26, 22, 21, 26, 34, 36, 38, 38, 38, 39, 37, 36, 35, 33, 31, 31, 30, 28,
        27, 27, 26, 25, 24, 24, 24, 23, 24, 22, 25, 22, 24, 25, 21, 24, 25, 20,
        32, 34, 36, 36, 36, 38, 36, 35, 34, 32, 30, 30, 29, 27, 26, 26, 25, 24,
        23, 23, 23, 22, 23, 23, 22, 22, 23, 21, 25, 22, 20, 25, 32, 34, 35, 36,
        36, 37, 35, 34, 34, 32, 30, 29, 29, 27, 26, 25, 25, 24, 23, 23, 23, 22,
        21, 23, 21, 23, 20, 24, 20, 22, 24, 18, 32, 33, 35, 35, 36, 37, 35, 34,
        33, 32, 30, 29, 29, 27, 26, 25, 25, 24, 23, 23, 22, 22, 22, 20, 22, 20,
        23, 19, 22, 22, 19, 24, 30, 32, 34, 34, 34, 36, 34, 33, 33, 31, 30, 29,
        28, 26, 26, 25, 24, 23, 22, 22, 22, 21, 21, 21, 20, 22, 19, 22, 20, 19,
        23, 18, 30, 32, 33, 33, 34, 35, 34, 33, 32, 30, 30, 29, 27, 27, 26, 25,
        24, 24, 23, 21, 22, 21, 21, 21, 21, 19, 20, 20, 20, 21, 17, 23, 29, 31,
        32, 33, 33, 34, 34, 33, 31, 31, 30, 28, 28, 27, 25, 25, 25, 22, 23, 23,
        20, 21, 21, 20, 21, 20, 19, 19, 20, 18, 22, 16, 28, 30, 31, 32, 33, 34,
        34, 31, 31, 31, 28, 29, 28, 26, 27, 25, 23, 25, 22, 21, 22, 20, 21, 21,
        19, 21, 19, 20, 17, 21, 17, 21, 28, 29, 31, 31, 32, 33, 33, 31, 32, 30,
        28, 29, 26, 27, 26, 23, 25, 22, 22, 23, 20, 22, 19, 20, 21, 19, 20, 19,
        20, 16, 21, 16, 27, 29, 30, 30, 32, 31, 33, 30, 31, 28, 30, 27, 27, 28,
        24, 26, 22, 24, 23, 20, 23, 19, 20, 19, 19, 20, 18, 20, 18, 20, 15, 20,
        27, 28, 29, 30, 31, 31, 33, 32, 30, 28, 30, 26, 29, 24, 27, 24, 23, 25,
        21, 24, 19, 22, 20, 19, 20, 19, 20, 18, 20, 18, 19, 14, 26, 28, 28, 29,
        30, 30, 32, 31, 28, 30, 27, 28, 27, 25, 27, 23, 26, 21, 25, 20, 22, 20,
        20, 20, 17, 20, 18, 20, 17, 19, 17, 19, 26, 27, 28, 28, 29, 29, 31, 31,
        27, 30, 26, 29, 25, 28, 23, 26, 22, 24, 22, 22, 22, 19, 21, 18, 21, 16,
        20, 18, 19, 17, 19, 17, 25, 27, 27, 28, 29, 29, 31, 30, 27, 29, 26, 28,
        24, 27, 23, 26, 21, 25, 20, 24, 19, 23, 17, 22, 17, 21, 15, 19, 17, 19,
        16, 19, 25, 26, 26, 28, 27, 29, 28, 32, 30, 25, 29, 25, 28, 23, 27, 21,
        26, 20, 25, 18, 24, 18, 23, 16, 21, 16, 20, 14, 19, 17, 19, 16,
        /* Size 4x8 */
        65, 44, 34, 29, 51, 45, 38, 31, 45, 37, 32, 29, 43, 34, 27, 25, 38, 31,
        24, 24, 34, 29, 22, 22, 31, 28, 22, 19, 28, 29, 24, 20,
        /* Size 8x4 */
        65, 51, 45, 43, 38, 34, 31, 28, 44, 45, 37, 34, 31, 29, 28, 29, 34, 38,
        32, 27, 24, 22, 22, 24, 29, 31, 29, 25, 24, 22, 19, 20,
        /* Size 8x16 */
        64, 55, 43, 39, 36, 31, 28, 26, 67, 51, 45, 43, 39, 34, 31, 29, 61, 48,
        44, 43, 40, 35, 33, 30, 49, 43, 41, 41, 38, 34, 34, 32, 42, 43, 39, 38,
        36, 33, 32, 28, 42, 45, 38, 33, 32, 29, 29, 27, 41, 44, 38, 32, 31, 28,
        26, 25, 38, 42, 37, 30, 28, 25, 26, 23, 36, 41, 36, 29, 27, 24, 26, 23,
        32, 37, 34, 27, 25, 22, 22, 22, 32, 37, 33, 27, 25, 22, 20, 21, 30, 34,
        32, 27, 24, 22, 20, 21, 29, 33, 31, 28, 23, 20, 21, 20, 27, 32, 29, 27,
        25, 23, 19, 20, 26, 30, 28, 25, 21, 19, 19, 20, 25, 29, 27, 24, 21, 19,
        17, 17,
        /* Size 16x8 */
        64, 67, 61, 49, 42, 42, 41, 38, 36, 32, 32, 30, 29, 27, 26, 25, 55, 51,
        48, 43, 43, 45, 44, 42, 41, 37, 37, 34, 33, 32, 30, 29, 43, 45, 44, 41,
        39, 38, 38, 37, 36, 34, 33, 32, 31, 29, 28, 27, 39, 43, 43, 41, 38, 33,
        32, 30, 29, 27, 27, 27, 28, 27, 25, 24, 36, 39, 40, 38, 36, 32, 31, 28,
        27, 25, 25, 24, 23, 25, 21, 21, 31, 34, 35, 34, 33, 29, 28, 25, 24, 22,
        22, 22, 20, 23, 19, 19, 28, 31, 33, 34, 32, 29, 26, 26, 26, 22, 20, 20,
        21, 19, 19, 17, 26, 29, 30, 32, 28, 27, 25, 23, 23, 22, 21, 21, 20, 20,
        20, 17,
        /* Size 16x32 */
        64, 66, 55, 49, 43, 42, 39, 38, 36, 32, 31, 29, 28, 27, 26, 25, 65, 65,
        53, 48, 44, 44, 41, 39, 38, 34, 33, 31, 30, 29, 28, 27, 67, 64, 51, 48,
        45, 45, 43, 41, 39, 36, 34, 33, 31, 30, 29, 27, 64, 60, 49, 47, 44, 45,
        43, 41, 40, 36, 34, 33, 32, 30, 29, 29, 61, 57, 48, 46, 44, 45, 43, 41,
        40, 37, 35, 34, 33, 32, 30, 28, 55, 51, 44, 43, 43, 45, 44, 42, 41, 38,
        36, 35, 33, 31, 30, 30, 49, 48, 43, 42, 41, 41, 41, 39, 38, 36, 34, 34,
        34, 34, 32, 29, 46, 46, 43, 41, 40, 40, 39, 38, 37, 35, 33, 33, 31, 32,
        32, 32, 42, 44, 43, 41, 39, 38, 38, 37, 36, 34, 33, 31, 32, 30, 28, 31,
        42, 45, 44, 41, 39, 37, 36, 35, 34, 32, 31, 31, 30, 29, 30, 26, 42, 45,
        45, 41, 38, 35, 33, 33, 32, 30, 29, 30, 29, 30, 27, 29, 41, 45, 45, 41,
        38, 35, 33, 32, 31, 30, 29, 28, 30, 27, 29, 25, 41, 44, 44, 41, 38, 34,
        32, 31, 31, 29, 28, 28, 26, 29, 25, 29, 39, 43, 43, 41, 38, 34, 31, 30,
        29, 27, 27, 27, 27, 25, 28, 23, 38, 41, 42, 39, 37, 33, 30, 29, 28, 26,
        25, 25, 26, 27, 23, 28, 37, 40, 41, 39, 37, 33, 29, 28, 27, 26, 25, 25,
        24, 25, 27, 22, 36, 39, 41, 38, 36, 32, 29, 28, 27, 25, 24, 25, 26, 24,
        23, 27, 34, 38, 39, 37, 35, 31, 28, 27, 26, 24, 23, 23, 23, 24, 24, 20,
        32, 36, 37, 35, 34, 30, 27, 26, 25, 23, 22, 23, 22, 21, 22, 25, 32, 35,
        37, 35, 34, 30, 27, 26, 25, 23, 22, 23, 23, 24, 22, 19, 32, 35, 37, 35,
        33, 30, 27, 26, 25, 23, 22, 21, 20, 20, 21, 24, 31, 34, 35, 34, 33, 29,
        27, 26, 24, 22, 21, 21, 21, 22, 20, 18, 30, 33, 34, 34, 32, 28, 27, 26,
        24, 22, 22, 21, 20, 20, 21, 23, 29, 32, 34, 34, 32, 28, 28, 26, 23, 23,
        22, 20, 20, 19, 19, 16, 29, 32, 33, 34, 31, 28, 28, 24, 23, 23, 20, 21,
        21, 20, 20, 22, 28, 31, 32, 33, 29, 28, 28, 24, 25, 22, 20, 21, 19, 19,
        16, 16, 27, 30, 32, 32, 29, 30, 27, 24, 25, 21, 23, 19, 19, 20, 20, 21,
        27, 30, 31, 32, 28, 30, 25, 27, 24, 21, 22, 18, 20, 18, 18, 14, 26, 29,
        30, 31, 28, 29, 25, 27, 21, 25, 19, 20, 19, 18, 20, 20, 26, 28, 29, 31,
        28, 29, 25, 27, 21, 24, 19, 22, 16, 20, 17, 17, 25, 28, 29, 30, 27, 28,
        24, 26, 21, 24, 19, 22, 17, 19, 17, 19, 25, 28, 27, 31, 29, 25, 28, 22,
        26, 19, 24, 16, 21, 15, 19, 16,
        /* Size 32x16 */
        64, 65, 67, 64, 61, 55, 49, 46, 42, 42, 42, 41, 41, 39, 38, 37, 36, 34,
        32, 32, 32, 31, 30, 29, 29, 28, 27, 27, 26, 26, 25, 25, 66, 65, 64, 60,
        57, 51, 48, 46, 44, 45, 45, 45, 44, 43, 41, 40, 39, 38, 36, 35, 35, 34,
        33, 32, 32, 31, 30, 30, 29, 28, 28, 28, 55, 53, 51, 49, 48, 44, 43, 43,
        43, 44, 45, 45, 44, 43, 42, 41, 41, 39, 37, 37, 37, 35, 34, 34, 33, 32,
        32, 31, 30, 29, 29, 27, 49, 48, 48, 47, 46, 43, 42, 41, 41, 41, 41, 41,
        41, 41, 39, 39, 38, 37, 35, 35, 35, 34, 34, 34, 34, 33, 32, 32, 31, 31,
        30, 31, 43, 44, 45, 44, 44, 43, 41, 40, 39, 39, 38, 38, 38, 38, 37, 37,
        36, 35, 34, 34, 33, 33, 32, 32, 31, 29, 29, 28, 28, 28, 27, 29, 42, 44,
        45, 45, 45, 45, 41, 40, 38, 37, 35, 35, 34, 34, 33, 33, 32, 31, 30, 30,
        30, 29, 28, 28, 28, 28, 30, 30, 29, 29, 28, 25, 39, 41, 43, 43, 43, 44,
        41, 39, 38, 36, 33, 33, 32, 31, 30, 29, 29, 28, 27, 27, 27, 27, 27, 28,
        28, 28, 27, 25, 25, 25, 24, 28, 38, 39, 41, 41, 41, 42, 39, 38, 37, 35,
        33, 32, 31, 30, 29, 28, 28, 27, 26, 26, 26, 26, 26, 26, 24, 24, 24, 27,
        27, 27, 26, 22, 36, 38, 39, 40, 40, 41, 38, 37, 36, 34, 32, 31, 31, 29,
        28, 27, 27, 26, 25, 25, 25, 24, 24, 23, 23, 25, 25, 24, 21, 21, 21, 26,
        32, 34, 36, 36, 37, 38, 36, 35, 34, 32, 30, 30, 29, 27, 26, 26, 25, 24,
        23, 23, 23, 22, 22, 23, 23, 22, 21, 21, 25, 24, 24, 19, 31, 33, 34, 34,
        35, 36, 34, 33, 33, 31, 29, 29, 28, 27, 25, 25, 24, 23, 22, 22, 22, 21,
        22, 22, 20, 20, 23, 22, 19, 19, 19, 24, 29, 31, 33, 33, 34, 35, 34, 33,
        31, 31, 30, 28, 28, 27, 25, 25, 25, 23, 23, 23, 21, 21, 21, 20, 21, 21,
        19, 18, 20, 22, 22, 16, 28, 30, 31, 32, 33, 33, 34, 31, 32, 30, 29, 30,
        26, 27, 26, 24, 26, 23, 22, 23, 20, 21, 20, 20, 21, 19, 19, 20, 19, 16,
        17, 21, 27, 29, 30, 30, 32, 31, 34, 32, 30, 29, 30, 27, 29, 25, 27, 25,
        24, 24, 21, 24, 20, 22, 20, 19, 20, 19, 20, 18, 18, 20, 19, 15, 26, 28,
        29, 29, 30, 30, 32, 32, 28, 30, 27, 29, 25, 28, 23, 27, 23, 24, 22, 22,
        21, 20, 21, 19, 20, 16, 20, 18, 20, 17, 17, 19, 25, 27, 27, 29, 28, 30,
        29, 32, 31, 26, 29, 25, 29, 23, 28, 22, 27, 20, 25, 19, 24, 18, 23, 16,
        22, 16, 21, 14, 20, 17, 19, 16,
        /* Size 4x16 */
        66, 42, 32, 27, 64, 45, 36, 30, 57, 45, 37, 32, 48, 41, 36, 34, 44, 38,
        34, 30, 45, 35, 30, 30, 44, 34, 29, 29, 41, 33, 26, 27, 39, 32, 25, 24,
        36, 30, 23, 21, 35, 30, 23, 20, 33, 28, 22, 20, 32, 28, 23, 20, 30, 30,
        21, 20, 29, 29, 25, 18, 28, 28, 24, 19,
        /* Size 16x4 */
        66, 64, 57, 48, 44, 45, 44, 41, 39, 36, 35, 33, 32, 30, 29, 28, 42, 45,
        45, 41, 38, 35, 34, 33, 32, 30, 30, 28, 28, 30, 29, 28, 32, 36, 37, 36,
        34, 30, 29, 26, 25, 23, 23, 22, 23, 21, 25, 24, 27, 30, 32, 34, 30, 30,
        29, 27, 24, 21, 20, 20, 20, 20, 18, 19,
        /* Size 8x32 */
        64, 55, 43, 39, 36, 31, 28, 26, 65, 53, 44, 41, 38, 33, 30, 28, 67, 51,
        45, 43, 39, 34, 31, 29, 64, 49, 44, 43, 40, 34, 32, 29, 61, 48, 44, 43,
        40, 35, 33, 30, 55, 44, 43, 44, 41, 36, 33, 30, 49, 43, 41, 41, 38, 34,
        34, 32, 46, 43, 40, 39, 37, 33, 31, 32, 42, 43, 39, 38, 36, 33, 32, 28,
        42, 44, 39, 36, 34, 31, 30, 30, 42, 45, 38, 33, 32, 29, 29, 27, 41, 45,
        38, 33, 31, 29, 30, 29, 41, 44, 38, 32, 31, 28, 26, 25, 39, 43, 38, 31,
        29, 27, 27, 28, 38, 42, 37, 30, 28, 25, 26, 23, 37, 41, 37, 29, 27, 25,
        24, 27, 36, 41, 36, 29, 27, 24, 26, 23, 34, 39, 35, 28, 26, 23, 23, 24,
        32, 37, 34, 27, 25, 22, 22, 22, 32, 37, 34, 27, 25, 22, 23, 22, 32, 37,
        33, 27, 25, 22, 20, 21, 31, 35, 33, 27, 24, 21, 21, 20, 30, 34, 32, 27,
        24, 22, 20, 21, 29, 34, 32, 28, 23, 22, 20, 19, 29, 33, 31, 28, 23, 20,
        21, 20, 28, 32, 29, 28, 25, 20, 19, 16, 27, 32, 29, 27, 25, 23, 19, 20,
        27, 31, 28, 25, 24, 22, 20, 18, 26, 30, 28, 25, 21, 19, 19, 20, 26, 29,
        28, 25, 21, 19, 16, 17, 25, 29, 27, 24, 21, 19, 17, 17, 25, 27, 29, 28,
        26, 24, 21, 19,
        /* Size 32x8 */
        64, 65, 67, 64, 61, 55, 49, 46, 42, 42, 42, 41, 41, 39, 38, 37, 36, 34,
        32, 32, 32, 31, 30, 29, 29, 28, 27, 27, 26, 26, 25, 25, 55, 53, 51, 49,
        48, 44, 43, 43, 43, 44, 45, 45, 44, 43, 42, 41, 41, 39, 37, 37, 37, 35,
        34, 34, 33, 32, 32, 31, 30, 29, 29, 27, 43, 44, 45, 44, 44, 43, 41, 40,
        39, 39, 38, 38, 38, 38, 37, 37, 36, 35, 34, 34, 33, 33, 32, 32, 31, 29,
        29, 28, 28, 28, 27, 29, 39, 41, 43, 43, 43, 44, 41, 39, 38, 36, 33, 33,
        32, 31, 30, 29, 29, 28, 27, 27, 27, 27, 27, 28, 28, 28, 27, 25, 25, 25,
        24, 28, 36, 38, 39, 40, 40, 41, 38, 37, 36, 34, 32, 31, 31, 29, 28, 27,
        27, 26, 25, 25, 25, 24, 24, 23, 23, 25, 25, 24, 21, 21, 21, 26, 31, 33,
        34, 34, 35, 36, 34, 33, 33, 31, 29, 29, 28, 27, 25, 25, 24, 23, 22, 22,
        22, 21, 22, 22, 20, 20, 23, 22, 19, 19, 19, 24, 28, 30, 31, 32, 33, 33,
        34, 31, 32, 30, 29, 30, 26, 27, 26, 24, 26, 23, 22, 23, 20, 21, 20, 20,
        21, 19, 19, 20, 19, 16, 17, 21, 26, 28, 29, 29, 30, 30, 32, 32, 28, 30,
        27, 29, 25, 28, 23, 27, 23, 24, 22, 22, 21, 20, 21, 19, 20, 16, 20, 18,
        20, 17, 17, 19 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        52, 25, 25, 13,
#endif
        /* Size 4x4 */
        63, 50, 30, 22, 50, 32, 23, 19, 30, 23, 16, 16, 22, 19, 16, 11,
        /* Size 8x8 */
        64, 63, 55, 43, 33, 26, 22, 19, 63, 58, 52, 44, 35, 28, 24, 21, 55, 52,
        40, 34, 29, 24, 22, 21, 43, 44, 34, 28, 24, 20, 20, 19, 33, 35, 29, 24,
        20, 17, 15, 16, 26, 28, 24, 20, 17, 15, 14, 12, 22, 24, 22, 20, 15, 14,
        12, 10, 19, 21, 21, 19, 16, 12, 10, 10,
        /* Size 16x16 */
        64, 65, 65, 63, 57, 52, 43, 38, 33, 29, 26, 24, 22, 20, 19, 18, 65, 64,
        63, 62, 59, 55, 47, 41, 36, 31, 28, 26, 24, 22, 20, 19, 65, 63, 61, 59,
        56, 53, 46, 41, 37, 32, 29, 27, 25, 23, 21, 20, 63, 62, 59, 56, 51, 49,
        44, 40, 36, 32, 29, 27, 26, 24, 23, 21, 57, 59, 56, 51, 42, 41, 37, 34,
        31, 28, 26, 24, 23, 23, 20, 19, 52, 55, 53, 49, 41, 38, 34, 32, 29, 26,
        24, 23, 21, 22, 19, 18, 43, 47, 46, 44, 37, 34, 29, 27, 25, 23, 22, 20,
        20, 19, 19, 17, 38, 41, 41, 40, 34, 32, 27, 25, 23, 21, 20, 19, 19, 17,
        19, 15, 33, 36, 37, 36, 31, 29, 25, 23, 21, 19, 18, 17, 16, 16, 18, 14,
        29, 31, 32, 32, 28, 26, 23, 21, 19, 17, 16, 16, 15, 16, 16, 13, 26, 28,
        29, 29, 26, 24, 22, 20, 18, 16, 15, 15, 15, 15, 15, 12, 24, 26, 27, 27,
        24, 23, 20, 19, 17, 16, 15, 14, 14, 13, 13, 11, 22, 24, 25, 26, 23, 21,
        20, 19, 16, 15, 15, 14, 13, 12, 11, 10, 20, 22, 23, 24, 23, 22, 19, 17,
        16, 16, 15, 13, 12, 12, 11, 9, 19, 20, 21, 23, 20, 19, 19, 19, 18, 16,
        15, 13, 11, 11, 11, 11, 18, 19, 20, 21, 19, 18, 17, 15, 14, 13, 12, 11,
        10, 9, 11, 10,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 64, 63, 60, 57, 54, 52, 47, 43, 41, 38, 35, 33, 31,
        29, 27, 26, 25, 24, 23, 22, 21, 20, 20, 19, 18, 18, 17, 65, 64, 64, 64,
        64, 63, 62, 60, 59, 55, 53, 48, 45, 43, 40, 36, 35, 33, 30, 28, 27, 26,
        25, 24, 23, 22, 21, 21, 20, 19, 19, 18, 65, 64, 64, 63, 63, 63, 62, 61,
        59, 56, 55, 50, 47, 45, 41, 38, 36, 34, 31, 29, 28, 27, 26, 25, 24, 23,
        22, 21, 20, 20, 19, 18, 65, 64, 63, 63, 63, 62, 61, 60, 59, 56, 54, 50,
        47, 45, 41, 38, 37, 34, 32, 30, 28, 27, 26, 25, 24, 23, 23, 22, 21, 20,
        19, 19, 65, 64, 63, 63, 61, 60, 59, 58, 56, 54, 53, 49, 46, 44, 41, 38,
        37, 35, 32, 30, 29, 28, 27, 26, 25, 24, 23, 22, 21, 21, 20, 19, 64, 63,
        63, 62, 60, 58, 57, 55, 53, 52, 51, 48, 46, 44, 41, 38, 37, 35, 33, 31,
        29, 28, 27, 26, 25, 24, 23, 22, 22, 21, 20, 20, 63, 62, 62, 61, 59, 57,
        56, 54, 51, 50, 49, 46, 44, 42, 40, 37, 36, 34, 32, 30, 29, 28, 27, 26,
        26, 25, 24, 24, 23, 22, 21, 19, 60, 60, 61, 60, 58, 55, 54, 52, 48, 47,
        46, 43, 41, 40, 38, 35, 34, 33, 30, 29, 28, 27, 26, 25, 24, 23, 22, 23,
        22, 22, 21, 21, 57, 59, 59, 59, 56, 53, 51, 48, 42, 41, 41, 38, 37, 36,
        34, 32, 31, 30, 28, 27, 26, 25, 24, 24, 23, 23, 23, 21, 20, 19, 19, 20,
        54, 55, 56, 56, 54, 52, 50, 47, 41, 40, 39, 37, 35, 34, 33, 31, 30, 29,
        27, 26, 25, 24, 24, 23, 23, 22, 21, 20, 21, 21, 20, 17, 52, 53, 55, 54,
        53, 51, 49, 46, 41, 39, 38, 35, 34, 33, 32, 30, 29, 28, 26, 25, 24, 24,
        23, 22, 21, 21, 22, 21, 19, 18, 18, 19, 47, 48, 50, 50, 49, 48, 46, 43,
        38, 37, 35, 32, 31, 30, 29, 27, 27, 26, 24, 23, 23, 22, 22, 21, 21, 21,
        20, 19, 19, 20, 19, 17, 43, 45, 47, 47, 46, 46, 44, 41, 37, 35, 34, 31,
        29, 29, 27, 26, 25, 24, 23, 22, 22, 21, 20, 21, 20, 19, 19, 20, 19, 17,
        17, 19, 41, 43, 45, 45, 44, 44, 42, 40, 36, 34, 33, 30, 29, 28, 26, 25,
        25, 24, 22, 21, 21, 20, 20, 20, 19, 20, 20, 17, 17, 19, 19, 15, 38, 40,
        41, 41, 41, 41, 40, 38, 34, 33, 32, 29, 27, 26, 25, 24, 23, 22, 21, 20,
        20, 19, 19, 18, 19, 18, 17, 19, 19, 16, 15, 18, 35, 36, 38, 38, 38, 38,
        37, 35, 32, 31, 30, 27, 26, 25, 24, 22, 22, 21, 20, 19, 19, 18, 18, 18,
        18, 17, 18, 17, 16, 18, 18, 14, 33, 35, 36, 37, 37, 37, 36, 34, 31, 30,
        29, 27, 25, 25, 23, 22, 21, 20, 19, 19, 18, 18, 17, 17, 16, 18, 16, 16,
        18, 15, 14, 17, 31, 33, 34, 34, 35, 35, 34, 33, 30, 29, 28, 26, 24, 24,
        22, 21, 20, 20, 18, 18, 17, 17, 17, 16, 17, 15, 16, 17, 14, 16, 17, 13,
        29, 30, 31, 32, 32, 33, 32, 30, 28, 27, 26, 24, 23, 22, 21, 20, 19, 18,
        17, 17, 16, 16, 16, 16, 15, 15, 16, 14, 16, 14, 13, 16, 27, 28, 29, 30,
        30, 31, 30, 29, 27, 26, 25, 23, 22, 21, 20, 19, 19, 18, 17, 16, 16, 15,
        15, 15, 15, 15, 14, 16, 13, 14, 16, 12, 26, 27, 28, 28, 29, 29, 29, 28,
        26, 25, 24, 23, 22, 21, 20, 19, 18, 17, 16, 16, 15, 15, 15, 14, 15, 13,
        15, 12, 15, 14, 12, 15, 25, 26, 27, 27, 28, 28, 28, 27, 25, 24, 24, 22,
        21, 20, 19, 18, 18, 17, 16, 15, 15, 15, 14, 14, 13, 14, 13, 14, 13, 12,
        15, 11, 24, 25, 26, 26, 27, 27, 27, 26, 24, 24, 23, 22, 20, 20, 19, 18,
        17, 17, 16, 15, 15, 14, 14, 14, 14, 13, 13, 13, 13, 14, 11, 15, 23, 24,
        25, 25, 26, 26, 26, 25, 24, 23, 22, 21, 21, 20, 18, 18, 17, 16, 16, 15,
        14, 14, 14, 13, 14, 13, 13, 12, 13, 11, 14, 10, 22, 23, 24, 24, 25, 25,
        26, 24, 23, 23, 21, 21, 20, 19, 19, 18, 16, 17, 15, 15, 15, 13, 14, 14,
        13, 13, 12, 13, 11, 13, 10, 14, 21, 22, 23, 23, 24, 24, 25, 23, 23, 22,
        21, 21, 19, 20, 18, 17, 18, 15, 15, 15, 13, 14, 13, 13, 13, 12, 13, 12,
        13, 10, 13, 10, 20, 21, 22, 23, 23, 23, 24, 22, 23, 21, 22, 20, 19, 20,
        17, 18, 16, 16, 16, 14, 15, 13, 13, 13, 12, 13, 12, 13, 11, 12, 9, 13,
        20, 21, 21, 22, 22, 22, 24, 23, 21, 20, 21, 19, 20, 17, 19, 17, 16, 17,
        14, 16, 12, 14, 13, 12, 13, 12, 13, 11, 12, 11, 12, 8, 19, 20, 20, 21,
        21, 22, 23, 22, 20, 21, 19, 19, 19, 17, 19, 16, 18, 14, 16, 13, 15, 13,
        13, 13, 11, 13, 11, 12, 11, 12, 11, 12, 18, 19, 20, 20, 21, 21, 22, 22,
        19, 21, 18, 20, 17, 19, 16, 18, 15, 16, 14, 14, 14, 12, 14, 11, 13, 10,
        12, 11, 12, 10, 12, 10, 18, 19, 19, 19, 20, 20, 21, 21, 19, 20, 18, 19,
        17, 19, 15, 18, 14, 17, 13, 16, 12, 15, 11, 14, 10, 13, 9, 12, 11, 12,
        10, 12, 17, 18, 18, 19, 19, 20, 19, 21, 20, 17, 19, 17, 19, 15, 18, 14,
        17, 13, 16, 12, 15, 11, 15, 10, 14, 10, 13, 8, 12, 10, 12, 9,
        /* Size 4x8 */
        64, 48, 29, 21, 62, 48, 32, 23, 56, 37, 27, 21, 45, 30, 22, 17, 34, 26,
        18, 17, 27, 22, 16, 14, 23, 21, 15, 12, 20, 20, 16, 12,
        /* Size 8x4 */
        64, 62, 56, 45, 34, 27, 23, 20, 48, 48, 37, 30, 26, 22, 21, 20, 29, 32,
        27, 22, 18, 16, 15, 16, 21, 23, 21, 17, 17, 14, 12, 12,
        /* Size 8x16 */
        64, 64, 57, 43, 31, 26, 22, 19, 65, 63, 59, 47, 34, 28, 24, 21, 64, 60,
        56, 46, 35, 29, 25, 22, 63, 58, 51, 44, 34, 29, 25, 23, 57, 55, 43, 37,
        30, 26, 24, 20, 52, 51, 41, 34, 28, 24, 21, 19, 44, 45, 37, 29, 24, 22,
        19, 17, 38, 41, 34, 27, 22, 20, 19, 16, 34, 37, 31, 25, 20, 18, 18, 15,
        29, 32, 28, 23, 18, 16, 15, 14, 26, 29, 26, 22, 17, 15, 14, 14, 24, 27,
        24, 20, 17, 15, 13, 13, 22, 25, 23, 20, 16, 13, 13, 13, 20, 23, 21, 19,
        17, 15, 12, 13, 19, 21, 20, 17, 14, 12, 12, 12, 18, 20, 19, 17, 14, 12,
        10, 10,
        /* Size 16x8 */
        64, 65, 64, 63, 57, 52, 44, 38, 34, 29, 26, 24, 22, 20, 19, 18, 64, 63,
        60, 58, 55, 51, 45, 41, 37, 32, 29, 27, 25, 23, 21, 20, 57, 59, 56, 51,
        43, 41, 37, 34, 31, 28, 26, 24, 23, 21, 20, 19, 43, 47, 46, 44, 37, 34,
        29, 27, 25, 23, 22, 20, 20, 19, 17, 17, 31, 34, 35, 34, 30, 28, 24, 22,
        20, 18, 17, 17, 16, 17, 14, 14, 26, 28, 29, 29, 26, 24, 22, 20, 18, 16,
        15, 15, 13, 15, 12, 12, 22, 24, 25, 25, 24, 21, 19, 19, 18, 15, 14, 13,
        13, 12, 12, 10, 19, 21, 22, 23, 20, 19, 17, 16, 15, 14, 14, 13, 13, 13,
        12, 10,
        /* Size 16x32 */
        64, 65, 64, 63, 57, 47, 43, 38, 31, 28, 26, 23, 22, 20, 19, 18, 65, 64,
        63, 62, 58, 48, 45, 40, 33, 29, 27, 24, 23, 21, 20, 19, 65, 63, 63, 62,
        59, 50, 47, 41, 34, 31, 28, 25, 24, 22, 21, 19, 65, 63, 62, 61, 59, 50,
        47, 41, 34, 31, 29, 26, 24, 23, 21, 20, 64, 63, 60, 59, 56, 49, 46, 41,
        35, 31, 29, 26, 25, 23, 22, 20, 64, 62, 59, 57, 53, 48, 46, 41, 35, 32,
        30, 27, 25, 23, 22, 21, 63, 62, 58, 56, 51, 46, 44, 40, 34, 31, 29, 27,
        25, 24, 23, 20, 60, 60, 57, 54, 48, 43, 41, 38, 33, 30, 28, 25, 24, 24,
        23, 22, 57, 59, 55, 51, 43, 38, 37, 34, 30, 28, 26, 24, 24, 22, 20, 21,
        54, 56, 53, 49, 41, 37, 35, 33, 29, 27, 25, 24, 22, 21, 21, 18, 52, 55,
        51, 48, 41, 35, 34, 32, 28, 26, 24, 23, 21, 22, 19, 20, 47, 50, 48, 46,
        38, 33, 31, 29, 26, 24, 23, 21, 22, 19, 21, 17, 44, 47, 45, 43, 37, 31,
        29, 27, 24, 23, 22, 21, 19, 21, 17, 20, 42, 45, 44, 42, 36, 30, 29, 26,
        24, 22, 21, 20, 20, 17, 20, 16, 38, 41, 41, 40, 34, 29, 27, 25, 22, 21,
        20, 18, 19, 19, 16, 19, 35, 38, 38, 37, 32, 27, 26, 24, 21, 20, 19, 18,
        17, 17, 18, 15, 34, 36, 37, 36, 31, 27, 25, 23, 20, 19, 18, 17, 18, 16,
        15, 18, 32, 34, 35, 34, 30, 26, 24, 22, 20, 18, 17, 16, 16, 17, 16, 13,
        29, 31, 32, 31, 28, 24, 23, 21, 18, 17, 16, 16, 15, 14, 14, 17, 27, 29,
        30, 30, 27, 23, 22, 20, 18, 17, 16, 15, 16, 16, 15, 12, 26, 28, 29, 29,
        26, 23, 22, 20, 17, 16, 15, 14, 14, 13, 14, 16, 25, 27, 28, 28, 25, 22,
        21, 19, 17, 16, 15, 14, 14, 14, 13, 12, 24, 26, 27, 27, 24, 22, 20, 19,
        17, 15, 15, 14, 13, 13, 13, 15, 23, 25, 26, 26, 24, 21, 21, 18, 16, 16,
        14, 13, 13, 12, 12, 10, 22, 24, 25, 25, 23, 21, 20, 17, 16, 16, 13, 14,
        13, 13, 13, 14, 21, 23, 24, 25, 22, 21, 20, 17, 18, 15, 13, 14, 12, 12,
        10, 10, 20, 23, 23, 24, 21, 22, 19, 17, 17, 14, 15, 12, 12, 13, 13, 13,
        20, 22, 22, 23, 21, 21, 17, 19, 16, 14, 15, 12, 13, 11, 11, 9, 19, 21,
        21, 22, 20, 20, 17, 19, 14, 16, 12, 13, 12, 11, 12, 12, 18, 20, 21, 21,
        20, 20, 17, 18, 14, 16, 12, 14, 10, 12, 10, 11, 18, 19, 20, 21, 19, 19,
        17, 18, 14, 16, 12, 14, 10, 12, 10, 12, 17, 19, 19, 20, 20, 17, 19, 14,
        17, 12, 15, 10, 14, 9, 12, 10,
        /* Size 32x16 */
        64, 65, 65, 65, 64, 64, 63, 60, 57, 54, 52, 47, 44, 42, 38, 35, 34, 32,
        29, 27, 26, 25, 24, 23, 22, 21, 20, 20, 19, 18, 18, 17, 65, 64, 63, 63,
        63, 62, 62, 60, 59, 56, 55, 50, 47, 45, 41, 38, 36, 34, 31, 29, 28, 27,
        26, 25, 24, 23, 23, 22, 21, 20, 19, 19, 64, 63, 63, 62, 60, 59, 58, 57,
        55, 53, 51, 48, 45, 44, 41, 38, 37, 35, 32, 30, 29, 28, 27, 26, 25, 24,
        23, 22, 21, 21, 20, 19, 63, 62, 62, 61, 59, 57, 56, 54, 51, 49, 48, 46,
        43, 42, 40, 37, 36, 34, 31, 30, 29, 28, 27, 26, 25, 25, 24, 23, 22, 21,
        21, 20, 57, 58, 59, 59, 56, 53, 51, 48, 43, 41, 41, 38, 37, 36, 34, 32,
        31, 30, 28, 27, 26, 25, 24, 24, 23, 22, 21, 21, 20, 20, 19, 20, 47, 48,
        50, 50, 49, 48, 46, 43, 38, 37, 35, 33, 31, 30, 29, 27, 27, 26, 24, 23,
        23, 22, 22, 21, 21, 21, 22, 21, 20, 20, 19, 17, 43, 45, 47, 47, 46, 46,
        44, 41, 37, 35, 34, 31, 29, 29, 27, 26, 25, 24, 23, 22, 22, 21, 20, 21,
        20, 20, 19, 17, 17, 17, 17, 19, 38, 40, 41, 41, 41, 41, 40, 38, 34, 33,
        32, 29, 27, 26, 25, 24, 23, 22, 21, 20, 20, 19, 19, 18, 17, 17, 17, 19,
        19, 18, 18, 14, 31, 33, 34, 34, 35, 35, 34, 33, 30, 29, 28, 26, 24, 24,
        22, 21, 20, 20, 18, 18, 17, 17, 17, 16, 16, 18, 17, 16, 14, 14, 14, 17,
        28, 29, 31, 31, 31, 32, 31, 30, 28, 27, 26, 24, 23, 22, 21, 20, 19, 18,
        17, 17, 16, 16, 15, 16, 16, 15, 14, 14, 16, 16, 16, 12, 26, 27, 28, 29,
        29, 30, 29, 28, 26, 25, 24, 23, 22, 21, 20, 19, 18, 17, 16, 16, 15, 15,
        15, 14, 13, 13, 15, 15, 12, 12, 12, 15, 23, 24, 25, 26, 26, 27, 27, 25,
        24, 24, 23, 21, 21, 20, 18, 18, 17, 16, 16, 15, 14, 14, 14, 13, 14, 14,
        12, 12, 13, 14, 14, 10, 22, 23, 24, 24, 25, 25, 25, 24, 24, 22, 21, 22,
        19, 20, 19, 17, 18, 16, 15, 16, 14, 14, 13, 13, 13, 12, 12, 13, 12, 10,
        10, 14, 20, 21, 22, 23, 23, 23, 24, 24, 22, 21, 22, 19, 21, 17, 19, 17,
        16, 17, 14, 16, 13, 14, 13, 12, 13, 12, 13, 11, 11, 12, 12, 9, 19, 20,
        21, 21, 22, 22, 23, 23, 20, 21, 19, 21, 17, 20, 16, 18, 15, 16, 14, 15,
        14, 13, 13, 12, 13, 10, 13, 11, 12, 10, 10, 12, 18, 19, 19, 20, 20, 21,
        20, 22, 21, 18, 20, 17, 20, 16, 19, 15, 18, 13, 17, 12, 16, 12, 15, 10,
        14, 10, 13, 9, 12, 11, 12, 10,
        /* Size 4x16 */
        65, 47, 28, 20, 63, 50, 31, 22, 63, 49, 31, 23, 62, 46, 31, 24, 59, 38,
        28, 22, 55, 35, 26, 22, 47, 31, 23, 21, 41, 29, 21, 19, 36, 27, 19, 16,
        31, 24, 17, 14, 28, 23, 16, 13, 26, 22, 15, 13, 24, 21, 16, 13, 23, 22,
        14, 13, 21, 20, 16, 11, 19, 19, 16, 12,
        /* Size 16x4 */
        65, 63, 63, 62, 59, 55, 47, 41, 36, 31, 28, 26, 24, 23, 21, 19, 47, 50,
        49, 46, 38, 35, 31, 29, 27, 24, 23, 22, 21, 22, 20, 19, 28, 31, 31, 31,
        28, 26, 23, 21, 19, 17, 16, 15, 16, 14, 16, 16, 20, 22, 23, 24, 22, 22,
        21, 19, 16, 14, 13, 13, 13, 13, 11, 12,
        /* Size 8x32 */
        64, 64, 57, 43, 31, 26, 22, 19, 65, 63, 58, 45, 33, 27, 23, 20, 65, 63,
        59, 47, 34, 28, 24, 21, 65, 62, 59, 47, 34, 29, 24, 21, 64, 60, 56, 46,
        35, 29, 25, 22, 64, 59, 53, 46, 35, 30, 25, 22, 63, 58, 51, 44, 34, 29,
        25, 23, 60, 57, 48, 41, 33, 28, 24, 23, 57, 55, 43, 37, 30, 26, 24, 20,
        54, 53, 41, 35, 29, 25, 22, 21, 52, 51, 41, 34, 28, 24, 21, 19, 47, 48,
        38, 31, 26, 23, 22, 21, 44, 45, 37, 29, 24, 22, 19, 17, 42, 44, 36, 29,
        24, 21, 20, 20, 38, 41, 34, 27, 22, 20, 19, 16, 35, 38, 32, 26, 21, 19,
        17, 18, 34, 37, 31, 25, 20, 18, 18, 15, 32, 35, 30, 24, 20, 17, 16, 16,
        29, 32, 28, 23, 18, 16, 15, 14, 27, 30, 27, 22, 18, 16, 16, 15, 26, 29,
        26, 22, 17, 15, 14, 14, 25, 28, 25, 21, 17, 15, 14, 13, 24, 27, 24, 20,
        17, 15, 13, 13, 23, 26, 24, 21, 16, 14, 13, 12, 22, 25, 23, 20, 16, 13,
        13, 13, 21, 24, 22, 20, 18, 13, 12, 10, 20, 23, 21, 19, 17, 15, 12, 13,
        20, 22, 21, 17, 16, 15, 13, 11, 19, 21, 20, 17, 14, 12, 12, 12, 18, 21,
        20, 17, 14, 12, 10, 10, 18, 20, 19, 17, 14, 12, 10, 10, 17, 19, 20, 19,
        17, 15, 14, 12,
        /* Size 32x8 */
        64, 65, 65, 65, 64, 64, 63, 60, 57, 54, 52, 47, 44, 42, 38, 35, 34, 32,
        29, 27, 26, 25, 24, 23, 22, 21, 20, 20, 19, 18, 18, 17, 64, 63, 63, 62,
        60, 59, 58, 57, 55, 53, 51, 48, 45, 44, 41, 38, 37, 35, 32, 30, 29, 28,
        27, 26, 25, 24, 23, 22, 21, 21, 20, 19, 57, 58, 59, 59, 56, 53, 51, 48,
        43, 41, 41, 38, 37, 36, 34, 32, 31, 30, 28, 27, 26, 25, 24, 24, 23, 22,
        21, 21, 20, 20, 19, 20, 43, 45, 47, 47, 46, 46, 44, 41, 37, 35, 34, 31,
        29, 29, 27, 26, 25, 24, 23, 22, 22, 21, 20, 21, 20, 20, 19, 17, 17, 17,
        17, 19, 31, 33, 34, 34, 35, 35, 34, 33, 30, 29, 28, 26, 24, 24, 22, 21,
        20, 20, 18, 18, 17, 17, 17, 16, 16, 18, 17, 16, 14, 14, 14, 17, 26, 27,
        28, 29, 29, 30, 29, 28, 26, 25, 24, 23, 22, 21, 20, 19, 18, 17, 16, 16,
        15, 15, 15, 14, 13, 13, 15, 15, 12, 12, 12, 15, 22, 23, 24, 24, 25, 25,
        25, 24, 24, 22, 21, 22, 19, 20, 19, 17, 18, 16, 15, 16, 14, 14, 13, 13,
        13, 12, 12, 13, 12, 10, 10, 14, 19, 20, 21, 21, 22, 22, 23, 23, 20, 21,
        19, 21, 17, 20, 16, 18, 15, 16, 14, 15, 14, 13, 13, 12, 13, 10, 13, 11,
        12, 10, 10, 12 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        42, 34, 34, 21,
#endif
        /* Size 4x4 */
        62, 46, 37, 31, 46, 35, 31, 28, 37, 31, 24, 24, 31, 28, 24, 19,
        /* Size 8x8 */
        65, 53, 43, 42, 38, 33, 31, 28, 53, 43, 44, 44, 41, 37, 34, 31, 43, 44,
        38, 37, 35, 32, 31, 31, 42, 44, 37, 33, 30, 28, 28, 28, 38, 41, 35, 30,
        27, 25, 23, 25, 33, 37, 32, 28, 25, 22, 22, 20, 31, 34, 31, 28, 23, 22,
        20, 17, 28, 31, 31, 28, 25, 20, 17, 18,
        /* Size 16x16 */
        64, 67, 61, 53, 42, 42, 41, 39, 37, 34, 32, 31, 29, 28, 27, 26, 67, 65,
        58, 50, 44, 45, 44, 43, 40, 37, 35, 34, 32, 31, 29, 28, 61, 58, 53, 47,
        43, 44, 44, 43, 41, 38, 36, 35, 34, 33, 31, 30, 53, 50, 47, 43, 42, 43,
        43, 43, 41, 38, 37, 35, 35, 34, 33, 32, 42, 44, 43, 42, 39, 39, 38, 38,
        37, 35, 34, 33, 32, 32, 29, 29, 42, 45, 44, 43, 39, 38, 36, 36, 35, 33,
        32, 31, 30, 31, 28, 27, 41, 44, 44, 43, 38, 36, 34, 33, 31, 30, 29, 28,
        29, 28, 28, 26, 39, 43, 43, 43, 38, 36, 33, 31, 29, 28, 27, 27, 27, 25,
        28, 24, 37, 40, 41, 41, 37, 35, 31, 29, 28, 27, 26, 25, 24, 24, 26, 23,
        34, 37, 38, 38, 35, 33, 30, 28, 27, 25, 24, 23, 23, 24, 25, 21, 32, 35,
        36, 37, 34, 32, 29, 27, 26, 24, 23, 22, 23, 23, 23, 20, 31, 34, 35, 35,
        33, 31, 28, 27, 25, 23, 22, 22, 21, 21, 21, 19, 29, 32, 34, 35, 32, 30,
        29, 27, 24, 23, 23, 21, 20, 20, 18, 18, 28, 31, 33, 34, 32, 31, 28, 25,
        24, 24, 23, 21, 20, 19, 19, 16, 27, 29, 31, 33, 29, 28, 28, 28, 26, 25,
        23, 21, 18, 19, 18, 18, 26, 28, 30, 32, 29, 27, 26, 24, 23, 21, 20, 19,
        18, 16, 18, 17,
        /* Size 32x32 */
        64, 66, 67, 66, 61, 56, 53, 49, 42, 42, 42, 42, 41, 40, 39, 38, 37, 36,
        34, 33, 32, 32, 31, 30, 29, 29, 28, 28, 27, 27, 26, 26, 66, 65, 65, 64,
        59, 53, 51, 48, 43, 43, 44, 44, 43, 42, 41, 39, 39, 38, 36, 35, 34, 33,
        33, 32, 31, 31, 30, 29, 29, 28, 28, 27, 67, 65, 65, 63, 58, 52, 50, 48,
        44, 45, 45, 45, 44, 44, 43, 41, 40, 39, 37, 36, 35, 35, 34, 33, 32, 32,
        31, 30, 29, 29, 28, 27, 66, 64, 63, 62, 57, 51, 49, 48, 44, 45, 45, 46,
        45, 44, 43, 41, 41, 40, 38, 37, 36, 35, 34, 33, 33, 32, 32, 31, 30, 30,
        29, 29, 61, 59, 58, 57, 53, 47, 47, 46, 43, 44, 44, 45, 44, 44, 43, 41,
        41, 40, 38, 37, 36, 36, 35, 34, 34, 33, 33, 32, 31, 30, 30, 28, 56, 53,
        52, 51, 47, 43, 43, 43, 43, 44, 44, 45, 45, 44, 44, 42, 42, 41, 39, 38,
        38, 37, 36, 35, 34, 34, 32, 32, 31, 31, 30, 30, 53, 51, 50, 49, 47, 43,
        43, 43, 42, 43, 43, 44, 43, 43, 43, 41, 41, 40, 38, 38, 37, 36, 35, 35,
        35, 34, 34, 34, 33, 32, 32, 29, 49, 48, 48, 48, 46, 43, 43, 42, 41, 41,
        41, 42, 41, 41, 41, 39, 39, 38, 37, 36, 36, 35, 34, 34, 33, 32, 32, 33,
        33, 32, 31, 32, 42, 43, 44, 44, 43, 43, 42, 41, 39, 39, 39, 38, 38, 38,
        38, 37, 37, 36, 35, 34, 34, 33, 33, 32, 32, 33, 32, 31, 29, 29, 29, 31,
        42, 43, 45, 45, 44, 44, 43, 41, 39, 38, 38, 37, 37, 37, 37, 36, 35, 35,
        34, 33, 33, 32, 32, 32, 32, 31, 30, 30, 31, 31, 30, 27, 42, 44, 45, 45,
        44, 44, 43, 41, 39, 38, 38, 37, 36, 36, 36, 35, 35, 34, 33, 33, 32, 32,
        31, 31, 30, 30, 31, 31, 28, 28, 27, 30, 42, 44, 45, 46, 45, 45, 44, 42,
        38, 37, 37, 35, 34, 34, 33, 33, 33, 32, 31, 31, 30, 30, 30, 29, 30, 30,
        28, 28, 29, 30, 29, 26, 41, 43, 44, 45, 44, 45, 43, 41, 38, 37, 36, 34,
        34, 33, 33, 32, 31, 31, 30, 30, 29, 29, 28, 29, 29, 27, 28, 29, 28, 26,
        26, 29, 40, 42, 44, 44, 44, 44, 43, 41, 38, 37, 36, 34, 33, 33, 32, 31,
        31, 30, 29, 29, 29, 28, 28, 28, 27, 28, 28, 26, 26, 28, 28, 24, 39, 41,
        43, 43, 43, 44, 43, 41, 38, 37, 36, 33, 33, 32, 31, 30, 29, 29, 28, 28,
        27, 27, 27, 26, 27, 27, 25, 27, 28, 24, 24, 28, 38, 39, 41, 41, 41, 42,
        41, 39, 37, 36, 35, 33, 32, 31, 30, 29, 28, 28, 27, 27, 26, 26, 26, 26,
        26, 25, 27, 25, 24, 27, 27, 23, 37, 39, 40, 41, 41, 42, 41, 39, 37, 35,
        35, 33, 31, 31, 29, 28, 28, 27, 27, 26, 26, 25, 25, 25, 24, 26, 24, 24,
        26, 24, 23, 27, 36, 38, 39, 40, 40, 41, 40, 38, 36, 35, 34, 32, 31, 30,
        29, 28, 27, 27, 26, 25, 25, 25, 24, 24, 25, 23, 24, 25, 22, 25, 26, 22,
        34, 36, 37, 38, 38, 39, 38, 37, 35, 34, 33, 31, 30, 29, 28, 27, 27, 26,
        25, 24, 24, 24, 23, 24, 23, 23, 24, 22, 25, 23, 21, 26, 33, 35, 36, 37,
        37, 38, 38, 36, 34, 33, 33, 31, 30, 29, 28, 27, 26, 25, 24, 24, 23, 23,
        23, 23, 22, 23, 22, 24, 21, 23, 25, 20, 32, 34, 35, 36, 36, 38, 37, 36,
        34, 33, 32, 30, 29, 29, 27, 26, 26, 25, 24, 23, 23, 23, 22, 22, 23, 21,
        23, 20, 23, 23, 20, 24, 32, 33, 35, 35, 36, 37, 36, 35, 33, 32, 32, 30,
        29, 28, 27, 26, 25, 25, 24, 23, 23, 22, 22, 22, 21, 22, 20, 22, 21, 20,
        23, 19, 31, 33, 34, 34, 35, 36, 35, 34, 33, 32, 31, 30, 28, 28, 27, 26,
        25, 24, 23, 23, 22, 22, 22, 22, 21, 20, 21, 21, 21, 22, 19, 23, 30, 32,
        33, 33, 34, 35, 35, 34, 32, 32, 31, 29, 29, 28, 26, 26, 25, 24, 24, 23,
        22, 22, 22, 21, 21, 21, 20, 20, 21, 19, 22, 18, 29, 31, 32, 33, 34, 34,
        35, 33, 32, 32, 30, 30, 29, 27, 27, 26, 24, 25, 23, 22, 23, 21, 21, 21,
        20, 21, 20, 20, 18, 21, 18, 22, 29, 31, 32, 32, 33, 34, 34, 32, 33, 31,
        30, 30, 27, 28, 27, 25, 26, 23, 23, 23, 21, 22, 20, 21, 21, 20, 21, 20,
        21, 17, 21, 17, 28, 30, 31, 32, 33, 32, 34, 32, 32, 30, 31, 28, 28, 28,
        25, 27, 24, 24, 24, 22, 23, 20, 21, 20, 20, 21, 19, 20, 19, 20, 16, 21,
        28, 29, 30, 31, 32, 32, 34, 33, 31, 30, 31, 28, 29, 26, 27, 25, 24, 25,
        22, 24, 20, 22, 21, 20, 20, 20, 20, 19, 20, 19, 20, 15, 27, 29, 29, 30,
        31, 31, 33, 33, 29, 31, 28, 29, 28, 26, 28, 24, 26, 22, 25, 21, 23, 21,
        21, 21, 18, 21, 19, 20, 18, 20, 18, 20, 27, 28, 29, 30, 30, 31, 32, 32,
        29, 31, 28, 30, 26, 28, 24, 27, 24, 25, 23, 23, 23, 20, 22, 19, 21, 17,
        20, 19, 20, 18, 20, 18, 26, 28, 28, 29, 30, 30, 32, 31, 29, 30, 27, 29,
        26, 28, 24, 27, 23, 26, 21, 25, 20, 23, 19, 22, 18, 21, 16, 20, 18, 20,
        17, 19, 26, 27, 27, 29, 28, 30, 29, 32, 31, 27, 30, 26, 29, 24, 28, 23,
        27, 22, 26, 20, 24, 19, 23, 18, 22, 17, 21, 15, 20, 18, 19, 17,
        /* Size 4x8 */
        65, 44, 36, 30, 51, 45, 39, 32, 45, 37, 34, 31, 44, 34, 29, 26, 39, 32,
        26, 25, 35, 30, 24, 22, 32, 30, 23, 20, 29, 30, 25, 20,
        /* Size 8x4 */
        65, 51, 45, 44, 39, 35, 32, 29, 44, 45, 37, 34, 32, 30, 30, 30, 36, 39,
        34, 29, 26, 24, 23, 25, 30, 32, 31, 26, 25, 22, 20, 20,
        /* Size 8x16 */
        64, 58, 43, 41, 36, 32, 29, 27, 67, 54, 45, 44, 39, 35, 32, 29, 61, 49,
        44, 44, 40, 37, 34, 31, 53, 45, 42, 43, 40, 37, 35, 33, 42, 43, 39, 38,
        36, 34, 33, 29, 42, 44, 39, 36, 34, 32, 30, 28, 41, 44, 38, 34, 31, 29,
        27, 26, 39, 43, 38, 32, 29, 27, 27, 25, 37, 41, 37, 31, 27, 26, 26, 24,
        34, 39, 35, 30, 26, 24, 23, 23, 32, 37, 34, 29, 25, 23, 21, 22, 31, 35,
        33, 28, 24, 22, 21, 21, 30, 34, 32, 29, 24, 21, 21, 21, 28, 33, 30, 28,
        26, 23, 20, 20, 27, 31, 30, 26, 23, 20, 20, 20, 26, 30, 29, 26, 23, 20,
        18, 18,
        /* Size 16x8 */
        64, 67, 61, 53, 42, 42, 41, 39, 37, 34, 32, 31, 30, 28, 27, 26, 58, 54,
        49, 45, 43, 44, 44, 43, 41, 39, 37, 35, 34, 33, 31, 30, 43, 45, 44, 42,
        39, 39, 38, 38, 37, 35, 34, 33, 32, 30, 30, 29, 41, 44, 44, 43, 38, 36,
        34, 32, 31, 30, 29, 28, 29, 28, 26, 26, 36, 39, 40, 40, 36, 34, 31, 29,
        27, 26, 25, 24, 24, 26, 23, 23, 32, 35, 37, 37, 34, 32, 29, 27, 26, 24,
        23, 22, 21, 23, 20, 20, 29, 32, 34, 35, 33, 30, 27, 27, 26, 23, 21, 21,
        21, 20, 20, 18, 27, 29, 31, 33, 29, 28, 26, 25, 24, 23, 22, 21, 21, 20,
        20, 18,
        /* Size 16x32 */
        64, 66, 58, 53, 43, 42, 41, 39, 36, 34, 32, 30, 29, 28, 27, 26, 65, 65,
        55, 51, 44, 44, 42, 41, 38, 36, 34, 32, 31, 30, 29, 28, 67, 64, 54, 51,
        45, 45, 44, 42, 39, 37, 35, 33, 32, 31, 29, 28, 66, 62, 53, 49, 45, 45,
        44, 43, 39, 37, 36, 34, 33, 32, 30, 30, 61, 57, 49, 47, 44, 45, 44, 43,
        40, 38, 37, 34, 34, 32, 31, 29, 55, 51, 45, 43, 43, 45, 45, 44, 41, 39,
        38, 36, 34, 32, 31, 31, 53, 50, 45, 43, 42, 44, 43, 42, 40, 38, 37, 35,
        35, 34, 33, 30, 49, 48, 44, 43, 41, 41, 41, 41, 38, 37, 36, 34, 33, 33,
        33, 33, 42, 44, 43, 42, 39, 38, 38, 38, 36, 35, 34, 32, 33, 31, 29, 32,
        42, 45, 44, 42, 39, 37, 37, 36, 35, 34, 33, 32, 31, 31, 31, 27, 42, 45,
        44, 43, 39, 37, 36, 36, 34, 33, 32, 31, 30, 31, 28, 30, 42, 45, 45, 44,
        38, 35, 34, 33, 32, 31, 30, 29, 30, 28, 30, 26, 41, 44, 44, 43, 38, 34,
        34, 32, 31, 30, 29, 29, 27, 29, 26, 29, 40, 44, 44, 43, 38, 34, 33, 32,
        30, 29, 29, 28, 28, 26, 29, 25, 39, 43, 43, 42, 38, 34, 32, 31, 29, 28,
        27, 26, 27, 27, 25, 28, 38, 41, 42, 41, 37, 33, 32, 30, 28, 27, 26, 26,
        25, 25, 27, 23, 37, 40, 41, 41, 37, 33, 31, 29, 27, 26, 26, 25, 26, 25,
        24, 27, 36, 39, 41, 40, 36, 32, 31, 29, 27, 26, 25, 24, 24, 25, 25, 22,
        34, 38, 39, 38, 35, 31, 30, 28, 26, 25, 24, 23, 23, 23, 23, 26, 33, 36,
        38, 37, 34, 31, 29, 28, 25, 24, 23, 23, 23, 24, 23, 20, 32, 36, 37, 37,
        34, 30, 29, 27, 25, 24, 23, 22, 21, 21, 22, 25, 32, 35, 36, 36, 33, 30,
        29, 27, 25, 24, 23, 22, 22, 22, 21, 20, 31, 34, 35, 35, 33, 30, 28, 27,
        24, 23, 22, 22, 21, 21, 21, 23, 30, 33, 35, 35, 32, 30, 29, 26, 24, 24,
        22, 21, 21, 20, 20, 18, 30, 33, 34, 34, 32, 30, 29, 25, 24, 24, 21, 21,
        21, 20, 21, 22, 29, 32, 33, 34, 31, 30, 29, 25, 26, 23, 21, 21, 20, 20,
        18, 17, 28, 31, 33, 33, 30, 31, 28, 25, 26, 22, 23, 20, 20, 21, 20, 21,
        28, 31, 32, 33, 30, 30, 26, 27, 24, 22, 23, 19, 21, 19, 19, 16, 27, 30,
        31, 32, 30, 30, 26, 28, 23, 25, 20, 21, 20, 19, 20, 20, 27, 29, 30, 32,
        29, 30, 26, 27, 23, 25, 20, 22, 18, 20, 18, 18, 26, 29, 30, 31, 29, 29,
        26, 27, 23, 25, 20, 22, 18, 20, 18, 20, 26, 29, 28, 32, 31, 26, 29, 23,
        27, 20, 24, 18, 22, 16, 20, 17,
        /* Size 32x16 */
        64, 65, 67, 66, 61, 55, 53, 49, 42, 42, 42, 42, 41, 40, 39, 38, 37, 36,
        34, 33, 32, 32, 31, 30, 30, 29, 28, 28, 27, 27, 26, 26, 66, 65, 64, 62,
        57, 51, 50, 48, 44, 45, 45, 45, 44, 44, 43, 41, 40, 39, 38, 36, 36, 35,
        34, 33, 33, 32, 31, 31, 30, 29, 29, 29, 58, 55, 54, 53, 49, 45, 45, 44,
        43, 44, 44, 45, 44, 44, 43, 42, 41, 41, 39, 38, 37, 36, 35, 35, 34, 33,
        33, 32, 31, 30, 30, 28, 53, 51, 51, 49, 47, 43, 43, 43, 42, 42, 43, 44,
        43, 43, 42, 41, 41, 40, 38, 37, 37, 36, 35, 35, 34, 34, 33, 33, 32, 32,
        31, 32, 43, 44, 45, 45, 44, 43, 42, 41, 39, 39, 39, 38, 38, 38, 38, 37,
        37, 36, 35, 34, 34, 33, 33, 32, 32, 31, 30, 30, 30, 29, 29, 31, 42, 44,
        45, 45, 45, 45, 44, 41, 38, 37, 37, 35, 34, 34, 34, 33, 33, 32, 31, 31,
        30, 30, 30, 30, 30, 30, 31, 30, 30, 30, 29, 26, 41, 42, 44, 44, 44, 45,
        43, 41, 38, 37, 36, 34, 34, 33, 32, 32, 31, 31, 30, 29, 29, 29, 28, 29,
        29, 29, 28, 26, 26, 26, 26, 29, 39, 41, 42, 43, 43, 44, 42, 41, 38, 36,
        36, 33, 32, 32, 31, 30, 29, 29, 28, 28, 27, 27, 27, 26, 25, 25, 25, 27,
        28, 27, 27, 23, 36, 38, 39, 39, 40, 41, 40, 38, 36, 35, 34, 32, 31, 30,
        29, 28, 27, 27, 26, 25, 25, 25, 24, 24, 24, 26, 26, 24, 23, 23, 23, 27,
        34, 36, 37, 37, 38, 39, 38, 37, 35, 34, 33, 31, 30, 29, 28, 27, 26, 26,
        25, 24, 24, 24, 23, 24, 24, 23, 22, 22, 25, 25, 25, 20, 32, 34, 35, 36,
        37, 38, 37, 36, 34, 33, 32, 30, 29, 29, 27, 26, 26, 25, 24, 23, 23, 23,
        22, 22, 21, 21, 23, 23, 20, 20, 20, 24, 30, 32, 33, 34, 34, 36, 35, 34,
        32, 32, 31, 29, 29, 28, 26, 26, 25, 24, 23, 23, 22, 22, 22, 21, 21, 21,
        20, 19, 21, 22, 22, 18, 29, 31, 32, 33, 34, 34, 35, 33, 33, 31, 30, 30,
        27, 28, 27, 25, 26, 24, 23, 23, 21, 22, 21, 21, 21, 20, 20, 21, 20, 18,
        18, 22, 28, 30, 31, 32, 32, 32, 34, 33, 31, 31, 31, 28, 29, 26, 27, 25,
        25, 25, 23, 24, 21, 22, 21, 20, 20, 20, 21, 19, 19, 20, 20, 16, 27, 29,
        29, 30, 31, 31, 33, 33, 29, 31, 28, 30, 26, 29, 25, 27, 24, 25, 23, 23,
        22, 21, 21, 20, 21, 18, 20, 19, 20, 18, 18, 20, 26, 28, 28, 30, 29, 31,
        30, 33, 32, 27, 30, 26, 29, 25, 28, 23, 27, 22, 26, 20, 25, 20, 23, 18,
        22, 17, 21, 16, 20, 18, 20, 17,
        /* Size 4x16 */
        66, 42, 34, 28, 64, 45, 37, 31, 57, 45, 38, 32, 50, 44, 38, 34, 44, 38,
        35, 31, 45, 37, 33, 31, 44, 34, 30, 29, 43, 34, 28, 27, 40, 33, 26, 25,
        38, 31, 25, 23, 36, 30, 24, 21, 34, 30, 23, 21, 33, 30, 24, 20, 31, 31,
        22, 21, 30, 30, 25, 19, 29, 29, 25, 20,
        /* Size 16x4 */
        66, 64, 57, 50, 44, 45, 44, 43, 40, 38, 36, 34, 33, 31, 30, 29, 42, 45,
        45, 44, 38, 37, 34, 34, 33, 31, 30, 30, 30, 31, 30, 29, 34, 37, 38, 38,
        35, 33, 30, 28, 26, 25, 24, 23, 24, 22, 25, 25, 28, 31, 32, 34, 31, 31,
        29, 27, 25, 23, 21, 21, 20, 21, 19, 20,
        /* Size 8x32 */
        64, 58, 43, 41, 36, 32, 29, 27, 65, 55, 44, 42, 38, 34, 31, 29, 67, 54,
        45, 44, 39, 35, 32, 29, 66, 53, 45, 44, 39, 36, 33, 30, 61, 49, 44, 44,
        40, 37, 34, 31, 55, 45, 43, 45, 41, 38, 34, 31, 53, 45, 42, 43, 40, 37,
        35, 33, 49, 44, 41, 41, 38, 36, 33, 33, 42, 43, 39, 38, 36, 34, 33, 29,
        42, 44, 39, 37, 35, 33, 31, 31, 42, 44, 39, 36, 34, 32, 30, 28, 42, 45,
        38, 34, 32, 30, 30, 30, 41, 44, 38, 34, 31, 29, 27, 26, 40, 44, 38, 33,
        30, 29, 28, 29, 39, 43, 38, 32, 29, 27, 27, 25, 38, 42, 37, 32, 28, 26,
        25, 27, 37, 41, 37, 31, 27, 26, 26, 24, 36, 41, 36, 31, 27, 25, 24, 25,
        34, 39, 35, 30, 26, 24, 23, 23, 33, 38, 34, 29, 25, 23, 23, 23, 32, 37,
        34, 29, 25, 23, 21, 22, 32, 36, 33, 29, 25, 23, 22, 21, 31, 35, 33, 28,
        24, 22, 21, 21, 30, 35, 32, 29, 24, 22, 21, 20, 30, 34, 32, 29, 24, 21,
        21, 21, 29, 33, 31, 29, 26, 21, 20, 18, 28, 33, 30, 28, 26, 23, 20, 20,
        28, 32, 30, 26, 24, 23, 21, 19, 27, 31, 30, 26, 23, 20, 20, 20, 27, 30,
        29, 26, 23, 20, 18, 18, 26, 30, 29, 26, 23, 20, 18, 18, 26, 28, 31, 29,
        27, 24, 22, 20,
        /* Size 32x8 */
        64, 65, 67, 66, 61, 55, 53, 49, 42, 42, 42, 42, 41, 40, 39, 38, 37, 36,
        34, 33, 32, 32, 31, 30, 30, 29, 28, 28, 27, 27, 26, 26, 58, 55, 54, 53,
        49, 45, 45, 44, 43, 44, 44, 45, 44, 44, 43, 42, 41, 41, 39, 38, 37, 36,
        35, 35, 34, 33, 33, 32, 31, 30, 30, 28, 43, 44, 45, 45, 44, 43, 42, 41,
        39, 39, 39, 38, 38, 38, 38, 37, 37, 36, 35, 34, 34, 33, 33, 32, 32, 31,
        30, 30, 30, 29, 29, 31, 41, 42, 44, 44, 44, 45, 43, 41, 38, 37, 36, 34,
        34, 33, 32, 32, 31, 31, 30, 29, 29, 29, 28, 29, 29, 29, 28, 26, 26, 26,
        26, 29, 36, 38, 39, 39, 40, 41, 40, 38, 36, 35, 34, 32, 31, 30, 29, 28,
        27, 27, 26, 25, 25, 25, 24, 24, 24, 26, 26, 24, 23, 23, 23, 27, 32, 34,
        35, 36, 37, 38, 37, 36, 34, 33, 32, 30, 29, 29, 27, 26, 26, 25, 24, 23,
        23, 23, 22, 22, 21, 21, 23, 23, 20, 20, 20, 24, 29, 31, 32, 33, 34, 34,
        35, 33, 33, 31, 30, 30, 27, 28, 27, 25, 26, 24, 23, 23, 21, 22, 21, 21,
        21, 20, 20, 21, 20, 18, 18, 22, 27, 29, 29, 30, 31, 31, 33, 33, 29, 31,
        28, 30, 26, 29, 25, 27, 24, 25, 23, 23, 22, 21, 21, 20, 21, 18, 20, 19,
        20, 18, 18, 20 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        52, 27, 27, 15,
#endif
        /* Size 4x4 */
        63, 53, 32, 23, 53, 36, 26, 20, 32, 26, 18, 17, 23, 20, 17, 12,
        /* Size 8x8 */
        64, 63, 59, 45, 36, 27, 24, 21, 63, 60, 55, 45, 38, 29, 26, 22, 59, 55,
        42, 36, 32, 26, 23, 22, 45, 45, 36, 29, 25, 21, 21, 20, 36, 38, 32, 25,
        22, 18, 17, 17, 27, 29, 26, 21, 18, 15, 15, 14, 24, 26, 23, 21, 17, 15,
        13, 11, 21, 22, 22, 20, 17, 14, 11, 11,
        /* Size 16x16 */
        64, 65, 65, 64, 59, 52, 47, 41, 35, 31, 29, 25, 23, 22, 20, 19, 65, 64,
        63, 63, 60, 54, 49, 44, 38, 34, 31, 27, 25, 24, 22, 20, 65, 63, 61, 60,
        57, 53, 49, 44, 39, 35, 32, 28, 26, 25, 23, 21, 64, 63, 60, 58, 55, 51,
        48, 44, 39, 35, 33, 29, 27, 25, 24, 22, 59, 60, 57, 55, 48, 44, 41, 38,
        34, 32, 29, 26, 25, 24, 22, 21, 52, 54, 53, 51, 44, 38, 35, 33, 30, 28,
        26, 24, 23, 23, 21, 20, 47, 49, 49, 48, 41, 35, 32, 30, 27, 26, 24, 22,
        21, 21, 20, 18, 41, 44, 44, 44, 38, 33, 30, 28, 25, 24, 22, 21, 20, 19,
        20, 17, 35, 38, 39, 39, 34, 30, 27, 25, 23, 21, 20, 19, 18, 17, 18, 16,
        31, 34, 35, 35, 32, 28, 26, 24, 21, 20, 18, 17, 16, 17, 17, 14, 29, 31,
        32, 33, 29, 26, 24, 22, 20, 18, 17, 16, 15, 16, 16, 14, 25, 27, 28, 29,
        26, 24, 22, 21, 19, 17, 16, 15, 14, 14, 14, 12, 23, 25, 26, 27, 25, 23,
        21, 20, 18, 16, 15, 14, 14, 13, 12, 12, 22, 24, 25, 25, 24, 23, 21, 19,
        17, 17, 16, 14, 13, 13, 12, 10, 20, 22, 23, 24, 22, 21, 20, 20, 18, 17,
        16, 14, 12, 12, 12, 11, 19, 20, 21, 22, 21, 20, 18, 17, 16, 14, 14, 12,
        12, 10, 11, 11,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 64, 64, 60, 59, 57, 52, 50, 47, 43, 41, 38, 35, 34,
        31, 29, 29, 26, 25, 25, 23, 23, 22, 21, 20, 20, 19, 18, 65, 64, 64, 64,
        64, 63, 63, 60, 59, 59, 53, 52, 48, 45, 43, 40, 37, 36, 33, 31, 30, 27,
        26, 26, 25, 24, 23, 22, 21, 21, 20, 20, 65, 64, 64, 64, 63, 63, 63, 61,
        60, 59, 54, 53, 49, 46, 44, 41, 38, 37, 34, 32, 31, 28, 27, 27, 25, 24,
        24, 23, 22, 21, 20, 19, 65, 64, 64, 63, 63, 63, 63, 61, 60, 59, 55, 53,
        50, 47, 45, 41, 39, 38, 34, 32, 32, 28, 28, 27, 26, 25, 24, 23, 23, 22,
        21, 21, 65, 64, 63, 63, 61, 61, 60, 58, 57, 56, 53, 51, 49, 46, 44, 41,
        39, 38, 35, 33, 32, 29, 28, 28, 26, 25, 25, 24, 23, 22, 21, 20, 64, 63,
        63, 63, 61, 60, 59, 57, 56, 55, 51, 51, 48, 45, 44, 41, 39, 38, 35, 33,
        32, 29, 28, 28, 27, 26, 25, 24, 23, 22, 22, 21, 64, 63, 63, 63, 60, 59,
        58, 55, 55, 53, 51, 50, 48, 45, 44, 41, 39, 38, 35, 33, 33, 29, 29, 28,
        27, 26, 25, 25, 24, 23, 22, 21, 60, 60, 61, 61, 58, 57, 55, 52, 50, 48,
        46, 45, 43, 41, 40, 38, 36, 35, 33, 31, 30, 28, 27, 27, 25, 25, 24, 25,
        24, 23, 23, 22, 59, 59, 60, 60, 57, 56, 55, 50, 48, 46, 44, 43, 41, 39,
        38, 36, 34, 34, 32, 30, 29, 27, 26, 26, 25, 25, 24, 23, 22, 21, 21, 22,
        57, 59, 59, 59, 56, 55, 53, 48, 46, 42, 41, 40, 38, 36, 36, 34, 33, 32,
        30, 29, 28, 26, 26, 25, 24, 23, 23, 22, 23, 22, 21, 19, 52, 53, 54, 55,
        53, 51, 51, 46, 44, 41, 38, 37, 35, 34, 33, 32, 30, 29, 28, 27, 26, 24,
        24, 24, 23, 23, 23, 22, 21, 20, 20, 21, 50, 52, 53, 53, 51, 51, 50, 45,
        43, 40, 37, 36, 34, 33, 32, 31, 29, 29, 27, 26, 26, 24, 23, 23, 22, 22,
        21, 20, 21, 21, 21, 18, 47, 48, 49, 50, 49, 48, 48, 43, 41, 38, 35, 34,
        32, 31, 30, 29, 27, 27, 26, 25, 24, 23, 22, 22, 21, 20, 21, 21, 20, 19,
        18, 20, 43, 45, 46, 47, 46, 45, 45, 41, 39, 36, 34, 33, 31, 29, 29, 27,
        26, 25, 24, 23, 23, 21, 21, 21, 20, 21, 20, 19, 19, 20, 20, 17, 41, 43,
        44, 45, 44, 44, 44, 40, 38, 36, 33, 32, 30, 29, 28, 26, 25, 25, 24, 23,
        22, 21, 21, 20, 20, 19, 19, 20, 20, 17, 17, 19, 38, 40, 41, 41, 41, 41,
        41, 38, 36, 34, 32, 31, 29, 27, 26, 25, 24, 23, 22, 21, 21, 20, 19, 19,
        19, 18, 19, 18, 17, 19, 19, 16, 35, 37, 38, 39, 39, 39, 39, 36, 34, 33,
        30, 29, 27, 26, 25, 24, 23, 22, 21, 20, 20, 19, 19, 18, 18, 18, 17, 17,
        18, 16, 16, 18, 34, 36, 37, 38, 38, 38, 38, 35, 34, 32, 29, 29, 27, 25,
        25, 23, 22, 22, 21, 20, 20, 18, 18, 18, 18, 17, 17, 18, 15, 17, 18, 15,
        31, 33, 34, 34, 35, 35, 35, 33, 32, 30, 28, 27, 26, 24, 24, 22, 21, 21,
        20, 19, 18, 17, 17, 17, 16, 17, 17, 16, 17, 16, 14, 17, 29, 31, 32, 32,
        33, 33, 33, 31, 30, 29, 27, 26, 25, 23, 23, 21, 20, 20, 19, 18, 18, 17,
        16, 16, 16, 16, 15, 17, 15, 15, 17, 13, 29, 30, 31, 32, 32, 32, 33, 30,
        29, 28, 26, 26, 24, 23, 22, 21, 20, 20, 18, 18, 17, 16, 16, 16, 15, 15,
        16, 14, 16, 15, 14, 16, 26, 27, 28, 28, 29, 29, 29, 28, 27, 26, 24, 24,
        23, 21, 21, 20, 19, 18, 17, 17, 16, 15, 15, 15, 14, 15, 14, 15, 14, 14,
        16, 13, 25, 26, 27, 28, 28, 28, 29, 27, 26, 26, 24, 23, 22, 21, 21, 19,
        19, 18, 17, 16, 16, 15, 15, 15, 14, 14, 14, 14, 14, 14, 12, 15, 25, 26,
        27, 27, 28, 28, 28, 27, 26, 25, 24, 23, 22, 21, 20, 19, 18, 18, 17, 16,
        16, 15, 15, 15, 14, 14, 13, 13, 14, 12, 15, 11, 23, 25, 25, 26, 26, 27,
        27, 25, 25, 24, 23, 22, 21, 20, 20, 19, 18, 18, 16, 16, 15, 14, 14, 14,
        14, 14, 13, 13, 12, 14, 12, 14, 23, 24, 24, 25, 25, 26, 26, 25, 25, 23,
        23, 22, 20, 21, 19, 18, 18, 17, 17, 16, 15, 15, 14, 14, 14, 13, 14, 13,
        13, 11, 14, 11, 22, 23, 24, 24, 25, 25, 25, 24, 24, 23, 23, 21, 21, 20,
        19, 19, 17, 17, 17, 15, 16, 14, 14, 13, 13, 14, 13, 13, 12, 13, 10, 13,
        21, 22, 23, 23, 24, 24, 25, 25, 23, 22, 22, 20, 21, 19, 20, 18, 17, 18,
        16, 17, 14, 15, 14, 13, 13, 13, 13, 12, 13, 12, 13, 9, 20, 21, 22, 23,
        23, 23, 24, 24, 22, 23, 21, 21, 20, 19, 20, 17, 18, 15, 17, 15, 16, 14,
        14, 14, 12, 13, 12, 13, 12, 13, 11, 12, 20, 21, 21, 22, 22, 22, 23, 23,
        21, 22, 20, 21, 19, 20, 17, 19, 16, 17, 16, 15, 15, 14, 14, 12, 14, 11,
        13, 12, 13, 11, 12, 11, 19, 20, 20, 21, 21, 22, 22, 23, 21, 21, 20, 21,
        18, 20, 17, 19, 16, 18, 14, 17, 14, 16, 12, 15, 12, 14, 10, 13, 11, 12,
        11, 12, 18, 20, 19, 21, 20, 21, 21, 22, 22, 19, 21, 18, 20, 17, 19, 16,
        18, 15, 17, 13, 16, 13, 15, 11, 14, 11, 13, 9, 12, 11, 12, 10,
        /* Size 4x8 */
        64, 53, 33, 23, 63, 51, 35, 24, 59, 40, 30, 23, 46, 34, 24, 19, 38, 29,
        21, 17, 28, 24, 17, 15, 25, 23, 16, 13, 22, 21, 17, 13,
        /* Size 8x4 */
        64, 63, 59, 46, 38, 28, 25, 22, 53, 51, 40, 34, 29, 24, 23, 21, 33, 35,
        30, 24, 21, 17, 16, 17, 23, 24, 23, 19, 17, 15, 13, 13,
        /* Size 8x16 */
        64, 64, 57, 47, 35, 26, 23, 20, 65, 63, 59, 49, 38, 28, 25, 22, 64, 61,
        56, 49, 39, 29, 26, 23, 64, 59, 53, 48, 39, 30, 27, 24, 59, 57, 46, 41,
        34, 27, 25, 22, 52, 52, 41, 35, 30, 24, 23, 20, 47, 48, 38, 33, 27, 23,
        21, 19, 42, 44, 36, 30, 25, 21, 20, 17, 36, 39, 33, 27, 23, 19, 18, 17,
        32, 35, 30, 26, 21, 17, 16, 16, 29, 32, 28, 24, 20, 16, 15, 15, 25, 28,
        25, 22, 19, 15, 14, 14, 23, 26, 24, 21, 18, 15, 14, 14, 22, 25, 23, 20,
        18, 16, 13, 13, 20, 23, 22, 19, 16, 14, 13, 13, 19, 21, 21, 18, 16, 13,
        11, 11,
        /* Size 16x8 */
        64, 65, 64, 64, 59, 52, 47, 42, 36, 32, 29, 25, 23, 22, 20, 19, 64, 63,
        61, 59, 57, 52, 48, 44, 39, 35, 32, 28, 26, 25, 23, 21, 57, 59, 56, 53,
        46, 41, 38, 36, 33, 30, 28, 25, 24, 23, 22, 21, 47, 49, 49, 48, 41, 35,
        33, 30, 27, 26, 24, 22, 21, 20, 19, 18, 35, 38, 39, 39, 34, 30, 27, 25,
        23, 21, 20, 19, 18, 18, 16, 16, 26, 28, 29, 30, 27, 24, 23, 21, 19, 17,
        16, 15, 15, 16, 14, 13, 23, 25, 26, 27, 25, 23, 21, 20, 18, 16, 15, 14,
        14, 13, 13, 11, 20, 22, 23, 24, 22, 20, 19, 17, 17, 16, 15, 14, 14, 13,
        13, 11,
        /* Size 16x32 */
        64, 65, 64, 64, 57, 52, 47, 38, 35, 31, 26, 25, 23, 22, 20, 19, 65, 64,
        63, 63, 58, 53, 48, 40, 37, 33, 27, 26, 24, 23, 21, 20, 65, 64, 63, 63,
        59, 54, 49, 41, 38, 34, 28, 27, 25, 23, 22, 20, 65, 63, 63, 62, 59, 55,
        50, 41, 39, 34, 28, 28, 25, 24, 23, 22, 64, 63, 61, 60, 56, 53, 49, 41,
        39, 35, 29, 28, 26, 24, 23, 21, 64, 63, 60, 59, 55, 51, 48, 41, 39, 35,
        29, 29, 26, 24, 23, 22, 64, 62, 59, 58, 53, 51, 48, 41, 39, 35, 30, 29,
        27, 25, 24, 22, 60, 60, 58, 55, 48, 46, 43, 38, 36, 33, 28, 27, 25, 25,
        24, 23, 59, 60, 57, 55, 46, 44, 41, 36, 34, 32, 27, 27, 25, 23, 22, 23,
        57, 59, 55, 53, 43, 40, 38, 34, 33, 30, 26, 26, 24, 23, 23, 20, 52, 55,
        52, 51, 41, 38, 35, 32, 30, 28, 24, 24, 23, 23, 20, 22, 51, 53, 51, 49,
        40, 37, 34, 31, 29, 27, 24, 24, 22, 21, 22, 19, 47, 50, 48, 48, 38, 35,
        33, 29, 27, 26, 23, 22, 21, 21, 19, 21, 43, 46, 45, 45, 37, 34, 31, 27,
        26, 24, 21, 21, 21, 19, 21, 17, 42, 45, 44, 44, 36, 33, 30, 26, 25, 24,
        21, 21, 20, 20, 17, 20, 38, 41, 41, 41, 34, 32, 29, 25, 24, 22, 20, 20,
        18, 18, 19, 16, 36, 39, 39, 39, 33, 30, 27, 24, 23, 21, 19, 19, 18, 18,
        17, 19, 35, 38, 38, 38, 32, 29, 27, 23, 22, 21, 18, 18, 17, 17, 17, 15,
        32, 34, 35, 35, 30, 28, 26, 22, 21, 20, 17, 17, 16, 16, 16, 18, 30, 32,
        33, 33, 29, 27, 25, 21, 20, 19, 17, 16, 16, 17, 16, 13, 29, 31, 32, 32,
        28, 26, 24, 21, 20, 18, 16, 16, 15, 14, 15, 17, 26, 28, 29, 29, 26, 24,
        23, 20, 19, 17, 15, 15, 15, 15, 14, 13, 25, 28, 28, 29, 25, 24, 22, 20,
        19, 17, 15, 15, 14, 14, 14, 16, 25, 27, 28, 28, 25, 24, 22, 19, 18, 17,
        15, 15, 14, 13, 13, 12, 23, 26, 26, 27, 24, 23, 21, 19, 18, 16, 15, 14,
        14, 13, 14, 15, 23, 25, 25, 26, 23, 23, 21, 19, 18, 16, 15, 14, 13, 13,
        11, 11, 22, 24, 25, 25, 23, 23, 20, 19, 18, 15, 16, 13, 13, 13, 13, 14,
        21, 23, 24, 24, 22, 22, 19, 20, 17, 15, 15, 13, 14, 12, 12, 10, 20, 23,
        23, 24, 22, 22, 19, 20, 16, 17, 14, 14, 13, 12, 13, 13, 20, 22, 22, 23,
        21, 21, 19, 19, 16, 17, 13, 15, 11, 13, 11, 11, 19, 21, 21, 22, 21, 21,
        18, 19, 16, 17, 13, 15, 11, 13, 11, 12, 18, 21, 20, 22, 22, 18, 20, 16,
        18, 13, 16, 11, 14, 10, 13, 10,
        /* Size 32x16 */
        64, 65, 65, 65, 64, 64, 64, 60, 59, 57, 52, 51, 47, 43, 42, 38, 36, 35,
        32, 30, 29, 26, 25, 25, 23, 23, 22, 21, 20, 20, 19, 18, 65, 64, 64, 63,
        63, 63, 62, 60, 60, 59, 55, 53, 50, 46, 45, 41, 39, 38, 34, 32, 31, 28,
        28, 27, 26, 25, 24, 23, 23, 22, 21, 21, 64, 63, 63, 63, 61, 60, 59, 58,
        57, 55, 52, 51, 48, 45, 44, 41, 39, 38, 35, 33, 32, 29, 28, 28, 26, 25,
        25, 24, 23, 22, 21, 20, 64, 63, 63, 62, 60, 59, 58, 55, 55, 53, 51, 49,
        48, 45, 44, 41, 39, 38, 35, 33, 32, 29, 29, 28, 27, 26, 25, 24, 24, 23,
        22, 22, 57, 58, 59, 59, 56, 55, 53, 48, 46, 43, 41, 40, 38, 37, 36, 34,
        33, 32, 30, 29, 28, 26, 25, 25, 24, 23, 23, 22, 22, 21, 21, 22, 52, 53,
        54, 55, 53, 51, 51, 46, 44, 40, 38, 37, 35, 34, 33, 32, 30, 29, 28, 27,
        26, 24, 24, 24, 23, 23, 23, 22, 22, 21, 21, 18, 47, 48, 49, 50, 49, 48,
        48, 43, 41, 38, 35, 34, 33, 31, 30, 29, 27, 27, 26, 25, 24, 23, 22, 22,
        21, 21, 20, 19, 19, 19, 18, 20, 38, 40, 41, 41, 41, 41, 41, 38, 36, 34,
        32, 31, 29, 27, 26, 25, 24, 23, 22, 21, 21, 20, 20, 19, 19, 19, 19, 20,
        20, 19, 19, 16, 35, 37, 38, 39, 39, 39, 39, 36, 34, 33, 30, 29, 27, 26,
        25, 24, 23, 22, 21, 20, 20, 19, 19, 18, 18, 18, 18, 17, 16, 16, 16, 18,
        31, 33, 34, 34, 35, 35, 35, 33, 32, 30, 28, 27, 26, 24, 24, 22, 21, 21,
        20, 19, 18, 17, 17, 17, 16, 16, 15, 15, 17, 17, 17, 13, 26, 27, 28, 28,
        29, 29, 30, 28, 27, 26, 24, 24, 23, 21, 21, 20, 19, 18, 17, 17, 16, 15,
        15, 15, 15, 15, 16, 15, 14, 13, 13, 16, 25, 26, 27, 28, 28, 29, 29, 27,
        27, 26, 24, 24, 22, 21, 21, 20, 19, 18, 17, 16, 16, 15, 15, 15, 14, 14,
        13, 13, 14, 15, 15, 11, 23, 24, 25, 25, 26, 26, 27, 25, 25, 24, 23, 22,
        21, 21, 20, 18, 18, 17, 16, 16, 15, 15, 14, 14, 14, 13, 13, 14, 13, 11,
        11, 14, 22, 23, 23, 24, 24, 24, 25, 25, 23, 23, 23, 21, 21, 19, 20, 18,
        18, 17, 16, 17, 14, 15, 14, 13, 13, 13, 13, 12, 12, 13, 13, 10, 20, 21,
        22, 23, 23, 23, 24, 24, 22, 23, 20, 22, 19, 21, 17, 19, 17, 17, 16, 16,
        15, 14, 14, 13, 14, 11, 13, 12, 13, 11, 11, 13, 19, 20, 20, 22, 21, 22,
        22, 23, 23, 20, 22, 19, 21, 17, 20, 16, 19, 15, 18, 13, 17, 13, 16, 12,
        15, 11, 14, 10, 13, 11, 12, 10,
        /* Size 4x16 */
        65, 52, 31, 22, 64, 54, 34, 23, 63, 53, 35, 24, 62, 51, 35, 25, 60, 44,
        32, 23, 55, 38, 28, 23, 50, 35, 26, 21, 45, 33, 24, 20, 39, 30, 21, 18,
        34, 28, 20, 16, 31, 26, 18, 14, 28, 24, 17, 14, 26, 23, 16, 13, 24, 23,
        15, 13, 23, 22, 17, 12, 21, 21, 17, 13,
        /* Size 16x4 */
        65, 64, 63, 62, 60, 55, 50, 45, 39, 34, 31, 28, 26, 24, 23, 21, 52, 54,
        53, 51, 44, 38, 35, 33, 30, 28, 26, 24, 23, 23, 22, 21, 31, 34, 35, 35,
        32, 28, 26, 24, 21, 20, 18, 17, 16, 15, 17, 17, 22, 23, 24, 25, 23, 23,
        21, 20, 18, 16, 14, 14, 13, 13, 12, 13,
        /* Size 8x32 */
        64, 64, 57, 47, 35, 26, 23, 20, 65, 63, 58, 48, 37, 27, 24, 21, 65, 63,
        59, 49, 38, 28, 25, 22, 65, 63, 59, 50, 39, 28, 25, 23, 64, 61, 56, 49,
        39, 29, 26, 23, 64, 60, 55, 48, 39, 29, 26, 23, 64, 59, 53, 48, 39, 30,
        27, 24, 60, 58, 48, 43, 36, 28, 25, 24, 59, 57, 46, 41, 34, 27, 25, 22,
        57, 55, 43, 38, 33, 26, 24, 23, 52, 52, 41, 35, 30, 24, 23, 20, 51, 51,
        40, 34, 29, 24, 22, 22, 47, 48, 38, 33, 27, 23, 21, 19, 43, 45, 37, 31,
        26, 21, 21, 21, 42, 44, 36, 30, 25, 21, 20, 17, 38, 41, 34, 29, 24, 20,
        18, 19, 36, 39, 33, 27, 23, 19, 18, 17, 35, 38, 32, 27, 22, 18, 17, 17,
        32, 35, 30, 26, 21, 17, 16, 16, 30, 33, 29, 25, 20, 17, 16, 16, 29, 32,
        28, 24, 20, 16, 15, 15, 26, 29, 26, 23, 19, 15, 15, 14, 25, 28, 25, 22,
        19, 15, 14, 14, 25, 28, 25, 22, 18, 15, 14, 13, 23, 26, 24, 21, 18, 15,
        14, 14, 23, 25, 23, 21, 18, 15, 13, 11, 22, 25, 23, 20, 18, 16, 13, 13,
        21, 24, 22, 19, 17, 15, 14, 12, 20, 23, 22, 19, 16, 14, 13, 13, 20, 22,
        21, 19, 16, 13, 11, 11, 19, 21, 21, 18, 16, 13, 11, 11, 18, 20, 22, 20,
        18, 16, 14, 13,
        /* Size 32x8 */
        64, 65, 65, 65, 64, 64, 64, 60, 59, 57, 52, 51, 47, 43, 42, 38, 36, 35,
        32, 30, 29, 26, 25, 25, 23, 23, 22, 21, 20, 20, 19, 18, 64, 63, 63, 63,
        61, 60, 59, 58, 57, 55, 52, 51, 48, 45, 44, 41, 39, 38, 35, 33, 32, 29,
        28, 28, 26, 25, 25, 24, 23, 22, 21, 20, 57, 58, 59, 59, 56, 55, 53, 48,
        46, 43, 41, 40, 38, 37, 36, 34, 33, 32, 30, 29, 28, 26, 25, 25, 24, 23,
        23, 22, 22, 21, 21, 22, 47, 48, 49, 50, 49, 48, 48, 43, 41, 38, 35, 34,
        33, 31, 30, 29, 27, 27, 26, 25, 24, 23, 22, 22, 21, 21, 20, 19, 19, 19,
        18, 20, 35, 37, 38, 39, 39, 39, 39, 36, 34, 33, 30, 29, 27, 26, 25, 24,
        23, 22, 21, 20, 20, 19, 19, 18, 18, 18, 18, 17, 16, 16, 16, 18, 26, 27,
        28, 28, 29, 29, 30, 28, 27, 26, 24, 24, 23, 21, 21, 20, 19, 18, 17, 17,
        16, 15, 15, 15, 15, 15, 16, 15, 14, 13, 13, 16, 23, 24, 25, 25, 26, 26,
        27, 25, 25, 24, 23, 22, 21, 21, 20, 18, 18, 17, 16, 16, 15, 15, 14, 14,
        14, 13, 13, 14, 13, 11, 11, 14, 20, 21, 22, 23, 23, 23, 24, 24, 22, 23,
        20, 22, 19, 21, 17, 19, 17, 17, 16, 16, 15, 14, 14, 13, 14, 11, 13, 12,
        13, 11, 11, 13 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        42, 35, 35, 22,
#endif
        /* Size 4x4 */
        64, 45, 38, 32, 45, 37, 33, 29, 38, 33, 26, 25, 32, 29, 25, 20,
        /* Size 8x8 */
        65, 56, 43, 42, 39, 34, 32, 29, 56, 48, 43, 44, 42, 37, 34, 32, 43, 43,
        39, 38, 37, 34, 32, 32, 42, 44, 38, 34, 32, 29, 29, 29, 39, 42, 37, 32,
        29, 26, 25, 26, 34, 37, 34, 29, 26, 23, 23, 22, 32, 34, 32, 29, 25, 23,
        21, 19, 29, 32, 32, 29, 26, 22, 19, 19,
        /* Size 16x16 */
        64, 67, 61, 56, 46, 42, 42, 40, 38, 36, 34, 32, 31, 29, 28, 27, 67, 65,
        59, 52, 47, 45, 45, 43, 41, 39, 37, 34, 33, 32, 31, 29, 61, 59, 53, 47,
        45, 44, 45, 44, 42, 40, 38, 36, 35, 34, 32, 31, 56, 52, 47, 43, 43, 44,
        45, 44, 43, 41, 39, 37, 36, 35, 34, 33, 46, 47, 45, 43, 41, 40, 40, 40,
        39, 38, 36, 34, 34, 33, 31, 30, 42, 45, 44, 44, 40, 38, 37, 36, 35, 34,
        33, 32, 31, 32, 30, 29, 42, 45, 45, 45, 40, 37, 35, 34, 33, 32, 31, 30,
        29, 29, 29, 27, 40, 43, 44, 44, 40, 36, 34, 33, 31, 30, 29, 28, 28, 27,
        28, 26, 38, 41, 42, 43, 39, 35, 33, 31, 29, 28, 27, 26, 26, 25, 27, 24,
        36, 39, 40, 41, 38, 34, 32, 30, 28, 27, 26, 25, 24, 25, 26, 23, 34, 37,
        38, 39, 36, 33, 31, 29, 27, 26, 25, 24, 23, 24, 24, 22, 32, 34, 36, 37,
        34, 32, 30, 28, 26, 25, 24, 23, 22, 22, 22, 20, 31, 33, 35, 36, 34, 31,
        29, 28, 26, 24, 23, 22, 21, 21, 20, 19, 29, 32, 34, 35, 33, 32, 29, 27,
        25, 25, 24, 22, 21, 20, 20, 18, 28, 31, 32, 34, 31, 30, 29, 28, 27, 26,
        24, 22, 20, 20, 19, 19, 27, 29, 31, 33, 30, 29, 27, 26, 24, 23, 22, 20,
        19, 18, 19, 18,
        /* Size 32x32 */
        64, 66, 67, 67, 61, 59, 56, 49, 46, 42, 42, 42, 42, 41, 40, 39, 38, 37,
        36, 35, 34, 32, 32, 32, 31, 30, 29, 29, 28, 28, 27, 27, 66, 65, 65, 66,
        59, 56, 53, 48, 46, 43, 44, 44, 44, 42, 42, 41, 40, 39, 38, 37, 36, 34,
        34, 33, 32, 32, 31, 31, 30, 29, 29, 29, 67, 65, 65, 65, 59, 55, 52, 48,
        47, 44, 45, 45, 45, 44, 43, 42, 41, 40, 39, 38, 37, 35, 34, 34, 33, 33,
        32, 31, 31, 30, 29, 28, 67, 66, 65, 64, 58, 55, 51, 48, 47, 44, 45, 45,
        46, 44, 44, 43, 41, 41, 39, 38, 38, 36, 35, 35, 34, 33, 33, 32, 32, 31,
        30, 30, 61, 59, 59, 58, 53, 50, 47, 46, 45, 43, 44, 45, 45, 44, 44, 43,
        42, 41, 40, 39, 38, 36, 36, 36, 35, 34, 34, 33, 32, 32, 31, 30, 59, 56,
        55, 55, 50, 48, 45, 45, 44, 43, 44, 45, 45, 44, 44, 44, 42, 42, 41, 39,
        39, 37, 37, 36, 35, 34, 34, 33, 32, 32, 31, 31, 56, 53, 52, 51, 47, 45,
        43, 43, 43, 43, 44, 45, 45, 45, 44, 44, 43, 42, 41, 40, 39, 38, 37, 37,
        36, 35, 35, 34, 34, 33, 33, 31, 49, 48, 48, 48, 46, 45, 43, 42, 41, 41,
        41, 41, 42, 41, 41, 41, 40, 39, 38, 38, 37, 36, 35, 35, 34, 34, 34, 34,
        34, 33, 33, 33, 46, 46, 47, 47, 45, 44, 43, 41, 41, 40, 40, 40, 40, 40,
        40, 39, 39, 38, 38, 37, 36, 35, 34, 34, 34, 34, 33, 32, 31, 31, 30, 32,
        42, 43, 44, 44, 43, 43, 43, 41, 40, 39, 39, 38, 38, 38, 38, 38, 37, 37,
        36, 35, 35, 34, 34, 33, 33, 32, 31, 31, 32, 32, 31, 29, 42, 44, 45, 45,
        44, 44, 44, 41, 40, 39, 38, 38, 37, 36, 36, 36, 35, 35, 34, 34, 33, 32,
        32, 32, 31, 31, 32, 31, 30, 29, 29, 31, 42, 44, 45, 45, 45, 45, 45, 41,
        40, 38, 38, 37, 36, 36, 36, 35, 34, 34, 34, 33, 33, 32, 31, 31, 31, 31,
        30, 29, 30, 31, 30, 27, 42, 44, 45, 46, 45, 45, 45, 42, 40, 38, 37, 36,
        35, 34, 34, 33, 33, 33, 32, 32, 31, 30, 30, 30, 29, 29, 29, 30, 29, 27,
        27, 30, 41, 42, 44, 44, 44, 44, 45, 41, 40, 38, 36, 36, 34, 34, 33, 32,
        32, 32, 31, 30, 30, 29, 29, 29, 28, 29, 29, 27, 27, 29, 29, 26, 40, 42,
        43, 44, 44, 44, 44, 41, 40, 38, 36, 36, 34, 33, 33, 32, 31, 31, 30, 30,
        29, 29, 28, 28, 28, 28, 27, 28, 28, 26, 26, 29, 39, 41, 42, 43, 43, 44,
        44, 41, 39, 38, 36, 35, 33, 32, 32, 31, 30, 30, 29, 28, 28, 27, 27, 27,
        27, 26, 27, 26, 26, 28, 28, 24, 38, 40, 41, 41, 42, 42, 43, 40, 39, 37,
        35, 34, 33, 32, 31, 30, 29, 29, 28, 27, 27, 26, 26, 26, 26, 26, 25, 26,
        27, 25, 24, 27, 37, 39, 40, 41, 41, 42, 42, 39, 38, 37, 35, 34, 33, 32,
        31, 30, 29, 29, 28, 27, 27, 26, 26, 26, 25, 25, 25, 26, 24, 26, 26, 23,
        36, 38, 39, 39, 40, 41, 41, 38, 38, 36, 34, 34, 32, 31, 30, 29, 28, 28,
        27, 26, 26, 25, 25, 25, 24, 25, 25, 24, 26, 24, 23, 26, 35, 37, 38, 38,
        39, 39, 40, 38, 37, 35, 34, 33, 32, 30, 30, 28, 27, 27, 26, 26, 25, 24,
        24, 24, 24, 24, 23, 25, 23, 24, 25, 21, 34, 36, 37, 38, 38, 39, 39, 37,
        36, 35, 33, 33, 31, 30, 29, 28, 27, 27, 26, 25, 25, 24, 24, 24, 23, 23,
        24, 22, 24, 23, 22, 25, 32, 34, 35, 36, 36, 37, 38, 36, 35, 34, 32, 32,
        30, 29, 29, 27, 26, 26, 25, 24, 24, 23, 23, 23, 22, 23, 22, 23, 22, 22,
        24, 21, 32, 34, 34, 35, 36, 37, 37, 35, 34, 34, 32, 31, 30, 29, 28, 27,
        26, 26, 25, 24, 24, 23, 23, 23, 22, 22, 22, 22, 22, 23, 20, 24, 32, 33,
        34, 35, 36, 36, 37, 35, 34, 33, 32, 31, 30, 29, 28, 27, 26, 26, 25, 24,
        24, 23, 23, 22, 22, 22, 21, 21, 22, 20, 23, 19, 31, 32, 33, 34, 35, 35,
        36, 34, 34, 33, 31, 31, 29, 28, 28, 27, 26, 25, 24, 24, 23, 22, 22, 22,
        21, 22, 21, 21, 20, 22, 19, 23, 30, 32, 33, 33, 34, 34, 35, 34, 34, 32,
        31, 31, 29, 29, 28, 26, 26, 25, 25, 24, 23, 23, 22, 22, 22, 21, 21, 21,
        21, 19, 22, 18, 29, 31, 32, 33, 34, 34, 35, 34, 33, 31, 32, 30, 29, 29,
        27, 27, 25, 25, 25, 23, 24, 22, 22, 21, 21, 21, 20, 21, 20, 21, 18, 21,
        29, 31, 31, 32, 33, 33, 34, 34, 32, 31, 31, 29, 30, 27, 28, 26, 26, 26,
        24, 25, 22, 23, 22, 21, 21, 21, 21, 20, 21, 20, 21, 17, 28, 30, 31, 32,
        32, 32, 34, 34, 31, 32, 30, 30, 29, 27, 28, 26, 27, 24, 26, 23, 24, 22,
        22, 22, 20, 21, 20, 21, 19, 20, 19, 20, 28, 29, 30, 31, 32, 32, 33, 33,
        31, 32, 29, 31, 27, 29, 26, 28, 25, 26, 24, 24, 23, 22, 23, 20, 22, 19,
        21, 20, 20, 19, 20, 19, 27, 29, 29, 30, 31, 31, 33, 33, 30, 31, 29, 30,
        27, 29, 26, 28, 24, 26, 23, 25, 22, 24, 20, 23, 19, 22, 18, 21, 19, 20,
        18, 20, 27, 29, 28, 30, 30, 31, 31, 33, 32, 29, 31, 27, 30, 26, 29, 24,
        27, 23, 26, 21, 25, 21, 24, 19, 23, 18, 21, 17, 20, 19, 20, 18,
        /* Size 4x8 */
        65, 44, 38, 31, 54, 44, 41, 33, 44, 38, 36, 32, 44, 36, 31, 27, 41, 35,
        28, 26, 36, 32, 25, 23, 33, 31, 24, 21, 31, 31, 25, 21,
        /* Size 8x4 */
        65, 54, 44, 44, 41, 36, 33, 31, 44, 44, 38, 36, 35, 32, 31, 31, 38, 41,
        36, 31, 28, 25, 24, 25, 31, 33, 32, 27, 26, 23, 21, 21,
        /* Size 8x16 */
        64, 60, 43, 42, 38, 32, 30, 28, 66, 57, 44, 45, 41, 35, 33, 31, 61, 51,
        44, 45, 42, 37, 34, 32, 55, 47, 43, 45, 43, 38, 36, 34, 46, 45, 40, 40,
        39, 35, 34, 31, 42, 44, 39, 37, 35, 32, 31, 29, 42, 45, 38, 35, 33, 30,
        29, 28, 40, 44, 38, 34, 31, 29, 28, 26, 38, 42, 37, 33, 29, 26, 26, 25,
        36, 40, 36, 32, 28, 25, 24, 24, 34, 38, 35, 31, 27, 24, 23, 23, 32, 36,
        34, 30, 26, 23, 22, 22, 31, 35, 33, 29, 26, 22, 22, 22, 29, 33, 32, 29,
        26, 24, 21, 21, 28, 32, 31, 27, 24, 22, 21, 21, 27, 31, 31, 27, 24, 22,
        19, 19,
        /* Size 16x8 */
        64, 66, 61, 55, 46, 42, 42, 40, 38, 36, 34, 32, 31, 29, 28, 27, 60, 57,
        51, 47, 45, 44, 45, 44, 42, 40, 38, 36, 35, 33, 32, 31, 43, 44, 44, 43,
        40, 39, 38, 38, 37, 36, 35, 34, 33, 32, 31, 31, 42, 45, 45, 45, 40, 37,
        35, 34, 33, 32, 31, 30, 29, 29, 27, 27, 38, 41, 42, 43, 39, 35, 33, 31,
        29, 28, 27, 26, 26, 26, 24, 24, 32, 35, 37, 38, 35, 32, 30, 29, 26, 25,
        24, 23, 22, 24, 22, 22, 30, 33, 34, 36, 34, 31, 29, 28, 26, 24, 23, 22,
        22, 21, 21, 19, 28, 31, 32, 34, 31, 29, 28, 26, 25, 24, 23, 22, 22, 21,
        21, 19,
        /* Size 16x32 */
        64, 66, 60, 55, 43, 42, 42, 39, 38, 36, 32, 32, 30, 29, 28, 27, 65, 65,
        58, 53, 44, 44, 44, 41, 40, 38, 34, 34, 32, 31, 30, 29, 66, 64, 57, 52,
        44, 45, 45, 42, 41, 39, 35, 34, 33, 32, 31, 29, 67, 64, 56, 51, 45, 45,
        45, 43, 41, 39, 36, 35, 34, 33, 32, 31, 61, 57, 51, 48, 44, 44, 45, 43,
        42, 40, 37, 36, 34, 33, 32, 30, 58, 54, 49, 46, 43, 44, 45, 43, 42, 41,
        37, 37, 35, 33, 32, 32, 55, 51, 47, 44, 43, 44, 45, 44, 43, 41, 38, 37,
        36, 35, 34, 31, 49, 48, 45, 43, 41, 41, 41, 41, 40, 38, 36, 35, 34, 34,
        34, 34, 46, 46, 45, 43, 40, 40, 40, 39, 39, 38, 35, 34, 34, 32, 31, 33,
        42, 44, 43, 43, 39, 38, 38, 38, 37, 36, 34, 34, 32, 32, 32, 29, 42, 45,
        44, 44, 39, 38, 37, 36, 35, 34, 32, 32, 31, 32, 29, 31, 42, 45, 45, 44,
        39, 37, 36, 35, 34, 34, 32, 31, 31, 29, 31, 28, 42, 45, 45, 45, 38, 37,
        35, 33, 33, 32, 30, 30, 29, 30, 28, 30, 41, 44, 44, 44, 38, 36, 34, 32,
        32, 31, 29, 29, 29, 27, 29, 26, 40, 44, 44, 44, 38, 36, 34, 32, 31, 30,
        29, 28, 28, 28, 26, 29, 39, 43, 43, 43, 38, 36, 34, 31, 30, 29, 27, 27,
        26, 27, 28, 25, 38, 41, 42, 42, 37, 35, 33, 30, 29, 28, 26, 26, 26, 26,
        25, 28, 38, 41, 41, 42, 37, 35, 33, 30, 29, 28, 26, 26, 25, 26, 26, 23,
        36, 39, 40, 41, 36, 34, 32, 29, 28, 27, 25, 25, 24, 24, 24, 26, 35, 38,
        39, 39, 35, 33, 32, 28, 27, 26, 24, 24, 24, 25, 24, 22, 34, 38, 38, 39,
        35, 33, 31, 28, 27, 26, 24, 24, 23, 22, 23, 25, 32, 36, 37, 37, 34, 32,
        30, 27, 26, 25, 23, 23, 23, 23, 22, 21, 32, 35, 36, 37, 34, 32, 30, 27,
        26, 25, 23, 23, 22, 22, 22, 24, 32, 35, 36, 37, 33, 32, 30, 27, 26, 25,
        23, 23, 22, 21, 21, 19, 31, 34, 35, 36, 33, 31, 29, 27, 26, 24, 22, 22,
        22, 21, 22, 23, 30, 33, 34, 35, 32, 31, 29, 27, 26, 24, 23, 22, 21, 21,
        19, 19, 29, 33, 33, 34, 32, 32, 29, 27, 26, 23, 24, 21, 21, 21, 21, 22,
        29, 32, 33, 34, 32, 31, 27, 28, 25, 24, 23, 21, 21, 20, 20, 17, 28, 31,
        32, 33, 31, 31, 27, 28, 24, 26, 22, 22, 21, 20, 21, 21, 28, 31, 32, 33,
        31, 31, 27, 28, 24, 25, 21, 23, 19, 21, 19, 19, 27, 30, 31, 32, 31, 30,
        27, 28, 24, 25, 22, 23, 19, 21, 19, 20, 27, 30, 30, 33, 32, 28, 30, 25,
        27, 22, 25, 19, 23, 17, 20, 18,
        /* Size 32x16 */
        64, 65, 66, 67, 61, 58, 55, 49, 46, 42, 42, 42, 42, 41, 40, 39, 38, 38,
        36, 35, 34, 32, 32, 32, 31, 30, 29, 29, 28, 28, 27, 27, 66, 65, 64, 64,
        57, 54, 51, 48, 46, 44, 45, 45, 45, 44, 44, 43, 41, 41, 39, 38, 38, 36,
        35, 35, 34, 33, 33, 32, 31, 31, 30, 30, 60, 58, 57, 56, 51, 49, 47, 45,
        45, 43, 44, 45, 45, 44, 44, 43, 42, 41, 40, 39, 38, 37, 36, 36, 35, 34,
        33, 33, 32, 32, 31, 30, 55, 53, 52, 51, 48, 46, 44, 43, 43, 43, 44, 44,
        45, 44, 44, 43, 42, 42, 41, 39, 39, 37, 37, 37, 36, 35, 34, 34, 33, 33,
        32, 33, 43, 44, 44, 45, 44, 43, 43, 41, 40, 39, 39, 39, 38, 38, 38, 38,
        37, 37, 36, 35, 35, 34, 34, 33, 33, 32, 32, 32, 31, 31, 31, 32, 42, 44,
        45, 45, 44, 44, 44, 41, 40, 38, 38, 37, 37, 36, 36, 36, 35, 35, 34, 33,
        33, 32, 32, 32, 31, 31, 32, 31, 31, 31, 30, 28, 42, 44, 45, 45, 45, 45,
        45, 41, 40, 38, 37, 36, 35, 34, 34, 34, 33, 33, 32, 32, 31, 30, 30, 30,
        29, 29, 29, 27, 27, 27, 27, 30, 39, 41, 42, 43, 43, 43, 44, 41, 39, 38,
        36, 35, 33, 32, 32, 31, 30, 30, 29, 28, 28, 27, 27, 27, 27, 27, 27, 28,
        28, 28, 28, 25, 38, 40, 41, 41, 42, 42, 43, 40, 39, 37, 35, 34, 33, 32,
        31, 30, 29, 29, 28, 27, 27, 26, 26, 26, 26, 26, 26, 25, 24, 24, 24, 27,
        36, 38, 39, 39, 40, 41, 41, 38, 38, 36, 34, 34, 32, 31, 30, 29, 28, 28,
        27, 26, 26, 25, 25, 25, 24, 24, 23, 24, 26, 25, 25, 22, 32, 34, 35, 36,
        37, 37, 38, 36, 35, 34, 32, 32, 30, 29, 29, 27, 26, 26, 25, 24, 24, 23,
        23, 23, 22, 23, 24, 23, 22, 21, 22, 25, 32, 34, 34, 35, 36, 37, 37, 35,
        34, 34, 32, 31, 30, 29, 28, 27, 26, 26, 25, 24, 24, 23, 23, 23, 22, 22,
        21, 21, 22, 23, 23, 19, 30, 32, 33, 34, 34, 35, 36, 34, 34, 32, 31, 31,
        29, 29, 28, 26, 26, 25, 24, 24, 23, 23, 22, 22, 22, 21, 21, 21, 21, 19,
        19, 23, 29, 31, 32, 33, 33, 33, 35, 34, 32, 32, 32, 29, 30, 27, 28, 27,
        26, 26, 24, 25, 22, 23, 22, 21, 21, 21, 21, 20, 20, 21, 21, 17, 28, 30,
        31, 32, 32, 32, 34, 34, 31, 32, 29, 31, 28, 29, 26, 28, 25, 26, 24, 24,
        23, 22, 22, 21, 22, 19, 21, 20, 21, 19, 19, 20, 27, 29, 29, 31, 30, 32,
        31, 34, 33, 29, 31, 28, 30, 26, 29, 25, 28, 23, 26, 22, 25, 21, 24, 19,
        23, 19, 22, 17, 21, 19, 20, 18,
        /* Size 4x16 */
        66, 42, 36, 29, 64, 45, 39, 32, 57, 44, 40, 33, 51, 44, 41, 35, 46, 40,
        38, 32, 45, 38, 34, 32, 45, 37, 32, 30, 44, 36, 30, 28, 41, 35, 28, 26,
        39, 34, 27, 24, 38, 33, 26, 22, 35, 32, 25, 22, 34, 31, 24, 21, 33, 32,
        23, 21, 31, 31, 26, 20, 30, 30, 25, 21,
        /* Size 16x4 */
        66, 64, 57, 51, 46, 45, 45, 44, 41, 39, 38, 35, 34, 33, 31, 30, 42, 45,
        44, 44, 40, 38, 37, 36, 35, 34, 33, 32, 31, 32, 31, 30, 36, 39, 40, 41,
        38, 34, 32, 30, 28, 27, 26, 25, 24, 23, 26, 25, 29, 32, 33, 35, 32, 32,
        30, 28, 26, 24, 22, 22, 21, 21, 20, 21,
        /* Size 8x32 */
        64, 60, 43, 42, 38, 32, 30, 28, 65, 58, 44, 44, 40, 34, 32, 30, 66, 57,
        44, 45, 41, 35, 33, 31, 67, 56, 45, 45, 41, 36, 34, 32, 61, 51, 44, 45,
        42, 37, 34, 32, 58, 49, 43, 45, 42, 37, 35, 32, 55, 47, 43, 45, 43, 38,
        36, 34, 49, 45, 41, 41, 40, 36, 34, 34, 46, 45, 40, 40, 39, 35, 34, 31,
        42, 43, 39, 38, 37, 34, 32, 32, 42, 44, 39, 37, 35, 32, 31, 29, 42, 45,
        39, 36, 34, 32, 31, 31, 42, 45, 38, 35, 33, 30, 29, 28, 41, 44, 38, 34,
        32, 29, 29, 29, 40, 44, 38, 34, 31, 29, 28, 26, 39, 43, 38, 34, 30, 27,
        26, 28, 38, 42, 37, 33, 29, 26, 26, 25, 38, 41, 37, 33, 29, 26, 25, 26,
        36, 40, 36, 32, 28, 25, 24, 24, 35, 39, 35, 32, 27, 24, 24, 24, 34, 38,
        35, 31, 27, 24, 23, 23, 32, 37, 34, 30, 26, 23, 23, 22, 32, 36, 34, 30,
        26, 23, 22, 22, 32, 36, 33, 30, 26, 23, 22, 21, 31, 35, 33, 29, 26, 22,
        22, 22, 30, 34, 32, 29, 26, 23, 21, 19, 29, 33, 32, 29, 26, 24, 21, 21,
        29, 33, 32, 27, 25, 23, 21, 20, 28, 32, 31, 27, 24, 22, 21, 21, 28, 32,
        31, 27, 24, 21, 19, 19, 27, 31, 31, 27, 24, 22, 19, 19, 27, 30, 32, 30,
        27, 25, 23, 20,
        /* Size 32x8 */
        64, 65, 66, 67, 61, 58, 55, 49, 46, 42, 42, 42, 42, 41, 40, 39, 38, 38,
        36, 35, 34, 32, 32, 32, 31, 30, 29, 29, 28, 28, 27, 27, 60, 58, 57, 56,
        51, 49, 47, 45, 45, 43, 44, 45, 45, 44, 44, 43, 42, 41, 40, 39, 38, 37,
        36, 36, 35, 34, 33, 33, 32, 32, 31, 30, 43, 44, 44, 45, 44, 43, 43, 41,
        40, 39, 39, 39, 38, 38, 38, 38, 37, 37, 36, 35, 35, 34, 34, 33, 33, 32,
        32, 32, 31, 31, 31, 32, 42, 44, 45, 45, 45, 45, 45, 41, 40, 38, 37, 36,
        35, 34, 34, 34, 33, 33, 32, 32, 31, 30, 30, 30, 29, 29, 29, 27, 27, 27,
        27, 30, 38, 40, 41, 41, 42, 42, 43, 40, 39, 37, 35, 34, 33, 32, 31, 30,
        29, 29, 28, 27, 27, 26, 26, 26, 26, 26, 26, 25, 24, 24, 24, 27, 32, 34,
        35, 36, 37, 37, 38, 36, 35, 34, 32, 32, 30, 29, 29, 27, 26, 26, 25, 24,
        24, 23, 23, 23, 22, 23, 24, 23, 22, 21, 22, 25, 30, 32, 33, 34, 34, 35,
        36, 34, 34, 32, 31, 31, 29, 29, 28, 26, 26, 25, 24, 24, 23, 23, 22, 22,
        22, 21, 21, 21, 21, 19, 19, 23, 28, 30, 31, 32, 32, 32, 34, 34, 31, 32,
        29, 31, 28, 29, 26, 28, 25, 26, 24, 24, 23, 22, 22, 21, 22, 19, 21, 20,
        21, 19, 19, 20 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        56, 29, 29, 15,
#endif
        /* Size 4x4 */
        63, 55, 35, 25, 55, 38, 28, 22, 35, 28, 20, 17, 25, 22, 17, 13,
        /* Size 8x8 */
        64, 64, 59, 48, 38, 30, 26, 22, 64, 61, 57, 49, 40, 32, 28, 24, 59, 57,
        45, 39, 34, 28, 26, 24, 48, 49, 39, 32, 28, 24, 22, 21, 38, 40, 34, 28,
        24, 20, 19, 18, 30, 32, 28, 24, 20, 17, 16, 15, 26, 28, 26, 22, 19, 16,
        15, 12, 22, 24, 24, 21, 18, 15, 12, 12,
        /* Size 16x16 */
        64, 65, 65, 64, 60, 57, 50, 43, 38, 35, 31, 27, 25, 23, 22, 21, 65, 64,
        63, 63, 60, 59, 52, 46, 41, 37, 34, 29, 27, 25, 23, 22, 65, 63, 62, 61,
        59, 57, 51, 46, 41, 38, 34, 31, 28, 26, 25, 23, 64, 63, 61, 58, 55, 53,
        49, 46, 41, 38, 35, 31, 29, 27, 25, 24, 60, 60, 59, 55, 52, 48, 45, 41,
        38, 35, 33, 29, 27, 25, 24, 23, 57, 59, 57, 53, 48, 42, 39, 37, 34, 32,
        30, 27, 25, 24, 23, 22, 50, 52, 51, 49, 45, 39, 36, 33, 30, 29, 27, 25,
        23, 22, 21, 20, 43, 46, 46, 46, 41, 37, 33, 29, 27, 26, 24, 22, 21, 20,
        20, 19, 38, 41, 41, 41, 38, 34, 30, 27, 25, 24, 22, 20, 19, 19, 19, 17,
        35, 37, 38, 38, 35, 32, 29, 26, 24, 22, 21, 19, 18, 18, 18, 16, 31, 34,
        34, 35, 33, 30, 27, 24, 22, 21, 20, 18, 17, 16, 17, 15, 27, 29, 31, 31,
        29, 27, 25, 22, 20, 19, 18, 17, 16, 15, 15, 14, 25, 27, 28, 29, 27, 25,
        23, 21, 19, 18, 17, 16, 15, 14, 13, 13, 23, 25, 26, 27, 25, 24, 22, 20,
        19, 18, 16, 15, 14, 14, 13, 12, 22, 23, 25, 25, 24, 23, 21, 20, 19, 18,
        17, 15, 13, 13, 13, 12, 21, 22, 23, 24, 23, 22, 20, 19, 17, 16, 15, 14,
        13, 12, 12, 12,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 65, 64, 63, 60, 58, 57, 52, 50, 47, 43, 42, 38, 37,
        35, 32, 31, 29, 27, 26, 25, 25, 23, 23, 22, 21, 21, 20, 65, 64, 64, 64,
        64, 64, 63, 62, 60, 59, 59, 53, 51, 48, 45, 44, 40, 38, 36, 34, 33, 30,
        29, 27, 26, 26, 25, 24, 23, 22, 22, 21, 65, 64, 64, 64, 63, 63, 63, 62,
        60, 59, 59, 54, 52, 49, 46, 45, 41, 39, 37, 34, 34, 31, 29, 27, 27, 27,
        25, 24, 23, 23, 22, 21, 65, 64, 64, 63, 63, 63, 63, 62, 61, 60, 59, 55,
        53, 50, 47, 46, 41, 40, 38, 35, 34, 32, 30, 28, 27, 27, 26, 25, 24, 23,
        23, 22, 65, 64, 63, 63, 62, 62, 61, 60, 59, 57, 57, 53, 51, 49, 46, 45,
        41, 40, 38, 35, 34, 32, 31, 29, 28, 28, 26, 25, 25, 24, 23, 22, 65, 64,
        63, 63, 62, 61, 60, 59, 58, 57, 56, 53, 51, 49, 46, 45, 41, 40, 38, 35,
        35, 32, 31, 29, 28, 28, 26, 25, 25, 24, 23, 23, 64, 63, 63, 63, 61, 60,
        58, 57, 55, 54, 53, 51, 49, 48, 46, 45, 41, 40, 38, 36, 35, 33, 31, 29,
        29, 28, 27, 26, 25, 25, 24, 23, 63, 62, 62, 62, 60, 59, 57, 56, 54, 52,
        51, 49, 48, 46, 44, 43, 40, 39, 37, 35, 34, 32, 30, 29, 28, 28, 26, 26,
        26, 25, 24, 24, 60, 60, 60, 61, 59, 58, 55, 54, 52, 49, 48, 46, 45, 43,
        41, 40, 38, 37, 35, 33, 33, 30, 29, 28, 27, 27, 25, 25, 24, 23, 23, 23,
        58, 59, 59, 60, 57, 57, 54, 52, 49, 45, 44, 41, 41, 39, 37, 37, 35, 34,
        33, 31, 31, 28, 28, 26, 26, 26, 24, 24, 24, 24, 23, 21, 57, 59, 59, 59,
        57, 56, 53, 51, 48, 44, 42, 41, 39, 38, 37, 36, 34, 33, 32, 31, 30, 28,
        27, 26, 25, 25, 24, 24, 23, 22, 22, 22, 52, 53, 54, 55, 53, 53, 51, 49,
        46, 41, 41, 38, 37, 35, 34, 34, 32, 31, 30, 28, 28, 26, 25, 24, 24, 24,
        23, 22, 22, 22, 22, 20, 50, 51, 52, 53, 51, 51, 49, 48, 45, 41, 39, 37,
        36, 34, 33, 32, 30, 30, 29, 27, 27, 25, 25, 24, 23, 23, 22, 22, 21, 20,
        20, 21, 47, 48, 49, 50, 49, 49, 48, 46, 43, 39, 38, 35, 34, 32, 31, 31,
        29, 28, 27, 26, 26, 24, 24, 23, 22, 22, 21, 20, 20, 21, 21, 19, 43, 45,
        46, 47, 46, 46, 46, 44, 41, 37, 37, 34, 33, 31, 29, 29, 27, 27, 26, 25,
        24, 23, 22, 22, 21, 21, 20, 21, 20, 19, 19, 20, 42, 44, 45, 46, 45, 45,
        45, 43, 40, 37, 36, 34, 32, 31, 29, 29, 27, 26, 25, 24, 24, 23, 22, 21,
        21, 21, 20, 19, 19, 20, 20, 17, 38, 40, 41, 41, 41, 41, 41, 40, 38, 35,
        34, 32, 30, 29, 27, 27, 25, 24, 24, 22, 22, 21, 20, 20, 19, 19, 19, 19,
        19, 18, 17, 19, 37, 38, 39, 40, 40, 40, 40, 39, 37, 34, 33, 31, 30, 28,
        27, 26, 24, 24, 23, 22, 22, 20, 20, 19, 19, 19, 18, 18, 17, 18, 18, 16,
        35, 36, 37, 38, 38, 38, 38, 37, 35, 33, 32, 30, 29, 27, 26, 25, 24, 23,
        22, 21, 21, 20, 19, 19, 18, 18, 18, 17, 18, 17, 16, 18, 32, 34, 34, 35,
        35, 35, 36, 35, 33, 31, 31, 28, 27, 26, 25, 24, 22, 22, 21, 20, 20, 19,
        18, 18, 17, 17, 17, 17, 16, 17, 17, 15, 31, 33, 34, 34, 34, 35, 35, 34,
        33, 31, 30, 28, 27, 26, 24, 24, 22, 22, 21, 20, 20, 18, 18, 17, 17, 17,
        16, 16, 17, 16, 15, 17, 29, 30, 31, 32, 32, 32, 33, 32, 30, 28, 28, 26,
        25, 24, 23, 23, 21, 20, 20, 19, 18, 17, 17, 16, 16, 16, 15, 16, 15, 15,
        16, 14, 27, 29, 29, 30, 31, 31, 31, 30, 29, 28, 27, 25, 25, 24, 22, 22,
        20, 20, 19, 18, 18, 17, 17, 16, 16, 16, 15, 15, 15, 15, 14, 16, 26, 27,
        27, 28, 29, 29, 29, 29, 28, 26, 26, 24, 24, 23, 22, 21, 20, 19, 19, 18,
        17, 16, 16, 15, 15, 15, 14, 14, 15, 14, 15, 13, 25, 26, 27, 27, 28, 28,
        29, 28, 27, 26, 25, 24, 23, 22, 21, 21, 19, 19, 18, 17, 17, 16, 16, 15,
        15, 15, 14, 14, 13, 15, 13, 15, 25, 26, 27, 27, 28, 28, 28, 28, 27, 26,
        25, 24, 23, 22, 21, 21, 19, 19, 18, 17, 17, 16, 16, 15, 15, 15, 14, 14,
        14, 12, 14, 12, 23, 25, 25, 26, 26, 26, 27, 26, 25, 24, 24, 23, 22, 21,
        20, 20, 19, 18, 18, 17, 16, 15, 15, 14, 14, 14, 14, 14, 13, 14, 12, 14,
        23, 24, 24, 25, 25, 25, 26, 26, 25, 24, 24, 22, 22, 20, 21, 19, 19, 18,
        17, 17, 16, 16, 15, 14, 14, 14, 14, 13, 14, 13, 13, 11, 22, 23, 23, 24,
        25, 25, 25, 26, 24, 24, 23, 22, 21, 20, 20, 19, 19, 17, 18, 16, 17, 15,
        15, 15, 13, 14, 13, 14, 13, 13, 12, 13, 21, 22, 23, 23, 24, 24, 25, 25,
        23, 24, 22, 22, 20, 21, 19, 20, 18, 18, 17, 17, 16, 15, 15, 14, 15, 12,
        14, 13, 13, 12, 13, 12, 21, 22, 22, 23, 23, 23, 24, 24, 23, 23, 22, 22,
        20, 21, 19, 20, 17, 18, 16, 17, 15, 16, 14, 15, 13, 14, 12, 13, 12, 13,
        12, 13, 20, 21, 21, 22, 22, 23, 23, 24, 23, 21, 22, 20, 21, 19, 20, 17,
        19, 16, 18, 15, 17, 14, 16, 13, 15, 12, 14, 11, 13, 12, 13, 11,
        /* Size 4x8 */
        64, 58, 34, 24, 63, 56, 36, 26, 59, 44, 32, 24, 50, 38, 26, 21, 40, 33,
        22, 18, 31, 28, 19, 16, 27, 25, 17, 14, 23, 22, 18, 14,
        /* Size 8x4 */
        64, 63, 59, 50, 40, 31, 27, 23, 58, 56, 44, 38, 33, 28, 25, 22, 34, 36,
        32, 26, 22, 19, 17, 18, 24, 26, 24, 21, 18, 16, 14, 14,
        /* Size 8x16 */
        64, 65, 59, 47, 38, 31, 25, 22, 65, 63, 59, 49, 41, 34, 27, 23, 65, 62,
        58, 49, 41, 35, 28, 24, 64, 60, 55, 48, 41, 35, 29, 25, 60, 59, 49, 43,
        38, 33, 27, 24, 57, 57, 45, 38, 34, 30, 25, 22, 50, 51, 41, 34, 30, 27,
        23, 21, 44, 46, 38, 31, 27, 24, 21, 19, 38, 41, 36, 29, 25, 22, 19, 18,
        35, 38, 33, 27, 24, 21, 18, 17, 32, 34, 31, 26, 22, 20, 17, 16, 27, 31,
        28, 24, 20, 18, 16, 15, 25, 28, 26, 22, 19, 17, 15, 15, 23, 26, 25, 21,
        19, 16, 14, 14, 22, 25, 24, 21, 17, 15, 14, 13, 21, 23, 23, 20, 17, 15,
        13, 12,
        /* Size 16x8 */
        64, 65, 65, 64, 60, 57, 50, 44, 38, 35, 32, 27, 25, 23, 22, 21, 65, 63,
        62, 60, 59, 57, 51, 46, 41, 38, 34, 31, 28, 26, 25, 23, 59, 59, 58, 55,
        49, 45, 41, 38, 36, 33, 31, 28, 26, 25, 24, 23, 47, 49, 49, 48, 43, 38,
        34, 31, 29, 27, 26, 24, 22, 21, 21, 20, 38, 41, 41, 41, 38, 34, 30, 27,
        25, 24, 22, 20, 19, 19, 17, 17, 31, 34, 35, 35, 33, 30, 27, 24, 22, 21,
        20, 18, 17, 16, 15, 15, 25, 27, 28, 29, 27, 25, 23, 21, 19, 18, 17, 16,
        15, 14, 14, 13, 22, 23, 24, 25, 24, 22, 21, 19, 18, 17, 16, 15, 15, 14,
        13, 12,
        /* Size 16x32 */
        64, 65, 65, 64, 59, 57, 47, 43, 38, 33, 31, 26, 25, 23, 22, 21, 65, 64,
        64, 63, 59, 58, 48, 45, 40, 34, 33, 27, 26, 24, 23, 22, 65, 64, 63, 63,
        59, 59, 49, 46, 41, 35, 34, 28, 27, 25, 23, 22, 65, 63, 63, 62, 60, 59,
        50, 47, 41, 36, 34, 28, 27, 25, 24, 23, 65, 63, 62, 60, 58, 57, 49, 46,
        41, 36, 35, 29, 28, 26, 24, 23, 64, 63, 62, 60, 57, 56, 49, 46, 41, 36,
        35, 29, 28, 26, 25, 24, 64, 62, 60, 58, 55, 53, 48, 46, 41, 37, 35, 30,
        29, 27, 25, 23, 63, 62, 59, 57, 53, 51, 46, 44, 40, 35, 34, 29, 28, 26,
        25, 25, 60, 60, 59, 55, 49, 48, 43, 41, 38, 34, 33, 28, 27, 25, 24, 24,
        58, 59, 57, 54, 46, 44, 39, 37, 35, 32, 31, 26, 26, 24, 24, 22, 57, 59,
        57, 53, 45, 43, 38, 37, 34, 31, 30, 26, 25, 24, 22, 23, 52, 55, 53, 51,
        43, 41, 35, 34, 32, 29, 28, 24, 24, 23, 23, 20, 50, 53, 51, 49, 41, 40,
        34, 33, 30, 28, 27, 24, 23, 22, 21, 22, 47, 50, 49, 48, 40, 38, 33, 31,
        29, 26, 26, 23, 22, 21, 22, 19, 44, 47, 46, 45, 38, 37, 31, 29, 27, 25,
        24, 22, 21, 21, 19, 21, 43, 46, 45, 44, 38, 36, 31, 29, 27, 25, 24, 21,
        21, 20, 20, 18, 38, 41, 41, 41, 36, 34, 29, 27, 25, 23, 22, 20, 19, 19,
        18, 20, 37, 40, 40, 40, 35, 33, 28, 27, 24, 22, 22, 19, 19, 18, 18, 16,
        35, 38, 38, 38, 33, 32, 27, 26, 24, 21, 21, 19, 18, 17, 17, 18, 32, 35,
        35, 36, 32, 31, 26, 25, 23, 20, 20, 18, 17, 17, 17, 15, 32, 34, 34, 35,
        31, 30, 26, 24, 22, 20, 20, 17, 17, 16, 16, 17, 29, 31, 32, 32, 29, 28,
        24, 23, 21, 19, 18, 16, 16, 16, 15, 14, 27, 30, 31, 31, 28, 27, 24, 22,
        20, 19, 18, 16, 16, 15, 15, 16, 26, 28, 29, 29, 27, 26, 23, 22, 20, 18,
        17, 15, 15, 14, 14, 13, 25, 27, 28, 28, 26, 25, 22, 21, 19, 18, 17, 15,
        15, 14, 15, 15, 25, 27, 28, 28, 26, 25, 22, 21, 19, 17, 17, 15, 15, 14,
        13, 12, 23, 26, 26, 27, 25, 24, 21, 20, 19, 17, 16, 14, 14, 14, 14, 14,
        23, 25, 25, 26, 24, 24, 21, 21, 18, 17, 16, 14, 14, 13, 13, 11, 22, 24,
        25, 25, 24, 23, 21, 20, 17, 18, 15, 15, 14, 13, 13, 13, 21, 23, 24, 25,
        23, 22, 20, 20, 17, 18, 15, 16, 13, 14, 12, 12, 21, 23, 23, 24, 23, 22,
        20, 20, 17, 17, 15, 15, 13, 13, 12, 13, 20, 22, 22, 24, 23, 20, 21, 17,
        19, 15, 17, 13, 15, 11, 13, 11,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 64, 64, 63, 60, 58, 57, 52, 50, 47, 44, 43, 38, 37,
        35, 32, 32, 29, 27, 26, 25, 25, 23, 23, 22, 21, 21, 20, 65, 64, 64, 63,
        63, 63, 62, 62, 60, 59, 59, 55, 53, 50, 47, 46, 41, 40, 38, 35, 34, 31,
        30, 28, 27, 27, 26, 25, 24, 23, 23, 22, 65, 64, 63, 63, 62, 62, 60, 59,
        59, 57, 57, 53, 51, 49, 46, 45, 41, 40, 38, 35, 34, 32, 31, 29, 28, 28,
        26, 25, 25, 24, 23, 22, 64, 63, 63, 62, 60, 60, 58, 57, 55, 54, 53, 51,
        49, 48, 45, 44, 41, 40, 38, 36, 35, 32, 31, 29, 28, 28, 27, 26, 25, 25,
        24, 24, 59, 59, 59, 60, 58, 57, 55, 53, 49, 46, 45, 43, 41, 40, 38, 38,
        36, 35, 33, 32, 31, 29, 28, 27, 26, 26, 25, 24, 24, 23, 23, 23, 57, 58,
        59, 59, 57, 56, 53, 51, 48, 44, 43, 41, 40, 38, 37, 36, 34, 33, 32, 31,
        30, 28, 27, 26, 25, 25, 24, 24, 23, 22, 22, 20, 47, 48, 49, 50, 49, 49,
        48, 46, 43, 39, 38, 35, 34, 33, 31, 31, 29, 28, 27, 26, 26, 24, 24, 23,
        22, 22, 21, 21, 21, 20, 20, 21, 43, 45, 46, 47, 46, 46, 46, 44, 41, 37,
        37, 34, 33, 31, 29, 29, 27, 27, 26, 25, 24, 23, 22, 22, 21, 21, 20, 21,
        20, 20, 20, 17, 38, 40, 41, 41, 41, 41, 41, 40, 38, 35, 34, 32, 30, 29,
        27, 27, 25, 24, 24, 23, 22, 21, 20, 20, 19, 19, 19, 18, 17, 17, 17, 19,
        33, 34, 35, 36, 36, 36, 37, 35, 34, 32, 31, 29, 28, 26, 25, 25, 23, 22,
        21, 20, 20, 19, 19, 18, 18, 17, 17, 17, 18, 18, 17, 15, 31, 33, 34, 34,
        35, 35, 35, 34, 33, 31, 30, 28, 27, 26, 24, 24, 22, 22, 21, 20, 20, 18,
        18, 17, 17, 17, 16, 16, 15, 15, 15, 17, 26, 27, 28, 28, 29, 29, 30, 29,
        28, 26, 26, 24, 24, 23, 22, 21, 20, 19, 19, 18, 17, 16, 16, 15, 15, 15,
        14, 14, 15, 16, 15, 13, 25, 26, 27, 27, 28, 28, 29, 28, 27, 26, 25, 24,
        23, 22, 21, 21, 19, 19, 18, 17, 17, 16, 16, 15, 15, 15, 14, 14, 14, 13,
        13, 15, 23, 24, 25, 25, 26, 26, 27, 26, 25, 24, 24, 23, 22, 21, 21, 20,
        19, 18, 17, 17, 16, 16, 15, 14, 14, 14, 14, 13, 13, 14, 13, 11, 22, 23,
        23, 24, 24, 25, 25, 25, 24, 24, 22, 23, 21, 22, 19, 20, 18, 18, 17, 17,
        16, 15, 15, 14, 15, 13, 14, 13, 13, 12, 12, 13, 21, 22, 22, 23, 23, 24,
        23, 25, 24, 22, 23, 20, 22, 19, 21, 18, 20, 16, 18, 15, 17, 14, 16, 13,
        15, 12, 14, 11, 13, 12, 13, 11,
        /* Size 4x16 */
        65, 57, 33, 23, 64, 59, 35, 25, 63, 57, 36, 26, 62, 53, 37, 27, 60, 48,
        34, 25, 59, 43, 31, 24, 53, 40, 28, 22, 47, 37, 25, 21, 41, 34, 23, 19,
        38, 32, 21, 17, 34, 30, 20, 16, 30, 27, 19, 15, 27, 25, 18, 14, 26, 24,
        17, 14, 24, 23, 18, 13, 23, 22, 17, 13,
        /* Size 16x4 */
        65, 64, 63, 62, 60, 59, 53, 47, 41, 38, 34, 30, 27, 26, 24, 23, 57, 59,
        57, 53, 48, 43, 40, 37, 34, 32, 30, 27, 25, 24, 23, 22, 33, 35, 36, 37,
        34, 31, 28, 25, 23, 21, 20, 19, 18, 17, 18, 17, 23, 25, 26, 27, 25, 24,
        22, 21, 19, 17, 16, 15, 14, 14, 13, 13,
        /* Size 8x32 */
        64, 65, 59, 47, 38, 31, 25, 22, 65, 64, 59, 48, 40, 33, 26, 23, 65, 63,
        59, 49, 41, 34, 27, 23, 65, 63, 60, 50, 41, 34, 27, 24, 65, 62, 58, 49,
        41, 35, 28, 24, 64, 62, 57, 49, 41, 35, 28, 25, 64, 60, 55, 48, 41, 35,
        29, 25, 63, 59, 53, 46, 40, 34, 28, 25, 60, 59, 49, 43, 38, 33, 27, 24,
        58, 57, 46, 39, 35, 31, 26, 24, 57, 57, 45, 38, 34, 30, 25, 22, 52, 53,
        43, 35, 32, 28, 24, 23, 50, 51, 41, 34, 30, 27, 23, 21, 47, 49, 40, 33,
        29, 26, 22, 22, 44, 46, 38, 31, 27, 24, 21, 19, 43, 45, 38, 31, 27, 24,
        21, 20, 38, 41, 36, 29, 25, 22, 19, 18, 37, 40, 35, 28, 24, 22, 19, 18,
        35, 38, 33, 27, 24, 21, 18, 17, 32, 35, 32, 26, 23, 20, 17, 17, 32, 34,
        31, 26, 22, 20, 17, 16, 29, 32, 29, 24, 21, 18, 16, 15, 27, 31, 28, 24,
        20, 18, 16, 15, 26, 29, 27, 23, 20, 17, 15, 14, 25, 28, 26, 22, 19, 17,
        15, 15, 25, 28, 26, 22, 19, 17, 15, 13, 23, 26, 25, 21, 19, 16, 14, 14,
        23, 25, 24, 21, 18, 16, 14, 13, 22, 25, 24, 21, 17, 15, 14, 13, 21, 24,
        23, 20, 17, 15, 13, 12, 21, 23, 23, 20, 17, 15, 13, 12, 20, 22, 23, 21,
        19, 17, 15, 13,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 64, 64, 63, 60, 58, 57, 52, 50, 47, 44, 43, 38, 37,
        35, 32, 32, 29, 27, 26, 25, 25, 23, 23, 22, 21, 21, 20, 65, 64, 63, 63,
        62, 62, 60, 59, 59, 57, 57, 53, 51, 49, 46, 45, 41, 40, 38, 35, 34, 32,
        31, 29, 28, 28, 26, 25, 25, 24, 23, 22, 59, 59, 59, 60, 58, 57, 55, 53,
        49, 46, 45, 43, 41, 40, 38, 38, 36, 35, 33, 32, 31, 29, 28, 27, 26, 26,
        25, 24, 24, 23, 23, 23, 47, 48, 49, 50, 49, 49, 48, 46, 43, 39, 38, 35,
        34, 33, 31, 31, 29, 28, 27, 26, 26, 24, 24, 23, 22, 22, 21, 21, 21, 20,
        20, 21, 38, 40, 41, 41, 41, 41, 41, 40, 38, 35, 34, 32, 30, 29, 27, 27,
        25, 24, 24, 23, 22, 21, 20, 20, 19, 19, 19, 18, 17, 17, 17, 19, 31, 33,
        34, 34, 35, 35, 35, 34, 33, 31, 30, 28, 27, 26, 24, 24, 22, 22, 21, 20,
        20, 18, 18, 17, 17, 17, 16, 16, 15, 15, 15, 17, 25, 26, 27, 27, 28, 28,
        29, 28, 27, 26, 25, 24, 23, 22, 21, 21, 19, 19, 18, 17, 17, 16, 16, 15,
        15, 15, 14, 14, 14, 13, 13, 15, 22, 23, 23, 24, 24, 25, 25, 25, 24, 24,
        22, 23, 21, 22, 19, 20, 18, 18, 17, 17, 16, 15, 15, 14, 15, 13, 14, 13,
        13, 12, 12, 13 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        43, 37, 37, 23,
#endif
        /* Size 4x4 */
        64, 45, 40, 33, 45, 38, 34, 31, 40, 34, 27, 25, 33, 31, 25, 21,
        /* Size 8x8 */
        65, 59, 44, 44, 41, 36, 33, 31, 59, 53, 44, 45, 43, 38, 36, 33, 44, 44,
        39, 39, 38, 35, 34, 33, 44, 45, 39, 35, 33, 31, 30, 30, 41, 43, 38, 33,
        30, 28, 27, 27, 36, 38, 35, 31, 28, 25, 24, 23, 33, 36, 34, 30, 27, 24,
        22, 20, 31, 33, 33, 30, 27, 23, 20, 20,
        /* Size 16x16 */
        64, 66, 62, 56, 49, 42, 42, 41, 39, 38, 36, 33, 32, 31, 29, 28, 66, 65,
        60, 53, 48, 44, 44, 43, 42, 40, 38, 36, 34, 33, 32, 31, 62, 60, 55, 48,
        46, 44, 45, 44, 43, 41, 40, 37, 36, 34, 33, 32, 56, 53, 48, 43, 43, 43,
        45, 45, 44, 42, 41, 39, 37, 36, 35, 33, 49, 48, 46, 43, 42, 41, 41, 41,
        41, 39, 38, 37, 35, 34, 33, 32, 42, 44, 44, 43, 41, 39, 38, 38, 38, 37,
        36, 34, 33, 33, 31, 31, 42, 44, 45, 45, 41, 38, 37, 36, 35, 34, 33, 32,
        31, 31, 30, 29, 41, 43, 44, 45, 41, 38, 36, 34, 33, 32, 31, 30, 29, 28,
        29, 27, 39, 42, 43, 44, 41, 38, 35, 33, 31, 30, 29, 28, 27, 27, 28, 26,
        38, 40, 41, 42, 39, 37, 34, 32, 30, 29, 28, 27, 26, 25, 26, 24, 36, 38,
        40, 41, 38, 36, 33, 31, 29, 28, 27, 26, 25, 24, 25, 23, 33, 36, 37, 39,
        37, 34, 32, 30, 28, 27, 26, 24, 23, 23, 23, 22, 32, 34, 36, 37, 35, 33,
        31, 29, 27, 26, 25, 23, 23, 22, 21, 21, 31, 33, 34, 36, 34, 33, 31, 28,
        27, 25, 24, 23, 22, 21, 21, 19, 29, 32, 33, 35, 33, 31, 30, 29, 28, 26,
        25, 23, 21, 21, 20, 20, 28, 31, 32, 33, 32, 31, 29, 27, 26, 24, 23, 22,
        21, 19, 20, 19,
        /* Size 32x32 */
        64, 66, 66, 67, 62, 61, 56, 53, 49, 43, 42, 42, 42, 42, 41, 41, 39, 39,
        38, 36, 36, 34, 33, 32, 32, 32, 31, 30, 29, 29, 28, 28, 66, 65, 65, 66,
        60, 59, 53, 51, 48, 44, 43, 44, 44, 44, 43, 42, 41, 41, 39, 38, 38, 36,
        35, 34, 33, 33, 32, 32, 31, 31, 30, 30, 66, 65, 65, 65, 60, 59, 53, 51,
        48, 45, 44, 44, 44, 45, 43, 43, 42, 41, 40, 39, 38, 37, 36, 35, 34, 34,
        33, 32, 32, 31, 31, 30, 67, 66, 65, 64, 59, 58, 51, 50, 48, 45, 44, 45,
        45, 46, 45, 44, 43, 42, 41, 40, 39, 38, 37, 36, 35, 35, 34, 33, 33, 32,
        32, 32, 62, 60, 60, 59, 55, 53, 48, 47, 46, 44, 44, 45, 45, 45, 44, 44,
        43, 42, 41, 40, 40, 38, 37, 36, 36, 36, 34, 34, 33, 33, 32, 31, 61, 59,
        59, 58, 53, 53, 47, 47, 46, 44, 43, 44, 45, 45, 44, 44, 43, 43, 41, 40,
        40, 38, 38, 36, 36, 36, 35, 34, 34, 33, 33, 33, 56, 53, 53, 51, 48, 47,
        43, 43, 43, 43, 43, 44, 45, 45, 45, 45, 44, 43, 42, 41, 41, 39, 39, 38,
        37, 37, 36, 35, 35, 34, 33, 32, 53, 51, 51, 50, 47, 47, 43, 43, 43, 42,
        42, 43, 43, 44, 43, 43, 43, 42, 41, 40, 40, 38, 38, 37, 36, 36, 35, 35,
        35, 34, 34, 34, 49, 48, 48, 48, 46, 46, 43, 43, 42, 41, 41, 41, 41, 42,
        41, 41, 41, 40, 39, 39, 38, 37, 37, 36, 35, 35, 34, 34, 33, 33, 32, 33,
        43, 44, 45, 45, 44, 44, 43, 42, 41, 39, 39, 39, 39, 39, 39, 39, 38, 38,
        37, 37, 37, 35, 35, 34, 34, 34, 33, 33, 33, 33, 32, 30, 42, 43, 44, 44,
        44, 43, 43, 42, 41, 39, 39, 39, 38, 38, 38, 38, 38, 37, 37, 36, 36, 35,
        34, 34, 33, 33, 33, 32, 31, 31, 31, 32, 42, 44, 44, 45, 45, 44, 44, 43,
        41, 39, 39, 38, 37, 37, 36, 36, 36, 35, 35, 34, 34, 33, 33, 32, 32, 32,
        31, 31, 31, 32, 31, 29, 42, 44, 44, 45, 45, 45, 45, 43, 41, 39, 38, 37,
        37, 36, 36, 35, 35, 34, 34, 34, 33, 33, 32, 32, 31, 31, 31, 31, 30, 29,
        29, 31, 42, 44, 45, 46, 45, 45, 45, 44, 42, 39, 38, 37, 36, 35, 34, 34,
        33, 33, 33, 32, 32, 31, 31, 30, 30, 30, 29, 29, 29, 30, 30, 27, 41, 43,
        43, 45, 44, 44, 45, 43, 41, 39, 38, 36, 36, 34, 34, 34, 33, 32, 32, 31,
        31, 30, 30, 29, 29, 29, 28, 29, 29, 27, 27, 29, 41, 42, 43, 44, 44, 44,
        45, 43, 41, 39, 38, 36, 35, 34, 34, 33, 32, 32, 32, 31, 31, 30, 29, 29,
        29, 29, 28, 27, 27, 29, 28, 26, 39, 41, 42, 43, 43, 43, 44, 43, 41, 38,
        38, 36, 35, 33, 33, 32, 31, 30, 30, 29, 29, 28, 28, 27, 27, 27, 27, 27,
        28, 26, 26, 28, 39, 41, 41, 42, 42, 43, 43, 42, 40, 38, 37, 35, 34, 33,
        32, 32, 30, 30, 29, 29, 29, 28, 27, 27, 27, 27, 26, 26, 25, 27, 27, 25,
        38, 39, 40, 41, 41, 41, 42, 41, 39, 37, 37, 35, 34, 33, 32, 32, 30, 29,
        29, 28, 28, 27, 27, 26, 26, 26, 25, 25, 26, 25, 24, 27, 36, 38, 39, 40,
        40, 40, 41, 40, 39, 37, 36, 34, 34, 32, 31, 31, 29, 29, 28, 27, 27, 26,
        26, 25, 25, 25, 25, 25, 24, 25, 26, 23, 36, 38, 38, 39, 40, 40, 41, 40,
        38, 37, 36, 34, 33, 32, 31, 31, 29, 29, 28, 27, 27, 26, 26, 25, 25, 25,
        24, 23, 25, 24, 23, 25, 34, 36, 37, 38, 38, 38, 39, 38, 37, 35, 35, 33,
        33, 31, 30, 30, 28, 28, 27, 26, 26, 25, 25, 24, 24, 24, 23, 24, 23, 23,
        25, 22, 33, 35, 36, 37, 37, 38, 39, 38, 37, 35, 34, 33, 32, 31, 30, 29,
        28, 27, 27, 26, 26, 25, 24, 24, 23, 23, 23, 23, 23, 23, 22, 24, 32, 34,
        35, 36, 36, 36, 38, 37, 36, 34, 34, 32, 32, 30, 29, 29, 27, 27, 26, 25,
        25, 24, 24, 23, 23, 23, 22, 22, 23, 22, 23, 21, 32, 33, 34, 35, 36, 36,
        37, 36, 35, 34, 33, 32, 31, 30, 29, 29, 27, 27, 26, 25, 25, 24, 23, 23,
        23, 22, 22, 22, 21, 23, 21, 23, 32, 33, 34, 35, 36, 36, 37, 36, 35, 34,
        33, 32, 31, 30, 29, 29, 27, 27, 26, 25, 25, 24, 23, 23, 22, 22, 22, 22,
        22, 20, 22, 20, 31, 32, 33, 34, 34, 35, 36, 35, 34, 33, 33, 31, 31, 29,
        28, 28, 27, 26, 25, 25, 24, 23, 23, 22, 22, 22, 21, 22, 21, 21, 19, 22,
        30, 32, 32, 33, 34, 34, 35, 35, 34, 33, 32, 31, 31, 29, 29, 27, 27, 26,
        25, 25, 23, 24, 23, 22, 22, 22, 22, 21, 21, 21, 21, 19, 29, 31, 32, 33,
        33, 34, 35, 35, 33, 33, 31, 31, 30, 29, 29, 27, 28, 25, 26, 24, 25, 23,
        23, 23, 21, 22, 21, 21, 20, 21, 20, 21, 29, 31, 31, 32, 33, 33, 34, 34,
        33, 33, 31, 32, 29, 30, 27, 29, 26, 27, 25, 25, 24, 23, 23, 22, 23, 20,
        21, 21, 21, 20, 21, 20, 28, 30, 31, 32, 32, 33, 33, 34, 32, 32, 31, 31,
        29, 30, 27, 28, 26, 27, 24, 26, 23, 25, 22, 23, 21, 22, 19, 21, 20, 21,
        19, 21, 28, 30, 30, 32, 31, 33, 32, 34, 33, 30, 32, 29, 31, 27, 29, 26,
        28, 25, 27, 23, 25, 22, 24, 21, 23, 20, 22, 19, 21, 20, 21, 19,
        /* Size 4x8 */
        65, 44, 38, 32, 57, 44, 41, 34, 45, 39, 37, 33, 45, 38, 32, 29, 42, 37,
        29, 26, 38, 35, 26, 24, 35, 33, 25, 22, 32, 31, 26, 21,
        /* Size 8x4 */
        65, 57, 45, 45, 42, 38, 35, 32, 44, 44, 39, 38, 37, 35, 33, 31, 38, 41,
        37, 32, 29, 26, 25, 26, 32, 34, 33, 29, 26, 24, 22, 21,
        /* Size 8x16 */
        64, 62, 45, 42, 39, 36, 32, 29, 66, 59, 46, 44, 42, 38, 34, 31, 62, 55,
        45, 45, 43, 40, 36, 33, 55, 48, 43, 45, 44, 41, 37, 34, 49, 46, 41, 41,
        41, 38, 35, 33, 42, 44, 39, 38, 38, 36, 33, 31, 42, 45, 40, 36, 35, 33,
        31, 29, 41, 44, 39, 34, 32, 31, 29, 27, 39, 43, 39, 34, 31, 29, 27, 26,
        38, 41, 38, 33, 30, 28, 26, 25, 36, 40, 37, 32, 29, 27, 25, 24, 34, 37,
        35, 31, 28, 26, 23, 23, 32, 36, 34, 30, 27, 25, 23, 22, 31, 34, 33, 29,
        27, 24, 22, 22, 30, 33, 33, 29, 26, 23, 21, 21, 29, 32, 32, 29, 26, 23,
        21, 20,
        /* Size 16x8 */
        64, 66, 62, 55, 49, 42, 42, 41, 39, 38, 36, 34, 32, 31, 30, 29, 62, 59,
        55, 48, 46, 44, 45, 44, 43, 41, 40, 37, 36, 34, 33, 32, 45, 46, 45, 43,
        41, 39, 40, 39, 39, 38, 37, 35, 34, 33, 33, 32, 42, 44, 45, 45, 41, 38,
        36, 34, 34, 33, 32, 31, 30, 29, 29, 29, 39, 42, 43, 44, 41, 38, 35, 32,
        31, 30, 29, 28, 27, 27, 26, 26, 36, 38, 40, 41, 38, 36, 33, 31, 29, 28,
        27, 26, 25, 24, 23, 23, 32, 34, 36, 37, 35, 33, 31, 29, 27, 26, 25, 23,
        23, 22, 21, 21, 29, 31, 33, 34, 33, 31, 29, 27, 26, 25, 24, 23, 22, 22,
        21, 20,
        /* Size 16x32 */
        64, 66, 62, 55, 45, 43, 42, 41, 39, 37, 36, 32, 32, 30, 29, 28, 65, 65,
        60, 53, 46, 44, 44, 42, 41, 38, 38, 34, 33, 32, 31, 30, 66, 64, 59, 53,
        46, 44, 44, 43, 42, 39, 38, 35, 34, 33, 31, 30, 67, 64, 59, 51, 46, 45,
        45, 44, 43, 40, 39, 36, 35, 34, 33, 32, 62, 59, 55, 48, 45, 44, 45, 44,
        43, 41, 40, 36, 36, 34, 33, 32, 61, 57, 53, 48, 45, 44, 45, 44, 43, 41,
        40, 37, 36, 34, 33, 33, 55, 51, 48, 44, 43, 43, 45, 45, 44, 41, 41, 38,
        37, 36, 34, 33, 53, 50, 47, 43, 42, 42, 44, 43, 42, 41, 40, 37, 36, 35,
        35, 34, 49, 48, 46, 43, 41, 41, 41, 41, 41, 39, 38, 36, 35, 34, 33, 34,
        43, 45, 44, 43, 40, 39, 39, 39, 38, 37, 37, 34, 34, 33, 33, 31, 42, 44,
        44, 43, 39, 39, 38, 38, 38, 36, 36, 34, 33, 32, 31, 32, 42, 45, 44, 44,
        40, 39, 37, 36, 36, 34, 34, 32, 32, 31, 32, 29, 42, 45, 45, 44, 40, 39,
        36, 36, 35, 34, 33, 32, 31, 31, 29, 31, 42, 45, 45, 45, 40, 38, 35, 34,
        33, 32, 32, 30, 30, 29, 30, 28, 41, 44, 44, 44, 39, 38, 34, 34, 32, 31,
        31, 29, 29, 29, 27, 30, 41, 44, 44, 44, 39, 38, 34, 33, 32, 31, 31, 29,
        29, 28, 29, 26, 39, 43, 43, 43, 39, 38, 34, 32, 31, 29, 29, 27, 27, 27,
        26, 28, 39, 42, 42, 43, 38, 37, 33, 32, 30, 29, 28, 27, 27, 26, 27, 25,
        38, 41, 41, 42, 38, 37, 33, 32, 30, 28, 28, 26, 26, 25, 25, 27, 36, 40,
        40, 41, 37, 36, 32, 31, 29, 27, 27, 25, 25, 25, 25, 23, 36, 39, 40, 41,
        37, 36, 32, 31, 29, 27, 27, 25, 25, 24, 24, 26, 34, 38, 38, 39, 36, 35,
        31, 30, 28, 26, 26, 24, 24, 24, 23, 22, 34, 37, 37, 38, 35, 34, 31, 30,
        28, 26, 26, 24, 23, 23, 23, 25, 32, 36, 36, 37, 34, 34, 30, 29, 27, 25,
        25, 23, 23, 22, 22, 21, 32, 35, 36, 37, 34, 33, 30, 29, 27, 25, 25, 23,
        23, 22, 22, 23, 32, 35, 35, 37, 34, 33, 30, 29, 27, 25, 25, 23, 22, 22,
        20, 20, 31, 34, 34, 36, 33, 33, 29, 28, 27, 25, 24, 22, 22, 22, 22, 22,
        30, 33, 34, 35, 33, 32, 29, 29, 26, 25, 24, 22, 22, 21, 21, 19, 30, 33,
        33, 34, 33, 32, 29, 29, 26, 26, 23, 23, 21, 21, 21, 21, 29, 32, 33, 34,
        33, 31, 29, 29, 26, 26, 23, 24, 20, 21, 20, 20, 29, 32, 32, 34, 32, 31,
        29, 28, 26, 26, 23, 23, 21, 21, 20, 21, 28, 31, 31, 34, 33, 29, 31, 26,
        28, 23, 25, 21, 23, 19, 21, 19,
        /* Size 32x16 */
        64, 65, 66, 67, 62, 61, 55, 53, 49, 43, 42, 42, 42, 42, 41, 41, 39, 39,
        38, 36, 36, 34, 34, 32, 32, 32, 31, 30, 30, 29, 29, 28, 66, 65, 64, 64,
        59, 57, 51, 50, 48, 45, 44, 45, 45, 45, 44, 44, 43, 42, 41, 40, 39, 38,
        37, 36, 35, 35, 34, 33, 33, 32, 32, 31, 62, 60, 59, 59, 55, 53, 48, 47,
        46, 44, 44, 44, 45, 45, 44, 44, 43, 42, 41, 40, 40, 38, 37, 36, 36, 35,
        34, 34, 33, 33, 32, 31, 55, 53, 53, 51, 48, 48, 44, 43, 43, 43, 43, 44,
        44, 45, 44, 44, 43, 43, 42, 41, 41, 39, 38, 37, 37, 37, 36, 35, 34, 34,
        34, 34, 45, 46, 46, 46, 45, 45, 43, 42, 41, 40, 39, 40, 40, 40, 39, 39,
        39, 38, 38, 37, 37, 36, 35, 34, 34, 34, 33, 33, 33, 33, 32, 33, 43, 44,
        44, 45, 44, 44, 43, 42, 41, 39, 39, 39, 39, 38, 38, 38, 38, 37, 37, 36,
        36, 35, 34, 34, 33, 33, 33, 32, 32, 31, 31, 29, 42, 44, 44, 45, 45, 45,
        45, 44, 41, 39, 38, 37, 36, 35, 34, 34, 34, 33, 33, 32, 32, 31, 31, 30,
        30, 30, 29, 29, 29, 29, 29, 31, 41, 42, 43, 44, 44, 44, 45, 43, 41, 39,
        38, 36, 36, 34, 34, 33, 32, 32, 32, 31, 31, 30, 30, 29, 29, 29, 28, 29,
        29, 29, 28, 26, 39, 41, 42, 43, 43, 43, 44, 42, 41, 38, 38, 36, 35, 33,
        32, 32, 31, 30, 30, 29, 29, 28, 28, 27, 27, 27, 27, 26, 26, 26, 26, 28,
        37, 38, 39, 40, 41, 41, 41, 41, 39, 37, 36, 34, 34, 32, 31, 31, 29, 29,
        28, 27, 27, 26, 26, 25, 25, 25, 25, 25, 26, 26, 26, 23, 36, 38, 38, 39,
        40, 40, 41, 40, 38, 37, 36, 34, 33, 32, 31, 31, 29, 28, 28, 27, 27, 26,
        26, 25, 25, 25, 24, 24, 23, 23, 23, 25, 32, 34, 35, 36, 36, 37, 38, 37,
        36, 34, 34, 32, 32, 30, 29, 29, 27, 27, 26, 25, 25, 24, 24, 23, 23, 23,
        22, 22, 23, 24, 23, 21, 32, 33, 34, 35, 36, 36, 37, 36, 35, 34, 33, 32,
        31, 30, 29, 29, 27, 27, 26, 25, 25, 24, 23, 23, 23, 22, 22, 22, 21, 20,
        21, 23, 30, 32, 33, 34, 34, 34, 36, 35, 34, 33, 32, 31, 31, 29, 29, 28,
        27, 26, 25, 25, 24, 24, 23, 22, 22, 22, 22, 21, 21, 21, 21, 19, 29, 31,
        31, 33, 33, 33, 34, 35, 33, 33, 31, 32, 29, 30, 27, 29, 26, 27, 25, 25,
        24, 23, 23, 22, 22, 20, 22, 21, 21, 20, 20, 21, 28, 30, 30, 32, 32, 33,
        33, 34, 34, 31, 32, 29, 31, 28, 30, 26, 28, 25, 27, 23, 26, 22, 25, 21,
        23, 20, 22, 19, 21, 20, 21, 19,
        /* Size 4x16 */
        66, 43, 37, 30, 64, 44, 39, 33, 59, 44, 41, 34, 51, 43, 41, 36, 48, 41,
        39, 34, 44, 39, 36, 32, 45, 39, 34, 31, 44, 38, 31, 29, 43, 38, 29, 27,
        41, 37, 28, 25, 39, 36, 27, 24, 37, 34, 26, 23, 35, 33, 25, 22, 34, 33,
        25, 22, 33, 32, 26, 21, 32, 31, 26, 21,
        /* Size 16x4 */
        66, 64, 59, 51, 48, 44, 45, 44, 43, 41, 39, 37, 35, 34, 33, 32, 43, 44,
        44, 43, 41, 39, 39, 38, 38, 37, 36, 34, 33, 33, 32, 31, 37, 39, 41, 41,
        39, 36, 34, 31, 29, 28, 27, 26, 25, 25, 26, 26, 30, 33, 34, 36, 34, 32,
        31, 29, 27, 25, 24, 23, 22, 22, 21, 21,
        /* Size 8x32 */
        64, 62, 45, 42, 39, 36, 32, 29, 65, 60, 46, 44, 41, 38, 33, 31, 66, 59,
        46, 44, 42, 38, 34, 31, 67, 59, 46, 45, 43, 39, 35, 33, 62, 55, 45, 45,
        43, 40, 36, 33, 61, 53, 45, 45, 43, 40, 36, 33, 55, 48, 43, 45, 44, 41,
        37, 34, 53, 47, 42, 44, 42, 40, 36, 35, 49, 46, 41, 41, 41, 38, 35, 33,
        43, 44, 40, 39, 38, 37, 34, 33, 42, 44, 39, 38, 38, 36, 33, 31, 42, 44,
        40, 37, 36, 34, 32, 32, 42, 45, 40, 36, 35, 33, 31, 29, 42, 45, 40, 35,
        33, 32, 30, 30, 41, 44, 39, 34, 32, 31, 29, 27, 41, 44, 39, 34, 32, 31,
        29, 29, 39, 43, 39, 34, 31, 29, 27, 26, 39, 42, 38, 33, 30, 28, 27, 27,
        38, 41, 38, 33, 30, 28, 26, 25, 36, 40, 37, 32, 29, 27, 25, 25, 36, 40,
        37, 32, 29, 27, 25, 24, 34, 38, 36, 31, 28, 26, 24, 23, 34, 37, 35, 31,
        28, 26, 23, 23, 32, 36, 34, 30, 27, 25, 23, 22, 32, 36, 34, 30, 27, 25,
        23, 22, 32, 35, 34, 30, 27, 25, 22, 20, 31, 34, 33, 29, 27, 24, 22, 22,
        30, 34, 33, 29, 26, 24, 22, 21, 30, 33, 33, 29, 26, 23, 21, 21, 29, 33,
        33, 29, 26, 23, 20, 20, 29, 32, 32, 29, 26, 23, 21, 20, 28, 31, 33, 31,
        28, 25, 23, 21,
        /* Size 32x8 */
        64, 65, 66, 67, 62, 61, 55, 53, 49, 43, 42, 42, 42, 42, 41, 41, 39, 39,
        38, 36, 36, 34, 34, 32, 32, 32, 31, 30, 30, 29, 29, 28, 62, 60, 59, 59,
        55, 53, 48, 47, 46, 44, 44, 44, 45, 45, 44, 44, 43, 42, 41, 40, 40, 38,
        37, 36, 36, 35, 34, 34, 33, 33, 32, 31, 45, 46, 46, 46, 45, 45, 43, 42,
        41, 40, 39, 40, 40, 40, 39, 39, 39, 38, 38, 37, 37, 36, 35, 34, 34, 34,
        33, 33, 33, 33, 32, 33, 42, 44, 44, 45, 45, 45, 45, 44, 41, 39, 38, 37,
        36, 35, 34, 34, 34, 33, 33, 32, 32, 31, 31, 30, 30, 30, 29, 29, 29, 29,
        29, 31, 39, 41, 42, 43, 43, 43, 44, 42, 41, 38, 38, 36, 35, 33, 32, 32,
        31, 30, 30, 29, 29, 28, 28, 27, 27, 27, 27, 26, 26, 26, 26, 28, 36, 38,
        38, 39, 40, 40, 41, 40, 38, 37, 36, 34, 33, 32, 31, 31, 29, 28, 28, 27,
        27, 26, 26, 25, 25, 25, 24, 24, 23, 23, 23, 25, 32, 33, 34, 35, 36, 36,
        37, 36, 35, 34, 33, 32, 31, 30, 29, 29, 27, 27, 26, 25, 25, 24, 23, 23,
        23, 22, 22, 22, 21, 20, 21, 23, 29, 31, 31, 33, 33, 33, 34, 35, 33, 33,
        31, 32, 29, 30, 27, 29, 26, 27, 25, 25, 24, 23, 23, 22, 22, 20, 22, 21,
        21, 20, 20, 21 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        58, 33, 33, 17,
#endif
        /* Size 4x4 */
        63, 59, 38, 27, 59, 42, 32, 25, 38, 32, 23, 18, 27, 25, 18, 15,
        /* Size 8x8 */
        64, 64, 60, 52, 41, 33, 27, 24, 64, 61, 58, 51, 42, 35, 29, 26, 60, 58,
        52, 45, 38, 33, 28, 25, 52, 51, 45, 36, 31, 27, 24, 22, 41, 42, 38, 31,
        26, 23, 20, 19, 33, 35, 33, 27, 23, 20, 17, 17, 27, 29, 28, 24, 20, 17,
        15, 14, 24, 26, 25, 22, 19, 17, 14, 13,
        /* Size 16x16 */
        64, 65, 65, 64, 61, 57, 52, 47, 42, 38, 35, 31, 27, 25, 24, 22, 65, 64,
        64, 63, 61, 59, 53, 49, 45, 40, 37, 33, 29, 27, 25, 24, 65, 64, 63, 62,
        60, 59, 54, 49, 46, 41, 38, 34, 30, 28, 26, 25, 64, 63, 62, 59, 57, 55,
        51, 48, 45, 41, 38, 34, 31, 29, 27, 25, 61, 61, 60, 57, 53, 49, 47, 44,
        41, 38, 36, 33, 29, 27, 26, 25, 57, 59, 59, 55, 49, 42, 41, 38, 36, 34,
        32, 30, 27, 26, 24, 24, 52, 53, 54, 51, 47, 41, 38, 35, 34, 32, 30, 28,
        25, 24, 23, 22, 47, 49, 49, 48, 44, 38, 35, 32, 31, 29, 27, 25, 24, 22,
        22, 21, 42, 45, 46, 45, 41, 36, 34, 31, 29, 27, 25, 24, 22, 21, 20, 19,
        38, 40, 41, 41, 38, 34, 32, 29, 27, 25, 24, 22, 20, 19, 19, 18, 35, 37,
        38, 38, 36, 32, 30, 27, 25, 24, 22, 21, 19, 18, 18, 17, 31, 33, 34, 34,
        33, 30, 28, 25, 24, 22, 21, 19, 18, 17, 16, 15, 27, 29, 30, 31, 29, 27,
        25, 24, 22, 20, 19, 18, 17, 16, 15, 15, 25, 27, 28, 29, 27, 26, 24, 22,
        21, 19, 18, 17, 16, 15, 14, 13, 24, 25, 26, 27, 26, 24, 23, 22, 20, 19,
        18, 16, 15, 14, 14, 13, 22, 24, 25, 25, 25, 24, 22, 21, 19, 18, 17, 15,
        15, 13, 13, 13,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 65, 64, 64, 61, 60, 57, 57, 52, 50, 47, 45, 42, 39,
        38, 35, 35, 31, 31, 29, 27, 26, 25, 25, 24, 23, 22, 22, 65, 64, 64, 64,
        64, 64, 63, 63, 61, 60, 59, 58, 53, 52, 48, 46, 44, 41, 40, 37, 36, 33,
        32, 30, 29, 27, 26, 26, 25, 24, 23, 23, 65, 64, 64, 64, 64, 63, 63, 63,
        61, 60, 59, 58, 53, 52, 49, 47, 45, 41, 40, 37, 37, 33, 33, 30, 29, 27,
        27, 26, 25, 24, 24, 23, 65, 64, 64, 63, 63, 63, 63, 63, 61, 61, 59, 59,
        55, 53, 50, 48, 46, 43, 41, 38, 38, 34, 34, 32, 30, 28, 28, 27, 26, 25,
        25, 24, 65, 64, 64, 63, 63, 62, 62, 62, 60, 59, 59, 58, 54, 53, 49, 48,
        46, 43, 41, 38, 38, 34, 34, 32, 30, 28, 28, 27, 26, 25, 25, 24, 65, 64,
        63, 63, 62, 61, 61, 60, 59, 58, 56, 56, 53, 51, 49, 47, 45, 42, 41, 38,
        38, 35, 34, 32, 31, 29, 28, 28, 27, 26, 25, 25, 64, 63, 63, 63, 62, 61,
        59, 59, 57, 57, 55, 54, 51, 51, 48, 47, 45, 42, 41, 39, 38, 35, 34, 32,
        31, 29, 29, 28, 27, 26, 25, 24, 64, 63, 63, 63, 62, 60, 59, 58, 56, 55,
        53, 53, 51, 50, 48, 47, 45, 42, 41, 39, 38, 35, 35, 33, 31, 29, 29, 28,
        27, 27, 26, 25, 61, 61, 61, 61, 60, 59, 57, 56, 53, 53, 49, 49, 47, 46,
        44, 43, 41, 39, 38, 36, 36, 33, 33, 31, 29, 28, 27, 27, 26, 26, 25, 25,
        60, 60, 60, 61, 59, 58, 57, 55, 53, 52, 48, 48, 46, 45, 43, 41, 40, 38,
        38, 35, 35, 33, 32, 30, 29, 28, 27, 27, 26, 25, 25, 23, 57, 59, 59, 59,
        59, 56, 55, 53, 49, 48, 42, 42, 41, 40, 38, 37, 36, 35, 34, 32, 32, 30,
        30, 28, 27, 26, 26, 25, 24, 24, 24, 24, 57, 58, 58, 59, 58, 56, 54, 53,
        49, 48, 42, 42, 40, 39, 38, 37, 36, 34, 34, 32, 32, 30, 29, 28, 27, 26,
        25, 25, 24, 24, 23, 22, 52, 53, 53, 55, 54, 53, 51, 51, 47, 46, 41, 40,
        38, 37, 35, 34, 34, 32, 32, 30, 30, 28, 28, 26, 25, 24, 24, 24, 23, 22,
        22, 23, 50, 52, 52, 53, 53, 51, 51, 50, 46, 45, 40, 39, 37, 36, 34, 34,
        33, 31, 31, 29, 29, 27, 27, 26, 25, 24, 23, 23, 23, 22, 22, 21, 47, 48,
        49, 50, 49, 49, 48, 48, 44, 43, 38, 38, 35, 34, 32, 32, 31, 29, 29, 27,
        27, 26, 25, 24, 24, 23, 22, 22, 22, 21, 21, 21, 45, 46, 47, 48, 48, 47,
        47, 47, 43, 41, 37, 37, 34, 34, 32, 31, 30, 28, 28, 26, 26, 25, 25, 23,
        23, 22, 22, 21, 21, 21, 21, 19, 42, 44, 45, 46, 46, 45, 45, 45, 41, 40,
        36, 36, 34, 33, 31, 30, 29, 27, 27, 26, 25, 24, 24, 23, 22, 21, 21, 21,
        20, 20, 19, 20, 39, 41, 41, 43, 43, 42, 42, 42, 39, 38, 35, 34, 32, 31,
        29, 28, 27, 26, 25, 24, 24, 23, 23, 21, 21, 20, 20, 20, 19, 19, 19, 18,
        38, 40, 40, 41, 41, 41, 41, 41, 38, 38, 34, 34, 32, 31, 29, 28, 27, 25,
        25, 24, 24, 22, 22, 21, 20, 20, 19, 19, 19, 18, 18, 19, 35, 37, 37, 38,
        38, 38, 39, 39, 36, 35, 32, 32, 30, 29, 27, 26, 26, 24, 24, 23, 22, 21,
        21, 20, 19, 19, 18, 18, 18, 18, 18, 17, 35, 36, 37, 38, 38, 38, 38, 38,
        36, 35, 32, 32, 30, 29, 27, 26, 25, 24, 24, 22, 22, 21, 21, 20, 19, 19,
        18, 18, 18, 17, 17, 18, 31, 33, 33, 34, 34, 35, 35, 35, 33, 33, 30, 30,
        28, 27, 26, 25, 24, 23, 22, 21, 21, 20, 19, 18, 18, 17, 17, 17, 17, 17,
        17, 16, 31, 32, 33, 34, 34, 34, 34, 35, 33, 32, 30, 29, 28, 27, 25, 25,
        24, 23, 22, 21, 21, 19, 19, 18, 18, 17, 17, 17, 16, 16, 15, 17, 29, 30,
        30, 32, 32, 32, 32, 33, 31, 30, 28, 28, 26, 26, 24, 23, 23, 21, 21, 20,
        20, 18, 18, 17, 17, 16, 16, 16, 16, 15, 16, 14, 27, 29, 29, 30, 30, 31,
        31, 31, 29, 29, 27, 27, 25, 25, 24, 23, 22, 21, 20, 19, 19, 18, 18, 17,
        17, 16, 16, 16, 15, 15, 15, 16, 26, 27, 27, 28, 28, 29, 29, 29, 28, 28,
        26, 26, 24, 24, 23, 22, 21, 20, 20, 19, 19, 17, 17, 16, 16, 15, 15, 15,
        15, 14, 15, 14, 25, 26, 27, 28, 28, 28, 29, 29, 27, 27, 26, 25, 24, 23,
        22, 22, 21, 20, 19, 18, 18, 17, 17, 16, 16, 15, 15, 15, 14, 14, 13, 15,
        25, 26, 26, 27, 27, 28, 28, 28, 27, 27, 25, 25, 24, 23, 22, 21, 21, 20,
        19, 18, 18, 17, 17, 16, 16, 15, 15, 15, 14, 14, 14, 13, 24, 25, 25, 26,
        26, 27, 27, 27, 26, 26, 24, 24, 23, 23, 22, 21, 20, 19, 19, 18, 18, 17,
        16, 16, 15, 15, 14, 14, 14, 14, 13, 14, 23, 24, 24, 25, 25, 26, 26, 27,
        26, 25, 24, 24, 22, 22, 21, 21, 20, 19, 18, 18, 17, 17, 16, 15, 15, 14,
        14, 14, 14, 13, 14, 13, 22, 23, 24, 25, 25, 25, 25, 26, 25, 25, 24, 23,
        22, 22, 21, 21, 19, 19, 18, 18, 17, 17, 15, 16, 15, 15, 13, 14, 13, 14,
        13, 13, 22, 23, 23, 24, 24, 25, 24, 25, 25, 23, 24, 22, 23, 21, 21, 19,
        20, 18, 19, 17, 18, 16, 17, 14, 16, 14, 15, 13, 14, 13, 13, 12,
        /* Size 4x8 */
        64, 58, 40, 26, 63, 56, 41, 28, 60, 48, 38, 27, 53, 40, 31, 24, 43, 35,
        25, 20, 34, 30, 22, 17, 28, 26, 20, 15, 25, 24, 18, 14,
        /* Size 8x4 */
        64, 63, 60, 53, 43, 34, 28, 25, 58, 56, 48, 40, 35, 30, 26, 24, 40, 41,
        38, 31, 25, 22, 20, 18, 26, 28, 27, 24, 20, 17, 15, 14,
        /* Size 8x16 */
        64, 65, 61, 51, 40, 31, 26, 23, 65, 64, 61, 53, 42, 33, 27, 25, 65, 63,
        60, 53, 43, 34, 29, 26, 64, 62, 57, 51, 43, 35, 29, 26, 61, 60, 53, 46,
        39, 33, 28, 26, 57, 58, 49, 40, 35, 30, 26, 24, 52, 53, 47, 38, 32, 28,
        24, 23, 47, 49, 44, 35, 29, 26, 23, 21, 43, 46, 41, 33, 28, 24, 21, 20,
        38, 41, 38, 31, 26, 22, 20, 18, 35, 38, 36, 29, 24, 21, 19, 17, 31, 34,
        33, 27, 23, 19, 17, 16, 27, 30, 29, 25, 21, 18, 16, 15, 25, 28, 27, 24,
        20, 17, 15, 14, 24, 26, 26, 23, 19, 17, 15, 14, 22, 25, 25, 22, 19, 17,
        15, 13,
        /* Size 16x8 */
        64, 65, 65, 64, 61, 57, 52, 47, 43, 38, 35, 31, 27, 25, 24, 22, 65, 64,
        63, 62, 60, 58, 53, 49, 46, 41, 38, 34, 30, 28, 26, 25, 61, 61, 60, 57,
        53, 49, 47, 44, 41, 38, 36, 33, 29, 27, 26, 25, 51, 53, 53, 51, 46, 40,
        38, 35, 33, 31, 29, 27, 25, 24, 23, 22, 40, 42, 43, 43, 39, 35, 32, 29,
        28, 26, 24, 23, 21, 20, 19, 19, 31, 33, 34, 35, 33, 30, 28, 26, 24, 22,
        21, 19, 18, 17, 17, 17, 26, 27, 29, 29, 28, 26, 24, 23, 21, 20, 19, 17,
        16, 15, 15, 15, 23, 25, 26, 26, 26, 24, 23, 21, 20, 18, 17, 16, 15, 14,
        14, 13,
        /* Size 16x32 */
        64, 65, 65, 64, 61, 57, 51, 47, 40, 38, 31, 31, 26, 25, 23, 22, 65, 64,
        64, 63, 61, 58, 52, 48, 41, 40, 33, 32, 27, 26, 24, 23, 65, 64, 64, 63,
        61, 59, 53, 49, 42, 40, 33, 33, 27, 27, 25, 23, 65, 63, 63, 62, 61, 59,
        54, 50, 43, 41, 34, 34, 28, 28, 26, 25, 65, 63, 63, 61, 60, 58, 53, 49,
        43, 41, 34, 34, 29, 28, 26, 24, 64, 63, 62, 60, 59, 56, 52, 49, 43, 41,
        35, 34, 29, 28, 26, 25, 64, 63, 62, 59, 57, 55, 51, 48, 43, 41, 35, 34,
        29, 29, 26, 25, 64, 62, 61, 58, 56, 53, 50, 48, 43, 41, 35, 35, 30, 29,
        27, 26, 61, 61, 60, 56, 53, 49, 46, 44, 39, 38, 33, 33, 28, 27, 26, 26,
        60, 60, 59, 55, 53, 48, 45, 43, 39, 38, 33, 32, 28, 27, 25, 24, 57, 59,
        58, 53, 49, 43, 40, 38, 35, 34, 30, 30, 26, 26, 24, 24, 57, 59, 58, 53,
        49, 43, 40, 38, 35, 34, 30, 29, 26, 25, 24, 22, 52, 55, 53, 51, 47, 41,
        38, 35, 32, 32, 28, 27, 24, 24, 23, 23, 51, 53, 53, 49, 46, 40, 37, 34,
        32, 31, 27, 27, 24, 24, 23, 21, 47, 50, 49, 48, 44, 38, 35, 33, 29, 29,
        26, 25, 23, 22, 21, 22, 45, 48, 47, 46, 42, 37, 34, 32, 28, 28, 25, 25,
        22, 22, 21, 19, 43, 46, 46, 44, 41, 36, 33, 31, 28, 27, 24, 24, 21, 21,
        20, 20, 39, 43, 42, 42, 39, 35, 32, 29, 26, 25, 23, 23, 20, 20, 20, 18,
        38, 41, 41, 41, 38, 34, 31, 29, 26, 25, 22, 22, 20, 20, 18, 19, 35, 38,
        38, 38, 36, 32, 29, 27, 25, 24, 21, 21, 19, 18, 18, 17, 35, 38, 38, 38,
        36, 32, 29, 27, 24, 24, 21, 21, 19, 18, 17, 18, 32, 34, 34, 35, 33, 30,
        27, 26, 23, 22, 20, 19, 17, 17, 17, 16, 31, 34, 34, 34, 33, 30, 27, 25,
        23, 22, 19, 19, 17, 17, 16, 17, 29, 31, 32, 32, 31, 28, 26, 24, 22, 21,
        18, 18, 16, 16, 15, 15, 27, 30, 30, 31, 29, 27, 25, 24, 21, 20, 18, 18,
        16, 16, 15, 16, 26, 28, 28, 29, 28, 26, 24, 23, 20, 20, 17, 17, 15, 15,
        14, 14, 25, 28, 28, 29, 27, 25, 24, 22, 20, 20, 17, 17, 15, 15, 14, 15,
        25, 27, 27, 28, 27, 25, 23, 22, 20, 19, 17, 17, 15, 15, 14, 13, 24, 26,
        26, 27, 26, 24, 23, 22, 19, 19, 17, 16, 15, 14, 14, 14, 23, 25, 25, 26,
        25, 24, 22, 21, 19, 18, 17, 16, 14, 14, 13, 13, 22, 25, 25, 26, 25, 23,
        22, 21, 19, 18, 17, 16, 15, 14, 13, 14, 22, 24, 24, 25, 25, 22, 23, 19,
        20, 17, 18, 15, 16, 13, 14, 13,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 64, 64, 64, 61, 60, 57, 57, 52, 51, 47, 45, 43, 39,
        38, 35, 35, 32, 31, 29, 27, 26, 25, 25, 24, 23, 22, 22, 65, 64, 64, 63,
        63, 63, 63, 62, 61, 60, 59, 59, 55, 53, 50, 48, 46, 43, 41, 38, 38, 34,
        34, 31, 30, 28, 28, 27, 26, 25, 25, 24, 65, 64, 64, 63, 63, 62, 62, 61,
        60, 59, 58, 58, 53, 53, 49, 47, 46, 42, 41, 38, 38, 34, 34, 32, 30, 28,
        28, 27, 26, 25, 25, 24, 64, 63, 63, 62, 61, 60, 59, 58, 56, 55, 53, 53,
        51, 49, 48, 46, 44, 42, 41, 38, 38, 35, 34, 32, 31, 29, 29, 28, 27, 26,
        26, 25, 61, 61, 61, 61, 60, 59, 57, 56, 53, 53, 49, 49, 47, 46, 44, 42,
        41, 39, 38, 36, 36, 33, 33, 31, 29, 28, 27, 27, 26, 25, 25, 25, 57, 58,
        59, 59, 58, 56, 55, 53, 49, 48, 43, 43, 41, 40, 38, 37, 36, 35, 34, 32,
        32, 30, 30, 28, 27, 26, 25, 25, 24, 24, 23, 22, 51, 52, 53, 54, 53, 52,
        51, 50, 46, 45, 40, 40, 38, 37, 35, 34, 33, 32, 31, 29, 29, 27, 27, 26,
        25, 24, 24, 23, 23, 22, 22, 23, 47, 48, 49, 50, 49, 49, 48, 48, 44, 43,
        38, 38, 35, 34, 33, 32, 31, 29, 29, 27, 27, 26, 25, 24, 24, 23, 22, 22,
        22, 21, 21, 19, 40, 41, 42, 43, 43, 43, 43, 43, 39, 39, 35, 35, 32, 32,
        29, 28, 28, 26, 26, 25, 24, 23, 23, 22, 21, 20, 20, 20, 19, 19, 19, 20,
        38, 40, 40, 41, 41, 41, 41, 41, 38, 38, 34, 34, 32, 31, 29, 28, 27, 25,
        25, 24, 24, 22, 22, 21, 20, 20, 20, 19, 19, 18, 18, 17, 31, 33, 33, 34,
        34, 35, 35, 35, 33, 33, 30, 30, 28, 27, 26, 25, 24, 23, 22, 21, 21, 20,
        19, 18, 18, 17, 17, 17, 17, 17, 17, 18, 31, 32, 33, 34, 34, 34, 34, 35,
        33, 32, 30, 29, 27, 27, 25, 25, 24, 23, 22, 21, 21, 19, 19, 18, 18, 17,
        17, 17, 16, 16, 16, 15, 26, 27, 27, 28, 29, 29, 29, 30, 28, 28, 26, 26,
        24, 24, 23, 22, 21, 20, 20, 19, 19, 17, 17, 16, 16, 15, 15, 15, 15, 14,
        15, 16, 25, 26, 27, 28, 28, 28, 29, 29, 27, 27, 26, 25, 24, 24, 22, 22,
        21, 20, 20, 18, 18, 17, 17, 16, 16, 15, 15, 15, 14, 14, 14, 13, 23, 24,
        25, 26, 26, 26, 26, 27, 26, 25, 24, 24, 23, 23, 21, 21, 20, 20, 18, 18,
        17, 17, 16, 15, 15, 14, 14, 14, 14, 13, 13, 14, 22, 23, 23, 25, 24, 25,
        25, 26, 26, 24, 24, 22, 23, 21, 22, 19, 20, 18, 19, 17, 18, 16, 17, 15,
        16, 14, 15, 13, 14, 13, 14, 13,
        /* Size 4x16 */
        65, 57, 38, 25, 64, 59, 40, 27, 63, 58, 41, 28, 63, 55, 41, 29, 61, 49,
        38, 27, 59, 43, 34, 26, 55, 41, 32, 24, 50, 38, 29, 22, 46, 36, 27, 21,
        41, 34, 25, 20, 38, 32, 24, 18, 34, 30, 22, 17, 30, 27, 20, 16, 28, 25,
        20, 15, 26, 24, 19, 14, 25, 23, 18, 14,
        /* Size 16x4 */
        65, 64, 63, 63, 61, 59, 55, 50, 46, 41, 38, 34, 30, 28, 26, 25, 57, 59,
        58, 55, 49, 43, 41, 38, 36, 34, 32, 30, 27, 25, 24, 23, 38, 40, 41, 41,
        38, 34, 32, 29, 27, 25, 24, 22, 20, 20, 19, 18, 25, 27, 28, 29, 27, 26,
        24, 22, 21, 20, 18, 17, 16, 15, 14, 14,
        /* Size 8x32 */
        64, 65, 61, 51, 40, 31, 26, 23, 65, 64, 61, 52, 41, 33, 27, 24, 65, 64,
        61, 53, 42, 33, 27, 25, 65, 63, 61, 54, 43, 34, 28, 26, 65, 63, 60, 53,
        43, 34, 29, 26, 64, 62, 59, 52, 43, 35, 29, 26, 64, 62, 57, 51, 43, 35,
        29, 26, 64, 61, 56, 50, 43, 35, 30, 27, 61, 60, 53, 46, 39, 33, 28, 26,
        60, 59, 53, 45, 39, 33, 28, 25, 57, 58, 49, 40, 35, 30, 26, 24, 57, 58,
        49, 40, 35, 30, 26, 24, 52, 53, 47, 38, 32, 28, 24, 23, 51, 53, 46, 37,
        32, 27, 24, 23, 47, 49, 44, 35, 29, 26, 23, 21, 45, 47, 42, 34, 28, 25,
        22, 21, 43, 46, 41, 33, 28, 24, 21, 20, 39, 42, 39, 32, 26, 23, 20, 20,
        38, 41, 38, 31, 26, 22, 20, 18, 35, 38, 36, 29, 25, 21, 19, 18, 35, 38,
        36, 29, 24, 21, 19, 17, 32, 34, 33, 27, 23, 20, 17, 17, 31, 34, 33, 27,
        23, 19, 17, 16, 29, 32, 31, 26, 22, 18, 16, 15, 27, 30, 29, 25, 21, 18,
        16, 15, 26, 28, 28, 24, 20, 17, 15, 14, 25, 28, 27, 24, 20, 17, 15, 14,
        25, 27, 27, 23, 20, 17, 15, 14, 24, 26, 26, 23, 19, 17, 15, 14, 23, 25,
        25, 22, 19, 17, 14, 13, 22, 25, 25, 22, 19, 17, 15, 13, 22, 24, 25, 23,
        20, 18, 16, 14,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 64, 64, 64, 61, 60, 57, 57, 52, 51, 47, 45, 43, 39,
        38, 35, 35, 32, 31, 29, 27, 26, 25, 25, 24, 23, 22, 22, 65, 64, 64, 63,
        63, 62, 62, 61, 60, 59, 58, 58, 53, 53, 49, 47, 46, 42, 41, 38, 38, 34,
        34, 32, 30, 28, 28, 27, 26, 25, 25, 24, 61, 61, 61, 61, 60, 59, 57, 56,
        53, 53, 49, 49, 47, 46, 44, 42, 41, 39, 38, 36, 36, 33, 33, 31, 29, 28,
        27, 27, 26, 25, 25, 25, 51, 52, 53, 54, 53, 52, 51, 50, 46, 45, 40, 40,
        38, 37, 35, 34, 33, 32, 31, 29, 29, 27, 27, 26, 25, 24, 24, 23, 23, 22,
        22, 23, 40, 41, 42, 43, 43, 43, 43, 43, 39, 39, 35, 35, 32, 32, 29, 28,
        28, 26, 26, 25, 24, 23, 23, 22, 21, 20, 20, 20, 19, 19, 19, 20, 31, 33,
        33, 34, 34, 35, 35, 35, 33, 33, 30, 30, 28, 27, 26, 25, 24, 23, 22, 21,
        21, 20, 19, 18, 18, 17, 17, 17, 17, 17, 17, 18, 26, 27, 27, 28, 29, 29,
        29, 30, 28, 28, 26, 26, 24, 24, 23, 22, 21, 20, 20, 19, 19, 17, 17, 16,
        16, 15, 15, 15, 15, 14, 15, 16, 23, 24, 25, 26, 26, 26, 26, 27, 26, 25,
        24, 24, 23, 23, 21, 21, 20, 20, 18, 18, 17, 17, 16, 15, 15, 14, 14, 14,
        14, 13, 13, 14 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        43, 39, 39, 25,
#endif
        /* Size 4x4 */
        64, 44, 41, 35, 44, 39, 37, 33, 41, 37, 29, 26, 35, 33, 26, 22,
        /* Size 8x8 */
        65, 59, 48, 44, 41, 38, 34, 32, 59, 53, 46, 45, 43, 40, 36, 34, 48, 46,
        42, 41, 41, 38, 36, 34, 44, 45, 41, 37, 35, 34, 32, 31, 41, 43, 41, 35,
        32, 29, 28, 27, 38, 40, 38, 34, 29, 27, 25, 25, 34, 36, 36, 32, 28, 25,
        23, 22, 32, 34, 34, 31, 27, 25, 22, 21,
        /* Size 16x16 */
        64, 66, 65, 58, 51, 42, 42, 42, 41, 39, 38, 36, 33, 32, 31, 30, 66, 65,
        63, 55, 49, 43, 44, 44, 43, 41, 40, 38, 36, 34, 33, 32, 65, 63, 59, 52,
        48, 44, 45, 46, 44, 43, 41, 39, 37, 36, 34, 33, 58, 55, 52, 47, 45, 43,
        44, 45, 44, 44, 42, 40, 38, 37, 36, 34, 51, 49, 48, 45, 43, 41, 42, 42,
        42, 41, 40, 39, 37, 36, 35, 34, 42, 43, 44, 43, 41, 39, 39, 38, 38, 38,
        37, 36, 34, 34, 33, 33, 42, 44, 45, 44, 42, 39, 38, 37, 36, 36, 35, 34,
        33, 32, 31, 31, 42, 44, 46, 45, 42, 38, 37, 35, 34, 33, 33, 32, 31, 30,
        30, 29, 41, 43, 44, 44, 42, 38, 36, 34, 33, 32, 32, 30, 29, 29, 28, 28,
        39, 41, 43, 44, 41, 38, 36, 33, 32, 31, 30, 29, 28, 27, 27, 26, 38, 40,
        41, 42, 40, 37, 35, 33, 32, 30, 29, 28, 27, 26, 26, 25, 36, 38, 39, 40,
        39, 36, 34, 32, 30, 29, 28, 26, 25, 25, 24, 23, 33, 36, 37, 38, 37, 34,
        33, 31, 29, 28, 27, 25, 24, 23, 23, 22, 32, 34, 36, 37, 36, 34, 32, 30,
        29, 27, 26, 25, 23, 23, 22, 21, 31, 33, 34, 36, 35, 33, 31, 30, 28, 27,
        26, 24, 23, 22, 22, 21, 30, 32, 33, 34, 34, 33, 31, 29, 28, 26, 25, 23,
        22, 21, 21, 21,
        /* Size 32x32 */
        64, 66, 66, 67, 65, 61, 58, 56, 51, 49, 42, 42, 42, 42, 42, 41, 41, 39,
        39, 38, 38, 36, 36, 34, 33, 32, 32, 32, 31, 30, 30, 29, 66, 65, 65, 66,
        63, 59, 55, 53, 49, 48, 43, 43, 44, 44, 44, 43, 42, 41, 41, 39, 39, 38,
        37, 36, 35, 34, 34, 33, 33, 32, 32, 31, 66, 65, 65, 65, 63, 59, 55, 53,
        49, 48, 43, 43, 44, 44, 44, 43, 43, 42, 41, 40, 40, 38, 38, 36, 36, 34,
        34, 34, 33, 32, 32, 31, 67, 66, 65, 64, 62, 58, 54, 51, 49, 48, 44, 44,
        45, 45, 46, 45, 44, 43, 43, 41, 41, 39, 39, 38, 37, 36, 35, 35, 34, 34,
        33, 33, 65, 63, 63, 62, 59, 55, 52, 49, 48, 47, 44, 44, 45, 45, 46, 45,
        44, 43, 43, 41, 41, 40, 39, 38, 37, 36, 36, 35, 34, 34, 33, 33, 61, 59,
        59, 58, 55, 53, 49, 47, 46, 46, 43, 44, 44, 45, 45, 45, 44, 43, 43, 42,
        41, 40, 40, 38, 38, 36, 36, 36, 35, 34, 34, 34, 58, 55, 55, 54, 52, 49,
        47, 45, 45, 44, 43, 43, 44, 45, 45, 45, 44, 44, 44, 42, 42, 41, 40, 39,
        38, 37, 37, 36, 36, 35, 34, 33, 56, 53, 53, 51, 49, 47, 45, 43, 43, 43,
        43, 43, 44, 45, 45, 45, 45, 44, 44, 43, 42, 41, 41, 39, 39, 38, 37, 37,
        36, 36, 35, 35, 51, 49, 49, 49, 48, 46, 45, 43, 43, 42, 41, 41, 42, 42,
        42, 42, 42, 41, 41, 40, 40, 39, 39, 38, 37, 36, 36, 35, 35, 34, 34, 35,
        49, 48, 48, 48, 47, 46, 44, 43, 42, 42, 41, 41, 41, 41, 42, 41, 41, 41,
        41, 39, 39, 38, 38, 37, 37, 36, 35, 35, 34, 34, 33, 32, 42, 43, 43, 44,
        44, 43, 43, 43, 41, 41, 39, 39, 39, 38, 38, 38, 38, 38, 38, 37, 37, 36,
        36, 35, 34, 34, 34, 33, 33, 33, 33, 33, 42, 43, 43, 44, 44, 44, 43, 43,
        41, 41, 39, 39, 38, 38, 38, 38, 38, 38, 38, 37, 37, 36, 36, 35, 34, 34,
        33, 33, 33, 32, 32, 31, 42, 44, 44, 45, 45, 44, 44, 44, 42, 41, 39, 38,
        38, 38, 37, 37, 36, 36, 36, 35, 35, 34, 34, 33, 33, 32, 32, 32, 31, 31,
        31, 32, 42, 44, 44, 45, 45, 45, 45, 45, 42, 41, 38, 38, 38, 37, 36, 36,
        36, 35, 35, 34, 34, 34, 33, 33, 32, 32, 31, 31, 31, 31, 31, 29, 42, 44,
        44, 46, 46, 45, 45, 45, 42, 42, 38, 38, 37, 36, 35, 35, 34, 34, 33, 33,
        33, 32, 32, 31, 31, 30, 30, 30, 30, 29, 29, 30, 41, 43, 43, 45, 45, 45,
        45, 45, 42, 41, 38, 38, 37, 36, 35, 34, 34, 33, 33, 32, 32, 31, 31, 31,
        30, 30, 29, 29, 29, 29, 29, 28, 41, 42, 43, 44, 44, 44, 44, 45, 42, 41,
        38, 38, 36, 36, 34, 34, 33, 33, 32, 32, 32, 31, 30, 30, 29, 29, 29, 29,
        28, 28, 28, 29, 39, 41, 42, 43, 43, 43, 44, 44, 41, 41, 38, 38, 36, 35,
        34, 33, 33, 32, 31, 30, 30, 29, 29, 29, 28, 28, 28, 27, 27, 27, 28, 26,
        39, 41, 41, 43, 43, 43, 44, 44, 41, 41, 38, 38, 36, 35, 33, 33, 32, 31,
        31, 30, 30, 29, 29, 28, 28, 27, 27, 27, 27, 26, 26, 27, 38, 39, 40, 41,
        41, 42, 42, 43, 40, 39, 37, 37, 35, 34, 33, 32, 32, 30, 30, 29, 29, 28,
        28, 27, 27, 26, 26, 26, 26, 26, 26, 25, 38, 39, 40, 41, 41, 41, 42, 42,
        40, 39, 37, 37, 35, 34, 33, 32, 32, 30, 30, 29, 29, 28, 28, 27, 27, 26,
        26, 26, 26, 25, 25, 26, 36, 38, 38, 39, 40, 40, 41, 41, 39, 38, 36, 36,
        34, 34, 32, 31, 31, 29, 29, 28, 28, 27, 27, 26, 26, 25, 25, 25, 24, 25,
        25, 24, 36, 37, 38, 39, 39, 40, 40, 41, 39, 38, 36, 36, 34, 33, 32, 31,
        30, 29, 29, 28, 28, 27, 26, 26, 25, 25, 25, 25, 24, 24, 23, 25, 34, 36,
        36, 38, 38, 38, 39, 39, 38, 37, 35, 35, 33, 33, 31, 31, 30, 29, 28, 27,
        27, 26, 26, 25, 25, 24, 24, 24, 23, 23, 24, 22, 33, 35, 36, 37, 37, 38,
        38, 39, 37, 37, 34, 34, 33, 32, 31, 30, 29, 28, 28, 27, 27, 26, 25, 25,
        24, 24, 23, 23, 23, 23, 22, 24, 32, 34, 34, 36, 36, 36, 37, 38, 36, 36,
        34, 34, 32, 32, 30, 30, 29, 28, 27, 26, 26, 25, 25, 24, 24, 23, 23, 23,
        22, 22, 23, 22, 32, 34, 34, 35, 36, 36, 37, 37, 36, 35, 34, 33, 32, 31,
        30, 29, 29, 28, 27, 26, 26, 25, 25, 24, 23, 23, 23, 23, 22, 22, 21, 23,
        32, 33, 34, 35, 35, 36, 36, 37, 35, 35, 33, 33, 32, 31, 30, 29, 29, 27,
        27, 26, 26, 25, 25, 24, 23, 23, 23, 22, 22, 22, 22, 20, 31, 33, 33, 34,
        34, 35, 36, 36, 35, 34, 33, 33, 31, 31, 30, 29, 28, 27, 27, 26, 26, 24,
        24, 23, 23, 22, 22, 22, 22, 22, 21, 22, 30, 32, 32, 34, 34, 34, 35, 36,
        34, 34, 33, 32, 31, 31, 29, 29, 28, 27, 26, 26, 25, 25, 24, 23, 23, 22,
        22, 22, 22, 21, 21, 21, 30, 32, 32, 33, 33, 34, 34, 35, 34, 33, 33, 32,
        31, 31, 29, 29, 28, 28, 26, 26, 25, 25, 23, 24, 22, 23, 21, 22, 21, 21,
        21, 21, 29, 31, 31, 33, 33, 34, 33, 35, 35, 32, 33, 31, 32, 29, 30, 28,
        29, 26, 27, 25, 26, 24, 25, 22, 24, 22, 23, 20, 22, 21, 21, 20,
        /* Size 4x8 */
        65, 44, 41, 34, 57, 44, 43, 36, 48, 41, 41, 35, 45, 39, 35, 31, 43, 38,
        31, 28, 39, 36, 29, 25, 36, 34, 27, 23, 33, 32, 26, 22,
        /* Size 8x4 */
        65, 57, 48, 45, 43, 39, 36, 33, 44, 44, 41, 39, 38, 36, 34, 32, 41, 43,
        41, 35, 31, 29, 27, 26, 34, 36, 35, 31, 28, 25, 23, 22,
        /* Size 8x16 */
        64, 64, 51, 42, 40, 36, 32, 30, 66, 62, 49, 44, 42, 38, 34, 32, 65, 59,
        48, 45, 43, 40, 36, 34, 58, 52, 45, 44, 44, 41, 37, 35, 50, 48, 43, 42,
        41, 39, 36, 34, 42, 44, 41, 38, 38, 36, 34, 33, 42, 45, 42, 38, 36, 34,
        32, 31, 42, 45, 42, 36, 34, 32, 30, 29, 41, 44, 41, 36, 33, 31, 29, 28,
        39, 43, 41, 35, 31, 29, 27, 26, 38, 41, 40, 34, 31, 28, 26, 25, 36, 39,
        39, 34, 29, 27, 25, 24, 34, 37, 37, 32, 28, 26, 24, 23, 32, 35, 35, 32,
        28, 25, 23, 22, 31, 34, 34, 31, 27, 24, 22, 22, 30, 33, 34, 31, 27, 25,
        22, 21,
        /* Size 16x8 */
        64, 66, 65, 58, 50, 42, 42, 42, 41, 39, 38, 36, 34, 32, 31, 30, 64, 62,
        59, 52, 48, 44, 45, 45, 44, 43, 41, 39, 37, 35, 34, 33, 51, 49, 48, 45,
        43, 41, 42, 42, 41, 41, 40, 39, 37, 35, 34, 34, 42, 44, 45, 44, 42, 38,
        38, 36, 36, 35, 34, 34, 32, 32, 31, 31, 40, 42, 43, 44, 41, 38, 36, 34,
        33, 31, 31, 29, 28, 28, 27, 27, 36, 38, 40, 41, 39, 36, 34, 32, 31, 29,
        28, 27, 26, 25, 24, 25, 32, 34, 36, 37, 36, 34, 32, 30, 29, 27, 26, 25,
        24, 23, 22, 22, 30, 32, 34, 35, 34, 33, 31, 29, 28, 26, 25, 24, 23, 22,
        22, 21,
        /* Size 16x32 */
        64, 66, 64, 55, 51, 43, 42, 42, 40, 39, 36, 35, 32, 32, 30, 29, 65, 65,
        62, 53, 49, 44, 44, 44, 41, 41, 38, 37, 34, 34, 32, 31, 66, 65, 62, 53,
        49, 44, 44, 44, 42, 41, 38, 38, 34, 34, 32, 31, 67, 64, 61, 51, 49, 45,
        45, 45, 43, 43, 39, 39, 36, 35, 34, 33, 65, 61, 59, 50, 48, 44, 45, 45,
        43, 43, 40, 39, 36, 36, 34, 33, 61, 57, 55, 48, 46, 44, 44, 45, 43, 43,
        40, 39, 37, 36, 34, 34, 58, 53, 52, 45, 45, 43, 44, 45, 44, 43, 41, 40,
        37, 37, 35, 34, 55, 51, 49, 44, 43, 43, 44, 45, 44, 44, 41, 41, 38, 37,
        36, 35, 50, 48, 48, 43, 43, 41, 42, 42, 41, 41, 39, 39, 36, 36, 34, 35,
        49, 48, 47, 43, 42, 41, 41, 41, 41, 41, 38, 38, 36, 35, 34, 32, 42, 44,
        44, 43, 41, 39, 38, 38, 38, 38, 36, 36, 34, 34, 33, 33, 42, 44, 44, 43,
        41, 39, 38, 38, 38, 37, 36, 36, 34, 33, 33, 31, 42, 45, 45, 44, 42, 39,
        38, 37, 36, 36, 34, 34, 32, 32, 31, 32, 42, 45, 45, 44, 42, 39, 37, 36,
        35, 35, 34, 33, 32, 31, 31, 29, 42, 45, 45, 45, 42, 38, 36, 35, 34, 33,
        32, 32, 30, 30, 29, 30, 41, 45, 45, 45, 42, 38, 36, 35, 33, 33, 31, 31,
        30, 29, 29, 28, 41, 44, 44, 44, 41, 38, 36, 34, 33, 32, 31, 30, 29, 29,
        28, 29, 40, 43, 43, 44, 41, 38, 35, 34, 32, 31, 29, 29, 28, 28, 27, 26,
        39, 43, 43, 43, 41, 38, 35, 34, 31, 31, 29, 29, 27, 27, 26, 27, 38, 41,
        41, 42, 40, 37, 34, 33, 31, 30, 28, 28, 26, 26, 26, 25, 38, 41, 41, 42,
        40, 37, 34, 33, 31, 30, 28, 28, 26, 26, 25, 26, 36, 39, 39, 41, 39, 36,
        34, 32, 30, 29, 27, 27, 25, 25, 25, 24, 36, 39, 39, 40, 39, 36, 34, 32,
        29, 29, 27, 26, 25, 25, 24, 25, 34, 38, 38, 39, 37, 35, 33, 31, 29, 28,
        26, 26, 24, 24, 23, 23, 34, 37, 37, 38, 37, 34, 32, 31, 28, 28, 26, 25,
        24, 23, 23, 24, 32, 36, 36, 37, 36, 34, 32, 30, 28, 27, 25, 25, 23, 23,
        22, 22, 32, 35, 35, 37, 35, 34, 32, 30, 28, 27, 25, 25, 23, 23, 22, 23,
        32, 35, 35, 37, 35, 33, 31, 30, 28, 27, 25, 25, 23, 23, 22, 20, 31, 34,
        34, 36, 34, 33, 31, 30, 27, 27, 24, 24, 22, 22, 22, 22, 30, 33, 34, 35,
        34, 32, 31, 29, 27, 26, 25, 24, 22, 22, 21, 21, 30, 33, 33, 35, 34, 32,
        31, 29, 27, 26, 25, 24, 22, 22, 21, 21, 29, 33, 32, 34, 34, 31, 32, 28,
        29, 25, 26, 22, 24, 20, 22, 20,
        /* Size 32x16 */
        64, 65, 66, 67, 65, 61, 58, 55, 50, 49, 42, 42, 42, 42, 42, 41, 41, 40,
        39, 38, 38, 36, 36, 34, 34, 32, 32, 32, 31, 30, 30, 29, 66, 65, 65, 64,
        61, 57, 53, 51, 48, 48, 44, 44, 45, 45, 45, 45, 44, 43, 43, 41, 41, 39,
        39, 38, 37, 36, 35, 35, 34, 33, 33, 33, 64, 62, 62, 61, 59, 55, 52, 49,
        48, 47, 44, 44, 45, 45, 45, 45, 44, 43, 43, 41, 41, 39, 39, 38, 37, 36,
        35, 35, 34, 34, 33, 32, 55, 53, 53, 51, 50, 48, 45, 44, 43, 43, 43, 43,
        44, 44, 45, 45, 44, 44, 43, 42, 42, 41, 40, 39, 38, 37, 37, 37, 36, 35,
        35, 34, 51, 49, 49, 49, 48, 46, 45, 43, 43, 42, 41, 41, 42, 42, 42, 42,
        41, 41, 41, 40, 40, 39, 39, 37, 37, 36, 35, 35, 34, 34, 34, 34, 43, 44,
        44, 45, 44, 44, 43, 43, 41, 41, 39, 39, 39, 39, 38, 38, 38, 38, 38, 37,
        37, 36, 36, 35, 34, 34, 34, 33, 33, 32, 32, 31, 42, 44, 44, 45, 45, 44,
        44, 44, 42, 41, 38, 38, 38, 37, 36, 36, 36, 35, 35, 34, 34, 34, 34, 33,
        32, 32, 32, 31, 31, 31, 31, 32, 42, 44, 44, 45, 45, 45, 45, 45, 42, 41,
        38, 38, 37, 36, 35, 35, 34, 34, 34, 33, 33, 32, 32, 31, 31, 30, 30, 30,
        30, 29, 29, 28, 40, 41, 42, 43, 43, 43, 44, 44, 41, 41, 38, 38, 36, 35,
        34, 33, 33, 32, 31, 31, 31, 30, 29, 29, 28, 28, 28, 28, 27, 27, 27, 29,
        39, 41, 41, 43, 43, 43, 43, 44, 41, 41, 38, 37, 36, 35, 33, 33, 32, 31,
        31, 30, 30, 29, 29, 28, 28, 27, 27, 27, 27, 26, 26, 25, 36, 38, 38, 39,
        40, 40, 41, 41, 39, 38, 36, 36, 34, 34, 32, 31, 31, 29, 29, 28, 28, 27,
        27, 26, 26, 25, 25, 25, 24, 25, 25, 26, 35, 37, 38, 39, 39, 39, 40, 41,
        39, 38, 36, 36, 34, 33, 32, 31, 30, 29, 29, 28, 28, 27, 26, 26, 25, 25,
        25, 25, 24, 24, 24, 22, 32, 34, 34, 36, 36, 37, 37, 38, 36, 36, 34, 34,
        32, 32, 30, 30, 29, 28, 27, 26, 26, 25, 25, 24, 24, 23, 23, 23, 22, 22,
        22, 24, 32, 34, 34, 35, 36, 36, 37, 37, 36, 35, 34, 33, 32, 31, 30, 29,
        29, 28, 27, 26, 26, 25, 25, 24, 23, 23, 23, 23, 22, 22, 22, 20, 30, 32,
        32, 34, 34, 34, 35, 36, 34, 34, 33, 33, 31, 31, 29, 29, 28, 27, 26, 26,
        25, 25, 24, 23, 23, 22, 22, 22, 22, 21, 21, 22, 29, 31, 31, 33, 33, 34,
        34, 35, 35, 32, 33, 31, 32, 29, 30, 28, 29, 26, 27, 25, 26, 24, 25, 23,
        24, 22, 23, 20, 22, 21, 21, 20,
        /* Size 4x16 */
        66, 43, 39, 32, 65, 44, 41, 34, 61, 44, 43, 36, 53, 43, 43, 37, 48, 41,
        41, 36, 44, 39, 38, 34, 45, 39, 36, 32, 45, 38, 33, 30, 44, 38, 32, 29,
        43, 38, 31, 27, 41, 37, 30, 26, 39, 36, 29, 25, 37, 34, 28, 23, 35, 34,
        27, 23, 34, 33, 27, 22, 33, 32, 26, 22,
        /* Size 16x4 */
        66, 65, 61, 53, 48, 44, 45, 45, 44, 43, 41, 39, 37, 35, 34, 33, 43, 44,
        44, 43, 41, 39, 39, 38, 38, 38, 37, 36, 34, 34, 33, 32, 39, 41, 43, 43,
        41, 38, 36, 33, 32, 31, 30, 29, 28, 27, 27, 26, 32, 34, 36, 37, 36, 34,
        32, 30, 29, 27, 26, 25, 23, 23, 22, 22,
        /* Size 8x32 */
        64, 64, 51, 42, 40, 36, 32, 30, 65, 62, 49, 44, 41, 38, 34, 32, 66, 62,
        49, 44, 42, 38, 34, 32, 67, 61, 49, 45, 43, 39, 36, 34, 65, 59, 48, 45,
        43, 40, 36, 34, 61, 55, 46, 44, 43, 40, 37, 34, 58, 52, 45, 44, 44, 41,
        37, 35, 55, 49, 43, 44, 44, 41, 38, 36, 50, 48, 43, 42, 41, 39, 36, 34,
        49, 47, 42, 41, 41, 38, 36, 34, 42, 44, 41, 38, 38, 36, 34, 33, 42, 44,
        41, 38, 38, 36, 34, 33, 42, 45, 42, 38, 36, 34, 32, 31, 42, 45, 42, 37,
        35, 34, 32, 31, 42, 45, 42, 36, 34, 32, 30, 29, 41, 45, 42, 36, 33, 31,
        30, 29, 41, 44, 41, 36, 33, 31, 29, 28, 40, 43, 41, 35, 32, 29, 28, 27,
        39, 43, 41, 35, 31, 29, 27, 26, 38, 41, 40, 34, 31, 28, 26, 26, 38, 41,
        40, 34, 31, 28, 26, 25, 36, 39, 39, 34, 30, 27, 25, 25, 36, 39, 39, 34,
        29, 27, 25, 24, 34, 38, 37, 33, 29, 26, 24, 23, 34, 37, 37, 32, 28, 26,
        24, 23, 32, 36, 36, 32, 28, 25, 23, 22, 32, 35, 35, 32, 28, 25, 23, 22,
        32, 35, 35, 31, 28, 25, 23, 22, 31, 34, 34, 31, 27, 24, 22, 22, 30, 34,
        34, 31, 27, 25, 22, 21, 30, 33, 34, 31, 27, 25, 22, 21, 29, 32, 34, 32,
        29, 26, 24, 22,
        /* Size 32x8 */
        64, 65, 66, 67, 65, 61, 58, 55, 50, 49, 42, 42, 42, 42, 42, 41, 41, 40,
        39, 38, 38, 36, 36, 34, 34, 32, 32, 32, 31, 30, 30, 29, 64, 62, 62, 61,
        59, 55, 52, 49, 48, 47, 44, 44, 45, 45, 45, 45, 44, 43, 43, 41, 41, 39,
        39, 38, 37, 36, 35, 35, 34, 34, 33, 32, 51, 49, 49, 49, 48, 46, 45, 43,
        43, 42, 41, 41, 42, 42, 42, 42, 41, 41, 41, 40, 40, 39, 39, 37, 37, 36,
        35, 35, 34, 34, 34, 34, 42, 44, 44, 45, 45, 44, 44, 44, 42, 41, 38, 38,
        38, 37, 36, 36, 36, 35, 35, 34, 34, 34, 34, 33, 32, 32, 32, 31, 31, 31,
        31, 32, 40, 41, 42, 43, 43, 43, 44, 44, 41, 41, 38, 38, 36, 35, 34, 33,
        33, 32, 31, 31, 31, 30, 29, 29, 28, 28, 28, 28, 27, 27, 27, 29, 36, 38,
        38, 39, 40, 40, 41, 41, 39, 38, 36, 36, 34, 34, 32, 31, 31, 29, 29, 28,
        28, 27, 27, 26, 26, 25, 25, 25, 24, 25, 25, 26, 32, 34, 34, 36, 36, 37,
        37, 38, 36, 36, 34, 34, 32, 32, 30, 30, 29, 28, 27, 26, 26, 25, 25, 24,
        24, 23, 23, 23, 22, 22, 22, 24, 30, 32, 32, 34, 34, 34, 35, 36, 34, 34,
        33, 33, 31, 31, 29, 29, 28, 27, 26, 26, 25, 25, 24, 23, 23, 22, 22, 22,
        22, 21, 21, 22 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        58, 35, 35, 20,
#endif
        /* Size 4x4 */
        63, 59, 41, 28, 59, 42, 34, 26, 41, 34, 25, 20, 28, 26, 20, 15,
        /* Size 8x8 */
        64, 64, 60, 53, 44, 36, 30, 26, 64, 61, 58, 53, 45, 38, 32, 28, 60, 58,
        52, 46, 40, 35, 30, 27, 53, 53, 46, 38, 34, 30, 26, 24, 44, 45, 40, 34,
        29, 25, 23, 21, 36, 38, 35, 30, 25, 22, 20, 18, 30, 32, 30, 26, 23, 20,
        17, 16, 26, 28, 27, 24, 21, 18, 16, 15,
        /* Size 16x16 */
        64, 65, 65, 65, 64, 60, 57, 52, 47, 42, 38, 35, 31, 29, 26, 25, 65, 64,
        64, 64, 63, 60, 59, 53, 48, 44, 40, 36, 33, 30, 27, 26, 65, 64, 63, 63,
        63, 61, 59, 55, 50, 46, 41, 38, 34, 32, 28, 27, 65, 64, 63, 61, 60, 58,
        56, 53, 49, 45, 41, 38, 35, 32, 29, 28, 64, 63, 63, 60, 58, 55, 53, 51,
        48, 45, 41, 38, 35, 33, 29, 28, 60, 60, 61, 58, 55, 52, 48, 46, 43, 40,
        38, 35, 33, 30, 28, 27, 57, 59, 59, 56, 53, 48, 42, 41, 38, 36, 34, 32,
        30, 28, 26, 25, 52, 53, 55, 53, 51, 46, 41, 38, 35, 34, 32, 30, 28, 26,
        24, 24, 47, 48, 50, 49, 48, 43, 38, 35, 32, 31, 29, 27, 26, 24, 23, 22,
        42, 44, 46, 45, 45, 40, 36, 34, 31, 29, 27, 25, 24, 23, 21, 21, 38, 40,
        41, 41, 41, 38, 34, 32, 29, 27, 25, 24, 22, 21, 20, 19, 35, 36, 38, 38,
        38, 35, 32, 30, 27, 25, 24, 22, 21, 20, 19, 18, 31, 33, 34, 35, 35, 33,
        30, 28, 26, 24, 22, 21, 20, 18, 17, 17, 29, 30, 32, 32, 33, 30, 28, 26,
        24, 23, 21, 20, 18, 17, 16, 16, 26, 27, 28, 29, 29, 28, 26, 24, 23, 21,
        20, 19, 17, 16, 15, 15, 25, 26, 27, 28, 28, 27, 25, 24, 22, 21, 19, 18,
        17, 16, 15, 15,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 65, 65, 64, 64, 60, 60, 57, 57, 52, 52, 47, 47, 42,
        42, 38, 38, 35, 35, 31, 31, 29, 29, 26, 26, 25, 25, 23, 65, 64, 64, 64,
        64, 64, 64, 63, 63, 60, 60, 59, 59, 53, 53, 48, 48, 44, 44, 40, 40, 36,
        36, 33, 33, 30, 30, 27, 27, 26, 26, 25, 65, 64, 64, 64, 64, 64, 64, 63,
        63, 60, 60, 59, 59, 53, 53, 48, 48, 44, 44, 40, 40, 36, 36, 33, 33, 30,
        30, 27, 27, 26, 26, 25, 65, 64, 64, 63, 63, 63, 63, 63, 63, 61, 61, 59,
        59, 55, 55, 50, 50, 46, 46, 41, 41, 38, 38, 34, 34, 32, 32, 28, 28, 27,
        27, 26, 65, 64, 64, 63, 63, 63, 63, 63, 63, 61, 61, 59, 59, 55, 55, 50,
        50, 46, 46, 41, 41, 38, 38, 34, 34, 32, 32, 28, 28, 27, 27, 26, 65, 64,
        64, 63, 63, 61, 61, 60, 60, 58, 58, 56, 56, 53, 53, 49, 49, 45, 45, 41,
        41, 38, 38, 35, 35, 32, 32, 29, 29, 28, 28, 26, 65, 64, 64, 63, 63, 61,
        61, 60, 60, 58, 58, 56, 56, 53, 53, 49, 49, 45, 45, 41, 41, 38, 38, 35,
        35, 32, 32, 29, 29, 28, 28, 26, 64, 63, 63, 63, 63, 60, 60, 58, 58, 55,
        55, 53, 53, 51, 51, 48, 48, 45, 45, 41, 41, 38, 38, 35, 35, 33, 33, 29,
        29, 28, 28, 27, 64, 63, 63, 63, 63, 60, 60, 58, 58, 55, 55, 53, 53, 51,
        51, 48, 48, 45, 45, 41, 41, 38, 38, 35, 35, 33, 33, 29, 29, 28, 28, 27,
        60, 60, 60, 61, 61, 58, 58, 55, 55, 52, 52, 48, 48, 46, 46, 43, 43, 40,
        40, 38, 38, 35, 35, 33, 33, 30, 30, 28, 28, 27, 27, 26, 60, 60, 60, 61,
        61, 58, 58, 55, 55, 52, 52, 48, 48, 46, 46, 43, 43, 40, 40, 38, 38, 35,
        35, 33, 33, 30, 30, 28, 28, 27, 27, 26, 57, 59, 59, 59, 59, 56, 56, 53,
        53, 48, 48, 42, 42, 41, 41, 38, 38, 36, 36, 34, 34, 32, 32, 30, 30, 28,
        28, 26, 26, 25, 25, 24, 57, 59, 59, 59, 59, 56, 56, 53, 53, 48, 48, 42,
        42, 41, 41, 38, 38, 36, 36, 34, 34, 32, 32, 30, 30, 28, 28, 26, 26, 25,
        25, 24, 52, 53, 53, 55, 55, 53, 53, 51, 51, 46, 46, 41, 41, 38, 38, 35,
        35, 34, 34, 32, 32, 30, 30, 28, 28, 26, 26, 24, 24, 24, 24, 23, 52, 53,
        53, 55, 55, 53, 53, 51, 51, 46, 46, 41, 41, 38, 38, 35, 35, 34, 34, 32,
        32, 30, 30, 28, 28, 26, 26, 24, 24, 24, 24, 23, 47, 48, 48, 50, 50, 49,
        49, 48, 48, 43, 43, 38, 38, 35, 35, 32, 32, 31, 31, 29, 29, 27, 27, 26,
        26, 24, 24, 23, 23, 22, 22, 21, 47, 48, 48, 50, 50, 49, 49, 48, 48, 43,
        43, 38, 38, 35, 35, 32, 32, 31, 31, 29, 29, 27, 27, 26, 26, 24, 24, 23,
        23, 22, 22, 21, 42, 44, 44, 46, 46, 45, 45, 45, 45, 40, 40, 36, 36, 34,
        34, 31, 31, 29, 29, 27, 27, 25, 25, 24, 24, 23, 23, 21, 21, 21, 21, 20,
        42, 44, 44, 46, 46, 45, 45, 45, 45, 40, 40, 36, 36, 34, 34, 31, 31, 29,
        29, 27, 27, 25, 25, 24, 24, 23, 23, 21, 21, 21, 21, 20, 38, 40, 40, 41,
        41, 41, 41, 41, 41, 38, 38, 34, 34, 32, 32, 29, 29, 27, 27, 25, 25, 24,
        24, 22, 22, 21, 21, 20, 20, 19, 19, 19, 38, 40, 40, 41, 41, 41, 41, 41,
        41, 38, 38, 34, 34, 32, 32, 29, 29, 27, 27, 25, 25, 24, 24, 22, 22, 21,
        21, 20, 20, 19, 19, 19, 35, 36, 36, 38, 38, 38, 38, 38, 38, 35, 35, 32,
        32, 30, 30, 27, 27, 25, 25, 24, 24, 22, 22, 21, 21, 20, 20, 19, 19, 18,
        18, 18, 35, 36, 36, 38, 38, 38, 38, 38, 38, 35, 35, 32, 32, 30, 30, 27,
        27, 25, 25, 24, 24, 22, 22, 21, 21, 20, 20, 19, 19, 18, 18, 18, 31, 33,
        33, 34, 34, 35, 35, 35, 35, 33, 33, 30, 30, 28, 28, 26, 26, 24, 24, 22,
        22, 21, 21, 20, 20, 18, 18, 17, 17, 17, 17, 16, 31, 33, 33, 34, 34, 35,
        35, 35, 35, 33, 33, 30, 30, 28, 28, 26, 26, 24, 24, 22, 22, 21, 21, 20,
        20, 18, 18, 17, 17, 17, 17, 16, 29, 30, 30, 32, 32, 32, 32, 33, 33, 30,
        30, 28, 28, 26, 26, 24, 24, 23, 23, 21, 21, 20, 20, 18, 18, 17, 17, 16,
        16, 16, 16, 16, 29, 30, 30, 32, 32, 32, 32, 33, 33, 30, 30, 28, 28, 26,
        26, 24, 24, 23, 23, 21, 21, 20, 20, 18, 18, 17, 17, 16, 16, 16, 16, 16,
        26, 27, 27, 28, 28, 29, 29, 29, 29, 28, 28, 26, 26, 24, 24, 23, 23, 21,
        21, 20, 20, 19, 19, 17, 17, 16, 16, 15, 15, 15, 15, 14, 26, 27, 27, 28,
        28, 29, 29, 29, 29, 28, 28, 26, 26, 24, 24, 23, 23, 21, 21, 20, 20, 19,
        19, 17, 17, 16, 16, 15, 15, 15, 15, 14, 25, 26, 26, 27, 27, 28, 28, 28,
        28, 27, 27, 25, 25, 24, 24, 22, 22, 21, 21, 19, 19, 18, 18, 17, 17, 16,
        16, 15, 15, 15, 15, 14, 25, 26, 26, 27, 27, 28, 28, 28, 28, 27, 27, 25,
        25, 24, 24, 22, 22, 21, 21, 19, 19, 18, 18, 17, 17, 16, 16, 15, 15, 15,
        15, 14, 23, 25, 25, 26, 26, 26, 26, 27, 27, 26, 26, 24, 24, 23, 23, 21,
        21, 20, 20, 19, 19, 18, 18, 16, 16, 16, 16, 14, 14, 14, 14, 14,
        /* Size 4x8 */
        64, 58, 40, 27, 63, 56, 41, 29, 60, 48, 38, 28, 55, 41, 32, 24, 46, 36,
        27, 21, 38, 32, 24, 19, 31, 28, 21, 16, 27, 25, 19, 15,
        /* Size 8x4 */
        64, 63, 60, 55, 46, 38, 31, 27, 58, 56, 48, 41, 36, 32, 28, 25, 40, 41,
        38, 32, 27, 24, 21, 19, 27, 29, 28, 24, 21, 19, 16, 15,
        /* Size 8x16 */
        64, 65, 64, 57, 47, 38, 31, 26, 65, 64, 63, 58, 48, 40, 33, 27, 65, 63,
        62, 59, 50, 41, 34, 28, 64, 63, 60, 56, 49, 41, 35, 29, 64, 62, 58, 53,
        48, 41, 35, 30, 60, 60, 55, 48, 43, 38, 33, 28, 57, 59, 53, 43, 38, 34,
        30, 26, 52, 55, 51, 41, 35, 32, 28, 24, 47, 50, 48, 38, 33, 29, 26, 23,
        43, 46, 44, 36, 31, 27, 24, 21, 38, 41, 41, 34, 29, 25, 22, 20, 35, 38,
        38, 32, 27, 24, 21, 19, 32, 34, 35, 30, 26, 22, 20, 17, 29, 31, 32, 28,
        24, 21, 18, 16, 26, 28, 29, 26, 23, 20, 17, 15, 25, 27, 28, 25, 22, 19,
        17, 15,
        /* Size 16x8 */
        64, 65, 65, 64, 64, 60, 57, 52, 47, 43, 38, 35, 32, 29, 26, 25, 65, 64,
        63, 63, 62, 60, 59, 55, 50, 46, 41, 38, 34, 31, 28, 27, 64, 63, 62, 60,
        58, 55, 53, 51, 48, 44, 41, 38, 35, 32, 29, 28, 57, 58, 59, 56, 53, 48,
        43, 41, 38, 36, 34, 32, 30, 28, 26, 25, 47, 48, 50, 49, 48, 43, 38, 35,
        33, 31, 29, 27, 26, 24, 23, 22, 38, 40, 41, 41, 41, 38, 34, 32, 29, 27,
        25, 24, 22, 21, 20, 19, 31, 33, 34, 35, 35, 33, 30, 28, 26, 24, 22, 21,
        20, 18, 17, 17, 26, 27, 28, 29, 30, 28, 26, 24, 23, 21, 20, 19, 17, 16,
        15, 15,
        /* Size 16x32 */
        64, 65, 65, 64, 64, 57, 57, 47, 47, 38, 38, 31, 31, 26, 26, 24, 65, 64,
        64, 63, 63, 58, 58, 48, 48, 40, 40, 33, 33, 27, 27, 25, 65, 64, 64, 63,
        63, 58, 58, 48, 48, 40, 40, 33, 33, 27, 27, 25, 65, 63, 63, 62, 62, 59,
        59, 50, 50, 41, 41, 34, 34, 28, 28, 26, 65, 63, 63, 62, 62, 59, 59, 50,
        50, 41, 41, 34, 34, 28, 28, 26, 64, 63, 63, 60, 60, 56, 56, 49, 49, 41,
        41, 35, 35, 29, 29, 27, 64, 63, 63, 60, 60, 56, 56, 49, 49, 41, 41, 35,
        35, 29, 29, 27, 64, 62, 62, 58, 58, 53, 53, 48, 48, 41, 41, 35, 35, 30,
        30, 27, 64, 62, 62, 58, 58, 53, 53, 48, 48, 41, 41, 35, 35, 30, 30, 27,
        60, 60, 60, 55, 55, 48, 48, 43, 43, 38, 38, 33, 33, 28, 28, 26, 60, 60,
        60, 55, 55, 48, 48, 43, 43, 38, 38, 33, 33, 28, 28, 26, 57, 59, 59, 53,
        53, 43, 43, 38, 38, 34, 34, 30, 30, 26, 26, 24, 57, 59, 59, 53, 53, 43,
        43, 38, 38, 34, 34, 30, 30, 26, 26, 24, 52, 55, 55, 51, 51, 41, 41, 35,
        35, 32, 32, 28, 28, 24, 24, 23, 52, 55, 55, 51, 51, 41, 41, 35, 35, 32,
        32, 28, 28, 24, 24, 23, 47, 50, 50, 48, 48, 38, 38, 33, 33, 29, 29, 26,
        26, 23, 23, 21, 47, 50, 50, 48, 48, 38, 38, 33, 33, 29, 29, 26, 26, 23,
        23, 21, 43, 46, 46, 44, 44, 36, 36, 31, 31, 27, 27, 24, 24, 21, 21, 20,
        43, 46, 46, 44, 44, 36, 36, 31, 31, 27, 27, 24, 24, 21, 21, 20, 38, 41,
        41, 41, 41, 34, 34, 29, 29, 25, 25, 22, 22, 20, 20, 19, 38, 41, 41, 41,
        41, 34, 34, 29, 29, 25, 25, 22, 22, 20, 20, 19, 35, 38, 38, 38, 38, 32,
        32, 27, 27, 24, 24, 21, 21, 19, 19, 18, 35, 38, 38, 38, 38, 32, 32, 27,
        27, 24, 24, 21, 21, 19, 19, 18, 32, 34, 34, 35, 35, 30, 30, 26, 26, 22,
        22, 20, 20, 17, 17, 16, 32, 34, 34, 35, 35, 30, 30, 26, 26, 22, 22, 20,
        20, 17, 17, 16, 29, 31, 31, 32, 32, 28, 28, 24, 24, 21, 21, 18, 18, 16,
        16, 16, 29, 31, 31, 32, 32, 28, 28, 24, 24, 21, 21, 18, 18, 16, 16, 16,
        26, 28, 28, 29, 29, 26, 26, 23, 23, 20, 20, 17, 17, 15, 15, 15, 26, 28,
        28, 29, 29, 26, 26, 23, 23, 20, 20, 17, 17, 15, 15, 15, 25, 27, 27, 28,
        28, 25, 25, 22, 22, 19, 19, 17, 17, 15, 15, 14, 25, 27, 27, 28, 28, 25,
        25, 22, 22, 19, 19, 17, 17, 15, 15, 14, 24, 26, 26, 27, 27, 24, 24, 21,
        21, 19, 19, 16, 16, 15, 15, 14,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 64, 64, 64, 64, 60, 60, 57, 57, 52, 52, 47, 47, 43,
        43, 38, 38, 35, 35, 32, 32, 29, 29, 26, 26, 25, 25, 24, 65, 64, 64, 63,
        63, 63, 63, 62, 62, 60, 60, 59, 59, 55, 55, 50, 50, 46, 46, 41, 41, 38,
        38, 34, 34, 31, 31, 28, 28, 27, 27, 26, 65, 64, 64, 63, 63, 63, 63, 62,
        62, 60, 60, 59, 59, 55, 55, 50, 50, 46, 46, 41, 41, 38, 38, 34, 34, 31,
        31, 28, 28, 27, 27, 26, 64, 63, 63, 62, 62, 60, 60, 58, 58, 55, 55, 53,
        53, 51, 51, 48, 48, 44, 44, 41, 41, 38, 38, 35, 35, 32, 32, 29, 29, 28,
        28, 27, 64, 63, 63, 62, 62, 60, 60, 58, 58, 55, 55, 53, 53, 51, 51, 48,
        48, 44, 44, 41, 41, 38, 38, 35, 35, 32, 32, 29, 29, 28, 28, 27, 57, 58,
        58, 59, 59, 56, 56, 53, 53, 48, 48, 43, 43, 41, 41, 38, 38, 36, 36, 34,
        34, 32, 32, 30, 30, 28, 28, 26, 26, 25, 25, 24, 57, 58, 58, 59, 59, 56,
        56, 53, 53, 48, 48, 43, 43, 41, 41, 38, 38, 36, 36, 34, 34, 32, 32, 30,
        30, 28, 28, 26, 26, 25, 25, 24, 47, 48, 48, 50, 50, 49, 49, 48, 48, 43,
        43, 38, 38, 35, 35, 33, 33, 31, 31, 29, 29, 27, 27, 26, 26, 24, 24, 23,
        23, 22, 22, 21, 47, 48, 48, 50, 50, 49, 49, 48, 48, 43, 43, 38, 38, 35,
        35, 33, 33, 31, 31, 29, 29, 27, 27, 26, 26, 24, 24, 23, 23, 22, 22, 21,
        38, 40, 40, 41, 41, 41, 41, 41, 41, 38, 38, 34, 34, 32, 32, 29, 29, 27,
        27, 25, 25, 24, 24, 22, 22, 21, 21, 20, 20, 19, 19, 19, 38, 40, 40, 41,
        41, 41, 41, 41, 41, 38, 38, 34, 34, 32, 32, 29, 29, 27, 27, 25, 25, 24,
        24, 22, 22, 21, 21, 20, 20, 19, 19, 19, 31, 33, 33, 34, 34, 35, 35, 35,
        35, 33, 33, 30, 30, 28, 28, 26, 26, 24, 24, 22, 22, 21, 21, 20, 20, 18,
        18, 17, 17, 17, 17, 16, 31, 33, 33, 34, 34, 35, 35, 35, 35, 33, 33, 30,
        30, 28, 28, 26, 26, 24, 24, 22, 22, 21, 21, 20, 20, 18, 18, 17, 17, 17,
        17, 16, 26, 27, 27, 28, 28, 29, 29, 30, 30, 28, 28, 26, 26, 24, 24, 23,
        23, 21, 21, 20, 20, 19, 19, 17, 17, 16, 16, 15, 15, 15, 15, 15, 26, 27,
        27, 28, 28, 29, 29, 30, 30, 28, 28, 26, 26, 24, 24, 23, 23, 21, 21, 20,
        20, 19, 19, 17, 17, 16, 16, 15, 15, 15, 15, 15, 24, 25, 25, 26, 26, 27,
        27, 27, 27, 26, 26, 24, 24, 23, 23, 21, 21, 20, 20, 19, 19, 18, 18, 16,
        16, 16, 16, 15, 15, 14, 14, 14,
        /* Size 4x16 */
        65, 57, 38, 26, 64, 58, 40, 27, 63, 59, 41, 28, 63, 56, 41, 29, 62, 53,
        41, 30, 60, 48, 38, 28, 59, 43, 34, 26, 55, 41, 32, 24, 50, 38, 29, 23,
        46, 36, 27, 21, 41, 34, 25, 20, 38, 32, 24, 19, 34, 30, 22, 17, 31, 28,
        21, 16, 28, 26, 20, 15, 27, 25, 19, 15,
        /* Size 16x4 */
        65, 64, 63, 63, 62, 60, 59, 55, 50, 46, 41, 38, 34, 31, 28, 27, 57, 58,
        59, 56, 53, 48, 43, 41, 38, 36, 34, 32, 30, 28, 26, 25, 38, 40, 41, 41,
        41, 38, 34, 32, 29, 27, 25, 24, 22, 21, 20, 19, 26, 27, 28, 29, 30, 28,
        26, 24, 23, 21, 20, 19, 17, 16, 15, 15,
        /* Size 8x32 */
        64, 65, 64, 57, 47, 38, 31, 26, 65, 64, 63, 58, 48, 40, 33, 27, 65, 64,
        63, 58, 48, 40, 33, 27, 65, 63, 62, 59, 50, 41, 34, 28, 65, 63, 62, 59,
        50, 41, 34, 28, 64, 63, 60, 56, 49, 41, 35, 29, 64, 63, 60, 56, 49, 41,
        35, 29, 64, 62, 58, 53, 48, 41, 35, 30, 64, 62, 58, 53, 48, 41, 35, 30,
        60, 60, 55, 48, 43, 38, 33, 28, 60, 60, 55, 48, 43, 38, 33, 28, 57, 59,
        53, 43, 38, 34, 30, 26, 57, 59, 53, 43, 38, 34, 30, 26, 52, 55, 51, 41,
        35, 32, 28, 24, 52, 55, 51, 41, 35, 32, 28, 24, 47, 50, 48, 38, 33, 29,
        26, 23, 47, 50, 48, 38, 33, 29, 26, 23, 43, 46, 44, 36, 31, 27, 24, 21,
        43, 46, 44, 36, 31, 27, 24, 21, 38, 41, 41, 34, 29, 25, 22, 20, 38, 41,
        41, 34, 29, 25, 22, 20, 35, 38, 38, 32, 27, 24, 21, 19, 35, 38, 38, 32,
        27, 24, 21, 19, 32, 34, 35, 30, 26, 22, 20, 17, 32, 34, 35, 30, 26, 22,
        20, 17, 29, 31, 32, 28, 24, 21, 18, 16, 29, 31, 32, 28, 24, 21, 18, 16,
        26, 28, 29, 26, 23, 20, 17, 15, 26, 28, 29, 26, 23, 20, 17, 15, 25, 27,
        28, 25, 22, 19, 17, 15, 25, 27, 28, 25, 22, 19, 17, 15, 24, 26, 27, 24,
        21, 19, 16, 15,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 64, 64, 64, 64, 60, 60, 57, 57, 52, 52, 47, 47, 43,
        43, 38, 38, 35, 35, 32, 32, 29, 29, 26, 26, 25, 25, 24, 65, 64, 64, 63,
        63, 63, 63, 62, 62, 60, 60, 59, 59, 55, 55, 50, 50, 46, 46, 41, 41, 38,
        38, 34, 34, 31, 31, 28, 28, 27, 27, 26, 64, 63, 63, 62, 62, 60, 60, 58,
        58, 55, 55, 53, 53, 51, 51, 48, 48, 44, 44, 41, 41, 38, 38, 35, 35, 32,
        32, 29, 29, 28, 28, 27, 57, 58, 58, 59, 59, 56, 56, 53, 53, 48, 48, 43,
        43, 41, 41, 38, 38, 36, 36, 34, 34, 32, 32, 30, 30, 28, 28, 26, 26, 25,
        25, 24, 47, 48, 48, 50, 50, 49, 49, 48, 48, 43, 43, 38, 38, 35, 35, 33,
        33, 31, 31, 29, 29, 27, 27, 26, 26, 24, 24, 23, 23, 22, 22, 21, 38, 40,
        40, 41, 41, 41, 41, 41, 41, 38, 38, 34, 34, 32, 32, 29, 29, 27, 27, 25,
        25, 24, 24, 22, 22, 21, 21, 20, 20, 19, 19, 19, 31, 33, 33, 34, 34, 35,
        35, 35, 35, 33, 33, 30, 30, 28, 28, 26, 26, 24, 24, 22, 22, 21, 21, 20,
        20, 18, 18, 17, 17, 17, 17, 16, 26, 27, 27, 28, 28, 29, 29, 30, 30, 28,
        28, 26, 26, 24, 24, 23, 23, 21, 21, 20, 20, 19, 19, 17, 17, 16, 16, 15,
        15, 15, 15, 15 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        43, 41, 41, 27,
#endif
        /* Size 4x4 */
        64, 44, 43, 36, 44, 39, 38, 34, 43, 38, 31, 27, 36, 34, 27, 23,
        /* Size 8x8 */
        65, 59, 48, 44, 42, 39, 36, 33, 59, 53, 46, 44, 44, 41, 38, 36, 48, 46,
        42, 41, 41, 39, 37, 35, 44, 44, 41, 38, 36, 35, 33, 32, 42, 44, 41, 36,
        33, 32, 30, 29, 39, 41, 39, 35, 32, 29, 27, 26, 36, 38, 37, 33, 30, 27,
        25, 24, 33, 36, 35, 32, 29, 26, 24, 22,
        /* Size 16x16 */
        64, 66, 67, 61, 56, 49, 42, 42, 42, 41, 39, 38, 36, 34, 32, 32, 66, 65,
        66, 59, 53, 48, 43, 44, 44, 42, 41, 39, 38, 36, 34, 33, 67, 66, 64, 58,
        51, 48, 44, 45, 46, 44, 43, 41, 39, 38, 36, 35, 61, 59, 58, 53, 47, 46,
        43, 44, 45, 44, 43, 41, 40, 38, 36, 36, 56, 53, 51, 47, 43, 43, 43, 44,
        45, 45, 44, 42, 41, 39, 38, 37, 49, 48, 48, 46, 43, 42, 41, 41, 42, 41,
        41, 39, 38, 37, 36, 35, 42, 43, 44, 43, 43, 41, 39, 39, 38, 38, 38, 37,
        36, 35, 34, 33, 42, 44, 45, 44, 44, 41, 39, 38, 37, 36, 36, 35, 34, 33,
        32, 32, 42, 44, 46, 45, 45, 42, 38, 37, 35, 34, 33, 33, 32, 31, 30, 30,
        41, 42, 44, 44, 45, 41, 38, 36, 34, 33, 32, 32, 31, 30, 29, 29, 39, 41,
        43, 43, 44, 41, 38, 36, 33, 32, 31, 30, 29, 28, 27, 27, 38, 39, 41, 41,
        42, 39, 37, 35, 33, 32, 30, 29, 28, 27, 26, 26, 36, 38, 39, 40, 41, 38,
        36, 34, 32, 31, 29, 28, 27, 26, 25, 25, 34, 36, 38, 38, 39, 37, 35, 33,
        31, 30, 28, 27, 26, 25, 24, 24, 32, 34, 36, 36, 38, 36, 34, 32, 30, 29,
        27, 26, 25, 24, 23, 23, 32, 33, 35, 36, 37, 35, 33, 32, 30, 29, 27, 26,
        25, 24, 23, 22,
        /* Size 32x32 */
        64, 66, 66, 67, 67, 61, 61, 56, 56, 49, 49, 42, 42, 42, 42, 42, 42, 41,
        41, 39, 39, 38, 38, 36, 36, 34, 34, 32, 32, 32, 32, 31, 66, 65, 65, 66,
        66, 59, 59, 53, 53, 48, 48, 43, 43, 44, 44, 44, 44, 42, 42, 41, 41, 39,
        39, 38, 38, 36, 36, 34, 34, 33, 33, 32, 66, 65, 65, 66, 66, 59, 59, 53,
        53, 48, 48, 43, 43, 44, 44, 44, 44, 42, 42, 41, 41, 39, 39, 38, 38, 36,
        36, 34, 34, 33, 33, 32, 67, 66, 66, 64, 64, 58, 58, 51, 51, 48, 48, 44,
        44, 45, 45, 46, 46, 44, 44, 43, 43, 41, 41, 39, 39, 38, 38, 36, 36, 35,
        35, 34, 67, 66, 66, 64, 64, 58, 58, 51, 51, 48, 48, 44, 44, 45, 45, 46,
        46, 44, 44, 43, 43, 41, 41, 39, 39, 38, 38, 36, 36, 35, 35, 34, 61, 59,
        59, 58, 58, 53, 53, 47, 47, 46, 46, 43, 43, 44, 44, 45, 45, 44, 44, 43,
        43, 41, 41, 40, 40, 38, 38, 36, 36, 36, 36, 35, 61, 59, 59, 58, 58, 53,
        53, 47, 47, 46, 46, 43, 43, 44, 44, 45, 45, 44, 44, 43, 43, 41, 41, 40,
        40, 38, 38, 36, 36, 36, 36, 35, 56, 53, 53, 51, 51, 47, 47, 43, 43, 43,
        43, 43, 43, 44, 44, 45, 45, 45, 45, 44, 44, 42, 42, 41, 41, 39, 39, 38,
        38, 37, 37, 36, 56, 53, 53, 51, 51, 47, 47, 43, 43, 43, 43, 43, 43, 44,
        44, 45, 45, 45, 45, 44, 44, 42, 42, 41, 41, 39, 39, 38, 38, 37, 37, 36,
        49, 48, 48, 48, 48, 46, 46, 43, 43, 42, 42, 41, 41, 41, 41, 42, 42, 41,
        41, 41, 41, 39, 39, 38, 38, 37, 37, 36, 36, 35, 35, 34, 49, 48, 48, 48,
        48, 46, 46, 43, 43, 42, 42, 41, 41, 41, 41, 42, 42, 41, 41, 41, 41, 39,
        39, 38, 38, 37, 37, 36, 36, 35, 35, 34, 42, 43, 43, 44, 44, 43, 43, 43,
        43, 41, 41, 39, 39, 39, 39, 38, 38, 38, 38, 38, 38, 37, 37, 36, 36, 35,
        35, 34, 34, 33, 33, 33, 42, 43, 43, 44, 44, 43, 43, 43, 43, 41, 41, 39,
        39, 39, 39, 38, 38, 38, 38, 38, 38, 37, 37, 36, 36, 35, 35, 34, 34, 33,
        33, 33, 42, 44, 44, 45, 45, 44, 44, 44, 44, 41, 41, 39, 39, 38, 38, 37,
        37, 36, 36, 36, 36, 35, 35, 34, 34, 33, 33, 32, 32, 32, 32, 31, 42, 44,
        44, 45, 45, 44, 44, 44, 44, 41, 41, 39, 39, 38, 38, 37, 37, 36, 36, 36,
        36, 35, 35, 34, 34, 33, 33, 32, 32, 32, 32, 31, 42, 44, 44, 46, 46, 45,
        45, 45, 45, 42, 42, 38, 38, 37, 37, 35, 35, 34, 34, 33, 33, 33, 33, 32,
        32, 31, 31, 30, 30, 30, 30, 30, 42, 44, 44, 46, 46, 45, 45, 45, 45, 42,
        42, 38, 38, 37, 37, 35, 35, 34, 34, 33, 33, 33, 33, 32, 32, 31, 31, 30,
        30, 30, 30, 30, 41, 42, 42, 44, 44, 44, 44, 45, 45, 41, 41, 38, 38, 36,
        36, 34, 34, 33, 33, 32, 32, 32, 32, 31, 31, 30, 30, 29, 29, 29, 29, 28,
        41, 42, 42, 44, 44, 44, 44, 45, 45, 41, 41, 38, 38, 36, 36, 34, 34, 33,
        33, 32, 32, 32, 32, 31, 31, 30, 30, 29, 29, 29, 29, 28, 39, 41, 41, 43,
        43, 43, 43, 44, 44, 41, 41, 38, 38, 36, 36, 33, 33, 32, 32, 31, 31, 30,
        30, 29, 29, 28, 28, 27, 27, 27, 27, 27, 39, 41, 41, 43, 43, 43, 43, 44,
        44, 41, 41, 38, 38, 36, 36, 33, 33, 32, 32, 31, 31, 30, 30, 29, 29, 28,
        28, 27, 27, 27, 27, 27, 38, 39, 39, 41, 41, 41, 41, 42, 42, 39, 39, 37,
        37, 35, 35, 33, 33, 32, 32, 30, 30, 29, 29, 28, 28, 27, 27, 26, 26, 26,
        26, 25, 38, 39, 39, 41, 41, 41, 41, 42, 42, 39, 39, 37, 37, 35, 35, 33,
        33, 32, 32, 30, 30, 29, 29, 28, 28, 27, 27, 26, 26, 26, 26, 25, 36, 38,
        38, 39, 39, 40, 40, 41, 41, 38, 38, 36, 36, 34, 34, 32, 32, 31, 31, 29,
        29, 28, 28, 27, 27, 26, 26, 25, 25, 25, 25, 24, 36, 38, 38, 39, 39, 40,
        40, 41, 41, 38, 38, 36, 36, 34, 34, 32, 32, 31, 31, 29, 29, 28, 28, 27,
        27, 26, 26, 25, 25, 25, 25, 24, 34, 36, 36, 38, 38, 38, 38, 39, 39, 37,
        37, 35, 35, 33, 33, 31, 31, 30, 30, 28, 28, 27, 27, 26, 26, 25, 25, 24,
        24, 24, 24, 23, 34, 36, 36, 38, 38, 38, 38, 39, 39, 37, 37, 35, 35, 33,
        33, 31, 31, 30, 30, 28, 28, 27, 27, 26, 26, 25, 25, 24, 24, 24, 24, 23,
        32, 34, 34, 36, 36, 36, 36, 38, 38, 36, 36, 34, 34, 32, 32, 30, 30, 29,
        29, 27, 27, 26, 26, 25, 25, 24, 24, 23, 23, 23, 23, 22, 32, 34, 34, 36,
        36, 36, 36, 38, 38, 36, 36, 34, 34, 32, 32, 30, 30, 29, 29, 27, 27, 26,
        26, 25, 25, 24, 24, 23, 23, 23, 23, 22, 32, 33, 33, 35, 35, 36, 36, 37,
        37, 35, 35, 33, 33, 32, 32, 30, 30, 29, 29, 27, 27, 26, 26, 25, 25, 24,
        24, 23, 23, 22, 22, 22, 32, 33, 33, 35, 35, 36, 36, 37, 37, 35, 35, 33,
        33, 32, 32, 30, 30, 29, 29, 27, 27, 26, 26, 25, 25, 24, 24, 23, 23, 22,
        22, 22, 31, 32, 32, 34, 34, 35, 35, 36, 36, 34, 34, 33, 33, 31, 31, 30,
        30, 28, 28, 27, 27, 25, 25, 24, 24, 23, 23, 22, 22, 22, 22, 21,
        /* Size 4x8 */
        65, 44, 41, 34, 57, 44, 43, 37, 48, 41, 41, 36, 45, 39, 36, 32, 44, 38,
        32, 29, 41, 37, 30, 26, 38, 35, 28, 24, 35, 33, 27, 23,
        /* Size 8x4 */
        65, 57, 48, 45, 44, 41, 38, 35, 44, 44, 41, 39, 38, 37, 35, 33, 41, 43,
        41, 36, 32, 30, 28, 27, 34, 37, 36, 32, 29, 26, 24, 23,
        /* Size 8x16 */
        64, 66, 55, 43, 42, 39, 36, 32, 65, 65, 53, 44, 44, 41, 38, 34, 67, 64,
        51, 45, 45, 43, 39, 36, 61, 57, 48, 44, 45, 43, 40, 37, 55, 51, 44, 43,
        45, 44, 41, 38, 49, 48, 43, 41, 41, 41, 38, 36, 42, 44, 43, 39, 38, 38,
        36, 34, 42, 45, 44, 39, 37, 36, 34, 32, 42, 45, 45, 38, 35, 33, 32, 30,
        41, 44, 44, 38, 34, 32, 31, 29, 39, 43, 43, 38, 34, 31, 29, 27, 38, 41,
        42, 37, 33, 30, 28, 26, 36, 39, 41, 36, 32, 29, 27, 25, 34, 38, 39, 35,
        31, 28, 26, 24, 32, 36, 37, 34, 30, 27, 25, 23, 32, 35, 37, 33, 30, 27,
        25, 23,
        /* Size 16x8 */
        64, 65, 67, 61, 55, 49, 42, 42, 42, 41, 39, 38, 36, 34, 32, 32, 66, 65,
        64, 57, 51, 48, 44, 45, 45, 44, 43, 41, 39, 38, 36, 35, 55, 53, 51, 48,
        44, 43, 43, 44, 45, 44, 43, 42, 41, 39, 37, 37, 43, 44, 45, 44, 43, 41,
        39, 39, 38, 38, 38, 37, 36, 35, 34, 33, 42, 44, 45, 45, 45, 41, 38, 37,
        35, 34, 34, 33, 32, 31, 30, 30, 39, 41, 43, 43, 44, 41, 38, 36, 33, 32,
        31, 30, 29, 28, 27, 27, 36, 38, 39, 40, 41, 38, 36, 34, 32, 31, 29, 28,
        27, 26, 25, 25, 32, 34, 36, 37, 38, 36, 34, 32, 30, 29, 27, 26, 25, 24,
        23, 23,
        /* Size 16x32 */
        64, 66, 66, 55, 55, 43, 43, 42, 42, 39, 39, 36, 36, 32, 32, 31, 65, 65,
        65, 53, 53, 44, 44, 44, 44, 41, 41, 38, 38, 34, 34, 33, 65, 65, 65, 53,
        53, 44, 44, 44, 44, 41, 41, 38, 38, 34, 34, 33, 67, 64, 64, 51, 51, 45,
        45, 45, 45, 43, 43, 39, 39, 36, 36, 34, 67, 64, 64, 51, 51, 45, 45, 45,
        45, 43, 43, 39, 39, 36, 36, 34, 61, 57, 57, 48, 48, 44, 44, 45, 45, 43,
        43, 40, 40, 37, 37, 35, 61, 57, 57, 48, 48, 44, 44, 45, 45, 43, 43, 40,
        40, 37, 37, 35, 55, 51, 51, 44, 44, 43, 43, 45, 45, 44, 44, 41, 41, 38,
        38, 36, 55, 51, 51, 44, 44, 43, 43, 45, 45, 44, 44, 41, 41, 38, 38, 36,
        49, 48, 48, 43, 43, 41, 41, 41, 41, 41, 41, 38, 38, 36, 36, 34, 49, 48,
        48, 43, 43, 41, 41, 41, 41, 41, 41, 38, 38, 36, 36, 34, 42, 44, 44, 43,
        43, 39, 39, 38, 38, 38, 38, 36, 36, 34, 34, 33, 42, 44, 44, 43, 43, 39,
        39, 38, 38, 38, 38, 36, 36, 34, 34, 33, 42, 45, 45, 44, 44, 39, 39, 37,
        37, 36, 36, 34, 34, 32, 32, 31, 42, 45, 45, 44, 44, 39, 39, 37, 37, 36,
        36, 34, 34, 32, 32, 31, 42, 45, 45, 45, 45, 38, 38, 35, 35, 33, 33, 32,
        32, 30, 30, 29, 42, 45, 45, 45, 45, 38, 38, 35, 35, 33, 33, 32, 32, 30,
        30, 29, 41, 44, 44, 44, 44, 38, 38, 34, 34, 32, 32, 31, 31, 29, 29, 28,
        41, 44, 44, 44, 44, 38, 38, 34, 34, 32, 32, 31, 31, 29, 29, 28, 39, 43,
        43, 43, 43, 38, 38, 34, 34, 31, 31, 29, 29, 27, 27, 27, 39, 43, 43, 43,
        43, 38, 38, 34, 34, 31, 31, 29, 29, 27, 27, 27, 38, 41, 41, 42, 42, 37,
        37, 33, 33, 30, 30, 28, 28, 26, 26, 25, 38, 41, 41, 42, 42, 37, 37, 33,
        33, 30, 30, 28, 28, 26, 26, 25, 36, 39, 39, 41, 41, 36, 36, 32, 32, 29,
        29, 27, 27, 25, 25, 24, 36, 39, 39, 41, 41, 36, 36, 32, 32, 29, 29, 27,
        27, 25, 25, 24, 34, 38, 38, 39, 39, 35, 35, 31, 31, 28, 28, 26, 26, 24,
        24, 23, 34, 38, 38, 39, 39, 35, 35, 31, 31, 28, 28, 26, 26, 24, 24, 23,
        32, 36, 36, 37, 37, 34, 34, 30, 30, 27, 27, 25, 25, 23, 23, 22, 32, 36,
        36, 37, 37, 34, 34, 30, 30, 27, 27, 25, 25, 23, 23, 22, 32, 35, 35, 37,
        37, 33, 33, 30, 30, 27, 27, 25, 25, 23, 23, 22, 32, 35, 35, 37, 37, 33,
        33, 30, 30, 27, 27, 25, 25, 23, 23, 22, 31, 34, 34, 36, 36, 33, 33, 29,
        29, 27, 27, 24, 24, 22, 22, 21,
        /* Size 32x16 */
        64, 65, 65, 67, 67, 61, 61, 55, 55, 49, 49, 42, 42, 42, 42, 42, 42, 41,
        41, 39, 39, 38, 38, 36, 36, 34, 34, 32, 32, 32, 32, 31, 66, 65, 65, 64,
        64, 57, 57, 51, 51, 48, 48, 44, 44, 45, 45, 45, 45, 44, 44, 43, 43, 41,
        41, 39, 39, 38, 38, 36, 36, 35, 35, 34, 66, 65, 65, 64, 64, 57, 57, 51,
        51, 48, 48, 44, 44, 45, 45, 45, 45, 44, 44, 43, 43, 41, 41, 39, 39, 38,
        38, 36, 36, 35, 35, 34, 55, 53, 53, 51, 51, 48, 48, 44, 44, 43, 43, 43,
        43, 44, 44, 45, 45, 44, 44, 43, 43, 42, 42, 41, 41, 39, 39, 37, 37, 37,
        37, 36, 55, 53, 53, 51, 51, 48, 48, 44, 44, 43, 43, 43, 43, 44, 44, 45,
        45, 44, 44, 43, 43, 42, 42, 41, 41, 39, 39, 37, 37, 37, 37, 36, 43, 44,
        44, 45, 45, 44, 44, 43, 43, 41, 41, 39, 39, 39, 39, 38, 38, 38, 38, 38,
        38, 37, 37, 36, 36, 35, 35, 34, 34, 33, 33, 33, 43, 44, 44, 45, 45, 44,
        44, 43, 43, 41, 41, 39, 39, 39, 39, 38, 38, 38, 38, 38, 38, 37, 37, 36,
        36, 35, 35, 34, 34, 33, 33, 33, 42, 44, 44, 45, 45, 45, 45, 45, 45, 41,
        41, 38, 38, 37, 37, 35, 35, 34, 34, 34, 34, 33, 33, 32, 32, 31, 31, 30,
        30, 30, 30, 29, 42, 44, 44, 45, 45, 45, 45, 45, 45, 41, 41, 38, 38, 37,
        37, 35, 35, 34, 34, 34, 34, 33, 33, 32, 32, 31, 31, 30, 30, 30, 30, 29,
        39, 41, 41, 43, 43, 43, 43, 44, 44, 41, 41, 38, 38, 36, 36, 33, 33, 32,
        32, 31, 31, 30, 30, 29, 29, 28, 28, 27, 27, 27, 27, 27, 39, 41, 41, 43,
        43, 43, 43, 44, 44, 41, 41, 38, 38, 36, 36, 33, 33, 32, 32, 31, 31, 30,
        30, 29, 29, 28, 28, 27, 27, 27, 27, 27, 36, 38, 38, 39, 39, 40, 40, 41,
        41, 38, 38, 36, 36, 34, 34, 32, 32, 31, 31, 29, 29, 28, 28, 27, 27, 26,
        26, 25, 25, 25, 25, 24, 36, 38, 38, 39, 39, 40, 40, 41, 41, 38, 38, 36,
        36, 34, 34, 32, 32, 31, 31, 29, 29, 28, 28, 27, 27, 26, 26, 25, 25, 25,
        25, 24, 32, 34, 34, 36, 36, 37, 37, 38, 38, 36, 36, 34, 34, 32, 32, 30,
        30, 29, 29, 27, 27, 26, 26, 25, 25, 24, 24, 23, 23, 23, 23, 22, 32, 34,
        34, 36, 36, 37, 37, 38, 38, 36, 36, 34, 34, 32, 32, 30, 30, 29, 29, 27,
        27, 26, 26, 25, 25, 24, 24, 23, 23, 23, 23, 22, 31, 33, 33, 34, 34, 35,
        35, 36, 36, 34, 34, 33, 33, 31, 31, 29, 29, 28, 28, 27, 27, 25, 25, 24,
        24, 23, 23, 22, 22, 22, 22, 21,
        /* Size 4x16 */
        66, 43, 39, 32, 65, 44, 41, 34, 64, 45, 43, 36, 57, 44, 43, 37, 51, 43,
        44, 38, 48, 41, 41, 36, 44, 39, 38, 34, 45, 39, 36, 32, 45, 38, 33, 30,
        44, 38, 32, 29, 43, 38, 31, 27, 41, 37, 30, 26, 39, 36, 29, 25, 38, 35,
        28, 24, 36, 34, 27, 23, 35, 33, 27, 23,
        /* Size 16x4 */
        66, 65, 64, 57, 51, 48, 44, 45, 45, 44, 43, 41, 39, 38, 36, 35, 43, 44,
        45, 44, 43, 41, 39, 39, 38, 38, 38, 37, 36, 35, 34, 33, 39, 41, 43, 43,
        44, 41, 38, 36, 33, 32, 31, 30, 29, 28, 27, 27, 32, 34, 36, 37, 38, 36,
        34, 32, 30, 29, 27, 26, 25, 24, 23, 23,
        /* Size 8x32 */
        64, 66, 55, 43, 42, 39, 36, 32, 65, 65, 53, 44, 44, 41, 38, 34, 65, 65,
        53, 44, 44, 41, 38, 34, 67, 64, 51, 45, 45, 43, 39, 36, 67, 64, 51, 45,
        45, 43, 39, 36, 61, 57, 48, 44, 45, 43, 40, 37, 61, 57, 48, 44, 45, 43,
        40, 37, 55, 51, 44, 43, 45, 44, 41, 38, 55, 51, 44, 43, 45, 44, 41, 38,
        49, 48, 43, 41, 41, 41, 38, 36, 49, 48, 43, 41, 41, 41, 38, 36, 42, 44,
        43, 39, 38, 38, 36, 34, 42, 44, 43, 39, 38, 38, 36, 34, 42, 45, 44, 39,
        37, 36, 34, 32, 42, 45, 44, 39, 37, 36, 34, 32, 42, 45, 45, 38, 35, 33,
        32, 30, 42, 45, 45, 38, 35, 33, 32, 30, 41, 44, 44, 38, 34, 32, 31, 29,
        41, 44, 44, 38, 34, 32, 31, 29, 39, 43, 43, 38, 34, 31, 29, 27, 39, 43,
        43, 38, 34, 31, 29, 27, 38, 41, 42, 37, 33, 30, 28, 26, 38, 41, 42, 37,
        33, 30, 28, 26, 36, 39, 41, 36, 32, 29, 27, 25, 36, 39, 41, 36, 32, 29,
        27, 25, 34, 38, 39, 35, 31, 28, 26, 24, 34, 38, 39, 35, 31, 28, 26, 24,
        32, 36, 37, 34, 30, 27, 25, 23, 32, 36, 37, 34, 30, 27, 25, 23, 32, 35,
        37, 33, 30, 27, 25, 23, 32, 35, 37, 33, 30, 27, 25, 23, 31, 34, 36, 33,
        29, 27, 24, 22,
        /* Size 32x8 */
        64, 65, 65, 67, 67, 61, 61, 55, 55, 49, 49, 42, 42, 42, 42, 42, 42, 41,
        41, 39, 39, 38, 38, 36, 36, 34, 34, 32, 32, 32, 32, 31, 66, 65, 65, 64,
        64, 57, 57, 51, 51, 48, 48, 44, 44, 45, 45, 45, 45, 44, 44, 43, 43, 41,
        41, 39, 39, 38, 38, 36, 36, 35, 35, 34, 55, 53, 53, 51, 51, 48, 48, 44,
        44, 43, 43, 43, 43, 44, 44, 45, 45, 44, 44, 43, 43, 42, 42, 41, 41, 39,
        39, 37, 37, 37, 37, 36, 43, 44, 44, 45, 45, 44, 44, 43, 43, 41, 41, 39,
        39, 39, 39, 38, 38, 38, 38, 38, 38, 37, 37, 36, 36, 35, 35, 34, 34, 33,
        33, 33, 42, 44, 44, 45, 45, 45, 45, 45, 45, 41, 41, 38, 38, 37, 37, 35,
        35, 34, 34, 34, 34, 33, 33, 32, 32, 31, 31, 30, 30, 30, 30, 29, 39, 41,
        41, 43, 43, 43, 43, 44, 44, 41, 41, 38, 38, 36, 36, 33, 33, 32, 32, 31,
        31, 30, 30, 29, 29, 28, 28, 27, 27, 27, 27, 27, 36, 38, 38, 39, 39, 40,
        40, 41, 41, 38, 38, 36, 36, 34, 34, 32, 32, 31, 31, 29, 29, 28, 28, 27,
        27, 26, 26, 25, 25, 25, 25, 24, 32, 34, 34, 36, 36, 37, 37, 38, 38, 36,
        36, 34, 34, 32, 32, 30, 30, 29, 29, 27, 27, 26, 26, 25, 25, 24, 24, 23,
        23, 23, 23, 22 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        60, 39, 39, 23,
#endif
        /* Size 4x4 */
        64, 61, 46, 33, 61, 52, 40, 32, 46, 40, 29, 24, 33, 32, 24, 19,
        /* Size 8x8 */
        65, 64, 63, 58, 48, 40, 34, 30, 64, 63, 61, 58, 49, 42, 36, 31, 63, 61,
        58, 53, 48, 41, 37, 32, 58, 58, 53, 42, 38, 34, 31, 28, 48, 49, 48, 38,
        32, 29, 26, 24, 40, 42, 41, 34, 29, 25, 23, 21, 34, 36, 37, 31, 26, 23,
        21, 19, 30, 31, 32, 28, 24, 21, 19, 17,
        /* Size 16x16 */
        64, 65, 65, 65, 64, 60, 58, 54, 50, 46, 42, 38, 35, 31, 29, 26, 65, 64,
        64, 64, 63, 60, 59, 55, 51, 48, 44, 40, 36, 33, 30, 27, 65, 64, 63, 63,
        63, 61, 60, 57, 53, 49, 46, 41, 38, 34, 32, 28, 65, 64, 63, 61, 60, 58,
        57, 54, 51, 48, 45, 41, 38, 35, 32, 29, 64, 63, 63, 60, 58, 55, 54, 52,
        49, 47, 45, 41, 38, 35, 33, 29, 60, 60, 61, 58, 55, 52, 49, 47, 45, 42,
        40, 38, 35, 33, 30, 28, 58, 59, 60, 57, 54, 49, 45, 42, 41, 39, 37, 35,
        33, 31, 28, 26, 54, 55, 57, 54, 52, 47, 42, 40, 38, 36, 34, 33, 31, 29,
        27, 25, 50, 51, 53, 51, 49, 45, 41, 38, 36, 34, 32, 30, 29, 27, 25, 24,
        46, 48, 49, 48, 47, 42, 39, 36, 34, 32, 30, 28, 27, 25, 24, 22, 42, 44,
        46, 45, 45, 40, 37, 34, 32, 30, 29, 27, 25, 24, 23, 21, 38, 40, 41, 41,
        41, 38, 35, 33, 30, 28, 27, 25, 24, 22, 21, 20, 35, 36, 38, 38, 38, 35,
        33, 31, 29, 27, 25, 24, 22, 21, 20, 19, 31, 33, 34, 35, 35, 33, 31, 29,
        27, 25, 24, 22, 21, 20, 18, 17, 29, 30, 32, 32, 33, 30, 28, 27, 25, 24,
        23, 21, 20, 18, 17, 16, 26, 27, 28, 29, 29, 28, 26, 25, 24, 22, 21, 20,
        19, 17, 16, 15,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 65, 65, 64, 64, 64, 60, 60, 58, 57, 54, 52, 50, 47,
        46, 42, 42, 38, 38, 36, 35, 33, 31, 31, 29, 28, 26, 26, 65, 65, 64, 64,
        64, 64, 64, 64, 63, 63, 60, 60, 59, 58, 55, 53, 51, 48, 47, 44, 44, 40,
        39, 37, 36, 34, 33, 32, 30, 30, 27, 27, 65, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 60, 60, 59, 59, 55, 53, 51, 48, 48, 44, 44, 40, 40, 37, 36, 35,
        33, 32, 30, 30, 27, 27, 65, 64, 64, 64, 64, 63, 63, 63, 63, 63, 61, 61,
        59, 59, 56, 54, 53, 49, 49, 46, 46, 41, 41, 39, 38, 36, 34, 33, 31, 31,
        28, 28, 65, 64, 64, 64, 63, 63, 63, 63, 63, 62, 61, 61, 60, 59, 57, 55,
        53, 50, 49, 46, 46, 42, 41, 39, 38, 36, 34, 34, 32, 31, 28, 28, 65, 64,
        64, 63, 63, 63, 62, 62, 61, 61, 59, 59, 59, 58, 55, 53, 52, 49, 49, 46,
        46, 42, 41, 39, 38, 36, 34, 34, 32, 31, 28, 28, 65, 64, 64, 63, 63, 62,
        61, 61, 60, 60, 58, 58, 57, 56, 54, 53, 51, 49, 48, 45, 45, 41, 41, 39,
        38, 36, 35, 34, 32, 32, 29, 29, 64, 64, 64, 63, 63, 62, 61, 60, 59, 59,
        58, 58, 56, 55, 53, 52, 51, 49, 48, 45, 45, 41, 41, 39, 38, 36, 35, 34,
        32, 32, 29, 29, 64, 63, 63, 63, 63, 61, 60, 59, 58, 58, 55, 55, 54, 53,
        52, 51, 49, 48, 47, 45, 45, 42, 41, 39, 38, 37, 35, 34, 33, 32, 29, 29,
        64, 63, 63, 63, 62, 61, 60, 59, 58, 58, 55, 55, 53, 53, 51, 50, 49, 48,
        47, 44, 44, 41, 41, 39, 38, 37, 35, 34, 32, 32, 29, 29, 60, 60, 60, 61,
        61, 59, 58, 58, 55, 55, 52, 52, 49, 48, 47, 46, 45, 43, 42, 40, 40, 38,
        38, 36, 35, 34, 33, 32, 30, 30, 28, 28, 60, 60, 60, 61, 61, 59, 58, 58,
        55, 55, 52, 52, 49, 48, 47, 46, 45, 43, 42, 40, 40, 38, 38, 36, 35, 34,
        33, 32, 30, 30, 28, 28, 58, 59, 59, 59, 60, 59, 57, 56, 54, 53, 49, 49,
        45, 44, 42, 41, 41, 39, 39, 37, 37, 35, 35, 33, 33, 32, 31, 30, 28, 28,
        26, 26, 57, 58, 59, 59, 59, 58, 56, 55, 53, 53, 48, 48, 44, 42, 41, 41,
        39, 38, 38, 36, 36, 34, 34, 33, 32, 31, 30, 29, 28, 28, 26, 26, 54, 55,
        55, 56, 57, 55, 54, 53, 52, 51, 47, 47, 42, 41, 40, 39, 38, 37, 36, 34,
        34, 33, 33, 31, 31, 30, 29, 28, 27, 27, 25, 25, 52, 53, 53, 54, 55, 53,
        53, 52, 51, 50, 46, 46, 41, 41, 39, 38, 37, 35, 35, 34, 34, 32, 32, 30,
        30, 29, 28, 27, 26, 26, 24, 24, 50, 51, 51, 53, 53, 52, 51, 51, 49, 49,
        45, 45, 41, 39, 38, 37, 36, 34, 34, 32, 32, 31, 30, 29, 29, 28, 27, 27,
        25, 25, 24, 24, 47, 48, 48, 49, 50, 49, 49, 49, 48, 48, 43, 43, 39, 38,
        37, 35, 34, 32, 32, 31, 31, 29, 29, 28, 27, 26, 26, 25, 24, 24, 23, 23,
        46, 47, 48, 49, 49, 49, 48, 48, 47, 47, 42, 42, 39, 38, 36, 35, 34, 32,
        32, 30, 30, 28, 28, 27, 27, 26, 25, 25, 24, 24, 22, 22, 42, 44, 44, 46,
        46, 46, 45, 45, 45, 44, 40, 40, 37, 36, 34, 34, 32, 31, 30, 29, 29, 27,
        27, 26, 25, 25, 24, 24, 23, 23, 21, 21, 42, 44, 44, 46, 46, 46, 45, 45,
        45, 44, 40, 40, 37, 36, 34, 34, 32, 31, 30, 29, 29, 27, 27, 26, 25, 25,
        24, 24, 23, 23, 21, 21, 38, 40, 40, 41, 42, 42, 41, 41, 42, 41, 38, 38,
        35, 34, 33, 32, 31, 29, 28, 27, 27, 25, 25, 24, 24, 23, 22, 22, 21, 21,
        20, 20, 38, 39, 40, 41, 41, 41, 41, 41, 41, 41, 38, 38, 35, 34, 33, 32,
        30, 29, 28, 27, 27, 25, 25, 24, 24, 23, 22, 22, 21, 21, 20, 20, 36, 37,
        37, 39, 39, 39, 39, 39, 39, 39, 36, 36, 33, 33, 31, 30, 29, 28, 27, 26,
        26, 24, 24, 23, 23, 22, 21, 21, 20, 20, 19, 19, 35, 36, 36, 38, 38, 38,
        38, 38, 38, 38, 35, 35, 33, 32, 31, 30, 29, 27, 27, 25, 25, 24, 24, 23,
        22, 22, 21, 21, 20, 20, 19, 19, 33, 34, 35, 36, 36, 36, 36, 36, 37, 37,
        34, 34, 32, 31, 30, 29, 28, 26, 26, 25, 25, 23, 23, 22, 22, 21, 20, 20,
        19, 19, 18, 18, 31, 33, 33, 34, 34, 34, 35, 35, 35, 35, 33, 33, 31, 30,
        29, 28, 27, 26, 25, 24, 24, 22, 22, 21, 21, 20, 20, 19, 18, 18, 17, 17,
        31, 32, 32, 33, 34, 34, 34, 34, 34, 34, 32, 32, 30, 29, 28, 27, 27, 25,
        25, 24, 24, 22, 22, 21, 21, 20, 19, 19, 18, 18, 17, 17, 29, 30, 30, 31,
        32, 32, 32, 32, 33, 32, 30, 30, 28, 28, 27, 26, 25, 24, 24, 23, 23, 21,
        21, 20, 20, 19, 18, 18, 17, 17, 16, 16, 28, 30, 30, 31, 31, 31, 32, 32,
        32, 32, 30, 30, 28, 28, 27, 26, 25, 24, 24, 23, 23, 21, 21, 20, 20, 19,
        18, 18, 17, 17, 16, 16, 26, 27, 27, 28, 28, 28, 29, 29, 29, 29, 28, 28,
        26, 26, 25, 24, 24, 23, 22, 21, 21, 20, 20, 19, 19, 18, 17, 17, 16, 16,
        15, 15, 26, 27, 27, 28, 28, 28, 29, 29, 29, 29, 28, 28, 26, 26, 25, 24,
        24, 23, 22, 21, 21, 20, 20, 19, 19, 18, 17, 17, 16, 16, 15, 15,
        /* Size 4x8 */
        64, 60, 47, 33, 63, 59, 49, 34, 62, 55, 47, 35, 59, 48, 38, 30, 49, 43,
        32, 26, 41, 38, 29, 22, 36, 34, 26, 20, 31, 30, 24, 18,
        /* Size 8x4 */
        64, 63, 62, 59, 49, 41, 36, 31, 60, 59, 55, 48, 43, 38, 34, 30, 47, 49,
        47, 38, 32, 29, 26, 24, 33, 34, 35, 30, 26, 22, 20, 18,
        /* Size 8x16 */
        64, 65, 64, 57, 47, 38, 33, 28, 65, 64, 63, 58, 48, 40, 34, 29, 65, 63,
        62, 59, 50, 41, 36, 31, 64, 63, 60, 56, 49, 41, 36, 31, 64, 62, 58, 53,
        48, 41, 37, 32, 60, 60, 55, 48, 43, 38, 34, 30, 58, 59, 54, 44, 39, 35,
        32, 28, 54, 56, 51, 41, 37, 33, 29, 27, 50, 53, 49, 40, 34, 30, 28, 25,
        46, 49, 47, 38, 32, 28, 26, 24, 43, 46, 44, 36, 31, 27, 25, 22, 38, 41,
        41, 34, 29, 25, 23, 21, 35, 38, 38, 32, 27, 24, 21, 20, 32, 34, 35, 30,
        26, 22, 20, 18, 29, 31, 32, 28, 24, 21, 19, 17, 26, 28, 29, 26, 23, 20,
        18, 16,
        /* Size 16x8 */
        64, 65, 65, 64, 64, 60, 58, 54, 50, 46, 43, 38, 35, 32, 29, 26, 65, 64,
        63, 63, 62, 60, 59, 56, 53, 49, 46, 41, 38, 34, 31, 28, 64, 63, 62, 60,
        58, 55, 54, 51, 49, 47, 44, 41, 38, 35, 32, 29, 57, 58, 59, 56, 53, 48,
        44, 41, 40, 38, 36, 34, 32, 30, 28, 26, 47, 48, 50, 49, 48, 43, 39, 37,
        34, 32, 31, 29, 27, 26, 24, 23, 38, 40, 41, 41, 41, 38, 35, 33, 30, 28,
        27, 25, 24, 22, 21, 20, 33, 34, 36, 36, 37, 34, 32, 29, 28, 26, 25, 23,
        21, 20, 19, 18, 28, 29, 31, 31, 32, 30, 28, 27, 25, 24, 22, 21, 20, 18,
        17, 16,
        /* Size 16x32 */
        64, 65, 65, 64, 64, 60, 57, 54, 47, 46, 38, 38, 33, 31, 28, 26, 65, 64,
        64, 63, 63, 60, 58, 55, 48, 47, 40, 40, 34, 33, 29, 27, 65, 64, 64, 63,
        63, 60, 58, 55, 48, 48, 40, 40, 34, 33, 29, 27, 65, 64, 64, 63, 63, 61,
        59, 56, 49, 49, 41, 41, 35, 34, 31, 28, 65, 63, 63, 63, 62, 61, 59, 56,
        50, 49, 41, 41, 36, 34, 31, 28, 65, 63, 63, 62, 61, 59, 58, 55, 49, 49,
        41, 41, 36, 34, 31, 29, 64, 63, 63, 61, 60, 58, 56, 54, 49, 48, 41, 41,
        36, 35, 31, 29, 64, 63, 63, 60, 59, 58, 55, 53, 48, 48, 41, 41, 36, 35,
        32, 29, 64, 63, 62, 59, 58, 55, 53, 52, 48, 47, 41, 41, 37, 35, 32, 30,
        64, 62, 62, 59, 58, 55, 53, 51, 48, 47, 41, 41, 36, 35, 32, 29, 60, 60,
        60, 57, 55, 52, 48, 47, 43, 42, 38, 38, 34, 33, 30, 28, 60, 60, 60, 57,
        55, 52, 48, 47, 43, 42, 38, 38, 34, 33, 30, 28, 58, 59, 59, 55, 54, 49,
        44, 42, 39, 39, 35, 35, 32, 31, 28, 26, 57, 59, 59, 55, 53, 48, 43, 41,
        38, 38, 34, 34, 31, 30, 28, 26, 54, 56, 56, 53, 51, 47, 41, 40, 37, 36,
        33, 33, 29, 29, 27, 25, 52, 54, 55, 51, 51, 46, 41, 39, 35, 35, 32, 32,
        29, 28, 26, 24, 50, 53, 53, 50, 49, 45, 40, 38, 34, 34, 30, 30, 28, 27,
        25, 24, 47, 49, 50, 48, 48, 43, 38, 37, 33, 32, 29, 29, 26, 26, 24, 23,
        46, 49, 49, 48, 47, 42, 38, 36, 32, 32, 28, 28, 26, 25, 24, 23, 43, 46,
        46, 45, 44, 40, 36, 34, 31, 30, 27, 27, 25, 24, 22, 21, 43, 46, 46, 45,
        44, 40, 36, 34, 31, 30, 27, 27, 25, 24, 22, 21, 39, 41, 42, 41, 41, 38,
        34, 33, 29, 29, 25, 25, 23, 22, 21, 20, 38, 41, 41, 41, 41, 38, 34, 33,
        29, 28, 25, 25, 23, 22, 21, 20, 36, 39, 39, 39, 39, 36, 33, 31, 28, 27,
        24, 24, 22, 21, 20, 19, 35, 38, 38, 38, 38, 35, 32, 31, 27, 27, 24, 24,
        21, 21, 20, 19, 33, 36, 36, 36, 37, 34, 31, 30, 26, 26, 23, 23, 21, 20,
        19, 18, 32, 34, 34, 35, 35, 33, 30, 29, 26, 25, 22, 22, 20, 20, 18, 17,
        31, 33, 33, 34, 34, 32, 29, 28, 25, 25, 22, 22, 20, 19, 18, 17, 29, 31,
        31, 32, 32, 30, 28, 27, 24, 24, 21, 21, 19, 18, 17, 16, 28, 31, 31, 32,
        32, 30, 28, 27, 24, 24, 21, 21, 19, 18, 17, 16, 26, 28, 28, 29, 29, 28,
        26, 25, 23, 22, 20, 20, 18, 17, 16, 15, 26, 28, 28, 29, 29, 28, 26, 25,
        23, 22, 20, 20, 18, 17, 16, 15,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 65, 64, 64, 64, 64, 60, 60, 58, 57, 54, 52, 50, 47,
        46, 43, 43, 39, 38, 36, 35, 33, 32, 31, 29, 28, 26, 26, 65, 64, 64, 64,
        63, 63, 63, 63, 63, 62, 60, 60, 59, 59, 56, 54, 53, 49, 49, 46, 46, 41,
        41, 39, 38, 36, 34, 33, 31, 31, 28, 28, 65, 64, 64, 64, 63, 63, 63, 63,
        62, 62, 60, 60, 59, 59, 56, 55, 53, 50, 49, 46, 46, 42, 41, 39, 38, 36,
        34, 33, 31, 31, 28, 28, 64, 63, 63, 63, 63, 62, 61, 60, 59, 59, 57, 57,
        55, 55, 53, 51, 50, 48, 48, 45, 45, 41, 41, 39, 38, 36, 35, 34, 32, 32,
        29, 29, 64, 63, 63, 63, 62, 61, 60, 59, 58, 58, 55, 55, 54, 53, 51, 51,
        49, 48, 47, 44, 44, 41, 41, 39, 38, 37, 35, 34, 32, 32, 29, 29, 60, 60,
        60, 61, 61, 59, 58, 58, 55, 55, 52, 52, 49, 48, 47, 46, 45, 43, 42, 40,
        40, 38, 38, 36, 35, 34, 33, 32, 30, 30, 28, 28, 57, 58, 58, 59, 59, 58,
        56, 55, 53, 53, 48, 48, 44, 43, 41, 41, 40, 38, 38, 36, 36, 34, 34, 33,
        32, 31, 30, 29, 28, 28, 26, 26, 54, 55, 55, 56, 56, 55, 54, 53, 52, 51,
        47, 47, 42, 41, 40, 39, 38, 37, 36, 34, 34, 33, 33, 31, 31, 30, 29, 28,
        27, 27, 25, 25, 47, 48, 48, 49, 50, 49, 49, 48, 48, 48, 43, 43, 39, 38,
        37, 35, 34, 33, 32, 31, 31, 29, 29, 28, 27, 26, 26, 25, 24, 24, 23, 23,
        46, 47, 48, 49, 49, 49, 48, 48, 47, 47, 42, 42, 39, 38, 36, 35, 34, 32,
        32, 30, 30, 29, 28, 27, 27, 26, 25, 25, 24, 24, 22, 22, 38, 40, 40, 41,
        41, 41, 41, 41, 41, 41, 38, 38, 35, 34, 33, 32, 30, 29, 28, 27, 27, 25,
        25, 24, 24, 23, 22, 22, 21, 21, 20, 20, 38, 40, 40, 41, 41, 41, 41, 41,
        41, 41, 38, 38, 35, 34, 33, 32, 30, 29, 28, 27, 27, 25, 25, 24, 24, 23,
        22, 22, 21, 21, 20, 20, 33, 34, 34, 35, 36, 36, 36, 36, 37, 36, 34, 34,
        32, 31, 29, 29, 28, 26, 26, 25, 25, 23, 23, 22, 21, 21, 20, 20, 19, 19,
        18, 18, 31, 33, 33, 34, 34, 34, 35, 35, 35, 35, 33, 33, 31, 30, 29, 28,
        27, 26, 25, 24, 24, 22, 22, 21, 21, 20, 20, 19, 18, 18, 17, 17, 28, 29,
        29, 31, 31, 31, 31, 32, 32, 32, 30, 30, 28, 28, 27, 26, 25, 24, 24, 22,
        22, 21, 21, 20, 20, 19, 18, 18, 17, 17, 16, 16, 26, 27, 27, 28, 28, 29,
        29, 29, 30, 29, 28, 28, 26, 26, 25, 24, 24, 23, 23, 21, 21, 20, 20, 19,
        19, 18, 17, 17, 16, 16, 15, 15,
        /* Size 4x16 */
        65, 60, 46, 31, 64, 60, 48, 33, 63, 61, 49, 34, 63, 58, 48, 35, 63, 55,
        47, 35, 60, 52, 42, 33, 59, 49, 39, 31, 56, 47, 36, 29, 53, 45, 34, 27,
        49, 42, 32, 25, 46, 40, 30, 24, 41, 38, 28, 22, 38, 35, 27, 21, 34, 33,
        25, 20, 31, 30, 24, 18, 28, 28, 22, 17,
        /* Size 16x4 */
        65, 64, 63, 63, 63, 60, 59, 56, 53, 49, 46, 41, 38, 34, 31, 28, 60, 60,
        61, 58, 55, 52, 49, 47, 45, 42, 40, 38, 35, 33, 30, 28, 46, 48, 49, 48,
        47, 42, 39, 36, 34, 32, 30, 28, 27, 25, 24, 22, 31, 33, 34, 35, 35, 33,
        31, 29, 27, 25, 24, 22, 21, 20, 18, 17,
        /* Size 8x32 */
        64, 65, 64, 57, 47, 38, 33, 28, 65, 64, 63, 58, 48, 40, 34, 29, 65, 64,
        63, 58, 48, 40, 34, 29, 65, 64, 63, 59, 49, 41, 35, 31, 65, 63, 62, 59,
        50, 41, 36, 31, 65, 63, 61, 58, 49, 41, 36, 31, 64, 63, 60, 56, 49, 41,
        36, 31, 64, 63, 59, 55, 48, 41, 36, 32, 64, 62, 58, 53, 48, 41, 37, 32,
        64, 62, 58, 53, 48, 41, 36, 32, 60, 60, 55, 48, 43, 38, 34, 30, 60, 60,
        55, 48, 43, 38, 34, 30, 58, 59, 54, 44, 39, 35, 32, 28, 57, 59, 53, 43,
        38, 34, 31, 28, 54, 56, 51, 41, 37, 33, 29, 27, 52, 55, 51, 41, 35, 32,
        29, 26, 50, 53, 49, 40, 34, 30, 28, 25, 47, 50, 48, 38, 33, 29, 26, 24,
        46, 49, 47, 38, 32, 28, 26, 24, 43, 46, 44, 36, 31, 27, 25, 22, 43, 46,
        44, 36, 31, 27, 25, 22, 39, 42, 41, 34, 29, 25, 23, 21, 38, 41, 41, 34,
        29, 25, 23, 21, 36, 39, 39, 33, 28, 24, 22, 20, 35, 38, 38, 32, 27, 24,
        21, 20, 33, 36, 37, 31, 26, 23, 21, 19, 32, 34, 35, 30, 26, 22, 20, 18,
        31, 33, 34, 29, 25, 22, 20, 18, 29, 31, 32, 28, 24, 21, 19, 17, 28, 31,
        32, 28, 24, 21, 19, 17, 26, 28, 29, 26, 23, 20, 18, 16, 26, 28, 29, 26,
        23, 20, 18, 16,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 65, 64, 64, 64, 64, 60, 60, 58, 57, 54, 52, 50, 47,
        46, 43, 43, 39, 38, 36, 35, 33, 32, 31, 29, 28, 26, 26, 65, 64, 64, 64,
        63, 63, 63, 63, 62, 62, 60, 60, 59, 59, 56, 55, 53, 50, 49, 46, 46, 42,
        41, 39, 38, 36, 34, 33, 31, 31, 28, 28, 64, 63, 63, 63, 62, 61, 60, 59,
        58, 58, 55, 55, 54, 53, 51, 51, 49, 48, 47, 44, 44, 41, 41, 39, 38, 37,
        35, 34, 32, 32, 29, 29, 57, 58, 58, 59, 59, 58, 56, 55, 53, 53, 48, 48,
        44, 43, 41, 41, 40, 38, 38, 36, 36, 34, 34, 33, 32, 31, 30, 29, 28, 28,
        26, 26, 47, 48, 48, 49, 50, 49, 49, 48, 48, 48, 43, 43, 39, 38, 37, 35,
        34, 33, 32, 31, 31, 29, 29, 28, 27, 26, 26, 25, 24, 24, 23, 23, 38, 40,
        40, 41, 41, 41, 41, 41, 41, 41, 38, 38, 35, 34, 33, 32, 30, 29, 28, 27,
        27, 25, 25, 24, 24, 23, 22, 22, 21, 21, 20, 20, 33, 34, 34, 35, 36, 36,
        36, 36, 37, 36, 34, 34, 32, 31, 29, 29, 28, 26, 26, 25, 25, 23, 23, 22,
        21, 21, 20, 20, 19, 19, 18, 18, 28, 29, 29, 31, 31, 31, 31, 32, 32, 32,
        30, 30, 28, 28, 27, 26, 25, 24, 24, 22, 22, 21, 21, 20, 20, 19, 18, 18,
        17, 17, 16, 16 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        49, 42, 42, 29,
#endif
        /* Size 4x4 */
        65, 48, 44, 38, 48, 42, 41, 38, 44, 41, 33, 30, 38, 38, 30, 26,
        /* Size 8x8 */
        65, 63, 53, 43, 44, 41, 38, 36, 63, 58, 49, 44, 46, 43, 41, 38, 53, 49,
        43, 43, 45, 44, 41, 39, 43, 44, 43, 39, 38, 38, 37, 35, 44, 46, 45, 38,
        35, 34, 33, 31, 41, 43, 44, 38, 34, 31, 30, 28, 38, 41, 41, 37, 33, 30,
        28, 26, 36, 38, 39, 35, 31, 28, 26, 25,
        /* Size 16x16 */
        64, 66, 67, 61, 56, 49, 43, 42, 42, 41, 41, 39, 38, 36, 34, 32, 66, 65,
        66, 59, 53, 48, 44, 43, 44, 44, 42, 41, 39, 38, 36, 34, 67, 66, 64, 58,
        51, 48, 45, 45, 45, 45, 44, 43, 41, 39, 38, 36, 61, 59, 58, 53, 47, 46,
        44, 44, 45, 45, 44, 43, 41, 40, 38, 36, 56, 53, 51, 47, 43, 43, 43, 44,
        45, 45, 45, 44, 42, 41, 39, 38, 49, 48, 48, 46, 43, 42, 41, 41, 41, 41,
        41, 41, 39, 38, 37, 36, 43, 44, 45, 44, 43, 41, 39, 39, 39, 39, 39, 38,
        37, 37, 35, 34, 42, 43, 45, 44, 44, 41, 39, 38, 38, 37, 37, 37, 36, 35,
        34, 33, 42, 44, 45, 45, 45, 41, 39, 38, 37, 36, 35, 35, 34, 33, 33, 32,
        41, 44, 45, 45, 45, 41, 39, 37, 36, 35, 34, 33, 33, 32, 31, 30, 41, 42,
        44, 44, 45, 41, 39, 37, 35, 34, 33, 32, 32, 31, 30, 29, 39, 41, 43, 43,
        44, 41, 38, 37, 35, 33, 32, 31, 30, 29, 28, 27, 38, 39, 41, 41, 42, 39,
        37, 36, 34, 33, 32, 30, 29, 28, 27, 26, 36, 38, 39, 40, 41, 38, 37, 35,
        33, 32, 31, 29, 28, 27, 26, 25, 34, 36, 38, 38, 39, 37, 35, 34, 33, 31,
        30, 28, 27, 26, 25, 24, 32, 34, 36, 36, 38, 36, 34, 33, 32, 30, 29, 27,
        26, 25, 24, 23,
        /* Size 32x32 */
        64, 66, 66, 67, 67, 64, 61, 59, 56, 55, 49, 49, 43, 42, 42, 42, 42, 42,
        41, 41, 41, 39, 39, 38, 38, 37, 36, 35, 34, 34, 32, 32, 66, 65, 65, 66,
        66, 63, 59, 58, 53, 53, 49, 49, 44, 43, 43, 44, 44, 44, 43, 42, 42, 41,
        41, 40, 39, 38, 38, 37, 36, 36, 34, 34, 66, 65, 65, 65, 66, 62, 59, 58,
        53, 53, 48, 48, 44, 43, 43, 44, 44, 44, 44, 42, 42, 41, 41, 40, 39, 39,
        38, 37, 36, 36, 34, 34, 67, 66, 65, 65, 65, 61, 59, 56, 52, 51, 48, 48,
        45, 44, 44, 45, 45, 45, 45, 44, 44, 43, 42, 41, 41, 40, 39, 38, 37, 37,
        35, 35, 67, 66, 66, 65, 64, 61, 58, 56, 51, 51, 48, 48, 45, 44, 45, 45,
        45, 46, 45, 44, 44, 43, 43, 42, 41, 40, 39, 39, 38, 38, 36, 36, 64, 63,
        62, 61, 61, 58, 55, 53, 49, 49, 47, 47, 45, 44, 44, 45, 45, 46, 45, 44,
        44, 43, 43, 42, 41, 41, 40, 39, 38, 38, 36, 36, 61, 59, 59, 59, 58, 55,
        53, 51, 47, 47, 46, 46, 44, 43, 44, 44, 45, 45, 45, 44, 44, 43, 43, 42,
        41, 41, 40, 39, 38, 38, 36, 36, 59, 58, 58, 56, 56, 53, 51, 49, 46, 46,
        45, 45, 44, 43, 44, 44, 45, 45, 45, 44, 44, 44, 43, 42, 42, 41, 40, 40,
        39, 38, 37, 37, 56, 53, 53, 52, 51, 49, 47, 46, 43, 43, 43, 43, 43, 43,
        44, 44, 45, 45, 45, 45, 45, 44, 44, 43, 42, 42, 41, 41, 39, 39, 38, 38,
        55, 53, 53, 51, 51, 49, 47, 46, 43, 43, 43, 43, 43, 43, 43, 44, 44, 45,
        45, 44, 44, 44, 44, 43, 42, 41, 41, 40, 39, 39, 37, 37, 49, 49, 48, 48,
        48, 47, 46, 45, 43, 43, 42, 42, 41, 41, 41, 41, 41, 42, 41, 41, 41, 41,
        41, 40, 39, 39, 38, 38, 37, 37, 36, 36, 49, 49, 48, 48, 48, 47, 46, 45,
        43, 43, 42, 42, 41, 41, 41, 41, 41, 42, 41, 41, 41, 41, 41, 40, 39, 39,
        38, 38, 37, 37, 36, 36, 43, 44, 44, 45, 45, 45, 44, 44, 43, 43, 41, 41,
        39, 39, 39, 39, 39, 39, 39, 39, 39, 38, 38, 38, 37, 37, 37, 36, 35, 35,
        34, 34, 42, 43, 43, 44, 44, 44, 43, 43, 43, 43, 41, 41, 39, 39, 39, 39,
        38, 38, 38, 38, 38, 38, 38, 37, 37, 37, 36, 36, 35, 35, 34, 34, 42, 43,
        43, 44, 45, 44, 44, 44, 44, 43, 41, 41, 39, 39, 38, 38, 38, 37, 37, 37,
        37, 37, 37, 36, 36, 35, 35, 35, 34, 34, 33, 33, 42, 44, 44, 45, 45, 45,
        44, 44, 44, 44, 41, 41, 39, 39, 38, 38, 37, 37, 37, 36, 36, 36, 36, 35,
        35, 34, 34, 34, 33, 33, 32, 32, 42, 44, 44, 45, 45, 45, 45, 45, 45, 44,
        41, 41, 39, 38, 38, 37, 37, 36, 36, 35, 35, 35, 35, 34, 34, 34, 33, 33,
        33, 32, 32, 32, 42, 44, 44, 45, 46, 46, 45, 45, 45, 45, 42, 42, 39, 38,
        37, 37, 36, 35, 35, 34, 34, 34, 33, 33, 33, 33, 32, 32, 31, 31, 30, 30,
        41, 43, 44, 45, 45, 45, 45, 45, 45, 45, 41, 41, 39, 38, 37, 37, 36, 35,
        35, 34, 34, 33, 33, 33, 33, 32, 32, 32, 31, 31, 30, 30, 41, 42, 42, 44,
        44, 44, 44, 44, 45, 44, 41, 41, 39, 38, 37, 36, 35, 34, 34, 33, 33, 32,
        32, 32, 32, 31, 31, 30, 30, 30, 29, 29, 41, 42, 42, 44, 44, 44, 44, 44,
        45, 44, 41, 41, 39, 38, 37, 36, 35, 34, 34, 33, 33, 32, 32, 32, 32, 31,
        31, 30, 30, 30, 29, 29, 39, 41, 41, 43, 43, 43, 43, 44, 44, 44, 41, 41,
        38, 38, 37, 36, 35, 34, 33, 32, 32, 31, 31, 30, 30, 30, 29, 29, 28, 28,
        27, 27, 39, 41, 41, 42, 43, 43, 43, 43, 44, 44, 41, 41, 38, 38, 37, 36,
        35, 33, 33, 32, 32, 31, 31, 30, 30, 29, 29, 29, 28, 28, 27, 27, 38, 40,
        40, 41, 42, 42, 42, 42, 43, 43, 40, 40, 38, 37, 36, 35, 34, 33, 33, 32,
        32, 30, 30, 29, 29, 29, 28, 28, 27, 27, 27, 27, 38, 39, 39, 41, 41, 41,
        41, 42, 42, 42, 39, 39, 37, 37, 36, 35, 34, 33, 33, 32, 32, 30, 30, 29,
        29, 28, 28, 28, 27, 27, 26, 26, 37, 38, 39, 40, 40, 41, 41, 41, 42, 41,
        39, 39, 37, 37, 35, 34, 34, 33, 32, 31, 31, 30, 29, 29, 28, 28, 27, 27,
        27, 26, 26, 26, 36, 38, 38, 39, 39, 40, 40, 40, 41, 41, 38, 38, 37, 36,
        35, 34, 33, 32, 32, 31, 31, 29, 29, 28, 28, 27, 27, 26, 26, 26, 25, 25,
        35, 37, 37, 38, 39, 39, 39, 40, 41, 40, 38, 38, 36, 36, 35, 34, 33, 32,
        32, 30, 30, 29, 29, 28, 28, 27, 26, 26, 26, 26, 25, 25, 34, 36, 36, 37,
        38, 38, 38, 39, 39, 39, 37, 37, 35, 35, 34, 33, 33, 31, 31, 30, 30, 28,
        28, 27, 27, 27, 26, 26, 25, 25, 24, 24, 34, 36, 36, 37, 38, 38, 38, 38,
        39, 39, 37, 37, 35, 35, 34, 33, 32, 31, 31, 30, 30, 28, 28, 27, 27, 26,
        26, 26, 25, 25, 24, 24, 32, 34, 34, 35, 36, 36, 36, 37, 38, 37, 36, 36,
        34, 34, 33, 32, 32, 30, 30, 29, 29, 27, 27, 27, 26, 26, 25, 25, 24, 24,
        23, 23, 32, 34, 34, 35, 36, 36, 36, 37, 38, 37, 36, 36, 34, 34, 33, 32,
        32, 30, 30, 29, 29, 27, 27, 27, 26, 26, 25, 25, 24, 24, 23, 23,
        /* Size 4x8 */
        65, 48, 43, 38, 61, 47, 45, 40, 51, 43, 45, 41, 44, 41, 38, 36, 45, 41,
        35, 32, 43, 41, 33, 29, 40, 39, 32, 27, 37, 37, 31, 26,
        /* Size 8x4 */
        65, 61, 51, 44, 45, 43, 40, 37, 48, 47, 43, 41, 41, 41, 39, 37, 43, 45,
        45, 38, 35, 33, 32, 31, 38, 40, 41, 36, 32, 29, 27, 26,
        /* Size 8x16 */
        64, 66, 55, 43, 42, 39, 37, 34, 65, 65, 53, 44, 44, 41, 38, 36, 67, 64,
        51, 45, 45, 43, 40, 37, 61, 57, 48, 44, 45, 43, 41, 38, 55, 51, 44, 43,
        45, 44, 41, 39, 49, 48, 43, 41, 41, 41, 39, 37, 43, 45, 43, 39, 39, 38,
        37, 35, 42, 45, 43, 39, 37, 36, 35, 34, 42, 45, 44, 39, 36, 35, 34, 32,
        42, 45, 45, 38, 35, 33, 32, 31, 41, 44, 44, 38, 34, 32, 31, 29, 39, 43,
        43, 38, 34, 31, 29, 28, 38, 41, 42, 37, 33, 30, 28, 27, 36, 39, 41, 36,
        32, 29, 27, 26, 34, 38, 39, 35, 31, 28, 26, 25, 32, 36, 37, 34, 30, 27,
        25, 24,
        /* Size 16x8 */
        64, 65, 67, 61, 55, 49, 43, 42, 42, 42, 41, 39, 38, 36, 34, 32, 66, 65,
        64, 57, 51, 48, 45, 45, 45, 45, 44, 43, 41, 39, 38, 36, 55, 53, 51, 48,
        44, 43, 43, 43, 44, 45, 44, 43, 42, 41, 39, 37, 43, 44, 45, 44, 43, 41,
        39, 39, 39, 38, 38, 38, 37, 36, 35, 34, 42, 44, 45, 45, 45, 41, 39, 37,
        36, 35, 34, 34, 33, 32, 31, 30, 39, 41, 43, 43, 44, 41, 38, 36, 35, 33,
        32, 31, 30, 29, 28, 27, 37, 38, 40, 41, 41, 39, 37, 35, 34, 32, 31, 29,
        28, 27, 26, 25, 34, 36, 37, 38, 39, 37, 35, 34, 32, 31, 29, 28, 27, 26,
        25, 24,
        /* Size 16x32 */
        64, 66, 66, 59, 55, 49, 43, 42, 42, 41, 39, 39, 37, 36, 34, 32, 65, 65,
        65, 57, 53, 48, 44, 43, 43, 43, 41, 41, 38, 38, 35, 34, 65, 65, 65, 57,
        53, 48, 44, 44, 44, 43, 41, 41, 38, 38, 36, 34, 67, 64, 64, 55, 52, 48,
        44, 45, 45, 45, 42, 42, 39, 39, 37, 35, 67, 64, 64, 55, 51, 48, 45, 45,
        45, 45, 43, 43, 40, 39, 37, 36, 64, 61, 60, 53, 49, 47, 44, 45, 45, 45,
        43, 43, 40, 40, 38, 36, 61, 58, 57, 51, 48, 46, 44, 44, 45, 45, 43, 43,
        41, 40, 38, 37, 59, 55, 55, 49, 46, 45, 44, 44, 45, 45, 43, 43, 41, 40,
        38, 37, 55, 51, 51, 46, 44, 43, 43, 44, 45, 45, 44, 44, 41, 41, 39, 38,
        55, 51, 51, 46, 44, 43, 43, 43, 45, 45, 43, 43, 41, 41, 39, 37, 49, 48,
        48, 45, 43, 42, 41, 41, 41, 41, 41, 41, 39, 38, 37, 36, 49, 48, 48, 45,
        43, 42, 41, 41, 41, 41, 41, 41, 39, 38, 37, 36, 43, 45, 45, 43, 43, 41,
        39, 39, 39, 39, 38, 38, 37, 37, 35, 34, 42, 44, 44, 43, 43, 41, 39, 39,
        38, 38, 38, 38, 36, 36, 35, 34, 42, 44, 45, 44, 43, 41, 39, 38, 37, 37,
        36, 36, 35, 35, 34, 33, 42, 45, 45, 44, 44, 41, 39, 38, 37, 37, 36, 36,
        34, 34, 33, 32, 42, 45, 45, 45, 44, 41, 39, 38, 36, 36, 35, 35, 34, 33,
        32, 32, 42, 45, 45, 45, 45, 41, 38, 37, 35, 35, 33, 33, 32, 32, 31, 30,
        42, 45, 45, 45, 45, 41, 38, 37, 35, 35, 33, 33, 32, 32, 31, 30, 41, 44,
        44, 44, 44, 41, 38, 37, 34, 34, 32, 32, 31, 31, 29, 29, 41, 44, 44, 44,
        44, 41, 38, 37, 34, 34, 32, 32, 31, 31, 29, 29, 39, 43, 43, 43, 44, 41,
        38, 37, 34, 33, 31, 31, 29, 29, 28, 27, 39, 42, 43, 43, 43, 41, 38, 37,
        34, 33, 31, 31, 29, 29, 28, 27, 38, 41, 41, 42, 42, 40, 37, 36, 33, 33,
        30, 30, 29, 28, 27, 27, 38, 41, 41, 42, 42, 39, 37, 36, 33, 33, 30, 30,
        28, 28, 27, 26, 37, 40, 40, 41, 41, 39, 37, 35, 33, 32, 29, 29, 28, 27,
        26, 26, 36, 39, 39, 40, 41, 38, 36, 35, 32, 32, 29, 29, 27, 27, 26, 25,
        35, 38, 39, 40, 40, 38, 36, 35, 32, 32, 29, 29, 27, 26, 25, 25, 34, 37,
        38, 39, 39, 37, 35, 34, 31, 31, 28, 28, 26, 26, 25, 24, 34, 37, 37, 38,
        39, 37, 35, 34, 31, 31, 28, 28, 26, 26, 25, 24, 32, 35, 36, 37, 37, 35,
        34, 33, 30, 30, 27, 27, 25, 25, 24, 23, 32, 35, 36, 37, 37, 35, 34, 33,
        30, 30, 27, 27, 25, 25, 24, 23,
        /* Size 32x16 */
        64, 65, 65, 67, 67, 64, 61, 59, 55, 55, 49, 49, 43, 42, 42, 42, 42, 42,
        42, 41, 41, 39, 39, 38, 38, 37, 36, 35, 34, 34, 32, 32, 66, 65, 65, 64,
        64, 61, 58, 55, 51, 51, 48, 48, 45, 44, 44, 45, 45, 45, 45, 44, 44, 43,
        42, 41, 41, 40, 39, 38, 37, 37, 35, 35, 66, 65, 65, 64, 64, 60, 57, 55,
        51, 51, 48, 48, 45, 44, 45, 45, 45, 45, 45, 44, 44, 43, 43, 41, 41, 40,
        39, 39, 38, 37, 36, 36, 59, 57, 57, 55, 55, 53, 51, 49, 46, 46, 45, 45,
        43, 43, 44, 44, 45, 45, 45, 44, 44, 43, 43, 42, 42, 41, 40, 40, 39, 38,
        37, 37, 55, 53, 53, 52, 51, 49, 48, 46, 44, 44, 43, 43, 43, 43, 43, 44,
        44, 45, 45, 44, 44, 44, 43, 42, 42, 41, 41, 40, 39, 39, 37, 37, 49, 48,
        48, 48, 48, 47, 46, 45, 43, 43, 42, 42, 41, 41, 41, 41, 41, 41, 41, 41,
        41, 41, 41, 40, 39, 39, 38, 38, 37, 37, 35, 35, 43, 44, 44, 44, 45, 44,
        44, 44, 43, 43, 41, 41, 39, 39, 39, 39, 39, 38, 38, 38, 38, 38, 38, 37,
        37, 37, 36, 36, 35, 35, 34, 34, 42, 43, 44, 45, 45, 45, 44, 44, 44, 43,
        41, 41, 39, 39, 38, 38, 38, 37, 37, 37, 37, 37, 37, 36, 36, 35, 35, 35,
        34, 34, 33, 33, 42, 43, 44, 45, 45, 45, 45, 45, 45, 45, 41, 41, 39, 38,
        37, 37, 36, 35, 35, 34, 34, 34, 34, 33, 33, 33, 32, 32, 31, 31, 30, 30,
        41, 43, 43, 45, 45, 45, 45, 45, 45, 45, 41, 41, 39, 38, 37, 37, 36, 35,
        35, 34, 34, 33, 33, 33, 33, 32, 32, 32, 31, 31, 30, 30, 39, 41, 41, 42,
        43, 43, 43, 43, 44, 43, 41, 41, 38, 38, 36, 36, 35, 33, 33, 32, 32, 31,
        31, 30, 30, 29, 29, 29, 28, 28, 27, 27, 39, 41, 41, 42, 43, 43, 43, 43,
        44, 43, 41, 41, 38, 38, 36, 36, 35, 33, 33, 32, 32, 31, 31, 30, 30, 29,
        29, 29, 28, 28, 27, 27, 37, 38, 38, 39, 40, 40, 41, 41, 41, 41, 39, 39,
        37, 36, 35, 34, 34, 32, 32, 31, 31, 29, 29, 29, 28, 28, 27, 27, 26, 26,
        25, 25, 36, 38, 38, 39, 39, 40, 40, 40, 41, 41, 38, 38, 37, 36, 35, 34,
        33, 32, 32, 31, 31, 29, 29, 28, 28, 27, 27, 26, 26, 26, 25, 25, 34, 35,
        36, 37, 37, 38, 38, 38, 39, 39, 37, 37, 35, 35, 34, 33, 32, 31, 31, 29,
        29, 28, 28, 27, 27, 26, 26, 25, 25, 25, 24, 24, 32, 34, 34, 35, 36, 36,
        37, 37, 38, 37, 36, 36, 34, 34, 33, 32, 32, 30, 30, 29, 29, 27, 27, 27,
        26, 26, 25, 25, 24, 24, 23, 23,
        /* Size 4x16 */
        66, 49, 41, 36, 65, 48, 43, 38, 64, 48, 45, 39, 58, 46, 45, 40, 51, 43,
        45, 41, 48, 42, 41, 38, 45, 41, 39, 37, 44, 41, 37, 35, 45, 41, 36, 33,
        45, 41, 35, 32, 44, 41, 34, 31, 42, 41, 33, 29, 41, 39, 33, 28, 39, 38,
        32, 27, 37, 37, 31, 26, 35, 35, 30, 25,
        /* Size 16x4 */
        66, 65, 64, 58, 51, 48, 45, 44, 45, 45, 44, 42, 41, 39, 37, 35, 49, 48,
        48, 46, 43, 42, 41, 41, 41, 41, 41, 41, 39, 38, 37, 35, 41, 43, 45, 45,
        45, 41, 39, 37, 36, 35, 34, 33, 33, 32, 31, 30, 36, 38, 39, 40, 41, 38,
        37, 35, 33, 32, 31, 29, 28, 27, 26, 25,
        /* Size 8x32 */
        64, 66, 55, 43, 42, 39, 37, 34, 65, 65, 53, 44, 43, 41, 38, 35, 65, 65,
        53, 44, 44, 41, 38, 36, 67, 64, 52, 44, 45, 42, 39, 37, 67, 64, 51, 45,
        45, 43, 40, 37, 64, 60, 49, 44, 45, 43, 40, 38, 61, 57, 48, 44, 45, 43,
        41, 38, 59, 55, 46, 44, 45, 43, 41, 38, 55, 51, 44, 43, 45, 44, 41, 39,
        55, 51, 44, 43, 45, 43, 41, 39, 49, 48, 43, 41, 41, 41, 39, 37, 49, 48,
        43, 41, 41, 41, 39, 37, 43, 45, 43, 39, 39, 38, 37, 35, 42, 44, 43, 39,
        38, 38, 36, 35, 42, 45, 43, 39, 37, 36, 35, 34, 42, 45, 44, 39, 37, 36,
        34, 33, 42, 45, 44, 39, 36, 35, 34, 32, 42, 45, 45, 38, 35, 33, 32, 31,
        42, 45, 45, 38, 35, 33, 32, 31, 41, 44, 44, 38, 34, 32, 31, 29, 41, 44,
        44, 38, 34, 32, 31, 29, 39, 43, 44, 38, 34, 31, 29, 28, 39, 43, 43, 38,
        34, 31, 29, 28, 38, 41, 42, 37, 33, 30, 29, 27, 38, 41, 42, 37, 33, 30,
        28, 27, 37, 40, 41, 37, 33, 29, 28, 26, 36, 39, 41, 36, 32, 29, 27, 26,
        35, 39, 40, 36, 32, 29, 27, 25, 34, 38, 39, 35, 31, 28, 26, 25, 34, 37,
        39, 35, 31, 28, 26, 25, 32, 36, 37, 34, 30, 27, 25, 24, 32, 36, 37, 34,
        30, 27, 25, 24,
        /* Size 32x8 */
        64, 65, 65, 67, 67, 64, 61, 59, 55, 55, 49, 49, 43, 42, 42, 42, 42, 42,
        42, 41, 41, 39, 39, 38, 38, 37, 36, 35, 34, 34, 32, 32, 66, 65, 65, 64,
        64, 60, 57, 55, 51, 51, 48, 48, 45, 44, 45, 45, 45, 45, 45, 44, 44, 43,
        43, 41, 41, 40, 39, 39, 38, 37, 36, 36, 55, 53, 53, 52, 51, 49, 48, 46,
        44, 44, 43, 43, 43, 43, 43, 44, 44, 45, 45, 44, 44, 44, 43, 42, 42, 41,
        41, 40, 39, 39, 37, 37, 43, 44, 44, 44, 45, 44, 44, 44, 43, 43, 41, 41,
        39, 39, 39, 39, 39, 38, 38, 38, 38, 38, 38, 37, 37, 37, 36, 36, 35, 35,
        34, 34, 42, 43, 44, 45, 45, 45, 45, 45, 45, 45, 41, 41, 39, 38, 37, 37,
        36, 35, 35, 34, 34, 34, 34, 33, 33, 33, 32, 32, 31, 31, 30, 30, 39, 41,
        41, 42, 43, 43, 43, 43, 44, 43, 41, 41, 38, 38, 36, 36, 35, 33, 33, 32,
        32, 31, 31, 30, 30, 29, 29, 29, 28, 28, 27, 27, 37, 38, 38, 39, 40, 40,
        41, 41, 41, 41, 39, 39, 37, 36, 35, 34, 34, 32, 32, 31, 31, 29, 29, 29,
        28, 28, 27, 27, 26, 26, 25, 25, 34, 35, 36, 37, 37, 38, 38, 38, 39, 39,
        37, 37, 35, 35, 34, 33, 32, 31, 31, 29, 29, 28, 28, 27, 27, 26, 26, 25,
        25, 25, 24, 24 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        61, 44, 44, 27,
#endif
        /* Size 4x4 */
        64, 61, 49, 37, 61, 53, 44, 36, 49, 44, 32, 27, 37, 36, 27, 22,
        /* Size 8x8 */
        65, 64, 64, 59, 53, 45, 39, 33, 64, 63, 63, 60, 55, 47, 41, 34, 64, 63,
        58, 55, 51, 46, 41, 35, 59, 60, 55, 48, 44, 39, 36, 32, 53, 55, 51, 44,
        38, 34, 32, 28, 45, 47, 46, 39, 34, 29, 27, 24, 39, 41, 41, 36, 32, 27,
        25, 22, 33, 34, 35, 32, 28, 24, 22, 20,
        /* Size 16x16 */
        64, 65, 65, 65, 64, 64, 60, 57, 53, 50, 47, 42, 38, 35, 33, 31, 65, 64,
        64, 64, 64, 63, 60, 59, 54, 51, 48, 44, 40, 37, 35, 33, 65, 64, 63, 63,
        63, 63, 61, 59, 55, 53, 50, 46, 41, 39, 37, 34, 65, 64, 63, 62, 61, 61,
        59, 57, 54, 51, 49, 45, 41, 39, 37, 34, 64, 64, 63, 61, 60, 59, 57, 55,
        53, 51, 48, 45, 41, 39, 37, 35, 64, 63, 63, 61, 59, 58, 55, 53, 51, 49,
        48, 45, 41, 39, 37, 35, 60, 60, 61, 59, 57, 55, 52, 48, 46, 45, 43, 40,
        38, 36, 34, 33, 57, 59, 59, 57, 55, 53, 48, 42, 41, 39, 38, 36, 34, 33,
        31, 30, 53, 54, 55, 54, 53, 51, 46, 41, 39, 38, 36, 34, 32, 31, 29, 28,
        50, 51, 53, 51, 51, 49, 45, 39, 38, 36, 34, 32, 30, 29, 28, 27, 47, 48,
        50, 49, 48, 48, 43, 38, 36, 34, 32, 31, 29, 27, 27, 26, 42, 44, 46, 45,
        45, 45, 40, 36, 34, 32, 31, 29, 27, 26, 25, 24, 38, 40, 41, 41, 41, 41,
        38, 34, 32, 30, 29, 27, 25, 24, 23, 22, 35, 37, 39, 39, 39, 39, 36, 33,
        31, 29, 27, 26, 24, 23, 22, 21, 33, 35, 37, 37, 37, 37, 34, 31, 29, 28,
        27, 25, 23, 22, 21, 20, 31, 33, 34, 34, 35, 35, 33, 30, 28, 27, 26, 24,
        22, 21, 20, 20,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 61, 60, 59, 57, 57, 53, 52,
        50, 47, 47, 43, 42, 41, 38, 38, 35, 35, 33, 31, 31, 29, 65, 65, 65, 64,
        64, 64, 64, 64, 64, 64, 64, 61, 60, 59, 58, 58, 54, 53, 51, 48, 48, 45,
        44, 42, 39, 39, 37, 36, 35, 33, 33, 30, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 63, 63, 61, 60, 59, 59, 59, 54, 53, 51, 48, 48, 45, 44, 42, 40, 40,
        37, 36, 35, 33, 33, 31, 65, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 61,
        60, 60, 59, 59, 55, 54, 52, 49, 49, 46, 45, 43, 41, 41, 38, 37, 36, 34,
        34, 31, 65, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 61, 61, 60, 59, 59,
        55, 55, 53, 50, 50, 47, 46, 44, 41, 41, 39, 38, 37, 34, 34, 32, 65, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 61, 61, 60, 59, 59, 55, 55, 53, 50,
        50, 47, 46, 44, 41, 41, 39, 38, 37, 34, 34, 32, 65, 64, 64, 63, 63, 63,
        62, 62, 61, 61, 61, 59, 59, 58, 57, 57, 54, 53, 51, 49, 49, 46, 45, 44,
        41, 41, 39, 38, 37, 34, 34, 32, 65, 64, 64, 63, 63, 63, 62, 61, 61, 60,
        60, 59, 58, 57, 56, 56, 53, 53, 51, 49, 49, 46, 45, 44, 41, 41, 39, 38,
        37, 35, 35, 32, 64, 64, 64, 63, 63, 63, 61, 61, 60, 59, 59, 58, 57, 56,
        55, 55, 53, 52, 51, 48, 48, 46, 45, 44, 41, 41, 39, 38, 37, 35, 35, 33,
        64, 64, 63, 63, 63, 63, 61, 60, 59, 58, 58, 56, 55, 55, 53, 53, 51, 51,
        49, 48, 48, 46, 45, 44, 41, 41, 39, 38, 37, 35, 35, 33, 64, 64, 63, 63,
        63, 63, 61, 60, 59, 58, 58, 56, 55, 55, 53, 53, 51, 51, 49, 48, 48, 46,
        45, 44, 41, 41, 39, 38, 37, 35, 35, 33, 61, 61, 61, 61, 61, 61, 59, 59,
        58, 56, 56, 53, 53, 51, 49, 49, 47, 47, 46, 44, 44, 42, 41, 40, 38, 38,
        36, 36, 35, 33, 33, 31, 60, 60, 60, 60, 61, 61, 59, 58, 57, 55, 55, 53,
        52, 50, 48, 48, 46, 46, 45, 43, 43, 41, 40, 39, 38, 38, 36, 35, 34, 33,
        33, 31, 59, 59, 59, 60, 60, 60, 58, 57, 56, 55, 55, 51, 50, 48, 46, 46,
        44, 44, 42, 41, 41, 39, 39, 38, 36, 36, 34, 34, 33, 32, 32, 30, 57, 58,
        59, 59, 59, 59, 57, 56, 55, 53, 53, 49, 48, 46, 42, 42, 41, 41, 39, 38,
        38, 37, 36, 35, 34, 34, 33, 32, 31, 30, 30, 28, 57, 58, 59, 59, 59, 59,
        57, 56, 55, 53, 53, 49, 48, 46, 42, 42, 41, 41, 39, 38, 38, 37, 36, 35,
        34, 34, 33, 32, 31, 30, 30, 28, 53, 54, 54, 55, 55, 55, 54, 53, 53, 51,
        51, 47, 46, 44, 41, 41, 39, 39, 38, 36, 36, 34, 34, 33, 32, 32, 31, 30,
        29, 28, 28, 27, 52, 53, 53, 54, 55, 55, 53, 53, 52, 51, 51, 47, 46, 44,
        41, 41, 39, 38, 37, 35, 35, 34, 34, 33, 32, 32, 30, 30, 29, 28, 28, 27,
        50, 51, 51, 52, 53, 53, 51, 51, 51, 49, 49, 46, 45, 42, 39, 39, 38, 37,
        36, 34, 34, 33, 32, 32, 30, 30, 29, 29, 28, 27, 27, 26, 47, 48, 48, 49,
        50, 50, 49, 49, 48, 48, 48, 44, 43, 41, 38, 38, 36, 35, 34, 32, 32, 31,
        31, 30, 29, 29, 27, 27, 27, 26, 26, 25, 47, 48, 48, 49, 50, 50, 49, 49,
        48, 48, 48, 44, 43, 41, 38, 38, 36, 35, 34, 32, 32, 31, 31, 30, 29, 29,
        27, 27, 27, 26, 26, 25, 43, 45, 45, 46, 47, 47, 46, 46, 46, 46, 46, 42,
        41, 39, 37, 37, 34, 34, 33, 31, 31, 29, 29, 28, 27, 27, 26, 26, 25, 24,
        24, 23, 42, 44, 44, 45, 46, 46, 45, 45, 45, 45, 45, 41, 40, 39, 36, 36,
        34, 34, 32, 31, 31, 29, 29, 28, 27, 27, 26, 25, 25, 24, 24, 23, 41, 42,
        42, 43, 44, 44, 44, 44, 44, 44, 44, 40, 39, 38, 35, 35, 33, 33, 32, 30,
        30, 28, 28, 27, 26, 26, 25, 25, 24, 23, 23, 22, 38, 39, 40, 41, 41, 41,
        41, 41, 41, 41, 41, 38, 38, 36, 34, 34, 32, 32, 30, 29, 29, 27, 27, 26,
        25, 25, 24, 24, 23, 22, 22, 21, 38, 39, 40, 41, 41, 41, 41, 41, 41, 41,
        41, 38, 38, 36, 34, 34, 32, 32, 30, 29, 29, 27, 27, 26, 25, 25, 24, 24,
        23, 22, 22, 21, 35, 37, 37, 38, 39, 39, 39, 39, 39, 39, 39, 36, 36, 34,
        33, 33, 31, 30, 29, 27, 27, 26, 26, 25, 24, 24, 23, 23, 22, 21, 21, 20,
        35, 36, 36, 37, 38, 38, 38, 38, 38, 38, 38, 36, 35, 34, 32, 32, 30, 30,
        29, 27, 27, 26, 25, 25, 24, 24, 23, 22, 22, 21, 21, 20, 33, 35, 35, 36,
        37, 37, 37, 37, 37, 37, 37, 35, 34, 33, 31, 31, 29, 29, 28, 27, 27, 25,
        25, 24, 23, 23, 22, 22, 21, 20, 20, 19, 31, 33, 33, 34, 34, 34, 34, 35,
        35, 35, 35, 33, 33, 32, 30, 30, 28, 28, 27, 26, 26, 24, 24, 23, 22, 22,
        21, 21, 20, 20, 20, 19, 31, 33, 33, 34, 34, 34, 34, 35, 35, 35, 35, 33,
        33, 32, 30, 30, 28, 28, 27, 26, 26, 24, 24, 23, 22, 22, 21, 21, 20, 20,
        20, 19, 29, 30, 31, 31, 32, 32, 32, 32, 33, 33, 33, 31, 31, 30, 28, 28,
        27, 27, 26, 25, 25, 23, 23, 22, 21, 21, 20, 20, 19, 19, 19, 18,
        /* Size 4x8 */
        64, 63, 48, 37, 64, 62, 50, 39, 63, 58, 48, 39, 59, 55, 41, 34, 54, 51,
        35, 30, 46, 45, 31, 26, 41, 41, 29, 24, 34, 35, 26, 21,
        /* Size 8x4 */
        64, 64, 63, 59, 54, 46, 41, 34, 63, 62, 58, 55, 51, 45, 41, 35, 48, 50,
        48, 41, 35, 31, 29, 26, 37, 39, 39, 34, 30, 26, 24, 21,
        /* Size 8x16 */
        64, 65, 64, 59, 53, 47, 38, 31, 65, 64, 63, 59, 54, 48, 40, 33, 65, 63,
        62, 60, 55, 50, 41, 34, 65, 63, 60, 58, 53, 49, 41, 35, 64, 63, 59, 56,
        53, 48, 41, 35, 64, 62, 58, 55, 51, 48, 41, 35, 60, 60, 55, 49, 46, 43,
        38, 33, 57, 59, 53, 45, 41, 38, 34, 30, 53, 55, 51, 43, 39, 36, 32, 28,
        50, 53, 49, 41, 38, 34, 30, 27, 47, 50, 48, 40, 36, 33, 29, 26, 43, 46,
        44, 38, 34, 31, 27, 24, 38, 41, 41, 36, 32, 29, 25, 22, 36, 39, 39, 34,
        31, 27, 24, 21, 34, 36, 37, 33, 29, 27, 23, 20, 32, 34, 35, 31, 28, 26,
        22, 20,
        /* Size 16x8 */
        64, 65, 65, 65, 64, 64, 60, 57, 53, 50, 47, 43, 38, 36, 34, 32, 65, 64,
        63, 63, 63, 62, 60, 59, 55, 53, 50, 46, 41, 39, 36, 34, 64, 63, 62, 60,
        59, 58, 55, 53, 51, 49, 48, 44, 41, 39, 37, 35, 59, 59, 60, 58, 56, 55,
        49, 45, 43, 41, 40, 38, 36, 34, 33, 31, 53, 54, 55, 53, 53, 51, 46, 41,
        39, 38, 36, 34, 32, 31, 29, 28, 47, 48, 50, 49, 48, 48, 43, 38, 36, 34,
        33, 31, 29, 27, 27, 26, 38, 40, 41, 41, 41, 41, 38, 34, 32, 30, 29, 27,
        25, 24, 23, 22, 31, 33, 34, 35, 35, 35, 33, 30, 28, 27, 26, 24, 22, 21,
        20, 20,
        /* Size 16x32 */
        64, 65, 65, 65, 64, 64, 59, 57, 53, 47, 47, 40, 38, 35, 31, 31, 65, 64,
        64, 64, 63, 63, 59, 58, 54, 48, 48, 41, 39, 37, 33, 33, 65, 64, 64, 64,
        63, 63, 59, 58, 54, 48, 48, 41, 40, 37, 33, 33, 65, 64, 64, 63, 63, 63,
        59, 59, 55, 49, 49, 42, 41, 38, 34, 34, 65, 64, 63, 63, 62, 62, 60, 59,
        55, 50, 50, 43, 41, 39, 34, 34, 65, 64, 63, 63, 62, 62, 60, 59, 55, 50,
        50, 43, 41, 39, 34, 34, 65, 63, 63, 62, 60, 60, 58, 57, 53, 49, 49, 43,
        41, 39, 35, 35, 64, 63, 63, 62, 60, 60, 57, 56, 53, 49, 49, 43, 41, 39,
        35, 35, 64, 63, 63, 61, 59, 59, 56, 55, 53, 48, 48, 43, 41, 39, 35, 35,
        64, 63, 62, 60, 58, 58, 55, 53, 51, 48, 48, 43, 41, 39, 35, 35, 64, 63,
        62, 60, 58, 58, 55, 53, 51, 48, 48, 43, 41, 39, 35, 35, 61, 61, 61, 59,
        56, 56, 51, 49, 47, 44, 44, 39, 38, 36, 33, 33, 60, 60, 60, 59, 55, 55,
        49, 48, 46, 43, 43, 39, 38, 36, 33, 33, 59, 59, 60, 58, 55, 55, 48, 46,
        44, 41, 41, 37, 36, 34, 32, 32, 57, 59, 59, 57, 53, 53, 45, 43, 41, 38,
        38, 35, 34, 33, 30, 30, 57, 59, 59, 57, 53, 53, 45, 43, 41, 38, 38, 35,
        34, 33, 30, 30, 53, 55, 55, 53, 51, 51, 43, 41, 39, 36, 36, 33, 32, 31,
        28, 28, 52, 54, 55, 53, 51, 51, 43, 41, 39, 35, 35, 32, 32, 30, 28, 28,
        50, 52, 53, 51, 49, 49, 41, 40, 38, 34, 34, 31, 30, 29, 27, 27, 47, 49,
        50, 49, 48, 48, 40, 38, 36, 33, 33, 29, 29, 27, 26, 26, 47, 49, 50, 49,
        48, 48, 40, 38, 36, 33, 33, 29, 29, 27, 26, 26, 44, 46, 47, 46, 45, 45,
        38, 37, 34, 31, 31, 28, 27, 26, 24, 24, 43, 45, 46, 45, 44, 44, 38, 36,
        34, 31, 31, 28, 27, 26, 24, 24, 41, 43, 44, 44, 43, 43, 37, 35, 33, 30,
        30, 27, 26, 25, 23, 23, 38, 41, 41, 41, 41, 41, 36, 34, 32, 29, 29, 26,
        25, 24, 22, 22, 38, 41, 41, 41, 41, 41, 36, 34, 32, 29, 29, 26, 25, 24,
        22, 22, 36, 38, 39, 39, 39, 39, 34, 33, 31, 27, 27, 25, 24, 23, 21, 21,
        35, 37, 38, 38, 38, 38, 33, 32, 30, 27, 27, 24, 24, 23, 21, 21, 34, 36,
        36, 37, 37, 37, 33, 31, 29, 27, 27, 24, 23, 22, 20, 20, 32, 34, 34, 34,
        35, 35, 31, 30, 28, 26, 26, 23, 22, 21, 20, 20, 32, 34, 34, 34, 35, 35,
        31, 30, 28, 26, 26, 23, 22, 21, 20, 20, 29, 32, 32, 32, 33, 33, 29, 28,
        27, 25, 25, 22, 21, 20, 19, 19,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 61, 60, 59, 57, 57, 53, 52,
        50, 47, 47, 44, 43, 41, 38, 38, 36, 35, 34, 32, 32, 29, 65, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 61, 60, 59, 59, 59, 55, 54, 52, 49, 49, 46,
        45, 43, 41, 41, 38, 37, 36, 34, 34, 32, 65, 64, 64, 64, 63, 63, 63, 63,
        63, 62, 62, 61, 60, 60, 59, 59, 55, 55, 53, 50, 50, 47, 46, 44, 41, 41,
        39, 38, 36, 34, 34, 32, 65, 64, 64, 63, 63, 63, 62, 62, 61, 60, 60, 59,
        59, 58, 57, 57, 53, 53, 51, 49, 49, 46, 45, 44, 41, 41, 39, 38, 37, 34,
        34, 32, 64, 63, 63, 63, 62, 62, 60, 60, 59, 58, 58, 56, 55, 55, 53, 53,
        51, 51, 49, 48, 48, 45, 44, 43, 41, 41, 39, 38, 37, 35, 35, 33, 64, 63,
        63, 63, 62, 62, 60, 60, 59, 58, 58, 56, 55, 55, 53, 53, 51, 51, 49, 48,
        48, 45, 44, 43, 41, 41, 39, 38, 37, 35, 35, 33, 59, 59, 59, 59, 60, 60,
        58, 57, 56, 55, 55, 51, 49, 48, 45, 45, 43, 43, 41, 40, 40, 38, 38, 37,
        36, 36, 34, 33, 33, 31, 31, 29, 57, 58, 58, 59, 59, 59, 57, 56, 55, 53,
        53, 49, 48, 46, 43, 43, 41, 41, 40, 38, 38, 37, 36, 35, 34, 34, 33, 32,
        31, 30, 30, 28, 53, 54, 54, 55, 55, 55, 53, 53, 53, 51, 51, 47, 46, 44,
        41, 41, 39, 39, 38, 36, 36, 34, 34, 33, 32, 32, 31, 30, 29, 28, 28, 27,
        47, 48, 48, 49, 50, 50, 49, 49, 48, 48, 48, 44, 43, 41, 38, 38, 36, 35,
        34, 33, 33, 31, 31, 30, 29, 29, 27, 27, 27, 26, 26, 25, 47, 48, 48, 49,
        50, 50, 49, 49, 48, 48, 48, 44, 43, 41, 38, 38, 36, 35, 34, 33, 33, 31,
        31, 30, 29, 29, 27, 27, 27, 26, 26, 25, 40, 41, 41, 42, 43, 43, 43, 43,
        43, 43, 43, 39, 39, 37, 35, 35, 33, 32, 31, 29, 29, 28, 28, 27, 26, 26,
        25, 24, 24, 23, 23, 22, 38, 39, 40, 41, 41, 41, 41, 41, 41, 41, 41, 38,
        38, 36, 34, 34, 32, 32, 30, 29, 29, 27, 27, 26, 25, 25, 24, 24, 23, 22,
        22, 21, 35, 37, 37, 38, 39, 39, 39, 39, 39, 39, 39, 36, 36, 34, 33, 33,
        31, 30, 29, 27, 27, 26, 26, 25, 24, 24, 23, 23, 22, 21, 21, 20, 31, 33,
        33, 34, 34, 34, 35, 35, 35, 35, 35, 33, 33, 32, 30, 30, 28, 28, 27, 26,
        26, 24, 24, 23, 22, 22, 21, 21, 20, 20, 20, 19, 31, 33, 33, 34, 34, 34,
        35, 35, 35, 35, 35, 33, 33, 32, 30, 30, 28, 28, 27, 26, 26, 24, 24, 23,
        22, 22, 21, 21, 20, 20, 20, 19,
        /* Size 4x16 */
        65, 64, 47, 35, 64, 63, 48, 37, 64, 62, 50, 39, 63, 60, 49, 39, 63, 59,
        48, 39, 63, 58, 48, 39, 60, 55, 43, 36, 59, 53, 38, 33, 55, 51, 36, 31,
        52, 49, 34, 29, 49, 48, 33, 27, 45, 44, 31, 26, 41, 41, 29, 24, 38, 39,
        27, 23, 36, 37, 27, 22, 34, 35, 26, 21,
        /* Size 16x4 */
        65, 64, 64, 63, 63, 63, 60, 59, 55, 52, 49, 45, 41, 38, 36, 34, 64, 63,
        62, 60, 59, 58, 55, 53, 51, 49, 48, 44, 41, 39, 37, 35, 47, 48, 50, 49,
        48, 48, 43, 38, 36, 34, 33, 31, 29, 27, 27, 26, 35, 37, 39, 39, 39, 39,
        36, 33, 31, 29, 27, 26, 24, 23, 22, 21,
        /* Size 8x32 */
        64, 65, 64, 59, 53, 47, 38, 31, 65, 64, 63, 59, 54, 48, 39, 33, 65, 64,
        63, 59, 54, 48, 40, 33, 65, 64, 63, 59, 55, 49, 41, 34, 65, 63, 62, 60,
        55, 50, 41, 34, 65, 63, 62, 60, 55, 50, 41, 34, 65, 63, 60, 58, 53, 49,
        41, 35, 64, 63, 60, 57, 53, 49, 41, 35, 64, 63, 59, 56, 53, 48, 41, 35,
        64, 62, 58, 55, 51, 48, 41, 35, 64, 62, 58, 55, 51, 48, 41, 35, 61, 61,
        56, 51, 47, 44, 38, 33, 60, 60, 55, 49, 46, 43, 38, 33, 59, 60, 55, 48,
        44, 41, 36, 32, 57, 59, 53, 45, 41, 38, 34, 30, 57, 59, 53, 45, 41, 38,
        34, 30, 53, 55, 51, 43, 39, 36, 32, 28, 52, 55, 51, 43, 39, 35, 32, 28,
        50, 53, 49, 41, 38, 34, 30, 27, 47, 50, 48, 40, 36, 33, 29, 26, 47, 50,
        48, 40, 36, 33, 29, 26, 44, 47, 45, 38, 34, 31, 27, 24, 43, 46, 44, 38,
        34, 31, 27, 24, 41, 44, 43, 37, 33, 30, 26, 23, 38, 41, 41, 36, 32, 29,
        25, 22, 38, 41, 41, 36, 32, 29, 25, 22, 36, 39, 39, 34, 31, 27, 24, 21,
        35, 38, 38, 33, 30, 27, 24, 21, 34, 36, 37, 33, 29, 27, 23, 20, 32, 34,
        35, 31, 28, 26, 22, 20, 32, 34, 35, 31, 28, 26, 22, 20, 29, 32, 33, 29,
        27, 25, 21, 19,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 61, 60, 59, 57, 57, 53, 52,
        50, 47, 47, 44, 43, 41, 38, 38, 36, 35, 34, 32, 32, 29, 65, 64, 64, 64,
        63, 63, 63, 63, 63, 62, 62, 61, 60, 60, 59, 59, 55, 55, 53, 50, 50, 47,
        46, 44, 41, 41, 39, 38, 36, 34, 34, 32, 64, 63, 63, 63, 62, 62, 60, 60,
        59, 58, 58, 56, 55, 55, 53, 53, 51, 51, 49, 48, 48, 45, 44, 43, 41, 41,
        39, 38, 37, 35, 35, 33, 59, 59, 59, 59, 60, 60, 58, 57, 56, 55, 55, 51,
        49, 48, 45, 45, 43, 43, 41, 40, 40, 38, 38, 37, 36, 36, 34, 33, 33, 31,
        31, 29, 53, 54, 54, 55, 55, 55, 53, 53, 53, 51, 51, 47, 46, 44, 41, 41,
        39, 39, 38, 36, 36, 34, 34, 33, 32, 32, 31, 30, 29, 28, 28, 27, 47, 48,
        48, 49, 50, 50, 49, 49, 48, 48, 48, 44, 43, 41, 38, 38, 36, 35, 34, 33,
        33, 31, 31, 30, 29, 29, 27, 27, 27, 26, 26, 25, 38, 39, 40, 41, 41, 41,
        41, 41, 41, 41, 41, 38, 38, 36, 34, 34, 32, 32, 30, 29, 29, 27, 27, 26,
        25, 25, 24, 24, 23, 22, 22, 21, 31, 33, 33, 34, 34, 34, 35, 35, 35, 35,
        35, 33, 33, 32, 30, 30, 28, 28, 27, 26, 26, 24, 24, 23, 22, 22, 21, 21,
        20, 20, 20, 19 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        53, 44, 44, 32,
#endif
        /* Size 4x4 */
        65, 49, 45, 40, 49, 43, 42, 40, 45, 42, 35, 33, 40, 40, 33, 29,
        /* Size 8x8 */
        65, 66, 54, 46, 43, 42, 41, 37, 66, 64, 51, 47, 45, 45, 43, 39, 54, 51,
        43, 43, 44, 45, 44, 41, 46, 47, 43, 41, 40, 40, 39, 38, 43, 45, 44, 40,
        38, 36, 36, 34, 42, 45, 45, 40, 36, 34, 33, 31, 41, 43, 44, 39, 36, 33,
        31, 29, 37, 39, 41, 38, 34, 31, 29, 27,
        /* Size 16x16 */
        64, 66, 67, 62, 59, 56, 49, 42, 42, 42, 42, 41, 39, 38, 37, 36, 66, 65,
        66, 60, 57, 53, 48, 43, 44, 44, 44, 42, 41, 40, 39, 38, 67, 66, 64, 59,
        55, 51, 48, 44, 45, 45, 46, 44, 43, 41, 41, 39, 62, 60, 59, 55, 51, 48,
        46, 44, 44, 45, 45, 44, 43, 42, 41, 40, 59, 57, 55, 51, 48, 46, 45, 43,
        44, 45, 45, 44, 44, 42, 41, 41, 56, 53, 51, 48, 46, 43, 43, 43, 44, 45,
        45, 45, 44, 43, 42, 41, 49, 48, 48, 46, 45, 43, 42, 41, 41, 41, 42, 41,
        41, 40, 39, 38, 42, 43, 44, 44, 43, 43, 41, 39, 39, 38, 38, 38, 38, 37,
        37, 36, 42, 44, 45, 44, 44, 44, 41, 39, 38, 38, 37, 37, 36, 35, 35, 35,
        42, 44, 45, 45, 45, 45, 41, 38, 38, 37, 36, 35, 35, 34, 34, 33, 42, 44,
        46, 45, 45, 45, 42, 38, 37, 36, 35, 34, 33, 33, 33, 32, 41, 42, 44, 44,
        44, 45, 41, 38, 37, 35, 34, 33, 32, 32, 31, 31, 39, 41, 43, 43, 44, 44,
        41, 38, 36, 35, 33, 32, 31, 30, 29, 29, 38, 40, 41, 42, 42, 43, 40, 37,
        35, 34, 33, 32, 30, 29, 29, 28, 37, 39, 41, 41, 41, 42, 39, 37, 35, 34,
        33, 31, 29, 29, 28, 27, 36, 38, 39, 40, 41, 41, 38, 36, 35, 33, 32, 31,
        29, 28, 27, 27,
        /* Size 32x32 */
        64, 65, 66, 66, 67, 67, 62, 61, 59, 56, 56, 51, 49, 46, 42, 42, 42, 42,
        42, 42, 42, 41, 41, 40, 39, 39, 38, 38, 37, 36, 36, 34, 65, 65, 65, 66,
        66, 66, 61, 59, 57, 54, 54, 49, 49, 46, 43, 43, 43, 43, 43, 43, 43, 42,
        42, 41, 41, 41, 39, 39, 38, 37, 37, 36, 66, 65, 65, 65, 66, 66, 60, 59,
        57, 53, 53, 49, 48, 46, 43, 43, 44, 44, 44, 44, 44, 43, 42, 42, 41, 41,
        40, 39, 39, 38, 38, 36, 66, 66, 65, 65, 65, 65, 60, 59, 56, 53, 53, 49,
        48, 47, 44, 44, 44, 44, 44, 45, 45, 43, 43, 43, 42, 42, 41, 40, 39, 38,
        38, 37, 67, 66, 66, 65, 64, 64, 59, 58, 55, 51, 51, 49, 48, 47, 44, 44,
        45, 45, 45, 46, 46, 45, 44, 44, 43, 43, 41, 41, 41, 39, 39, 38, 67, 66,
        66, 65, 64, 64, 59, 58, 55, 51, 51, 49, 48, 47, 44, 44, 45, 45, 45, 46,
        46, 45, 44, 44, 43, 43, 41, 41, 41, 39, 39, 38, 62, 61, 60, 60, 59, 59,
        55, 53, 51, 48, 48, 47, 46, 45, 44, 44, 44, 45, 45, 45, 45, 44, 44, 44,
        43, 43, 42, 41, 41, 40, 40, 39, 61, 59, 59, 59, 58, 58, 53, 53, 51, 47,
        47, 46, 46, 45, 43, 43, 44, 44, 45, 45, 45, 44, 44, 44, 43, 43, 42, 41,
        41, 40, 40, 39, 59, 57, 57, 56, 55, 55, 51, 51, 48, 46, 46, 45, 45, 44,
        43, 43, 44, 44, 45, 45, 45, 45, 44, 44, 44, 44, 42, 42, 41, 41, 41, 39,
        56, 54, 53, 53, 51, 51, 48, 47, 46, 43, 43, 43, 43, 43, 43, 43, 44, 44,
        45, 45, 45, 45, 45, 44, 44, 44, 43, 42, 42, 41, 41, 40, 56, 54, 53, 53,
        51, 51, 48, 47, 46, 43, 43, 43, 43, 43, 43, 43, 44, 44, 45, 45, 45, 45,
        45, 44, 44, 44, 43, 42, 42, 41, 41, 40, 51, 49, 49, 49, 49, 49, 47, 46,
        45, 43, 43, 43, 42, 42, 41, 41, 42, 42, 42, 42, 42, 42, 42, 41, 41, 41,
        40, 40, 40, 39, 39, 38, 49, 49, 48, 48, 48, 48, 46, 46, 45, 43, 43, 42,
        42, 41, 41, 41, 41, 41, 41, 42, 42, 41, 41, 41, 41, 41, 40, 39, 39, 38,
        38, 37, 46, 46, 46, 47, 47, 47, 45, 45, 44, 43, 43, 42, 41, 41, 40, 40,
        40, 40, 40, 40, 40, 40, 40, 40, 39, 39, 39, 38, 38, 38, 38, 37, 42, 43,
        43, 44, 44, 44, 44, 43, 43, 43, 43, 41, 41, 40, 39, 39, 39, 39, 38, 38,
        38, 38, 38, 38, 38, 38, 37, 37, 37, 36, 36, 35, 42, 43, 43, 44, 44, 44,
        44, 43, 43, 43, 43, 41, 41, 40, 39, 39, 39, 39, 38, 38, 38, 38, 38, 38,
        38, 38, 37, 37, 37, 36, 36, 35, 42, 43, 44, 44, 45, 45, 44, 44, 44, 44,
        44, 42, 41, 40, 39, 39, 38, 38, 38, 37, 37, 37, 37, 36, 36, 36, 35, 35,
        35, 35, 35, 34, 42, 43, 44, 44, 45, 45, 45, 44, 44, 44, 44, 42, 41, 40,
        39, 39, 38, 38, 37, 37, 37, 36, 36, 36, 36, 36, 35, 35, 35, 34, 34, 33,
        42, 43, 44, 44, 45, 45, 45, 45, 45, 45, 45, 42, 41, 40, 38, 38, 38, 37,
        37, 36, 36, 36, 35, 35, 35, 35, 34, 34, 34, 33, 33, 33, 42, 43, 44, 45,
        46, 46, 45, 45, 45, 45, 45, 42, 42, 40, 38, 38, 37, 37, 36, 35, 35, 34,
        34, 34, 33, 33, 33, 33, 33, 32, 32, 32, 42, 43, 44, 45, 46, 46, 45, 45,
        45, 45, 45, 42, 42, 40, 38, 38, 37, 37, 36, 35, 35, 34, 34, 34, 33, 33,
        33, 33, 33, 32, 32, 32, 41, 42, 43, 43, 45, 45, 44, 44, 45, 45, 45, 42,
        41, 40, 38, 38, 37, 36, 36, 34, 34, 34, 34, 33, 33, 33, 32, 32, 31, 31,
        31, 30, 41, 42, 42, 43, 44, 44, 44, 44, 44, 45, 45, 42, 41, 40, 38, 38,
        37, 36, 35, 34, 34, 34, 33, 33, 32, 32, 32, 32, 31, 31, 31, 30, 40, 41,
        42, 43, 44, 44, 44, 44, 44, 44, 44, 41, 41, 40, 38, 38, 36, 36, 35, 34,
        34, 33, 33, 32, 32, 32, 31, 31, 31, 30, 30, 29, 39, 41, 41, 42, 43, 43,
        43, 43, 44, 44, 44, 41, 41, 39, 38, 38, 36, 36, 35, 33, 33, 33, 32, 32,
        31, 31, 30, 30, 29, 29, 29, 28, 39, 41, 41, 42, 43, 43, 43, 43, 44, 44,
        44, 41, 41, 39, 38, 38, 36, 36, 35, 33, 33, 33, 32, 32, 31, 31, 30, 30,
        29, 29, 29, 28, 38, 39, 40, 41, 41, 41, 42, 42, 42, 43, 43, 40, 40, 39,
        37, 37, 35, 35, 34, 33, 33, 32, 32, 31, 30, 30, 29, 29, 29, 28, 28, 27,
        38, 39, 39, 40, 41, 41, 41, 41, 42, 42, 42, 40, 39, 38, 37, 37, 35, 35,
        34, 33, 33, 32, 32, 31, 30, 30, 29, 29, 28, 28, 28, 27, 37, 38, 39, 39,
        41, 41, 41, 41, 41, 42, 42, 40, 39, 38, 37, 37, 35, 35, 34, 33, 33, 31,
        31, 31, 29, 29, 29, 28, 28, 27, 27, 27, 36, 37, 38, 38, 39, 39, 40, 40,
        41, 41, 41, 39, 38, 38, 36, 36, 35, 34, 33, 32, 32, 31, 31, 30, 29, 29,
        28, 28, 27, 27, 27, 26, 36, 37, 38, 38, 39, 39, 40, 40, 41, 41, 41, 39,
        38, 38, 36, 36, 35, 34, 33, 32, 32, 31, 31, 30, 29, 29, 28, 28, 27, 27,
        27, 26, 34, 36, 36, 37, 38, 38, 39, 39, 39, 40, 40, 38, 37, 37, 35, 35,
        34, 33, 33, 32, 32, 30, 30, 29, 28, 28, 27, 27, 27, 26, 26, 25,
        /* Size 4x8 */
        65, 54, 43, 39, 64, 51, 45, 41, 52, 44, 45, 43, 46, 43, 40, 39, 44, 44,
        37, 35, 44, 44, 34, 32, 42, 43, 34, 30, 39, 41, 32, 28,
        /* Size 8x4 */
        65, 64, 52, 46, 44, 44, 42, 39, 54, 51, 44, 43, 44, 44, 43, 41, 43, 45,
        45, 40, 37, 34, 34, 32, 39, 41, 43, 39, 35, 32, 30, 28,
        /* Size 8x16 */
        64, 66, 55, 45, 42, 42, 39, 36, 65, 65, 53, 46, 44, 44, 41, 38, 67, 64,
        51, 46, 45, 45, 43, 39, 62, 59, 48, 45, 44, 45, 43, 40, 59, 55, 46, 44,
        44, 45, 43, 40, 55, 51, 44, 43, 44, 45, 44, 41, 49, 48, 43, 41, 41, 41,
        41, 38, 42, 44, 43, 39, 39, 38, 38, 36, 42, 45, 44, 40, 38, 37, 36, 34,
        42, 45, 44, 40, 38, 36, 35, 33, 42, 45, 45, 40, 37, 35, 33, 32, 41, 44,
        44, 39, 37, 34, 32, 31, 39, 43, 43, 39, 36, 34, 31, 29, 38, 41, 42, 38,
        35, 33, 30, 28, 37, 40, 41, 38, 35, 33, 29, 27, 36, 39, 41, 37, 34, 32,
        29, 27,
        /* Size 16x8 */
        64, 65, 67, 62, 59, 55, 49, 42, 42, 42, 42, 41, 39, 38, 37, 36, 66, 65,
        64, 59, 55, 51, 48, 44, 45, 45, 45, 44, 43, 41, 40, 39, 55, 53, 51, 48,
        46, 44, 43, 43, 44, 44, 45, 44, 43, 42, 41, 41, 45, 46, 46, 45, 44, 43,
        41, 39, 40, 40, 40, 39, 39, 38, 38, 37, 42, 44, 45, 44, 44, 44, 41, 39,
        38, 38, 37, 37, 36, 35, 35, 34, 42, 44, 45, 45, 45, 45, 41, 38, 37, 36,
        35, 34, 34, 33, 33, 32, 39, 41, 43, 43, 43, 44, 41, 38, 36, 35, 33, 32,
        31, 30, 29, 29, 36, 38, 39, 40, 40, 41, 38, 36, 34, 33, 32, 31, 29, 28,
        27, 27,
        /* Size 16x32 */
        64, 66, 66, 62, 55, 55, 45, 43, 42, 42, 42, 40, 39, 38, 36, 36, 65, 65,
        65, 60, 54, 54, 46, 43, 43, 43, 43, 41, 41, 39, 37, 37, 65, 65, 65, 60,
        53, 53, 46, 44, 44, 44, 44, 41, 41, 40, 38, 38, 66, 65, 64, 59, 53, 53,
        46, 44, 44, 44, 44, 42, 42, 40, 38, 38, 67, 64, 64, 59, 51, 51, 46, 45,
        45, 45, 45, 43, 43, 41, 39, 39, 67, 64, 64, 59, 51, 51, 46, 45, 45, 45,
        45, 43, 43, 41, 39, 39, 62, 59, 59, 55, 48, 48, 45, 44, 44, 45, 45, 43,
        43, 42, 40, 40, 61, 58, 57, 53, 48, 48, 45, 44, 44, 45, 45, 43, 43, 42,
        40, 40, 59, 55, 55, 51, 46, 46, 44, 43, 44, 45, 45, 44, 43, 42, 40, 40,
        55, 52, 51, 48, 44, 44, 43, 43, 44, 45, 45, 44, 44, 43, 41, 41, 55, 52,
        51, 48, 44, 44, 43, 43, 44, 45, 45, 44, 44, 43, 41, 41, 50, 49, 48, 47,
        43, 43, 42, 41, 42, 42, 42, 41, 41, 40, 39, 39, 49, 48, 48, 46, 43, 43,
        41, 41, 41, 41, 41, 41, 41, 40, 38, 38, 46, 46, 46, 45, 43, 43, 41, 40,
        40, 40, 40, 39, 39, 39, 38, 38, 42, 44, 44, 44, 43, 43, 39, 39, 39, 38,
        38, 38, 38, 37, 36, 36, 42, 44, 44, 44, 43, 43, 39, 39, 39, 38, 38, 38,
        38, 37, 36, 36, 42, 44, 45, 44, 44, 44, 40, 39, 38, 37, 37, 36, 36, 35,
        34, 34, 42, 44, 45, 44, 44, 44, 40, 39, 38, 37, 37, 36, 36, 35, 34, 34,
        42, 45, 45, 45, 44, 44, 40, 39, 38, 36, 36, 35, 35, 34, 33, 33, 42, 45,
        45, 45, 45, 45, 40, 38, 37, 35, 35, 34, 33, 33, 32, 32, 42, 45, 45, 45,
        45, 45, 40, 38, 37, 35, 35, 34, 33, 33, 32, 32, 41, 44, 44, 44, 44, 44,
        39, 38, 37, 34, 34, 33, 32, 32, 31, 31, 41, 43, 44, 44, 44, 44, 39, 38,
        37, 34, 34, 33, 32, 32, 31, 31, 40, 43, 44, 44, 44, 44, 39, 38, 36, 34,
        34, 32, 32, 31, 30, 30, 39, 42, 43, 43, 43, 43, 39, 38, 36, 34, 34, 31,
        31, 30, 29, 29, 39, 42, 43, 43, 43, 43, 39, 38, 36, 34, 34, 31, 31, 30,
        29, 29, 38, 41, 41, 42, 42, 42, 38, 37, 35, 33, 33, 31, 30, 29, 28, 28,
        38, 40, 41, 41, 42, 42, 38, 37, 35, 33, 33, 31, 30, 29, 28, 28, 37, 40,
        40, 41, 41, 41, 38, 37, 35, 33, 33, 30, 29, 29, 27, 27, 36, 39, 39, 40,
        41, 41, 37, 36, 34, 32, 32, 30, 29, 28, 27, 27, 36, 39, 39, 40, 41, 41,
        37, 36, 34, 32, 32, 30, 29, 28, 27, 27, 35, 37, 38, 38, 39, 39, 36, 35,
        34, 32, 32, 29, 28, 27, 26, 26,
        /* Size 32x16 */
        64, 65, 65, 66, 67, 67, 62, 61, 59, 55, 55, 50, 49, 46, 42, 42, 42, 42,
        42, 42, 42, 41, 41, 40, 39, 39, 38, 38, 37, 36, 36, 35, 66, 65, 65, 65,
        64, 64, 59, 58, 55, 52, 52, 49, 48, 46, 44, 44, 44, 44, 45, 45, 45, 44,
        43, 43, 42, 42, 41, 40, 40, 39, 39, 37, 66, 65, 65, 64, 64, 64, 59, 57,
        55, 51, 51, 48, 48, 46, 44, 44, 45, 45, 45, 45, 45, 44, 44, 44, 43, 43,
        41, 41, 40, 39, 39, 38, 62, 60, 60, 59, 59, 59, 55, 53, 51, 48, 48, 47,
        46, 45, 44, 44, 44, 44, 45, 45, 45, 44, 44, 44, 43, 43, 42, 41, 41, 40,
        40, 38, 55, 54, 53, 53, 51, 51, 48, 48, 46, 44, 44, 43, 43, 43, 43, 43,
        44, 44, 44, 45, 45, 44, 44, 44, 43, 43, 42, 42, 41, 41, 41, 39, 55, 54,
        53, 53, 51, 51, 48, 48, 46, 44, 44, 43, 43, 43, 43, 43, 44, 44, 44, 45,
        45, 44, 44, 44, 43, 43, 42, 42, 41, 41, 41, 39, 45, 46, 46, 46, 46, 46,
        45, 45, 44, 43, 43, 42, 41, 41, 39, 39, 40, 40, 40, 40, 40, 39, 39, 39,
        39, 39, 38, 38, 38, 37, 37, 36, 43, 43, 44, 44, 45, 45, 44, 44, 43, 43,
        43, 41, 41, 40, 39, 39, 39, 39, 39, 38, 38, 38, 38, 38, 38, 38, 37, 37,
        37, 36, 36, 35, 42, 43, 44, 44, 45, 45, 44, 44, 44, 44, 44, 42, 41, 40,
        39, 39, 38, 38, 38, 37, 37, 37, 37, 36, 36, 36, 35, 35, 35, 34, 34, 34,
        42, 43, 44, 44, 45, 45, 45, 45, 45, 45, 45, 42, 41, 40, 38, 38, 37, 37,
        36, 35, 35, 34, 34, 34, 34, 34, 33, 33, 33, 32, 32, 32, 42, 43, 44, 44,
        45, 45, 45, 45, 45, 45, 45, 42, 41, 40, 38, 38, 37, 37, 36, 35, 35, 34,
        34, 34, 34, 34, 33, 33, 33, 32, 32, 32, 40, 41, 41, 42, 43, 43, 43, 43,
        44, 44, 44, 41, 41, 39, 38, 38, 36, 36, 35, 34, 34, 33, 33, 32, 31, 31,
        31, 31, 30, 30, 30, 29, 39, 41, 41, 42, 43, 43, 43, 43, 43, 44, 44, 41,
        41, 39, 38, 38, 36, 36, 35, 33, 33, 32, 32, 32, 31, 31, 30, 30, 29, 29,
        29, 28, 38, 39, 40, 40, 41, 41, 42, 42, 42, 43, 43, 40, 40, 39, 37, 37,
        35, 35, 34, 33, 33, 32, 32, 31, 30, 30, 29, 29, 29, 28, 28, 27, 36, 37,
        38, 38, 39, 39, 40, 40, 40, 41, 41, 39, 38, 38, 36, 36, 34, 34, 33, 32,
        32, 31, 31, 30, 29, 29, 28, 28, 27, 27, 27, 26, 36, 37, 38, 38, 39, 39,
        40, 40, 40, 41, 41, 39, 38, 38, 36, 36, 34, 34, 33, 32, 32, 31, 31, 30,
        29, 29, 28, 28, 27, 27, 27, 26,
        /* Size 4x16 */
        66, 55, 42, 38, 65, 53, 44, 40, 64, 51, 45, 41, 59, 48, 45, 42, 55, 46,
        45, 42, 52, 44, 45, 43, 48, 43, 41, 40, 44, 43, 38, 37, 44, 44, 37, 35,
        45, 44, 36, 34, 45, 45, 35, 33, 43, 44, 34, 32, 42, 43, 34, 30, 41, 42,
        33, 29, 40, 41, 33, 29, 39, 41, 32, 28,
        /* Size 16x4 */
        66, 65, 64, 59, 55, 52, 48, 44, 44, 45, 45, 43, 42, 41, 40, 39, 55, 53,
        51, 48, 46, 44, 43, 43, 44, 44, 45, 44, 43, 42, 41, 41, 42, 44, 45, 45,
        45, 45, 41, 38, 37, 36, 35, 34, 34, 33, 33, 32, 38, 40, 41, 42, 42, 43,
        40, 37, 35, 34, 33, 32, 30, 29, 29, 28,
        /* Size 8x32 */
        64, 66, 55, 45, 42, 42, 39, 36, 65, 65, 54, 46, 43, 43, 41, 37, 65, 65,
        53, 46, 44, 44, 41, 38, 66, 64, 53, 46, 44, 44, 42, 38, 67, 64, 51, 46,
        45, 45, 43, 39, 67, 64, 51, 46, 45, 45, 43, 39, 62, 59, 48, 45, 44, 45,
        43, 40, 61, 57, 48, 45, 44, 45, 43, 40, 59, 55, 46, 44, 44, 45, 43, 40,
        55, 51, 44, 43, 44, 45, 44, 41, 55, 51, 44, 43, 44, 45, 44, 41, 50, 48,
        43, 42, 42, 42, 41, 39, 49, 48, 43, 41, 41, 41, 41, 38, 46, 46, 43, 41,
        40, 40, 39, 38, 42, 44, 43, 39, 39, 38, 38, 36, 42, 44, 43, 39, 39, 38,
        38, 36, 42, 45, 44, 40, 38, 37, 36, 34, 42, 45, 44, 40, 38, 37, 36, 34,
        42, 45, 44, 40, 38, 36, 35, 33, 42, 45, 45, 40, 37, 35, 33, 32, 42, 45,
        45, 40, 37, 35, 33, 32, 41, 44, 44, 39, 37, 34, 32, 31, 41, 44, 44, 39,
        37, 34, 32, 31, 40, 44, 44, 39, 36, 34, 32, 30, 39, 43, 43, 39, 36, 34,
        31, 29, 39, 43, 43, 39, 36, 34, 31, 29, 38, 41, 42, 38, 35, 33, 30, 28,
        38, 41, 42, 38, 35, 33, 30, 28, 37, 40, 41, 38, 35, 33, 29, 27, 36, 39,
        41, 37, 34, 32, 29, 27, 36, 39, 41, 37, 34, 32, 29, 27, 35, 38, 39, 36,
        34, 32, 28, 26,
        /* Size 32x8 */
        64, 65, 65, 66, 67, 67, 62, 61, 59, 55, 55, 50, 49, 46, 42, 42, 42, 42,
        42, 42, 42, 41, 41, 40, 39, 39, 38, 38, 37, 36, 36, 35, 66, 65, 65, 64,
        64, 64, 59, 57, 55, 51, 51, 48, 48, 46, 44, 44, 45, 45, 45, 45, 45, 44,
        44, 44, 43, 43, 41, 41, 40, 39, 39, 38, 55, 54, 53, 53, 51, 51, 48, 48,
        46, 44, 44, 43, 43, 43, 43, 43, 44, 44, 44, 45, 45, 44, 44, 44, 43, 43,
        42, 42, 41, 41, 41, 39, 45, 46, 46, 46, 46, 46, 45, 45, 44, 43, 43, 42,
        41, 41, 39, 39, 40, 40, 40, 40, 40, 39, 39, 39, 39, 39, 38, 38, 38, 37,
        37, 36, 42, 43, 44, 44, 45, 45, 44, 44, 44, 44, 44, 42, 41, 40, 39, 39,
        38, 38, 38, 37, 37, 37, 37, 36, 36, 36, 35, 35, 35, 34, 34, 34, 42, 43,
        44, 44, 45, 45, 45, 45, 45, 45, 45, 42, 41, 40, 38, 38, 37, 37, 36, 35,
        35, 34, 34, 34, 34, 34, 33, 33, 33, 32, 32, 32, 39, 41, 41, 42, 43, 43,
        43, 43, 43, 44, 44, 41, 41, 39, 38, 38, 36, 36, 35, 33, 33, 32, 32, 32,
        31, 31, 30, 30, 29, 29, 29, 28, 36, 37, 38, 38, 39, 39, 40, 40, 40, 41,
        41, 39, 38, 38, 36, 36, 34, 34, 33, 32, 32, 31, 31, 30, 29, 29, 28, 28,
        27, 27, 27, 26 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        62, 48, 48, 32,
#endif
        /* Size 4x4 */
        64, 63, 53, 40, 63, 58, 51, 42, 53, 51, 38, 32, 40, 42, 32, 25,
        /* Size 8x8 */
        65, 64, 64, 60, 58, 49, 44, 38, 64, 63, 63, 61, 59, 51, 46, 41, 64, 63,
        60, 58, 55, 49, 45, 40, 60, 61, 58, 52, 48, 44, 40, 37, 58, 59, 55, 48,
        42, 39, 36, 34, 49, 51, 49, 44, 39, 34, 32, 29, 44, 46, 45, 40, 36, 32,
        29, 26, 38, 41, 40, 37, 34, 29, 26, 24,
        /* Size 16x16 */
        64, 65, 65, 65, 65, 64, 63, 60, 57, 54, 52, 47, 43, 41, 38, 35, 65, 64,
        64, 64, 64, 63, 62, 60, 59, 55, 53, 48, 45, 43, 40, 36, 65, 64, 64, 63,
        63, 63, 62, 61, 59, 56, 55, 50, 47, 45, 41, 38, 65, 64, 63, 63, 63, 62,
        61, 60, 59, 56, 54, 50, 47, 45, 41, 38, 65, 64, 63, 63, 61, 60, 59, 58,
        56, 54, 53, 49, 46, 44, 41, 38, 64, 63, 63, 62, 60, 58, 57, 55, 53, 52,
        51, 48, 46, 44, 41, 38, 63, 62, 62, 61, 59, 57, 56, 54, 51, 50, 49, 46,
        44, 42, 40, 37, 60, 60, 61, 60, 58, 55, 54, 52, 48, 47, 46, 43, 41, 40,
        38, 35, 57, 59, 59, 59, 56, 53, 51, 48, 42, 41, 41, 38, 37, 36, 34, 32,
        54, 55, 56, 56, 54, 52, 50, 47, 41, 40, 39, 37, 35, 34, 33, 31, 52, 53,
        55, 54, 53, 51, 49, 46, 41, 39, 38, 35, 34, 33, 32, 30, 47, 48, 50, 50,
        49, 48, 46, 43, 38, 37, 35, 32, 31, 30, 29, 27, 43, 45, 47, 47, 46, 46,
        44, 41, 37, 35, 34, 31, 29, 29, 27, 26, 41, 43, 45, 45, 44, 44, 42, 40,
        36, 34, 33, 30, 29, 28, 26, 25, 38, 40, 41, 41, 41, 41, 40, 38, 34, 33,
        32, 29, 27, 26, 25, 24, 35, 36, 38, 38, 38, 38, 37, 35, 32, 31, 30, 27,
        26, 25, 24, 22,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 63, 60, 60, 59, 57, 57,
        54, 52, 52, 48, 47, 46, 43, 42, 41, 38, 38, 37, 35, 35, 65, 65, 65, 65,
        64, 64, 64, 64, 64, 64, 64, 64, 62, 60, 60, 59, 58, 58, 55, 53, 53, 49,
        48, 47, 45, 44, 43, 40, 39, 38, 36, 36, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 62, 60, 60, 59, 59, 59, 55, 53, 53, 50, 48, 48, 45, 44,
        43, 40, 40, 39, 36, 36, 65, 65, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63,
        62, 60, 60, 59, 59, 59, 55, 53, 53, 50, 49, 48, 45, 44, 44, 40, 40, 39,
        37, 37, 65, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 62, 61, 61, 60,
        59, 59, 56, 55, 55, 51, 50, 49, 47, 46, 45, 41, 41, 40, 38, 38, 65, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 61, 61, 60, 59, 59, 57, 55,
        55, 51, 50, 50, 47, 46, 45, 42, 41, 41, 38, 38, 65, 64, 64, 64, 63, 63,
        63, 63, 63, 62, 62, 62, 61, 60, 60, 59, 59, 59, 56, 54, 54, 51, 50, 49,
        47, 46, 45, 42, 41, 41, 38, 38, 65, 64, 64, 64, 63, 63, 63, 62, 62, 61,
        60, 60, 59, 59, 59, 58, 56, 56, 54, 53, 53, 50, 49, 48, 46, 45, 44, 41,
        41, 40, 38, 38, 65, 64, 64, 64, 63, 63, 63, 62, 61, 61, 60, 60, 59, 58,
        58, 57, 56, 56, 54, 53, 53, 50, 49, 48, 46, 45, 44, 41, 41, 40, 38, 38,
        64, 64, 64, 63, 63, 63, 62, 61, 61, 60, 59, 59, 59, 58, 58, 56, 55, 55,
        53, 52, 52, 49, 49, 48, 46, 45, 44, 41, 41, 40, 38, 38, 64, 64, 63, 63,
        63, 63, 62, 60, 60, 59, 58, 58, 57, 55, 55, 55, 53, 53, 52, 51, 51, 49,
        48, 48, 46, 45, 44, 42, 41, 41, 38, 38, 64, 64, 63, 63, 63, 63, 62, 60,
        60, 59, 58, 58, 57, 55, 55, 55, 53, 53, 52, 51, 51, 49, 48, 48, 46, 45,
        44, 42, 41, 41, 38, 38, 63, 62, 62, 62, 62, 62, 61, 59, 59, 59, 57, 57,
        56, 54, 54, 53, 51, 51, 50, 49, 49, 47, 46, 46, 44, 43, 42, 40, 40, 39,
        37, 37, 60, 60, 60, 60, 61, 61, 60, 59, 58, 58, 55, 55, 54, 52, 52, 50,
        48, 48, 47, 46, 46, 44, 43, 43, 41, 40, 40, 38, 38, 37, 35, 35, 60, 60,
        60, 60, 61, 61, 60, 59, 58, 58, 55, 55, 54, 52, 52, 50, 48, 48, 47, 46,
        46, 44, 43, 43, 41, 40, 40, 38, 38, 37, 35, 35, 59, 59, 59, 59, 60, 60,
        59, 58, 57, 56, 55, 55, 53, 50, 50, 48, 45, 45, 44, 43, 43, 41, 41, 40,
        39, 38, 38, 36, 36, 35, 34, 34, 57, 58, 59, 59, 59, 59, 59, 56, 56, 55,
        53, 53, 51, 48, 48, 45, 42, 42, 41, 41, 41, 39, 38, 38, 37, 36, 36, 34,
        34, 34, 32, 32, 57, 58, 59, 59, 59, 59, 59, 56, 56, 55, 53, 53, 51, 48,
        48, 45, 42, 42, 41, 41, 41, 39, 38, 38, 37, 36, 36, 34, 34, 34, 32, 32,
        54, 55, 55, 55, 56, 57, 56, 54, 54, 53, 52, 52, 50, 47, 47, 44, 41, 41,
        40, 39, 39, 37, 37, 36, 35, 34, 34, 33, 33, 32, 31, 31, 52, 53, 53, 53,
        55, 55, 54, 53, 53, 52, 51, 51, 49, 46, 46, 43, 41, 41, 39, 38, 38, 36,
        35, 35, 34, 34, 33, 32, 32, 31, 30, 30, 52, 53, 53, 53, 55, 55, 54, 53,
        53, 52, 51, 51, 49, 46, 46, 43, 41, 41, 39, 38, 38, 36, 35, 35, 34, 34,
        33, 32, 32, 31, 30, 30, 48, 49, 50, 50, 51, 51, 51, 50, 50, 49, 49, 49,
        47, 44, 44, 41, 39, 39, 37, 36, 36, 34, 33, 33, 32, 32, 31, 30, 29, 29,
        28, 28, 47, 48, 48, 49, 50, 50, 50, 49, 49, 49, 48, 48, 46, 43, 43, 41,
        38, 38, 37, 35, 35, 33, 32, 32, 31, 31, 30, 29, 29, 28, 27, 27, 46, 47,
        48, 48, 49, 50, 49, 48, 48, 48, 48, 48, 46, 43, 43, 40, 38, 38, 36, 35,
        35, 33, 32, 32, 31, 30, 30, 29, 28, 28, 27, 27, 43, 45, 45, 45, 47, 47,
        47, 46, 46, 46, 46, 46, 44, 41, 41, 39, 37, 37, 35, 34, 34, 32, 31, 31,
        29, 29, 29, 27, 27, 27, 26, 26, 42, 44, 44, 44, 46, 46, 46, 45, 45, 45,
        45, 45, 43, 40, 40, 38, 36, 36, 34, 34, 34, 32, 31, 30, 29, 29, 28, 27,
        27, 26, 25, 25, 41, 43, 43, 44, 45, 45, 45, 44, 44, 44, 44, 44, 42, 40,
        40, 38, 36, 36, 34, 33, 33, 31, 30, 30, 29, 28, 28, 27, 26, 26, 25, 25,
        38, 40, 40, 40, 41, 42, 42, 41, 41, 41, 42, 42, 40, 38, 38, 36, 34, 34,
        33, 32, 32, 30, 29, 29, 27, 27, 27, 25, 25, 25, 24, 24, 38, 39, 40, 40,
        41, 41, 41, 41, 41, 41, 41, 41, 40, 38, 38, 36, 34, 34, 33, 32, 32, 29,
        29, 28, 27, 27, 26, 25, 25, 25, 24, 24, 37, 38, 39, 39, 40, 41, 41, 40,
        40, 40, 41, 41, 39, 37, 37, 35, 34, 34, 32, 31, 31, 29, 28, 28, 27, 26,
        26, 25, 25, 24, 23, 23, 35, 36, 36, 37, 38, 38, 38, 38, 38, 38, 38, 38,
        37, 35, 35, 34, 32, 32, 31, 30, 30, 28, 27, 27, 26, 25, 25, 24, 24, 23,
        22, 22, 35, 36, 36, 37, 38, 38, 38, 38, 38, 38, 38, 38, 37, 35, 35, 34,
        32, 32, 31, 30, 30, 28, 27, 27, 26, 25, 25, 24, 24, 23, 22, 22,
        /* Size 4x8 */
        64, 63, 55, 39, 64, 62, 56, 41, 63, 59, 53, 41, 60, 55, 47, 38, 59, 53,
        41, 34, 51, 48, 37, 29, 45, 44, 34, 27, 39, 40, 32, 25,
        /* Size 8x4 */
        64, 64, 63, 60, 59, 51, 45, 39, 63, 62, 59, 55, 53, 48, 44, 40, 55, 56,
        53, 47, 41, 37, 34, 32, 39, 41, 41, 38, 34, 29, 27, 25,
        /* Size 8x16 */
        64, 65, 64, 63, 57, 47, 43, 38, 65, 64, 63, 62, 58, 48, 45, 40, 65, 63,
        63, 62, 59, 50, 47, 41, 65, 63, 62, 61, 59, 50, 47, 41, 64, 63, 60, 59,
        56, 49, 46, 41, 64, 62, 59, 57, 53, 48, 46, 41, 63, 62, 58, 56, 51, 46,
        44, 40, 60, 60, 57, 54, 48, 43, 41, 38, 57, 59, 55, 51, 43, 38, 37, 34,
        54, 56, 53, 49, 41, 37, 35, 33, 52, 55, 51, 48, 41, 35, 34, 32, 47, 50,
        48, 46, 38, 33, 31, 29, 44, 47, 45, 43, 37, 31, 29, 27, 42, 45, 44, 42,
        36, 30, 29, 26, 38, 41, 41, 40, 34, 29, 27, 25, 35, 38, 38, 37, 32, 27,
        26, 24,
        /* Size 16x8 */
        64, 65, 65, 65, 64, 64, 63, 60, 57, 54, 52, 47, 44, 42, 38, 35, 65, 64,
        63, 63, 63, 62, 62, 60, 59, 56, 55, 50, 47, 45, 41, 38, 64, 63, 63, 62,
        60, 59, 58, 57, 55, 53, 51, 48, 45, 44, 41, 38, 63, 62, 62, 61, 59, 57,
        56, 54, 51, 49, 48, 46, 43, 42, 40, 37, 57, 58, 59, 59, 56, 53, 51, 48,
        43, 41, 41, 38, 37, 36, 34, 32, 47, 48, 50, 50, 49, 48, 46, 43, 38, 37,
        35, 33, 31, 30, 29, 27, 43, 45, 47, 47, 46, 46, 44, 41, 37, 35, 34, 31,
        29, 29, 27, 26, 38, 40, 41, 41, 41, 41, 40, 38, 34, 33, 32, 29, 27, 26,
        25, 24,
        /* Size 16x32 */
        64, 65, 65, 65, 64, 64, 63, 58, 57, 54, 47, 47, 43, 38, 38, 35, 65, 64,
        64, 64, 64, 63, 62, 59, 58, 55, 48, 48, 45, 39, 39, 36, 65, 64, 64, 64,
        63, 63, 62, 59, 58, 55, 48, 48, 45, 40, 40, 36, 65, 64, 64, 64, 63, 63,
        62, 59, 58, 55, 49, 49, 45, 40, 40, 37, 65, 64, 63, 63, 63, 63, 62, 59,
        59, 56, 50, 50, 47, 41, 41, 38, 65, 64, 63, 63, 63, 62, 62, 59, 59, 56,
        50, 50, 47, 41, 41, 38, 65, 64, 63, 63, 62, 62, 61, 59, 59, 56, 50, 50,
        47, 41, 41, 38, 64, 63, 63, 63, 61, 60, 59, 57, 56, 54, 49, 49, 46, 41,
        41, 38, 64, 63, 63, 62, 60, 60, 59, 56, 56, 54, 49, 49, 46, 41, 41, 38,
        64, 63, 63, 62, 60, 59, 59, 55, 55, 53, 48, 48, 46, 41, 41, 38, 64, 63,
        62, 62, 59, 58, 57, 54, 53, 52, 48, 48, 46, 41, 41, 38, 64, 63, 62, 62,
        59, 58, 57, 54, 53, 52, 48, 48, 46, 41, 41, 38, 63, 62, 62, 61, 58, 57,
        56, 52, 51, 49, 46, 46, 44, 40, 40, 37, 60, 60, 60, 60, 57, 55, 54, 49,
        48, 47, 43, 43, 41, 38, 38, 35, 60, 60, 60, 60, 57, 55, 54, 49, 48, 47,
        43, 43, 41, 38, 38, 35, 59, 59, 59, 59, 55, 55, 53, 46, 46, 44, 41, 41,
        39, 36, 36, 34, 57, 59, 59, 59, 55, 53, 51, 44, 43, 41, 38, 38, 37, 34,
        34, 32, 57, 59, 59, 59, 55, 53, 51, 44, 43, 41, 38, 38, 37, 34, 34, 32,
        54, 55, 56, 55, 53, 51, 49, 42, 41, 40, 37, 37, 35, 33, 33, 31, 52, 54,
        55, 54, 51, 51, 48, 42, 41, 39, 35, 35, 34, 32, 32, 30, 52, 54, 55, 54,
        51, 51, 48, 42, 41, 39, 35, 35, 34, 32, 32, 30, 48, 51, 51, 51, 49, 48,
        47, 40, 39, 37, 33, 33, 32, 29, 29, 28, 47, 49, 50, 49, 48, 48, 46, 39,
        38, 37, 33, 33, 31, 29, 29, 27, 47, 48, 49, 49, 48, 47, 46, 39, 38, 36,
        32, 32, 31, 28, 28, 27, 44, 46, 47, 46, 45, 45, 43, 38, 37, 35, 31, 31,
        29, 27, 27, 26, 43, 45, 46, 46, 45, 44, 43, 37, 36, 34, 31, 31, 29, 27,
        27, 25, 42, 44, 45, 45, 44, 44, 42, 37, 36, 34, 30, 30, 29, 26, 26, 25,
        39, 41, 42, 42, 41, 41, 40, 35, 34, 33, 29, 29, 27, 25, 25, 24, 38, 40,
        41, 41, 41, 41, 40, 35, 34, 33, 29, 29, 27, 25, 25, 24, 37, 39, 40, 40,
        40, 40, 39, 34, 34, 32, 28, 28, 27, 25, 25, 23, 35, 37, 38, 38, 38, 38,
        37, 33, 32, 31, 27, 27, 26, 24, 24, 22, 35, 37, 38, 38, 38, 38, 37, 33,
        32, 31, 27, 27, 26, 24, 24, 22,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 63, 60, 60, 59, 57, 57,
        54, 52, 52, 48, 47, 47, 44, 43, 42, 39, 38, 37, 35, 35, 65, 64, 64, 64,
        64, 64, 64, 63, 63, 63, 63, 63, 62, 60, 60, 59, 59, 59, 55, 54, 54, 51,
        49, 48, 46, 45, 44, 41, 40, 39, 37, 37, 65, 64, 64, 64, 63, 63, 63, 63,
        63, 63, 62, 62, 62, 60, 60, 59, 59, 59, 56, 55, 55, 51, 50, 49, 47, 46,
        45, 42, 41, 40, 38, 38, 65, 64, 64, 64, 63, 63, 63, 63, 62, 62, 62, 62,
        61, 60, 60, 59, 59, 59, 55, 54, 54, 51, 49, 49, 46, 46, 45, 42, 41, 40,
        38, 38, 64, 64, 63, 63, 63, 63, 62, 61, 60, 60, 59, 59, 58, 57, 57, 55,
        55, 55, 53, 51, 51, 49, 48, 48, 45, 45, 44, 41, 41, 40, 38, 38, 64, 63,
        63, 63, 63, 62, 62, 60, 60, 59, 58, 58, 57, 55, 55, 55, 53, 53, 51, 51,
        51, 48, 48, 47, 45, 44, 44, 41, 41, 40, 38, 38, 63, 62, 62, 62, 62, 62,
        61, 59, 59, 59, 57, 57, 56, 54, 54, 53, 51, 51, 49, 48, 48, 47, 46, 46,
        43, 43, 42, 40, 40, 39, 37, 37, 58, 59, 59, 59, 59, 59, 59, 57, 56, 55,
        54, 54, 52, 49, 49, 46, 44, 44, 42, 42, 42, 40, 39, 39, 38, 37, 37, 35,
        35, 34, 33, 33, 57, 58, 58, 58, 59, 59, 59, 56, 56, 55, 53, 53, 51, 48,
        48, 46, 43, 43, 41, 41, 41, 39, 38, 38, 37, 36, 36, 34, 34, 34, 32, 32,
        54, 55, 55, 55, 56, 56, 56, 54, 54, 53, 52, 52, 49, 47, 47, 44, 41, 41,
        40, 39, 39, 37, 37, 36, 35, 34, 34, 33, 33, 32, 31, 31, 47, 48, 48, 49,
        50, 50, 50, 49, 49, 48, 48, 48, 46, 43, 43, 41, 38, 38, 37, 35, 35, 33,
        33, 32, 31, 31, 30, 29, 29, 28, 27, 27, 47, 48, 48, 49, 50, 50, 50, 49,
        49, 48, 48, 48, 46, 43, 43, 41, 38, 38, 37, 35, 35, 33, 33, 32, 31, 31,
        30, 29, 29, 28, 27, 27, 43, 45, 45, 45, 47, 47, 47, 46, 46, 46, 46, 46,
        44, 41, 41, 39, 37, 37, 35, 34, 34, 32, 31, 31, 29, 29, 29, 27, 27, 27,
        26, 26, 38, 39, 40, 40, 41, 41, 41, 41, 41, 41, 41, 41, 40, 38, 38, 36,
        34, 34, 33, 32, 32, 29, 29, 28, 27, 27, 26, 25, 25, 25, 24, 24, 38, 39,
        40, 40, 41, 41, 41, 41, 41, 41, 41, 41, 40, 38, 38, 36, 34, 34, 33, 32,
        32, 29, 29, 28, 27, 27, 26, 25, 25, 25, 24, 24, 35, 36, 36, 37, 38, 38,
        38, 38, 38, 38, 38, 38, 37, 35, 35, 34, 32, 32, 31, 30, 30, 28, 27, 27,
        26, 25, 25, 24, 24, 23, 22, 22,
        /* Size 4x16 */
        65, 64, 54, 38, 64, 63, 55, 40, 64, 63, 56, 41, 64, 62, 56, 41, 63, 60,
        54, 41, 63, 58, 52, 41, 62, 57, 49, 40, 60, 55, 47, 38, 59, 53, 41, 34,
        55, 51, 40, 33, 54, 51, 39, 32, 49, 48, 37, 29, 46, 45, 35, 27, 44, 44,
        34, 26, 40, 41, 33, 25, 37, 38, 31, 24,
        /* Size 16x4 */
        65, 64, 64, 64, 63, 63, 62, 60, 59, 55, 54, 49, 46, 44, 40, 37, 64, 63,
        63, 62, 60, 58, 57, 55, 53, 51, 51, 48, 45, 44, 41, 38, 54, 55, 56, 56,
        54, 52, 49, 47, 41, 40, 39, 37, 35, 34, 33, 31, 38, 40, 41, 41, 41, 41,
        40, 38, 34, 33, 32, 29, 27, 26, 25, 24,
        /* Size 8x32 */
        64, 65, 64, 63, 57, 47, 43, 38, 65, 64, 64, 62, 58, 48, 45, 39, 65, 64,
        63, 62, 58, 48, 45, 40, 65, 64, 63, 62, 58, 49, 45, 40, 65, 63, 63, 62,
        59, 50, 47, 41, 65, 63, 63, 62, 59, 50, 47, 41, 65, 63, 62, 61, 59, 50,
        47, 41, 64, 63, 61, 59, 56, 49, 46, 41, 64, 63, 60, 59, 56, 49, 46, 41,
        64, 63, 60, 59, 55, 48, 46, 41, 64, 62, 59, 57, 53, 48, 46, 41, 64, 62,
        59, 57, 53, 48, 46, 41, 63, 62, 58, 56, 51, 46, 44, 40, 60, 60, 57, 54,
        48, 43, 41, 38, 60, 60, 57, 54, 48, 43, 41, 38, 59, 59, 55, 53, 46, 41,
        39, 36, 57, 59, 55, 51, 43, 38, 37, 34, 57, 59, 55, 51, 43, 38, 37, 34,
        54, 56, 53, 49, 41, 37, 35, 33, 52, 55, 51, 48, 41, 35, 34, 32, 52, 55,
        51, 48, 41, 35, 34, 32, 48, 51, 49, 47, 39, 33, 32, 29, 47, 50, 48, 46,
        38, 33, 31, 29, 47, 49, 48, 46, 38, 32, 31, 28, 44, 47, 45, 43, 37, 31,
        29, 27, 43, 46, 45, 43, 36, 31, 29, 27, 42, 45, 44, 42, 36, 30, 29, 26,
        39, 42, 41, 40, 34, 29, 27, 25, 38, 41, 41, 40, 34, 29, 27, 25, 37, 40,
        40, 39, 34, 28, 27, 25, 35, 38, 38, 37, 32, 27, 26, 24, 35, 38, 38, 37,
        32, 27, 26, 24,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 63, 60, 60, 59, 57, 57,
        54, 52, 52, 48, 47, 47, 44, 43, 42, 39, 38, 37, 35, 35, 65, 64, 64, 64,
        63, 63, 63, 63, 63, 63, 62, 62, 62, 60, 60, 59, 59, 59, 56, 55, 55, 51,
        50, 49, 47, 46, 45, 42, 41, 40, 38, 38, 64, 64, 63, 63, 63, 63, 62, 61,
        60, 60, 59, 59, 58, 57, 57, 55, 55, 55, 53, 51, 51, 49, 48, 48, 45, 45,
        44, 41, 41, 40, 38, 38, 63, 62, 62, 62, 62, 62, 61, 59, 59, 59, 57, 57,
        56, 54, 54, 53, 51, 51, 49, 48, 48, 47, 46, 46, 43, 43, 42, 40, 40, 39,
        37, 37, 57, 58, 58, 58, 59, 59, 59, 56, 56, 55, 53, 53, 51, 48, 48, 46,
        43, 43, 41, 41, 41, 39, 38, 38, 37, 36, 36, 34, 34, 34, 32, 32, 47, 48,
        48, 49, 50, 50, 50, 49, 49, 48, 48, 48, 46, 43, 43, 41, 38, 38, 37, 35,
        35, 33, 33, 32, 31, 31, 30, 29, 29, 28, 27, 27, 43, 45, 45, 45, 47, 47,
        47, 46, 46, 46, 46, 46, 44, 41, 41, 39, 37, 37, 35, 34, 34, 32, 31, 31,
        29, 29, 29, 27, 27, 27, 26, 26, 38, 39, 40, 40, 41, 41, 41, 41, 41, 41,
        41, 41, 40, 38, 38, 36, 34, 34, 33, 32, 32, 29, 29, 28, 27, 27, 26, 25,
        25, 25, 24, 24 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        53, 45, 45, 35,
#endif
        /* Size 4x4 */
        65, 53, 44, 41, 53, 43, 44, 44, 44, 44, 38, 36, 41, 44, 36, 31,
        /* Size 8x8 */
        65, 66, 58, 49, 43, 43, 42, 40, 66, 64, 56, 48, 44, 46, 44, 42, 58, 56,
        49, 45, 43, 45, 44, 43, 49, 48, 45, 42, 41, 41, 41, 40, 43, 44, 43, 41,
        39, 38, 38, 38, 43, 46, 45, 41, 38, 36, 35, 34, 42, 44, 44, 41, 38, 35,
        33, 32, 40, 42, 43, 40, 38, 34, 32, 30,
        /* Size 16x16 */
        64, 66, 67, 66, 61, 56, 53, 49, 42, 42, 42, 42, 41, 40, 39, 38, 66, 65,
        65, 64, 59, 53, 51, 48, 43, 43, 44, 44, 43, 42, 41, 39, 67, 65, 65, 63,
        58, 52, 50, 48, 44, 45, 45, 45, 44, 44, 43, 41, 66, 64, 63, 62, 57, 51,
        49, 48, 44, 45, 45, 46, 45, 44, 43, 41, 61, 59, 58, 57, 53, 47, 47, 46,
        43, 44, 44, 45, 44, 44, 43, 41, 56, 53, 52, 51, 47, 43, 43, 43, 43, 44,
        44, 45, 45, 44, 44, 42, 53, 51, 50, 49, 47, 43, 43, 43, 42, 43, 43, 44,
        43, 43, 43, 41, 49, 48, 48, 48, 46, 43, 43, 42, 41, 41, 41, 42, 41, 41,
        41, 39, 42, 43, 44, 44, 43, 43, 42, 41, 39, 39, 39, 38, 38, 38, 38, 37,
        42, 43, 45, 45, 44, 44, 43, 41, 39, 38, 38, 37, 37, 37, 37, 36, 42, 44,
        45, 45, 44, 44, 43, 41, 39, 38, 38, 37, 36, 36, 36, 35, 42, 44, 45, 46,
        45, 45, 44, 42, 38, 37, 37, 35, 34, 34, 33, 33, 41, 43, 44, 45, 44, 45,
        43, 41, 38, 37, 36, 34, 34, 33, 33, 32, 40, 42, 44, 44, 44, 44, 43, 41,
        38, 37, 36, 34, 33, 33, 32, 31, 39, 41, 43, 43, 43, 44, 43, 41, 38, 37,
        36, 33, 33, 32, 31, 30, 38, 39, 41, 41, 41, 42, 41, 39, 37, 36, 35, 33,
        32, 31, 30, 29,
        /* Size 32x32 */
        64, 65, 66, 66, 67, 67, 66, 62, 61, 59, 56, 56, 53, 49, 49, 46, 42, 42,
        42, 42, 42, 42, 42, 42, 41, 41, 40, 39, 39, 39, 38, 38, 65, 65, 65, 65,
        66, 66, 65, 60, 59, 58, 54, 54, 52, 49, 49, 46, 43, 43, 43, 43, 43, 43,
        43, 43, 42, 42, 41, 41, 41, 40, 39, 39, 66, 65, 65, 65, 65, 66, 64, 59,
        59, 58, 53, 53, 51, 48, 48, 46, 43, 43, 43, 44, 44, 44, 44, 44, 43, 42,
        42, 41, 41, 41, 39, 39, 66, 65, 65, 65, 65, 65, 64, 59, 59, 57, 53, 53,
        51, 48, 48, 46, 43, 43, 44, 44, 44, 44, 44, 44, 43, 43, 42, 41, 41, 41,
        39, 39, 67, 66, 65, 65, 65, 65, 63, 59, 58, 56, 52, 52, 50, 48, 48, 46,
        44, 44, 45, 45, 45, 45, 45, 45, 44, 44, 44, 43, 43, 42, 41, 41, 67, 66,
        66, 65, 65, 64, 63, 59, 58, 56, 51, 51, 50, 48, 48, 46, 44, 44, 45, 45,
        45, 46, 46, 46, 45, 44, 44, 43, 43, 42, 41, 41, 66, 65, 64, 64, 63, 63,
        62, 58, 57, 55, 51, 51, 49, 48, 48, 46, 44, 44, 45, 45, 45, 45, 46, 46,
        45, 44, 44, 43, 43, 43, 41, 41, 62, 60, 59, 59, 59, 59, 58, 53, 53, 51,
        48, 48, 47, 46, 46, 45, 44, 44, 44, 45, 45, 45, 45, 45, 44, 44, 44, 43,
        43, 43, 41, 41, 61, 59, 59, 59, 58, 58, 57, 53, 53, 51, 47, 47, 47, 46,
        46, 45, 43, 43, 44, 44, 44, 45, 45, 45, 44, 44, 44, 43, 43, 43, 41, 41,
        59, 58, 58, 57, 56, 56, 55, 51, 51, 49, 46, 46, 46, 45, 45, 44, 43, 43,
        44, 44, 44, 45, 45, 45, 44, 44, 44, 44, 43, 43, 42, 42, 56, 54, 53, 53,
        52, 51, 51, 48, 47, 46, 43, 43, 43, 43, 43, 43, 43, 43, 44, 44, 44, 45,
        45, 45, 45, 45, 44, 44, 44, 44, 42, 42, 56, 54, 53, 53, 52, 51, 51, 48,
        47, 46, 43, 43, 43, 43, 43, 43, 43, 43, 44, 44, 44, 45, 45, 45, 45, 45,
        44, 44, 44, 44, 42, 42, 53, 52, 51, 51, 50, 50, 49, 47, 47, 46, 43, 43,
        43, 43, 43, 42, 42, 42, 43, 43, 43, 44, 44, 44, 43, 43, 43, 43, 43, 42,
        41, 41, 49, 49, 48, 48, 48, 48, 48, 46, 46, 45, 43, 43, 43, 42, 42, 41,
        41, 41, 41, 41, 41, 41, 42, 41, 41, 41, 41, 41, 41, 40, 39, 39, 49, 49,
        48, 48, 48, 48, 48, 46, 46, 45, 43, 43, 43, 42, 42, 41, 41, 41, 41, 41,
        41, 41, 42, 41, 41, 41, 41, 41, 41, 40, 39, 39, 46, 46, 46, 46, 46, 46,
        46, 45, 45, 44, 43, 43, 42, 41, 41, 41, 40, 40, 40, 40, 40, 40, 40, 40,
        39, 39, 39, 39, 39, 39, 38, 38, 42, 43, 43, 43, 44, 44, 44, 44, 43, 43,
        43, 43, 42, 41, 41, 40, 39, 39, 39, 39, 39, 38, 38, 38, 38, 38, 38, 38,
        38, 38, 37, 37, 42, 43, 43, 43, 44, 44, 44, 44, 43, 43, 43, 43, 42, 41,
        41, 40, 39, 39, 39, 39, 39, 38, 38, 38, 38, 38, 38, 38, 38, 38, 37, 37,
        42, 43, 43, 44, 45, 45, 45, 44, 44, 44, 44, 44, 43, 41, 41, 40, 39, 39,
        38, 38, 38, 38, 37, 37, 37, 37, 37, 37, 37, 36, 36, 36, 42, 43, 44, 44,
        45, 45, 45, 45, 44, 44, 44, 44, 43, 41, 41, 40, 39, 39, 38, 38, 38, 37,
        37, 37, 36, 36, 36, 36, 36, 35, 35, 35, 42, 43, 44, 44, 45, 45, 45, 45,
        44, 44, 44, 44, 43, 41, 41, 40, 39, 39, 38, 38, 38, 37, 37, 37, 36, 36,
        36, 36, 36, 35, 35, 35, 42, 43, 44, 44, 45, 46, 45, 45, 45, 45, 45, 45,
        44, 41, 41, 40, 38, 38, 38, 37, 37, 36, 36, 35, 35, 35, 35, 34, 34, 34,
        33, 33, 42, 43, 44, 44, 45, 46, 46, 45, 45, 45, 45, 45, 44, 42, 42, 40,
        38, 38, 37, 37, 37, 36, 35, 35, 34, 34, 34, 34, 33, 33, 33, 33, 42, 43,
        44, 44, 45, 46, 46, 45, 45, 45, 45, 45, 44, 41, 41, 40, 38, 38, 37, 37,
        37, 35, 35, 35, 34, 34, 34, 33, 33, 33, 33, 33, 41, 42, 43, 43, 44, 45,
        45, 44, 44, 44, 45, 45, 43, 41, 41, 39, 38, 38, 37, 36, 36, 35, 34, 34,
        34, 34, 33, 33, 33, 32, 32, 32, 41, 42, 42, 43, 44, 44, 44, 44, 44, 44,
        45, 45, 43, 41, 41, 39, 38, 38, 37, 36, 36, 35, 34, 34, 34, 33, 33, 32,
        32, 32, 32, 32, 40, 41, 42, 42, 44, 44, 44, 44, 44, 44, 44, 44, 43, 41,
        41, 39, 38, 38, 37, 36, 36, 35, 34, 34, 33, 33, 33, 32, 32, 32, 31, 31,
        39, 41, 41, 41, 43, 43, 43, 43, 43, 44, 44, 44, 43, 41, 41, 39, 38, 38,
        37, 36, 36, 34, 34, 33, 33, 32, 32, 31, 31, 31, 30, 30, 39, 41, 41, 41,
        43, 43, 43, 43, 43, 43, 44, 44, 43, 41, 41, 39, 38, 38, 37, 36, 36, 34,
        33, 33, 33, 32, 32, 31, 31, 31, 30, 30, 39, 40, 41, 41, 42, 42, 43, 43,
        43, 43, 44, 44, 42, 40, 40, 39, 38, 38, 36, 35, 35, 34, 33, 33, 32, 32,
        32, 31, 31, 30, 30, 30, 38, 39, 39, 39, 41, 41, 41, 41, 41, 42, 42, 42,
        41, 39, 39, 38, 37, 37, 36, 35, 35, 33, 33, 33, 32, 32, 31, 30, 30, 30,
        29, 29, 38, 39, 39, 39, 41, 41, 41, 41, 41, 42, 42, 42, 41, 39, 39, 38,
        37, 37, 36, 35, 35, 33, 33, 33, 32, 32, 31, 30, 30, 30, 29, 29,
        /* Size 4x8 */
        65, 54, 43, 41, 65, 51, 45, 43, 56, 46, 44, 43, 48, 43, 41, 41, 43, 43,
        39, 38, 44, 45, 38, 34, 43, 44, 37, 32, 41, 43, 36, 31,
        /* Size 8x4 */
        65, 65, 56, 48, 43, 44, 43, 41, 54, 51, 46, 43, 43, 45, 44, 43, 43, 45,
        44, 41, 39, 38, 37, 36, 41, 43, 43, 41, 38, 34, 32, 31,
        /* Size 8x16 */
        64, 66, 58, 53, 43, 42, 41, 39, 65, 65, 55, 51, 44, 44, 42, 41, 67, 64,
        54, 51, 45, 45, 44, 42, 66, 62, 53, 49, 45, 45, 44, 43, 61, 57, 49, 47,
        44, 45, 44, 43, 55, 51, 45, 43, 43, 45, 45, 44, 53, 50, 45, 43, 42, 44,
        43, 42, 49, 48, 44, 43, 41, 41, 41, 41, 42, 44, 43, 42, 39, 38, 38, 38,
        42, 45, 44, 42, 39, 37, 37, 36, 42, 45, 44, 43, 39, 37, 36, 36, 42, 45,
        45, 44, 38, 35, 34, 33, 41, 44, 44, 43, 38, 34, 34, 32, 40, 44, 44, 43,
        38, 34, 33, 32, 39, 43, 43, 42, 38, 34, 32, 31, 38, 41, 42, 41, 37, 33,
        32, 30,
        /* Size 16x8 */
        64, 65, 67, 66, 61, 55, 53, 49, 42, 42, 42, 42, 41, 40, 39, 38, 66, 65,
        64, 62, 57, 51, 50, 48, 44, 45, 45, 45, 44, 44, 43, 41, 58, 55, 54, 53,
        49, 45, 45, 44, 43, 44, 44, 45, 44, 44, 43, 42, 53, 51, 51, 49, 47, 43,
        43, 43, 42, 42, 43, 44, 43, 43, 42, 41, 43, 44, 45, 45, 44, 43, 42, 41,
        39, 39, 39, 38, 38, 38, 38, 37, 42, 44, 45, 45, 45, 45, 44, 41, 38, 37,
        37, 35, 34, 34, 34, 33, 41, 42, 44, 44, 44, 45, 43, 41, 38, 37, 36, 34,
        34, 33, 32, 32, 39, 41, 42, 43, 43, 44, 42, 41, 38, 36, 36, 33, 32, 32,
        31, 30,
        /* Size 16x32 */
        64, 66, 66, 65, 58, 55, 53, 44, 43, 42, 42, 42, 41, 39, 39, 38, 65, 65,
        65, 64, 56, 54, 52, 44, 43, 43, 43, 43, 42, 41, 41, 39, 65, 65, 65, 64,
        55, 53, 51, 45, 44, 44, 44, 44, 42, 41, 41, 39, 66, 65, 65, 64, 55, 53,
        51, 45, 44, 44, 44, 44, 43, 41, 41, 39, 67, 65, 64, 63, 54, 52, 51, 45,
        45, 45, 45, 45, 44, 42, 42, 41, 67, 65, 64, 62, 54, 51, 50, 45, 45, 45,
        45, 45, 44, 43, 43, 41, 66, 63, 62, 61, 53, 51, 49, 45, 45, 45, 45, 45,
        44, 43, 43, 41, 61, 59, 58, 57, 50, 48, 47, 44, 44, 44, 45, 45, 44, 43,
        43, 41, 61, 59, 57, 56, 49, 48, 47, 44, 44, 44, 45, 45, 44, 43, 43, 41,
        59, 56, 55, 55, 48, 46, 46, 44, 44, 44, 45, 45, 44, 43, 43, 42, 55, 53,
        51, 50, 45, 44, 43, 43, 43, 44, 45, 45, 45, 44, 44, 42, 55, 53, 51, 50,
        45, 44, 43, 43, 43, 44, 45, 45, 45, 44, 44, 42, 53, 51, 50, 49, 45, 43,
        43, 42, 42, 43, 44, 44, 43, 42, 42, 41, 49, 48, 48, 47, 44, 43, 43, 41,
        41, 41, 41, 41, 41, 41, 41, 39, 49, 48, 48, 47, 44, 43, 43, 41, 41, 41,
        41, 41, 41, 41, 41, 39, 46, 46, 46, 46, 44, 43, 42, 40, 40, 40, 40, 40,
        39, 39, 39, 38, 42, 43, 44, 44, 43, 43, 42, 39, 39, 39, 38, 38, 38, 38,
        38, 37, 42, 43, 44, 44, 43, 43, 42, 39, 39, 39, 38, 38, 38, 38, 38, 37,
        42, 44, 45, 44, 44, 43, 42, 39, 39, 38, 37, 37, 37, 36, 36, 36, 42, 44,
        45, 45, 44, 44, 43, 39, 39, 38, 37, 37, 36, 36, 36, 35, 42, 44, 45, 45,
        44, 44, 43, 39, 39, 38, 37, 37, 36, 36, 36, 35, 42, 44, 45, 45, 45, 45,
        43, 39, 38, 38, 36, 36, 35, 34, 34, 33, 42, 44, 45, 45, 45, 45, 44, 39,
        38, 37, 35, 35, 34, 33, 33, 33, 42, 44, 45, 45, 45, 45, 44, 39, 38, 37,
        35, 35, 34, 33, 33, 33, 41, 43, 44, 44, 44, 44, 43, 39, 38, 37, 34, 34,
        34, 32, 32, 32, 41, 43, 44, 44, 44, 44, 43, 39, 38, 37, 34, 34, 33, 32,
        32, 31, 40, 43, 44, 44, 44, 44, 43, 39, 38, 37, 34, 34, 33, 32, 32, 31,
        39, 42, 43, 43, 43, 44, 42, 38, 38, 37, 34, 34, 33, 31, 31, 30, 39, 42,
        43, 43, 43, 43, 42, 38, 38, 37, 34, 34, 32, 31, 31, 30, 39, 41, 42, 42,
        43, 43, 42, 38, 38, 36, 33, 33, 32, 31, 31, 29, 38, 40, 41, 41, 42, 42,
        41, 37, 37, 36, 33, 33, 32, 30, 30, 29, 38, 40, 41, 41, 42, 42, 41, 37,
        37, 36, 33, 33, 32, 30, 30, 29,
        /* Size 32x16 */
        64, 65, 65, 66, 67, 67, 66, 61, 61, 59, 55, 55, 53, 49, 49, 46, 42, 42,
        42, 42, 42, 42, 42, 42, 41, 41, 40, 39, 39, 39, 38, 38, 66, 65, 65, 65,
        65, 65, 63, 59, 59, 56, 53, 53, 51, 48, 48, 46, 43, 43, 44, 44, 44, 44,
        44, 44, 43, 43, 43, 42, 42, 41, 40, 40, 66, 65, 65, 65, 64, 64, 62, 58,
        57, 55, 51, 51, 50, 48, 48, 46, 44, 44, 45, 45, 45, 45, 45, 45, 44, 44,
        44, 43, 43, 42, 41, 41, 65, 64, 64, 64, 63, 62, 61, 57, 56, 55, 50, 50,
        49, 47, 47, 46, 44, 44, 44, 45, 45, 45, 45, 45, 44, 44, 44, 43, 43, 42,
        41, 41, 58, 56, 55, 55, 54, 54, 53, 50, 49, 48, 45, 45, 45, 44, 44, 44,
        43, 43, 44, 44, 44, 45, 45, 45, 44, 44, 44, 43, 43, 43, 42, 42, 55, 54,
        53, 53, 52, 51, 51, 48, 48, 46, 44, 44, 43, 43, 43, 43, 43, 43, 43, 44,
        44, 45, 45, 45, 44, 44, 44, 44, 43, 43, 42, 42, 53, 52, 51, 51, 51, 50,
        49, 47, 47, 46, 43, 43, 43, 43, 43, 42, 42, 42, 42, 43, 43, 43, 44, 44,
        43, 43, 43, 42, 42, 42, 41, 41, 44, 44, 45, 45, 45, 45, 45, 44, 44, 44,
        43, 43, 42, 41, 41, 40, 39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 38,
        38, 38, 37, 37, 43, 43, 44, 44, 45, 45, 45, 44, 44, 44, 43, 43, 42, 41,
        41, 40, 39, 39, 39, 39, 39, 38, 38, 38, 38, 38, 38, 38, 38, 38, 37, 37,
        42, 43, 44, 44, 45, 45, 45, 44, 44, 44, 44, 44, 43, 41, 41, 40, 39, 39,
        38, 38, 38, 38, 37, 37, 37, 37, 37, 37, 37, 36, 36, 36, 42, 43, 44, 44,
        45, 45, 45, 45, 45, 45, 45, 45, 44, 41, 41, 40, 38, 38, 37, 37, 37, 36,
        35, 35, 34, 34, 34, 34, 34, 33, 33, 33, 42, 43, 44, 44, 45, 45, 45, 45,
        45, 45, 45, 45, 44, 41, 41, 40, 38, 38, 37, 37, 37, 36, 35, 35, 34, 34,
        34, 34, 34, 33, 33, 33, 41, 42, 42, 43, 44, 44, 44, 44, 44, 44, 45, 45,
        43, 41, 41, 39, 38, 38, 37, 36, 36, 35, 34, 34, 34, 33, 33, 33, 32, 32,
        32, 32, 39, 41, 41, 41, 42, 43, 43, 43, 43, 43, 44, 44, 42, 41, 41, 39,
        38, 38, 36, 36, 36, 34, 33, 33, 32, 32, 32, 31, 31, 31, 30, 30, 39, 41,
        41, 41, 42, 43, 43, 43, 43, 43, 44, 44, 42, 41, 41, 39, 38, 38, 36, 36,
        36, 34, 33, 33, 32, 32, 32, 31, 31, 31, 30, 30, 38, 39, 39, 39, 41, 41,
        41, 41, 41, 42, 42, 42, 41, 39, 39, 38, 37, 37, 36, 35, 35, 33, 33, 33,
        32, 31, 31, 30, 30, 29, 29, 29,
        /* Size 4x16 */
        66, 55, 42, 39, 65, 53, 44, 41, 65, 52, 45, 42, 63, 51, 45, 43, 59, 48,
        44, 43, 53, 44, 44, 44, 51, 43, 43, 42, 48, 43, 41, 41, 43, 43, 39, 38,
        44, 43, 38, 36, 44, 44, 38, 36, 44, 45, 37, 33, 43, 44, 37, 32, 43, 44,
        37, 32, 42, 43, 37, 31, 40, 42, 36, 30,
        /* Size 16x4 */
        66, 65, 65, 63, 59, 53, 51, 48, 43, 44, 44, 44, 43, 43, 42, 40, 55, 53,
        52, 51, 48, 44, 43, 43, 43, 43, 44, 45, 44, 44, 43, 42, 42, 44, 45, 45,
        44, 44, 43, 41, 39, 38, 38, 37, 37, 37, 37, 36, 39, 41, 42, 43, 43, 44,
        42, 41, 38, 36, 36, 33, 32, 32, 31, 30,
        /* Size 8x32 */
        64, 66, 58, 53, 43, 42, 41, 39, 65, 65, 56, 52, 43, 43, 42, 41, 65, 65,
        55, 51, 44, 44, 42, 41, 66, 65, 55, 51, 44, 44, 43, 41, 67, 64, 54, 51,
        45, 45, 44, 42, 67, 64, 54, 50, 45, 45, 44, 43, 66, 62, 53, 49, 45, 45,
        44, 43, 61, 58, 50, 47, 44, 45, 44, 43, 61, 57, 49, 47, 44, 45, 44, 43,
        59, 55, 48, 46, 44, 45, 44, 43, 55, 51, 45, 43, 43, 45, 45, 44, 55, 51,
        45, 43, 43, 45, 45, 44, 53, 50, 45, 43, 42, 44, 43, 42, 49, 48, 44, 43,
        41, 41, 41, 41, 49, 48, 44, 43, 41, 41, 41, 41, 46, 46, 44, 42, 40, 40,
        39, 39, 42, 44, 43, 42, 39, 38, 38, 38, 42, 44, 43, 42, 39, 38, 38, 38,
        42, 45, 44, 42, 39, 37, 37, 36, 42, 45, 44, 43, 39, 37, 36, 36, 42, 45,
        44, 43, 39, 37, 36, 36, 42, 45, 45, 43, 38, 36, 35, 34, 42, 45, 45, 44,
        38, 35, 34, 33, 42, 45, 45, 44, 38, 35, 34, 33, 41, 44, 44, 43, 38, 34,
        34, 32, 41, 44, 44, 43, 38, 34, 33, 32, 40, 44, 44, 43, 38, 34, 33, 32,
        39, 43, 43, 42, 38, 34, 33, 31, 39, 43, 43, 42, 38, 34, 32, 31, 39, 42,
        43, 42, 38, 33, 32, 31, 38, 41, 42, 41, 37, 33, 32, 30, 38, 41, 42, 41,
        37, 33, 32, 30,
        /* Size 32x8 */
        64, 65, 65, 66, 67, 67, 66, 61, 61, 59, 55, 55, 53, 49, 49, 46, 42, 42,
        42, 42, 42, 42, 42, 42, 41, 41, 40, 39, 39, 39, 38, 38, 66, 65, 65, 65,
        64, 64, 62, 58, 57, 55, 51, 51, 50, 48, 48, 46, 44, 44, 45, 45, 45, 45,
        45, 45, 44, 44, 44, 43, 43, 42, 41, 41, 58, 56, 55, 55, 54, 54, 53, 50,
        49, 48, 45, 45, 45, 44, 44, 44, 43, 43, 44, 44, 44, 45, 45, 45, 44, 44,
        44, 43, 43, 43, 42, 42, 53, 52, 51, 51, 51, 50, 49, 47, 47, 46, 43, 43,
        43, 43, 43, 42, 42, 42, 42, 43, 43, 43, 44, 44, 43, 43, 43, 42, 42, 42,
        41, 41, 43, 43, 44, 44, 45, 45, 45, 44, 44, 44, 43, 43, 42, 41, 41, 40,
        39, 39, 39, 39, 39, 38, 38, 38, 38, 38, 38, 38, 38, 38, 37, 37, 42, 43,
        44, 44, 45, 45, 45, 45, 45, 45, 45, 45, 44, 41, 41, 40, 38, 38, 37, 37,
        37, 36, 35, 35, 34, 34, 34, 34, 34, 33, 33, 33, 41, 42, 42, 43, 44, 44,
        44, 44, 44, 44, 45, 45, 43, 41, 41, 39, 38, 38, 37, 36, 36, 35, 34, 34,
        34, 33, 33, 33, 32, 32, 32, 32, 39, 41, 41, 41, 42, 43, 43, 43, 43, 43,
        44, 44, 42, 41, 41, 39, 38, 38, 36, 36, 36, 34, 33, 33, 32, 32, 32, 31,
        31, 31, 30, 30 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        63, 54, 54, 38,
#endif
        /* Size 4x4 */
        64, 63, 59, 48, 63, 59, 55, 48, 59, 55, 42, 38, 48, 48, 38, 32,
        /* Size 8x8 */
        65, 65, 64, 64, 60, 55, 48, 44, 65, 63, 63, 63, 60, 57, 50, 46, 64, 63,
        61, 60, 58, 54, 49, 45, 64, 63, 60, 58, 55, 52, 48, 45, 60, 60, 58, 55,
        50, 46, 42, 39, 55, 57, 54, 52, 46, 40, 37, 34, 48, 50, 49, 48, 42, 37,
        32, 31, 44, 46, 45, 45, 39, 34, 31, 29,
        /* Size 16x16 */
        64, 65, 65, 65, 65, 65, 64, 63, 60, 58, 57, 52, 50, 47, 43, 42, 65, 64,
        64, 64, 64, 64, 63, 62, 60, 59, 59, 53, 51, 48, 45, 44, 65, 64, 64, 64,
        63, 63, 63, 62, 60, 59, 59, 54, 52, 49, 46, 45, 65, 64, 64, 63, 63, 63,
        63, 62, 61, 60, 59, 55, 53, 50, 47, 46, 65, 64, 63, 63, 62, 62, 61, 60,
        59, 57, 57, 53, 51, 49, 46, 45, 65, 64, 63, 63, 62, 61, 60, 59, 58, 57,
        56, 53, 51, 49, 46, 45, 64, 63, 63, 63, 61, 60, 58, 57, 55, 54, 53, 51,
        49, 48, 46, 45, 63, 62, 62, 62, 60, 59, 57, 56, 54, 52, 51, 49, 48, 46,
        44, 43, 60, 60, 60, 61, 59, 58, 55, 54, 52, 49, 48, 46, 45, 43, 41, 40,
        58, 59, 59, 60, 57, 57, 54, 52, 49, 45, 44, 41, 41, 39, 37, 37, 57, 59,
        59, 59, 57, 56, 53, 51, 48, 44, 42, 41, 39, 38, 37, 36, 52, 53, 54, 55,
        53, 53, 51, 49, 46, 41, 41, 38, 37, 35, 34, 34, 50, 51, 52, 53, 51, 51,
        49, 48, 45, 41, 39, 37, 36, 34, 33, 32, 47, 48, 49, 50, 49, 49, 48, 46,
        43, 39, 38, 35, 34, 32, 31, 31, 43, 45, 46, 47, 46, 46, 46, 44, 41, 37,
        37, 34, 33, 31, 29, 29, 42, 44, 45, 46, 45, 45, 45, 43, 40, 37, 36, 34,
        32, 31, 29, 29,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 63, 60, 60, 60,
        58, 57, 57, 54, 52, 52, 50, 47, 47, 46, 43, 42, 42, 40, 65, 65, 65, 65,
        65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 62, 60, 60, 60, 59, 58, 58, 55,
        53, 53, 51, 48, 48, 47, 44, 44, 44, 41, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 63, 63, 63, 62, 60, 60, 60, 59, 59, 59, 55, 53, 53, 51, 48,
        48, 48, 45, 44, 44, 41, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63,
        63, 63, 62, 60, 60, 60, 59, 59, 59, 55, 53, 53, 51, 48, 48, 48, 45, 44,
        44, 41, 65, 65, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 62, 60,
        60, 60, 59, 59, 59, 56, 54, 54, 52, 49, 49, 48, 46, 45, 45, 42, 65, 65,
        64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 61, 61, 60, 60, 59,
        59, 57, 55, 55, 53, 50, 50, 49, 47, 46, 46, 43, 65, 65, 64, 64, 64, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 62, 61, 61, 60, 60, 59, 59, 57, 55, 55,
        53, 50, 50, 49, 47, 46, 46, 43, 65, 64, 64, 64, 64, 63, 63, 63, 63, 63,
        63, 62, 62, 62, 61, 60, 60, 60, 59, 59, 59, 56, 54, 54, 53, 50, 50, 49,
        47, 46, 46, 43, 65, 64, 64, 64, 63, 63, 63, 63, 62, 62, 62, 61, 61, 61,
        60, 59, 59, 59, 57, 57, 57, 55, 53, 53, 51, 49, 49, 48, 46, 45, 45, 43,
        65, 64, 64, 64, 63, 63, 63, 63, 62, 61, 61, 61, 60, 60, 59, 58, 58, 58,
        57, 56, 56, 54, 53, 53, 51, 49, 49, 48, 46, 45, 45, 43, 65, 64, 64, 64,
        63, 63, 63, 63, 62, 61, 61, 61, 60, 60, 59, 58, 58, 58, 57, 56, 56, 54,
        53, 53, 51, 49, 49, 48, 46, 45, 45, 43, 64, 64, 63, 63, 63, 63, 63, 62,
        61, 61, 61, 59, 59, 59, 58, 57, 57, 56, 55, 55, 55, 53, 51, 51, 50, 48,
        48, 48, 46, 45, 45, 43, 64, 64, 63, 63, 63, 63, 63, 62, 61, 60, 60, 59,
        58, 58, 57, 55, 55, 55, 54, 53, 53, 52, 51, 51, 49, 48, 48, 47, 46, 45,
        45, 43, 64, 64, 63, 63, 63, 63, 63, 62, 61, 60, 60, 59, 58, 58, 57, 55,
        55, 55, 54, 53, 53, 52, 51, 51, 49, 48, 48, 47, 46, 45, 45, 43, 63, 62,
        62, 62, 62, 62, 62, 61, 60, 59, 59, 58, 57, 57, 56, 54, 54, 54, 52, 51,
        51, 50, 49, 49, 48, 46, 46, 46, 44, 43, 43, 41, 60, 60, 60, 60, 60, 61,
        61, 60, 59, 58, 58, 57, 55, 55, 54, 52, 52, 51, 49, 48, 48, 47, 46, 46,
        45, 43, 43, 42, 41, 40, 40, 39, 60, 60, 60, 60, 60, 61, 61, 60, 59, 58,
        58, 57, 55, 55, 54, 52, 52, 51, 49, 48, 48, 47, 46, 46, 45, 43, 43, 42,
        41, 40, 40, 39, 60, 60, 60, 60, 60, 60, 60, 60, 59, 58, 58, 56, 55, 55,
        54, 51, 51, 50, 48, 47, 47, 46, 45, 45, 44, 42, 42, 41, 40, 39, 39, 38,
        58, 59, 59, 59, 59, 60, 60, 59, 57, 57, 57, 55, 54, 54, 52, 49, 49, 48,
        45, 44, 44, 42, 41, 41, 41, 39, 39, 39, 37, 37, 37, 36, 57, 58, 59, 59,
        59, 59, 59, 59, 57, 56, 56, 55, 53, 53, 51, 48, 48, 47, 44, 42, 42, 41,
        41, 41, 39, 38, 38, 38, 37, 36, 36, 35, 57, 58, 59, 59, 59, 59, 59, 59,
        57, 56, 56, 55, 53, 53, 51, 48, 48, 47, 44, 42, 42, 41, 41, 41, 39, 38,
        38, 38, 37, 36, 36, 35, 54, 55, 55, 55, 56, 57, 57, 56, 55, 54, 54, 53,
        52, 52, 50, 47, 47, 46, 42, 41, 41, 40, 39, 39, 38, 37, 37, 36, 35, 34,
        34, 33, 52, 53, 53, 53, 54, 55, 55, 54, 53, 53, 53, 51, 51, 51, 49, 46,
        46, 45, 41, 41, 41, 39, 38, 38, 37, 35, 35, 35, 34, 34, 34, 32, 52, 53,
        53, 53, 54, 55, 55, 54, 53, 53, 53, 51, 51, 51, 49, 46, 46, 45, 41, 41,
        41, 39, 38, 38, 37, 35, 35, 35, 34, 34, 34, 32, 50, 51, 51, 51, 52, 53,
        53, 53, 51, 51, 51, 50, 49, 49, 48, 45, 45, 44, 41, 39, 39, 38, 37, 37,
        36, 34, 34, 34, 33, 32, 32, 31, 47, 48, 48, 48, 49, 50, 50, 50, 49, 49,
        49, 48, 48, 48, 46, 43, 43, 42, 39, 38, 38, 37, 35, 35, 34, 32, 32, 32,
        31, 31, 31, 29, 47, 48, 48, 48, 49, 50, 50, 50, 49, 49, 49, 48, 48, 48,
        46, 43, 43, 42, 39, 38, 38, 37, 35, 35, 34, 32, 32, 32, 31, 31, 31, 29,
        46, 47, 48, 48, 48, 49, 49, 49, 48, 48, 48, 48, 47, 47, 46, 42, 42, 41,
        39, 38, 38, 36, 35, 35, 34, 32, 32, 32, 31, 30, 30, 29, 43, 44, 45, 45,
        46, 47, 47, 47, 46, 46, 46, 46, 46, 46, 44, 41, 41, 40, 37, 37, 37, 35,
        34, 34, 33, 31, 31, 31, 29, 29, 29, 28, 42, 44, 44, 44, 45, 46, 46, 46,
        45, 45, 45, 45, 45, 45, 43, 40, 40, 39, 37, 36, 36, 34, 34, 34, 32, 31,
        31, 30, 29, 29, 29, 28, 42, 44, 44, 44, 45, 46, 46, 46, 45, 45, 45, 45,
        45, 45, 43, 40, 40, 39, 37, 36, 36, 34, 34, 34, 32, 31, 31, 30, 29, 29,
        29, 28, 40, 41, 41, 41, 42, 43, 43, 43, 43, 43, 43, 43, 43, 43, 41, 39,
        39, 38, 36, 35, 35, 33, 32, 32, 31, 29, 29, 29, 28, 28, 28, 26,
        /* Size 4x8 */
        65, 63, 58, 48, 64, 62, 59, 50, 63, 60, 56, 49, 63, 58, 53, 48, 60, 55,
        47, 42, 55, 51, 41, 37, 49, 48, 38, 33, 45, 44, 36, 31,
        /* Size 8x4 */
        65, 64, 63, 63, 60, 55, 49, 45, 63, 62, 60, 58, 55, 51, 48, 44, 58, 59,
        56, 53, 47, 41, 38, 36, 48, 50, 49, 48, 42, 37, 33, 31,
        /* Size 8x16 */
        64, 65, 65, 64, 59, 57, 47, 43, 65, 64, 64, 63, 59, 58, 48, 45, 65, 64,
        63, 63, 59, 59, 49, 46, 65, 63, 63, 62, 60, 59, 50, 47, 65, 63, 62, 60,
        58, 57, 49, 46, 64, 63, 62, 60, 57, 56, 49, 46, 64, 62, 60, 58, 55, 53,
        48, 46, 63, 62, 59, 57, 53, 51, 46, 44, 60, 60, 59, 55, 49, 48, 43, 41,
        58, 59, 57, 54, 46, 44, 39, 37, 57, 59, 57, 53, 45, 43, 38, 37, 52, 55,
        53, 51, 43, 41, 35, 34, 50, 53, 51, 49, 41, 40, 34, 33, 47, 50, 49, 48,
        40, 38, 33, 31, 44, 47, 46, 45, 38, 37, 31, 29, 43, 46, 45, 44, 38, 36,
        31, 29,
        /* Size 16x8 */
        64, 65, 65, 65, 65, 64, 64, 63, 60, 58, 57, 52, 50, 47, 44, 43, 65, 64,
        64, 63, 63, 63, 62, 62, 60, 59, 59, 55, 53, 50, 47, 46, 65, 64, 63, 63,
        62, 62, 60, 59, 59, 57, 57, 53, 51, 49, 46, 45, 64, 63, 63, 62, 60, 60,
        58, 57, 55, 54, 53, 51, 49, 48, 45, 44, 59, 59, 59, 60, 58, 57, 55, 53,
        49, 46, 45, 43, 41, 40, 38, 38, 57, 58, 59, 59, 57, 56, 53, 51, 48, 44,
        43, 41, 40, 38, 37, 36, 47, 48, 49, 50, 49, 49, 48, 46, 43, 39, 38, 35,
        34, 33, 31, 31, 43, 45, 46, 47, 46, 46, 46, 44, 41, 37, 37, 34, 33, 31,
        29, 29,
        /* Size 16x32 */
        64, 65, 65, 65, 65, 64, 64, 63, 59, 57, 57, 51, 47, 47, 43, 38, 65, 65,
        64, 64, 64, 63, 63, 62, 59, 58, 58, 52, 48, 48, 44, 39, 65, 64, 64, 64,
        64, 63, 63, 62, 59, 58, 58, 52, 48, 48, 45, 40, 65, 64, 64, 64, 64, 63,
        63, 62, 59, 58, 58, 52, 48, 48, 45, 40, 65, 64, 64, 64, 63, 63, 63, 62,
        59, 59, 59, 53, 49, 49, 46, 41, 65, 64, 63, 63, 63, 62, 62, 62, 60, 59,
        59, 54, 50, 50, 47, 41, 65, 64, 63, 63, 63, 62, 62, 62, 60, 59, 59, 54,
        50, 50, 47, 41, 65, 64, 63, 63, 63, 62, 62, 61, 59, 59, 59, 53, 50, 50,
        47, 41, 65, 64, 63, 63, 62, 60, 60, 59, 58, 57, 57, 52, 49, 49, 46, 41,
        64, 63, 63, 63, 62, 60, 60, 59, 57, 56, 56, 52, 49, 49, 46, 41, 64, 63,
        63, 63, 62, 60, 60, 59, 57, 56, 56, 52, 49, 49, 46, 41, 64, 63, 63, 63,
        61, 59, 59, 58, 55, 55, 55, 51, 48, 48, 46, 41, 64, 63, 62, 62, 60, 58,
        58, 57, 55, 53, 53, 50, 48, 48, 46, 41, 64, 63, 62, 62, 60, 58, 58, 57,
        55, 53, 53, 50, 48, 48, 46, 41, 63, 62, 62, 62, 59, 57, 57, 56, 53, 51,
        51, 48, 46, 46, 44, 40, 60, 60, 60, 60, 59, 55, 55, 54, 49, 48, 48, 45,
        43, 43, 41, 38, 60, 60, 60, 60, 59, 55, 55, 54, 49, 48, 48, 45, 43, 43,
        41, 38, 60, 60, 60, 60, 59, 55, 55, 53, 49, 47, 47, 44, 42, 42, 40, 37,
        58, 59, 59, 59, 57, 54, 54, 52, 46, 44, 44, 41, 39, 39, 37, 35, 57, 59,
        59, 59, 57, 53, 53, 51, 45, 43, 43, 40, 38, 38, 37, 34, 57, 59, 59, 59,
        57, 53, 53, 51, 45, 43, 43, 40, 38, 38, 37, 34, 54, 55, 56, 56, 55, 51,
        51, 49, 44, 41, 41, 38, 37, 37, 35, 33, 52, 53, 55, 55, 53, 51, 51, 48,
        43, 41, 41, 38, 35, 35, 34, 32, 52, 53, 55, 55, 53, 51, 51, 48, 43, 41,
        41, 38, 35, 35, 34, 32, 50, 51, 53, 53, 51, 49, 49, 47, 41, 40, 40, 36,
        34, 34, 33, 30, 47, 49, 50, 50, 49, 48, 48, 46, 40, 38, 38, 35, 33, 33,
        31, 29, 47, 49, 50, 50, 49, 48, 48, 46, 40, 38, 38, 35, 33, 33, 31, 29,
        46, 48, 49, 49, 48, 47, 47, 45, 40, 38, 38, 34, 32, 32, 31, 28, 44, 45,
        47, 47, 46, 45, 45, 43, 38, 37, 37, 33, 31, 31, 29, 27, 43, 45, 46, 46,
        45, 44, 44, 43, 38, 36, 36, 33, 31, 31, 29, 27, 43, 45, 46, 46, 45, 44,
        44, 43, 38, 36, 36, 33, 31, 31, 29, 27, 40, 42, 43, 43, 43, 42, 42, 41,
        37, 35, 35, 32, 29, 29, 28, 26,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 63, 60, 60, 60,
        58, 57, 57, 54, 52, 52, 50, 47, 47, 46, 44, 43, 43, 40, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 62, 60, 60, 60, 59, 59, 59, 55,
        53, 53, 51, 49, 49, 48, 45, 45, 45, 42, 65, 64, 64, 64, 64, 63, 63, 63,
        63, 63, 63, 63, 62, 62, 62, 60, 60, 60, 59, 59, 59, 56, 55, 55, 53, 50,
        50, 49, 47, 46, 46, 43, 65, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63,
        62, 62, 62, 60, 60, 60, 59, 59, 59, 56, 55, 55, 53, 50, 50, 49, 47, 46,
        46, 43, 65, 64, 64, 64, 63, 63, 63, 63, 62, 62, 62, 61, 60, 60, 59, 59,
        59, 59, 57, 57, 57, 55, 53, 53, 51, 49, 49, 48, 46, 45, 45, 43, 64, 63,
        63, 63, 63, 62, 62, 62, 60, 60, 60, 59, 58, 58, 57, 55, 55, 55, 54, 53,
        53, 51, 51, 51, 49, 48, 48, 47, 45, 44, 44, 42, 64, 63, 63, 63, 63, 62,
        62, 62, 60, 60, 60, 59, 58, 58, 57, 55, 55, 55, 54, 53, 53, 51, 51, 51,
        49, 48, 48, 47, 45, 44, 44, 42, 63, 62, 62, 62, 62, 62, 62, 61, 59, 59,
        59, 58, 57, 57, 56, 54, 54, 53, 52, 51, 51, 49, 48, 48, 47, 46, 46, 45,
        43, 43, 43, 41, 59, 59, 59, 59, 59, 60, 60, 59, 58, 57, 57, 55, 55, 55,
        53, 49, 49, 49, 46, 45, 45, 44, 43, 43, 41, 40, 40, 40, 38, 38, 38, 37,
        57, 58, 58, 58, 59, 59, 59, 59, 57, 56, 56, 55, 53, 53, 51, 48, 48, 47,
        44, 43, 43, 41, 41, 41, 40, 38, 38, 38, 37, 36, 36, 35, 57, 58, 58, 58,
        59, 59, 59, 59, 57, 56, 56, 55, 53, 53, 51, 48, 48, 47, 44, 43, 43, 41,
        41, 41, 40, 38, 38, 38, 37, 36, 36, 35, 51, 52, 52, 52, 53, 54, 54, 53,
        52, 52, 52, 51, 50, 50, 48, 45, 45, 44, 41, 40, 40, 38, 38, 38, 36, 35,
        35, 34, 33, 33, 33, 32, 47, 48, 48, 48, 49, 50, 50, 50, 49, 49, 49, 48,
        48, 48, 46, 43, 43, 42, 39, 38, 38, 37, 35, 35, 34, 33, 33, 32, 31, 31,
        31, 29, 47, 48, 48, 48, 49, 50, 50, 50, 49, 49, 49, 48, 48, 48, 46, 43,
        43, 42, 39, 38, 38, 37, 35, 35, 34, 33, 33, 32, 31, 31, 31, 29, 43, 44,
        45, 45, 46, 47, 47, 47, 46, 46, 46, 46, 46, 46, 44, 41, 41, 40, 37, 37,
        37, 35, 34, 34, 33, 31, 31, 31, 29, 29, 29, 28, 38, 39, 40, 40, 41, 41,
        41, 41, 41, 41, 41, 41, 41, 41, 40, 38, 38, 37, 35, 34, 34, 33, 32, 32,
        30, 29, 29, 28, 27, 27, 27, 26,
        /* Size 4x16 */
        65, 64, 57, 47, 64, 63, 58, 48, 64, 63, 59, 49, 64, 62, 59, 50, 64, 60,
        57, 49, 63, 60, 56, 49, 63, 58, 53, 48, 62, 57, 51, 46, 60, 55, 48, 43,
        59, 54, 44, 39, 59, 53, 43, 38, 53, 51, 41, 35, 51, 49, 40, 34, 49, 48,
        38, 33, 45, 45, 37, 31, 45, 44, 36, 31,
        /* Size 16x4 */
        65, 64, 64, 64, 64, 63, 63, 62, 60, 59, 59, 53, 51, 49, 45, 45, 64, 63,
        63, 62, 60, 60, 58, 57, 55, 54, 53, 51, 49, 48, 45, 44, 57, 58, 59, 59,
        57, 56, 53, 51, 48, 44, 43, 41, 40, 38, 37, 36, 47, 48, 49, 50, 49, 49,
        48, 46, 43, 39, 38, 35, 34, 33, 31, 31,
        /* Size 8x32 */
        64, 65, 65, 64, 59, 57, 47, 43, 65, 64, 64, 63, 59, 58, 48, 44, 65, 64,
        64, 63, 59, 58, 48, 45, 65, 64, 64, 63, 59, 58, 48, 45, 65, 64, 63, 63,
        59, 59, 49, 46, 65, 63, 63, 62, 60, 59, 50, 47, 65, 63, 63, 62, 60, 59,
        50, 47, 65, 63, 63, 62, 59, 59, 50, 47, 65, 63, 62, 60, 58, 57, 49, 46,
        64, 63, 62, 60, 57, 56, 49, 46, 64, 63, 62, 60, 57, 56, 49, 46, 64, 63,
        61, 59, 55, 55, 48, 46, 64, 62, 60, 58, 55, 53, 48, 46, 64, 62, 60, 58,
        55, 53, 48, 46, 63, 62, 59, 57, 53, 51, 46, 44, 60, 60, 59, 55, 49, 48,
        43, 41, 60, 60, 59, 55, 49, 48, 43, 41, 60, 60, 59, 55, 49, 47, 42, 40,
        58, 59, 57, 54, 46, 44, 39, 37, 57, 59, 57, 53, 45, 43, 38, 37, 57, 59,
        57, 53, 45, 43, 38, 37, 54, 56, 55, 51, 44, 41, 37, 35, 52, 55, 53, 51,
        43, 41, 35, 34, 52, 55, 53, 51, 43, 41, 35, 34, 50, 53, 51, 49, 41, 40,
        34, 33, 47, 50, 49, 48, 40, 38, 33, 31, 47, 50, 49, 48, 40, 38, 33, 31,
        46, 49, 48, 47, 40, 38, 32, 31, 44, 47, 46, 45, 38, 37, 31, 29, 43, 46,
        45, 44, 38, 36, 31, 29, 43, 46, 45, 44, 38, 36, 31, 29, 40, 43, 43, 42,
        37, 35, 29, 28,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 63, 60, 60, 60,
        58, 57, 57, 54, 52, 52, 50, 47, 47, 46, 44, 43, 43, 40, 65, 64, 64, 64,
        64, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 60, 60, 60, 59, 59, 59, 56,
        55, 55, 53, 50, 50, 49, 47, 46, 46, 43, 65, 64, 64, 64, 63, 63, 63, 63,
        62, 62, 62, 61, 60, 60, 59, 59, 59, 59, 57, 57, 57, 55, 53, 53, 51, 49,
        49, 48, 46, 45, 45, 43, 64, 63, 63, 63, 63, 62, 62, 62, 60, 60, 60, 59,
        58, 58, 57, 55, 55, 55, 54, 53, 53, 51, 51, 51, 49, 48, 48, 47, 45, 44,
        44, 42, 59, 59, 59, 59, 59, 60, 60, 59, 58, 57, 57, 55, 55, 55, 53, 49,
        49, 49, 46, 45, 45, 44, 43, 43, 41, 40, 40, 40, 38, 38, 38, 37, 57, 58,
        58, 58, 59, 59, 59, 59, 57, 56, 56, 55, 53, 53, 51, 48, 48, 47, 44, 43,
        43, 41, 41, 41, 40, 38, 38, 38, 37, 36, 36, 35, 47, 48, 48, 48, 49, 50,
        50, 50, 49, 49, 49, 48, 48, 48, 46, 43, 43, 42, 39, 38, 38, 37, 35, 35,
        34, 33, 33, 32, 31, 31, 31, 29, 43, 44, 45, 45, 46, 47, 47, 47, 46, 46,
        46, 46, 46, 46, 44, 41, 41, 40, 37, 37, 37, 35, 34, 34, 33, 31, 31, 31,
        29, 29, 29, 28 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        62, 45, 45, 38,
#endif
        /* Size 4x4 */
        65, 55, 43, 44, 55, 47, 43, 45, 43, 43, 39, 38, 44, 45, 38, 35,
        /* Size 8x8 */
        65, 66, 60, 55, 48, 43, 43, 41, 66, 64, 58, 51, 47, 45, 46, 44, 60, 58,
        53, 47, 45, 44, 45, 44, 55, 51, 47, 43, 43, 44, 45, 45, 48, 47, 45, 43,
        41, 41, 41, 41, 43, 45, 44, 44, 41, 38, 37, 37, 43, 46, 45, 45, 41, 37,
        35, 34, 41, 44, 44, 45, 41, 37, 34, 33,
        /* Size 16x16 */
        64, 66, 66, 67, 62, 61, 56, 53, 49, 43, 42, 42, 42, 42, 41, 41, 66, 65,
        65, 66, 60, 59, 53, 51, 48, 44, 43, 44, 44, 44, 43, 42, 66, 65, 65, 65,
        60, 59, 53, 51, 48, 45, 44, 44, 44, 45, 43, 43, 67, 66, 65, 64, 59, 58,
        51, 50, 48, 45, 44, 45, 45, 46, 45, 44, 62, 60, 60, 59, 55, 53, 48, 47,
        46, 44, 44, 45, 45, 45, 44, 44, 61, 59, 59, 58, 53, 53, 47, 47, 46, 44,
        43, 44, 45, 45, 44, 44, 56, 53, 53, 51, 48, 47, 43, 43, 43, 43, 43, 44,
        45, 45, 45, 45, 53, 51, 51, 50, 47, 47, 43, 43, 43, 42, 42, 43, 43, 44,
        43, 43, 49, 48, 48, 48, 46, 46, 43, 43, 42, 41, 41, 41, 41, 42, 41, 41,
        43, 44, 45, 45, 44, 44, 43, 42, 41, 39, 39, 39, 39, 39, 39, 39, 42, 43,
        44, 44, 44, 43, 43, 42, 41, 39, 39, 39, 38, 38, 38, 38, 42, 44, 44, 45,
        45, 44, 44, 43, 41, 39, 39, 38, 37, 37, 36, 36, 42, 44, 44, 45, 45, 45,
        45, 43, 41, 39, 38, 37, 37, 36, 36, 35, 42, 44, 45, 46, 45, 45, 45, 44,
        42, 39, 38, 37, 36, 35, 34, 34, 41, 43, 43, 45, 44, 44, 45, 43, 41, 39,
        38, 36, 36, 34, 34, 34, 41, 42, 43, 44, 44, 44, 45, 43, 41, 39, 38, 36,
        35, 34, 34, 33,
        /* Size 32x32 */
        64, 65, 66, 66, 66, 67, 67, 66, 62, 61, 61, 58, 56, 56, 53, 49, 49, 48,
        43, 42, 42, 42, 42, 42, 42, 42, 42, 41, 41, 41, 41, 40, 65, 65, 65, 65,
        66, 66, 66, 65, 61, 60, 60, 57, 55, 55, 52, 49, 49, 48, 44, 43, 43, 43,
        43, 43, 43, 43, 43, 43, 42, 41, 41, 41, 66, 65, 65, 65, 65, 66, 66, 64,
        60, 59, 59, 55, 53, 53, 51, 48, 48, 47, 44, 43, 43, 43, 44, 44, 44, 44,
        44, 44, 43, 42, 42, 41, 66, 65, 65, 65, 65, 66, 66, 64, 60, 59, 59, 55,
        53, 53, 51, 48, 48, 47, 44, 43, 43, 43, 44, 44, 44, 44, 44, 44, 43, 42,
        42, 41, 66, 66, 65, 65, 65, 65, 65, 64, 60, 59, 59, 55, 53, 53, 51, 48,
        48, 47, 45, 44, 44, 44, 44, 44, 44, 45, 45, 44, 43, 43, 43, 42, 67, 66,
        66, 66, 65, 64, 64, 63, 59, 58, 58, 54, 51, 51, 50, 48, 48, 47, 45, 44,
        44, 45, 45, 45, 45, 46, 46, 45, 45, 44, 44, 44, 67, 66, 66, 66, 65, 64,
        64, 63, 59, 58, 58, 54, 51, 51, 50, 48, 48, 47, 45, 44, 44, 45, 45, 45,
        45, 46, 46, 45, 45, 44, 44, 44, 66, 65, 64, 64, 64, 63, 63, 62, 58, 57,
        57, 53, 51, 51, 49, 48, 48, 47, 45, 44, 44, 45, 45, 45, 45, 46, 46, 45,
        45, 44, 44, 44, 62, 61, 60, 60, 60, 59, 59, 58, 55, 53, 53, 50, 48, 48,
        47, 46, 46, 46, 44, 44, 44, 44, 45, 45, 45, 45, 45, 45, 44, 44, 44, 44,
        61, 60, 59, 59, 59, 58, 58, 57, 53, 53, 53, 49, 47, 47, 47, 46, 46, 45,
        44, 43, 43, 44, 44, 44, 45, 45, 45, 45, 44, 44, 44, 44, 61, 60, 59, 59,
        59, 58, 58, 57, 53, 53, 53, 49, 47, 47, 47, 46, 46, 45, 44, 43, 43, 44,
        44, 44, 45, 45, 45, 45, 44, 44, 44, 44, 58, 57, 55, 55, 55, 54, 54, 53,
        50, 49, 49, 47, 45, 45, 45, 44, 44, 44, 43, 43, 43, 44, 44, 44, 45, 45,
        45, 45, 45, 44, 44, 44, 56, 55, 53, 53, 53, 51, 51, 51, 48, 47, 47, 45,
        43, 43, 43, 43, 43, 43, 43, 43, 43, 44, 44, 44, 45, 45, 45, 45, 45, 45,
        45, 44, 56, 55, 53, 53, 53, 51, 51, 51, 48, 47, 47, 45, 43, 43, 43, 43,
        43, 43, 43, 43, 43, 44, 44, 44, 45, 45, 45, 45, 45, 45, 45, 44, 53, 52,
        51, 51, 51, 50, 50, 49, 47, 47, 47, 45, 43, 43, 43, 43, 43, 43, 42, 42,
        42, 43, 43, 43, 43, 44, 44, 44, 43, 43, 43, 43, 49, 49, 48, 48, 48, 48,
        48, 48, 46, 46, 46, 44, 43, 43, 43, 42, 42, 42, 41, 41, 41, 41, 41, 41,
        41, 42, 42, 41, 41, 41, 41, 41, 49, 49, 48, 48, 48, 48, 48, 48, 46, 46,
        46, 44, 43, 43, 43, 42, 42, 42, 41, 41, 41, 41, 41, 41, 41, 42, 42, 41,
        41, 41, 41, 41, 48, 48, 47, 47, 47, 47, 47, 47, 46, 45, 45, 44, 43, 43,
        43, 42, 42, 41, 41, 40, 40, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 40,
        43, 44, 44, 44, 45, 45, 45, 45, 44, 44, 44, 43, 43, 43, 42, 41, 41, 41,
        39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 38, 42, 43, 43, 43,
        44, 44, 44, 44, 44, 43, 43, 43, 43, 43, 42, 41, 41, 40, 39, 39, 39, 39,
        39, 39, 38, 38, 38, 38, 38, 38, 38, 38, 42, 43, 43, 43, 44, 44, 44, 44,
        44, 43, 43, 43, 43, 43, 42, 41, 41, 40, 39, 39, 39, 39, 39, 39, 38, 38,
        38, 38, 38, 38, 38, 38, 42, 43, 43, 43, 44, 45, 45, 45, 44, 44, 44, 44,
        44, 44, 43, 41, 41, 41, 39, 39, 39, 38, 38, 38, 38, 37, 37, 37, 37, 37,
        37, 37, 42, 43, 44, 44, 44, 45, 45, 45, 45, 44, 44, 44, 44, 44, 43, 41,
        41, 41, 39, 39, 39, 38, 38, 38, 37, 37, 37, 37, 36, 36, 36, 36, 42, 43,
        44, 44, 44, 45, 45, 45, 45, 44, 44, 44, 44, 44, 43, 41, 41, 41, 39, 39,
        39, 38, 38, 38, 37, 37, 37, 37, 36, 36, 36, 36, 42, 43, 44, 44, 44, 45,
        45, 45, 45, 45, 45, 45, 45, 45, 43, 41, 41, 41, 39, 38, 38, 38, 37, 37,
        37, 36, 36, 36, 36, 35, 35, 35, 42, 43, 44, 44, 45, 46, 46, 46, 45, 45,
        45, 45, 45, 45, 44, 42, 42, 41, 39, 38, 38, 37, 37, 37, 36, 35, 35, 35,
        34, 34, 34, 34, 42, 43, 44, 44, 45, 46, 46, 46, 45, 45, 45, 45, 45, 45,
        44, 42, 42, 41, 39, 38, 38, 37, 37, 37, 36, 35, 35, 35, 34, 34, 34, 34,
        41, 43, 44, 44, 44, 45, 45, 45, 45, 45, 45, 45, 45, 45, 44, 41, 41, 41,
        39, 38, 38, 37, 37, 37, 36, 35, 35, 35, 34, 34, 34, 34, 41, 42, 43, 43,
        43, 45, 45, 45, 44, 44, 44, 45, 45, 45, 43, 41, 41, 41, 39, 38, 38, 37,
        36, 36, 36, 34, 34, 34, 34, 34, 34, 33, 41, 41, 42, 42, 43, 44, 44, 44,
        44, 44, 44, 44, 45, 45, 43, 41, 41, 41, 39, 38, 38, 37, 36, 36, 35, 34,
        34, 34, 34, 33, 33, 33, 41, 41, 42, 42, 43, 44, 44, 44, 44, 44, 44, 44,
        45, 45, 43, 41, 41, 41, 39, 38, 38, 37, 36, 36, 35, 34, 34, 34, 34, 33,
        33, 33, 40, 41, 41, 41, 42, 44, 44, 44, 44, 44, 44, 44, 44, 44, 43, 41,
        41, 40, 38, 38, 38, 37, 36, 36, 35, 34, 34, 34, 33, 33, 33, 32,
        /* Size 4x8 */
        65, 54, 43, 43, 65, 51, 45, 45, 59, 48, 44, 45, 53, 44, 43, 45, 47, 43,
        41, 41, 44, 43, 39, 37, 44, 45, 38, 35, 43, 44, 38, 34,
        /* Size 8x4 */
        65, 65, 59, 53, 47, 44, 44, 43, 54, 51, 48, 44, 43, 43, 45, 44, 43, 45,
        44, 43, 41, 39, 38, 38, 43, 45, 45, 45, 41, 37, 35, 34,
        /* Size 8x16 */
        64, 66, 62, 55, 45, 43, 42, 41, 65, 65, 60, 53, 46, 44, 44, 42, 66, 64,
        59, 53, 46, 44, 44, 43, 67, 64, 59, 51, 46, 45, 45, 44, 62, 59, 55, 48,
        45, 44, 45, 44, 61, 57, 53, 48, 45, 44, 45, 44, 55, 51, 48, 44, 43, 43,
        45, 45, 53, 50, 47, 43, 42, 42, 44, 43, 49, 48, 46, 43, 41, 41, 41, 41,
        43, 45, 44, 43, 40, 39, 39, 39, 42, 44, 44, 43, 39, 39, 38, 38, 42, 45,
        44, 44, 40, 39, 37, 36, 42, 45, 45, 44, 40, 39, 36, 36, 42, 45, 45, 45,
        40, 38, 35, 34, 41, 44, 44, 44, 39, 38, 34, 34, 41, 44, 44, 44, 39, 38,
        34, 33,
        /* Size 16x8 */
        64, 65, 66, 67, 62, 61, 55, 53, 49, 43, 42, 42, 42, 42, 41, 41, 66, 65,
        64, 64, 59, 57, 51, 50, 48, 45, 44, 45, 45, 45, 44, 44, 62, 60, 59, 59,
        55, 53, 48, 47, 46, 44, 44, 44, 45, 45, 44, 44, 55, 53, 53, 51, 48, 48,
        44, 43, 43, 43, 43, 44, 44, 45, 44, 44, 45, 46, 46, 46, 45, 45, 43, 42,
        41, 40, 39, 40, 40, 40, 39, 39, 43, 44, 44, 45, 44, 44, 43, 42, 41, 39,
        39, 39, 39, 38, 38, 38, 42, 44, 44, 45, 45, 45, 45, 44, 41, 39, 38, 37,
        36, 35, 34, 34, 41, 42, 43, 44, 44, 44, 45, 43, 41, 39, 38, 36, 36, 34,
        34, 33,
        /* Size 16x32 */
        64, 66, 66, 66, 62, 55, 55, 53, 45, 43, 43, 42, 42, 42, 41, 39, 65, 65,
        65, 65, 61, 54, 54, 52, 46, 43, 43, 43, 43, 43, 42, 40, 65, 65, 65, 65,
        60, 53, 53, 51, 46, 44, 44, 44, 44, 44, 42, 41, 65, 65, 65, 65, 60, 53,
        53, 51, 46, 44, 44, 44, 44, 44, 42, 41, 66, 65, 64, 64, 59, 53, 53, 51,
        46, 44, 44, 44, 44, 44, 43, 42, 67, 65, 64, 64, 59, 51, 51, 50, 46, 45,
        45, 45, 45, 45, 44, 43, 67, 65, 64, 64, 59, 51, 51, 50, 46, 45, 45, 45,
        45, 45, 44, 43, 66, 64, 62, 62, 58, 51, 51, 49, 46, 45, 45, 45, 45, 45,
        44, 43, 62, 60, 59, 59, 55, 48, 48, 48, 45, 44, 44, 45, 45, 45, 44, 43,
        61, 59, 57, 57, 53, 48, 48, 47, 45, 44, 44, 44, 45, 45, 44, 43, 61, 59,
        57, 57, 53, 48, 48, 47, 45, 44, 44, 44, 45, 45, 44, 43, 58, 55, 53, 53,
        50, 45, 45, 45, 44, 43, 43, 44, 45, 45, 44, 43, 55, 53, 51, 51, 48, 44,
        44, 43, 43, 43, 43, 44, 45, 45, 45, 44, 55, 53, 51, 51, 48, 44, 44, 43,
        43, 43, 43, 44, 45, 45, 45, 44, 53, 51, 50, 50, 47, 43, 43, 43, 42, 42,
        42, 43, 44, 44, 43, 42, 49, 48, 48, 48, 46, 43, 43, 43, 41, 41, 41, 41,
        41, 41, 41, 41, 49, 48, 48, 48, 46, 43, 43, 43, 41, 41, 41, 41, 41, 41,
        41, 41, 48, 47, 47, 47, 46, 43, 43, 43, 41, 41, 41, 41, 41, 41, 41, 40,
        43, 44, 45, 45, 44, 43, 43, 42, 40, 39, 39, 39, 39, 39, 39, 38, 42, 43,
        44, 44, 44, 43, 43, 42, 39, 39, 39, 38, 38, 38, 38, 38, 42, 43, 44, 44,
        44, 43, 43, 42, 39, 39, 39, 38, 38, 38, 38, 38, 42, 44, 45, 45, 44, 43,
        43, 42, 39, 39, 39, 38, 37, 37, 37, 36, 42, 44, 45, 45, 44, 44, 44, 43,
        40, 39, 39, 38, 37, 37, 36, 36, 42, 44, 45, 45, 44, 44, 44, 43, 40, 39,
        39, 38, 37, 37, 36, 36, 42, 44, 45, 45, 45, 44, 44, 43, 40, 39, 39, 37,
        36, 36, 36, 35, 42, 44, 45, 45, 45, 45, 45, 44, 40, 38, 38, 36, 35, 35,
        34, 33, 42, 44, 45, 45, 45, 45, 45, 44, 40, 38, 38, 36, 35, 35, 34, 33,
        42, 44, 45, 45, 45, 45, 45, 43, 39, 38, 38, 36, 35, 35, 34, 33, 41, 43,
        44, 44, 44, 44, 44, 43, 39, 38, 38, 36, 34, 34, 34, 32, 41, 43, 44, 44,
        44, 44, 44, 43, 39, 38, 38, 36, 34, 34, 33, 32, 41, 43, 44, 44, 44, 44,
        44, 43, 39, 38, 38, 36, 34, 34, 33, 32, 40, 42, 43, 43, 43, 44, 44, 42,
        39, 38, 38, 35, 34, 34, 33, 31,
        /* Size 32x16 */
        64, 65, 65, 65, 66, 67, 67, 66, 62, 61, 61, 58, 55, 55, 53, 49, 49, 48,
        43, 42, 42, 42, 42, 42, 42, 42, 42, 42, 41, 41, 41, 40, 66, 65, 65, 65,
        65, 65, 65, 64, 60, 59, 59, 55, 53, 53, 51, 48, 48, 47, 44, 43, 43, 44,
        44, 44, 44, 44, 44, 44, 43, 43, 43, 42, 66, 65, 65, 65, 64, 64, 64, 62,
        59, 57, 57, 53, 51, 51, 50, 48, 48, 47, 45, 44, 44, 45, 45, 45, 45, 45,
        45, 45, 44, 44, 44, 43, 66, 65, 65, 65, 64, 64, 64, 62, 59, 57, 57, 53,
        51, 51, 50, 48, 48, 47, 45, 44, 44, 45, 45, 45, 45, 45, 45, 45, 44, 44,
        44, 43, 62, 61, 60, 60, 59, 59, 59, 58, 55, 53, 53, 50, 48, 48, 47, 46,
        46, 46, 44, 44, 44, 44, 44, 44, 45, 45, 45, 45, 44, 44, 44, 43, 55, 54,
        53, 53, 53, 51, 51, 51, 48, 48, 48, 45, 44, 44, 43, 43, 43, 43, 43, 43,
        43, 43, 44, 44, 44, 45, 45, 45, 44, 44, 44, 44, 55, 54, 53, 53, 53, 51,
        51, 51, 48, 48, 48, 45, 44, 44, 43, 43, 43, 43, 43, 43, 43, 43, 44, 44,
        44, 45, 45, 45, 44, 44, 44, 44, 53, 52, 51, 51, 51, 50, 50, 49, 48, 47,
        47, 45, 43, 43, 43, 43, 43, 43, 42, 42, 42, 42, 43, 43, 43, 44, 44, 43,
        43, 43, 43, 42, 45, 46, 46, 46, 46, 46, 46, 46, 45, 45, 45, 44, 43, 43,
        42, 41, 41, 41, 40, 39, 39, 39, 40, 40, 40, 40, 40, 39, 39, 39, 39, 39,
        43, 43, 44, 44, 44, 45, 45, 45, 44, 44, 44, 43, 43, 43, 42, 41, 41, 41,
        39, 39, 39, 39, 39, 39, 39, 38, 38, 38, 38, 38, 38, 38, 43, 43, 44, 44,
        44, 45, 45, 45, 44, 44, 44, 43, 43, 43, 42, 41, 41, 41, 39, 39, 39, 39,
        39, 39, 39, 38, 38, 38, 38, 38, 38, 38, 42, 43, 44, 44, 44, 45, 45, 45,
        45, 44, 44, 44, 44, 44, 43, 41, 41, 41, 39, 38, 38, 38, 38, 38, 37, 36,
        36, 36, 36, 36, 36, 35, 42, 43, 44, 44, 44, 45, 45, 45, 45, 45, 45, 45,
        45, 45, 44, 41, 41, 41, 39, 38, 38, 37, 37, 37, 36, 35, 35, 35, 34, 34,
        34, 34, 42, 43, 44, 44, 44, 45, 45, 45, 45, 45, 45, 45, 45, 45, 44, 41,
        41, 41, 39, 38, 38, 37, 37, 37, 36, 35, 35, 35, 34, 34, 34, 34, 41, 42,
        42, 42, 43, 44, 44, 44, 44, 44, 44, 44, 45, 45, 43, 41, 41, 41, 39, 38,
        38, 37, 36, 36, 36, 34, 34, 34, 34, 33, 33, 33, 39, 40, 41, 41, 42, 43,
        43, 43, 43, 43, 43, 43, 44, 44, 42, 41, 41, 40, 38, 38, 38, 36, 36, 36,
        35, 33, 33, 33, 32, 32, 32, 31,
        /* Size 4x16 */
        66, 55, 43, 42, 65, 53, 44, 44, 65, 53, 44, 44, 65, 51, 45, 45, 60, 48,
        44, 45, 59, 48, 44, 45, 53, 44, 43, 45, 51, 43, 42, 44, 48, 43, 41, 41,
        44, 43, 39, 39, 43, 43, 39, 38, 44, 44, 39, 37, 44, 44, 39, 36, 44, 45,
        38, 35, 43, 44, 38, 34, 43, 44, 38, 34,
        /* Size 16x4 */
        66, 65, 65, 65, 60, 59, 53, 51, 48, 44, 43, 44, 44, 44, 43, 43, 55, 53,
        53, 51, 48, 48, 44, 43, 43, 43, 43, 44, 44, 45, 44, 44, 43, 44, 44, 45,
        44, 44, 43, 42, 41, 39, 39, 39, 39, 38, 38, 38, 42, 44, 44, 45, 45, 45,
        45, 44, 41, 39, 38, 37, 36, 35, 34, 34,
        /* Size 8x32 */
        64, 66, 62, 55, 45, 43, 42, 41, 65, 65, 61, 54, 46, 43, 43, 42, 65, 65,
        60, 53, 46, 44, 44, 42, 65, 65, 60, 53, 46, 44, 44, 42, 66, 64, 59, 53,
        46, 44, 44, 43, 67, 64, 59, 51, 46, 45, 45, 44, 67, 64, 59, 51, 46, 45,
        45, 44, 66, 62, 58, 51, 46, 45, 45, 44, 62, 59, 55, 48, 45, 44, 45, 44,
        61, 57, 53, 48, 45, 44, 45, 44, 61, 57, 53, 48, 45, 44, 45, 44, 58, 53,
        50, 45, 44, 43, 45, 44, 55, 51, 48, 44, 43, 43, 45, 45, 55, 51, 48, 44,
        43, 43, 45, 45, 53, 50, 47, 43, 42, 42, 44, 43, 49, 48, 46, 43, 41, 41,
        41, 41, 49, 48, 46, 43, 41, 41, 41, 41, 48, 47, 46, 43, 41, 41, 41, 41,
        43, 45, 44, 43, 40, 39, 39, 39, 42, 44, 44, 43, 39, 39, 38, 38, 42, 44,
        44, 43, 39, 39, 38, 38, 42, 45, 44, 43, 39, 39, 37, 37, 42, 45, 44, 44,
        40, 39, 37, 36, 42, 45, 44, 44, 40, 39, 37, 36, 42, 45, 45, 44, 40, 39,
        36, 36, 42, 45, 45, 45, 40, 38, 35, 34, 42, 45, 45, 45, 40, 38, 35, 34,
        42, 45, 45, 45, 39, 38, 35, 34, 41, 44, 44, 44, 39, 38, 34, 34, 41, 44,
        44, 44, 39, 38, 34, 33, 41, 44, 44, 44, 39, 38, 34, 33, 40, 43, 43, 44,
        39, 38, 34, 33,
        /* Size 32x8 */
        64, 65, 65, 65, 66, 67, 67, 66, 62, 61, 61, 58, 55, 55, 53, 49, 49, 48,
        43, 42, 42, 42, 42, 42, 42, 42, 42, 42, 41, 41, 41, 40, 66, 65, 65, 65,
        64, 64, 64, 62, 59, 57, 57, 53, 51, 51, 50, 48, 48, 47, 45, 44, 44, 45,
        45, 45, 45, 45, 45, 45, 44, 44, 44, 43, 62, 61, 60, 60, 59, 59, 59, 58,
        55, 53, 53, 50, 48, 48, 47, 46, 46, 46, 44, 44, 44, 44, 44, 44, 45, 45,
        45, 45, 44, 44, 44, 43, 55, 54, 53, 53, 53, 51, 51, 51, 48, 48, 48, 45,
        44, 44, 43, 43, 43, 43, 43, 43, 43, 43, 44, 44, 44, 45, 45, 45, 44, 44,
        44, 44, 45, 46, 46, 46, 46, 46, 46, 46, 45, 45, 45, 44, 43, 43, 42, 41,
        41, 41, 40, 39, 39, 39, 40, 40, 40, 40, 40, 39, 39, 39, 39, 39, 43, 43,
        44, 44, 44, 45, 45, 45, 44, 44, 44, 43, 43, 43, 42, 41, 41, 41, 39, 39,
        39, 39, 39, 39, 39, 38, 38, 38, 38, 38, 38, 38, 42, 43, 44, 44, 44, 45,
        45, 45, 45, 45, 45, 45, 45, 45, 44, 41, 41, 41, 39, 38, 38, 37, 37, 37,
        36, 35, 35, 35, 34, 34, 34, 34, 41, 42, 42, 42, 43, 44, 44, 44, 44, 44,
        44, 44, 45, 45, 43, 41, 41, 41, 39, 38, 38, 37, 36, 36, 36, 34, 34, 34,
        34, 33, 33, 33 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        63, 59, 59, 42,
#endif
        /* Size 4x4 */
        64, 64, 60, 53, 64, 61, 58, 53, 60, 58, 52, 46, 53, 53, 46, 38,
        /* Size 8x8 */
        65, 65, 64, 64, 62, 59, 55, 50, 65, 64, 63, 63, 62, 59, 56, 52, 64, 63,
        63, 62, 60, 59, 55, 51, 64, 63, 62, 60, 58, 56, 53, 50, 62, 62, 60, 58,
        55, 53, 49, 47, 59, 59, 59, 56, 53, 48, 44, 42, 55, 56, 55, 53, 49, 44,
        40, 38, 50, 52, 51, 50, 47, 42, 38, 35,
        /* Size 16x16 */
        64, 65, 65, 65, 65, 65, 65, 64, 64, 60, 60, 57, 57, 52, 52, 47, 65, 64,
        64, 64, 64, 64, 64, 63, 63, 60, 60, 59, 59, 53, 53, 48, 65, 64, 64, 64,
        64, 64, 64, 63, 63, 60, 60, 59, 59, 53, 53, 48, 65, 64, 64, 63, 63, 63,
        63, 63, 63, 61, 61, 59, 59, 55, 55, 50, 65, 64, 64, 63, 63, 63, 63, 63,
        63, 61, 61, 59, 59, 55, 55, 50, 65, 64, 64, 63, 63, 61, 61, 60, 60, 58,
        58, 56, 56, 53, 53, 49, 65, 64, 64, 63, 63, 61, 61, 60, 60, 58, 58, 56,
        56, 53, 53, 49, 64, 63, 63, 63, 63, 60, 60, 58, 58, 55, 55, 53, 53, 51,
        51, 48, 64, 63, 63, 63, 63, 60, 60, 58, 58, 55, 55, 53, 53, 51, 51, 48,
        60, 60, 60, 61, 61, 58, 58, 55, 55, 52, 52, 48, 48, 46, 46, 43, 60, 60,
        60, 61, 61, 58, 58, 55, 55, 52, 52, 48, 48, 46, 46, 43, 57, 59, 59, 59,
        59, 56, 56, 53, 53, 48, 48, 42, 42, 41, 41, 38, 57, 59, 59, 59, 59, 56,
        56, 53, 53, 48, 48, 42, 42, 41, 41, 38, 52, 53, 53, 55, 55, 53, 53, 51,
        51, 46, 46, 41, 41, 38, 38, 35, 52, 53, 53, 55, 55, 53, 53, 51, 51, 46,
        46, 41, 41, 38, 38, 35, 47, 48, 48, 50, 50, 49, 49, 48, 48, 43, 43, 38,
        38, 35, 35, 32,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 62,
        60, 60, 60, 59, 57, 57, 57, 55, 52, 52, 52, 49, 47, 47, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 62, 60, 60, 60, 59,
        58, 58, 58, 55, 53, 53, 53, 50, 48, 48, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 63, 63, 63, 63, 62, 60, 60, 60, 59, 59, 59, 59, 55,
        53, 53, 53, 51, 48, 48, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 63, 63, 63, 63, 62, 60, 60, 60, 59, 59, 59, 59, 55, 53, 53, 53, 51,
        48, 48, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63,
        63, 62, 60, 60, 60, 59, 59, 59, 59, 55, 53, 53, 53, 51, 48, 48, 65, 65,
        64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 60, 60,
        60, 59, 59, 59, 59, 56, 54, 54, 54, 52, 49, 49, 65, 65, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 61, 61, 61, 60, 59, 59,
        59, 57, 55, 55, 55, 53, 50, 50, 65, 65, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 62, 61, 61, 61, 60, 59, 59, 59, 57, 55, 55,
        55, 53, 50, 50, 65, 65, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 62, 61, 61, 61, 60, 59, 59, 59, 57, 55, 55, 55, 53, 50, 50,
        65, 64, 64, 64, 64, 63, 63, 63, 63, 63, 62, 62, 62, 62, 61, 61, 61, 60,
        59, 59, 59, 59, 58, 58, 58, 55, 53, 53, 53, 51, 49, 49, 65, 64, 64, 64,
        64, 63, 63, 63, 63, 62, 61, 61, 61, 61, 60, 60, 60, 59, 58, 58, 58, 57,
        56, 56, 56, 54, 53, 53, 53, 51, 49, 49, 65, 64, 64, 64, 64, 63, 63, 63,
        63, 62, 61, 61, 61, 61, 60, 60, 60, 59, 58, 58, 58, 57, 56, 56, 56, 54,
        53, 53, 53, 51, 49, 49, 65, 64, 64, 64, 64, 63, 63, 63, 63, 62, 61, 61,
        61, 61, 60, 60, 60, 59, 58, 58, 58, 57, 56, 56, 56, 54, 53, 53, 53, 51,
        49, 49, 64, 64, 63, 63, 63, 63, 63, 63, 63, 62, 61, 61, 61, 60, 59, 59,
        59, 58, 57, 57, 57, 56, 55, 55, 55, 53, 51, 51, 51, 50, 48, 48, 64, 64,
        63, 63, 63, 63, 63, 63, 63, 61, 60, 60, 60, 59, 58, 58, 58, 57, 55, 55,
        55, 55, 53, 53, 53, 52, 51, 51, 51, 49, 48, 48, 64, 64, 63, 63, 63, 63,
        63, 63, 63, 61, 60, 60, 60, 59, 58, 58, 58, 57, 55, 55, 55, 55, 53, 53,
        53, 52, 51, 51, 51, 49, 48, 48, 64, 64, 63, 63, 63, 63, 63, 63, 63, 61,
        60, 60, 60, 59, 58, 58, 58, 57, 55, 55, 55, 55, 53, 53, 53, 52, 51, 51,
        51, 49, 48, 48, 62, 62, 62, 62, 62, 62, 62, 62, 62, 60, 59, 59, 59, 58,
        57, 57, 57, 55, 54, 54, 54, 53, 51, 51, 51, 49, 48, 48, 48, 47, 46, 46,
        60, 60, 60, 60, 60, 60, 61, 61, 61, 59, 58, 58, 58, 57, 55, 55, 55, 54,
        52, 52, 52, 50, 48, 48, 48, 47, 46, 46, 46, 44, 43, 43, 60, 60, 60, 60,
        60, 60, 61, 61, 61, 59, 58, 58, 58, 57, 55, 55, 55, 54, 52, 52, 52, 50,
        48, 48, 48, 47, 46, 46, 46, 44, 43, 43, 60, 60, 60, 60, 60, 60, 61, 61,
        61, 59, 58, 58, 58, 57, 55, 55, 55, 54, 52, 52, 52, 50, 48, 48, 48, 47,
        46, 46, 46, 44, 43, 43, 59, 59, 59, 59, 59, 59, 60, 60, 60, 59, 57, 57,
        57, 56, 55, 55, 55, 53, 50, 50, 50, 48, 45, 45, 45, 44, 43, 43, 43, 42,
        41, 41, 57, 58, 59, 59, 59, 59, 59, 59, 59, 58, 56, 56, 56, 55, 53, 53,
        53, 51, 48, 48, 48, 45, 42, 42, 42, 41, 41, 41, 41, 39, 38, 38, 57, 58,
        59, 59, 59, 59, 59, 59, 59, 58, 56, 56, 56, 55, 53, 53, 53, 51, 48, 48,
        48, 45, 42, 42, 42, 41, 41, 41, 41, 39, 38, 38, 57, 58, 59, 59, 59, 59,
        59, 59, 59, 58, 56, 56, 56, 55, 53, 53, 53, 51, 48, 48, 48, 45, 42, 42,
        42, 41, 41, 41, 41, 39, 38, 38, 55, 55, 55, 55, 55, 56, 57, 57, 57, 55,
        54, 54, 54, 53, 52, 52, 52, 49, 47, 47, 47, 44, 41, 41, 41, 40, 39, 39,
        39, 38, 37, 37, 52, 53, 53, 53, 53, 54, 55, 55, 55, 53, 53, 53, 53, 51,
        51, 51, 51, 48, 46, 46, 46, 43, 41, 41, 41, 39, 38, 38, 38, 37, 35, 35,
        52, 53, 53, 53, 53, 54, 55, 55, 55, 53, 53, 53, 53, 51, 51, 51, 51, 48,
        46, 46, 46, 43, 41, 41, 41, 39, 38, 38, 38, 37, 35, 35, 52, 53, 53, 53,
        53, 54, 55, 55, 55, 53, 53, 53, 53, 51, 51, 51, 51, 48, 46, 46, 46, 43,
        41, 41, 41, 39, 38, 38, 38, 37, 35, 35, 49, 50, 51, 51, 51, 52, 53, 53,
        53, 51, 51, 51, 51, 50, 49, 49, 49, 47, 44, 44, 44, 42, 39, 39, 39, 38,
        37, 37, 37, 35, 34, 34, 47, 48, 48, 48, 48, 49, 50, 50, 50, 49, 49, 49,
        49, 48, 48, 48, 48, 46, 43, 43, 43, 41, 38, 38, 38, 37, 35, 35, 35, 34,
        32, 32, 47, 48, 48, 48, 48, 49, 50, 50, 50, 49, 49, 49, 49, 48, 48, 48,
        48, 46, 43, 43, 43, 41, 38, 38, 38, 37, 35, 35, 35, 34, 32, 32,
        /* Size 4x8 */
        65, 64, 60, 53, 64, 63, 60, 54, 64, 62, 59, 53, 63, 61, 57, 51, 62, 59,
        54, 48, 59, 57, 50, 43, 55, 54, 47, 39, 51, 51, 44, 37,
        /* Size 8x4 */
        65, 64, 64, 63, 62, 59, 55, 51, 64, 63, 62, 61, 59, 57, 54, 51, 60, 60,
        59, 57, 54, 50, 47, 44, 53, 54, 53, 51, 48, 43, 39, 37,
        /* Size 8x16 */
        64, 65, 65, 64, 64, 57, 57, 47, 65, 64, 64, 63, 63, 58, 58, 48, 65, 64,
        64, 63, 63, 58, 58, 48, 65, 63, 63, 62, 62, 59, 59, 50, 65, 63, 63, 62,
        62, 59, 59, 50, 64, 63, 63, 60, 60, 56, 56, 49, 64, 63, 63, 60, 60, 56,
        56, 49, 64, 62, 62, 58, 58, 53, 53, 48, 64, 62, 62, 58, 58, 53, 53, 48,
        60, 60, 60, 55, 55, 48, 48, 43, 60, 60, 60, 55, 55, 48, 48, 43, 57, 59,
        59, 53, 53, 43, 43, 38, 57, 59, 59, 53, 53, 43, 43, 38, 52, 55, 55, 51,
        51, 41, 41, 35, 52, 55, 55, 51, 51, 41, 41, 35, 47, 50, 50, 48, 48, 38,
        38, 33,
        /* Size 16x8 */
        64, 65, 65, 65, 65, 64, 64, 64, 64, 60, 60, 57, 57, 52, 52, 47, 65, 64,
        64, 63, 63, 63, 63, 62, 62, 60, 60, 59, 59, 55, 55, 50, 65, 64, 64, 63,
        63, 63, 63, 62, 62, 60, 60, 59, 59, 55, 55, 50, 64, 63, 63, 62, 62, 60,
        60, 58, 58, 55, 55, 53, 53, 51, 51, 48, 64, 63, 63, 62, 62, 60, 60, 58,
        58, 55, 55, 53, 53, 51, 51, 48, 57, 58, 58, 59, 59, 56, 56, 53, 53, 48,
        48, 43, 43, 41, 41, 38, 57, 58, 58, 59, 59, 56, 56, 53, 53, 48, 48, 43,
        43, 41, 41, 38, 47, 48, 48, 50, 50, 49, 49, 48, 48, 43, 43, 38, 38, 35,
        35, 33,
        /* Size 16x32 */
        64, 65, 65, 65, 65, 64, 64, 64, 64, 60, 57, 57, 57, 52, 47, 47, 65, 65,
        65, 65, 65, 64, 64, 64, 64, 60, 58, 58, 58, 53, 48, 48, 65, 64, 64, 64,
        64, 64, 63, 63, 63, 60, 58, 58, 58, 53, 48, 48, 65, 64, 64, 64, 64, 64,
        63, 63, 63, 60, 58, 58, 58, 53, 48, 48, 65, 64, 64, 64, 64, 64, 63, 63,
        63, 60, 58, 58, 58, 53, 48, 48, 65, 64, 64, 64, 64, 63, 63, 63, 63, 60,
        59, 59, 59, 54, 49, 49, 65, 64, 63, 63, 63, 63, 62, 62, 62, 61, 59, 59,
        59, 55, 50, 50, 65, 64, 63, 63, 63, 63, 62, 62, 62, 61, 59, 59, 59, 55,
        50, 50, 65, 64, 63, 63, 63, 63, 62, 62, 62, 61, 59, 59, 59, 55, 50, 50,
        65, 64, 63, 63, 63, 62, 61, 61, 61, 59, 58, 58, 58, 53, 49, 49, 64, 64,
        63, 63, 63, 61, 60, 60, 60, 58, 56, 56, 56, 53, 49, 49, 64, 64, 63, 63,
        63, 61, 60, 60, 60, 58, 56, 56, 56, 53, 49, 49, 64, 64, 63, 63, 63, 61,
        60, 60, 60, 58, 56, 56, 56, 53, 49, 49, 64, 63, 63, 63, 63, 61, 59, 59,
        59, 57, 55, 55, 55, 51, 48, 48, 64, 63, 62, 62, 62, 60, 58, 58, 58, 55,
        53, 53, 53, 51, 48, 48, 64, 63, 62, 62, 62, 60, 58, 58, 58, 55, 53, 53,
        53, 51, 48, 48, 64, 63, 62, 62, 62, 60, 58, 58, 58, 55, 53, 53, 53, 51,
        48, 48, 62, 62, 61, 61, 61, 59, 57, 57, 57, 54, 51, 51, 51, 48, 46, 46,
        60, 60, 60, 60, 60, 58, 55, 55, 55, 52, 48, 48, 48, 46, 43, 43, 60, 60,
        60, 60, 60, 58, 55, 55, 55, 52, 48, 48, 48, 46, 43, 43, 60, 60, 60, 60,
        60, 58, 55, 55, 55, 52, 48, 48, 48, 46, 43, 43, 59, 59, 59, 59, 59, 57,
        55, 55, 55, 50, 46, 46, 46, 43, 41, 41, 57, 59, 59, 59, 59, 56, 53, 53,
        53, 48, 43, 43, 43, 40, 38, 38, 57, 59, 59, 59, 59, 56, 53, 53, 53, 48,
        43, 43, 43, 40, 38, 38, 57, 59, 59, 59, 59, 56, 53, 53, 53, 48, 43, 43,
        43, 40, 38, 38, 55, 55, 57, 57, 57, 54, 52, 52, 52, 47, 42, 42, 42, 39,
        37, 37, 52, 53, 55, 55, 55, 53, 51, 51, 51, 46, 41, 41, 41, 38, 35, 35,
        52, 53, 55, 55, 55, 53, 51, 51, 51, 46, 41, 41, 41, 38, 35, 35, 52, 53,
        55, 55, 55, 53, 51, 51, 51, 46, 41, 41, 41, 38, 35, 35, 49, 51, 52, 52,
        52, 51, 49, 49, 49, 44, 39, 39, 39, 37, 34, 34, 47, 48, 50, 50, 50, 49,
        48, 48, 48, 43, 38, 38, 38, 35, 33, 33, 47, 48, 50, 50, 50, 49, 48, 48,
        48, 43, 38, 38, 38, 35, 33, 33,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 62,
        60, 60, 60, 59, 57, 57, 57, 55, 52, 52, 52, 49, 47, 47, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 62, 60, 60, 60, 59,
        59, 59, 59, 55, 53, 53, 53, 51, 48, 48, 65, 65, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 63, 63, 63, 62, 62, 62, 61, 60, 60, 60, 59, 59, 59, 59, 57,
        55, 55, 55, 52, 50, 50, 65, 65, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 63, 62, 62, 62, 61, 60, 60, 60, 59, 59, 59, 59, 57, 55, 55, 55, 52,
        50, 50, 65, 65, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62,
        62, 61, 60, 60, 60, 59, 59, 59, 59, 57, 55, 55, 55, 52, 50, 50, 64, 64,
        64, 64, 64, 63, 63, 63, 63, 62, 61, 61, 61, 61, 60, 60, 60, 59, 58, 58,
        58, 57, 56, 56, 56, 54, 53, 53, 53, 51, 49, 49, 64, 64, 63, 63, 63, 63,
        62, 62, 62, 61, 60, 60, 60, 59, 58, 58, 58, 57, 55, 55, 55, 55, 53, 53,
        53, 52, 51, 51, 51, 49, 48, 48, 64, 64, 63, 63, 63, 63, 62, 62, 62, 61,
        60, 60, 60, 59, 58, 58, 58, 57, 55, 55, 55, 55, 53, 53, 53, 52, 51, 51,
        51, 49, 48, 48, 64, 64, 63, 63, 63, 63, 62, 62, 62, 61, 60, 60, 60, 59,
        58, 58, 58, 57, 55, 55, 55, 55, 53, 53, 53, 52, 51, 51, 51, 49, 48, 48,
        60, 60, 60, 60, 60, 60, 61, 61, 61, 59, 58, 58, 58, 57, 55, 55, 55, 54,
        52, 52, 52, 50, 48, 48, 48, 47, 46, 46, 46, 44, 43, 43, 57, 58, 58, 58,
        58, 59, 59, 59, 59, 58, 56, 56, 56, 55, 53, 53, 53, 51, 48, 48, 48, 46,
        43, 43, 43, 42, 41, 41, 41, 39, 38, 38, 57, 58, 58, 58, 58, 59, 59, 59,
        59, 58, 56, 56, 56, 55, 53, 53, 53, 51, 48, 48, 48, 46, 43, 43, 43, 42,
        41, 41, 41, 39, 38, 38, 57, 58, 58, 58, 58, 59, 59, 59, 59, 58, 56, 56,
        56, 55, 53, 53, 53, 51, 48, 48, 48, 46, 43, 43, 43, 42, 41, 41, 41, 39,
        38, 38, 52, 53, 53, 53, 53, 54, 55, 55, 55, 53, 53, 53, 53, 51, 51, 51,
        51, 48, 46, 46, 46, 43, 40, 40, 40, 39, 38, 38, 38, 37, 35, 35, 47, 48,
        48, 48, 48, 49, 50, 50, 50, 49, 49, 49, 49, 48, 48, 48, 48, 46, 43, 43,
        43, 41, 38, 38, 38, 37, 35, 35, 35, 34, 33, 33, 47, 48, 48, 48, 48, 49,
        50, 50, 50, 49, 49, 49, 49, 48, 48, 48, 48, 46, 43, 43, 43, 41, 38, 38,
        38, 37, 35, 35, 35, 34, 33, 33,
        /* Size 4x16 */
        65, 64, 60, 52, 64, 64, 60, 53, 64, 64, 60, 53, 64, 63, 61, 55, 64, 63,
        61, 55, 64, 61, 58, 53, 64, 61, 58, 53, 63, 60, 55, 51, 63, 60, 55, 51,
        60, 58, 52, 46, 60, 58, 52, 46, 59, 56, 48, 40, 59, 56, 48, 40, 53, 53,
        46, 38, 53, 53, 46, 38, 48, 49, 43, 35,
        /* Size 16x4 */
        65, 64, 64, 64, 64, 64, 64, 63, 63, 60, 60, 59, 59, 53, 53, 48, 64, 64,
        64, 63, 63, 61, 61, 60, 60, 58, 58, 56, 56, 53, 53, 49, 60, 60, 60, 61,
        61, 58, 58, 55, 55, 52, 52, 48, 48, 46, 46, 43, 52, 53, 53, 55, 55, 53,
        53, 51, 51, 46, 46, 40, 40, 38, 38, 35,
        /* Size 8x32 */
        64, 65, 65, 64, 64, 57, 57, 47, 65, 65, 65, 64, 64, 58, 58, 48, 65, 64,
        64, 63, 63, 58, 58, 48, 65, 64, 64, 63, 63, 58, 58, 48, 65, 64, 64, 63,
        63, 58, 58, 48, 65, 64, 64, 63, 63, 59, 59, 49, 65, 63, 63, 62, 62, 59,
        59, 50, 65, 63, 63, 62, 62, 59, 59, 50, 65, 63, 63, 62, 62, 59, 59, 50,
        65, 63, 63, 61, 61, 58, 58, 49, 64, 63, 63, 60, 60, 56, 56, 49, 64, 63,
        63, 60, 60, 56, 56, 49, 64, 63, 63, 60, 60, 56, 56, 49, 64, 63, 63, 59,
        59, 55, 55, 48, 64, 62, 62, 58, 58, 53, 53, 48, 64, 62, 62, 58, 58, 53,
        53, 48, 64, 62, 62, 58, 58, 53, 53, 48, 62, 61, 61, 57, 57, 51, 51, 46,
        60, 60, 60, 55, 55, 48, 48, 43, 60, 60, 60, 55, 55, 48, 48, 43, 60, 60,
        60, 55, 55, 48, 48, 43, 59, 59, 59, 55, 55, 46, 46, 41, 57, 59, 59, 53,
        53, 43, 43, 38, 57, 59, 59, 53, 53, 43, 43, 38, 57, 59, 59, 53, 53, 43,
        43, 38, 55, 57, 57, 52, 52, 42, 42, 37, 52, 55, 55, 51, 51, 41, 41, 35,
        52, 55, 55, 51, 51, 41, 41, 35, 52, 55, 55, 51, 51, 41, 41, 35, 49, 52,
        52, 49, 49, 39, 39, 34, 47, 50, 50, 48, 48, 38, 38, 33, 47, 50, 50, 48,
        48, 38, 38, 33,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 62,
        60, 60, 60, 59, 57, 57, 57, 55, 52, 52, 52, 49, 47, 47, 65, 65, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 61, 60, 60, 60, 59,
        59, 59, 59, 57, 55, 55, 55, 52, 50, 50, 65, 65, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 63, 63, 63, 62, 62, 62, 61, 60, 60, 60, 59, 59, 59, 59, 57,
        55, 55, 55, 52, 50, 50, 64, 64, 63, 63, 63, 63, 62, 62, 62, 61, 60, 60,
        60, 59, 58, 58, 58, 57, 55, 55, 55, 55, 53, 53, 53, 52, 51, 51, 51, 49,
        48, 48, 64, 64, 63, 63, 63, 63, 62, 62, 62, 61, 60, 60, 60, 59, 58, 58,
        58, 57, 55, 55, 55, 55, 53, 53, 53, 52, 51, 51, 51, 49, 48, 48, 57, 58,
        58, 58, 58, 59, 59, 59, 59, 58, 56, 56, 56, 55, 53, 53, 53, 51, 48, 48,
        48, 46, 43, 43, 43, 42, 41, 41, 41, 39, 38, 38, 57, 58, 58, 58, 58, 59,
        59, 59, 59, 58, 56, 56, 56, 55, 53, 53, 53, 51, 48, 48, 48, 46, 43, 43,
        43, 42, 41, 41, 41, 39, 38, 38, 47, 48, 48, 48, 48, 49, 50, 50, 50, 49,
        49, 49, 49, 48, 48, 48, 48, 46, 43, 43, 43, 41, 38, 38, 38, 37, 35, 35,
        35, 34, 33, 33 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        64, 44, 44, 39,
#endif
        /* Size 4x4 */
        65, 59, 48, 44, 59, 53, 46, 44, 48, 46, 42, 41, 44, 44, 41, 38,
        /* Size 8x8 */
        65, 66, 63, 58, 52, 46, 43, 43, 66, 65, 62, 55, 50, 46, 44, 45, 63, 62,
        58, 52, 48, 46, 44, 45, 58, 55, 52, 48, 45, 44, 44, 45, 52, 50, 48, 45,
        43, 42, 42, 43, 46, 46, 46, 44, 42, 41, 40, 40, 43, 44, 44, 44, 42, 40,
        38, 38, 43, 45, 45, 45, 43, 40, 38, 37,
        /* Size 16x16 */
        64, 66, 66, 67, 67, 61, 61, 56, 56, 49, 49, 42, 42, 42, 42, 42, 66, 65,
        65, 66, 66, 59, 59, 53, 53, 48, 48, 43, 43, 44, 44, 44, 66, 65, 65, 66,
        66, 59, 59, 53, 53, 48, 48, 43, 43, 44, 44, 44, 67, 66, 66, 64, 64, 58,
        58, 51, 51, 48, 48, 44, 44, 45, 45, 46, 67, 66, 66, 64, 64, 58, 58, 51,
        51, 48, 48, 44, 44, 45, 45, 46, 61, 59, 59, 58, 58, 53, 53, 47, 47, 46,
        46, 43, 43, 44, 44, 45, 61, 59, 59, 58, 58, 53, 53, 47, 47, 46, 46, 43,
        43, 44, 44, 45, 56, 53, 53, 51, 51, 47, 47, 43, 43, 43, 43, 43, 43, 44,
        44, 45, 56, 53, 53, 51, 51, 47, 47, 43, 43, 43, 43, 43, 43, 44, 44, 45,
        49, 48, 48, 48, 48, 46, 46, 43, 43, 42, 42, 41, 41, 41, 41, 42, 49, 48,
        48, 48, 48, 46, 46, 43, 43, 42, 42, 41, 41, 41, 41, 42, 42, 43, 43, 44,
        44, 43, 43, 43, 43, 41, 41, 39, 39, 39, 39, 38, 42, 43, 43, 44, 44, 43,
        43, 43, 43, 41, 41, 39, 39, 39, 39, 38, 42, 44, 44, 45, 45, 44, 44, 44,
        44, 41, 41, 39, 39, 38, 38, 37, 42, 44, 44, 45, 45, 44, 44, 44, 44, 41,
        41, 39, 39, 38, 38, 37, 42, 44, 44, 46, 46, 45, 45, 45, 45, 42, 42, 38,
        38, 37, 37, 35,
        /* Size 32x32 */
        64, 65, 66, 66, 66, 66, 67, 67, 67, 64, 61, 61, 61, 59, 56, 56, 56, 53,
        49, 49, 49, 46, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 65, 65, 65, 65,
        65, 66, 66, 66, 66, 63, 60, 60, 60, 58, 55, 55, 55, 52, 49, 49, 49, 46,
        42, 42, 42, 43, 43, 43, 43, 43, 43, 43, 66, 65, 65, 65, 65, 65, 66, 66,
        66, 62, 59, 59, 59, 56, 53, 53, 53, 51, 48, 48, 48, 46, 43, 43, 43, 43,
        44, 44, 44, 44, 44, 44, 66, 65, 65, 65, 65, 65, 66, 66, 66, 62, 59, 59,
        59, 56, 53, 53, 53, 51, 48, 48, 48, 46, 43, 43, 43, 43, 44, 44, 44, 44,
        44, 44, 66, 65, 65, 65, 65, 65, 66, 66, 66, 62, 59, 59, 59, 56, 53, 53,
        53, 51, 48, 48, 48, 46, 43, 43, 43, 43, 44, 44, 44, 44, 44, 44, 66, 66,
        65, 65, 65, 65, 65, 65, 65, 62, 59, 59, 59, 55, 53, 53, 53, 50, 48, 48,
        48, 46, 44, 44, 44, 44, 44, 44, 44, 45, 45, 45, 67, 66, 66, 66, 66, 65,
        64, 64, 64, 61, 58, 58, 58, 55, 51, 51, 51, 49, 48, 48, 48, 46, 44, 44,
        44, 45, 45, 45, 45, 45, 46, 46, 67, 66, 66, 66, 66, 65, 64, 64, 64, 61,
        58, 58, 58, 55, 51, 51, 51, 49, 48, 48, 48, 46, 44, 44, 44, 45, 45, 45,
        45, 45, 46, 46, 67, 66, 66, 66, 66, 65, 64, 64, 64, 61, 58, 58, 58, 55,
        51, 51, 51, 49, 48, 48, 48, 46, 44, 44, 44, 45, 45, 45, 45, 45, 46, 46,
        64, 63, 62, 62, 62, 62, 61, 61, 61, 58, 55, 55, 55, 52, 49, 49, 49, 48,
        47, 47, 47, 46, 44, 44, 44, 44, 45, 45, 45, 45, 46, 46, 61, 60, 59, 59,
        59, 59, 58, 58, 58, 55, 53, 53, 53, 50, 47, 47, 47, 47, 46, 46, 46, 45,
        43, 43, 43, 44, 44, 44, 44, 45, 45, 45, 61, 60, 59, 59, 59, 59, 58, 58,
        58, 55, 53, 53, 53, 50, 47, 47, 47, 47, 46, 46, 46, 45, 43, 43, 43, 44,
        44, 44, 44, 45, 45, 45, 61, 60, 59, 59, 59, 59, 58, 58, 58, 55, 53, 53,
        53, 50, 47, 47, 47, 47, 46, 46, 46, 45, 43, 43, 43, 44, 44, 44, 44, 45,
        45, 45, 59, 58, 56, 56, 56, 55, 55, 55, 55, 52, 50, 50, 50, 48, 45, 45,
        45, 45, 45, 45, 45, 44, 43, 43, 43, 44, 44, 44, 44, 45, 45, 45, 56, 55,
        53, 53, 53, 53, 51, 51, 51, 49, 47, 47, 47, 45, 43, 43, 43, 43, 43, 43,
        43, 43, 43, 43, 43, 44, 44, 44, 44, 45, 45, 45, 56, 55, 53, 53, 53, 53,
        51, 51, 51, 49, 47, 47, 47, 45, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43,
        43, 44, 44, 44, 44, 45, 45, 45, 56, 55, 53, 53, 53, 53, 51, 51, 51, 49,
        47, 47, 47, 45, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 44, 44, 44,
        44, 45, 45, 45, 53, 52, 51, 51, 51, 50, 49, 49, 49, 48, 47, 47, 47, 45,
        43, 43, 43, 43, 43, 43, 43, 42, 42, 42, 42, 42, 43, 43, 43, 43, 44, 44,
        49, 49, 48, 48, 48, 48, 48, 48, 48, 47, 46, 46, 46, 45, 43, 43, 43, 43,
        42, 42, 42, 41, 41, 41, 41, 41, 41, 41, 41, 41, 42, 42, 49, 49, 48, 48,
        48, 48, 48, 48, 48, 47, 46, 46, 46, 45, 43, 43, 43, 43, 42, 42, 42, 41,
        41, 41, 41, 41, 41, 41, 41, 41, 42, 42, 49, 49, 48, 48, 48, 48, 48, 48,
        48, 47, 46, 46, 46, 45, 43, 43, 43, 43, 42, 42, 42, 41, 41, 41, 41, 41,
        41, 41, 41, 41, 42, 42, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 45, 45,
        45, 44, 43, 43, 43, 42, 41, 41, 41, 41, 40, 40, 40, 40, 40, 40, 40, 40,
        40, 40, 42, 42, 43, 43, 43, 44, 44, 44, 44, 44, 43, 43, 43, 43, 43, 43,
        43, 42, 41, 41, 41, 40, 39, 39, 39, 39, 39, 39, 39, 38, 38, 38, 42, 42,
        43, 43, 43, 44, 44, 44, 44, 44, 43, 43, 43, 43, 43, 43, 43, 42, 41, 41,
        41, 40, 39, 39, 39, 39, 39, 39, 39, 38, 38, 38, 42, 42, 43, 43, 43, 44,
        44, 44, 44, 44, 43, 43, 43, 43, 43, 43, 43, 42, 41, 41, 41, 40, 39, 39,
        39, 39, 39, 39, 39, 38, 38, 38, 42, 43, 43, 43, 43, 44, 45, 45, 45, 44,
        44, 44, 44, 44, 44, 44, 44, 42, 41, 41, 41, 40, 39, 39, 39, 38, 38, 38,
        38, 38, 38, 38, 42, 43, 44, 44, 44, 44, 45, 45, 45, 45, 44, 44, 44, 44,
        44, 44, 44, 43, 41, 41, 41, 40, 39, 39, 39, 38, 38, 38, 38, 37, 37, 37,
        42, 43, 44, 44, 44, 44, 45, 45, 45, 45, 44, 44, 44, 44, 44, 44, 44, 43,
        41, 41, 41, 40, 39, 39, 39, 38, 38, 38, 38, 37, 37, 37, 42, 43, 44, 44,
        44, 44, 45, 45, 45, 45, 44, 44, 44, 44, 44, 44, 44, 43, 41, 41, 41, 40,
        39, 39, 39, 38, 38, 38, 38, 37, 37, 37, 42, 43, 44, 44, 44, 45, 45, 45,
        45, 45, 45, 45, 45, 45, 45, 45, 45, 43, 41, 41, 41, 40, 38, 38, 38, 38,
        37, 37, 37, 37, 36, 36, 42, 43, 44, 44, 44, 45, 46, 46, 46, 46, 45, 45,
        45, 45, 45, 45, 45, 44, 42, 42, 42, 40, 38, 38, 38, 38, 37, 37, 37, 36,
        35, 35, 42, 43, 44, 44, 44, 45, 46, 46, 46, 46, 45, 45, 45, 45, 45, 45,
        45, 44, 42, 42, 42, 40, 38, 38, 38, 38, 37, 37, 37, 36, 35, 35,
        /* Size 4x8 */
        65, 60, 49, 43, 65, 59, 48, 44, 62, 55, 47, 45, 56, 50, 45, 44, 51, 47,
        43, 43, 46, 45, 41, 40, 43, 44, 41, 38, 44, 45, 41, 37,
        /* Size 8x4 */
        65, 65, 62, 56, 51, 46, 43, 44, 60, 59, 55, 50, 47, 45, 44, 45, 49, 48,
        47, 45, 43, 41, 41, 41, 43, 44, 45, 44, 43, 40, 38, 37,
        /* Size 8x16 */
        64, 66, 66, 55, 55, 43, 43, 42, 65, 65, 65, 53, 53, 44, 44, 44, 65, 65,
        65, 53, 53, 44, 44, 44, 67, 64, 64, 51, 51, 45, 45, 45, 67, 64, 64, 51,
        51, 45, 45, 45, 61, 57, 57, 48, 48, 44, 44, 45, 61, 57, 57, 48, 48, 44,
        44, 45, 55, 51, 51, 44, 44, 43, 43, 45, 55, 51, 51, 44, 44, 43, 43, 45,
        49, 48, 48, 43, 43, 41, 41, 41, 49, 48, 48, 43, 43, 41, 41, 41, 42, 44,
        44, 43, 43, 39, 39, 38, 42, 44, 44, 43, 43, 39, 39, 38, 42, 45, 45, 44,
        44, 39, 39, 37, 42, 45, 45, 44, 44, 39, 39, 37, 42, 45, 45, 45, 45, 38,
        38, 35,
        /* Size 16x8 */
        64, 65, 65, 67, 67, 61, 61, 55, 55, 49, 49, 42, 42, 42, 42, 42, 66, 65,
        65, 64, 64, 57, 57, 51, 51, 48, 48, 44, 44, 45, 45, 45, 66, 65, 65, 64,
        64, 57, 57, 51, 51, 48, 48, 44, 44, 45, 45, 45, 55, 53, 53, 51, 51, 48,
        48, 44, 44, 43, 43, 43, 43, 44, 44, 45, 55, 53, 53, 51, 51, 48, 48, 44,
        44, 43, 43, 43, 43, 44, 44, 45, 43, 44, 44, 45, 45, 44, 44, 43, 43, 41,
        41, 39, 39, 39, 39, 38, 43, 44, 44, 45, 45, 44, 44, 43, 43, 41, 41, 39,
        39, 39, 39, 38, 42, 44, 44, 45, 45, 45, 45, 45, 45, 41, 41, 38, 38, 37,
        37, 35,
        /* Size 16x32 */
        64, 65, 66, 66, 66, 61, 55, 55, 55, 49, 43, 43, 43, 42, 42, 42, 65, 65,
        66, 66, 66, 60, 55, 55, 55, 49, 43, 43, 43, 43, 43, 43, 65, 65, 65, 65,
        65, 59, 53, 53, 53, 48, 44, 44, 44, 44, 44, 44, 65, 65, 65, 65, 65, 59,
        53, 53, 53, 48, 44, 44, 44, 44, 44, 44, 65, 65, 65, 65, 65, 59, 53, 53,
        53, 48, 44, 44, 44, 44, 44, 44, 66, 65, 64, 64, 64, 59, 53, 53, 53, 48,
        44, 44, 44, 44, 45, 45, 67, 65, 64, 64, 64, 58, 51, 51, 51, 48, 45, 45,
        45, 45, 45, 45, 67, 65, 64, 64, 64, 58, 51, 51, 51, 48, 45, 45, 45, 45,
        45, 45, 67, 65, 64, 64, 64, 58, 51, 51, 51, 48, 45, 45, 45, 45, 45, 45,
        64, 62, 60, 60, 60, 55, 49, 49, 49, 47, 44, 44, 44, 45, 45, 45, 61, 59,
        57, 57, 57, 53, 48, 48, 48, 46, 44, 44, 44, 44, 45, 45, 61, 59, 57, 57,
        57, 53, 48, 48, 48, 46, 44, 44, 44, 44, 45, 45, 61, 59, 57, 57, 57, 53,
        48, 48, 48, 46, 44, 44, 44, 44, 45, 45, 58, 56, 54, 54, 54, 50, 46, 46,
        46, 45, 43, 43, 43, 44, 45, 45, 55, 53, 51, 51, 51, 47, 44, 44, 44, 43,
        43, 43, 43, 44, 45, 45, 55, 53, 51, 51, 51, 47, 44, 44, 44, 43, 43, 43,
        43, 44, 45, 45, 55, 53, 51, 51, 51, 47, 44, 44, 44, 43, 43, 43, 43, 44,
        45, 45, 52, 51, 49, 49, 49, 47, 43, 43, 43, 43, 42, 42, 42, 43, 43, 43,
        49, 48, 48, 48, 48, 46, 43, 43, 43, 42, 41, 41, 41, 41, 41, 41, 49, 48,
        48, 48, 48, 46, 43, 43, 43, 42, 41, 41, 41, 41, 41, 41, 49, 48, 48, 48,
        48, 46, 43, 43, 43, 42, 41, 41, 41, 41, 41, 41, 46, 46, 46, 46, 46, 45,
        43, 43, 43, 41, 40, 40, 40, 40, 40, 40, 42, 43, 44, 44, 44, 43, 43, 43,
        43, 41, 39, 39, 39, 38, 38, 38, 42, 43, 44, 44, 44, 43, 43, 43, 43, 41,
        39, 39, 39, 38, 38, 38, 42, 43, 44, 44, 44, 43, 43, 43, 43, 41, 39, 39,
        39, 38, 38, 38, 42, 43, 44, 44, 44, 44, 43, 43, 43, 41, 39, 39, 39, 38,
        38, 38, 42, 44, 45, 45, 45, 44, 44, 44, 44, 41, 39, 39, 39, 38, 37, 37,
        42, 44, 45, 45, 45, 44, 44, 44, 44, 41, 39, 39, 39, 38, 37, 37, 42, 44,
        45, 45, 45, 44, 44, 44, 44, 41, 39, 39, 39, 38, 37, 37, 42, 44, 45, 45,
        45, 45, 44, 44, 44, 41, 39, 39, 39, 37, 36, 36, 42, 44, 45, 45, 45, 45,
        45, 45, 45, 41, 38, 38, 38, 37, 35, 35, 42, 44, 45, 45, 45, 45, 45, 45,
        45, 41, 38, 38, 38, 37, 35, 35,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 66, 67, 67, 67, 64, 61, 61, 61, 58, 55, 55, 55, 52,
        49, 49, 49, 46, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 62, 59, 59, 59, 56, 53, 53, 53, 51, 48, 48, 48, 46,
        43, 43, 43, 43, 44, 44, 44, 44, 44, 44, 66, 66, 65, 65, 65, 64, 64, 64,
        64, 60, 57, 57, 57, 54, 51, 51, 51, 49, 48, 48, 48, 46, 44, 44, 44, 44,
        45, 45, 45, 45, 45, 45, 66, 66, 65, 65, 65, 64, 64, 64, 64, 60, 57, 57,
        57, 54, 51, 51, 51, 49, 48, 48, 48, 46, 44, 44, 44, 44, 45, 45, 45, 45,
        45, 45, 66, 66, 65, 65, 65, 64, 64, 64, 64, 60, 57, 57, 57, 54, 51, 51,
        51, 49, 48, 48, 48, 46, 44, 44, 44, 44, 45, 45, 45, 45, 45, 45, 61, 60,
        59, 59, 59, 59, 58, 58, 58, 55, 53, 53, 53, 50, 47, 47, 47, 47, 46, 46,
        46, 45, 43, 43, 43, 44, 44, 44, 44, 45, 45, 45, 55, 55, 53, 53, 53, 53,
        51, 51, 51, 49, 48, 48, 48, 46, 44, 44, 44, 43, 43, 43, 43, 43, 43, 43,
        43, 43, 44, 44, 44, 44, 45, 45, 55, 55, 53, 53, 53, 53, 51, 51, 51, 49,
        48, 48, 48, 46, 44, 44, 44, 43, 43, 43, 43, 43, 43, 43, 43, 43, 44, 44,
        44, 44, 45, 45, 55, 55, 53, 53, 53, 53, 51, 51, 51, 49, 48, 48, 48, 46,
        44, 44, 44, 43, 43, 43, 43, 43, 43, 43, 43, 43, 44, 44, 44, 44, 45, 45,
        49, 49, 48, 48, 48, 48, 48, 48, 48, 47, 46, 46, 46, 45, 43, 43, 43, 43,
        42, 42, 42, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 43, 43, 44, 44,
        44, 44, 45, 45, 45, 44, 44, 44, 44, 43, 43, 43, 43, 42, 41, 41, 41, 40,
        39, 39, 39, 39, 39, 39, 39, 39, 38, 38, 43, 43, 44, 44, 44, 44, 45, 45,
        45, 44, 44, 44, 44, 43, 43, 43, 43, 42, 41, 41, 41, 40, 39, 39, 39, 39,
        39, 39, 39, 39, 38, 38, 43, 43, 44, 44, 44, 44, 45, 45, 45, 44, 44, 44,
        44, 43, 43, 43, 43, 42, 41, 41, 41, 40, 39, 39, 39, 39, 39, 39, 39, 39,
        38, 38, 42, 43, 44, 44, 44, 44, 45, 45, 45, 45, 44, 44, 44, 44, 44, 44,
        44, 43, 41, 41, 41, 40, 38, 38, 38, 38, 38, 38, 38, 37, 37, 37, 42, 43,
        44, 44, 44, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 43, 41, 41,
        41, 40, 38, 38, 38, 38, 37, 37, 37, 36, 35, 35, 42, 43, 44, 44, 44, 45,
        45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 43, 41, 41, 41, 40, 38, 38,
        38, 38, 37, 37, 37, 36, 35, 35,
        /* Size 4x16 */
        65, 61, 49, 42, 65, 59, 48, 44, 65, 59, 48, 44, 65, 58, 48, 45, 65, 58,
        48, 45, 59, 53, 46, 44, 59, 53, 46, 44, 53, 47, 43, 44, 53, 47, 43, 44,
        48, 46, 42, 41, 48, 46, 42, 41, 43, 43, 41, 38, 43, 43, 41, 38, 44, 44,
        41, 38, 44, 44, 41, 38, 44, 45, 41, 37,
        /* Size 16x4 */
        65, 65, 65, 65, 65, 59, 59, 53, 53, 48, 48, 43, 43, 44, 44, 44, 61, 59,
        59, 58, 58, 53, 53, 47, 47, 46, 46, 43, 43, 44, 44, 45, 49, 48, 48, 48,
        48, 46, 46, 43, 43, 42, 42, 41, 41, 41, 41, 41, 42, 44, 44, 45, 45, 44,
        44, 44, 44, 41, 41, 38, 38, 38, 38, 37,
        /* Size 8x32 */
        64, 66, 66, 55, 55, 43, 43, 42, 65, 66, 66, 55, 55, 43, 43, 43, 65, 65,
        65, 53, 53, 44, 44, 44, 65, 65, 65, 53, 53, 44, 44, 44, 65, 65, 65, 53,
        53, 44, 44, 44, 66, 64, 64, 53, 53, 44, 44, 45, 67, 64, 64, 51, 51, 45,
        45, 45, 67, 64, 64, 51, 51, 45, 45, 45, 67, 64, 64, 51, 51, 45, 45, 45,
        64, 60, 60, 49, 49, 44, 44, 45, 61, 57, 57, 48, 48, 44, 44, 45, 61, 57,
        57, 48, 48, 44, 44, 45, 61, 57, 57, 48, 48, 44, 44, 45, 58, 54, 54, 46,
        46, 43, 43, 45, 55, 51, 51, 44, 44, 43, 43, 45, 55, 51, 51, 44, 44, 43,
        43, 45, 55, 51, 51, 44, 44, 43, 43, 45, 52, 49, 49, 43, 43, 42, 42, 43,
        49, 48, 48, 43, 43, 41, 41, 41, 49, 48, 48, 43, 43, 41, 41, 41, 49, 48,
        48, 43, 43, 41, 41, 41, 46, 46, 46, 43, 43, 40, 40, 40, 42, 44, 44, 43,
        43, 39, 39, 38, 42, 44, 44, 43, 43, 39, 39, 38, 42, 44, 44, 43, 43, 39,
        39, 38, 42, 44, 44, 43, 43, 39, 39, 38, 42, 45, 45, 44, 44, 39, 39, 37,
        42, 45, 45, 44, 44, 39, 39, 37, 42, 45, 45, 44, 44, 39, 39, 37, 42, 45,
        45, 44, 44, 39, 39, 36, 42, 45, 45, 45, 45, 38, 38, 35, 42, 45, 45, 45,
        45, 38, 38, 35,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 66, 67, 67, 67, 64, 61, 61, 61, 58, 55, 55, 55, 52,
        49, 49, 49, 46, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 66, 66, 65, 65,
        65, 64, 64, 64, 64, 60, 57, 57, 57, 54, 51, 51, 51, 49, 48, 48, 48, 46,
        44, 44, 44, 44, 45, 45, 45, 45, 45, 45, 66, 66, 65, 65, 65, 64, 64, 64,
        64, 60, 57, 57, 57, 54, 51, 51, 51, 49, 48, 48, 48, 46, 44, 44, 44, 44,
        45, 45, 45, 45, 45, 45, 55, 55, 53, 53, 53, 53, 51, 51, 51, 49, 48, 48,
        48, 46, 44, 44, 44, 43, 43, 43, 43, 43, 43, 43, 43, 43, 44, 44, 44, 44,
        45, 45, 55, 55, 53, 53, 53, 53, 51, 51, 51, 49, 48, 48, 48, 46, 44, 44,
        44, 43, 43, 43, 43, 43, 43, 43, 43, 43, 44, 44, 44, 44, 45, 45, 43, 43,
        44, 44, 44, 44, 45, 45, 45, 44, 44, 44, 44, 43, 43, 43, 43, 42, 41, 41,
        41, 40, 39, 39, 39, 39, 39, 39, 39, 39, 38, 38, 43, 43, 44, 44, 44, 44,
        45, 45, 45, 44, 44, 44, 44, 43, 43, 43, 43, 42, 41, 41, 41, 40, 39, 39,
        39, 39, 39, 39, 39, 39, 38, 38, 42, 43, 44, 44, 44, 45, 45, 45, 45, 45,
        45, 45, 45, 45, 45, 45, 45, 43, 41, 41, 41, 40, 38, 38, 38, 38, 37, 37,
        37, 36, 35, 35 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        64, 61, 61, 52,
#endif
        /* Size 4x4 */
        64, 64, 63, 59, 64, 63, 62, 59, 63, 62, 58, 54, 59, 59, 54, 45,
        /* Size 8x8 */
        65, 65, 65, 64, 64, 63, 60, 58, 65, 64, 64, 64, 63, 62, 60, 59, 65, 64,
        63, 63, 63, 62, 61, 59, 64, 64, 63, 61, 60, 59, 58, 56, 64, 63, 63, 60,
        59, 58, 56, 54, 63, 62, 62, 59, 58, 56, 54, 51, 60, 60, 61, 58, 56, 54,
        52, 48, 58, 59, 59, 56, 54, 51, 48, 42,
        /* Size 16x16 */
        64, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 61, 60, 59, 57, 57, 65, 65,
        65, 64, 64, 64, 64, 64, 64, 64, 64, 61, 60, 59, 58, 58, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 63, 63, 61, 60, 59, 59, 59, 65, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 61, 60, 60, 59, 59, 65, 64, 64, 64, 63, 63, 63, 63,
        63, 63, 63, 61, 61, 60, 59, 59, 65, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 61, 61, 60, 59, 59, 65, 64, 64, 63, 63, 63, 62, 62, 61, 61, 61, 59,
        59, 58, 57, 57, 65, 64, 64, 63, 63, 63, 62, 61, 61, 60, 60, 59, 58, 57,
        56, 56, 64, 64, 64, 63, 63, 63, 61, 61, 60, 59, 59, 58, 57, 56, 55, 55,
        64, 64, 63, 63, 63, 63, 61, 60, 59, 58, 58, 56, 55, 55, 53, 53, 64, 64,
        63, 63, 63, 63, 61, 60, 59, 58, 58, 56, 55, 55, 53, 53, 61, 61, 61, 61,
        61, 61, 59, 59, 58, 56, 56, 53, 53, 51, 49, 49, 60, 60, 60, 60, 61, 61,
        59, 58, 57, 55, 55, 53, 52, 50, 48, 48, 59, 59, 59, 60, 60, 60, 58, 57,
        56, 55, 55, 51, 50, 48, 46, 46, 57, 58, 59, 59, 59, 59, 57, 56, 55, 53,
        53, 49, 48, 46, 42, 42, 57, 58, 59, 59, 59, 59, 57, 56, 55, 53, 53, 49,
        48, 46, 42, 42,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64,
        64, 64, 64, 63, 61, 60, 60, 60, 59, 58, 57, 57, 57, 55, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63,
        61, 60, 60, 60, 59, 59, 58, 58, 58, 55, 65, 65, 65, 65, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62, 61, 60, 60, 60,
        59, 59, 58, 58, 58, 56, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 63, 63, 63, 63, 62, 61, 60, 60, 60, 59, 59, 59, 59,
        59, 56, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 63, 63, 63, 63, 62, 61, 60, 60, 60, 59, 59, 59, 59, 59, 56, 65, 65,
        65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63,
        63, 62, 61, 60, 60, 60, 59, 59, 59, 59, 59, 56, 65, 65, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 61, 60,
        60, 60, 60, 59, 59, 59, 59, 57, 65, 65, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 61, 61, 61, 61, 60, 59,
        59, 59, 59, 58, 65, 65, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 62, 61, 61, 61, 61, 60, 60, 59, 59, 59, 58,
        65, 65, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 62, 61, 61, 61, 61, 60, 60, 59, 59, 59, 58, 65, 65, 64, 64,
        64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62,
        61, 61, 61, 61, 60, 60, 59, 59, 59, 58, 65, 65, 64, 64, 64, 64, 64, 63,
        63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 61, 60, 59, 59, 59,
        59, 59, 59, 59, 59, 56, 65, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 62,
        62, 62, 62, 62, 61, 61, 61, 61, 61, 60, 59, 59, 59, 59, 58, 57, 57, 57,
        57, 55, 65, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 62, 62, 61, 61, 61,
        61, 60, 60, 60, 60, 59, 59, 58, 58, 58, 57, 57, 56, 56, 56, 55, 65, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 63, 62, 62, 61, 61, 61, 61, 60, 60, 60,
        60, 59, 59, 58, 58, 58, 57, 57, 56, 56, 56, 55, 65, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 62, 62, 61, 61, 61, 61, 60, 60, 60, 60, 59, 59, 58,
        58, 58, 57, 57, 56, 56, 56, 55, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 62, 61, 61, 61, 61, 60, 59, 59, 59, 59, 59, 58, 57, 57, 57, 56, 55,
        55, 55, 55, 54, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 61, 60,
        60, 60, 59, 59, 59, 59, 59, 58, 57, 56, 56, 56, 55, 55, 54, 54, 54, 53,
        64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 61, 60, 60, 60, 59, 59,
        58, 58, 58, 57, 56, 55, 55, 55, 55, 54, 53, 53, 53, 53, 64, 64, 64, 63,
        63, 63, 63, 63, 63, 63, 63, 62, 61, 60, 60, 60, 59, 59, 58, 58, 58, 57,
        56, 55, 55, 55, 55, 54, 53, 53, 53, 53, 64, 64, 64, 63, 63, 63, 63, 63,
        63, 63, 63, 62, 61, 60, 60, 60, 59, 59, 58, 58, 58, 57, 56, 55, 55, 55,
        55, 54, 53, 53, 53, 53, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 61,
        60, 59, 59, 59, 59, 58, 57, 57, 57, 56, 55, 54, 54, 54, 53, 52, 51, 51,
        51, 50, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 60, 59, 59, 59, 59,
        58, 57, 56, 56, 56, 55, 53, 53, 53, 53, 51, 50, 49, 49, 49, 48, 60, 60,
        60, 60, 60, 60, 60, 61, 61, 61, 61, 59, 59, 58, 58, 58, 57, 56, 55, 55,
        55, 54, 53, 52, 52, 52, 50, 49, 48, 48, 48, 47, 60, 60, 60, 60, 60, 60,
        60, 61, 61, 61, 61, 59, 59, 58, 58, 58, 57, 56, 55, 55, 55, 54, 53, 52,
        52, 52, 50, 49, 48, 48, 48, 47, 60, 60, 60, 60, 60, 60, 60, 61, 61, 61,
        61, 59, 59, 58, 58, 58, 57, 56, 55, 55, 55, 54, 53, 52, 52, 52, 50, 49,
        48, 48, 48, 47, 59, 59, 59, 59, 59, 59, 60, 60, 60, 60, 60, 59, 58, 57,
        57, 57, 56, 55, 55, 55, 55, 53, 51, 50, 50, 50, 48, 47, 46, 46, 46, 45,
        58, 59, 59, 59, 59, 59, 59, 59, 60, 60, 60, 59, 57, 57, 57, 57, 55, 55,
        54, 54, 54, 52, 50, 49, 49, 49, 47, 45, 44, 44, 44, 43, 57, 58, 58, 59,
        59, 59, 59, 59, 59, 59, 59, 59, 57, 56, 56, 56, 55, 54, 53, 53, 53, 51,
        49, 48, 48, 48, 46, 44, 42, 42, 42, 41, 57, 58, 58, 59, 59, 59, 59, 59,
        59, 59, 59, 59, 57, 56, 56, 56, 55, 54, 53, 53, 53, 51, 49, 48, 48, 48,
        46, 44, 42, 42, 42, 41, 57, 58, 58, 59, 59, 59, 59, 59, 59, 59, 59, 59,
        57, 56, 56, 56, 55, 54, 53, 53, 53, 51, 49, 48, 48, 48, 46, 44, 42, 42,
        42, 41, 55, 55, 56, 56, 56, 56, 57, 58, 58, 58, 58, 56, 55, 55, 55, 55,
        54, 53, 53, 53, 53, 50, 48, 47, 47, 47, 45, 43, 41, 41, 41, 41,
        /* Size 4x8 */
        65, 65, 64, 58, 64, 64, 63, 58, 64, 63, 62, 59, 64, 63, 60, 56, 64, 62,
        59, 54, 62, 62, 57, 51, 60, 60, 55, 48, 58, 59, 53, 43,
        /* Size 8x4 */
        65, 64, 64, 64, 64, 62, 60, 58, 65, 64, 63, 63, 62, 62, 60, 59, 64, 63,
        62, 60, 59, 57, 55, 53, 58, 58, 59, 56, 54, 51, 48, 43,
        /* Size 8x16 */
        64, 65, 65, 65, 64, 64, 59, 57, 65, 64, 64, 64, 63, 63, 59, 58, 65, 64,
        64, 64, 63, 63, 59, 58, 65, 64, 64, 63, 63, 63, 59, 59, 65, 64, 63, 63,
        62, 62, 60, 59, 65, 64, 63, 63, 62, 62, 60, 59, 65, 63, 63, 62, 60, 60,
        58, 57, 64, 63, 63, 62, 60, 60, 57, 56, 64, 63, 63, 61, 59, 59, 56, 55,
        64, 63, 62, 60, 58, 58, 55, 53, 64, 63, 62, 60, 58, 58, 55, 53, 61, 61,
        61, 59, 56, 56, 51, 49, 60, 60, 60, 59, 55, 55, 49, 48, 59, 59, 60, 58,
        55, 55, 48, 46, 57, 59, 59, 57, 53, 53, 45, 43, 57, 59, 59, 57, 53, 53,
        45, 43,
        /* Size 16x8 */
        64, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 61, 60, 59, 57, 57, 65, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 63, 61, 60, 59, 59, 59, 65, 64, 64, 64,
        63, 63, 63, 63, 63, 62, 62, 61, 60, 60, 59, 59, 65, 64, 64, 63, 63, 63,
        62, 62, 61, 60, 60, 59, 59, 58, 57, 57, 64, 63, 63, 63, 62, 62, 60, 60,
        59, 58, 58, 56, 55, 55, 53, 53, 64, 63, 63, 63, 62, 62, 60, 60, 59, 58,
        58, 56, 55, 55, 53, 53, 59, 59, 59, 59, 60, 60, 58, 57, 56, 55, 55, 51,
        49, 48, 45, 45, 57, 58, 58, 59, 59, 59, 57, 56, 55, 53, 53, 49, 48, 46,
        43, 43,
        /* Size 16x32 */
        64, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 61, 59, 57, 57, 57, 65, 65,
        65, 65, 65, 65, 64, 64, 64, 64, 64, 61, 59, 58, 58, 58, 65, 65, 64, 64,
        64, 64, 64, 63, 63, 63, 63, 61, 59, 58, 58, 58, 65, 64, 64, 64, 64, 64,
        64, 63, 63, 63, 63, 61, 59, 58, 58, 58, 65, 64, 64, 64, 64, 64, 64, 63,
        63, 63, 63, 61, 59, 58, 58, 58, 65, 64, 64, 64, 64, 64, 64, 63, 63, 63,
        63, 61, 59, 58, 58, 58, 65, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 61,
        59, 59, 59, 59, 65, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 61, 59, 59,
        59, 59, 65, 64, 64, 63, 63, 63, 63, 63, 62, 62, 62, 61, 60, 59, 59, 59,
        65, 64, 64, 63, 63, 63, 63, 63, 62, 62, 62, 61, 60, 59, 59, 59, 65, 64,
        64, 63, 63, 63, 63, 63, 62, 62, 62, 61, 60, 59, 59, 59, 65, 64, 63, 63,
        63, 63, 62, 62, 61, 61, 61, 60, 59, 58, 58, 58, 65, 64, 63, 63, 63, 63,
        62, 61, 60, 60, 60, 59, 58, 57, 57, 57, 64, 64, 63, 63, 63, 63, 62, 60,
        60, 60, 60, 59, 57, 56, 56, 56, 64, 64, 63, 63, 63, 63, 62, 60, 60, 60,
        60, 59, 57, 56, 56, 56, 64, 64, 63, 63, 63, 63, 62, 60, 60, 60, 60, 59,
        57, 56, 56, 56, 64, 64, 63, 63, 63, 63, 61, 60, 59, 59, 59, 58, 56, 55,
        55, 55, 64, 64, 63, 62, 62, 62, 61, 59, 59, 59, 59, 57, 55, 54, 54, 54,
        64, 64, 63, 62, 62, 62, 60, 59, 58, 58, 58, 56, 55, 53, 53, 53, 64, 64,
        63, 62, 62, 62, 60, 59, 58, 58, 58, 56, 55, 53, 53, 53, 64, 64, 63, 62,
        62, 62, 60, 59, 58, 58, 58, 56, 55, 53, 53, 53, 63, 62, 62, 62, 62, 62,
        59, 58, 57, 57, 57, 55, 53, 51, 51, 51, 61, 61, 61, 61, 61, 61, 59, 57,
        56, 56, 56, 53, 51, 49, 49, 49, 60, 60, 60, 60, 60, 60, 59, 57, 55, 55,
        55, 53, 49, 48, 48, 48, 60, 60, 60, 60, 60, 60, 59, 57, 55, 55, 55, 53,
        49, 48, 48, 48, 60, 60, 60, 60, 60, 60, 59, 57, 55, 55, 55, 53, 49, 48,
        48, 48, 59, 59, 59, 60, 60, 60, 58, 55, 55, 55, 55, 51, 48, 46, 46, 46,
        58, 59, 59, 59, 59, 59, 57, 55, 54, 54, 54, 50, 46, 44, 44, 44, 57, 58,
        59, 59, 59, 59, 57, 55, 53, 53, 53, 49, 45, 43, 43, 43, 57, 58, 59, 59,
        59, 59, 57, 55, 53, 53, 53, 49, 45, 43, 43, 43, 57, 58, 59, 59, 59, 59,
        57, 55, 53, 53, 53, 49, 45, 43, 43, 43, 55, 56, 57, 57, 57, 57, 55, 53,
        52, 52, 52, 48, 44, 42, 42, 42,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64,
        64, 64, 64, 63, 61, 60, 60, 60, 59, 58, 57, 57, 57, 55, 65, 65, 65, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62,
        61, 60, 60, 60, 59, 59, 58, 58, 58, 56, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 61, 60, 60, 60,
        59, 59, 59, 59, 59, 57, 65, 65, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 61, 60, 60, 60, 60, 59, 59, 59,
        59, 57, 65, 65, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 62, 62, 62, 62, 62, 61, 60, 60, 60, 60, 59, 59, 59, 59, 57, 65, 65,
        64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62,
        62, 62, 61, 60, 60, 60, 60, 59, 59, 59, 59, 57, 65, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 61, 61, 60, 60, 60, 59, 59, 59,
        59, 59, 58, 57, 57, 57, 57, 55, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 62, 61, 60, 60, 60, 60, 59, 59, 59, 59, 58, 57, 57, 57, 57, 55, 55,
        55, 55, 55, 53, 64, 64, 63, 63, 63, 63, 63, 63, 62, 62, 62, 61, 60, 60,
        60, 60, 59, 59, 58, 58, 58, 57, 56, 55, 55, 55, 55, 54, 53, 53, 53, 52,
        64, 64, 63, 63, 63, 63, 63, 63, 62, 62, 62, 61, 60, 60, 60, 60, 59, 59,
        58, 58, 58, 57, 56, 55, 55, 55, 55, 54, 53, 53, 53, 52, 64, 64, 63, 63,
        63, 63, 63, 63, 62, 62, 62, 61, 60, 60, 60, 60, 59, 59, 58, 58, 58, 57,
        56, 55, 55, 55, 55, 54, 53, 53, 53, 52, 61, 61, 61, 61, 61, 61, 61, 61,
        61, 61, 61, 60, 59, 59, 59, 59, 58, 57, 56, 56, 56, 55, 53, 53, 53, 53,
        51, 50, 49, 49, 49, 48, 59, 59, 59, 59, 59, 59, 59, 59, 60, 60, 60, 59,
        58, 57, 57, 57, 56, 55, 55, 55, 55, 53, 51, 49, 49, 49, 48, 46, 45, 45,
        45, 44, 57, 58, 58, 58, 58, 58, 59, 59, 59, 59, 59, 58, 57, 56, 56, 56,
        55, 54, 53, 53, 53, 51, 49, 48, 48, 48, 46, 44, 43, 43, 43, 42, 57, 58,
        58, 58, 58, 58, 59, 59, 59, 59, 59, 58, 57, 56, 56, 56, 55, 54, 53, 53,
        53, 51, 49, 48, 48, 48, 46, 44, 43, 43, 43, 42, 57, 58, 58, 58, 58, 58,
        59, 59, 59, 59, 59, 58, 57, 56, 56, 56, 55, 54, 53, 53, 53, 51, 49, 48,
        48, 48, 46, 44, 43, 43, 43, 42,
        /* Size 4x16 */
        65, 65, 64, 57, 65, 64, 63, 58, 64, 64, 63, 58, 64, 64, 63, 59, 64, 63,
        62, 59, 64, 63, 62, 59, 64, 63, 60, 57, 64, 63, 60, 56, 64, 63, 59, 55,
        64, 62, 58, 53, 64, 62, 58, 53, 61, 61, 56, 49, 60, 60, 55, 48, 59, 60,
        55, 46, 58, 59, 53, 43, 58, 59, 53, 43,
        /* Size 16x4 */
        65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 61, 60, 59, 58, 58, 65, 64,
        64, 64, 63, 63, 63, 63, 63, 62, 62, 61, 60, 60, 59, 59, 64, 63, 63, 63,
        62, 62, 60, 60, 59, 58, 58, 56, 55, 55, 53, 53, 57, 58, 58, 59, 59, 59,
        57, 56, 55, 53, 53, 49, 48, 46, 43, 43,
        /* Size 8x32 */
        64, 65, 65, 65, 64, 64, 59, 57, 65, 65, 65, 64, 64, 64, 59, 58, 65, 64,
        64, 64, 63, 63, 59, 58, 65, 64, 64, 64, 63, 63, 59, 58, 65, 64, 64, 64,
        63, 63, 59, 58, 65, 64, 64, 64, 63, 63, 59, 58, 65, 64, 64, 63, 63, 63,
        59, 59, 65, 64, 63, 63, 63, 63, 59, 59, 65, 64, 63, 63, 62, 62, 60, 59,
        65, 64, 63, 63, 62, 62, 60, 59, 65, 64, 63, 63, 62, 62, 60, 59, 65, 63,
        63, 62, 61, 61, 59, 58, 65, 63, 63, 62, 60, 60, 58, 57, 64, 63, 63, 62,
        60, 60, 57, 56, 64, 63, 63, 62, 60, 60, 57, 56, 64, 63, 63, 62, 60, 60,
        57, 56, 64, 63, 63, 61, 59, 59, 56, 55, 64, 63, 62, 61, 59, 59, 55, 54,
        64, 63, 62, 60, 58, 58, 55, 53, 64, 63, 62, 60, 58, 58, 55, 53, 64, 63,
        62, 60, 58, 58, 55, 53, 63, 62, 62, 59, 57, 57, 53, 51, 61, 61, 61, 59,
        56, 56, 51, 49, 60, 60, 60, 59, 55, 55, 49, 48, 60, 60, 60, 59, 55, 55,
        49, 48, 60, 60, 60, 59, 55, 55, 49, 48, 59, 59, 60, 58, 55, 55, 48, 46,
        58, 59, 59, 57, 54, 54, 46, 44, 57, 59, 59, 57, 53, 53, 45, 43, 57, 59,
        59, 57, 53, 53, 45, 43, 57, 59, 59, 57, 53, 53, 45, 43, 55, 57, 57, 55,
        52, 52, 44, 42,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64,
        64, 64, 64, 63, 61, 60, 60, 60, 59, 58, 57, 57, 57, 55, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62,
        61, 60, 60, 60, 59, 59, 59, 59, 59, 57, 65, 65, 64, 64, 64, 64, 64, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 61, 60, 60, 60,
        60, 59, 59, 59, 59, 57, 65, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 62,
        62, 62, 62, 62, 61, 61, 60, 60, 60, 59, 59, 59, 59, 59, 58, 57, 57, 57,
        57, 55, 64, 64, 63, 63, 63, 63, 63, 63, 62, 62, 62, 61, 60, 60, 60, 60,
        59, 59, 58, 58, 58, 57, 56, 55, 55, 55, 55, 54, 53, 53, 53, 52, 64, 64,
        63, 63, 63, 63, 63, 63, 62, 62, 62, 61, 60, 60, 60, 60, 59, 59, 58, 58,
        58, 57, 56, 55, 55, 55, 55, 54, 53, 53, 53, 52, 59, 59, 59, 59, 59, 59,
        59, 59, 60, 60, 60, 59, 58, 57, 57, 57, 56, 55, 55, 55, 55, 53, 51, 49,
        49, 49, 48, 46, 45, 45, 45, 44, 57, 58, 58, 58, 58, 58, 59, 59, 59, 59,
        59, 58, 57, 56, 56, 56, 55, 54, 53, 53, 53, 51, 49, 48, 48, 48, 46, 44,
        43, 43, 43, 42 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        65, 48, 48, 42,
#endif
        /* Size 4x4 */
        65, 63, 53, 44, 63, 59, 49, 45, 53, 49, 43, 43, 44, 45, 43, 39,
        /* Size 8x8 */
        65, 65, 67, 60, 56, 53, 49, 42, 65, 65, 66, 59, 55, 51, 48, 43, 67, 66,
        64, 58, 53, 50, 48, 44, 60, 59, 58, 53, 48, 47, 46, 43, 56, 55, 53, 48,
        45, 44, 44, 43, 53, 51, 50, 47, 44, 43, 43, 42, 49, 48, 48, 46, 44, 43,
        42, 41, 42, 43, 44, 43, 43, 42, 41, 39,
        /* Size 16x16 */
        64, 65, 66, 66, 67, 67, 62, 61, 59, 56, 56, 51, 49, 46, 42, 42, 65, 65,
        65, 66, 66, 66, 61, 59, 57, 54, 54, 49, 49, 46, 43, 43, 66, 65, 65, 65,
        66, 66, 60, 59, 57, 53, 53, 49, 48, 46, 43, 43, 66, 66, 65, 65, 65, 65,
        60, 59, 56, 53, 53, 49, 48, 47, 44, 44, 67, 66, 66, 65, 64, 64, 59, 58,
        55, 51, 51, 49, 48, 47, 44, 44, 67, 66, 66, 65, 64, 64, 59, 58, 55, 51,
        51, 49, 48, 47, 44, 44, 62, 61, 60, 60, 59, 59, 55, 53, 51, 48, 48, 47,
        46, 45, 44, 44, 61, 59, 59, 59, 58, 58, 53, 53, 51, 47, 47, 46, 46, 45,
        43, 43, 59, 57, 57, 56, 55, 55, 51, 51, 48, 46, 46, 45, 45, 44, 43, 43,
        56, 54, 53, 53, 51, 51, 48, 47, 46, 43, 43, 43, 43, 43, 43, 43, 56, 54,
        53, 53, 51, 51, 48, 47, 46, 43, 43, 43, 43, 43, 43, 43, 51, 49, 49, 49,
        49, 49, 47, 46, 45, 43, 43, 43, 42, 42, 41, 41, 49, 49, 48, 48, 48, 48,
        46, 46, 45, 43, 43, 42, 42, 41, 41, 41, 46, 46, 46, 47, 47, 47, 45, 45,
        44, 43, 43, 42, 41, 41, 40, 40, 42, 43, 43, 44, 44, 44, 44, 43, 43, 43,
        43, 41, 41, 40, 39, 39, 42, 43, 43, 44, 44, 44, 44, 43, 43, 43, 43, 41,
        41, 40, 39, 39,
        /* Size 32x32 */
        64, 65, 65, 66, 66, 66, 66, 67, 67, 67, 67, 65, 62, 61, 61, 61, 59, 57,
        56, 56, 56, 53, 51, 49, 49, 49, 46, 43, 42, 42, 42, 42, 65, 65, 65, 65,
        65, 65, 66, 66, 67, 67, 67, 64, 62, 60, 60, 60, 59, 56, 55, 55, 55, 53,
        50, 49, 49, 49, 46, 44, 42, 42, 42, 42, 65, 65, 65, 65, 65, 65, 66, 66,
        66, 66, 66, 63, 61, 59, 59, 59, 57, 55, 54, 54, 54, 52, 49, 49, 49, 49,
        46, 44, 43, 43, 43, 43, 66, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 63,
        60, 59, 59, 59, 57, 55, 53, 53, 53, 51, 49, 48, 48, 48, 46, 44, 43, 43,
        43, 43, 66, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 63, 60, 59, 59, 59,
        57, 55, 53, 53, 53, 51, 49, 48, 48, 48, 46, 44, 43, 43, 43, 43, 66, 65,
        65, 65, 65, 65, 65, 65, 66, 66, 66, 63, 60, 59, 59, 59, 57, 55, 53, 53,
        53, 51, 49, 48, 48, 48, 46, 44, 43, 43, 43, 43, 66, 66, 66, 65, 65, 65,
        65, 65, 65, 65, 65, 62, 60, 59, 59, 59, 56, 54, 53, 53, 53, 51, 49, 48,
        48, 48, 47, 45, 44, 44, 44, 44, 67, 66, 66, 65, 65, 65, 65, 65, 65, 65,
        65, 62, 59, 58, 58, 58, 55, 53, 52, 52, 52, 50, 49, 48, 48, 48, 47, 45,
        44, 44, 44, 44, 67, 67, 66, 66, 66, 66, 65, 65, 64, 64, 64, 62, 59, 58,
        58, 58, 55, 53, 51, 51, 51, 50, 49, 48, 48, 48, 47, 45, 44, 44, 44, 45,
        67, 67, 66, 66, 66, 66, 65, 65, 64, 64, 64, 62, 59, 58, 58, 58, 55, 53,
        51, 51, 51, 50, 49, 48, 48, 48, 47, 45, 44, 44, 44, 45, 67, 67, 66, 66,
        66, 66, 65, 65, 64, 64, 64, 62, 59, 58, 58, 58, 55, 53, 51, 51, 51, 50,
        49, 48, 48, 48, 47, 45, 44, 44, 44, 45, 65, 64, 63, 63, 63, 63, 62, 62,
        62, 62, 62, 59, 57, 55, 55, 55, 53, 51, 49, 49, 49, 49, 48, 47, 47, 47,
        46, 45, 44, 44, 44, 44, 62, 62, 61, 60, 60, 60, 60, 59, 59, 59, 59, 57,
        55, 53, 53, 53, 51, 49, 48, 48, 48, 47, 47, 46, 46, 46, 45, 44, 44, 44,
        44, 44, 61, 60, 59, 59, 59, 59, 59, 58, 58, 58, 58, 55, 53, 53, 53, 53,
        51, 48, 47, 47, 47, 47, 46, 46, 46, 46, 45, 44, 43, 43, 43, 44, 61, 60,
        59, 59, 59, 59, 59, 58, 58, 58, 58, 55, 53, 53, 53, 53, 51, 48, 47, 47,
        47, 47, 46, 46, 46, 46, 45, 44, 43, 43, 43, 44, 61, 60, 59, 59, 59, 59,
        59, 58, 58, 58, 58, 55, 53, 53, 53, 53, 51, 48, 47, 47, 47, 47, 46, 46,
        46, 46, 45, 44, 43, 43, 43, 44, 59, 59, 57, 57, 57, 57, 56, 55, 55, 55,
        55, 53, 51, 51, 51, 51, 48, 47, 46, 46, 46, 45, 45, 45, 45, 45, 44, 44,
        43, 43, 43, 44, 57, 56, 55, 55, 55, 55, 54, 53, 53, 53, 53, 51, 49, 48,
        48, 48, 47, 45, 44, 44, 44, 44, 44, 44, 44, 44, 44, 43, 43, 43, 43, 44,
        56, 55, 54, 53, 53, 53, 53, 52, 51, 51, 51, 49, 48, 47, 47, 47, 46, 44,
        43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 56, 55, 54, 53,
        53, 53, 53, 52, 51, 51, 51, 49, 48, 47, 47, 47, 46, 44, 43, 43, 43, 43,
        43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 56, 55, 54, 53, 53, 53, 53, 52,
        51, 51, 51, 49, 48, 47, 47, 47, 46, 44, 43, 43, 43, 43, 43, 43, 43, 43,
        43, 43, 43, 43, 43, 43, 53, 53, 52, 51, 51, 51, 51, 50, 50, 50, 50, 49,
        47, 47, 47, 47, 45, 44, 43, 43, 43, 43, 43, 43, 43, 43, 43, 42, 42, 42,
        42, 42, 51, 50, 49, 49, 49, 49, 49, 49, 49, 49, 49, 48, 47, 46, 46, 46,
        45, 44, 43, 43, 43, 43, 43, 42, 42, 42, 42, 41, 41, 41, 41, 41, 49, 49,
        49, 48, 48, 48, 48, 48, 48, 48, 48, 47, 46, 46, 46, 46, 45, 44, 43, 43,
        43, 43, 42, 42, 42, 42, 41, 41, 41, 41, 41, 41, 49, 49, 49, 48, 48, 48,
        48, 48, 48, 48, 48, 47, 46, 46, 46, 46, 45, 44, 43, 43, 43, 43, 42, 42,
        42, 42, 41, 41, 41, 41, 41, 41, 49, 49, 49, 48, 48, 48, 48, 48, 48, 48,
        48, 47, 46, 46, 46, 46, 45, 44, 43, 43, 43, 43, 42, 42, 42, 42, 41, 41,
        41, 41, 41, 41, 46, 46, 46, 46, 46, 46, 47, 47, 47, 47, 47, 46, 45, 45,
        45, 45, 44, 44, 43, 43, 43, 43, 42, 41, 41, 41, 41, 40, 40, 40, 40, 40,
        43, 44, 44, 44, 44, 44, 45, 45, 45, 45, 45, 45, 44, 44, 44, 44, 44, 43,
        43, 43, 43, 42, 41, 41, 41, 41, 40, 39, 39, 39, 39, 39, 42, 42, 43, 43,
        43, 43, 44, 44, 44, 44, 44, 44, 44, 43, 43, 43, 43, 43, 43, 43, 43, 42,
        41, 41, 41, 41, 40, 39, 39, 39, 39, 39, 42, 42, 43, 43, 43, 43, 44, 44,
        44, 44, 44, 44, 44, 43, 43, 43, 43, 43, 43, 43, 43, 42, 41, 41, 41, 41,
        40, 39, 39, 39, 39, 39, 42, 42, 43, 43, 43, 43, 44, 44, 44, 44, 44, 44,
        44, 43, 43, 43, 43, 43, 43, 43, 43, 42, 41, 41, 41, 41, 40, 39, 39, 39,
        39, 39, 42, 42, 43, 43, 43, 43, 44, 44, 45, 45, 45, 44, 44, 44, 44, 44,
        44, 44, 43, 43, 43, 42, 41, 41, 41, 41, 40, 39, 39, 39, 39, 38,
        /* Size 4x8 */
        65, 66, 55, 43, 65, 65, 53, 44, 66, 64, 51, 45, 59, 57, 48, 44, 55, 52,
        44, 43, 52, 50, 43, 42, 48, 48, 43, 41, 43, 44, 43, 39,
        /* Size 8x4 */
        65, 65, 66, 59, 55, 52, 48, 43, 66, 65, 64, 57, 52, 50, 48, 44, 55, 53,
        51, 48, 44, 43, 43, 43, 43, 44, 45, 44, 43, 42, 41, 39,
        /* Size 8x16 */
        64, 66, 66, 62, 55, 55, 45, 43, 65, 65, 65, 60, 54, 54, 46, 43, 65, 65,
        65, 60, 53, 53, 46, 44, 66, 65, 64, 59, 53, 53, 46, 44, 67, 64, 64, 59,
        51, 51, 46, 45, 67, 64, 64, 59, 51, 51, 46, 45, 62, 59, 59, 55, 48, 48,
        45, 44, 61, 58, 57, 53, 48, 48, 45, 44, 59, 55, 55, 51, 46, 46, 44, 43,
        55, 52, 51, 48, 44, 44, 43, 43, 55, 52, 51, 48, 44, 44, 43, 43, 50, 49,
        48, 47, 43, 43, 42, 41, 49, 48, 48, 46, 43, 43, 41, 41, 46, 46, 46, 45,
        43, 43, 41, 40, 42, 44, 44, 44, 43, 43, 39, 39, 42, 44, 44, 44, 43, 43,
        39, 39,
        /* Size 16x8 */
        64, 65, 65, 66, 67, 67, 62, 61, 59, 55, 55, 50, 49, 46, 42, 42, 66, 65,
        65, 65, 64, 64, 59, 58, 55, 52, 52, 49, 48, 46, 44, 44, 66, 65, 65, 64,
        64, 64, 59, 57, 55, 51, 51, 48, 48, 46, 44, 44, 62, 60, 60, 59, 59, 59,
        55, 53, 51, 48, 48, 47, 46, 45, 44, 44, 55, 54, 53, 53, 51, 51, 48, 48,
        46, 44, 44, 43, 43, 43, 43, 43, 55, 54, 53, 53, 51, 51, 48, 48, 46, 44,
        44, 43, 43, 43, 43, 43, 45, 46, 46, 46, 46, 46, 45, 45, 44, 43, 43, 42,
        41, 41, 39, 39, 43, 43, 44, 44, 45, 45, 44, 44, 43, 43, 43, 41, 41, 40,
        39, 39,
        /* Size 16x32 */
        64, 65, 66, 66, 66, 66, 62, 58, 55, 55, 55, 51, 45, 43, 43, 43, 65, 65,
        66, 66, 66, 66, 61, 57, 55, 55, 55, 50, 45, 43, 43, 43, 65, 65, 65, 65,
        65, 65, 60, 56, 54, 54, 54, 49, 46, 43, 43, 43, 65, 65, 65, 65, 65, 65,
        60, 55, 53, 53, 53, 49, 46, 44, 44, 44, 65, 65, 65, 65, 65, 65, 60, 55,
        53, 53, 53, 49, 46, 44, 44, 44, 65, 65, 65, 65, 65, 65, 60, 55, 53, 53,
        53, 49, 46, 44, 44, 44, 66, 65, 65, 64, 64, 64, 59, 55, 53, 53, 53, 49,
        46, 44, 44, 44, 67, 66, 65, 64, 64, 64, 59, 54, 52, 52, 52, 49, 46, 45,
        45, 45, 67, 66, 64, 64, 64, 64, 59, 54, 51, 51, 51, 49, 46, 45, 45, 45,
        67, 66, 64, 64, 64, 64, 59, 54, 51, 51, 51, 49, 46, 45, 45, 45, 67, 66,
        64, 64, 64, 64, 59, 54, 51, 51, 51, 49, 46, 45, 45, 45, 65, 63, 62, 61,
        61, 61, 57, 52, 50, 50, 50, 48, 46, 44, 44, 44, 62, 60, 59, 59, 59, 59,
        55, 51, 48, 48, 48, 47, 45, 44, 44, 44, 61, 59, 58, 57, 57, 57, 53, 49,
        48, 48, 48, 46, 45, 44, 44, 44, 61, 59, 58, 57, 57, 57, 53, 49, 48, 48,
        48, 46, 45, 44, 44, 44, 61, 59, 58, 57, 57, 57, 53, 49, 48, 48, 48, 46,
        45, 44, 44, 44, 59, 57, 55, 55, 55, 55, 51, 48, 46, 46, 46, 45, 44, 43,
        43, 43, 56, 55, 53, 52, 52, 52, 49, 46, 44, 44, 44, 44, 43, 43, 43, 43,
        55, 54, 52, 51, 51, 51, 48, 45, 44, 44, 44, 43, 43, 43, 43, 43, 55, 54,
        52, 51, 51, 51, 48, 45, 44, 44, 44, 43, 43, 43, 43, 43, 55, 54, 52, 51,
        51, 51, 48, 45, 44, 44, 44, 43, 43, 43, 43, 43, 53, 52, 51, 50, 50, 50,
        47, 45, 43, 43, 43, 43, 42, 42, 42, 42, 50, 49, 49, 48, 48, 48, 47, 45,
        43, 43, 43, 43, 42, 41, 41, 41, 49, 48, 48, 48, 48, 48, 46, 44, 43, 43,
        43, 42, 41, 41, 41, 41, 49, 48, 48, 48, 48, 48, 46, 44, 43, 43, 43, 42,
        41, 41, 41, 41, 49, 48, 48, 48, 48, 48, 46, 44, 43, 43, 43, 42, 41, 41,
        41, 41, 46, 46, 46, 46, 46, 46, 45, 44, 43, 43, 43, 42, 41, 40, 40, 40,
        43, 44, 45, 45, 45, 45, 44, 43, 43, 43, 43, 41, 40, 39, 39, 39, 42, 43,
        44, 44, 44, 44, 44, 43, 43, 43, 43, 41, 39, 39, 39, 39, 42, 43, 44, 44,
        44, 44, 44, 43, 43, 43, 43, 41, 39, 39, 39, 39, 42, 43, 44, 44, 44, 44,
        44, 43, 43, 43, 43, 41, 39, 39, 39, 39, 42, 43, 44, 44, 44, 44, 44, 43,
        43, 43, 43, 41, 39, 39, 39, 39,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 65, 66, 67, 67, 67, 67, 65, 62, 61, 61, 61, 59, 56,
        55, 55, 55, 53, 50, 49, 49, 49, 46, 43, 42, 42, 42, 42, 65, 65, 65, 65,
        65, 65, 65, 66, 66, 66, 66, 63, 60, 59, 59, 59, 57, 55, 54, 54, 54, 52,
        49, 48, 48, 48, 46, 44, 43, 43, 43, 43, 66, 66, 65, 65, 65, 65, 65, 65,
        64, 64, 64, 62, 59, 58, 58, 58, 55, 53, 52, 52, 52, 51, 49, 48, 48, 48,
        46, 45, 44, 44, 44, 44, 66, 66, 65, 65, 65, 65, 64, 64, 64, 64, 64, 61,
        59, 57, 57, 57, 55, 52, 51, 51, 51, 50, 48, 48, 48, 48, 46, 45, 44, 44,
        44, 44, 66, 66, 65, 65, 65, 65, 64, 64, 64, 64, 64, 61, 59, 57, 57, 57,
        55, 52, 51, 51, 51, 50, 48, 48, 48, 48, 46, 45, 44, 44, 44, 44, 66, 66,
        65, 65, 65, 65, 64, 64, 64, 64, 64, 61, 59, 57, 57, 57, 55, 52, 51, 51,
        51, 50, 48, 48, 48, 48, 46, 45, 44, 44, 44, 44, 62, 61, 60, 60, 60, 60,
        59, 59, 59, 59, 59, 57, 55, 53, 53, 53, 51, 49, 48, 48, 48, 47, 47, 46,
        46, 46, 45, 44, 44, 44, 44, 44, 58, 57, 56, 55, 55, 55, 55, 54, 54, 54,
        54, 52, 51, 49, 49, 49, 48, 46, 45, 45, 45, 45, 45, 44, 44, 44, 44, 43,
        43, 43, 43, 43, 55, 55, 54, 53, 53, 53, 53, 52, 51, 51, 51, 50, 48, 48,
        48, 48, 46, 44, 44, 44, 44, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43,
        55, 55, 54, 53, 53, 53, 53, 52, 51, 51, 51, 50, 48, 48, 48, 48, 46, 44,
        44, 44, 44, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 55, 55, 54, 53,
        53, 53, 53, 52, 51, 51, 51, 50, 48, 48, 48, 48, 46, 44, 44, 44, 44, 43,
        43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 51, 50, 49, 49, 49, 49, 49, 49,
        49, 49, 49, 48, 47, 46, 46, 46, 45, 44, 43, 43, 43, 43, 43, 42, 42, 42,
        42, 41, 41, 41, 41, 41, 45, 45, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46,
        45, 45, 45, 45, 44, 43, 43, 43, 43, 42, 42, 41, 41, 41, 41, 40, 39, 39,
        39, 39, 43, 43, 43, 44, 44, 44, 44, 45, 45, 45, 45, 44, 44, 44, 44, 44,
        43, 43, 43, 43, 43, 42, 41, 41, 41, 41, 40, 39, 39, 39, 39, 39, 43, 43,
        43, 44, 44, 44, 44, 45, 45, 45, 45, 44, 44, 44, 44, 44, 43, 43, 43, 43,
        43, 42, 41, 41, 41, 41, 40, 39, 39, 39, 39, 39, 43, 43, 43, 44, 44, 44,
        44, 45, 45, 45, 45, 44, 44, 44, 44, 44, 43, 43, 43, 43, 43, 42, 41, 41,
        41, 41, 40, 39, 39, 39, 39, 39,
        /* Size 4x16 */
        65, 66, 55, 43, 65, 65, 54, 43, 65, 65, 53, 44, 65, 64, 53, 44, 66, 64,
        51, 45, 66, 64, 51, 45, 60, 59, 48, 44, 59, 57, 48, 44, 57, 55, 46, 43,
        54, 51, 44, 43, 54, 51, 44, 43, 49, 48, 43, 41, 48, 48, 43, 41, 46, 46,
        43, 40, 43, 44, 43, 39, 43, 44, 43, 39,
        /* Size 16x4 */
        65, 65, 65, 65, 66, 66, 60, 59, 57, 54, 54, 49, 48, 46, 43, 43, 66, 65,
        65, 64, 64, 64, 59, 57, 55, 51, 51, 48, 48, 46, 44, 44, 55, 54, 53, 53,
        51, 51, 48, 48, 46, 44, 44, 43, 43, 43, 43, 43, 43, 43, 44, 44, 45, 45,
        44, 44, 43, 43, 43, 41, 41, 40, 39, 39,
        /* Size 8x32 */
        64, 66, 66, 62, 55, 55, 45, 43, 65, 66, 66, 61, 55, 55, 45, 43, 65, 65,
        65, 60, 54, 54, 46, 43, 65, 65, 65, 60, 53, 53, 46, 44, 65, 65, 65, 60,
        53, 53, 46, 44, 65, 65, 65, 60, 53, 53, 46, 44, 66, 65, 64, 59, 53, 53,
        46, 44, 67, 65, 64, 59, 52, 52, 46, 45, 67, 64, 64, 59, 51, 51, 46, 45,
        67, 64, 64, 59, 51, 51, 46, 45, 67, 64, 64, 59, 51, 51, 46, 45, 65, 62,
        61, 57, 50, 50, 46, 44, 62, 59, 59, 55, 48, 48, 45, 44, 61, 58, 57, 53,
        48, 48, 45, 44, 61, 58, 57, 53, 48, 48, 45, 44, 61, 58, 57, 53, 48, 48,
        45, 44, 59, 55, 55, 51, 46, 46, 44, 43, 56, 53, 52, 49, 44, 44, 43, 43,
        55, 52, 51, 48, 44, 44, 43, 43, 55, 52, 51, 48, 44, 44, 43, 43, 55, 52,
        51, 48, 44, 44, 43, 43, 53, 51, 50, 47, 43, 43, 42, 42, 50, 49, 48, 47,
        43, 43, 42, 41, 49, 48, 48, 46, 43, 43, 41, 41, 49, 48, 48, 46, 43, 43,
        41, 41, 49, 48, 48, 46, 43, 43, 41, 41, 46, 46, 46, 45, 43, 43, 41, 40,
        43, 45, 45, 44, 43, 43, 40, 39, 42, 44, 44, 44, 43, 43, 39, 39, 42, 44,
        44, 44, 43, 43, 39, 39, 42, 44, 44, 44, 43, 43, 39, 39, 42, 44, 44, 44,
        43, 43, 39, 39,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 65, 66, 67, 67, 67, 67, 65, 62, 61, 61, 61, 59, 56,
        55, 55, 55, 53, 50, 49, 49, 49, 46, 43, 42, 42, 42, 42, 66, 66, 65, 65,
        65, 65, 65, 65, 64, 64, 64, 62, 59, 58, 58, 58, 55, 53, 52, 52, 52, 51,
        49, 48, 48, 48, 46, 45, 44, 44, 44, 44, 66, 66, 65, 65, 65, 65, 64, 64,
        64, 64, 64, 61, 59, 57, 57, 57, 55, 52, 51, 51, 51, 50, 48, 48, 48, 48,
        46, 45, 44, 44, 44, 44, 62, 61, 60, 60, 60, 60, 59, 59, 59, 59, 59, 57,
        55, 53, 53, 53, 51, 49, 48, 48, 48, 47, 47, 46, 46, 46, 45, 44, 44, 44,
        44, 44, 55, 55, 54, 53, 53, 53, 53, 52, 51, 51, 51, 50, 48, 48, 48, 48,
        46, 44, 44, 44, 44, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 55, 55,
        54, 53, 53, 53, 53, 52, 51, 51, 51, 50, 48, 48, 48, 48, 46, 44, 44, 44,
        44, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 45, 45, 46, 46, 46, 46,
        46, 46, 46, 46, 46, 46, 45, 45, 45, 45, 44, 43, 43, 43, 43, 42, 42, 41,
        41, 41, 41, 40, 39, 39, 39, 39, 43, 43, 43, 44, 44, 44, 44, 45, 45, 45,
        45, 44, 44, 44, 44, 44, 43, 43, 43, 43, 43, 42, 41, 41, 41, 41, 40, 39,
        39, 39, 39, 39 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        64, 63, 63, 59,
#endif
        /* Size 4x4 */
        65, 64, 64, 63, 64, 63, 63, 62, 64, 63, 61, 60, 63, 62, 60, 58,
        /* Size 8x8 */
        65, 65, 65, 65, 64, 64, 64, 61, 65, 64, 64, 64, 64, 64, 63, 61, 65, 64,
        64, 64, 63, 63, 63, 61, 65, 64, 64, 63, 63, 63, 63, 61, 64, 64, 63, 63,
        61, 61, 60, 59, 64, 64, 63, 63, 61, 60, 59, 58, 64, 63, 63, 63, 60, 59,
        58, 56, 61, 61, 61, 61, 59, 58, 56, 54,
        /* Size 16x16 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 63, 60, 65, 65,
        65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 62, 60, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 62, 60, 65, 65, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 63, 63, 63, 62, 60, 65, 65, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 63, 62, 60, 65, 65, 64, 64, 64, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 62, 61, 65, 65, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 62, 61, 65, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 62, 62, 62,
        61, 60, 65, 64, 64, 64, 63, 63, 63, 63, 62, 62, 62, 61, 61, 61, 60, 59,
        65, 64, 64, 64, 63, 63, 63, 63, 62, 61, 61, 61, 60, 60, 59, 58, 65, 64,
        64, 64, 63, 63, 63, 63, 62, 61, 61, 61, 60, 60, 59, 58, 64, 64, 63, 63,
        63, 63, 63, 62, 61, 61, 61, 59, 59, 59, 58, 57, 64, 64, 63, 63, 63, 63,
        63, 62, 61, 60, 60, 59, 58, 58, 57, 55, 64, 64, 63, 63, 63, 63, 63, 62,
        61, 60, 60, 59, 58, 58, 57, 55, 63, 62, 62, 62, 62, 62, 62, 61, 60, 59,
        59, 58, 57, 57, 56, 54, 60, 60, 60, 60, 60, 61, 61, 60, 59, 58, 58, 57,
        55, 55, 54, 52,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 63, 62, 60, 60, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 61, 60, 60, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 63, 62, 61, 60, 60, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 62, 61,
        60, 60, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 62, 61, 60, 60, 65, 65,
        65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 62, 61, 60, 60, 65, 65, 65, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 63, 62, 61, 60, 60, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63,
        62, 61, 60, 60, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 61, 60, 60,
        65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 61, 61, 61, 65, 65, 65, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 62, 62, 61, 61, 61, 65, 65, 65, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 62, 62, 61, 61, 61, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 61,
        61, 61, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 61, 61, 61, 65, 65,
        64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 62, 62, 62, 62, 62, 62, 62, 61, 61, 60, 60, 65, 65, 64, 64, 64, 64,
        64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 61,
        61, 61, 61, 61, 60, 60, 59, 59, 65, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 61, 61, 61, 61, 61, 61, 60,
        60, 59, 59, 59, 65, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 62, 62, 61, 61, 61, 61, 61, 61, 60, 60, 60, 60, 60, 59, 59, 58, 58,
        65, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 62, 62, 61,
        61, 61, 61, 61, 61, 60, 60, 60, 60, 60, 59, 59, 58, 58, 65, 64, 64, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 62, 62, 61, 61, 61, 61, 61,
        61, 60, 60, 60, 60, 60, 59, 59, 58, 58, 65, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 63, 63, 62, 62, 61, 61, 61, 61, 61, 61, 60, 60, 60,
        60, 60, 59, 59, 58, 58, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63,
        63, 63, 62, 62, 61, 61, 61, 61, 61, 60, 60, 59, 59, 59, 59, 59, 59, 58,
        58, 58, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62,
        61, 61, 61, 61, 61, 60, 59, 59, 59, 59, 59, 59, 58, 58, 57, 57, 64, 64,
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 61, 61, 60, 60, 60,
        60, 59, 59, 59, 59, 59, 59, 58, 58, 57, 56, 56, 64, 64, 64, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 62, 61, 61, 60, 60, 60, 60, 59, 59, 59,
        58, 58, 58, 58, 57, 56, 55, 55, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 62, 61, 61, 60, 60, 60, 60, 59, 59, 59, 58, 58, 58, 58,
        57, 56, 55, 55, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        62, 61, 61, 60, 60, 60, 60, 59, 59, 59, 58, 58, 58, 58, 57, 56, 55, 55,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 61, 60, 60,
        60, 60, 60, 59, 59, 58, 58, 58, 58, 58, 57, 56, 55, 55, 63, 63, 62, 62,
        62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 61, 60, 60, 59, 59, 59, 59, 59,
        58, 58, 57, 57, 57, 57, 56, 55, 54, 54, 62, 61, 61, 61, 61, 61, 61, 61,
        61, 61, 61, 61, 61, 61, 61, 60, 59, 59, 59, 59, 59, 58, 58, 57, 56, 56,
        56, 56, 55, 54, 53, 53, 60, 60, 60, 60, 60, 60, 60, 60, 60, 61, 61, 61,
        61, 61, 60, 59, 59, 58, 58, 58, 58, 58, 57, 56, 55, 55, 55, 55, 54, 53,
        52, 52, 60, 60, 60, 60, 60, 60, 60, 60, 60, 61, 61, 61, 61, 61, 60, 59,
        59, 58, 58, 58, 58, 58, 57, 56, 55, 55, 55, 55, 54, 53, 52, 52,
        /* Size 4x8 */
        65, 65, 64, 64, 65, 64, 63, 63, 64, 64, 63, 63, 64, 63, 63, 62, 64, 63,
        61, 60, 64, 63, 60, 59, 64, 62, 59, 58, 61, 61, 58, 56,
        /* Size 8x4 */
        65, 65, 64, 64, 64, 64, 64, 61, 65, 64, 64, 63, 63, 63, 62, 61, 64, 63,
        63, 63, 61, 60, 59, 58, 64, 63, 63, 62, 60, 59, 58, 56,
        /* Size 8x16 */
        64, 65, 65, 65, 65, 64, 64, 63, 65, 65, 64, 64, 64, 63, 63, 62, 65, 64,
        64, 64, 64, 63, 63, 62, 65, 64, 64, 64, 64, 63, 63, 62, 65, 64, 64, 64,
        63, 63, 63, 62, 65, 64, 63, 63, 63, 62, 62, 62, 65, 64, 63, 63, 63, 62,
        62, 62, 65, 64, 63, 63, 63, 62, 62, 61, 65, 64, 63, 63, 62, 60, 60, 59,
        64, 63, 63, 63, 62, 60, 60, 59, 64, 63, 63, 63, 62, 60, 60, 59, 64, 63,
        63, 63, 61, 59, 59, 58, 64, 63, 62, 62, 60, 58, 58, 57, 64, 63, 62, 62,
        60, 58, 58, 57, 63, 62, 62, 62, 59, 57, 57, 56, 60, 60, 60, 60, 59, 55,
        55, 54,
        /* Size 16x8 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 63, 60, 65, 65,
        64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 62, 60, 65, 64, 64, 64,
        64, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 60, 65, 64, 64, 64, 64, 63,
        63, 63, 63, 63, 63, 63, 62, 62, 62, 60, 65, 64, 64, 64, 63, 63, 63, 63,
        62, 62, 62, 61, 60, 60, 59, 59, 64, 63, 63, 63, 63, 62, 62, 62, 60, 60,
        60, 59, 58, 58, 57, 55, 64, 63, 63, 63, 63, 62, 62, 62, 60, 60, 60, 59,
        58, 58, 57, 55, 63, 62, 62, 62, 62, 62, 62, 61, 59, 59, 59, 58, 57, 57,
        56, 54,
        /* Size 16x32 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 63, 60, 65, 65,
        65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 62, 60, 65, 65, 65, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 62, 60, 65, 65, 64, 64, 64, 64,
        64, 64, 64, 63, 63, 63, 63, 63, 62, 60, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 63, 63, 63, 63, 63, 62, 60, 65, 65, 64, 64, 64, 64, 64, 64, 64, 63,
        63, 63, 63, 63, 62, 60, 65, 65, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63,
        63, 63, 62, 60, 65, 65, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63,
        62, 60, 65, 65, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 62, 60,
        65, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 62, 61, 65, 64,
        64, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 61, 65, 64, 64, 63,
        63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 61, 65, 64, 64, 63, 63, 63,
        63, 63, 63, 63, 62, 62, 62, 62, 62, 61, 65, 64, 64, 63, 63, 63, 63, 63,
        63, 63, 62, 62, 62, 62, 62, 61, 65, 64, 64, 63, 63, 63, 63, 63, 63, 62,
        62, 62, 62, 62, 61, 60, 65, 64, 64, 63, 63, 63, 63, 63, 62, 62, 61, 61,
        61, 61, 60, 59, 65, 64, 64, 63, 63, 63, 63, 63, 62, 61, 60, 60, 60, 60,
        59, 59, 64, 64, 63, 63, 63, 63, 63, 62, 62, 61, 60, 60, 60, 60, 59, 58,
        64, 64, 63, 63, 63, 63, 63, 62, 62, 61, 60, 60, 60, 60, 59, 58, 64, 64,
        63, 63, 63, 63, 63, 62, 62, 61, 60, 60, 60, 60, 59, 58, 64, 64, 63, 63,
        63, 63, 63, 62, 62, 61, 60, 60, 60, 60, 59, 58, 64, 64, 63, 63, 63, 63,
        63, 62, 61, 60, 59, 59, 59, 59, 59, 58, 64, 64, 63, 63, 63, 63, 63, 62,
        61, 60, 59, 59, 59, 59, 58, 57, 64, 64, 63, 63, 62, 62, 62, 62, 61, 59,
        59, 59, 59, 59, 57, 56, 64, 64, 63, 63, 62, 62, 62, 62, 60, 59, 58, 58,
        58, 58, 57, 55, 64, 64, 63, 63, 62, 62, 62, 62, 60, 59, 58, 58, 58, 58,
        57, 55, 64, 64, 63, 63, 62, 62, 62, 62, 60, 59, 58, 58, 58, 58, 57, 55,
        64, 63, 63, 62, 62, 62, 62, 62, 60, 59, 58, 58, 58, 58, 57, 55, 63, 62,
        62, 62, 62, 62, 62, 61, 59, 59, 57, 57, 57, 57, 56, 54, 61, 61, 61, 61,
        61, 61, 61, 60, 59, 58, 56, 56, 56, 56, 55, 53, 60, 60, 60, 60, 60, 60,
        60, 60, 59, 57, 55, 55, 55, 55, 54, 52, 60, 60, 60, 60, 60, 60, 60, 60,
        59, 57, 55, 55, 55, 55, 54, 52,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 61, 60, 60, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 63, 62, 61, 60, 60, 65, 65, 65, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 62, 61, 60, 60, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 61,
        60, 60, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 61, 60, 60, 65, 65,
        64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 62, 62, 62, 62, 62, 62, 61, 60, 60, 65, 65, 64, 64, 64, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62,
        62, 62, 62, 62, 62, 61, 60, 60, 65, 65, 64, 64, 64, 64, 64, 64, 64, 63,
        63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62,
        61, 60, 60, 60, 65, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 62, 62, 62, 62, 62, 62, 61, 61, 61, 60, 60, 60, 60, 59, 59, 59, 59,
        64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 61, 61,
        61, 61, 61, 60, 60, 59, 59, 59, 59, 59, 59, 58, 57, 57, 64, 64, 63, 63,
        63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 61, 60, 60, 60, 60, 60, 59,
        59, 59, 58, 58, 58, 58, 57, 56, 55, 55, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 63, 62, 62, 62, 62, 62, 61, 60, 60, 60, 60, 60, 59, 59, 59, 58, 58,
        58, 58, 57, 56, 55, 55, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62,
        62, 62, 62, 61, 60, 60, 60, 60, 60, 59, 59, 59, 58, 58, 58, 58, 57, 56,
        55, 55, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 61,
        60, 60, 60, 60, 60, 59, 59, 59, 58, 58, 58, 58, 57, 56, 55, 55, 63, 62,
        62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 61, 60, 59, 59, 59, 59,
        59, 59, 58, 57, 57, 57, 57, 57, 56, 55, 54, 54, 60, 60, 60, 60, 60, 60,
        60, 60, 60, 61, 61, 61, 61, 61, 60, 59, 59, 58, 58, 58, 58, 58, 57, 56,
        55, 55, 55, 55, 54, 53, 52, 52,
        /* Size 4x16 */
        65, 65, 64, 64, 65, 64, 64, 63, 65, 64, 63, 63, 65, 64, 63, 63, 65, 64,
        63, 63, 64, 63, 63, 62, 64, 63, 63, 62, 64, 63, 62, 62, 64, 63, 61, 60,
        64, 63, 61, 60, 64, 63, 61, 60, 64, 63, 60, 59, 64, 62, 59, 58, 64, 62,
        59, 58, 62, 62, 59, 57, 60, 60, 57, 55,
        /* Size 16x4 */
        65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62, 60, 65, 64,
        64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 60, 64, 64, 63, 63,
        63, 63, 63, 62, 61, 61, 61, 60, 59, 59, 59, 57, 64, 63, 63, 63, 63, 62,
        62, 62, 60, 60, 60, 59, 58, 58, 57, 55,
        /* Size 8x32 */
        64, 65, 65, 65, 65, 64, 64, 63, 65, 65, 65, 65, 64, 64, 64, 62, 65, 65,
        64, 64, 64, 63, 63, 62, 65, 64, 64, 64, 64, 63, 63, 62, 65, 64, 64, 64,
        64, 63, 63, 62, 65, 64, 64, 64, 64, 63, 63, 62, 65, 64, 64, 64, 64, 63,
        63, 62, 65, 64, 64, 64, 64, 63, 63, 62, 65, 64, 64, 64, 63, 63, 63, 62,
        65, 64, 64, 64, 63, 63, 63, 62, 65, 64, 63, 63, 63, 62, 62, 62, 65, 64,
        63, 63, 63, 62, 62, 62, 65, 64, 63, 63, 63, 62, 62, 62, 65, 64, 63, 63,
        63, 62, 62, 62, 65, 64, 63, 63, 63, 62, 62, 61, 65, 64, 63, 63, 62, 61,
        61, 60, 65, 64, 63, 63, 62, 60, 60, 59, 64, 63, 63, 63, 62, 60, 60, 59,
        64, 63, 63, 63, 62, 60, 60, 59, 64, 63, 63, 63, 62, 60, 60, 59, 64, 63,
        63, 63, 62, 60, 60, 59, 64, 63, 63, 63, 61, 59, 59, 59, 64, 63, 63, 63,
        61, 59, 59, 58, 64, 63, 62, 62, 61, 59, 59, 57, 64, 63, 62, 62, 60, 58,
        58, 57, 64, 63, 62, 62, 60, 58, 58, 57, 64, 63, 62, 62, 60, 58, 58, 57,
        64, 63, 62, 62, 60, 58, 58, 57, 63, 62, 62, 62, 59, 57, 57, 56, 61, 61,
        61, 61, 59, 56, 56, 55, 60, 60, 60, 60, 59, 55, 55, 54, 60, 60, 60, 60,
        59, 55, 55, 54,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 61, 60, 60, 65, 65, 65, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 62, 61, 60, 60, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62,
        62, 62, 62, 61, 60, 60, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 61,
        60, 60, 65, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 62,
        62, 62, 62, 62, 62, 61, 61, 61, 60, 60, 60, 60, 59, 59, 59, 59, 64, 64,
        63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 61, 60, 60, 60, 60,
        60, 59, 59, 59, 58, 58, 58, 58, 57, 56, 55, 55, 64, 64, 63, 63, 63, 63,
        63, 63, 63, 63, 62, 62, 62, 62, 62, 61, 60, 60, 60, 60, 60, 59, 59, 59,
        58, 58, 58, 58, 57, 56, 55, 55, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62,
        62, 62, 62, 62, 61, 60, 59, 59, 59, 59, 59, 59, 58, 57, 57, 57, 57, 57,
        56, 55, 54, 54 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        65, 54, 54, 44,
#endif
        /* Size 4x4 */
        65, 66, 59, 53, 66, 64, 58, 51, 59, 58, 53, 47, 53, 51, 47, 43,
        /* Size 8x8 */
        65, 66, 66, 67, 60, 59, 55, 51, 66, 65, 65, 66, 59, 58, 53, 50, 66, 65,
        65, 65, 59, 56, 52, 49, 67, 66, 65, 64, 58, 56, 51, 49, 60, 59, 59, 58,
        53, 51, 47, 46, 59, 58, 56, 56, 51, 49, 46, 45, 55, 53, 52, 51, 47, 46,
        43, 43, 51, 50, 49, 49, 46, 45, 43, 43,
        /* Size 16x16 */
        64, 65, 66, 66, 66, 67, 67, 66, 62, 61, 61, 58, 56, 56, 53, 49, 65, 65,
        65, 65, 66, 66, 66, 65, 61, 60, 60, 57, 55, 55, 52, 49, 66, 65, 65, 65,
        65, 66, 66, 64, 60, 59, 59, 55, 53, 53, 51, 48, 66, 65, 65, 65, 65, 66,
        66, 64, 60, 59, 59, 55, 53, 53, 51, 48, 66, 66, 65, 65, 65, 65, 65, 64,
        60, 59, 59, 55, 53, 53, 51, 48, 67, 66, 66, 66, 65, 64, 64, 63, 59, 58,
        58, 54, 51, 51, 50, 48, 67, 66, 66, 66, 65, 64, 64, 63, 59, 58, 58, 54,
        51, 51, 50, 48, 66, 65, 64, 64, 64, 63, 63, 62, 58, 57, 57, 53, 51, 51,
        49, 48, 62, 61, 60, 60, 60, 59, 59, 58, 55, 53, 53, 50, 48, 48, 47, 46,
        61, 60, 59, 59, 59, 58, 58, 57, 53, 53, 53, 49, 47, 47, 47, 46, 61, 60,
        59, 59, 59, 58, 58, 57, 53, 53, 53, 49, 47, 47, 47, 46, 58, 57, 55, 55,
        55, 54, 54, 53, 50, 49, 49, 47, 45, 45, 45, 44, 56, 55, 53, 53, 53, 51,
        51, 51, 48, 47, 47, 45, 43, 43, 43, 43, 56, 55, 53, 53, 53, 51, 51, 51,
        48, 47, 47, 45, 43, 43, 43, 43, 53, 52, 51, 51, 51, 50, 50, 49, 47, 47,
        47, 45, 43, 43, 43, 43, 49, 49, 48, 48, 48, 48, 48, 48, 46, 46, 46, 44,
        43, 43, 43, 42,
        /* Size 32x32 */
        64, 65, 65, 66, 66, 66, 66, 66, 66, 67, 67, 67, 67, 67, 66, 64, 62, 61,
        61, 61, 61, 59, 58, 56, 56, 56, 56, 55, 53, 51, 49, 49, 65, 65, 65, 65,
        66, 66, 66, 66, 66, 66, 67, 67, 67, 67, 66, 64, 62, 60, 60, 60, 60, 59,
        57, 55, 55, 55, 55, 55, 53, 51, 49, 49, 65, 65, 65, 65, 65, 65, 65, 65,
        66, 66, 66, 66, 66, 66, 65, 63, 61, 60, 60, 60, 60, 59, 57, 55, 55, 55,
        55, 54, 52, 51, 49, 49, 66, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66,
        66, 66, 64, 63, 60, 59, 59, 59, 59, 58, 56, 54, 53, 53, 53, 53, 51, 50,
        49, 49, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 64, 62,
        60, 59, 59, 59, 59, 58, 55, 54, 53, 53, 53, 53, 51, 50, 48, 48, 66, 66,
        65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 64, 62, 60, 59, 59, 59,
        59, 58, 55, 54, 53, 53, 53, 53, 51, 50, 48, 48, 66, 66, 65, 65, 65, 65,
        65, 65, 65, 65, 66, 66, 66, 66, 64, 62, 60, 59, 59, 59, 59, 58, 55, 54,
        53, 53, 53, 53, 51, 50, 48, 48, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 64, 62, 60, 59, 59, 59, 59, 57, 55, 54, 53, 53, 53, 53,
        51, 50, 48, 48, 66, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        64, 62, 60, 59, 59, 59, 59, 57, 55, 53, 53, 53, 53, 52, 51, 49, 48, 48,
        67, 66, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 63, 61, 59, 59,
        59, 59, 59, 56, 55, 53, 52, 52, 52, 51, 51, 49, 48, 48, 67, 67, 66, 66,
        66, 66, 66, 65, 65, 65, 64, 64, 64, 64, 63, 61, 59, 58, 58, 58, 58, 56,
        54, 52, 51, 51, 51, 51, 50, 49, 48, 48, 67, 67, 66, 66, 66, 66, 66, 65,
        65, 65, 64, 64, 64, 64, 63, 61, 59, 58, 58, 58, 58, 56, 54, 52, 51, 51,
        51, 51, 50, 49, 48, 48, 67, 67, 66, 66, 66, 66, 66, 65, 65, 65, 64, 64,
        64, 64, 63, 61, 59, 58, 58, 58, 58, 56, 54, 52, 51, 51, 51, 51, 50, 49,
        48, 48, 67, 67, 66, 66, 66, 66, 66, 65, 65, 65, 64, 64, 64, 64, 63, 61,
        59, 58, 58, 58, 58, 56, 54, 52, 51, 51, 51, 51, 50, 49, 48, 48, 66, 66,
        65, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 62, 60, 58, 57, 57, 57,
        57, 55, 53, 51, 51, 51, 51, 50, 49, 48, 48, 48, 64, 64, 63, 63, 62, 62,
        62, 62, 62, 61, 61, 61, 61, 61, 60, 58, 56, 55, 55, 55, 55, 53, 52, 50,
        49, 49, 49, 49, 48, 48, 47, 47, 62, 62, 61, 60, 60, 60, 60, 60, 60, 59,
        59, 59, 59, 59, 58, 56, 55, 53, 53, 53, 53, 52, 50, 49, 48, 48, 48, 48,
        47, 47, 46, 46, 61, 60, 60, 59, 59, 59, 59, 59, 59, 59, 58, 58, 58, 58,
        57, 55, 53, 53, 53, 53, 53, 51, 49, 48, 47, 47, 47, 47, 47, 46, 46, 46,
        61, 60, 60, 59, 59, 59, 59, 59, 59, 59, 58, 58, 58, 58, 57, 55, 53, 53,
        53, 53, 53, 51, 49, 48, 47, 47, 47, 47, 47, 46, 46, 46, 61, 60, 60, 59,
        59, 59, 59, 59, 59, 59, 58, 58, 58, 58, 57, 55, 53, 53, 53, 53, 53, 51,
        49, 48, 47, 47, 47, 47, 47, 46, 46, 46, 61, 60, 60, 59, 59, 59, 59, 59,
        59, 59, 58, 58, 58, 58, 57, 55, 53, 53, 53, 53, 53, 51, 49, 48, 47, 47,
        47, 47, 47, 46, 46, 46, 59, 59, 59, 58, 58, 58, 58, 57, 57, 56, 56, 56,
        56, 56, 55, 53, 52, 51, 51, 51, 51, 49, 48, 47, 46, 46, 46, 46, 46, 45,
        45, 45, 58, 57, 57, 56, 55, 55, 55, 55, 55, 55, 54, 54, 54, 54, 53, 52,
        50, 49, 49, 49, 49, 48, 47, 45, 45, 45, 45, 45, 45, 45, 44, 44, 56, 55,
        55, 54, 54, 54, 54, 54, 53, 53, 52, 52, 52, 52, 51, 50, 49, 48, 48, 48,
        48, 47, 45, 44, 44, 44, 44, 44, 44, 44, 44, 44, 56, 55, 55, 53, 53, 53,
        53, 53, 53, 52, 51, 51, 51, 51, 51, 49, 48, 47, 47, 47, 47, 46, 45, 44,
        43, 43, 43, 43, 43, 43, 43, 43, 56, 55, 55, 53, 53, 53, 53, 53, 53, 52,
        51, 51, 51, 51, 51, 49, 48, 47, 47, 47, 47, 46, 45, 44, 43, 43, 43, 43,
        43, 43, 43, 43, 56, 55, 55, 53, 53, 53, 53, 53, 53, 52, 51, 51, 51, 51,
        51, 49, 48, 47, 47, 47, 47, 46, 45, 44, 43, 43, 43, 43, 43, 43, 43, 43,
        55, 55, 54, 53, 53, 53, 53, 53, 52, 51, 51, 51, 51, 51, 50, 49, 48, 47,
        47, 47, 47, 46, 45, 44, 43, 43, 43, 43, 43, 43, 43, 43, 53, 53, 52, 51,
        51, 51, 51, 51, 51, 51, 50, 50, 50, 50, 49, 48, 47, 47, 47, 47, 47, 46,
        45, 44, 43, 43, 43, 43, 43, 43, 43, 43, 51, 51, 51, 50, 50, 50, 50, 50,
        49, 49, 49, 49, 49, 49, 48, 48, 47, 46, 46, 46, 46, 45, 45, 44, 43, 43,
        43, 43, 43, 43, 43, 43, 49, 49, 49, 49, 48, 48, 48, 48, 48, 48, 48, 48,
        48, 48, 48, 47, 46, 46, 46, 46, 46, 45, 44, 44, 43, 43, 43, 43, 43, 43,
        42, 42, 49, 49, 49, 49, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 47,
        46, 46, 46, 46, 46, 45, 44, 44, 43, 43, 43, 43, 43, 43, 42, 42,
        /* Size 4x8 */
        65, 66, 59, 55, 65, 65, 57, 53, 66, 64, 55, 52, 66, 64, 55, 51, 59, 57,
        51, 48, 58, 55, 49, 46, 54, 51, 46, 44, 50, 49, 45, 43,
        /* Size 8x4 */
        65, 65, 66, 66, 59, 58, 54, 50, 66, 65, 64, 64, 57, 55, 51, 49, 59, 57,
        55, 55, 51, 49, 46, 45, 55, 53, 52, 51, 48, 46, 44, 43,
        /* Size 8x16 */
        64, 66, 66, 66, 62, 55, 55, 53, 65, 65, 65, 65, 61, 54, 54, 52, 65, 65,
        65, 65, 60, 53, 53, 51, 65, 65, 65, 65, 60, 53, 53, 51, 66, 65, 64, 64,
        59, 53, 53, 51, 67, 65, 64, 64, 59, 51, 51, 50, 67, 65, 64, 64, 59, 51,
        51, 50, 66, 64, 62, 62, 58, 51, 51, 49, 62, 60, 59, 59, 55, 48, 48, 48,
        61, 59, 57, 57, 53, 48, 48, 47, 61, 59, 57, 57, 53, 48, 48, 47, 58, 55,
        53, 53, 50, 45, 45, 45, 55, 53, 51, 51, 48, 44, 44, 43, 55, 53, 51, 51,
        48, 44, 44, 43, 53, 51, 50, 50, 47, 43, 43, 43, 49, 48, 48, 48, 46, 43,
        43, 43,
        /* Size 16x8 */
        64, 65, 65, 65, 66, 67, 67, 66, 62, 61, 61, 58, 55, 55, 53, 49, 66, 65,
        65, 65, 65, 65, 65, 64, 60, 59, 59, 55, 53, 53, 51, 48, 66, 65, 65, 65,
        64, 64, 64, 62, 59, 57, 57, 53, 51, 51, 50, 48, 66, 65, 65, 65, 64, 64,
        64, 62, 59, 57, 57, 53, 51, 51, 50, 48, 62, 61, 60, 60, 59, 59, 59, 58,
        55, 53, 53, 50, 48, 48, 47, 46, 55, 54, 53, 53, 53, 51, 51, 51, 48, 48,
        48, 45, 44, 44, 43, 43, 55, 54, 53, 53, 53, 51, 51, 51, 48, 48, 48, 45,
        44, 44, 43, 43, 53, 52, 51, 51, 51, 50, 50, 49, 48, 47, 47, 45, 43, 43,
        43, 43,
        /* Size 16x32 */
        64, 65, 66, 66, 66, 66, 66, 65, 62, 59, 55, 55, 55, 55, 53, 49, 65, 65,
        65, 66, 66, 66, 66, 65, 61, 59, 55, 55, 55, 55, 53, 49, 65, 65, 65, 65,
        65, 65, 65, 64, 61, 58, 54, 54, 54, 54, 52, 49, 65, 65, 65, 65, 65, 65,
        65, 64, 60, 57, 53, 53, 53, 53, 51, 48, 65, 65, 65, 65, 65, 65, 65, 64,
        60, 57, 53, 53, 53, 53, 51, 48, 65, 65, 65, 65, 65, 65, 65, 64, 60, 57,
        53, 53, 53, 53, 51, 48, 65, 65, 65, 65, 65, 65, 65, 64, 60, 57, 53, 53,
        53, 53, 51, 48, 66, 65, 65, 65, 65, 65, 65, 64, 60, 57, 53, 53, 53, 53,
        51, 48, 66, 66, 65, 65, 64, 64, 64, 63, 59, 56, 53, 53, 53, 53, 51, 48,
        67, 66, 65, 64, 64, 64, 64, 63, 59, 55, 52, 52, 52, 52, 51, 48, 67, 66,
        65, 64, 64, 64, 64, 62, 59, 55, 51, 51, 51, 51, 50, 48, 67, 66, 65, 64,
        64, 64, 64, 62, 59, 55, 51, 51, 51, 51, 50, 48, 67, 66, 65, 64, 64, 64,
        64, 62, 59, 55, 51, 51, 51, 51, 50, 48, 67, 66, 65, 64, 64, 64, 64, 62,
        59, 55, 51, 51, 51, 51, 50, 48, 66, 65, 64, 63, 62, 62, 62, 61, 58, 54,
        51, 51, 51, 51, 49, 48, 64, 63, 62, 61, 60, 60, 60, 59, 56, 53, 49, 49,
        49, 49, 48, 47, 62, 61, 60, 59, 59, 59, 59, 58, 55, 51, 48, 48, 48, 48,
        48, 46, 61, 59, 59, 58, 57, 57, 57, 56, 53, 51, 48, 48, 48, 48, 47, 46,
        61, 59, 59, 58, 57, 57, 57, 56, 53, 51, 48, 48, 48, 48, 47, 46, 61, 59,
        59, 58, 57, 57, 57, 56, 53, 51, 48, 48, 48, 48, 47, 46, 61, 59, 59, 58,
        57, 57, 57, 56, 53, 51, 48, 48, 48, 48, 47, 46, 59, 58, 57, 55, 55, 55,
        55, 55, 52, 49, 46, 46, 46, 46, 46, 45, 58, 56, 55, 54, 53, 53, 53, 53,
        50, 48, 45, 45, 45, 45, 45, 44, 56, 55, 53, 52, 52, 52, 52, 51, 48, 46,
        44, 44, 44, 44, 44, 44, 55, 54, 53, 51, 51, 51, 51, 50, 48, 46, 44, 44,
        44, 44, 43, 43, 55, 54, 53, 51, 51, 51, 51, 50, 48, 46, 44, 44, 44, 44,
        43, 43, 55, 54, 53, 51, 51, 51, 51, 50, 48, 46, 44, 44, 44, 44, 43, 43,
        55, 53, 53, 51, 51, 51, 51, 50, 48, 46, 44, 44, 44, 44, 43, 43, 53, 52,
        51, 50, 50, 50, 50, 49, 47, 45, 43, 43, 43, 43, 43, 43, 51, 50, 49, 49,
        49, 49, 49, 48, 47, 45, 43, 43, 43, 43, 43, 42, 49, 49, 48, 48, 48, 48,
        48, 47, 46, 45, 43, 43, 43, 43, 43, 42, 49, 49, 48, 48, 48, 48, 48, 47,
        46, 45, 43, 43, 43, 43, 43, 42,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 65, 65, 66, 66, 67, 67, 67, 67, 67, 66, 64, 62, 61,
        61, 61, 61, 59, 58, 56, 55, 55, 55, 55, 53, 51, 49, 49, 65, 65, 65, 65,
        65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 65, 63, 61, 59, 59, 59, 59, 58,
        56, 55, 54, 54, 54, 53, 52, 50, 49, 49, 66, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 64, 62, 60, 59, 59, 59, 59, 57, 55, 53, 53, 53,
        53, 53, 51, 49, 48, 48, 66, 66, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64,
        64, 64, 63, 61, 59, 58, 58, 58, 58, 55, 54, 52, 51, 51, 51, 51, 50, 49,
        48, 48, 66, 66, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 62, 60,
        59, 57, 57, 57, 57, 55, 53, 52, 51, 51, 51, 51, 50, 49, 48, 48, 66, 66,
        65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 62, 60, 59, 57, 57, 57,
        57, 55, 53, 52, 51, 51, 51, 51, 50, 49, 48, 48, 66, 66, 65, 65, 65, 65,
        65, 65, 64, 64, 64, 64, 64, 64, 62, 60, 59, 57, 57, 57, 57, 55, 53, 52,
        51, 51, 51, 51, 50, 49, 48, 48, 65, 65, 64, 64, 64, 64, 64, 64, 63, 63,
        62, 62, 62, 62, 61, 59, 58, 56, 56, 56, 56, 55, 53, 51, 50, 50, 50, 50,
        49, 48, 47, 47, 62, 61, 61, 60, 60, 60, 60, 60, 59, 59, 59, 59, 59, 59,
        58, 56, 55, 53, 53, 53, 53, 52, 50, 48, 48, 48, 48, 48, 47, 47, 46, 46,
        59, 59, 58, 57, 57, 57, 57, 57, 56, 55, 55, 55, 55, 55, 54, 53, 51, 51,
        51, 51, 51, 49, 48, 46, 46, 46, 46, 46, 45, 45, 45, 45, 55, 55, 54, 53,
        53, 53, 53, 53, 53, 52, 51, 51, 51, 51, 51, 49, 48, 48, 48, 48, 48, 46,
        45, 44, 44, 44, 44, 44, 43, 43, 43, 43, 55, 55, 54, 53, 53, 53, 53, 53,
        53, 52, 51, 51, 51, 51, 51, 49, 48, 48, 48, 48, 48, 46, 45, 44, 44, 44,
        44, 44, 43, 43, 43, 43, 55, 55, 54, 53, 53, 53, 53, 53, 53, 52, 51, 51,
        51, 51, 51, 49, 48, 48, 48, 48, 48, 46, 45, 44, 44, 44, 44, 44, 43, 43,
        43, 43, 55, 55, 54, 53, 53, 53, 53, 53, 53, 52, 51, 51, 51, 51, 51, 49,
        48, 48, 48, 48, 48, 46, 45, 44, 44, 44, 44, 44, 43, 43, 43, 43, 53, 53,
        52, 51, 51, 51, 51, 51, 51, 51, 50, 50, 50, 50, 49, 48, 48, 47, 47, 47,
        47, 46, 45, 44, 43, 43, 43, 43, 43, 43, 43, 43, 49, 49, 49, 48, 48, 48,
        48, 48, 48, 48, 48, 48, 48, 48, 48, 47, 46, 46, 46, 46, 46, 45, 44, 44,
        43, 43, 43, 43, 43, 42, 42, 42,
        /* Size 4x16 */
        65, 66, 59, 55, 65, 65, 58, 54, 65, 65, 57, 53, 65, 65, 57, 53, 66, 64,
        56, 53, 66, 64, 55, 51, 66, 64, 55, 51, 65, 62, 54, 51, 61, 59, 51, 48,
        59, 57, 51, 48, 59, 57, 51, 48, 56, 53, 48, 45, 54, 51, 46, 44, 54, 51,
        46, 44, 52, 50, 45, 43, 49, 48, 45, 43,
        /* Size 16x4 */
        65, 65, 65, 65, 66, 66, 66, 65, 61, 59, 59, 56, 54, 54, 52, 49, 66, 65,
        65, 65, 64, 64, 64, 62, 59, 57, 57, 53, 51, 51, 50, 48, 59, 58, 57, 57,
        56, 55, 55, 54, 51, 51, 51, 48, 46, 46, 45, 45, 55, 54, 53, 53, 53, 51,
        51, 51, 48, 48, 48, 45, 44, 44, 43, 43,
        /* Size 8x32 */
        64, 66, 66, 66, 62, 55, 55, 53, 65, 65, 66, 66, 61, 55, 55, 53, 65, 65,
        65, 65, 61, 54, 54, 52, 65, 65, 65, 65, 60, 53, 53, 51, 65, 65, 65, 65,
        60, 53, 53, 51, 65, 65, 65, 65, 60, 53, 53, 51, 65, 65, 65, 65, 60, 53,
        53, 51, 66, 65, 65, 65, 60, 53, 53, 51, 66, 65, 64, 64, 59, 53, 53, 51,
        67, 65, 64, 64, 59, 52, 52, 51, 67, 65, 64, 64, 59, 51, 51, 50, 67, 65,
        64, 64, 59, 51, 51, 50, 67, 65, 64, 64, 59, 51, 51, 50, 67, 65, 64, 64,
        59, 51, 51, 50, 66, 64, 62, 62, 58, 51, 51, 49, 64, 62, 60, 60, 56, 49,
        49, 48, 62, 60, 59, 59, 55, 48, 48, 48, 61, 59, 57, 57, 53, 48, 48, 47,
        61, 59, 57, 57, 53, 48, 48, 47, 61, 59, 57, 57, 53, 48, 48, 47, 61, 59,
        57, 57, 53, 48, 48, 47, 59, 57, 55, 55, 52, 46, 46, 46, 58, 55, 53, 53,
        50, 45, 45, 45, 56, 53, 52, 52, 48, 44, 44, 44, 55, 53, 51, 51, 48, 44,
        44, 43, 55, 53, 51, 51, 48, 44, 44, 43, 55, 53, 51, 51, 48, 44, 44, 43,
        55, 53, 51, 51, 48, 44, 44, 43, 53, 51, 50, 50, 47, 43, 43, 43, 51, 49,
        49, 49, 47, 43, 43, 43, 49, 48, 48, 48, 46, 43, 43, 43, 49, 48, 48, 48,
        46, 43, 43, 43,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 65, 65, 66, 66, 67, 67, 67, 67, 67, 66, 64, 62, 61,
        61, 61, 61, 59, 58, 56, 55, 55, 55, 55, 53, 51, 49, 49, 66, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 62, 60, 59, 59, 59, 59, 57,
        55, 53, 53, 53, 53, 53, 51, 49, 48, 48, 66, 66, 65, 65, 65, 65, 65, 65,
        64, 64, 64, 64, 64, 64, 62, 60, 59, 57, 57, 57, 57, 55, 53, 52, 51, 51,
        51, 51, 50, 49, 48, 48, 66, 66, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64,
        64, 64, 62, 60, 59, 57, 57, 57, 57, 55, 53, 52, 51, 51, 51, 51, 50, 49,
        48, 48, 62, 61, 61, 60, 60, 60, 60, 60, 59, 59, 59, 59, 59, 59, 58, 56,
        55, 53, 53, 53, 53, 52, 50, 48, 48, 48, 48, 48, 47, 47, 46, 46, 55, 55,
        54, 53, 53, 53, 53, 53, 53, 52, 51, 51, 51, 51, 51, 49, 48, 48, 48, 48,
        48, 46, 45, 44, 44, 44, 44, 44, 43, 43, 43, 43, 55, 55, 54, 53, 53, 53,
        53, 53, 53, 52, 51, 51, 51, 51, 51, 49, 48, 48, 48, 48, 48, 46, 45, 44,
        44, 44, 44, 44, 43, 43, 43, 43, 53, 53, 52, 51, 51, 51, 51, 51, 51, 51,
        50, 50, 50, 50, 49, 48, 48, 47, 47, 47, 47, 46, 45, 44, 43, 43, 43, 43,
        43, 43, 43, 43 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        64, 64, 64, 62,
#endif
        /* Size 4x4 */
        65, 65, 65, 64, 65, 64, 64, 63, 65, 64, 63, 63, 64, 63, 63, 61,
        /* Size 8x8 */
        65, 65, 65, 65, 65, 65, 64, 64, 65, 64, 64, 64, 64, 64, 64, 64, 65, 64,
        64, 64, 64, 64, 64, 64, 65, 64, 64, 64, 64, 64, 63, 63, 65, 64, 64, 64,
        63, 63, 63, 63, 65, 64, 64, 64, 63, 63, 63, 63, 64, 64, 64, 63, 63, 63,
        61, 61, 64, 64, 64, 63, 63, 63, 61, 61,
        /* Size 16x16 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 65, 65, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 63, 65, 65, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        65, 65, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 65,
        64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 65, 64, 64,
        64, 64, 64, 63, 63, 63, 63, 63, 62, 62, 62, 62, 65, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 65, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 62, 62, 61, 61, 61, 65, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 62, 62, 61, 61, 61, 65, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 62,
        62, 61, 61, 61,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63,
        63, 63, 63, 63, 63, 63, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 63, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 65,
        65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 65, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 65, 65, 65,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 65, 65, 65, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 62, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 65, 65,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62, 62, 65, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62,
        62, 62, 62, 62, 62, 62, 62, 61, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 61, 61, 61,
        61, 61, 61, 61, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 61, 61, 61, 61, 61, 61, 61,
        65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 62, 62, 62, 61, 61, 61, 61, 61, 61, 61, 65, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        62, 62, 62, 61, 61, 61, 61, 61, 61, 61, 65, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 61,
        61, 61, 61, 61, 61, 61, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 61, 61, 61, 61, 61,
        61, 61, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 62, 62, 62, 61, 61, 61, 61, 61, 61, 61, 61,
        /* Size 4x8 */
        65, 65, 65, 64, 65, 64, 64, 63, 65, 64, 64, 63, 65, 64, 64, 63, 65, 63,
        63, 63, 65, 63, 63, 62, 64, 63, 63, 61, 64, 63, 63, 61,
        /* Size 8x4 */
        65, 65, 65, 65, 65, 65, 64, 64, 65, 64, 64, 64, 63, 63, 63, 63, 65, 64,
        64, 64, 63, 63, 63, 63, 64, 63, 63, 63, 63, 62, 61, 61,
        /* Size 8x16 */
        64, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 64, 64, 65, 65,
        64, 64, 64, 64, 64, 63, 65, 64, 64, 64, 64, 64, 64, 63, 65, 64, 64, 64,
        64, 64, 64, 63, 65, 64, 64, 64, 64, 64, 64, 63, 65, 64, 64, 64, 64, 64,
        63, 63, 65, 64, 64, 63, 63, 63, 63, 63, 65, 64, 64, 63, 63, 63, 63, 63,
        65, 64, 64, 63, 63, 63, 63, 63, 65, 64, 64, 63, 63, 63, 63, 63, 65, 64,
        63, 63, 63, 63, 62, 62, 65, 64, 63, 63, 63, 63, 62, 61, 64, 64, 63, 63,
        63, 63, 62, 60, 64, 64, 63, 63, 63, 63, 62, 60, 64, 64, 63, 63, 63, 63,
        62, 60,
        /* Size 16x8 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 65, 65,
        65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 65, 65, 64, 64, 64, 64,
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 65, 64, 64, 64, 64, 64, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 65, 65, 64, 64, 64, 64, 64, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 65, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 62,
        62, 62, 62, 62, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 61, 60,
        60, 60,
        /* Size 16x32 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 65, 65, 65, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 63, 63, 63, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 63, 63, 63, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63,
        63, 63, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63,
        65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 65, 65,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 65, 65, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 63, 63, 63, 63, 63, 65, 65, 64, 64, 64, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 65, 65, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 62, 65, 65, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 62, 65, 65, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62,
        65, 65, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 65, 65,
        64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 65, 65, 64, 64,
        64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 65, 65, 64, 64, 64, 63,
        63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 65, 64, 64, 64, 63, 63, 63, 63,
        63, 63, 63, 63, 62, 62, 62, 61, 65, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 63, 62, 62, 61, 61, 65, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 62,
        62, 61, 61, 60, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 61,
        60, 60, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 61, 60, 60,
        64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 61, 60, 60, 64, 64,
        64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 61, 60, 60, 64, 64, 64, 63,
        63, 63, 63, 63, 63, 63, 63, 62, 62, 61, 60, 60, 64, 64, 64, 63, 63, 63,
        63, 63, 63, 63, 63, 62, 62, 61, 60, 60, 64, 64, 64, 63, 63, 63, 63, 63,
        63, 63, 63, 62, 61, 61, 60, 59,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63,
        63, 63, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 65,
        65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 65, 65, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 65, 65, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62,
        62, 62, 62, 62, 62, 62, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 62,
        62, 61, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 62, 62, 62, 61, 61, 61, 61, 61, 61, 61, 61, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 62, 62, 61, 61, 60, 60, 60, 60, 60, 60, 60, 64, 64, 64, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 62, 61, 61,
        60, 60, 60, 60, 60, 60, 60, 59,
        /* Size 4x16 */
        65, 65, 65, 64, 65, 65, 65, 64, 65, 64, 64, 64, 65, 64, 64, 63, 65, 64,
        64, 63, 65, 64, 64, 63, 65, 64, 64, 63, 65, 63, 63, 63, 65, 63, 63, 63,
        65, 63, 63, 63, 65, 63, 63, 63, 64, 63, 63, 62, 64, 63, 63, 61, 64, 63,
        63, 61, 64, 63, 63, 61, 64, 63, 63, 61,
        /* Size 16x4 */
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 65, 65,
        64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 65, 64, 64,
        64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 63, 63, 63,
        63, 63, 63, 63, 63, 62, 61, 61, 61, 61,
        /* Size 8x32 */
        64, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 64, 64, 65, 65,
        65, 65, 65, 65, 64, 64, 65, 65, 64, 64, 64, 64, 64, 64, 65, 65, 64, 64,
        64, 64, 64, 63, 65, 64, 64, 64, 64, 64, 64, 63, 65, 64, 64, 64, 64, 64,
        64, 63, 65, 64, 64, 64, 64, 64, 64, 63, 65, 64, 64, 64, 64, 64, 64, 63,
        65, 64, 64, 64, 64, 64, 64, 63, 65, 64, 64, 64, 64, 64, 64, 63, 65, 64,
        64, 64, 64, 64, 64, 63, 65, 64, 64, 64, 64, 64, 63, 63, 65, 64, 64, 64,
        64, 64, 63, 63, 65, 64, 64, 63, 63, 63, 63, 63, 65, 64, 64, 63, 63, 63,
        63, 63, 65, 64, 64, 63, 63, 63, 63, 63, 65, 64, 64, 63, 63, 63, 63, 63,
        65, 64, 64, 63, 63, 63, 63, 63, 65, 64, 64, 63, 63, 63, 63, 63, 65, 64,
        64, 63, 63, 63, 63, 63, 65, 64, 64, 63, 63, 63, 63, 62, 65, 64, 63, 63,
        63, 63, 62, 62, 65, 64, 63, 63, 63, 63, 62, 61, 65, 64, 63, 63, 63, 63,
        62, 61, 64, 64, 63, 63, 63, 63, 62, 60, 64, 64, 63, 63, 63, 63, 62, 60,
        64, 64, 63, 63, 63, 63, 62, 60, 64, 64, 63, 63, 63, 63, 62, 60, 64, 64,
        63, 63, 63, 63, 62, 60, 64, 64, 63, 63, 63, 63, 62, 60, 64, 64, 63, 63,
        63, 63, 61, 60,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
        63, 63, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 65,
        65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 65, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62,
        62, 62, 62, 62, 62, 62, 62, 61, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63,
        63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 61, 61, 60, 60, 60,
        60, 60, 60, 60 },
      { /* Chroma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        65, 62, 62, 57,
#endif
        /* Size 4x4 */
        65, 66, 66, 60, 66, 65, 65, 59, 66, 65, 64, 58, 60, 59, 58, 53,
        /* Size 8x8 */
        65, 66, 66, 66, 67, 66, 61, 61, 66, 65, 65, 65, 66, 64, 59, 59, 66, 65,
        65, 65, 66, 64, 59, 59, 66, 65, 65, 65, 65, 64, 59, 59, 67, 66, 66, 65,
        64, 63, 58, 58, 66, 64, 64, 64, 63, 62, 57, 57, 61, 59, 59, 59, 58, 57,
        53, 53, 61, 59, 59, 59, 58, 57, 53, 53,
        /* Size 16x16 */
        64, 65, 65, 66, 66, 66, 66, 67, 67, 67, 67, 65, 62, 61, 61, 61, 65, 65,
        65, 65, 65, 65, 66, 66, 67, 67, 67, 64, 62, 60, 60, 60, 65, 65, 65, 65,
        65, 65, 66, 66, 66, 66, 66, 63, 61, 59, 59, 59, 66, 65, 65, 65, 65, 65,
        65, 65, 66, 66, 66, 63, 60, 59, 59, 59, 66, 65, 65, 65, 65, 65, 65, 65,
        66, 66, 66, 63, 60, 59, 59, 59, 66, 65, 65, 65, 65, 65, 65, 65, 66, 66,
        66, 63, 60, 59, 59, 59, 66, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 62,
        60, 59, 59, 59, 67, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 62, 59, 58,
        58, 58, 67, 67, 66, 66, 66, 66, 65, 65, 64, 64, 64, 62, 59, 58, 58, 58,
        67, 67, 66, 66, 66, 66, 65, 65, 64, 64, 64, 62, 59, 58, 58, 58, 67, 67,
        66, 66, 66, 66, 65, 65, 64, 64, 64, 62, 59, 58, 58, 58, 65, 64, 63, 63,
        63, 63, 62, 62, 62, 62, 62, 59, 57, 55, 55, 55, 62, 62, 61, 60, 60, 60,
        60, 59, 59, 59, 59, 57, 55, 53, 53, 53, 61, 60, 59, 59, 59, 59, 59, 58,
        58, 58, 58, 55, 53, 53, 53, 53, 61, 60, 59, 59, 59, 59, 59, 58, 58, 58,
        58, 55, 53, 53, 53, 53, 61, 60, 59, 59, 59, 59, 59, 58, 58, 58, 58, 55,
        53, 53, 53, 53,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 67, 67, 67, 67, 67,
        67, 67, 67, 66, 65, 64, 62, 61, 61, 61, 61, 61, 61, 60, 65, 65, 65, 65,
        65, 66, 66, 66, 66, 66, 66, 66, 66, 66, 67, 67, 67, 67, 67, 67, 67, 66,
        65, 63, 62, 61, 61, 61, 61, 61, 61, 59, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 66, 66, 66, 66, 67, 67, 67, 67, 67, 67, 65, 64, 63, 62, 60,
        60, 60, 60, 60, 60, 59, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66,
        66, 66, 66, 66, 66, 66, 66, 66, 66, 65, 64, 62, 61, 60, 60, 60, 60, 60,
        60, 59, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66,
        66, 66, 66, 66, 66, 65, 63, 62, 61, 59, 59, 59, 59, 59, 59, 59, 66, 66,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66,
        66, 64, 63, 62, 60, 59, 59, 59, 59, 59, 59, 58, 66, 66, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 64, 63, 62,
        60, 59, 59, 59, 59, 59, 59, 58, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 64, 63, 62, 60, 59, 59, 59,
        59, 59, 59, 58, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 66, 66, 66, 66, 66, 66, 64, 63, 62, 60, 59, 59, 59, 59, 59, 59, 58,
        66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66,
        66, 66, 66, 64, 63, 62, 60, 59, 59, 59, 59, 59, 59, 58, 66, 66, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 64,
        63, 62, 60, 59, 59, 59, 59, 59, 59, 58, 66, 66, 66, 66, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 63, 61, 60, 59,
        59, 59, 59, 59, 59, 58, 66, 66, 66, 66, 66, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 62, 61, 60, 59, 59, 59, 59, 59,
        59, 58, 67, 66, 66, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 64, 62, 61, 59, 59, 59, 59, 59, 59, 59, 57, 67, 67,
        66, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 63, 62, 60, 59, 58, 58, 58, 58, 58, 58, 57, 67, 67, 67, 66, 66, 66,
        66, 66, 66, 66, 66, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 63, 62, 60,
        59, 58, 58, 58, 58, 58, 58, 56, 67, 67, 67, 66, 66, 66, 66, 66, 66, 66,
        66, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 63, 62, 60, 59, 58, 58, 58,
        58, 58, 58, 56, 67, 67, 67, 66, 66, 66, 66, 66, 66, 66, 66, 65, 65, 65,
        65, 64, 64, 64, 64, 64, 64, 63, 62, 60, 59, 58, 58, 58, 58, 58, 58, 56,
        67, 67, 67, 66, 66, 66, 66, 66, 66, 66, 66, 65, 65, 65, 65, 64, 64, 64,
        64, 64, 64, 63, 62, 60, 59, 58, 58, 58, 58, 58, 58, 56, 67, 67, 67, 66,
        66, 66, 66, 66, 66, 66, 66, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 63,
        62, 60, 59, 58, 58, 58, 58, 58, 58, 56, 67, 67, 67, 66, 66, 66, 66, 66,
        66, 66, 66, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 63, 62, 60, 59, 58,
        58, 58, 58, 58, 58, 56, 66, 66, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 63, 63, 63, 63, 63, 62, 60, 59, 58, 57, 57, 57, 57, 57,
        57, 55, 65, 65, 64, 64, 63, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 62,
        62, 62, 62, 62, 62, 60, 59, 58, 57, 55, 55, 55, 55, 55, 55, 55, 64, 63,
        63, 62, 62, 62, 62, 62, 62, 62, 62, 61, 61, 61, 60, 60, 60, 60, 60, 60,
        60, 59, 58, 57, 55, 55, 55, 55, 55, 55, 55, 53, 62, 62, 62, 61, 61, 60,
        60, 60, 60, 60, 60, 60, 60, 59, 59, 59, 59, 59, 59, 59, 59, 58, 57, 55,
        55, 53, 53, 53, 53, 53, 53, 53, 61, 61, 60, 60, 59, 59, 59, 59, 59, 59,
        59, 59, 59, 59, 58, 58, 58, 58, 58, 58, 58, 57, 55, 55, 53, 53, 53, 53,
        53, 53, 53, 51, 61, 61, 60, 60, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59,
        58, 58, 58, 58, 58, 58, 58, 57, 55, 55, 53, 53, 53, 53, 53, 53, 53, 51,
        61, 61, 60, 60, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 58, 58, 58, 58,
        58, 58, 58, 57, 55, 55, 53, 53, 53, 53, 53, 53, 53, 51, 61, 61, 60, 60,
        59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 58, 58, 58, 58, 58, 58, 58, 57,
        55, 55, 53, 53, 53, 53, 53, 53, 53, 51, 61, 61, 60, 60, 59, 59, 59, 59,
        59, 59, 59, 59, 59, 59, 58, 58, 58, 58, 58, 58, 58, 57, 55, 55, 53, 53,
        53, 53, 53, 53, 53, 51, 61, 61, 60, 60, 59, 59, 59, 59, 59, 59, 59, 59,
        59, 59, 58, 58, 58, 58, 58, 58, 58, 57, 55, 55, 53, 53, 53, 53, 53, 53,
        53, 51, 60, 59, 59, 59, 59, 58, 58, 58, 58, 58, 58, 58, 58, 57, 57, 56,
        56, 56, 56, 56, 56, 55, 55, 53, 53, 51, 51, 51, 51, 51, 51, 51,
        /* Size 4x8 */
        65, 66, 66, 59, 65, 65, 65, 58, 65, 65, 65, 58, 66, 64, 64, 57, 66, 64,
        64, 56, 65, 62, 62, 55, 60, 57, 57, 51, 60, 57, 57, 51,
        /* Size 8x4 */
        65, 65, 65, 66, 66, 65, 60, 60, 66, 65, 65, 64, 64, 62, 57, 57, 66, 65,
        65, 64, 64, 62, 57, 57, 59, 58, 58, 57, 56, 55, 51, 51,
        /* Size 8x16 */
        64, 65, 66, 66, 66, 66, 62, 58, 65, 65, 66, 66, 66, 66, 61, 57, 65, 65,
        65, 65, 65, 65, 60, 56, 65, 65, 65, 65, 65, 65, 60, 55, 65, 65, 65, 65,
        65, 65, 60, 55, 65, 65, 65, 65, 65, 65, 60, 55, 66, 65, 65, 64, 64, 64,
        59, 55, 67, 66, 65, 64, 64, 64, 59, 54, 67, 66, 64, 64, 64, 64, 59, 54,
        67, 66, 64, 64, 64, 64, 59, 54, 67, 66, 64, 64, 64, 64, 59, 54, 65, 63,
        62, 61, 61, 61, 57, 52, 62, 60, 59, 59, 59, 59, 55, 51, 61, 59, 58, 57,
        57, 57, 53, 49, 61, 59, 58, 57, 57, 57, 53, 49, 61, 59, 58, 57, 57, 57,
        53, 49,
        /* Size 16x8 */
        64, 65, 65, 65, 65, 65, 66, 67, 67, 67, 67, 65, 62, 61, 61, 61, 65, 65,
        65, 65, 65, 65, 65, 66, 66, 66, 66, 63, 60, 59, 59, 59, 66, 66, 65, 65,
        65, 65, 65, 65, 64, 64, 64, 62, 59, 58, 58, 58, 66, 66, 65, 65, 65, 65,
        64, 64, 64, 64, 64, 61, 59, 57, 57, 57, 66, 66, 65, 65, 65, 65, 64, 64,
        64, 64, 64, 61, 59, 57, 57, 57, 66, 66, 65, 65, 65, 65, 64, 64, 64, 64,
        64, 61, 59, 57, 57, 57, 62, 61, 60, 60, 60, 60, 59, 59, 59, 59, 59, 57,
        55, 53, 53, 53, 58, 57, 56, 55, 55, 55, 55, 54, 54, 54, 54, 52, 51, 49,
        49, 49,
        /* Size 16x32 */
        64, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 64, 62, 60, 58, 55, 65, 65,
        65, 66, 66, 66, 66, 66, 66, 66, 66, 64, 62, 59, 58, 55, 65, 65, 65, 65,
        66, 66, 66, 66, 66, 66, 66, 64, 61, 59, 57, 55, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 63, 61, 59, 56, 54, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 63, 60, 59, 56, 54, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 62, 60, 58, 55, 53, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 62,
        60, 58, 55, 53, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 62, 60, 58,
        55, 53, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 62, 60, 58, 55, 53,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 62, 60, 58, 55, 53, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 62, 60, 58, 55, 53, 66, 66, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 62, 60, 58, 55, 53, 66, 66, 65, 65, 65, 64,
        64, 64, 64, 64, 64, 62, 59, 57, 55, 53, 66, 66, 66, 65, 65, 64, 64, 64,
        64, 64, 64, 62, 59, 57, 55, 52, 67, 66, 66, 65, 65, 64, 64, 64, 64, 64,
        64, 61, 59, 57, 54, 52, 67, 66, 66, 65, 64, 64, 64, 64, 64, 64, 64, 61,
        59, 56, 54, 51, 67, 66, 66, 65, 64, 64, 64, 64, 64, 64, 64, 61, 59, 56,
        54, 51, 67, 66, 66, 65, 64, 64, 64, 64, 64, 64, 64, 61, 59, 56, 54, 51,
        67, 66, 66, 65, 64, 64, 64, 64, 64, 64, 64, 61, 59, 56, 54, 51, 67, 66,
        66, 65, 64, 64, 64, 64, 64, 64, 64, 61, 59, 56, 54, 51, 67, 66, 66, 65,
        64, 64, 64, 64, 64, 64, 64, 61, 59, 56, 54, 51, 66, 65, 65, 64, 63, 62,
        62, 62, 62, 62, 62, 60, 58, 55, 53, 51, 65, 64, 63, 62, 62, 61, 61, 61,
        61, 61, 61, 59, 57, 55, 52, 50, 63, 63, 62, 61, 60, 60, 60, 60, 60, 60,
        60, 58, 55, 53, 51, 49, 62, 61, 60, 60, 59, 59, 59, 59, 59, 59, 59, 56,
        55, 53, 51, 48, 61, 60, 59, 59, 58, 57, 57, 57, 57, 57, 57, 55, 53, 51,
        49, 48, 61, 60, 59, 59, 58, 57, 57, 57, 57, 57, 57, 55, 53, 51, 49, 48,
        61, 60, 59, 59, 58, 57, 57, 57, 57, 57, 57, 55, 53, 51, 49, 48, 61, 60,
        59, 59, 58, 57, 57, 57, 57, 57, 57, 55, 53, 51, 49, 48, 61, 60, 59, 59,
        58, 57, 57, 57, 57, 57, 57, 55, 53, 51, 49, 48, 61, 60, 59, 59, 58, 57,
        57, 57, 57, 57, 57, 55, 53, 51, 49, 48, 59, 59, 59, 58, 57, 56, 56, 56,
        56, 56, 56, 54, 53, 51, 49, 47,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 67, 67, 67, 67,
        67, 67, 67, 66, 65, 63, 62, 61, 61, 61, 61, 61, 61, 59, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 65,
        64, 63, 61, 60, 60, 60, 60, 60, 60, 59, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 65, 63, 62, 60, 59,
        59, 59, 59, 59, 59, 59, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 62, 61, 60, 59, 59, 59, 59, 59,
        59, 58, 66, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64,
        64, 64, 64, 64, 64, 63, 62, 60, 59, 58, 58, 58, 58, 58, 58, 57, 66, 66,
        66, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 62, 61, 60, 59, 57, 57, 57, 57, 57, 57, 56, 66, 66, 66, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62, 61, 60,
        59, 57, 57, 57, 57, 57, 57, 56, 66, 66, 66, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62, 61, 60, 59, 57, 57, 57,
        57, 57, 57, 56, 66, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 62, 61, 60, 59, 57, 57, 57, 57, 57, 57, 56,
        66, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 62, 61, 60, 59, 57, 57, 57, 57, 57, 57, 56, 66, 66, 66, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62,
        61, 60, 59, 57, 57, 57, 57, 57, 57, 56, 64, 64, 64, 63, 63, 62, 62, 62,
        62, 62, 62, 62, 62, 62, 61, 61, 61, 61, 61, 61, 61, 60, 59, 58, 56, 55,
        55, 55, 55, 55, 55, 54, 62, 62, 61, 61, 60, 60, 60, 60, 60, 60, 60, 60,
        59, 59, 59, 59, 59, 59, 59, 59, 59, 58, 57, 55, 55, 53, 53, 53, 53, 53,
        53, 53, 60, 59, 59, 59, 59, 58, 58, 58, 58, 58, 58, 58, 57, 57, 57, 56,
        56, 56, 56, 56, 56, 55, 55, 53, 53, 51, 51, 51, 51, 51, 51, 51, 58, 58,
        57, 56, 56, 55, 55, 55, 55, 55, 55, 55, 55, 55, 54, 54, 54, 54, 54, 54,
        54, 53, 52, 51, 51, 49, 49, 49, 49, 49, 49, 49, 55, 55, 55, 54, 54, 53,
        53, 53, 53, 53, 53, 53, 53, 52, 52, 51, 51, 51, 51, 51, 51, 51, 50, 49,
        48, 48, 48, 48, 48, 48, 48, 47,
        /* Size 4x16 */
        65, 66, 66, 60, 65, 66, 66, 59, 65, 65, 65, 59, 65, 65, 65, 58, 65, 65,
        65, 58, 65, 65, 65, 58, 66, 64, 64, 57, 66, 64, 64, 57, 66, 64, 64, 56,
        66, 64, 64, 56, 66, 64, 64, 56, 64, 61, 61, 55, 61, 59, 59, 53, 60, 57,
        57, 51, 60, 57, 57, 51, 60, 57, 57, 51,
        /* Size 16x4 */
        65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 64, 61, 60, 60, 60, 66, 66,
        65, 65, 65, 65, 64, 64, 64, 64, 64, 61, 59, 57, 57, 57, 66, 66, 65, 65,
        65, 65, 64, 64, 64, 64, 64, 61, 59, 57, 57, 57, 60, 59, 59, 58, 58, 58,
        57, 57, 56, 56, 56, 55, 53, 51, 51, 51,
        /* Size 8x32 */
        64, 65, 66, 66, 66, 66, 62, 58, 65, 65, 66, 66, 66, 66, 62, 58, 65, 65,
        66, 66, 66, 66, 61, 57, 65, 65, 65, 65, 65, 65, 61, 56, 65, 65, 65, 65,
        65, 65, 60, 56, 65, 65, 65, 65, 65, 65, 60, 55, 65, 65, 65, 65, 65, 65,
        60, 55, 65, 65, 65, 65, 65, 65, 60, 55, 65, 65, 65, 65, 65, 65, 60, 55,
        65, 65, 65, 65, 65, 65, 60, 55, 65, 65, 65, 65, 65, 65, 60, 55, 66, 65,
        65, 65, 65, 65, 60, 55, 66, 65, 65, 64, 64, 64, 59, 55, 66, 66, 65, 64,
        64, 64, 59, 55, 67, 66, 65, 64, 64, 64, 59, 54, 67, 66, 64, 64, 64, 64,
        59, 54, 67, 66, 64, 64, 64, 64, 59, 54, 67, 66, 64, 64, 64, 64, 59, 54,
        67, 66, 64, 64, 64, 64, 59, 54, 67, 66, 64, 64, 64, 64, 59, 54, 67, 66,
        64, 64, 64, 64, 59, 54, 66, 65, 63, 62, 62, 62, 58, 53, 65, 63, 62, 61,
        61, 61, 57, 52, 63, 62, 60, 60, 60, 60, 55, 51, 62, 60, 59, 59, 59, 59,
        55, 51, 61, 59, 58, 57, 57, 57, 53, 49, 61, 59, 58, 57, 57, 57, 53, 49,
        61, 59, 58, 57, 57, 57, 53, 49, 61, 59, 58, 57, 57, 57, 53, 49, 61, 59,
        58, 57, 57, 57, 53, 49, 61, 59, 58, 57, 57, 57, 53, 49, 59, 59, 57, 56,
        56, 56, 53, 49,
        /* Size 32x8 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 67, 67, 67, 67,
        67, 67, 67, 66, 65, 63, 62, 61, 61, 61, 61, 61, 61, 59, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 65,
        63, 62, 60, 59, 59, 59, 59, 59, 59, 59, 66, 66, 66, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 63, 62, 60, 59, 58,
        58, 58, 58, 58, 58, 57, 66, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 62, 61, 60, 59, 57, 57, 57, 57, 57,
        57, 56, 66, 66, 66, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 62, 61, 60, 59, 57, 57, 57, 57, 57, 57, 56, 66, 66,
        66, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 62, 61, 60, 59, 57, 57, 57, 57, 57, 57, 56, 62, 62, 61, 61, 60, 60,
        60, 60, 60, 60, 60, 60, 59, 59, 59, 59, 59, 59, 59, 59, 59, 58, 57, 55,
        55, 53, 53, 53, 53, 53, 53, 53, 58, 58, 57, 56, 56, 55, 55, 55, 55, 55,
        55, 55, 55, 55, 54, 54, 54, 54, 54, 54, 54, 53, 52, 51, 51, 49, 49, 49,
        49, 49, 49, 49 },
  },
  {
      { /* Luma */
#if CONFIG_CB4X4
        /* Size 2x2 */
        65, 65, 65, 64,
#endif
        /* Size 4x4 */
        65, 65, 65, 65, 65, 64, 64, 64, 65, 64, 64, 64, 65, 64, 64, 64,
        /* Size 8x8 */
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 64, 64, 64, 64, 64, 65, 65, 64, 64, 64, 64, 64, 64, 65, 65, 64, 64,
        64, 64, 64, 64, 65, 65, 64, 64, 64, 64, 64, 64, 65, 65, 64, 64, 64, 64,
        64, 64, 65, 65, 64, 64, 64, 64, 64, 63,
        /* Size 16x16 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 63, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63,
        /* Size 32x32 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65,
        65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 65, 65, 65, 65, 65, 65, 65, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 63, 63, 63, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63,
        63, 63, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63,
        /* Size 4x8 */
        65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 65, 64, 64, 64, 65, 64,
        64, 64, 65, 64, 64, 64, 65, 64, 64, 64, 65, 64, 64, 63,
        /* Size 8x4 */
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 65, 65,
        64, 64, 64, 64, 64, 64, 65, 65, 64, 64, 64, 64, 64, 63,
        /* Size 8x16 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 65, 65, 65, 64,
        64, 64, 64, 64, 65, 65, 64, 64, 64, 64, 64, 64, 65, 65, 64, 64, 64, 64,
        64, 64, 65, 65, 64, 64, 64, 64, 64, 64, 65, 65, 64, 64, 64, 64, 64, 64,
        65, 65, 64, 64, 64, 64, 64, 64, 65, 65, 64, 64, 64, 64, 64, 64, 65, 65,
        64, 64, 64, 64, 64, 64, 65, 65, 64, 64, 64, 64, 64, 64, 65, 65, 64, 64,
        64, 64, 64, 64, 65, 65, 64, 64, 64, 63, 63, 63, 65, 65, 64, 64, 64, 63,
        63, 63,
        /* Size 16x8 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 63, 63, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 63, 63, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        63, 63,
        /* Size 16x32 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 64, 64, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
        65, 65, 65, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65,
        65, 64, 64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 65, 65, 65, 64,
        64, 64, 64, 64, 64, 64, 63, 63, 63, 63, 63, 63, 65, 65, 65, 64, 64, 64,
        64, 64, 64, 63, 63, 63, 63, 63, 63, 63, 65, 65, 65, 64, 64, 64, 64, 64,
        64, 63, 63, 63, 63, 63, 63, 63,
        /* Size 32x16 */
        64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
        65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64,
        64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 64, 64, 64, 64, 64, 64,
        64, 64, 6