/* Functions related to building and playing with classes.
   Copyright (C) 1987 Free Software Foundation, Inc.
   Contributed by Michael Tiemann (tiemann@mcc.com)

This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */


/* High-level class interface. */

#include "config.h"
#include "tree.h"
#include "cplus-tree.h"
#include "flags.h"
#include "rtl.h"
#include "assert.h"
#include <stdio.h>

#define OVERLOAD_MAX_LEN 1024

#define NULL 0
#define MIN(X,Y) ((X) < (Y) ? (X) : (Y))
#define MAX(X,Y) ((X) > (Y) ? (X) : (Y))

/* some statistics gathering help.  */
static int n_convert_harshness, n_compute_conversion_costs, n_build_method_call;
static int n_inner_fields_searched;

/* Compute the ease with which a conversion can be performed
   between an expected and the given type.  */
static int convert_harshness ();

/* in decl.c.  */
extern tree lookup_tag_current_binding_level ();

/* in method.c.  */
extern void do_inline_function_hair ();

/* Way of stacking class names.  */
static tree *current_class_base, *current_class_stack;
static int current_class_stacksize;

tree current_class_decl, C_C_D;	/* PARM_DECL: the class instance variable */
tree current_vtable_decl;

/* The following two can be derived from the previous one */
tree current_class_name;	/* IDENTIFIER_NODE: name of current class */
tree current_class_type;	/* _TYPE: the type of the current class */
tree prev_class_type;		/* _TYPE: the previous type that was a class */

static tree get_vtable_name (), get_vfield_name ();
tree the_null_vtable_entry;

/* Way of stacking langauge names.  */
static tree *current_lang_base, *current_lang_stack;
static int current_lang_stacksize;

/* Names of languages we recognize.  */
tree lang_name_c, lang_name_cplusplus;
tree current_lang_name;

tree minus_one_node;

#if 0
/* Make sure that the tag NAME is defined *in the current binding level*
   at least as a forward reference.
   CODE says which kind of tag NAME ought to be.

   Not used for C++.  Not maintained.  */

tree
start_struct (code, name)
     enum tree_code code;
     tree name;
{
  /* If there is already a tag defined at this binding level
     (as a forward reference), just return it.  */
  register tree ref = 0;

  if (name != 0)
    ref = lookup_tag (code, name, current_binding_level, 1);
  if (ref && TREE_CODE (ref) == code)
    {
      if (TYPE_FIELDS (ref))
	error ((code == UNION_TYPE ? "redefinition of `union %s'"
		: "redefinition of `struct %s'"),
	       IDENTIFIER_POINTER (name));

      return ref;
    }

  /* Otherwise create a forward-reference just so the tag is in scope.  */

  ref = make_lang_type (code);
  /* Must re-synch this with xref_tag if you are going to use it.  */
  assert (0);
  pushtag (name, ref);
  return ref;
}
#endif

/* Virtual baseclass things.  */
tree
build_vbase_pointer (exp, type)
     tree exp, type;
{
  char *name;

  if (exp == error_mark_node)
    return error_mark_node;

  name = (char *) alloca (TYPE_NAME_LENGTH (type) + sizeof (VBASE_NAME) + 1);
  sprintf (name, VBASE_NAME_FORMAT, TYPE_NAME_STRING (type));
  return build_component_ref (exp, get_identifier (name), 0, 0);
}

/* Virtual function things.  */

/* Virtual functions to be dealt with after laying out our
   virtual base classes (only if the type has any).  */
static tree pending_hard_virtuals;

/* The names of the entries in the virtual table structure.  */
static tree delta_name, pfn_name;

/* Build an entry in the virtual function table.
   Note that the index in the virtual function table
   is always 0.  */
tree
build_vtable_entry (delta, pfn)
     tree delta, pfn;
{
  tree elems = tree_cons (NULL_TREE, delta,
			  tree_cons (NULL_TREE, integer_zero_node,
				     build_tree_list (NULL_TREE, pfn)));
  tree entry = build (CONSTRUCTOR, vtable_entry_type, NULL_TREE, elems);
  TREE_LITERAL (entry) = 1;
  TREE_STATIC (entry) = 1;
  TREE_READONLY (entry) = 1;
  return entry;
}

/* Given an object INSTANCE, return an expression which yields
   the virtual function corresponding to INDEX.  There are many special
   cases for INSTANCE which we take care of here, mainly to avoid
   creating extra tree nodes when we don't have to.  */
tree
build_vfn_ref (ptr_to_instptr, instance, index)
     tree *ptr_to_instptr, instance;
     tree index;
{
  tree vtbl, aref;
  tree basetype = TREE_TYPE (instance);

  if (TREE_CODE (basetype) == REFERENCE_TYPE)
    basetype = TREE_TYPE (basetype);

  if (instance == C_C_D)
    {
      if (flag_this_is_variable)
	vtbl = build_indirect_ref (build_vfield_ref (instance, basetype));
      else
	vtbl = current_vtable_decl;
    }
  else
    {
      if (optimize)
	{
	  /* Try to figure out what a reference refers to, and
	     access its virtual function table directly.  */
	  tree ref = 0;

	  if (TREE_CODE (instance) == INDIRECT_REF
	      && TREE_CODE (TREE_TYPE (TREE_OPERAND (instance, 0))) == REFERENCE_TYPE)
	    ref = TREE_OPERAND (instance, 0);
	  else if (TREE_CODE (TREE_TYPE (instance)) == REFERENCE_TYPE)
	    ref = instance;

	  if (ref && TREE_CODE (ref) == VAR_DECL
	      && DECL_INITIAL (ref)
	      && DECL_INITIAL (ref) != error_mark_node)
	    {
	      tree init = DECL_INITIAL (ref);

	      while (TREE_CODE (init) == NOP_EXPR
		     || TREE_CODE (init) == REFERENCE_EXPR)
		init = TREE_OPERAND (init, 0);
	      if (TREE_CODE (init) == ADDR_EXPR)
		{
		  init = TREE_OPERAND (init, 0);
		  if (IS_AGGR_TYPE (TREE_TYPE (init))
		      && (TREE_CODE (init) == PARM_DECL
			  || TREE_CODE (init) == VAR_DECL))
		    instance = init;
		}
	    }
	}

      if (IS_AGGR_TYPE (instance)
	  && (TREE_CODE (instance) == RESULT_DECL
	      || TREE_CODE (instance) == PARM_DECL
	      || TREE_CODE (instance) == VAR_DECL))
	vtbl = CLASSTYPE_VTABLE (basetype);
      else
	vtbl = build_indirect_ref (build_vfield_ref (instance, basetype), 0);
    }
  aref = save_expr (build_array_ref (vtbl, index));
  *ptr_to_instptr = build (PLUS_EXPR, TREE_TYPE (*ptr_to_instptr),
			   build (NOP_EXPR, ptr_type_node, *ptr_to_instptr),
			   convert (integer_type_node, build_component_ref (aref, delta_name, 0, 0)));
  return build_component_ref (aref, pfn_name, 0, 0);
}

/* NOTE!!  BASETYPE and TYPE may be aliased.  */
static tree
build_vtable (basetype, type)
     tree basetype, type;
{
  tree name = get_vtable_name (type);
  /* This is just a guess at the type, since we don't know it yet.  */
  tree decl;

  if (basetype)
    decl = build_decl (VAR_DECL, name, TREE_TYPE (CLASSTYPE_VTABLE (basetype)));
  else
    decl = build_decl (VAR_DECL, name, void_type_node);

  if (write_virtuals == 1)
    TREE_PUBLIC (decl) = 1;
  else if (write_virtuals == -1)
    TREE_EXTERNAL (decl) = 1;
  IDENTIFIER_GLOBAL_VALUE (name) = decl = pushdecl_top_level (decl);
  CLASSTYPE_VTABLE (type) = decl;
  TREE_STATIC (decl) = 1;
  DECL_ALIGN (decl) = MAX (TYPE_ALIGN (double_type_node),
			   DECL_ALIGN (decl));

  if (basetype)
    CLASSTYPE_VIRTUALS (type) = copy_list (CLASSTYPE_VIRTUALS (basetype));
  if (basetype && write_virtuals >= 0)
    DECL_VIRTUAL_P (decl) = 1;
  /* Remember which class this vtable is really for.  */
  DECL_VPARENT (decl) = type;
  CLASSTYPE_MARKED3 (type) = 1;
  CLASSTYPE_MARKED4 (type) = 1;
  return decl;
}

/* Give TYPE a new virtual function table which is initialized
   with a skeleton-copy of its original initialization.  The only
   entry that changes is the `delta' entry, so we can really
   share a lot of structure.

   FOR_TYPE is the derived type which caused this table to
   be needed.  */
static void
prepare_fresh_vtable (type, for_type)
     tree type, for_type;
{
  tree orig_decl = CLASSTYPE_VTABLE (type);
  tree name = build_type_pathname (VTABLE_NAME_FORMAT,
				   CLASSTYPE_THIS_VARIANT (type),
				   for_type);
  tree new_decl = build_decl (VAR_DECL, name, TREE_TYPE (orig_decl));

  /* Remember which class this vtable is really for.  */
  DECL_VPARENT (new_decl) = type;

  /* `new_decl' has the right PUBLIC and EXTERNAL bits set.  */
  TREE_STATIC (new_decl) = 1;
  CLASSTYPE_VTABLE (type) = pushdecl_top_level (new_decl);
  DECL_VIRTUAL_P (new_decl) = 1;
  DECL_ALIGN (new_decl) = DECL_ALIGN (orig_decl);

  /* Make fresh virtual list, so we can smash it later.  */
  CLASSTYPE_VIRTUALS (type) = copy_list (CLASSTYPE_VIRTUALS (type));

  CLASSTYPE_MARKED3 (type) = 1;
  CLASSTYPE_MARKED4 (type) = 1;
}

static tree this_fndecl, this_base_fndecl;
static tree this_vcontext, this_context;
static tree this_offset;
static tree vtable_fnaddr, static_vfield_name;

static void
prepare_to_modify_vtable_entry (x, base_fndecl, vfn)
     tree x, base_fndecl, vfn;
{
  tree base_offset, offset;
  tree vcontext, context;

  this_fndecl = x;
  this_base_fndecl = base_fndecl;
  vtable_fnaddr = vfn;

  DECL_VCONTEXT (x) = DECL_VCONTEXT (this_base_fndecl);
  vcontext = get_base_type (DECL_VCONTEXT (this_base_fndecl), current_class_type, 0);
  context = get_base_type (DECL_CONTEXT (this_fndecl), current_class_type, 0);

  if (vcontext == NULL_TREE)
    vcontext = current_class_type;
  if (context == NULL_TREE)
    context = current_class_type;

  if (! TREE_VIA_VIRTUAL (vcontext))
    base_offset = CLASSTYPE_OFFSET (vcontext);
  else
    {
      tree offset_info = value_member (CLASSTYPE_THIS_VARIANT (vcontext),
				       CLASSTYPE_VBASECLASSES (current_class_type));
      base_offset = TREE_PURPOSE (offset_info);
    }
  if (! TREE_VIA_VIRTUAL (context))
    offset = CLASSTYPE_OFFSET (context);
  else
    {
      tree offset_info = value_member (CLASSTYPE_THIS_VARIANT (context),
				       CLASSTYPE_VBASECLASSES (current_class_type));
      offset = TREE_PURPOSE (offset_info);
    }
  this_vcontext = vcontext;
  this_context = context;
  this_offset = genop (MINUS_EXPR, offset, base_offset);
}

static int
modify_vtable_entry (type, i)
     tree type;
     int i;
{
  tree basetype = i == 0 ? (abort (), type) : CLASSTYPE_THIS_BASECLASS (type, i);
  tree tmp;
  tree pfn;
  tree vcontext = this_vcontext;
  tree context = this_context;
  tree base_fndecl = this_base_fndecl;

  if (TREE_VIA_VIRTUAL (basetype) == 0
      && DECL_NAME (CLASSTYPE_VFIELD (basetype)) == static_vfield_name)
    {
      if (! CLASSTYPE_MARKED4 (current_class_type))
	build_vtable (basetype, current_class_type);

      /* In this case, it is *type*'s vtable we are modifying.  */
      basetype = current_class_type;
    }
  else
    {
      /* This is our very own copy of `basetype' to play with.  */
      if (! CLASSTYPE_MARKED4 (basetype))
	prepare_fresh_vtable (basetype, current_class_type);
    }

  tmp = CLASSTYPE_VIRTUALS (basetype);

  for (i = (HOST_BITS_PER_INT >= BITS_PER_WORD
#ifdef VTABLE_USES_MASK
	    && 0
#endif
	    ? (TREE_INT_CST_LOW (DECL_VINDEX (base_fndecl))
	       & ((1<<(BITS_PER_WORD-1))-1))
	    : TREE_INT_CST_LOW (DECL_VINDEX (base_fndecl)));
       i > 0;
       i--, tmp = TREE_CHAIN (tmp))
    ;

  /* Now change it in place.  */
  pfn = FNADDR_FROM_VTABLE_ENTRY (TREE_VALUE (tmp));

  /* We can't put in the really right offset information
     here, since we have not yet laid out the class to
     take into account virtual base classes.  */
  TREE_VALUE (tmp) = build_vtable_entry (this_offset, vtable_fnaddr);
  if (TREE_CODE (DECL_VINDEX (this_fndecl)) == INTEGER_CST)
    {
      if (! tree_int_cst_equal (DECL_VINDEX (this_fndecl),
				DECL_VINDEX (TREE_OPERAND (pfn, 0))))
	{
	  tree elts = CONSTRUCTOR_ELTS (TREE_VALUE (tmp));
	  tree this_vfield = CLASSTYPE_VFIELD (type);
	  tree index = DECL_VINDEX (TREE_OPERAND (pfn, 0));
	  /* Compute the relative offset of vtable we are really looking for.  */
	  TREE_VALUE (elts) = genop (PLUS_EXPR,
				     build_int (DECL_OFFSET (this_vfield)
						/ DECL_SIZE_UNIT (this_vfield)),
				     this_offset);
	  /* Say what index to use when we use that vtable.  */
#ifndef VTABLE_USES_MASK
	  index = build_int_2 (TREE_INT_CST_LOW (index) & ~(1 << (BITS_PER_WORD -1)), 0);
#endif
	  TREE_VALUE (TREE_CHAIN (elts)) = index;
	}
    }
  else
    SET_DECL_VINDEX (this_fndecl, DECL_VINDEX (TREE_OPERAND (pfn, 0)));

  return 0;
}

static int
q_base_with_virtual_p (type, i)
     tree type;
     int i;
{
  tree basetype;
  tree vfield;

  if (static_vfield_name == 0)
    abort ();
  basetype = CLASSTYPE_THIS_BASECLASS (type, i);
  if ((vfield = CLASSTYPE_VFIELD (basetype)) == 0)
    return 0;
  if (DECL_VCONTEXT (this_fndecl) == CLASSTYPE_MAIN_VARIANT (basetype)
      || get_base_type (DECL_VCONTEXT (this_fndecl), basetype, 0) != 0)
    return 1;
  return 0;
}

static tree
add_virtual_function (pending_virtuals, has_virtual, x, first)
     tree pending_virtuals;
     int *has_virtual;
     tree x;
     int first;
{
  int debug_vbase = 1;

  /* FUNCTION_TYPEs and OFFSET_TYPEs no longer freely
     convert to void *.  Make such a conversion here.  */
  tree vfn = build (ADDR_EXPR, ptr_type_node, x);
  TREE_LITERAL (vfn) = 1;
  TREE_ADDRESSABLE (x) = 1;

  /* If the virtual function is a redefinition of a prior one,
     figure out in which base class the new definition goes,
     and if necessary, make a fresh virtual function table
     to hold that entry.  */
  if (DECL_VINDEX (x) == NULL_TREE)
    {
      tree entry = build_vtable_entry (integer_zero_node, vfn);

      /* Build a new INT_CST for this DECL_VINDEX.  */
#ifdef VTABLE_USES_MASK
      SET_DECL_VINDEX (x, build_int_2 (++(*has_virtual), 0));
#else
      SET_DECL_VINDEX (x, build_int_2 (((1 << (BITS_PER_WORD - 1)) | ++(*has_virtual)), ~0));
#endif
      pending_virtuals = tree_cons (DECL_VINDEX (x), entry, pending_virtuals);
    }
  else if (debug_vbase && TYPE_USES_VIRTUAL_BASECLASSES (current_class_type))
    {
      /* Need an entry in some other virtual function table.
         Deal with this after we have laid out our virtual base classes.  */
      pending_hard_virtuals = temp_tree_cons (x, vfn, pending_hard_virtuals);
    }
  else
    {
      /* Need an entry in some other virtual function table.
         We can do this now.  */
      prepare_to_modify_vtable_entry (x, DECL_VINDEX (x), vfn);

      /* Get the base class which is the immediate predecessor
	 owning this virtual function.  If there is more than
	 one, an error will be reported later.  */
      if (CLASSTYPE_N_BASECLASSES (current_class_type) == 1)
	modify_vtable_entry (current_class_type, 1);
      else
	breadth_first_search (current_class_type, modify_vtable_entry,
			      q_base_with_virtual_p);
    }
  return pending_virtuals;
}

void
merge_virtual_function (type, x, this_context, that_context, this_before_that, entry)
     tree type;
     tree x;
     tree this_context, that_context;
     int this_before_that;
     tree entry;
{
  /* Need an entry in some other virtual function table.  */
  int i = 1;			/* this is the index we want.  */
  int j;
  tree tmp1, tmp2;		/* These walk through entries in arrays.  */
  tree pfn;
  tree btype = this_before_that ? this_context : that_context;
  tree dtype = this_before_that ? that_context : this_context;
  tree basetype;

  /* Get the base class which is the immediate predecessor
     owning this virtual function.  If there is more than
     one, an error will be reported later.  */

  assert (CLASSTYPE_N_BASECLASSES (type) != 1);
  for (i = 2; i <= CLASSTYPE_N_BASECLASSES (type); i++)
    {
      basetype = CLASSTYPE_THIS_BASECLASS (type, i);

      if (CLASSTYPE_MAIN_VARIANT (dtype) == CLASSTYPE_MAIN_VARIANT (basetype)
	  || get_base_type (dtype, basetype, 0))
	{
	  tmp1 = CLASSTYPE_VIRTUALS (dtype);
	  for (j = (HOST_BITS_PER_INT >= BITS_PER_WORD
#ifdef VTABLE_USES_MASK
		    && 0
#endif
		    ? (TREE_INT_CST_LOW (DECL_VINDEX (x))
		       & ((1<<(BITS_PER_WORD-1))-1))
		    : TREE_INT_CST_LOW (DECL_VINDEX (x)));
	       j > 0;
	       j--)
	    {
	      tmp1 = TREE_CHAIN (tmp1);
	    }
	  pfn = FNADDR_FROM_VTABLE_ENTRY (TREE_VALUE (tmp1));
	  if (DECL_ORIGINAL_NAME (x) == DECL_ORIGINAL_NAME (TREE_OPERAND (pfn, 0))
	      && compparms (TREE_CHAIN (TYPE_ARG_TYPES (TREE_TYPE (x))),
			    TREE_CHAIN (TYPE_ARG_TYPES (TREE_TYPE (TREE_OPERAND (pfn, 0)))), 1))
	    break;
	}
    }

  if (! CLASSTYPE_MARKED4 (type))
    build_vtable (type, type);

  tmp1 = CLASSTYPE_VIRTUALS (type);
  tmp2 = CLASSTYPE_VIRTUALS (dtype);

  for (i = (HOST_BITS_PER_INT >= BITS_PER_WORD
#ifdef VTABLE_USES_MASK
	    && 0
#endif
	    ? (TREE_INT_CST_LOW (DECL_VINDEX (x))
	       & ((1<<(BITS_PER_WORD-1))-1))
	    : TREE_INT_CST_LOW (DECL_VINDEX (x)));
       i > 0;
       i--)
    {
      tmp1 = TREE_CHAIN (tmp1);
      tmp2 = TREE_CHAIN (tmp2);
    }

  /* Now change it in place.  */
  pfn = FNADDR_FROM_VTABLE_ENTRY (TREE_VALUE (tmp1));
  TREE_VALUE (tmp1) = build_vtable_entry (CLASSTYPE_OFFSET (basetype),
					  FNADDR_FROM_VTABLE_ENTRY (TREE_VALUE (tmp2)));
}

/* Subroutines of finish_struct.  */

/* Look through the list of fields for this struct, deleting
   duplicates as we go.  This must be recursive to handle
   anonymous unions.

   FIELD is the field which may not appear anywhere in FIELDS.
   FIELD_PTR, if non-null, is the starting point at which
   chained deletions may take place.
   The value returned is the first acceptable entry found
   in FIELDS.

   Note that anonymous fields which are not of UNION_TYPE are
   not duplicates, they are just anonymous fields.  This happens
   when we have unnamed bitfields, for example.  */
static tree
delete_duplicate_fields_1 (field, field_ptr, fields)
     tree field, *field_ptr, fields;
{
  tree x;
  tree prev = field_ptr ? *field_ptr : 0;
  if (DECL_NAME (field) == 0)
    {
      if (TREE_CODE (TREE_TYPE (field)) != UNION_TYPE)
	return fields;

      for (x = TYPE_FIELDS (TREE_TYPE (field)); x; x = TREE_CHAIN (x))
	fields = delete_duplicate_fields_1 (x, field_ptr, fields);
      if (prev)
	TREE_CHAIN (prev) = fields;
      return fields;
    }
  else
    {
      for (x = fields; x; prev = x, x = TREE_CHAIN (x))
	{
	  if (DECL_NAME (x) == 0)
	    {
	      if (TREE_CODE (TREE_TYPE (x)) != UNION_TYPE)
		continue;
	      TYPE_FIELDS (TREE_TYPE (x))
		= delete_duplicate_fields_1 (field, 0, TYPE_FIELDS (TREE_TYPE (x)));
	      if (TYPE_FIELDS (TREE_TYPE (x)) == 0)
		{
		  if (prev == 0)
		    fields = TREE_CHAIN (fields);
		  else
		    TREE_CHAIN (prev) = TREE_CHAIN (x);
		}
	    }
	  else
	    {
	      if (DECL_NAME (field) == DECL_NAME (x))
		{
		  if (TREE_CODE (field) == CONST_DECL
		      && TREE_CODE (x) == CONST_DECL)
		    error_with_decl (x, "duplicate enum value `%s'");
		  else if (TREE_CODE (field) == CONST_DECL
			   || TREE_CODE (x) == CONST_DECL)
		    error_with_decl (x, "duplicate field `%s' (as enum and non-enum)");
		  else
		    error_with_decl (x, "duplicate member `%s'");
		  if (prev == 0)
		    fields = TREE_CHAIN (fields);
		  else
		    TREE_CHAIN (prev) = TREE_CHAIN (x);
		}
	    }
	}
    }
  return fields;
}

static void
delete_duplicate_fields (fields)
     tree fields;
{
  tree x;
  for (x = fields; x && TREE_CHAIN (x); x = TREE_CHAIN (x))
    TREE_CHAIN (x) = delete_duplicate_fields_1 (x, &x, TREE_CHAIN (x));
}

/* Change the visibility of T::FDECL to VISIBILITY.
   Return 1 if change was legit, otherwise return 0.  */
static int
alter_visibility (t, fdecl, visibility)
     tree t;
     tree fdecl;
     enum visibility_type visibility;
{
  tree elem = purpose_member (t, DECL_VISIBILITY (fdecl));
  if (elem && TREE_VALUE (elem) != (tree)visibility)
    {
      if (TREE_CODE (TREE_TYPE (fdecl)) == FUNCTION_DECL)
	{
	  error_with_decl (TREE_TYPE (fdecl), "conflicting visibility specifications for method `%s', ignored");
	}
      else error ("conflicting visibility specifications for field `%s', ignored", IDENTIFIER_POINTER (DECL_NAME (fdecl)));
    }
  else if (TREE_PRIVATE (fdecl) && visibility != visibility_private)
    error_with_decl (fdecl, "cannot make private %s non-private");
  else if (TREE_PROTECTED (fdecl) && visibility == visibility_public)
		    
    error_with_decl (fdecl, "cannot make protected %s public");
  else if (elem == NULL_TREE)
    {
      DECL_VISIBILITY (fdecl) = tree_cons (t, (tree)visibility,
					   DECL_VISIBILITY (fdecl));
      return 1;
    }
  return 0;
}

static void
build_class_init_list (type, first)
     tree type;
     int first;
{
  /* If this type does not have a constructor, then the
     compiler must manually deal with all of the initialization
     this type requires.
     
     If a base initializer exists only to fill in the
     virtual function table pointer, then we mark that fact
     with the TREE_VIRTUAL bit.  This way, we avoid
     multiple initializations of the same field by
     each virtual function table up the class hierarchy.

     Virtual base class pointers are not initialized here.  They are
     initialized only at the "top level" of object creation.
     If we initialized them here, we would have to skip a lot of work.  */

  tree base_init_list = NULL_TREE;
  tree member_init_list = NULL_TREE;

  /* Since we build member_init_list and base_init_list using
     tree_cons, backwards fields the all through work.  */
  tree x;
  int i, n_baseclasses = CLASSTYPE_N_BASECLASSES (type);

  for (x = TYPE_FIELDS (type); x; x = TREE_CHAIN (x))
    {
      if (TREE_CODE (x) == VAR_DECL || TREE_CODE (x) == CONST_DECL)
	continue;

      if (TYPE_NEEDS_CONSTRUCTING (TREE_TYPE (x))
	  || DECL_INITIAL (x) != NULL_TREE)
	member_init_list = tree_cons (x, type, member_init_list);
    }
  member_init_list = nreverse (member_init_list);

  /* We will end up doing this last.  Need special marker
     to discourage infinite recursion.  */
  if (TYPE_VIRTUAL_P (type))
    {
      base_init_list = build_tree_list (error_mark_node, type);
      TREE_ADDRESSABLE (base_init_list) = 1;
    }

  /* Each base class which needs to have initialization
     of some kind gets to make such requests known here.  */
  for (i = n_baseclasses; i > 0; i--)
    {
      tree basetype = CLASSTYPE_THIS_BASECLASS (type, i);
      tree blist;

      /* Don't initialize virtual baseclasses this way.  */
      if (TREE_VIA_VIRTUAL (basetype))
	continue;

      if (TYPE_HAS_CONSTRUCTOR (basetype))
	{
	  base_init_list = tree_cons (NULL_TREE, basetype,
				      base_init_list);
	  continue;
	}

      blist = CLASSTYPE_BASE_INIT_LIST (basetype);
      if (blist != NULL_TREE)
	{
	  /* This is normally true for single inheritance.
	     The win is we can shrink the chain of initializations
	     to be done by only converting to the actual type
	     we are interested in.  */
	  if (TREE_CODE (TREE_VALUE (blist)) == RECORD_TYPE
	      && (DECL_OFFSET (TYPE_NAME (basetype))
		  == DECL_OFFSET (TYPE_NAME (TREE_VALUE (blist)))))
	    {
	      if (base_init_list)
		{
		  /* Does it do more than just fill in a
		     virtual function table pointer?  */
		  if (! TREE_ADDRESSABLE (blist))
		    base_init_list = build_tree_list (blist, base_init_list);
		}
	      else
		base_init_list = blist;
	    }
	  else
	    {
	      /* The function expand_aggr_init knows how to do the
		 initialization of `basetype' without getting
		 an explicit `blist'.  */
	      if (base_init_list)
		base_init_list = tree_cons (NULL_TREE, basetype, base_init_list);
	      else
		base_init_list = CLASSTYPE_AS_LIST (basetype);
	    }
	}
    }

  if (base_init_list)
    if (member_init_list)
      CLASSTYPE_BASE_INIT_LIST (type) = build_tree_list (base_init_list, member_init_list);
    else
      CLASSTYPE_BASE_INIT_LIST (type) = base_init_list;
  else if (member_init_list)
    CLASSTYPE_BASE_INIT_LIST (type) = member_init_list;
}

/* Create a RECORD_TYPE or UNION_TYPE node for a C struct or union declaration
   (or C++ class declaration).

   For C++, we must handle the building of derived classes.
   Also, C++ allows static class members.  The way that this is
   handled is to keep the field name where it is (as the DECL_NAME
   of the field), and place the overloaded decl in the DECL_OFFSET
   of the field.  layout_record and layout_union will know about this.

   More C++ hair: inline functions have text in their
   DECL_PENDING_INLINE_INFO nodes which must somehow be parsed into
   meaningful tree structure.  After the struct has been laid out, set
   things up so that this can happen.

   And still more: virtual functions.  In the case of single inheritance,
   when a new virtual function is seen which redefines a virtual function
   from the base class, the new virtual function is placed into
   the virtual function table at exactly the same address that
   it had in the base class.  When this is extended to multiple
   inheritance, the same thing happens, except that multiple virtual
   function tables must be maintained.  The first virtual function
   table is treated in exactly the same way as in the case of single
   inheritance.  Additional virtual function tables have different
   DELTAs, which tell how to adjust `this' to point to the right thing.

   LIST_OF_FIELDLISTS is just that.  The elements of the list are
   TREE_LIST elements, whose TREE_PURPOSE field tells what visibility
   the list has, and the TREE_VALUE slot gives the actual fields.

   EMPTY is non-zero if this structure has no declarations following it.

   If flag_all_virtual == 1, then we lay all functions into
   the virtual function table, as though they were declared
   virtual.  Constructors do not lay down in the virtual function table.

   If flag_all_virtual == 2, then we lay all functions into
   the virtual function table, such that virtual functions
   occupy a space by themselves, and then all functions
   of the class occupy a space by themselves.  This is illustrated
   in the following diagram:

   class A; class B : A;

	Class A's vtbl:			Class B's vtbl:
    --------------------------------------------------------------------
   | A's virtual functions|		| B's virtual funcitions	|
   |			  |		| (may inherit some from A).	|
    --------------------------------------------------------------------
   | All of A's functions |		| All of A's functions		|
   | (such as a->A::f).	  |		| (such as b->A::f)		|
    --------------------------------------------------------------------
					| B's new virtual functions	|
					| (not defined in A.)		|
					 -------------------------------
					| All of B's functions		|
					| (such as b->B::f)		|
					 -------------------------------

   this allows the program to make references to any function, virtual
   or otherwise in a type-consistant manner.  */

tree
finish_struct (t, list_of_fieldlists, empty, warn_anon)
     tree t;
     tree list_of_fieldlists;
     int empty;
     int warn_anon;
{
  tree head;
  int old;
  int round_up_size = 1;

  enum tree_code code = TREE_CODE (t);
  register tree x, y, tail;
  int needs_ctor = 0, needs_dtor = 0, needs_wrapper = 0;
  int members_need_dtors = 0;
  tree name = TYPE_NAME (t), fields, fn_fields;
  enum visibility_type visibility;
  int all_virtual;
  int has_virtual = 0;
  int max_has_virtual = 0;
  tree pending_virtuals = NULL_TREE;
  tree vfield = NULL_TREE;
  int first_vfn_base_index = 0;
  int i, n_baseclasses;
  int any_assigns_this = 0;
  int any_default_members = 0;
  int gets_assignment = 0;
  int n_ancestors = 0;
  char *err_name;
  int const_sans_init = 0;
  int ref_sans_init = 0;
  tree friend_list = 0;
  int nonprivate_method = 0;

  if (TREE_CODE (name) == TYPE_DECL)
    name = DECL_NAME (name);
  err_name = IDENTIFIER_POINTER (name);

  if (warn_anon && code != UNION_TYPE
      && ANON_AGGRNAME_P (name))
    {
      warning ("un-usable class ignored (anonymous classes and unions are useless)");
      err_name = "(anon)";
    }

  if (TYPE_SIZE (t))
    {
      error ((TREE_CODE (t) == UNION_TYPE ? "redefinition of `union %s'"
	      : "redefinition of `struct %s'"),
	     err_name);
      popclass (0);
      return t;
    }

  /* If this type was previously laid out as a forward reference,
     make sure we lay it out again.  */

  TYPE_SIZE (t) = 0;

  old = suspend_momentary ();

  /* If using multiple inheritance, this may cause variants of our
     basetypes to be used (instead of their canonical forms).  */
  fields = layout_basetypes (t);
  y = tree_last (fields);

  /* Install struct as DECL_FIELD_CONTEXT of each field decl.
     Also process specified field sizes.
     Set DECL_SIZE_UNIT to the specified size, or 0 if none specified.
     The specified size is found in the DECL_INITIAL.
     Store 0 there, except for ": 0" fields (so we can find them
     and delete them, below).  */

  n_baseclasses = CLASSTYPE_N_BASECLASSES (t);
  if (n_baseclasses >= 1)
    {
      if (n_baseclasses == 1)
	TYPE_USES_MULTIPLE_INHERITANCE (t)
	  = TYPE_USES_MULTIPLE_INHERITANCE (CLASSTYPE_BASECLASS (t, 1));
      else
	TYPE_USES_MULTIPLE_INHERITANCE (t) = 1;
    }

  static_vfield_name = NULL_TREE;

  for (i = 1; i <= n_baseclasses; i++)
    {
      tree basetype = CLASSTYPE_BASECLASS (t, i);

      /* If the type of basetype is incomplete, then
	 we already complained about that fact.  */
      if (TYPE_SIZE (basetype))
	{
	  TYPE_USES_VIRTUAL_BASECLASSES (t)
	    |= (TYPE_USES_VIRTUAL_BASECLASSES (basetype) | CLASSTYPE_VIA_VIRTUAL (t, i));
	  CLASSTYPE_HAS_UNINHERITABLE_VIRTUALS (t)
	    |= CLASSTYPE_HAS_UNINHERITABLE_VIRTUALS (basetype) | (CLASSTYPE_UNINHERITED_VIRTUALS (basetype) != 0);
	  CLASSTYPE_ALTERS_VISIBILITIES_P (t)
	    |= CLASSTYPE_ALTERS_VISIBILITIES_P (basetype);

	  /* This does not work if the MI lattice has
	     virtual baseclasses.  Must use a bfs or dfs
	     routine instead.  */
	  n_ancestors += CLASSTYPE_N_SUPERCLASSES (basetype);
	  needs_ctor |= TYPE_NEEDS_CONSTRUCTOR (basetype);
	  needs_dtor |= TYPE_NEEDS_DESTRUCTOR (basetype);
	  needs_wrapper |= TYPE_NEEDS_WRAPPER (basetype) | TYPE_HAS_WRAPPER (basetype);
	  any_assigns_this |= TREE_ANY_ASSIGNS_THIS (basetype);
	  gets_assignment |= TREE_GETS_ASSIGNMENT (basetype);

	  TYPE_OVERLOADS_CALL_EXPR (t) |= TYPE_OVERLOADS_CALL_EXPR (basetype);
	  TYPE_OVERLOADS_ARRAY_REF (t) |= TYPE_OVERLOADS_ARRAY_REF (basetype);
	  if (TYPE_VIRTUAL_P (basetype))
	    {
	      /* We cannot share virtual information for any node
		 which has a non-zero offset.  This is because for
		 such nodes, we rewrite that node's virtual information
		 in place--mostly.  */
	      if (CLASSTYPE_OFFSET (basetype) != integer_zero_node
		  || CLASSTYPE_VIA_VIRTUAL (t, i))
		{
		  CLASSTYPE_THIS_BASECLASS (t, i) = basetype = copy_node (basetype);
		  TREE_VIA_VIRTUAL (basetype) = CLASSTYPE_VIA_VIRTUAL (t, i);
		  copy_type_lang_specific (basetype);
		}
	      if (first_vfn_base_index == 0)
		{
		  first_vfn_base_index = i;

		  /* The size of the virtual function table we get
		     from the base class is this.  */
		  max_has_virtual = has_virtual = CLASSTYPE_VSIZE (basetype);

		  /* Don't borrow vfields from virtual baseclasses,
		     but do borrow everything else.  */
		  if (! TREE_VIA_VIRTUAL (basetype))
		    {
		      vfield = CLASSTYPE_VFIELD (basetype);
		      CLASSTYPE_VFIELD (t) = vfield;
		      static_vfield_name = DECL_NAME (vfield);
		    }
		}
	      else
		{
		  if (CLASSTYPE_VSIZE (basetype) > max_has_virtual)
		    max_has_virtual = CLASSTYPE_VSIZE (basetype);
		}
	    }
	}
      else continue;
    }

  CLASSTYPE_N_SUPERCLASSES (t) = n_ancestors + n_baseclasses;
  fn_fields = NULL_TREE;
  tail = NULL_TREE;
  if (y && list_of_fieldlists)
    TREE_CHAIN (y) = TREE_VALUE (list_of_fieldlists);

#ifdef SOS
  if (flag_all_virtual == 2)
    all_virtual = 2;
  else
#endif
    {
      if (flag_all_virtual == 1 && (needs_wrapper || TYPE_OVERLOADS_METHOD_CALL_EXPR (t)))
	all_virtual = 1;
      else
	all_virtual = 0;
    }

  if (list_of_fieldlists
      && TREE_PURPOSE (list_of_fieldlists) == (tree)visibility_default)
    TREE_PURPOSE (list_of_fieldlists) =
      (tree)(CLASSTYPE_DECLARED_CLASS (t) ? visibility_private : visibility_public);
  while (list_of_fieldlists)
    {
      visibility = (enum visibility_type)TREE_PURPOSE (list_of_fieldlists);

      for (x = TREE_VALUE (list_of_fieldlists); x; x = TREE_CHAIN (x))
	{
	  TREE_PRIVATE (x) = visibility == visibility_private;
	  TREE_PROTECTED (x) = visibility == visibility_protected;
	  if (TREE_CODE (x) == FUNCTION_DECL)
	    {
	      if (y) TREE_CHAIN (y) = TREE_CHAIN (x);

	      /* Clear out this flag.

	         @@ Doug may figure out how to break
		 @@ this with nested classes and friends.  */
	      DECL_IN_AGGR_P (x) = 0;

	      /* Don't add friends to member function chains,
	         but do queue them if they have inline functions
		 to worry about.  */
	      nonprivate_method |= ! TREE_PRIVATE (x);

	      if (DECL_FRIEND_P (x))
		{
		  if (DECL_PENDING_INLINE_INFO (x))
		    friend_list = temp_tree_cons (NULL_TREE, x, friend_list);
		  continue;
		}

	      if (! fn_fields) fn_fields = x;
	      else TREE_CHAIN (tail) = x;
	      tail = x;
	      if (DECL_FIELD_CONTEXT (x))
		continue;

	      DECL_CONTEXT (x) = t;
	      DECL_VCONTEXT (x) = t;

	      DECL_SIZE_UNIT (x) = 0;

	      /* The name of the field is the original field name
		 Save this in auxilliary field for later overloading.  */
	      if (DECL_VIRTUAL_P (x)
		  || (all_virtual == 1 && ! DECL_CONSTRUCTOR_P (x)))
		pending_virtuals = add_virtual_function (pending_virtuals,
							 &has_virtual, x,
							 first_vfn_base_index);
	      continue;
	    }

	  /* Handle visibility declarations.  */
	  if (DECL_NAME (x) && TREE_CODE (DECL_NAME (x)) == SCOPE_REF)
	    {
	      tree type = TREE_OPERAND (DECL_NAME (x), 0);
	      tree fdecl = TREE_OPERAND (DECL_NAME (x), 1);
	      tree elem;

	      if (y) TREE_CHAIN (y) = TREE_CHAIN (x);
	      /* Make type T see field decl FDECL with
		 the visibility VISIBILITY.  */
	      if (TREE_CODE (fdecl) == TREE_LIST)
		{
		  fdecl = TREE_VALUE (TREE_VALUE (fdecl));
		  while (fdecl)
		    {
		      if (alter_visibility (t, fdecl, visibility) == 0)
			break;
		      fdecl = TREE_CHAIN (fdecl);
		    }
		}
	      else alter_visibility (t, fdecl, visibility);
	      CLASSTYPE_ALTERS_VISIBILITIES_P (t) = 1;
	      continue;
	    }

	  /* Perform error checking that did not get done in grokdeclarator.  */
	  if (TREE_CODE (TREE_TYPE (x)) == FUNCTION_TYPE)
	    {
	      error_with_decl (x, "field `%s' invalidly declared function type");
	      TREE_TYPE (x) = build_pointer_type (TREE_TYPE (x));
	    }
	  else if (TREE_CODE (TREE_TYPE (x)) == METHOD_TYPE)
	    {
	      error_with_decl (x, "field `%s' invalidly declared method type");
	      TREE_TYPE (x) = build_pointer_type (TREE_TYPE (x));
	    }
	  else if (TREE_CODE (TREE_TYPE (x)) == OFFSET_TYPE)
	    {
	      error_with_decl (x, "field `%s' invalidly declared offset type");
	      TREE_TYPE (x) = build_pointer_type (TREE_TYPE (x));
	    }
	  /* If this is of reference type, check if it needs an init.  */
	  if (TREE_CODE (TREE_TYPE (x)) == REFERENCE_TYPE
	      && DECL_INITIAL (x) == 0)
	    ref_sans_init = 1;

	  /* When this goes into scope, it will be a non-local reference.  */
	  TREE_NONLOCAL (x) = 1;

	  /* Static class members do not affect us in this way.
	     Neither do CONST_DECLs.  */
	  if (TREE_CODE (x) == FIELD_DECL)
	    {
	      /* Never let anything with uninheritable virutals
		 make it through without complaint.  */
	      if (IS_AGGR_TYPE (TREE_TYPE (x))
		  && CLASSTYPE_UNINHERITED_VIRTUALS (TREE_TYPE (x)))
		uninheritable_virtuals_error (x, TREE_TYPE (x));

	      /* If any field is const, the structure type is pseudo-const.  */
	      if (TREE_READONLY (x))
		{
		  C_TYPE_FIELDS_READONLY (t) = 1;
		  if (DECL_INITIAL (x) == 0)
		    const_sans_init = 1;
		}
	      else 
		{
		  /* A field that is pseudo-const makes the structure likewise.  */
		  tree t1 = TREE_TYPE (x);
		  while (TREE_CODE (t1) == ARRAY_TYPE)
		    t1 = TREE_TYPE (t1);
		  if (IS_AGGR_TYPE (t1))
		    {
		      if (C_TYPE_FIELDS_READONLY (t1))
			C_TYPE_FIELDS_READONLY (t) = 1;
		      if (CLASSTYPE_READONLY_FIELDS_NEED_INIT (t1))
			const_sans_init = 1;
		    }
		}
	    }

	  if (! fields) fields = x;
	  DECL_FIELD_CONTEXT (x) = t;
	  DECL_SIZE_UNIT (x) = 0;

	  if (TREE_PACKED (x))
	    {
	      /* Invalid bit-field size done by grokfield.  */
	      /* Detect invalid bit-field type.  */
	      if (DECL_INITIAL (x)
		  && TREE_CODE (TREE_TYPE (x)) != INTEGER_TYPE
		  && TREE_CODE (TREE_TYPE (x)) != ENUMERAL_TYPE)
		{
		  error_with_decl (x, "bit-field `%s' has invalid type");
		  DECL_INITIAL (x) = NULL;
		}
	      if (DECL_INITIAL (x) && pedantic
		  && TREE_TYPE (x) != integer_type_node
		  && TREE_TYPE (x) != unsigned_type_node)
		warning_with_decl (x, "bit-field `%s' type invalid in ANSI C");

	      /* Detect and ignore out of range field width.  */
	      if (DECL_INITIAL (x))
		{
		  register int width = TREE_INT_CST_LOW (DECL_INITIAL (x));

		  if (width < 0)
		    {
		      DECL_INITIAL (x) = NULL;
		      warning_with_decl (x, "negative width in bit-field `%s'");
		    }
		  else if (width == 0 && DECL_NAME (x) != 0)
		    {
		      error_with_decl (x, "zero width for bit-field `%s'");
		      DECL_INITIAL (x) = NULL;
		    }
		  else if (width > TYPE_PRECISION (TREE_TYPE (x)))
		    {
		      DECL_INITIAL (x) = NULL;
		      warning_with_decl (x, "width of `%s' exceeds its type");
		    }
		}

	      /* Process valid field width.  */
	      if (DECL_INITIAL (x))
		{
		  register int width = TREE_INT_CST_LOW (DECL_INITIAL (x));

		  if (width == 0)
		    {
		      /* field size 0 => mark following field as "aligned" */
		      if (TREE_CHAIN (x))
			DECL_ALIGN (TREE_CHAIN (x))
			  = MAX (DECL_ALIGN (TREE_CHAIN (x)), EMPTY_FIELD_BOUNDARY);
		      /* field of size 0 at the end => round up the size.  */
		      else
			round_up_size = EMPTY_FIELD_BOUNDARY;
		    }
		  else
		    {
		      DECL_INITIAL (x) = NULL_TREE;
		      DECL_SIZE_UNIT (x) = width;
		      TREE_PACKED (x) = 1;
		      /* Traditionally a bit field is unsigned
			 even if declared signed.  */
		      if (flag_traditional
			  && TREE_CODE (TREE_TYPE (x)) == INTEGER_TYPE)
			TREE_TYPE (x) = unsigned_type_node;
		    }
		}
	      else
		/* Non-bit-fields are aligned for their type.  */
		DECL_ALIGN (x) = MAX (DECL_ALIGN (x), TYPE_ALIGN (TREE_TYPE (x)));
	    }
	  else
	    {
	      tree type = TREE_TYPE (x);
	      if (TREE_CODE (type) == ARRAY_TYPE)
		type = TREE_TYPE (type);
	      if (code == UNION_TYPE)
		{
		  if (TYPE_NEEDS_CONSTRUCTING (type))
		    error ("member %s::%s with constructor not allowed in union",
			   IDENTIFIER_POINTER (name), IDENTIFIER_POINTER (DECL_NAME (x)));
		  if (TYPE_NEEDS_DESTRUCTOR (type))
		    error ("member %s::%s with destructor (also) not allowed in union",
			   IDENTIFIER_POINTER (name), IDENTIFIER_POINTER (DECL_NAME (x)));
		}
	      else if (IS_AGGR_TYPE_CODE (code)
		       && IS_AGGR_TYPE (type))
		{
		  needs_ctor |= TYPE_NEEDS_CONSTRUCTOR (type);
		  needs_dtor |= TYPE_NEEDS_DESTRUCTOR (type);
		  members_need_dtors |= TYPE_NEEDS_DESTRUCTOR (type);
		  gets_assignment |= TREE_GETS_ASSIGNMENT (type);
		}
	      if (TREE_CODE (x) != CONST_DECL && DECL_INITIAL (x) != NULL_TREE)
		{
		  if (code == UNION_TYPE && any_default_members != 0)
		    error ("multiple fields in union initialized");
		  any_default_members = 1;
		}
	    }
	  y = x;
	}
      list_of_fieldlists = TREE_CHAIN (list_of_fieldlists);
      /* link the tail while we have it! */
      if (y)
	{
	  TREE_CHAIN (y) = NULL_TREE;

	  if (list_of_fieldlists
	      && TREE_VALUE (list_of_fieldlists)
	      && TREE_CODE (TREE_VALUE (list_of_fieldlists)) != FUNCTION_DECL)
	    TREE_CHAIN (y) = TREE_VALUE (list_of_fieldlists);
	}
    }

  if (tail) TREE_CHAIN (tail) = NULL_TREE;

  /* If this type has any constant members which did not come
     with their own initialization, mark that fact here.  It is
     not an error here, since such types can be saved either by their
     constructors, or by fortuitous initialization.  */
  CLASSTYPE_READONLY_FIELDS_NEED_INIT (t) = const_sans_init;
  CLASSTYPE_REF_FIELDS_NEED_INIT (t) = ref_sans_init;

  if (fields == 0 && pedantic)
    warning ((code == UNION_TYPE ? "union has no data members"
	      : "structure has no data members"));

  if (vfield == 0
      && (has_virtual
#ifdef SOS
	  || TYPE_DYNAMIC (t)
#endif
	  ))
    {
      /* We build this decl with ptr_type_node, and
	 change the type when we know what it should be.  */
      vfield = build_lang_decl (FIELD_DECL, get_vfield_name (t), ptr_type_node);
      DECL_FIELD_CONTEXT (vfield) = t;
      DECL_VCONTEXT (vfield) = t;
      DECL_SIZE_UNIT (vfield) = 0;
      if (y)
	{
	  assert (TREE_CHAIN (y) == 0);
	  TREE_CHAIN (y) = vfield;
	  y = vfield;
	}
      else fields = vfield;
    }

  /* Now DECL_INITIAL is null on all members except for zero-width bit-fields.
     And they have already done their work.

     C++: maybe we will support default field initialization some day...  */

  /* Delete all zero-width bit-fields from the front of the fieldlist */
  while (fields && TREE_PACKED (fields)
	 && DECL_INITIAL (fields))
    fields = TREE_CHAIN (fields);
  /* Delete all such fields from the rest of the fields.  */
  for (x = fields; x;)
    {
      if (TREE_CHAIN (x) && TREE_PACKED (TREE_CHAIN (x))
	  && DECL_INITIAL (TREE_CHAIN (x)))
	TREE_CHAIN (x) = TREE_CHAIN (TREE_CHAIN (x));
      else x = TREE_CHAIN (x);
    }
  /* Delete all duplicate fields from the fields */
  delete_duplicate_fields (fields);

  /* Now we have the final fieldlist for the data fields.  Record it,
     then lay out the structure or union (including the fields).  */

  TYPE_FIELDS (t) = fields;

  /* If there's a :0 field at the end, round the size to the
     EMPTY_FIELD_BOUNDARY.  */
  TYPE_ALIGN (t) = round_up_size;

  /* Warn about duplicate methods in fn_fields.  Also compact
     method lists so that lookup can be made faster.

     Algorithm:  Outer loop builds lists by method name.
     Inner loop checks for redundant method names within a list.

     Data Structure:  List of method lists.  The outer list
     is a TREE_LIST, whose TREE_PURPOSE field is the field name
     and the TREE_VALUE is the TREE_CHAIN of the FIELD_DECLs.
     Friends are chained in the same way as member functions, but
     they live in the TREE_TYPE field of the outer list.
     That allows them to be quicky deleted, and requires
     no extra storage.

     If there are any constructors/destructors, they are moved to
     the front of the list.  This makes pushclass more efficient.

     We also link each field which has shares a name with its
     baseclass to the head of the list of fields for that base class.
     This allows us to reduce search time in places like `build_method_call'
     to consider only reasonably likely functions.  */

  head = NULL_TREE;
  tail = NULL_TREE;
  if (fn_fields
      && nonprivate_method == 0
      && DECL_FRIEND_CLASSES (TYPE_NAME (t)) == NULL_TREE
      && DECL_FRIENDLIST (TYPE_NAME (t)) == NULL_TREE)
    warning ("all class member functions are private");
  while (fn_fields)
    {
      /* NEXT Pointer, TEST Pointer, and BASE Pointer.  */
      tree nextp, testp;

      nextp = TREE_CHAIN (fn_fields);
      TREE_CHAIN (fn_fields) = NULL_TREE;
      testp = head;
      while (testp && TREE_PURPOSE (testp) != DECL_ORIGINAL_NAME (fn_fields))
	testp = TREE_CHAIN (testp);
      if (testp)
	{
	  tree fndecl = fn_fields;
	  for (x = TREE_VALUE (testp); x; x = TREE_CHAIN (x))
	    {
	      if (DECL_NAME (fndecl) == DECL_NAME (x))
		{
		  /* We complain about multiple destructors on sight,
		     so we do not repeat the warning here.  Friend-friend
		     ambiguities are warned about outside this loop.  */
		  if (! DESTRUCTOR_NAME_P (DECL_NAME (fndecl)))
		    yylineerror (DECL_SOURCE_LINE (fndecl),
				 "ambiguous method `%s' in structure",
				 lang_printable_name (fndecl));
		  break;
		}
	      y = x;
	    }
	  if (x == 0)
	    if (TREE_VALUE (testp))
	      TREE_CHAIN (y) = fn_fields;
	    else
	      TREE_VALUE (testp) = fn_fields;
	}
      else
	{
	  /* Constrcutors are handled easily in search routines.
	     Besides, we know we wont find any, so do not bother looking.  */
	  if (head)
	    {
	      if (DECL_ORIGINAL_NAME (fn_fields) == name)
		head = tree_cons (name, fn_fields, head);
	      else
		{
		  TREE_CHAIN (tail) = build_tree_list (DECL_ORIGINAL_NAME (fn_fields), fn_fields);

		  tail = TREE_CHAIN (tail);
		}
	    }
	  else
	    {
	      head = build_tree_list (DECL_ORIGINAL_NAME (fn_fields), fn_fields);

	      tail = head;
	    }
	  TREE_OVERLOADED (tail) = TREE_OVERLOADED (fn_fields);
	}
      fn_fields = nextp;
    }

  /* If there are constructors (and destructors), they are at the
     front.  Place destructors at very front.  Also warn if all
     constructors and/or destructors are private (in which case this
     class is effectively unusable.  */
  if (TYPE_HAS_DESTRUCTOR (t))
    {
      tree dtor, prev;

      for (dtor = TREE_VALUE (head); dtor; prev = dtor, dtor = TREE_CHAIN (dtor))
	{
	  if (DESTRUCTOR_NAME_P (DECL_NAME (dtor)))
	    {
	      if (TREE_PRIVATE (dtor)
		  && DECL_FRIEND_CLASSES (TYPE_NAME (t)) == NULL_TREE
		  && DECL_FRIENDLIST (TYPE_NAME (t)) == NULL_TREE)
		error ("class `%s' only defines a private destructor and has no friends",
		       TYPE_NAME_STRING (t));
	      break;
	    }
	}
      if (dtor != TREE_VALUE (head))
	{
	  TREE_CHAIN (prev) = TREE_CHAIN (dtor);
	  TREE_CHAIN (dtor) = TREE_VALUE (head);
	  TREE_VALUE (head) = dtor;
	}
    }
  else if (members_need_dtors)
    {
      /* Here we must cons up a destructor on the fly.  */
      tree dtor = cons_up_dtor_for_type (t);
      DECL_CONTEXT (dtor) = t;
      DECL_VCONTEXT (dtor) = t;
      if (DECL_VIRTUAL_P (dtor))
	pending_virtuals = add_virtual_function (pending_virtuals,
						 &has_virtual, dtor);
      if (TYPE_HAS_CONSTRUCTOR (t))
	{
	  TREE_CHAIN (dtor) = TREE_VALUE (head);
	  TREE_VALUE (head) = dtor;
	}
      else
	head = tree_cons (name,  dtor, head);
      TYPE_HAS_DESTRUCTOR (t) = 1;
    }

  if (TYPE_HAS_CONSTRUCTOR (t)
      && DECL_FRIEND_CLASSES (TYPE_NAME (t)) == NULL_TREE
      && DECL_FRIENDLIST (TYPE_NAME (t)) == NULL_TREE)
    {
      int nonprivate_ctor = 0;
      tree ctor;

      for (ctor = TREE_VALUE (head); ctor; ctor = TREE_CHAIN (ctor))
	if (! TREE_PRIVATE (ctor))
	  {
	    nonprivate_ctor = 1;
	    break;
	  }
      if (nonprivate_ctor == 0)
	warning ("class %s only defines private constructors and has no friends",
		 TYPE_NAME_STRING (t));
    }

  /* Now for each member function (except for constructors and
     destructors), compute where member functions of the same
     name reside in base classes.  */
  if (n_baseclasses != 0)
    {
      tree x = head;

      if (TYPE_HAS_CONSTRUCTOR (t))
	x = TREE_CHAIN (x);

      for (; x; x = TREE_CHAIN (x))
	TREE_BASELINK (x) = get_baselinks (t, TREE_PURPOSE (x));
    }

  /* We can't know this information until we have seen all of the
     constructors.  */
  TREE_NONE_ASSIGN_THIS (t) = 0;

  layout_type (t);

  /* Set the TYPE_DECL for this type to contain the right
     value for DECL_OFFSET, so that we can use it as part
     of a COMPONENT_REF for multiple inheritance.  */

  if (TREE_CODE (TYPE_NAME (t)) == TYPE_DECL)
    layout_decl (TYPE_NAME (t));

  /* Now fix up any virtual base class types that we
     left lying around.  We must get these done
     before we try to lay out the virtual function table.  */
  if (TYPE_USES_VIRTUAL_BASECLASSES (t))
    {
      tree vbases;
      layout_vbasetypes (t);
      pending_hard_virtuals = nreverse (pending_hard_virtuals);
      vbases = CLASSTYPE_VBASECLASSES (t);
      while (vbases)
	{
	  tree virtuals = CLASSTYPE_VIRTUALS (TREE_VALUE (vbases));

	  if (virtuals)
	    virtuals = TREE_CHAIN (virtuals);

	  while (virtuals)
	    {
	      tree pfn = FNADDR_FROM_VTABLE_ENTRY (TREE_VALUE (virtuals));
	      tree base_fndecl = TREE_OPERAND (pfn, 0);
	      tree decl = get_first_matching_virtual (t, base_fndecl, 0);
	      if (decl != base_fndecl)
		{
		  int i = TREE_INT_CST_LOW (DECL_VINDEX (base_fndecl)) & ((1<<(BITS_PER_WORD-1))-1);
		  tree these_virtuals = CLASSTYPE_VIRTUALS (DECL_CONTEXT (decl));
		  while (i-- > 0)
		    these_virtuals = TREE_CHAIN (these_virtuals);
		  pfn = FNADDR_FROM_VTABLE_ENTRY (TREE_VALUE (these_virtuals));
		  prepare_to_modify_vtable_entry (decl, base_fndecl, pfn);

		  if (n_baseclasses == 1)
		    modify_vtable_entry (t, 1);
		  else breadth_first_search (t, modify_vtable_entry, q_base_with_virtual_p);
		}
	      virtuals = TREE_CHAIN (virtuals);
	    }
	  vbases = TREE_CHAIN (vbases);
	}
      while (pending_hard_virtuals)
	{
	  /* Need an entry in some other virtual function table.  */
	  prepare_to_modify_vtable_entry (TREE_PURPOSE (pending_hard_virtuals),
					  DECL_VINDEX (TREE_PURPOSE (pending_hard_virtuals)),
					  TREE_VALUE (pending_hard_virtuals));

	  /* Get the base class which is the immediate predecessor
	     owning this virtual function.  If there is more than
	     one, an error will be reported later.  */
	  if (n_baseclasses == 1)
	    modify_vtable_entry (t, 1);
	  else breadth_first_search (t, modify_vtable_entry, q_base_with_virtual_p);
	  pending_hard_virtuals = TREE_CHAIN (pending_hard_virtuals);
	}
    }
  else
    CLASSTYPE_FULL_SIZE (t) = TYPE_SIZE (t);

  if (pending_virtuals)
    {
      pending_virtuals = nreverse (pending_virtuals);
      /* We must enter these virtuals into the table.  */
      if (first_vfn_base_index == 0)
	{
	  pending_virtuals = tree_cons (NULL_TREE, the_null_vtable_entry,
					pending_virtuals);
	  build_vtable (0, t);
	}
      else
	{
	  /* Here we know enough to change the type of our virtual
	     function table, but we will wait until later this function.  */
	  if (! CLASSTYPE_MARKED4 (t))
	    build_vtable (CLASSTYPE_THIS_BASECLASS (t, first_vfn_base_index), t);
	}
    }
  else if (first_vfn_base_index)
    {
      tree basetype = CLASSTYPE_THIS_BASECLASS (t, first_vfn_base_index);
      /* This class contributes nothing new to the virtual function
	 machinery.  However, it must be able to point its
	 vtbl_ptr somewhere.  We create a declaration which is
	 the same as the one from the base class.  In fact,
	 it is the one from the (leftmost) base class.  */

#ifdef SOS
      /* Don't define this ahead of time if we have more
	 fields to add later.  */
      if (all_virtual == 2 && fn_fields != NULL_TREE)
	;
      else
#endif
	{
	  /* See if we should steal the virtual info from base class.  */
	  if (CLASSTYPE_VTABLE (t) == NULL_TREE)
	    CLASSTYPE_VTABLE (t) = CLASSTYPE_VTABLE (basetype);
	  if (CLASSTYPE_VIRTUALS (t) == NULL_TREE)
	    CLASSTYPE_VIRTUALS (t) = CLASSTYPE_VIRTUALS (basetype);
	}
    }
  if (has_virtual)
    {
#ifdef VTABLE_USES_MASK
      if (has_virtual >= VINDEX_MAX)
	{
	  error ("too many virtual functions for class `%s' (VINDEX_MAX < %d)", TYPE_NAME_STRING (t), has_virtual);
	}
#endif
      TYPE_VIRTUAL_P (t) = 1;
      CLASSTYPE_VSIZE (t) = has_virtual;
      if (first_vfn_base_index)
	CLASSTYPE_VIRTUALS (t) = chainon (CLASSTYPE_VIRTUALS (t),
					  pending_virtuals);
      else
	{
	  CLASSTYPE_VIRTUALS (t) = pending_virtuals;
	  if (write_virtuals >= 0)
	    DECL_VIRTUAL_P (CLASSTYPE_VTABLE (t)) = 1;
	}
    }
  else CLASSTYPE_VSIZE (t) = 0;

#ifdef SOS
  if (all_virtual == 2 && (has_virtual || head))
    {
      /* Now that we know the size of the virtual table, lay out
	 the absolute table following it.  */
      int i;
      tree tmp;
      tree pending_absolutes = NULL_TREE;
      int has_absolute = has_virtual;

      /* Local variables for building a table filled with strings
	 containing the names of interesting things.  */
      tree decl, init;
      tree start = NULL_TREE, next = NULL_TREE;
      tree outer = head;

      while (outer)
	{
	  tree inner;
	  for (inner = TREE_VALUE (outer); inner; inner = TREE_CHAIN (inner))
	    {
	      tree entry;
	      tree fn;

	      /* Don't bother with functions which appear
		 for visibility reasons.  */
	      if (DECL_FIELD_CONTEXT (inner) != t)
		continue;

	      /* Must lay this function into its absolute table as well.
		 This forces an inline function to be written out.  */
	      fn = build (ADDR_EXPR, ptr_type_node, inner);
	      TREE_LITERAL (fn) = 1;
	      DECL_DINDEX (inner) = build_int_2 (++has_absolute, 0);
	      entry = build_vtable_entry (integer_zero_node, fn);
	      pending_absolutes = tree_cons (DECL_DINDEX (inner), entry,
					     pending_absolutes);
	    }
	  outer = TREE_CHAIN (outer);
	}

      CLASSTYPE_VIRTUALS (t) = chainon (CLASSTYPE_VIRTUALS (t), nreverse (pending_absolutes));
      if (TYPE_DYNAMIC (t))
	{
	  for (outer = head; outer; outer = TREE_CHAIN (outer))
	    {
	      tree inner;
	      for (inner = TREE_VALUE (outer); inner; inner = TREE_CHAIN (inner))
		{
		  tree str = make_node (STRING_CST);
		  TREE_STRING_LENGTH (str) = IDENTIFIER_LENGTH (DECL_NAME (inner));
		  TREE_STRING_POINTER (str) = IDENTIFIER_POINTER (DECL_NAME (inner));
		  TREE_LITERAL (str) = 1;
		  TREE_STATIC (str) = 1;
		  TREE_TYPE (str)
		    = build_array_type (char_type_node,
					build_index_type (build_int_2 (TREE_STRING_LENGTH (str) - 1, 0)));
	      
		  if (start)
		    {
		      TREE_CHAIN (next) = build_tree_list (NULL_TREE, str);
		      next = TREE_CHAIN (next);
		    }
		  else
		    {
		      start = build_tree_list (NULL_TREE, str);
		      next = start;
		    }
		}
	    }

	  /* Lay out dynamic link table for SOS.  */

	  decl = finish_table (get_linktable_name (t),
			       string_type_node, start, 0);
	}
      has_virtual = has_absolute;
      CLASSTYPE_VSIZE (t) = has_virtual;
    }
#endif

  /* Now lay out the virtual function table.  */
  if (has_virtual)
    {
      tree atype, itype, decl, init;

      if (TREE_TYPE (vfield) == ptr_type_node)
	{
	  /* We must create a pointer to this table because
	     the one inherited from base class does not exist.
	     We will fill in the type when we know what it
	     should really be.  */
	  itype = build_index_type (build_int_2 (has_virtual, 0));
	  atype = build_array_type (vtable_entry_type, itype);
	  layout_type (atype);
	  TREE_TYPE (vfield) = build_pointer_type (atype);
	}
      else
	{
	  atype = TREE_TYPE (TREE_TYPE (vfield));

	  if (has_virtual != TREE_INT_CST_LOW (TYPE_MAX_VALUE (TYPE_DOMAIN (atype))))
	    {
	      /* We must extend (or create) the boundaries on this array,
		 because we picked up virtual functions from multiple
		 base classes.  */
	      itype = build_index_type (build_int_2 (has_virtual, 0));
	      atype = build_array_type (vtable_entry_type, itype);
	      layout_type (atype);
	      vfield = copy_node (vfield);
	      TREE_TYPE (vfield) = build_pointer_type (atype);
	      DECL_FIELD_CONTEXT (vfield) = t;
	    }
	}

      CLASSTYPE_VFIELD (t) = vfield;
      if (TREE_TYPE (CLASSTYPE_VTABLE (t)) != atype)
	{
	  TREE_TYPE (CLASSTYPE_VTABLE (t)) = atype;
	  layout_decl (CLASSTYPE_VTABLE (t));
	  DECL_ALIGN (CLASSTYPE_VTABLE (t))
	    = MAX (TYPE_ALIGN (double_type_node),
		   DECL_ALIGN (CLASSTYPE_VTABLE (t)));
	}

      /* Now that we know all virtual functions defined for this class,
	 make sure we don't inherit any from a base class we are not
	 supposed to inherit.  */
      if (CLASSTYPE_HAS_UNINHERITABLE_VIRTUALS (t))
	{
	  x = TREE_CHAIN (CLASSTYPE_VIRTUALS (t));
	  while (x != NULL_TREE)
	    {
	      tree context;
	      decl = TREE_OPERAND (FNADDR_FROM_VTABLE_ENTRY (TREE_VALUE (x)), 0);
	      context = TYPE_MAIN_VARIANT (DECL_CONTEXT (decl));
	      if (context != t
		  && value_member (decl, CLASSTYPE_UNINHERITED_VIRTUALS (context)))
		{
		  error_with_decl (decl, "Virtual function `%s' cannot be inherited from base class");
		  error ("at this point in file");
		}
	      x = TREE_CHAIN (x);
	    }
	}
    }

  TYPE_NEEDS_CONSTRUCTOR (t) |= needs_ctor || TYPE_HAS_CONSTRUCTOR (t);
  TYPE_NEEDS_CONSTRUCTING (t)
    = (TYPE_NEEDS_CONSTRUCTOR (t) | TYPE_USES_VIRTUAL_BASECLASSES (t))
      || has_virtual;
  TYPE_NEEDS_DESTRUCTOR (t) |= needs_dtor || TYPE_HAS_DESTRUCTOR (t);

  TYPE_NEEDS_WRAPPER (t) |= needs_wrapper;
  TREE_ANY_ASSIGNS_THIS (t) = any_assigns_this;
  TREE_GETS_ASSIGNMENT (t) |= gets_assignment;

  /* Notice whether this class has type conversion functions defined.
     Also report whether joining two types yields an ambiguity in the
     virtual function table, e.g.,

     struct A { virtual int f (); };
     struct B { virtual int f (); };
     struct C : A, B { / * no f (); * / };	/ / error, ambiguous
  */

  build_mi_virtuals (n_baseclasses, max_has_virtual);
  for (i = n_baseclasses; i > 0; i--)
    {
      tree basetype = CLASSTYPE_BASECLASS (t, i);

      if (TYPE_HAS_CONVERSION (basetype))
	{
	  TYPE_HAS_CONVERSION (t) = 1;
	  TYPE_HAS_INT_CONVERSION (t) |= TYPE_HAS_INT_CONVERSION (basetype);
	  TYPE_HAS_REAL_CONVERSION (t) |= TYPE_HAS_REAL_CONVERSION (basetype);
	}
      if (CLASSTYPE_VIA_VIRTUAL (t, i))
	/* Virtual functions from virtual baseclasses are done above.  */;
      else if (i == first_vfn_base_index)
	add_mi_virtuals (i, TREE_CHAIN (CLASSTYPE_VIRTUALS (t)));
      else if (TYPE_VIRTUAL_P (basetype))
	add_mi_virtuals (i, TREE_CHAIN (CLASSTYPE_VIRTUALS (basetype)));
    }
  report_ambiguous_mi_virtuals (n_baseclasses, t);

  /* Now that we know what the virtual functiond table looks like,
     fix up offsets in the presence of virtual base classes.  */
  if (has_virtual && TYPE_USES_VIRTUAL_BASECLASSES (t))
    fixup_vbase_offsets (t);

  if (TYPE_HAS_INT_CONVERSION (t) == 0 || TYPE_HAS_REAL_CONVERSION (t) == 0)
    {
      tree tmp = head;
      int need_int = ! TYPE_HAS_INT_CONVERSION (t);
      int need_real = ! TYPE_HAS_REAL_CONVERSION (t);

      while (tmp)
	{
	  if (OPERATOR_TYPENAME_P (TREE_PURPOSE (tmp)))
	    {
	      tree fntype = TREE_TYPE (TREE_VALUE (tmp));
	      tree return_type = TREE_TYPE (fntype);
	      assert (TREE_CODE (fntype) == METHOD_TYPE);

	      TYPE_HAS_CONVERSION (t) = 1;
	      if (need_int
		  && (TREE_CODE (return_type) == INTEGER_TYPE
		      || (TREE_CODE (return_type) == REFERENCE_TYPE
			  && TREE_CODE (TREE_TYPE (return_type)) == INTEGER_TYPE)))
		{
		  TYPE_HAS_INT_CONVERSION (t) = 1;
		  need_int = 0;
		  if (need_real == 0)
		    break;
		}
	      else if (need_real
		       && (TREE_CODE (return_type) == REAL_TYPE
			   || (TREE_CODE (return_type) == REFERENCE_TYPE
			       && TREE_CODE (TREE_TYPE (return_type)) == REAL_TYPE)))
		{
		  TYPE_HAS_REAL_CONVERSION (t) = 1;
		  need_real = 0;
		  if (need_int == 0)
		    break;
		}
	    }
	  tmp = TREE_CHAIN (tmp);
	}
    }

  if (head)
    CLASSTYPE_FN_FIELDS (t) = head;

  /* If this type has constructors, force its mode to be BLKmode,
     and force its TREE_ADDRESSABLE bit to be nonzero.  */
  if (TYPE_NEEDS_CONSTRUCTING (t) || TYPE_NEEDS_DESTRUCTOR (t))
    {
      TYPE_MODE (t) = BLKmode;
      DECL_MODE (TYPE_NAME (t)) = BLKmode;
      TREE_ADDRESSABLE (t) = 1;
    }

  /* Promote each bit-field's type to int if it is narrower than that.
     Also warn (or error) if static members are specified for a class
     which takes a constructor.  */
  for (x = fields; x; x = TREE_CHAIN (x))
    {
      if (TREE_PACKED (x)
	  && TREE_CODE (TREE_TYPE (x)) == INTEGER_TYPE
	  && (TREE_INT_CST_LOW (DECL_SIZE (x)) * DECL_SIZE_UNIT (x)
	      < TYPE_PRECISION (integer_type_node)))
	TREE_TYPE (x) = integer_type_node;
      if (TREE_STATIC (x))
	{
	  /* According to Stroustrup, page 275, static members
	     cannot be of a class with a constructor.  We include in
	     this case types which have virtual function tables.  Sigh.  */
	  if (TYPE_NEEDS_CONSTRUCTING (TREE_TYPE (x)))
	    if (pedantic)
	      error_with_decl (x, "member `%s' cannot be static (type `%s' needs constructing)", TYPE_NAME_STRING (TREE_TYPE (x)));
	    else
	      warning_with_decl (x, "member `%s' cannot be static (type `%s' needs constructing)", TYPE_NAME_STRING (TREE_TYPE (x)));
	}
    }

  if ((! TYPE_HAS_CONSTRUCTOR (t)
       && TYPE_NEEDS_CONSTRUCTING (t))
      || any_default_members != 0)
    build_class_init_list (t, first_vfn_base_index);

  if (current_lang_name == lang_name_cplusplus)
    {
      embrace_waiting_friends (t);
      /* Write out inline function definitions.  */
      do_inline_function_hair (name, head, friend_list);
    }

  if (CLASSTYPE_VSIZE (t) != 0)
    {
      TYPE_NONCOPIED_PARTS (t) = build_tree_list (NULL_TREE, vfield);

      if (! flag_this_is_variable)
	{
	  tree vtbl_ptr = build_decl (VAR_DECL, get_identifier (VPTR_NAME),
				      TREE_TYPE (vfield));
	  TREE_REGDECL (vtbl_ptr) = 1;
	  SET_DECL_VTBL_PTR (TYPE_NAME (t), vtbl_ptr);
	}
    }

  /* Now out of this class's scope.  However, if this class defined
     any new typedefs, then we must export those to the outer
     binding level.  This is unpleasant.  */
  x = gettags ();

  popclass (0);
  unmark_finished_struct (t);

#if 0
  /* Remove aggregate types from the list of tags,
     since these appear at global scope.  */
  while (x && IS_AGGR_TYPE (TREE_VALUE (x)))
    x = TREE_CHAIN (x);
  TYPE_TAGS (t) = x;
  y = x;
  while (x)
    {
      if (IS_AGGR_TYPE (TREE_VALUE (x)))
	TREE_CHAIN (y) = TREE_CHAIN (x);
      x = TREE_CHAIN (x);
    }
#endif

  hack_incomplete_structures (t);

  resume_momentary (old);

  return t;
}

/* Return non-zero if the effective type of INSTANCE is static.
   Used to determine whether the virtual function table is needed
   or not.  */
int
resolves_to_fixed_type_p (instance)
     tree instance;
{
  switch (TREE_CODE (instance))
    {
    case ADDR_EXPR:
      return resolves_to_fixed_type_p (TREE_OPERAND (instance, 0));
    case VAR_DECL:
    case PARM_DECL:
    case COMPONENT_REF:
    case NEW_EXPR:
      if (IS_AGGR_TYPE (TREE_TYPE (instance)))
	return 1;
    default:
      return 0;
    }
}

/* Ordering function for overload resolution.  */
int
rank_for_overload (x, y)
     struct candidate *x, *y;
{
  if (y->evil - x->evil)
    return y->evil - x->evil;
  if (y->user - x->user)
    return y->user - x->user;
  if (y->b_or_d - x->b_or_d)
    return y->b_or_d - x->b_or_d;
  return y->easy - x->easy;
}

/* TYPE is the type we wish to convert to.  PARM is the parameter
   we have to work with.  We use a somewhat arbitrary cost function
   to measure this conversion.  */
static int
convert_harshness (type, parmtype, parm)
     register tree type, parmtype;
     tree parm;
{
  register enum tree_code codel = TREE_CODE (type);
  register enum tree_code coder = TREE_CODE (parmtype);

  n_convert_harshness++;

  if (TYPE_MAIN_VARIANT (parmtype) == TYPE_MAIN_VARIANT (type))
    return 0;

  if (coder == ERROR_MARK)
    return 1;

  if (codel == POINTER_TYPE
      && (coder == METHOD_TYPE || coder == FUNCTION_TYPE))
    {
      tree p1, p2;
      int harshness, new_harshness;

      /* Not a pointer-to-function or pointer-to-member type.  */
      if (TREE_CODE (type) != POINTER_TYPE)
	return 1;

      /* Get to the OFFSET_TYPE or FUNCTION_TYPE that this might be.  */
      type = TREE_TYPE (type);

      if (coder != TREE_CODE (type))
	return 1;

      if (coder == OFFSET_TYPE)
	{
	  if (TYPE_OFFSET_BASETYPE (type) == TYPE_OFFSET_BASETYPE (parmtype))
	    harshness = 0;
	  else if (get_base_type (TYPE_OFFSET_BASETYPE (TREE_TYPE (type)),
				  TYPE_OFFSET_BASETYPE (TREE_TYPE (parmtype)), 0))
	    harshness = (1<<3);
	  else
	    return 1;
	  type = TREE_TYPE (type);
	  parmtype = TREE_TYPE (parmtype);
	}
      else
	{
	  harshness = 0;
	}

      /* We allow the default conversion between function type
	 and pointer-to-function type for free.  */
      if (type == parmtype)
	return 0;

      /* Compare return types.  */
      harshness |= convert_harshness (TREE_TYPE (type), TREE_TYPE (parmtype), 0);
      if (harshness & 1)
	return 1;
      p1 = TYPE_ARG_TYPES (type);
      p2 = TYPE_ARG_TYPES (parmtype);
      while (p1 && p2)
	{
	  new_harshness = convert_harshness (TREE_VALUE (p1), TREE_VALUE (p2), 0);
	  if (new_harshness & 1)
	    return 1;
	  if ((new_harshness & 7) == 0)
	    harshness += new_harshness;
	  else
	    harshness |= new_harshness;
	  p1 = TREE_CHAIN (p1);
	  p2 = TREE_CHAIN (p2);
	}
      if (p1 == p2)
	return harshness;
      if (p2)
	return 1;
      if (p1)
	return harshness | (TREE_PURPOSE (p1) == NULL_TREE);
    }

  if (coder == UNKNOWN_TYPE)
    {
      if (codel == FUNCTION_TYPE
	  || codel == METHOD_TYPE
	  || (codel == POINTER_TYPE
	      && (TREE_CODE (TREE_TYPE (type)) == FUNCTION_TYPE
		  || TREE_CODE (TREE_TYPE (type)) == METHOD_TYPE)))
	return 0;
      return 1;
    }

  if (coder == VOID_TYPE)
    return 1;

#if 0
  /* Enums can be converted to ints, but not vice-versa.  */
  if (codel == ENUMERAL_TYPE && coder == INTEGER_TYPE)
    return 1;
#endif

  if (codel == INTEGER_TYPE || codel == ENUMERAL_TYPE)
    if (coder == INTEGER_TYPE || coder == ENUMERAL_TYPE)
      {
	int easy = TREE_UNSIGNED (type) ^ TREE_UNSIGNED (parmtype);
	if (TYPE_MODE (type) != TYPE_MODE (parmtype))
	  easy += 2;
	return (easy << 4);
      }
    else if (coder == REAL_TYPE)
      return (4<<4);

  if (codel == REAL_TYPE)
    if (coder == REAL_TYPE)
      /* Shun converting between float and double if a choice exists.  */
      {
	if (TYPE_MODE (type) != TYPE_MODE (parmtype))
	  return (2<<4);
	return 0;
      }
    else if (coder == INTEGER_TYPE || coder == ENUMERAL_TYPE)
      return (4<<4);

  /* convert arrays which have not previously been converted.  */
  if (codel == ARRAY_TYPE)
    codel = POINTER_TYPE;
  if (coder == ARRAY_TYPE)
    coder = POINTER_TYPE;

  /* Conversions among pointers */
  if (codel == POINTER_TYPE && coder == POINTER_TYPE)
    {
      register tree ttl = TYPE_MAIN_VARIANT (TREE_TYPE (type));
      register tree ttr = TYPE_MAIN_VARIANT (TREE_TYPE (parmtype));
      int penalty = 4 * (ttl != ttr);
      /* Anything converts to void *.  void * converts to anything.
	 Since these may be `const void *' (etc.) use VOID_TYPE
	 instead of void_type_node.
	 Otherwise, the targets must be the same,
	 except that we do allow (at some cost) conversion
	 between signed and unsinged pointer types.  */

      if ((TREE_CODE (ttl) == METHOD_TYPE
	   || TREE_CODE (ttl) == FUNCTION_TYPE)
	  && TREE_CODE (ttl) == TREE_CODE (ttr))
	{
	  if (comptypes (ttl, ttr, -1))
	    return penalty<<4;
	  return 1;
	}

      if (!(TREE_CODE (ttl) == VOID_TYPE
	    || TREE_CODE (ttr) == VOID_TYPE
	    || (TREE_UNSIGNED (ttl) ^ TREE_UNSIGNED (ttr)
		&& (ttl = unsigned_type (ttl),
		    ttr = unsigned_type (ttr),
		    penalty = 10, 0))
	    || (comp_target_types (ttl, ttr, 0))))
	return 1;

      if (ttr == ttl)
	return 4;

      if (IS_AGGR_TYPE (ttl) && IS_AGGR_TYPE (ttr))
	{
	  int b_or_d = get_base_distance (ttl, ttr, 0, 0);
	  if (b_or_d < 0)
	    return 1;
	  return (b_or_d<<3) | 4;
	}

      return (penalty<<4);
    }

  if (codel == POINTER_TYPE && coder == INTEGER_TYPE)
    {
      /* This is not a bad match, but don't let it beat
	 integer-enum combinations.  */
      if (parm && integer_zerop (parm))
	return (4<<4);
    }

  /* C++: one of the types must be a reference type.  */
  {
    tree ttl, ttr;
    register tree intype = TYPE_MAIN_VARIANT (parmtype);
    register enum tree_code form = TREE_CODE (intype);
    int penalty;

    if (codel == REFERENCE_TYPE || coder == REFERENCE_TYPE)
      {
	ttl = TYPE_MAIN_VARIANT (type);

	if (codel == REFERENCE_TYPE)
	  {
	    ttl = TYPE_MAIN_VARIANT (TREE_TYPE (ttl));

	    if (form == OFFSET_TYPE)
	      {
		intype = TREE_TYPE (intype);
		form = TREE_CODE (intype);
	      }

	    if (form == REFERENCE_TYPE)
	      {
		intype = TYPE_MAIN_VARIANT (TREE_TYPE (intype));

		if (ttl == intype)
		  return 0;
		penalty = 2;
	      }
	    else
	      {
		/* Can reference be built up?  */
		if (ttl == intype)
		  {
		    return 0;
		  }
		else
		  penalty = 2;
	      }
	  }
	else if (form == REFERENCE_TYPE)
	  {
	    if (parm)
	      {
		tree tmp = convert_from_reference (parm);
		intype = TYPE_MAIN_VARIANT (TREE_TYPE (tmp));
	      }
	    else
	      {
		intype = parmtype;
		do
		  {
		    intype = TREE_TYPE (intype);
		  }
		while (TREE_CODE (intype) == REFERENCE_TYPE);
		intype = TYPE_MAIN_VARIANT (intype);
	      }

	    if (ttl == intype)
	      return 0;
	    else
	      penalty = 2;
	  }

	if (TREE_UNSIGNED (ttl) ^ TREE_UNSIGNED (intype))
	  {
	    ttl = unsigned_type (ttl);
	    intype = unsigned_type (intype);
	    penalty += 2;
	  }

	ttr = intype;

	/* If the initializer is not an lvalue, then it does not
	   matter if we make life easier for the programmer
	   by creating a temporary variable with which to
	   hold the result.  */
	if (parm && (coder == INTEGER_TYPE
		     || coder == ENUMERAL_TYPE
		     || coder == REAL_TYPE)
	    && ! lvalue_p (parm))
	  return (convert_harshness (ttl, ttr, 0) | (penalty << 4));

	if (ttl == ttr)
	  return 4;

	/* Pointers to voids always convert for pointers.  But
	   make them less natural than more specific matches.  */
	if (TREE_CODE (ttl) == POINTER_TYPE && TREE_CODE (ttr) == POINTER_TYPE)
	  if (TREE_TYPE (ttl) == void_type_node
	      || TREE_TYPE (ttr) == void_type_node)
	    return ((penalty+1)<<4);

	if (parm && codel != REFERENCE_TYPE)
	  return (convert_harshness (ttl, ttr, 0) | (penalty << 4));

	/* Here it does matter.  If this conversion is from
	   derived to base, allow it.  Otherwise, types must
	   be compatible in the strong sense.  */
	if (IS_AGGR_TYPE (ttl) && IS_AGGR_TYPE (ttr))
	  {
	    int b_or_d = get_base_distance (ttl, ttr, 0, 0);
	    if (b_or_d < 0)
	      return 1;
#if AMBIGUOUS_WORKING
	    if (ttl == TYPE_MAIN_VARIANT (type)
		&& TYPE_GETS_INIT_REF (type))
	      return (b_or_d<<3) | 6;
#endif
	    return (b_or_d<<3) | 4;
	  }

	if (comp_target_types (ttl, intype, 1))
	  return (penalty<<4);
      }
  }
  if (codel == RECORD_TYPE && coder == RECORD_TYPE)
    {
      int b_or_d = get_base_distance (type, parmtype, 0, 0);
      if (b_or_d < 0)
	return 1;
#if AMBIGUOUS_WORKING
      if (TYPE_GETS_INIT_REF (type))
	return (b_or_d<<3) | 6;
#endif
      return (b_or_d<<3) | 4;
    }
  return 1;
}

/* Algorithm: Start out with no stikes against.  For each argument
   which requires a (subjective) hard conversion (such as between
   floating point and integer), issue a strike.  If there are the same
   number of formal and actual parameters in the list, there will be at
   least on strike, otherwise an exact match would have been found.  If
   there are not the same number of arguments in the type lists, we are
   not dead yet: a `...' means that we can have more parms then were
   declared, and if we wind up in the default argument section of the
   list those can be used as well.  If an exact match could be found for
   one of those cases, return it immediately.  Otherwise, Rank the fields
   so that fields with fewer strikes are tried first.

   Conversions between builtin and user-defined types are allowed, but
   no function involving such a conversion is prefered to one which
   does not require such a conversion.  Furthermore, such conversions
   must be unique.  */

void
compute_conversion_costs (function, tta_in, cp, arglen)
     tree function;
     tree tta_in;
     struct candidate *cp;
     int arglen;
{
  tree ttf = TYPE_ARG_TYPES (TREE_TYPE (function));
  tree tta = tta_in;

  /* Start out with no strikes against.  */
  int evil_strikes = 0;
  int user_strikes = 0;
  int b_or_d_strikes = 0;
  int easy_strikes = 0;

  int strike_index = 0, win, lose;

  n_compute_conversion_costs++;

  cp->function = function;
  cp->arg = tta ? TREE_VALUE (tta) : NULL_TREE;
  cp->u.bad_arg = 0;		/* optimistic!  */

  bzero (cp->harshness, (arglen+1) * sizeof (short));

  while (ttf && tta)
    {
      int harshness;

      if (TREE_VALUE (ttf) == void_type_node)
	break;

      if (TREE_TYPE (TREE_VALUE (tta)) == unknown_type_node)
	{	  
	  /* Must perform some instantiation here.  */
	  tree rhs = TREE_VALUE (tta);
	  tree lhstype = TREE_VALUE (ttf);

	  /* @@ This is to undo what `grokdeclarator' does to
	     parameter types.  It really should go through
	     something more general.  */

	  TREE_TYPE (tta) = unknown_type_node;
	  if (TREE_CODE (rhs) == OP_IDENTIFIER)
	    rhs = build_instantiated_decl (lhstype, rhs);
	  else
	    rhs = instantiate_type (lhstype, rhs, 0);

	  if (TREE_CODE (rhs) == ERROR_MARK)
	    harshness = 1;
	  else
	    {
	      harshness = convert_harshness (lhstype, TREE_TYPE (rhs), rhs);
	      /* harshness |= 2; */
	    }
	}
      else
	harshness = convert_harshness (TREE_VALUE (ttf), TREE_TYPE (TREE_VALUE (tta)), TREE_VALUE (tta));

      cp->harshness[strike_index] = harshness;
      if (harshness & 1)
	{
	  cp->u.bad_arg = strike_index;
	  evil_strikes = 1;
	}
      else if (harshness & 2)
	{
	  user_strikes += 1;
	}
      else if (harshness & 4)
	{
	  b_or_d_strikes += (harshness >> 3);
	}
      else
	easy_strikes += harshness >> 4;
      ttf = TREE_CHAIN (ttf);
      tta = TREE_CHAIN (tta);
      strike_index += 1;
    }

  if (tta)
    {
      /* ran out of formals, and parmlist is fixed size.  */
      if (ttf /* == void_type_node */)
	{
	  cp->evil = 1;
	  cp->u.bad_arg = -1;
	  return;
	}
    }
  else if (ttf && TREE_VALUE (ttf) != void_type_node)
    {
      /* ran out of actuals, and no defaults.  */
      if (TREE_PURPOSE (ttf) == NULL_TREE)
	{
	  cp->evil = 1;
	  cp->u.bad_arg = -2;
	  return;
	}
      /* Store index of first default.  */
      cp->harshness[arglen] = strike_index+1;
    }
  else cp->harshness[arglen] = 0;

  /* Argument list lengths work out, so don't need to check them again.  */
  if (evil_strikes)
    {
      /* We do not check for derived->base conversions here, since in
	 no case would they give evil strike counts, unless such conversions
	 are somehow ambiguous.  */

      /* See if any user-defined conversions apply.
         But make sure that we do not loop.  */
      static int dont_convert_types = 0;

      if (dont_convert_types)
	{
	  cp->evil = 1;
	  return;
	}

      win = 0;			/* Only get one chance to win.  */
      ttf = TYPE_ARG_TYPES (TREE_TYPE (function));
      tta = tta_in;
      strike_index = 0;
      evil_strikes = 0;

      while (ttf && tta)
	{
	  if (TREE_VALUE (ttf) == void_type_node)
	    break;

	  lose = cp->harshness[strike_index];
	  if (lose&1)
	    {
	      tree actual_type = TREE_TYPE (TREE_VALUE (tta));
	      tree formal_type = TREE_VALUE (ttf);

	      dont_convert_types = 1;

	      if (TREE_CODE (formal_type) == REFERENCE_TYPE)
		formal_type = TREE_TYPE (formal_type);
	      if (TREE_CODE (actual_type) == REFERENCE_TYPE)
		actual_type = TREE_TYPE (actual_type);

	      formal_type = TYPE_MAIN_VARIANT (formal_type);
	      actual_type = TYPE_MAIN_VARIANT (actual_type);

	      if (TYPE_HAS_CONSTRUCTOR (formal_type))
		{
		  /* If it has a constructor for this type, try to use it.  */
		  if (convert_to_aggr (formal_type, TREE_VALUE (tta), 0, 1)
		      != error_mark_node)
		    {
		      /* @@ There is no way to save this result yet.
		         @@ So success is NULL_TREE for now.  */
		      win++;
		    }
		}
	      if (IS_AGGR_TYPE (actual_type) && TYPE_HAS_CONVERSION (actual_type))
		{
		  if (TREE_CODE (formal_type) == INTEGER_TYPE
		      && TYPE_HAS_INT_CONVERSION (actual_type))
		    win++;
		  else if (TREE_CODE (formal_type) == REAL_TYPE
			   && TYPE_HAS_REAL_CONVERSION (actual_type))
		    win++;
		  else if (build_type_conversion (formal_type, TREE_VALUE (tta), 0))
		    win++;
		}

	      dont_convert_types = 0;

	      if (win == 1)
		{
		  user_strikes += 1;
		  cp->harshness[strike_index] = 2;
		  win = 0;
		}
	      else
		{
		  if (cp->u.bad_arg > strike_index)
		    cp->u.bad_arg = strike_index;

		  evil_strikes = win ? 2 : 1;
		  break;
		}
	    }

	  ttf = TREE_CHAIN (ttf);
	  tta = TREE_CHAIN (tta);
	  strike_index += 1;
	}
    }
  cp->evil = evil_strikes;
  cp->user = user_strikes;
  cp->b_or_d = b_or_d_strikes;
  cp->easy = easy_strikes;
}

struct candidate *
ideal_candidate (basetype, candidates, n_candidates, parms, len)
     tree basetype;
     struct candidate *candidates;
     int n_candidates;
     tree parms;
     int len;
{
  struct candidate *cp = candidates + n_candidates;
  qsort (candidates,		/* char *base */
	 n_candidates,		/* int nel */
	 sizeof (struct candidate), /* int width */
	 rank_for_overload);	/* int (*compar)() */

  /* If the best two candidates we find require user-defined
     conversions, we may need to report and error message.  */
  if (cp[-1].user && cp[-2].user)
    {
      /* If the best two methods found involved user-defined
	 type conversions, then we must see whether one
	 of them is exactly what we wanted.  If not, then
	 we have an ambiguity.  */
      int i = 0, index, best = 0;
      tree tta = parms;
      tree ttf;

#if AMBIGUOUS_WORKING
      if (cp[-1].b_or_d == 0
	  && cp[-1].easy == 0
	  && (cp[-2].b_or_d | cp[-2].easy) > 0)
	return cp - 1;
#endif

      /* Stash all of our parameters in safe places
	 so that we can perform type conversions in place.  */
      while (tta)
	{
	  TREE_PURPOSE (tta) = TREE_VALUE (tta);
	  tta = TREE_CHAIN (tta);
	}

      do
	{
	  int exact_conversions = 0;

	  i -= 1;
	  tta = parms;
	  if (DECL_STATIC_FUNCTION_P (cp[i].function))
	    tta = TREE_CHAIN (tta);
	  for (ttf = TYPE_ARG_TYPES (TREE_TYPE (cp[i].function)), index = 0;
	       index < len;
	       tta = TREE_CHAIN (tta), ttf = TREE_CHAIN (ttf), index++)
	    {
	      if (cp[i].harshness[index] & 2)
		{
		  TREE_VALUE (tta)
		    = build_type_conversion (TREE_VALUE (ttf), TREE_PURPOSE (tta), 2);
		  if (TREE_VALUE (tta))
		    {
		      if (TREE_CODE (TREE_VALUE (tta)) != CONVERT_EXPR
			  && (TREE_CODE (TREE_VALUE (tta)) != NOP_EXPR
			      || comp_target_types (TREE_TYPE (TREE_VALUE (tta)),
						    TREE_TYPE (TREE_OPERAND (TREE_VALUE (tta), 0)), 1)))
			exact_conversions += 1;
		    }
		  else if (IS_AGGR_TYPE (TREE_VALUE (ttf))
			   || (TREE_CODE (TREE_VALUE (ttf)) == REFERENCE_TYPE
			       && IS_AGGR_TYPE (TREE_TYPE (TREE_VALUE (ttf)))))
		    {
		      /* To get here we had to have succeeded via
			 a constructor.  */
		      TREE_VALUE (tta) = TREE_PURPOSE (tta);
		      exact_conversions += 1;
		    }
		}
	    }
	  if (exact_conversions == cp[i].user)
	    {
	      if (best == 0)
		best = i;
	      else
		return 0;
	    }
	} while (cp + i != candidates);

      if (best)
	{
	  int exact_conversions = cp[best].user;
	  tta = parms;
	  if (DECL_STATIC_FUNCTION_P (cp[best].function))
	    tta = TREE_CHAIN (parms);
	  for (ttf = TYPE_ARG_TYPES (TREE_TYPE (cp[best].function)), index = 0;
	       exact_conversions > 0;
	       tta = TREE_CHAIN (tta), ttf = TREE_CHAIN (ttf), index++)
	    if (cp[best].harshness[index] & 2)
	      {
		/* We must now fill in the slot we left behind.
		   @@ This could be optimized to use the value previously
		   @@ computed by build_type_conversion in some cases.  */
		TREE_VALUE (tta) = convert (TREE_VALUE (ttf), TREE_PURPOSE (tta));
		exact_conversions -= 1;
	      }
	  return cp + best;
	}
      return 0;
    }
  /* If the best two candidates we find both use default parameters,
     we may need to report and error.  Don't need to worry if next-best
     candidate is forced to use user-defined conversion when best is not.  */
  if (cp[-2].user == 0
      && cp[-1].harshness[len] != 0 && cp[-2].harshness[len] != 0)
    {
      tree tt1 = TYPE_ARG_TYPES (TREE_TYPE (cp[-1].u.field));
      tree tt2 = TYPE_ARG_TYPES (TREE_TYPE (cp[-2].u.field));
      int i = cp[-1].harshness[len];
      if (cp[-2].harshness[len] < i)
	i = cp[-2].harshness[len];
      while (--i > 0)
	{
	  if (TYPE_MAIN_VARIANT (TREE_VALUE (tt1))
	      != TYPE_MAIN_VARIANT (TREE_VALUE (tt2)))
	    /* These lists are not identical, so we can choose our best candidate.  */
	    return cp - 1;
	  tt1 = TREE_CHAIN (tt1);
	  tt2 = TREE_CHAIN (tt2);
	}
      /* To get here, both lists had the same parameters up to the defaults
	 which were used.  This is an ambiguous request.  */
      return 0;
    }

  /* Otherwise, return our best candidate.  Note that if we get candidates
     from independent base classes, we have an ambiguity, even if one
     argument list look a little better than another one.  */
  if (basetype && TYPE_USES_MULTIPLE_INHERITANCE (basetype))
    {
      int i = n_candidates - 1, best;
      tree base1 = NULL_TREE, base2 = NULL_TREE;

      if (TREE_CODE (TREE_TYPE (candidates[i].function)) == FUNCTION_TYPE)
	return cp - 1;

      for (; i >= 0 && candidates[i].user == 0 && candidates[i].evil == 0; i--)
	{
	  if (TREE_CODE (TREE_TYPE (candidates[i].function)) == METHOD_TYPE)
	    {
	      tree newbase = TYPE_METHOD_BASETYPE (TREE_TYPE (candidates[i].function));

	      if (base1 != NULL_TREE)
		{
		  if (newbase != base1
		      && ! get_base_type (newbase, base1, 0))
		    {
		      error ("ambiguous request for function from distinct base classes of type `%s'", TYPE_NAME_STRING (basetype));
		      error ("first candidate is `%s'", fndecl_as_string (0, 0, candidates[best].function, 1));
		      error ("second candidates is `%s'", fndecl_as_string (0, 0, candidates[i].function, 1));
		      return cp - 1;
		    }
		}
	      else
		{
		  best = i;
		  base1 = newbase;
		}
	    }
	  else return cp - 1;
	}
    }

#if AMBIGUOUS_WORKING
  if (cp[-1].user == cp[-2].user
      && cp[-1].b_or_d == cp[-2].b_or_d
      && cp[-1].easy == cp[-2].easy)
    return 0;
#endif

  return cp - 1;
}

/* Assume that if the class referred to is not in the
   current class hierarchy, that it may be remote.
   PARENT is assumed to be of aggregate type here.  */
static int
may_be_remote (parent)
     tree parent;
{
  if (TYPE_OVERLOADS_METHOD_CALL_EXPR (parent) == 0)
    return 0;

  if (current_class_type == NULL_TREE)
    return 0;
  if (parent == current_class_type)
    return 0;

  if (get_base_type (parent, current_class_type, 0))
    return 0;
  return 1;
}

/* Return the number of bytes that the arglist in PARMS would
   occupy on the stack.  */
int
get_arglist_len_in_bytes (parms)
     tree parms;
{
  register tree parm;
  register int bytecount = 0;

  for (parm = parms; parm; parm = TREE_CHAIN (parm))
    {
      register tree pval = TREE_VALUE (parm);
      register int used, size;

      if (TREE_CODE (pval) == ERROR_MARK)
	continue;
      else if (TYPE_MODE (TREE_TYPE (pval)) != BLKmode)
	{
	  used = size = GET_MODE_SIZE (TYPE_MODE (TREE_TYPE (pval)));
#ifdef PUSH_ROUNDING
	  size = PUSH_ROUNDING (size);
#endif
	  used = (((size + PARM_BOUNDARY / BITS_PER_UNIT - 1)
		   / (PARM_BOUNDARY / BITS_PER_UNIT))
		  * (PARM_BOUNDARY / BITS_PER_UNIT));
	}
      else
	{
	  register tree size = size_in_bytes (TREE_TYPE (pval));
	  register tree used_t = convert_units (convert_units (size, BITS_PER_UNIT, PARM_BOUNDARY),
						PARM_BOUNDARY, BITS_PER_UNIT);
	  used = TREE_INT_CST_LOW (used_t);
	}
      bytecount += used;
    }
  return bytecount;
}

tree
build_vfield_ref (datum, type)
     tree datum, type;
{
  if (TREE_CODE (TREE_TYPE (datum)) == REFERENCE_TYPE)
    datum = convert_from_reference (datum);
  if (datum == error_mark_node)
    return error_mark_node;

  if (! TYPE_USES_VIRTUAL_BASECLASSES (type))
    return build (COMPONENT_REF, TREE_TYPE (CLASSTYPE_VFIELD (type)),
		  datum, CLASSTYPE_VFIELD (type));
  return build_component_ref (datum, DECL_NAME (CLASSTYPE_VFIELD (type)), 0, 0);
}

/* Build something of the form ptr->method (args)
   or object.method (args).  This can also build
   calls to constructors, and find friends.

   Member functions always take their class variable
   as a pointer.

   INSTANCE is a class instance.

   NAME is the NAME field of the struct, union, or class
   whose type is that of INSTANCE.

   PARMS help to figure out what that NAME really refers to.

   BASETYPE_PATH, if non-NULL, tells which basetypes of INSTANCE
   we should be traversed before starting our search.  We need
   this information to get protected accesses correct.

   FLAGS is the logical disjunction of zero or more LOOKUP_
   flags.  See cplus-tree.h for more information about these
   flags.

   If this is all OK, calls build_function_call with the resolved
   member function.

   This function must also handle being called to perform
   initialization, promotion/coercion of arguments, and
   instantiation of default parameters.

   Note that NAME may refer to an instance variable name.  If
   `operator()()' is defined for the type of that field, then we return
   that result.

   When converting the class instance variable from REFERENCE_TYPE to
   POINTER_TYPE, we must do so with a REFERENCE_EXPR, *not* a NOP_EXPR.
   This is because the conversion is for initialization.  */
tree
build_method_call (instance, name, parms, basetype_path, flags)
     tree instance, name, parms, basetype_path;
     int flags;
{
  register tree function, fntype, value_type;
  register tree classname, basetype, save_basetype;
  register enum tree_code form;
  register tree fields, result, method_name, parmtypes, parm;
  int pass;
  enum visibility_type visibility;
  int rank_for_overload ();
  int need_vtbl = 0;
  char *err_name;
  char *name_kind;
  int ever_seen = 0;
  int wrap;
  tree wrap_type;
  tree instance_ptr = NULL_TREE;
  int all_virtual = flag_all_virtual;
  int static_call_context
    = ! (instance == NULL_TREE
	 || TREE_CODE (instance) != NOP_EXPR
	 || TREE_OPERAND (instance, 0) != error_mark_node);
  tree saw_private = 0;
  tree saw_protected = 0;
#ifdef SOS
  /* If call is a call to a constructor, then `dtbl'
     will first be initialized with the function table pointer
     of the appropriate type (calling "sosFindCode" as a last
     resort), the the call to the constructor will go through there.  */
  tree dtbl = (flags & LOOKUP_DYNAMIC) ? TREE_VALUE (parms) : NULL_TREE;

  /* Flag saying whether or not `dtbl' has been inserted into the
     parameter list.  This is needed because we cannot tell (until
     we have a match) whether this parameter should go in or not.

     If 1, then `dtbl' is living naturally.
     If 0, then `dtbl' is not among the parms that we know about.
     If -1, the `dtbl' was place into the parms unnaturally.

     Note that we may side-effect the parameter list, but in such a way
     that the caller of this function would never know.  */
  int dtbl_inserted = (flags & LOOKUP_DYNAMIC);
#endif

  n_build_method_call++;

  if (instance == error_mark_node
      || name == error_mark_node
      || parms == error_mark_node
      || (instance != 0 && TREE_TYPE (instance) == error_mark_node))
    return error_mark_node;

  if (TREE_CODE (name) == WRAPPER_EXPR)
    {
      wrap_type = TREE_OPERAND (name, 0);
      name = TREE_OPERAND (name, 1);
      wrap = 1;
    }
  else if (TREE_CODE (name) == ANTI_WRAPPER_EXPR)
    {
      wrap_type = TREE_OPERAND (name, 0);
      name = TREE_OPERAND (name, 1);
      wrap = -1;
    }
  else
    {
      wrap_type = NULL_TREE;
      wrap = 0;
    }

  /* Initialize name for error reporting.  */
  if (TREE_CODE (name) == OP_IDENTIFIER)
    name = build_operator_fnname (name, parms, 1);

  if (OPERATOR_NAME_P (name))
    {
      err_name = (char *)alloca (OVERLOAD_MAX_LEN);
      sprintf (err_name, "operator %s", operator_name_string (name));
    }
  else if (name == wrapper_name)
    err_name = "wrapper";
  else if (OPERATOR_TYPENAME_P (name))
    err_name = "type conversion operator";
  else
    err_name = IDENTIFIER_POINTER (name);

  if (wrap)
    {
      char *p = (char *)alloca (OVERLOAD_MAX_LEN);
      sprintf (p, "%s for `%s'", wrap < 0 ? "anti-wrapper" : "wrapper", err_name);
      err_name = p;
    }

  if (instance == NULL_TREE)
    {
      /* call to a constructor... */
      if (TREE_TYPE (name))
	basetype = TREE_TYPE (TREE_TYPE (name));
      else
	{
	  tree typedef_name = lookup_name (name);
	  if (typedef_name && TREE_CODE (typedef_name) == TYPE_DECL)
	    {
	      /* Cannonicalize the typedef name.  */
	      basetype = TREE_TYPE (typedef_name);
	      name = DECL_NAME (TYPE_NAME (basetype));
	    }
	  else
	    {
	      error ("no constructor named `%s' in visible scope",
		     IDENTIFIER_POINTER (name));
	      return error_mark_node;
	    }
	}
      if (wrap_type && wrap_type != basetype)
	{
	  error_with_aggr_type (wrap_type, "invalid constructor `%s::%s'",
				TYPE_NAME_STRING (basetype));
	  return error_mark_node;
	}
      if (TYPE_VIRTUAL_P (basetype))
	{
	  wrap_type = basetype;
	}

      form = TREE_CODE (basetype);

      if (! IS_AGGR_TYPE_CODE (form))
	{
	non_aggr_error:
	  if ((flags & LOOKUP_COMPLAIN) && form != ERROR_MARK)
	    error ("request for member `%s' in something not a structure or union", err_name);

	  return error_mark_node;
	}
    }
  else if (instance == C_C_D || instance == current_class_decl)
    {
      /* Check to see if we really have a reference to an instance variable
	 with `operator()()' overloaded.  */
#if 1
      tree field = IDENTIFIER_CLASS_VALUE (name);
#else
      tree field = identifier_class_value (name);
#endif

      if (field == NULL_TREE)
	{
	  error ("`this' has no member named `%s'", err_name);
	  return error_mark_node;
	}

      if (TREE_CODE (field) == FIELD_DECL)
	{
	  /* If it's a field, try overloading operator (),
	     or calling if the field is a pointer-to-function.  */
	  instance = build_component_ref_1 (C_C_D, field, 0, 1);
	  if (instance == error_mark_node)
	    return error_mark_node;

	  if (IS_AGGR_TYPE (TREE_TYPE (instance))
	      && TYPE_OVERLOADS_CALL_EXPR (TREE_TYPE (instance)))
	    return build_opfncall (CALL_EXPR, instance, parms);

	  if (TREE_CODE (TREE_TYPE (instance)) == POINTER_TYPE)
	    if (TREE_CODE (TREE_TYPE (TREE_TYPE (instance))) == FUNCTION_TYPE)
	      return build_function_call (instance, parms);
	    else if (TREE_CODE (TREE_TYPE (TREE_TYPE (instance))) == METHOD_TYPE)
	      return build_function_call (instance, tree_cons (NULL_TREE, current_class_decl, parms));
	}

      /* When doing initialization, we side-effect the TREE_TYPE of
	 C_C_D, hence we cannot set up BASETYPE from CURRENT_CLASS_TYPE.  */

      instance = C_C_D;
      basetype = TREE_TYPE (instance);
      instance_ptr = current_class_decl;
      if (!(flags & LOOKUP_NONVIRTUAL) && TYPE_VIRTUAL_P (basetype)
	  && wrap_type == NULL_TREE)
	need_vtbl = 1;
    }
  else if (TREE_CODE (instance) == RESULT_DECL)
    {
      basetype = TYPE_MAIN_VARIANT (TREE_TYPE (instance));
      if (wrap_type)
	{
	  if (basetype == wrap_type || get_base_type (basetype, wrap_type, 0))
	    basetype = wrap_type;
	  else
	    {
	      error_not_derived_type (basetype, wrap_type);
	      return error_mark_node;
	    }
	}
      /* Should we ever have to make a virtual function reference
	 from a RESULT_DECL, know that it must be of fixed type
	 within the scope of this function.  */
      else if (!(flags & LOOKUP_NONVIRTUAL) && TYPE_VIRTUAL_P (basetype))
	need_vtbl = 1;
      instance_ptr = build_unary_op (ADDR_EXPR, instance, 0);
    }
  else
    {
      /* The MAIN_VARIANT of the type that `instance_ptr' winds up being.  */
      tree inst_ptr_basetype;

      /* from the file "cplus-typeck.c".  */
      extern tree unary_complex_lvalue ();

      /* the base type of an instance variable is pointer to class */
      basetype = TYPE_MAIN_VARIANT (TREE_TYPE (instance));

      if (basetype == error_mark_node)
	return error_mark_node;

      if (TREE_CODE (basetype) == REFERENCE_TYPE)
	{
	  basetype = TYPE_MAIN_VARIANT (TREE_TYPE (basetype));
	  if (! IS_AGGR_TYPE (basetype))
	    goto non_aggr_error;
	  /* Call to convert not needed because we are remaining
	     within the same type.  */
#if 0
	  instance_ptr = build (REFERENCE_EXPR, TYPE_POINTER_TO (basetype), instance);
#else
	  instance_ptr = build (NOP_EXPR, TYPE_POINTER_TO (basetype), instance);
#endif
	  inst_ptr_basetype = basetype;
	}
      else
	{
	  if (TREE_CODE (basetype) == POINTER_TYPE)
	    {
	      basetype = TREE_TYPE (basetype);
	      instance_ptr = instance;
	    }

	  if (! IS_AGGR_TYPE (basetype))
	    goto non_aggr_error;

	  if (! instance_ptr)
	    {
	      if ((lvalue_p (instance)
		   && (instance_ptr = build_unary_op (ADDR_EXPR, instance, 0)))
		  || (instance_ptr = unary_complex_lvalue (ADDR_EXPR, instance)))
		{
		  if (instance_ptr == error_mark_node)
		    return error_mark_node;
		}
	      else if (TREE_CODE (instance) == NOP_EXPR)
		{
		  /* A cast is not an lvalue.  Initialize a fresh temp
		     with the value we are casting from, and proceed with
		     that temporary.  We can't cast to a reference type,
		     so that simplifies the initialization to something
		     we can manage.  */
		  tree temp = get_temp_name (TYPE_MAIN_VARIANT (TREE_TYPE (instance)));
		  if (IS_AGGR_TYPE (TREE_TYPE (instance)))
		    expand_aggr_init (temp, instance, 0);
		  else
		    {
		      store_init_value (temp, instance);
		      expand_decl_init (temp);
		    }
		  instance = temp;
		  instance_ptr = build_unary_op (ADDR_EXPR, instance, 0);
		}
	      else
		{
		  assert (TREE_CODE (instance) == CALL_EXPR);
		  if (TYPE_NEEDS_CONSTRUCTOR (basetype))
		    instance = build (NEW_EXPR, basetype,
				      TREE_OPERAND (instance, 0),
				      TREE_OPERAND (instance, 1));
		  else
		    instance = get_temp_name (basetype);
		  TREE_ADDRESSABLE (instance) = 1;
		  instance_ptr = build_unary_op (ADDR_EXPR, instance, 0);
		}
	      /* @@ Should we call comp_target_types here?  */
	      inst_ptr_basetype = TYPE_MAIN_VARIANT (TREE_TYPE (TREE_TYPE (instance_ptr)));
	      if (basetype != inst_ptr_basetype)
		instance_ptr = convert (TYPE_POINTER_TO (basetype), instance_ptr);
	    }
	  else
	    inst_ptr_basetype = TYPE_MAIN_VARIANT (TREE_TYPE (TREE_TYPE (instance_ptr)));
	}

      if (basetype_path == NULL_TREE)
	basetype_path = CLASSTYPE_AS_LIST (inst_ptr_basetype);
      {
	/* Check to see if this is not really a reference to an instance variable
	   with `operator()()' overloaded.  */
	tree field = lookup_field (basetype_path, name, 1);

	/* This can happen if the reference was ambiguous
	   or for visibility violations.  */
	if (field == error_mark_node)
	  return error_mark_node;

	if (field && IS_AGGR_TYPE (TREE_TYPE (field))
	    && TYPE_OVERLOADS_CALL_EXPR (TREE_TYPE (field)))
	  {
	    /* Make the next search for this field very short.  */
	    basetype = DECL_FIELD_CONTEXT (field);
	    instance_ptr = convert_to_nonzero_pointer (TYPE_POINTER_TO (basetype),
						       instance_ptr);

	    instance = build_indirect_ref (instance_ptr, 0);
	    return build_opfncall (CALL_EXPR,
				   build_component_ref_1 (instance, field, 0, 0),
				   parms);
	  }
	if (field && TREE_CODE (TREE_TYPE (field)) == POINTER_TYPE)
	  if (TREE_CODE (TREE_TYPE (TREE_TYPE (field))) == FUNCTION_TYPE
	      || TREE_CODE (TREE_TYPE (TREE_TYPE (field))) == METHOD_TYPE)
	  {
	    /* This is a member which is a pointer to function.  */
	    tree ref = build_component_ref_1 (build_indirect_ref (instance_ptr, 0, 0),
					      field, flags & LOOKUP_COMPLAIN);
	    if (ref == error_mark_node)
	      return error_mark_node;
	    return build_function_call (ref, parms);
	  }
	else if (TREE_CODE (TREE_TYPE (TREE_TYPE (field))) == METHOD_TYPE)
	  {
	    error ("invalid call via pointer-to-member function");
	    return error_mark_node;
	  }
      }

      if (wrap_type)
	{
	  if (basetype == wrap_type || get_base_type (basetype, wrap_type, 0))
	    basetype = wrap_type;
	  else
	    {
	      error_not_derived_type (basetype, wrap_type);
	      return error_mark_node;
	    }
	}
      else if (!(flags & LOOKUP_NONVIRTUAL) && TYPE_VIRTUAL_P (basetype))
	{
	  tree ref = instance_ptr;

	  /* No need to make a SAVE_EXPR for really easy things.  */
	  if (TREE_CODE (ref) == ADDR_EXPR)
	    ref = TREE_OPERAND (ref, 0);
	  while (TREE_CODE (ref) == NOP_EXPR
		 /* @@ How might stripping REFERENCE_EXPRs
		    @@ do us in?  */
		 || TREE_CODE (ref) == REFERENCE_EXPR)
	    ref = TREE_OPERAND (ref, 0);

	  if (TREE_CODE (ref) != PARM_DECL && TREE_CODE (ref) != VAR_DECL)
	    {
	      /* This action is needed because the instance is needed
		 for providing the base of the virtual function table.
		 Without using a SAVE_EXPR, the function we are building
		 may be called twice, or side effects on the instance
		 variable (such as a post-increment), may happen twice.  */
	      ref = instance_ptr;

	      if (TREE_CODE (TREE_TYPE (ref)) == REFERENCE_TYPE)
		{
#if 0
		  ref = build (REFERENCE_EXPR, build_pointer_type (TREE_TYPE (TREE_TYPE (ref))), ref);
#else
		  ref = build (NOP_EXPR, build_pointer_type (TREE_TYPE (TREE_TYPE (ref))), ref);
#endif
		  /* Don't need to do a save_expr on references,
		     because they cannot change from under us.  */
		}
	      else ref = save_expr (ref);

	      if (TREE_TYPE (ref) == TREE_TYPE (instance_ptr))
		instance_ptr = ref;
	      else
		instance_ptr = convert_to_nonzero_pointer (TREE_TYPE (instance_ptr), ref);
	      instance = instance_ptr;
	    }

	  if (TREE_CODE (TREE_TYPE (instance)) == POINTER_TYPE)
	    instance = build_indirect_ref (instance, 0);

	  need_vtbl = 1;
	}
    }

  /* Are we building a non-virtual wrapper?  */
  if (flags & LOOKUP_NONVIRTUAL)
    {
      if (all_virtual)
	sorry ("non-virtual call with -fall-virtual");
      if (wrap)
	wrap_type = basetype;
    }

  save_basetype = basetype;

  if (all_virtual == 1
      && (! strncmp (IDENTIFIER_POINTER (name), "op$method_call", 13)
	  || instance_ptr == NULL_TREE
	  || (TYPE_OVERLOADS_METHOD_CALL_EXPR (basetype) == 0
	      && TYPE_NEEDS_WRAPPER (basetype) == 0)))
    all_virtual = 0;
		  
  if (TYPE_SIZE (save_basetype) == 0)
    {
      /* This is worth complaining about, I think.  */
      error_with_aggr_type (save_basetype, "cannot lookup method in incomplete type `%s'");
      return error_mark_node;
    }

  parmtypes = build_tree_list (NULL_TREE, TYPE_POINTER_TO (basetype));
  for (parm = parms; parm; parm = TREE_CHAIN (parm))
    {
      tree t = TREE_TYPE (TREE_VALUE (parm));
      if (t == error_mark_node)
	return error_mark_node;

      if (TREE_CODE (t) == ARRAY_TYPE || TREE_CODE (t) == OFFSET_TYPE)
	{
	  /* Perform the conversion from ARRAY_TYPE to POINTER_TYPE in place.
	     Also convert OFFSET_TYPE entities to their normal selves.
	     This eliminates needless calls to `compute_conversion_costs'.  */
	  TREE_VALUE (parm) = default_conversion (TREE_VALUE (parm));
	  t = TREE_TYPE (TREE_VALUE (parm));
	}
      parmtypes = chainon (parmtypes, build_tree_list (NULL_TREE, t));
    }

  if (instance)
    {
      parms = tree_cons (NULL_TREE, instance_ptr, parms);
    }
  else
    {
#if 0
      parms = tree_cons (NULL_TREE, build (REFERENCE_EXPR, TYPE_POINTER_TO (basetype), integer_zero_node), parms);
#else
      parms = tree_cons (NULL_TREE, build (NOP_EXPR, TYPE_POINTER_TO (basetype), integer_zero_node), parms);
#endif
    }

  /* Look up function name in the structure type definition.  */

  if (wrap)
    {
      if (wrap > 0)
	name_kind = "wrapper";
      else
	name_kind = "anti-wrapper";
      fields = get_wrapper (basetype);
    }
  else
    {
      if (TREE_TYPE (name) && ! OPERATOR_TYPENAME_P (name))
	{
	  tree tmp = NULL_TREE;
	  if (TREE_TYPE (name) == TYPE_NAME (basetype))
	    tmp = basetype;
	  else
	    tmp = get_base_type (TREE_TYPE (TREE_TYPE (name)), basetype, 0);
	  if (tmp != 0)
	    {
	      name_kind = "constructor";
#ifdef SOS
	      if (TYPE_DYNAMIC (basetype) && dtbl_inserted == 0)
		{
		  tree parm, parmtype;
		  dtbl = get_sos_dtable (basetype);
		  parm = tree_cons (NULL_TREE, dtbl, TREE_CHAIN (parms));
		  parmtype = tree_cons (NULL_TREE, build_pointer_type (ptr_type_node), TREE_CHAIN (parmtypes));
		  TREE_CHAIN (parms) = parm;
		  TREE_CHAIN (parmtypes) = parmtype;
		  dtbl_inserted = -1;
		}
#endif
	      /* constructors are in very specific places.  */
#ifdef SOS
	      if (dtbl_inserted == -1)
		{
		  TREE_CHAIN (parmtypes) = TREE_CHAIN (TREE_CHAIN (parmtypes));
		  TREE_CHAIN (parms) = TREE_CHAIN (TREE_CHAIN (parms));
		  dtbl_inserted = 0;
		}
#endif
	      basetype = tmp;
	    }
	  else
	    name_kind = "method";
	}
      else name_kind = "method";

      if (basetype_path == NULL_TREE)
	basetype_path = CLASSTYPE_AS_LIST (basetype);
      result = lookup_fnfields (basetype_path, name,
				(flags & LOOKUP_COMPLAIN));
      if (result == error_mark_node)
	return error_mark_node;
    }

  /* Now, go look for this method name.  We do not find destructors here.  */
  method_name = build_decl_overload (IDENTIFIER_POINTER (name), parmtypes);

  for (pass = 0; pass < 2; pass++)
    {
      struct candidate *candidates;
      struct candidate *cp;
      int len;

      /* This increments every time we go up the type hierarchy.
	 The idea is to prefer a function of the derived class if possible.  */
      int b_or_d;

      fields = result;

      if (pass == 1)
	{
	  candidates = (struct candidate *) alloca ((ever_seen+1) * sizeof (struct candidate));
	  cp = candidates;
	  len = list_length (parms);
	  b_or_d = 0;

	  /* First see if a global function has a shot at it.  */
	  if (flags & LOOKUP_GLOBAL)
	    {
	      tree friend_parms;
	      tree parm = TREE_VALUE (parms);

	      if (TREE_CODE (TREE_TYPE (parm)) == REFERENCE_TYPE)
		friend_parms = parms;
	      else if (TREE_CODE (TREE_TYPE (parm)) == POINTER_TYPE)
		{
		  parm = build_indirect_ref (parm, "friendifying parms (compiler error)");
		  parm = convert (build_reference_type (TREE_TYPE (parm)), parm);
		  friend_parms = tree_cons (NULL_TREE, parm, TREE_CHAIN (parms));
		}
	      else
		assert (0);

	      result = do_actual_overload (name, friend_parms, 0, cp);
	      /* If it turns out to be the one we were actually looking for
		 (it was probably a friend function), the return the
		 good result.  */
	      if (TREE_CODE (result) == CALL_EXPR)
		return result;

	      while (cp->evil == 0)
		{
		  /* non-standard uses: set the field to 0 to indicate
		     we are using a non-member function.  */
		  cp->u.field = 0;
		  cp += 1;
		}
	    }
	}

      while (fields)
	{
	  /* We have a hit (of sorts). If the parameter list is
	     "error_mark_node", or some variant thereof, it won't
	     match any methods. Since we have verified that the is
	     some method vaguely matching this one (in name at least),
	     silently return.
	     
	     Don't stop for friends, however.  */
	  tree baselink = fields;
	  tree field = TREE_VALUE (TREE_VALUE (baselink));
	  tree basetypes = TREE_PURPOSE (baselink);

	  basetype = TREE_VALUE (basetypes);

	  /* Cast the instance variable to the approriate type.  */
	  TREE_VALUE (parmtypes) = TYPE_POINTER_TO (basetype);

	  if (DESTRUCTOR_NAME_P (DECL_NAME (field)))
	    field = TREE_CHAIN (field);

	  while (field)
	    {
	      n_inner_fields_searched++;
	      ever_seen++;

	      /* Not looking for friends here.  */
	      if (TREE_CODE (TREE_TYPE (field)) == FUNCTION_TYPE
		  && ! DECL_STATIC_FUNCTION_P (field))
		{
		  field = TREE_CHAIN (field);
		  continue;
		}

	      if (pass == 0
		  && DECL_NAME (field) == method_name)
		{
		  if (flags & LOOKUP_PROTECT)
		    {
		      visibility = compute_visibility (basetypes, field);
		      if (visibility == visibility_protected
			  && flags & LOOKUP_PROTECTED_OK)
			visibility = visibility_public;
		    }

		  if (instance_ptr && ! DECL_STATIC_FUNCTION_P (field))
		    {
		      instance_ptr = convert_to_nonzero_pointer (TYPE_POINTER_TO (DECL_VCONTEXT (field)), instance_ptr);
		      if (! static_call_context)
			instance_ptr = save_expr (instance_ptr);
		      TREE_VALUE (parms) = instance_ptr;
		      instance = build_indirect_ref (instance_ptr);
		    }

		  function = field;
		  if ((flags & LOOKUP_PROTECT) == 0
		      || visibility == visibility_public)
		    goto found_and_ok;
		  else if (visibility == visibility_private)
		    saw_private = field;
		  else if (visibility == visibility_protected)
		    saw_protected = field;
		  /* If we fail on the exact match, we have
		     an immediate failure.  */
		  goto found;
		}
	      if (pass != 0)
		{
		  tree these_parms = parms;

		  n_inner_fields_searched++;
		  function = field;
		  cp->harshness
		    = (unsigned short *)alloca ((len+1) * sizeof (short));
		  if (DECL_STATIC_FUNCTION_P (field))
		    these_parms = TREE_CHAIN (these_parms);
		  compute_conversion_costs (function, these_parms, cp, len);
		  cp->b_or_d += b_or_d;
		  if (cp->evil == 0)
		    {
		      cp->u.field = field;
		      cp->function = field;
		      if (flags & LOOKUP_PROTECT)
			{
			  enum visibility_type this_v;
			  this_v = compute_visibility (basetypes, field);
			  if (this_v == visibility_protected
			      && (flags & LOOKUP_PROTECTED_OK))
			    this_v = visibility_public;
			  if (this_v != visibility_public)
			    {
			      if (this_v == visibility_private)
				saw_private = field;
			      else
				saw_protected = field;
			      /* Initialize these fields for error reporting.  */
			      cp->u.field = field;
			      cp->function = field;
			      field = TREE_CHAIN (field);
			      continue;
			    }
			}
		      /* If we used default parameters, we must
			 check to see whether anyone else might
			 use them also, and report a possible
			 ambiguity.  */
		      if (! TYPE_USES_MULTIPLE_INHERITANCE (save_basetype)
			  && cp->harshness[len] == 0
			  && cp->user == 0 && cp->b_or_d == 0
			  && cp->easy <= 1)
			{
			  if (! DECL_STATIC_FUNCTION_P (function))
			    TREE_VALUE (parms) = cp->arg;
			  goto found_and_ok;
			}
		      cp++;
		    }
		}
	      field = TREE_CHAIN (field);
	    }
	  /* Now we have run through one link's member functions.
	     arrange to head-insert this link's links.  */
	  {
	    tree tmp = TREE_BASELINK (baselink);
	    fields = TREE_CHAIN (fields);
	    while (tmp)
	      {
		/* @@ does not yet add previous base types.  */
		fields = tree_cons (TREE_PURPOSE (tmp), TREE_VALUE (tmp),
				    fields);
		TREE_TYPE (fields) = TREE_TYPE (tmp);
		tmp = TREE_CHAIN (tmp);
	      }
	  }
	}
      if (pass == 0)
	{
	  /* No exact match could be found.  Now try to find match
	     using default conversions.  */
	  if ((flags & LOOKUP_GLOBAL) && IDENTIFIER_GLOBAL_VALUE (name))
	    if (TREE_CODE (IDENTIFIER_GLOBAL_VALUE (name)) == FUNCTION_DECL)
	      ever_seen += 1;
	    else if (TREE_CODE (IDENTIFIER_GLOBAL_VALUE (name)) == TREE_LIST)
	      ever_seen += list_length (IDENTIFIER_GLOBAL_VALUE (name));

	  if (ever_seen == 0)
	    {
	      if (flags & LOOKUP_GLOBAL)
		error ("no global or member function `%s' defined",
		       IDENTIFIER_POINTER (name));
	      else
		error_with_aggr_type (save_basetype, "no member function `%s' defined within type `%s'",
				      IDENTIFIER_POINTER (name));
	      return error_mark_node;
	    }
	}

      if (pass != 0 && (cp - candidates) != 0)
	{
	  /* Rank from worst to best.  Then cp will point to best one.
	     Private fields have their bits flipped.  For unsigned
	     numbers, this should make them look very large.
	     If the best alternate has a (signed) negative value,
	     then all we ever saw were private members.  */
	  if (cp - candidates > 1)
	    {
	      cp = ideal_candidate (save_basetype, candidates,
				    cp - candidates, parms, len);
	      if (cp == 0)
		{
		  error ("ambiguous type conversion requested for %s `%s'",
			 name_kind, err_name);
		  return error_mark_node;
		}
	    }
	  else if (cp[-1].evil > 1)
	    {
	      error ("ambiguous type conversion requested for %s `%s'",
		     name_kind, err_name);
	      return error_mark_node;
	    }
	  else cp--;

	  /* The global function was the best, so use it.  */
	  if (cp->u.field == 0)
	    {
	      /* We must convert the instance pointer into a reference type.
		 Global overloaded functions can only either take
		 aggregate objects (which come for free from references)
		 or reference data types anyway.  */
	      TREE_VALUE (parms) = copy_node (instance_ptr);
	      TREE_TYPE (TREE_VALUE (parms)) = build_reference_type (TREE_TYPE (TREE_TYPE (instance_ptr)));
	      return build_function_call (cp->function, parms);
	    }

	  function = cp->function;
	  if (DECL_STATIC_FUNCTION_P (function))
	    basetype = NULL_TREE;
	  else
	    {
	      basetype = TREE_TYPE (TREE_TYPE (cp->arg));
	      TREE_VALUE (parms) = cp->arg;
	    }
	  goto found_and_ok;
	}

      if (pass != 0 && ((flags & ~LOOKUP_GLOBAL) & (LOOKUP_COMPLAIN|LOOKUP_SPECULATIVELY)))
	{
	  char *tag_name, *buf;

	  if (DECL_STATIC_FUNCTION_P (cp->function))
	    parms = TREE_CHAIN (parms);
	  if (ever_seen)
	    {
	      if (saw_protected == 0 && saw_private == 0)
		{
		  if (flags & LOOKUP_SPECULATIVELY)
		    return NULL_TREE;
		  if (static_call_context && cp->u.bad_arg >= 0)
		    error_with_aggr_type (TREE_TYPE (TREE_TYPE (instance_ptr)),
					  "object missing in call to `%s::%s'",
					  IDENTIFIER_POINTER (name));
		  else
		    report_type_mismatch (cp, parms, name_kind, err_name);
		}
	      else
		{
		  char buf[80];
		  char *msg;
		  tree seen = saw_private;

		  if (saw_private)
		    if (saw_protected)
		      msg = "%s %%s (and the like) are private or protected";
		    else
		      msg = "the %s %%s is private";
		  else
		    {
		      msg = "the %s %%s is protected";
		      seen = saw_protected;
		    }
		  sprintf (buf, msg, name_kind);
		  error_with_decl (seen, buf);
		  error ("within this context");
		}
	      return error_mark_node;
	    }

	  if ((flags & ~LOOKUP_SPECULATIVELY) & LOOKUP_COMPLAIN)
	    {
	      if (TREE_CODE (save_basetype) == RECORD_TYPE)
		tag_name = "structure";
	      else
		tag_name = "union";

	      if (wrap)
		buf = "%s has no appropriate wrapper function defined";
	      else
		{
		  buf = (char *)alloca (30 + strlen (err_name));
		  strcpy (buf, "%s has no method named `%s'");
		}

	      error (buf, tag_name, err_name);
	      return error_mark_node;
	    }
	  return NULL_TREE;
	}
    }
  /* Silently return error_mark_node.  */
  return error_mark_node;

 found:
  if (visibility == visibility_private)
    {
      if (flags & LOOKUP_COMPLAIN)
	error (TREE_PRIVATE (function)
	       ? "%s `%s' is private"
	       : "%s `%s' is from private base class",
	       name_kind,
	       lang_printable_name (function));
      return error_mark_node;
    }
  else if (visibility == visibility_protected)
    {
      if (flags & LOOKUP_COMPLAIN)
	error (TREE_PROTECTED (function)
	       ? "%s `%s' is protected"
	       : "%s `%s' has protected visibility from this point",
	       name_kind,
	       lang_printable_name (function));
      return error_mark_node;
    }
  abort ();
 found_and_ok:

  fntype = TREE_TYPE (function);

  if (TREE_CODE (fntype) == POINTER_TYPE)
    fntype = TREE_TYPE (fntype);

  if (TREE_CODE (fntype) == METHOD_TYPE && static_call_context)
    {
      /* Let's be nice to the user for now, and give reasonable
	 default behavior.  */
      if (current_class_decl)
	{
	  if (DECL_CONTEXT (function) == current_class_type)
	    instance_ptr = current_class_decl;
	  else
	    {
	      basetype = get_base_type (DECL_CONTEXT (function),
					current_class_type, 1);
	      if (basetype == 0)
		{
		  error_with_aggr_type (DECL_CONTEXT (function), "type `%s' not derived from type `%s'",
					IDENTIFIER_POINTER (current_class_name));
		  return error_mark_node;
		}
	      else if (basetype == error_mark_node)
		return error_mark_node;
	      instance_ptr = convert (TYPE_POINTER_TO (basetype),
				      current_class_decl);
	    }
	  TREE_VALUE (parms) = instance_ptr;
	}
      else
	{
	  error_with_aggr_type (DECL_CONTEXT (function),
				"cannot call member function `%s::%s' without object",
				IDENTIFIER_POINTER (name));
	  return error_mark_node;
	}
    }

  value_type = TREE_TYPE (fntype) ? TREE_TYPE (fntype) : void_type_node;

  if (TYPE_SIZE (value_type) == 0)
    {
      incomplete_type_error (0, value_type);
      return error_mark_node;
    }

  /* We do not pass FUNCTION into `actualparameterlist', because by
     now everything should be ok.  If not, then we have a serious error.  */
  if (DECL_STATIC_FUNCTION_P (function))
    parms = actualparameterlist (NULL_TREE, TYPE_ARG_TYPES (fntype),
				 TREE_CHAIN (parms), NULL_TREE);
  else
    {
      instance_ptr = convert_to_nonzero_pointer (TYPE_POINTER_TO (DECL_VCONTEXT (function)), TREE_VALUE (parms));
      if (TREE_CODE (instance_ptr) == COND_EXPR)
	instance_ptr = save_expr (instance_ptr);
      instance = build_indirect_ref (instance_ptr);
      parms = tree_cons (NULL_TREE, instance_ptr, actualparameterlist (NULL_TREE, TREE_CHAIN (TYPE_ARG_TYPES (fntype)), TREE_CHAIN (parms), NULL_TREE));
    }

  /* See if there is a wrapper for this thing.  */
  if (wrap < 0 || basetype == NULL_TREE
      || name == wrapper_name
      || name == DECL_NAME (TYPE_NAME (basetype)))
    ;
  else if (wrap > 0 || TYPE_NEEDS_WRAPPER (basetype))
    {
      if (wrap == 0)
	{
	  wrap = TYPE_NEEDS_WRAPPER (basetype);
	  /* If no wrapper specified, wrapper may be virtual.  */
	  flags &= ~LOOKUP_NONVIRTUAL;
	}

      if (wrap)
	{
	  tree wrapped_result, unwrapped_result;
	  register int bytecount = get_arglist_len_in_bytes (parms);

	  if (!all_virtual && TREE_CODE (function) == FUNCTION_DECL)
	    parm = build_unary_op (ADDR_EXPR, function, 0);
	  else
	    {
              fntype = build_pointer_type (build_method_type (basetype, fntype));
	      parm = build (NOP_EXPR, fntype, DECL_VINDEX (function));
	    }

	  if (TYPE_HAS_WRAPPER_PRED (basetype))
	    {
	      unwrapped_result = build_nt (CALL_EXPR, default_conversion (function), parms, NULL_TREE);

	      assert (TREE_OPERAND (unwrapped_result, 1) != error_mark_node);

	      TREE_TYPE (unwrapped_result) = value_type;
	      TREE_VOLATILE (unwrapped_result) = 1;
	    }

	  /* If this pointer walked as a result of multiple inheritance,
	     keep its displaced value.  */
	  parms = tree_cons (NULL_TREE, build_int_2 (bytecount, 0),
			     tree_cons (NULL_TREE, parm, TREE_CHAIN (parms)));

	  wrapped_result = build_method_call (instance, wrapper_name, parms, basetype_path, flags);
#if 0
	  /* Do this if we want the result of operator->() to inherit
	     the type of the function it is subbing for.  */
	  if (wrapped_result != error_mark_node)
	    TREE_TYPE (wrapped_result) = value_type;
#endif

	  if (TYPE_HAS_WRAPPER_PRED (basetype))
	    {
	      result = build_conditional_expr
		(build_method_call (instance, wrapper_pred_name, build_tree_list (NULL_TREE, parm), basetype_path, LOOKUP_NORMAL),
		 wrapped_result,
		 unwrapped_result);

	    }
	  else
	    {
	      result = wrapped_result;
	    }

	  TREE_VOLATILE (result) = 1;
	  return result;
	}
    }
  /* Constructors do not overload method calls.  */
  else if (basetype != NULL_TREE
	   && TYPE_OVERLOADS_METHOD_CALL_EXPR (basetype)
	   && name != DECL_NAME (TYPE_NAME (basetype))
	   && (TREE_CODE (function) != FUNCTION_DECL
	       || strncmp (IDENTIFIER_POINTER (DECL_NAME (function)),
			   "op$method_call", 13))
	   && (may_be_remote (basetype)
	       || (C_C_D ? TREE_TYPE (instance) != current_class_type : 1)))
    {
      register int used, size;
#ifdef ESKIT
      register int bytecount = 0;
#else
      register int bytecount = get_arglist_len_in_bytes (parms);
#endif

      parms = tree_cons (NULL_TREE, build_int_2 (bytecount, 0),
			 TREE_CHAIN (parms));
      if (!all_virtual && TREE_CODE (function) == FUNCTION_DECL)
	result = build_opfncall (METHOD_CALL_EXPR, instance,
				 build_unary_op (ADDR_EXPR, function, 0),
				 parms);
      else
	result = build_opfncall (METHOD_CALL_EXPR, instance,
				 convert (TREE_TYPE (default_conversion (function)), DECL_VINDEX (function)),
				 parms);
      if (result == NULL_TREE)
	{
	  compiler_error ("could not overload `operator->()(...)'");
	  return error_mark_node;
	}
      else if (result == error_mark_node)
	return error_mark_node;

#if 0
      /* Do this if we want the result of operator->() to inherit
	 the type of the function it is subbing for.  */
      TREE_TYPE (result) = value_type;
#endif

#ifdef ESKIT
      /* Count the number of bytes of arguements to operator->(),
	 not to the method itself.  In the tally, don't count bytes
	 for pointer to member function or for the bytecount.  */
      parms = TREE_OPERAND (result, 1);
      bytecount = get_arglist_len_in_bytes (TREE_CHAIN (TREE_CHAIN (TREE_CHAIN (parms))));
      used = size = GET_MODE_SIZE (TYPE_MODE (TREE_TYPE (TREE_VALUE (parms))));
#ifdef PUSH_ROUNDING
      size = PUSH_ROUNDING (size);
#endif
      used = (((size + PARM_BOUNDARY / BITS_PER_UNIT - 1)
	       / (PARM_BOUNDARY / BITS_PER_UNIT))
	      * (PARM_BOUNDARY / BITS_PER_UNIT));
      bytecount += used;
      TREE_CHAIN (TREE_CHAIN (parms))
	= tree_cons (NULL_TREE, build_int_2 (bytecount, 0),
		     TREE_CHAIN (TREE_CHAIN (TREE_CHAIN (parms))));
#endif

      return result;
    }

  /* If we are referencing a virtual function from an object
     of effectively static type, then there is no need
     to go through the virtual function table.  */
  if (need_vtbl
      && ((all_virtual == 1
	   && DECL_VINDEX (function)
	   && may_be_remote (basetype))
	  || (DECL_VIRTUAL_P (function)
	      && ! resolves_to_fixed_type_p (instance))))
    {
      function = build_vfn_ref (&TREE_VALUE (parms), instance, DECL_VINDEX (function));
      TREE_TYPE (function) = build_pointer_type (fntype);
    }
#ifdef SOS
  else if (basetype && TYPE_DYNAMIC (basetype))
    {
      function = build_array_ref (dtbl, DECL_DINDEX (function));
      TREE_TYPE (function) = build_pointer_type (fntype);
    }
#endif

  if (TREE_INLINE (function) && TREE_CODE (function) == FUNCTION_DECL)
    function = build (ADDR_EXPR, build_pointer_type (TREE_TYPE (function)),
		      function);
  else function = default_conversion (function);

  result =
    build_nt (CALL_EXPR, function, parms, NULL_TREE);

  assert (TREE_OPERAND (result, 1) != error_mark_node);

  TREE_TYPE (result) = value_type;
  TREE_VOLATILE (result) = 1;
  return result;
}

void
init_class_processing ()
{
  current_class_stacksize = 10;
  current_class_base = (tree *)xmalloc(current_class_stacksize * sizeof (tree));
  current_class_stack = current_class_base;

  current_lang_stacksize = 10;
  current_lang_base = (tree *)xmalloc(current_lang_stacksize * sizeof (tree));
  current_lang_stack = current_lang_base;

  lang_name_cplusplus = get_identifier ("C++");
  lang_name_c = get_identifier ("C");
  current_lang_name = lang_name_cplusplus;

  delta_name = get_identifier (VTABLE_DELTA_NAME);
  pfn_name = get_identifier (VTABLE_PFN_NAME);

  /* Keep these values lying around.  */
  minus_one_node = build_int_2 (-1, 0);
  the_null_vtable_entry = build_vtable_entry (integer_zero_node, integer_zero_node);
}

/* Set current scope to NAME. CODE tells us if this is a
   STRUCT, UNION, or ENUM environment.

   NAME may end up being NULL_TREE if this is an anonymous or
   late-bound struct (as in "struct { ... } foo;")  */

/* Here's a subroutine we need because C lacks lambdas.  */
void
unuse_fields (type)
     tree type;
{
  tree fields;

  for (fields = TYPE_FIELDS (type); fields; fields = TREE_CHAIN (fields))
    {
      if (TREE_CODE (fields) == VAR_DECL || TREE_CODE (fields) == CONST_DECL)
	continue;
      TREE_USED (fields) = 0;
      if (DECL_ANON_UNION_ELEM (fields))
	unuse_fields (TREE_TYPE (fields));
    }
}
/* Set global variables CURRENT_CLASS_NAME and CURRENT_CLASS_TYPE to
   appropriate values, found by looking up the type definition of
   NAME (as a CODE).

   If MODIFY is 1, we set IDENTIFIER_CLASS_VALUE's of names
   which can be seen locally to the class. They are shadowed by
   any subsequent local declaration (including parameter names).

   If MODIFY is 2, we set IDENTIFIER_CLASS_VALUE's of names
   which have static meaning (i.e., static members, static
   member functions, enum declarations, etc).

   So that we may avoid calls to lookup_name, we cache the TYPE_DECL
   in the TREE_TYPE field of the name.

   For multiple inheritance, we perform a two-pass depth-first search
   of the type lattice.  The first pass performs a pre-order search,
   marking types after the type has had its fields installed in
   the appropriate IDENTIFIER_CLASS_VALUE slot.  The second pass merely
   unmarks the marked types.  If a field or member function name
   appears in an ambiguous way, the IDENTIFIER_CLASS_VALUE of
   that name becomes `error_mark_node'.  */

void
pushclass (type, modify)
     tree type;
     int modify;
{
  push_memoized_context (type, modify);

  *current_class_stack++ = current_class_name;
  if (current_class_stack >= current_class_base + current_class_stacksize)
    {
      current_class_base =
	(tree *)xrealloc (current_class_base, current_class_stacksize + 10);
      current_class_stack = current_class_base + current_class_stacksize;
      current_class_stacksize += 10;
    }

  type = TYPE_MAIN_VARIANT (type);
  current_class_name = DECL_NAME (TYPE_NAME (type));
  current_class_type = type;

  if (type != prev_class_type && prev_class_type != NULL_TREE)
    {
      popclass (-1);
      prev_class_type = 0;
    }

  if (modify)
    {
      tree tags;

      if (type != prev_class_type)
	{
	  build_mi_matrix (type);
	  push_class_decls (type);
	  free_mi_matrix ();
	  prev_class_type = type;
	}
      else
	unuse_fields (type);

      tags = CLASSTYPE_TAGS (type);
      while (tags)
	{
	  pushtag (TREE_PURPOSE (tags), TREE_VALUE (tags));
	  TREE_NONLOCAL (TREE_VALUE (tags)) = 1;
	  if (IDENTIFIER_CLASS_VALUE (TREE_PURPOSE (tags)) == NULL_TREE)
	    IDENTIFIER_CLASS_VALUE (TREE_PURPOSE (tags)) = TREE_VALUE (tags);
	  tags = TREE_CHAIN (tags);
	}
    }
  else
    pushlevel_class ();
}
 
/* Get out of the current class scope. If we were in a class scope
   previously, that is the one popped to.  The flag MODIFY tells
   whether the current scope declarations needs to be modified
   as a result of popping to the new scope.  */
void
popclass (modify)
     int modify;
{
  tree t;
  tree fields;

  if (modify < 0)
    {
      /* Back this old class out completely.  */
      tree tags = CLASSTYPE_TAGS (prev_class_type);

      pop_class_decls (prev_class_type);
      while (tags)
	{
	  TREE_NONLOCAL (TREE_VALUE (tags)) = 0;
	  IDENTIFIER_CLASS_VALUE (TREE_PURPOSE (tags)) = NULL_TREE;
	  tags = TREE_CHAIN (tags);
	}
      return;
    }
  if (modify)
    {
      /* Just remove from this class what didn't make
	 it into IDENTIFIER_CLASS_VALUE.  */
      tree tags = CLASSTYPE_TAGS (current_class_type);

      while (tags)
	{
	  TREE_NONLOCAL (TREE_VALUE (tags)) = 0;
	  IDENTIFIER_CLASS_VALUE (TREE_PURPOSE (tags)) = NULL_TREE;
	  tags = TREE_CHAIN (tags);
	}
    }
  else
    poplevel_class ();

  t = *--current_class_stack;

  if (t)
    {
      current_vtable_decl = DECL_VTBL_PTR (t);
      t = TREE_TYPE (t);
      current_class_type = TREE_TYPE (t);
      current_class_name = DECL_NAME (t);
      current_class_decl = lookup_name (get_identifier (THIS_NAME));
      if (current_class_decl)
	{
	  if (TREE_CODE (TREE_TYPE (current_class_decl)) == POINTER_TYPE)
	    C_C_D = build (INDIRECT_REF, current_class_type, current_class_decl);
	  else
	    C_C_D = current_class_decl;
	}
      else C_C_D = NULL_TREE;
    }
  else
    {
      current_class_type = NULL_TREE;
      current_class_name = NULL_TREE;
      current_class_decl = NULL_TREE;
      current_vtable_decl = NULL_TREE;
      C_C_D = NULL_TREE;
    }

  pop_memoized_context (modify);
}

/* Set global variables CURRENT_LANG_NAME to appropriate value
   so that behavior of name-mangline machinery is correct.  */

void
push_lang_context (name)
     tree name;
{
  *current_lang_stack++ = current_lang_name;
  if (current_lang_stack >= current_lang_base + current_lang_stacksize)
    {
      current_lang_base =
	(tree *)xrealloc (current_lang_base, current_lang_stacksize + 10);
      current_lang_stack = current_lang_base + current_lang_stacksize;
      current_lang_stacksize += 10;
    }

  current_lang_name = name;
  if (name == lang_name_cplusplus)
    strict_prototype = strict_prototypes_lang_cplusplus;
  else if (name == lang_name_c)
    strict_prototype = strict_prototypes_lang_c;
}
  
/* Get out of the current language scope.  */
void
pop_lang_context ()
{
  current_lang_name = *--current_lang_stack;
  if (current_lang_name == lang_name_cplusplus)
    strict_prototype = strict_prototypes_lang_cplusplus;
  else if (current_lang_name == lang_name_c)
    strict_prototype = strict_prototypes_lang_c;
}

int
root_lang_context_p ()
{
  return current_lang_stack == current_lang_base;
}

/* This function will instantiate the type of the expression given
   in RHS to match the type of LHSTYPE.  If LHSTYPE is NULL_TREE,
   or other errors exist, the TREE_TYPE of RHS will be ERROR_MARK_NODE.

   This function is used in build_modify_expr, actualparameterlist,
   build_c_cast, and compute_conversion_costs.  */
tree
instantiate_type (lhstype, rhs, complain)
     tree lhstype, rhs;
     int complain;
{
  if (TREE_CODE (lhstype) == UNKNOWN_TYPE)
    {
      if (complain)
	error ("not enough type information");
      return error_mark_node;
    }

  if (TREE_TYPE (rhs) != NULL_TREE
      && TREE_TYPE (rhs) != unknown_type_node
      && (TREE_CODE (TREE_TYPE (rhs)) != OFFSET_TYPE
	  || TREE_TYPE (TREE_TYPE (rhs)) != unknown_type_node))
    return rhs;

  /* This should really only be used when attempting to distinguish
     what sort of a pointer to function we have.  For now, any
     arithmethic operation which is not supported on pointers
     is rejected as an error.  */

  switch (TREE_CODE (rhs))
    {
    case OP_IDENTIFIER:
    case TYPE_EXPR:
    case CONVERT_EXPR:
    case SAVE_EXPR:
    case CONSTRUCTOR:
    case BUFFER_REF:
      assert (0);
      return error_mark_node;

    case INDIRECT_REF:
    case ARRAY_REF:
      TREE_TYPE (rhs) = lhstype;
      lhstype = build_pointer_type (lhstype);
      if (TREE_CODE (TREE_OPERAND (rhs, 0)) == OP_IDENTIFIER)
	TREE_OPERAND (rhs, 0)
	  = build_instantiated_decl (lhstype, TREE_OPERAND (rhs, 0));
      else
	TREE_OPERAND (rhs, 0) = instantiate_type (lhstype, TREE_OPERAND (rhs, 0), complain);
      if (TREE_OPERAND (rhs, 0) == error_mark_node)
	return error_mark_node;

      return rhs;

    case NOP_EXPR:
      rhs = copy_node (TREE_OPERAND (rhs, 0));
      TREE_TYPE (rhs) = unknown_type_node;
      return instantiate_type (lhstype, rhs, complain);

    case COMPONENT_REF:
      {
	tree field = TREE_OPERAND (rhs, 1);
	assert (TREE_CODE (field) == FIELD_DECL);
	assert (!(TREE_CODE (TREE_TYPE (field)) == FUNCTION_TYPE
		  || TREE_CODE (TREE_TYPE (field)) == METHOD_TYPE));

	TREE_TYPE (rhs) = lhstype;
	/* First look for an exact match  */

	while (field && TREE_TYPE (field) != lhstype)
	  field = TREE_CHAIN (field);
	if (field)
	  {
	    TREE_OPERAND (rhs, 1) = field;
	    return rhs;
	  }

	/* No exact match found, look for a compatible function.  */
	field = TREE_OPERAND (rhs, 1);
	while (field && ! comptypes (lhstype, TREE_TYPE (field), 0))
	  field = TREE_CHAIN (field);
	if (field)
	  {
	    TREE_OPERAND (rhs, 1) = field;
	    field = TREE_CHAIN (field);
	    while (field && ! comptypes (lhstype, TREE_TYPE (field), 0))
	      field = TREE_CHAIN (field);
	    if (field)
	      {
		if (complain)
		  error ("ambiguous type instantiation for COMPONENT_REF requested");
		return error_mark_node;
	      }
	  }
	else
	  {
	    if (complain)
	      error ("no appropriate type instantiation exists for COMPONENT_REF");
	    return error_mark_node;
	  }
	return rhs;
      }

    case TREE_LIST:
      {
	tree elem = rhs;

	/* First look for an exact match.  Search either overloaded
	   functions or member functions.  May have to undo what
	   `default_conversion' or `datatype' might do to lhstype.  */

	if (TREE_CODE (lhstype) == POINTER_TYPE)
	  if (TREE_CODE (TREE_TYPE (lhstype)) == FUNCTION_TYPE
	      || TREE_CODE (TREE_TYPE (lhstype)) == METHOD_TYPE)
	    lhstype = TREE_TYPE (lhstype);
	  else
	    {
	      if (complain)
		error ("invalid type combination in `instantiate_type'");
	      return error_mark_node;
	    }

	/* Our list of functions can come in two flavors:
	   a list of function decls, or a list of lists of
	   function decls.  */
	if (TREE_CODE (TREE_VALUE (rhs)) == FUNCTION_DECL)
	  {
	    if (TREE_CODE (lhstype) != FUNCTION_TYPE)
	      {
		error ("cannot resolve overloaded function `%s' based on non-function type",
		       IDENTIFIER_POINTER (TREE_PURPOSE (rhs)));
		return error_mark_node;
	      }

	    while (elem)
	      if (TREE_TYPE (TREE_VALUE (elem)) != lhstype)
		elem = TREE_CHAIN (elem);
	      else
		return TREE_VALUE (elem);

	    /* No exact match found, look for a compatible function.  */
	    elem = rhs;
	    while (elem && ! comp_target_types (lhstype, TREE_TYPE (TREE_VALUE (elem)), 1))
	      elem = TREE_CHAIN (elem);
	    if (elem)
	      {
		tree save_elem = TREE_VALUE (elem);
		elem = TREE_CHAIN (elem);
		while (elem && ! comp_target_types (lhstype, TREE_TYPE (TREE_VALUE (elem)), 0))
		  elem = TREE_CHAIN (elem);
		if (elem)
		  {
		    if (complain)
		      error ("ambiguous type instantiation for overloaded function requested");
		    return error_mark_node;
		  }
		return save_elem;
	      }
	    else
	      {
		if (complain)
		  error ("no appropriate type instantiation for overloaded function `%s' exists",
			 IDENTIFIER_POINTER (TREE_PURPOSE (rhs)));
		return error_mark_node;
	      }
	  }
	else if (TREE_CODE (TREE_VALUE (rhs)) == TREE_LIST)
	  {
	    char *name = IDENTIFIER_POINTER (TREE_PURPOSE (rhs));
	    /* rhs is a BASELINK from derived downto base types.  */
	    while (rhs)
	      {
		/* The TREE_VALUE of rhs is a pointer to the method
		   chain we are interested in.  Its TREE_CHAIN points
		   to other method chains not of interest.  */
		elem = TREE_VALUE (TREE_VALUE (rhs));

		/* First look for an exact match among member functions.  */
		while (elem && TREE_TYPE (elem) != lhstype)
		  elem = TREE_CHAIN (elem);
		if (elem)
		  return elem;

		/* No exact match found, look for compatible member function.  */
		elem = TREE_VALUE (TREE_VALUE (rhs));
		while (elem && ! comp_target_types (lhstype, TREE_TYPE (elem), 0))
		  elem = TREE_CHAIN (elem);
		if (elem)
		  {
		    tree save_elem = elem;
		    elem = TREE_CHAIN (elem);
		    while (elem && ! comp_target_types (lhstype, TREE_TYPE (elem), 0))
		      elem = TREE_CHAIN (elem);
		    if (elem)
		      {
			if (complain)
			  error ("ambiguous type instantiation for member function requested");
			return error_mark_node;
		      }
		    return save_elem;
		  }
		rhs = TREE_CHAIN (rhs);
	      }
	    if (complain)
	      error ("no appropriate type instantiation for member function `%s' exists", name);
	    return error_mark_node;
	  }
	else abort ();
      }

    case CALL_EXPR:
      /* This is too hard for now.  */
      assert (0);
      return error_mark_node;

    case PLUS_EXPR:
    case MINUS_EXPR:
    case COMPOUND_EXPR:
      if (TREE_CODE (TREE_OPERAND (rhs, 0)) == OP_IDENTIFIER)
	TREE_OPERAND (rhs, 0)
	  = build_instantiated_decl (lhstype, TREE_OPERAND (rhs, 0));
      else
	TREE_OPERAND (rhs, 0) = instantiate_type (lhstype, TREE_OPERAND (rhs, 0), complain);
      if (TREE_OPERAND (rhs, 0) == error_mark_node)
	return error_mark_node;

      if (TREE_CODE (TREE_OPERAND (rhs, 1)) == OP_IDENTIFIER)
	TREE_OPERAND (rhs, 1)
	  = build_instantiated_decl (lhstype, TREE_OPERAND (rhs, 1));
      else
	TREE_OPERAND (rhs, 1) = instantiate_type (lhstype, TREE_OPERAND (rhs, 1), complain);
      if (TREE_OPERAND (rhs, 1) == error_mark_node)
	return error_mark_node;

      TREE_TYPE (rhs) = lhstype;
      return rhs;

    case MULT_EXPR:
    case TRUNC_DIV_EXPR:
    case FLOOR_DIV_EXPR:
    case CEIL_DIV_EXPR:
    case ROUND_DIV_EXPR:
    case RDIV_EXPR:
    case TRUNC_MOD_EXPR:
    case FLOOR_MOD_EXPR:
    case CEIL_MOD_EXPR:
    case ROUND_MOD_EXPR:
    case FIX_ROUND_EXPR:
    case FIX_FLOOR_EXPR:
    case FIX_CEIL_EXPR:
    case FIX_TRUNC_EXPR:
    case FLOAT_EXPR:
    case NEGATE_EXPR:
    case ABS_EXPR:
    case MAX_EXPR:
    case MIN_EXPR:
    case FFS_EXPR:

    case BIT_AND_EXPR:
    case BIT_IOR_EXPR:
    case BIT_XOR_EXPR:
    case LSHIFT_EXPR:
    case RSHIFT_EXPR:
    case LROTATE_EXPR:
    case RROTATE_EXPR:

    case PREINCREMENT_EXPR:
    case PREDECREMENT_EXPR:
    case POSTINCREMENT_EXPR:
    case POSTDECREMENT_EXPR:
      if (complain)
	error ("illegal operation on uninstantiated type");
      return error_mark_node;

    case TRUTH_AND_EXPR:
    case TRUTH_OR_EXPR:
    case LT_EXPR:
    case LE_EXPR:
    case GT_EXPR:
    case GE_EXPR:
    case EQ_EXPR:
    case NE_EXPR:
    case TRUTH_ANDIF_EXPR:
    case TRUTH_ORIF_EXPR:
    case TRUTH_NOT_EXPR:
      if (complain)
	error ("not enough type information");
      return error_mark_node;

    case COND_EXPR:
      if (TREE_TYPE (TREE_OPERAND (rhs, 0)) == unknown_type_node)
	{
	  if (complain)
	    error ("not enough type information");
	  return error_mark_node;
	}
      if (TREE_CODE (TREE_OPERAND (rhs, 1)) == OP_IDENTIFIER)
	TREE_OPERAND (rhs, 1)
	  = build_instantiated_decl (lhstype, TREE_OPERAND (rhs, 1));
      else
	TREE_OPERAND (rhs, 1) = instantiate_type (lhstype, TREE_OPERAND (rhs, 1), complain);
      if (TREE_OPERAND (rhs, 1) == error_mark_node)
	return error_mark_node;

      if (TREE_CODE (TREE_OPERAND (rhs, 2)) == OP_IDENTIFIER)
	TREE_OPERAND (rhs, 2)
	  = build_instantiated_decl (lhstype, TREE_OPERAND (rhs, 2));
      else
	TREE_OPERAND (rhs, 2) = instantiate_type (lhstype, TREE_OPERAND (rhs, 2), complain);
      if (TREE_OPERAND (rhs, 2) == error_mark_node)
	return error_mark_node;

      TREE_TYPE (rhs) = lhstype;
      return rhs;

    case MODIFY_EXPR:
      if (TREE_CODE (TREE_OPERAND (rhs, 1)) == OP_IDENTIFIER)
	TREE_OPERAND (rhs, 1)
	  = build_instantiated_decl (lhstype, TREE_OPERAND (rhs, 1));
      else
	TREE_OPERAND (rhs, 1) = instantiate_type (lhstype, TREE_OPERAND (rhs, 1), complain);
      if (TREE_OPERAND (rhs, 1) == error_mark_node)
	return error_mark_node;

      TREE_TYPE (rhs) = lhstype;
      return rhs;
      
    case ADDR_EXPR:
      TREE_TYPE (rhs) = lhstype;
      lhstype = TREE_TYPE (lhstype);
      if (TREE_CODE (TREE_OPERAND (rhs, 0)) == OP_IDENTIFIER)
	TREE_OPERAND (rhs, 0)
	  = build_instantiated_decl (lhstype, TREE_OPERAND (rhs, 0));
      else
	TREE_OPERAND (rhs, 0) = instantiate_type (lhstype, TREE_OPERAND (rhs, 0), complain);
      if (TREE_OPERAND (rhs, 0) == error_mark_node)
	return error_mark_node;

      return rhs;

    case ENTRY_VALUE_EXPR:
      assert (0);
      return error_mark_node;

    case ERROR_MARK:
      return error_mark_node;

    default:
      assert (0);
      return error_mark_node;
    }
}

/* This routine is called when we finally know the type of expression
   we are looking for.  If the operator encoded by EXP can take an
   argument of type TYPE, return the FUNCTION_DECL for that operator.  */
tree
build_instantiated_decl (type, exp)
     tree type, exp;
{
  tree parmtypes, decl, name;

  if (TREE_CODE (exp) != OP_IDENTIFIER)
    {
      compiler_error ("bad argument to build_instantiated_decl");
      return error_mark_node;
    }
  type = datatype (type);
  if (TREE_CODE (type) != POINTER_TYPE
      || (TREE_CODE (TREE_TYPE (type)) != FUNCTION_TYPE
	  && TREE_CODE (TREE_TYPE (type)) != METHOD_TYPE))
    {
      error ("invalid type used to resolve overloaded function");
      return error_mark_node;
    }


  /* Now we know the type of this function, so overload it.  */
  parmtypes = TYPE_ARG_TYPES (TREE_TYPE (type));
  name = build_operator_fnname (TREE_VALUE (exp), parmtypes, 0);
  if (name)
    {
      name = build_decl_overload (IDENTIFIER_POINTER (name), parmtypes);
      decl = lookup_name (name);
      if (decl)
	return decl;
      error ("type instantiation failed to find a suitable declaration for `operator %s'",
	     operator_name_string (name));
      return error_mark_node;
    }
  return error_mark_node;
}

/* Return the name of the virtual function table (as an IDENTIFIER_NODE)
   for the given TYPE.  */
static tree
get_vtable_name (type)
     tree type;
{
  char *buf = (char *)alloca (sizeof (VTABLE_NAME_FORMAT)
			      + TYPE_NAME_LENGTH (type)
			      + 2);
  if (! IS_AGGR_TYPE (type))
    abort ();

  sprintf (buf, VTABLE_NAME_FORMAT, TYPE_NAME_STRING (type));
  return get_identifier (buf);
}

/* Return the name of the virtual function pointer field
   (as an IDENTIFIER_NODE) for the given TYPE.  Note that
   this may have to look back through base types to find the
   ultimate field name.  (For single inheritance, these could
   all be the same name.  Who knows for multiple inheritance).  */
static tree
get_vfield_name (type)
     tree type;
{
  char *buf;

  while (CLASSTYPE_N_BASECLASSES (type)
	 && TYPE_VIRTUAL_P (CLASSTYPE_BASECLASS (type, 1))
	 && ! CLASSTYPE_VIA_VIRTUAL (type, 1))
    type = CLASSTYPE_BASECLASS (type, 1);

  buf = (char *)alloca (sizeof (VFIELD_NAME_FORMAT)
			+ TYPE_NAME_LENGTH (type)
			+ 2);
  sprintf (buf, VFIELD_NAME_FORMAT, TYPE_NAME_STRING (type));
  return get_identifier (buf);
}

void
print_class_statistics ()
{
  fprintf (stderr, "convert_harshness = %d\n", n_convert_harshness);
  fprintf (stderr, "compute_conversion_costs = %d\n", n_compute_conversion_costs);
  fprintf (stderr, "build_method_call = %d (inner = %d)\n",
	   n_build_method_call, n_inner_fields_searched);
}
