// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#ifndef _BitSet_h

#define _BitSet_h 1

#include <stream.h>

struct _BitSetrep
{
  unsigned short  len;          // number of longs in s
  unsigned short  virt;         // virtual 0 or 1
  unsigned short  sz;           // allocated slots
  short           ref;          // reference count
  unsigned long   s[1];         // bits start here
};

extern _BitSetrep    _nil_BitSetrep;

class BitSet
{
  _BitSetrep*        rep;

  void               setlength(int, int = 0);
  void               copy(unsigned long*, int, int);

public:

// constructors
                     BitSet();
                     BitSet(BitSet&);
                     BitSet(unsigned long w);

                     ~BitSet();

  BitSet&            operator =  (BitSet& y);

// equality & subset tests

  friend int         operator == (BitSet& x, BitSet& y);
  friend int         operator != (BitSet& x, BitSet& y);
  friend int         operator <  (BitSet& x, BitSet& y);
  friend int         operator <= (BitSet& x, BitSet& y);
  friend int         operator >  (BitSet& x, BitSet& y);
  friend int         operator >= (BitSet& x, BitSet& y);

// set operators

  friend BitSet      operator |  (BitSet& x, BitSet& y);
  friend BitSet      operator &  (BitSet& x, BitSet& y);
  friend BitSet      operator -  (BitSet& x, BitSet& y);
  friend BitSet      operator ^  (BitSet& x, BitSet& y);

  BitSet&            operator |= (BitSet& y);
  BitSet&            operator &= (BitSet& y);
  BitSet&            operator -= (BitSet& y);
  BitSet&            operator ^= (BitSet& y);

  BitSet             operator ~  ();
  BitSet&            complement();

// individual bit manipulation

  void               set(int pos);
  void               set(int from, int to);

  void               clear(int pos);
  void               clear(int from, int to);

  void               invert(int pos);
  void               invert(int from, int to);

  int                test(int pos);
  int                test(int from, int to);

// iterators

  int                first(int b = 1);
  int                last(int b = 1);

  int                next(int pos, int b = 1);
  int                previous(int pos, int b = 1);

// status

  int                empty();
  int                virtual_bit();
  int                count(int b = 1);
  
// convertors & IO

  friend BitSet      atoBitSet(const char* s, 
                               char f='0', char t='1', char star='*');
  friend const char* BitSettoa(BitSet& x, 
                               char f='0', char t='1', char star='*');

  friend ostream&    operator << (ostream& s, BitSet& x);

// misc

  void               copy(BitSet& y);
  friend void        setop(BitSet& x, BitSet& y, BitSet& r, char op);
  void               make_unique();
  void               trim();
  void               error(char* msg);
};

// error handlers

extern void default_BitSet_error_handler(char*);
extern one_arg_error_handler_t BitSet_error_handler;

extern one_arg_error_handler_t 
        set_BitSet_error_handler(one_arg_error_handler_t f);


//#ifdef __OPTIMIZE__

inline BitSet::BitSet()
{ 
  rep = &_nil_BitSetrep;
}

inline BitSet::BitSet(BitSet& x)
{ 
  rep = x.rep; if (rep->ref > 0) rep->ref++;
}

inline BitSet::BitSet(unsigned long w)
{
  rep = &_nil_BitSetrep; setlength(1); rep->s[0] = w;
}

inline BitSet::~BitSet()
{ 
  if (rep->ref > 0 && --rep->ref == 0) delete rep;
}

inline BitSet& BitSet::operator =  (BitSet& y)
{ 
  y.rep->ref++;
  if (rep->ref > 0 && --rep->ref == 0) delete rep;
  rep = y.rep;
  return *this; 
}

inline int operator != (BitSet& x, BitSet& y)
{
  return !(x == y);
}

inline int operator>(BitSet& x, BitSet& y)
{
  return y < x;
}

inline int operator>=(BitSet& x, BitSet& y)
{
  return y <= x;
}

inline BitSet operator & (BitSet& x, BitSet& y)
{
  BitSet r; setop(x, y, r, '&'); return r;
}

inline BitSet operator | (BitSet& x, BitSet& y)
{
  BitSet r; setop(x, y, r, '|'); return r;
}

inline BitSet operator ^ (BitSet& x, BitSet& y)
{
  BitSet r; setop(x, y, r, '^'); return r;
}

inline BitSet operator - (BitSet& x, BitSet& y)
{
  BitSet r; setop(x, y, r, '-'); return r;
}

inline BitSet& BitSet::operator &= (BitSet& y)
{
  setop(*this, y, *this, '&'); return *this;
}

inline BitSet& BitSet::operator |= (BitSet& y)
{
  setop(*this, y, *this, '|'); return *this;
}

inline BitSet& BitSet::operator ^= (BitSet& y)
{
  setop(*this, y, *this, '^'); return *this;
}

inline BitSet& BitSet::operator -= (BitSet& y)
{
  setop(*this, y, *this, '-'); return *this;
}

inline void BitSet::make_unique()
{
  if (rep->ref != 1) setlength(rep->len);
}

inline void BitSet::copy(BitSet& y)
{
  copy(y.rep->s, y.rep->len, y.rep->virt);
}

inline int BitSet::virtual_bit()
{
  return rep->virt;
}

inline int BitSet::first(int b = 1)
{
  return next(-1, b);
}


//#endif

#endif

