// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#ifndef _BitVec_h

#define _BitVec_h 1

#include <stream.h>

// Bits per word. Done here to allow inlines to use.
// This will be done in a nicer way when gnu libc .h files available
#define BITVEC_BPW  32

class BitVec;

class OneBit
{
  friend class     BitVec;

  unsigned long*   wd;
  int              pos;

 public:
                   OneBit(BitVec& v, int p);
                   OneBit(unsigned long* w, int p);
                   OneBit(OneBit& b);
                   ~OneBit();

  int              operator = (int b);
  friend int       operator = (int& a, OneBit& b);
  int              operator int();
};


class BitVec
{
  friend class       OneBit;

  unsigned long*     s;
  int                len;

                     BitVec(unsigned long* w, int l);
public:

// constructors
                     BitVec(BitVec&);
                     BitVec(int l);

                     ~BitVec();

  BitVec&            operator =  (BitVec& y);

// equality & subset tests

  friend int         operator == (BitVec& x, BitVec& y);
  friend int         operator != (BitVec& x, BitVec& y);
  friend int         operator <  (BitVec& x, BitVec& y);
  friend int         operator <= (BitVec& x, BitVec& y);
  friend int         operator >  (BitVec& x, BitVec& y);
  friend int         operator >= (BitVec& x, BitVec& y);

// set operators

  friend BitVec      operator |  (BitVec& x, BitVec& y);
  friend BitVec      operator &  (BitVec& x, BitVec& y);
  friend BitVec      operator -  (BitVec& x, BitVec& y);
  friend BitVec      operator ^  (BitVec& x, BitVec& y);

  BitVec&            operator |= (BitVec& y);
  BitVec&            operator &= (BitVec& y);
  BitVec&            operator -= (BitVec& y);
  BitVec&            operator ^= (BitVec& y);

  BitVec             operator ~  ();
  BitVec&            complement();

// status

  int                empty();
  int                count(int b = 1);
  int                capacity();
  void               resize(int newlen);

  OneBit             operator [] (int i);
  OneBit             elem (int i);

  friend BitVec      atoBitVec(const char* s, char f='0', char t='1');
  friend const char* BitVectoa(BitVec& x, char f='0', char t='1');

  friend ostream&    operator << (ostream& s, BitVec& x);

  void               error(char* msg);
};

// error handlers

extern void default_BitVec_error_handler(char*);
extern one_arg_error_handler_t BitVec_error_handler;

extern one_arg_error_handler_t 
        set_BitVec_error_handler(one_arg_error_handler_t f);


//#ifdef __OPTIMIZE__

inline BitVec::BitVec(int l)
{ 
  int sz = (unsigned)(len = l) / BITVEC_BPW + ((l & (BITVEC_BPW - 1))? 1 : 0);
  s = new unsigned long[sz];
  bzero(s, sz * sizeof(unsigned long));
}

inline BitVec::BitVec(unsigned long* w, int l)
{ 
  s = w;
  len = l;
}

inline BitVec::BitVec(BitVec& x)
{ 
  int sz = (unsigned)(len = x.len) / BITVEC_BPW + ((x.len & (BITVEC_BPW - 1))? 1 : 0);
  s = new unsigned long[sz];
  bcopy(x.s, s, sz * sizeof(unsigned long));
}


inline BitVec::~BitVec()
{ 
  delete s;
}

inline BitVec& BitVec::operator =  (BitVec& x)
{ 
  if (s != x.s)
  {
    delete s;
    int sz = (unsigned)(len = x.len) / BITVEC_BPW + ((x.len & (BITVEC_BPW - 1))? 1 : 0);
    s = new unsigned long[sz];
    bcopy(x.s, s, sz * sizeof(unsigned long));
  }
  return *this; 
}

inline int operator != (BitVec& x, BitVec& y)
{
  return !(x == y);
}

inline int operator>(BitVec& x, BitVec& y)
{
  return y < x;
}

inline int operator>=(BitVec& x, BitVec& y)
{
  return y <= x;
}


inline OneBit::OneBit(OneBit& b)
{
  wd = b.wd;
  pos = b.pos;
}

inline OneBit::OneBit(BitVec& v, int p)
{
  wd = v.s;
  pos = p;
}

inline OneBit::OneBit(unsigned long* w, int p)
{
  wd = w;
  pos = p;
}

inline OneBit::~OneBit() {}

inline int OneBit::operator int()
{
  return (wd[(unsigned)(pos) / BITVEC_BPW] & (1 << (pos & (BITVEC_BPW - 1)))) != 0;
}

inline int operator = (int& a, OneBit& b)
{
  return a = (b.wd[(unsigned)(b.pos) / BITVEC_BPW] & (1 << (b.pos & (BITVEC_BPW - 1)))) != 0;
}

inline int OneBit::operator = (int b)
{
  if (b)
    wd[(unsigned)(pos) / BITVEC_BPW] |= (1 << (pos & (BITVEC_BPW - 1)));
  else
    wd[(unsigned)(pos) / BITVEC_BPW] &= ~(1 << (pos & (BITVEC_BPW - 1)));
  return b;
}

inline OneBit BitVec::operator [] (int i)
{
  if ((unsigned)(i) >= (unsigned)(len))
    error("index out of range");
  return OneBit(s, i);
}

inline OneBit BitVec::elem (int i)
{
  return OneBit(s, i);
}



//#endif

#endif

