// Iostreams base classes -*- C++ -*-

// Copyright (C) 1997,1998 Cygnus Solutions
//
// This file is part of the libstdc++ version 3 distribution.
//
// This software is a copyrighted work licensed under the terms of the
// Cygnus libstdc++ license. Please consult the file LICENSE.STD for
// details.

//
// ISO C++ working draft paper: 27.4  Iostreams base classes
//

#include <bits/std_ios.h>
#include <libio.h>  /* non-standard */
#include <bits/std_new.h>
#include <bits/std_string.h>

#include <unistd.h> /* non-standard */

#if _G_USE_NAMESPACE
namespace std {
#endif

template class basic_ios<char>;
template class basic_ios<wchar_t>;

void
ios_base::clear (iostate __state)
{
  if (((_M_state = __state) & exceptions ()) != 0)
    {
#ifdef _G_USE_EXCEPTIONS
       throw failure ("unmasked state event in basic_ios object");
#endif
    }
}


// 27.4.2.5  ios_base storage functions

int
ios_base::xalloc () throw()
{
  // XXX MT
  static int top = 4; // XXX should be a symbol. (Reserve 0..3 for builtins.)
  return top++;
}

// 27.4.2.5  iword/pword storage

ios_base::_Words&
ios_base::_M_grow_words(int ix)
{
  // precondition: _M_word_limit <= ix
  _Words zero = { 0, 0 };
  int newlimit = _S_local_words;
  _Words* words = _M_word_array;
  int i = 0;
  if (_S_local_words <= ix)
    {
      newlimit = ix+1;
      try
        { words = new _Words[ix+1]; }
      catch (...)
	{
	  _M_dummy = zero;  // XXX MT? Not on "normal" machines.
	  clear(rdstate () | badbit);  // may throw
	  return _M_dummy;
	}
      do { words[i] = _M_words[i]; } while (++i < _M_word_limit);
      if (_M_words != _M_word_array) delete [] _M_words;
    }

  do { words[i] = zero; } while (++i < newlimit);
  _M_words = words;
  _M_word_limit = newlimit;
  return words[ix];
}

// 27.4.2.6  ios_base callbacks

struct ios_base::_Callback_list
{
  _Callback_list* _M_next;
  ios_base::event_callback _M_fn;
  int _M_index;
  int _M_refcount;  // 0 means one reference.

  _Callback_list (ios_base::event_callback fn, int index, _Callback_list* __cb)
    : _M_next (__cb), _M_fn (fn), _M_index (index), _M_refcount (0) { }
  void _M_add_reference () { ++_M_refcount; } // XXX MT
  int _M_remove_reference () { return _M_refcount--; }  // 0 => OK to delete
};

void
ios_base::register_callback (event_callback fn, int index)
{
  _M_callbacks = new _Callback_list (fn, index, _M_callbacks);
}

void
ios_base::_M_call_callbacks (event ev) throw()
{
  for (_Callback_list* run = _M_callbacks; run; run = run->_M_next)
    {
      try { (*run->_M_fn) (ev, *this, run->_M_index); } catch (...) {}
    }
}

void
ios_base::_M_dispose_callbacks ()
{
  _Callback_list* __p = _M_callbacks;
  while (__p && __p->_M_remove_reference () == 0)
    {
      _Callback_list* __next = __p->_M_next;
      delete __p;
      __p = __next;
    }
    _M_callbacks = 0;
}

void
ios_base::_M_init()   // called only by basic_ios<>::init.
{
  // XXX may be called more than once
  _M_flags = skipws | dec;
  _M_width = 0;
  _M_precision = 6;
  _M_callbacks = 0;
  _M_words = 0;
  _M_word_limit = 0;
  _M_state = goodbit;
  _M_exceptions = goodbit;
  _M_locale = locale ();
  // no init needed for _M_word_array or _M_dummy.
}

void
ios_base::_M_copy_base(ios_base& __rhs)
{
  // alloc any new word array first, so if it fails we have "rollback".
  _Words* __words = (__rhs._M_word_limit <= _S_local_words) ?
     _M_word_array : new _Words[__rhs._M_word_limit];
  // bump refs before doing callbacks, for safety.
  _Callback_list* __cb = __rhs._M_callbacks;
  if (__cb) __cb->_M_add_reference ();

  _M_call_callbacks (erase_event);

  if (_M_words != _M_word_array) delete [] _M_words;
  _M_dispose_callbacks ();
  _M_callbacks = __cb;  // note: don't want any added during above.
  for (int __i = 0; __i < __rhs._M_word_limit; ++__i)
    __words[__i] = __rhs._M_words[__i];
  if (_M_words != _M_word_array) delete [] _M_words;
  _M_words = __words;
  _M_word_limit = __rhs._M_word_limit;
#if 0
  // Per 27.1.1.1, does not call imbue(); but must trash all
  // caches associated with imbue(); must not imbue on the sbuf.
  _M_locale = __rhs._M_locale;
#endif
  flags (__rhs.flags ());
  width (__rhs.width ());
  precision (__rhs.precision ());
}

// 27.4.2.3  ios_base locale functions
locale
ios_base::imbue (const locale& loc)
{
  locale tmp = _M_locale;
  // make sure there's a callback for the format caches so they will be
  // marked dirty.o
  _Format_cache<char>::_S_get(*this);
  _Format_cache<wchar_t>::_S_get(*this);
  _M_call_callbacks (imbue_event);
  _M_locale = loc;
  // XXX not done?
  return tmp;
}

ios_base::ios_base ()
{
  // do nothing; init() does it.  Static init to 0 makes everything sane.
}

// 27.4.2.7  ios_base constructors/destructors
ios_base::~ios_base ()
{
  _M_call_callbacks (erase_event);
  _M_dispose_callbacks ();
  if (_M_words != _M_word_array) delete [] _M_words;

  // XXX done?
}

// Manipulators:
  ios_base&
  boolalpha (ios_base& __str)
  {
    __str.setf (ios_base::boolalpha);
    return __str;
  }

  ios_base&
  noboolalpha (ios_base& __str)
  {
    __str.unsetf (ios_base::boolalpha);
    return __str;
  }

  ios_base&
  showbase (ios_base& __str)
  {
    __str.setf (ios_base::showbase);
    return __str;
  }

  ios_base&
  noshowbase (ios_base& __str)
  {
    __str.unsetf (ios_base::showbase);
    return __str;
  }

  ios_base&
  showpoint (ios_base& __str)
  {
    __str.setf (ios_base::showpoint);
    return __str;
  }

  ios_base&
  noshowpoint (ios_base& __str)
  {
    __str.unsetf (ios_base::showpoint);
    return __str;
  }

  ios_base&
  showpos (ios_base& __str)
  {
    __str.setf (ios_base::showpos);
    return __str;
  }

  ios_base&
  noshowpos (ios_base& __str)
  {
    __str.unsetf (ios_base::showpos);
    return __str;
  }

  ios_base&
  skipws (ios_base& __str)
  {
    __str.setf (ios_base::skipws);
    return __str;
  }

  ios_base&
  noskipws (ios_base& __str)
  {
    __str.unsetf (ios_base::skipws);
    return __str;
  }

  ios_base&
  uppercase (ios_base& __str)
  {
    __str.setf (ios_base::uppercase);
    return __str;
  }

  ios_base&
  nouppercase (ios_base& __str)
  {
    __str.unsetf (ios_base::uppercase);
    return __str;
  }

  ios_base&
  unitbuf (ios_base& __str)
  {
    __str.setf (ios_base::unitbuf);
    return __str;
  }

  ios_base&
  nounitbuf (ios_base& __str)
  {
    __str.unsetf (ios_base::unitbuf);
    return __str;
  }


  // Adjustfield:
  ios_base&
  internal (ios_base& __str)
  {
    __str.setf (ios_base::internal, ios_base::adjustfield);
    return __str;
  }

  ios_base&
  left (ios_base& __str)
  {
    __str.setf (ios_base::left, ios_base::adjustfield);
    return __str;
  }

  ios_base&
  right (ios_base& __str)
  {
    __str.setf (ios_base::right, ios_base::adjustfield);
    return __str;
  }


  // Basefield:
  ios_base&
  dec (ios_base& __str)
  {
    __str.setf (ios_base::dec, ios_base::basefield);
    return __str;
  }

  ios_base&
  hex (ios_base& __str)
  {
    __str.setf (ios_base::hex, ios_base::basefield);
    return __str;
  }

  ios_base&
  oct (ios_base& __str)
  {
    __str.setf (ios_base::oct, ios_base::basefield);
    return __str;
  }


  // Floatfield:
  ios_base&
  fixed (ios_base& __str)
  {
    __str.setf (ios_base::fixed, ios_base::floatfield);
    return __str;
  }

  ios_base&
  scientific (ios_base& __str)
  {
    __str.setf (ios_base::scientific, ios_base::floatfield);
    return __str;
  }

  template<>
    _Format_cache<char>::_Format_cache ()
  : _M_valid(true)
  , _M_decsep('.')
  , _M_grsep(',')
  , _M_use_grouping(false)
  //  , _M_grouping(string())
  {
    memcpy(_M_literals,_FORMAT_CACHE_NUMERIC_LITERALS, _S_literal_count);
    _M_boolnames[0] = "false";
    _M_boolnames[1] = "true";
  }

  template<>
    _Format_cache<wchar_t>::_Format_cache ()
  : _M_valid(true)
  , _M_decsep(L'.')
  , _M_grsep(L',')
  , _M_use_grouping(false)
  //  , _M_grouping(wstring())
  {
    static char const* lits = _FORMAT_CACHE_NUMERIC_LITERALS;
    for (int i = 0; i < _S_literal_count; ++i)
      _M_literals[i] = wchar_t(lits[i]);
    _M_boolnames[0] = L"false";
    _M_boolnames[1] = L"true";
  }

#if _G_USE_NAMESPACE
}
#endif
