/* Serial interface for mailbox connections on VMS systems
   Copyright 1997 Free Software Foundation, Inc.

   Contributed by Douglas B. Rupp, rupp@gnat.com

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include "defs.h"
#include "serial.h"
#include "signals.h"
#include "gdb_string.h"
#include "inferior.h"

#include <descrip.h>
#include <dvidef.h>
#include <iodef.h>
#include <iosbdef.h>
#include <lib$routines.h>
#include <jpidef.h>
#include <lnmdef.h>
#include <ssdef.h>
#include <starlet.h>

#define BUFMAX 2048 /* coordinate with evax-stub.c BUFMAX */

struct mbox_ttystate
{
  int bogus;
};

static int mbox_open PARAMS ((serial_t scb, const char *name));
static void mbox_raw PARAMS ((serial_t scb));
static int mbox_readchar PARAMS ((serial_t scb, int timeout));
static int mbox_setbaudrate PARAMS ((serial_t scb, int rate));
static int mbox_setstopbits PARAMS ((serial_t scb, int num));
static int mbox_write PARAMS ((serial_t scb, const char *str, int len));
/* FIXME: static void mbox_restore PARAMS ((serial_t scb)); */
static void mbox_close PARAMS ((serial_t scb));
static serial_ttystate mbox_get_tty_state PARAMS ((serial_t scb));
static int mbox_set_tty_state PARAMS ((serial_t scb, serial_ttystate state));

/* Open up a mailbox */

static struct {
  unsigned short buflen, item_code;
  void *bufaddr;
  void *retlenaddr;
  unsigned int terminator;
} itm_lst;

static int
mbox_open(scb, name)
     serial_t scb;
     const char *name;
{
  int status;
  struct dsc$descriptor_s mbox;
  unsigned int attr = LNM$M_CASE_BLIND;
  $DESCRIPTOR(tabledsc, "LNM$GDB");
  int gdb_mbx_exists = 0;
  unsigned int mbx_dev_len;
  char mbx_dev_buff [64];
  struct dsc$descriptor_s mbx_dev;

  mbox.dsc$w_length = strlen (name);
  mbox.dsc$b_dtype = DSC$K_DTYPE_T;
  mbox.dsc$b_class = DSC$K_CLASS_S;
  mbox.dsc$a_pointer = (char *) name;

  itm_lst.buflen = 64;
  itm_lst.item_code = LNM$_STRING;
  itm_lst.bufaddr = mbx_dev_buff;
  itm_lst.retlenaddr = &mbx_dev_len;
  itm_lst.terminator = 0;

  status = sys$trnlnm
    (&attr,      /* attr */
     &tabledsc,  /* tabnam */
     &mbox,      /* lognam */
     0,          /* acmode */
     &itm_lst);

  if (status == SS$_NOLOGNAM)
    {
      gdb_mbx_exists = 0;
    }
  else if (status == SS$_NORMAL)
    {
      gdb_mbx_exists = 1;
    }

  if (!gdb_mbx_exists)
    {
      status = sys$crembx
	(0,          /* pmmflg */
	 (unsigned short *)&scb->fd, /* chan */
	 BUFMAX,     /* maxmsg */
	 BUFMAX,     /* bufquo */
	 0,          /* promsk */
	 0,          /* acmode */
	 0,          /* lognam */
	 0,          /* flags */
	 0);         /* nullarg */
      if ((status & 1) != 1)
	return -1;

      itm_lst.buflen = 64;
      itm_lst.item_code = DVI$_DEVNAM;
      itm_lst.bufaddr = mbx_dev_buff;
      itm_lst.retlenaddr = &mbx_dev_len;
      itm_lst.terminator = 0;

      status = sys$getdviw
	(0,          /* efn */
	 (unsigned short)scb->fd, /* chan */
	 0,          /* devnam */
	 &itm_lst,   /* itmlst */
	 0,          /* iosb */
	 0,          /* astadr */
	 0,          /* astprm */
	 0);         /* nullarg */
      if ((status & 1) != 1)
	return -1;

      mbx_dev.dsc$w_length = strlen (mbx_dev_buff);
      mbx_dev.dsc$b_dtype = DSC$K_DTYPE_T;
      mbx_dev.dsc$b_class = DSC$K_CLASS_S;
      mbx_dev.dsc$a_pointer = mbx_dev_buff;;

      status = lib$set_logical (&mbox, &mbx_dev, &tabledsc, 0, 0);

      if (status == SS$_NOLOGTAB)
	error ("GDB Logical Name Table not found, see release notes.");

      if ((status & 1) != 1)
	return -1;
    }
  else
    {
      mbx_dev.dsc$w_length = strlen (mbx_dev_buff);
      mbx_dev.dsc$b_dtype = DSC$K_DTYPE_T;
      mbx_dev.dsc$b_class = DSC$K_CLASS_S;
      mbx_dev.dsc$a_pointer = mbx_dev_buff;;

      status = sys$assign (&mbx_dev, (unsigned short *)&scb->fd, 0, 0, 0);
      if ((status & 1) != 1)
	return -1;
    }

  if (scb->fd < 0)
    return -1;

  return 0;
}

static serial_ttystate
mbox_get_tty_state(scb)
     serial_t scb;
{
  struct mbox_ttystate *state;

  state = (struct mbox_ttystate *)xmalloc(sizeof *state);

  return (serial_ttystate)state;
}

static int
mbox_set_tty_state(scb, ttystate)
     serial_t scb;
     serial_ttystate ttystate;
{
  struct mbox_ttystate *state;

  state = (struct mbox_ttystate *)ttystate;

  return 0;
}

static int
mbox_return_0 (scb)
     serial_t scb;
{
  return 0;
}

static void
mbox_raw(scb)
     serial_t scb;
{
  return;			/* Always in raw mode */
}

/* Read a character with user-specified timeout.  TIMEOUT is number of seconds
   to wait, or -1 to wait forever.  Use timeout of 0 to effect a poll.  Returns
   char if successful.  Returns -2 if timeout expired, EOF if line dropped
   dead, or -3 for any other error (see errno in that case). */

static int
mbox_readchar(scb, timeout)
     serial_t scb;
     int timeout;
{
  unsigned status;
  int ioflags;
  IOSB mbxiosb;
  int i;

  if (scb->bufcnt-- > 0)
    return *scb->bufp++;

  ioflags = IO$_READVBLK;

  status = sys$qio (MBXQIO_EF, scb->fd, ioflags, &mbxiosb, 0, 0,
		     &scb->buf[0], BUFMAX, 0, 0, 0, 0);
  if ((status & 1) != 1)
    {
      LIB$SIGNAL (status);
      return -3;
    }

  sys$wflor (MBXQIO_EF, MBXQIO_EF_MASK | TIMER_EF_MASK);
  sys$readef (MBXQIO_EF, &status);

  if (status & TIMER_EF_MASK)
    {
      /* Timer went off */

      int targpid = inferior_pid;
      int jpi_pid;
      int jpi_itemcode = JPI$_PID;

      /* See if process still exists */
      status = lib$getjpi
	(&jpi_itemcode, &targpid, 0, &jpi_pid, 0, 0);

      if ((status & 1) == 1 && targpid == jpi_pid)
	{
	  /* Need to wait a little longer */
	  fprintf (stderr, "waiting ...\n");
	  if (query ("Target process doesn't respond, delete it? "))
	    {
	      sys$delprc ((unsigned int *)&inferior_pid, 0);
	      return SERIAL_TIMEOUT;
	    }
	  else
	    {
	      error ("Not confirmed.");
	      return SERIAL_TIMEOUT;
	    }
	}
      else if (status == SS$_NONEXPR)
	{
	  /* No longer exists, so return a time out error */
	  return SERIAL_TIMEOUT;
	}
      else
	{
	  /* Don't know what else it could be */
	  LIB$SIGNAL (status);
	}
    }

  scb->bufcnt = mbxiosb.iosb$w_bcnt;

  if ((mbxiosb.iosb$w_status & 1) != 1)
    {
      if (mbxiosb.iosb$w_status == 0 && mbxiosb.iosb$w_bcnt == 0)
	return SERIAL_TIMEOUT;
      else
	return SERIAL_ERROR;
    }

  if (scb->bufcnt <= 0)
    return SERIAL_ERROR;	/* Got an error from read */

  scb->bufcnt--;
  scb->bufp = scb->buf;
  return *scb->bufp++;
}

static int
mbox_noflush_set_tty_state (scb, new_ttystate, old_ttystate)
     serial_t scb;
     serial_ttystate new_ttystate;
     serial_ttystate old_ttystate;
{
  return 0;
}

static void
mbox_print_tty_state (scb, ttystate)
     serial_t scb;
     serial_ttystate ttystate;
{
  /* Nothing to print.  */
  return;
}

static int
mbox_setbaudrate(scb, rate)
     serial_t scb;
     int rate;
{
  return 0;			/* Never fails! */
}

static int
mbox_setstopbits(scb, num)
     serial_t scb;
     int num;
{
  return 0;			/* Never fails! */
}

static int
mbox_write(scb, str, len)
     serial_t scb;
     const char *str;
     int len;
{
  int status;
  IOSB mbxiosb;

  while (len > 0)
    {
      status = sys$qiow (0, scb->fd, IO$_WRITEVBLK, &mbxiosb, 0, 0,
			 (char *)str, len, 0, 0, 0, 0);
      if ((status & 1) != 1) return 1;
      if ((mbxiosb.iosb$w_status & 1) != 1) return 1;
      len -= mbxiosb.iosb$w_bcnt;
      str += mbxiosb.iosb$w_bcnt;
    }
  return 0;
}

static void
mbox_close(scb)
     serial_t scb;
{
  if (scb->fd < 0)
    return;

  sys$dassgn (scb->fd);
  scb->fd = -1;
}

static struct serial_ops mbox_ops =
{
  "gdbmbox",
  0,
  mbox_open,
  mbox_close,
  mbox_readchar,
  mbox_write,
  mbox_return_0, /* flush output */
  mbox_return_0, /* flush input */
  mbox_return_0, /* send break */
  mbox_raw,
  mbox_get_tty_state,
  mbox_set_tty_state,
  mbox_print_tty_state,
  mbox_noflush_set_tty_state,
  mbox_setbaudrate,
  mbox_setstopbits,
};

void
_initialize_ser_mbox ()
{
  serial_add_interface (&mbox_ops);
}
