
/*--------------------------------------------------------------------*/
/*--- A header file containing architecture-specific definitions   ---*/
/*--- for Valgrind on PowerPC-architecture processors.		   ---*/
/*---                                               vg_archdefs.h  ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Valgrind, an extensible
   emulator for monitoring program execution on Unixes.

   Copyright (C) 2000-2004 Julian Seward 
      jseward@acm.org
   Copyright (C) 2004 Paul Mackerras
      paulus@samba.org

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#ifndef __VG_ARCHDEFS_H
#define __VG_ARCHDEFS_H

/* Total number of spill slots available for allocation, if a TempReg
   doesn't make it into a RealReg.  Just bomb the entire system if
   this value is too small; we don't expect it will ever get
   particularly high. */
#define VG_MAX_SPILLSLOTS 64

/* Valgrind's stack sizes, in words */
#define VG_SIGSTACK_SIZE_W    16384

/* Architecture-specific part of a ThreadState */
typedef struct arch_thread {
   UInt m_gpr[32];
   UInt m_eip;
   UInt m_ctr;
   UInt m_lr;
   UInt m_xer;
   UInt m_cr;
   UInt sh_gpr[32];
   UInt sh_ctr;
   UInt sh_lr;
   UInt sh_xer;
   UInt sh_cr;
   UInt m_orig_gpr3;
   UInt m_result;

   double m_fpr[33];		/* includes fpscr as [32] */

   /* vector state; needs to be aligned on a 16-byte boundary */
   UInt m_vr[33*4] __attribute__((__aligned__(16)));
} arch_thread_t;

/* Accessors for the arch_thread_t */
#define ARCH_INSTR_POINTER(regs)	((regs).m_eip)
#define ARCH_STACK_POINTER(regs)	((regs).m_gpr[1])
#define ARCH_FRAME_POINTER(regs)	((regs).m_gpr[1])

/* are these linux-specific? */
#define ARCH_SYSCALL_NUM(regs)		((regs).m_gpr[0])
#define ARCH_SYSCALL_RET(regs)		((regs).m_result)
#define ARCH_SYSCALL_ARG1(regs)		((regs).m_orig_gpr3)
#define ARCH_SYSCALL_ARG2(regs)		((regs).m_gpr[4])
#define ARCH_SYSCALL_ARG3(regs)		((regs).m_gpr[5])
#define ARCH_SYSCALL_ARG4(regs)		((regs).m_gpr[6])
#define ARCH_SYSCALL_ARG5(regs)		((regs).m_gpr[7])
#define ARCH_SYSCALL_ARG6(regs)		((regs).m_gpr[8])

/* XXX should this be setting shadow regs? */
#define ARCH_PRE_SYSCALL_RESULT(regs, val)			\
   do {								\
      (regs).m_result = (val);					\
      (regs).m_gpr[3] = (regs).m_result;			\
      (regs).m_cr &= ~0x10000000;	/* clear cr0.SO */	\
      if ((regs).m_result > -4096) {				\
	 (regs).m_gpr[3] = - (regs).m_result;			\
	 (regs).m_cr |= 0x10000000;				\
      }								\
      (regs).m_gpr[0] = 0;					\
   } while (0)

#define ARCH_CLREQ_ARGS(regs)		((regs).m_gpr[3])

/* Interesting register numbers */
#define R_SYSCALL_NUM	0
#define R_SYSCALL_ARG1	3
#define R_SYSCALL_RET	3
#define R_STACK_PTR	1
#define R_FRAME_PTR	1

/* Stack frame layout and linkage */
#define FIRST_STACK_FRAME(sp)	(((UInt*)(sp))[0])
#define STACK_FRAME_RET(sp)	(((UInt*)(sp))[1])
#define STACK_FRAME_NEXT(sp)	(((UInt*)(sp))[0])

#define GET_STACK_ADDRS(bp, sp)		do {		\
   asm("mr %0,1; mr %1,1" : "=r" (bp), "=r" (sp));	\
} while (0)

/* Write a value to a client's thread register, and shadow (if necessary) */
#define SET_THREAD_REG( zztid, zzval, zzreg, zzREG, zzevent, zzargs... ) \
   do { VG_(threads)[zztid].arch.m_##zzreg = (zzval);			 \
        VG_TRACK( zzevent, zztid, zzREG, ##zzargs );			 \
   } while (0)

#define SET_SYSCALL_RETVAL(zztid, zzval)				\
   do {									\
      if ((UInt)(zzval) > -4096) {					\
	 /* error return */						\
	 SET_THREAD_REG(zztid, -zzval,					\
			gpr[3], 3, post_reg_write_syscall_return);	\
	 SET_THREAD_REG(zztid, VG_(threads)[zztid].arch.m_cr | 0x10000000,\
			cr, R_CR, post_reg_write_syscall_return);	\
      } else {								\
	 SET_THREAD_REG(zztid, zzval,					\
			gpr[3], 3, post_reg_write_syscall_return);	\
      }									\
   } while (0)

#define SET_SIGNAL_GPR(zztid, n, zzval) \
   SET_THREAD_REG(zztid, zzval, gpr[n], n, post_reg_write_deliver_signal)

#define SET_SIGNAL_LR(zztid, zzval) \
   SET_THREAD_REG(zztid, zzval, lr, R_LR, post_reg_write_deliver_signal)

#define SET_CLREQ_RETVAL(zztid, zzval) \
   SET_THREAD_REG(zztid, zzval, gpr[3], 3, post_reg_write_clientreq_return)

#define SET_CLCALL_RETVAL(zztid, zzval, f) \
   SET_THREAD_REG(zztid, zzval, gpr[3], 3, post_reg_write_clientcall_return, f)

#define SET_PTHREQ_ESP(zztid, zzval) \
   SET_THREAD_REG(zztid, zzval, gpr[1], 1, post_reg_write_pthread_return)

#define SET_PTHREQ_ARG(zztid, zzval) \
   SET_THREAD_REG(zztid, zzval, gpr[3], 3, post_reg_write_pthread_return)

#define SET_PTHREQ_RETADDR(zztid, zzval) \
   SET_THREAD_REG(zztid, zzval, lr, R_LR, post_reg_write_pthread_return)

#define SET_PTHREQ_RETVAL(zztid, zzval) \
   SET_THREAD_REG(zztid, zzval, gpr[3], 3, post_reg_write_pthread_return)

/* ---------------------------------------------------------------------
   Exports of vg_ppc_helpers.S
   ------------------------------------------------------------------ */

extern void VG_(helper_loadfpu)(double *fprs);
extern void VG_(helper_storefpu)(double *fprs);
extern void VG_(load_fpu_state)(double *fprs);
extern void VG_(save_fpu_state)(double *fprs);

extern void VG_(helper_loadvec)(UInt *vrs);
extern void VG_(load_vec_state)(UInt *vrs);
extern void VG_(save_vec_state)(UInt *vrs);

extern void VG_(helper_cache_inval)(Addr address);

/* -----------------------------------------------------
   Exports of arch/powerpc-common/vg_ppc_state.c
   -------------------------------------------------- */

/* For setting up baseBlock */
extern void arch_alloc_low_baseBlock ( Addr client_eip, Addr esp_at_startup );
extern void arch_alloc_baseBlock( Addr client_eip, Addr esp_at_startup );

extern void VG_(arch_process_options)(void);

extern void VG_(arch_load_state) ( arch_thread_t *, ThreadId );
extern void VG_(arch_save_state) ( arch_thread_t *, ThreadId );
extern void VG_(arch_clear_thread) ( arch_thread_t * );
extern void VG_(arch_thread_init) ( arch_thread_t * );
extern void VG_(arch_thread_cleanup) ( arch_thread_t * );
extern void VG_(arch_child_setup) ( arch_thread_t *, arch_thread_t * );
extern Int VG_(arch_set_child_regs_from_BB)( Int );
extern Int VG_(arch_set_child_regs)( Int pid, arch_thread_t * );
extern UInt *VG_(arch_reg_addr_from_BB)( Int reg );
extern UInt *VG_(arch_reg_addr)( Int reg, arch_thread_t * );
extern void arch_set_arg_and_ret( ThreadId tid, UInt arg, Addr ret );
extern void arch_thread_initial_stack( ThreadId tid, UInt arg, Addr ret );

/* -----------------------------------------------------
   Exports of arch/powerpc-common/vg_ppc_signal.c
   -------------------------------------------------- */

extern void VG_(arch_deliver_signal)(ThreadId tid, Addr esp_top_of_frame,
				     const vki_ksiginfo_t *siginfo,
				     void *handler, UInt flags,
				     const vki_ksigset_t *mask);
extern Int VG_(arch_signal_return)(ThreadId tid, Bool has_siginfo);
extern void VG_(arch_fill_elfregs_from_BB)(struct user_regs_struct *);
extern void VG_(arch_fill_elfregs)(struct user_regs_struct *,
				   const arch_thread_t *);
extern void VG_(arch_fill_elffpregs_from_BB)(elf_fpregset_t *);
extern void VG_(arch_fill_elffpregs)(elf_fpregset_t *, const arch_thread_t *);
extern void VG_(arch_fill_elffpxregs_from_BB)(elf_fpxregset_t *);
extern void VG_(arch_fill_elffpxregs)(elf_fpxregset_t *,
				      const arch_thread_t *);
extern void VG_(arch_flush_state)( struct vki_ucontext * );

/* -----------------------------------------------------
   Exports of arch/powerpc-common/vg_ppc_syscall.c
   -------------------------------------------------- */

extern void VG_(arch_restart_syscall)(arch_thread_t *);

/* -----------------------------------------------------
   Read-write parts of baseBlock.
   -------------------------------------------------- */

/* State of the simulated CPU. */
extern Int VGOFF_(m_gpr);
extern Int VGOFF_(m_esp);
extern Int VGOFF_(m_cr);
extern Int VGOFF_(m_lr);
extern Int VGOFF_(m_ctr);
extern Int VGOFF_(m_xer);
extern Int VGOFF_(m_eip);
extern Int VGOFF_(spillslots);
extern Int VGOFF_(sh_gpr);
extern Int VGOFF_(sh_cr);
extern Int VGOFF_(sh_lr);
extern Int VGOFF_(sh_ctr);
extern Int VGOFF_(sh_xer);
extern Int VGOFF_(fpu_state_ptr);
extern Int VGOFF_(vec_state_ptr);
extern Int VGOFF_(helper_cache_inval);
extern Int VGOFF_(total_synth_instrs);
extern Int VGOFF_(total_real_instrs);

#define VGOFF_INSTR_POINTER	VGOFF_(m_eip)
#define VGOFF_STACK_POINTER	(VGOFF_(m_gpr)+1)
#define VGOFF_FRAME_POINTER	(VGOFF_(m_gpr)+1)

/* -----------------------------------------------------
   Read-only parts of baseBlock.
   -------------------------------------------------- */

extern Int VGOFF_(helper_loadfpu);
extern Int VGOFF_(helper_storefpu);
extern Int VGOFF_(helper_loadvec);
extern Int VGOFF_(helper_cache_inval);

/* Nb: Most helper offsets are in include/vg_skin.h, for use by skins */
extern Int VGOFF_(helper_undefined_instruction);

extern Addr VG_(dispatch_sp);

/* Struct used to convey architecture-specific info from pthread_create
   to thread_wrapper in vg_libpthread.c.  */
typedef
   struct arch_thread_info {
      void *tls_data;
   } arch_thread_info_t;

/* Exports of vg_ppc_libpthread.c */
void __vg_arch_thread_exit(void);
void __vg_arch_thread_create(arch_thread_info_t *info);
void __vg_arch_thread_wrapper(arch_thread_info_t *info);
int  __vg_arch_has_tls(void);

/* How vg_tt and VG_(tt_fast) are indexed. */
#define VG_TT_HASH(addr)        (((UInt)(addr) >> 2) % VG_TT_SIZE)
#define VG_TT_FAST_INDEX(addr)  (((UInt)(addr) >> 2) & VG_TT_FAST_MASK)

#define DO_ARCH_REDIRECTS

#endif /* __VG_ARCHDEFS_H */
