
/*--------------------------------------------------------------------*/
/*--- PowerPC-specific stuff for the core.    powerpc/core_arch.h  ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Valgrind, an extensible
   emulator for monitoring program execution on Unixes.

   Copyright (C) 2004 Paul Mackerras
      paulus@samba.org

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#ifndef __PPC_CORE_ARCH_H
#define __PPC_CORE_ARCH_H

#include "core_arch_asm.h"
#include "tool_arch.h"

/* ---------------------------------------------------------------------
   Interesting registers
   ------------------------------------------------------------------ */

// Accessors for the arch_thread_t
#define ARCH_INSTR_PTR(regs)	((regs).m_eip)
#define ARCH_STACK_PTR(regs)	((regs).m_gpr[1])
#define ARCH_FRAME_PTR(regs)	((regs).m_gpr[1])

#define ARCH_CLREQ_ARGS(regs)	((regs).m_gpr[3])
#define ARCH_PTHREQ_RET(regs)	((regs).m_gpr[3])
#define ARCH_CLREQ_RET(regs)	((regs).m_gpr[3])

// Accessors for the baseBlock
#define R_STACK_PTR                    1
#define R_FRAME_PTR                    1

#define R_CLREQ_RET                    3
#define R_PTHREQ_RET                   3

// Stack frame layout and linkage
#define FIRST_STACK_FRAME(sp)	(((UWord*)(sp))[0])
#define STACK_FRAME_RET(sp)	(((UWord*)(sp))[1])
#define STACK_FRAME_NEXT(sp)	(((UWord*)(sp))[0])

// Offsets of interesting registers
#define VGOFF_INSTR_PTR	VGOFF_(m_eip)
#define VGOFF_STACK_PTR	(VGOFF_(m_gpr)+1)
#define VGOFF_FRAME_PTR	(VGOFF_(m_gpr)+1)

// Get stack pointer and frame pointer
#define ARCH_GET_REAL_STACK_PTR(sp) do {	\
   asm("mr %0,1" : "=r" (sp));			\
} while (0)

#define ARCH_GET_REAL_FRAME_PTR(bp) do {	\
   asm("mr %0,1" : "=r" (bp));			\
} while (0)


/* -----------------------------------------------------
   Read-write parts of baseBlock.
   -------------------------------------------------- */

/* State of the simulated CPU. */
extern Int VGOFF_(m_gpr);
extern Int VGOFF_(m_esp);
extern Int VGOFF_(m_cr);
extern Int VGOFF_(m_lr);
extern Int VGOFF_(m_ctr);
extern Int VGOFF_(m_xer);
extern Int VGOFF_(m_eip);
extern Int VGOFF_(spillslots);
extern Int VGOFF_(sh_gpr);
extern Int VGOFF_(sh_cr);
extern Int VGOFF_(sh_lr);
extern Int VGOFF_(sh_ctr);
extern Int VGOFF_(sh_xer);
extern Int VGOFF_(fpu_state_ptr);
extern Int VGOFF_(vec_state_ptr);
extern Int VGOFF_(helper_cache_inval);
extern Int VGOFF_(total_synth_instrs);
extern Int VGOFF_(total_real_instrs);

/* -----------------------------------------------------
   Read-only parts of baseBlock.
   -------------------------------------------------- */

extern Int VGOFF_(helper_loadfpu);

extern Int VGOFF_(helper_storefpu);
extern Int VGOFF_(helper_loadvec);
extern Int VGOFF_(helper_cache_inval);

/* Nb: Most helper offsets are in include/vg_skin.h, for use by skins */
extern Int VGOFF_(helper_undefined_instruction);

/* Architecture-specific part of a ThreadState */
typedef struct arch_thread {
   UInt m_gpr[32];
   UInt m_eip;
   UInt m_ctr;
   UInt m_lr;
   UInt m_xer;
   UInt m_cr;
   UInt sh_gpr[32];
   UInt sh_ctr;
   UInt sh_lr;
   UInt sh_xer;
   UInt sh_cr;
   UInt m_orig_gpr3;
   UInt m_result;

   double m_fpr[33];		/* includes fpscr as [32] */

   /* vector state; needs to be aligned on a 16-byte boundary */
   UInt m_vr[33*4] __attribute__((__aligned__(16)));
} arch_thread_t;


/* ---------------------------------------------------------------------
   Elf stuff
   ------------------------------------------------------------------ */

#define VG_ELF_ENDIANNESS     ELFDATA2MSB
#define VG_ELF_MACHINE        EM_PPC
#define VG_ELF_CLASS          ELFCLASS32


/* ---------------------------------------------------------------------
   Exports of vg_ppc_helpers.S
   ------------------------------------------------------------------ */

extern void VG_(helper_loadvec)(UInt *vrs);
extern void VG_(load_vec_state)(UInt *vrs);
extern void VG_(save_vec_state)(UInt *vrs);

extern void VG_(helper_cache_inval)(Addr address);


extern Addr VG_(dispatch_sp);

/* ---------------------------------------------------------------------
   libpthread stuff
   ------------------------------------------------------------------ */

struct arch_thread_aux {
   void *tls_data;
};

/* How vg_tt and VG_(tt_fast) are indexed. */
#define VG_TT_HASH(addr)        (((UInt)(addr) >> 2) % VG_TT_SIZE)
#define VG_TT_FAST_INDEX(addr)  (((UInt)(addr) >> 2) & VG_TT_FAST_MASK)

#define DO_ARCH_REDIRECTS

/* ---------------------------------------------------------------------
   Miscellaneous constants
   ------------------------------------------------------------------ */

// Total number of spill slots available for register allocation.
#define VG_MAX_SPILLSLOTS 64

// Valgrind's signal stack size, in words
#define VG_SIGSTACK_SIZE_W    16384

// base address of client address space
#define CLIENT_BASE	0x00000000ul

void call_on_stack(void (*func)(void), unsigned int stacktop);

#endif /* __PPC_CORE_ARCH_H */
