
/*--------------------------------------------------------------------*/
/*--- PowerPC-specific functions for handling signals.
                                                      ppc/signal.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Valgrind, an extensible
   emulator for monitoring program execution on Unixes.

   Copyright (C) 2000-2004 Julian Seward 
      jseward@acm.org
   Copyright (C) 2004 Paul Mackerras
      paulus@samba.org

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#include "core.h"

#define SET_SIGNAL_GPR(zztid, n, zzval) \
   do { VG_(threads)[zztid].arch.m_gpr[n] = (zzval);		\
        VG_TRACK( post_reg_write_deliver_signal, zztid, n);	\
   } while (0)

#define SET_SIGNAL_LR(zztid, zzval) \
   do { VG_(threads)[zztid].arch.m_lr = (zzval);		\
        VG_TRACK( post_reg_write_deliver_signal, zztid, R_LR);	\
   } while (0)

/* Structure containing bits of information that we want to save
   on signal delivery. */
struct vg_sig_private {
   UInt magicPI;
   UInt sigNo_private;
   ThreadStatus status;
   void *associated_mx;
   void *associated_cv;
   UInt sh_gpr[32];
   UInt sh_ctr;
   UInt sh_lr;
   UInt sh_xer;
   UInt sh_cr;
};

/* Structure put on stack for signal handlers with SA_SIGINFO clear. */
struct nonrt_sigframe {
   UInt gap1[16];
   struct vki_sigcontext sigcontext;
   struct vki_mcontext mcontext;
   struct vg_sig_private priv;
   unsigned char abigap[224];
};

/* Structure put on stack for signal handlers with SA_SIGINFO set. */
struct rt_sigframe {
   UInt gap1[20];
   vki_siginfo_t siginfo;
   struct vki_ucontext ucontext;
   struct vg_sig_private priv;
   unsigned char abigap[224];
};

static void stack_mcontext(struct vki_mcontext *mc, ThreadId tid, int ret,
			   UInt fault_addr)
{
   ThreadState *tst = &VG_(threads)[tid];

   VG_TRACK(pre_mem_write, Vg_CoreSignal, tid, "signal frame mcontext",
	    (Addr)mc, sizeof(struct vki_pt_regs));
   VG_(memcpy)(&mc->mc_gregs[VKI_PT_R0], tst->arch.m_gpr, 32 * sizeof(UWord));
   mc->mc_gregs[VKI_PT_NIP] = tst->arch.m_eip;
   mc->mc_gregs[VKI_PT_MSR] = 0xf032;	/* pretty arbitrary */
   mc->mc_gregs[VKI_PT_ORIG_R3] = tst->arch.m_orig_gpr3;
   mc->mc_gregs[VKI_PT_CTR] = tst->arch.m_ctr;
   mc->mc_gregs[VKI_PT_LNK] = tst->arch.m_lr;
   mc->mc_gregs[VKI_PT_XER] = tst->arch.m_xer;
   mc->mc_gregs[VKI_PT_CCR] = tst->arch.m_cr;
   mc->mc_gregs[VKI_PT_MQ] = 0;
   mc->mc_gregs[VKI_PT_TRAP] = 0;
   mc->mc_gregs[VKI_PT_DAR] = fault_addr;
   mc->mc_gregs[VKI_PT_DSISR] = 0;
   mc->mc_gregs[VKI_PT_RESULT] = 0;
   VG_TRACK(post_mem_write, (Addr)mc, sizeof(struct vki_pt_regs));

   /* XXX should do FP and vector regs */

   /* set up signal return trampoline */
   VG_TRACK(pre_mem_write, Vg_CoreSignal, tid, "signal frame mcontext",
	    (Addr)&mc->mc_pad, sizeof(mc->mc_pad));
   mc->mc_pad[0] = 0x38000000U + ret;	/* li 0,ret */
   mc->mc_pad[1] = 0x44000002U;		/* sc */
   VG_TRACK(post_mem_write, (Addr)&mc->mc_pad, sizeof(mc->mc_pad));
   /* probably should invalidate any translation of this area */

   /* set the signal handler to return to the trampoline */
   SET_SIGNAL_LR(tid, (Addr) &mc->mc_pad[0]);
}

void VGA_(push_signal_frame)(ThreadId tid, Addr sp_top_of_frame,
			     const vki_siginfo_t *siginfo,
			     void *handler, UInt flags,
			     const vki_sigset_t *mask)
{
   struct vg_sig_private *priv;
   Addr sp;
   ThreadState *tst;
   Int sigNo = siginfo->si_signo;

   /* Stack must be 16-byte aligned */
   sp_top_of_frame &= ~0xf;

   if (flags & VKI_SA_SIGINFO) {
      sp = sp_top_of_frame - sizeof(struct rt_sigframe);
   } else {
      sp = sp_top_of_frame - sizeof(struct nonrt_sigframe);
   }

   tst = & VG_(threads)[tid];

   /* For tracking memory events, indicate the entire frame has been
    * allocated, but pretend that only some of the words are written */
   VG_TRACK( new_mem_stack_signal, sp, sp_top_of_frame - sp );

   vg_assert(((UWord)sp & 0xf) == 0);

   /* Set up the stack chain pointer */
   VG_TRACK(pre_mem_write, Vg_CoreSignal, tid, "signal handler frame",
	    sp, sizeof(UWord));
   *(Addr *)sp = tst->arch.m_gpr[1];
   VG_TRACK(post_mem_write, sp, sizeof(UWord));
   
   if (flags & VKI_SA_SIGINFO) {
      struct rt_sigframe *frame = (struct rt_sigframe *) sp;
      struct vki_ucontext *ucp = &frame->ucontext;

      VG_TRACK(pre_mem_write, Vg_CoreSignal, tid, "signal frame siginfo",
	       (Addr)&frame->siginfo, sizeof(frame->siginfo));
      VG_(memcpy)(&frame->siginfo, siginfo, sizeof(*siginfo));
      VG_TRACK(post_mem_write, (Addr)&frame->siginfo, sizeof(frame->siginfo));

      VG_TRACK(pre_mem_write, Vg_CoreSignal, tid, "signal frame ucontext",
	       (Addr)ucp, offsetof(struct vki_ucontext, uc_pad));
      ucp->uc_flags = 0;
      ucp->uc_link = 0;
      ucp->uc_stack = tst->altstack;
      VG_TRACK(post_mem_write, (Addr)ucp,
	       offsetof(struct vki_ucontext, uc_pad));

      VG_TRACK(pre_mem_write, Vg_CoreSignal, tid, "signal frame ucontext",
	       (Addr)&ucp->uc_regs,
	       sizeof(ucp->uc_regs) + sizeof(ucp->uc_sigmask));
      ucp->uc_regs = &ucp->uc_mcontext;
      ucp->uc_sigmask = tst->sig_mask;
      VG_TRACK(post_mem_write, (Addr)&ucp->uc_regs,
	       sizeof(ucp->uc_regs) + sizeof(ucp->uc_sigmask));

      stack_mcontext(&ucp->uc_mcontext, tid, __NR_rt_sigreturn,
		     (UWord)siginfo->_sifields._sigfault._addr);
      priv = &frame->priv;

      SET_SIGNAL_GPR(tid, 4, (Addr) &frame->siginfo);
      SET_SIGNAL_GPR(tid, 5, (Addr) ucp);
      /* the kernel sets this, though it doesn't seem to be in the ABI */
      SET_SIGNAL_GPR(tid, 6, (Addr) &frame->siginfo);

   } else {
      /* non-RT signal delivery */
      struct nonrt_sigframe *frame = (struct nonrt_sigframe *) sp;
      struct vki_sigcontext *scp = &frame->sigcontext;

      VG_TRACK(pre_mem_write, Vg_CoreSignal, tid, "signal frame sigcontext",
	       (Addr)&scp->_unused[3], sizeof(*scp) - 3 * sizeof(UInt));
      scp->signal = sigNo;
      scp->handler = (Addr) handler;
      scp->oldmask = tst->sig_mask.sig[0];
      scp->_unused[3] = tst->sig_mask.sig[1];
      VG_TRACK(post_mem_write, (Addr)&scp->_unused[3],
	       sizeof(*scp) - 3 * sizeof(UInt));

      stack_mcontext(&frame->mcontext, tid, __NR_sigreturn,
		     (UWord)siginfo->_sifields._sigfault._addr);
      priv = &frame->priv;

      SET_SIGNAL_GPR(tid, 4, (Addr) scp);
   }

   priv->magicPI = 0x31415927;
   priv->sigNo_private = sigNo;

   priv->status = tst->status;
   if (priv->status == VgTs_WaitSys)
      priv->status = VgTs_Runnable;

   priv->associated_mx = tst->associated_mx;
   priv->associated_cv = tst->associated_cv;

   if (VG_(needs).shadow_regs) {
      VG_(memcpy)(priv->sh_gpr, tst->arch.sh_gpr, 32 * sizeof(UWord));
      priv->sh_cr = tst->arch.sh_cr;
      priv->sh_lr = tst->arch.sh_lr;
      priv->sh_ctr = tst->arch.sh_ctr;
      priv->sh_xer = tst->arch.sh_xer;
   }

   SET_SIGNAL_GPR(tid, 1, sp);
   SET_SIGNAL_GPR(tid, 3, sigNo);
   tst->arch.m_eip = (Addr) handler;

}

Int VGA_(pop_signal_frame)(ThreadId tid, Bool has_siginfo)
{
   ThreadState *tst;
   struct vg_sig_private *priv;
   Addr sp;
   UInt frame_size;
   struct vki_mcontext *mc;

   vg_assert(VG_(is_valid_tid)(tid));
   tst = &VG_(threads)[tid];

   /* Check that the stack frame looks valid */
   sp = tst->arch.m_gpr[1];
   vg_assert((sp & 0xf) == 0);
   frame_size = *(Addr *)sp - sp;

   if (has_siginfo) {
      struct rt_sigframe *frame = (struct rt_sigframe *)sp;
      vg_assert(frame_size == sizeof(*frame));
      mc = &frame->ucontext.uc_mcontext;
      priv = &frame->priv;
      tst->sig_mask = frame->ucontext.uc_sigmask;
   } else {
      struct nonrt_sigframe *frame = (struct nonrt_sigframe *)sp;
      vg_assert(frame_size == sizeof(*frame));
      mc = &frame->mcontext;
      priv = &frame->priv;
      tst->sig_mask.sig[0] = frame->sigcontext.oldmask;
      tst->sig_mask.sig[1] = frame->sigcontext._unused[3];
   }

   vg_assert(priv->magicPI == 0x31415927);

   VG_TRACK(die_mem_stack_signal, sp, frame_size);

   VG_(memcpy)(tst->arch.m_gpr, &mc->mc_gregs[VKI_PT_R0], 32 * sizeof(UWord));
   tst->arch.m_eip = mc->mc_gregs[VKI_PT_NIP];
   tst->arch.m_orig_gpr3 = mc->mc_gregs[VKI_PT_ORIG_R3];
   tst->arch.m_cr = mc->mc_gregs[VKI_PT_CCR];
   tst->arch.m_lr = mc->mc_gregs[VKI_PT_LNK];
   tst->arch.m_ctr = mc->mc_gregs[VKI_PT_CTR];
   tst->arch.m_xer = mc->mc_gregs[VKI_PT_XER];
   if (VG_(needs).shadow_regs) {
      VG_(memcpy)(tst->arch.sh_gpr, priv->sh_gpr, 32 * sizeof(UWord));
      tst->arch.sh_cr = priv->sh_cr;
      tst->arch.sh_lr = priv->sh_lr;
      tst->arch.sh_ctr = priv->sh_ctr;
      tst->arch.sh_xer = priv->sh_xer;
   }

   tst->status = priv->status;

   tst->associated_mx = priv->associated_mx;
   tst->associated_cv = priv->associated_cv;

   return priv->sigNo_private;
}


/* The generated code keeps some ArchRegs in real CPU registers.
   If that is the case at present, save them back to the baseBlock. */
void VGA_(flush_state)(struct vki_ucontext *uc)
{
   unsigned long *regs = uc->uc_regs->mc_gregs;

   if (VG_(dispatch_sp) && VG_(dispatch_sp) == regs[1]) {
	   VG_(dispatch_ctr) = regs[17];
	   VG_(baseBlock)[VGOFF_(m_cr)]  = regs[18];
	   VG_(baseBlock)[VGOFF_(m_xer)] = regs[19];
	   VG_(baseBlock)[VGOFF_(m_eip)] = regs[30];
   }

#if 0
   /* Maybe the floating point state is in the FP register image
      on the stack.  If so, copy it off before we longjmp. */
   if (VG_(baseBlock)[VGOFF_(fpu_state_ptr)] == 0) {
	   ThreadState *tst = &VG_(threads)[VG_(get_current_or_recent_tid)()];

	   VG_(memcpy)(tst->arch.m_fpr, uc->uc_regs->fpr, 33 * sizeof(double));
	   VG_(baseBlock)[VGOFF_(fpu_state_ptr)] = (Addr) tst->arch.m_fpr;
   }
#endif

   /* Copy altivec state back to the thread state */
   if ((VG_(hardware_capabilities) & VKI_HWCAP_ALTIVEC)
       && VG_(baseBlock)[VGOFF_(vec_state_ptr)] == 0) {
	   ThreadState *tst = &VG_(threads)[VG_(get_current_or_recent_tid)()];

	   VG_(save_vec_state)(&tst->arch.m_vr[0]);
	   VG_(baseBlock)[VGOFF_(vec_state_ptr)] = (Addr) &tst->arch.m_vr[0];
   }
}

void VGA_(fill_elfregs_from_BB)(struct vki_user_regs_struct *regs)
{
   /* XXX implement me */
}

void VGA_(fill_elfregs_from_tst)(struct vki_user_regs_struct *regs,
				 const arch_thread_t *tst)
{
   /* XXX implement me */
}

void VGA_(fill_elffpregs_from_BB)(vki_elf_fpregset_t *fpu)
{
   /* XXX implement me */
}

void VGA_(fill_elffpxregs_from_BB)(vki_elf_fpxregset_t *xfpu)
{
   /* XXX implement me */
}

void VGA_(fill_elffpregs_from_tst)(vki_elf_fpregset_t *fpu,
				   const arch_thread_t *tst)
{
   /* XXX implement me */
}

void VGA_(fill_elffpxregs_from_tst)(vki_elf_fpxregset_t *xfpu,
				    const arch_thread_t *tst)
{
   /* XXX implement me */
}
