
/*--------------------------------------------------------------------*/
/*--- PowerPC-Linux-specific stuff for the core.                   ---*/
/*				    powerpc-linux/core_platform.h  ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Valgrind, an extensible
   emulator for monitoring program execution on Unixes.

   Copyright (C) 2004 Paul Mackerras
      paulus@samba.org

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#ifndef __PPC_LINUX_CORE_PLATFORM_H
#define __PPC_LINUX_CORE_PLATFORM_H

/* ---------------------------------------------------------------------
   Interesting registers
   ------------------------------------------------------------------ */

// Accessors for the arch_thread_t
#define PLATFORM_SYSCALL_NUM(regs)	((regs).m_gpr[0])
#define PLATFORM_SYSCALL_RET(regs)	((regs).m_result)
#define PLATFORM_SYSCALL_ARG1(regs)	((regs).m_orig_gpr3)
#define PLATFORM_SYSCALL_ARG2(regs)	((regs).m_gpr[4])
#define PLATFORM_SYSCALL_ARG3(regs)	((regs).m_gpr[5])
#define PLATFORM_SYSCALL_ARG4(regs)	((regs).m_gpr[6])
#define PLATFORM_SYSCALL_ARG5(regs)	((regs).m_gpr[7])
#define PLATFORM_SYSCALL_ARG6(regs)	((regs).m_gpr[8])

// Interesting register numbers
#define R_SYSCALL_NUM	0
#define R_SYSCALL_ARG1	3
#define R_SYSCALL_ARG2	4
#define R_SYSCALL_ARG3	5
#define R_SYSCALL_ARG4	6
#define R_SYSCALL_ARG5	7
#define R_SYSCALL_ARG6	8
#define R_SYSCALL_RET	3

/* XXX should this be setting shadow regs? */
/* N.B. this uses val more than once */
#define PLATFORM_SET_SYSCALL_RESULT(regs, val)			\
   do {								\
      UInt __val = (val);					\
      if (__val <= -4096) {					\
         (regs).m_gpr[3] = __val;				\
         (regs).m_cr &= ~0x10000000;	/* clear cr0.SO */	\
      } else {							\
	 (regs).m_gpr[3] = - __val;				\
	 (regs).m_cr |= 0x10000000;				\
      }								\
      (regs).m_result = __val;					\
      (regs).m_gpr[0] = 0;					\
   } while (0)

#define PLATFORM_SYSCALL_GPR3(regs)	((regs).m_gpr[3])
#define PLATFORM_SYSCALL_CR(regs)	((regs).m_cr)

#define SET_SYSCALL_RETVAL(zztid, zzval)				\
   do {									\
      UInt __cr = VG_(threads)[zztid].arch.m_cr & ~0x10000000;		\
      UInt __val = (zzval);						\
      VG_(threads)[zztid].arch.m_result = __val;			\
      if (__val > -4096) {						\
	 __val = - __val;						\
	 __cr |= 0x10000000;						\
      }									\
      SET_THREAD_REG(zztid, __val, PLATFORM_SYSCALL_GPR3, 3,		\
		     post_reg_write_syscall_return);			\
      /* XXX this isn't right because it makes the whole CR valid. */	\
      SET_THREAD_REG(zztid, __cr, PLATFORM_SYSCALL_CR, R_CR,		\
		     post_reg_write_syscall_return);			\
   } while (0)

/* ---------------------------------------------------------------------
   ucontext stuff
   ------------------------------------------------------------------ */

#define UCONTEXT_INSTR_PTR(ucp)		((ucp)->uc_regs->mc_gregs[VKI_PT_NIP])
#define UCONTEXT_STACK_PTR(ucp)		((ucp)->uc_regs->mc_gregs[1])
#define UCONTEXT_FRAME_PTR(ucp)		((ucp)->uc_regs->mc_gregs[1])
#define UCONTEXT_SYSCALL_NUM(ucp)	((ucp)->uc_regs->mc_gregs[0])
#define UCONTEXT_SYSCALL_RET(ucp)	((ucp)->uc_regs->mc_gregs[3])

/* ---------------------------------------------------------------------
   mmap() stuff
   ------------------------------------------------------------------ */

#define PLATFORM_DO_MMAP(ret, start, length, prot, flags, fd, offset) {	\
   ret = VG_(do_syscall)(__NR_mmap, (UWord)(start), (length),		\
			 (prot), (flags), (fd), (offset));		\
}

#define PLATFORM_GET_MMAP_ARGS(tst, a1, a2, a3, a4, a5, a6) do {	\
   a1 = PLATFORM_SYSCALL_ARG1(tst->arch);				\
   a2 = PLATFORM_SYSCALL_ARG2(tst->arch);				\
   a3 = PLATFORM_SYSCALL_ARG3(tst->arch);				\
   a4 = PLATFORM_SYSCALL_ARG4(tst->arch);				\
   a5 = PLATFORM_SYSCALL_ARG5(tst->arch);				\
   a6 = PLATFORM_SYSCALL_ARG6(tst->arch);				\
} while (0)

#define PLATFORM_DO_SYSCALL_DIRECT(sysnr, tst) {	\
   VGA_(do_syscall_direct)(sysnr, tst);			\
} while (0)

extern void VGA_(do_syscall_direct)(Int sysnr, arch_thread_t *tp);

/* Atomic operations with futexes. */

/* atomic decrement if non-negative, returns new value */
static __inline__ Int __futex_down(Int *counter)
{
    Int val;

    __asm__ __volatile__(
	"1: lwarx %0,0,%1\n"
	"   cmpwi 0,%0,0\n"
	"   blt 2f\n"
	"   addi %0,%0,-1\n"
	"   stwcx. %0,0,%1\n"
	"   bne- 1b\n"
	"   isync\n"
	"2:"
	: "=&b" (val)
	: "r" (counter)
	: "cr0", "memory");
    return val;
}

/* atomic inc, return new value */
static __inline__ Int __atomic_inc(Int *counter)
{
    Int val;

    __asm__ __volatile__(
	"1: lwarx %0,0,%1\n"
	"   addi %0,%0,1\n"
	"   stwcx. %0,0,%1\n"
	"   bne- 1b\n"
	"   isync"
	: "=&b" (val)
	: "r" (counter)
	: "cr0", "memory");
    return val;
}

static __inline__ Int __futex_up(Int *counter)
{
    return __atomic_inc(counter) == 1;
}

static __inline__ void __futex_commit(void)
{
    __asm__ __volatile__("sync" : : : "memory");
}

/* Use libc setjmp/longjmp. */
#include <setjmp.h>

#define SETJMP(env)		setjmp(env)
#define LONGJMP(env, val)	longjmp(env, val)

extern Int VG_(clone) ( Int (*fn)(void *), void *stack, Int flags, void *arg, 
			Int *child_tid, Int *parent_tid, void * );

#endif /* __PPC_LINUX_CORE_PLATFORM_H */
