
/*--------------------------------------------------------------------*/
/*--- PowerPC-specific stuff for the core.    powerpc/core_arch.h  ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Valgrind, an extensible
   emulator for monitoring program execution on Unixes.

   Copyright (C) 2004-2005 Paul Mackerras
      paulus@samba.org

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#ifndef __PPC_CORE_ARCH_H
#define __PPC_CORE_ARCH_H

#include "core_arch_asm.h"
#include "tool_arch.h"

/* ---------------------------------------------------------------------
   Interesting registers
   ------------------------------------------------------------------ */

// Accessors for the arch_thread_t
#define ARCH_INSTR_PTR(regs)	((regs).m_eip)
#define ARCH_STACK_PTR(regs)	((regs).m_gpr[1])
#define ARCH_FRAME_PTR(regs)	((regs).m_gpr[1])
#define ARCH_RETADDR(regs)	((regs).m_lr)
#define ARCH_RETVAL(regs)	((regs).m_gpr[3])

#define ARCH_CLREQ_ARGS(regs)	((regs).m_gpr[4])
#define ARCH_PTHREQ_RET(regs)	((regs).m_gpr[3])
#define ARCH_CLREQ_RET(regs)	((regs).m_gpr[3])

// Accessors for the ThreadState
#define R_STACK_PTR                    1
#define R_FRAME_PTR                    1

#define R_CLREQ_RET                    3
#define R_PTHREQ_RET                   3

// Stack frame layout and linkage
#define FIRST_STACK_FRAME(sp)	(((UWord*)(sp))[0])
#define STACK_FRAME_RET(sp)	(((UWord*)(sp))[1])
#define STACK_FRAME_NEXT(sp)	(((UWord*)(sp))[0])

// Get stack pointer and frame pointer
#define ARCH_GET_REAL_STACK_PTR(sp) do {	\
   asm("mr %0,1" : "=r" (sp));			\
} while (0)

#define ARCH_GET_REAL_FRAME_PTR(bp) do {	\
   asm("mr %0,1" : "=r" (bp));			\
} while (0)

#define ARCH_MAKE_VA_LIST(args, regs)	\
   va_list args = {{ 0, 0, (regs).m_gpr[1] + 8, &(regs).m_gpr[3] }};

/* ---------------------------------------------------------------------
   Miscellaneous constants
   ------------------------------------------------------------------ */

// Total number of spill slots available for register allocation.
#define VG_MAX_SPILLSLOTS 64

// Valgrind's stack size, in words
#define VG_STACK_SIZE_W    16384

// base address of client address space
#define CLIENT_BASE	0x00000000ul

/* ---------------------------------------------------------------------
   Architecture-specific part of a ThreadState
   ------------------------------------------------------------------ */
typedef struct arch_thread {
   UInt m_gpr[32];
   UInt m_eip;
   UInt m_ctr;
   UInt m_lr;
   UInt m_xer;
   UInt m_cr;
   UInt sh_gpr[32];
   UInt sh_ctr;
   UInt sh_lr;
   UInt sh_xer;
   UInt sh_cr;
   UInt m_orig_gpr3;
   UInt m_result;

   Word spillslots[VG_MAX_SPILLSLOTS];

   double m_fpr[33];		/* includes fpscr as [32] */

   /* vector state; needs to be aligned on a 16-byte boundary */
   UInt m_vr[33*4] __attribute__((__aligned__(16)));
   Bool vr_live;		/* vector state is in machine regs */

   Addr dispatch_sp;
} arch_thread_t;


/* ---------------------------------------------------------------------
   Elf stuff
   ------------------------------------------------------------------ */

#define VG_ELF_ENDIANNESS     ELFDATA2MSB
#define VG_ELF_MACHINE        EM_PPC
#define VG_ELF_CLASS          ELFCLASS32


/* ---------------------------------------------------------------------
   Exports of vg_ppc_helpers.S
   ------------------------------------------------------------------ */

extern void VG_(load_vec_state)(ThreadState *tst);
extern void VG_(save_vec_state)(ThreadState *tst);

extern void VG_(helper_cache_inval)(Addr address);

/* How vg_tt and VG_(tt_fast) are indexed. */
#define VG_TT_HASH(addr)        (((UInt)(addr) >> 2) % VG_TT_SIZE)
#define VG_TT_FAST_INDEX(addr)  (((UInt)(addr) >> 2) & VG_TT_FAST_MASK)

#define DO_ARCH_REDIRECTS

void call_on_stack(void (*func)(void), unsigned int stacktop)
     __attribute__((noreturn));
void call_on_stack1(void (*func)(UInt), unsigned int stacktop, UInt arg)
     __attribute__((noreturn));

void VGA_(signal_return)(ThreadId tid, Bool isRT);
void VGA_(flush_state)(struct vki_ucontext *uc);

/*
 * If USE_MACHINE_RESERVATION is defined, we use real lwarx and
 * stwcx. for doing lwarx and stwcx. on the synthetic cpu, i.e., we
 * use the real machine reservation as the synthetic cpu's
 * reservation.  If it is not defined, we emulate the reservation
 * using the VG_(reservation) variable, which gets set non-zero when
 * the reservation is taken, and cleared on context switch or stwcx.
 */
#define USE_MACHINE_RESERVATION

#ifndef USE_MACHINE_RESERVATION
extern Addr VG_(reservation);
#define ARCH_SWITCH_TO(tid)	VG_(reservation) = 0

#else
/* On context switch, do a dummy stwcx. to clear any reservation. */
#define ARCH_SWITCH_TO(tid)	do {				\
	int __x;						\
	__asm__ __volatile__("stwcx. 0,0,%0" : : "r" (&__x));	\
} while (0)

#endif

#endif /* __PPC_CORE_ARCH_H */
